/**
 * @file errors.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <glib.h>
#include <log.h>
#include <errors.h>
#include "inc/declarations.h"

void
wapdman_error_failed (GDBusMethodInvocation *invocation,
                      int errnum)
{
    errnum = abs (errnum);
    const char *str = strerror (errnum);

    switch (errnum) {
    case ENXIO:
        wapdman_error_no_such_device (invocation);
        break;

    case EACCES:
        wapdman_error_permission_denied (invocation);
        break;

    case EEXIST:
        wapdman_error_already_exists (invocation);
        break;

    case EINVAL:
        wapdman_error_invalid_arguments (invocation);
        break;

    case ENOSYS:
        wapdman_error_not_implemented (invocation);
        break;

    case ENODATA:
        wapdman_error_no_data (invocation);
        break;

    case ENOLINK:
        wapdman_error_no_carrier (invocation);
        break;

    case ENOTUNIQ:
        wapdman_error_not_unique (invocation);
        break;

    case EOPNOTSUPP:
        wapdman_error_not_supported (invocation);
        break;

    case ENOTCONN:
    case ECONNREFUSED:
        wapdman_error_not_connected (invocation);
        break;

    case ETIMEDOUT:
        wapdman_error_operation_timeout (invocation);
        break;

    case EALREADY:
    case EINPROGRESS:
        wapdman_error_in_progress (invocation);
        break;

    case EBUSY:
        wapdman_error_busy (invocation);
        break;

    case ENOMEM:
        wapdman_error_no_memory (invocation);
        break;

    case ECANCELED:
        wapdman_error_operation_cancelled (invocation);
        break;

    case ERROR_INTERNAL_FAILURE:
    	wapdman_error_internal_failure (invocation);
    	break;

    default:
        g_dbus_method_invocation_return_dbus_error
                (invocation, WAPDMAN_BUS_NAME".Failed", str);
        break;
    }
}

void
wapdman_error_internal_failure (GDBusMethodInvocation *invocation)
{
    return_if_fail (invocation);

    g_dbus_method_invocation_return_dbus_error
            (invocation, WAPDMAN_BUS_NAME".InternalFailure",
             "Internal failure");
}

void
wapdman_error_invalid_arguments (GDBusMethodInvocation *invocation)
{
    return_if_fail (invocation);

    g_dbus_method_invocation_return_dbus_error
            (invocation, WAPDMAN_BUS_NAME".InvalidArguments",
             "Invalid arguments");
}

void
wapdman_error_permission_denied (GDBusMethodInvocation *invocation)
{
    return_if_fail (invocation);

    g_dbus_method_invocation_return_dbus_error
            (invocation, WAPDMAN_BUS_NAME".PermissionDenied",
             "Permission denied");
}

void
wapdman_error_not_unique (GDBusMethodInvocation *invocation)
{
    return_if_fail (invocation);

    g_dbus_method_invocation_return_dbus_error
            (invocation, WAPDMAN_BUS_NAME".NotUnique",
             "Not unique");
}

void
wapdman_error_not_supported (GDBusMethodInvocation *invocation)
{
    return_if_fail (invocation);

    g_dbus_method_invocation_return_dbus_error
            (invocation, WAPDMAN_BUS_NAME".NotSupported",
             "Not supported");
}

void
wapdman_error_not_implemented (GDBusMethodInvocation *invocation)
{
    return_if_fail (invocation);

    g_dbus_method_invocation_return_dbus_error
            (invocation, WAPDMAN_BUS_NAME".NotImplemented",
             "Not implemented");
}

void
wapdman_error_no_such_device (GDBusMethodInvocation *invocation)
{
    return_if_fail (invocation);

    g_dbus_method_invocation_return_dbus_error
            (invocation, WAPDMAN_BUS_NAME".NoSuchDevice",
             "No such device");
}

void
wapdman_error_no_carrier (GDBusMethodInvocation *invocation)
{
    return_if_fail (invocation);

    g_dbus_method_invocation_return_dbus_error
            (invocation, WAPDMAN_BUS_NAME".NoCarrier",
             "No carrier");
}

void
wapdman_error_no_data (GDBusMethodInvocation *invocation)
{
    return_if_fail (invocation);

    g_dbus_method_invocation_return_dbus_error
            (invocation, WAPDMAN_BUS_NAME".NoData",
             "No Data Available");
}

void
wapdman_error_in_progress(GDBusMethodInvocation *invocation)
{
    return_if_fail (invocation);

    g_dbus_method_invocation_return_dbus_error
            (invocation, WAPDMAN_BUS_NAME".InProgress",
             "In progress");
}

void
wapdman_error_already_exists (GDBusMethodInvocation *invocation)
{
    return_if_fail (invocation);

    g_dbus_method_invocation_return_dbus_error
            (invocation, WAPDMAN_BUS_NAME".AlreadyExists",
             "Already exists");
}

void
wapdman_error_already_same (GDBusMethodInvocation *invocation)
{
    return_if_fail (invocation);

    g_dbus_method_invocation_return_dbus_error
            (invocation, WAPDMAN_BUS_NAME".AlreadySame",
             "The requested value is already the same");
}

void
wapdman_error_already_enabled (GDBusMethodInvocation *invocation)
{
    return_if_fail (invocation);

    g_dbus_method_invocation_return_dbus_error
            (invocation, WAPDMAN_BUS_NAME".AlreadyEnabled",
             "Already enabled");
}

void
wapdman_error_already_disabled (GDBusMethodInvocation *invocation)
{
    return_if_fail (invocation);

    g_dbus_method_invocation_return_dbus_error
            (invocation, WAPDMAN_BUS_NAME".AlreadyDisabled",
             "Already disabled");
}

void
wapdman_error_not_connected (GDBusMethodInvocation *invocation)
{
    return_if_fail (invocation);

    g_dbus_method_invocation_return_dbus_error
            (invocation, WAPDMAN_BUS_NAME".NotConnected",
             "Not connected");
}

void
wapdman_error_operation_timeout (GDBusMethodInvocation *invocation)
{
    return_if_fail (invocation);

    g_dbus_method_invocation_return_dbus_error
            (invocation, WAPDMAN_BUS_NAME".OperationTimeout",
             "Operation timeout, Maybe too many requests in the queue, Try again");
}

void
wapdman_error_invalid_property (GDBusMethodInvocation *invocation)
{
    return_if_fail (invocation);

    g_dbus_method_invocation_return_dbus_error
            (invocation, WAPDMAN_BUS_NAME".InvalidProperty",
             "Invalid property");
}

void
wapdman_error_invalid_interface (GDBusMethodInvocation *invocation)
{
    return_if_fail (invocation);

    g_dbus_method_invocation_return_dbus_error
            (invocation, WAPDMAN_BUS_NAME".InvalidInterface",
             "Invalid interface");
}

void
wapdman_error_no_memory (GDBusMethodInvocation *invocation)
{
    return_if_fail (invocation);

    g_dbus_method_invocation_return_dbus_error
            (invocation, WAPDMAN_BUS_NAME".OutOfMemory",
             "Out of Memory");
}

void
wapdman_error_busy (GDBusMethodInvocation *invocation)
{
    return_if_fail (invocation);

    g_dbus_method_invocation_return_dbus_error
            (invocation, WAPDMAN_BUS_NAME".Busy",
             "Resource is busy");
}

void
wapdman_error_operation_cancelled (GDBusMethodInvocation *invocation)
{
    return_if_fail (invocation);

    g_dbus_method_invocation_return_dbus_error
            (invocation, WAPDMAN_BUS_NAME".OperationCancelled",
             "Operation cancelled");
}


void
wapdman_error_read_only (GDBusMethodInvocation *invocation)
{
    return_if_fail (invocation);

    g_dbus_method_invocation_return_dbus_error
            (invocation, WAPDMAN_BUS_NAME".ReadOnly",
             "Read only");
}

/** @} */
