/**
 * @file accesspoint_if.c
 * @author RBEI/ECO3-Karthikeyan Madeswaran
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <glib.h>
#include <gio/gio.h>
#include <errno.h>
#include <string.h>
#include <ap.h>
#include <errors.h>
#include <log.h>
#include <aptasks.h>
#include <accesspoint.h>
#include <manager_if.h>
#include <accesspoint_if.h>
#include <objectmanager_if.h>
#include <associatedstation_if.h>
#include <servicereg.h>
#include "inc/utils.h"
#include <org-bosch-wapdman-accesspoint-generated.h>

#define DBUS_AP_SERVICE_EXPORTED                BIT(0)
#define DBUS_PROP_SERVICE_EXPORTED              BIT(1)
#define DBUS_DHCP_SERVICE_EXPORTED              BIT(2)
#define DBUS_TETHERING_SERVICE_EXPORTED         BIT(3)

#define AP_DBUS_INTERFACES_SIZE                 4

typedef AccesspointDHCPDNSConfiguration dhcp_dns_service_t;
typedef AccesspointTetheringSettings tethering_service_t;

struct apdbuservice
{
    char *path;
    struct access_point *ap;
    GDBusConnection *conn;
    unsigned int exportedifs;
    Accesspoint *apservice;
    OrgFreedesktopDBusProperties *propservice;
    dhcp_dns_service_t *dhcpdnsservice;
    tethering_service_t *tetheringservice;
};

static int
access_point_property_get_address (const PropertyTable *property,
                                   GVariant **variant,
                                   void *data)
{
    char *address;
    struct apdbuservice *service = data;

    return_val_if_fail (service && variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    address = access_point_get_address (service->ap);
    return_val_if_fail (address, -ENODATA);

    *variant = g_variant_new (property->type, address);
    return 0;
}

static int
access_point_property_get_interface (const PropertyTable *property,
                                     GVariant **variant,
                                     void *data)
{
    char *interface;
    struct apdbuservice *service = data;

    return_val_if_fail (service && variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    interface = access_point_get_interface (service->ap);
    return_val_if_fail (interface, -ENODATA);

    *variant = g_variant_new (property->type, interface);
    return 0;
}

static int
access_point_property_get_ssid (const PropertyTable *property,
                                GVariant **variant,
                                void *data)
{
    int ret;
    char ssid [SSID_MAX_LEN];
    GVariantBuilder *builder;
    struct apdbuservice *service = data;
    size_t length = 0, index = 0;

    return_val_if_fail (service && variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    memset (ssid, 0, sizeof (ssid));
    ret = access_point_get_ssid (service->ap, ssid, &length);
    return_val_if_fail (ret == 0, -ENODATA);

    builder = g_variant_builder_new (G_VARIANT_TYPE (property->type));
    for ( ; index < length; index++)
        g_variant_builder_add (builder, "y", ssid [index]);

    *variant = g_variant_builder_end (builder);
    g_variant_builder_unref (builder);
    return 0;
}

static int
access_point_property_get_powered (const PropertyTable *property,
                                   GVariant **variant,
                                   void *data)
{
    struct apdbuservice *service = data;

    return_val_if_fail (service && variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    *variant = g_variant_new (property->type, access_point_get_powered (service->ap));
    return 0;
}

static int
access_point_property_get_power_state (const PropertyTable *property,
                                       GVariant **variant,
                                       void *data)
{
    char *powerstate;
    struct apdbuservice *service = data;

    return_val_if_fail (service && variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    powerstate = access_point_get_power_state (service->ap);
    return_val_if_fail (powerstate, -ENODATA);

    *variant = g_variant_new (property->type, powerstate);
    return 0;
}

static int
access_point_property_get_power_failure_reason (const PropertyTable *property,
                                                GVariant **variant,
                                                void *data)
{
    char *failurereason;
    struct apdbuservice *service = data;

    return_val_if_fail (service && variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    failurereason = access_point_get_power_failure_reason (service->ap);
    return_val_if_fail (failurereason, -ENODATA);

    *variant = g_variant_new (property->type, failurereason);
    return 0;
}

static int
access_point_property_get_hidden (const PropertyTable *property,
                                  GVariant **variant,
                                  void *data)
{
    struct apdbuservice *service = data;

    return_val_if_fail (service && variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    *variant = g_variant_new (property->type, access_point_get_hidden (service->ap));
    return 0;
}

static int
access_point_property_get_security (const PropertyTable *property,
                                    GVariant **variant,
                                    void *data)
{
    const char *security;
    struct apdbuservice *service = data;

    return_val_if_fail (service && variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    security = access_point_get_security (service->ap);
    return_val_if_fail (security, -ENODATA);

    *variant = g_variant_new (property->type, security);
    return 0;
}

static int
access_point_property_get_passphrase (const PropertyTable *property,
                                      GVariant **variant,
                                      void *data)
{
    char *passphrase;
    struct apdbuservice *service = data;

    return_val_if_fail (service && variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    passphrase = access_point_get_passphrase (service->ap);
    return_val_if_fail (passphrase, -ENODATA);

    *variant = g_variant_new (property->type, passphrase);
    return 0;
}

static int
access_point_property_get_supported_channels (const PropertyTable *property,
                                              GVariant **variant,
                                              void *data)
{
    size_t i = 0;
    unsigned int *channels, *frequencies,
            chanlen = 0, freqlen = 0;
    char channel [128];
    GVariantBuilder *builder;
    struct apdbuservice *service = data;

    return_val_if_fail (service && variant, -EINVAL);
    return_val_if_fail (!*variant, -EALREADY);

    channels = access_point_get_supported_channels (service->ap, &chanlen);
    frequencies = access_point_get_supported_frequencies (service->ap, &freqlen);
    builder = g_variant_builder_new (G_VARIANT_TYPE (property->type));
    for ( ; (i < chanlen) && frequencies && channels; i++) {

        memset (channel, 0, sizeof (channel));
        sprintf (channel, "channel %u", channels [i]);

        g_variant_builder_add (builder, "{sv}", channel,
                               g_variant_new_uint16 (
                                   (unsigned short int) frequencies [i]));
    }

    *variant = g_variant_builder_end (builder);
    g_variant_builder_unref (builder);

    return 0;
}

static unsigned int
next_valid_mode (unsigned int *modes)
{
    unsigned int i = 0;

    if (*modes == 0)
        return IEEE80211_MODE_MAX;

    for ( ; i <= IEEE80211_MODE_AD; i++) {
        if (*modes & ((unsigned int) 1 << i)) {
            *modes ^= ((unsigned int) 1 << i);
            break;
        }
    }

    return i;
}

static const char*
access_point_modes2string (unsigned int mode)
{
    mode = ((unsigned int) 1 << mode);

    switch (mode) {
    case IEEE80211_MODE_A:
        return "Mode A";
    case IEEE80211_MODE_AD:
        return "Mode AD";
    case IEEE80211_MODE_B:
        return "Mode B";
    case IEEE80211_MODE_G:
        return "Mode G";
    }

    return "UNKNOWN";
}

static int
access_point_property_get_hwfeatures (const PropertyTable *property,
                                      GVariant **variant,
                                      void *data)
{
    char *ifname;
    int ret;
    unsigned int length = 0,
            index = 0;
    unsigned int modes, mode;
    unsigned int *rates = NULL;
    GVariantBuilder *hwmodes, *rtbuilder;
    struct apdbuservice *service = data;

    return_val_if_fail (service && variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    ifname = access_point_get_interface (service->ap);
    return_val_if_fail (ifname, -ENODATA);

    modes = access_point_get_hwmodes (service->ap);
    return_val_if_fail (modes != IEEE80211_MODE_MAX, -ENODATA);

    hwmodes = g_variant_builder_new (G_VARIANT_TYPE (property->type));
    while ((mode = next_valid_mode (&modes)) != IEEE80211_MODE_MAX) {

        rates = NULL; index = 0; length = 0;
        const char *mde = access_point_modes2string (mode);
        continue_if_fail (g_strcmp0 (mde, "UNKNOWN") != 0);

        ret = accesspoint_get_rates_from_mode (ifname, mode, &rates, &length);
        if (ret < 0) {
            ERROR ("Failed to get the rates supported by the mode: %s [%s] "
                   "[error: %s/%d]", mde, ifname, strerror (-ret), -ret);
            continue;
        }

        rtbuilder = g_variant_builder_new (G_VARIANT_TYPE ("aq"));
        for ( ; index < length; index++)
            g_variant_builder_add (rtbuilder, "q", rates [index]);

        g_variant_builder_add (hwmodes, "{sv}", mde, g_variant_builder_end (rtbuilder));
        g_variant_builder_unref (rtbuilder);

        g_free (rates);
    }

    *variant = g_variant_builder_end (hwmodes);
    g_variant_builder_unref (hwmodes);

    return 0;
}

static int
access_point_property_get_maxstations (const PropertyTable *property,
                                       GVariant **variant,
                                       void *data)
{
    int maxstations;
    struct apdbuservice *service = data;

    return_val_if_fail (service && variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    maxstations = access_point_get_maxstations (service->ap);
    return_val_if_fail ((maxstations >= 0), -ENODATA);

    *variant = g_variant_new (property->type, maxstations);
    return 0;
}

static int
access_point_property_get_operatingchannel (const PropertyTable *property,
                                            GVariant **variant,
                                            void *data)
{
    unsigned int channel;
    struct apdbuservice *service = data;

    return_val_if_fail (service && variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    channel = access_point_get_operating_channel (service->ap);
    *variant = g_variant_new (property->type, channel);
    return 0;
}

static int
access_point_property_get_hwmode (const PropertyTable *property,
                                  GVariant **variant,
                                  void *data)
{
    int ret;
    char *hw_mode = NULL;
    struct apdbuservice *service = data;

    return_val_if_fail (service && variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    ret = access_point_get_current_hwmode (service->ap, &hw_mode);
    return_val_if_fail (ret == 0, -ENODATA);

    *variant = g_variant_new (property->type, hw_mode);

    g_free (hw_mode);
    return 0;
}

static int
access_point_property_get_country_code (const PropertyTable *property,
                                        GVariant **variant,
                                        void *data)
{
    int ret;
    char *country_code = NULL;
    struct apdbuservice *service = data;

    return_val_if_fail (service && variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    country_code = access_point_get_country (service->ap);

    ret = is_valid_regdom (country_code);
    return_val_if_fail (ret == 0, -EINVAL);

    *variant = g_variant_new (property->type, country_code);
    return 0;
}

static int
access_point_property_get_dhcp_pools (const PropertyTable *property,
                                      GVariant **variant,
                                      void *data)
{
    int ret;
    GVariantBuilder *inner, *outer;
    unsigned int size = 0, index, i;
    void *value;
    char *fields [] = {
      "router", "start", "end",
        "netmask", "network",
        "broadcast", "leasetime"
    };
    struct apdbuservice *service = data;

    return_val_if_fail (service && variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    size = access_point_get_dhcp_pools (service->ap);
    outer = g_variant_builder_new (G_VARIANT_TYPE (property->type));

    for (index = 0; index < size; index++) {

        inner = g_variant_builder_new (G_VARIANT_TYPE ("a{sv}"));
        for (i = 0, value = NULL; i < ARRAY_SIZE (fields); ++i,
             value = NULL) {
            ret = access_point_get_dhcp_data_from_index
                    (service->ap, index, fields [i], &value);
            if (0 == ret && value) {
                if (g_str_equal (fields [i], "leasetime"))
                    g_variant_builder_add
                            (inner, "{sv}", fields [i], g_variant_new ("u", GPOINTER_TO_UINT (value)));
                else {
                    g_variant_builder_add (inner, "{sv}", fields [i], g_variant_new ("s", (char *) value));
                }
            }
        }

        g_variant_builder_add (outer, "@a{sv}", g_variant_builder_end (inner));
        g_variant_builder_unref (inner);
    }

    *variant = g_variant_builder_end (outer);
    g_variant_builder_unref (outer);
    return 0;
}

static int
access_point_property_get_tethering (const PropertyTable *property,
                                     GVariant **variant,
                                     void *data)
{
    struct apdbuservice *service = data;

    return_val_if_fail (service && variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    *variant = g_variant_new (property->type, access_point_get_tethering (service->ap));
    return 0;
}

static int
access_point_property_get_accnettype (const PropertyTable *property,
                                        GVariant **variant,
                                        void *data)
{
    unsigned int  acc_net_type = 0;
    struct apdbuservice *service = data;

    return_val_if_fail (service && variant, -EINVAL);
    return_val_if_fail (!*variant, -EEXIST);

    acc_net_type = access_point_get_accnettype (service->ap);

    *variant = g_variant_new (property->type, acc_net_type);
    return 0;
}

static int
access_point_property_set_country_code (const PropertyTable *property,
                                        GVariant *value,
                                        void *invocation,
                                        void *data)
{
    char *actual, *target = NULL;
    struct apdbuservice *service = data;

    return_val_if_fail (value && invocation && service, -EINVAL);

    DEBUG ("Property: %s value: %p invocation: %p service: %p",
           property->name, value, invocation, service);

    actual = access_point_get_country (service->ap);
    g_variant_get (value, "s", &target);

    INFO ("Target: %s Actual: %s", target ? target : "", actual ? actual : "");

    if (is_valid_regdom (target) != 0) {
        wapdman_error_invalid_arguments (invocation);
        return 0;
    }

    if (actual && target && !g_strcmp0 (actual, target)) {
        wapdman_error_already_same (invocation);
        return 0;
    }

    return ap_process_request (service->ap, invocation,
                               TASK_AP_SET_COUNTRY_CODE,
                               g_strdup (target));
}

static int
access_point_property_set_maxstations (const PropertyTable *property,
                                        GVariant *value,
                                        void *invocation,
                                        void *data)
{
    int actual, allowed;
    u_int16_t target = 0;
    struct apdbuservice *service = data;

    return_val_if_fail (value && invocation && service, -EINVAL);

    DEBUG ("Property: %s value: %p invocation: %p service: %p [%s]",
           property->name, value, invocation, service,
           access_point_get_interface (service->ap));

    actual = access_point_get_maxstations (service->ap);
    allowed = access_point_get_max_allowed_stations (service->ap);
    g_variant_get (value, "q", &target);

    INFO ("Max allowed: %d Target: %u Actual: %d", allowed, target, actual);

    if (target > allowed) {
        wapdman_error_invalid_arguments (invocation);
        return 0;
    }

    if (actual == (int) target) {
        wapdman_error_already_same (invocation);
        return 0;
    }

    return ap_process_request (service->ap, invocation,
                               TASK_AP_SET_MAX_STATIONS_ALLOWED,
                               GUINT_TO_POINTER (target));
}

static int
access_point_property_set_tethering (const PropertyTable *property,
                                     GVariant *value,
                                     void *invocation,
                                     void *data)
{
    gboolean target, actual;
    struct apdbuservice *service = data;

    return_val_if_fail (value && invocation && service, -EINVAL);

    DEBUG ("Property: %s value: %p invocation: %p service: %p",
           property->name, value, invocation, service);

    target = actual = FALSE;
    actual = access_point_get_tethering (service->ap);
    g_variant_get (value, "b", &target);

    INFO ("Target: %s Actual: %s", target ? "ON" : "OFF",
          actual ? "ON" : "OFF");

    if (actual && target) {
        wapdman_error_already_enabled (invocation);
        return 0;
    }
    else if (!actual && !target){
        wapdman_error_already_disabled (invocation);
        return 0;
    }

    return ap_process_request (service->ap, invocation,
                              TASK_AP_SET_TETHERING,
                              target ? GINT_TO_POINTER (1)
                                     : GINT_TO_POINTER (0));
}

static int
access_point_property_set_powered (const PropertyTable *property,
                                   GVariant *value,
                                   void *invocation,
                                   void *data)
{
    gboolean target, actual;
    struct apdbuservice *service = data;

    return_val_if_fail (value && invocation && service, -EINVAL);

    DEBUG ("Property: %s value: %p invocation: %p service: %p",
           property->name, value, invocation, service);

    target = actual = FALSE;
    actual = access_point_get_powered (service->ap);
    g_variant_get (value, "b", &target);

    INFO ("Target: %s Actual: %s", target ? "ON" : "OFF",
          actual ? "ON" : "OFF");

    if (actual && target) {
        wapdman_error_already_enabled (invocation);
        return 0;
    }
    else if (!actual && !target){
        wapdman_error_already_disabled (invocation);
        return 0;
    }

    return ap_process_request (service->ap, invocation,
                              TASK_AP_SET_POWERED,
                              target ? GINT_TO_POINTER (1)
                                     : GINT_TO_POINTER (0));
}

static int
access_point_property_set_hidden (const PropertyTable *property,
                                  GVariant *value,
                                  void *invocation,
                                  void *data)
{
    gboolean target, actual;
    struct apdbuservice *service = data;

    return_val_if_fail (value && invocation && service, -EINVAL);

    DEBUG ("Property: %s value: %p invocation: %p service: %p",
           property->name, value, invocation, service);

    target = actual = FALSE;
    actual = access_point_get_hidden (service->ap);
    g_variant_get (value, "b", &target);

    INFO ("Target: %s Actual: %s", target ? "ON" : "OFF",
          actual ? "ON" : "OFF");

    if (actual == target) {
        wapdman_error_already_same (invocation);
        return 0;
    }

    return ap_process_request (service->ap, invocation,
                              TASK_AP_SET_VISIBILITY,
                              target ? GINT_TO_POINTER (1)
                                     : GINT_TO_POINTER (0));
}

static int
access_point_property_set_accnettype (const PropertyTable *property,
                                  GVariant *value,
                                  void *invocation,
                                  void *data)
{
    unsigned int actual = 0;
    u_int16_t target = 0;
    struct apdbuservice *service = data;

    return_val_if_fail (value && invocation && service, -EINVAL);

    DEBUG ("Property: %s value: %p invocation: %p service: %p",
           property->name, value, invocation, service);

    actual = access_point_get_accnettype (service->ap);
    g_variant_get (value, "q", &target);

    INFO ("AccessNetworkType: Target: %u Actual: %d", target, actual);

    if (actual == target) {
        wapdman_error_already_same (invocation);
        return 0;
    }

    if(target > 15) {
        wapdman_error_invalid_arguments(invocation);
        return 0;
    }

    return ap_process_request (service->ap, invocation,
                              TASK_AP_SET_ACCESS_NETWORK_TYPE,
                              GUINT_TO_POINTER (target));
}

static int
access_point_property_set_passphrase (const PropertyTable *property,
                                      GVariant *value,
                                      void *invocation,
                                      void *data)
{
    size_t len = 0;
    int proceed = 0;
    const char *security;
    char *target = NULL, *actual;
    struct apdbuservice *service = data;;

    return_val_if_fail (value && invocation && service, -EINVAL);

    DEBUG ("Property: %s value: %p invocation: %p service: %p",
           property->name, value, invocation, service);

    actual = access_point_get_passphrase (service->ap);
    g_variant_get (value, "s", &target);

    INFO ("Target: %s Actual: %s", target, actual);

    if (actual && target && !g_strcmp0 (actual, target)) {
        wapdman_error_already_same (invocation);
        return 0;
    }

    if (!target || ((len = strlen (target)) == 0)) {
        wapdman_error_invalid_arguments (invocation);
        return 0;
    }

    security = access_point_get_security (service->ap);
    if (!g_strcmp0 (security, "wep")) {
        if (len == 5 || len == 13 || len == 16)
            proceed = 1;
        else if (len == 10 || len == 26 || len == 32) {
            if (!is_hexa_decimal (target, len))
                proceed = 1;
        }
    }
    else if (!g_strcmp0 (security, "wpa-psk") ||
             !g_strcmp0 (security, "wpa2-psk") ||
             !g_strcmp0 (security, "wpa/wpa2-psk")) {
        if (len >= 8 && len < 64)
            proceed = 1;
        else if (len == 64) {
            if (!is_hexa_decimal (target, len))
                proceed = 1;
        }
    }

    if (!proceed) {
        wapdman_error_invalid_arguments (invocation);
        return 0;
    }

    return ap_process_request (service->ap, invocation,
                              TASK_AP_SET_PASSPHRASE,
                              g_strdup (target));
}

static const PropertyTable access_point_properties [] = {
    { "MacAddress", "s", PROPERTY_FLAG_READONLY,
      NULL, access_point_property_get_address },
    { "Interface", "s", PROPERTY_FLAG_READONLY,
      NULL, access_point_property_get_interface },
    { "SSID", "ay", PROPERTY_FLAG_READONLY,
      NULL, access_point_property_get_ssid },
    { "Powered", "b", PROPERTY_FLAG_READWRITE,
      access_point_property_set_powered,
      access_point_property_get_powered },
    { "PowerState", "s", PROPERTY_FLAG_READONLY,
      NULL, access_point_property_get_power_state },
    { "PowerFailureReason", "s", PROPERTY_FLAG_READONLY,
      NULL, access_point_property_get_power_failure_reason },
    { "Hidden", "b", PROPERTY_FLAG_READWRITE,
      access_point_property_set_hidden,
      access_point_property_get_hidden },
    { "AccessNetworkType", "q", PROPERTY_FLAG_READWRITE,
      access_point_property_set_accnettype,
      access_point_property_get_accnettype },
    { "Security", "s", PROPERTY_FLAG_READWRITE,
      NULL, access_point_property_get_security },
    { "Passphrase", "s", PROPERTY_FLAG_READWRITE,
      access_point_property_set_passphrase,
      access_point_property_get_passphrase },
    { "CurrentOperatingChannel", "q", PROPERTY_FLAG_READONLY,
      NULL, access_point_property_get_operatingchannel },
    { "CurrentHwMode", "s", PROPERTY_FLAG_READONLY,
      NULL, access_point_property_get_hwmode },
    { "SupportedChannels", "a{sv}", PROPERTY_FLAG_READONLY,
      NULL, access_point_property_get_supported_channels },
    { "HWFeatures", "a{sv}", PROPERTY_FLAG_READONLY,
      NULL, access_point_property_get_hwfeatures },
    { "MaximumStationsAllowed", "q", PROPERTY_FLAG_READWRITE,
      access_point_property_set_maxstations,
      access_point_property_get_maxstations },
    { "CountryCode", "s", PROPERTY_FLAG_READWRITE,
      access_point_property_set_country_code,
      access_point_property_get_country_code },
    { NULL }
};

static const PropertyTable properties [] = {
    { NULL }
};

static const PropertyTable dhcp_dns_conf_properties [] = {
    { "IPv4Pool", "aa{sv}", PROPERTY_FLAG_READONLY,
      NULL, access_point_property_get_dhcp_pools },
    { NULL }
};

static const PropertyTable tethering_settings_properties [] = {
    { "Tethering", "b", PROPERTY_FLAG_READWRITE,
      access_point_property_set_tethering,
      access_point_property_get_tethering },
    { NULL }
};

static InterfaceTable access_point_interfaces [] = {
    { WAPDMAN_ACCESSPOINT_INTERFACE_NAME,
      access_point_properties, NULL},
    { WAPDMAN_ACCESSPOINT_DHCPDNSCONF_INTERFACE_NAME,
      dhcp_dns_conf_properties, NULL},
    { WAPDMAN_ACCESSPOINT_TETHERING_SETTINGS_INTERFACE_NAME,
      tethering_settings_properties, NULL},
    { ORG_FREEDESKTOP_PROPERTIES, properties, NULL},
    { NULL }
};

static int
access_point_register_interfaces (struct apdbuservice *service)
{
    int ret = 0;
    const InterfaceTable *interface;

    return_val_if_fail (service, -EINVAL);

    interface = access_point_interfaces;
    for ( ; interface && interface->name; interface++) {
        ret = objman_register_interfaces (service->path, interface->name,
                                          interface->properties, service);
        if (ret < 0) {
            ERROR ("Failed to register the interface [\"%s\"]"
                   "for the object : %s [error: %s/%d]", interface->name,
                   service->path, strerror (-ret), -ret);
        }
    }

    return ret;
}

static int
access_point_unregister_interfaces (struct apdbuservice *service)
{
    int ret = 0;
    const InterfaceTable *interface;

    return_val_if_fail (service, -EINVAL);

    interface = access_point_interfaces;
    for ( ; interface && interface->name; interface++) {
        ret = objman_unregister_interfaces (service->path, interface->name);
        if (ret < 0) {
            ERROR ("Failed to register the interface [\"%s\"]"
                   "for the object : %s [error: %s/%d]", interface->name,
                   service->path, strerror (-ret), -ret);
        }
    }

    return ret;
}

static gboolean
access_point_handle_get (OrgFreedesktopDBusProperties *object,
                         GDBusMethodInvocation *invocation,
                         const char *interfacename,
                         const char *property,
                         gpointer data)
{
    int found = 0;
    GVariant *value = NULL, *var;
    const PropertyTable *properties;
    const InterfaceTable *interfaces;
    struct apdbuservice *service = data;

    return_val_if_fail (service && object && invocation &&
                        interfacename && property, FALSE);

    DEBUG ("Object: %p Interface: \"%s\" Property: \"%s\"",
           object, interfacename, property);

    interfaces = access_point_interfaces;
    for ( ; interfaces && interfaces->name; interfaces++)
        if (!g_strcmp0 (interfacename, interfaces->name)) {
            found = 1;
            break;
        }

    if (!found) {
        wapdman_error_invalid_interface (invocation);
        return TRUE;
    }

    found = 0;
    properties = interfaces->properties;
    for ( ; properties && properties->name; properties++)
        if (!g_strcmp0 (properties->name, property)) {
            found = 1;
            break;
        }

    if (!found) {
        wapdman_error_invalid_property (invocation);
        return TRUE;
    }

    if (!properties->get) {
        wapdman_error_not_implemented (invocation);
        return TRUE;
    }

    found = properties->get (properties, &value, service);
    if (found < 0) {
        ERROR ("Failed to get the \"%s\" value", property);
        wapdman_error_failed (invocation, -1);
        return TRUE;
    }

    var = g_variant_new_variant (value);
    org_freedesktop_dbus_properties_complete_get (object, invocation, var);
    return TRUE;
}

static gboolean
access_point_handle_set (OrgFreedesktopDBusProperties* object,
                         GDBusMethodInvocation* invocation,
                         const char *interface,
                         const char *property,
                         GVariant *value,
                         gpointer data)
{
    int found = 0;
    GVariant *variant = NULL;
    const PropertyTable *properties;
    const InterfaceTable *interfaces;

    return_val_if_fail (object && invocation && interface
                        && property && value, FALSE);

    DEBUG ("Object: %p Interface: \"%s\" Property: \"%s\"",
           object, interface, property);

    interfaces = access_point_interfaces;
    for ( ; interfaces && interfaces->name; interfaces++)
        if (!g_strcmp0 (interface, interfaces->name)) {
            found = 1;
            break;
        }

    if (!found) {
        wapdman_error_invalid_interface (invocation);
        return TRUE;
    }

    properties = interfaces->properties;
    found = 0; //Reset the value of found , set in Ln 752
    for ( ; properties && properties->name; properties++)
        if (!g_strcmp0 (properties->name, property)) {
            found = 1;
            break;
        }

    if (!found) {
        wapdman_error_invalid_property (invocation);
        return TRUE;
    }

    if (!properties->set) {
        wapdman_error_read_only (invocation);
        return TRUE;
    }

    variant = g_variant_get_variant (value);
    DEBUG ("Signature of the arguments: %s",
          g_variant_get_type_string (variant));
    if (g_strcmp0 (properties->type,
                   g_variant_get_type_string (variant))) {
        wapdman_error_invalid_arguments (invocation);
        return TRUE;
    }

    found = properties->set (properties, variant, invocation, data);
    if (found < 0) {
        if (found == -EINPROGRESS) {
            wapdman_error_in_progress (invocation);
        } else
            ERROR ("Failed to set the \"%s\": %s/%d",
                   property, strerror (-found), -found);
    }

    g_variant_unref (variant);
    return TRUE;
}

static gboolean
access_point_handle_getall (OrgFreedesktopDBusProperties *object,
                            GDBusMethodInvocation *invocation,
                            const char *interface,
                            gpointer data)
{
    int found = 0, ret;
    GVariant *variant = NULL;
    const PropertyTable *property;
    const InterfaceTable *interfaces;
    GVariantBuilder *propbuilder;

    return_val_if_fail (object && invocation &&
                        interface && data, FALSE);

    DEBUG ("Object: %p invocation: %p interface: %s",
           object, invocation, interface);

    interfaces = access_point_interfaces;
    for ( ; interfaces && interfaces->name; interfaces++) {
        if (!g_strcmp0 (interface, interfaces->name)) {
            found = 1;
            break;
        }
    }

    if (!found) {
        wapdman_error_invalid_interface (invocation);
        return TRUE;
    }

    propbuilder = g_variant_builder_new (G_VARIANT_TYPE ("a{sv}"));
    property = interfaces->properties;
    for ( ; property && property->name; property++, variant = NULL) {

        continue_if_fail (property->get && property->type);
        ret = property->get (property, &variant, data);
        if (ret < 0) {
            ERROR ("Failed to get the \"%s\" value: %s/%d",
                   property->name, strerror (-ret), -ret);
            continue;
        }

        g_variant_builder_add (propbuilder, "{sv}", property->name, variant);
    }

    org_freedesktop_dbus_properties_complete_get_all
            (object, invocation, g_variant_builder_end (propbuilder));
    g_variant_builder_unref (propbuilder) ;

    return TRUE;
}

static gboolean
access_point_handle_getassociatedstations (Accesspoint *object,
                                           GDBusMethodInvocation *invocation,
                                           gpointer data)
{
    int ret;
    GVariantBuilder *builder;
    struct apdbuservice *service = data;

    return_val_if_fail (object && invocation && service, FALSE);

    DEBUG ("Object: %p invocation: %p AP: %s", object, invocation,
           service->path ? service->path : "unknown");

    builder = g_variant_builder_new (G_VARIANT_TYPE ("a(oa{sv})"));
    ret = access_point_get_associatedstations (service, builder);
    if (ret < 0) {

        ERROR ("Failed to get the stations associated to the ap: %s [error: "
               "%s/%d]", service->path ? service->path : "unknown",
               strerror (-ret), -ret);
        wapdman_error_failed (invocation, ret);

        g_variant_builder_unref (builder);
        return TRUE;
    }

    accesspoint_complete_get_associated_stations (object, invocation, g_variant_builder_end (builder));
    g_variant_builder_unref (builder);

    return TRUE;
}

int
access_point_handle_results (struct apdbuservice *ap,
                             GDBusMethodInvocation *invocation,
                             int result,
                             int opcode,
                             void *data)
{
    int ret;
    aptask id = (aptask) opcode;

    return_val_if_fail (ap, -EINVAL);

    DEBUG ("AP: %s invocation: %p result: %d opcode: %s data: %p",
           ap->path, invocation, result, ap_task2string (id),
           data);

    switch (id) {

    case TASK_AP_REGISTER_VENDOR_IE:
    case TASK_AP_UNREGISTER_VENDOR_IE:
    {
        struct vendor_element *vendorie = data;

        if (!result && vendorie) {
            ret = access_point_set_vendor_elements (ap->ap, vendorie->ies, vendorie->length);
            if (ret < 0)
                ERROR ("Failed to update the vendor elements : %s/%d", strerror (-ret), -ret);
        }

        if (invocation) {
            if (result < 0) {
                wapdman_error_failed (invocation, result);
            } else {
                if (TASK_AP_REGISTER_VENDOR_IE == id) {
                    accesspoint_complete_register_vendor_service (ap->apservice, invocation);
                } else
                    accesspoint_complete_unregister_vendor_service (ap->apservice, invocation);
            }
        }

        if (vendorie) {
            g_free (vendorie->ies);
            g_free (vendorie);
        }
    }
    break;

    case TASK_AP_SET_POWERED:
    {
        if (result < 0) {
            wapdman_error_failed (invocation, result);
        } else {
            if (invocation)
                org_freedesktop_dbus_properties_complete_set (ap->propservice, invocation);
        }
    }
    break;

    case TASK_AP_SET_TETHERING:
    {
        gboolean tethering = GPOINTER_TO_INT (data) ? TRUE : FALSE;

        ret = access_point_set_tethering (ap->ap,  tethering);
        if (ret < 0)
            ERROR ("Failed to store the visibility: %s/%d",
                   strerror (-ret), -ret);

        if (result < 0) {
            wapdman_error_failed (invocation, result);
        } else
            org_freedesktop_dbus_properties_complete_set (ap->propservice, invocation);
    }
    break;

    case TASK_AP_SET_VISIBILITY:
    {
        gboolean hidden = GPOINTER_TO_INT (data) ? TRUE : FALSE;

        ret = access_point_set_hidden (ap->ap,  hidden);
        if (ret < 0)
            ERROR ("Failed to store the visibility: %s", strerror (-ret));

        if (result < 0) {
            wapdman_error_failed (invocation, result);
        } else
            org_freedesktop_dbus_properties_complete_set (ap->propservice, invocation);
    }
    break;

    case TASK_AP_SET_SSID:
    {
        struct set_ssid_data *ssiddata = data;

        if (result < 0) {
            wapdman_error_failed (invocation, result);
        } else {
            if (ssiddata) {
                ret = access_point_set_ssid (ap->ap, ssiddata->ssid, ssiddata->length);
                if (ret < 0)
                    ERROR ("Failed to store the SSID : %s/%d", strerror (-ret), -ret);
            }
            accesspoint_complete_set_ssid (ap->apservice, invocation);
        }

        g_free (ssiddata);
    }
    break;

    case TASK_AP_DEAUTHENTICATE_STATION:
    {
        if (result < 0) {
            wapdman_error_failed (invocation, result);
        } else
            accesspoint_complete_de_authenticate_station (ap->apservice, invocation);
        g_free (data);
    }
    break;

    case TASK_AP_SET_PASSPHRASE:
    {
        if (result < 0) {
            wapdman_error_failed (invocation, result);
        } else {
            ret = access_point_set_passphrase (ap->ap, (char *) data);
            if (ret < 0)
                ERROR ("Failed to save the passphrase [%s]: %s/%d", (char *) data,
                       strerror (-ret), -ret);
            org_freedesktop_dbus_properties_complete_set (ap->propservice, invocation);
        }
        g_free (data);
    }
    break;

    case TASK_AP_SET_COUNTRY_CODE:
    {
        if (result < 0) {
            wapdman_error_failed (invocation, result);
        } else {
            ret = access_point_set_country (ap->ap, (char *)data);
            if (ret < 0)
                ERROR ("Failed to save the passphrase [%s]: %s/%d", (char *) data,
                       strerror (-ret), -ret);
            org_freedesktop_dbus_properties_complete_set (ap->propservice, invocation);
        }
        g_free (data);
    }
    break;

    case TASK_AP_SET_OPERATING_CHANNELS:
    {
        struct set_operating_channels *channels = data;

        if (result < 0) {
            wapdman_error_failed (invocation, result);
        } else {
            if (channels) {
                ret = access_point_set_channels (ap->ap, channels->channels, channels->length);
                if (ret < 0)
                    ERROR ("Failed to store the channels: %s/%d", strerror (-ret), -ret);
            }
            accesspoint_complete_set_operating_channels (ap->apservice, invocation);
        }

        if (channels) {
            g_free (channels->channels);
            g_free (channels);
        }
    }
    break;

    case TASK_AP_SET_ACCESS_NETWORK_TYPE:
    {
        unsigned int accnettype = GPOINTER_TO_UINT (data);
        if (result < 0) {
            wapdman_error_failed (invocation, result);
        } else {
            ret = access_point_set_acc_net_type (ap->ap, accnettype);
            if (ret < 0)
                ERROR ("Failed to update access_network_type %s/%d", strerror (-ret), -ret);
            org_freedesktop_dbus_properties_complete_set (ap->propservice, invocation);
        }
    }
    break;

    case TASK_AP_CLEAR_DATA:
    {
       if (result < 0) {
           wapdman_error_failed (invocation, result);
         } else
           accesspoint_complete_clear_private_data (ap->apservice, invocation);
    }
    break;

    case TASK_AP_RESTORE_DEFAULT_PRIVATE_DATA:
    {
       if (result < 0) {
           wapdman_error_failed (invocation, result);
         } else
            accesspoint_complete_restore_default_apsettings (ap->apservice, invocation);
    }
    break;

    case TASK_AP_SET_MAX_STATIONS_ALLOWED:
    {
        unsigned int max_num_sta = GPOINTER_TO_UINT (data);

        if (result < 0) {
            wapdman_error_failed (invocation, result);
        } else {
            ret = access_point_set_max_num_station (ap->ap, (int) max_num_sta);
            if (ret < 0)
                ERROR ("Failed to update max_num_sta: %s/%d", strerror (-ret), -ret);
            org_freedesktop_dbus_properties_complete_set (ap->propservice, invocation);
        }
    }
    break;

    default:
        break;
    }

    return 0;
}

static gboolean
access_point_handle_setssid (Accesspoint *object,
                             GDBusMethodInvocation *invocation,
                             GVariant *argssid,
                             gboolean utf_ssid,
                             gpointer data)
{
    int ret;
    struct set_ssid_data *ssid_data;
    char cur_ssid [SSID_MAX_LEN];
    size_t len, length = 0;
    const char *temp;
    gboolean signature = FALSE;
    struct apdbuservice *service = data;

    return_val_if_fail (object && invocation &&
                        argssid && service, FALSE);

    DEBUG ("Object: %p Invocation: %p ssid: %p utf_ssid: %d service: %s",
           object, invocation, argssid, utf_ssid, service->path ?
               service->path : "unknown");

    signature = g_variant_is_of_type (argssid, G_VARIANT_TYPE ("ay"));
    if (!signature) {
        wapdman_error_invalid_arguments (invocation);
        return TRUE;
    }

    temp = g_variant_get_fixed_array (argssid, &len, 1);
    if (!len || len > SSID_MAX_LEN ||
            !g_utf8_validate (temp, (long) len, NULL)) {
        wapdman_error_invalid_arguments (invocation);
        return TRUE;
    }

    HEXDUMP (temp, len);
    ret = access_point_get_ssid (service->ap, cur_ssid, &length);
    if (!ret) {
        if (length == len && !memcmp (cur_ssid, temp, len)) {
            wapdman_error_already_same (invocation);
            return TRUE;
        }
    }

    ssid_data = g_try_malloc0 (sizeof (*ssid_data));
    if (!ssid_data) {
        wapdman_error_no_memory (invocation);
        return TRUE;
    }

    ssid_data->length = len;
    memcpy (ssid_data->ssid, temp, len);

    ret = ap_process_request (service->ap, invocation,
                              TASK_AP_SET_SSID, ssid_data);
    if (ret < 0) {
        if (ret == -EINPROGRESS) {
            g_free (ssid_data);
            wapdman_error_in_progress (invocation);
        } else
            ERROR ("Error failed to process the ssid request: %s",
                   strerror (-ret));
    }

    return TRUE;
}

static gboolean
access_point_handle_deauthenticatestation (Accesspoint *object,
                                           GDBusMethodInvocation *invocation,
                                           const char *address,
                                           gpointer data)
{
    char temp [12];
    int ret;
    size_t len, index, iter;
    char *station = NULL;
    struct apdbuservice *service = data;

    return_val_if_fail (object && invocation &&
                        address && service, FALSE);

    DEBUG ("Access point: %s Station to be deAuthenticated : %s",
           service->path ? service->path : "unknown", address);

    /* e.g. valid address = a1:ae:f0:3d:0e:ef */
    len = strlen (address);
    if (len != 17) {
        wapdman_error_invalid_arguments (invocation);
        return TRUE;
    }

    index = iter = 0;
    memset (temp, 0, sizeof (temp));
    for ( ; index < len; index++)
        if (address [index] != ':')
            temp [iter++] = address [index];

    ret = is_hexa_decimal (temp, sizeof (temp));
    if (ret < 0) {
        wapdman_error_invalid_arguments (invocation);
        return TRUE;
    }

    ret = is_station_connected (service->ap, address);
    if (ret < 0) {
        wapdman_error_not_connected (invocation);
        return TRUE;
    }

    station = g_strdup (address);
    ret = ap_process_request (service->ap, invocation,
                              TASK_AP_DEAUTHENTICATE_STATION,
                              station);
    if (ret < 0) {
        if (ret == -EINPROGRESS) {
            g_free (station);
            wapdman_error_in_progress (invocation);
        } else
            ERROR ("Error failed to process the deauthenticate request for "
                   "station [%s], error : %s", station, strerror (-ret));
    }

    return TRUE;
}

static gboolean
access_point_handle_setoperatingchannels (Accesspoint *object,
                                          GDBusMethodInvocation *invocation,
                                          GVariant *argchannels,
                                          gpointer data)
{
    gboolean signature = FALSE;
    unsigned int *channels;
    size_t len = 0;
    int ret;
    unsigned int index = 0;
    const u_int16_t *chans;
    struct set_operating_channels *operchans;
    struct apdbuservice *service = data;

    return_val_if_fail (object && invocation &&
                        argchannels && service, FALSE);

    DEBUG ("Object: %p invocation: %p argchannels: %p service: %s",
           object, invocation, argchannels, service->path ?
               service->path : "unknown");

    signature = g_variant_is_of_type (argchannels, G_VARIANT_TYPE ("aq"));
    if (!signature) {
        wapdman_error_invalid_arguments (invocation);
        return TRUE;
    }

    chans = g_variant_get_fixed_array (argchannels, &len, sizeof (u_int16_t));
    if (!len) {
        wapdman_error_invalid_arguments (invocation);
        return TRUE;
    }

    ret = check_valid_channels (service->ap, chans, len);
    if (ret < 0) {
        wapdman_error_invalid_arguments (invocation);
        return TRUE;
    }

    channels = g_try_malloc0 (sizeof (*channels) * len);
    if (!channels) {
        wapdman_error_no_memory (invocation);
        return TRUE;
    }

    for ( ; index < len; index++)
        channels [index] = chans [index];

    operchans = g_try_malloc0 (sizeof (*operchans));
    if (!operchans) {
        g_free (channels);
        wapdman_error_no_memory (invocation);
        return TRUE;
    }

    operchans->is5ghz = (channels_is_5ghz (channels, index) == 0)
            ? TRUE : FALSE;
    operchans->channels = channels;
    operchans->length = index;

    ret = ap_process_request (service->ap, invocation,
                              TASK_AP_SET_OPERATING_CHANNELS,
                              operchans);
    if (ret < 0) {
        if (ret == -EINPROGRESS) {
            g_free (operchans->channels);
            g_free (operchans);
            wapdman_error_in_progress (invocation);
        } else
            ERROR ("Error failed to process the ssid request: %s",
                   strerror (-ret));
    }

    return TRUE;
}

static void
vendor_elements_cleanup (gpointer data)
{
    struct vendor_element *element = data;
    return_if_fail (element);
    g_free (element->ies);
    g_free (element);
}

static gboolean
access_point_handle_registervendorservice (Accesspoint *object,
                                           GDBusMethodInvocation *invocation,
                                           GVariant *vendor_ies,
                                           gpointer data)
{
    int ret;
    gsize length = 0;
    gboolean signature;
    GVariant *temp;
    GVariantIter iter;
    struct vendor_element *element;
    const char *val;
    const char *owner;
    GSList *list = NULL;
    struct apdbuservice *service = data;

    return_val_if_fail (object && invocation &&
                        vendor_ies && service, FALSE);

    DEBUG ("Object: %p invocation: %p vendor_ies: %p [sig: %s] "
           "service: %s", object, invocation, vendor_ies,
           g_variant_get_type_string (vendor_ies), service->path ?
               service->path : "unknown");

    signature = g_variant_is_of_type (vendor_ies, G_VARIANT_TYPE ("aay"));
    if (!signature) {
        wapdman_error_invalid_arguments (invocation);
        return TRUE;
    }

    g_variant_iter_init (&iter, vendor_ies);
    while (g_variant_iter_next (&iter, "@ay", &temp)) {

        length = 0;
        val = g_variant_get_fixed_array (temp, &length, 1);
        if (!length) {
            g_variant_unref (temp);
            continue;
        }

        element = g_try_malloc0 (sizeof (*element));
        if (!element) {
            g_variant_unref (temp);
            continue;
        }

        element->length = length;
        element->ies = g_try_malloc0 (sizeof (*element->ies)
                                      * length);
        if (!element->ies) {
            g_free (element);
            g_variant_unref (temp);
            continue;
        }

        memcpy (element->ies, val, length);
        list = g_slist_append (list, element);
        g_variant_unref (temp);
    }

    if (!g_slist_length (list)) {
        wapdman_error_invalid_arguments (invocation);
        return TRUE;
    }


    owner = g_dbus_method_invocation_get_sender (invocation);
    ret = register_vendor_service (NL80211_IFTYPE_AP, owner,
                                   list, service->ap, invocation);
    if (ret < 0) {
        if (ret == -EINPROGRESS) {
            wapdman_error_in_progress (invocation);
        } else {
            ERROR ("Error failed to process the vendor elements"
                   " request : %s", strerror (-ret));
            wapdman_error_failed (invocation, ret);
        }
    }

    g_slist_free_full (list, vendor_elements_cleanup);
    return TRUE;
}

static gboolean
access_point_handle_unregistervendorservice (Accesspoint *object,
                                             GDBusMethodInvocation *invocation,
                                             GVariant *vendor_ies,
                                             gpointer data)
{
    int ret;
    gsize length = 0;
    gboolean signature;
    GVariant *temp;
    GVariantIter iter;
    struct vendor_element *element;
    const char *val;
    const char *owner;
    GSList *list = NULL;
    struct apdbuservice *service = data;

    return_val_if_fail (object && invocation &&
                        vendor_ies && service, FALSE);

    DEBUG ("Object: %p Invocation: %p vendor_ies: %p [sig: %s] "
           "Service: %s", object, invocation, vendor_ies,
           g_variant_get_type_string (vendor_ies), service->path ?
               service->path : "unknown");

    signature = g_variant_is_of_type (vendor_ies, G_VARIANT_TYPE ("aay"));
    if (!signature) {
        wapdman_error_invalid_arguments (invocation);
        return TRUE;
    }

    g_variant_iter_init (&iter, vendor_ies);
    while (g_variant_iter_next (&iter, "@ay", &temp)) {

        length = 0;
        val = g_variant_get_fixed_array (temp, &length, 1);
        if (!length) {
            g_variant_unref (temp);
            continue;
        }

        element = g_try_malloc0 (sizeof (*element));
        if (!element) {
            g_variant_unref (temp);
            continue;
        }

        element->length = length;
        element->ies = g_try_malloc0 (sizeof (*element->ies)
                                      * length);
        if (!element->ies) {
            g_free (element);
            g_variant_unref (temp);
            continue;
        }

        memcpy (element->ies, val, length);
        list = g_slist_append (list, element);
        g_variant_unref (temp);
    }

    if (!g_slist_length (list)) {
        wapdman_error_invalid_arguments (invocation);
        return TRUE;
    }


    owner = g_dbus_method_invocation_get_sender (invocation);
    ret = unregister_vendor_service (NL80211_IFTYPE_AP, owner,
                                     list, service->ap, invocation);
    if (ret < 0) {
        if (ret == -EINPROGRESS) {
            wapdman_error_in_progress (invocation);
        } else {
            ERROR ("Error failed to process the vendor elements"
                   " request: %s", strerror (-ret));
            wapdman_error_failed (invocation, ret);
        }
    }

    g_slist_free_full (list, vendor_elements_cleanup);
    return TRUE;
}

static gboolean
access_point_handle_clearprivatedata(Accesspoint *object,
                                     GDBusMethodInvocation *invocation,
                                     gpointer data)
{
    int ret;
    struct apdbuservice *service = data;

    return_val_if_fail (object && service && invocation, FALSE);

    DEBUG ("Object: %p Invocation: %p Service: %s", object, invocation,
           service->path ? service->path : "unknown");

    ret = ap_process_request (service->ap, invocation, TASK_AP_CLEAR_DATA, NULL);

    if (ret < 0) {
        if (ret == -EINPROGRESS)
            wapdman_error_in_progress (invocation);
        else {
            ERROR("Error failed to process clear AP data request: %s/%d",
                    strerror(-ret), -ret);
            wapdman_error_failed (invocation, ret);
        }
    }

    return TRUE;
}

static gboolean
access_point_handle_restoredefaultapsettings(Accesspoint *object,
                                     GDBusMethodInvocation *invocation,
                                     gpointer data)
{
    int ret = 0;
    struct apdbuservice *service = data;

    return_val_if_fail (object && service && invocation, FALSE);

    DEBUG ("Object: %p Invocation: %p Service: %s", object, invocation,
           service->path ? service->path : "unknown");

    ret = ap_process_request (service->ap, invocation,
                  TASK_AP_RESTORE_DEFAULT_PRIVATE_DATA, NULL);

   if (ret < 0) {
        if (ret == -EINPROGRESS)
            wapdman_error_in_progress (invocation);
        else {
            ERROR("Error failed to process restore default "
                  "AP Settings data request: %s/%d",
                    strerror(-ret), -ret);
            wapdman_error_failed (invocation, ret);
      }
   }

   return TRUE;
}

static void
access_point_services_create (struct apdbuservice *service)
{
    return_if_fail (service);

    service->apservice = accesspoint_skeleton_new ();
    service->dhcpdnsservice = accesspoint_dhcpdnsconfiguration_skeleton_new ();
    service->propservice = org_freedesktop_dbus_properties_skeleton_new ();
    service->tetheringservice = accesspoint_tethering_settings_skeleton_new ();

    DEBUG ("Constructed Object Path: \"%s\", "
           "Access point skeleton: \"%p\" "
           "DHCP DNS Conf skeleton: \"%p\" "
           "Tethering Settings skeleton: \"%p\" "
           "dbus properties skeleton: \"%p\"",
           service->path, service->apservice,
           service->dhcpdnsservice,
           service->tetheringservice, service->propservice);
}

static void
access_point_services_cleanup (struct apdbuservice *service)
{
    return_if_fail (service);

    if (!!(service->exportedifs & DBUS_AP_SERVICE_EXPORTED))
        g_dbus_interface_skeleton_unexport (G_DBUS_INTERFACE_SKELETON (service->apservice));
    if (!!(service->exportedifs & DBUS_DHCP_SERVICE_EXPORTED))
        g_dbus_interface_skeleton_unexport (G_DBUS_INTERFACE_SKELETON (service->dhcpdnsservice));
    if (!!(service->exportedifs & DBUS_PROP_SERVICE_EXPORTED))
        g_dbus_interface_skeleton_unexport (G_DBUS_INTERFACE_SKELETON (service->propservice));
    if (!!(service->exportedifs & DBUS_TETHERING_SERVICE_EXPORTED))
        g_dbus_interface_skeleton_unexport (G_DBUS_INTERFACE_SKELETON (service->tetheringservice));

    g_object_unref (service->apservice);
    g_object_unref (service->dhcpdnsservice);
    g_object_unref (service->propservice);
    g_object_unref (service->tetheringservice);

    service->apservice = NULL;
    service->dhcpdnsservice = NULL;
    service->propservice = NULL;
    service->tetheringservice = NULL;

    service->exportedifs = 0;
}

static int
access_point_services_export (struct apdbuservice *service)
{
    GError *error = NULL;
    const InterfaceTable *interface;
    GDBusInterfaceSkeleton *ifskeleton = NULL;
    gboolean expose = FALSE;
    unsigned int exporting = 0;

    return_val_if_fail (service, -EINVAL);

    interface = access_point_interfaces;
    for ( ; interface && interface->name; interface++,
          error = NULL, ifskeleton = NULL, exporting = 0) {

        if (!g_strcmp0 (interface->name, WAPDMAN_ACCESSPOINT_INTERFACE_NAME)) {
            ifskeleton = G_DBUS_INTERFACE_SKELETON (service->apservice);
            exporting |= DBUS_AP_SERVICE_EXPORTED;
        }
        else if (!g_strcmp0 (interface->name,
                             WAPDMAN_ACCESSPOINT_DHCPDNSCONF_INTERFACE_NAME)) {
            ifskeleton = G_DBUS_INTERFACE_SKELETON (service->dhcpdnsservice);
            exporting |= DBUS_DHCP_SERVICE_EXPORTED;
        }
        else if (!g_strcmp0 (interface->name, ORG_FREEDESKTOP_PROPERTIES)) {
            ifskeleton = G_DBUS_INTERFACE_SKELETON (service->propservice);
            exporting |= DBUS_PROP_SERVICE_EXPORTED;
        }
        else if (!g_strcmp0 (interface->name,
                             WAPDMAN_ACCESSPOINT_TETHERING_SETTINGS_INTERFACE_NAME)) {
            ifskeleton = G_DBUS_INTERFACE_SKELETON (service->tetheringservice);
            exporting |= DBUS_TETHERING_SERVICE_EXPORTED;
        }

        continue_if_fail (ifskeleton);
        expose = g_dbus_interface_skeleton_export
                (ifskeleton, service->conn, service->path, &error);
        if (!expose) {
            ERROR ("Failed to export the \"%s\" interface to the dbus: %s",
                   interface->name, error ? error->message : "unknown");
            if (error)
                g_error_free (error);
            break;
        } else { service->exportedifs |= exporting; }
    }

    if (!expose)
        access_point_services_cleanup (service);

    INFO ("Exported dbus interfaces for the ap \"%s\": %02x",
          access_point_get_interface (service->ap), service->exportedifs);

    return expose == TRUE ? 0 : -EIO;
}

static int
access_point_service_init (struct apdbuservice *service)
{
    return_val_if_fail (service, -EINVAL);
    return_val_if_fail (!service->apservice && !service->propservice, -EEXIST);
    return_val_if_fail (!service->dhcpdnsservice &&
                        !service->tetheringservice, -EEXIST);

    access_point_services_create (service);

    g_signal_connect (service->propservice, "handle-get",
                      G_CALLBACK (access_point_handle_get),
                      service);
    g_signal_connect (service->propservice, "handle-set",
                      G_CALLBACK (access_point_handle_set),
                      service);
    g_signal_connect (service->propservice, "handle-get-all",
                      G_CALLBACK (access_point_handle_getall),
                      service);

    g_signal_connect (service->apservice, "handle-get-associated-stations",
                      G_CALLBACK (access_point_handle_getassociatedstations),
                      service);
    g_signal_connect (service->apservice, "handle-set-ssid",
                      G_CALLBACK (access_point_handle_setssid),
                      service);
    g_signal_connect (service->apservice, "handle-de-authenticate-station",
                      G_CALLBACK (access_point_handle_deauthenticatestation),
                      service);
    g_signal_connect (service->apservice, "handle-set-operating-channels",
                      G_CALLBACK (access_point_handle_setoperatingchannels),
                      service);
    g_signal_connect (service->apservice, "handle-register-vendor-service",
                      G_CALLBACK (access_point_handle_registervendorservice),
                      service);
    g_signal_connect (service->apservice, "handle-unregister-vendor-service",
                      G_CALLBACK (access_point_handle_unregistervendorservice),
                      service);
    g_signal_connect (service->apservice, "handle-clear-private-data",
                      G_CALLBACK (access_point_handle_clearprivatedata),
                      service);
    g_signal_connect (service->apservice, "handle-restore-default-apsettings",
                      G_CALLBACK (access_point_handle_restoredefaultapsettings),
                      service);

    return access_point_services_export (service);
}

static int
access_point_notify (struct apdbuservice *service)
{
    int ret;

    return_val_if_fail (service, -EINVAL);

    ret = access_point_register_interfaces (service);
    if (ret < 0) {
        ERROR ("Failed to notify the object manager: %s/%d",
               strerror (-ret), -ret);
        return ret;
    }

    ret = manager_accesspoint_added (service->path);
    if (ret < 0)
        ERROR ("Failed to emit the access point added signal: %s/%d",
               strerror (-ret), -ret);

    return ret;
}

char*
access_point_get_objpath (struct apdbuservice *ap)
{
    return_val_if_fail (ap, NULL);
    return ap->path;
}

int
access_point_get_properties (struct apdbuservice *ap,
                             GVariantBuilder *builder)
{
    int ret = 0;
    GVariant *variant = NULL;
    const PropertyTable *properties;

    return_val_if_fail (ap && builder, -EINVAL);

    properties = access_point_properties;
    for ( ; properties && properties->name; properties++, variant = NULL) {

        continue_if_fail (properties->get);
        ret = properties->get (properties, &variant, ap);
        if (ret < 0) {
            ERROR ("Failed to get the property value: %s [error: %s/%d]",
                   properties->name, strerror (-ret), -ret);
            continue;
        }

        g_variant_builder_add (builder, "{sv}", properties->name, variant);
    }

    return ret;
}

void*
access_point_get_approxy (struct apdbuservice *ap)
{
    return_val_if_fail (ap, NULL);
    return ap->apservice;
}

static void
stations_added (gpointer value,
                gpointer data)
{
    int ret;
    char *path;
    GVariantBuilder *builder;
    GVariantBuilder *changed = data;
    struct station_changed *station = value;

    path = associated_station_get_objectpath (station->station);
    builder = g_variant_builder_new (G_VARIANT_TYPE ("a{sv}"));

    if (station->reason == STATION_CHANGE_ADDED) {
        ret = associated_station_get_properties (station->station, builder);
        if (ret < 0) {
            ERROR ("Failed to get the properties for the station: %s "
                   "[error: %s/%d]", associated_station_get_macaddress (
                       station->station), strerror (-ret), -ret);
        }
    }

    g_variant_builder_add (changed, "(oa{sv})", path, builder);
    g_variant_builder_unref (builder);
}

int
access_point_emit_stationsChanged (void *added,
                                   void *removed,
                                   struct apdbuservice *ap)
{
    char *path;
    size_t length, index = 0;
    GVariantBuilder *changed;
    GSList *addedstations = added;
    GSList *removedstations = removed;
    const char **removedpaths = NULL;
    struct station_changed *station;

    DEBUG ("Added stations: %u Removed stations: %u ap: %s",
           g_slist_length (addedstations),
           g_slist_length (removedstations),
           ap->path ? ap->path : "unknown");

    length = g_slist_length (removedstations);
    removedpaths = g_try_malloc0 (sizeof (char *)*(length + 1));
    return_val_if_fail (removedpaths, -ENOMEM);

    changed = g_variant_builder_new (G_VARIANT_TYPE ("a(oa{sv})"));
    g_slist_foreach (addedstations, stations_added, changed);

    for ( ; removedstations; removedstations = removedstations->next) {
        station = removedstations->data;
        continue_if_fail (station);
        path = associated_station_get_objectpath (station->station);
        removedpaths [index++] = path;
    }

    accesspoint_emit_associated_stations_changed
            (ap->apservice, g_variant_builder_end (changed), removedpaths);

    g_variant_builder_unref (changed);
    g_free (removedpaths);
    return 0;
}

int
access_point_update_property (struct apdbuservice *service,
                              const char *property)
{
    int found = 0, ret;
    GVariant *value = NULL;
    GVariantBuilder *builder;
    const PropertyTable *properties;
    const InterfaceTable *table;
    const char *invalidated [1];

    return_val_if_fail (service && property, -EINVAL);

    table = access_point_interfaces;
    for ( ; table && table->name; table++) {

        properties = table->properties;
        for ( ; properties && properties->name; properties++) {
            if (!g_strcmp0 (properties->name, property)) {
                found = 1;
                break;
            }
        }

        if (found)
            break;
    }

    return_val_if_fail (found, -ENOENT);
    ret = properties->get (properties, &value, service);
    if (ret < 0)
        return ret;

    invalidated [0] = NULL;

    builder = g_variant_builder_new (G_VARIANT_TYPE ("a{sv}"));
    g_variant_builder_add (builder, "{sv}", properties->name, value);

    org_freedesktop_dbus_properties_emit_properties_changed
            (service->propservice, table->name,
             g_variant_builder_end (builder), invalidated);

    g_variant_builder_unref (builder);
    return 0;
}

struct apdbuservice*
access_point_create_service (GDBusConnection *conn,
                             struct access_point *ap)
{
    int ret;
    struct apdbuservice *service;
    char *address, *ifname, objpath [128];

    return_val_if_fail (ap, NULL);
    address = access_point_get_address (ap);
    return_val_if_fail (address, NULL);
    ifname = access_point_get_interface (ap);
    return_val_if_fail (ifname, NULL);

    DEBUG ("New Access point service entry, Address: %s ifname: %s",
           address, ifname);

    memset (objpath, 0, sizeof (objpath));
    ret = objpath_from_addr (address, objpath);
    if (ret < 0) {
        ERROR ("Failed to compose the object path from the "
               "address: %s/%d", strerror (-ret), -ret);
        return NULL;
    }

    service = g_try_malloc0 (sizeof (*service));
    return_val_if_fail (service, NULL);

    service->ap = ap;
    service->conn = conn;
    service->path = g_strdup_printf ("%s/%s/%s_%s",
                                     WAPDMAN_BUS_PATH,
                                     ifname,
                                     WAPDMAN_ACCESSPOINT_INTERFACE_PATH,
                                     objpath);

    ret = access_point_service_init (service);
    if (ret < 0) {
        ERROR ("Failed to initialize the access point service interface: %s/%d",
               strerror (-ret), -ret);
        goto error;
    }

    ret = access_point_notify (service);
    if (ret < 0)
        ERROR ("Failed to throw a notification via dbus: %s/%d",
               strerror (-ret), -ret);

    return service;

error:
    g_free (service->path);
    g_free (service);
    return NULL;
}

int
access_point_service_destroy (struct apdbuservice *service)
{
    int ret;

    return_val_if_fail (service, -EINVAL);

    ret = manager_accesspoint_removed (service->path);
    if (ret < 0) {
        ERROR ("Failed to emit the access point added signal: %s/%d",
               strerror (-ret), -ret);
    }

    ret = access_point_unregister_interfaces (service);
    if (ret < 0) {
        ERROR ("Failed to unregister the interfaces from the "
               "object manager: %s/%d", strerror (-ret), -ret);
    }

    access_point_services_cleanup (service);
    g_free (service->path);
    g_free (service);

    return ret;
}

/** @} */
