/**
 * @file rtnl.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <arpa/inet.h>
#include <linux/if.h>
#include <linux/netlink.h>
#include <linux/rtnetlink.h>
#include <linux/if_addr.h>
#include <linux/wireless.h>
#include <netinet/ether.h>
#include <netinet/icmp6.h>
#include <net/if_arp.h>
#include <netlink/socket.h>
#include <netlink/netlink.h>
#include <netlink/msg.h>
#include <sys/ioctl.h>
#include <glib.h>
#include <rtnl.h>
#include <ipconfig.h>
#include <device.h>
#include <log.h>
#include <nlutils.h>

#define IFQDISCSIZ	32

typedef
struct __rtnl_client
{
    struct nl_sock *sk, *s_rw;
    struct nl_cb *cb;
    GIOChannel *iochannel;
    unsigned int watch;
} rtnlclient;

typedef
struct __network_interface
{
    int index;
    char *ifname;
    devicetype devtype;
} netif;

struct nla_policy rtnl_link_policy [IFLA_MAX+1] = {
    [IFLA_IFNAME]		= { .type = NLA_STRING,
                            .maxlen = IFNAMSIZ },
    [IFLA_MTU]		= { .type = NLA_U32 },
    [IFLA_TXQLEN]		= { .type = NLA_U32 },
    [IFLA_LINK]		= { .type = NLA_U32 },
    [IFLA_WEIGHT]		= { .type = NLA_U32 },
    [IFLA_MASTER]		= { .type = NLA_U32 },
    [IFLA_OPERSTATE]	= { .type = NLA_U8 },
    [IFLA_LINKMODE] 	= { .type = NLA_U8 },
    [IFLA_LINKINFO]		= { .type = NLA_NESTED },
    [IFLA_QDISC]		= { .type = NLA_STRING,
                            .maxlen = IFQDISCSIZ },
    [IFLA_STATS]		= { .minlen = sizeof (struct rtnl_link_stats) },
    [IFLA_STATS64]		= { .minlen = sizeof (struct rtnl_link_stats64) },
    [IFLA_MAP]		= { .minlen = sizeof (struct rtnl_link_ifmap) },
    [IFLA_IFALIAS]		= { .type = NLA_STRING, .maxlen = IFALIASZ },
    [IFLA_NUM_VF]		= { .type = NLA_U32 },
    [IFLA_AF_SPEC]		= { .type = NLA_NESTED },
    [IFLA_PROMISCUITY]	= { .type = NLA_U32 },
    [IFLA_NUM_TX_QUEUES]	= { .type = NLA_U32 },
    [IFLA_NUM_RX_QUEUES]	= { .type = NLA_U32 },
    [IFLA_GROUP]		= { .type = NLA_U32 },
    [IFLA_CARRIER]		= { .type = NLA_U8 },
    [IFLA_PHYS_PORT_ID]	= { .type = NLA_UNSPEC },
    [IFLA_NET_NS_PID]	= { .type = NLA_U32 },
    [IFLA_NET_NS_FD]	= { .type = NLA_U32 },
};

static GHashTable *interfacelist;
static rtnlclient *rtclient;
static GSList *registeredclients;
static GSList *rtnlrequests;
static gboolean rtnlrequested = FALSE;

static const char*
rtnl_msgtype_tostring (int type)
{
    switch (type) {
    case RTM_NEWLINK:
        return ENUMTOSTR (RTM_NEWLINK);
    case RTM_DELLINK:
        return ENUMTOSTR (RTM_DELLINK);
    case RTM_GETLINK:
        return ENUMTOSTR (RTM_GETLINK);
    case RTM_SETLINK:
        return ENUMTOSTR (RTM_SETLINK);
    case RTM_NEWADDR:
        return ENUMTOSTR (RTM_NEWADDR);
    case RTM_DELADDR:
        return ENUMTOSTR (RTM_DELADDR);
    case RTM_GETADDR:
        return ENUMTOSTR (RTM_GETADDR);
    case RTM_NEWROUTE:
        return ENUMTOSTR (RTM_NEWROUTE);
    case RTM_DELROUTE:
        return ENUMTOSTR (RTM_DELROUTE);
    case RTM_GETROUTE:
        return ENUMTOSTR (RTM_GETROUTE);
    case RTM_NEWNEIGH:
        return ENUMTOSTR (RTM_NEWNEIGH);
    case RTM_DELNEIGH:
        return ENUMTOSTR (RTM_DELNEIGH);
    case RTM_GETNEIGH:
        return ENUMTOSTR (RTM_GETNEIGH);
    case RTM_NEWRULE:
        return ENUMTOSTR (RTM_NEWRULE);
    case RTM_DELRULE:
        return ENUMTOSTR (RTM_DELRULE);
    case RTM_GETRULE:
        return ENUMTOSTR (RTM_GETRULE);
    case RTM_NEWQDISC:
        return ENUMTOSTR (RTM_NEWQDISC);
    case RTM_DELQDISC:
        return ENUMTOSTR (RTM_DELQDISC);
    case RTM_GETQDISC:
        return ENUMTOSTR (RTM_GETQDISC);
    case RTM_NEWTCLASS:
        return ENUMTOSTR (RTM_NEWTCLASS);
    case RTM_DELTCLASS:
        return ENUMTOSTR (RTM_DELTCLASS);
    case RTM_GETTCLASS:
        return ENUMTOSTR (RTM_GETTCLASS);
    case RTM_NEWTFILTER:
        return ENUMTOSTR (RTM_NEWTFILTER);
    case RTM_DELTFILTER:
        return ENUMTOSTR (RTM_DELTFILTER);
    case RTM_GETTFILTER:
        return ENUMTOSTR (RTM_GETTFILTER);
    case RTM_NEWACTION:
        return ENUMTOSTR (RTM_NEWACTION);
    case RTM_DELACTION:
        return ENUMTOSTR (RTM_DELACTION);
    case RTM_GETACTION:
        return ENUMTOSTR (RTM_GETACTION);
    case RTM_NEWPREFIX:
        return ENUMTOSTR (RTM_NEWPREFIX);
    case RTM_GETMULTICAST:
        return ENUMTOSTR (RTM_GETMULTICAST);
    case RTM_GETANYCAST:
        return ENUMTOSTR (RTM_GETANYCAST);
    case RTM_NEWNEIGHTBL:
        return ENUMTOSTR (RTM_NEWNEIGHTBL);
    case RTM_GETNEIGHTBL:
        return ENUMTOSTR (RTM_GETNEIGHTBL);
    case RTM_SETNEIGHTBL:
        return ENUMTOSTR (RTM_SETNEIGHTBL);
    case RTM_NEWNDUSEROPT:
        return ENUMTOSTR (RTM_NEWNDUSEROPT);
    case RTM_NEWADDRLABEL:
        return ENUMTOSTR (RTM_NEWADDRLABEL);
    case RTM_DELADDRLABEL:
        return ENUMTOSTR (RTM_DELADDRLABEL);
    case RTM_GETADDRLABEL:
        return ENUMTOSTR (RTM_GETADDRLABEL);
    case RTM_GETDCB:
        return ENUMTOSTR (RTM_GETDCB);
    case RTM_SETDCB:
        return ENUMTOSTR (RTM_SETDCB);
    case RTM_NEWNETCONF:
        return ENUMTOSTR (RTM_NEWNETCONF);
    case RTM_GETNETCONF:
        return ENUMTOSTR (RTM_GETNETCONF);
    case RTM_NEWMDB:
        return ENUMTOSTR (RTM_NEWMDB);
    case RTM_DELMDB:
        return ENUMTOSTR (RTM_DELMDB);
    case RTM_GETMDB:
        return ENUMTOSTR (RTM_GETMDB);
    }

    return ENUMTOSTR (UNKNOWN);
}

const char*
rtnl_scope2string (int scope)
{
    switch (scope) {
    case RT_SCOPE_UNIVERSE:
        return ENUMTOSTR (RT_SCOPE_UNIVERSE);
    case RT_SCOPE_SITE:
        return ENUMTOSTR (RT_SCOPE_SITE);
    case RT_SCOPE_LINK:
        return ENUMTOSTR (RT_SCOPE_LINK);
    case RT_SCOPE_HOST:
        return ENUMTOSTR (RT_SCOPE_HOST);
    case RT_SCOPE_NOWHERE:
        return ENUMTOSTR (RT_SCOPE_NOWHERE);
    }

    return ENUMTOSTR (RT_SCOPE_UNKNOWN);
}

const char*
rtnl_table2string (int table)
{
    switch (table) {
    case RT_TABLE_UNSPEC:
        return ENUMTOSTR (RT_TABLE_UNSPEC);
    case RT_TABLE_COMPAT:
        return ENUMTOSTR (RT_TABLE_COMPAT);
    case RT_TABLE_DEFAULT:
        return ENUMTOSTR (RT_TABLE_DEFAULT);
    case RT_TABLE_MAIN:
        return ENUMTOSTR (RT_TABLE_MAIN);
    case RT_TABLE_LOCAL:
        return ENUMTOSTR (RT_TABLE_LOCAL);
    case (int) RT_TABLE_MAX:
        return ENUMTOSTR (RT_TABLE_MAX);
    }

    return ENUMTOSTR (RT_TABLE_UNKNOWN);
}

static void
rtnl_ifdata_cleanup (gpointer data)
{
    netif *nif = data;

    return_if_fail (nif);

    DEBUG ("Cleaning up interface : %s", nif->ifname);
    g_free (nif->ifname);
    g_free (nif);
}

static int
rtnl_process_next (rtnlclient *client,
                   unsigned int sequence)
{
    int ret;
    GSList *list;
    int found = 0;
    struct nl_msg *req;
    struct nlmsghdr *nlhdr;

    DEBUG ("Message seq no.: %u", sequence);

    list = rtnlrequests;
    for ( ; list; list = list->next) {
        req = list->data;
        if (req) {
            nlhdr = nlmsg_hdr (req);
            if (nlhdr->nlmsg_seq ==  sequence) {
                found = 1;
                break;
            }
        }
    }

    if (found) {
        rtnlrequests = g_slist_remove (rtnlrequests, req);
        INFO ("Removing %s from the request queue", rtnl_msgtype_tostring (nlhdr->nlmsg_type));
        nlmsg_free (req);
    }

    /* Request the next message */
    req = g_slist_nth_data (rtnlrequests, 0);
    if (!req)
        return 0;

    return_val_if_fail (client, -EINVAL);
    nlhdr = nlmsg_hdr (req);

    INFO ("Requesting %s dump", rtnl_msgtype_tostring (nlhdr->nlmsg_type));
    ret = nl_send_auto (client->sk, req);
    if (ret < 0)
        ERROR ("Failed to post the nl msg to get %s dump, error:%s",
               rtnl_msgtype_tostring (nlhdr->nlmsg_type), nl_geterror (ret));

    return ret;
}

static int
rtnl_error_handler (struct sockaddr_nl *nla,
                    struct nlmsgerr *err,
                    void *arg)
{
    (void) nla;
    (void) arg;

    return_val_if_fail (err, NL_STOP);

    DEBUG ("Error received for Nl message with sequence no [%d] : %s/%d "
           "system error : %s/%d", err->msg.nlmsg_seq, nl_geterror (err->error),
           err->error, strerror (-err->error), -err->error);
    return NL_STOP;
}

static int
rtnl_finish_handler (struct nl_msg *msg,
                     void *arg)
{
    int ret;
    struct nlmsghdr *nlhdr;

    return_val_if_fail (msg, NL_SKIP);

    nlhdr = nlmsg_hdr (msg);
    ret = rtnl_process_next ((rtnlclient *) arg, nlhdr->nlmsg_seq);
    if (ret < 0)
        ERROR ("Failed to send the next rtnl request %s", strerror (-ret));

    return NL_SKIP;
}

static int
rtnl_ack_handler (struct nl_msg *msg,
                  void *arg)
{
    (void) arg;
    struct nlmsghdr *nlhdr;

    return_val_if_fail (msg, NL_STOP);

    nlhdr = nlmsg_hdr (msg);
    DEBUG ("Message with seq no [%d] acknowledged", nlhdr->nlmsg_seq);
    return NL_STOP;
}

static int
rtnl_seq_check (struct nl_msg *msg,
                void *arg)
{
    (void) arg;
    struct nlmsghdr *nlhdr;

    return_val_if_fail (msg, NL_OK);

    nlhdr = nlmsg_hdr(msg);
    DEBUG ("Sequence check for Message with seq no [%d]", nlhdr->nlmsg_seq);
    return NL_OK;
}

static int
rtnl_wext_interface (char *ifname)
{
    struct iwreq wrq;
    int fd, ret;

    fd = socket (PF_INET, SOCK_DGRAM | SOCK_CLOEXEC, 0);
    if (fd < 0)
        return -errno;

    memset(&wrq, 0, sizeof(wrq));
    strncpy(wrq.ifr_name, ifname, sizeof (wrq.ifr_name) - 1);

    ret = ioctl (fd, SIOCGIWNAME, &wrq);

    (void)close (fd);
    if (!ret)
        DEBUG ("%s is a wireless device", ifname);
    return ret;
}

static int
rtnl_classify_device (netif *netiface)
{
    int found = 0;
    FILE *pfile;
    char *filename, line[128];
    devicetype devtype = DEVICE_TYPE_NONE;

    return_val_if_fail (netiface, -EINVAL);

    filename = g_strdup_printf ("/sys/class/net/%s/uevent",
                                netiface->ifname);

    pfile = fopen (filename, "re");
    g_free (filename);

    if (!pfile) {
        netiface->devtype = devtype;
        return 0;
    }

    devtype = DEVICE_TYPE_ETHERNET;
    while (fgets (line, sizeof (line), pfile)) {

        char *pos;
        pos = strchr (line, '\n');
        if (!pos)
            continue;

        pos[0] = '\0';

        if (strncmp (line, "DEVTYPE=", 8) != 0)
            continue;

        found = 1;

        if (!g_strcmp0 (line + 8, "bluetooth"))
            devtype = DEVICE_TYPE_BLUETOOTH;
        else if (!g_strcmp0 (line + 8, "bond"))
            devtype = DEVICE_TYPE_BOND;
        else if (!g_strcmp0 (line + 8, "bridge"))
            devtype = DEVICE_TYPE_BRDIGE;
        else if (!g_strcmp0 (line + 8, "wwan"))
            devtype = DEVICE_TYPE_CELLULAR;
        else if (!g_strcmp0 (line + 8, "ethernet"))
            devtype = DEVICE_TYPE_ETHERNET;
        else if (!g_strcmp0 (line + 8, "gadget"))
            devtype = DEVICE_TYPE_GADGET;
        else if (!g_strcmp0 (line + 8, "hsr"))
            devtype = DEVICE_TYPE_HSR;
        else if (!g_strcmp0 (line + 8, "vlan"))
            devtype = DEVICE_TYPE_VLAN;
        else if (!g_strcmp0 (line + 8, "wimax"))
            devtype = DEVICE_TYPE_WIMAX;
        else if (!g_strcmp0 (line + 8, "wlan"))
            devtype = DEVICE_TYPE_WLAN;
        else if (!g_strcmp0 (line + 8, "vxlan"))
            devtype = DEVICE_TYPE_VXLAN;
    }

    fclose (pfile);

    if (!found) {
        if (netiface->ifname &&
                !rtnl_wext_interface (netiface->ifname)) {
            devtype = DEVICE_TYPE_WLAN;
            WARNING ("%s runs an obselete 802.11 driver (WEXT)",
                     netiface->ifname);
        }
    }

    INFO ("Identified device type : %s", device_type2str (devtype));
    netiface->devtype = devtype;
    return 0;
}

static int
rtnl_process_newlink (rtnlclient *rtnl,
                      unsigned char family,
                      unsigned short l2type,
                      int index,
                      unsigned int flags,
                      unsigned int change,
                      struct nlattr **attr)
{
    GSList *temp;
    netif *netiface;
    devicetype oldtype;
    char ifname [IFNAMSIZ];
    rtnlwatch *ops;

    return_val_if_fail (rtnl && attr, -EINVAL);

    memset (ifname, 0, IFNAMSIZ);

    if (!attr [IFLA_IFNAME])
        return 0;

    nla_strlcpy (ifname, attr [IFLA_IFNAME], IFNAMSIZ);
    if (!ifname [0])
        return 0;

    INFO ("{newlink} name : %s family : %d l2type : %d index : %d flags : %d"
          " change : %d", ifname, family, l2type, index, flags, change);

    if (flags & IFF_SLAVE) {
        INFO ("%s {newlink} ignoring slave, index %d",
              ifname, index);
        return 0;
    }

    if (l2type == ARPHRD_LOOPBACK) {
        INFO ("%s {newlink} ignoring the loopback device, index %d",
              ifname, index);
        return 0;
    }

    netiface = g_hash_table_lookup (interfacelist, GINT_TO_POINTER (index));
    if (!netiface) {

        netiface = g_try_malloc0 (sizeof (*netiface));
        return_val_if_fail (netiface, -ENOMEM);

        netiface->index = index;
        netiface->ifname = g_strdup (ifname);

        g_hash_table_insert (interfacelist, GINT_TO_POINTER (index),
                             netiface);

        if (l2type == ARPHRD_ETHER)
            rtnl_classify_device (netiface);
    } else {

        oldtype = netiface->devtype;
        /* renamed? */
        if (netiface->ifname &&
                g_strcmp0 (ifname, netiface->ifname) != 0) {

            INFO ("Device \"%s\" renamed to \"%s\"",
                  netiface->ifname, ifname);
            g_free (netiface->ifname);
            netiface->ifname = g_strdup (ifname);
        }

        /* Really possible? Device type change on the fly? */
        if (l2type == ARPHRD_ETHER) {
            rtnl_classify_device (netiface);

            if (oldtype != netiface->devtype)
                INFO ("old device type : %s new device type : %s",
                      device_type2str (oldtype),
                      device_type2str (netiface->devtype));
        }
    }

    temp = registeredclients;
    for ( ; temp; temp = temp->next) {
        ops = temp->data;
        if (ops && ops->newlink)
            ops->newlink (index, l2type, family, flags, change, attr);
    }

    switch (l2type) {
    case ARPHRD_ETHER:
    case ARPHRD_NONE:
        ipconfig_newlink (index, flags, ifname);
        break;
    }

    return 0;
}

static int
rtnl_process_dellink (rtnlclient *rtnl,
                      unsigned char family,
                      unsigned short l2type,
                      int index,
                      unsigned int flags,
                      unsigned int change,
                      struct nlattr **attr)
{
    (void) attr;
    (void) rtnl;

    GSList *temp;
    netif *interface;
    rtnlwatch *ops;

    return_val_if_fail (rtnl, -EINVAL);

    interface = g_hash_table_lookup (interfacelist, GINT_TO_POINTER(index));
    return_val_if_fail (interface, -ENOENT);

    INFO ("{dellink} name: %s family : %d l2type : %d index : %d flags : %d",
          interface->ifname, family, l2type, index, flags);

    temp = registeredclients;
    for ( ; temp; temp = temp->next) {
        ops = temp->data;
        if (ops && ops->dellink)
            ops->dellink (index, l2type, family, flags, change, interface->ifname);
    }

    switch (l2type) {
    case ARPHRD_ETHER:
    case ARPHRD_NONE:
        ipconfig_dellink (index);
        break;
    }

    g_hash_table_remove (interfacelist, GINT_TO_POINTER(index));
    return 0;
}

static int
rtnl_route_msg_parser (struct nlmsghdr *nlhdr,
                       const rtnlclient *rtnl,
                       unsigned short msgtype)
{
    (void) rtnl;

    GSList *temp;

    temp = registeredclients;
    for ( ; temp; temp = temp->next) {
        rtnlwatch *ops = temp->data;
        if (RTM_NEWROUTE == msgtype)
            if (ops && ops->newgw)
                ops->newgw (nlhdr);
        if (RTM_DELROUTE == msgtype)
            if (ops && ops->delgw)
                ops->delgw (nlhdr);
    }

    return 0;
}

static int
rtnl_addr_msg_parser (struct nlmsghdr *nlhdr,
                      const rtnlclient *rtnl,
                      unsigned short msgtype)
{
    (void) rtnl;
    int ret;
    GSList *temp;
    struct ifaddrmsg *ifa;
    struct in6_addr dst6;
    struct nlattr *tb [IFA_MAX + 1];
    struct nl_addr *addr = NULL;
    char label[IFNAMSIZ];
    char address [INET6_ADDRSTRLEN];
    char local [INET6_ADDRSTRLEN];
    char broadcast [INET6_ADDRSTRLEN];
    char anycast [INET6_ADDRSTRLEN];
    char multicast [INET6_ADDRSTRLEN];

    ret = nlmsg_parse (nlhdr, sizeof (struct ifaddrmsg),
                       tb, IFA_MAX, 0);
    if (ret < 0)
        return -ENOMSG;

    ifa = nlmsg_data (nlhdr);

    INFO ("family : %d, flags : %d, index : %d,"
          "prefixlen : %d scope : %d", ifa->ifa_family,
          ifa->ifa_flags, ifa->ifa_index,
          ifa->ifa_prefixlen, ifa->ifa_scope);

    memset (label, 0, IFNAMSIZ);
    memset (address, 0, INET6_ADDRSTRLEN);
    memset (local, 0, INET6_ADDRSTRLEN);
    memset (broadcast, 0, INET6_ADDRSTRLEN);
    memset (anycast, 0, INET6_ADDRSTRLEN);
    memset (multicast, 0, INET6_ADDRSTRLEN);

    if (tb [IFA_LABEL])
        nla_strlcpy (label, tb [IFA_LABEL], IFNAMSIZ);
    if (tb [IFA_LOCAL]) {
        if ((addr = nl_addr_alloc_attr (tb [IFA_LOCAL], ifa->ifa_family))) {
            nl_addr2str (addr, local, INET6_ADDRSTRLEN);
            nl_addr_put (addr);
        } else
            ERROR ("Failed to parse the local address");
    }
    if (tb [IFA_ADDRESS]) {
        if ((addr = nl_addr_alloc_attr (tb [IFA_ADDRESS], ifa->ifa_family))) {
            nl_addr2str (addr, address, INET6_ADDRSTRLEN);
            nl_addr_put (addr);
        } else
            ERROR ("Failed to parse the address");
    }
    if (tb [IFA_BROADCAST]) {
        if ((addr = nl_addr_alloc_attr (tb [IFA_BROADCAST], ifa->ifa_family))) {
            nl_addr2str (addr, broadcast, INET6_ADDRSTRLEN);
            nl_addr_put (addr);
        } else
            ERROR ("Failed to parse the broadcast address");
    }
    if (tb [IFA_ANYCAST]) {
        if ((addr = nl_addr_alloc_attr (tb [IFA_ANYCAST], ifa->ifa_family))) {
            nl_addr2str (addr, anycast, INET6_ADDRSTRLEN);
            nl_addr_put (addr);
        } else
            ERROR ("Failed to parse the broadcast address");
    }
    if (tb [IFA_MULTICAST]) {
        if ((addr = nl_addr_alloc_attr (tb [IFA_MULTICAST], ifa->ifa_family))) {
            nl_addr2str (addr, multicast, INET6_ADDRSTRLEN);
            nl_addr_put (addr);
        } else
            ERROR ("Failed to parse the broadcast address");
    }

    if (AF_INET6 == ifa->ifa_family) {
        inet_ntop (ifa->ifa_family, &dst6, address, sizeof(address));
        if (IN6_IS_ADDR_LINKLOCAL (&dst6))
            return 0;
    }

    INFO ("label : \"%s\" address : \"%s\" broadcast : "
          "\"%s\" local : \"%s\" anycast : \"%s\" "
          "multiast : \"%s\"", local, address, broadcast,
          local, anycast, multicast);

    temp = registeredclients;
    for ( ; temp; temp = temp->next) {

        rtnlwatch *ops = temp->data;
        if (RTM_NEWADDR == msgtype)
            if (ops && ops->newaddr)
                ops->newaddr (ifa->ifa_index, ifa->ifa_family,
                              label, local, address,
                              anycast, multicast);
        if (RTM_DELADDR == msgtype)
            if (ops && ops->deladdr)
                ops->deladdr (ifa->ifa_index, ifa->ifa_family,
                              label, local, address,
                              anycast, multicast);

    }

    return 0;
}

static int
rtnl_rule_msg_parser (struct nlmsghdr *nlhdr,
                      const rtnlclient *rtnl,
                      unsigned short msgtype)
{
    (void) rtnl;

    GSList *temp;

    temp = registeredclients;
    for ( ; temp; temp = temp->next) {
        rtnlwatch *ops = temp->data;
        if (RTM_NEWRULE == msgtype)
            if (ops && ops->newrule)
                ops->newrule (nlhdr);
        if (RTM_DELRULE == msgtype)
            if (ops && ops->delrule)
                ops->delrule (nlhdr);
    }

    return 0;
}

static int
rtnl_process_event (struct nl_msg *msg,
                    void *arg)
{
    int size, ret;
    struct rtmsg rhdr;
    struct ifaddrmsg ifhdr;
    rtnlclient *rtnl = arg;
    struct nlattr *tb [IFLA_MAX + 1];
    struct nlmsghdr *nlhdr;
    struct ifinfomsg *nlmsg;
    struct fib_rule_hdr *rule;

    return_val_if_fail (msg && rtnl, NL_SKIP);

    if (nlmsg_get_proto (msg) != NETLINK_ROUTE)
        return 0;

    nlhdr = nlmsg_hdr (msg);

    DEBUG ("Event received for the socket [%d] [seq %u] : "
           "%s [%d] sent by [%u]", nl_socket_get_fd (rtnl->sk),
           nlhdr->nlmsg_seq, rtnl_msgtype_tostring (nlhdr->nlmsg_type),
           nlhdr->nlmsg_type, nlhdr->nlmsg_pid);

    switch (nlhdr->nlmsg_type) {
    case RTM_NEWLINK:
    {
        size = sizeof (struct ifinfomsg);
        return_val_if_fail (nlmsg_valid_hdr (nlhdr, size), NL_SKIP);

        nlmsg = nlmsg_data (nlhdr);
        return_val_if_fail (nlmsg, NL_SKIP);
        ret = nlmsg_parse (nlhdr, size, tb, IFLA_MAX, rtnl_link_policy);
        if (ret < 0) {
            ERROR ("Error occured while parsing the nlmsg : %s", nl_geterror (-ret));
            return NL_SKIP;
        }

        rtnl_process_newlink (rtnl,
                              nlmsg->ifi_family,
                              nlmsg->ifi_type,
                              nlmsg->ifi_index,
                              nlmsg->ifi_flags,
                              nlmsg->ifi_change,
                              tb);
    }
    break;

    case RTM_DELLINK:
    {
        size = sizeof (struct ifinfomsg);
        return_val_if_fail (nlmsg_valid_hdr (nlhdr, size), NL_SKIP);

        nlmsg = nlmsg_data (nlhdr);
        return_val_if_fail (nlmsg, NL_SKIP);
        ret = nlmsg_parse (nlhdr, size, tb, IFLA_MAX, rtnl_link_policy);
        if (ret < 0) {
            ERROR ("Error occured while parsing the nlmsg : %s",
                   nl_geterror (-ret));
            return NL_SKIP;
        }

        rtnl_process_dellink (rtnl,
                              nlmsg->ifi_family,
                              nlmsg->ifi_type,
                              nlmsg->ifi_index,
                              nlmsg->ifi_flags,
                              nlmsg->ifi_change,
                              tb);
    }
    break;

    case RTM_DELADDR:
    case RTM_NEWADDR:
    {
        size = sizeof (rhdr);
        return_val_if_fail (nlmsg_valid_hdr (nlhdr, size), NL_SKIP);
        rtnl_addr_msg_parser (nlhdr, rtnl, nlhdr->nlmsg_type);
    }
    break;

    case RTM_DELROUTE:
    case RTM_NEWROUTE:
    {
        size = sizeof (ifhdr);
        return_val_if_fail (nlmsg_valid_hdr (nlhdr, size), NL_SKIP);
        rtnl_route_msg_parser (nlhdr, rtnl, nlhdr->nlmsg_type);
    }
    break;

    case RTM_NEWRULE:
    case RTM_DELRULE:
    {
        size = sizeof (rule);
        return_val_if_fail (nlmsg_valid_hdr (nlhdr, size), NL_SKIP);
        rtnl_rule_msg_parser (nlhdr, rtnl, nlhdr->nlmsg_type);
    }
    break;

    default:
        ERROR ("unhandled rtnl message");
        break;
    }

    return NL_SKIP;
}


static gboolean
rtnl_event (GIOChannel *chan,
            GIOCondition cond,
            gpointer data)
{
    int ret;
    rtnlclient *rtnl = data;

    return_val_if_fail (rtnl, TRUE);

    /* IO Channel no more valid */
    if (cond & (G_IO_NVAL | G_IO_HUP | G_IO_ERR)) {
        ERROR ("Error occured in GIOChannel %p", chan);
        rtnl->watch = 0;
        return FALSE;
    }

    ret = nl_recvmsgs (rtnl->sk, rtnl->cb);
    if (ret < 0) {
        ERROR ("nl80211: global message receive failed : %s [%d]",
               nl_geterror (-ret), -ret);
    }

    return TRUE;
}

static int
rtnl_send_and_recv (rtnlclient *rtnl,
                    int msgtype,
                    unsigned char family,
                    int (*handler) (struct nl_msg *, void *),
                    void *data)
{
    (void) handler;
    (void) data;

    struct nl_msg *msg;
    struct nlmsghdr *hdr;
    struct rtgenmsg rtmsg;
    int err = -ENOMEM;
    int flags = 0;

    return_val_if_fail (rtnl, -EINVAL);

    DEBUG ("Requesting the dump of: %s", rtnl_msgtype_tostring (msgtype));

    flags = NLM_F_REQUEST | NLM_F_DUMP | NLM_F_ACK;
    rtmsg.rtgen_family = family;

    msg = nlmsg_alloc ();
    return_val_if_fail (msg, -ENOMEM);

    hdr = nlmsg_put (msg, 0, 0, msgtype, sizeof (struct rtgenmsg), flags);
    if (!hdr) {
        nlmsg_free (msg);
        return -ENOMSG;
    }

    memcpy (nlmsg_data (hdr), &rtmsg, sizeof (struct rtgenmsg));
    rtnlrequests = g_slist_append (rtnlrequests, msg);

    if (g_slist_length (rtnlrequests) > 1)
        return 0;

    err = nl_send_auto (rtnl->sk, msg);
    if (err < 0)
        ERROR ("Failed to aquire the %s information , error:%s",
               rtnl_msgtype_tostring (msgtype),
               nl_geterror (err));

    return err;
}

int
rtnl_get_information ()
{
    int ret;
    size_t index = 0;
    rtnlclient *rtnl = rtclient;
    int rtnlreqs [4] = {
        RTM_GETLINK, RTM_GETADDR,
        RTM_GETROUTE, RTM_GETRULE
    };

    return_val_if_fail (rtnl, -EINVAL);

    DEBUG ("Requesting all available links, addresses, routes and rules");

    if (rtnlrequested)
        return 0;

    for ( ; index < ARRAY_SIZE (rtnlreqs); ++index) {
        ret = rtnl_send_and_recv (rtnl, rtnlreqs [index], AF_INET,
                                  rtnl_process_event, rtnl);
        if (ret < 0) {
            ERROR ("Failed to get the %s information: %s/%d",
                   rtnl_msgtype_tostring (rtnlreqs [index]),
                   strerror (-ret), -ret);
        }
    }

    rtnlrequested = TRUE;
    return ret;
}

static int
rtnl_socket_add_membership (const rtnlclient *rtnl)
{
    int ret;

    return_val_if_fail (rtnl, -EINVAL);

    ret = nl_socket_add_memberships (rtnl->sk,
                                     RTNLGRP_LINK,
                                     RTNLGRP_TC,
                                     RTNLGRP_IPV4_IFADDR,
                                     RTNLGRP_IPV4_ROUTE,
                                     RTNLGRP_IPV4_RULE,
                                     RTNLGRP_IPV4_NETCONF,
                                     RTNLGRP_IPV6_IFADDR,
                                     RTNLGRP_IPV6_ROUTE,
                                     RTNLGRP_IPV6_RULE,
                                     RTNLGRP_IPV6_NETCONF,
                                     0);

    if (ret < 0)
        ERROR ("Failed to add link memberships to the nl socket [%d] : %s",
               nl_socket_get_fd (rtnl->sk), nl_geterror (-ret));

    return ret;
}

static int
rtnl_client_deinit (rtnlclient **rtnl)
{
    rtnlclient *client = *rtnl;

    return_val_if_fail (rtnl, -EINVAL);
    return_val_if_fail (*rtnl, -EALREADY);

    if (client->watch > 0)
        g_source_remove (client->watch);

    g_io_channel_unref (client->iochannel);
    nl_cb_put (client->cb);

    nl_socket_free (client->sk);
    nl_socket_free (client->s_rw);

    g_free (client);
    *rtnl = NULL;

    return 0;
}

static int
rtnl_client_init (rtnlclient **rtnl)
{
    int ret, sk;
    rtnlclient *client;

    return_val_if_fail (rtnl && !*rtnl, -EINVAL);

    client = g_try_malloc0 (sizeof (*client));
    return_val_if_fail (client, -ENOMEM);

    client->cb = nl_cb_alloc (NL_CB_DEFAULT);
    if (!client->cb) {
        ERROR ("Failed to allocate the callback for nl socket");
        ret = -ENOMEM;
        goto error;
    }

    ret = nl_socket_init (&client->sk, NETLINK_ROUTE, FALSE, 64, 8);
    if (ret < 0)
        goto cb_cleanup;

    ret = nl_socket_init (&client->s_rw, NETLINK_ROUTE, TRUE, 8, 8);
    if (ret < 0) {
        nl_socket_free (client->sk);
        goto cb_cleanup;
    }

    ret = rtnl_socket_add_membership (client);
    if (ret < 0) {
        nl_socket_free (client->sk);
        nl_socket_free (client->s_rw);
        goto cb_cleanup;
    }

    sk = nl_socket_get_fd (client->sk);
    client->iochannel = g_io_channel_unix_new (sk);

    g_io_channel_set_close_on_unref (client->iochannel, TRUE);
    g_io_channel_set_encoding (client->iochannel, NULL, NULL);
    g_io_channel_set_buffered (client->iochannel, FALSE);

    client->watch = g_io_add_watch (client->iochannel,
                                   G_IO_IN | G_IO_NVAL |
                                   G_IO_HUP | G_IO_ERR,
                                   rtnl_event,
                                   client);

    nl_cb_err (client->cb, NL_CB_CUSTOM,
               (nl_recvmsg_err_cb_t) rtnl_error_handler,
               NULL);
    nl_cb_set (client->cb, NL_CB_FINISH, NL_CB_CUSTOM,
               rtnl_finish_handler, client);
    nl_cb_set (client->cb, NL_CB_ACK, NL_CB_CUSTOM,
               rtnl_ack_handler, NULL);
    nl_cb_set (client->cb, NL_CB_SEQ_CHECK, NL_CB_CUSTOM,
               rtnl_seq_check, NULL);
    nl_cb_set (client->cb, NL_CB_VALID, NL_CB_CUSTOM,
               rtnl_process_event, client);

    INFO ("Event socket created and the io channel setup "
          "successfully for : %d", sk);
    *rtnl = client;
    return ret;

cb_cleanup:
    nl_cb_put (client->cb);
error:
    g_free (client);
    *rtnl = NULL;
    return ret;
}

int
rtnl_notifier_register (rtnlwatch *ops)
{
    int found = 0;
    GSList *temp;

    return_val_if_fail (ops, -EINVAL);

    DEBUG ("Registering the client for rtnl update : %s [%p]",
           ops->client ? ops->client : "UNKNOWN", ops);

    temp = registeredclients;
    for ( ; temp; temp = temp->next) {
        rtnlwatch *registeredops = temp->data;
        if (ops == registeredops) {
            found = 1;
            break;
        }
    }

    if (!found) {
        registeredclients = g_slist_append (registeredclients, ops);
        return 0;
    }

    return -EALREADY;
}

int
rtnl_notifier_unregister (rtnlwatch *ops)
{
    int found = 0;
    GSList *temp;

    return_val_if_fail (ops, -EINVAL);

    DEBUG ("UnRegistering the client for rtnl update : %s [%p]",
           ops->client ? ops->client : "UNKNOWN", ops);

    temp = registeredclients;
    for ( ; temp; temp = temp->next) {
        rtnlwatch *registeredops = temp->data;
        if (ops == registeredops) {
            found = 1;
            break;
        }
    }

    if (found) {
        registeredclients = g_slist_remove_link (registeredclients,
                                                 temp);
        return 0;
    }

    return -ENOENT;
}

int
rtnl_get_device_type (int index)
{
    netif *interface;

    interface = g_hash_table_lookup (interfacelist, GINT_TO_POINTER(index));
    return_val_if_fail (interface, DEVICE_TYPE_NONE);
    return (int) interface->devtype;
}

int
rtnl_init ()
{
    int ret;

    DEBUG ("");

    ret = rtnl_client_init (&rtclient);
    if (ret < 0) {
        ERROR ("Failed to initialize the rtnl client: %s/%d",
               strerror (-ret), -ret);
        return ret;
    }

    interfacelist = g_hash_table_new_full (g_direct_hash, g_direct_equal,
                                           NULL, rtnl_ifdata_cleanup);

    return ret;
}

int
rtnl_deinit ()
{
    int ret = 0;

    DEBUG ("");

    ret = rtnl_client_deinit (&rtclient);
    if (ret < 0)
        ERROR ("Failed to deinitialize the rtnl client : %s",
               strerror (-ret));

    g_hash_table_destroy (interfacelist);
    g_slist_free (registeredclients);
    rtnlrequested = FALSE;

    return ret;
}

/** @} */
