/**
 * @file fib.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2018 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <getopt.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <linux/netlink.h>
#include <linux/fib_rules.h>
#include <linux/if.h>
#include <netlink/socket.h>
#include <netlink/msg.h>
#include <glib.h>
#include <rtnl.h>
#include <rttables.h>
#include <log.h>
#include <string.h>
#include <fib.h>
#include <nlutils.h>
#include "inc/utils.h"

#define IP_RULE_ATTR_FAMILY     BIT(0)
#define IP_RULE_ATTR_TABLE		BIT(1)
#define IP_RULE_ATTR_ACTION     BIT(2)
#define IP_RULE_ATTR_FLAGS		BIT(3)
#define IP_RULE_ATTR_IIFNAME	BIT(4)
#define IP_RULE_ATTR_OIFNAME	BIT(5)
#define IP_RULE_ATTR_PRIO		BIT(6)
#define IP_RULE_ATTR_MARK		BIT(7)
#define IP_RULE_ATTR_MASK		BIT(8)
#define IP_RULE_ATTR_GOTO		BIT(9)
#define IP_RULE_ATTR_SRC		BIT(10)
#define IP_RULE_ATTR_DST		BIT(11)
#define IP_RULE_ATTR_DSFIELD	BIT(12)
#define IP_RULE_ATTR_FLOW		BIT(13)
#define IP_RULE_ATTR_TOS		BIT(14)
#define IP_RULE_ATTR_SPRE_LEN	BIT(15)
#define IP_RULE_ATTR_IF_GRP		BIT(16)
#define IP_RULE_ATTR_INV		BIT(17)
#define IP_RULE_ATTR_SLEN		BIT(18)
#define IP_RULE_ATTR_DLEN		BIT(19)

struct _routing_rule
{
    int refcnt;
    unsigned int flags;
    unsigned int prio;
    unsigned int mark;
    unsigned int mask;
    unsigned int jump;
    unsigned int flow;
    char *src;
    char *dst;
    char *table_name;
    unsigned char family;
    unsigned char dlen;
    unsigned char slen;
    unsigned char tos;
    unsigned int table;
    unsigned char action;
    unsigned int suppress_ifgroup;
    unsigned int suppress_prefixlen;
    unsigned int realms;
    char iifname [IFNAMSIZ];
    char oifname [IFNAMSIZ];
};

typedef
struct _nl_socket
{
    struct nl_sock *sk;
    struct nl_cb *cb;
} fibsocket;

typedef
enum _rule_opcode
{
    IP_RULE_OPCODE_INV,
    IP_RULE_OPCODE_ADD,
    IP_RULE_OPCODE_INS,
    IP_RULE_OPCODE_DEL
} ruleop_t;

typedef
struct _rule_parser
{
    char **argv;
    routing_rule_t *rule;
    ruleop_t opcode;
    int argc;
    int refcnt;
    unsigned int options;
} ruleparser_t;

static struct option rule_options [] = {
    { "add",         no_argument,       NULL, 'A'},
    { "del",         no_argument,       NULL, 'D'},
    { "priority",    required_argument, NULL, 'p'},
    { "from",        required_argument, NULL, 's'},
    { "to",          required_argument, NULL, 'd'},
    { "tos",         required_argument, NULL, 'q'},
    { "fwmark",      required_argument, NULL, 'm'},
    { "table",       required_argument, NULL, 't'},
    { "suppress_prefixlength",  required_argument, NULL, 'l'},
    { "suppress_ifgroup",       required_argument, NULL, 'g'},
    { "iif",         required_argument, NULL, 'i'},
    { "oif",         required_argument, NULL, 'o'},
    { "goto",        required_argument, NULL, 'j'},
    { "realms",      required_argument, NULL, 'r'},
    { "family",      required_argument, NULL, 'f'},
    { NULL },
};

static GHashTable *routing_rules;
static fibsocket *fibsk;

static void
fib_rule_parser_unref_impl (ruleparser_t *parser,
                            const char *file,
                            int line,
                            const char *caller);
static ruleparser_t*
fib_rule_parser_ref_impl (ruleparser_t *parser,
                          const char *file,
                          int line,
                          const char *caller);

#define fib_rule_parser_ref(parser) fib_rule_parser_ref_impl(parser, __FILE__, __LINE__, __func__)
#define fib_rule_parser_unref(parser) fib_rule_parser_unref_impl(parser, __FILE__, __LINE__, __func__)

static const char*
fib_rule_opcode2str (ruleop_t cmd)
{
    switch (cmd) {
    case IP_RULE_OPCODE_INV:
        return "Invalid";
    case IP_RULE_OPCODE_ADD:
        return "Add";
    case IP_RULE_OPCODE_INS:
        return "Insert";
    case IP_RULE_OPCODE_DEL:
        return "Delete";
    }

    return "Invalid";
}

static void
fib_rule_cleanup (void *data)
{
    routing_rule_t *rule = data;

    return_if_fail (rule);

    g_free (rule->table_name);
    g_free (rule->src);
    g_free (rule->dst);
    g_free (rule);
}

static void
fib_rule_parser_cleanup (void *data)
{
    int index = 0;
    ruleparser_t *parser = data;

    return_if_fail (parser);

    fib_rule_unref (parser->rule);
    for ( ; index < parser->argc; index++)
        g_free (parser->argv [index]);
    g_free (parser);
}

static void
fib_rule_delete (void *rule)
{
    fib_rule_unref (rule);
}

static void
fib_rules_cleanup (void *rules)
{
    g_list_free_full (rules, fib_rule_delete);
}

routing_rule_t *
fib_rule_ref_impl (routing_rule_t *rule,
                   const char *file,
                   int line,
                   const char *caller)
{
    return_val_if_fail (rule, NULL);

    DEBUG ("Rule %p ref %d by %s:%d:%s()",
           rule, rule->refcnt + 1,
           file ? file : "unknown",
           line, caller ? caller : "unknown");

    __sync_fetch_and_add (&rule->refcnt, 1);
    return rule;
}

void
fib_rule_unref_impl (routing_rule_t *rule,
                     const char *file,
                     int line,
                     const char *caller)
{
    return_if_fail (rule);

    DEBUG ("Rule %p ref %d by %s:%d:%s()",
           rule, rule->refcnt - 1,
           file ? file : "unknown",
           line, caller ? caller : "unknown");

    if (__sync_fetch_and_sub (&rule->refcnt, 1) == 1)
        fib_rule_cleanup (rule);
}

static void
fib_rule_parser_unref_impl (ruleparser_t *parser,
                            const char *file,
                            int line,
                            const char *caller)
{
    return_if_fail (parser);

    DEBUG ("Rule Parser %p ref %d by %s:%d:%s()",
           parser, parser->refcnt - 1,
           file ? file : "unknown",
           line, caller ? caller : "unknown");

    if (__sync_fetch_and_sub (&parser->refcnt, 1) == 1)
        fib_rule_parser_cleanup (parser);
}

static ruleparser_t*
fib_rule_parser_ref_impl (ruleparser_t *parser,
                          const char *file,
                          int line,
                          const char *caller)
{
    return_val_if_fail (parser, NULL);

    DEBUG ("Rule Parser %p ref %d by %s:%d:%s()",
           parser, parser->refcnt + 1,
           file ? file : "unknown",
           line, caller ? caller : "unknown");

    __sync_fetch_and_add (&parser->refcnt, 1);
    return parser;
}

static int
fib_find_routing_rule (GList *rules,
                       routing_rule_t *rule,
                       routing_rule_t **fetch)
{
    GList *temp;
    int ret = -ENOENT;
    routing_rule_t *rrule;

    return_val_if_fail (rules && rule, -EINVAL);
    return_val_if_fail (fetch, -EINVAL);
    return_val_if_fail (!*fetch, -EINVAL);

    for (temp = rules; temp; temp = temp->next) {
        rrule = temp->data;

        if (rrule->family != rule->family)
            continue;
        if (rrule->table != rule->table)
            continue;
//        if (rrule->prio != rule->prio)
//            continue;
        if (rrule->mark != rule->mark)
            continue;
        if (rrule->mask != rule->mask)
            continue;
        if (g_strcmp0 (rrule->iifname, rule->iifname))
            continue;
        if (g_strcmp0 (rrule->oifname, rule->oifname))
            continue;

        if (rule->family == AF_INET) {
            if (rrule->tos != rule->tos)
                continue;
            if (rrule->dlen != rule->dlen)
                continue;
            if (rrule->slen != rule->slen)
                continue;
            if (rrule->src && !rule->src)
                continue;
            if (!rrule->src && rule->src)
                continue;
            if (rrule->src && rule->src &&
                    g_strcmp0 (rrule->src, rule->src) != 0)
                continue;
            if (rrule->dst && !rule->dst)
                continue;
            if (!rrule->dst && rule->dst)
                continue;
            if (rrule->dst && rule->dst &&
                    g_strcmp0 (rrule->dst, rule->dst) != 0)
                continue;
        }

        DEBUG ("Rule Match identified: %p == %p", rrule, rule);
        *fetch = rrule;
        ret = 0;
        break;
    }

    return ret;
}

static void
fib_dump_rule (routing_rule_t *rule)
{
    return_if_fail (rule);

    INFO (LMLN_FMAT"Dumping FIB Rule [%p], "
          LMLN_FMAT"family: \"%u\" "
          LMLN_FMAT"Routing table: \"%u\" [\"%s\"] "
          LMLN_FMAT"Routing action: \"%u\" "
          LMLN_FMAT"from: \"%s/%u\" "
          LMLN_FMAT"to: \"%s/%u\" "
          LMLN_FMAT"TOS: \"%u\" "
          LMLN_FMAT"iifname: \"%s\" "
          LMLN_FMAT"oifname: \"%s\" "
          LMLN_FMAT"priority: \"%u\" "
          LMLN_FMAT"fwmark: \"%u\" "
          LMLN_FMAT"fwmask: \"%u\" "
          LMLN_FMAT"target: \"%u\" "
          LMLN_FMAT"suppress prefix len: \"%u\" "
          LMLN_FMAT"flags: \"%02x\" %s%s%s%s%s%s",
          rule,
          rule->family,
          rule->table,
          rule->table_name ?
              rule->table_name : "unknown",
          rule->action,
          rule->src,
          rule->slen,
          rule->dst,
          rule->dlen,
          rule->tos,
          *rule->iifname ?
              rule->iifname : "unknown",
          *rule->oifname ?
              rule->oifname : "unknown",
          rule->prio,
          rule->mark,
          rule->mask,
          rule->jump,
          rule->suppress_prefixlen,
          rule->flags,
          !!(rule->flags & FIB_RULE_PERMANENT) ? "(permanent) " : "",
          !!(rule->flags & FIB_RULE_INVERT) ? "(invert) " : "",
          !!(rule->flags & FIB_RULE_UNRESOLVED) ? "(unresolved) " : "",
          !!(rule->flags & FIB_RULE_IIF_DETACHED) ? "(iif_detached) " : "",
          !!(rule->flags & FIB_RULE_DEV_DETACHED) ? "(dev_detached) " : "",
          !!(rule->flags & FIB_RULE_OIF_DETACHED) ? "(oif_detached)" : "");
}

static int
fib_create_rule (struct nlmsghdr *hdr,
                 int family,
                 routing_rule_t **route_rule)
{
    int ret;
	size_t length = 0;
    struct nlattr *tb [FRA_MAX + 1];
    routing_rule_t *rule;
    struct nla_policy fib_rule_policy [FRA_MAX + 1] = {
        [FRA_IIFNAME]	= { .type = NLA_STRING,
                            .maxlen = IFNAMSIZ },
        [FRA_OIFNAME]	= { .type = NLA_STRING,
                            .maxlen = IFNAMSIZ },
        [FRA_PRIORITY]	= { .type = NLA_U32 },
        [FRA_FWMARK]	= { .type = NLA_U32 },
        [FRA_FWMASK]	= { .type = NLA_U32 },
        [FRA_TABLE]     = { .type = NLA_U32 },
        [FRA_SUPPRESS_PREFIXLEN] = { .type = NLA_U32 },
        [FRA_SUPPRESS_IFGROUP] = { .type = NLA_U32 },
        [FRA_GOTO]	= { .type = NLA_U32 },
        [FRA_FLOW]	= { .type = NLA_U32 },
    };

    return_val_if_fail (hdr, -EINVAL);
    return_val_if_fail (route_rule, -EINVAL);
    return_val_if_fail (!*route_rule, -EALREADY);

    ret = nlmsg_parse (hdr, sizeof (struct fib_rule_hdr),
                       tb, FRA_MAX, fib_rule_policy);
    if (ret < 0)
        return -ENOMSG;

    rule = g_try_malloc0 (sizeof (*rule));
    return_val_if_fail (rule, -ENOMEM);

    rule = fib_rule_ref (rule);
    __sync_synchronize();

    if (tb [FRA_IIFNAME]) {
        length = strlen (nla_get_string (tb [FRA_IIFNAME]));
        if (length < IFNAMSIZ)
            strncpy (rule->iifname, nla_get_string (tb [FRA_IIFNAME]), length);
        else {
            ERROR ("The interface \"%s\" has name with invalid length: %u",
                   nla_get_string (tb [FRA_IIFNAME]), length);
        }
    }
    if (tb [FRA_OIFNAME]) {
        length = strlen (nla_get_string (tb [FRA_OIFNAME]));
        if (length < IFNAMSIZ)
            strncpy (rule->oifname, nla_get_string (tb [FRA_OIFNAME]), length);
        else {
            ERROR ("The interface \"%s\" has name with invalid length: %u",
                   nla_get_string (tb [FRA_OIFNAME]), length);
        }
    }
    if (tb [FRA_PRIORITY])
        rule->prio = nla_get_u32 (tb [FRA_PRIORITY]);
    if (tb [FRA_FWMARK])
        rule->mark = nla_get_u32 (tb [FRA_FWMARK]);
    if (tb [FRA_FWMASK])
        rule->mask = nla_get_u32 (tb [FRA_FWMASK]);
    if (tb [FRA_GOTO])
        rule->jump = nla_get_u32 (tb [FRA_GOTO]);
    if (tb [FRA_TABLE])
        rule->table = nla_get_u32 (tb [FRA_TABLE]);
    if (tb [FRA_SRC])
        rule->src = nl_addr2string (tb [FRA_SRC], family);
    if (tb [FRA_DST])
        rule->dst = nl_addr2string (tb [FRA_DST], family);
    if (tb [FRA_SUPPRESS_IFGROUP])
        rule->suppress_ifgroup = nla_get_u32 (tb [FRA_SUPPRESS_IFGROUP]);
    if (tb [FRA_SUPPRESS_PREFIXLEN])
        rule->suppress_prefixlen = nla_get_u32 (tb [FRA_SUPPRESS_PREFIXLEN]);

    *route_rule = rule;
    return 0;
}

static int
fib_rule_changed (void *payload,
                  routing_rule_t **rule)
{
    int ret, wlan_rule = 0;
    struct fib_rule_hdr *rhdr;
    struct nlmsghdr *nlhdr = payload;
    char *table;
    routing_rule_t *temp = NULL;

    return_val_if_fail (payload, -EINVAL);
    rhdr = nlmsg_data (nlhdr);
    return_val_if_fail (rhdr, -EINVAL);
    table = get_rt_table_name_from_id (rhdr->table);

    if (table && g_str_has_prefix (table, "WLAN_"))
        wlan_rule = 1;
    else if (rhdr->table >= 1 && rhdr->table <= 100)
        wlan_rule = 1;

    INFO ("Routing table: %s table id: %d",
          table ? table : "unknown", rhdr->table);

    if (!wlan_rule) {
        g_free (table);
        return -ENOTSUP;
    }

    ret = fib_create_rule (nlhdr, rhdr->family, rule);
    if (ret < 0) {
        ERROR ("Failed to create an internal rule entry: %s/%d",
               strerror (-ret), -ret);
        g_free (table);
        return ret;
    }

    temp = *rule;
    temp->table = rhdr->table;
    temp->tos = rhdr->tos;
    temp->flags = rhdr->flags;
    temp->table_name = table;
    temp->slen = rhdr->src_len;
    temp->dlen = rhdr->dst_len;
    temp->family = rhdr->family;

    fib_dump_rule (temp);
    return ret;
}

static int
fib_rule_added (void *payload)
{
    routing_rule_t *rule = NULL,
            *pentry = NULL;;
    int ret = 0, added = 0;
    GList *rules = NULL;

    return_val_if_fail (payload, -EINVAL);

    ret = fib_rule_changed (payload, &rule);
    return_val_if_fail (!ret, ret);

    rules = g_hash_table_lookup (routing_rules, GUINT_TO_POINTER (rule->table));
    if (!rules) {
        rules = g_list_append (rules, rule);
        g_hash_table_replace (routing_rules, GUINT_TO_POINTER (rule->table), rules);
        added = 1;
    } else {
        ret = fib_find_routing_rule (rules, rule, &pentry);
        if (ret == 0) {
            ret = -EEXIST;
            fib_rule_unref (rule);
        } else {
            rules = g_list_append (rules, rule);
            g_hash_table_replace (routing_rules, GUINT_TO_POINTER (rule->table), rules);
            added = 1;
        }
    }

    if (added)
        DEBUG ("Routing rule %p has been added to the internal list of "
               "table: %u, size: %u", rule, rule->table,
               g_list_length (rules));

    return ret;
}

static int
fib_rule_deleted (void *payload)
{
    routing_rule_t *rule = NULL,
            *pentry = NULL;
    int ret = 0;
    GList *rules = NULL;

    return_val_if_fail (payload, -EINVAL);

    ret = fib_rule_changed (payload, &rule);
    return_val_if_fail (!ret, ret);

    rules = g_hash_table_lookup (routing_rules, GUINT_TO_POINTER (rule->table));
    if (!rules) {
        fib_rule_unref (rule);
        return -ENOENT;
    }

    ret = fib_find_routing_rule (rules, rule, &pentry);
    if (ret == 0) {
        if (g_list_length (rules) == 1) {
            g_hash_table_remove (routing_rules, GUINT_TO_POINTER (rule->table));
            g_list_free_full (rules, fib_rule_delete);
            rules = NULL;
        } else {
            rules = g_list_remove (rules, pentry);
            g_hash_table_replace (routing_rules, GUINT_TO_POINTER (rule->table), rules);
            fib_rule_unref (pentry);
        }
        DEBUG ("Routing rule %p has been deleted from the internal list of "
               "table: %u, size: %u", pentry, rule->table,
               g_list_length (rules));
    }

    fib_rule_unref (rule);
    return ret;
}

static int
fib_try_allocate_parser (ruleparser_t **parser)
{
    routing_rule_t *temp;
    ruleparser_t *tparser;

    return_val_if_fail (parser && !*parser, -EINVAL);

    tparser = g_try_malloc0 (sizeof (*tparser));
    return_val_if_fail (tparser, -ENOMEM);

    tparser = fib_rule_parser_ref (tparser);
    temp = g_try_malloc0 (sizeof (*temp));
    if (!temp) {
        fib_rule_parser_unref (tparser);
        return -ENOMEM;
    }

    temp = fib_rule_ref (temp);
    tparser->rule = temp;

    *parser = tparser;
    return 0;
}

static int
fib_prepare_rule_arguments (const char *rule,
                            ruleparser_t **parser)
{
    char **argv;
    ruleparser_t *temp = NULL;
    int ret, index, length = 0;

    return_val_if_fail (rule && parser && !*parser, -EINVAL);

    ret = fib_try_allocate_parser (&temp);
    return_val_if_fail (ret == 0, -ENOMEM);

    argv = g_strsplit_set (rule, " ", -1);

    ret = calculate_argc (argv, &length);
    if (ret < 0) {
        fib_rule_parser_unref (temp);
        return ret;
    }

    temp->argc = length + 1;
    temp->argv = g_try_malloc0 ((unsigned int) (temp->argc + 1)
                                * sizeof (char *));
    if (!temp->argv) {
        fib_rule_parser_unref (temp);
        g_strfreev (argv);
        return -ENOMEM;
    }

    temp->argv[0] = g_strdup ("argh");
    for (index = 1; index < temp->argc; index++)
        temp->argv [index] = argv [index-1];

    *parser = temp;
    g_free (argv);
    return ret;
}

static int
fib_parse_rule (const char *rstring,
                ruleparser_t *rparser)
{
    char **address;
    int ret, cleanup = 0, bk,
            length = 0, invert = 0;
    unsigned int value = 0;
    size_t size = 0;

    return_val_if_fail (rstring && rparser, -EINVAL);

    opterr = 0;
    optind = 0;

    while ((ret = getopt_long (rparser->argc,
                               rparser->argv,
                               "-ADp:s:d:q:m:t:l:g:i:o:j:r:f:",
                               rule_options, NULL)) != -1) {

        DEBUG ("Supplied option: %d argument: %s", ret, optarg);

        switch (ret) {
        case 'A':
            rparser->opcode = IP_RULE_OPCODE_ADD;
            break;

        case 'D':
            rparser->opcode = IP_RULE_OPCODE_DEL;
            break;

        case 1:
        {
            if (optarg[0] == '!' && optarg [1] == '\0') {

                if (invert) {
                    cleanup = 1;
                    break;
                }

                invert = 1;
                optarg[0] = '\0';
                rparser->options |= IP_RULE_ATTR_FLAGS;
                rparser->rule->flags |= FIB_RULE_INVERT;
                continue;
            }

            cleanup = 1;
        }
        break;

        case 'i':
        case 'o':
        {
            size = strlen (optarg);
            if (*optarg == '\0' || size + 1 > IFNAMSIZ) {
                cleanup = 1;
                break;
            }

            if (ret == 'i') {
                memcpy (rparser->rule->iifname, optarg, size);
                rparser->options |= IP_RULE_ATTR_IIFNAME;
            }
            else if (ret == 'o') {
                memcpy (rparser->rule->oifname, optarg, size);
                rparser->options |= IP_RULE_ATTR_OIFNAME;
            }
        }
        break;

        case 'p':
        case 'q':
        case 'm':
        case 't':
        case 'l':
        case 'g':
        case 'j':
        case 'f':
        case 'r':
        {
            bk = ret;
            ret = convert_strtoumax (optarg, &value);
            if (ret < 0) {
                cleanup = 1;
                break;
            }

            switch (bk) {
            case 'p':
            {
                rparser->rule->prio = value;
                rparser->options |= IP_RULE_ATTR_PRIO;
            }
            break;

            case 't':
            case 'q':
            case 'f':
            {
                if (value > 255) {
                    cleanup = 1;
                    break;
                }

                if (bk == 'q') {
                    rparser->rule->tos = (unsigned char) value;
                    rparser->options |= IP_RULE_ATTR_TOS;
                }
                else if (bk == 't') {

                    if (value > 100) {
                        cleanup = 1;
                        break;
                    }

                    rparser->rule->table = value;
                    rparser->options |= IP_RULE_ATTR_TABLE;
                }
                else if (bk == 'f') {

                    if (AF_INET != value && AF_INET6 != value) {
                        cleanup = 1;
                        break;
                    }

                    rparser->rule->family = (unsigned char) value;
                    rparser->options |= IP_RULE_ATTR_FAMILY;
                }
            }
            break;

            case 'm':
            {
                rparser->rule->mark = value;
                rparser->options |= IP_RULE_ATTR_MARK;
            }
            break;

            case 'l':
            {
                rparser->rule->suppress_prefixlen = value;
                rparser->options |= IP_RULE_ATTR_SPRE_LEN;
            }
            break;

            case 'g':
            {
                rparser->rule->suppress_ifgroup = value;
                rparser->options |= IP_RULE_ATTR_IF_GRP;
            }
            break;

            case 'j':
            {
                rparser->rule->jump = value;
                rparser->options |= IP_RULE_ATTR_GOTO;
            }
            break;

            case 'r':
            {
                rparser->rule->realms = value;
                rparser->options |= IP_RULE_ATTR_FLOW;
            }
            break;
            }
        }
        break;

        case 's':
        case 'd':
        {
            bk = ret;
            length = 0;
            value = 0;
            address = g_strsplit_set (optarg, "/", -1);

            ret = calculate_argc (address, &length);
            if (ret != 0 || length > 2) {
                g_strfreev (address);
                cleanup = 1;
                break;
            }

            ret = get_family_from_addr (address [0]);
            if (ret < 0 || (ret != AF_INET &&
                            ret != AF_INET6)) {
                g_strfreev (address);
                cleanup = 1;
                break;
            }

            if (length == 2) {
                ret = convert_strtoumax (address [1], &value);
                if (ret < 0 || value > 255) {
                    g_strfreev (address);
                    cleanup = 1;
                    break;
                }
            }

            switch (bk) {
            case 's':
            {
                rparser->rule->src = g_strdup (address [0]);
                rparser->options |= IP_RULE_ATTR_SRC;

                if (value) {
                    rparser->rule->slen = (unsigned char) value;
                    rparser->options |= IP_RULE_ATTR_SLEN;
                }
            }
            break;

            case 'd':
            {
                rparser->rule->dst = g_strdup (address [0]);
                rparser->options |= IP_RULE_ATTR_DST;

                if (value) {
                    rparser->rule->dlen = (unsigned char) value;
                    rparser->options |= IP_RULE_ATTR_DLEN;
                }
            }
            break;
            }

            g_strfreev (address);
        }
        break;

        default:
            cleanup = 1;
            break;
        }

        invert = 0;
        if (cleanup)
            break;
    }

    return (cleanup == 0) ? 0 : -EINVAL;
}

static int
fib_build_nlmsg (const ruleparser_t *rparse,
                 int cmd,
                 struct nl_msg **nlmsg)
{
    struct nl_msg *msg;
    struct fib_rule_hdr rhr;
    int flags = 0;

    return_val_if_fail (rparse && nlmsg && !*nlmsg, -EINVAL);

    if (!(rparse->options & IP_RULE_ATTR_FAMILY) ||
            !(rparse->options & IP_RULE_ATTR_TABLE))
        return -EINVAL;

    memset (&rhr, 0, sizeof (struct fib_rule_hdr));

    flags = NLM_F_REQUEST | NLM_F_ACK;
    if (RTM_NEWRULE == cmd)
        flags |= NLM_F_CREATE | NLM_F_EXCL;

    msg = nlmsg_alloc_simple (cmd, flags);
    return_val_if_fail (msg, -ENOMEM);

    /* struct rtmsg == struct fib_rule_hdr */
    rhr.action = RTN_UNSPEC;
    rhr.res1 = RTPROT_BOOT;
    rhr.res2 = RT_SCOPE_UNIVERSE;
    rhr.family = rparse->rule->family;
    if (RTM_NEWRULE == cmd)
        rhr.action = RTN_UNICAST;

    if (rparse->options & IP_RULE_ATTR_TOS)
        rhr.tos = rparse->rule->tos;
    if (rparse->options & IP_RULE_ATTR_SLEN)
        rhr.src_len = rparse->rule->slen;
    if (rparse->options & IP_RULE_ATTR_DLEN)
        rhr.dst_len = rparse->rule->dlen;
    if (rparse->options & IP_RULE_ATTR_FLAGS)
        rhr.flags = rparse->rule->flags;

    if (nlmsg_append (msg, &rhr, sizeof (struct fib_rule_hdr),
                      NLMSG_ALIGNTO) < 0) {
        nlmsg_free (msg);
        return -ENOMSG;
    }

    rhr.table = rparse->rule->table
            != RT_TABLE_MAIN ?
                RT_TABLE_UNSPEC : RT_TABLE_MAIN;

    nla_put_u32 (msg, FRA_TABLE, rparse->rule->table);

    if (rparse->options & IP_RULE_ATTR_SRC)
        nl_add_address (msg, rparse->rule->family,
                         FRA_SRC, rparse->rule->src);

    if (rparse->options & IP_RULE_ATTR_DST)
        nl_add_address (msg, rparse->rule->family,
                         FRA_DST, rparse->rule->dst);

    if (rparse->options & IP_RULE_ATTR_PRIO)
        nla_put_u32 (msg, FRA_PRIORITY, rparse->rule->prio);

    if (rparse->options & IP_RULE_ATTR_MARK)
        nla_put_u32 (msg, FRA_FWMARK, rparse->rule->mark);

    if (rparse->options & IP_RULE_ATTR_MASK)
        nla_put_u32 (msg, FRA_FWMASK, rparse->rule->mask);

    if (rparse->options & IP_RULE_ATTR_GOTO)
        nla_put_u32 (msg, FRA_GOTO, rparse->rule->jump);

    if (rparse->options & IP_RULE_ATTR_FLOW)
        nla_put_u32 (msg, FRA_FLOW, rparse->rule->flow);

    if (rparse->options & IP_RULE_ATTR_IIFNAME)
        nla_put_string (msg, FRA_IIFNAME, rparse->rule->iifname);

    if (rparse->options & IP_RULE_ATTR_OIFNAME)
        nla_put_string (msg, FRA_OIFNAME, rparse->rule->oifname);

    *nlmsg = msg;
    return 0;
}

static int
fib_add_rule (const ruleparser_t *rparse)
{
    struct nl_msg *msg = NULL;
    routing_rule_t *pentry = NULL;
    GList *rules = NULL;
    int cmd = RTM_NEWRULE, ret;

    return_val_if_fail (fibsk, -ENOTCONN);

    rules = g_hash_table_lookup (routing_rules, GUINT_TO_POINTER (rparse->rule->table));
    if (rules) {
        ret = fib_find_routing_rule (rules, rparse->rule, &pentry);
        if (ret == 0)
            return -EEXIST;
    }

    ret = fib_build_nlmsg (rparse, cmd, &msg);
    if (ret < 0) {
        ERROR ("Failed to build the netlink message: %s/%d",
               strerror (-ret), -ret);
        return ret;
    }

    ret = nl_send_auto_complete (fibsk->sk, msg);
    nlmsg_free (msg);
    if (ret < 0) {
        ERROR ("Failed to send the nl msg: %s/%d",
               nl_geterror (-ret), -ret);
        return ret;
    }

    ret = nl_recvmsgs (fibsk->sk, fibsk->cb);
    if (ret < 0)
        ERROR ("Failed to add the rule: %p", msg);

    return ret;
}

static int
fib_del_rule (const ruleparser_t *rparse)
{
    struct nl_msg *msg = NULL;
    routing_rule_t *pentry = NULL;
    GList *rules = NULL;
    int cmd = RTM_DELRULE, ret;

    return_val_if_fail (fibsk, -ENOTCONN);

    rules = g_hash_table_lookup (routing_rules, GUINT_TO_POINTER (rparse->rule->table));
    return_val_if_fail (rules, -ENOENT);

    ret = fib_find_routing_rule (rules, rparse->rule, &pentry);
    return_val_if_fail (ret == 0, -ENOENT);

    ret = fib_build_nlmsg (rparse, cmd, &msg);
    if (ret < 0) {
        ERROR ("Failed to build the netlink message: %s/%d",
               strerror (-ret), -ret);
        return ret;
    }

    ret = nl_send_auto_complete (fibsk->sk, msg);
    nlmsg_free (msg);
    if (ret < 0) {
        ERROR ("Failed to send the nl msg: %s/%d",
               nl_geterror (-ret), -ret);
        return ret;
    }

    ret = nl_recvmsgs (fibsk->sk, fibsk->cb);
    if (ret < 0)
        ERROR ("Failed to del the rule: %p", msg);

    return ret;
}

static int
fib_commit_rule (const ruleparser_t *rparse)
{
    int ret = -EINVAL;

    return_val_if_fail (rparse && rparse->rule, -EINVAL);

    DEBUG ("Rule Action: %s", fib_rule_opcode2str (rparse->opcode));

    if (rparse->opcode == IP_RULE_OPCODE_ADD)
        ret = fib_add_rule (rparse);
    else if (rparse->opcode == IP_RULE_OPCODE_DEL)
        ret = fib_del_rule (rparse);

    return ret;
}

static int
fib_modify_rule (const char *rule)
{
    int ret;
    ruleparser_t *parser = NULL;

    DEBUG ("IP Rule: %s", rule);

    ret = fib_prepare_rule_arguments (rule, &parser);
    if (ret < 0) {
        ERROR ("Failed to prepare the rule arguments: %s/%d",
               strerror (-ret), -ret);
        return ret;
    }

    ret = fib_parse_rule (rule, parser);
    if (ret < 0) {
        ERROR ("Failed to parse the rule [%s]: %s/%d", rule,
               strerror (-ret), -ret);
        fib_rule_parser_unref (parser);
        return ret;
    }

    ret = fib_commit_rule (parser);
    if (ret < 0)
        ERROR ("Failed to commit the rule [%s]: %s/%d", rule,
               strerror (-ret), -ret);

    fib_rule_parser_unref (parser);
    return ret;
}

int
fib_modify_ip_rule (const char *rule)
{
    return_val_if_fail (rule, -EINVAL);
    return fib_modify_rule (rule);
}

static rtnlwatch fibops = {
    .client  = "fib",
    .newrule = fib_rule_added,
    .delrule = fib_rule_deleted
};

static int
fib_error_handler (struct sockaddr_nl *nla,
                   struct nlmsgerr *err,
                   void *arg)
{
    (void) nla;
    (void) arg;

    return_val_if_fail (err, NL_STOP);

    DEBUG ("Error received for Nl message with "
           "sequence no [%u] : %s/%d "
           "system error : %s/%d",
           err->msg.nlmsg_seq,
           nl_geterror (err->error),
           err->error, strerror (-err->error), -err->error);

    return NL_STOP;
}

static int
fib_finish_handler (struct nl_msg *msg,
                    void *arg)
{
    (void) arg;
    struct nlmsghdr *nlhdr;

    return_val_if_fail (msg, NL_SKIP);
    nlhdr = nlmsg_hdr (msg);
    DEBUG ("Message with seq no [%u] finished successfully",
           nlhdr->nlmsg_seq);
    return NL_SKIP;
}

static int
fib_ack_handler (struct nl_msg *msg,
                 void *arg)
{
    (void) arg;
    struct nlmsghdr *nlhdr;

    return_val_if_fail (msg, NL_STOP);

    nlhdr = nlmsg_hdr (msg);
    DEBUG ("Message with seq no [%u] acknowledged", nlhdr->nlmsg_seq);
    return NL_STOP;
}

static int
fib_init_transport (fibsocket **sk)
{
    int ret;
    fibsocket *fsk;

    return_val_if_fail (sk, -EINVAL);
    return_val_if_fail (!*sk, -EEXIST);

    fsk = g_try_malloc0 (sizeof (*fsk));
    return_val_if_fail (fsk, -ENOMEM);

    fsk->cb = nl_cb_alloc (NL_CB_DEFAULT);
    if (!fsk->cb) {
        g_free (fsk);
        return -ENOMEM;
    }

    ret = nl_socket_init (&fsk->sk, NETLINK_ROUTE,
                          TRUE, 0, 0);
    if (ret < 0) {
        nl_cb_put (fsk->cb);
        g_free (fsk);
        return ret;
    }

    nl_cb_set (fsk->cb, NL_CB_ACK, NL_CB_CUSTOM, fib_ack_handler, NULL);
    nl_cb_set (fsk->cb, NL_CB_FINISH, NL_CB_CUSTOM, fib_finish_handler, NULL);
    nl_cb_err (fsk->cb, NL_CB_CUSTOM, (nl_recvmsg_err_cb_t) fib_error_handler, NULL);

    *sk = fsk;
    return ret;
}

static int
fib_deinit_transport (fibsocket **sk)
{
    fibsocket *socket;

    return_val_if_fail (sk, -EINVAL);
    return_val_if_fail (*sk, -EALREADY);

    socket = *sk;
    nl_cb_put (socket->cb);
    nl_socket_free (socket->sk);

    g_free (socket);
    *sk = NULL;

    return 0;
}

static void
destroy_rules (void *table)
{
   gpointer key, value;
   GHashTableIter iter;
   GHashTable *rules = table;

   return_if_fail (rules);

   g_hash_table_iter_init (&iter, rules);
   while (g_hash_table_iter_next (&iter, &key, &value)) {
      DEBUG ("Clearing the table %u, size : %u", GPOINTER_TO_UINT (key),
             g_list_length (value));
      fib_rules_cleanup (value);
   }
}

int
fib_init ()
{
    int ret;

    DEBUG ("");

    ret = fib_init_transport (&fibsk);
    if (ret < 0) {
        ERROR ("Failed to initialize the fib socket: %s/%d",
               strerror (-ret), -ret);
        return ret;
    }

    routing_rules = g_hash_table_new_full (g_direct_hash,
                                           g_direct_equal,
                                           NULL, NULL);

    ret = rtnl_notifier_register (&fibops);
    if (ret < 0)
        ERROR ("Failed to register to rtnl watch list: %s/%d",
               strerror (-ret), -ret);

    return ret;
}

int
fib_deinit ()
{
    int ret;

    DEBUG ("");

    ret = fib_deinit_transport (&fibsk);
    if (ret < 0)
        ERROR ("Failed to deinitialize the fib socket: %s/%d",
               strerror (-ret), -ret);

    ret = rtnl_notifier_unregister (&fibops);
    if (ret < 0)
        ERROR ("Failed to unregister from rtnl watch list: %s/%d",
               strerror (-ret), -ret);

    destroy_rules (routing_rules);
    g_hash_table_destroy (routing_rules);

    return ret;
}

/** @} */
