/**
 * @file device.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <unistd.h>
#include <errno.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <linux/netlink.h>
#include <linux/rtnetlink.h>
#include <linux/if_addr.h>
#include <linux/if.h>
#include <linux/if_tun.h>
#include <netinet/ether.h>
#include <netinet/icmp6.h>
#include <net/if_arp.h>
#include <netlink/attr.h>
#include <sys/ioctl.h>
#include <glib.h>
#include <device.h>
#include <rtnl.h>
#include <log.h>

#define IFQDISCSIZ	32

struct net_device
{
    char *name;
    char *qdisc;
    char *address;
    char *bcast;
    unsigned int family;
    unsigned int arptype;
    unsigned int index;
    unsigned int flags;
    unsigned int change;
    unsigned int mtu;
    unsigned int link;
    unsigned int txqlen;
    unsigned int weight;
    unsigned int master;
    unsigned int promiscuity;
    unsigned int numtxqueues;
    unsigned int numrxqueues;
    unsigned int netdevgroup;
    unsigned int carrier;
    struct rtnl_link_stats64 *stats64;
    unsigned char operstate;
    unsigned char linkmode;
    devicetype type;
    devicestate state;
};

typedef
enum __device_event
{
    EVENT_DEVICE_ADDED,
    EVENT_DEVICE_REMOVED,
    EVENT_DEVICE_CHANGED,
    MAX_DEVICE_EVENTS
} deviceevent;

static GList *devicelist;
static GList *registeredclients;

static const char*
device_arptype_type2str (const unsigned int l2type)
{
    switch (l2type) {
    case ARPHRD_NETROM:
        return ENUMTOSTR (NETROM);
    case ARPHRD_ETHER:
        return ENUMTOSTR (ETHER);
    case ARPHRD_EETHER:
        return ENUMTOSTR (EETHER);
    case ARPHRD_AX25:
        return ENUMTOSTR (AX25);
    case ARPHRD_PRONET:
        return ENUMTOSTR (PRONET);
    case ARPHRD_CHAOS:
        return ENUMTOSTR (CHAOS);
    case ARPHRD_IEEE802:
        return ENUMTOSTR (IEEE802);
    case ARPHRD_ARCNET:
        return ENUMTOSTR (ARCNET);
    case ARPHRD_APPLETLK:
        return ENUMTOSTR (APPLETLK);
    case ARPHRD_DLCI:
        return ENUMTOSTR (DLCI);
    case ARPHRD_ATM:
        return ENUMTOSTR (ATM);
    case ARPHRD_METRICOM:
        return ENUMTOSTR (METRICOM);
    case ARPHRD_IEEE1394:
        return ENUMTOSTR (IEEE1394);
    case ARPHRD_EUI64:
        return ENUMTOSTR (EUI64);
    case ARPHRD_INFINIBAND:
        return ENUMTOSTR (INFINIBAND);
    case ARPHRD_VOID:
        return ENUMTOSTR (VOID);
    case ARPHRD_NONE:
        return ENUMTOSTR (NONE);
    }

    return ENUMTOSTR (UNKNOWN);
}

const char*
device_type2str (const devicetype type)
{
    switch (type) {
    case DEVICE_TYPE_NONE:
    case DEVICE_TYPE_MAX:
        return ENUMTOSTR (UNKNOWN);
    case DEVICE_TYPE_BLUETOOTH:
        return ENUMTOSTR (BLUETOOTH);
    case DEVICE_TYPE_BOND:
        return ENUMTOSTR (BOND);
    case DEVICE_TYPE_BRDIGE:
        return ENUMTOSTR (BRDIGE);
    case DEVICE_TYPE_CELLULAR:
        return ENUMTOSTR (CELLULAR);
    case DEVICE_TYPE_ETHERNET:
        return ENUMTOSTR (ETHERNET);
    case DEVICE_TYPE_GADGET:
        return ENUMTOSTR (GADGET);
    case DEVICE_TYPE_HSR:
        return ENUMTOSTR (HSR);
    case DEVICE_TYPE_VLAN:
        return ENUMTOSTR (VLAN);
    case DEVICE_TYPE_WIMAX:
        return ENUMTOSTR (WIMAX);
    case DEVICE_TYPE_WLAN:
        return ENUMTOSTR (WLAN);
    case DEVICE_TYPE_VXLAN:
        return ENUMTOSTR (VXLAN);
    }

    return ENUMTOSTR (UNKNOWN);
}

const char*
device_state2str (const devicestate state)
{
    switch (state) {
    case DEVSTATE_UNKNOWN:
        return ENUMTOSTR (UNKNOWN);
    case DEVSTATE_UNMANAGED:
        return ENUMTOSTR (UNMANAGED);
    case DEVSTATE_UNAVAILABLE:
        return ENUMTOSTR (UNAVAILABLE);
    case DEVSTATE_MANAGED:
        return ENUMTOSTR (MANAGED);
    case DEVSTATE_PREPARING:
        return ENUMTOSTR (PREPARING);
    case DEVSTATE_STARTING:
        return ENUMTOSTR (STARTING);
    case DEVSTATE_CONFIGURING_IP:
        return ENUMTOSTR (CONFIGURING_IP);
    case DEVSTATE_ACTIVATED:
        return ENUMTOSTR (ACTIVATED);
    case DEVSTATE_DEACTIVATING:
        return ENUMTOSTR (DEACTIVATING);
    case DEVSTATE_FAILED:
        return ENUMTOSTR (FAILED);
    }

    return ENUMTOSTR (UNKNOWN);
}

static const char*
device_operstate2str (unsigned char operstate)
{
    switch (operstate) {
    case IF_OPER_UNKNOWN:
        return ENUMTOSTR (OPER_UNKNOWN);
    case IF_OPER_NOTPRESENT:
        return ENUMTOSTR (OPER_NOTPRESENT);
    case IF_OPER_DOWN:
        return ENUMTOSTR (OPER_DOWN);
    case IF_OPER_LOWERLAYERDOWN:
        return ENUMTOSTR (OPER_LOWERLAYERDOWN);
    case IF_OPER_TESTING:
        return ENUMTOSTR (OPER_TESTING);
    case IF_OPER_DORMANT:
        return ENUMTOSTR (OPER_DORMANT);
    case IF_OPER_UP:
        return ENUMTOSTR (OPER_UP);
    }

    return ENUMTOSTR (OPER_UNKNOWN);
}

static void
device_cleanup (gpointer data)
{
    struct net_device *device = data;

    return_if_fail (device);

    DEBUG ("Cleaning up the device: %s", device->name);
    g_free (device->name);
    g_free (device->qdisc);
    g_free (device->address);
    g_free (device->bcast);
    g_free (device->stats64);
    g_free (device);
}

static int
device_index2name (int index,
                   char *ifname)
{
    int sk, ret;
    struct ifreq ifr;

    return_val_if_fail (ifname, -EINVAL);

    memset (&ifr, 0, sizeof (struct ifreq));
    ifr.ifr_ifindex = index;
    sk = socket (AF_INET, SOCK_DGRAM | SOCK_CLOEXEC, 0);
    if (sk < 0)
        return -errno;

    ret = ioctl (sk, SIOCGIFNAME, &ifr);
    if (ret < 0) {
        ERROR ("Failed to get the inerface name for"
               " index [%d]: %s", index, strerror (errno));
        (void)close (sk);
        return -errno;
    }

    memcpy (ifname, ifr.ifr_name, IFNAMSIZ);
    (void)close (sk);
    return 0;
}

static void
device_dump_info (struct net_device *device)
{
    return_if_fail (device);

    INFO ("****************************************************************");
    INFO ("Name: %s Index: %d Link Type: %s Hwaddr: %s",
          device->name, device->index,
          device_arptype_type2str (device->arptype),
          device->address);
    INFO ("MTU: %d Txqlen: %d qdisc: %s NumTxqueues: %d NumRxqueues: %d",
          device->mtu, device->txqlen, device->qdisc, device->numtxqueues,
          device->numrxqueues);
    INFO ("Operstate: %s RX Packets: %llu TX Packets: %llu ",
          device_operstate2str (device->operstate),
          device->stats64->rx_packets, device->stats64->tx_packets);
    INFO ("****************************************************************");
}

static void
device_notify_clients (const deviceevent event,
                       const struct net_device *device)
{
    GList *temp;

    return_if_fail (device);

    temp = registeredclients;
    for ( ; temp; temp = temp->next) {

        deviceops *ops = temp->data;
        if (!ops)
            continue;

        switch (event) {
        case EVENT_DEVICE_ADDED:
            if (ops->device_added)
                ops->device_added (device->index,
                                   device->name,
                                   device->address);
            break;
        case EVENT_DEVICE_REMOVED:
            if (ops->device_removed)
                ops->device_removed (device->index,
                                     device->name,
                                     device->address);
            break;
        case EVENT_DEVICE_CHANGED:
            break;
        default:
            break;
        }
    }
}

static struct net_device*
device_find (int index)
{
    GList *temp;
    struct net_device *device;

    temp = devicelist;
    for ( ; temp; temp = temp->next) {
        device = temp->data;
        if (device && device->index == (unsigned int) index)
            return device;
    }

    return NULL;
}

int
device_can_manage (int index)
{
    devicetype devtype;
    char ifname [IFNAMSIZ];

    if (index < 0)
        return -EINVAL;

    memset (ifname, 0, IFNAMSIZ);
    if (device_index2name (index, ifname) < 0)
        return -ENODEV;

    devtype = (devicetype) rtnl_get_device_type (index);
    if (devtype == DEVICE_TYPE_WLAN)
        return 0;

    return -ENOTSUP;
}

static int
device_new_link (int index,
                 unsigned short type,
                 unsigned int family,
                 unsigned int flags,
                 unsigned int change,
                 struct nlattr **attr)
{
    int ret, notify = 0;
    char ifname [IFNAMSIZ + 1] = {0,};
    char qdisc [IFQDISCSIZ + 1] = {0,};
    struct ether_addr address = {{0,}};
    struct ether_addr broadcast = {{0,}};
    struct net_device *device;

    return_val_if_fail (attr, -EINVAL);

    ret = device_can_manage (index);
    if (ret < 0)
        return ret;

    device = device_find (index);
    if (!device) {
        device = g_try_malloc0 (sizeof (*device));
        return_val_if_fail (device, -ENOMEM);

        device->stats64 = g_try_malloc0 (sizeof (struct rtnl_link_stats64));
        if (!device->stats64) {
            g_free (device);
            return -ENOMEM;
        }
        notify = 1;
        devicelist = g_list_append (devicelist, device);
    }

    device->family = family;
    device->arptype = type;
    device->index = (unsigned int) index;
    device->flags = flags;
    device->change = change;

    if (attr [IFLA_ADDRESS]) {
        g_free (device->address);
        nla_memcpy (&address, attr [IFLA_ADDRESS], ETH_ALEN);
        device->address =
                g_strdup_printf ("%02x:%02x:%02x:%02x:%02x:%02x",
                                 address.ether_addr_octet [0],
                                 address.ether_addr_octet [1],
                                 address.ether_addr_octet [2],
                                 address.ether_addr_octet [3],
                                 address.ether_addr_octet [4],
                                 address.ether_addr_octet [5]);
    }
    if (attr [IFLA_BROADCAST]) {
        g_free (device->bcast);
        nla_memcpy (&broadcast, attr [IFLA_BROADCAST], ETH_ALEN);
        device->bcast =
                g_strdup_printf ("%02x:%02x:%02x:%02x:%02x:%02x",
                                 broadcast.ether_addr_octet [0],
                                 broadcast.ether_addr_octet [1],
                                 broadcast.ether_addr_octet [2],
                                 broadcast.ether_addr_octet [3],
                                 broadcast.ether_addr_octet [4],
                                 broadcast.ether_addr_octet [5]);
    }
    if (attr [IFLA_IFNAME]) {

        strncpy (ifname, nla_get_string (attr [IFLA_IFNAME]),
                 IFNAMSIZ);

        if (device->name &&
                g_strcmp0 (device->name, ifname)) {
            notify = 1;
            g_free (device->name);
        }
        device->name = g_strdup (ifname);
    }
    if (attr [IFLA_MTU])
        device->mtu = nla_get_u32 (attr [IFLA_MTU]);
    if (attr [IFLA_LINK])
        device->link = nla_get_u32 (attr [IFLA_MTU]);
    if (attr [IFLA_OPERSTATE])
        device->operstate = nla_get_u8 (attr [IFLA_OPERSTATE]);
    if (attr [IFLA_LINKMODE])
        device->linkmode = nla_get_u8 (attr [IFLA_LINKMODE]);
    if (attr [IFLA_TXQLEN])
        device->txqlen = nla_get_u32 (attr [IFLA_TXQLEN]);
    if (attr [IFLA_NUM_TX_QUEUES])
        device->numtxqueues = nla_get_u32 (attr [IFLA_NUM_TX_QUEUES]);
    if (attr [IFLA_NUM_RX_QUEUES])
        device->numrxqueues = nla_get_u32 (attr [IFLA_NUM_RX_QUEUES]);
    if (attr [IFLA_QDISC]) {
        g_free (device->qdisc);
        strncpy (qdisc, nla_get_string (attr [IFLA_QDISC]),
                 IFQDISCSIZ);
        device->qdisc = g_strdup (qdisc);
    }
    if (attr [IFLA_CARRIER])
        device->carrier = nla_get_u32 (attr [IFLA_CARRIER]);
    if (attr [IFLA_STATS64])
        memcpy (device->stats64, nla_data (attr [IFLA_STATS64]),
                sizeof (struct rtnl_link_stats64));
    if (attr [IFLA_GROUP])
        device->netdevgroup = nla_get_u32 (attr [IFLA_GROUP]);
    if (attr [IFLA_WEIGHT])
        device->weight = nla_get_u32 (attr [IFLA_WEIGHT]);
    if (attr [IFLA_MASTER])
        device->master = nla_get_u32 (attr [IFLA_MASTER]);
    if (attr [IFLA_PROMISCUITY])
        device->promiscuity = nla_get_u32 (attr [IFLA_PROMISCUITY]);

    if (notify) {
        device_notify_clients (EVENT_DEVICE_ADDED, device);
        device_dump_info (device);
    }

    return 0;
}

static int
device_del_link (int index,
                 unsigned short type,
                 unsigned int family,
                 unsigned int flags,
                 unsigned int change,
                 const char *name)
{
    (void) type;
    (void) family;
    (void) flags;
    (void) change;

    struct net_device *device;

    DEBUG ("index: %d name: %s", index, name);

    device = device_find (index);
    return_val_if_fail (device, -ENOENT);
    device_notify_clients (EVENT_DEVICE_REMOVED, device);
    devicelist = g_list_remove (devicelist, device);
    device_cleanup (device);
    return 0;
}


static rtnlwatch devicewatch = {
  .client  = "device",
  .newlink = device_new_link,
  .dellink = device_del_link,
};

const char *
device_get_name (const struct net_device *device)
{
    return_val_if_fail (device, NULL);
    return device->name;
}

const char *
device_get_qdisc (const struct net_device *device)
{
    return_val_if_fail (device, NULL);
    return device->qdisc;
}

const char *
device_get_address (const struct net_device *device)
{
    return_val_if_fail (device, NULL);
    return device->address;
}

const char *
device_get_bcastaddress (const struct net_device *device)
{
    return_val_if_fail (device, NULL);
    return device->bcast;
}

unsigned int
device_get_family (const struct net_device *device)
{
    return_val_if_fail (device, 0);
    return device->family;
}

unsigned int
device_get_arptype (const struct net_device *device)
{
    return_val_if_fail (device, 0);
    return device->arptype;
}

unsigned int
device_get_index (const struct net_device *device)
{
    return_val_if_fail (device, 0);
    return device->index;
}

unsigned int
device_get_flags (const struct net_device *device)
{
    return_val_if_fail (device, 0);
    return device->flags;
}

unsigned int
device_get_mtu (const struct net_device *device)
{
    return_val_if_fail (device, 0);
    return device->mtu;
}

unsigned int
device_get_txqlen (const struct net_device *device)
{
    return_val_if_fail (device, 0);
    return device->txqlen;
}

unsigned int
device_get_numtxqueues (const struct net_device *device)
{
    return_val_if_fail (device, 0);
    return device->numtxqueues;
}

unsigned int
device_get_numrxqueues (const struct net_device *device)
{
    return_val_if_fail (device, 0);
    return device->numrxqueues;
}

unsigned int
device_get_group (const struct net_device *device)
{
    return_val_if_fail (device, 0);
    return device->netdevgroup;
}

unsigned int
device_get_carrier (const struct net_device *device)
{
    return_val_if_fail (device, 0);
    return device->carrier;
}

unsigned int
device_get_operstate (const struct net_device *device)
{
    return_val_if_fail (device, 0);
    return device->operstate;
}

devicetype
device_get_type (const struct net_device *device)
{
    return_val_if_fail (device, 0);
    return device->type;
}

struct net_device*
device_get (const unsigned int index)
{
    GList *temp;
    struct net_device *device;

    temp = devicelist;
    for ( ; temp; temp = temp->next) {
        device = temp->data;
        if (device && device->index == index)
            return device;
    }

    return NULL;
}

int
device_notifier_register (deviceops *ops)
{
    int found = 0;
    GList *temp;

    return_val_if_fail (ops, -EINVAL);

    DEBUG ("Registering the client for device update : %s [%p]",
           ops->client ? ops->client : "UNKNOWN", ops);

    temp = registeredclients;
    for ( ; temp; temp = temp->next) {
        deviceops *registeredops = temp->data;
        if (ops == registeredops) {
            found = 1;
            break;
        }
    }

    if (!found) {
        registeredclients = g_list_append (registeredclients, ops);
        return 0;
    }

    return -EALREADY;
}

int
device_notifier_unregister (deviceops *ops)
{
    int found = 0;
    GList *temp;

    return_val_if_fail (ops, -EINVAL);

    DEBUG ("UnRegistering the client for device update : %s [%p]",
           ops->client ? ops->client : "UNKNOWN", ops);

    temp = registeredclients;
    for ( ; temp; temp = temp->next) {
        deviceops *registeredops = temp->data;
        if (ops == registeredops) {
            found = 1;
            break;
        }
    }

    if (found) {
        registeredclients = g_list_remove_link (registeredclients, temp);
        return 0;
    }

    return -ENOENT;
}

int
device_init ()
{
    int ret;

    DEBUG ("");

    ret = rtnl_notifier_register (&devicewatch);
    if (ret < 0)
        ERROR ("Failed to register to rtnl watch list");

    return ret;
}

int
device_deinit ()
{
    int ret;

    ret = rtnl_notifier_unregister (&devicewatch);
    if (ret < 0)
        ERROR ("Failed to unregister from rtnl watch list");

    g_list_free_full (devicelist, device_cleanup);
    g_list_free (registeredclients);
    return ret;
}
