/**
 * @file nat.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief NAT handling.
 *
 * @{
 */

#include <glib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <linux/limits.h>
#include <sys/inotify.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <session.h>
#include <firewall.h>
#include <inotifier.h>
#include <log.h>
#include <ipconfig.h>
#include <route.h>
#include <fib.h>
#include <rtnl.h>
#include <main.h>

/* global */
#define WAPMAN_NAT_IP_FORWARD_DIR       "/proc/sys/net/ipv4/"
#define WAPMAN_NAT_IP_FORWARD_FILE      "ip_forward"
#define WAPMAN_NAT_IP_FORWARD_PATH      "/proc/sys/net/ipv4/ip_forward"

/* per device */
#define WAPMAN_NAT_IP_FORWARD_DEVICE_DIR    "/proc/sys/net/ipv4/conf/%s"
#define WAPMAN_NAT_IP_FORWARD_DEVICE_FILE   "forwarding"
#define WAPMAN_NAT_IP_FORWARD_DEVICE        "/proc/sys/net/ipv4/conf/%s/forwarding"

#define NAT_ENABLE_ICM_FORWARD			    "-i %s -s %s/%d -o %s -j ACCEPT"
#define NAT_ENABLE_OGNG_FORWARD			    "-o %s -d %s/%d -i %s -j ACCEPT"
#define NAT_ENABLE_MASQUERADE		        "-s %s/%d ! -d %s/%d -o %s -j MASQUERADE"

#define NAT_RT_TABLE_ID_MIN                 50
#define NAT_RT_TABLE_ID_MAX                 99

#define NAT_IP_RULE                         "%s -f %d -s %s -t %d"
#define NAT_ADD_GATEWAY                     "%s -f %d -g %s -i %d -t %d"
#define NAT_ADD_ADDR                        "%s -f %d -d %s/%d -s %s -i %d -t %d"

typedef
struct __masquerade_details
{
    char *target_device;
    char *tethered_device;
    gboolean ip_fwd_enabled;
    gboolean conf_fw_watch;
} masquerade_details_t;

typedef
struct __wapman_nat
{
    /* operating interface of the AP */
    char *interface;

    masquerade_details_t *masquerade;

    unsigned int rt_table;

    gboolean fwd_enabled;
    gboolean conf_fw_watch;

    /* nat is enabled or disabled */
    gboolean enabled;

    /* IP address of the DHCP pool */
    char *address;

    /* subnet size of the operating IP pool */
    unsigned int prefix_len;

    char *router;

    /* firewall context with respect to
     * this nat */
    fwcontext *fwctx;
} wapdnat;

static GHashTable *nat_hash;
static unsigned int rt_table = NAT_RT_TABLE_ID_MIN;
static GList *rt_tables;

static int nat_inotify_register (gboolean enable, void *data, inotify_cb cb,
                                 const char *file, const char *format, ...);

static int
nat_get_unused_rt_table_id (unsigned int *table)
{
    GList *temp;
    unsigned int id = 0;

    return_val_if_fail (table, -EINVAL);

    id = rt_table;

    if (id >= NAT_RT_TABLE_ID_MAX)
        id = rt_table = NAT_RT_TABLE_ID_MIN;

    while ((temp = g_list_find (rt_tables, GUINT_TO_POINTER (id)))) {
        id = ++rt_table;
        if (id >= NAT_RT_TABLE_ID_MAX)
            return -ENOENT;
    }

    rt_tables = g_list_append (rt_tables, GUINT_TO_POINTER (id));
    *table = id;

    DEBUG ("Next unsed table id: %u", id);
    return 0;
}

static int
nat_ip_forward (const gboolean enable,
                const char *path)
{
    int err = 0, fd;
    unsigned char forward = '0';

    return_val_if_fail (path, -EINVAL);

    DEBUG ("IP Forwarding to be: %s [for %s]", enable ? "Enabled" : "Disabled",
           path);

    fd = open (path, O_CLOEXEC | O_RDWR);
    if (fd < 0) {
        ERROR ("Failed to open the file \"%s\", error : %s", path, strerror (errno));
        return -errno;
    }

    if (enable)
        forward = '1';

    if (write (fd, &forward, sizeof (forward)) < 0)
        err = -errno;

    (void)close (fd);
    DEBUG ("%s %s ip forwarding for: %s", (err < 0) ? "Failed to" : "Successfully",
           enable ? "enabled" : "disabled", path);
    return err;
}

static int
nat_read_ip_forward (const char *path)
{
    int err = 0;
    unsigned char forward = 0;
    int fd;

    return_val_if_fail (path, -EINVAL);

    fd = open (path, O_CLOEXEC | O_RDONLY);
    if (fd < 0) {
        ERROR ("Failed to open the file %s, error : %s", path,
               strerror (errno));
        return -errno;
    }

    if (read (fd, &forward, sizeof (forward)) < 0)
        err = -EIO;
    else {
        INFO ("IP forward value for %s: %c [%d]", path, forward, forward);
        if (forward == '1')
            err = 1;
    }

    (void)close (fd);
    return err;
}

static int
nat_toggle_ip_forward (gboolean enable,
                       const char *format,
                       ...)
{
    int ret = -EINVAL;
    va_list args;
    char path [BUFSIZ];

    va_start (args, format);
    memset (path, 0, sizeof (path));
    if (vsnprintf (path, sizeof (path), format, args) > 0)
        ret = nat_ip_forward (enable, path);

    va_end (args);
    return ret;
}

static void
nat_inotify_iface (struct inotify_event *event,
				   const char *dir,
                   void *data)
{
    int err;
    char temp [200];
    wapdnat *nat = data;

    return_if_fail (event && dir && nat);

    DEBUG ("Event %p Nat %p [enabled: %s] changed dir [%s] file %s", event, nat,
    	   nat->enabled ? "YES" : "NO", dir, event->len ? event->name : "UNKNOWN");

    if (g_strcmp0 (event->name, WAPMAN_NAT_IP_FORWARD_DEVICE_FILE)
            || !(event->mask & IN_MODIFY))
        return;

    memset (temp, 0, sizeof (temp));
    snprintf (temp, sizeof (temp), "%s/%s", dir, WAPMAN_NAT_IP_FORWARD_DEVICE_FILE);
    err = nat_read_ip_forward (temp);
    if (err < 0) {
        ERROR ("Failed to read %s : %s/%d", temp, strerror (-err), -err);
        return;
    }

    if (nat->enabled && !err)
        nat_ip_forward (TRUE, temp);
    else if (!nat->enabled && err == 1)
        nat_ip_forward (FALSE, temp);
}

static int
nat_fw_rule_add (wapdnat *nat,
                 const char *table,
                 const char *chain,
                 const char *format,
                 ...)
{
    int ret = -EINVAL;
    va_list args;
    char rule [1024];

    return_val_if_fail (nat && table && chain, -EINVAL);

    va_start (args, format);
    memset (rule, 0, sizeof (rule));

    if (vsnprintf (rule, sizeof (rule), format, args) > 0) {
        ret = firewall_add_rule (nat->fwctx, table, chain, rule);
        if (ret < 0)
            ERROR ("Failed to add the \"%s\" to the nat context: %s "
                   "[%s/%d]", rule, nat->interface, strerror (-ret), -ret);
    } else
        ERROR ("Invalid rule format : %s", format);

    va_end (args);
    return ret;
}

static int
nat_add_default_route (wapdnat *nat,
                       gboolean add)
{
    char *gateway;
    int ret;
    unsigned int index = 0;

    index = if_nametoindex (nat->masquerade->tethered_device);
    if (!index) {
        ERROR ("Failed to get the index for net_device: %s [%s/%d]",
               nat->masquerade->tethered_device, strerror (errno), errno);
        return -errno;
    }

    gateway = g_strdup_printf (NAT_ADD_GATEWAY, add ? "-A" : "-D", AF_INET,
                               nat->router, index, nat->rt_table);
    ret = modify_ip_route (gateway);
    if (ret < 0)
        ERROR ("Failed to add ip route [%s]: %s/%d", gateway, strerror (-ret), -ret);

    g_free (gateway);
    return ret;
}

static int
nat_update_routes (wapdnat *nat,
                   gboolean add)
{
    char *addr, *network;
    int ret;
    unsigned int index = 0, start_ip = 0;

    index = if_nametoindex (nat->interface);
    if (!index) {
        ERROR ("Failed to get the index for net_device: %s [%s/%d]",
               nat->interface, strerror (errno), errno);
        return -errno;
    }

    start_ip = ntohl (inet_addr (nat->address));
    network = g_strdup_printf ("%u.%u.%u.%u", (start_ip >> 24) & 0xff,
                               (start_ip >> 16) & 0xff, (start_ip >> 8) & 0xff, 1);
    addr = g_strdup_printf (NAT_ADD_ADDR, add ? "-A" : "-D", AF_INET, nat->address,
                            nat->prefix_len, network,
                            index, nat->rt_table);

    ret = modify_ip_route (addr);
    if (ret < 0)
        ERROR ("Failed to add ip route [%s]: %s/%d", addr, strerror (-ret), -ret);

    g_free (network);
    g_free (addr);
    return ret;
}

static int
nat_update_fib_rules (wapdnat *nat,
                      gboolean add)
{
    int ret;
    char *rule, *source;

    return_val_if_fail (nat, -EINVAL);

    source = g_strdup_printf ("%s/%d", nat->address, nat->prefix_len);
    rule = g_strdup_printf (NAT_IP_RULE, add ? "-A" : "-D", AF_INET,
                            source, nat->rt_table);

    ret = fib_modify_ip_rule (rule);
    if (ret < 0)
        ERROR ("Failed to add ip rule [%s]: %s/%d", rule, strerror (-ret), -ret);

    g_free (rule);
    g_free (source);
    return ret;
}

static void
nat_clean_routing_table (wapdnat *nat)
{
    int ret;
    struct in_addr link;

    return_if_fail (nat);

    memset (&link, 0, sizeof (struct in_addr));
    ret = inet_aton (nat->address, &link);
    if (ret == 0)
        ERROR ("Invalid IP address provided: %s", nat->address);

    (void) nat_add_default_route (nat, FALSE);
    (void) nat_update_routes (nat, FALSE);
    (void) nat_update_fib_rules (nat, FALSE);
}

static void
nat_init_routing_table (wapdnat *nat)
{
    int ret;
    struct in_addr link;

    return_if_fail (nat);

    memset (&link, 0, sizeof (struct in_addr));
    ret = inet_aton (nat->address, &link);
    return_if_fail (ret != 0);

    (void) nat_update_fib_rules (nat, TRUE);
    (void) nat_update_routes (nat, TRUE);
    (void) nat_add_default_route (nat, TRUE);
}

static int
nat_configure_netfilter (wapdnat *nat)
{
    int ret;

    if (FALSE == get_firewall_configuration ())
        return 0;

    nat->fwctx = firewall_create_context ();
    return_val_if_fail (nat->fwctx, -ENOMEM);

    nat_fw_rule_add (nat, "filter", "FORWARD", NAT_ENABLE_ICM_FORWARD,
                     nat->interface, nat->address, nat->prefix_len,
                     nat->masquerade->tethered_device);

    nat_fw_rule_add (nat, "filter", "FORWARD", NAT_ENABLE_OGNG_FORWARD,
                     nat->interface, nat->address, nat->prefix_len,
                     nat->masquerade->tethered_device);

    nat_fw_rule_add (nat, "nat", "POSTROUTING", NAT_ENABLE_MASQUERADE,
                     nat->address, nat->prefix_len, nat->address,
                     nat->prefix_len, nat->masquerade->tethered_device);

    ret = firewall_enable (nat->fwctx);
    if (ret < 0) {
        ERROR ("Failed to enable firewall context of %s: %s/%d",
               nat->interface,
               strerror (-ret), -ret);
        firewall_disable (nat->fwctx);
        firewall_cleanup_context (nat->fwctx);
        nat->fwctx = NULL;
    }

    return ret;
}

static int
nat_configure_forwarding (wapdnat *nat)
{
    int ret;

    ret = nat_toggle_ip_forward (TRUE, WAPMAN_NAT_IP_FORWARD_DEVICE, nat->interface);
    return_val_if_fail (0 == ret, ret);

    ret = nat_toggle_ip_forward (TRUE, WAPMAN_NAT_IP_FORWARD_DEVICE,
                                 nat->masquerade->tethered_device);
    if (ret < 0) {
        (void) nat_toggle_ip_forward (FALSE, WAPMAN_NAT_IP_FORWARD_DEVICE, nat->interface);
        return ret;
    }

    INFO ("Enabled IP forwarding between [%s <---> %s] for: %s", nat->interface,
          nat->masquerade->tethered_device, nat->interface);

    nat->fwd_enabled = TRUE;
    nat->masquerade->ip_fwd_enabled = TRUE;

    return ret;
}

static void
nat_deconfigure_forwarding (wapdnat *nat)
{
    int ret;

    INFO ("IP forwarding status for \"%s\" "
          "Master-Dev Forwarding: \"%s\" "
          "Tethered-Dev Forwarding: \"%s\"",
          nat->interface,
          nat->fwd_enabled ? "Y" : "N",
          nat->masquerade->ip_fwd_enabled ? "Y" : "N");

    if (nat->fwd_enabled) {
        ret = nat_toggle_ip_forward (FALSE, WAPMAN_NAT_IP_FORWARD_DEVICE,
                                     nat->interface);
        if (0 == ret)
            nat->fwd_enabled = FALSE;
    }

    if (nat->masquerade->ip_fwd_enabled) {
        ret = nat_toggle_ip_forward (FALSE, WAPMAN_NAT_IP_FORWARD_DEVICE,
                                     nat->masquerade->tethered_device);
        if (0 == ret)
            nat->masquerade->ip_fwd_enabled = FALSE;
    }
}

static void
nat_deconfigure_fw_watch (wapdnat *nat)
{
    int ret;

    INFO ("IP forwarding Conf Watch status for \"%s\" "
          "Master-Dev Watch: \"%s\" "
          "Tethered-Dev Watch: \"%s\"",
          nat->interface,
          nat->conf_fw_watch ? "Y" : "N",
          nat->masquerade->conf_fw_watch ? "Y" : "N");

    if (nat->conf_fw_watch) {
        ret = nat_inotify_register (FALSE, nat, nat_inotify_iface, NULL, WAPMAN_NAT_IP_FORWARD_DEVICE_DIR,
                                    nat->interface);
        if (0 == ret)
            nat->conf_fw_watch = FALSE;
    }

    if (nat->masquerade->conf_fw_watch) {
        ret = nat_inotify_register (FALSE, nat, nat_inotify_iface, NULL, WAPMAN_NAT_IP_FORWARD_DEVICE_DIR,
                                    nat->masquerade->tethered_device);
        if (0 == ret)
            nat->masquerade->conf_fw_watch = FALSE;
    }
}

static int
nat_enable_configurations (wapdnat *nat)
{
    int ret = 0;
    unsigned int index = 0;

    index = if_nametoindex (nat->masquerade->tethered_device);
    return_val_if_fail (index != 0, -errno);

    ret = nat_configure_forwarding (nat);
    if (ret < 0) {
        ERROR ("Failed to enable IP forwarding between [%s <--> %s]",
               nat->interface, nat->masquerade->tethered_device);
        return ret;
    }

    nat_init_routing_table (nat);
    ret = nat_configure_netfilter (nat);
    if (ret < 0) {
        ERROR ("Failed to configure netfilter for: %s", nat->interface);
        nat_deconfigure_forwarding (nat);
        nat_clean_routing_table (nat);
        return ret;
    }

    nat->enabled = TRUE;
    return ret;
}

static int
nat_disable_configurations (wapdnat *nat)
{
    int err = 0;

    return_val_if_fail (nat, -EINVAL);

    DEBUG ("Disable firewall for nat: %s [status: %s]", nat->interface,
           nat->enabled ? "enabled" : "disabled");

    nat_deconfigure_forwarding (nat);
    nat_clean_routing_table (nat);

    if (TRUE == get_firewall_configuration()) {
        err = firewall_disable (nat->fwctx);
        if (err < 0)
            ERROR ("Failed to disable firewall for nat [%s]: %s", nat->interface,
                   strerror (-err));
        firewall_cleanup_context (nat->fwctx);
        nat->fwctx = NULL;
    }

    return err;
}

static int
nat_inotify_register (gboolean enable,
					  void *data,
                      inotify_cb cb,
					  const char *file,
                      const char *format,
                      ...)
{
    va_list args;
    char path [PATH_MAX];
    int ret = -EINVAL;

    va_start (args, format);
    memset (path, 0, sizeof (path));

    if (vsnprintf (path, sizeof (path), format, args) > 0) {
    	if (enable)
    		ret = inotify_register (path, cb, file, data);
        else {
    		ret = inotify_unregister (path, cb, file);
        }

        if (ret < 0)
        	ERROR ("Failed to %s to inotifier for: %s [%s/%d]", enable ? "register" : "unregister",
        		   path, strerror (-ret), -ret);
    }

    va_end (args);
    return ret;
}

int
nat_disable (const char *interface,
             gboolean remove)
{
    int err = 0;
    wapdnat *nat = NULL;

    return_val_if_fail (interface, -EINVAL);

    DEBUG ("Disabling nat for: %s Remove the nat configuration: %s",
           interface, remove ? "Y" : "N");

    nat = g_hash_table_lookup (nat_hash, interface);
    return_val_if_fail (nat, -ENOENT);

    DEBUG ("Nat Status on \"%s\" --> "
           "FW conf: \"%s\" "
           "Master Dev conf watch: \"%s\" "
           "Tethered Dev conf watch: \"%s\" "
           "IP Fw on master Dev: \"%s\" "
           "IP Fw on tethered Dev: \"%s\"",
           nat->interface,
           nat->enabled ? "Y" : "N",
           nat->conf_fw_watch ? "Y" : "N",
           nat->masquerade->conf_fw_watch ? "Y" : "N",
           nat->fwd_enabled ? "Y" : "N",
           nat->masquerade->ip_fwd_enabled ? "Y" : "N");

    nat_deconfigure_fw_watch (nat);
    if (nat->enabled) {
        err = nat_disable_configurations (nat);
        if (err < 0)
            ERROR ("Failed to disable nat configurations for: %s [%s/%d]",
                   interface, strerror (-err), -err);
        nat->enabled = FALSE;
    }

    if (remove)
        g_hash_table_remove (nat_hash, interface);

    return err;
}

static wapdnat*
nat_get_instance (const char *iface)
{
    int ret;
    unsigned int table = 0;
    wapdnat *nat;
    masquerade_details_t *masq = NULL;

    return_val_if_fail (iface, NULL);

    nat = g_hash_table_lookup (nat_hash, iface);
    if (nat)
        return nat;

    nat = g_try_malloc0 (sizeof (*nat));
    return_val_if_fail (nat, NULL);

    ret = nat_get_unused_rt_table_id (&table);
    if (ret < 0) {
        ERROR ("Failed to get an ununsed routing table id: %s/%d",
               strerror (-ret), -ret);
        g_free (nat);
        return NULL;
    }

    masq = g_try_malloc0 (sizeof (*masq));
    if (!masq) {
        rt_tables = g_list_remove (rt_tables, GUINT_TO_POINTER (table));
        g_free (nat);
        return NULL;
    }

    nat->masquerade = masq;
    nat->rt_table = table;
    nat->interface = g_strdup (iface);
    g_hash_table_replace (nat_hash, g_strdup (iface), nat);

    return nat;
}

static int
nat_compare_details (wapdnat *nat,
                     const char *outgng_iface,
                     const char *address,
                     const char *router,
                     unsigned char prefixlen)
{
    int differ = 0;

    return_val_if_fail (nat && nat->masquerade, -EINVAL);

    if (g_strcmp0 (nat->address, address))
        differ = TRUE;
    else if (prefixlen != nat->prefix_len)
        differ = TRUE;
    if (router && !nat->router)
        differ = 1;
    else if (!router && nat->router)
        differ = 1;
    else if (router && nat->router &&
             g_strcmp0 (router, nat->router))
        differ = 1;
    else if (outgng_iface && !nat->masquerade->target_device)
        differ = 1;
    else if (!outgng_iface && nat->masquerade->target_device)
        differ = 1;
    else if (outgng_iface && nat->masquerade->target_device &&
             g_strcmp0 (nat->masquerade->target_device, outgng_iface))
        differ = 1;

    return differ;
}

static void
nat_cleanup_details (wapdnat *nat)
{
    return_if_fail (nat);

    g_free (nat->address);
    g_free (nat->router);
    g_free (nat->masquerade->target_device);
    g_free (nat->masquerade->tethered_device);

    nat->address = NULL;
    nat->router = NULL;
    nat->masquerade->target_device = NULL;
    nat->masquerade->tethered_device = NULL;
}

int
nat_enable (const char *interface,
            const char *outgng_iface,
            const char *address,
            const char *router,
            unsigned char prefixlen)
{
    wapdnat *nat;
    char *addr;
    int err = 0, differ = 0;
    unsigned int index = 0;
    gboolean remove = FALSE;

    return_val_if_fail (interface && address, -EINVAL);
    return_val_if_fail (outgng_iface && router, 0);

    INFO ("Actual Interface: \"%s\" "
          "Intended Outgoing device: \"%s\" "
          "Router Address: \"%s\" "
          "ipv4 address: \"%s\" "
          "prefixlen: %d", interface,
          outgng_iface ? outgng_iface : "None",
          router ? router : "None", address, prefixlen);

    nat = nat_get_instance (interface);
    return_val_if_fail (nat, -ENOMEM);

    if (nat->enabled) {
        differ = nat_compare_details (nat, outgng_iface, address, router, prefixlen);
        if (!differ)
            return -EALREADY;

        err = nat_disable (nat->interface, FALSE);
        if (err < 0)
            ERROR ("Failed to disable the nat for the current configuration: "
                   "%s [%s/%d]", nat->interface, strerror (-err), -err);

        nat_cleanup_details (nat);
    }

    nat->router = g_strdup (router);
    nat->prefix_len = prefixlen;
    nat->address = g_strdup (address);
    nat->masquerade->target_device = g_strdup (outgng_iface);
    nat->masquerade->tethered_device = g_strdup (outgng_iface);

    /* Start watching the wlan interface till nat_disable */
    if (!nat->conf_fw_watch) {
        err = nat_inotify_register (TRUE, nat, nat_inotify_iface, NULL, WAPMAN_NAT_IP_FORWARD_DEVICE_DIR,
                                    nat->interface);
        if (0 == err)
            nat->conf_fw_watch = TRUE;
    }

    if (outgng_iface) {
        index = if_nametoindex (outgng_iface);
        if (!index) {
            INFO ("Failed to fetch the index for the interface: %s %s/%d",
                  outgng_iface, strerror (errno), errno);
            return 0;
        }
    }

    /*if (!index)
        return 0;*/

    if (!nat->masquerade->conf_fw_watch) {
        err = nat_inotify_register (TRUE, nat, nat_inotify_iface, NULL, WAPMAN_NAT_IP_FORWARD_DEVICE_DIR,
                                    nat->masquerade->tethered_device);
        if (0 == err)
            nat->masquerade->conf_fw_watch = TRUE;
    }

    addr = ipconfig_get_address (index);
    if (!addr)
        return 0;

    err = nat_enable_configurations (nat);
    if (err < 0) {
        ERROR ("Failed to enable the network configurations for: %s", nat->interface);
        nat_deconfigure_fw_watch (nat);
        remove = g_hash_table_remove (nat_hash, interface);
        if (FALSE == remove)
            ERROR ("Failed to remove nat configuration for: %s", interface);
    }

    return err;
}

static void
nat_inotify_global (struct inotify_event *event,
					const char *dir,
                    void *data)
{
	(void) data;

    int err;

    return_if_fail (event && dir);

    DEBUG ("event %p changed file %s", event,
           event->len ? event->name : "UNKNOWN");

    if (g_strcmp0 (event->name, WAPMAN_NAT_IP_FORWARD_FILE) &&
    		g_strcmp0 (event->name, WAPMAN_NAT_IP_FORWARD_PATH))
        return;

    err = nat_read_ip_forward (WAPMAN_NAT_IP_FORWARD_PATH);
    if (err < 0)
        ERROR ("Failed to read %s : %s", WAPMAN_NAT_IP_FORWARD_PATH,
               strerror (-err));
}

static void
nat_shutdown (gpointer key,
              gpointer value,
              gpointer user_data)
{
    (void) value;
    (void) user_data;

    const char *interface = key;
    (void) nat_disable (interface, FALSE);
}

static void
nat_cleanup (gpointer data)
{
    wapdnat *nat = data;

    return_if_fail (nat);

    INFO ("Freeing up nat client : %s [%p]",
          nat->interface, nat);

    rt_tables = g_list_remove (rt_tables, GUINT_TO_POINTER (nat->rt_table));

    g_free (nat->interface);
    g_free (nat->address);
    g_free (nat->router);
    g_free (nat->masquerade->target_device);
    g_free (nat->masquerade->tethered_device);
    g_free (nat->masquerade);
    g_free (nat);
}

static int
find_nat_configuration (const char *name,
                        wapdnat **nat)
{
    gpointer key, value;
    GHashTableIter iter;

    return_val_if_fail (name, -EINVAL);
    return_val_if_fail (nat, -EINVAL);
    return_val_if_fail (!*nat, -EEXIST);

    g_hash_table_iter_init (&iter, nat_hash);
    while (g_hash_table_iter_next (&iter, &key, &value)) {
        if (value && ((wapdnat *) value)->masquerade &&
                ((wapdnat *) value)->masquerade->target_device &&
                !g_strcmp0 (((wapdnat *) value)->masquerade->target_device,
                            name)) {
            *nat = value;
            return 0;
        }
    }

    return -ENOENT;
}

static int
nat_handle_dev_conf (int index,
                     const char *name,
                     int family,
                     const char *address)
{
    int ret;
    wapdnat *nat = NULL;

    return_val_if_fail (name && address, -EINVAL);

    DEBUG ("Index: %d ifname: %s family: %d address: %s",
           index, name, family, address);

    ret = find_nat_configuration (name, &nat);
    if (ret < 0)
        return 0;

    INFO ("Nat dev available for: %s [%p], enabled ? \"%s\"",
          nat->interface, nat, nat->enabled ? "Y" : "N");

    if (nat->enabled)
        return 0;

    return nat_enable_configurations (nat);
}

static int
nat_handle_dev_deconf (int index,
                       const char *name,
                       int family,
                       const char *address)
{
    wapdnat *nat = NULL;
    int ret;

    return_val_if_fail (name && address && index > 0, -EINVAL);

    DEBUG ("Index: %d ifname: %s family: %d address: %s",
           index, name, family, address);

    ret = find_nat_configuration (name, &nat);
    if (ret < 0)
        return 0;

    INFO ("Nat dev available for: %s [%p], enabled ? \"%s\"",
          nat->interface, nat, nat->enabled ? "Y" : "N");

    if (nat->enabled) {
        ret = nat_disable_configurations (nat);
        if (ret < 0)
            ERROR ("Failed to disable configurations for: %s [%s/%d]",
                   name, strerror (-ret), -ret);
        nat->enabled = FALSE;
    }

    return ret;
}

static ipconfig_ops nat_notify = {
    .name = "nat",
    .addr_added = nat_handle_dev_conf,
    .addr_removed = nat_handle_dev_deconf,
};

static int
nat_new_link (int index,
              unsigned short type,
              unsigned int family,
              unsigned int flags,
              unsigned int change,
              struct nlattr **attr)
{
    wapdnat *nat = NULL;
    int ret = 0;
    char name [IFNAMSIZ + 1];

    DEBUG ("Index: %d type: %u family: %u flags: %u change: %u",
           index, type, family, flags, change);

    return_val_if_fail (index > 0 && attr, -EINVAL);
    return_val_if_fail (attr [IFLA_IFNAME], -EINVAL);

    memset (name, 0, sizeof(name));
    if (attr [IFLA_IFNAME])
        strncpy (name, nla_get_string (attr [IFLA_IFNAME]),
                 IFNAMSIZ);

    ret = find_nat_configuration (name, &nat);
    if (ret < 0)
        return 0;

    DEBUG ("Nat dev: %s conf_fw_watch: %s", nat->interface,
           nat->masquerade->conf_fw_watch ? "Y" : "N");

    if (!nat->masquerade->conf_fw_watch) {
        ret = nat_inotify_register (TRUE, nat, nat_inotify_iface, NULL, WAPMAN_NAT_IP_FORWARD_DEVICE_DIR,
                                    nat->masquerade->tethered_device);
        if (0 == ret)
            nat->masquerade->conf_fw_watch = TRUE;
    }

    return ret;
}

static int
nat_del_link (int index,
              unsigned short type,
              unsigned int family,
              unsigned int flags,
              unsigned int change,
              const char *name)
{
    wapdnat *nat = NULL;
    int ret = 0;

    DEBUG ("Index: %d type: %u family: %u flags: %u change: %u name: %s",
           index, type, family, flags, change, name);

    return_val_if_fail (index > 0 && name, -EINVAL);

    ret = find_nat_configuration (name, &nat);
    if (ret < 0)
        return 0;

    DEBUG ("Nat dev: %s conf_fw_watch: %s", nat->interface,
           nat->masquerade->conf_fw_watch ? "Y" : "N");

    if (nat->masquerade->conf_fw_watch) {
        ret = nat_inotify_register (FALSE, nat, nat_inotify_iface, NULL, WAPMAN_NAT_IP_FORWARD_DEVICE_DIR,
                                    nat->masquerade->tethered_device);
        if (0 == ret)
            nat->masquerade->conf_fw_watch = FALSE;
    }

    return ret;
}

static rtnlwatch linkops = {
  .client  = "nat",
  .newlink = nat_new_link,
  .dellink = nat_del_link,
};

int
nat_init ()
{
    int err = 0;

    DEBUG ("");

    err = rtnl_notifier_register (&linkops);
    if (err < 0) {
        ERROR ("Failed to register to rtnl watch list");
        return err;
    }

    err = ipconfig_notifier_register (&nat_notify);
    if (err < 0) {
        ERROR ("Failed to register the device ops: %s/%d",
               strerror (-err), -err);
        (void) rtnl_notifier_unregister (&linkops);
        return err;
    }

    err = inotify_register (WAPMAN_NAT_IP_FORWARD_DIR, nat_inotify_global,
                            WAPMAN_NAT_IP_FORWARD_FILE, NULL);
    if (err  < 0) {
        ERROR ("Failed to register to inotifier for monitoring \"%s\"' : %s",
               WAPMAN_NAT_IP_FORWARD_FILE, strerror (-err));
        (void) rtnl_notifier_unregister (&linkops);
        (void) ipconfig_notifier_unregister (&nat_notify);
        return err;
    }

    err = nat_read_ip_forward (WAPMAN_NAT_IP_FORWARD_PATH);
    if (err < 0) {
        ERROR ("Failed to read %s : %s", WAPMAN_NAT_IP_FORWARD_FILE, strerror (-err));
        goto failure;
    }

    nat_hash = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, nat_cleanup);
    return err;

failure:
    (void) rtnl_notifier_unregister (&linkops);
    (void) ipconfig_notifier_unregister (&nat_notify);
    (void) inotify_unregister (WAPMAN_NAT_IP_FORWARD_DIR, nat_inotify_global,
                        WAPMAN_NAT_IP_FORWARD_FILE);
	return err;
}

int
nat_deinit ()
{
    int err = 0;

    DEBUG ("");

    err = rtnl_notifier_unregister (&linkops);
    if (err < 0)
        ERROR ("Failed to unregister the rtnl ops: %s/%d",
               strerror (-err), -err);

    err = ipconfig_notifier_unregister (&nat_notify);
    if (err < 0)
        ERROR ("Failed to unregister the device ops: %s/%d",
               strerror (-err), -err);

    err = inotify_unregister (WAPMAN_NAT_IP_FORWARD_DIR, nat_inotify_global,
                              WAPMAN_NAT_IP_FORWARD_FILE);
    if (err < 0)
        ERROR ("Failed to unregister to inotifier for monitoring \"%s\"' : %s",
               WAPMAN_NAT_IP_FORWARD_FILE, strerror (-err));

    g_hash_table_foreach (nat_hash, nat_shutdown, NULL);
    g_hash_table_destroy (nat_hash);

    nat_hash = NULL;
    return err;
}

/** @} */
