/**
 * @file session.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief Creates a session with connman to track changes in the
 *        various bearers.
 *
 * @{
 */

#include <glib.h>
#include <gio/gio.h>
#include <errno.h>
#include <string.h>
#include <log.h>
#include "inc/utils.h"
#include "session.h"
#include "connman-manager.h"
#include "connman-session.h"

typedef
struct session_data
{
    /* connection to the system dbus */
    GDBusConnection *system_bus;

    /* proxy for net.connman.Session */
    NetConnmanSession *session;

    /* object path of the active session */
    char *session_path;

    /* object path of the notifier object */
    char *notifier_path;

    /* A registered dbus service id of the notifier
     * service */
    guint register_id;

    /* introspection data for notifier */
    GDBusNodeInfo *notifier_introspection_data;

    /* active techs */
    techinfo *tech;
} sessiondata;

static const char
notifier_introspection_xml [] =
        "<node>"
        "  <interface name='net.connman.Notification'>"
        "    <method name='Release'>"
        "    </method>"
        "    <method name='Update'>"
        "      <arg type='a{sv}' name='settings' direction='in'/>"
        "    </method>"
        "  </interface>"
        "</node>";

static const char *
default_bearers [] = {
    "ethernet",
    "wifi",
    "bluetooth",
    NULL
};

/* dbus proxy to net.connman.Manager */
static NetConnmanManager *manager;

/* separate sessions for each bearers */
static GHashTable *session_hash;

/* watch id for connman */
static guint watch;

/* Registered clients for the notification of the
 * changes in the sessions */
static GList *notifier_clients;

/* forward declarations */
static void
notifier_method_call (GDBusConnection       *connection,
                      const gchar           *sender,
                      const gchar           *object_path,
                      const gchar           *interface_name,
                      const gchar           *method_name,
                      GVariant              *parameters,
                      GDBusMethodInvocation *invocation,
                      gpointer               user_data);

/* Virtual table for handling method calls for
 * net.connman.Notification */
static const GDBusInterfaceVTable
notifier_interface_vtable = {
      notifier_method_call,
      NULL,
      NULL,
      {NULL}
};

static NetConnmanSession*
session_create_session_proxy (GDBusConnection *connection,
                              const gchar* path)
{

    NetConnmanSession* sess_proxy = NULL;
    GError* error = NULL;

    return_val_if_fail (path, NULL);
    DEBUG ("Creating proxy for the interface %s", CONNMAN_SESSION_INTERFACE_NAME);

    sess_proxy = net_connman_session_proxy_new_sync (connection,           /* System bus */
                                                     G_DBUS_PROXY_FLAGS_NONE,    /* proxy flags */
                                                     CONNMAN_BUS_NAME,           /* service name */
                                                     path,                       /* object path */
                                                     NULL,                       /* GCancellable */
                                                     &error);                    /* GError */

    if (!sess_proxy) {
        ERROR ("Failed to create proxy for the interface %s error message is : %s",
              CONNMAN_SESSION_INTERFACE_NAME, error->message);
        /* free the error */
        g_error_free (error);
        return NULL;
    }

    INFO ("Proxy successfully created for the interface %s", CONNMAN_SESSION_INTERFACE_NAME);
    return sess_proxy;
}

static NetConnmanManager*
session_create_manager_proxy (GDBusConnection *connection)
{
    NetConnmanManager* manager_proxy = NULL;
    GError* error = NULL;

    DEBUG ("Creating proxy for the interface %s", CONNMAN_MANAGER_INTERFACE_NAME);

    manager_proxy = net_connman_manager_proxy_new_sync (connection,
                                                        G_DBUS_PROXY_FLAGS_NONE,
                                                        CONNMAN_BUS_NAME,
                                                        CONNMAN_MANAGER_INTERFACE_PATH,
                                                        NULL,
                                                        &error);

    if (!manager_proxy) {
        ERROR ("Failed to create proxy for the interface %s, error message is : %s",
               CONNMAN_MANAGER_INTERFACE_NAME, error->message);
        g_error_free (error);
        return NULL;
    }

    INFO ("Proxy successfully created for the interface %s", CONNMAN_MANAGER_INTERFACE_NAME);
    return manager_proxy;
}

static void
session_notify_clients (void)
{
    GHashTableIter iter;
    gpointer key, value;
    GList *temp;

    DEBUG ("");

    temp = notifier_clients;
    g_hash_table_iter_init (&iter, session_hash);
    while (g_hash_table_iter_next (&iter, &key, &value)) {

        const char *bearer = key;
        sessiondata *session = value;

        if (!session)
            continue;

        if (!session->tech)
            continue;

        if (session->tech->state == STATE_ONLINE ||
                session->tech->state == STATE_CONNECTED) {
            for ( ; temp; temp = temp->next) {
                notifier_ops *ops = temp->data;
                if (!ops || !ops->default_changed)
                    continue;
                INFO ("Notifying the client \"%s\" on the change in the default interface %s",
                      ops->name ? ops->name : "UNKNOWN", bearer);
                ops->default_changed (session->tech);
            }

            break;
        }
    }
}

static void
notifier_method_call (GDBusConnection       *connection,
                      const gchar           *sender,
                      const gchar           *object_path,
                      const gchar           *interface_name,
                      const gchar           *method_name,
                      GVariant              *parameters,
                      GDBusMethodInvocation *invocation,
                      gpointer               user_data)
{
    (void) connection;
    (void) invocation;

    sessiondata *session = user_data;

    return_if_fail (session && session->tech);

    INFO ("***********************************************************")
    INFO ("session update for session %p bearer : \"%s\"", session, session->tech->bearer);
    INFO ("Sender name : %s, Object path : %s, Interface name : %s, Method name : %s",
          sender, object_path, interface_name, method_name);

    if (g_strcmp0 (method_name, "Update") == 0) {

        GVariantIter iter,val;
        GVariant *value,*child;
        gchar *key;
        const gchar *st;
        gsize length;

        INFO("Parameters type of the update : %s", g_variant_get_type_string(parameters));

        g_variant_iter_init (&iter, parameters);
        while ((child = g_variant_iter_next_value (&iter))) {

            g_variant_iter_init (&val, child);
            while (g_variant_iter_next (&val, "{sv}", &key, &value)) {

                /* Session state update */
                if (g_strcmp0 (key, "State") == 0) {

                    st = g_variant_get_string (value,&length);
                    if (length > 0) {

                        INFO ("State of the session : %s",st);
                        if (!g_strcmp0 (st, SESSION_STATE_DISCONNECTED))
                            session->tech->state = STATE_DISCONNECTED;
                        else if (!g_strcmp0 (st, SESSION_STATE_CONNECTED))
                            session->tech->state = STATE_CONNECTED;
                        else if (!g_strcmp0 (st, SESSION_STATE_ONLINE))
                            session->tech->state = STATE_ONLINE;
                    } else
                        INFO ("Invalid string length for the state");
                }
                else if (g_strcmp0 (key, "Name") == 0) {

                    st = g_variant_get_string (value, &length);
                    if (length > 0) {

                        INFO ("Name of the service to which the session is connected : %s",st);
                        /* We already have the service name, check whether there is a
                         * change in the service */
                        if (session->tech->service) {
                            if (g_strcmp0 (st,session->tech->service) != 0) {
                                g_free (session->tech->service);
                                session->tech->service = g_strdup (st);
                            }
                        } else
                            session->tech->service = g_strdup (st);
                    } else {
                        /* could be that session is disconnected and there
                         * aint any active service connection */
                        if (session->tech->service) {
                            /* free the service name */
                            g_free (session->tech->service);
                            session->tech->service = NULL;
                        }
                    }
                }
                else if (g_strcmp0 (key, "Bearer") == 0) {

                    st = g_variant_get_string (value, &length);
                    if (length > 0)
                        INFO ("Active bearer of the session : %s", st);
                }
                else if (g_strcmp0 (key, "Interface") == 0) {

                    st = g_variant_get_string (value, &length);
                    if (length > 0) {

                        INFO ("Currently used interface of the session : %s", st);
                        /* We already have the interface name, check whether there is a
                         * change in the network interface */
                        if (session->tech->interface) {
                            if (g_strcmp0 (st,session->tech->interface) != 0) {
                                g_free (session->tech->interface);
                                session->tech->interface = g_strdup (st);
                            }
                        } else
                            session->tech->interface = g_strdup (st);
                    } else {
                        /* could be that session is disconnected and there
                         * aint any interface active */
                        if (session->tech->interface) {
                            g_free (session->tech->interface);
                            session->tech->interface = NULL;
                        }
                    }

                }
                else if (g_strcmp0 (key, "IPv4") == 0) {

                    GVariantIter ipiter;
                    GVariant *ipvalue;
                    gchar* ipkey;
                    const gchar* ipstr;

                    INFO ("IPv4 parameters type : %s", g_variant_get_type_string (value));

                    g_variant_iter_init (&ipiter, value);
                    while (g_variant_iter_next (&ipiter, "{sv}", &ipkey, &ipvalue)) {

                        ipstr = g_variant_get_string (ipvalue, &length);
                        if (length > 0) {

                            INFO ("IPv4 Settings, key : %s value : %s", ipkey, ipstr);

                            if (g_strcmp0 (ipkey, "Address")) {
                                if (session->tech->ipdetails->address)
                                    g_free (session->tech->ipdetails->address);
                                session->tech->ipdetails->address = g_strdup (ipstr);
                            }
                            else if (g_strcmp0 (ipkey, "Netmask")) {
                                if (session->tech->ipdetails->netmask)
                                    g_free (session->tech->ipdetails->netmask);
                                session->tech->ipdetails->netmask = g_strdup (ipstr);
                            }
                            else if (g_strcmp0 (ipkey, "Gateway")) {
                                if (session->tech->ipdetails->gw)
                                    g_free (session->tech->ipdetails->gw);
                                session->tech->ipdetails->gw = g_strdup (ipstr);
                            }
                        }

                        /* free the contents */
                        g_variant_unref (ipvalue);
                        g_free (ipkey);
                    }

                }
                else if (g_strcmp0 (key, "IPv6") == 0) {

                    GVariantIter ipiter;
                    GVariant *ipvalue;
                    gchar* ipkey;
                    const gchar* ipstr;

                    INFO ("IPv6 parameters type : %s", g_variant_get_type_string (value));

                    g_variant_iter_init (&ipiter, value);
                    while (g_variant_iter_next (&ipiter, "{sv}", &ipkey, &ipvalue)) {

                        ipstr = g_variant_get_string (ipvalue, &length);

                        if (length > 0)
                            INFO ("IPv6 Settings, key : %s value : %s", ipkey, ipstr);

                        /* free the contents */
                        g_variant_unref (ipvalue);
                        g_free (ipkey);
                    }
                }
                else if (g_strcmp0 (key, "ConnectionType") == 0) {

                    st = g_variant_get_string (value, &length);

                    if (length > 0)
                        INFO ("Connection type of the session : %s", st);
                }
                else if (g_strcmp0 (key, "AllowedBearers") == 0) {

                    const gchar** allowedbearers = NULL;
                    const gchar** temp = NULL;

                    allowedbearers = g_variant_get_strv (value,&length);

                    if (length > 0) {
                        temp = allowedbearers;

                        /* null terminated string array */
                        while (*temp)
                            INFO ("Allowed bearers for this sesssion : %s", *temp++);

                        g_free (allowedbearers);
                    }
                }

                g_variant_unref (value);
                g_free (key);
            }
        }

        /* notify the clients if there is a change
         * in the default interface */
        session_notify_clients ();
    }
    else if (g_strcmp0 (method_name, "Release") == 0) {

        /* Session has been released i.e., session is
           no more active */
        g_hash_table_remove (session_hash, session->tech->bearer);

    } else {

        /* unknown method i.e., not supported */
    }
    INFO ("***********************************************************")
}

static void
session_create_callback (GObject *source_object,
                         GAsyncResult *res,
                         gpointer user_data)
{
    sessiondata *session;
    GError* error = NULL;
    gchar* session_path = NULL;
    gboolean result = FALSE;

    DEBUG ("");

    session = user_data;
    return_if_fail (session);

    result = net_connman_manager_call_create_session_finish ((NetConnmanManager*)source_object,
                                                             &session_path,
                                                             res,
                                                             &error);

    if (!result) {
        ERROR ("Failed to create the session error message : %s", error->message);
        /* free the error */
        g_error_free(error);
        return;
    }

    INFO ("successfully created the session for %p bearer %s , session path is : %s",
          session, session->tech->bearer, session_path);
    session->session_path = g_strdup (session_path);
    session->session = session_create_session_proxy (session->system_bus,
                                                     session->session_path);

    if (!session->session) {
        ERROR("Failed to create the proxy for net.connman.session");
        g_free (session_path);
        g_free (session->session_path);
        return;
    }

    /* free the session path */
    g_free (session_path);
}

static int
session_create_connman_session (sessiondata *session)
{
    GVariantBuilder *builder, *dict;
    GVariant *bearers,*input,*conntype;

    return_val_if_fail (session && session->tech, -EINVAL);

    DEBUG ("session %p bearer %s", session, session->tech->bearer);

    return_val_if_fail (manager, -EINVAL);
    /* adding bearers */
    builder = g_variant_builder_new (G_VARIANT_TYPE ("as"));
    g_variant_builder_add (builder, "s", session->tech->bearer);
    bearers = g_variant_new ("as", builder);
    g_variant_builder_unref (builder);

    /* adding connection type */
    conntype = g_variant_new ("s", SESSION_CONNECTION_TYPE);

    dict = g_variant_builder_new (G_VARIANT_TYPE ("a{sv}"));
    g_variant_builder_add (dict, "{sv}", SESSION_PARAMS_BEARERS,
                           bearers);
    g_variant_builder_add (dict, "{sv}", SESSION_PARAMS_CONNTYPE,
                           conntype);

    input = g_variant_new ("a{sv}",dict);
    g_variant_builder_unref (dict);

    net_connman_manager_call_create_session (manager,
                                             input,
                                             session->notifier_path,
                                             NULL,
                                             session_create_callback,
                                             session);

    INFO ("Successfully posted create session method call");
    return 0;
}

static void
session_techinfo_cleanup (techinfo *tech)
{
    return_if_fail (tech);

    INFO ("cleaning up techinfo : %s [%p]",
          tech->bearer, tech);

    g_free (tech->bearer);
    g_free (tech->interface);
    g_free (tech->service);

    if (tech->ipdetails) {
        g_free (tech->ipdetails->address);
        g_free (tech->ipdetails->netmask);
        g_free (tech->ipdetails->gw);
        g_free (tech->ipdetails);
    }

    g_free (tech);
}

static void
session_cleanup (gpointer data)
{
    sessiondata *session = data;

    return_if_fail (session);

    INFO ("cleaning up session for the bearer : %s [%p]",
          session->tech == NULL ? "UNKNOWN"
          : session->tech->bearer == NULL ? "UNKNOWN"
          : session->tech->bearer,
          session);

    if (g_dbus_connection_unregister_object (session->system_bus,
                                             session->register_id)) {
        INFO ("Successfully unregistered the object with id : %d",
              session->register_id);
        session->register_id = 0;
    }

    g_free (session->notifier_path);
    g_free (session->session_path);
    g_dbus_node_info_unref (session->notifier_introspection_data);
    g_object_unref (session->session);
    session_techinfo_cleanup (session->tech);
    g_free (session);
}

static sessiondata*
session_create (GDBusConnection *connection,
                const char *const bearer)
{
    GError *error = NULL;
    sessiondata *session;

    return_val_if_fail (bearer, NULL);

    DEBUG ("Dbus connection [%p], creating session for the bearer : \"%s\"",
           connection, bearer);

    session = g_try_malloc0 (sizeof (*session));
    return_val_if_fail (session, NULL);
    session->system_bus = connection;
    session->notifier_path = g_strdup_printf ("/org/bosch/wapdman/%s/notifier_%d",
                                              bearer, getpid());
    INFO ("session %p bearer %s notifier path %s", session,
          bearer, session->notifier_path);
    session->tech = g_try_malloc0 (sizeof (techinfo));
    if (!session->tech) {
        g_free (session);
        return NULL;
    }

    session->tech->ipdetails = g_try_malloc0 (sizeof (ipv4));
    if (!session->tech->ipdetails) {
        g_free (session->tech);
        g_free (session);
        return NULL;
    }

    session->tech->bearer = g_strdup (bearer);

    session->notifier_introspection_data =
            g_dbus_node_info_new_for_xml (notifier_introspection_xml, NULL);
    if (!session->notifier_introspection_data) {
        ERROR ("Failed to export the introspection xml of notifier");
        goto out;
    }

    session->register_id =
            g_dbus_connection_register_object (connection,
                                               session->notifier_path,
                                               session->notifier_introspection_data->interfaces[0],
                                               &notifier_interface_vtable,
                                               session,
                                               NULL,
                                               &error);

    INFO ("Notifier object registration id : %d", session->register_id);
    if (!session->register_id) {
        ERROR ("Error occurred while registering notifier object to system bus : %s", error->message);
        g_error_free (error);
        goto out;
    }

    return session;
out:
    g_free (session->notifier_path);
    g_free (session->tech->bearer);
    g_free (session->tech->ipdetails);
    g_free (session->tech);
    g_free (session);
    return NULL;
}

static int
session_create_all (GHashTable *hash)
{
    int err = 0;
    GHashTableIter iter;
    gpointer key, value;

    return_val_if_fail (hash, -EINVAL);

    g_hash_table_iter_init(&iter, hash);
    while (g_hash_table_iter_next (&iter, &key, &value)) {
        const char *bearer = key;
        sessiondata *session = value;
        INFO ("Requesting session %p for bearer : %s", session, bearer);
        if ((err = session_create_connman_session (session)) < 0)
            ERROR ("Failed to create session for %p bearer %s", session,
                   session->tech->bearer);
    }

    return err;
}

static int
session_initialise (GDBusConnection *connection)
{
    int err = 0, index = 0;
    sessiondata *session;

    for ( ; default_bearers [index]; index ++) {
        session = session_create (connection, default_bearers [index]);
        if (!session) {
            err = -ENOTCONN;
            continue;
        }

        g_hash_table_replace (session_hash, g_strdup (default_bearers [index]),
                              session);
    }

    return err;
}

static void
session_connman_appeared (GDBusConnection *connection,
                          const gchar     *name,
                          const gchar     *name_owner,
                          gpointer         user_data)
{
    int err;

    (void) connection;
    (void) user_data;

    INFO ("Name %s is owned by %s", name, name_owner);

    /* Try to create proxy to net.connman.Manager if connman is
     * already added to the system bus */
    if (!manager) {
        manager = session_create_manager_proxy (connection);
        if (!manager) {
            CRITICAL ("Failed to create the proxy object for %s", CONNMAN_MANAGER_INTERFACE_NAME);
            return;
        }
    }

    if ((err = session_initialise (connection)) < 0)
        ERROR ("Error occured in initializing various sessions : %s", strerror (-err));

    /* Even then we proceed to create the initialized sessions */
    (void) session_create_all (session_hash);
}

static void
session_connman_vanished (GDBusConnection *connection,
                          const gchar     *name,
                          gpointer         user_data)
{
    (void) connection;
    (void) user_data;

    INFO ("Name %s does not exist anymore", name);

    /* Not necessary to cleanup sessions here, if connman gets crashed or
     * deregistered from system dbus, it would have already called the
     * release method of net.connman.Notification where we would have
     * cleaned up the sessions */

    /* clear the invalid proxy to net.connman.Manager */
    if (manager) {
        INFO ("Clearing the proxy for net.connman.Manager");
        g_object_unref (manager);
        manager = NULL;
    }
}

int
session_notifier_register (notifier_ops *ops)
{
    int found = 0;
    notifier_ops *registeredops;
    GList *temp;

    return_val_if_fail (ops, -EINVAL);

    DEBUG ("Registering notifier client : %s [%p]",
           ops->name ? ops->name : "UNKNOWN", ops);

    temp = notifier_clients;
    for ( ; temp; temp = temp->next) {
        registeredops = temp->data;
        if (ops == registeredops) {
            found = 1;
            break;
        }
    }

    if (!found) {
        notifier_clients = g_list_append (notifier_clients, ops);
        return 0;
    }

    return -EALREADY;
}

int
session_notifier_unregister (notifier_ops *ops)
{
    int found = 0;
    GList *temp;

    return_val_if_fail (ops, -EINVAL);

    DEBUG ("Unregistering the notifier client : %s [%p]",
           ops->name ? ops->name : "UNKNOWN", ops);

    temp = notifier_clients;
    for ( ; temp; temp = temp->next) {
        notifier_ops *registeredops = temp->data;
        if (ops == registeredops) {
            found = 1;
            break;
        }
    }

    if (found) {
        notifier_clients = g_list_remove_link (notifier_clients, temp);
        return 0;
    }

    return -ENOENT;
}

int
session_init ()
{
    DEBUG ("");

    /* Start watching ConnMan */
    watch = g_bus_watch_name (G_BUS_TYPE_SYSTEM,
                              CONNMAN_BUS_NAME,
                              G_BUS_NAME_WATCHER_FLAGS_NONE,
                              session_connman_appeared,
                              session_connman_vanished,
                              NULL, NULL);

    /* valid watch ids are never 0 */
    if (!watch)
        return -ENOTCONN;

    session_hash = g_hash_table_new_full (g_str_hash, g_str_equal,
                                          g_free, session_cleanup);
    return 0;
}


int
session_deinit (void)
{
    DEBUG ("");

    if (manager) {
        INFO ("Clearing the proxy for net.connman.Manager");
        g_object_unref (manager);
        manager = NULL;
    }

    if (watch > 0)
        g_bus_unwatch_name (watch);

    g_list_free (notifier_clients);
    g_hash_table_destroy (session_hash);

    return 0;
}

/** @} */
