/**
 * @file log.h
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#ifndef LOG_H
#define LOG_H

#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>

#define TIMESTAMP   20
#define BUFFER      128

#define LOG_LEVEL_0  LOG_EMERG
#define LOG_LEVEL_1  LOG_ALERT
#define LOG_LEVEL_2  LOG_CRIT
#define LOG_LEVEL_3  LOG_ERR
#define LOG_LEVEL_4  LOG_WARNING
#define LOG_LEVEL_5  LOG_NOTICE
#define LOG_LEVEL_6  LOG_INFO
#define LOG_LEVEL_7  LOG_DEBUG
/* debug priorities for syslog */
typedef
enum {
    /* emergency messages */
    DEBUG_EMERG,

    /* alerts */
    DEBUG_ALERT,

    /* critical messages */
    DEBUG_CRITICAL,

    /* error messages */
    DEBUG_ERR,

    /* warning messages */
    DEBUG_WARNING,

    /* significant conditions */
    DEBUG_NOTICE,

    /* informational */
    DEBUG_INFO,

    /* debug-level messages */
    DEBUG_TRACE,
} debug_prio;

#define ENUMTOSTR(exp)    #exp

#define \
DEBUG(fmt, arg...) \
    debug (DEBUG_TRACE, "%s:[%d]:%s() " fmt, __FILE__, __LINE__ , __FUNCTION__ , ## arg);
#define \
WARNING(fmt, arg...) \
    debug (DEBUG_WARNING, "%s:[%d]:%s() " fmt, __FILE__, __LINE__ , __FUNCTION__ , ## arg);
#define \
ERROR(fmt, arg...) \
    debug (DEBUG_ERR, "%s:[%d]:%s() " fmt, __FILE__, __LINE__ , __FUNCTION__ , ## arg);
#define \
INFO(fmt, arg...) \
    debug (DEBUG_INFO, "%s:[%d]:%s() " fmt, __FILE__, __LINE__ , __FUNCTION__ , ## arg);
#define \
CRITICAL(fmt, arg...) \
    debug (DEBUG_CRITICAL, "%s:[%d]:%s() " fmt, __FILE__, __LINE__ , __FUNCTION__ , ## arg);
#define \
ALERT(fmt, arg...) \
    debug (DEBUG_ALERT, "%s:[%d]:%s() " fmt, __FILE__, __LINE__ , __FUNCTION__ , ## arg);
#define \
HEXDUMP(data, length) \
    hexdump (DEBUG_INFO, data, length);

#define likely(x)       __builtin_expect(!!(x), 1)
#define unlikely(x)     __builtin_expect(!!(x), 0)

#define assert_if_fail(expr) \
    do { if (likely (expr)) { } else { WARNING ("Expected condition \"%s\" failed", #expr); } } while (0)

#define return_if_fail(expr) \
    do { if (likely (expr)) { } else { ERROR ("Expected condition \"%s\" failed", #expr); return; } } while (0)

#define return_val_if_fail(expr, val) \
    do { if (likely (expr)) { } else { ERROR ("Expected condition \"%s\" failed", #expr); return (val); } } while (0)

#define continue_if_fail(expr)  \
   if (likely (expr)) { } else { DEBUG ("Expected condition \"%s\" failed", #expr); continue; }

#define ARRAY_SIZE(X)   (sizeof (X) / sizeof (X [0]))
#define LMLN_FMAT       "\n\t\t\t\t\t\t "

#define BIT(x)          (1U << (x))

#define WARN_UNUSED     __attribute__((warn_unused_result))

extern void debug (int prio, const char* format, ...);
extern void vdebug (int prio, const char* msg, va_list args);
extern void hexdump (int prio, const char *data, size_t size);
extern int timestamp (char *tme);


/** Logging utility operations */
int log_init (char *program_name, const int log_level, const char *debugfile);
int log_deinit ();
char* log_get_program_name(void);

/* Utility Operations */
typedef
struct __log_reg_ops
{
    char* util_name;
    int (*util_open) ();
    int (*util_close) ();
    int (*util_log)(int, const char*, va_list);
} logops;

/* Utility Registration And Deregistering */
int log_util_register (logops *util);
int log_util_unregister (logops *util);

#endif // LOG_H
/** @} */
