/**
 * @file firewall.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <stdlib.h>
#include <string.h>
#include <sys/errno.h>
#include <glib.h>
#include "firewall.h"
#include "iptables.h"
#include <log.h>

#define FW_ALL_RULES    -1

typedef
enum __fw_commit {

    /*invalid*/
    FW_COMMIT_NONE,

    /* commit all rules of a fw context */
    FW_COMMIT_ALL,

    /* commit a single rule from the given id
     * of a fw context */
    FW_COMMIT_ID
} fw_commit_t;

/* Status of a particular rule */
typedef
enum _fw_rule_status
{
    /* initial status */
    FW_RULE_NONE = 0,

    /* commit filed as the rule is
     * not valid, next time commit shall
     * not be tried for this rule */
    FW_RULE_INVALID = 1,

    /* Successfully commited to netfilter
     * module and the rule is currently active */
    FW_RULE_ENABLED = 2,

    /* valid rule, the current status is disabled
     * i.e., not active, can be commited again */
    FW_RULE_DISABLED = 3,

    /* Rule already exists i.e., not added by us
     * and in such case dont duplicate and delete */
    FW_RULE_EXISTS = 4
} fwrulestatus;

/* represents a firewall rule
 * i.e., ipt_entry */
typedef
struct __fw_rule
{
    /* unique id for this rule */
    unsigned int id;

    /* current status of the rule
     * i.e., invalid/none/enabled/disabled */
    fwrulestatus status;

    /* table to be manipulated */
    char *table;

    /* chain that this rule meant for */
    char *chain;

    /* rule_spec i.e., contains the matches and
     * the targets to be performed if the rule
     * matches */
    char *rule;
} fwrule;

static firewall_ops *driver;
static unsigned int rule_id;

static const char*
firewall_rule_status_to_string (fwrulestatus status)
{
    switch (status) {
    case FW_RULE_NONE:
        return ENUMTOSTR (FW_RULE_NONE);
    case FW_RULE_INVALID:
        return ENUMTOSTR (FW_RULE_INVALID);
    case FW_RULE_ENABLED:
        return ENUMTOSTR (FW_RULE_ENABLED);
    case FW_RULE_DISABLED:
        return ENUMTOSTR (FW_RULE_DISABLED);
    case FW_RULE_EXISTS:
        return ENUMTOSTR (FW_RULE_EXISTS);
    }

    return ENUMTOSTR (UNKNOWN);
}

static const char*
firewall_commit_status2string (fw_commit_t commit)
{
    switch (commit) {
    case FW_COMMIT_NONE:
        return ENUMTOSTR (FW_COMMIT_NONE);
    case FW_COMMIT_ID:
        return ENUMTOSTR (FW_COMMIT_ID);
    case FW_COMMIT_ALL:
        return ENUMTOSTR (FW_COMMIT_ALL);
    }

    return ENUMTOSTR (UNKNOWN);
}

static int
firewall_valid_table (const char *const table)
{
    return_val_if_fail (table, -EINVAL);
    if (!g_strcmp0 (table, "filter") || !g_strcmp0 (table, "mangle") ||
            !g_strcmp0 (table, "nat") )
        return 0;
    return -EINVAL;
}

static void
firewall_free_rule (gpointer data)
{
    fwrule *rul = data;

    return_if_fail (rul);

    DEBUG ("Freeing the rule [%p] table : %s, chain : %s, id : %d, "
           "rule_spec : %s, status %s", rul, rul->table, rul->chain,
           rul->id, rul->rule, firewall_rule_status_to_string (rul->status));

    g_free (rul->table);
    g_free (rul->chain);
    g_free (rul->rule);
    g_free (rul);
}

void
firewall_cleanup_context (fwcontext *ctx)
{
    DEBUG ("cleaning up the fwcontext : %p", ctx);
    g_list_free_full (ctx->rules, firewall_free_rule);
}

fwcontext*
firewall_create_context (void)
{
    fwcontext *ctx;

    ctx = g_try_malloc0 (sizeof (*ctx));
    DEBUG ("New FW context created: %p", ctx);
    /* responsibility of the caller to check whether
     * the returned ctx is NULL */
    return ctx;
}

int
firewall_add_rule (fwcontext *ctx,
                   const char *table,
                   const char *chain,
                   const char *rule_fmt,
                   ...)
{
    int err = 0;
    va_list args;
    char *rule_spec;
    fwrule *rule;

    /* clients shall have a valid
     * context */
    return_val_if_fail (ctx, -ENOTSUP);
    /* invalid table && chain information is necessary */
    return_val_if_fail (table && chain, -EINVAL);

    DEBUG ("table %s chain %s ctx %p", table, chain, ctx);

    err = firewall_valid_table (table);
    if (err < 0) {
        ERROR ("Request to add a rule to an invalid table");
        return err;
    }

    va_start (args, rule_fmt);
    rule_spec = g_strdup_vprintf (rule_fmt, args);
    va_end (args);

    INFO ("Rule spec : \"%s\"", rule_spec);

    rule = g_try_malloc0 (sizeof (*rule));
    if (!rule) {
        g_free (rule_spec);
        return -ENOMEM;
    }

    rule->id = ++rule_id;
    rule->status = FW_RULE_NONE;
    rule->table = g_strdup (table);
    rule->chain = g_strdup (chain);
    rule->rule = rule_spec;

    ctx->rules = g_list_append (ctx->rules, rule);
    return err;
}

int
firewall_remove_rule (fwcontext *ctx,
                      const unsigned int id)
{
    int found = 0;
    fwrule *rule;
    GList *rules;

    return_val_if_fail (ctx, -EINVAL);

    DEBUG ("Request to remove rule with id [%d] from ctx %p", id, ctx);

    rules = ctx->rules;
    for ( ; rules; rules = rules->next) {
        rule = rules->data;
        if (rule && rule->id == id) {
            found = 1;
            break;
        }
    }

    if (found) {
        ctx->rules = g_list_remove_link (ctx->rules, rules);
        firewall_free_rule ((void *)rule);
        return 0;
    }

    return -ENOENT;
}

static int
firewall_commit_rule (fwrule *rule,
                      const char *option)
{
    int err = 0;
    char *r = NULL;

    return_val_if_fail (rule, -EINVAL);
    /* no firewall driver found */
    return_val_if_fail (driver, -ENODEV);
    /* firewall driver does not support to commit a rule */
    return_val_if_fail (driver->commitrule, -ENOTSUP);

    r = g_strdup_printf ("-t %s %s %s %s", rule->table, option,
                         rule->chain, rule->rule);

    err = driver->commitrule (r);
    if (err < 0 && err != -EEXIST)
        ERROR ("Failed to commit the rule: %s [%s/%d]", r,
               strerror (-err), -err);

    g_free (r);
    return err;
}

static fwrule*
firewall_get_rule_from_id (fwcontext *ctx,
                           const unsigned int id)
{
    GList *rules;
    fwrule *r;

    return_val_if_fail (ctx, NULL);
    rules = ctx->rules;
    for ( ; rules; rules = rules->next) {
        r = rules->data;
        if (r && r->id == id)
            return r;
    }

    return NULL;
}

int
firewall_enable_rule (fwcontext *ctx,
                      const int commit,
                      const unsigned int id)
{
    int err = 0;
    fwrule *rule;
    GList *rules;
    /* append */
    const char *option = "-A";

    return_val_if_fail (ctx, -EINVAL);

    DEBUG ("fw ctx %p commit : %s id : %u", ctx,
           firewall_commit_status2string (commit), id);

    if (commit == FW_COMMIT_NONE)
        return 0;

    if (commit == FW_COMMIT_ID) {
        rule = firewall_get_rule_from_id (ctx, id);
        return_val_if_fail (rule, -ENOENT);

        /* Try to commit only if the status of rule is
         * disabled/none */
        if (rule->status == FW_RULE_ENABLED ||
                rule->status == FW_RULE_EXISTS)
            return -EALREADY;
        /* invalid rule */
        else if (rule->status == FW_RULE_INVALID)
            return -EINVAL;

        err = firewall_commit_rule (rule, option);
        if (err < 0) {
            if (err == -EEXIST) {
                err = 0;
                rule->status = FW_RULE_EXISTS;
            } else { rule->status = FW_RULE_INVALID; }
        } else {
            rule->status = FW_RULE_ENABLED;
        }
    } else {
        rules = ctx->rules;
        for ( ; rules; rules = rules->next) {
            rule = rules->data;
            continue_if_fail (rule);

            /* skip the rules which are already commited or
             * invalid */
            if (rule->status == FW_RULE_ENABLED ||
                    rule->status == FW_RULE_INVALID ||
                    rule->status == FW_RULE_EXISTS)
                continue;
            err = firewall_commit_rule (rule, option);
            if (err < 0) {
                if (err == -EEXIST) {
                    err = 0;
                    rule->status = FW_RULE_EXISTS;
                } else { rule->status = FW_RULE_INVALID; }
            } else {
                rule->status = FW_RULE_ENABLED;
            }
        }
    }

    return err;
}

int
firewall_disable_rule (fwcontext *ctx,
                       const int commit,
                       const unsigned int id)
{
    int err = 0;
    fwrule *rule;
    GList *rules;
    const char *option = "-D"; /* delete the rule */

    return_val_if_fail (ctx, -EINVAL);

    DEBUG ("fw ctx %p commit : %s id : %u", ctx,
           firewall_commit_status2string (commit), id);

    if (commit == FW_COMMIT_NONE)
        return 0;

    if (commit == FW_COMMIT_ID) {
        rule = firewall_get_rule_from_id (ctx, id);
        return_val_if_fail (rule, -ENOENT);

        /* Disable the rule only if it is enabled */
        if (rule->status != FW_RULE_ENABLED)
            return rule->status == FW_RULE_EXISTS ? 0 : -EINVAL;

        err = firewall_commit_rule (rule, option);
        if (err < 0)
            rule->status = FW_RULE_INVALID;
        else
            rule->status = FW_RULE_DISABLED;
    } else {
        rules = ctx->rules;
        for ( ; rules; rules = rules->next) {
            rule = rules->data;
            continue_if_fail (rule);

            /* only disable the rules which are already
             * commited by us */
            if (rule->status != FW_RULE_ENABLED)
                continue;
            err = firewall_commit_rule (rule, option);
            if (err < 0)
                rule->status = FW_RULE_INVALID;
            else
                rule->status = FW_RULE_DISABLED;
        }
    }

    return err;
}

int
firewall_enable (fwcontext *ctx)
{
    int err = 0;

    return_val_if_fail (ctx, -EINVAL);

    DEBUG ("Enabling the firewall context %p", ctx);

    err = firewall_enable_rule (ctx, FW_COMMIT_ALL, 0);
    if (err < 0) {
        ERROR ("Failed to install iptables rules: %s",
               strerror (-err));
//        (void) firewall_disable_rule (ctx, FW_ALL_RULES);
    }

    return err;
}

int
firewall_disable (fwcontext *ctx)
{
    return_val_if_fail (ctx, -EINVAL);
    DEBUG ("Disabling the firewall context %p", ctx);
    return firewall_disable_rule (ctx, FW_COMMIT_ALL, 0);
}

int
firewall_driver_register(firewall_ops *ops)
{
    return_val_if_fail (ops, -EINVAL);

    if (driver == ops)
        return -EEXIST;

    DEBUG ("Registering firewall driver %s", ops->name ? ops->name : "UNKNOWN");
    driver = ops;
    return 0;
}

int
firewall_driver_unregister(firewall_ops *ops)
{
    return_val_if_fail (ops, -EINVAL);

    if (driver != ops)
        return -ENOENT;

    DEBUG ("Unregistering firewall driver %s", ops->name ? ops->name : "UNKNOWN");
    driver = NULL;
    return 0;
}

int
firewall_init()
{
    DEBUG ("");

    return 0;
}

int
firewall_deinit()
{
    DEBUG ("");

    return 0;
}

/** @} */
