/**
 * @file iptables.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/errno.h>
#include <sys/types.h>
#include <xtables.h>
#include <syslog.h>
#include <linux/capability.h>
#include <sys/capability.h>
#include <getopt.h>
#include <netdb.h>
#include <glib.h>
#include "firewall.h"
#include "capabilities.h"
#include <log.h>
#include "inc/utils.h"
#include <linux/netfilter_ipv4/ip_tables.h>

/* valid tables */
#define IPTABLES             "ip_tables"
#define IPTABLES_FILTER      "iptable_filter"
#define IPTABLES_NAT         "iptable_nat"
#define IPTABLES_MANLGE      "iptable_mangle"
#define IPTABLES_RAW         "iptable_raw"
#define IPTABLES_SECUTIRY    "iptable_security"

/* Netfilter proto family and socket type */
#define NETFILTER_FAMILY    AF_INET
#define NETFILTER_PROTO     IPPROTO_RAW

/* for reading the entries from /proc/modules */
#define KERNEL_MOD_PATH "/proc/modules"
#define MODULENAME_SIZE 256
#define O_READONLY      "r"
#define SPACE           ' '
#define NEWLINE         '\n'

/* debugging the IP address */
#define IP_ADDRESS(n)			\
    (unsigned int) ((n)>>24)&0xFF,			\
    (unsigned int) ((n)>>16)&0xFF,			\
    (unsigned int) ((n)>>8)&0xFF,			\
    (unsigned int) ((n)&0xFF)

/* Type of the targets */
#define IPT_ENTRY_STANDARD          (1 << 0)
#define IPT_ENTRY_FALLTHROUGH       (1 << 1)
#define IPT_ENTRY_JUMP              (1 << 2)
#define IPT_ENTRY_MODULE            (1 << 3)

#define IPTC_STANDARD_TARGET        ""
#define IPTC_ERROR_TARGET           "ERROR"

/* Standard targets */
#define LABEL_ACCEPT  "ACCEPT"
#define LABEL_DROP    "DROP"
#define LABEL_QUEUE   "QUEUE"
#define LABEL_RETURN  "RETURN"
#define LABEL_STOP    "STOP"
#define LABEL_STOLEN  "STOLEN"

/* Needed for parser. Indicates the number of
 * options specified in the rule */
#define IPT_BUILD_OPTION_NONE       (1 << 0)
#define IPT_BUILD_OPTION_APPEND     (1 << 1)
#define IPT_BUILD_OPTION_DELETE     (1 << 2)
#define IPT_BUILD_OPTION_SET_PROTO  (1 << 3)
#define IPT_BUILD_OPTION_DST_ADDR   (1 << 4)
#define IPT_BUILD_OPTION_SRC_ADDR   (1 << 5)
#define IPT_BUILD_OPTION_IN_IFACE   (1 << 6)
#define IPT_BUILD_OPTION_SET_TARGET (1 << 7)
#define IPT_BUILD_OPTION_OUT_IFACE  (1 << 8)
#define IPT_BUILD_OPTION_TABLE      (1 << 9)
#define IPT_BUILD_OPTION_MATCH      (1 << 10)

#define MIN_ALIGN (__alignof__(struct ipt_entry))
#define ALIGN(s) (((s) + ((MIN_ALIGN)-1)) & ~((MIN_ALIGN)-1))

#define XT_OPTION_OFFSET_SCALE 256

/* Ipt entries comparison macros */
#define IPT_COMPARE_SUCCESS     0
#define IPT_COMPARE_FAILURE     1

/* builtin hooks */
static const char*
builtin_hooks[] = {
    [NF_IP_PRE_ROUTING]	= "PREROUTING",
    [NF_IP_LOCAL_IN]	= "INPUT",
    [NF_IP_FORWARD]		= "FORWARD",
    [NF_IP_LOCAL_OUT]	= "OUTPUT",
    [NF_IP_POST_ROUTING]	= "POSTROUTING",
};

/* ipt standard targets */
static const char*
standard_target[] = {
    [NF_DROP]   = "NF_DROP",
    [NF_ACCEPT] = "NF_ACCEPT",
    [NF_STOLEN] = "NF_STOLEN",
    [NF_QUEUE]  = "NF_QUEUE",
    [NF_REPEAT] = "NF_REPEAT",
    [NF_STOP]   = "NF_STOP",
};

/* Is this a replacement of iptables? Does it includes everything
 * that iptables has?
 *
 * Well, the answer is big no. It just implements what is required
 * for a router i.e.,
 *
 * this version just implements
 *       -  Appending a rule to an existing chain
 *       -  Deleting a rule from the chain (ofcourse only if it is present)
 *       -  Loads the required match and target extensions..
 *
 * Currently these are the only supported rules by this component,
 * to be extended on need basis */

static struct option options[] = {
    { "append",         required_argument, NULL, 'A'},
    { "delete",         required_argument, NULL, 'D'},
    { "protocol",       required_argument, NULL, 'p'},
    { "destination",    required_argument, NULL, 'd'},
    { "in-interface",   required_argument, NULL, 'i'},
    { "jump",           required_argument, NULL, 'j'},
    { "match",          required_argument, NULL, 'm'},
    { "out-interface",  required_argument, NULL, 'o'},
    { "source",         required_argument, NULL, 's'},
    { "table",          required_argument, NULL, 't'},
    { NULL },
};

/* Needed for replacing entries and
 * compiling counters */
typedef
enum __ip_rule_opcode
{
    /* none */
    IP_RULE_OPCODE_NONE,

    /* an existing entry provided
     * by kernel */
    IP_RULE_OPCODE_EXIST,

    /* an appended entry */
    IP_RULE_OPCODE_APPEND,

    /* deleted entry */
    IP_RULE_OPCODE_DELETE
} ruleopcode;

/* internal representation of an ipt rule */
typedef
struct __ip_rule
{
    /* head offset of this rule */
    unsigned int offset;

    /* type of this rule i.e., jump/fallthrough/
     * standard/module */
    unsigned int type;

    /* an existing rule/new appended entry/
     * deleted entry  */
    ruleopcode opcode;

    /* rule entry */
    struct ipt_entry *entry;

    /* for jumps */
    void *jump;
} rule_t;

/* internal representation of an ipt chain */
typedef
struct __ip_chain
{
    /* name of the chain */
    char *name;

    /* builtin chain */
    unsigned int builtin;

    /* no. of rules in the chain */
    unsigned int num_rules;

    /* actual rules */
    GList *rules;
} chain_t;

/* internal representation of an ipt chain */
typedef
struct __ip_table
{
    /* table name */
    char *name;

    /* socket to AF_INET family */
    int sock;

    /* details of the table i.e., valid hooks,
     * no. of entries, size of the table... */
    struct ipt_getinfo *info;

    /* valid entries (ipt_entries aka ruleset) of the
     * table */
    struct ipt_get_entries *ipt_entries;

    /* no. of chains in the table */
    unsigned int num_chains;

    /* actual chains */
    GList *chains;
} table_t;

/* parser */
typedef
struct _build_ipt_entry
{
    /* count of the arguments parsed from the rule */
    int argc;

    /* argument list */
    char **argv;

    /* list of options specified in the rule i.e., append,
     * target, match.. */
    unsigned int options;

    /* table to be manipulated */
    char *table;

    /* SRC IP address specified in the rule */
    char *sourceaddressmask;

    /* DST IP address specified in the rule */
    char *destaddressmask;

    /* Chain of the table to be manipulated */
    char *chain;

    /* Protocol specified in the rule */
    char *protocol;

    /* whether the specified protocol extension
     * loaded already */
    unsigned int proto_used;

    /* real me */
    struct ipt_entry entry;

    /* target of the rule */
    struct xtables_target *target;

    /* list of the matches specified in the rule */
    GList *matches;

    /* xtables match */
    struct xtables_rule_match *xt_rule_match;
} ipt_entry_builder;

/* forward declaration */
static void
ipt_xtglobals_error (enum xtables_exittype status,
                     const char *msg, ...)
__attribute__((noreturn, format(printf,2,3)));

struct xtables_globals *xtglobals = NULL;

static const char*
ipt_xtables_exit_error2string (enum xtables_exittype status)
{
    /* Xtables errors */
    switch (status) {
    case OTHER_PROBLEM:
        return ENUMTOSTR (OTHER_PROBLEM);
    case PARAMETER_PROBLEM:
        return ENUMTOSTR (PARAMETER_PROBLEM);
    case VERSION_PROBLEM:
        return ENUMTOSTR (VERSION_PROBLEM);
    case RESOURCE_PROBLEM:
        return ENUMTOSTR (RESOURCE_PROBLEM);
    case XTF_ONLY_ONCE:
        return ENUMTOSTR (XTF_ONLY_ONCE);
    case XTF_NO_INVERT:
        return ENUMTOSTR (XTF_NO_INVERT);
    case XTF_BAD_VALUE:
        return ENUMTOSTR (XTF_BAD_VALUE);
    case XTF_ONE_ACTION:
        return ENUMTOSTR (XTF_ONE_ACTION);
    }

    return ENUMTOSTR (UNKNOWN);
}

static void
ipt_debug (int prio,
           const char *msg,
           ...)
{
    va_list args;
    char buffer [BUFSIZ];

    if (!getenv ("WAPDMAN_MAKE_IPT_VERBOSE"))
        return;

    va_start (args, msg);
    memset (buffer, 0, sizeof (buffer));
    if (vsnprintf (buffer, sizeof (buffer), msg, args) > 0)
        vdebug (prio, buffer, args);

    va_end (args);
}

static void
ipt_xtglobals_error (enum xtables_exittype status,
                     const char *msg, ...)
{
    char str[BUFSIZ];
    va_list args;

    va_start (args, msg);

    ipt_debug (DEBUG_CRITICAL, "FATAL ERROR: xtables exit: %s",
               ipt_xtables_exit_error2string (status));

    if ( vsnprintf (str, sizeof(str), msg, args) > 0) {
#ifdef WAPDMAN_DEBUG_ENABLE_SYSLOG
        vsyslog (LOG_ALERT, str, args);
#elif WAPDMAN_DEBUG_ENABLE_STDOUT
        vprintf(str, args);
        printf ("\n");
#endif
    }

    va_end (args);

    /* FATAL : something very bad has happened while using xtables (probably
     * modifying/inserting the ruleset). since xtables has made this function
     * as not returnable, it is expected to abort/exit the program here (well,
     * if you dont, you will receive a SIGSEGV and thus forced to quit ).
     *
     * Thus initiate the clean-up activities before we exit */

    /* cleanup code */

    exit (status);
}

static int
ipt_init_xtglobals()
{
    return_val_if_fail (!xtglobals, -EALREADY);

    xtglobals = g_try_malloc0 (sizeof (*xtglobals));
    return_val_if_fail (xtglobals, -ENOMEM);

    xtglobals->program_name = g_strdup (COMPONENT_NAME);
    xtglobals->program_version = g_strdup (VERSION);
    xtglobals->option_offset = 0;
    xtglobals->opts = xtglobals->orig_opts = options;
    xtglobals->exit_err = ipt_xtglobals_error;
#if XTABLES_VERSION_CODE > 10
    xtglobals->compat_rev = xtables_compatible_revision; // same approach as connman
#endif

    return 0;
}

static int
ipt_deinit_xtglobals(void)
{
    return_val_if_fail (xtglobals, -EALREADY);

    g_free ((void *) xtglobals->program_name);
    g_free ((void *) xtglobals->program_version);
    g_free (xtglobals);

    xtglobals = NULL;

    return 0;
}

static int
ipt_ismoduleloaded (const char *module)
{
    FILE *pfile;
    size_t size = MODULENAME_SIZE, count = 0;
    int ch = 0, loaded = 0, neglect = 0;
    char *buffer = NULL;

    return_val_if_fail (module, -EINVAL);

    pfile = fopen (KERNEL_MOD_PATH, O_READONLY);
    if (!pfile) {
        ipt_debug (DEBUG_ERR, "Failed to open %s, error: %d/%s",
                   KERNEL_MOD_PATH, errno,
                   strerror (errno));
        return -errno;
    }

    buffer = g_try_malloc0 (size * sizeof (*buffer));
    if (!buffer) {
        fclose (pfile);
        return -ENOMEM;
    }

    memset (buffer, 0, size);
    while ((ch = fgetc (pfile)) != EOF) {

        if (count == size) {
            char *oldbuf = buffer;
            size *= 2; count = 0;

            buffer = g_try_realloc (buffer, size * (sizeof (char)));

            if (!buffer) {
                g_free (oldbuf);
                fclose (pfile);
                return -ENOMEM;
            }
        }

        /* xxxxxx@g3gdev-xxxxxx-2:~$ cat /proc/modules
         * nls_utf8 12493 1 - Live 0x00000000
         * isofs 39555 2 - Live 0x00000000
         * ...
         *
         * wherein "nls_utf8" and "isofs" are the modules */

        if (!neglect && ch == SPACE) {

            neglect = 1;
            if ( 0 == g_strcmp0 (buffer, module) ) {
                loaded = 1;
                break;
            }

            memset (buffer, 0, size);
            count = 0;
        }
        else if ( ch == NEWLINE &&  neglect ) {
            neglect = 0;
        } else {
            if (!neglect)
                buffer[count++] = (char) ch;
        }
    }

    g_free (buffer);
    fclose (pfile);

    if (loaded)
        return 0;
    return -ENOENT;
}

static int
ipt_verifymodule (const char *module)
{
    return_val_if_fail (module, -EINVAL);

    if ( !g_strcmp0 (IPTABLES, module) ||
         !g_strcmp0 (IPTABLES_FILTER, module) ||
         !g_strcmp0 (IPTABLES_MANLGE, module) ||
         !g_strcmp0 (IPTABLES_NAT, module) ||
         !g_strcmp0 (IPTABLES_RAW, module) ||
         !g_strcmp0 (IPTABLES_SECUTIRY, module) )
        return 0;

    return -EINVAL;
}

static int
ipt_loadmodule (const char *module)
{
    int ret = 0;

    return_val_if_fail (module, -EINVAL);

    ret = ipt_verifymodule (module);
    if (ret < 0) {
        ipt_debug (DEBUG_ERR, "Invalid netfilter module given, error: %d/%s",
                   -ret, strerror (-ret));
        return ret;
    }

    ret = ipt_ismoduleloaded (module);
    if (ret < 0 ) {
        ipt_debug (DEBUG_TRACE, "\"%s\" module is not yet loaded, error: %d/%s",
                   module, -ret, strerror (-ret));
        if ( -ENOENT == ret ) {
            ret = xtables_insmod (module, NULL, TRUE);
            if (!ret) {
                ipt_debug (DEBUG_INFO, "Successfully inserted the module: \"%s\"",
                           module);
                return 0;
            }
        }
        /* xtables_insmod returns -1 on failure*/
        return ret;
    }

    return 0;
}

static int
ipt_get_table_entries (table_t *table)
{
    int err;
    socklen_t len;

    return_val_if_fail (table && table->info, -EINVAL);

    ipt_debug (DEBUG_TRACE, "Dump the table [%s (%p)] from kernel",
               table->name, table);

    len = (socklen_t) (sizeof (struct ipt_get_entries) + table->info->size);

    table->ipt_entries = g_try_malloc0 (len);
    return_val_if_fail (table->ipt_entries, -ENOMEM);

    g_stpcpy (table->ipt_entries->name, table->name);
    table->ipt_entries->size = table->info->size;

    err = getsockopt (table->sock, IPPROTO_IP, IPT_SO_GET_ENTRIES,
                      table->ipt_entries, &len);
    if (err < 0) {
        g_free (table->ipt_entries);
        table->ipt_entries = NULL;
        return -errno;
    }

    return 0;
}

static table_t*
ipt_get_table (const char *module)
{
    int err;
    socklen_t len;
    table_t *table;
    char *module_name;

    ipt_debug (DEBUG_TRACE, "Construct the table: %s", module);

    return_val_if_fail (module, NULL);

    /* first "ip_tables" has to loaded to be used by others */
    err = ipt_loadmodule (IPTABLES);
    if (err < 0) {
        ipt_debug (DEBUG_ERR, "Failed to load \"%s\" module, error : %d/%s",
                   IPTABLES, -err, strerror(-err));
        return NULL;
    }

    module_name = g_strconcat ("iptable_", module, NULL);
    err = ipt_loadmodule (module_name);
    if (err < 0 ) {
        ipt_debug (DEBUG_ERR, "Failed to load the module \"%s\", error : %d/%s",
                   module, -err, strerror(-err));
        g_free (module_name);
        return NULL;
    }

    g_free (module_name);
    table = g_try_malloc0 (sizeof (*table));
    return_val_if_fail (table, NULL);

    table->info = g_try_malloc0 (sizeof (struct ipt_getinfo));
    if (!table->info) {
        ipt_debug (DEBUG_ERR, "Failed to allocate memory for ipt_getinfo");
        g_free (table);
        return NULL;
    }

    table->sock = socket (NETFILTER_FAMILY, SOCK_RAW | SOCK_CLOEXEC,
                          NETFILTER_PROTO);
    if (table->sock < 0) {
        ipt_debug (DEBUG_ERR,
                   "Failed to create a socket to \"IPPROTO_RAW\", "
                   "error: %d/%s", errno, strerror (errno));
        goto out;
    }

    ipt_debug (DEBUG_INFO, "Created socket: %d", table->sock);

    table->name = g_strdup (module);
    g_stpcpy (table->info->name, table->name);
    len = sizeof (*(table->info));

    err = getsockopt (table->sock, IPPROTO_IP, IPT_SO_GET_INFO,
                      table->info, &len);
    if (err < 0) {
        ipt_debug (DEBUG_ERR,
                   "Failed to get the table info for \"%s\", "
                   "error : %d/%s", table->name,
                   errno, strerror(errno));
        g_free (table->name);
        (void)close (table->sock);
        goto out;
    }

    err = ipt_get_table_entries (table);
    if (err < 0) {
        ipt_debug (DEBUG_ERR,
                   "Failed to get the table entries for \"%s\","
                   " error : %d/%s", table->name,
                   -err, strerror(-err));
        g_free (table->name);
        (void)close (table->sock);
        goto out;
    }

    return table;

out:
    g_free (table->info);
    g_free (table);
    return NULL;
}

static unsigned int
next_valid_hook (unsigned int *validhooks)
{
    unsigned int i = 0;

    if (*validhooks == 0)
        return NF_INET_NUMHOOKS;

    for ( ; i < NF_INET_NUMHOOKS; i++) {
        if (*validhooks & ((unsigned int) 1 << i)) {
            *validhooks ^= ((unsigned int) 1 << i);
            break;
        }
    }

    ipt_debug (DEBUG_INFO, "next valid hook: %x", i);
    return i;
}

static int
ipt_add_entry_to_chain (chain_t *chain,
                        struct ipt_entry **entry,
                        unsigned int offset)
{
    struct xt_entry_target *target;
    rule_t *rule;
    const char* verdict;
    struct xt_standard_target *starget;

    return_val_if_fail (chain && entry && *entry, -EINVAL);

    ipt_debug (DEBUG_TRACE,
               "Adding %p [head offset : %d, "
               "foot offset : %d]"
               " to chain %s [%p]",
               *entry, offset,
               (*entry)->next_offset,
               chain->name, chain);

    rule = g_try_malloc0 (sizeof (*rule));
    return_val_if_fail (rule, -ENOMEM);

    rule->entry = g_try_malloc0 ((*entry)->next_offset);
    if (!rule->entry) {
        g_free (rule);
        return -ENOMEM;
    }

    /* head offset of this rule */
    rule->offset = offset;
    rule->opcode = IP_RULE_OPCODE_EXIST;
    memcpy (rule->entry, *entry, (*entry)->next_offset);

    target = ipt_get_target (rule->entry);
    starget = (struct xt_standard_target *) target;

    ipt_debug (DEBUG_INFO, "Name of the target : %s",
               (target->u.user.name[0] == '\0') ?
                "standard" : target->u.user.name );
    ipt_debug (DEBUG_INFO, "Verdict of the target : %d",
               starget->verdict);

    if (!g_strcmp0 (target->u.user.name, IPTC_STANDARD_TARGET)) {

        if (starget->verdict < 0) {

            switch (starget->verdict) {
            case XT_RETURN:
                verdict = standard_target [NF_REPEAT];
                break;
            case -NF_ACCEPT - 1:
                verdict = standard_target [NF_ACCEPT];
                break;
            case -NF_DROP - 1:
                verdict = standard_target [NF_DROP];
                break;
            case -NF_QUEUE - 1:
                verdict = standard_target [NF_QUEUE];
                break;
            case -NF_STOP - 1:
                verdict = standard_target [NF_STOP];
                break;
            default:
                verdict = "UNKNOWN";
                break;
            }

            ipt_debug (DEBUG_INFO, "Standard Verdict: %s", verdict);
            rule->type |= IPT_ENTRY_STANDARD;
        }
        else if (starget->verdict == (int) (rule->offset +
                                            rule->entry->next_offset))
            /* fallthrough to the next ipt_entry i.e., next rule */
            rule->type |= IPT_ENTRY_FALLTHROUGH;
        else
            /* else a jump to some chain */
            rule->type |= IPT_ENTRY_JUMP;
    } else {
        /* not a standard target, a different module  */
        rule->type |= IPT_ENTRY_MODULE;
    }

    ipt_debug (DEBUG_INFO, "Type of the rule : %s",
               rule->type & IPT_ENTRY_STANDARD ? "Standard"
               : rule->type & IPT_ENTRY_FALLTHROUGH ? "Fallthrough"
               : rule->type & IPT_ENTRY_JUMP ? "JUMP"
               : rule->type & IPT_ENTRY_MODULE ? "MODULE"
               : "Unknown");

    chain->num_rules++;
    /* in the same order as presented by kernel */
    chain->rules = g_list_append (chain->rules, rule);
    return 0;
}

static int
ipt_build_new_chain (const char *name,
                     unsigned int builtin,
                     chain_t **chain)
{
    chain_t *temp;

    return_val_if_fail (name && chain, -EINVAL);
    return_val_if_fail (!*chain, -EEXIST);

    ipt_debug (DEBUG_TRACE, "Adding a new chain : %s,"
               " builtin : %s", name,
               (builtin < NF_INET_NUMHOOKS) ?
               builtin_hooks [builtin] : "no");

    temp = g_try_malloc0 (sizeof (chain_t));
    return_val_if_fail (temp, -ENOMEM);

    temp->name = g_strdup (name);
    temp->builtin = builtin;

    *chain = temp;
    ipt_debug (DEBUG_INFO, "New chain created : %s [%p] "
               "a builtin chain: %s", temp->name, temp,
               temp->builtin != 0xFFFFFFFF ? "Y" : "N");
    return 0;
}

static void
ipt_dump_rule (gpointer data,
               gpointer user_data)
{
    (void) user_data;
    rule_t *rule = data;
    struct xt_entry_target *t;
    struct xt_entry_match *match;
    unsigned int offset = sizeof (struct ipt_entry);

    return_if_fail (rule);

    if (rule->opcode == IP_RULE_OPCODE_DELETE)
        return;

    ipt_debug (DEBUG_INFO,
               "Rule: %p "
               "head offset : %u "
               "foot offset : %u",
               rule,
               rule->offset,
               rule->entry->next_offset);

    ipt_debug (DEBUG_INFO,
               "SRC IP: %u.%u.%u.%u/%u.%u.%u.%u "
               "DST IP: %u.%u.%u.%u/%u.%u.%u.%u "
               "Interface: `%s' to `%s'",
               IP_ADDRESS (ntohl(rule->entry->ip.src.s_addr)),
               IP_ADDRESS (ntohl(rule->entry->ip.smsk.s_addr)),
               IP_ADDRESS (ntohl(rule->entry->ip.dst.s_addr)),
               IP_ADDRESS (ntohl(rule->entry->ip.dmsk.s_addr)),
               rule->entry->ip.iniface,
               rule->entry->ip.outiface);

    ipt_debug (DEBUG_INFO,
               "Protocol: %u, "
               "Flags: %02X, "
               "Invflags: %02X",
               rule->entry->ip.proto,
               rule->entry->ip.flags,
               rule->entry->ip.invflags);

    for ( ; offset < rule->entry->target_offset;
          offset += match->u.match_size ) {
        match = (void *) rule->entry + offset;
        ipt_debug (DEBUG_INFO, "Match name: \"%s\"",
                   match->u.user.name);
    }

    ipt_debug (DEBUG_INFO, "Counters Info "
               "[packets: %llu bytes: %llu]",
               rule->entry->counters.pcnt,
               rule->entry->counters.bcnt);

    t = ipt_get_target (rule->entry);
    ipt_debug (DEBUG_INFO, "Target name: `%s' [%u]",
               t->u.user.name, t->u.target_size);

    /* STANDARD target */
    if ( 0 == g_strcmp0 (t->u.user.name, IPTC_STANDARD_TARGET)) {
        const unsigned char *data = t->data;
        int pos = *(const int *)data;
        if (pos < 0) {
            ipt_debug (DEBUG_INFO, "verdict = %s",
                       pos == -NF_ACCEPT-1 ? "NF_ACCEPT"
                       : pos == -NF_DROP-1 ? "NF_DROP"
                       : pos == -NF_QUEUE-1 ? "NF_QUEUE"
                       : pos == -NF_REPEAT-1 ? "RETURN"
                       : "UNKNOWN");
        } else
            ipt_debug (DEBUG_INFO, "verdict = %u", pos);
    }
    /* ERROR target */
    else if (g_strcmp0 (t->u.user.name, IPTC_ERROR_TARGET) == 0)
        ipt_debug (DEBUG_INFO, "error =`%s'", t->data);
}

static void
ipt_dump_chain (gpointer data,
                gpointer user_data)
{
    (void) user_data;
    chain_t *chain = data;

    return_if_fail (chain);

    ipt_debug (DEBUG_INFO, "*************************************************************");
    ipt_debug (DEBUG_INFO,
               "Chain Name: %s, "
               "no. of rules: %u, "
               "builtin: %s ",
               chain->name,
               chain->num_rules,
               chain->builtin != 0xFFFFFFFF ? "Y" : "N");
    ipt_debug (DEBUG_INFO, "*************************************************************");

    /* dump each and every rule in the chain */
    g_list_foreach (chain->rules, ipt_dump_rule, NULL);
}

static void
ipt_dump_table (const table_t *const table)
{
    return_if_fail (table);

    ipt_debug (DEBUG_TRACE, "******************************************************");
    ipt_debug (DEBUG_INFO,
               "Table name: `%s', "
               "valid chains : %u",
               table->info->name,
               table->num_chains);
    ipt_debug (DEBUG_TRACE, "******************************************************");

    ipt_debug (DEBUG_INFO, "Hooks: pre-routing/input/forward/output/post-routing"
               " = %d/%d/%d/%d/%d",
               table->info->hook_entry[NF_INET_PRE_ROUTING],
               table->info->hook_entry[NF_INET_LOCAL_IN],
               table->info->hook_entry[NF_INET_FORWARD],
               table->info->hook_entry[NF_INET_LOCAL_OUT],
               table->info->hook_entry[NF_INET_POST_ROUTING]);

    ipt_debug (DEBUG_INFO, "Underflows: pre-routing/input/forward/output/post-routing"
               " = %d/%d/%d/%d/%d",
               table->info->underflow[NF_INET_PRE_ROUTING],
               table->info->underflow[NF_INET_LOCAL_IN],
               table->info->underflow[NF_INET_FORWARD],
               table->info->underflow[NF_INET_LOCAL_OUT],
               table->info->underflow[NF_INET_POST_ROUTING]);

    g_list_foreach (table->chains, ipt_dump_chain, NULL);
}

static int
ipt_parse_table (table_t *table)
{
    chain_t *current_chain = NULL;
    struct ipt_entry *entry;
    struct xt_entry_target *target;
    const char *chain_name = NULL;
    unsigned int offset = 0, builtin_hook = 0,
            newchain = 0, validhooks = 0, builtin = 0xFFFFFFFF;
    int err = 0;

    return_val_if_fail (table && table->info && table->ipt_entries, -EINVAL);

    ipt_debug (DEBUG_TRACE,
               "Parsing the table: %s, "
               "valid hooks: %u, size : %u, "
               "no of entries: %u",
               table->name,
               table->info->valid_hooks,
               table->info->size,
               table->info->num_entries);

    validhooks = table->info->valid_hooks;
    builtin_hook = next_valid_hook (&validhooks);

    for ( ; offset < table->info->size;
          offset += entry->next_offset) {

        /* next entry */
        entry = (void *) table->ipt_entries->entrytable + offset;
        target = ipt_get_target(entry);
        newchain = 0; builtin = 0xFFFFFFFF;

        /* user defined chain */
        if (builtin_hook == NF_INET_NUMHOOKS &&
                g_strcmp0 (target->u.user.name, "ERROR") == 0) {
            newchain = 1;
            chain_name = (const char *) target->data;
        }

        /* new builtin chain */
        if (builtin_hook < NF_INET_NUMHOOKS &&
                offset == table->info->hook_entry[builtin_hook]) {
            newchain = 1; builtin = builtin_hook;
            ipt_debug (DEBUG_INFO, "builtin: %d, builtin_hook: %u",
                       builtin, builtin_hook);
            chain_name = builtin_hooks [builtin_hook];
        }

        if (newchain) {
            chain_t *chain = NULL;

            err = ipt_build_new_chain (chain_name, builtin, &chain);
            if (err < 0) {
                ipt_debug (DEBUG_ERR, "Failed to add a chain: %s/%d",
                           strerror (-err), -err);
                return err;
            }

            table->num_chains++;
            table->chains = g_list_append (table->chains, chain);
            current_chain = chain;
        }

        /* underflow: we have reached the end of the chain */
        if (builtin_hook < NF_INET_NUMHOOKS &&
                table->info->underflow[builtin_hook] <= offset) {
            ipt_debug (DEBUG_INFO, "Start of a new chain");
            builtin_hook = next_valid_hook (&validhooks);
        }

        if (current_chain &&
                (err = ipt_add_entry_to_chain (current_chain, &entry, offset)) < 0) {
            ipt_debug (DEBUG_ERR,
                       "Failed to add rules to the chain: %s, "
                       "error: %s/%d",
                       current_chain->name,
                       strerror (-err), -err);
            return err;
        }
    }

    return 0;
}

static void
ipt_free_rule (gpointer data)
{
    rule_t *rule = data;

    return_if_fail (rule);
    ipt_debug (DEBUG_TRACE, "Rule: %p", rule);
    g_free (rule->entry);
    g_free (rule);
}

static void
ipt_free_chain (gpointer data)
{
    chain_t *chain = data;

    return_if_fail (chain);
    ipt_debug (DEBUG_TRACE, "Chain: %s [%p]", chain->name, chain);
    g_list_free_full (chain->rules, ipt_free_rule);
    g_free (chain->name);
    g_free (chain);
}

static int
ipt_free_table (table_t *table)
{
    int ret;

    return_val_if_fail (table, -EALREADY);

    ipt_debug (DEBUG_TRACE, "Table: %s [%p] ", table->name, table);

    g_free (table->name);
    g_free (table->info);
    g_free (table->ipt_entries);
    g_list_free_full (table->chains, ipt_free_chain);

    ret = close (table->sock);
    if (ret < 0)
        ipt_debug (DEBUG_ERR,
                   "Failed to close the socket [%d], "
                   "error: %s/%d",
                   table->sock,
                   strerror (errno), errno);

    g_free (table);
    return 0;
}

static int
ipt_calculate_length (char **strv)
{
    int length;

    return_val_if_fail (strv, -EINVAL);

    for (length = 0; *strv; length++, strv++)
        ;
    return length;
}

static int
ipt_prepare_entry_builder (const char *const rule,
                           ipt_entry_builder **builder)
{
    int i, length = 0;
    char **argv;
    ipt_entry_builder *temp;

    return_val_if_fail (rule && builder, -EINVAL);
    return_val_if_fail (!*builder, -EEXIST);

    ipt_debug (DEBUG_TRACE, "Preparing ipt_builder for: %s", rule);

    temp = g_try_malloc0 (sizeof (*temp));
    return_val_if_fail (temp, -ENOMEM);

    argv = g_strsplit_set (rule, " ", -1);

    /* Normally argc shall be the number of arguments
     * including the process name itself. Thus we need
     * to add argv[0] as well!! */
    length = ipt_calculate_length (argv);
    if (length < 0) {
        g_free (temp);
        return length;
    }

    temp->argc = length + 1;
    temp->argv = g_try_malloc0 ((unsigned int) (temp->argc + 1)
                                 * sizeof(char *));
    if (!temp->argv) {
        g_free (temp);
        g_strfreev (argv);
        return -ENOMEM;
    }

    /* start building the args */
    temp->argv[0] = g_strdup ("argh");
    for (i = 1; i < temp->argc; i++)
        temp->argv[i] = argv[i-1];

    *builder = temp;
    g_free (argv);
    return 0;
}

static const char*
ipt_xtables_load_option_to_string (const enum xtables_tryload load_option)
{
    switch (load_option) {
    case XTF_DONT_LOAD:
        return "XTF_DONT_LOAD";
    case XTF_DURING_LOAD:
        return "XTF_DURING_LOAD";
    case XTF_TRY_LOAD:
        return "XTF_TRY_LOAD";
    case XTF_LOAD_MUST_SUCCEED:
        return "XTF_LOAD_MUST_SUCCEED";
    }

    return "UNKNOWN";
}

static int
ipt_load_target (const char *const target_name,
                 const enum xtables_tryload load_option,
                 ipt_entry_builder *builder)
{
    size_t size;

    return_val_if_fail (target_name && builder, -EINVAL);

    ipt_debug (DEBUG_TRACE,
               "Builder: %p "
               "loading target: %s "
               "load option: %s",
               builder, target_name,
               ipt_xtables_load_option_to_string (load_option));

    builder->target = xtables_find_target (target_name, load_option);
    if (!builder->target)
        return -EPROTOTYPE;

    size = ALIGN (sizeof (struct xt_entry_target)) + builder->target->size;
    ipt_debug (DEBUG_INFO,
               "Target found, Name: \"%s\" size : %u",
               builder->target->real_name ?
                   builder->target->real_name :
                   builder->target->name,
               size);

    builder->target->t = g_try_malloc0 (size);
    if (!builder->target->t)
        return -ENOMEM;

    builder->target->t->u.target_size = (unsigned short int) size;

    /* if the target has the real name, the real name (the real target behind
     * this target) shall be copied, else the targets will not match and kernel
     * rejects it */
    if (builder->target->real_name) {
        g_stpcpy (builder->target->t->u.user.name, builder->target->real_name);
    } else g_stpcpy (builder->target->t->u.user.name, target_name);

    builder->target->t->u.user.revision = builder->target->revision;

    /* initiate the target extension if the extension
     * supports */
    if (builder->target->init)
        builder->target->init (builder->target->t);

    if (builder->target->x6_options != NULL)
        xtglobals->opts = xtables_options_xfrm (xtglobals->orig_opts,
                                                xtglobals->opts,
                                                builder->target->x6_options,
                                                &builder->target->option_offset);
    else
        xtglobals->opts = xtables_merge_options (xtglobals->orig_opts,
                                                 xtglobals->opts,
                                                 builder->target->extra_opts,
                                                 &builder->target->option_offset);

    if (!xtglobals->opts) {
        ipt_debug (DEBUG_ERR, "Failed to extend the options for target: \"%s\"",
                   target_name);
        g_free (builder->target->t);
        return -ENOMEM;
    }

    /* successfully loaded the target and its extensions (options) */
    return 0;
}

static int
ipt_load_matches (const char *const match_name,
                  const enum xtables_tryload load_option,
                  ipt_entry_builder *builer)
{
    size_t size;
    struct xtables_match *match;

    return_val_if_fail (match_name && builer, -EINVAL);

    ipt_debug (DEBUG_TRACE,
               "Builder: %p "
               "loading match extension: \"%s\" "
               "load option: %s",
               builer, match_name,
               ipt_xtables_load_option_to_string (load_option));

    /* find and load the match given from the ruleset */
    match = xtables_find_match (match_name, load_option,
                                /* to be filled by the xtables lib*/
                                &builer->xt_rule_match);

    /* we shall stop if we dont load the match and if the load
     * type is other than XTF_TRY_LOAD (like XTF_LOAD_MUST_SUCCEED) */
    if (load_option != XTF_TRY_LOAD)
        if (!match)
            return -ENOENT;

    size = ALIGN (sizeof (struct xt_entry_match)) + match->size;

    ipt_debug (DEBUG_INFO,
               "Match Name: \"%s\" "
               "Real Name: \"%s\" "
               "Revision: \"%u\" "
               "size: \"%u\"",
               match->name,
               match->real_name,
               match->revision, size);

    /* allocate for the entry match */
    match->m = g_try_malloc0 (size);
    return_val_if_fail (match->m, -ENOMEM);

    match->m->u.match_size = (unsigned short int) size;

    /* if the target has the real name, the real name (the real target behind
     * this target) shall be copied, else the targets will not match and kernel
     * will reject it */
    if (match->real_name) {
        g_stpcpy (match->m->u.user.name, match->real_name);
    } else g_stpcpy (match->m->u.user.name, match->name);

    match->m->u.user.revision = match->revision;

    ipt_debug (DEBUG_INFO, "Initialize match extensions for: %s",
               match->m->u.user.name);
    /* init the module */
    if (match->init)
        match->init (match->m);

    /* creates new option entries (extending the current options)
     * depending on the match extension for getopt */
    if (match->x6_options)
        xtglobals->opts = xtables_options_xfrm ( xtglobals->orig_opts,
                                                 xtglobals->opts,
                                                 match->x6_options,
                                                 &match->option_offset );
    else if (match->extra_opts != NULL)
        xtglobals->opts = xtables_merge_options ( xtglobals->orig_opts,
                                                  xtglobals->opts,
                                                  match->extra_opts,
                                                  &match->option_offset );

    if (!xtglobals->opts) {
        ipt_debug (DEBUG_ERR, "Failed to extend the options set for the "
                   "match extension: %s", match->name);
        g_free (match->m);
        return -ENOMEM;
    }

    /* successfully loaded the match and its extensions */
    builer->matches = g_list_append (builer->matches, match);
    return 0;
}

static void
ipt_cleanup_xtables_matches_target (void)
{
    struct xtables_match *xt_m;
    struct xtables_target *xt_t;

    ipt_debug (DEBUG_TRACE, "");

    for (xt_m = xtables_matches; xt_m; xt_m = xt_m->next) {
        ipt_debug (DEBUG_INFO, "Global Match: \"%s\"",
                   xt_m->name);
        xt_m->mflags = 0;
    }

    for (xt_t = xtables_targets; xt_t; xt_t = xt_t->next) {
        ipt_debug (DEBUG_INFO, "Global Target: \"%s\"",
                   xt_t->name);
        xt_t->tflags = 0;
        xt_t->used = 0;
    }

    /* We need also to free the memory implicitly allocated
     * during parsing (see xtables_options_xfrm()).
     * Note xt_params is actually iptables_globals. */
    if (xt_params->opts != xt_params->orig_opts) {
        ipt_debug (DEBUG_INFO, "Freeing up the extended options "
                   "as part of loading match extensions");
        g_free (xt_params->opts);
        xt_params->opts = xt_params->orig_opts;
    }

    xt_params->option_offset = 0;
    ipt_debug (DEBUG_TRACE, "Cleaned successfully");
}

static int
ipt_load_defaults_proto (ipt_entry_builder *builder)
{
    int err;

    return_val_if_fail (builder && builder->protocol, -EINVAL);

    ipt_debug (DEBUG_TRACE, "Protocol: %s Builder: %p",
               builder->protocol, builder);

    /* we would only have builder->protocol filled if it is
     * a valid protocol */

    /* if there is an implicit request like "--dport" which needs the
     * protocol "tcp" match extension to be loaded. Thus we need to
     * - check if the match is already loaded
     * - if not loaded, load the match */

    /* check if the match is already loaded */
    if (xtables_find_match (builder->protocol, XTF_DONT_LOAD, NULL) != NULL) {

        ipt_debug (DEBUG_INFO, "Match already loaded for: %s",
                   builder->protocol);
        /* check if it is being used i.e., we shall not load it again
         * if it is already loaded */
        if (builder->proto_used) {
            ipt_debug (DEBUG_INFO, "Match already used for: %s",
                       builder->protocol);
            return -EALREADY;
        }
    }

    err = ipt_load_matches (builder->protocol, XTF_TRY_LOAD, builder);
    if (!err)
        builder->proto_used = 1;

    return err;
}

static void
ipt_dump_options (const struct option *const options)
{
    int i = 0;

    return_if_fail (options);

    for ( ; options[i].name; i++ )
        ipt_debug (DEBUG_INFO,
                   "options [%d]: name[\"%s\"] value[%d]",
                   i, options[i].name, options[i].val);
}

static int
ipt_load_defaults (int ret,
                   int invert,
                   ipt_entry_builder *builder)
{
    struct xtables_rule_match *rm;
    struct xtables_match *m;

    ipt_debug (DEBUG_TRACE, "Ret: %d Invert: %d Builder: %p",
               ret, invert, builder);

    /* all protocols */
    if (!builder->entry.ip.proto)
        builder->entry.ip.proto = IPPROTO_IP;

    /* we are just supplying the additional options of the
     * target to the target parser */
    if ( builder->target != NULL &&
         /* atleast one parser the target shall have */
         (builder->target->parse != NULL ||
          builder->target->x6_parse != NULL) &&
         /* option shall be within the target offset */
         (unsigned int) ret >= builder->target->option_offset &&
         (unsigned int) ret < builder->target->option_offset
         + XT_OPTION_OFFSET_SCALE) {

        ipt_debug (DEBUG_INFO, "Dispatching arguments to the appropriate "
                   "parse function of target extension: \"%s\"",
                   builder->target->name);
        /* dispatch it to the parser of the target */
        xtables_option_tpcall ((unsigned int) ret,
                               builder->argv, invert,
                               builder->target,
                               &builder->entry);

        return 0;
    }

    rm = builder->xt_rule_match;
    for ( ; rm; rm = rm->next) {
        if (rm->completed != 0)
            continue;

        m = rm->match;

        /* additional match shall have the parser  */
        if (!m->x6_parse && !m->parse)
            continue;

        /* option shall be within the target offset */
        if (ret < (int) m->option_offset ||
                ret >= (int) m->option_offset + XT_OPTION_OFFSET_SCALE)
            continue;

        ipt_debug (DEBUG_INFO, "Dispatching arguments to the appropriate "
                   "parse function of match extension: \"%s\"", m->name);
        /* dispatch it to the parser of the match */
        xtables_option_mpcall ((unsigned int) ret,
                               builder->argv, invert,
                               m, &builder->entry);
        return 0;
    }

    /* Try loading protocol */
    if (!ipt_load_defaults_proto (builder)) {
        ipt_debug (DEBUG_INFO,
                   "Match loaded successfully for the "
                   "protocol: \"%s\" "
                   "optind : %d",
                   builder->protocol, optind);
        /* just to debug the extended options */
        ipt_dump_options (xtglobals->opts);

        /* once you have the match loaded which might have introduced/added
         * new options to the option list, thus recall the getopt_long again
         * by redirecting it to again parse the option which was unknown till
         * now will have an answer from the option set */
        optind--;
        return 0;
    }

    /* rule has invalid long options which is not yet supported.
     * Well, ofcourse free to extend it based on the requirement :) */
    if (ret == '?' || ret == ':') {
        ipt_debug (DEBUG_ERR, "Invalid option provided (in the rule)");
        return -EINVAL;
    }

    return -ECANCELED;
}

static int
ipt_parse_protocol (const char *const protocol,
                    unsigned int *protono,
                    ipt_entry_builder *builder)
{
    const struct protoent *pent;
    unsigned int proto;

    return_val_if_fail (protocol && builder && protono, -EINVAL);

    ipt_debug (DEBUG_TRACE, "Protocol: \"%s\" Builder: %p",
               protocol, builder);

    /* xtables version of strtoui */
    if (xtables_strtoui (protocol, NULL, &proto, 0, UINT8_MAX)) {
        ipt_debug (DEBUG_INFO, "protocol numeric identified as: %u",
                   proto);
        *protono = proto;
        return 0;
    }

    /* special case if the protocol is specified as "all" */
    if (g_strcmp0 (protocol, "all") == 0) {
        ipt_debug (DEBUG_INFO, "protocol numeric identified as: %d", 0);
        *protono = 0;
        return 0;
    }

    /* try to get the protocol from the protocol db
     * i.e., /etc/protocols */
    pent = getprotobyname (protocol);
    return_val_if_fail (pent, -ENOENT);

    /* we have the valid entry in the protocol db */
    *protono = (unsigned int) pent->p_proto;

    ipt_debug (DEBUG_INFO, "protocol numeric identified as: %u",
               *protono);
    return 0;
}

static int
ipt_parse_source_destination_addresses (ipt_entry_builder *builder)
{
    unsigned int i, j;
    unsigned int nsaddrs = 0, ndaddrs = 0;
    struct in_addr *saddrs = NULL, *smasks = NULL;
    struct in_addr *daddrs = NULL, *dmasks = NULL;

    return_val_if_fail (builder, -EINVAL);

    ipt_debug (DEBUG_TRACE,
               "Builder [%p] "
               "Source: \"%s\" "
               "Destination: \"%s\"",
               builder,
               builder->sourceaddressmask,
               builder->destaddressmask);

    if (builder->sourceaddressmask)
        /* well, you know that there is a possibility of having multiple
         * ip addresses (source) in a given rule, xtables helper to parse them */
        xtables_ipparse_multiple (builder->sourceaddressmask, &saddrs,
                                  &smasks, &nsaddrs);

    if (builder->destaddressmask)
        /* well, you know that there is a possibility of having multiple
         * ip addresses (destination) in a given rule, xtables helper to parse them */
        xtables_ipparse_multiple (builder->destaddressmask, &daddrs,
                                  &dmasks, &ndaddrs);

    ipt_debug (DEBUG_INFO, "nsaddrs: %u ndaddrs: %u", nsaddrs, ndaddrs);

    for (i = 0; i < nsaddrs ; i++)
        ipt_debug (DEBUG_INFO,
                   "Source IP address [%d]: %u.%u.%u.%u/%u.%u.%u.%u",
                   i, IP_ADDRESS (ntohl ((saddrs)[i].s_addr)),
                   IP_ADDRESS (ntohl ((smasks)[i].s_addr)));

    for (i = 0; i < ndaddrs ; i++)
        ipt_debug (DEBUG_INFO,
                   "Destination IP address [%d]: %u.%u.%u.%u/%u.%u.%u.%u",
                   i, IP_ADDRESS (ntohl ((daddrs[i]).s_addr)),
                   IP_ADDRESS (ntohl ((dmasks)[i].s_addr)));

    if ((nsaddrs > 1 || ndaddrs > 1) &&
            (builder->entry.ip.invflags & (IPT_INV_SRCIP | IPT_INV_DSTIP))) {
        ipt_debug (DEBUG_ERR, "Invert flag has been set for multiple "
                   "addresses which is not allowed!!");
        return -EINVAL;
    }

    /* copy the SRC and DST IP addresses to the entry, we here copy the
     * last address */
    for (i = 0; i < nsaddrs; i++) {
        builder->entry.ip.src.s_addr = saddrs[i].s_addr;
        builder->entry.ip.smsk.s_addr = smasks[i].s_addr;
        for (j = 0; j < ndaddrs; j++) {
            builder->entry.ip.dst.s_addr = daddrs[j].s_addr;
            builder->entry.ip.dmsk.s_addr = dmasks[j].s_addr;
        }
    }

    ipt_debug (DEBUG_INFO,
               "SRC IP %u.%u.%u.%u/%u.%u.%u.%u "
               "DST IP %u.%u.%u.%u/%u.%u.%u.%u ",
               IP_ADDRESS (ntohl (builder->entry.ip.src.s_addr)),
               IP_ADDRESS (ntohl (builder->entry.ip.smsk.s_addr)),
               IP_ADDRESS (ntohl (builder->entry.ip.dst.s_addr)),
               IP_ADDRESS (ntohl (builder->entry.ip.dmsk.s_addr)));

    free (saddrs);
    free (smasks);
    free (daddrs);
    free (dmasks);

    return 0;
}

static void
ipt_clean_xtable_matches (ipt_entry_builder *builder)
{
    struct xtables_rule_match *matchp, *tmp;

    return_if_fail (builder);

    ipt_debug (DEBUG_TRACE, "Builder [%p] xtables_rule_match [%p]", builder,
           builder->xt_rule_match);

    for (matchp = builder->xt_rule_match; matchp;) {

        tmp = matchp->next;

        /* free the xt_entry_match */
        if (matchp->match->m) {
            ipt_debug (DEBUG_INFO, "Freeing up xt_entry_match: %s [%p]",
                       matchp->match->m->u.user.name, matchp->match->m);
            g_free (matchp->match->m);
            matchp->match->m = NULL;
        }

        /* free the xtables_match */
        if (matchp->match == matchp->match->next) {
            ipt_debug (DEBUG_INFO, "Freeing up xtables_match: %s [%p]",
                       matchp->match->real_name ?
                           matchp->match->real_name : matchp->match->name,
                       matchp->match);
            free (matchp->match);
            matchp->match = NULL;
        }

        ipt_debug (DEBUG_INFO, "Freeing up xtables_rule_match [%p]", matchp);
        free (matchp);
        matchp = tmp;
    }

    builder->xt_rule_match = NULL;
}

static void
ipt_cleanup_builder (ipt_entry_builder *builder)
{
    int i = 0;

    ipt_debug (DEBUG_TRACE, "Builder: %p", builder);

    if (builder->options & IPT_BUILD_OPTION_TABLE) {
        ipt_debug (DEBUG_INFO, "Freeing the table entry: %s[%p]",
                   builder->table, builder->table);
        g_free (builder->table);
        builder->table = NULL;
    }

    if (builder->options & IPT_BUILD_OPTION_APPEND ||
            builder->options & IPT_BUILD_OPTION_DELETE) {
        ipt_debug (DEBUG_INFO, "Freeing the chain entry: %s[%p]",
                   builder->chain, builder->chain);
        g_free (builder->chain);
        builder->chain = NULL;
    }

    if (builder->options & IPT_BUILD_OPTION_SET_PROTO) {
        ipt_debug (DEBUG_INFO, "Freeing the protocol entry: %s[%p]",
                   builder->protocol, builder->protocol);
        g_free (builder->protocol);
        builder->protocol = NULL;
    }

    if (builder->options & IPT_BUILD_OPTION_SRC_ADDR) {
        ipt_debug (DEBUG_INFO, "Freeing the source address entry: %s[%p]",
                   builder->sourceaddressmask,
                   builder->sourceaddressmask);
        g_free (builder->sourceaddressmask);
        builder->sourceaddressmask = NULL;
    }

    if (builder->options & IPT_BUILD_OPTION_DST_ADDR) {
        ipt_debug (DEBUG_INFO, "Freeing the destination address entry %s[%p]",
              builder->destaddressmask,
              builder->destaddressmask);
        g_free (builder->destaddressmask);
        builder->destaddressmask = NULL;
    }

    while (builder->argc) {
        g_free (builder->argv[i]);
        i++; builder->argc--;
    }

    ipt_clean_xtable_matches (builder);
    ipt_cleanup_xtables_matches_target ();
    g_free (builder);
}

static int
ipt_generate_rule (rule_t **rule,
                   const ipt_entry_builder *const builder)
{
    size_t size;
    struct xtables_rule_match *rm;
    struct ipt_entry *entry;
    
    return_val_if_fail (builder && rule, -EINVAL);
    return_val_if_fail (!*rule, -EEXIST);

    ipt_debug (DEBUG_TRACE, "Builder: %p", builder);

    /* compute the size of the entry to be allocated */
    size = sizeof (struct ipt_entry);

    for (rm = builder->xt_rule_match; rm; rm = rm->next) {
        ipt_debug (DEBUG_INFO,
                   "Size of the match \"%s\": %u",
                   rm->match->real_name ?
                       rm->match->real_name : rm->match->name,
                   rm->match->m->u.match_size);
        size += rm->match->m->u.match_size;
    }

    /* at least we will have the standard target */
    ipt_debug (DEBUG_INFO, "Total size of the target \"%s\": [%d]",
               builder->target->real_name ?
                   builder->target->real_name : builder->target->name,
               builder->target->t->u.target_size);
    entry = g_try_malloc0 (size + builder->target->t->u.target_size);
    return_val_if_fail (entry, -ENOMEM);

    *entry = builder->entry;
    entry->target_offset = (unsigned short int) size;
    entry->next_offset = (unsigned short int) (size + builder->target->t->u.target_size);

    size = 0;
    for (rm = builder->xt_rule_match; rm; rm = rm->next) {
        ipt_debug (DEBUG_INFO, "Copying match: \"%s\"", rm->match->m->u.user.name);
        memcpy (entry->elems + size, rm->match->m,
                rm->match->m->u.match_size);
        size += rm->match->m->u.match_size;
    }

    ipt_debug (DEBUG_INFO, "Copying target: \"%s\"", builder->target->t->u.user.name);
    memcpy (entry->elems + size, builder->target->t, builder->target->t->u.target_size);

    *rule = g_try_malloc0 (sizeof (rule_t));
    if (!*rule) {
        ipt_debug (DEBUG_ERR, "Failed to allocate memory for the rule");
        g_free (entry);
        return -ENOMEM;
    }

    (*rule)->entry = entry;
    return 0;
}

static int
ipt_build_entry_from_rule (const char *const rule,
                           ipt_entry_builder *builder)
{
    struct xtables_rule_match *rm;
    size_t size, mask;
    unsigned int cleanup = 0;
    int ret, err, invert = 0;

    return_val_if_fail (builder, -EINVAL);

    ipt_debug (DEBUG_TRACE, "Supplied Rule: \"%s\"", rule);

    /* dont write to stderr on error, we will
     * writing in that case */
    opterr = 0;
    /* iterate from the begining i.e., whenever the function
     * gets called, it will start to iterate from the first
     * argument */
    optind = 0;

    while ((ret = getopt_long (builder->argc, builder->argv,
                               "-:A:D:o:p:s:m:d:j:i:t:",
                               xtglobals->opts, NULL)) != -1) {
        ipt_debug (DEBUG_TRACE, "Supplied option: %d argument: %s", ret, optarg);
        switch (ret) {
        case 't':
        {
            /* table to be manipulated */
            ipt_debug (DEBUG_TRACE, "Optind: %d Table: %s", optind, optarg);
            builder->table = g_strdup (optarg);
            builder->options |= IPT_BUILD_OPTION_TABLE;
        }
        break;

        case 'A':
        {
            /* Append entry */
            if (builder->chain) {
                ipt_debug (DEBUG_ERR, "Chain entry is already there %s, "
                           "anyway updating it to %s",
                           builder->chain, optarg);
                g_free (builder->chain);
                builder->chain = NULL;
            }
            builder->chain  = g_strdup (optarg);
            ipt_debug (DEBUG_TRACE, "optind: %d append entry to the chain: %s",
                       optind, optarg);
            builder->options |= IPT_BUILD_OPTION_APPEND;
        }
        break;

        case 'D':
        {
            /* Delete entry */
            if (builder->chain) {
                ipt_debug (DEBUG_ERR, "chain entry is already there %s, "
                           "anyway updating it to %s",
                           builder->chain, optarg);
                g_free (builder->chain);
                builder->chain = NULL;
            }
            builder->chain  = g_strdup (optarg);
            ipt_debug (DEBUG_TRACE, "optind:%d, delete entry from the chain: %s",
                       optind, optarg);
            builder->options |= IPT_BUILD_OPTION_DELETE;
        }
        break;

        case 'o':
        {
            /* outgoing interface */
            mask = 0; size = 0;
            size = strlen (optarg);

            if (*optarg == '\0' || size + 1 > IFNAMSIZ) {
                ipt_debug (DEBUG_ERR, "Invalid interface supplied as part "
                           "of outgoing interface switch");
                cleanup = 1;
                break;
            }

            ipt_debug (DEBUG_TRACE, "optind:%d, Outgoing interface: %s",
                       optind, optarg);
            g_stpcpy (builder->entry.ip.outiface, optarg);

            /* entry.ip.outiface_mask tells match key i.e., whether to match
             * complete interface name given or just few chars of it. An example
             * would be to specify "wlan+" option and that would include all the
             * interfaces start with "wlan" under radar */

            (optarg[size - 1] == '+' /* wildcard match */) ?
                        (mask = size - 1) : (mask = size + 1);
            memset (builder->entry.ip.outiface_mask, 0xFF, mask);

            if (invert) {
                ipt_debug (DEBUG_TRACE, "Setting the invert flag for incoming interface");
                builder->entry.ip.invflags |= IPT_INV_VIA_OUT;
            }

            builder->options |= IPT_BUILD_OPTION_OUT_IFACE;
        }
        break;

        case 'i':
        {
            /* incoming interface */
            mask = 0; size = 0;
            size = strlen (optarg);

            if (*optarg == '\0' || size + 1 > IFNAMSIZ) {
                ipt_debug (DEBUG_ERR, "Invalid interface supplied as part "
                           "of outgoing interface switch");
                cleanup = 1;
                break;
            }

            ipt_debug (DEBUG_INFO, "optind: %d, Incoming interface: %s",
                       optind, optarg);
            g_stpcpy (builder->entry.ip.iniface, optarg);

            /* entry.ip.iniface_mask tells match key i.e., whether to match
             * complete interface name given or just few chars of it. An example
             * would be to specify "wlan+" option and that would include all the
             * interfaces start with "wlan" under radar */

            (optarg[size - 1] == '+' /* wildcard match */) ?
                        (mask = size - 1) : (mask = size + 1);
            memset (builder->entry.ip.iniface_mask, 0xFF, mask);

            if (invert) {
                ipt_debug (DEBUG_TRACE, "Setting the invert flag for incoming interface");
                builder->entry.ip.invflags |= IPT_INV_VIA_IN;
            }

            builder->options |= IPT_BUILD_OPTION_IN_IFACE;
        }
        break;

        case 'p':
        {
            /* protocol */
            unsigned int proto;

            ipt_debug (DEBUG_TRACE, "Optind:%d, Specified protocol: %s", optind, optarg);
            if (ipt_parse_protocol (optarg, &proto, builder) < 0) {
                ipt_debug (DEBUG_ERR, "Invalid protocol specified: %s",
                           optarg);
                cleanup = 1;
                break;
            }

            builder->entry.ip.proto = (unsigned short int) proto;

            if (invert) {
                ipt_debug (DEBUG_TRACE, "Setting invert flag to the protocol");
                builder->entry.ip.invflags |= IPT_INV_PROTO;
            }

            if (builder->entry.ip.proto == 0 /* all protocols */ &&
                    builder->entry.ip.invflags & IPT_INV_PROTO) {
                ipt_debug (DEBUG_ERR, "Invalid rule, because rule would not "
                           "match protocol");
                cleanup = 1;
                break;
            }

            builder->protocol = g_strdup (optarg);
            builder->options |= IPT_BUILD_OPTION_SET_PROTO;
        }
        break;

        case 's':
        {
            /* source address */
            ipt_debug (DEBUG_TRACE, "optind:%d, Source address: %s", optind, optarg);
            builder->sourceaddressmask = g_strdup (optarg);

            if (invert) {
                ipt_debug (DEBUG_TRACE, "Setting the invert flag for source address");
                builder->entry.ip.invflags |= IPT_INV_SRCIP;
            }

            builder->options |= IPT_BUILD_OPTION_SRC_ADDR;
        }
        break;

        case 'm':
        {
            /* match extension */
            int err;

            ipt_debug (DEBUG_TRACE, "optind: %d, Match extension: %s",
                       optind, optarg);

            if (invert) {
                ipt_debug (DEBUG_ERR, "Invalid ! flag before --match");
                cleanup = 1;
                break;
            }

            err = ipt_load_matches (optarg, XTF_LOAD_MUST_SUCCEED, builder);
            if (err < 0) {
                ipt_debug (DEBUG_ERR, "Failed to load and init the provided matches: %s/%d",
                           strerror (-err), -err);
                cleanup = 1;
                break;
            }

            ipt_debug (DEBUG_TRACE, "Matches loaded successfully for the match: %s", optarg);
            ipt_dump_options (xtglobals->opts);
            builder->options |= IPT_BUILD_OPTION_MATCH;
        }
        break;

        case 'd':
        {
            /* destination address */
            ipt_debug (DEBUG_TRACE, "optind: %d, Destination address: %s", optind, optarg);
            builder->destaddressmask = g_strdup (optarg);

            if (invert) {
                ipt_debug (DEBUG_TRACE, "Setting the invert flag for destination address");
                builder->entry.ip.invflags |= IPT_INV_DSTIP;
            }

            builder->options |= IPT_BUILD_OPTION_DST_ADDR;
        }
        break;

        case 'j':
        {
            /* prepare jump aka target */
            int err;

            ipt_debug (DEBUG_TRACE, "optind: %d, jump/target: %s", optind, optarg);
            if (invert) {
                ipt_debug (DEBUG_ERR, "Invalid ! flag before -j (target)!!");
                cleanup = 1;
                break;
            }

            err = ipt_load_target (optarg, XTF_TRY_LOAD, builder);
            if (err < 0) {

                ipt_debug (DEBUG_ERR, "Failed to load and init the target %s [%s/%d]",
                           optarg, strerror (-err), -err);
                /* try to load the standard target */
                err = ipt_load_target (XT_STANDARD_TARGET, XTF_LOAD_MUST_SUCCEED, builder);
                if (err < 0) {
                    ipt_debug (DEBUG_ERR, "Failed to load and init the standard target: %s/%d",
                               strerror (-err), -err);
                    cleanup = 1;
                    break;
                }
                ipt_debug (DEBUG_TRACE, "Successfully loaded the standard target");
            }

            ipt_debug (DEBUG_TRACE, "Target loaded successfully for the target: %s", optarg);
            ipt_dump_options (xtglobals->opts);
            builder->options |= IPT_BUILD_OPTION_SET_TARGET;
        }
        break;

        case 1:
        {
            ipt_debug (DEBUG_TRACE, "Ret: %d optind: %d optarg: %s",
                       ret, optind, optarg);
            if (optarg[0] == '!' && optarg [1] == '\0') {

                ipt_debug (DEBUG_TRACE, "invert flag set!!");
                if (invert) {
                    ipt_debug (DEBUG_ERR, "There is already an invert flag set, "
                               "multiple invert flags seen!!");
                    /* clean up the built rule and return error */
                    cleanup = 1;
                    break;
                }
                /* set the invert flag */
                invert = 1;
                /* recall getopt_long */
                optarg[0] = '\0';
                continue;
            } else {
                /* invalid option given exit rule parser?*/
            }
        }
        break;

        default:
        {
            /* the rest of the arguments including the extended options
             * depending on the matches loaded */

            ipt_debug (DEBUG_INFO, "Default Ret: %d optind: %d optarg: %s",
                       ret, optind, optarg);
            if ((err = ipt_load_defaults (ret, invert, builder)) == 0) {
                ipt_debug (DEBUG_INFO, "defaults loaded successfully, continue to "
                           "check the other options");
                continue;
            } else {
                /* something has gone wrong while loading the extensions or
                 * dispatching the arguments, thus cleanup the builder */
                cleanup = 1;
            }
            break;
        }
        break;
        }

        invert = 0;
        if (cleanup)
            /* something gone wrong break outta of while to
             * cleanup builder*/
            break;
    }

    if (!cleanup) {

        for (rm = builder->xt_rule_match; rm; rm = rm->next) {
            ipt_debug (DEBUG_INFO, "Dispatching arguments to the final check function "
                       "of the match extension: %s", rm->match->name);
            xtables_option_mfcall (rm->match);
        }

        if (builder->target) {
            ipt_debug (DEBUG_INFO, "Dispatching arguments to the final check function of the "
                       "target extension: %s", builder->target->name);
            xtables_option_tfcall (builder->target);
        }

        /* If the rule is to append/delete an ipt_entry, we shall add the src
         * and dst IP addresses even if they aren'e specified */
        if ((builder->options & IPT_BUILD_OPTION_APPEND ||
             builder->options & IPT_BUILD_OPTION_DELETE)) {

            if (!builder->sourceaddressmask) {
                builder->sourceaddressmask = g_strdup ("0.0.0.0/0");
                builder->options |= IPT_BUILD_OPTION_SRC_ADDR;
            }

            if (!builder->destaddressmask) {
                builder->destaddressmask = g_strdup ("0.0.0.0/0");
                builder->options |= IPT_BUILD_OPTION_DST_ADDR;
            }

            /* parse the source and destination addresses */
            if ((err = ipt_parse_source_destination_addresses (builder)) < 0) {
                ipt_debug (DEBUG_ERR, "Failed to parse the source and destination addresses: %s/%d",
                           strerror(-err), -err);
                cleanup = 1;
            }
        }
    }

    if (cleanup) {
        ipt_cleanup_builder (builder);
        return -EINVAL;
    }

    return 0;
}

static int
ipt_is_builtin_chain (const char *chain_name)
{
    int i = 0;

    return_val_if_fail (chain_name, -EINVAL);

    ipt_debug (DEBUG_TRACE, "Is \"%s\" a builtin chain?", chain_name);

    /*  Below are the only builtin chains
     *   - PREROUTING
     *   - INPUT
     *   - FORWARD
     *   - OUTPUT
     *   - POSTROUTING */

    for ( ; i < NF_INET_NUMHOOKS; i++)
        if (!g_strcmp0 (chain_name, builtin_hooks[i]))
            return 0;

    return -EINVAL;
}

static int
ipt_is_builtin_target (const char *target_name)
{
    return_val_if_fail (target_name, -EINVAL);

    ipt_debug (DEBUG_TRACE, "Is \"%s\" a builtin target?", target_name);

    /* builtin target only if any of the below
     *      - ACCEPT
     *      - DROP
     *      - QUEUE
     *      - RETURN
     *      - STOLEN
     *      - STOP
     */
    if ( !g_strcmp0 (target_name, LABEL_ACCEPT) ||
         !g_strcmp0 (target_name, LABEL_DROP) ||
         !g_strcmp0 (target_name, LABEL_QUEUE) ||
         !g_strcmp0 (target_name, LABEL_RETURN) ||
         !g_strcmp0 (target_name, LABEL_STOLEN) ||
         !g_strcmp0 (target_name, LABEL_STOP))
        return 0;

    return -EINVAL;
}

static int
ipt_target_to_verdict (const char *const target_name,
                       int *verdict)
{
    return_val_if_fail (target_name, -EINVAL);

    ipt_debug (DEBUG_TRACE, "Target name: %s", target_name);

    if (!g_strcmp0(target_name, LABEL_ACCEPT))
        *verdict = -NF_ACCEPT - 1;
    else if (!g_strcmp0(target_name, LABEL_DROP))
        *verdict = -NF_DROP - 1;
    else if (!g_strcmp0(target_name, LABEL_QUEUE))
        *verdict = -NF_QUEUE - 1;
    else if (!g_strcmp0(target_name, LABEL_RETURN))
        *verdict = XT_RETURN;
    else
        return -EINVAL;

    return 0;
}

static chain_t*
ipt_get_chain (const table_t *const table,
               const char *const chain_name)
{
    chain_t *chain = NULL;
    GList *list = NULL;

    return_val_if_fail (table && chain_name, NULL);

    ipt_debug (DEBUG_TRACE, "Table: \"%s\" Chain: \"%s\"",
               table->name, chain_name);
    list = table->chains;
    for ( ; list; list = list->next) {
        chain = list->data;
        if (chain && !g_strcmp0 (chain->name, chain_name))
            return chain;
    }

    return NULL;
}

static int
ipt_prepare_target_verdict (const table_t *const table,
                            const chain_t *const chain,
                            rule_t *rule)
{
    GList *rules;
    struct xt_entry_target *etarget;
    struct xt_standard_target *sttarget;
    int verdict, err;

    return_val_if_fail (table && chain && rule, -EINVAL);

    if (!rule->entry) {
        ipt_debug (DEBUG_ERR, "Rule [%p] does not have the "
                   "ipt_entry constructed", rule);
        return -EINVAL;
    }

    ipt_debug (DEBUG_TRACE, "Table: \"%s\" Rule: %p", table->name, rule);

    etarget = ipt_get_target (rule->entry);

    ipt_debug (DEBUG_INFO, "Target name: \"%s\"", etarget->u.user.name);
    if ( !ipt_is_builtin_chain (etarget->u.user.name)) {
        ipt_debug (DEBUG_ERR, "jumping/target to a builtin chain is not allowed");
        return -EINVAL;
    }

    /* no target specified, thus a fallthrough rule */
    if (!g_strcmp0 (etarget->u.user.name, "")) {
        rule->type |= IPT_ENTRY_FALLTHROUGH;
        return 0;
    }

    sttarget = (struct xt_standard_target *) etarget;
    if (!ipt_is_builtin_target (etarget->u.user.name)) {

        ipt_debug (DEBUG_INFO,
                   "Target size: %u, "
                   "Standard target size : %u",
                   sttarget->target.u.target_size,
                   ALIGN (sizeof(struct xt_standard_target)));

        /* If it is a standard target it has to match the size of
         * the standard target */
        if (sttarget->target.u.target_size !=
                ALIGN (sizeof(struct xt_standard_target))) {
            ipt_debug (DEBUG_ERR, "Invalid match between the standard target "
                       "and the target size");
            return -EINVAL;
        }

        err = ipt_target_to_verdict (etarget->u.user.name, &verdict);
        if (err < 0) {
            ipt_debug (DEBUG_ERR, "Failed to get the verdict for standard target: %s/%d",
                       strerror (-err), -err);
            return err;
        }

        memset (sttarget->target.u.user.name, 0, sizeof (sttarget->target.u.user.name));
        g_stpcpy (sttarget->target.u.user.name, IPTC_STANDARD_TARGET);

        sttarget->verdict = verdict;
        rule->type |= IPT_ENTRY_STANDARD;

        return 0;
    } else {
        /* It could be a jump to any of the user defined chain */
        chain_t *chain = ipt_get_chain (table, etarget->u.user.name);

        if (chain) {
            ipt_debug (DEBUG_INFO, "Found user defined chain for the target: %s [%p]",
                       chain->name, chain);

            rules = g_list_first (chain->rules);
            return_val_if_fail (rules, -ECANCELED);
            rule = rules->data;
            /* start of first entry of the jump chain */
            sttarget->verdict = (int) rule->offset;
            rule->type |= IPT_ENTRY_JUMP;

            return 0;
        }
    }

    /* could be a extension target */
    memset (etarget->u.user.name + strlen(etarget->u.user.name), 0,
            XT_FUNCTION_MAXNAMELEN - 1 - strlen(etarget->u.user.name));
    rule->type = IPT_ENTRY_MODULE;
    return 0;
}

static int
ipt_compare_targets (const rule_t *const r1,
                     const rule_t *const r2)
{
    unsigned int i;
    struct xt_entry_target *t1, *t2;

    return_val_if_fail (r1 && r2, IPT_COMPARE_FAILURE);

    ipt_debug (DEBUG_TRACE, "Comparing targets for r1 [%p] r2 [%p]", r1, r2);

    if (r1->type != r2->type)
        return IPT_COMPARE_FAILURE;

    t1 = ipt_get_target (r1->entry);
    t2 = ipt_get_target (r2->entry);

    ipt_debug (DEBUG_INFO, "r1 target type: %u r2 target type: %u",
               r1->type, r2->type);
    ipt_debug (DEBUG_INFO, "Comparing targets t1: %s  and t2: %s",
               t1->u.user.name, t2->u.user.name);

    switch (r1->type) {
    case IPT_ENTRY_FALLTHROUGH:
        ipt_debug (DEBUG_INFO, "Fallthrough target");
        return IPT_COMPARE_SUCCESS;
        /* verdict shall match for the standard
         * and jump targets  */

    case IPT_ENTRY_JUMP:
    case IPT_ENTRY_STANDARD:
    {
        struct xt_standard_target *st1, *st2;

        st1 = (struct xt_standard_target *) t1;
        st2 = (struct xt_standard_target *) t2;

        if (st1->verdict == st2->verdict)
            return IPT_COMPARE_SUCCESS;
        else
            return IPT_COMPARE_FAILURE;
    }
        /* the complete target shall be checked
         * if it is a module */
    case IPT_ENTRY_MODULE:

        /* target size */
        if (t1->u.target_size != t2->u.target_size)
            return IPT_COMPARE_FAILURE;

        /* name shall match */
        if (g_strcmp0 (t1->u.user.name, t2->u.user.name) != 0)
            return IPT_COMPARE_FAILURE;

        /* compare byte by byte data */
        for (i = 0; i < (t1->u.target_size -
             sizeof (struct xt_standard_target)); i++)
            if ((t1->data[i] ^ t2->data[i]) != 0)
                return IPT_COMPARE_FAILURE;

        return IPT_COMPARE_SUCCESS;

    default:
        /* unknown type of the target */
        ipt_debug (DEBUG_ERR, "ERROR: bad type %d", r1->type);
        return IPT_COMPARE_FAILURE;
    }

    return IPT_COMPARE_FAILURE;
}

static int
ipt_compare_xt_entry_matches (struct xt_entry_match *m1,
                              struct xt_entry_match *m2)
{
    unsigned int i = 0;

    return_val_if_fail (m1 && m2, IPT_COMPARE_FAILURE);

    /* size match */
    if (m1->u.match_size != m2->u.match_size)
        return IPT_COMPARE_FAILURE;

    /* revision match */
    if (m1->u.user.revision != m2->u.user.revision)
        return IPT_COMPARE_FAILURE;

    /* name shall match */
    if (g_strcmp0 (m1->u.user.name, m2->u.user.name) != 0)
        return IPT_COMPARE_FAILURE;

    /* byte by byte match for the match data */
    for ( ; i < m1->u.match_size - sizeof(struct xt_entry_match); i++)
        if ((m1->data[i] ^ m2->data[i]) != 0)
            return IPT_COMPARE_FAILURE;

    return IPT_COMPARE_SUCCESS;
}

static int
ipt_compare_matches (const rule_t *const r1,
                     const rule_t *const r2)
{
    unsigned int m1_offset = 0,
            m2_offset = 0,
            r1_matches = 0,
            r2_matches = 0;
    struct xt_entry_match *m1, *m2;

    return_val_if_fail (r1 && r2, IPT_COMPARE_FAILURE);

    ipt_debug (DEBUG_TRACE, "Comparing matches for r1 [%p] r2 [%p]",
               r1, r2);

    m1_offset = sizeof (struct ipt_entry);
    m2_offset = sizeof (struct ipt_entry);

    /* compiling the no. of matches in the rule 1 */
    for ( ; m1_offset < r1->entry->target_offset ;
          m1_offset += m1->u.match_size ) {
        m1 = (void *)r1->entry + m1_offset;
        r1_matches++;
    }

    /* compiling the no. of matches in the rule 2 */
    for ( ; m2_offset < r2->entry->target_offset ;
          m2_offset += m2->u.match_size ) {
        m2 = (void *)r2->entry + m2_offset;
        r2_matches++;
    }

    ipt_debug (DEBUG_INFO, "No. of matches in the rule 1 [%p]: %u",
               r1, r1_matches);
    ipt_debug (DEBUG_INFO, "No. of matches in the rule 2 [%p]: %u",
               r2, r2_matches);

    /* r1 and r2 shall have the same no. of matches */
    if (r1_matches != r2_matches)
        return IPT_COMPARE_FAILURE;

    m1_offset = sizeof (struct ipt_entry);
    m2_offset = sizeof (struct ipt_entry);

    for ( ; m1_offset < r1->entry->target_offset &&
          m2_offset < r2->entry->target_offset;
          m1_offset += m1->u.match_size,
          m2_offset += m2->u.match_size ) {

        m1 = (void *)r1->entry + m1_offset;
        m2 = (void *)r2->entry + m2_offset;

        ipt_debug (DEBUG_INFO, "comparing match m1: \"%s\" and m2: \"%s\"",
                   m1->u.user.name, m2->u.user.name);

        if (ipt_compare_xt_entry_matches (m1, m2) != IPT_COMPARE_SUCCESS)
            return IPT_COMPARE_FAILURE;
    }

    return IPT_COMPARE_SUCCESS;
}

static int
ipt_compare_rules (const rule_t *const entry1,
                   const rule_t *const entry2)
{
    return_val_if_fail (entry1 && entry2, IPT_COMPARE_FAILURE);

    ipt_debug (DEBUG_INFO, "Comparing rules: \"%p\" and \"%p\"", entry1, entry2);

    /* compare the struct ipt_ip blob */
    if (memcmp (&entry1->entry->ip, &entry2->entry->ip,
                sizeof(struct ipt_ip)) != 0)
        return IPT_COMPARE_FAILURE;

    /* offsets (target and next) shall match */
    if (entry1->entry->target_offset != entry2->entry->target_offset)
        return IPT_COMPARE_FAILURE;
    if (entry1->entry->next_offset != entry2->entry->next_offset)
        return IPT_COMPARE_FAILURE;

    if ( ipt_compare_targets (entry1, entry2) != IPT_COMPARE_SUCCESS) {
        ipt_debug (DEBUG_INFO, "Targets do not match for rule 1 [%p] and rule 2 [%p]",
              entry1, entry2);
        return IPT_COMPARE_FAILURE;
    }

    if ( ipt_compare_matches (entry1, entry2) != IPT_COMPARE_SUCCESS) {
        ipt_debug (DEBUG_INFO, "Matches do not match for rule 1 [%p] and rule 2 [%p]",
              entry1, entry2);
        return IPT_COMPARE_FAILURE;
    }

    return IPT_COMPARE_SUCCESS;
}

static void
ipt_table_update_offsets (const table_t *const table)
{
    /* assuming that the first rule always has the
     * head offset as 0 (i.e., starts at 0) */
    int index = 0;
    unsigned int offset = 0;
    GList *chains, *rules;

    return_if_fail (table);

    ipt_debug (DEBUG_TRACE, "Updating the offsets for the table: %s",
               table->name);

    chains = table->chains;
    for ( ; chains; chains = chains->next) {

        rules = ((chain_t *) chains->data)->rules;
        for ( ; rules; rules = rules->next) {
            rule_t *rule = rules->data;
            /* skip the deleted rule */
            if (rule->opcode == IP_RULE_OPCODE_DELETE)
                continue;

            ipt_debug (DEBUG_INFO,
                       "Rule [%p, no :%d] "
                       "previous offset : %u "
                       "recalculated offset : %u",
                       rule, index++,
                       rule->offset, offset);
            rule->offset = offset;
            offset += rule->entry->next_offset;
        }
    }
}

static int
ipt_restore_jumps (const table_t *table)
{
    rule_t *rule;
    chain_t *chain;
    GList *chains, *rules;
    struct xt_standard_target *target;

    return_val_if_fail (table, -EINVAL);

    ipt_debug (DEBUG_TRACE, "Updating jumps for the table: %s",
               table->name ? table->name : "Unknown");

    chains = table->chains;
    for ( ; chains; chains = chains->next) {

        chain = chains->data;
        continue_if_fail (chain);

        ipt_debug (DEBUG_INFO, "Examining the chain: %s", chain->name ?
                  chain->name : "Unknown");
        rules = chain->rules;
        for ( ; rules; rules = rules->next) {

            rule = rules->data;
            continue_if_fail (rule);

            if (rule->opcode == IP_RULE_OPCODE_DELETE)
                continue;

            if (rule->type != IPT_ENTRY_JUMP &&
                    rule->type != IPT_ENTRY_FALLTHROUGH)
                continue;

            /* screwed, ??? */
            if (!rule->jump) {
                ipt_debug (DEBUG_ERR, "No jump found for the jump rule [%p]",
                           rule);
                return -ENXIO;
            }

            target = (struct xt_standard_target *)
                    ipt_get_target (rule->entry);

            ipt_debug (DEBUG_INFO,
                       "Rule: %p "
                       "previous verdict: %d "
                       "new verdict: %u",
                       rule,
                       target->verdict,
                       ((rule_t *) rule->jump)->offset);

            target->verdict = (int) ((rule_t *) rule->jump)->offset;
        }
    }

    return 0;
}

static int
ipt_delete_rule_from_chain (rule_t *rule,
                            const table_t *const table,
                            chain_t *chain)
{
    GList *rules;
    rule_t *r = NULL;
    int found = 0;

    return_val_if_fail (table && chain && rule , -EINVAL);

    ipt_debug (DEBUG_TRACE,
               "Deleting rule %p "
               "from the chain: \"%s\" "
               "of table: \"%s\" "
               "builtin: %d",
               rule, chain->name,
               table->name, chain->builtin);

    /* - builtin chain atleast should have the policy entry
     * - user defined chains atleast shall be two entries
     *      i.e., error and policy */
    if (chain->builtin < NF_INET_NUMHOOKS) {
        if (g_list_length (chain->rules) <= 1)
            return -ENOENT;
    } else
        if (g_list_length (chain->rules) <= 2)
            return -ENOENT;

    rules = chain->rules;
    for ( ; rules; rules = rules->next) {
        r = rules->data;
        if (ipt_compare_rules (r, rule) == IPT_COMPARE_SUCCESS) {
            ipt_debug (DEBUG_INFO, "Rule %p == Rule %p", r, rule);
            found = 1;
            break;
        }
    }

    if (found) {
        r->opcode = IP_RULE_OPCODE_DELETE;
        return 0;
    }

    return -ENOENT;
}

static int
ipt_find_chain_from_offset (const table_t *const table,
                            unsigned int offset,
                            chain_t **chain)
{
    chain_t *c;
    GList *chains, *rules;
    rule_t *rule;
    int offset_is_a_chain = 0, index;

    return_val_if_fail (table && chain && !(*chain), -EINVAL);

    for (chains = table->chains; chains; chains = chains->next) {

        c = chains->data;
        continue_if_fail (c);

        ipt_debug (DEBUG_TRACE,"Examining the chain: %s length: %u", c->name,
                   g_list_length (c->rules));

        for (index = 0, rules = c->rules; rules; rules = rules->next, index++) {

            rule = rules->data;
            continue_if_fail (rule);

            if (offset == rule->offset)
                offset_is_a_chain = 1;

            /* For builtin chains, the first ipt_entry
             * is a rule but for an unser defined chain,
             * the first entry is always an error and
             * then the real rule. Therefore we allow
             * two iterations for user defined chains but
             * only one for builtins */
            if (c->builtin != 0xFFFFFFFF)
                break;
            else if (1 == index)
                break;
        }

        if (offset_is_a_chain) {
            *chain = c;
            break;
        }
    }

    return offset_is_a_chain == 1 ? 0 : -ENOTUNIQ;
}

static int
ipt_examine_chain (rule_t *rule,
                   const table_t *const table,
                   chain_t *chain,
                   int *available)
{
    int ret;
    GList *rules;
    rule_t *r;
    chain_t *c;

    /* Here we check if the rule is already part
     * of the chain */
    for (rules = chain->rules; rules; rules = rules->next) {

        r = rules->data;
        continue_if_fail (r);

        /* If a rule is a jump to any chain then
         * we traverse the chain and check if the
         * user defined chain already has this rule */
        c = NULL;
        if (IPT_ENTRY_JUMP == r->type) {

            if (!r->jump)
                continue;

            ret = ipt_find_chain_from_offset (table, ((rule_t *) r->jump)->offset, &c);
            if (ret == 0)
                /* recursive, examine the chain again */
                ipt_examine_chain (rule, table, c, available);
        } else {
            if (ipt_compare_rules (r, rule) == IPT_COMPARE_SUCCESS) {
                ipt_debug (DEBUG_INFO, "Match identified: Rule %p == Rule %p",
                           r, rule);
                *available = 1;
            }
        }

        if (*available)
            break;
    }

    return *available == 1 ? 0 : -ENOENT;
}

static int
ipt_append_rule_to_chain (rule_t *rule,
                          const table_t *const table,
                          chain_t *chain)
{
    int ret, available = 0;
    GList *bfe, *nxt;
    rule_t *nxt_entry = NULL, *bfe_entry = NULL;

    return_val_if_fail (table && chain && rule , -EINVAL);

    ipt_debug (DEBUG_TRACE, "Adding Rule: %p to chain: \"%s\"",
               rule, chain->name);

    ret = ipt_examine_chain (rule, table, chain, &available);
    if (0 == ret)
        return -EEXIST;

    /* Note: We are here appending the rule to a chain,
     *       - the last entry of a chain is the policy
     * Thus the new rule has to be inserted here
     * (n-2), (n-1) --> new entry, n (policy)--> last entry */

    /* nth entry */
    nxt = g_list_nth (chain->rules, chain->num_rules-1);
    /* n-1 entry */
    bfe = g_list_nth (chain->rules, chain->num_rules-2);

    if (!nxt) {
        ipt_debug (DEBUG_ERR, "Invalid last entry, chain shall have "
                   "atleast one entry");
        return -EINVAL;
    }

    if (!bfe)
        ipt_debug (DEBUG_INFO, "Empty n-1 entry, only one rule (policy) in the chain");

    nxt_entry = (rule_t *) nxt->data;
    bfe_entry = bfe ? (rule_t *) bfe->data : NULL;

    ipt_debug (DEBUG_INFO, "Chain: %s "
               "bfe [head %d foot %d] "
               "nxt bfe [head %d foot %d]",
               chain->name,
               bfe_entry ? bfe_entry->offset : (unsigned int) -1 ,
               bfe_entry ? bfe_entry->entry->next_offset : (unsigned int) -1,
               nxt_entry->offset, nxt_entry->entry->next_offset);

    if (bfe_entry)
        rule->offset = bfe_entry->offset + bfe_entry->entry->next_offset;
    nxt_entry->offset = rule->offset + rule->entry->next_offset;
    chain->rules = g_list_insert_before (chain->rules, nxt, rule);
    /* new rule added to the chain increment the count */
    chain->num_rules++;

    /* just for debugging purpose */
    ipt_dump_table (table);
    return 0;
}

static void
ipt_compile_new_table_size_for_replacement (unsigned int *size,
                                            const table_t *const table)
{
    GList *chain_list, *rule_list;

    return_if_fail (table);

    chain_list = table->chains;
    for ( ; chain_list; chain_list = chain_list->next) {

        chain_t *chain = chain_list->data;
        rule_list = chain->rules;
        for ( ; rule_list; rule_list = rule_list->next) {
            rule_t *rule = rule_list->data;
            /* skip the deleted rule */
            if (rule->opcode != IP_RULE_OPCODE_DELETE)
                *size += rule->entry->next_offset;
        }
    }

    ipt_debug (DEBUG_INFO, "Total size of the table %s: %u",
               table->name, *size);
}

static void
ipt_replace_update_hook_underflow (struct ipt_replace *replace,
                                   const table_t *const table)
{
    chain_t *chain;
    GList *chains, *rules;
    rule_t *r = NULL;
    unsigned int hook_entry = 0,
            underflow = 0, i = 0;

    return_if_fail (table && replace);

    ipt_debug (DEBUG_TRACE, "Table: %s Replace: %s",
               table->name, replace->name);

    chains = table->chains;

    /* initialize the hooks and the underflows of the builtin chains
     * with an impossible value */
    for ( ; i < NF_INET_NUMHOOKS; i++) {
        replace->hook_entry[i] = 0xFFFFFFFF;
        replace->underflow[i] = 0xFFFFFFFF;
    }

    for ( ; chains; chains = chains->next) {

        chain = chains->data;

        ipt_debug (DEBUG_INFO,
                   "Chain: \"%s\" [%p] "
                   "no. of rules: %u",
                   chain->name,
                   chain, chain->num_rules);

        if (ipt_is_builtin_chain (chain->name) != 0)
            continue;

        /* finding the hook entry */
        rules = chain->rules;
        for ( ; rules; rules = rules->next) {
            r = rules->data;
            /* the first rule without IP_RULE_OPCODE_DELETE is
             * the head of this chain */
            if (r->opcode != IP_RULE_OPCODE_DELETE) {
                hook_entry = r->offset;
                break;
            }
        }

        rules = chain->rules;
        for ( ; rules; rules = rules->next) {
            r = rules->data;
            if (r->opcode != IP_RULE_OPCODE_DELETE)
                underflow += r->entry->next_offset;
        }

        replace->hook_entry[chain->builtin] = hook_entry;
        replace->underflow[chain->builtin] = underflow - r->entry->next_offset;
    }

    ipt_debug (DEBUG_INFO, "Table: %s Hooks: pre-routing/input/forward/output/post-routing"
               " = %d/%d/%d/%d/%d",
               replace->name,
               replace->hook_entry[NF_INET_PRE_ROUTING],
               replace->hook_entry[NF_INET_LOCAL_IN],
               replace->hook_entry[NF_INET_FORWARD],
               replace->hook_entry[NF_INET_LOCAL_OUT],
               replace->hook_entry[NF_INET_POST_ROUTING]);

    ipt_debug (DEBUG_INFO, "Table : %s Underflows: pre-routing/input/forward/output/post-"
               "routing = %d/%d/%d/%d/%d",
               replace->name,
               replace->underflow[NF_INET_PRE_ROUTING],
               replace->underflow[NF_INET_LOCAL_IN],
               replace->underflow[NF_INET_FORWARD],
               replace->underflow[NF_INET_LOCAL_OUT],
               replace->underflow[NF_INET_POST_ROUTING]);
}

static void
ipt_replace_update_entries (struct ipt_replace *replace,
                            const table_t *const table)
{
    chain_t *chain;
    GList *chains, *rules;
    rule_t *r;
    unsigned char *entry_index;

    return_if_fail (replace && table);

    ipt_debug (DEBUG_TRACE, "Table: %s Replace: %s",
               table->name, replace->name);

    entry_index = (unsigned char *) replace->entries;
    chains = table->chains;

    for ( ; chains; chains = chains->next) {

        chain = chains->data;
        ipt_debug (DEBUG_INFO,
                   "Chain: \"%s\" [%p] "
                   "No. of rules: %u",
                   chain->name,
                   chain, chain->num_rules);

        rules = chain->rules;
        for ( ; rules; rules = rules->next) {
            r = rules->data;
            /* skip the deleted entry */
            if (r->opcode != IP_RULE_OPCODE_DELETE) {
                memcpy (entry_index, r->entry, r->entry->next_offset);
                entry_index += r->entry->next_offset;
            }
        }
    }

    ipt_debug (DEBUG_TRACE, "ipt_entries ready for replacement");
}

static int
ipt_compile_table_entries (unsigned int *no,
                           const table_t *const table)
{
    chain_t *chain;
    GList *chains, *rules;

    return_val_if_fail (table && no, -EINVAL);

    chains = table->chains;
    for ( ; chains; chains = chains->next) {
        chain = chains->data;
        rules = chain->rules;
        for ( ; rules; rules = rules->next) {
            rule_t *r = rules->data;
            /* skip the deleted entry */
            if (r->opcode != IP_RULE_OPCODE_DELETE)
                (*no)++;
        }
    }

    return 0;
}

static int
ipt_commit_counters(const table_t *table,
                    struct xt_counters_info *c)
{
    int ret;

    DEBUG("Table: %s Counters: %p", table->name, c);

    ret = setsockopt (table->sock, IPPROTO_IP, IPT_SO_SET_ADD_COUNTERS,
                      c, (socklen_t) (sizeof(*c) + (sizeof(struct xt_counters) * c->num_counters)));
    if (ret < 0) {
        ipt_debug (DEBUG_ERR, "Error occurred while commiting the counters "
                   "[table: %s]: %s/%d", table->name,
                   strerror (errno), errno);
        return -errno;
    }

    return 0;
}

static int
ipt_compile_counters (const table_t *const table,
                      const struct ipt_replace *repl)
{
    int err, i = 0, j = 0;
    unsigned int num_entries = 0;
    size_t counterlen;
    chain_t *chain;
    rule_t *r;
    GList *chains, *rules;
    struct xt_counters_info *counters;

    return_val_if_fail (table && repl, -EINVAL);

    err = ipt_compile_table_entries (&num_entries, table);
    if (err < 0) {
        ipt_debug (DEBUG_ERR, "Failed to compile the number of entries: %s/%d",
                   strerror (-err), -err);
        return err;
    }

    ipt_debug (DEBUG_INFO, "Total entries in the table %s: %u", table->name,
               num_entries);

    counterlen = sizeof (struct xt_counters_info) +
            sizeof (struct xt_counters) * num_entries;
    counters = g_try_malloc0 (counterlen);
    return_val_if_fail (counters, -ENOMEM);
    memset(counters, 0, counterlen);
    g_stpcpy (counters->name, table->info->name);
    counters->num_counters = num_entries;

    chains = table->chains;
    for ( ; chains; chains = chains->next) {

        chain = chains->data;
        rules = chain->rules;
        for ( ; rules; rules = rules->next, i++) {
            r = rules->data;

            /* new entry the counters shall be empty, just
             * skip the newly appended entry */
            if (r->opcode == IP_RULE_OPCODE_APPEND)
                continue;
            /* a deleted entry, thus skip the current old counter and
             * traverse to the new old counter and make sure that we
             * stay in the current new counter */
            else if (r->opcode == IP_RULE_OPCODE_DELETE) {
                i--; j++;
                continue;
            }

            ipt_debug (DEBUG_INFO,
                       "New counter: %u "
                       "Old counter: %u "
                       "Chain: %s "
                       "counters [packet %llu byte %llu]",
                       i, j, chain->name,
                       repl->counters[j].pcnt,
                       repl->counters[j].bcnt);

            counters->counters[i] = repl->counters[j];
            j++;
        }
    }

    err = ipt_commit_counters (table, counters);
    g_free (counters);
    return err;
}

static struct ipt_replace *
ipt_prepare_table_for_replacement (table_t *table)
{
    int err;
    unsigned int size = 0, new_entries = 0;
    struct ipt_replace *replace;

    return_val_if_fail (table, NULL);

    ipt_debug (DEBUG_TRACE, "Table: %s", table->name);

    ipt_compile_new_table_size_for_replacement (&size, table);

    err = ipt_compile_table_entries (&new_entries, table);
    if (err < 0) {
        ipt_debug (DEBUG_ERR, "Failed to get the number of entries: %s/%d",
                   strerror (-err), -err);
        return NULL;
    }

    ipt_debug (DEBUG_INFO,
               "No. of entries: %u "
               "Total size of the table (entries) \"%s\": %u",
               new_entries, table->name, size);

    replace = g_try_malloc0 (sizeof (struct ipt_replace) + size);
    if (!replace) {
        ipt_debug (DEBUG_ERR, "Failed to allocate for struct ipt_replace");
        return NULL;
    }

    /* old counters */
    replace->counters = g_try_malloc0 (sizeof (struct xt_counters)
                                       * table->info->num_entries);
    if (!replace->counters) {
        ipt_debug (DEBUG_ERR, "Failed to allocate for struct xt_counters: %s",
                   strerror (ENOMEM));
        g_free (replace);
        return NULL;
    }

    g_stpcpy (replace->name, table->info->name);
    replace->num_entries = new_entries;
    replace->num_counters = table->info->num_entries;
    replace->valid_hooks  = table->info->valid_hooks;
    replace->size = size;

    ipt_debug (DEBUG_INFO,
               "IPT Replace Table: %s "
               "Num of entries: %u "
               "size: %u "
               "num_counters: %u "
               "valid_hooks: %x",
               replace->name,
               replace->num_entries,
               replace->size,
               replace->num_counters,
               replace->valid_hooks);

    ipt_replace_update_entries (replace, table);
    /* update the builtin hooks hook_entry and underflow offsets */
    ipt_replace_update_hook_underflow (replace, table);

    return replace;
}

static int
ipt_replace_entries (const table_t *const table,
                     struct ipt_replace *replace)
{
    int ret;

    return_val_if_fail (replace && table, -EINVAL);

    ipt_debug (DEBUG_TRACE, "Table: %s Socket: %d Replace: %s",
               table->name,
               table->sock, replace->name);

    ret = setsockopt (table->sock, IPPROTO_IP, IPT_SO_SET_REPLACE,
                      replace, (socklen_t) (sizeof(*replace) + replace->size));
    if (ret < 0) {
        ipt_debug (DEBUG_ERR, "Failed to replace the %s: %s/%d",
                   table->name, strerror (errno), -errno);
        return -errno;
    }

    return 0;
}

static int
ipt_prepare_rule_for_manipulation (const char *const rule,
                                   ipt_entry_builder **builder,
                                   rule_t **r)
{
    int err;

    return_val_if_fail (rule && builder && r, -EINVAL);
    return_val_if_fail (!*builder, -EEXIST);
    return_val_if_fail (!*r, -EEXIST);

    /* parse the rule and load the necesary matches
     * and targets if necessary */
    err = ipt_prepare_entry_builder (rule, builder);
    if (err < 0) {
        ipt_debug (DEBUG_ERR, "Failed to parse the rule: %s/%d",
                   strerror (-err), -err);
        return err;
    }

    err = ipt_build_entry_from_rule (rule, *builder);
    if (err < 0) {
        ipt_debug (DEBUG_ERR, "Failed to parse and extract the rule params: %s/%d",
                   strerror (-err), -err);
        g_free (*builder);
        *builder = NULL;
        return err;
    }

    /* since now we have parsed the rule and every
     * detail is available, we can now build an ipt_entry
     * to be appened to the chain */
    err = ipt_generate_rule (r, *builder);
    if (err < 0) {
        ipt_debug (DEBUG_ERR, "Failed to create an ipt_entry for a new"
                   " rule inclusion: %s/%d", strerror (-err), -err);
        ipt_cleanup_builder (*builder);
        *builder = NULL;
        return err;
    }

    return 0;
}

static rule_t*
get_rule_from_offset (const table_t *table,
                      const int offset)
{
    rule_t *rule;
    chain_t *chain;
    GList *chains, *rules;

    return_val_if_fail (table, NULL);

    ipt_debug (DEBUG_TRACE, "Table: %s Offset: %d", table->name ?
               table->name : "Unknown", offset);

    chains = table->chains;
    for ( ; chains; chains = chains->next) {
        chain = chains->data;
        rules = (chain != NULL) ? chain->rules : NULL;
        for ( ; rules; rules = rules->next) {
            rule = rules->data;
            if (rule && rule->offset == (unsigned int) offset)
                return rule;
        }
    }

    return NULL;
}

static int
ipt_update_jumps (const table_t *table)
{
    rule_t *rule, *jump;
    chain_t *chain;
    GList *chains, *rules;
    struct xt_entry_target *target;
    struct xt_standard_target *starget;

    return_val_if_fail (table, -EINVAL);

    ipt_debug (DEBUG_TRACE, "Updating jumps for the table: %s",
               table->name ? table->name : "Unknown");

    chains = table->chains;
    for ( ; chains; chains = chains->next) {

        chain = chains->data;
        continue_if_fail (chain);

        ipt_debug (DEBUG_INFO, "Examining the chain: %s [%p], size: %u",
                   chain->name ? chain->name : "Unknown", chain,
                   g_list_length (chain->rules));
        rules = chain->rules;
        for ( ; rules; rules = rules->next) {

            rule = rules->data;
            continue_if_fail (rule);

            if (rule->type != IPT_ENTRY_JUMP &&
                    rule->type != IPT_ENTRY_FALLTHROUGH)
                continue;

            target = ipt_get_target (rule->entry);
            starget = (struct xt_standard_target *) target;
            ipt_debug (DEBUG_TRACE, "Rule %p is a jump to %d",
                       rule, starget->verdict);

            jump = get_rule_from_offset (table, starget->verdict);
            /* screwed, shall not happen */
            if (!jump) {
                ipt_debug (DEBUG_ERR, "Rule [%p] is jump but there "
                           "is no jump identified", rule);
                return -ENXIO;
            }

            ipt_debug (DEBUG_INFO, "Rule [%p] is a jump to rule [%p]",
                       rule, jump);
            rule->jump = jump;
        }
    }

    return 0;
}

static int
ipt_prepare_table (const char *const table_name,
                   table_t **table)
{
    int err;

    return_val_if_fail (table_name && table, -EINVAL);
    return_val_if_fail (!*table, -EEXIST);

    /* try to read the ip table */
    if (!(*table = ipt_get_table (table_name)))
        return -ENOENT;

    err = ipt_parse_table (*table);
    if (err < 0) {
        ipt_debug (DEBUG_ERR, "Failed to parse the table: %s, error: %s/%d",
                   (*table)->name, strerror (-err), -err);
        ipt_free_table (*table);
        return err;
    }

    err = ipt_update_jumps (*table);
    if (err < 0) {
        ipt_debug (DEBUG_ERR, "Failed to update the jumps for the table "
                   "\"%s\", err: %s/%d", (*table)->name ?
                   (*table)->name : "Unknown",
                   strerror (-err), -err);
        ipt_free_table (*table);
        return err;
    }

    /* dump the complete table for debugging purposes */
    ipt_dump_table (*table);
    return 0;
}

static void
ipt_cleanup_table (table_t *table,
                   ipt_entry_builder *builder)
{
    ipt_debug (DEBUG_INFO, "cleaning up table %p builder %p",
               table, builder);
    ipt_free_table (table);
    ipt_cleanup_builder (builder);
}

static int
ipt_append_entry (const char *const rule,
                  rule_t *r,
                  table_t *table,
                  chain_t *chain)
{
    int err;
    struct ipt_replace *repl;

    return_val_if_fail (rule && r && table && chain, -EINVAL);

    DEBUG ("Rule to be appended: \"%s\"", rule);

    r->opcode = IP_RULE_OPCODE_APPEND;
    err = ipt_append_rule_to_chain (r,table, chain);
    if (err < 0) {
        if (err != -EEXIST) {
            ERROR ("Failed to add the rule to chain: %s",
                   chain->name);
        } else
            INFO ("Rule is already part of netfilter");
        return err;
    }

    /* update the head offsets of various rules */
    ipt_table_update_offsets (table);

    DEBUG ("Successfully added the rule \"%s\" "
           "to chain: %s "
           "in table: %s",
           rule, chain->name, table->name);

    /* Lets update the jumps */
    err = ipt_restore_jumps (table);
    if (err < 0) {
        ERROR ("Failed to restore the jumps: %s/%d",
               strerror (-err), -err);
        return err;
    }

    if (!(repl = ipt_prepare_table_for_replacement (table)))
        return -ECANCELED;

    ipt_dump_table (table);
    err = ipt_replace_entries (table, repl);
    if (err < 0) {
        ERROR ("Failed to replace the ipt_entries: %s/%d",
               strerror (-err), -err);
        g_free (repl);
        return err;
    }

    err = ipt_compile_counters (table, repl);
    if (err < 0) {
        ERROR ("Failed to replace the ipt_entries: %s/%d",
               strerror (-err), -err);
        g_free (repl);
        return err;
    }

    INFO ("Successfully appended the rule [%s] to the table: %s",
          rule, table->name);
    g_free (repl->counters);
    g_free (repl);
    return 0;
}

static int
ipt_delete_entry (const char *const rule,
                  rule_t *r,
                  table_t *table,
                  chain_t *chain)
{
    int err;
    struct ipt_replace *repl;

    return_val_if_fail (rule && r && table && chain, -EINVAL);

    DEBUG ("Rule to be deleted: \"%s\"", rule);

    err = ipt_delete_rule_from_chain (r, table, chain);
    if (err < 0) {
        ERROR ("Failed to add the rule to chain %s "
               "error: %s/%d", chain->name,
               strerror (-err), -err);
        return err;
    }

    ipt_table_update_offsets (table);

    /* Lets update the jumps */
    err = ipt_restore_jumps (table);
    if (err < 0) {
        ERROR ("Failed to restore the jumps: %s/%d",
               strerror (-err), -err);
        return err;
    }

    ipt_dump_table (table);

    INFO ("Successfully deleted the rule \"%s\" to "
          "chain: %s "
          "of table: %s",
          rule, chain->name,
          table->name);

    if (!(repl = ipt_prepare_table_for_replacement (table)))
        return -ECANCELED;

    err = ipt_replace_entries (table, repl);
    if (err < 0) {
        ERROR ("Failed to replace the ipt_entries: %s/%d",
               strerror (-err), -err);
        g_free (repl);
        return err;
    }

    err = ipt_compile_counters (table, repl);
    if (err < 0) {
        ERROR ("Failed to replace the ipt_entries: %s/%d",
               strerror (-err), -err);
        g_free (repl);
        return err;
    }

    DEBUG ("Successfully deleted the rule [%s] to the table: %s",
           rule, table->name);
    g_free (repl->counters);
    g_free (repl);
    return 0;
}

static int
ipt_manipulate_table (const char *const rule)
{

    int err;
    table_t *table = NULL;
    chain_t *chain = NULL;
    rule_t *r = NULL;
    ipt_entry_builder *builder = NULL;

    return_val_if_fail (rule, -EINVAL);

    err = ipt_prepare_rule_for_manipulation (rule, &builder, &r);
    if (err < 0) {
        ERROR ("Failed to create a new rule for inclusion: %s/%d",
               strerror (-err), -err);
        return err;
    }

    if (!(builder->options & IPT_BUILD_OPTION_APPEND ||
            builder->options & IPT_BUILD_OPTION_DELETE)) {
        ERROR ("Only append/delete operations are supported");
        err = -EINVAL;
        goto build_cleanup;
    }

    err = ipt_prepare_table (builder->table, &table);
    if (err < 0) {
        ERROR ("Failed to prepare the table for appending the "
               "new rule :%s/%d", strerror (-err), -err);
        goto build_cleanup;
    }

    /* try to find the chain from the table */
    if (!(chain = ipt_get_chain (table, builder->chain))) {
        ERROR ("There is no such chain to insert the rule");
        goto out;
    }

    /* prepare the verdict for the target */
    err = ipt_prepare_target_verdict (table, chain, r);
    if (err < 0) {
        ERROR ("Failed to prepare the target: %s/%d",
               strerror (-err), -err);
        goto out;
    }

    if (builder->options & IPT_BUILD_OPTION_APPEND) {

        err = ipt_append_entry (rule, r, table, chain);
        if (err < 0) {
            if (err != -EEXIST) {
                ERROR ("Failed to append the rule [\"%s\"] "
                       "to the table: %s"
                       " chain : %s ",
                       rule, table->name, chain->name);
            }
            goto out;
        }
    }
    else if (builder->options & IPT_BUILD_OPTION_DELETE) {

        err = ipt_delete_entry (rule, r, table, chain);
        if (err < 0){
            ERROR ("Failed to delete the rule [\"%s\"] "
                   "to the table : %s "
                   "chain : %s ", rule,
                   table->name, chain->name);
            g_free (r);
            goto out;
        }
        g_free (r);
    }

    ipt_cleanup_table (table, builder);
    return 0;
out:
    ipt_free_table (table);
build_cleanup:
    ipt_cleanup_builder (builder);
    return err;
}

static int
ipt_init(void)
{
    int err = 0;

    DEBUG("");

    err = ipt_init_xtglobals();
    if (err < 0) {
        ERROR ("Failed to initialize xtables_globals error: %s/%d",
               strerror (-err), -err);
        return err;
    }

    err = xtables_init_all (xtglobals, NFPROTO_IPV4);
    if (err < 0) {
        ERROR ("Failed to initialize xtables");
        return err;
    }

    return err;
}

static int
ipt_deinit(void)
{
    return ipt_deinit_xtglobals ();
}

int
ipt_commit_rule (const char *const rule)
{
    return ipt_manipulate_table (rule);
}

static firewall_ops
ipt_firewall_ops = {
    .name		= "iptables",
    .commitrule	= ipt_commit_rule,
};

int
iptables_init (void)
{
    int err = 0;

    DEBUG ("");

    /* No CAP_NET_ADMIN permissions */
    if (get_capabilities (CAP_NET_ADMIN, CAP_EFFECTIVE) != 1)
        return err;

    err = ipt_init ();
    if (err < 0) {
        ERROR ("Failed to initialize the iptables: %s/%d",
               strerror (-err), -err);
        return err;
    }

    return firewall_driver_register (&ipt_firewall_ops);
}

int
iptables_deinit (void)
{
    int err = 0;

    DEBUG ("");

    if (get_capabilities (CAP_NET_ADMIN, CAP_EFFECTIVE) != 1)
        return err;

    err = ipt_deinit();
    if (err < 0)
        ERROR ("Failed to deinitialize the iptables: %s/%d",
               strerror (-err), -err);

    return firewall_driver_unregister (&ipt_firewall_ops);
}

/** @} */
