/**
 * @file dnsmasq-man.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <stdio.h>
#include <unistd.h>
#include <dirent.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include <linux/capability.h>
#include <sys/capability.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <glib.h>
#include <gio/gio.h>
#include <string.h>
#include <errno.h>
#include <log.h>
#include "inc/utils.h"
#include <capabilities.h>
#include <firewall.h>
#include <dnsmasq-man.h>
#include <server.h>
#include <db_defines.h>

/**
 * To get the PID of a particular process
 */
#define PROC_DIR            "/proc"
#define PROC_PROCESS_INFO   "cmdline"
#define PROC_NAME_SIZE      1024
#define O_READONLY          "r"

/**
 * Unused ports taken from http://www.iana.org/. Thus
 * dnsmasq will be asked to use one of these ports for listening
 * DNS queries (instead of the standard port 53) and port
 * forwrding will be enabled to redirect the incoming DNS
 * requests to alternate port.
 */
#define DNSMASQ_PORT_MIN        33657
#define DNSMASQ_PORT_MAX        34248

/**
 * Various DNSMASQ Dbus signals
 */
#define DNSMASQ_SIGNAL_UP                       "Up"
#define DNSMASQ_SIGNAL_DHCPLEASE_ADDED          "DhcpLeaseAdded"
#define DNSMASQ_SIGNAL_DHCPLEASE_UPDATED        "DhcpLeaseUpdated"
#define DNSMASQ_SIGNAL_DHCPLEASE_DELETED        "DhcpLeaseDeleted"

/**
 * Whether the DNSMASQ dbus proxy is shared across various
 * dnsmasq instances (well, internally)
 */
#define DNSMASQ_PROXY_SHARED        1
#define DNSMASQ_PROXY_UNSHARED      0

/* Below two files moved from dynamic to static partition
#define DNSMASQ_CONFILE             DNSMASQ_DB_DIR"/dnsmasq_%s.conf"
#define DNSMASQ_HOSTFILE            DNSMASQ_DB_DIR"/hosts_%s.txt"
*/
#define DNSMASQ_CONFILE             STATIC_DNSMASQ_DB_DIR"/dnsmasq_%s.conf"
#define DNSMASQ_HOSTFILE            STATIC_DNSMASQ_DB_DIR"/hosts_%s.txt"

#define DNSMASQ_LEASEFILE           DNSMASQ_DB_DIR"/dnsmasq_%s.leases"
#define DNSMASQ_TEMP_LEASEFILE      DNSMASQ_DB_TEMP_DIR"/dnsmasq_%s.leases"
#define DNSMASQ_PIDFILE             PID_FILE_DIR"/dnsmasq_%d.pid"
#define DNSMASQ_DBUS_SERVICENAME    "uk.org.thekelleys.dnsmasq_%s"

#define DNSMASQ_ENABLE_DHCP_INCOMING        "-i %s -p udp --dport 67 --sport 68 -j ACCEPT"
#define DNSMASQ_ENABLE_DHCP_OUTGOING        "-o %s -p udp --dport 68 --sport 67 -j ACCEPT"
#define DNSMASQ_ENABLE_DNS_TCP              "-i %s -p tcp -s %s/%d --dport 53 -j REDIRECT --to-port %d"
#define DNSMASQ_ENABLE_DNS_UDP              "-i %s -p udp -s %s/%d --dport 53 -j REDIRECT --to-port %d"

/**
 * Checks whether the process has CAP_KILL privileges
 */
#define CAP_KILL_EFFECTIVE(CAP)    \
    ({get_capabilities (CAP, CAP_EFFECTIVE);})

/**
 * struct DhcpRange - DHCP Range information
 */
typedef
struct __dhcp_range
{
    /**
     * start - start IP address of the DHCP range
     */
    char *start;

    /**
     * end - end IP address of the DHCP range
     */
    char *end;

    /**
     * subnet - subnet mask
     */
    char *subnet;

    /**
     * router - Router IP address of this network.
     * Normally the mode running the dnsmasq
     */
    char *router;

    /**
     * broadcast - broadcast IP address of the network
     */
    char *broadcast;

    /**
     * leasetime - DHCP lease time. Shall not short
     * It is always good to have a considerable lease
     * period
     */
    unsigned int leasetime;
} DhcpRange;

/**
 * struct DnsPort - Alternate DNS port information
 */
typedef
struct __dns_port
{
    /**
     * alternateport - Alternate DNS port instead of
     * the standard port 53
     */
    unsigned int alternateport;
} DnsPort;

/**
 * struct DnsmasqProxy - DBus Proxy information
 */
typedef
struct __dnsmasq_proxy
{
    /**
     * busname - Well known name (or bus/service name) of
     * of the service provider
     */
    char *busname;

    /**
     * proxy - The actual dbus proxy
     */
    GDBusProxy *proxy;

    /**
     * proxywatch - Watch id for monitoring the owner of
     * the service
     */
    guint proxywatch;

    /**
     * shared - Whether the proxy is shared across various
     * dbus instances
     */
    int shared;
} DnsmasqProxy;

/**
 * struct clientdata - Client data for an async operation
 */
typedef
struct __client_data
{
    /**
     * ifname - Name of the network interface where the DHCP
     * server is running
     */
    char *ifname;

    /**
     * callback - Callback of the caller wherein the result
     * shall be returned
     */
    dhcp_server_callback callback;

    /**
     * userdata - Clients could pass some user data which
     * would be handed over to the "cb" along with the result
     */
    void *userdata;
} clientdata;

typedef
struct __dns_settings
{
    char **dns_servers;
    char **ntp_servers;
} dnssettings_t;

/**
 * struct DnsMasqInstance - Represents a single instance of
 * DHCP Server i.e., dnsmasq
 */
typedef
struct __dnsmasq_instance
{
    /**
     * proxy - DBus proxy
     */
    DnsmasqProxy *proxy;

    gboolean dnssupport;

    /**
     * pidfile - path for dnsmasq to record its process id
     */
    char *pidfile;

    /**
     * conffile - path for dnsmasq configuration file. Normally
     * dnsmasq reads its configuration from its standard path
     * /etc/dnsmasq.conf. Alternate file path can be given if it
     * is not desired
     */
    char *conffile;

    /**
     * hostfile - path for dnsmasq hostfile which has the information
     * about various hosts
     */
    char *hostfile;

    /**
     * pid - PID of the spawned dnsmasq process
     */
    int pid;

    /**
     * watch - watch id to watch the spwaned dnsmasq
     */
    unsigned int watch;

    /**
     * ifname - network interface name where this dnsmasq
     * shall control
     */
    char *ifname;

    /**
     * port - Alternate DNS port information if the standard
     * port is not desired
     */
    DnsPort *port;

    /**
     * ctx - firewall context specific to this dnsmasq instance
     */
    fwcontext *ctx;

    /**
     * dhcprange - DHCP range information used by this dnsmasq
     * instance to serve the clients
     */
    DhcpRange *dhcprange;

    dnssettings_t *dnsoptions;

    /**
     * data - callback data
     */
    clientdata *data;
} DnsMasqInstance;

/**
 * struct dnsmasqcmdline - command line data to start the
 * dnsmasq
 */
typedef
struct __dnsmasq_cmdline
{
    GPtrArray *array;
    GStringChunk *chunk;
} dnsmasqcmdline;

/**
 * struct dhcpoption - DHCP option information
 */
typedef
struct _dhcp_options
{
    /**
     * option - DHCP option
     */
    char *option;

    /**
     * optioncode - DHCP option code
     */
    int optioncode;
} dhcpoption;

/**
 * Various DHCP options. Normally Dnsmasq uses the interface
 * address (where it is operating) as the router address and
 * includes the same in the DHCP lease. If this is not desired
 * and there are other routers in the network, the default
 * behaviour can be changed by asking the dnsmasq to use a different
 * router address in the lease information to all the clients.
 *
 * It can also be configured to supply customized router,dns-server,
 * ntp-server addresses to only specific host or set of hosts. This
 * is particularly useful where if some of the clients in the network
 * are supposed to use a different router/dns-server.
 *
 */
/*
static const dhcpoption
dnsmasq_dhcp_options []=  {
    { "router",         3},
    { "dns-server",     6},
    { "ntp-server",     42},
    { NULL }
};
*/

/**
 * various executable paths of linux
 */
static const char *const
dnsmasq_executable_paths [] = {
    "/sbin",
    "/bin",
    "/usr/sbin",
    "/usr/local/sbin",
    "/usr/bin",
    "/usr/local/bin",
    NULL,
};

static GHashTable *dnsmasq_manager;
static DnsmasqProxy *proxy;
static GList *clients;
static unsigned int dnsport = DNSMASQ_PORT_MIN;
static GList *usedports;

static GDBusProxy*
dnsmasq_create_proxy (const char *name,
                      const char *interface,
                      const char *path,
                      GDBusConnection *connection)
{
    GDBusProxy *proxy = NULL;
    GError *error = NULL;

    return_val_if_fail (name && interface && path && connection, NULL);

    INFO ("Creating proxy for interface : %s [owner : %s]", interface, name);
    proxy = g_dbus_proxy_new_sync (connection,
                                   G_DBUS_PROXY_FLAGS_NONE,
                                   NULL,
                                   name,
                                   path,
                                   interface,
                                   NULL,
                                   &error);
    if (!proxy) {
        ERROR ("Failed to create proxy for the interface %s, error message is : %s",
               interface, error->message);
        g_error_free (error);
        return NULL;
    }

    INFO ("Proxy successfully created for the interface : \"%s\"", interface);
    return proxy;
}

static int
dnsmasq_find_executable (const char *const exe,
                         char **binpath)
{
    int i = 0, ret;
    struct stat st;
    const char *path;
    char exepath [PATH_MAX];

    return_val_if_fail (exe && binpath, -EINVAL);

    for ( ; (path = dnsmasq_executable_paths [i]); i++) {
        memset (exepath, 0, sizeof (exepath));
        sprintf (exepath, "%s/%s", dnsmasq_executable_paths [i], exe);
        if ((ret = stat (exepath, &st)) < 0)
            continue;

        /* We dont care whether the file is really an
         * executable. i.e., exec will report it */
        if (S_ISREG (st.st_mode)) {
            *binpath = g_strdup (exepath);
            return 0;
        }
    }

    return -ENOENT;
}

static dnsmasqevent
dnsmasq_signal_to_event (const char *evt)
{
    dnsmasqevent event = NUM_DNSMASQ_EVENTS;

    return_val_if_fail (evt, event);

    if (!g_strcmp0 (evt, DNSMASQ_SIGNAL_DHCPLEASE_ADDED))
        event = DNSMASQ_DHCP_LEASE_ADDED;
    else if (!g_strcmp0 (evt, DNSMASQ_SIGNAL_DHCPLEASE_UPDATED))
        event = DNSMASQ_DHCP_LEASE_UPDATED;
    else if (!g_strcmp0 (evt, DNSMASQ_SIGNAL_DHCPLEASE_DELETED))
        event = DNSMASQ_DHCP_LEASE_DELETED;

    return event;
}

static void
dnsmasq_update_clients (char *interface,
                        const dnsmasqevent event,
                        char *ipaddress,
                        char *macaddress,
                        char *hostname)
{
    if (event == NUM_DNSMASQ_EVENTS)
        return;
    dhcp_server_dhcp_event (event, interface, ipaddress,
                            macaddress, hostname);
}

static void
dnsmasq_signal_emitted (GDBusProxy *proxy,
                        gchar      *sender,
                        gchar      *signalname,
                        GVariant   *value,
                        gpointer    userdata)
{
    (void) proxy;
    DnsMasqInstance *dnsmasq;
    char *ipaddress = NULL, *mac = NULL,
            *hostname = NULL, *interface = NULL;

    dnsmasq = userdata;
    return_if_fail (dnsmasq);

    INFO ("Signal Info : \"%s\" \"%s\" of type \"%s\"", sender,
          signalname, g_variant_get_type_string (value));

    if (!g_strcmp0 (signalname, DNSMASQ_SIGNAL_UP)) {
        INFO ("DnsMasq Process is UP!!!");
    }
    else if (!g_strcmp0 (signalname, DNSMASQ_SIGNAL_DHCPLEASE_ADDED) ||
             !g_strcmp0 (signalname, DNSMASQ_SIGNAL_DHCPLEASE_UPDATED) ||
             !g_strcmp0 (signalname, DNSMASQ_SIGNAL_DHCPLEASE_DELETED)) {

        if (!g_variant_check_format_string (value, "(sss)", FALSE))
            ERROR ("Invalid format for the parameters");

        g_variant_get (value, "(sss)", &ipaddress, &mac, &hostname);
		interface =  (dnsmasq->ifname)? dnsmasq->ifname: NULL;
        INFO ("Interface : \"%s\" IPAddress : \"%s\" MACAddress : \"%s\""
              "Hostname : \"%s\"", interface ? interface : "UNKNOWN",
              ipaddress, mac, hostname);

        dnsmasq_update_clients (interface,
                                dnsmasq_signal_to_event (signalname),
                                ipaddress, mac, hostname);
	
    }
}

static DnsmasqProxy*
dnsmasq_proxy_init (const char *busname,
                    const int shared)
{
    DnsmasqProxy *proxy;

    return_val_if_fail (busname, NULL);

    proxy = g_try_malloc0 (sizeof (*proxy));
    return_val_if_fail (proxy, NULL);
    proxy->busname = g_strdup (busname);
    proxy->shared = shared;
    return proxy;
}

static int
dnsmasq_initialize_proxy (const char *name,
                          const char *interface,
                          const char *objpath,
                          GDBusConnection *connection,
                          GDBusProxy **proxy,
                          void *userdata)
{
    size_t id;

    return_val_if_fail (interface && objpath && proxy, -EINVAL);
    return_val_if_fail (!*proxy, -EEXIST);
    *proxy = dnsmasq_create_proxy (name, interface, objpath, connection);
    return_val_if_fail (*proxy, -ENOLINK);
    id = g_signal_connect (*proxy, "g-signal",
                           G_CALLBACK (dnsmasq_signal_emitted),
                           userdata);
    if (id <= 0)
        return -ENOTCONN;
    return 0;
}

static int
dnsmasq_disable_fw (DnsMasqInstance *dnsmasq)
{
    int err = 0;

    return_val_if_fail (dnsmasq, -EINVAL);

    DEBUG ("Dnsmasq disabling firewall : %s", dnsmasq->ifname);
    err = firewall_disable (dnsmasq->ctx);
    if (err < 0) {
        ERROR ("Failed to disable firewall for dnsmasq [%s] : %s",
               dnsmasq->ifname, strerror (-err));
        return err;
    }

    firewall_cleanup_context (dnsmasq->ctx);
    dnsmasq->ctx = NULL;
    return err;
}

static int
dnsmasq_fw_rule_add (DnsMasqInstance *dnsmasq,
                     const char *table,
                     const char *chain,
                     const char *format,
                     ...)
{
    int ret = -EINVAL;
    va_list args;
    char rule [BUFSIZ];

    return_val_if_fail (dnsmasq && table && chain, -EINVAL);

    va_start (args, format);
    memset (rule, 0, sizeof (rule));

    if (vsnprintf (rule, sizeof (rule), format, args) > 0) {
        ret = firewall_add_rule (dnsmasq->ctx, table, chain, rule);
        if (ret < 0)
            ERROR ("Failed to add the \"%s\" to the dnsmasq instance: %s "
                   "[%s/%d]", rule, dnsmasq->ifname, strerror (-ret), -ret);
    } else
        ERROR ("Invalid rule format : %s", format);

    va_end (args);
    return ret;
}

static int
dnsmasq_enable_fw (DnsMasqInstance *dnsmasq)
{
    int ret;
    char *iprange = NULL;
    unsigned int ipblock;
    unsigned char prefix;

    return_val_if_fail (dnsmasq, -EINVAL);

    DEBUG ("Enable DHCP ports and DNS port forwarding for the dnsmasq : %s",
           dnsmasq->ifname);

    dnsmasq_fw_rule_add (dnsmasq, "filter", "INPUT", DNSMASQ_ENABLE_DHCP_INCOMING, dnsmasq->ifname);
    dnsmasq_fw_rule_add (dnsmasq, "filter", "OUTPUT", DNSMASQ_ENABLE_DHCP_OUTGOING, dnsmasq->ifname);

    if (dnsmasq->dnssupport) {
        if (dnsmasq->dhcprange) {
            prefix = calculate_netmask (dnsmasq->dhcprange->subnet);
            if (prefix != 255 && dnsmasq->dhcprange->router) {
                ipblock = ntohl (inet_addr (dnsmasq->dhcprange->router));
                iprange = g_strdup_printf ("%d.%d.%d.0", ((ipblock >> 24) & 0xFF),
                                           ((ipblock >> 16) & 0xFF), ((ipblock >> 8) & 0xFF));
                dnsmasq_fw_rule_add (dnsmasq, "nat", "PREROUTING", DNSMASQ_ENABLE_DNS_TCP,
                                     dnsmasq->ifname, iprange, prefix, dnsmasq->port->alternateport);
                dnsmasq_fw_rule_add (dnsmasq, "nat", "PREROUTING", DNSMASQ_ENABLE_DNS_UDP,
                                     dnsmasq->ifname, iprange, prefix, dnsmasq->port->alternateport);
                g_free (iprange);
            } else {
                ERROR ("Invalid subnet mask detected, could not insert the NAT rules for port forwarding");
            }
        } else
            ERROR ("Invalid DHCP details for dnsmasq instance: %s", dnsmasq->ifname);
    }

    ret = firewall_enable (dnsmasq->ctx);
    if (ret < 0)
        ERROR ("Failed to enable firewall %s", strerror (-ret));

    return ret;
}

static void
dnsmasq_cleanup_clientdata (gpointer data)
{
    clientdata *clidata = data;
    return_if_fail (clidata);
    g_free (clidata->ifname);
    g_free (clidata);
}

static void
dnsmasq_proxy_cleanup (gpointer data)
{
    DnsmasqProxy *pxy = data;

    return_if_fail (pxy);
    g_free (pxy->busname);
    if (pxy->proxywatch > 0) {
        g_bus_unwatch_name (pxy->proxywatch);
        if (pxy->proxy)
            g_object_unref (pxy->proxy);
    }
    g_free (pxy);
}

static void
dnsmasq_cleanup (gpointer data)
{
    int ret;
    DnsMasqInstance *dnsmasq = data;

    return_if_fail (dnsmasq);

    DEBUG ("Cleaning up the dnsmasq instance for interface: %s",
           dnsmasq->ifname);
    if (dnsmasq->proxy) {
        if (!dnsmasq->proxy->shared)
            dnsmasq_proxy_cleanup (dnsmasq->proxy);
    }
    if (dnsmasq->ctx) {
        ret = dnsmasq_disable_fw (dnsmasq);
        if (ret < 0)
            ERROR ("Failed to disable the firewall context "
                   "for dnsmasq interface: %s", dnsmasq->ifname);
    }
    if (dnsmasq->dhcprange) {
        g_free (dnsmasq->dhcprange->broadcast);
        g_free (dnsmasq->dhcprange->subnet);
        g_free (dnsmasq->dhcprange->start);
        g_free (dnsmasq->dhcprange->end);
        g_free (dnsmasq->dhcprange->router);
    }

    if (dnsmasq->port) {
        usedports =
                g_list_remove (usedports, GUINT_TO_POINTER (dnsmasq->port->alternateport));
        g_free (dnsmasq->port);
    }

    if (dnsmasq->dnsoptions) {
        g_strfreev (dnsmasq->dnsoptions->dns_servers);
        g_strfreev (dnsmasq->dnsoptions->ntp_servers);
    }

    g_free (dnsmasq->dnsoptions);
    g_free (dnsmasq->dhcprange);
    g_free (dnsmasq->pidfile);
    g_free (dnsmasq->conffile);
    g_free (dnsmasq->hostfile);
    g_free (dnsmasq->ifname);
    g_free (dnsmasq);
}

static void
dnsmasq_args_cleanup (gpointer data)
{
    g_free (data);
}

static char**
dnsmasq_get_args_from_cmdline (const char *cmdline,
                               int delimiter,
                               unsigned int count)
{
    char **args = NULL;
    GSList *slist = NULL, *templist;
    char *buf, *oldbuf;
    unsigned int size = 512;
    unsigned int index, filler = 0;
    size_t length = 0;

    buf = g_try_malloc0 (size);
    return_val_if_fail (buf, NULL);

    for (index = 0; index < count; index++) {

        if (filler == size) {
            oldbuf = buf;
            size *= 2;
            buf = g_try_realloc (buf, size * (sizeof (*buf)));
            if (!buf) {
                g_free (oldbuf);
                goto failure;
            }
        }

        if (delimiter != cmdline [index])
            buf [filler++] = cmdline [index];
        else {
            if (strlen (buf) > 0)
                slist = g_slist_append (slist, g_strdup (buf));
            filler = 0;
            memset (buf, 0, size);
        }
    }

    length = g_slist_length (slist);
    if (length == 0)
        goto failure;

    args = g_try_malloc0 ((length + 1) * sizeof (*args));
    if (!args)
        goto failure;

    for (index = 0, templist = slist; templist;
         templist = templist->next, index++)
        args [index] = g_strdup ((char *) templist->data);

failure:
    g_free (buf);
    g_slist_free_full (slist, dnsmasq_args_cleanup);
    return args;
}

static int
dnsmasq_match_args (const char *process,
                    char **args,
                    char **match_args)
{
    int proceed = 0, nomatch = 0;
    char **temp, *name, *path, **tmatch;

    return_val_if_fail (process && args, -EINVAL);
    return_val_if_fail (g_strv_length (args) > 0, -EINVAL);

    path = args [0];
    if (path) {

        name = strrchr (path, '/');
        if (name)
            name++;

        if(!name)
            return -EINVAL;

        if (!g_strcmp0 (name, process))
            proceed = 1;
    }

    if (!proceed)
        return -ENXIO;

    if (!match_args)
        return 0;

    for (tmatch = match_args; *tmatch; tmatch++) {

        proceed = 0;
        for (temp = args; *temp; temp++) {
            if (strstr (*temp, *tmatch)) {
                proceed = 1;
                break;
            }
        }

        if (!proceed) {
            nomatch = 1;
            break;
        }
    }

    return nomatch == 0 ? 0 : -ENXIO;
}

static int
dnsmasq_get_pid (const char *processname,
                 char **args,
                 pid_t *pid)
{
    int found = 0, ch = 0, ret;
    FILE *pfile;
    DIR *dir, *internaldir;
    struct dirent *ent, *internalent;
    unsigned int size = 0, count = 0;
    char *buffer = NULL,
            dirname [PATH_MAX];
    char **cmdline;

    return_val_if_fail (processname, -EINVAL);

    dir = opendir (PROC_DIR);
    return_val_if_fail (dir, -errno);
    while ((ent = readdir (dir))) {

        /* current directory */
        if (!g_strcmp0 (ent->d_name, "."))
            continue;
        /* parent directory */
        if (!g_strcmp0 (ent->d_name, ".."))
            continue;

        memset (dirname, 0, sizeof (dirname));
        sprintf (dirname,"%s/%s", PROC_DIR, ent->d_name);
        internaldir = opendir (dirname);
        if (!internaldir)
            continue;

        while ((internalent = readdir (internaldir))) {

            ch = 0;
            size = count = 0;
            if (g_strcmp0 (internalent->d_name, PROC_PROCESS_INFO) != 0)
                continue;

            memset (dirname, 0, sizeof (dirname));
            sprintf (dirname, "%s/%s/%s", PROC_DIR,
                     ent->d_name, internalent->d_name);

            pfile = fopen (dirname, O_READONLY);
            if (!pfile)
                continue;

            /* initial size and we grow */
            size = PROC_NAME_SIZE;
            buffer = g_try_malloc0 (size * sizeof (char));
            memset (buffer, 0, size);
            /* first null byte */
            while ((ch = fgetc (pfile)) != EOF) {

                if (count == size) {
                    /* if the realloc fails, we need to clear old buffer */
                    char *oldbuf = buffer;
                    size *= 2;

                    buffer = (char *) g_try_realloc (buffer, size * (sizeof(char)));
                    if (!buffer) {
                        g_free (oldbuf);
                        fclose (pfile);
                        closedir (internaldir);
                        closedir (dir);
                        return -ENOMEM;
                    }
                }

                buffer [count++] = (char) ch;
            }

            cmdline = dnsmasq_get_args_from_cmdline (buffer, '\0', count);
            if (cmdline) {
                ret = dnsmasq_match_args (processname, cmdline, args);
                if (ret == 0) {
                    *pid = atoi (ent->d_name);
                    INFO ("Process identifier of \"%s\" is: %d", processname, *pid);
                    found = 1;
                }
                g_strfreev (cmdline);
            }

            g_free (buffer);
            fclose (pfile);

            if (found)
                break;
        }

        closedir (internaldir);
        if (found)
            break;
    }

    closedir (dir);
    if (found)
        return 0;

    return -ENOENT;
}

static void
dnsmasq_name_appeared (GDBusConnection *connection,
                       const gchar     *name,
                       const gchar     *name_owner,
                       gpointer         userdata)
{
    int ret;
    DnsmasqProxy *pxy;

    INFO ("Name %s is owned by %s", name, name_owner);

    pxy = userdata;
    return_if_fail (pxy && pxy->proxy);
    ret = dnsmasq_initialize_proxy (name,
                                    DNSMASQ_INTERFACE_NAME,
                                    DNSMASQ_OBJECT_PATH,
                                    connection,
                                    &pxy->proxy,
                                    NULL);
    if (ret < 0)
        CRITICAL ("Failed to create the dbus proxy to : %s [error : %s]",
                  name, strerror (-ret));
}

static void
dnsmasq_name_vanished (GDBusConnection *connection,
                       const gchar     *name,
                       gpointer         userdata)
{
    DnsmasqProxy *pxy;
    (void) connection;

    INFO("Name %s does not exist anymore", name);

    pxy = userdata;
    return_if_fail (pxy);
    if (pxy->proxy) {
        INFO ("Cleaning up the proxy for interface : \"%s\"",
              g_dbus_proxy_get_interface_name (pxy->proxy));
        g_object_unref (pxy->proxy);
        pxy->proxy = NULL;
    }
}

static int
dnsmasq_getused_port (unsigned int *port)
{
    GList *temp;
    unsigned int pt = 0;

    return_val_if_fail (port, -EINVAL);

    pt = ++dnsport;

    if (pt >= DNSMASQ_PORT_MAX)
        pt = dnsport = DNSMASQ_PORT_MIN;

    while ((temp = g_list_find (usedports, GUINT_TO_POINTER (pt)))) {
        pt = ++dnsport;
        if (pt >= DNSMASQ_PORT_MIN)
            return -ENOENT;
    }

    usedports = g_list_append (usedports, GUINT_TO_POINTER (pt));
    *port = pt;

    DEBUG ("Next unsed port : %d", pt);
    return 0;
}

static void
dnsmasq_cname_appeared (GDBusConnection *connection,
                        const gchar     *name,
                        const gchar     *name_owner,
                        gpointer         userdata)
{
    int ret;
    DnsMasqInstance *dnsmasq;

    INFO ("Name %s is owned by %s", name, name_owner);

    dnsmasq = userdata;
    return_if_fail (dnsmasq && dnsmasq->proxy);
    ret = dnsmasq_initialize_proxy (name,
                                    dnsmasq->proxy->busname,
                                    DNSMASQ_OBJECT_PATH,
                                    connection,
                                    &dnsmasq->proxy->proxy,
                                    dnsmasq);
    if (ret < 0)
        CRITICAL ("Failed to create the dbus proxy to : %s [error : %s]",
                  name, strerror (-ret));
}

static void
dnsmasq_cname_vanished (GDBusConnection *connection,
                        const gchar     *name,
                        gpointer         userdata)
{
    DnsMasqInstance *dnsmasq;
    (void) connection;

    INFO("Name %s does not exist anymore", name);

    dnsmasq = userdata;
    return_if_fail (dnsmasq && dnsmasq->proxy);
    if (dnsmasq->proxy->proxy) {
        INFO ("Cleaning up the proxy for interface : \"%s\"",
              g_dbus_proxy_get_interface_name (dnsmasq->proxy->proxy));
        g_object_unref (dnsmasq->proxy->proxy);
        dnsmasq->proxy->proxy = NULL;
    }
}

static int
dnsmasq_create_instance (const char *interface,
                         const char *router,
                         const char *subnet,
                         const char *start_ip,
                         const char *end_ip,
                         const char *broadcast,
                         char **nameservers,
                         char **ntpservers,
                         unsigned int lease_time,
                         gboolean dnssupport,
                         DnsMasqInstance **dnsmasq)
{
    DnsMasqInstance *dm;
    int ret = -ENOMEM;
    unsigned int port = 0;

    return_val_if_fail (dnsmasq, -EINVAL);
    return_val_if_fail (!*dnsmasq, -EEXIST);

    dm = g_try_malloc0 (sizeof (*dm));
    return_val_if_fail (dm, ret);

    dm->dnssupport = dnssupport;
    dm->ifname = g_strdup (interface);

    dm->dnsoptions = g_try_malloc0 (sizeof (*dm->dnsoptions));
    if (!dm->dnsoptions)
        goto error;

    dm->ctx = firewall_create_context ();
    if (!dm->ctx)
        goto servers_cleanup;

    if (dnssupport) {
        dm->port = g_try_malloc0 (sizeof (DnsPort));
        if (!dm->port)
            goto firewall_cleanup;

        ret = dnsmasq_getused_port (&dm->port->alternateport);
        if (ret < 0) {
            g_free (dm->port);
            goto firewall_cleanup;
        }
    } else {
    	ret = dnsmasq_getused_port (&port);
    	if (ret < 0)
    		goto firewall_cleanup;
    }

    dm->proxy = dnsmasq_proxy_init (DNSMASQ_BUS_NAME,
                                    DNSMASQ_PROXY_UNSHARED);
    if (!dm->proxy) {
        if (dnssupport)
            g_free (dm->port);
        goto firewall_cleanup;
    }

    dm->proxy->busname =
            g_strdup_printf (DNSMASQ_DBUS_SERVICENAME, interface);
    dm->proxy->proxywatch =
            g_bus_watch_name (G_BUS_TYPE_SYSTEM,
                              dm->proxy->busname,
                              G_BUS_NAME_WATCHER_FLAGS_NONE,
                              dnsmasq_cname_appeared,
                              dnsmasq_cname_vanished,
                              dm, NULL);
    if (dm->proxy->proxywatch == 0) {
        if (dnssupport)
            g_free (dm->port);
        dnsmasq_proxy_cleanup (dm->proxy);
        goto firewall_cleanup;
    }

    dm->dhcprange = g_try_malloc0 (sizeof (*dm->dhcprange));
    if (!dm->dhcprange) {
        if (dnssupport)
            g_free (dm->port);
        dnsmasq_proxy_cleanup (dm->proxy);
        goto firewall_cleanup;
    }

    dm->dhcprange->start = g_strdup (start_ip);
    dm->dhcprange->end = g_strdup (end_ip);
    dm->dhcprange->router = g_strdup (router);
    dm->dhcprange->subnet = g_strdup (subnet);
    dm->dhcprange->broadcast = g_strdup (broadcast);
    dm->dhcprange->leasetime = lease_time;
    dm->conffile = g_strdup_printf (DNSMASQ_CONFILE, dm->ifname);
    dm->hostfile = g_strdup_printf (DNSMASQ_HOSTFILE, dm->ifname);
    dm->pidfile = g_strdup_printf (DNSMASQ_PIDFILE, dnssupport ?
    		dm->port->alternateport : port);
    dm->dnsoptions->dns_servers = g_strdupv (nameservers);
    dm->dnsoptions->ntp_servers = g_strdupv (ntpservers);

    *dnsmasq = dm;
    return 0;

firewall_cleanup:
    firewall_cleanup_context (dm->ctx);
servers_cleanup:
    g_free (dm->dnsoptions);
error:
    g_free (dm->ifname);
    g_free (dm);
    return ret;
}

static void
dnsmasq_cmdline_destroy (dnsmasqcmdline *cmdline)
{
    g_ptr_array_free (cmdline->array, TRUE);
    g_string_chunk_free (cmdline->chunk);
    g_slice_free (dnsmasqcmdline, cmdline);
}

static dnsmasqcmdline*
dnsmasq_cmdline_new (void)
{
    dnsmasqcmdline *cmdline;
    cmdline = g_slice_new (dnsmasqcmdline);
    cmdline->array = g_ptr_array_new ();
    cmdline->chunk = g_string_chunk_new (1024);
    return cmdline;
}

static int
dnsmasq_cmd_line_add_string (dnsmasqcmdline *cmdline,
                             const char *str)
{
    return_val_if_fail (cmdline, -EINVAL);
    g_ptr_array_add (cmdline->array, g_string_chunk_insert (cmdline->chunk, str));
    return 0;
}

static int
dnsmasq_cmd_line_add_option (dnsmasqcmdline *cmdline,
                             const char* format,
                             ...)
{
    va_list args;
    char str [BUFSIZ];

    return_val_if_fail (cmdline, -EINVAL);

    va_start (args, format);
    if (vsnprintf (str, sizeof(str), format, args) > 0)
        dnsmasq_cmd_line_add_string (cmdline, str);

    va_end (args);
    return 0;
}

static char *
dnsmasq_cmd_line_to_string (dnsmasqcmdline *cmdline)
{
    char *str;

    g_ptr_array_add (cmdline->array, NULL);
    str = g_strjoinv (" ", (gchar **) cmdline->array->pdata);
    g_ptr_array_remove_index (cmdline->array, cmdline->array->len - 1);

    return str;
}

static void
dnsmasq_add_dns_options (dnsmasqcmdline *cmd,
                         const char *option,
                         char **servers)
{
    size_t length = 0, index = 0;

    return_if_fail (cmd && servers && option);

    length = g_strv_length (servers);
    for ( ; index < length; index++)
        dnsmasq_cmd_line_add_option (cmd, "%s=%s:%s,%s", "--dhcp-option",
                                     "option", option, servers [index]);
}

static int
dnsmasq_cmd_line (DnsMasqInstance *dnsmasq,
                  dnsmasqcmdline **cmd)
{
    int ret;
    char *path = NULL;
    unsigned int dnsport = 0;
    dnsmasqcmdline *cmdline;

    return_val_if_fail (dnsmasq && cmd, -EINVAL);
    return_val_if_fail (!*cmd, -EEXIST);

    ret = dnsmasq_find_executable ("dnsmasq", &path);
    if (ret < 0)
        return -ENOENT;

    cmdline = dnsmasq_cmdline_new ();

    /* dnsmasq exe absoulte path */
    dnsmasq_cmd_line_add_option (cmdline, "%s", path);

    /* extra logging for DHCP */
    dnsmasq_cmd_line_add_option (cmdline, "%s", "--log-dhcp");

    if (dnsmasq->dnssupport)
        /* Log the results of DNS queries handled by dnsmasq */
        dnsmasq_cmd_line_add_option (cmdline, "%s", "--log-queries");

    /* By default dnsmasq reads the configuration file from its
     * standard path (/etc/dnsmasq/dnsmasq.conf) and which may
     * cause undesirable side-effects and which we obviously
     * dont want*/
    dnsmasq_cmd_line_add_option (cmdline, "%s=%s", "--conf-file",
                                 dnsmasq->conffile);

    /* Server only on the intended interface */
    dnsmasq_cmd_line_add_option (cmdline, "%s=%s", "--interface",
                                 dnsmasq->ifname);

    /* Lease file i.e., dnsmasq's db sort of thing
     * which has history about the offered leases */
    dnsmasq_cmd_line_add_option (cmdline, "%s="DNSMASQ_TEMP_LEASEFILE,
                                 "--dhcp-leasefile", dnsmasq->ifname);

    /* Don't read the hostnames in /etc/hosts */
    dnsmasq_cmd_line_add_option (cmdline, "%s", "--no-hosts");

    /* Do not go into the background at startup but otherwise
     * run as normal. This is intended for use when dnsmasq is
     * run under daemontools like us */
    dnsmasq_cmd_line_add_option (cmdline, "%s", "--keep-in-foreground");

    /* bind only to the interfaces meant for */
    dnsmasq_cmd_line_add_option (cmdline, "%s", "--bind-interfaces");

    /* Whenever /etc/resolv.conf is re-read or the upstream servers
     * are set via DBus, clear the DNS cache. This is useful when new
     * nameservers may have different data than that held in cache. */
    dnsmasq_cmd_line_add_option (cmdline, "%s", "--clear-on-reload");

    /* Operating DHCP range */
    dnsmasq_cmd_line_add_option (cmdline, "%s=%s,%s,%s,%s,%d", "--dhcp-range",
                                 dnsmasq->dhcprange->start,
                                 dnsmasq->dhcprange->end,
                                 dnsmasq->dhcprange->subnet,
                                 dnsmasq->dhcprange->broadcast,
                                 dnsmasq->dhcprange->leasetime);

    /* Alternate path to log its pid */
    dnsmasq_cmd_line_add_option (cmdline, "%s=%s", "--pid-file",
                                 dnsmasq->pidfile);

    /* Dhcp hosts file carries information about the various
     * special hosts wherein we have to push some options
     * different from the default behaviour */
    dnsmasq_cmd_line_add_option (cmdline, "%s=%s", "--dhcp-hostsfile",
                                 dnsmasq->hostfile);

    /* Enable the dbus support with the customized name */
    dnsmasq_cmd_line_add_option (cmdline, "%s=%s", "--enable-dbus",
                                 dnsmasq->proxy->busname);

    /* alternate dns port */
    if (dnsmasq->dnssupport)
        dnsport = dnsmasq->port->alternateport;

    dnsmasq_cmd_line_add_option (cmdline, "%s=%d", "--port", dnsport);

    /* Dnsmasq is definitely the only DHCP server on this network.
     * This allows new hosts to get a lease without a tedious timeout
     * under all circumstances. */
    dnsmasq_cmd_line_add_option (cmdline, "%s", "--dhcp-authoritative");

    if (dnsmasq->dhcprange->router)
        dnsmasq_cmd_line_add_option (cmdline, "%s=%s:%s,%s", "--dhcp-option",
                                     "option", "router", dnsmasq->dhcprange->router);

    if (dnsmasq->dnsoptions) {
        dnsmasq_add_dns_options (cmdline, "dns-server", dnsmasq->dnsoptions->dns_servers);
        dnsmasq_add_dns_options (cmdline, "ntp-server", dnsmasq->dnsoptions->ntp_servers);
    }

    *cmd = cmdline;
    g_free (path);
    return 0;
}

static void
dnsmasq_exit_code (int status)
{
    char *msg = "Unknown error";

    /* DNSMASQ exit codes */
    switch (status) {
    case 1:
        msg = "Configuration problem";
        break;
    case 2:
        msg = "Network access problem (address in use; permissions; etc)";
        break;
    case 3:
        msg = "Filesystem problem (missing file/directory; permissions; etc)";
        break;
    case 4:
        msg = "Memory allocation failure";
        break;
    case 5:
        msg = "Other problem";
        break;
    default:
        if (status >= 11)
            msg = "Lease-script 'init' process failure";
        break;
    }

    INFO ("dnsmasq exited with error: %s (%d)", msg, status);
}

static gboolean
dnsmasq_remove_source (unsigned int *id)
{
    if (id && *id) {
        g_source_remove (*id);
        *id = 0;
        return TRUE;
    }
    return FALSE;
}

static void
dnsmasq_watch_cb (GPid pid,
                  int status,
                  gpointer userdata)
{
    int ret, err;
    clientdata *data;
    DnsMasqInstance *dnsmasq = userdata;

    INFO ("Dnsmasq exited for the interface %s with pid [%ld]",
          dnsmasq->ifname, (long) pid);

    if (WIFEXITED (status)) {
        INFO ("Dnsmasq Exited!!");
        err = WEXITSTATUS (status);
        if (err != 0)
            dnsmasq_exit_code (err);
    } else if (WIFSTOPPED (status)) {
        INFO ("dnsmasq stopped unexpectedly with signal %d",
              WSTOPSIG (status));
    } else if (WIFSIGNALED (status)) {
        INFO ("dnsmasq died with signal %d", WTERMSIG (status));
    } else {
        INFO ("dnsmasq died from an unknown cause");
    }

    /* remove the gsource */
    dnsmasq_remove_source (&(dnsmasq->watch));

    /* unlink the pid file */
    ret = unlink (dnsmasq->pidfile);
    if (ret < 0)
        ERROR ("Failed to unlink the pid file of ifname [%s] : %s",
               dnsmasq->ifname, strerror (errno));

    data = dnsmasq->data;
    g_hash_table_remove (dnsmasq_manager, dnsmasq->ifname);

    if (data) {
        if (data->callback)
            data->callback (data->ifname, 0,
                            NULL, data->userdata);

        dnsmasq_cleanup_clientdata (data);
    }
}

static int
dnsmasq_reap_child (pid_t pid,
                    int signal)
{
    int status = 0,
            err, errsv;
    pid_t ret;

    /* already terminated? */
    ret = waitpid (pid, &status, WNOHANG);
    if (ret > 0) {
        INFO ("process %ld already terminated", (long) ret);
        dnsmasq_exit_code (status);
        return 0;
    }
    else if (ret != 0) {
        errsv = errno;
        /* ECHILD means, the process is not a child/does not exist or
         * it has SIGCHILD blocked. */
        if (errsv != ECHILD) {
            INFO ("Error while waitpid for the process [%ld] error : %s",
                  (long) pid, strerror (errno));
            return -errsv;
        }
    }

    /* send the signal */
    err = kill (pid, signal);
    /* Failed to send the signal */
    if (err != 0) {
        errsv = errno;
        /* ESRCH means, process does not exist or is already a zombie. */
        if (errsv != ESRCH) {
            ERROR ("Error occured while sending the signale [%d] to the pid [%ld]"
                   "error : %s", signal, (long) pid, strerror (errno))
            return -errsv;
        }

        /* let's try again with waitpid, probably there was a race... */
        ret = waitpid (pid, &status, 0);
        if (ret > 0) {
            INFO ("process %ld already terminated", (long) ret);
            dnsmasq_exit_code (status);
        } else {
            errsv = errno;
            ERROR ("Error occured while sending the signale [%d] to the pid [%ld]"
                   "error : %s", signal, (long) pid, strerror (errno))
        }
        return -errsv;
    }

    return err;
}

static gboolean
dnsmasq_stop_cb (gpointer data)
{
    clientdata *clidata = data;

    return_val_if_fail (clidata, G_SOURCE_REMOVE);
    if (clidata->callback)
        clidata->callback (clidata->ifname, 0 ,
                        NULL, clidata->userdata);

    dnsmasq_cleanup_clientdata (clidata);
    return G_SOURCE_REMOVE;
}

static int
dnsmasq_dhcpserver_stop (const char *ifname,
                         dhcp_server_callback cb,
                         void *userdata)
{
    int ret;
    DnsMasqInstance *dnsmasq;
    clientdata *data;

    return_val_if_fail (ifname, -EINVAL);

    DEBUG ("Stop dnsmasq server for the interface : %s",
           ifname);

    data = g_try_malloc0 (sizeof (*data));
    return_val_if_fail (data, -ENOMEM);
    data->ifname = g_strdup (ifname);
    data->callback = cb;
    data->userdata = userdata;

    /* just fake */
    if (CAP_KILL_EFFECTIVE (CAP_KILL) != 1) {
        g_timeout_add_seconds (0, dnsmasq_stop_cb, data);
        return 0;
    }

    dnsmasq = g_hash_table_lookup (dnsmasq_manager, ifname);
    if (!dnsmasq) {
        dnsmasq_cleanup_clientdata (data);
        return -ENOENT;
    }

    if (dnsmasq->pid) {
        ret = dnsmasq_reap_child (dnsmasq->pid, SIGKILL);
        if (ret < 0) {
            ERROR ("Failed to kill the dnsmasq process [%ld] of ifname : %s"
                   "error : %s", dnsmasq->pid, dnsmasq->ifname,
                   strerror (-ret));
            return ret;
        }
    }

    dnsmasq->data = data;
    return 0;
}

static int
dnsmasq_dhcpserver_start (const char *interface,
                          const char *router,
                          const char *subnet,
                          const char *start_ip,
                          const char *end_ip,
                          const char *broadcast,
                          char **nameservers,
                          char **ntpservers,
                          gboolean dnssupport,
                          unsigned int lease_time)
{
    int ret;
    gboolean start = FALSE;
    GError *error = NULL;
    DnsMasqInstance *dnsmasq;
    dnsmasqcmdline *cmdline = NULL;
    char *cmdstring;

    INFO ("Interface: \"%s\", Router: \"%s\", "
          "subnet: \"%s\", start_ip: \"%s\", "
          "end_ip: \"%s\", broadcast: \"%s\", "
          "lease_time: %d",
          interface, router, subnet, start_ip,
          end_ip, broadcast, lease_time);

    /* just fake */
    if (CAP_KILL_EFFECTIVE (CAP_KILL) != 1)
        return 0;

    dnsmasq = g_hash_table_lookup (dnsmasq_manager, interface);
    return_val_if_fail (!dnsmasq, -EALREADY);

    ret = dnsmasq_create_instance (interface,
                                   router,
                                   subnet,
                                   start_ip,
                                   end_ip,
                                   broadcast,
                                   nameservers,
                                   ntpservers,
                                   lease_time,
                                   dnssupport,
                                   &dnsmasq);
    if (ret < 0)
        return ret;

    ret = dnsmasq_cmd_line (dnsmasq, &cmdline);
    if (ret < 0)
        goto dnsmasqcleanup;

    g_ptr_array_add (cmdline->array, NULL);

    INFO ("Starting dnsmasq for the interface : %s", interface);
    INFO ("Cmdline : %s", (cmdstring = dnsmasq_cmd_line_to_string (cmdline)));

    /* Spawing a dnsmasq process which will bind to the
     * particular interface and only serve those requests
     * from that particular interface. This is particular
     * required when dnsmasq is supposed to operate with
     * different subnets and this is not possible if dont
     * restart the dnsmasq process (dnsmasq while startup
     * needs root privileges, once everything is setup it
     * drops its root privileges and changes to nobody and
     * dnsmasq does not reads the configuration file if it
     * changes dynamically).
     *
     * We dont let glib automatically reap the child as we
     * want to know the exit status of the child.
     *
     * Also the child will stay within the process group
     * of the parent and thus will be killed once the
     * parent exits */
    start = g_spawn_async (NULL, (char **) cmdline->array->pdata,
                           NULL, G_SPAWN_DO_NOT_REAP_CHILD,
                           NULL, NULL,
                           &dnsmasq->pid, &error);

    if (!start) {
        ERROR ("Dnsmasq process cannot be started : %s", error->message);
        ret = -errno;
        goto dnsmasqcleanup;
    }

    /* Either wait/waitpid/waitid or g_child_watch_add has to
     * be performed to free up the child resources else
     * you know about zombies!! */
    dnsmasq->watch =
            g_child_watch_add (dnsmasq->pid,
                               (GChildWatchFunc) dnsmasq_watch_cb,
                               dnsmasq);
    dnsmasq_enable_fw (dnsmasq);
    dnsmasq_cmdline_destroy (cmdline);

    g_hash_table_replace (dnsmasq_manager, dnsmasq->ifname, dnsmasq);
    return 0;

dnsmasqcleanup:
    dnsmasq_cleanup ((void *) dnsmasq);
    return ret;
}

static int
dnsmasq_startup_flush ()
{
    int ret;
    pid_t pid = -1;
    char *args [2] = {
        DNSMASQ_BUS_NAME,
        NULL
    };

    ret = dnsmasq_get_pid ("dnsmasq", args, &pid);
    if (!ret) {
        /* not a parent and dont wait */
        ret = kill (pid, SIGKILL);
        if (ret < 0)
            ERROR ("Failed to terminate the already running dnsmasq : %s [pid : %ld]",
                  strerror (errno), (long) pid);
    }

    if (ret == -ENOENT)
        ret = 0;
    return ret;
}

static int
dnsmasq_get_proxy (const char *ifname,
                   DnsmasqProxy **dnspxy)
{
    DnsMasqInstance *dnsmasq;

    return_val_if_fail (ifname && dnspxy, -EINVAL);

    if (CAP_KILL_EFFECTIVE (CAP_KILL) != 1) {
        if (!proxy)
            return -ENOTCONN;
        *dnspxy = proxy;
    } else {
        dnsmasq = g_hash_table_lookup (dnsmasq_manager, ifname);
        return_val_if_fail (dnsmasq, -ENOENT);
        return_val_if_fail (dnsmasq->proxy, -ENOTCONN);
        *dnspxy = dnsmasq->proxy;
    }

    return 0;
}

static void
dnsmasq_clearcache_cb (GObject *object,
                       GAsyncResult *res,
                       gpointer user_data)
{
    int ret = 0;
    GVariant *result;
    GDBusProxy *dnspxy;
    clientdata *data = NULL;
    GError *error = NULL;

    DEBUG ("");

    dnspxy = (GDBusProxy *) object;
    return_if_fail (dnspxy);
    data = user_data;
    result = g_dbus_proxy_call_finish (dnspxy, res, &error);
    if (error) {
        ERROR ("\"ClearCache\" Call failed : %s", error->message);
        g_error_free (error);
        ret = -1;
    }

    return_if_fail (data);
    if (data->callback)
        data->callback (data->ifname, ret,
                        NULL, data->userdata);

    if (!ret)
        g_variant_unref (result);
    dnsmasq_cleanup_clientdata (data);
}

static int
dnsmasq_dhcpserver_clear_cache (const char *ifname,
                                dhcp_server_callback cb,
                                void *userdata)
{
    int ret;
    DnsmasqProxy *dnspxy = NULL;
    clientdata *data;

    return_val_if_fail (ifname && cb, -EINVAL);

    DEBUG ("%s", ifname);

    ret = dnsmasq_get_proxy (ifname, &dnspxy);
    if (ret < 0)
        return ret;

    data = g_try_malloc0 (sizeof (*data));
    return_val_if_fail (data, -ENOMEM);
    data->ifname = g_strdup (ifname);
    data->callback = cb;
    data->userdata = userdata;

    g_dbus_proxy_call (dnspxy->proxy,
                       "ClearCache",
                       NULL,
                       G_DBUS_CALL_FLAGS_NONE,
                       -1, NULL,
                       dnsmasq_clearcache_cb,
                       data);

    return 0;
}

static void
dnsmasq_setservers_cb (GObject *object,
                       GAsyncResult *res,
                       gpointer user_data)
{
    int ret = 0;
    GVariant *result;
    GDBusProxy *dnspxy;
    clientdata *data = NULL;
    GError *error = NULL;

    DEBUG ("");

    dnspxy = (GDBusProxy *) object;
    return_if_fail (dnspxy);
    data = (clientdata *) user_data;
    result = g_dbus_proxy_call_finish (dnspxy, res, &error);
    if (error) {
        ERROR ("\"SetServers\" Call failed : %s", error->message);
        g_error_free (error);
        ret = -1;
    }

    return_if_fail (data);
    if (data->callback)
        data->callback (data->ifname, ret,
                        NULL, data->userdata);

    if (!ret)
        g_variant_unref (result);
    dnsmasq_cleanup_clientdata (data);
}

static int
dnsmasq_dhcpserver_set_servers (const char *ifname,
                                GList *servers,
                                dhcp_server_callback cb,
                                void *userdata)
{
    int ret;
    char *server;
    DnsmasqProxy *dnspxy = NULL;
    clientdata *data;
    GList *temp;
    GVariantBuilder builder;
    GVariant *variant;

    return_val_if_fail (ifname && cb, -EINVAL);

    DEBUG ("%s", ifname);

    ret = dnsmasq_get_proxy (ifname, &dnspxy);
    if (ret < 0)
        return ret;

    data = g_try_malloc0 (sizeof (*data));
    return_val_if_fail (data, -ENOMEM);
    data->ifname = g_strdup (ifname);
    data->callback = cb;
    data->userdata = userdata;

    temp = servers;
    g_variant_builder_init (&builder, G_VARIANT_TYPE ("av"));

    for ( ; temp; temp = temp->next) {
        server = (char *) temp->data;
        if (server)
            g_variant_builder_add (&builder, "v", g_variant_new_string (server));
    }

    variant = g_variant_builder_end (&builder);
    g_dbus_proxy_call (dnspxy->proxy,
                       "SetServers",
                       g_variant_new ("(@av)",
                                      variant),
                       G_DBUS_CALL_FLAGS_NONE,
                       -1, NULL,
                       dnsmasq_setservers_cb,
                       data);

    return 0;
}

static void
dnsmasq_addlease_cb (GObject *object,
                     GAsyncResult *res,
                     gpointer user_data)
{
    int ret = 0;
    GVariant *result;
    GDBusProxy *dnspxy;
    clientdata *data = NULL;
    GError *error = NULL;

    DEBUG ("");

    dnspxy = (GDBusProxy *) object;
    return_if_fail (dnspxy);
    data = (clientdata *) user_data;
    result = g_dbus_proxy_call_finish (dnspxy, res, &error);
    if (error) {
        ERROR ("\"AddDhcpLease\" Call failed : %s", error->message);
        g_error_free (error);
        ret = -1;
    }

    return_if_fail (data);
    if (data->callback)
        data->callback (data->ifname, ret,
                        NULL, data->userdata);

    if (!ret)
        g_variant_unref (result);
    dnsmasq_cleanup_clientdata (data);
}

static int
dnsmasq_dhcpserver_add_lease (const char *ifname,
                              const char *ipaddress,
                              const char *macaddress,
                              const char *hostname,
                              const char *clientid,
                              unsigned int leaseduration,
                              unsigned int iaid,
                              gboolean istemporary,
                              dhcp_server_callback cb,
                              void *userdata)
{
    int ret;
    DnsmasqProxy *dnspxy = NULL;
    clientdata *data;

    return_val_if_fail (ifname && cb, -EINVAL);

    DEBUG ("%s", ifname);

    ret = dnsmasq_get_proxy (ifname, &dnspxy);
    if (ret < 0)
        return ret;

    data = g_try_malloc0 (sizeof (*data));
    return_val_if_fail (data, -ENOMEM);
    data->ifname = g_strdup (ifname);
    data->callback = cb;
    data->userdata = userdata;

    g_dbus_proxy_call (dnspxy->proxy,
                       "AddDhcpLease",
                       g_variant_new ("(ss^ay^ayuub)",
                                      ipaddress,
                                      macaddress,
                                      hostname,
                                      clientid,
                                      leaseduration,
                                      iaid,
                                      istemporary),
                       G_DBUS_CALL_FLAGS_NONE,
                       -1, NULL,
                       dnsmasq_addlease_cb,
                       data);

    return 0;
}

static void
dnsmasq_rmlease_cb (GObject *object,
                    GAsyncResult *res,
                    gpointer user_data)
{
    int ret = 0;
    GVariant *result;
    GDBusProxy *dnspxy;
    clientdata *data = NULL;
    GError *error = NULL;

    DEBUG ("");

    dnspxy = (GDBusProxy *) object;
    return_if_fail (dnspxy);
    data = (clientdata *) user_data;
    result = g_dbus_proxy_call_finish (dnspxy, res, &error);
    if (error) {
        ERROR ("\"DeleteDhcpLease\" Call failed : %s", error->message);
        g_error_free (error);
        ret = -1;
    }

    return_if_fail (data);
    if (data->callback)
        data->callback (data->ifname, ret,
                        NULL, data->userdata);

    if (!ret)
        g_variant_unref (result);
    dnsmasq_cleanup_clientdata (data);
}

static int
dnsmasq_dhcpserver_remove_lease (const char *ifname,
                                 const char *ipaddress,
                                 dhcp_server_callback cb,
                                 void *userdata)
{
    int ret;
    DnsmasqProxy *dnspxy = NULL;
    clientdata *data;

    return_val_if_fail (ifname && cb, -EINVAL);

    DEBUG ("%s", ifname);

    ret = dnsmasq_get_proxy (ifname, &dnspxy);
    if (ret < 0)
        return ret;

    data = g_try_malloc0 (sizeof (*data));
    return_val_if_fail (data, -ENOMEM);
    data->ifname = g_strdup (ifname);
    data->callback = cb;
    data->userdata = userdata;

    g_dbus_proxy_call (dnspxy->proxy,
                       "DeleteDhcpLease",
                       g_variant_new ("(s)",
                                      ipaddress),
                       G_DBUS_CALL_FLAGS_NONE,
                       -1, NULL,
                       dnsmasq_rmlease_cb,
                       data);

    return 0;
}

static dhcpserver_ops
server_ops = {
    .clientname = "DNSMASQ",
    .dhcp_server_start = dnsmasq_dhcpserver_start,
    .dhcp_server_stop = dnsmasq_dhcpserver_stop,
    .dhcp_server_clear_cache = dnsmasq_dhcpserver_clear_cache,
    .dhcp_server_set_servers = dnsmasq_dhcpserver_set_servers,
    .dhcp_server_add_lease = dnsmasq_dhcpserver_add_lease,
    .dhcp_server_remove_lease = dnsmasq_dhcpserver_remove_lease,
};

int
dnsmasq_init ()
{
    int ret;

    DEBUG ("");

    ret = dhcp_server_driver_register (&server_ops);
    if (ret < 0) {
        ERROR ("Failed to register the DHCP server driver");
        return ret;
    }

    dnsmasq_manager = g_hash_table_new_full (g_str_hash, g_str_equal,
                                             NULL, dnsmasq_cleanup);

    /* cleanup all the dnsmasq processes while we
     * start up. This is bit awkward to do but it is with
     * the assumption that this wifi_ap_direct_manager
     * is the only component which manages the dnsmasq process.
     * Revisit (remove) this if there is an another guy who wants
     * the dnsmasq process */
    if (CAP_KILL_EFFECTIVE (CAP_KILL) == 1)
        return dnsmasq_startup_flush();

    /* If the process does not have CAP_KILL capability
     * required for the dynamic management we assume that
     * the dnsmasq process is/will be started with the
     * required configurations and thus we share the proxy
     * across the dnsmasq instances */
    proxy = dnsmasq_proxy_init (DNSMASQ_BUS_NAME,
                                DNSMASQ_PROXY_SHARED);
    if (!proxy)
        return -ENOMEM;

    INFO ("Does not have CAP_KILL capabilities, will be using dnsmasq as it is!!");
    proxy->proxywatch = g_bus_watch_name (G_BUS_TYPE_SYSTEM,
                                          proxy->busname,
                                          G_BUS_NAME_WATCHER_FLAGS_NONE,
                                          dnsmasq_name_appeared,
                                          dnsmasq_name_vanished,
                                          proxy, NULL);

    if (proxy->proxywatch == 0) {
        dnsmasq_proxy_cleanup (proxy);
        proxy = NULL;
        return -EIO;
    }
    return 0;
}

int
dnsmasq_deinit ()
{
    int ret;

    DEBUG ("");

    ret = dhcp_server_driver_unregister (&server_ops);
    if (ret < 0)
        ERROR ("Failed to register the DHCP server driver");

    dnsmasq_proxy_cleanup (proxy);
    g_hash_table_destroy (dnsmasq_manager);
    g_list_free (clients);
    return 0;
}

/** @} */
