/**
 * @file database.c
 * @author RBEI/ECO32 Jiji Anna Jacob
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_mw\wifi_ap_direct_manager
 *
 * @brief DB Handling for Wifi_AP_Direct_Manager
 *	
 * @{
 */

#include <glib.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <sys/inotify.h>
#include <sys/types.h>
#include <dirent.h>
#include <database.h>
#include <crypto.h>
#include <ap.h>
#include <log.h>
#include <db_defines.h>

/**
 * db_destroy_temp_file
 *@pathname: Path to file to be destroyed
 * 
 * Destroys file located @pathname
 * Returns: 0 on success,
 * -errno on failure
 *
 **/
int 
db_destroy_temp_file (const char *pathname)
{
    int ret;

    return_val_if_fail (pathname, -EINVAL);

    DEBUG ("Removing file: %s", pathname);

    ret = unlink (pathname);
    if (ret < 0) {
        ERROR ("Failed to delete the file \"%s\", error : %s",
               pathname, strerror (errno));
        return -errno;
    }

    return ret;
}

/**
 * db_load_key_file
 *@path: Path to conf file 
 * 
 * Converts txt file @path to a 
 * GKeyFile
 * Returns: GKeyFile
 *
 **/
GKeyFile*
db_load_key_file (const char* path)
{
    GKeyFile *gkf = NULL;
    GError *error = NULL;

    return_val_if_fail (path, NULL);

    DEBUG ("Loading keyfile from : %s", path);

    gkf = g_key_file_new ();
    if (!g_key_file_load_from_file (gkf, path, 0, &error)) {
        DEBUG ("Unable to load GKeyFile from the file [\"%s\"], "
               "error: %s", path, error->message);
        g_error_free (error);
        g_key_file_free (gkf);
        gkf = NULL;
    }

    return gkf;
}
/**
 * db_exists
 *@ifname: 
 *@file:
 * Checks if conf file exists for
 * an interface specified by "ifname"
 * Returns: 0 on success,
 * -errno on failure
 **/
int
db_exists (const char *ifname,
           char **file)
{
    DIR *dir;
    char filepath [PATH_MAX];
    struct dirent *ent;
    int found = 0;

    return_val_if_fail (ifname && file, -EINVAL);
    return_val_if_fail (!*file, -EEXIST);

    DEBUG ("Checking for the existence of the configuration"
           " file for the ap : %s", ifname);

    dir = opendir (WAPDMAN_DB_DIR);
    if (!dir) {
        ERROR ("Failed to open the dir \"%s\" : %s", WAPDMAN_DB_DIR, strerror (errno));
        return -errno;
    }

    memset (filepath, 0, sizeof (filepath));
    while ((ent = readdir (dir))) {

        if (!g_strcmp0 (ent->d_name, "."))
            continue;
        if (!g_strcmp0 (ent->d_name, ".."))
            continue;

        if (g_str_has_prefix (ent->d_name, ifname) &&
                g_str_has_suffix (ent->d_name, ".conf")) {
            sprintf (filepath, "%s/%s", WAPDMAN_DB_DIR, ent->d_name);
            found = 1;
            break;
        }
    }

    if (closedir (dir) < 0)
        ERROR ("Failed to close the dir %s: %s",
               WAPDMAN_DB_DIR, strerror (errno));

    if (found) {
        *file = g_strdup (filepath);
        return 0;
    }

    return -ENOENT;
}

/**
 * db_dump_file
 *@file: File that contains the data 
 * Dumps contents of file
 * Returns: void 
 *
 **/
static void
db_dump_file (const char *file)
{
    FILE *fp;
    char line [512];

    return_if_fail (file);

    fp = fopen (file, "r");
    if (!fp) {
        ERROR ("Failed to open the file [%s] : %s", file, strerror (errno));
        return;
    }

    memset (line, 0, sizeof (line));
    while (fgets (line, sizeof (line), fp)) {
    	line [strlen (line) - 1] = '\0';
        INFO ("%s", line);
        memset (line, 0, sizeof (line));
    }

    if (fclose (fp) < 0)
        ERROR ("Failed to close the file [%s] : %s", file, strerror (errno));
}
/**
 * db_create_temp_file
 * @ ap : Access point conf params
 * @path:
 * @keyfile:
 * @keys[]:
 * @groupid:
 * Creates a temp plain text file
 * with hostapd conf params. 
 * Returns: 0 on success,
 * -errno on failure
 *
 **/
int
db_create_temp_file (struct access_point *ap,
                     char **path,
                     GKeyFile *keyfile,
                     char *keys[],
                     const char *groupid)
{
    size_t size;
    char tempfile [PATH_MAX];
    char configuration [PATH_MAX];
    int ret = 0, fd, error = 0;
    char *ifname, **temp, *value;

    return_val_if_fail (ap && path && keyfile, -EINVAL);
    return_val_if_fail (!*path, -EEXIST);
    return_val_if_fail (groupid, -ENOKEY);

    ifname = access_point_get_interface (ap);
    return_val_if_fail (ifname, -ENODATA);
    memset (tempfile, 0, sizeof (tempfile));
    sprintf (tempfile, "%s/%s.conf", WAPDMAN_TEMP_CONF_DIR, ifname);

    fd = open (tempfile, O_CLOEXEC | O_CREAT | O_RDWR, S_IRUSR | S_IWUSR
               | S_IRGRP | S_IROTH);
    if (fd < 0) {
        ERROR ("Failed to open the file \"%s\", error : %s", tempfile,
               strerror(errno));
        return -errno;
    }

    temp = keys;
    for ( ; *temp; temp++) {
        memset (configuration, 0, sizeof (configuration));
        if (strcmp(*temp, "wpa_passphrase") == 0) {
        	value = g_key_file_get_string (keyfile, groupid, *temp, NULL);
        } else
        	value = g_key_file_get_value (keyfile, groupid, *temp, NULL);
        
        if (!value)
            continue;

        sprintf (configuration, "%s=%s", *temp, value);
        size = strlen (configuration);
        configuration [size] = '\n';
        ret = (int) write (fd, configuration, strlen (configuration));
        if (ret < 0) {
            CRITICAL ("Failed to write the configuration [%s] to file %s [%s]",
                      configuration, tempfile, strerror (errno));
            error = 1;
            g_free (value);
            ret = -errno;
            break;
        }

        g_free (value);
    }

    if (close (fd) < 0)
        ERROR ("Failed to close the temp file : %s", tempfile);

    if (!error) {
        /* dump for debugging purpose */
        db_dump_file (tempfile);
        *path = g_strdup (tempfile);
        ret = 0;
    } else
        db_destroy_temp_file (tempfile);

    return ret;
}
/**
 * db_load
 * @ ap : Access point conf params
 * @ kfile: GKeyFile with key value pairs
 * Loads contents of GKeyFile into 
 * struct ap for AP conf params.
 * Returns: 0 on success,
 * -errno on failure
 *
 **/
int
db_load (struct access_point *ap,
         GKeyFile **kfile)
{
    int ret;
    char *ifname, *dbfile = NULL,
            tempfile [PATH_MAX];
    GKeyFile *keyfile = NULL;
    struct stat info;

    return_val_if_fail (ap && kfile, -EINVAL);
    return_val_if_fail (!*kfile, -EEXIST);

    ifname = access_point_get_interface (ap);
    ret = db_exists (ifname, &dbfile);
    if (ret < 0) {
        ERROR ("Failed to load the configuration for "
               "ap [\"%s\"] : %s", ifname, strerror (-ret));
        return ret;
    }

    ret = lstat (dbfile, &info);
    if (ret < 0) {
        ERROR ("lstat failed for [%s]: %s/%d", dbfile,
               strerror (errno), errno);
        g_free (dbfile);
        return -errno;
    }

    INFO ("Size of the db file: %lld", (long long) info.st_size);
    if (info.st_size == 0) {
        g_free (dbfile);
        return -ENOENT;
    }

    DEBUG ("Interface : %s", ifname);
    memset (tempfile, 0, sizeof (tempfile));
    sprintf (tempfile, "%s/%s_tmp.conf", WAPDMAN_TEMP_CONF_DIR,
             ifname);

    ret = decrypt_file (dbfile, tempfile);
    if (ret < 0) {
        ERROR ("Failed to decrypt the configuration file : %s [error : %s]",
              dbfile, strerror (-ret));
        g_free (dbfile);
        return ret;
    }

    db_dump_file (tempfile);
    keyfile = db_load_key_file (tempfile);
    if (!keyfile) {
        g_free (dbfile);
        return -EIO;
    }

    *kfile = keyfile;
    ret = db_destroy_temp_file (tempfile);
    if (ret < 0)
        ERROR ("Failed to remove the temp file %s [error : %s]",
               tempfile, strerror (-ret));

    g_free (dbfile);
    return ret;
}
/**
 * db_save_ap
 * @ interface:Access point conf params
 * @ keyfile:GKeyFile with key value pairs
 * @ pathname: Pathname for encryped file
 * Saves GKeyFile for AP conf params on the 
 * file system as an encrypted file.
 * Returns: 0 on success,
 * -errno on failure
 *
 **/
static int
db_save_ap (const char *interface,
            GKeyFile *keyfile,
            char *pathname)
{
    int ret;
    char dbfile [PATH_MAX];
    char *data = NULL;
    gsize length = 0;
    gboolean set = FALSE;
    GError *error = NULL;

    return_val_if_fail (interface, -ENOKEY);
    return_val_if_fail (keyfile && pathname, -EINVAL);

    data = g_key_file_to_data (keyfile, &length, NULL);
    set = g_file_set_contents (pathname, data, (gssize) length, &error);
    if (!set) {
        ERROR ("Failed to set the file contents for ap : %s "
               "[error : %s]", interface, error->message);
        g_free (data);
        g_error_free (error);
        return -EIO;
    }

    /* just for debugging */
    db_dump_file (pathname);
    memset (dbfile, 0 , sizeof (dbfile));
    sprintf (dbfile, "%s/%s.conf", WAPDMAN_DB_DIR, interface);
    ret = encrypt_file (pathname, dbfile);
    if (ret < 0)
        ERROR ("Failed to encrypt the file for ap [%s] : %s",
               interface, dbfile);

    g_free (data);
    ret = db_destroy_temp_file (pathname);
    if (ret < 0)
        ERROR ("Failed to remove the temp db file : %s [%s]",
               pathname, strerror (errno));

    return ret;
}
/**
 * db_save_ap_conf
 * @ ap:Access point conf params
 * @ keyfile:GKeyFile with key value pairs
 * @ pathname: Pathname for encryped file
 * Saves struct for AP conf params on the 
 * file system as an encrypted file.
 * Returns: 0 on success,
 * -errno on failure
 *
 **/
int
db_save_ap_conf (struct access_point *ap,
                 GKeyFile *keyfile)
{
    int ret = 0;
    char *interface, path [255];

    return_val_if_fail (ap && keyfile, -EINVAL);

    interface = access_point_get_interface (ap);
    DEBUG ("access point : %s", interface);
    memset (path, 0, sizeof (path));
    sprintf (path, "%s/%s_tmp.conf" , WAPDMAN_TEMP_CONF_DIR, interface);
    ret = db_save_ap (interface, keyfile, path);
    if (ret < 0)
        ERROR ("Error occured while saving the configuration file "
               "for ap : %s [error : %s]", interface, strerror (-ret));

    return ret;
}



