/**
 * @file task.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <string.h>
#include <errno.h>
#include <glib.h>
#include <gio/gio.h>
#include <task.h>
#include <time.h>
#include <log.h>
#include "inc/utils.h"

/*
 * We talk about the dbus requests here as thats the only
 * IPC currently supported by wifi_ap_direct_manager. The
 * default timeout period of the dbus method request is 25 (if
 * the clients have not modified it) and therefore as a safe
 * bet we consider a task (i.e., dbus request) has to be completed
 * within 24 seconds, otherwise the request shall timeout and a
 * appropriate error shall be returned to the requestor.
 */
#define TASK_DBUS_TIMEOUT   24

static unsigned int nextrequestid;

struct wapdman_task
{
    int priority;
    tasktype type;
    unsigned int id;
    int opcode;
    int ongoingtask;
    void *value;
    void *invocation;
    void *userdata;
    taskstate state;
    GSList *subtasks;
    struct timeval receive,
            request,
            complete;
    unsigned int timeout;
    timeout_func func;
};

const char*
task_type2string (tasktype type)
{
    switch (type) {
    case TASK_TYPE_AP:
        return ENUMTOSTR (AP);
    case TASK_TYPE_P2P:
        return ENUMTOSTR (P2P);
    default:
        return ENUMTOSTR (Unknown);
    }

    return ENUMTOSTR (Unknown);
}

const char*
task_state2string (taskstate state)
{
    switch (state) {
    case TASK_STATE_INVALID:
        return ENUMTOSTR (STATE_INVALID);
    case TASK_STATE_IDLE:
        return ENUMTOSTR (STATE_IDLE);
    case TASK_STATE_REQUESTED:
        return ENUMTOSTR (STATE_REQUESTED);
    case TASK_STATE_INTERRUPTED:
        return ENUMTOSTR (STATE_INTERRUPTED);
    case TASK_STATE_COMPLETED:
        return ENUMTOSTR (STATE_COMPLETED);
    }

    return ENUMTOSTR (STATE_INVALID);
}

int
task_set_state (struct wapdman_task *task,
                taskstate state)
{
    struct timeval tv;

    return_val_if_fail (task, -EINVAL);

    DEBUG ("Task [%p] old state: %s new state: %s", task,
           task_state2string (task->state), task_state2string (state));

    if (task->state == state)
        return -EALREADY;

    if (TASK_STATE_REQUESTED == state) {
        get_monotonic_time (&task->request);
        get_time_difference (&task->request, &task->receive, &tv);
        INFO ("Task [%p] requested at %lu.%06lu seconds [%lu.%06lu after "
              "its trigger]", task, task->request.tv_sec, task->request.tv_usec,
              tv.tv_sec, tv.tv_usec);
    }
    else if (TASK_STATE_COMPLETED == state) {
        get_monotonic_time (&task->complete);
        get_time_difference (&task->complete, &task->receive, &tv);
        INFO ("Task [%p] completed at %lu.%06lu seconds [%lu.%06lu after "
              "its trigger]", task, task->complete.tv_sec, task->complete.tv_usec,
              tv.tv_sec, tv.tv_usec);

        if (task->timeout > 0) {
            INFO ("Removing the timeout source [%u] for task: %p", task->timeout,
                  task);
            g_source_remove (task->timeout);
        }
        task->timeout = 0;
    }

    task->state = state;
    return 0;
}

int
task_set_ongoingoperation (struct wapdman_task *task,
                           int opcode)
{
    return_val_if_fail (task, -EINVAL);

    DEBUG ("Task [%p] original opcode: %d ongoing opcode: %d "
           "requested ongoing opcode: %d", task, task->opcode,
           task->ongoingtask, opcode);

    if (task->ongoingtask == opcode)
        return -EALREADY;

    task->ongoingtask = opcode;
    return 0;
}

int
task_get_ongoingoperation (struct wapdman_task *task)
{
    return_val_if_fail (task, -EINVAL);
    return task->ongoingtask;
}

int
task_get_state (struct wapdman_task *task)
{
    return_val_if_fail (task, -EINVAL);
    return task->state;
}

void*
task_get_invocation (struct wapdman_task *task)
{
    return_val_if_fail (task, NULL);
    return task->invocation;
}

int
task_get_opcode (struct wapdman_task *task)
{
    return_val_if_fail (task, -1);
    return task->opcode;
}

void*
task_get_data (struct wapdman_task *task)
{
    return_val_if_fail (task, NULL);
    return task->value;
}

static gint
task_compare_priority (gconstpointer a,
                       gconstpointer b)
{
    const struct wapdman_task *task1 = a;
    const struct wapdman_task *task2 = b;
    return task1->priority - task2->priority;
}

void
task_cleanup (struct wapdman_task *task)
{
	struct timeval tv;

    return_if_fail (task);

    DEBUG ("Cleaning up task: %p", task);

    if (task->timeout > 0) {
    	get_monotonic_time (&task->complete);
    	get_time_difference (&task->complete, &task->receive, &tv);
    	INFO ("Task [%p] completed at %lu.%06lu seconds [%lu.%06lu after its trigger]",
    			task, task->complete.tv_sec, task->complete.tv_usec, tv.tv_sec, tv.tv_usec);
    	INFO ("Removing the timeout source [%u] for task: %p", task->timeout, task);
    	g_source_remove (task->timeout);
    }
    task->timeout = 0;

    g_slist_free (task->subtasks);
    g_free (task);
}

struct wapdman_task*
task_find (GList **queue,
           tasktype type,
           int opcode)
{
    (void) type;
    GList *temp;
    struct wapdman_task *task;

    return_val_if_fail (queue, NULL);

    temp = *queue;
    for ( ; temp; temp = temp->next) {
        task = temp->data;
        if (task && task->opcode == opcode) {
            DEBUG ("opcode: %d task: %p", opcode, task);
            return task;
        }
    }

    return NULL;
}

static gboolean
task_completion_timeout (gpointer data)
{
    struct wapdman_task *task = data;
    taskstate state = task_get_state (task);

    INFO ("Timeout event for the task: %p [state: %s]", task,
          task_state2string (state));

    if (TASK_STATE_REQUESTED != state) {
        task->timeout = 0;
        if (task->func)
            task->func (task, task->userdata);
        return G_SOURCE_REMOVE;
    }

    return G_SOURCE_CONTINUE;
}

struct wapdman_task*
task_create (tasktype type,
             int opcode,
             int priority,
             void *invocation,
             void *value,
             void *userdata,
             timeout_func func)
{
    int create = 0;
    struct wapdman_task *task;
    const char *requestor, *object,
            *interface, *method;

    if (type == TASK_TYPE_AP || type == TASK_TYPE_P2P)
        create = 1;

    return_val_if_fail (create, NULL);
    task = g_try_malloc0 (sizeof (*task));
    return_val_if_fail (task, NULL);

    task->timeout = 0;
    task->id = nextrequestid++;
    task->type = type;
    task->opcode = opcode;
    task->priority = priority;
    task->userdata = userdata;
    task->value = value;
    task->invocation = invocation;
    task->state = TASK_STATE_IDLE;

    get_monotonic_time (&task->receive);

    INFO ("A new %s task [%p] created at %lu.%06lu seconds",
          invocation ? "external" : "internal", task,
          task->receive.tv_sec, task->receive.tv_usec);

    if (!invocation)
        goto complete;

    task->func = func;
    /* Only for external triggers */
    task->timeout = g_timeout_add_seconds (TASK_DBUS_TIMEOUT,
                                           task_completion_timeout,
                                           task);

    object = g_dbus_method_invocation_get_object_path (invocation);
    method = g_dbus_method_invocation_get_method_name (invocation);
    requestor = g_dbus_method_invocation_get_sender (invocation);
    interface = g_dbus_method_invocation_get_interface_name (invocation);

    DEBUG ("Requestor of the task [%p]: %s [invocation: %p] [timeout id: %u]",
           task, requestor, invocation, task->timeout);
    DEBUG ("Request details of the task [%p]: %s.%s [of object: %s]",
           task, interface, method, object);

complete:
    return task;
}

int
task_get_next_subtask (struct wapdman_task *task)
{
    gpointer data;

    return_val_if_fail (task, -EINVAL);

    data = g_slist_nth_data (task->subtasks, 0);
    if (!data) {
        DEBUG ("No subtaks available for the task : %p", task);
        return -ENODATA;
    }

    DEBUG ("Task [%p] next task id: %d", task, GPOINTER_TO_INT (data));
    return GPOINTER_TO_INT (data);
}

int
task_add_subtask (struct wapdman_task *task,
                  int opcode)
{
    return_val_if_fail (task, -EINVAL);
    DEBUG ("Task [%p] original opcode: %d subtask: %d", task, task->opcode, opcode);
    task->subtasks = g_slist_append (task->subtasks, GINT_TO_POINTER (opcode));
    return 0;
}

int
task_rm_subtask (struct wapdman_task *task,
                 int opcode)
{
    return_val_if_fail (task, -EINVAL);
    DEBUG ("Task [%p] original opcode: %d subtask: %d", task, task->opcode, opcode);
    task->subtasks = g_slist_remove (task->subtasks, GINT_TO_POINTER (opcode));
    return 0;
}

int
task_add_toqueue (GList **taskqueue,
                  struct wapdman_task *task)
{
    return_val_if_fail (taskqueue && task, -EINVAL);
    *taskqueue = g_list_insert_sorted (*taskqueue, task, task_compare_priority);
    return 0;
}


struct wapdman_task*
task_process_next (GList **queue)
{
    GList *temp;
    struct wapdman_task *task;

    return_val_if_fail (queue, NULL);

    temp = g_list_first (*queue);
    if (!temp)
        return NULL;

    task = temp->data;
    DEBUG ("Next task in the queue: %p", task);
    return task;
}

int
task_remove (GList **queue,
             struct wapdman_task *task)
{
    return_val_if_fail (queue && *queue, -EINVAL);
    *queue = g_list_remove (*queue, task);
    return 0;
}

gboolean
task_is_ongoing (GList **queue)
{
    GList *temp;
    struct wapdman_task *task;

    return_val_if_fail (queue, -EINVAL);

    temp = *queue;
    for ( ; temp; temp = temp->next) {
        task = temp->data;
        if (task && task->state == TASK_STATE_REQUESTED) {
            DEBUG ("Task [%p] is currently in progress", task);
            return TRUE;
        }
    }

    return FALSE;
}

/** @} */
