/**
 * @file ippool.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <glib.h>
#include <ippool.h>

#define IP_BLOCK_DUMP(block)			\
    (unsigned int) ((block)>>24)&0xFF,			\
    (unsigned int) ((block)>>16)&0xFF,			\
    (unsigned int) ((block)>>8)&0xFF,			\
    (unsigned int) ((block)&0xFF)

#define PRIVATE_IP(block)   \
    ((block & 0xff000000) + (block & 0x00ff0000) +  (block & 0x0000ff00))

#define COMPARE(block, shif)    ((unsigned int) (block >> shif))
#define SUBNET_MAX              255
#define VALIDATE_SUBNET(mask)   (mask & (~mask >> 1))

/* Last byte does not matter */
#define COMPARE_BLOCKS(blk1, blk2)      \
    (((COMPARE (blk1, 24) & 0xff) ==    \
    (COMPARE (blk2, 24) & 0xff)) &&     \
    ((COMPARE (blk1, 16) & 0xff) ==     \
    (COMPARE (blk2, 16) & 0xff)) &&   \
    ((COMPARE (blk1, 8) & 0xff) ==      \
    (COMPARE (blk2, 8) & 0xff)))

/**
 * struct ippool - ippool information
 */
struct ippool
{
    /**
     * block - ip block of this pool
     */
    unsigned int block;

    /**
     * ifname - network interface name
     */
    char *ifname;

    /**
     * gateway - gateway ip of the network
     */
    char *gateway;

    /**
     * broadcast - broadcast ip of the network
     */
    char *broadcast;

    /**
     * start_ip - start ip of this network
     */
    char *start_ip;

    /**
     * end_ip - end ip of this network
     */
    char *end_ip;

    char *network;

    /**
     * subnet_mask - subnet mask
     */
    char *subnet_mask;
};

/**
 * List of the Private IP addresses
 *
 * 192.168.0.0 – 192.168.255.255
 * 172.16.0.0 –  172.31.255.255
 * 10.0.0.0 –  10.255.255.255
 *
 * Whenever an dhcp range is asked ippool tries to find
 * /allocate a range from the above list of private ip
 * addresses. The idea is that whenever an AP is started
 * it will be operating with a different dhcp range (except
 * otherwise it is told to run with a fixed dhcp range).
 *
 * This idea is shamelessly and completely taken/stolen from
 * the OSS component ConnMan.
 */

GHashTable *ippools;
GSList *usedblocks;

static unsigned int last_block;
static unsigned int block_16_bits;
static unsigned int block_20_bits;
static unsigned int block_24_bits;
static unsigned int subnet_mask_24;

static void
ippool_cleanup (gpointer data)
{
    struct ippool *pool = data;

    return_if_fail (pool);

    DEBUG ("ippool cleanup: %s [%p]",
           pool->ifname, pool);

    g_free (pool->broadcast);
    g_free (pool->end_ip);
    g_free (pool->gateway);
    g_free (pool->ifname);
    g_free (pool->start_ip);
    g_free (pool->subnet_mask);
    g_free (pool->network);
    g_free (pool);
}

static int
ippool_char_to_ip (const char *address,
                   unsigned int *ip)
{
    int ret;
    struct in_addr inp;

    return_val_if_fail (address && ip, -EINVAL);
    ret = inet_aton (address, &inp);
    return_val_if_fail (ret, -ENXIO);
    *ip = ntohl (inet_addr (address));
    return 0;
}

static char*
ippool_ip_to_char (unsigned int block)
{
    struct in_addr addr;
    addr.s_addr = htonl (block);
    return g_strdup (inet_ntoa (addr));
}

static unsigned int
ippool_get_next_block (unsigned int block)
{
    uint32_t next;

    next = (block & 0x0000ff00) >> 8;
    next += 1;

    if (next == 255) {
        if ((block & 0xffff0000) == block_16_bits) {
            /*
             * Reached the end of the 16 bit block, switch
             * to the 20-bit block.
             */
            return block_20_bits;
        }

        if ((block & 0xffff0000) >= block_20_bits) {
            next = (block & 0x00ff0000) >> 16;
            if (next >= 16 && next < 32)
                next += 1;

            if (next == 32) {
                /*
                 * Reached the end of the 20 bit
                 * block, switch to the 24-bit block.
                 */
                return block_24_bits;
            }

            return (block & 0xff000000) |
                ((next << 16) & 0x00ff0000);
        }

        if ((block & 0xff000000) == block_24_bits) {
            next = (block & 0x00ff0000) >> 16;
            if (next < 255)
                next += 1;

            if (next == 255) {
                /*
                 * Reached the end of the 24 bit
                 * block, switch to the 16-bit block.
                 */
                return block_16_bits;
            }

            return (block & 0xff000000) |
                ((next << 16) & 0x00ff0000);
        }
    }

    return (block & 0xffff0000) | ((next << 8) & 0x0000ff00);
}

static unsigned int
ippool_get_free_block ()
{
    unsigned int block;
    GHashTableIter iter;
    gpointer key, value;
    int used = 0;

    if (last_block)
        block = last_block;
    else
        block = block_16_bits;

    INFO ("Last block: \"%u.%u.%u.%u\"", IP_BLOCK_DUMP (block));
    do {
        block = ippool_get_next_block (block);
        g_hash_table_iter_init (&iter, ippools);
        while (g_hash_table_iter_next (&iter, &key, &value)) {
            struct ippool *pool = value;
            if (pool && (pool->block == block ||
                    g_slist_find (usedblocks, GUINT_TO_POINTER (block))))
                used = 1;
        }

        if (!used)
            return block;
    } while (block != last_block);

    return 0;
}

gboolean
is_private_ipaddress (const char *ip)
{
    int valid = 0;
    unsigned int ipblock = 0;

    return_val_if_fail (ip, FALSE);

    ipblock = PRIVATE_IP (ntohl (inet_addr ("192.168.0.0")));
    if (ipblock >= PRIVATE_IP (ntohl (inet_addr ("192.168.0.0"))) &&
            ipblock <= PRIVATE_IP (ntohl (inet_addr ("192.168.255.255"))))
        valid = 1;
    else if (ipblock >= PRIVATE_IP (ntohl (inet_addr ("172.16.0.0"))) &&
             ipblock <= PRIVATE_IP (ntohl (inet_addr ("172.31.255.255"))))
        valid = 1;
    else if (ipblock >= PRIVATE_IP (ntohl (inet_addr ("10.0.0.0"))) &&
             ipblock <= PRIVATE_IP (ntohl (inet_addr ("10.255.255.255"))))
        valid = 1;

    if (valid)
        return TRUE;

    return FALSE;
}

static int
is_block_unused (unsigned int block)
{
    GHashTableIter iter;
    gpointer key, value;

    g_hash_table_iter_init (&iter, ippools);
    while (g_hash_table_iter_next (&iter, &key, &value)) {
        if (value && COMPARE_BLOCKS (block, ((struct ippool *) value)->block))
            return 0;
    }

    return -ENOENT;
}

static int
calculate_subnet_mask (unsigned int block,
                       unsigned int start,
                       unsigned int end,
                       unsigned int *smask)
{
    (void) block;

    char buf [128];
    unsigned int max = SUBNET_MAX, mask;

    return_val_if_fail (smask, -EINVAL);

    memset (buf, 0, sizeof (buf));
    sprintf (buf, "%u.%u.%u.%u", 255, 255, 255, max - (end - start));

    DEBUG ("Calculated Subnet Mask: %s", buf);
    mask = ntohl (inet_addr (buf));
    if (VALIDATE_SUBNET (mask))
        return -EINVAL;

    *smask = mask;
    return 0;
}

struct ippool*
ippool_create_from_ipaddress (const char *ifname,
                              const char *ip,
                              unsigned int start,
                              unsigned int end)
{
    int ret;
    unsigned int block = 0;
    struct ippool *pool;
    unsigned int mask = 0;

    return_val_if_fail (ifname && ip, NULL);

    DEBUG ("Interface: \"%s\" IP address: \"%s\" start: \"%u\" end: \"%u\"",
           ifname, ip, start, end);

    ret = ippool_char_to_ip (ip, &block);
    if (ret < 0)
        return NULL;

    INFO ("IP Address: %s, block: \"%u.%u.%u.%u\"", ip,
          IP_BLOCK_DUMP (block));

    if ((end - start) > 255)
        return NULL;

    /* dont mess up */
    if (start < (block & 0x000000ff))
        return NULL;

    if (((block & 0x000000ff) + (end - start)) > 255)
        return NULL;

    if (!is_block_unused (block))
        return NULL;

    pool = g_try_malloc0 (sizeof (*pool));
    return_val_if_fail (pool, NULL);

    ret = calculate_subnet_mask (block, start, end, &mask);
    if (ret < 0) {
        ALERT ("Address Range provide does not provide a right subnet");
        mask = subnet_mask_24;
    }

    pool->ifname = g_strdup (ifname);
    pool->block = block;
    pool->subnet_mask = ippool_ip_to_char (mask);
    pool->gateway = ippool_ip_to_char (block + 1);
    pool->start_ip = ippool_ip_to_char (block + 2);
    pool->end_ip = ippool_ip_to_char ((block & 0xffffff00) + (end - 1));
    pool->broadcast = ippool_ip_to_char ((block & mask) | ~mask);
    pool->network = ippool_ip_to_char ((block & mask));

    DEBUG ("Gateway: \"%s\", Broadcast: \"%s\" "
           "Start: \"%s\" End: \"%s\" "
           "Subnet: \"%s\" "
           "Network address: \"%u.%u.%u.%u\"",
           pool->gateway, pool->broadcast,
           pool->start_ip, pool->end_ip,
           pool->subnet_mask,
           IP_BLOCK_DUMP (block & mask));

    usedblocks = g_slist_append (usedblocks, GUINT_TO_POINTER (block));
    g_hash_table_replace (ippools, pool->ifname, pool);
    return pool;
}

int
ippool_destroy (const char *ifname)
{
    struct ippool *pool;

    return_val_if_fail (ifname, -EINVAL);
    pool = g_hash_table_lookup (ippools, ifname);
    return_val_if_fail (pool, -ENOENT);
    usedblocks = g_slist_remove (usedblocks, GUINT_TO_POINTER (pool->block));
    g_hash_table_remove (ippools, ifname);
    return 0;
}

struct ippool*
ippool_create_from_nextblock (const char *ifname,
                              unsigned int start,
                              unsigned int end)
{
    unsigned int freeblock;
    struct ippool *pool;

    return_val_if_fail (ifname, NULL);

    DEBUG ("Creating ip pool for: %s [Start : %d End : %d]",
           ifname, start, end);

    /* Excluding the network, gateway and
     * broadcast addresses */
    if ((end - start) > 255)
        return NULL;

    pool = g_hash_table_lookup (ippools, ifname);
    if (pool)
        return pool;

    freeblock = ippool_get_free_block ();
    if (!freeblock)
        return NULL;

    INFO ("Next Free IP subnet: \"%u.%u.%u.%u\"",
          IP_BLOCK_DUMP (freeblock));

    pool = g_try_malloc0 (sizeof (*pool));
    return_val_if_fail (pool, NULL);

    last_block = freeblock;

    pool->ifname = g_strdup (ifname);
    pool->block = freeblock;
    pool->network = ippool_ip_to_char (freeblock);
    pool->gateway = ippool_ip_to_char (freeblock + 1);
    pool->broadcast = ippool_ip_to_char (freeblock + 255);
    pool->start_ip = ippool_ip_to_char (freeblock + start);
    pool->end_ip = ippool_ip_to_char (freeblock + end);
    pool->subnet_mask = ippool_ip_to_char (subnet_mask_24);

    DEBUG ("Gateway: \"%s\", Broadcast: \"%s\" "
           "Start: \"%s\", End: \"%s\" Subnet: \"%s\" "
           "Network Address: \"%s\"",
           pool->gateway, pool->broadcast,
           pool->start_ip, pool->end_ip, pool->subnet_mask,
           pool->network);

    g_hash_table_replace (ippools, pool->ifname, pool);
    return pool;
}

char*
ippool_get_ifname (struct ippool *pool)
{
    return_val_if_fail (pool, NULL);
    return pool->ifname;
}

char*
ippool_get_broadcast (struct ippool *pool)
{
    return_val_if_fail (pool, NULL);
    return pool->broadcast;
}

char*
ippool_get_gateway (struct ippool *pool)
{
    return_val_if_fail (pool, NULL);
    return pool->gateway;
}

char*
ippool_get_start (struct ippool *pool)
{
    return_val_if_fail (pool, NULL);
    return pool->start_ip;
}

char*
ippool_get_end (struct ippool *pool)
{
    return_val_if_fail (pool, NULL);
    return pool->end_ip;
}

char*
ippool_get_subnet (struct ippool *pool)
{
    return_val_if_fail (pool, NULL);
    return pool->subnet_mask;
}

unsigned int
ippool_get_block (struct ippool* pool)
{
	return_val_if_fail (pool, 0);
	return pool->block;
}

char*
ippool_get_network (struct ippool* pool)
{
    return_val_if_fail (pool, NULL);
    return pool->network;
}

const char*
ippool_type2string (ippooltype type)
{
    switch (type) {
    case IPPOOL_TYPE_UNKNOWN:
        return "unknown";
    case IPPOOL_TYPE_FIXED:
        return "fixed";
    case IPPOOL_TYPE_DYNAMIC:
        return "dynamic";
    }

    return "unknown";
}

int
ippool_init (void)
{
    DEBUG ("");

    ippools = g_hash_table_new_full (g_str_hash, g_str_equal,
                                     NULL, ippool_cleanup);

    block_16_bits = ntohl (inet_addr ("192.168.0.0"));
    block_20_bits = ntohl (inet_addr ("172.16.0.0"));
    block_24_bits = ntohl (inet_addr ("10.0.0.0"));

    subnet_mask_24 = ntohl (inet_addr ("255.255.255.0"));

    return 0;
}

int
ippool_deinit (void)
{
    DEBUG ("");

    g_hash_table_destroy (ippools);
    g_slist_free (usedblocks);
    return 0;
}

/** @} */
