/**
 * @file inotifier.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>
#include <dirent.h>
#include <sys/inotify.h>
#include <stdarg.h>
#include <glib.h>
#include "inotifier.h"
#include <log.h>

/* intereted inotify events */
#define WAPMAN_INOTIFY_EVENTS \
    (IN_MODIFY | \
     IN_CREATE | \
     IN_DELETE | \
     IN_MOVED_TO |  \
     IN_MOVED_FROM)

typedef
enum {

    WATCH_NONE  = 0,

    /* watch a single file or a list of
     * files */
    WATCH_FILES = 1,

    /* watch the entire directory */
    WATCH_DIR   = 2
} inot_watch_type ;

typedef
struct _inotify_client
{
    /* directory to be watched */
    char *path;

    /* watch type that the client
     * registered */
    inot_watch_type watchtype;

    /* interested list of files in the
     * directory to be watched for */
    GList *files;

    /* callback funtion to be called when an
     * interested event occurs */
    inotify_cb callback;

    void *data;
} inotifyclient;

typedef
struct _inotify_data
{
    /* path to be watched */
    char *path;

    /* GIOChannel for the particular
     * directory */
    GIOChannel *channel;

    /* GIOWatch */
    unsigned int watch;

    /* fd returned by inotify */
    int wd;

    /* no. of clients registered for
     * monitoring this particular path (most
     * probably a dir) */
    GList *clients;
} inotifydata;

static
GHashTable *inotify_watches = NULL;

static const char*
inotify_watch_type_to_string(const inot_watch_type type)
{
    switch (type) {
    case WATCH_NONE:
        return ENUMTOSTR (WATCH_NONE);
    case WATCH_DIR:
        return ENUMTOSTR (WATCH_DIR);
    case WATCH_FILES:
        return ENUMTOSTR (WATCH_FILES);
    }

    return ENUMTOSTR (UNKNOWN);
}

static gboolean
inotify_io_function(GIOChannel *channel, GIOCondition cond,
                    gpointer user_data)
{
    inotifydata *data = user_data;
    char buffer [sizeof(struct inotify_event) + NAME_MAX + 1];
    char *next_event;
    gsize bytes_read;
    GIOStatus status;
    GList *list, *files;
    int notify = 0;

    if (cond & (G_IO_NVAL | G_IO_ERR | G_IO_HUP)) {
        ERROR ("Error occured in GIOChannel %p", channel);
        data->watch = 0;
        return FALSE;
    }

    status = g_io_channel_read_chars (channel,
                                      buffer,
                                      sizeof (buffer),
                                      &bytes_read,
                                      NULL);

    switch (status) {
    case G_IO_STATUS_NORMAL:
        break;
    case G_IO_STATUS_AGAIN:
        return TRUE;
    default:
        ERROR ("Reading from inotify channel failed");
        data->watch = 0;
        return FALSE;
    }

    next_event = buffer;

    while (bytes_read > 0) {
        struct inotify_event *event;
        char *changedfile = NULL;
        gsize len;

        event = (struct inotify_event *) next_event;
        if (event->len)
            changedfile = next_event + sizeof(struct inotify_event);

        INFO ("Directory [%s] Changed file is: \"%s\"",
              data->path, changedfile ? changedfile : "UNKNOWN");

        /* next event */
        len = sizeof(struct inotify_event) + event->len;

        /* check if inotify_event block fit */
        if (len > bytes_read)
            break;

        next_event += len;
        bytes_read -= len;

        /* start notifying clients depending inline to their
         * registration */
        list = data->clients;
        for ( ; list; list = list->next) {

            inotifyclient *client = list->data;
            if (!client)
                continue;

            INFO ("client %p watch type : %s", client,
                  inotify_watch_type_to_string (client->watchtype));

            if (!client->callback)
                continue;

            /* notify only if there is a change in one of
             * the interested files (from the client registration)
             * if the client has registration type is WATCH_FILES  */
            if (client->watchtype == WATCH_FILES) {
                files = client->files;
                for ( ; files; files = files->next) {
                    char *file = files->data;
                    if (file && !g_strcmp0 (file, changedfile)) {
                        notify = 1;
                        break;
                    }
                }
            }
            /* notify whenever there is a change in the directory
             * if the registration type is WATCH_DIR */
            else if (client->watchtype == WATCH_DIR)
                notify = 1;

            if (notify)
                client->callback (event, data->path, client->data);
        }
    }

    return TRUE;
}

static int
inotify_create_watch (const char *const path,
                      inotifyclient *client,
                      inotifydata **data)
{
    int fd;

    return_val_if_fail (data, -EINVAL);
    return_val_if_fail (!*data, -EEXIST);

    DEBUG ("Add directory watch for \"%s\" for client [%p]", path, client);

    *data = g_try_malloc0 (sizeof (inotifydata));
    return_val_if_fail (*data, -ENOMEM);
    fd = inotify_init();
    if (fd < 0) {
        g_free (*data);
        return -EIO;
    }

    (*data)->wd = inotify_add_watch (fd, path, WAPMAN_INOTIFY_EVENTS);
    if ((*data)->wd < 0) {
        ERROR ("Creation of %s watch failed", path);
        (void)close (fd);
        g_free (*data);
        *data = NULL;
        return -EIO;
    }

    (*data)->path = g_strdup (path);
    (*data)->channel = g_io_channel_unix_new (fd);

    if (!(*data)->channel) {
        ERROR ("Creation of inotify channel failed");
        inotify_rm_watch (fd, (*data)->wd);
        (*data)->wd = 0;
        (void)close (fd);
        g_free (*data);
        *data = NULL;
        return -EIO;
    }

    g_io_channel_set_close_on_unref ((*data)->channel, TRUE);
    g_io_channel_set_encoding ((*data)->channel, NULL, NULL);
    g_io_channel_set_buffered ((*data)->channel, FALSE);

    (*data)->watch = g_io_add_watch ((*data)->channel,
                                     G_IO_IN | G_IO_HUP | G_IO_NVAL | G_IO_ERR,
                                     inotify_io_function, *data);

    (*data)->clients = g_list_append ((*data)->clients, client);
    INFO ("Successfully created watch for path : \"%s\" and added client"
          " [%p] to inotifydata [%p]", path, client, *data);
    return 0;
}

static void
inotify_key_destroy (gpointer data)
{
    char *path = data;
    DEBUG ("key to be freed path: %s [%p]", path, path);
    g_free (path);
}

static void
inotify_free_client_file (gpointer data)
{
    char *file = data;
    DEBUG ("client file path to be freed: %s [%p]", file, file);
    g_free (file);
}

void
inotify_free_client (gpointer data)
{
    inotifyclient *client = data;

    return_if_fail (client);

    DEBUG ("Freeing client %p", client);

    g_free (client->path);
    g_list_free_full (client->files, inotify_free_client_file);
    client->callback = NULL;
    g_free (client);
}

void
inotify_val_destroy (gpointer data)
{
    int fd;
    inotifydata *inodata = data;

    return_if_fail (inodata);

    DEBUG ("freeing %p No. of clients %d", inodata,
           g_list_length (inodata->clients));

    /* If the channel is broken (e.g., G_IO_HUP) we would
     * have set the watch id to 0. */
    if (inodata->watch > 0)
        g_source_remove (inodata->watch);
    fd = g_io_channel_unix_get_fd (inodata->channel);
    inotify_rm_watch (fd, inodata->wd);
    g_io_channel_unref (inodata->channel);
    g_list_free_full (inodata->clients, inotify_free_client);
    inodata->clients = NULL;
    g_free (inodata);
}

int
inotify_register(const char *const path,
                 inotify_cb callback,
                 const char *const file,
                 void *userdata)
{
    int err = 0, found = 1, same_client = 0;
    inotifyclient *client, *registeredclient;
    inotifydata *data;
    GList *clients, *registeredfiles;

    return_val_if_fail (path && callback, -EINVAL);

    DEBUG ("Directory to be watched : %s [File : \"%s\"]", path,
           file ? file : "All files");

    client = g_try_malloc0 (sizeof (*client));
    return_val_if_fail (client, -ENOMEM);

    client->path = g_strdup (path);
    client->data = userdata;

    /* If the client has requested watch for a specific
     * file, send notifications on the changed events only if
     * there is a change in its interested file */
    if (file) {
        client->files = g_list_append (client->files, g_strdup (file));
        client->watchtype = WATCH_FILES;
    } else {
        /* No specific file mentioned, watch for the changes
         * in the complete directory and notify the same to
         * the client */
        client->watchtype = WATCH_DIR;
    }

    client->callback = callback;
    data = g_hash_table_lookup (inotify_watches, client->path);
    if (!data) {
        err = inotify_create_watch (client->path, client, &data);
        if (err < 0) {
            inotify_free_client ((void *)client);
            return err;
        }

        INFO ("data %p path %s", data, data->path);
        g_hash_table_replace (inotify_watches, data->path, data);
    } else {

        clients = data->clients;
        for ( ; clients; clients = clients->next) {
            registeredclient = clients->data;
            if (!registeredclient)
                continue;

            if (WATCH_FILES == client->watchtype) {

                registeredfiles = registeredclient->files;
                for ( ; registeredfiles; registeredfiles = registeredfiles->next) {
                    if (registeredfiles->data && !g_strcmp0 ((char *) registeredfiles->data, file)) {
                        INFO ("Already the file %s is registered", file);
                        found = 0;
                        break;
                    }
                }
            }

            if (callback == registeredclient->callback) {
                INFO ("Already a client with same callback %p has registered", callback);
                same_client = 1;
                if (file && found)
                    registeredclient->files = g_list_append (registeredclient->files, g_strdup (file));
                inotify_free_client ((void *)client);
                break;
            }
        }

        if (!same_client) {
            INFO ("New client has registered for watching file \"%s\" in dir \"%s\", callback : %p",
                  file, path, callback);
            data->clients = g_list_append (data->clients, client);
            INFO ("No. of clients registered for path %s : %d", path, g_list_length (data->clients));
        }
    }

    return err;
}

int
inotify_unregister (const char *const path,
                    inotify_cb callback,
                    const char *const file)
{
    inotifydata *data;
    inotifyclient *client;
    GList *clients, *files;
    char *registeredfile = NULL;
    int found = 0;

    return_val_if_fail (path, -EINVAL);

    data = g_hash_table_lookup (inotify_watches, path);
    if (!data) {
        ERROR ("Watch for the directory %s was not created", path);
        return -ENOENT;
    }

    clients = data->clients;
    for ( ; clients; clients = clients->next) {
       client = clients->data;
       if (!client)
           continue;

       if (!file && client->watchtype == WATCH_DIR) {
           if (client->callback == callback) {
               INFO ("client found for the path %s with callback %p [%p]",
                     path, callback, client);
               found = 1;
           }
       }
       else if (file && client->watchtype == WATCH_FILES) {

           files = client->files;
           for ( ; files; files = files->next) {

               registeredfile = files->data;
               if (registeredfile && !g_strcmp0 (file, registeredfile)) {
                   INFO ("client found for the path %s file %s with callback %p [%p]",
                         path, file, callback, client);
                   found = 1;
                   break;
               }
           }
       }

       if (found)
           break;
    }

    if (found) {

        /* free the client only if there are no more watch files
         * left if the watch type is WATCH_FILES */
        if (client->watchtype == WATCH_FILES) {
            client->files = g_list_remove_link (client->files, files);
            g_free (registeredfile);

            if (!client->files) {
                inotify_free_client (client);
                data->clients = g_list_remove_link (data->clients, clients);
            }
        }
        /* free the client if the watch type is WATCH_DIR */
        else if (client->watchtype == WATCH_DIR) {
            inotify_free_client (client);
            data->clients = g_list_remove_link (data->clients, clients);
        }
    }

    /* Stop watching for the directory if there are no more
     * clients left and interested */
    if (!data->clients){
        INFO ("No more clients for %s, thus deleting the inotifydata %p",
              path, data);
        g_hash_table_remove (inotify_watches, path);
    }

    return 0;
}

int
inot_init(void)
{
    DEBUG ("");

    inotify_watches = g_hash_table_new_full (g_str_hash, g_str_equal,
                                             inotify_key_destroy,
                                             inotify_val_destroy);
    return 0;
}

int
inot_deinit(void)
{
    DEBUG ("");

    g_hash_table_destroy (inotify_watches);
    return 0;
}

/** @} */
