/**
 * @file capabilities.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <sys/capability.h>
#include <linux/capability.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include "capabilities.h"
#include <log.h>

/* valid capabaility flags */
#define flag_valid(x) ((x >= (int) CAP_EFFECTIVE) && \
    x <= (int) CAP_INHERITABLE)

/* various available (possible) linux capabilities */
static const char*
capabilities_to_string [] = {
    [CAP_CHOWN]            = "CAP_CHOWN",
    [CAP_DAC_OVERRIDE]     = "CAP_DAC_OVERRIDE",
    [CAP_DAC_READ_SEARCH]  = "CAP_DAC_READ_SEARCH",
    [CAP_FOWNER]           = "CAP_FOWNER",
    [CAP_FSETID]           = "CAP_FSETID",
    [CAP_KILL]             = "CAP_KILL",
    [CAP_SETGID]           = "CAP_SETGID",
    [CAP_SETUID]           = "CAP_SETUID",
    [CAP_SETPCAP]          = "CAP_SETPCAP",
    [CAP_LINUX_IMMUTABLE]  = "CAP_LINUX_IMMUTABLE",
    [CAP_NET_BIND_SERVICE] = "CAP_NET_BIND_SERVICE",
    [CAP_NET_BROADCAST]    = "CAP_NET_BROADCAST",
    [CAP_NET_ADMIN]        = "CAP_NET_ADMIN",
    [CAP_NET_RAW]          = "CAP_NET_RAW",
    [CAP_IPC_LOCK]         = "CAP_IPC_LOCK",
    [CAP_IPC_OWNER]        = "CAP_IPC_OWNER",
    [CAP_SYS_MODULE]       = "CAP_SYS_MODULE",
    [CAP_SYS_RAWIO]        = "CAP_SYS_RAWIO",
    [CAP_SYS_CHROOT]       = "CAP_SYS_CHROOT",
    [CAP_SYS_PTRACE]       = "CAP_SYS_PTRACE",
    [CAP_SYS_PACCT]        = "CAP_SYS_PACCT",
    [CAP_SYS_ADMIN]        = "CAP_SYS_ADMIN",
    [CAP_SYS_BOOT]         = "CAP_SYS_BOOT",
    [CAP_SYS_NICE]         = "CAP_SYS_NICE",
    [CAP_SYS_RESOURCE]     = "CAP_SYS_RESOURCE",
    [CAP_SYS_TIME]         = "CAP_SYS_TIME",
    [CAP_SYS_TTY_CONFIG]   = "CAP_SYS_TTY_CONFIG",
    [CAP_MKNOD]            = "CAP_MKNOD",
    [CAP_LEASE]            = "CAP_LEASE",
    [CAP_AUDIT_WRITE]      = "CAP_AUDIT_WRITE",
    [CAP_AUDIT_CONTROL]    = "CAP_AUDIT_CONTROL",
    [CAP_SETFCAP]	       = "CAP_SETFCAP",
    [CAP_MAC_OVERRIDE]     = "CAP_MAC_OVERRIDE",
    [CAP_MAC_ADMIN]        = "CAP_MAC_ADMIN",
    [CAP_SYSLOG]           = "CAP_SYSLOG",
    [CAP_WAKE_ALARM]       = "CAP_WAKE_ALARM",
#ifdef CAP_BLOCK_SUSPEND
    [CAP_BLOCK_SUSPEND]    = "CAP_BLOCK_SUSPEND",
#endif
#ifdef CAP_AUDIT_READ
    [CAP_AUDIT_READ]       = "CAP_AUDIT_READ",
#endif
};

static const char*
cap_flag_to_string [] = {
    [CAP_EFFECTIVE]     = "CAP_EFFECTIVE",
    [CAP_PERMITTED]     = "CAP_PERMITTED",
    [CAP_INHERITABLE]   = "CAP_INHERITABLE",
};

int
get_capabilities (const int cap,
                  const int flag)
{
    pid_t pid;
    cap_t capabilities;
    cap_flag_value_t flags_value;

    /* invalid arguments */
    if (!cap_valid (cap))
        return -EINVAL;

    /* invalid arguments */
    if (!flag_valid (flag))
        return -EINVAL;

    pid = getpid ();
    capabilities = cap_get_pid (pid);
    return_val_if_fail (capabilities, -ENOMEM);

    DEBUG ("PID of the process : %d", pid);
    DEBUG ("capability check for : %s [%s]", capabilities_to_string [cap],
           cap_flag_to_string [flag]);

    if (cap_get_flag (capabilities, (cap_value_t) cap, (cap_flag_t) flag, &flags_value) < 0) {
        ERROR ("Failed to get the capabilities of the process, error : %s", strerror (errno));
        cap_free (capabilities);
        return -errno;
    }

    INFO ("Does this process has \"%s\" capability? : %s", capabilities_to_string [cap],
          flags_value == CAP_SET ? "YES" : "NO");

    cap_free (capabilities);
    return (flags_value == CAP_SET);
}

/** @} */
