/**
 * @file ippool.h
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#ifndef IPPOOL_H
#define IPPOOL_H

#include <log.h>

/* opaque */
struct ippool;

typedef
enum {
    IPPOOL_TYPE_UNKNOWN,
    IPPOOL_TYPE_FIXED,
    IPPOOL_TYPE_DYNAMIC
} ippooltype;

/**
 * ippool_init - Initialize the ippool
 */
int ippool_init (void);

/**
 * ippool_deinit - Deinitialize the ippool
 */
int ippool_deinit (void);

/**
 * ippool_create_from_nextblock - Create an ippool with a given size
 * with the next available free block
 * @ifname: Name of the network interface
 * @start: start ip of the ippool
 * @end: end ip of the ippool
 * Returns: Pointer to abstract ippool or %NULL on failure
 */
struct ippool* WARN_UNUSED
ippool_create_from_nextblock (const char *ifname,
                              unsigned int start,
                              unsigned int end);

struct ippool* ippool_create_from_ipaddress (const char *ifname,
                                             const char *ip,
                                             unsigned int start,
                                             unsigned int end);

/**
 * ippool_destroy - Destroy the created ippool
 * @ifname: Name of the network interface
 * Returns: 0 on success or negative error code on failure
 */
int ippool_destroy (const char *ifname);

/**
 * ippool_get_ifname - Get the interface name
 * @pool: ippool
 * Returns: interface name on success or %NULL on failure
 */
char* ippool_get_ifname (struct ippool *pool);

/**
 * ippool_get_broadcast - Get the broadcast ip of the network
 * @pool: ippool
 * Returns: broadcast ip on success or %NULL on failure
 */
char* ippool_get_broadcast (struct ippool *pool);
char* ippool_get_gateway (struct ippool *pool);
/**
 * ippool_get_start - Get the start ip of the network
 * @pool: ippool
 * Returns: start ip address on success or %NULL on failure
 */
char* ippool_get_start (struct ippool *pool);

/**
 * ippool_get_end - Get the end ip of the network
 * @pool: ippool
 * Returns: end ip on success or %NULL on failure
 */
char* ippool_get_end (struct ippool *pool);

/**
 * ippool_get_subnet - Get the subnet mask
 * @pool: ippool
 * Returns: subnet mask on success or %NULL on failure
 */
char* ippool_get_subnet (struct ippool *pool);

unsigned int ippool_get_block (struct ippool* pool);

char *ippool_get_network (struct ippool* pool);

gboolean is_private_ipaddress (const char *ip);

const char *ippool_type2string (ippooltype type);

#endif  //IPPOOL_H

/** @} */
