/**
 * @file hostapd_cli.c
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <sys/syscall.h>
#include <sys/inotify.h>
#include <linux/limits.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <dirent.h>
#include <string.h>
#include <errno.h>
#include <glib.h>
#include <gio/gio.h>
#include <genl.h>
#include <inotifier.h>
#include <log.h>
#include "inc/utils.h"
#include <wpa_ctrl.h>
#include <accesspoint.h>

/**
 * UNIX_SOCK_ROOT_DIR_PATH - Run-time variable data of the system
 */
#define UNIX_SOCK_ROOT_DIR_PATH         "/var/run"

/**
 * HOSTAPD_SOCK_DIR_PATH - Direcroty path of run time variable data of
 * Hostapd
 */
#define HOSTAPD_SOCK_DIR_PATH           UNIX_SOCK_ROOT_DIR_PATH"/hostapd"

/**
 * HOSTAPD_SOCK_DIR_NAME - Direcroty name of run time variable data of
 * Hostapd
 */
#define HOSTAPD_SOCK_DIR_NAME           "hostapd"


/**
 * HOSTAPD_REQUEST_BUFFER - Buffer size of a request to hostapd
 */
#define HOSTAPD_REQUEST_BUFFER      512

/**
 * HOSTAPD_REPLY_BUFFER - Buffer size of a reply from hostapd
 */
#define HOSTAPD_REPLY_BUFFER        4096

/**
 * HOSTAPD_REQUEST_SUCCESS - Hostapd has successfully processed the
 * request
 */
#define HOSTAPD_REQUEST_SUCCESS     "OK"

/**
 * HOSTAPD_REQUEST_SUCCESS - Hostapd has failed to process the request
 */
#define HOSTAPD_REQUEST_FAILURE     "FAIL"

/* Generic control interface commands */
#define HOSTAPD_CTRL_CMD_PING                       "PING"
#define HOSTAPD_CTRL_CMD_PING_REPLY                 "PONG"
#define HOSTAPD_CTRL_CMD_FAILED                     "FAIL"

/* Global control interface commands */
#define HOSTAPD_GLOBAL_CTRL_CMD_ADD_INTERFACE       "ADD"
#define HOSTAPD_GLOBAL_CTRL_CMD_REMOVE_INTERFACE    "REMOVE"
#define HOSTAPD_GLOBAL_CTRL_CMD_ADD_BSS_CONFIG      "bss_config"

/* control interface commands related to a particular interface */
#define HOSTAPD_AP_CTRL_CMD_ENABLE                  "ENABLE"
#define HOSTAPD_AP_CTRL_CMD_DISABLE                 "DISABLE"
#define HOSTAPD_AP_CTRL_CMD_RELOAD                  "RELOAD"
#define HOSTAPD_AP_CTRL_CMD_DISASSOCIATE_STATION    "DISASSOCIATE"
#define HOSTAPD_AP_CTRL_CMD_DEAUTHENTICATE_STATION  "DEAUTHENTICATE"
#define HOSTAPD_AP_CTRL_CMD_SET                     "SET"

#define HOSTAPD_AP_CTRL_CMD_SET_PASSPHRASE          "wpa_passphrase"
#define HOSTAPD_AP_CTRL_CMD_SET_SECURITY            "wpa"
#define HOSTAPD_AP_CTRL_CMD_SET_KEYMGMT             "wpa_key_mgmt"
#define HOSTAPD_AP_CTRL_CMD_SET_WPA_PAIRWISE        "wpa_pairwise"
#define HOSTAPD_AP_CTRL_CMD_SET_RSN_PAIRWISE        "rsn_pairwise"
#define HOSTAPD_AP_CTRL_CMD_SET_WEPDEFAULT          "wep_default_key"
#define HOSTAPD_AP_CTRL_CMD_SET_WEPKEY              "wep_key0"
#define HOSTAPD_AP_CTRL_CMD_SET_AUTH_ALGS           "auth_algs"
#define HOSTAPD_AP_CTRL_CMD_SET_CHANNEL             "channel"
#define HOSTAPD_AP_CTRL_CMD_SET_CHANNELLIST         "chanlist"
#define HOSTAPD_AP_CTRL_CMD_SET_SSID                "ssid"
#define HOSTAPD_AP_CTRL_CMD_SET_SSID2               "ssid2"
#define HOSTAPD_AP_CTRL_CMD_SET_UTF8SSID            "utf8_ssid"
#define HOSTAPD_AP_CTRL_CMD_SET_HWMODE              "hw_mode"
#define HOSTAPD_AP_CTRL_CMD_SET_HIDDEN              "ignore_broadcast_ssid"
#define HOSTAPD_AP_CTRL_CMD_SET_VENDOR_IE           "vendor_elements"
#define HOSTAPD_AP_CTRL_CMD_SET_ASSOCRESP_IE        "assocresp_elements"
#define HOSTAPD_AP_CTRL_CMD_MAX_NUM_STA             "max_num_sta"
#define HOSTAPD_AP_CTRL_CMD_SET_ACCNETTYPE          "access_network_type"

/* WPA Key Mgmt types */
#define HOSTAPD_KEY_MGMT_PSK        "WPA-PSK"
#define HOSTAPD_KEY_MGMT_EAP        "WPA-EAP"

/* WPA Pairwise and RSN Pairwise ciphers */
#define HOSTAPD_KEY_PAIRWISE_CCMP   "CCMP"
#define HOSTAPD_KEY_PAIRWISE_TKIP   "TKIP"

#define SSID_MAX_LEN            32
#define PASSPHRASE_MIN_LENGTH   8
#define PASSPHRASE_MAX_LENGTH   63

#define MAX_STATIONS_ALLOWED    10

#define WPS_ENROLLEE_PARAM_LENGTH       128
#define ACS_PARAM_LENGTH                WPS_ENROLLEE_PARAM_LENGTH

#define HOSTAPD_REPLY_REMOVE_NEWLINE(reply, length)    \
    ({   \
    if (length) \
        reply [length - 1] = '\0';   \
    })

#define HOSTAPD_ALREADY_ACQUIRED(sk)    (!!sk->acquired)

#define HOSTAPD_MAX_THREADS_PER_POOL    3

/**
 * struct wpa_ctrl - hostapd control socket
 */
typedef struct wpa_ctrl wpasocket;

/* forward declaration */
static void
hostapd_inotify (struct inotify_event *event, const char *dir, void *data);

/**
 * enum sockettype - different socket types
 */
typedef
enum __socket_type
{
    HOSTAPD_SOCKET_TYPE_GLOBAL,
    HOSTAPD_SOCKET_TYPE_IFACE,
    NUM_HOSTAPS_SOCKET_TYPES
} sockettype;

/**
 * struct hostapdsocket - represents a hostapd control
 * interface socket
 */
typedef
struct _hostapd_sk
{
    /**
     * ifname - network interface name
     */
    char *ifname;

    /**
     * sk - wpa control interface socket to the
     * interface. This socket is used for requests!!
     */
    wpasocket *ctrl_sk;

    /**
     * sk - wpa control interface socket to the
     * interface. This socket is used to monitor events!!
     */
    wpasocket *monitor_sk;

    /**
     * acquired - A request is currently in progress!!
     * This ctrl_sk is already acquired
     */
    int acquired;

    /**
     * type - type of the socket i.e., whether a
     * global socket or to a particular interface
     */
    sockettype type;

    /**
     * iochannel - integrating this socket to the
     * main event loop of the application
     */
    GIOChannel *iochannel;

    /**
     * watch - event source id for adding the iochannel
     * to the default main loop context with default priority
     */
    unsigned int watch;
} hostapdsocket;

/**
 * struct accesspoint - represents an access point of the
 * system
 */
typedef
struct _accesspoint
{
    /**
     * started - whether the access point is started
     */
    int started;

    int acs_due, acs_channels;

    acsevent event;

    /**
     * sock - the control interface socket related to this
     * accesspoint
     */
    hostapdsocket *sock;

    /**
     * wpsenrollees - wps enrollees detail i.e., requested
     * for an association with this access point
     */
    GHashTable *wpsenrollees;

    /**
     * regdom_requests - no. of regulatory change requests
     * currently in progress
     */
    GSList *acs_requests;
} accesspoint;

/**
 * struct clientdata - Client data for an async function
 */
typedef
struct __client_data
{
    /**
     * cb - Callback of the caller wherein the result shall be
     * returned
     */
    apresultcb cb;

    /**
     * userdata - Clients could pass some user data which would be
     * handed over to the "cb" along with the result
     */
    void *userdata;
} clientdata;

typedef
struct __acs_request
{
    /**
     * iface - network interface name
     */
    char *iface;

    /**
     * phyname - The wireless device that this iface
     * belong to
     */
    unsigned int timeout;

    /**
     * regdom_source_event - source id for a reg dom change
     * timeout event
     */
    unsigned int timeout_event;

    /**
     * client_data - client data for this request
     */
    clientdata *data;

    /**
     * sk - socket to be blocked till the reg dom
     * request complete
     */
    hostapdsocket *sk;

    /**
     * result - result of this request
     */
    int result;
} acs_request;


typedef
struct __regulatory_change_request
{
    /**
     * iface - network interface name
     */
    char *iface;

    /**
     * phyname - The wireless device that this iface
     * belong to
     */
    char *phyname;

    /**
     * regdom_source_event - source id for a reg dom change
     * timeout event
     */
    unsigned int regdom_source_event;

    /**
     * client_data - client data for this request
     */
    clientdata *data;

    /**
     * sk - socket to be blocked till the reg dom
     * request complete
     */
    hostapdsocket *sk;

    /**
     * result - result of this request
     */
    int result;
} regdom_request;

/**
 * struct hostapdglobal - global instance
 */
typedef
struct _accesspoint_global
{
    /**
     * global - global control interface
     */
    hostapdsocket *global;

    /**
     * accesspoints - no. of access points available
     */
    GHashTable *accesspoints;

    /**
     * threadpool - All of the hostapd requests are done
     * via the wpa_ctrl library (wpa_ctrl_request()) which
     * is a blocking call (wpa_client blocks the whenever a request
     * is made even though the socket is of type of non-blocking),
     * thus the main thread could be blocked whenever a request to
     * hostapd is made.
     *
     * The idea here is to fork out and execute the blocking calls in
     * the background. We use here the GThreadPool wherein we
     * create upto 3 worker threads (at max) (1 for the global
     * operations and one for each access point). The thread pool
     * will only create the max thread if the existing resources aka
     * threads are already utilized else the same resources (threads)
     * are shared across the accesspoints.
     */
    GThreadPool *threadpool;

    /**
     * regdom_requests - no. of regulatory change requests
     * currently in progress
     */
    GSList *regdom_requests;
} hostapdglobal;

/**
 * struct securitydata - Security data required for the async
 * operations
 */
typedef
struct __security_data
{
    /**
     * type - security type
     */
    securitytype type;
} securitydata;

/**
 * struct ssiddata - SSID data required for the async
 * operations
 */
typedef
struct __ssid_data
{
    /**
     * wpa/wpa2 passphrase
     */
	char *wpa_passphrase;
} ssiddata;

/**
 * struct hostapdreq - An async hostapd request
 */
typedef
struct __hostapd_req
{
    /**
     * ifname - interface name of the access point
     */
    char *ifname;

    /**
     * req - The actual request
     */
    char *reqcmd;

    /**
     * result - result of the operation
     */
    int result;

    /**
     * reply - reply from the hostapd
     */
    char *reply;

    /**
     * sk - ctrl socket to be used for hostapd
     * requests
     */
    hostapdsocket *sk;

    /**
     * pool - thread pool to be used
     */
    GThreadPool *pool;

    /**
     * context - context in which the callback shall be invoked.
     * If the context is the current executing thread, then the
     * callback will be invoked directly else a source will be created
     * and attached to the target thread's context.
     */
    void *context;

    /**
     * cb - callback to be invoked
     */
    GSourceFunc cb;

    /**
     * data - requested client data
     */
    clientdata *data;

    /**
     * security - Additional information on the security if the
     * request is to set the security
     */
    securitydata *security;

    /**
     * ssid - Additional information on the passphrase if the
     * request is to set/change the SSID
     */
    ssiddata *ssid;

    int acs_required, acs_channels;
    unsigned int acs_timeout_event;
} hostapdreq;

static const char *aperrmsg [NUM_APE_MAX+1] = {
    [APE_SUCCESS]		= "Success",
    [APE_SENDRECEIVE_FAILED]		= "Send/Receive failue",
    [APE_TIMEOUT]		= "Timeout occurred",
    [APE_FAILURE]		= "Unspecific failure",
    [APE_INVAL]         = "Invalid parameters",
    [APE_ALREADY]		= "Already exists",
    [APE_SYSNOTREADY]   = "System not ready",
    [APE_NOLINK]		= "Link has been severed",
    [APE_NO80211IFACE]	= "Not a wireless interface",
    [APE_INTERNALFAIL]	= "Internal failure",
    [APE_NOENT]         = "No such entry",
    [APE_NOTSTARTED]	= "Accesspoint not started",
    [APE_NOTSUP]    	= "Operation not supported",
    [APE_BUSY]          = "Device or resource busy",
    [APE_IO]        	= "I/O error",
};

static hostapdglobal *hostapd;
static int hostapd_started;
static int hostapd_retry;
static const accesspointcallbacks *callbacks;

static const char*
hostapd_send_error_tostring (const int err)
{
    switch (err) {
    case -1:
        return "SEND/RECEIVE ERROR";
    case -2:
        return "TIMED OUT";
    }

    return "UNKNOWN";
}

static void
hostapd_enrolle_cleanup (gpointer data)
{
    wpsenrollee *dev = data;

    return_if_fail (dev);

    INFO ("Cleaning up the enrollee: %s [%s]",
          dev->uuid, dev->device_name);

    g_free (dev->devicecategory);
    g_free (dev->device_name);
    g_free (dev->macaddress);
    g_free (dev->manufacturer);
    g_free (dev->modelname);
    g_free (dev->modelnumber);
    g_free (dev->serialnumber);
    g_free (dev->uuid);

    g_free (dev);
}

static void
hostapd_accesspoint_cleanup (gpointer data)
{
    accesspoint *ap = data;

    return_if_fail (ap);

    INFO ("Cleaning up the access point : %s",
          ap->sock->ifname);

    /* we would not have attached for global socket */
    if (HOSTAPD_SOCKET_TYPE_IFACE == ap->sock->type) {
        wpa_ctrl_detach (ap->sock->monitor_sk);
        if (ap->sock->watch > 0)
            g_source_remove (ap->sock->watch);
        g_io_channel_unref (ap->sock->iochannel);
        wpa_ctrl_close (ap->sock->monitor_sk);
    }
    g_free (ap->sock->ifname);
    if (ap->wpsenrollees)
        g_hash_table_destroy (ap->wpsenrollees);
    wpa_ctrl_close (ap->sock->ctrl_sk);
    g_free (ap->sock);
    g_free (ap);
}

static acsevent
hostapd_acs_event (const char *evt)
{
    return_val_if_fail (evt, NUM_ACS_EVENTS);

    if (!g_strcmp0 (ACS_EVENT_STARTED, evt))
        return EVENT_ACS_STARTED;
    else if (!g_strcmp0 (ACS_EVENT_FAILED, evt))
        return EVENT_ACS_FAILED;
    else if (!g_strcmp0 (ACS_EVENT_COMPLETED, evt))
        return EVENT_ACS_COMPLETED;

    return NUM_ACS_EVENTS;
}

static void
hostapd_callback_station_associated (const char *iface,
                                     const char *station)
{
    return_if_fail (callbacks && callbacks->station_associated);
    callbacks->station_associated (iface, station);
}

static void
hostapd_callback_station_disconnected (const char *iface,
                                     const char *station)
{
    return_if_fail (callbacks && callbacks->station_disconnected);
    callbacks->station_disconnected (iface, station);
}

static void
hostapd_callback_system_status (const int status)
{
    DEBUG ("Hostapd system status: %s", status
           ? "available" : "unavailable");
    return_if_fail (callbacks && callbacks->system_status);
    callbacks->system_status (status);
}

static void
hostapd_callback_ap_started (const char *iface)
{
    return_if_fail (callbacks && callbacks->accesspoint_started);
    callbacks->accesspoint_started (iface);
}

static void
hostapd_callback_ap_stopped (const char *iface)
{
    return_if_fail (callbacks && callbacks->accesspoint_stopped);
    callbacks->accesspoint_stopped (iface);
}

static void
acs_request_cleanup (gpointer data)
{
    acs_request *req = data;

    return_if_fail (req);

    INFO ("Freeing up the acs request: %s [%p]",
          req->iface, req);

    g_free (req->iface);
    g_free (req->data);
    g_free (req);
}

static void
hostapd_acs_request_cleanup (accesspoint *ap)
{
    GSList *temp;
    gboolean avail = FALSE;
    acs_request *req = NULL;

    return_if_fail (ap);

    for (temp = ap->acs_requests; temp; temp = temp->next) {

        req = temp->data;
        continue_if_fail (req);

        avail = TRUE;
        if (req->timeout_event > 0)
            g_source_remove (req->timeout_event);

        if (req->sk)
            req->sk->acquired = 0;

        req->result = ap->event == EVENT_ACS_COMPLETED ? 0 : -APE_FAILURE;
        if (req->data && req->data->cb)
            req->data->cb (req->result, req->iface, req->data->userdata);

        acs_request_cleanup (req);
        break;
    }

    if (avail) {
        ap->acs_due = ap->acs_channels = 0;
        ap->acs_requests = g_slist_remove (ap->acs_requests, req);
    }
}

static void
hostapd_callback_acs_event (const char *iface,
                            const char *event,
                            char *buf)
{
    char *msg, *tmp = NULL,
            *frequency;
    acsevent acsevt;
    unsigned int chan = 0,
            freq = 0;
    accesspoint *ap = NULL;
    gboolean acs_cleanup = FALSE;

    DEBUG ("AccessPoint: %s Event: %s msg: %s", iface, event, buf);

    msg = buf;
    acsevt = hostapd_acs_event (event);

    return_if_fail (callbacks);

    if (callbacks->acs_event)
        callbacks->acs_event (iface, acsevt);

    if (acsevt == EVENT_ACS_COMPLETED && msg) {
        /* buf */
        /* "freq=xxxx channel=xx" */
        strtok_r (msg, "=", &tmp);
        frequency = strtok_r (NULL, " ", &tmp);
        strtok_r (NULL, "=", &tmp);

        if (frequency)
            freq = (unsigned int) atoi (frequency);
        if (tmp)
            chan = (unsigned int) atoi (tmp);

        INFO ("Changed operating frequency: \"%u\" channel: \"%u\"",
              freq, chan);

        if (callbacks->operatingchannel_changed)
            callbacks->operatingchannel_changed (iface, freq, chan);
    }
    else if (acsevt == EVENT_ACS_FAILED)
        acs_cleanup = TRUE;

    return_if_fail (hostapd);
    ap = g_hash_table_lookup (hostapd->accesspoints, iface);
    return_if_fail (ap);
    ap->event = acsevt;

    /* ACS completed on FAILURE (On completed the
     * cleanup is handled on the AP_ENABLED event */
    if (acs_cleanup)
        hostapd_acs_request_cleanup (ap);
}

static void
hostapd_process_enrollee (accesspoint *ap,
                          char *buf)
{
    char *msg, *tmp;
//    wpsenrollee *enrollee;
    char *uuid, *macaddress, *devicename,
            *manufacturer, *modelname, *modelnumber,
            *serialnumber, *devicecategory;

    return_if_fail (ap && buf);

    /* buf */
    /* "UUID-E MACAddress [Device Name|Manufacturer|Model Name|Model Number|Serial Number|Device Category]" */

    msg = buf;
    uuid = strtok_r (msg, " ", &tmp);
    macaddress = strtok_r (NULL, " ", &tmp);
    tmp++;
    devicename = strtok_r (NULL, "|", &tmp);
    manufacturer = strtok_r (NULL, "|", &tmp);
    modelname = strtok_r (NULL, "|", &tmp);
    modelnumber = strtok_r (NULL, "|", &tmp);
    serialnumber = strtok_r (NULL, "|", &tmp);
    devicecategory = strtok_r (NULL, "]", &tmp);

    INFO ("UUID: \"%s\" "
          "MAC Address: \"%s\" "
          "DevName: \"%s\" "
          "Manufacturer: \"%s\""
          "ModelName: \"%s\" "
          "ModelNumber: \"%s\" "
          "SerialNumber: \"%s\" "
          "DeviceCategory: \"%s\"",
          uuid,
          macaddress,
          devicename,
          manufacturer,
          modelname,
          modelnumber,
          serialnumber, devicecategory);

    return_if_fail (callbacks && callbacks->wps_pin_needed);
#if 0
    enrollee = g_hash_table_lookup (ap->wpsenrollees, uuid);
    if (enrollee)
        return;

    enrollee = g_try_malloc0 (sizeof (wpsenrollee));
    if (!enrollee)
        return;

    enrollee->uuid = g_strdup (uuid);
    enrollee->device_name = g_strdup (devicename);
    enrollee->macaddress = g_strdup (macaddress);
    enrollee->manufacturer = g_strdup (manufacturer);
    enrollee->modelname = g_strdup (modelname);
    enrollee->modelnumber = g_strdup (modelnumber);
    enrollee->serialnumber = g_strdup (serialnumber);
    enrollee->devicecategory = g_strdup (devicecategory);

    g_hash_table_replace (ap->wpsenrollees, enrollee->uuid, enrollee);
    callbacks->wps_pin_needed (ap->sock->ifname, enrollee);
#endif
}

static int
hostapd_process_accesspoint_events (accesspoint *ap,
                                    char *buf)
{
    char *msg;
    char event [HOSTAPD_REPLY_BUFFER];
    int offset = 0;

    return_val_if_fail (ap, -EINVAL);
    return_val_if_fail (buf, -ENOKEY);

    /* "<3>WPS-ENROLLEE-SEEN 50:2e:5c:28:0b:cd 60f92dbc-eb1b-563d-a39c-f458dca5b157 10-0050F204-5 0x4288 0 0 [htc_asia_india]" */
    msg = buf;
    if (*msg == '<') {
        msg = strchr (msg, '>');
        if (msg)
            msg++;
        else
            msg = buf;
    }

    memset (event, 0, sizeof (event));
    while (*msg != '\0') {
        if (*msg == ' ') {
            msg++;
            event [offset] = ' ';
            break;
        }
        event [offset++] = *msg++;
    }

    INFO ("Event: \"%s\", Type: \"%s\" Msg: \"%s\"", buf,
          event, msg);

    if (!g_strcmp0 (AP_STA_CONNECTED, event))
        hostapd_callback_station_associated (ap->sock->ifname, msg);
    else if (!g_strcmp0 (AP_STA_DISCONNECTED, event))
        hostapd_callback_station_disconnected (ap->sock->ifname, msg);
    else if (!g_strcmp0 (WPS_EVENT_PIN_NEEDED, event))
        hostapd_process_enrollee (ap, msg);
    else if (!g_strcmp0 (AP_EVENT_ENABLED, event)) {
        ap->started = 1;
        hostapd_callback_ap_started (ap->sock->ifname);
        if (ap->acs_due)
            hostapd_acs_request_cleanup (ap);
    }
    else if (!g_strcmp0 (AP_EVENT_DISABLED, event)) {
        ap->started = 0;
        hostapd_callback_ap_stopped (ap->sock->ifname);
    }
    else if (!g_strcmp0 (ACS_EVENT_STARTED, event) ||
             !g_strcmp0 (ACS_EVENT_FAILED, event) ||
             !g_strcmp0 (ACS_EVENT_COMPLETED, event)) {
        hostapd_callback_acs_event (ap->sock->ifname, event, msg);
    }

    return 0;
}

static gboolean
hostapd_accesspoint_receive (GIOChannel *source,
                             GIOCondition condition,
                             gpointer data)
{
    size_t len;
    char reply [HOSTAPD_REPLY_BUFFER];
    accesspoint *ap = data;

    return_val_if_fail (ap, TRUE);

    if (condition & (G_IO_NVAL | G_IO_ERR | G_IO_HUP)) {
        ERROR ("Error occured in GIOChannel of interface %p",
               source);
        ap->sock->watch = 0;
        return FALSE;
    }

    DEBUG ("Event for the access point interface: %s",
           ap->sock->ifname);

    while (wpa_ctrl_pending (ap->sock->monitor_sk)) {
		memset (reply, 0, sizeof (reply));
		len = sizeof (reply) - 1;
        if (!wpa_ctrl_recv (ap->sock->monitor_sk, reply, &len)) {
			reply [len] = '\0';
            DEBUG ("EVENT Message: %s [length: %u]", reply, len);
            hostapd_process_accesspoint_events (ap, reply);
        }
    }

    return TRUE;
}

static void
regulatory_request_cleanup (gpointer data)
{
    regdom_request *req = data;

    return_if_fail (req);

    INFO ("Freeing up the regdom request: %s/%s [%p]",
          req->iface, req->phyname, req);

    g_free (req->iface);
    g_free (req->phyname);
    g_free (req->data);
    g_free (req);
}

static void
hostapd_task_cleanup (gpointer data)
{
    hostapdreq *req = data;

    return_if_fail (req);

    INFO ("Freeing up the hostapd request: %s [%p]",
          req->reqcmd, req);

    g_free (req->ifname);
    g_free (req->reqcmd);
    g_free (req->reply);
    g_free (req->data);
    g_free (req->security);
    if (req->ssid)
    	g_free (req->ssid->wpa_passphrase);
    g_free (req->ssid);
    g_free (req);
}

static void
hostapd_request_async (gpointer data,
                       gpointer userdata)
{
    (void) userdata;
    int ret;
    size_t len;
    hostapdreq *req = data;
    char reply [HOSTAPD_REPLY_BUFFER];

    return_if_fail (req);

    INFO ("Request: %s for iface [%s] gets executed in thread: %ld",
          req->reqcmd, req->ifname, syscall (SYS_gettid));

    memset (reply, 0, sizeof (reply));
    len = sizeof (reply);
    ret = wpa_ctrl_request (req->sk->ctrl_sk,
                            req->reqcmd,
                            strlen (req->reqcmd),
                            reply, &len,
                            NULL);

    if (ret < 0) {
        ERROR ("Could not request the command: %s [error: %s]",
               req->reqcmd, hostapd_send_error_tostring (-ret));
        req->result = -APE_IO;
    } else {
        HOSTAPD_REPLY_REMOVE_NEWLINE (reply, len);
        req->reply = g_strdup (reply);
    }

    g_main_context_invoke_full (req->context,
                                G_PRIORITY_DEFAULT,
                                req->cb,
                                req,
                                hostapd_task_cleanup);
}

static int
hostapd_initialize_socket (const char *ifname,
                           wpasocket *sk,
                           const sockettype type,
                           hostapdsocket **hpdsock,
                           GIOFunc iofunc,
                           gpointer userdata)
{
    int socket, ret;
    hostapdsocket *hapsock;

    return_val_if_fail (ifname && sk && hpdsock, -EINVAL);
    return_val_if_fail (!*hpdsock, -EALREADY);

    /* invalid socket type */
    if (type != HOSTAPD_SOCKET_TYPE_GLOBAL &&
            type != HOSTAPD_SOCKET_TYPE_IFACE)
        return -EINVAL;

    /* Need a callback for ap socket to receive
     * the events */
    if (HOSTAPD_SOCKET_TYPE_IFACE == type && !iofunc)
        return -EINVAL;

    socket = wpa_ctrl_get_fd (sk);

    ret = fcntl (socket, F_SETFL, O_NONBLOCK | O_CLOEXEC);
    if (ret < 0)
        return -errno;

    hapsock = g_try_malloc0 (sizeof (hostapdsocket));
    return_val_if_fail (hapsock, -ENOMEM);

    hapsock->ifname = g_strdup (ifname);
    hapsock->type = type;

    if (HOSTAPD_SOCKET_TYPE_GLOBAL == type)
        hapsock->ctrl_sk = sk;
    else if (HOSTAPD_SOCKET_TYPE_IFACE == type) {

        hapsock->monitor_sk = sk;

        INFO ("Trying to create an IOChannel for the ctrl socket [%d] for "
              "the iface \"%s\"", socket, ifname);

        /* attach to the control interface socket for monitoring
         * asynchronous events */
        ret = wpa_ctrl_attach (sk);
        if (ret < 0) {
            CRITICAL ("Failed to attach to the hostapd for the iface: "
                      "%s [%d]", ifname, socket);
            goto error;
        }

        hapsock->iochannel = g_io_channel_unix_new (socket);

        g_io_channel_set_close_on_unref (hapsock->iochannel, TRUE);
        g_io_channel_set_encoding (hapsock->iochannel, NULL, NULL);
        g_io_channel_set_buffered (hapsock->iochannel, FALSE);

        hapsock->watch = g_io_add_watch (hapsock->iochannel,
                                         G_IO_IN | G_IO_NVAL
                                         | G_IO_HUP | G_IO_ERR,
                                         iofunc, userdata);

        INFO ("Successfully created an IOChannel for the ctrl socket [%d] [%p] "
              "for the iface \"%s\"", socket, hapsock->iochannel, ifname);
    }

    *hpdsock = hapsock;
    return 0;

error:
    g_free (hapsock->ifname);
    g_free (hapsock);
    *hpdsock = NULL;

    return ret;
}

static int
hostapd_is_global (const char *ctrl_path,
                   wpasocket **sk)
{
    int ret;
    size_t len;
    char reply [HOSTAPD_REPLY_BUFFER];
    wpasocket *sock;

    return_val_if_fail (ctrl_path && sk, -EINVAL);
    return_val_if_fail (!*sk, -EALREADY);

    sock = wpa_ctrl_open (ctrl_path);
    return_val_if_fail (sock, -ENOMEM);

    memset (reply, 0, sizeof (reply));
    /* space for null terminator */
    len = sizeof (reply) - 1;

    if ((ret = wpa_ctrl_request (sock, HOSTAPD_CTRL_CMD_PING,
                                 sizeof (HOSTAPD_CTRL_CMD_PING),
                                 reply, &len, NULL)) < 0) {
        ERROR ("Failed to ping the ctrl interface: %s [error: %s]",
               ctrl_path, hostapd_send_error_tostring (ret));
        wpa_ctrl_close (sock);
        return -ENOLINK;
    }

    HOSTAPD_REPLY_REMOVE_NEWLINE (reply, len);
    if (g_strcmp0 (reply, HOSTAPD_CTRL_CMD_PING_REPLY) != 0) {
        ERROR ("Unknown reply for ping ctrl command: [%s] [not expected]",
               reply);
        wpa_ctrl_close (sock);
        return -ENOTCONN;
    }

    memset (reply, 0, sizeof (reply));
    len = sizeof (reply) - 1;
    ret = wpa_ctrl_request (sock, HOSTAPD_GLOBAL_CTRL_CMD_ADD_INTERFACE,
                            sizeof (HOSTAPD_GLOBAL_CTRL_CMD_ADD_INTERFACE),
                            reply, &len, NULL);

    if (ret < 0) {
        ERROR ("Failed to send \"%s\" request to the ctrl interface: %s "
               "[error: %s]",
               HOSTAPD_GLOBAL_CTRL_CMD_ADD_INTERFACE,
               ctrl_path, hostapd_send_error_tostring (ret));
        wpa_ctrl_close (sock);
        return -ENOPROTOOPT;
    }

    HOSTAPD_REPLY_REMOVE_NEWLINE (reply, len);
    /* only global interface supports the command "ADD" */
    if (!g_strcmp0 (reply, HOSTAPD_CTRL_CMD_FAILED)) {
        INFO ("Found the global interface socket: \"%s\"", ctrl_path);
        *sk = sock;
        return 0;
    }

    wpa_ctrl_close (sock);
    return -ENOENT;
}

static int
hostapd_create_global (const char *hpddir)
{
    int ret;
    char path [PATH_MAX];
    DIR *dir;
    struct dirent *ent;
    GError *error = NULL;
    wpasocket *sk = NULL;

    return_val_if_fail (hpddir, -EINVAL);

    DEBUG ("hostapd socket dir: %s", hpddir);

    if (!(dir = opendir (hpddir))) {
        ERROR ("Failed to open the hostapd socket dir: %s [%s/%d]",
               hpddir, strerror (errno), errno);
        return -errno;
    }

    /* Heuristic for the global socket */
    while ((ent = readdir (dir))) {

        /* current directory */
        if (!g_strcmp0 (ent->d_name, "."))
            continue;

        /* parent directory */
        if (!g_strcmp0 (ent->d_name, ".."))
            continue;

        INFO ("Checking whether the interface \"%s\" is a global interface",
              ent->d_name);

        memset (path, 0, sizeof (path));
        sprintf (path, "%s/%s", HOSTAPD_SOCK_DIR_PATH, ent->d_name);
        ret = file_exists (path, FILE_TYPE_SOCK);
        if (ret <= 0) /* not a socket */
            continue;

        if (hostapd_is_global (path, &sk) < 0)
            continue;

        /* global interface found and initialized, we have
         * the system ready for access point management */
        hostapd = g_try_malloc0 (sizeof (hostapdglobal));
        if (!hostapd) {
            wpa_ctrl_close (sk);
            closedir (dir);
            return -ENOMEM;
        }

        hostapd->accesspoints = g_hash_table_new_full (g_str_hash,
                                                       g_str_equal,
                                                       NULL,
                                                       hostapd_accesspoint_cleanup);

        hostapd->threadpool =  g_thread_pool_new (hostapd_request_async,
                                                  NULL,
                                                  HOSTAPD_MAX_THREADS_PER_POOL,
                                                  FALSE,
                                                  &error);
        if (error) {
            ERROR ("Thread pool creation error: %s", error->message);
            g_error_free (error);
        }

        ret = hostapd_initialize_socket (ent->d_name, sk,
                                         HOSTAPD_SOCKET_TYPE_GLOBAL,
                                         &hostapd->global,
                                         NULL, NULL);
        if (ret < 0) {
            ERROR ("Failed to initialize the global ctrl socket of hostapd: %s",
                   ent->d_name);
            wpa_ctrl_close (sk);
            g_thread_pool_free (hostapd->threadpool,
                                TRUE, TRUE);
            g_free (hostapd);
            hostapd = NULL;
            closedir (dir);
            return -ENOTCONN;
        }

        hostapd_started = 1;
        closedir (dir);
        return 0;
    }

    closedir (dir);
    return -ENOENT;
}

static gboolean
hostapd_source_func (gpointer data)
{
    int ret;
    char *hpddir = data;

    return_val_if_fail (hpddir, FALSE);

    /* give up */
    if (hostapd_retry >= 10) {
        hostapd_retry = 0;
        g_free (hpddir);
        return FALSE;
    }

    ret = hostapd_create_global (hpddir);
    if (ret < 0) {
        ERROR ("Failed to create the hostapd global client : %s",
               strerror (-ret));
        hostapd_retry++;
        return TRUE;
    }

    hostapd_started = 1;
    hostapd_retry = 0;
    hostapd_callback_system_status (hostapd_started);
    g_free (hpddir);
    return FALSE;
}

static void
hostapd_global_cleanup (hostapdglobal **global)
{
    return_if_fail (global && *global);

    INFO ("Cleaning up the global socket : %s",
          (*global)->global->ifname);
    g_hash_table_destroy ((*global)->accesspoints);
    g_free ((*global)->global->ifname);
    wpa_ctrl_close ((*global)->global->ctrl_sk);
    g_thread_pool_free ((*global)->threadpool,
                        TRUE, TRUE);
    g_free ((*global)->global);
    g_free (*global);
    *global = NULL;
}

static void
hostapd_inotify (struct inotify_event *event,
				 const char *dir,
				 void *data)
{
	(void) data;

    int ret;

    return_if_fail (event && dir);

    INFO ("Inofity event for \"%s\" changed dir: %s file : %s",
          UNIX_SOCK_ROOT_DIR_PATH, dir, event->name ? event->name : "UNKNOWN");

    if (!hostapd_started) {
        if (event->mask & IN_CREATE) {
            INFO ("Hostapd process is started!!");
            ret = file_exists (HOSTAPD_SOCK_DIR_PATH, FILE_TYPE_DIR);
            if (ret < 0)
                return;

            /* Possible chances that the directory (/var/run/hostapd)
             * be empty due to the quick fire of inotify and the context
             * change. Thus we would end up in getting an empty directory.
             * Try some time before we decide something out of it!! */
            g_timeout_add_seconds (0, hostapd_source_func,
                                   g_strdup (HOSTAPD_SOCK_DIR_PATH));
        }
    } else {
        if (event->mask & IN_DELETE) {
            /* R.I.P */
            INFO ("Hostapd process is no more!!");
            /* <TODO> : Deintialize the hostapd global */
            hostapd_global_cleanup (&hostapd);
            hostapd_started = 0;
            hostapd_callback_system_status (hostapd_started);
        }
    }
}

static int
hostapd_global_deinit (void)
{
    int ret;

    ret = inotify_unregister (UNIX_SOCK_ROOT_DIR_PATH,
                              hostapd_inotify,
                              HOSTAPD_SOCK_DIR_NAME);
    if (ret < 0) {
        CRITICAL ("Failed to unregister to inotifier for \"%s\" "
                  "[error: %s/%d]",
                  UNIX_SOCK_ROOT_DIR_PATH, strerror (-ret), -ret);
    }

    hostapd_global_cleanup (&hostapd);
    return ret;
}

static int
hostapd_global_init ()
{
    int ret;

    ret = file_exists (UNIX_SOCK_ROOT_DIR_PATH, FILE_TYPE_DIR);
    if (ret < 0)
        return ret;

    ret = inotify_register (UNIX_SOCK_ROOT_DIR_PATH, hostapd_inotify,
                            HOSTAPD_SOCK_DIR_NAME, NULL);
    if (ret < 0) {
        CRITICAL ("Failed to register to inotifier for \"%s\" [%s/%d]",
                  UNIX_SOCK_ROOT_DIR_PATH, strerror (-ret), -ret);
        return ret;
    }

    ret = file_exists (HOSTAPD_SOCK_DIR_PATH, FILE_TYPE_DIR);
    if (ret < 0) {
        INFO ("Hostapd daemon is not yet started");
        ret = 0;
    } else {
        /* hostapd socket dir is present i.e., hostapd is already
         * started */
        ret = hostapd_create_global (HOSTAPD_SOCK_DIR_PATH);
        if (!ret) {
            hostapd_started = 1;
            ret = 0;
        }
    }

    return ret;
}

static int
hostapd_reply_to_result (const char *reply)
{
    return_val_if_fail (reply, -APE_FAILURE);

    if (!g_strcmp0 (reply, HOSTAPD_REQUEST_SUCCESS))
        return APE_SUCCESS;
    else if (!g_strcmp0 (reply, HOSTAPD_REQUEST_FAILURE))
        return -APE_FAILURE;

    return -APE_FAILURE;
}

static accesspoint*
hostapd_create_accesspoint (const char *ifname)
{
    int ret;
    wpasocket *ctrl_sk, *monitor_sk;
    accesspoint *ap;
    char path [PATH_MAX] = {0,};

    return_val_if_fail (ifname, NULL);

    sprintf (path, "%s/%s", HOSTAPD_SOCK_DIR_PATH, ifname);

    INFO ("Creating a wpa socket to control path : %s", path);
    ctrl_sk = wpa_ctrl_open (path);
    return_val_if_fail (ctrl_sk, NULL);
    monitor_sk = wpa_ctrl_open (path);
    if (!monitor_sk) {
        wpa_ctrl_close (ctrl_sk);
        return NULL;
    }

    ap = g_try_malloc0 (sizeof (*ap));
    if (!ap) {
        wpa_ctrl_close (ctrl_sk);
        wpa_ctrl_close (monitor_sk);
        return NULL;
    }

    ap->wpsenrollees = g_hash_table_new_full (g_str_hash,
                                              g_str_equal,
                                              NULL, hostapd_enrolle_cleanup);

    ret = hostapd_initialize_socket (ifname, monitor_sk,
                                     HOSTAPD_SOCKET_TYPE_IFACE,
                                     &ap->sock,
                                     hostapd_accesspoint_receive,
                                     ap);

    if (ret < 0) {
        wpa_ctrl_close (ctrl_sk);
        wpa_ctrl_close (monitor_sk);
        g_free (ap);
        return NULL;
    }

    ap->sock->acquired = 0;
    ap->sock->monitor_sk = monitor_sk;
    ap->sock->ctrl_sk = ctrl_sk;
    ap->started = 1;

    return ap;
}

static void
hostapd_regulatory_changed (const char *interface,
                            const unsigned int *frequencies,
                            const unsigned int *channels,
                            const unsigned int len)
{
    regdom_request *req = NULL;
    GSList *temp;
    gboolean avail = FALSE;

    return_if_fail (interface);
    return_if_fail (hostapd && hostapd_started);
    return_if_fail (callbacks && callbacks->regulatory_change);

    callbacks->regulatory_change (interface, frequencies, channels, len);

    for (temp = hostapd->regdom_requests; temp; temp = temp->next) {

        req = temp->data;
        if (!g_strcmp0 (interface, req->iface)) {

            avail = TRUE;
            if (req->regdom_source_event > 0)
                g_source_remove (req->regdom_source_event);

            if (req->sk)
                req->sk->acquired = 0;
            if (req->data && req->data->cb)
                req->data->cb (req->result, req->iface, req->data->userdata);

            regulatory_request_cleanup (req);
            break;
        }
    }

    if (avail)
        hostapd->regdom_requests = g_slist_remove (hostapd->regdom_requests, req);
}

static void
hostapd_scan_state_changed (const char *wiphy,
                            const char *ifname,
                            nl80211_scan_state state)
{
    return_if_fail (ifname && wiphy);
    return_if_fail (hostapd && hostapd_started);
    return_if_fail (callbacks && callbacks->scan_state_change);

    DEBUG ("Scan state change for the device: %s [interface: %s]",
           wiphy, ifname);

    callbacks->scan_state_change (wiphy, ifname, (int) state);
}

static regulatory_ops regulatoryops = {
    .clientname = "accesspoint",
    .reg_changed = hostapd_regulatory_changed,
};

static scanstate_ops scanops = {
    .clientname = "accesspoint",
    .scan_state_changed = hostapd_scan_state_changed,
};

static hostapdreq*
hostapd_create_request (const char *ifname,
                        const char *cmd,
                        GThreadPool *pool,
                        hostapdsocket *sk,
                        GMainContext *ctx,
                        apresultcb cb,
                        void *userdata)
{
    hostapdreq *req;

    return_val_if_fail (ifname && cmd && sk && ctx, NULL);

    req = g_try_malloc0 (sizeof (*req));
    return_val_if_fail (req, NULL);

    req->data = g_try_malloc0 (sizeof (*req->data));
    if (!req->data) {
        g_free (req);
        return NULL;
    }

    req->ifname = g_strdup (ifname);
    req->reqcmd = g_strdup (cmd);
    req->context = (void *) ctx;
    req->pool = pool;
    req->sk = sk;
    req->data->cb = cb;
    req->data->userdata  = userdata;

    return req;
}

static int
hostapd_request (GThreadPool *pool,
                 hostapdreq *request,
                 GSourceFunc cb)
{
    gboolean push;
    GError *error = NULL;

    return_val_if_fail (request && pool && cb, -APE_INVAL);

    request->cb = cb;
    push = g_thread_pool_push (pool, request, &error);
    if (!push) {
        ERROR ("Failed to add the task [%s] to interface [%s] threadpool, error: %s",
               request->reqcmd, request->ifname, error->message);
        g_error_free (error);
        return -APE_IO;
    }

    return APE_SUCCESS;
}

static gboolean
hostapd_addiface_failure_cb (gpointer data)
{
    int ret;
    hostapdreq *req = data;

    return_val_if_fail (req, G_SOURCE_REMOVE);

    INFO ("Interface: %s Request: %s result: %s thread: %ld",
          req->ifname, req->reqcmd, req->reply, syscall (SYS_gettid));

    ret = hostapd_reply_to_result (req->reply);
    if (!ret)
        ret = -APE_INTERNALFAIL;

    req->sk->acquired = 0;
    if (req->data->cb)
        req->data->cb (ret, req->ifname, req->data->userdata);

    if (hostapd && g_hash_table_contains
            (hostapd->accesspoints, req->ifname))
        g_hash_table_remove (hostapd->accesspoints, req->ifname);

    return G_SOURCE_REMOVE;
}

static gboolean
hostapd_addiface_cb (gpointer data)
{
    int ret = 0;
    char cmd [HOSTAPD_REQUEST_BUFFER];
    accesspoint *ap;
    hostapdreq *req, *removereq;

    req = data;
    return_val_if_fail (req, G_SOURCE_REMOVE);

    INFO ("Interface: %s Request: %s result: %s thread: %ld",
          req->ifname, req->reqcmd, req->reply, syscall (SYS_gettid));

    if (req->result < 0) {
        ERROR ("Failed to add the interface %s to hostapd error: %d",
               req->ifname, req->result);
        ret = req->result;
    } else {
        ret = hostapd_reply_to_result (req->reply);

        if (!ret) {
            ap = hostapd_create_accesspoint (req->ifname);
            if (!ap) {

                memset (cmd, 0, sizeof (cmd));
                sprintf (cmd, "%s %s", HOSTAPD_GLOBAL_CTRL_CMD_REMOVE_INTERFACE,
                         req->ifname);

                removereq = hostapd_create_request (req->ifname, cmd,
                                                    req->pool,
                                                    req->sk,
                                                    g_main_context_default(),
                                                    req->data->cb,
                                                    req->data->userdata);
                if (!removereq) {
                    ERROR ("Failed to clone the remove request");
                    ret = -APE_INTERNALFAIL;
                    goto error;
                }
                ret = hostapd_request (req->pool, removereq, hostapd_addiface_failure_cb);
                if (ret == APE_SUCCESS)
                    return G_SOURCE_REMOVE;
            } else {
                g_hash_table_replace (hostapd->accesspoints,
                                      ap->sock->ifname, ap);
            }
        }
    }

error:
    req->sk->acquired = 0;
    if (req->data->cb)
        req->data->cb (ret, req->ifname, req->data->userdata);
    return G_SOURCE_REMOVE;
}

#define PERFORM_GLOBAL_SANITY_CHECK() \
    do {    \
        return_val_if_fail (hostapd_started, -APE_SYSNOTREADY); \
        return_val_if_fail (hostapd, -APE_NOLINK);  \
        return_val_if_fail (!HOSTAPD_ALREADY_ACQUIRED (hostapd->global),    \
                            -APE_BUSY); \
    } while (0)

static gboolean
hostapd_acs_cb (gpointer data)
{
    acs_request *req = data;
    accesspoint *ap;

    return_val_if_fail (req, G_SOURCE_REMOVE);

    if (req->sk)
        req->sk->acquired = 0;

    if (req->data && req->data->cb)
        req->data->cb (-APE_TIMEOUT, req->iface, req->data->userdata);

    if (hostapd) {
        ap = g_hash_table_lookup (hostapd->accesspoints, req->iface);
        if (ap) {
            ap->acs_due = ap->acs_channels = 0;
            ap->acs_requests = g_slist_remove (ap->acs_requests, req);
        }
    }

    acs_request_cleanup (req);
    return G_SOURCE_REMOVE;
}

static gboolean
hostapd_addif_acs_timeout_cb (gpointer data)
{
    int ret = 0;
    acs_request *req = data;
    accesspoint *ap;
    char cmd [HOSTAPD_REQUEST_BUFFER];
    hostapdreq *removereq;

    return_val_if_fail (req, G_SOURCE_REMOVE);

    if (!hostapd)
        goto cleanup;

    ap = g_hash_table_lookup (hostapd->accesspoints, req->iface);
    if (!ap)
        goto cleanup;

    ap->acs_due = ap->acs_channels = 0;
    ap->acs_requests = g_slist_remove (ap->acs_requests, req);

    memset (cmd, 0, sizeof (cmd));
    sprintf (cmd, "%s %s", HOSTAPD_GLOBAL_CTRL_CMD_REMOVE_INTERFACE,
             req->iface);

    removereq = hostapd_create_request (req->iface, cmd,
                                        hostapd->threadpool,
                                        req->sk,
                                        g_main_context_default(),
                                        req->data->cb,
                                        req->data->userdata);
    if (!removereq) {
        ERROR ("Failed to create the remove request for: %s",
               req->iface);
        goto cleanup;
    }

    ret = hostapd_request (removereq->pool, removereq, hostapd_addiface_failure_cb);
    if (ret == APE_SUCCESS) {
        acs_request_cleanup (req);
        return G_SOURCE_REMOVE;
    }

cleanup:
    if (req->sk)
        req->sk->acquired = 0;

    if (req->data && req->data->cb)
        req->data->cb (-APE_TIMEOUT, req->iface, req->data->userdata);

    acs_request_cleanup (req);
    return G_SOURCE_REMOVE;
}

static gboolean
hostapd_addiface_acs_cb (gpointer data)
{
    int ret = 0;
    char cmd [HOSTAPD_REQUEST_BUFFER];
    accesspoint *ap;
    hostapdreq *req, *removereq;
    gboolean acs = FALSE;
    acs_request *acsreq;
    unsigned int timeout = 2;

    req = data;
    return_val_if_fail (req, G_SOURCE_REMOVE);

    INFO ("Interface: %s Request: %s result: %s thread: %ld",
          req->ifname, req->reqcmd, req->reply, syscall (SYS_gettid));

    if (req->result < 0) {
        ERROR ("Failed to add the interface %s to hostapd error: %d",
               req->ifname, req->result);
        ret = req->result;
    } else {
        ret = hostapd_reply_to_result (req->reply);

        if (ret != APE_SUCCESS)
            goto error;

        ap = hostapd_create_accesspoint (req->ifname);
        if (!ap) {

            memset (cmd, 0, sizeof (cmd));
            sprintf (cmd, "%s %s", HOSTAPD_GLOBAL_CTRL_CMD_REMOVE_INTERFACE,
                     req->ifname);

            removereq = hostapd_create_request (req->ifname, cmd,
                                                req->pool,
                                                req->sk,
                                                g_main_context_default(),
                                                req->data->cb,
                                                req->data->userdata);
            if (!removereq) {
                ERROR ("Failed to clone the remove request");
                ret = -APE_INTERNALFAIL;
                goto error;
            }
            ret = hostapd_request (req->pool, removereq, hostapd_addiface_failure_cb);
            if (ret == APE_SUCCESS)
                return G_SOURCE_REMOVE;
        } else {

            g_hash_table_replace (hostapd->accesspoints, ap->sock->ifname, ap);
            acsreq = g_try_malloc0 (sizeof (*acsreq));
            if (!acsreq)
                goto error;

            acsreq->data = g_try_malloc0 (sizeof (*acsreq->data));
            if (!acsreq->data) {
                g_free (acsreq);
                goto error;
            }

            acsreq->sk = req->sk;
            acsreq->iface = g_strdup (req->ifname);
            acsreq->data->cb = req->data->cb;
            acsreq->data->userdata = req->data->userdata;

            /* Most wifi devices take a minimum of 100ms to perform
             * a scan. So we set out a timeout of
             * (num_of_channels * 100ms * number_of_scans * 1 (round_up))
             * + 2 (delta) */
            if (req->acs_channels)
                acsreq->timeout = timeout + (unsigned int)
                        ((((float)(req->acs_channels * 100) / 1000) * 5.0f) + 1.0f);

            INFO ("Setting a timeout of \"%u\" seconds for ACS on AP: %s "
                  "[req: %p]", acsreq->timeout, acsreq->iface, acsreq);
            acsreq->timeout_event = g_timeout_add_seconds
                    (acsreq->timeout, hostapd_addif_acs_timeout_cb, acsreq);
            ap->acs_due = 1;
            acs = TRUE;
            ap->acs_channels = req->acs_channels;
            ap->acs_requests = g_slist_append (ap->acs_requests, acsreq);
        }
    }

    if (acs)
        return G_SOURCE_REMOVE;

error:
    req->sk->acquired = 0;
    if (req->data->cb)
        req->data->cb (ret, req->ifname, req->data->userdata);
    return G_SOURCE_REMOVE;
}

int
accesspoint_add_interface (const char *ifname,
                           const char *configfile,
                           int acs,
                           int num_acs_chans,
                           apresultcb cb,
                           void *userdata)
{
    hostapdreq *req;
    accesspoint *ap;
    const char *phyname;
    char cmd [HOSTAPD_REQUEST_BUFFER];
    int ret;

    return_val_if_fail (ifname && configfile, -APE_INVAL);
    PERFORM_GLOBAL_SANITY_CHECK ();

    ap = g_hash_table_lookup (hostapd->accesspoints, ifname);
    return_val_if_fail (!ap, -APE_ALREADY);
    ret = genl_interface_supported (ifname);
    if (ret < 0)
        return -APE_NO80211IFACE;

    memset (cmd, 0, sizeof (cmd));
    phyname = genl_get_phyname (ifname);
    if (!phyname)
        phyname = "phy0"; /* generic wireless device name */

    sprintf (cmd, "%s %s=%s:%s", HOSTAPD_GLOBAL_CTRL_CMD_ADD_INTERFACE,
             HOSTAPD_GLOBAL_CTRL_CMD_ADD_BSS_CONFIG, phyname,
             configfile);

    req = hostapd_create_request (ifname, cmd,
                                  hostapd->threadpool,
                                  hostapd->global,
                                  g_main_context_default(),
                                  cb, userdata);
    return_val_if_fail (req, -APE_INTERNALFAIL);

    req->acs_required = acs;
    req->acs_channels = num_acs_chans;

    ret = hostapd_request (req->pool, req, acs == 0 ?
                               hostapd_addiface_cb : hostapd_addiface_acs_cb);
    if (APE_SUCCESS == ret)
        hostapd->global->acquired = 1;

    return ret;
}

static gboolean
hostapd_rmiface_cb (gpointer data)
{
    int ret = 0;
    hostapdreq *req = data;
    gboolean remove;

    return_val_if_fail (req, G_SOURCE_REMOVE);

    INFO ("Interface: %s Request: %s result: %s thread: %ld",
          req->ifname, req->reqcmd, req->reply, syscall (SYS_gettid));

    if (req->result < 0) {
        ERROR ("Failed to remove the interface %s to hostapd error : %d",
               req->ifname, req->result);
        ret = req->result;
    } else {
        ret = hostapd_reply_to_result (req->reply);
        if (!ret) {
            remove = g_hash_table_remove (hostapd->accesspoints, req->ifname);
            if (remove)
                INFO ("Successfully removed \"%s\" from hostapd", req->ifname);
        }
    }

    req->sk->acquired = 0;
    if (req->data->cb)
        req->data->cb (ret, req->ifname, req->data->userdata);
    return G_SOURCE_REMOVE;
}

int
accesspoint_remove_interface (const char *ifname,
                              apresultcb cb,
                              void *userdata)
{
    int ret;
    hostapdreq *req;
    accesspoint *ap;
    char cmd [HOSTAPD_REQUEST_BUFFER];

    return_val_if_fail (ifname, -APE_INVAL);
    PERFORM_GLOBAL_SANITY_CHECK ();

    ap = g_hash_table_lookup (hostapd->accesspoints, ifname);
    return_val_if_fail (ap, -APE_NOENT);

    memset (cmd, 0, sizeof (cmd));
    sprintf (cmd, "%s %s", HOSTAPD_GLOBAL_CTRL_CMD_REMOVE_INTERFACE,
             ifname);

    req = hostapd_create_request (ifname, cmd,
                                  hostapd->threadpool,
                                  hostapd->global,
                                  g_main_context_default(),
                                  cb, userdata);
    return_val_if_fail (req, -APE_INTERNALFAIL);
    ret = hostapd_request (req->pool, req, hostapd_rmiface_cb);
    if (APE_SUCCESS == ret)
        hostapd->global->acquired = 1;

    return ret;
}

static gboolean
hostapd_apgeneric_cb (gpointer data)
{
    int ret = 0;
    hostapdreq *req = data;
    accesspoint *ap;

    return_val_if_fail (req, G_SOURCE_REMOVE);

    INFO ("Interface: %s Request: %s result: %s thread: %ld",
          req->ifname, req->reqcmd, req->reply, syscall (SYS_gettid));

    if (req->result < 0) {
        ERROR ("Command failed for the interface %s to hostapd error: %d",
               req->ifname, req->result);
        ret = req->result;
    } else {
        ret = hostapd_reply_to_result (req->reply);
        if (APE_SUCCESS == ret && hostapd &&
                req->acs_required) {
            ap = g_hash_table_lookup (hostapd->accesspoints, req->ifname);
            if (ap) {
                INFO ("ACS is due for the access point: %s [channels: %d]",
                      req->ifname, req->acs_channels);
                ap->acs_due = 1;
                ap->acs_channels = req->acs_channels;
            }
        }
    }

    req->sk->acquired = 0;
    if (req->data->cb)
        req->data->cb (ret, req->ifname, req->data->userdata);
    return G_SOURCE_REMOVE;
}

#define PERFORM_AP_SANITY_CHECK(ifname, start, global, ap)   \
    do {    \
        return_val_if_fail (ifname, -APE_INVAL);    \
        return_val_if_fail (start, -APE_SYSNOTREADY); \
        return_val_if_fail (global, -APE_NOLINK);  \
        *ap = g_hash_table_lookup (global->accesspoints, ifname);   \
        return_val_if_fail (*ap, -APE_NOENT);    \
    } while (0)

static gboolean
hostapd_configure_acs_cb (gpointer data)
{
    int ret = 0;
    hostapdreq *req = data;
    gboolean acs_operation = FALSE;
    accesspoint *ap = NULL;
    unsigned int timeout = 2;
    acs_request *acsreq = NULL;

    return_val_if_fail (req, G_SOURCE_REMOVE);

    INFO ("Interface: %s Request: %s result: %s thread: %ld",
          req->ifname, req->reqcmd, req->reply, syscall (SYS_gettid));

    if (hostapd)
        ap = g_hash_table_lookup (hostapd->accesspoints, req->ifname);

    if (req->result < 0) {
        ERROR ("Command failed for the interface %s to hostapd error: %d",
               req->ifname, req->result);
        ret = req->result;
    } else {
        ret = hostapd_reply_to_result (req->reply);
        if (ret != APE_SUCCESS)
            goto failure;

        if (!req->acs_required)
            goto failure;

        if (!ap)
            goto failure;

        acsreq = g_try_malloc0 (sizeof (*acsreq));
        if (!acsreq)
            goto failure;

        acsreq->data = g_try_malloc0 (sizeof (*acsreq->data));
        if (!acsreq->data) {
            g_free (acsreq);
            goto failure;
        }

        acsreq->sk = req->sk;
        acsreq->iface = g_strdup (req->ifname);
        acsreq->data->cb = req->data->cb;
        acsreq->data->userdata = req->data->userdata;

        if (req->acs_channels)
            acsreq->timeout = timeout + (unsigned int)
                    ((((float)(req->acs_channels * 100) / 1000) * 5.0f) + 1.0f);

        INFO ("Setting a timeout of \"%u\" seconds for ACS on AP: %s [req: %p]",
              acsreq->timeout, acsreq->iface, acsreq);
        acsreq->timeout_event = g_timeout_add_seconds
                (acsreq->timeout, hostapd_acs_cb, acsreq);
        acs_operation = TRUE;
        ap->acs_requests = g_slist_append (ap->acs_requests, acsreq);
    }

    if (acs_operation)
        return G_SOURCE_REMOVE;

failure:
    if (ap) {
        ap->acs_due = 0;
        ap->acs_channels = 0;
    }
    req->sk->acquired = 0;
    if (req->data->cb)
        req->data->cb (ret, req->ifname, req->data->userdata);
    return G_SOURCE_REMOVE;
}

int
accesspoint_enable (const char *ifname,
                    apresultcb cb,
                    void *userdata)
{
    hostapdreq *req;
    accesspoint *ap = NULL;
    int ret;
    char cmd [HOSTAPD_REQUEST_BUFFER];

    PERFORM_AP_SANITY_CHECK (ifname, hostapd_started, hostapd, &ap);
    return_val_if_fail (!ap->started, -APE_ALREADY);
    return_val_if_fail (!HOSTAPD_ALREADY_ACQUIRED (ap->sock), -APE_BUSY);

    memset (cmd, 0, sizeof (cmd));
    sprintf (cmd, "%s", HOSTAPD_AP_CTRL_CMD_ENABLE);

    req = hostapd_create_request (ifname, cmd,
                                  hostapd->threadpool,
                                  ap->sock,
                                  g_main_context_default(),
                                  cb, userdata);
    return_val_if_fail (req, -APE_INTERNALFAIL);

    req->acs_required = ap->acs_channels;
    req->acs_channels = ap->acs_channels;

    ret = hostapd_request (req->pool, req, ap->acs_due ?
                               hostapd_configure_acs_cb : hostapd_apgeneric_cb);
    if (APE_SUCCESS == ret)
        ap->sock->acquired = 1;

    return ret;
}

int
accesspoint_disable (const char *ifname,
                     apresultcb cb,
                     void *userdata)
{
    hostapdreq *req;
    accesspoint *ap = NULL;
    int ret;
    char cmd [HOSTAPD_REQUEST_BUFFER];

    PERFORM_AP_SANITY_CHECK (ifname, hostapd_started, hostapd, &ap);
    return_val_if_fail (ap->started, -APE_NOTSTARTED);
    return_val_if_fail (!HOSTAPD_ALREADY_ACQUIRED (ap->sock), -APE_BUSY);

    memset (cmd, 0, sizeof (cmd));
    sprintf (cmd, "%s", HOSTAPD_AP_CTRL_CMD_DISABLE);

    req = hostapd_create_request (ifname, cmd,
                                  hostapd->threadpool,
                                  ap->sock,
                                  g_main_context_default(),
                                  cb, userdata);
    return_val_if_fail (req, -APE_INTERNALFAIL);
    ret = hostapd_request (req->pool, req, hostapd_apgeneric_cb);
    if (APE_SUCCESS == ret)
        ap->sock->acquired = 1;

    return ret;
}

int
accesspoint_reload (const char *ifname,
                    apresultcb cb,
                    void *userdata)
{
    hostapdreq *req;
    accesspoint *ap = NULL;
    int ret;
    char cmd [HOSTAPD_REQUEST_BUFFER];

    PERFORM_AP_SANITY_CHECK (ifname, hostapd_started, hostapd, &ap);
    return_val_if_fail (ap->started, -APE_NOTSTARTED);
    return_val_if_fail (!HOSTAPD_ALREADY_ACQUIRED (ap->sock), -APE_BUSY);

    memset (cmd, 0, sizeof (cmd));
    sprintf (cmd, "%s", HOSTAPD_AP_CTRL_CMD_RELOAD);

    req = hostapd_create_request (ifname, cmd,
                                  hostapd->threadpool,
                                  ap->sock,
                                  g_main_context_default(),
                                  cb, userdata);
    return_val_if_fail (req, -APE_INTERNALFAIL);
    ret = hostapd_request (req->pool, req, hostapd_apgeneric_cb);
    if (APE_SUCCESS == ret)
        ap->sock->acquired = 1;

    return ret;
}

int
accesspoint_disassociate_station (const char *ifname,
                                  const char *station,
                                  apresultcb cb,
                                  void *userdata)
{
    hostapdreq *req;
    accesspoint *ap = NULL;
    int ret;
    char cmd [HOSTAPD_REQUEST_BUFFER];

    return_val_if_fail (station, -APE_INVAL);
    PERFORM_AP_SANITY_CHECK (ifname, hostapd_started, hostapd, &ap);
    return_val_if_fail (ap->started, -APE_NOTSTARTED);
    return_val_if_fail (!HOSTAPD_ALREADY_ACQUIRED (ap->sock), -APE_BUSY);

    memset (cmd, 0, sizeof (cmd));
    sprintf (cmd, "%s %s", HOSTAPD_AP_CTRL_CMD_DISASSOCIATE_STATION,
             station);

    req = hostapd_create_request (ifname, cmd,
                                  hostapd->threadpool,
                                  ap->sock,
                                  g_main_context_default(),
                                  cb, userdata);
    return_val_if_fail (req, -APE_INTERNALFAIL);
    ret = hostapd_request (req->pool, req, hostapd_apgeneric_cb);
    if (APE_SUCCESS == ret)
        ap->sock->acquired = 1;

    return ret;
}

int
accesspoint_deauthenticate_station (const char *ifname,
                                    const char *station,
                                    apresultcb cb,
                                    void *userdata)
{
    hostapdreq *req;
    accesspoint *ap = NULL;
    int ret;
    char cmd [HOSTAPD_REQUEST_BUFFER];

    return_val_if_fail (station, -APE_INVAL);
    PERFORM_AP_SANITY_CHECK (ifname, hostapd_started, hostapd, &ap);
    return_val_if_fail (ap->started, -APE_NOTSTARTED);
    return_val_if_fail (!HOSTAPD_ALREADY_ACQUIRED (ap->sock), -APE_BUSY);

    memset (cmd, 0, sizeof (cmd));
    sprintf (cmd, "%s %s", HOSTAPD_AP_CTRL_CMD_DEAUTHENTICATE_STATION,
             station);

    req = hostapd_create_request (ifname, cmd,
                                  hostapd->threadpool,
                                  ap->sock,
                                  g_main_context_default(),
                                  cb, userdata);
    return_val_if_fail (req, -APE_INTERNALFAIL);
    ret = hostapd_request (req->pool, req, hostapd_apgeneric_cb);
    if (APE_SUCCESS == ret)
        ap->sock->acquired = 1;

    return ret;
}


static gboolean
hostapd_setreq_cb (gpointer data)
{
    int ret = 0;
    hostapdreq *req, *reload;
    char cmd [HOSTAPD_REQUEST_BUFFER];

    req = data;
    return_val_if_fail (req, G_SOURCE_REMOVE);

    INFO ("Interface: %s Request: %s result: %s thread: %ld",
          req->ifname, req->reqcmd, req->reply, syscall (SYS_gettid));

    if (req->result < 0) {
        ERROR ("Command failed for the interface %s to hostapd error: %d",
               req->ifname, req->result);
        ret = req->result;
    } else {
        ret = hostapd_reply_to_result (req->reply);
        if (ret == APE_SUCCESS) {
            memset (cmd, 0, sizeof (cmd));
            sprintf (cmd, "%s", HOSTAPD_AP_CTRL_CMD_RELOAD);

            reload = hostapd_create_request (req->ifname, cmd,
                                             req->pool,
                                             req->sk,
                                             req->context,
                                             req->data->cb,
                                             req->data->userdata);
            if (!reload) {
                ERROR ("Failed to reload the accesspoint \"%s\" after setting: %s",
                       req->ifname, req->reqcmd);
                ret = -APE_INTERNALFAIL;
            } else {
                ret = hostapd_request (reload->pool, reload, hostapd_apgeneric_cb);
                if (ret == APE_SUCCESS)
                    return G_SOURCE_REMOVE;
            }
        }
    }

    req->sk->acquired = 0;
    if (req->data->cb)
        req->data->cb (ret, req->ifname, req->data->userdata);
    return G_SOURCE_REMOVE;
}

int
accesspoint_set_passphrase (const char *ifname,
                            const char *passphrase,
                            apresultcb cb,
                            void *userdata)
{
    accesspoint *ap = NULL;
    size_t length;
    int ret;
    hostapdreq *req;
    char cmd [HOSTAPD_REQUEST_BUFFER];

    return_val_if_fail (passphrase, -APE_INVAL);

    length = strlen (passphrase);
    if (length < PASSPHRASE_MIN_LENGTH ||
            length > PASSPHRASE_MAX_LENGTH)
        return -APE_INVAL;

    PERFORM_AP_SANITY_CHECK (ifname, hostapd_started, hostapd, &ap);
    return_val_if_fail (!HOSTAPD_ALREADY_ACQUIRED (ap->sock), -APE_BUSY);

    memset (cmd, 0, sizeof (cmd));
    sprintf (cmd, "%s %s %s", HOSTAPD_AP_CTRL_CMD_SET,
             HOSTAPD_AP_CTRL_CMD_SET_PASSPHRASE, passphrase);

    req = hostapd_create_request (ifname, cmd,
                                  hostapd->threadpool,
                                  ap->sock,
                                  g_main_context_default(),
                                  cb, userdata);
    return_val_if_fail (req, -APE_INTERNALFAIL);
    ret = hostapd_request (req->pool, req, hostapd_apgeneric_cb);
    if (APE_SUCCESS == ret)
        ap->sock->acquired = 1;

    return ret;
}

static const char*
hostapd_security (securitytype type)
{
    switch (type) {

    case NUM_AP_SECURITIES:
        return "2";
    case AP_SECURITY_OPEN:
    case AP_SECURITY_WEP:
        return "0";
    case AP_SECURITY_WPA_PSK:
        return "1";
    case AP_SECURITY_WPA2_PSK:
        return "2";
    case AP_SECURITY_WPA_WPA2_PSK:
        return "3";
    default:
    return "2";
    }
}

static gboolean
hostapd_setauthalgs_cb (gpointer data)
{
    int ret = 0;
    hostapdreq *req, *reload;
    char cmd [HOSTAPD_REQUEST_BUFFER];

    req = data;
    return_val_if_fail (req, G_SOURCE_REMOVE);

    INFO ("Interface: %s Request: %s result: %s thread: %ld",
          req->ifname, req->reqcmd, req->reply, syscall (SYS_gettid));

    if (req->result < 0) {
        ERROR ("Command failed for the interface %s to hostapd error: %d",
               req->ifname, req->result);
        ret = req->result;
    } else {
        ret = hostapd_reply_to_result (req->reply);
        if (ret == APE_SUCCESS) {
            memset (cmd, 0, sizeof (cmd));
            sprintf (cmd, "%s %s %s", HOSTAPD_AP_CTRL_CMD_SET,
                     HOSTAPD_AP_CTRL_CMD_SET_SECURITY,
                     hostapd_security (req->security->type));

            reload = hostapd_create_request (req->ifname, cmd,
                                             req->pool,
                                             req->sk,
                                             req->context,
                                             req->data->cb,
                                             req->data->userdata);
            if (!reload) {
                ERROR ("Failed to reload the accesspoint: %s", req->ifname);
                ret = -APE_INTERNALFAIL;
            } else {
                ret = hostapd_request (reload->pool, reload, hostapd_setreq_cb);
                if (ret == APE_SUCCESS)
                    return G_SOURCE_REMOVE;
            }
        }
    }

    req->sk->acquired = 0;
    if (req->data->cb)
        req->data->cb (ret, req->ifname, req->data->userdata);
    return G_SOURCE_REMOVE;
}

static gboolean
hostapd_setwepdefault_cb (gpointer data)
{
    int ret = 0;
    hostapdreq *req, *wep;
    securitydata *sec;
    char cmd [HOSTAPD_REQUEST_BUFFER];

    req = data;
    return_val_if_fail (req, G_SOURCE_REMOVE);

    INFO ("Interface: %s Request: %s Result: %s thread: %ld",
          req->ifname, req->reqcmd, req->reply, syscall (SYS_gettid));

    if (req->result < 0) {
        ERROR ("Command failed for the interface %s to hostapd error: %d",
               req->ifname, req->result);
        ret = req->result;
    } else {
        ret = hostapd_reply_to_result (req->reply);
        if (ret == APE_SUCCESS) {
            memset (cmd, 0, sizeof (cmd));
            sprintf (cmd, "%s %s %s", HOSTAPD_AP_CTRL_CMD_SET,
                     HOSTAPD_AP_CTRL_CMD_SET_WEPDEFAULT,
                     (req->security->type == AP_SECURITY_WEP) ?
                         "0" : "1");

            wep = hostapd_create_request (req->ifname, cmd,
                                          req->pool,
                                          req->sk,
                                          req->context,
                                          req->data->cb,
                                          req->data->userdata);
            if (!wep) {
                ERROR ("Failed to reload the accesspoint: %s", req->ifname);
                ret = -APE_INTERNALFAIL;
            } else {
                sec = g_try_malloc0 (sizeof (*sec));
                if (!sec) {
                    hostapd_task_cleanup (wep);
                    ret = -APE_INTERNALFAIL;
                    goto error;
                }
                sec->type = req->security->type;
                wep->security = sec;
                ret = hostapd_request (wep->pool, wep, hostapd_setauthalgs_cb);
                if (ret == APE_SUCCESS)
                    return G_SOURCE_REMOVE;
            }
        }
    }

error:
    req->sk->acquired = 0;
    if (req->data->cb)
        req->data->cb (ret, req->ifname, req->data->userdata);
    return G_SOURCE_REMOVE;
}

static gboolean
hostapd_setwepkey_cb (gpointer data)
{
    int ret = 0;
    hostapdreq *req, *wep;
    securitydata *sec;
    char cmd[HOSTAPD_REQUEST_BUFFER];

    req = data;
    return_val_if_fail (req, G_SOURCE_REMOVE);

    INFO ("Interface: %s Request: %s result: %s thread: %ld",
          req->ifname, req->reqcmd, req->reply, syscall (SYS_gettid));

    if (req->result < 0) {
        ERROR ("Command failed for the interface %s to hostapd error: %d",
               req->ifname, req->result);
        ret = req->result;
    } else {
        ret = hostapd_reply_to_result (req->reply);
        if (ret == APE_SUCCESS) {
            memset (cmd, 0, sizeof (cmd));
            sprintf (cmd, "%s %s %s", HOSTAPD_AP_CTRL_CMD_SET,
                     HOSTAPD_AP_CTRL_CMD_SET_AUTH_ALGS,
                     (AP_SECURITY_WEP == req->security->type) ? "2" : "1");

            wep = hostapd_create_request (req->ifname, cmd,
                                          req->pool,
                                          req->sk,
                                          req->context,
                                          req->data->cb,
                                          req->data->userdata);
            if (!wep) {
                ERROR ("Failed to reload the accesspoint: %s", req->ifname);
                ret = -APE_INTERNALFAIL;
            } else {
                sec = g_try_malloc0 (sizeof (*sec));
                if (!sec) {
                    hostapd_task_cleanup (wep);
                    ret = -APE_INTERNALFAIL;
                    goto error;
                }
                sec->type = req->security->type;
                wep->security = sec;
                ret = hostapd_request (wep->pool, wep, hostapd_setwepdefault_cb);
                if (ret == APE_SUCCESS)
                    return G_SOURCE_REMOVE;
            }
        }
    }

error:
    req->sk->acquired = 0;
    if (req->data->cb)
        req->data->cb (ret, req->ifname, req->data->userdata);
    return G_SOURCE_REMOVE;
}

int
accesspoint_set_security (const char *ifname,
                          const securitytype sec,
                          const char *wep_key,
                          apresultcb cb,
                          void *userdata)
{
    accesspoint *ap = NULL;
    size_t len;
    int ret;
    hostapdreq *req;
    securitydata *data;
    char cmd [HOSTAPD_REQUEST_BUFFER];
    GSourceFunc callback;

    if (NUM_AP_SECURITIES == sec)
        return -APE_INVAL;

    PERFORM_AP_SANITY_CHECK (ifname, hostapd_started, hostapd, &ap);
    return_val_if_fail (!HOSTAPD_ALREADY_ACQUIRED (ap->sock), -APE_BUSY);

    if (AP_SECURITY_WEP == sec) {

        if (!wep_key)
            return -APE_INVAL;

        len = strlen (wep_key);
        if (len != 5 && len != 13 && len != 16 &&
                len != 10 && len != 26 && len != 32)
            return -APE_INVAL;
    }

    data = g_try_malloc0 (sizeof (*data));
    if (!data)
        return -APE_INTERNALFAIL;

    data->type  = sec;
    memset (cmd, 0, sizeof (cmd));

    if (AP_SECURITY_WEP == sec) {
        sprintf (cmd, "%s %s \"%s\"", HOSTAPD_AP_CTRL_CMD_SET,
                 HOSTAPD_AP_CTRL_CMD_SET_WEPKEY, wep_key);
        callback = hostapd_setwepkey_cb;
    }
    else if (AP_SECURITY_OPEN == sec){
        sprintf (cmd, "%s %s %s", HOSTAPD_AP_CTRL_CMD_SET,
                 HOSTAPD_AP_CTRL_CMD_SET_AUTH_ALGS,
                 "1");
        callback = hostapd_setauthalgs_cb;
    } else {
        sprintf (cmd, "%s %s %s", HOSTAPD_AP_CTRL_CMD_SET,
                 HOSTAPD_AP_CTRL_CMD_SET_RSN_PAIRWISE,
                 HOSTAPD_KEY_PAIRWISE_CCMP);
        callback = hostapd_setwepkey_cb;
    }

    req = hostapd_create_request (ifname, cmd,
                                  hostapd->threadpool,
                                  ap->sock,
                                  g_main_context_default(),
                                  cb, userdata);
    if (!req) {
        g_free (data);
        return -APE_INTERNALFAIL;
    }

    req->security = data;
    ret = hostapd_request (req->pool, req, callback);
    if (APE_SUCCESS == ret)
        ap->sock->acquired = 1;

    return ret;
}

static gboolean
hostapd_setopchannels_cb (gpointer data)
{
    int ret = 0;
    hostapdreq *req, *channel;
    char cmd [HOSTAPD_REQUEST_BUFFER];

    req = data;
    return_val_if_fail (req, G_SOURCE_REMOVE);

    INFO ("Interface: %s Request: %s result: %s thread : %ld",
          req->ifname, req->reqcmd, req->reply, syscall (SYS_gettid));

    if (req->result < 0) {
        ERROR ("Command failed for the interface %s to hostapd error: %d",
               req->ifname, req->result);
        ret = req->result;
    } else {
        ret = hostapd_reply_to_result (req->reply);
        if (ret == APE_SUCCESS) {
            memset (cmd, 0, sizeof (cmd));
            sprintf (cmd, "%s %s %d", HOSTAPD_AP_CTRL_CMD_SET,
                             HOSTAPD_AP_CTRL_CMD_SET_CHANNEL, 0);

            channel = hostapd_create_request (req->ifname, cmd,
                                              req->pool,
                                              req->sk,
                                              req->context,
                                              req->data->cb,
                                              req->data->userdata);
            if (!channel) {
                ERROR ("Failed to configure acs for the access point: %s",
                       req->ifname);
                ret = -APE_INTERNALFAIL;
            } else {
                channel->acs_required = req->acs_required;
                channel->acs_channels = req->acs_channels;
                ret = hostapd_request (channel->pool, channel, hostapd_apgeneric_cb);
                if (ret == APE_SUCCESS)
                    return G_SOURCE_REMOVE;
            }
        }
    }

    req->sk->acquired = 0;
    if (req->data->cb)
        req->data->cb (ret, req->ifname, req->data->userdata);
    return G_SOURCE_REMOVE;
}

int
accesspoint_set_operating_channel (const char *ifname,
                                   const unsigned int channel [],
                                   const unsigned int chanlen,
                                   apresultcb cb,
                                   void *userdata)
{
    accesspoint *ap = NULL;
    hostapdreq *req;
    char cmd [HOSTAPD_REQUEST_BUFFER];
    char chan [33];
    int ret, acs = 0;
    GSourceFunc callback;
    int index = 0; unsigned int length = 0;

    if (chanlen <= 0)
        return -APE_INVAL;

    PERFORM_AP_SANITY_CHECK (ifname, hostapd_started, hostapd, &ap);
    return_val_if_fail (!HOSTAPD_ALREADY_ACQUIRED (ap->sock), -APE_BUSY);

    memset (cmd, 0, sizeof (cmd));
    if (1 == chanlen) {
        sprintf (cmd, "%s %s %d", HOSTAPD_AP_CTRL_CMD_SET,
                 HOSTAPD_AP_CTRL_CMD_SET_CHANNEL, channel [0]);
        callback = hostapd_apgeneric_cb;
    } else {

        sprintf (cmd, "%s %s", HOSTAPD_AP_CTRL_CMD_SET,
                 HOSTAPD_AP_CTRL_CMD_SET_CHANNELLIST);

        length = chanlen;
        while (length) {
            memset (chan, 0, sizeof (chan));
            sprintf (chan, " %d", channel [index]);
            index++; length--;
            strcat (cmd, chan);
        }

        acs = 1;
        callback = hostapd_setopchannels_cb;
    }

    req = hostapd_create_request (ifname, cmd,
                                  hostapd->threadpool,
                                  ap->sock,
                                  g_main_context_default(),
                                  cb, userdata);
    return_val_if_fail (req, -APE_INTERNALFAIL);
    req->acs_required = acs;
    req->acs_channels = index;

    /* In order to complete the ACS, after setting the chanlist,
     * it is necessary to disable and reenable the AP which in
     * fact is the responsibility of the caller, but in case if
     * the AP was already disabled and there are just calls which
     * plays around operating channels, make sure that we dont
     * fall apart */
    if (!req->acs_required) {
        ap->acs_due = ap->acs_channels = 0;
    }

    ret = hostapd_request (req->pool, req, callback);
    if (APE_SUCCESS == ret)
        ap->sock->acquired = 1;

    return ret;
}

int
accesspoint_set_hwmode (const char *ifname,
                        const char *mode,
                        apresultcb cb,
                        void *userdata)
{
    accesspoint *ap = NULL;
    hostapdreq *req;
    int ret;
    char cmd [HOSTAPD_REQUEST_BUFFER];
    GSourceFunc callback;

    PERFORM_AP_SANITY_CHECK (ifname, hostapd_started, hostapd, &ap);
    return_val_if_fail (!HOSTAPD_ALREADY_ACQUIRED (ap->sock), -APE_BUSY);

    memset (cmd, 0, sizeof (cmd));
    sprintf (cmd, "%s %s %s", HOSTAPD_AP_CTRL_CMD_SET,
             HOSTAPD_AP_CTRL_CMD_SET_HWMODE, mode);
    callback = hostapd_apgeneric_cb;

    req = hostapd_create_request (ifname, cmd,
                                  hostapd->threadpool,
                                  ap->sock,
                                  g_main_context_default(),
                                  cb, userdata);
    return_val_if_fail (req, -APE_INTERNALFAIL);
    ret = hostapd_request (req->pool, req, callback);
    if (APE_SUCCESS == ret)
        ap->sock->acquired = 1;

    return ret;
}

int
accesspoint_set_hidden (const char *ifname,
                        const int hidden,
                        apresultcb cb,
                        void *userdata)
{
    accesspoint *ap = NULL;
    hostapdreq *req;
    int ret;
    char cmd [HOSTAPD_REQUEST_BUFFER];
    GSourceFunc callback;

    PERFORM_AP_SANITY_CHECK (ifname, hostapd_started, hostapd, &ap);
    return_val_if_fail (!HOSTAPD_ALREADY_ACQUIRED (ap->sock), -APE_BUSY);

    memset (cmd, 0, sizeof (cmd));
    sprintf (cmd, "%s %s %d", HOSTAPD_AP_CTRL_CMD_SET,
             HOSTAPD_AP_CTRL_CMD_SET_HIDDEN, hidden);
    callback = hostapd_apgeneric_cb;

    req = hostapd_create_request (ifname, cmd,
                                  hostapd->threadpool,
                                  ap->sock,
                                  g_main_context_default(),
                                  cb, userdata);
    return_val_if_fail (req, -APE_INTERNALFAIL);
    ret = hostapd_request (req->pool, req, callback);
    if (APE_SUCCESS == ret)
        ap->sock->acquired = 1;

    return ret;
}

int
accesspoint_set_accnettype (const char *ifname,
                        const unsigned int accnettype,
                        apresultcb cb,
                        void *userdata)
{
    accesspoint *ap = NULL;
    hostapdreq *req;
    int ret;
    char cmd [HOSTAPD_REQUEST_BUFFER];
    GSourceFunc callback;

    PERFORM_AP_SANITY_CHECK (ifname, hostapd_started, hostapd, &ap);
    return_val_if_fail (!HOSTAPD_ALREADY_ACQUIRED (ap->sock), -APE_BUSY);

    memset (cmd, 0, sizeof (cmd));
    sprintf (cmd, "%s %s %d", HOSTAPD_AP_CTRL_CMD_SET,
            HOSTAPD_AP_CTRL_CMD_SET_ACCNETTYPE, accnettype);
    callback = hostapd_apgeneric_cb;

    req = hostapd_create_request (ifname, cmd,
                                  hostapd->threadpool,
                                  ap->sock,
                                  g_main_context_default(),
                                  cb, userdata);
    return_val_if_fail (req, -APE_INTERNALFAIL);
    ret = hostapd_request (req->pool, req, callback);
    if (APE_SUCCESS == ret)
        ap->sock->acquired = 1;

    return ret;
}

int
accesspoint_set_vendor_ie (const char *ifname,
                           const char *vendor_element,
                           const unsigned long length,
                           gboolean isAssocResp,
                           apresultcb cb,
                           void *userdata)
{
    gboolean newstring = FALSE;
    accesspoint *ap = NULL;
    hostapdreq *req;
    int ret;
    char cmd [HOSTAPD_REQUEST_BUFFER],
            *hexstring;
    GSourceFunc callback;

    PERFORM_AP_SANITY_CHECK (ifname, hostapd_started, hostapd, &ap);
    return_val_if_fail (!HOSTAPD_ALREADY_ACQUIRED (ap->sock), -APE_BUSY);

    if (length) {
        newstring = TRUE;
        hexstring = byte_to_hexstring (vendor_element, length);
        if (!hexstring)
            return -APE_INVAL;
    } else
        hexstring = "";

    memset (cmd, 0, sizeof (cmd));
    if(!isAssocResp){
        sprintf (cmd, "%s %s %s", HOSTAPD_AP_CTRL_CMD_SET,
                HOSTAPD_AP_CTRL_CMD_SET_VENDOR_IE, hexstring);
    }
    else{
        sprintf (cmd, "%s %s %s", HOSTAPD_AP_CTRL_CMD_SET,
                HOSTAPD_AP_CTRL_CMD_SET_ASSOCRESP_IE, hexstring);
    }
    callback = hostapd_setreq_cb;

    if (newstring)
        g_free (hexstring);

    req = hostapd_create_request (ifname, cmd,
                                  hostapd->threadpool,
                                  ap->sock,
                                  g_main_context_default(),
                                  cb, userdata);
    return_val_if_fail (req, -APE_INTERNALFAIL);
    ret = hostapd_request (req->pool, req, callback);
    if (APE_SUCCESS == ret)
        ap->sock->acquired = 1;

    return ret;
}

static gboolean
hostapd_setssid_utf8_cb (gpointer data)
{
    int ret = 0;
    hostapdreq *req, *channel;
    char cmd [HOSTAPD_REQUEST_BUFFER];

    req = data;
    return_val_if_fail (req, G_SOURCE_REMOVE);

    INFO ("Interface: %s Request: %s result: %s thread: %ld",
          req->ifname, req->reqcmd, req->reply, syscall (SYS_gettid));

    if (req->result < 0) {
        ERROR ("Command failed for the interface %s to hostapd error: %d",
               req->ifname, req->result);
        ret = req->result;
    } else {
        ret = hostapd_reply_to_result (req->reply);
        if (ret == APE_SUCCESS) {

            memset (cmd, 0, sizeof (cmd));
            sprintf (cmd, "%s %s %s", HOSTAPD_AP_CTRL_CMD_SET,
            		HOSTAPD_AP_CTRL_CMD_SET_PASSPHRASE, req->ssid->wpa_passphrase);

            channel = hostapd_create_request (req->ifname, cmd,
                                              req->pool,
                                              req->sk,
                                              req->context,
                                              req->data->cb,
                                              req->data->userdata);
            if (!channel) {
                ERROR ("Failed to change the passphrase after ssid change: %s", req->ifname);
                ret = -APE_INTERNALFAIL;
            } else {
                ret = hostapd_request (channel->pool, channel, hostapd_apgeneric_cb);
                if (ret == APE_SUCCESS)
                    return G_SOURCE_REMOVE;
            }
        }
    }

    req->sk->acquired = 0;
    if (req->data->cb)
        req->data->cb (ret, req->ifname, req->data->userdata);
    return G_SOURCE_REMOVE;
}

static gboolean
hostapd_setssid_cb (gpointer data)
{
    int ret = 0;
    hostapdreq *req, *channel;
    char cmd [HOSTAPD_REQUEST_BUFFER];

    req = data;
    return_val_if_fail (req, G_SOURCE_REMOVE);

    INFO ("Interface: %s Request: %s result: %s thread: %ld",
          req->ifname, req->reqcmd, req->reply, syscall (SYS_gettid));

    if (req->result < 0) {
        ERROR ("Command failed for the interface %s to hostapd error: %d",
               req->ifname, req->result);
        ret = req->result;
    } else {
        ret = hostapd_reply_to_result (req->reply);
        if (ret == APE_SUCCESS) {
            memset (cmd, 0, sizeof (cmd));
            sprintf (cmd, "%s %s %d", HOSTAPD_AP_CTRL_CMD_SET,
                     HOSTAPD_AP_CTRL_CMD_SET_UTF8SSID, 1);

            channel = hostapd_create_request (req->ifname, cmd,
                                              req->pool,
                                              req->sk,
                                              req->context,
                                              req->data->cb,
                                              req->data->userdata);
            if (!channel) {
                ERROR ("Failed to set the utf8_ssid data: %s", req->ifname);
                ret = -APE_INTERNALFAIL;
            } else {
            	if (req->ssid) {
            		channel->ssid = g_try_malloc0 (sizeof (*channel->ssid));
            		if (channel->ssid)
            			channel->ssid->wpa_passphrase = g_strdup (req->ssid->wpa_passphrase);
            	}
                ret = hostapd_request (channel->pool, channel,
                					   channel->ssid ? hostapd_setssid_utf8_cb :
                							   hostapd_apgeneric_cb);
                if (ret == APE_SUCCESS)
                    return G_SOURCE_REMOVE;
            }
        }
    }

    req->sk->acquired = 0;
    if (req->data->cb)
        req->data->cb (ret, req->ifname, req->data->userdata);
    return G_SOURCE_REMOVE;
}

int
accesspoint_set_ssid (const char *ifname,
                      const char *ssid,
                      size_t length,
					  const char *passphrase,
                      apresultcb cb,
                      void *userdata)
{
    accesspoint *ap = NULL;
    int ret;
    size_t plength;
    hostapdreq *req;
    char cmd [HOSTAPD_REQUEST_BUFFER],
            *hexstring;
    ssiddata *asyncdata = NULL;

    if (!ssid || length > SSID_MAX_LEN)
        return -APE_INVAL;

    if (passphrase) {
    	plength = strlen (passphrase);
    	if (plength < PASSPHRASE_MIN_LENGTH ||
    			plength > PASSPHRASE_MAX_LENGTH)
    		return -APE_INVAL;
    }

    PERFORM_AP_SANITY_CHECK (ifname, hostapd_started, hostapd, &ap);
    return_val_if_fail (!HOSTAPD_ALREADY_ACQUIRED (ap->sock), -APE_BUSY);

    memset (cmd, 0, sizeof (cmd));
    hexstring = byte_to_hexstring (ssid, length);
    if (!hexstring)
        return -APE_INVAL;

    sprintf (cmd, "%s %s %s", HOSTAPD_AP_CTRL_CMD_SET,
             HOSTAPD_AP_CTRL_CMD_SET_SSID2, hexstring);
    g_free (hexstring);

    if (passphrase) {
    	asyncdata = g_try_malloc0 (sizeof (*asyncdata));
    	return_val_if_fail (asyncdata, -APE_INTERNALFAIL);
    	asyncdata->wpa_passphrase = g_strdup (passphrase);
    }

    req = hostapd_create_request (ifname, cmd,
                                  hostapd->threadpool,
                                  ap->sock,
                                  g_main_context_default(),
                                  cb, userdata);

    if (!req) {
    	if (passphrase) {
    		g_free (asyncdata->wpa_passphrase);
    		g_free (asyncdata);
    	}
    	return -APE_INTERNALFAIL;
    }

    if (passphrase)
    	req->ssid = asyncdata;

    ret = hostapd_request (req->pool, req, hostapd_setssid_cb);
    if (APE_SUCCESS == ret)
        ap->sock->acquired = 1;

    return ret;
}

int
accesspoint_set_max_num_sta (const char *ifname,
                             unsigned int max_num_sta,
                             apresultcb cb,
                             void *userdata)
{
    accesspoint *ap = NULL;
    int ret;
    hostapdreq *req;
    char cmd [HOSTAPD_REQUEST_BUFFER];

    if (max_num_sta > MAX_STATIONS_ALLOWED)
        return -APE_INVAL;

    PERFORM_AP_SANITY_CHECK (ifname, hostapd_started, hostapd, &ap);
    return_val_if_fail (!HOSTAPD_ALREADY_ACQUIRED (ap->sock), -APE_BUSY);

    memset (cmd, 0, sizeof (cmd));
    sprintf (cmd, "%s %s %u", HOSTAPD_AP_CTRL_CMD_SET,
             HOSTAPD_AP_CTRL_CMD_MAX_NUM_STA, max_num_sta);

    req = hostapd_create_request (ifname, cmd,
                                  hostapd->threadpool,
                                  ap->sock,
                                  g_main_context_default(),
                                  cb, userdata);
    return_val_if_fail (req, -APE_INTERNALFAIL);
    ret = hostapd_request (req->pool, req, hostapd_setreq_cb);
    if (APE_SUCCESS == ret)
        ap->sock->acquired = 1;

    return ret;
}

static gboolean
accesspoint_regdom_cb (gpointer data)
{
    regdom_request *req = data;

    return_val_if_fail (req, G_SOURCE_REMOVE);

    if (req->sk)
        req->sk->acquired = 0;

    if (req->data && req->data->cb)
        req->data->cb (req->result, req->iface, req->data->userdata);

    regulatory_request_cleanup (req);

    if (hostapd)
        hostapd->regdom_requests = g_slist_remove (hostapd->regdom_requests, req);

    return G_SOURCE_REMOVE;
}

static int
accesspoint_check_duplicates (const char *phyname,
                              GSList *list)
{
    GSList *temp;
    regdom_request *req;

    for (temp = list; temp; temp = temp->next) {
        req = temp->data;
        if (req && !g_strcmp0 (req->phyname, phyname))
            return 0;
    }

    return -ENOENT;
}

int
accesspoint_set_country_code (const char *ifname,
                              const char *alpha2,
                              apresultcb cb,
                              void *userdata)
{
    const char *phyname = NULL;
    accesspoint *ap = NULL;
    regdom_request *req = NULL;
    int ret;

    return_val_if_fail (ifname && alpha2, -APE_INVAL);
    return_val_if_fail (hostapd_started, -APE_SYSNOTREADY);
    return_val_if_fail (hostapd, -APE_NOLINK);

    phyname = genl_get_phyname (ifname);
    return_val_if_fail (phyname, -APE_NOTSUP);

    /* Every interface of that particular device gets affected,
     * therefore */
    ret = accesspoint_check_duplicates (phyname, hostapd->regdom_requests);
    if (0 == ret)
        return -APE_BUSY;

    /* We block the hostapd socket if there is an entry
     * to make sure that we donot do any AP operations
     * with when a regulatory change is in progress. Else
     * we simply go ahead and change the regdom. */
    ap = g_hash_table_lookup (hostapd->accesspoints, ifname);
    if (ap && ap->sock->acquired)
        return -APE_BUSY;

    req = g_try_malloc0 (sizeof (*req));
    return_val_if_fail (req, -APE_INTERNALFAIL);

    req->data = g_try_malloc0 (sizeof (*req->data));
    if (!req->data) {
        g_free (req);
        return -APE_INTERNALFAIL;
    }

    req->iface = g_strdup (ifname);
    req->phyname = g_strdup (phyname);
    req->data->cb = cb;
    req->data->userdata = userdata;

    if (ap) {
        req->sk = ap->sock;
        req->sk->acquired = 1;
    }

    ret = genl_set_country_code (ifname, alpha2);
    if (0 == ret) {
        req->result = ret;
        /* why 5 seconds? enough I think. Also hostapd uses it.
         * If fails,something terribly wrong with the system and
         * dont expect proper wifi operations */
        req->regdom_source_event = g_timeout_add_seconds
                (5, accesspoint_regdom_cb, req);
        hostapd->regdom_requests = g_slist_append (hostapd->regdom_requests, req);
    } else {
        if (ap)
            req->sk->acquired = 0;
        regulatory_request_cleanup (req);
    }

    return ret;
}

int
accesspoint_get_supported_channels (const char *ifname,
                                    unsigned int **channels,
                                    unsigned int *size)
{
    int result;

    return_val_if_fail (ifname && channels, -APE_INVAL);
    return_val_if_fail (!*channels, -APE_ALREADY);

    result = genl_get_supported_channels (ifname, channels, size);
    if (result < 0)
        result = -APE_FAILURE;

    return result;
}

int
accesspoint_get_supported_frequencies (const char *ifname,
                                       unsigned int **frequencies,
                                       unsigned int *size)
{
    int result;

    return_val_if_fail (ifname && frequencies, -APE_INVAL);
    return_val_if_fail (!*frequencies, -APE_ALREADY);

    result = genl_get_supported_frequencies (ifname, frequencies, size);
    if (result < 0)
        result = -APE_FAILURE;

    return result;
}

int
accesspoint_get_supported_hwmodes (const char *ifname,
                                   unsigned int *modes)
{
    int result;

    return_val_if_fail (ifname && modes, -APE_INVAL);

    result = genl_get_supported_hwmodes (ifname, modes);
    if (result < 0)
        result = -APE_FAILURE;

    return result;
}

int
accesspoint_get_supported_rates (const char *ifname,
                                 unsigned int **rates,
                                 unsigned int *len)
{
    int result;

    return_val_if_fail (ifname && rates, -APE_INVAL);
    return_val_if_fail (!*rates, -APE_ALREADY);

    result = genl_get_supported_rates (ifname, rates, len);
    if (result < 0)
        result = -APE_FAILURE;

    return result;
}

int
accesspoint_get_rates_from_mode (const char *ifname,
                                 unsigned int mde,
                                 unsigned int **rates,
                                 unsigned int *len)
{
    int result;

    return_val_if_fail (ifname && rates, -APE_INVAL);
    return_val_if_fail (!*rates, -APE_ALREADY);

    result = genl_get_rates_from_mode (ifname, ((unsigned int)1 << mde),
                                       rates, len);
    if (result < 0)
        result = -APE_FAILURE;

    return result;
}

char*
accesspoint_get_country_code (const char *ifname)
{
    return_val_if_fail (ifname, NULL);
    return genl_get_country_code (ifname);
}

const char*
accesspoint_strerror (int error)
{
    error = abs (error);

    if (error >= NUM_APE_MAX)
        error = APE_FAILURE;

    return aperrmsg [error];
}

int
accesspoint_init (const accesspointcallbacks *cbacks)
{
    int ret;

    DEBUG ("");

    return_val_if_fail (cbacks, -APE_INVAL);

    ret = genl_regnotifier_register (&regulatoryops);
    if (ret < 0) {
        ERROR ("Failed to register to the genl client for update on the "
               "regulatory information: %s/%d", strerror (-ret), -ret);
        return ret;
    }

    ret = genl_scannotifier_register (&scanops);
    if (ret < 0) {
        genl_regnotifier_unregister (&regulatoryops);
        return ret;
    }

    ret = hostapd_global_init();
    if (ret < 0) {
        genl_regnotifier_unregister (&regulatoryops);
        genl_scannotifier_unregister (&scanops);
        return ret;
    }

    callbacks = cbacks;
    return ret;
}

int
accesspoint_deinit (void)
{
    int ret;

    DEBUG ("");

    callbacks = NULL;

    ret = genl_regnotifier_unregister (&regulatoryops);
    if (ret < 0)
        ERROR ("Failed to unregister to the genl client regulatory "
               "callbacks: %s/%d", strerror (-ret), -ret);

    ret = genl_scannotifier_unregister (&scanops);
    if (ret < 0)
        ERROR ("Failed to unregister from scan state changes: %s%d",
               strerror (-ret), -ret);

    ret = hostapd_global_deinit();
    if (ret < 0)
        ERROR ("Failed to deinitialise the global hostapd: %s/%d",
               strerror (-ret), -ret);

    return ret;
}

/** @} */
