/**
 * @file genl.h
 * @author RBEI/ECO3-Usman Sheik
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#ifndef GENL_H
#define GENL_H

/**
 * struct regulatory_ops - regulatory callbacks
 *
 * This struct shall be registered/subscribed by the client
 * to get the updates on regulatory.
 *
 * @clientname - name of the client
 *
 * @ifname - name of the interface. If the client is controlling
 * a single interface, the client could only register for the updates
 * on that particular interface
 *
 * @reg_changed - change in the regulatory
 *  @ifname: name of the interface
 *  @frequencies: changed frequency list
 *  @channels: changed channel list
 *  @len: length of the list
 */
typedef
struct __genl_notifier {
    const char *clientname;
    const char *ifname;
    void (*reg_changed) (const char *ifname,
                         const unsigned int *frequencies,
                         const unsigned int *channels,
                         const unsigned int len);
} regulatory_ops;

/**
 * enum dfsevent - DFS events
 */
typedef
enum __nl80211_scan_state {
    /**
     * SCAN has been triggered and currently in
     * progress
     */
    NL80211_SCAN_STATE_SCAN_STARTED,

    /**
     * SCAN has been triggered but aborted before
     * its completion
     */
    NL80211_SCAN_STATE_SCAN_ABORTED,

    /**
     * Requested scan has been completed successfully
     */
    NL80211_SCAN_STATE_SCAN_COMPLETED,

    /**
     * SCHEDULED SCAN has been started
     */
    NL80211_SCAN_STATE_SCHED_SCAN_STARTED,

    /**
     * SCHEDULED SCAN has been stopped
     */
    NL80211_SCAN_STATE_SCHED_SCAN_STOPPED,

    /**
     * SCHEDULED SCAN has been completed
     */
    NL80211_SCAN_STATE_SCHED_SCAN_COMPLETED,
    NL80211_NUM_SCAN_STATES
} nl80211_scan_state;

/**
 * struct scanstate_ops - scan state callbacks
 *
 * This struct shall be registered/subscribed by the client
 * to get the updates on scan states of a device.
 *
 * @clientname - name of the client
 *
 * @ifname - name of the interface. If the client is controlling
 * a single interface, the client could only register for the updates
 * on that particular interface
 *
 * @nl80211_scan_state - change in the scan state of a particular wiphy
 *  @wiphy: name of the device/wiphy
 *  @ifname: name of the network interface where the scan is
 *           requested
 *  @state: changed scan state
 */
typedef
struct __scan_state_notifier {
    const char *clientname;
    const char *ifname;
    void (*scan_state_changed) (const char *wiphy,
                                const char *ifname,
                                nl80211_scan_state state);
} scanstate_ops;

/**
 * genl_init - Initialize the genl client internals i.e., open a netlink
 * socket and connect to netlink protocol for monitoring regulatory
 * updates..
 * Returns: 0 on success and negative error code on failure
 */
int genl_init (void);

/**
 * genl_deinit - Deinitialize the genl client i.e., close the netlink
 * socket and disconnect from netlink protocol
 * Returns: 0 on success and negative error code on failure
 */
int genl_deinit (void);

/**
 * genl_regnotifier_register - Register a client for regulatory updates
 * @ops: regulatory_ops callbacks
 * Returns: 0 on success and negative error code on failure
 */
int genl_regnotifier_register (regulatory_ops *ops);

/**
 * genl_regnotifier_unregister - Unregister a client from regulatory updates
 * @ops: regulatory_ops callbacks
 * Returns: 0 on success and negative error code on failure
 */
int genl_regnotifier_unregister (regulatory_ops *ops);

/**
 * genl_scannotifier_register - Register a client for scan reated updates
 * @ops: scanstate_ops callbacks
 * Returns: 0 on success and negative error code on failure
 */
int genl_scannotifier_register (scanstate_ops *ops);

/**
 * genl_regnotifier_unregister - Unregister a client from scan related updates
 * @ops: scanstate_ops callbacks
 * Returns: 0 on success and negative error code on failure
 */
int genl_scannotifier_unregister (scanstate_ops *ops);

/**
 * genl_interface_supported - Check if the network interface is wireless i.e.,
 * whether it is exposed by any of the wireless devices attached to the system
 * @ifname: name of the network interface
 * Returns: 0 on success and negative error code on failure
 */
int genl_interface_supported (const char *ifname);

/**
 * genl_get_phyname - Get the name of the wireless device which has exposed this
 * wireless interface
 * @ifname: name of the network interface
 * Returns: device name on success and %NULL on failure
 */
const char* genl_get_phyname (const char *ifname);

/**
 * genl_get_supported_channels - Get the list of supported frequencies
 * by the access point
 * @ifname: network interface name
 * @channels: List of channels. It is the responsibility of the user to free the
 * channels
 * @size: size of the channel list
 * Returns: 0 on success and negative error code on failure
 */
int genl_get_supported_channels (const char *ifname,
                                 unsigned int **channels,
                                 unsigned int *size);

/**
 * genl_get_supported_frequencies - Get the list of supported frequencies
 * by the access point
 * @ifname: network interface name
 * @frequencies: List of frequencies. It is the responsibility of the user to free the
 * frequencies
 * @size: size of the frequency list
 * Returns: 0 on success and negative error code on failure
 */
int genl_get_supported_frequencies (const char *ifname,
                                    unsigned int **frequencies,
                                    unsigned int *size);

/**
 * genl_get_supported_hwmodes - Get the IEEE 802.11 modes supported by the
 * interface
 * @ifname: network interface name
 * @modes: IEEE 802.11 modes i.e., a bitwise-or'd IEEE80211 MODES
 * Returns: 0 on success and negative error on failure
 */
int genl_get_supported_hwmodes (const char *ifname,
                                 unsigned int *modes);

/**
 * genl_get_supported_rates - Get the list of supported rates
 * by the interface
 * @ifname: network interface name
 * @rates: List of rates. It is the responsibility of the user to free the
 * rates
 * @len: length of the rates list
 * Returns: 0 on success and negative error code on failure
 */
int genl_get_supported_rates (const char *ifname,
                              unsigned int **rates,
                              unsigned int *len);

/**
 * genl_get_rates_from_mode - Get the list of supported rates by this mode
 * @ifname: network interface name
 * @mode: HW mode
 * @rates: List of rates. It is the responsibility of the user to free the
 * rates
 * @len: length of the rates list
 * Returns: 0 on success and negative error code on failure
 */
int genl_get_rates_from_mode (const char *ifname,
                              unsigned int mode,
                              unsigned int **rates,
                              unsigned int *len);

/**
 * genl_get_interface_type - Get the WLAN mode of this interface
 * @ifname: network interface name
 * Returns: 0 on success and negative error code on failure
 */
int genl_get_interface_type (const char *ifname);

/**
 * genl_get_country_code - Get the reg domain alpha2
 * @ifname: network interface name
 * Returns: NULL on failure or ISO/IEC 3166-1 alpha2 country code
 */
char *genl_get_country_code (const char *ifname);

/**
 * genl_set_country_code - Set the reg domain alpha2
 * @ifname: network interface name
 * @alpha2: specify ISO/IEC 3166-1 alpha2 country code, should be null terminated
 * Returns: 0 on success and negative error code on failure
 */
int genl_set_country_code (const char *ifname, const char *alpha2);

/**
 * genl_scan_state_to_string - scan state to string
 * @state: nl80211 scan state
 * Returns: a string corresponding to the nl80211 scan state
 */
const char* genl_scan_state_to_string (nl80211_scan_state state);

#endif //GENL_H

/** @} */
