
#include "iap_bt_plugin_common.h"
#include <adit_typedef.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/select.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <unistd.h>
#include <poll.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <termios.h>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_IAP_PLUGIN_COMMON
#include "trcGenProj/Header/iap_bt_plugin_common.c.trc.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_MSG_PREFIX_IAP1
static const U8 IAP_VERSION = 1;
#elif VARIANT_S_FTR_ENABLE_TRC_MSG_PREFIX_IAP2
static const U8 IAP_VERSION = 2;
#else
static const U8 IAP_VERSION = 0;
#endif

S32 iAPBTOpenPlugin (const U8* device_name)
{
   /* Calling function is responsible for parameter check ! */
   ETG_TRACE_USR3(("IAP%u: iAPBTOpenPlugin::entered", IAP_VERSION));
   S32 device_fd = -1;
   S32 rc = IPOD_DATACOM_SUCCESS;
   struct termios termios_p;

   device_fd = open ((char *)device_name, O_RDWR | O_NOCTTY | O_SYNC | O_NONBLOCK);
   if (device_fd < 0)
   {
      ETG_TRACE_ERR(("IAP%u: iAPBTOpenPlugin::open() device failed: %d", IAP_VERSION, errno));
      rc = IPOD_DATACOM_ERROR;
   }
   else
   {
      ETG_TRACE_USR3(("IAP%u: iAPBTOpenPlugin::Device %s opened", IAP_VERSION, device_name));
      ETG_TRACE_USR3(("IAP%u: iAPBTOpenPlugin::Filedesciptor is %d", IAP_VERSION, device_fd));

      if (tcgetattr (device_fd, &termios_p) < 0)
      {
         ETG_TRACE_ERR(("IAP%u: iAPBTOpenPlugin::tcgetattr failed: %d", IAP_VERSION, errno));
         rc = IPOD_DATACOM_ERROR;
      }
   }

   if(rc == IPOD_DATACOM_SUCCESS)  // Now we have opened the SPP device successfully and flush it
   {
      termios_p.c_cc[VMIN]  =  1;
      termios_p.c_cc[VTIME] =  0;
      termios_p.c_lflag &= (tcflag_t)(~(ECHO|ICANON|ISIG|ECHOE|ECHOK|ECHONL));
      termios_p.c_oflag &= (tcflag_t)(~(OPOST));

      tcflush (device_fd, TCIOFLUSH);
      cfmakeraw (&termios_p);

      if (tcsetattr (device_fd, TCSANOW, &termios_p) < 0)
      {
         ETG_TRACE_ERR(("IAP%u: iAPBTOpenPlugin::tcsetattr failed: %d - ignored", IAP_VERSION, errno));
         /* tcsetattr errors are printed out but ignored */
      }
      rc = device_fd;
   }

   ETG_TRACE_USR3(("IAP%u: iAPBTOpenPlugin::leaving with 'device_fd' (>0) or 'error code' (<0) as return value= %d", IAP_VERSION, rc));
   return rc;
}

S32 iAPBTClosePlugin (S32 device_fd)
{
   ETG_TRACE_USR3(("IAP%u: iAPBTClosePlugin::entered", IAP_VERSION));

   /* close /dev/rfcommX */
   (void)close (device_fd);

   ETG_TRACE_USR3(("IAP%u: iAPBTClosePlugin::leaving", IAP_VERSION));
   return IPOD_DATACOM_SUCCESS;
}

S32 iAPBTDeviceIoCtl (S32 request, void* argp)
{
   /* Calling function is responsible for parameter check ! */
   ETG_TRACE_USR3(("IAP%u: iAPBTDeviceIoCtl::entered", IAP_VERSION));
   S32 rc = IPOD_DATACOM_SUCCESS;

   /* Ioctl is currently not required and therefore not yet implemented */

   /* avoid lint warnings */
   request = request;
   argp = argp;

   rc = IPOD_DATACOM_SUCCESS;

   ETG_TRACE_USR3 (("IAP%u: iAPBTDeviceIoCtl::leaving", IAP_VERSION));
   return rc;
}

S32 iAPBTSendMessage (S32 device_fd, U32 msgLenTotal, const U8 *iPod_msg)
{
   /* Calling function is responsible for parameter check ! */
   ETG_TRACE_USR3(("IAP%u: iAPBTSendMessage::entered", IAP_VERSION));
   ETG_TRACE_USR3(("IAP%u: iAPBTSendMessage::Write request received with %d", IAP_VERSION, msgLenTotal));
   S32 rc = IPOD_DATACOM_SUCCESS;
   S32 bytes_sent = -1;
   S32 i = 0;
   U32 retryCount = 0;
   U32 overallRetryCount = 0;

   // calculate maximum overall retry counter:
   // - transfer rate = 20 bytes / ms (assumption for lowest value)
   // - delay between retries = 10 ms
   // - required time to transfer whole message = msgLenTotal / 20 bytes / ms = (msgLenTotal / 20) ms
   // - maximum overall retry counter = required time to transfer whole message / delay between retries = (msgLenTotal / 20) ms / 10 ms = msgLenTotal / 200
   U32 maxOverallRetryCount = (U32)(msgLenTotal / 200);

   // minimum value for maximum overall retry counter is 5
   if(5 > maxOverallRetryCount)
   {
      maxOverallRetryCount = 5;
   }

   // maximum value for maximum overall retry counter is 1000 that equals 10s
   if(1000 < maxOverallRetryCount)
   {
      maxOverallRetryCount = 1000;
   }

   while((rc < (S32)msgLenTotal) && (overallRetryCount < maxOverallRetryCount))
   {
      bytes_sent = (S32)write(device_fd, (iPod_msg + rc), (msgLenTotal - (U32)rc));

      if (bytes_sent >= 0)
      {
         ETG_TRACE_USR3(("IAP%u: iAPBTSendMessage::Sent data %d of %d: ", IAP_VERSION, bytes_sent, msgLenTotal));
         for(i=0; i<bytes_sent; i++)
         {
            ETG_TRACE_USR4(("IAP%u: Data: %x ", IAP_VERSION, iPod_msg[i]));
         }
         ETG_TRACE_USR4(("---"));
         rc += bytes_sent;
         retryCount = 0; // reset retry counter after successful write
      }
      else
      {
         ETG_TRACE_USR3(("IAP%u: iAPBTSendMessage::errno: %d %s", IAP_VERSION, errno, strerror(errno)));

         if(errno == EAGAIN)
         {
            //if write returns EAGAIN, resource temporarily unavailable try again
            ETG_TRACE_USR3(("IAP%u: iAPBTSendMessage::retryCount %d overallRetryCount %d", IAP_VERSION, retryCount, overallRetryCount));
            retryCount++;
            overallRetryCount++;

            // check overall retry counter
            if(overallRetryCount < maxOverallRetryCount)
            {
               usleep(10000); //retry after 10 msec
            }
            else
            {
               // no retries left, end the loop
               ETG_TRACE_ERR(("IAP%u: iAPBTSendMessage::Error: No retries left", IAP_VERSION));
               rc = IPOD_DATACOM_NOT_CONNECTED;
               break;
            }
         }
         else
         {
            ETG_TRACE_ERR(("IAP%u: iAPBTSendMessage::Error: Write request but device is not longer connected", IAP_VERSION));
            rc = IPOD_DATACOM_NOT_CONNECTED;
            break;
         }
      }
   }

   ETG_TRACE_USR3(("IAP%u: iAPBTSendMessage::Write request returned with %d", IAP_VERSION, rc));
   ETG_TRACE_USR3(("IAP%u: iAPBTSendMessage::leaving", IAP_VERSION));

   return rc;
}

S32 iAPBTReceiveMessage (S32 device_fd, U32 buffer_size, U8 *msgBuffer)
{
   /* Calling function is responsible for parameter check ! */
   ETG_TRACE_USR3(("IAP%u: iAPBTReceiveMessage::entered", IAP_VERSION));
   S32 rc = IPOD_DATACOM_SUCCESS;
   S32 bytes_read = -1;
   S32 i = 0;

   bytes_read = (S32)read(device_fd, msgBuffer, buffer_size);

   ETG_TRACE_USR3(("IAP%u: iAPBTReceiveMessage::bytes_read: %d", IAP_VERSION, bytes_read));

   if (bytes_read > 0)
   {
      ETG_TRACE_USR4(("IAP%u: iAPBTReceiveMessage::Received data: ", IAP_VERSION));
      for(i=0; i<bytes_read; i++)
      {
         ETG_TRACE_USR4(("IAP%u: Data %x", IAP_VERSION, msgBuffer[i]));
      }
      ETG_TRACE_USR4(("---"));
      rc = bytes_read;
   }
   else
   {
      ETG_TRACE_USR3(("IAP%u: iAPBTReceiveMessage::errno: %d %s", IAP_VERSION, errno, strerror(errno)));

      if(errno == EAGAIN)
      {
         //if read returns EAGAIN, resource temporarily unavailable try again
         rc = 0;
      }
      else
      {
         ETG_TRACE_ERR(("IAP%u: iAPBTReceiveMessage::Error: Receive_message request but device is not longer connected", IAP_VERSION));
         rc = IPOD_DATACOM_NOT_CONNECTED;
      }
   }

   ETG_TRACE_USR3(("IAP%u: iAPBTReceiveMessage::Receive_message request returned with %d", IAP_VERSION, rc));
   ETG_TRACE_USR3(("IAP%u: iAPBTReceiveMessage::leaving", IAP_VERSION));
   return rc;
}
