#ifndef _BT_IAP1_PLUGIN_H
#define _BT_IAP1_PLUGIN_H

#ifdef __cplusplus
extern "C" {
#endif

typedef struct
{
    /*! Maxmum size that device is possible to read and write the data. */
    U32 maxSize;
} IPOD_DATACOM_PROPERTY;


/*!
 * \fn typedef S32 (*iPodDataComOpen)(const U8 *deviceName, S32 flags, S32 mode)
 * \param [in] *deviceName - name of opened driver. It must be NULL terminated strings.<br>
 * \param [in] flags - It must be set #IPOD_DATACOM_FLAGS_RDONLY, #IPOD_DATACOM_FLAGS_WRONLY or #IPOD_DATACOM_FLAGS_RDWR<br>
 * \param [in] mode - It does not have affect currently. If neccessary, pugin can add<br>
 * \retval fd : file descriptor. This value equal to or greater than 0.<br>
 * \retval error : Value is negative. This function is failed.<br>
 * \par DESCRIPTION
 * This function is used for open the data communication driver by name.
 * This function returns a value equal to or greater than 0 if this function succeed.
 * Otherwise this function will returns an negative value
 * \attention
 * <b> This function must guarantee that it is thread safe.</b>
 */
typedef S32 (*iPodDataComOpen)(const U8 *deviceName, S32 flags, S32 mode);



/*!
 * \fn typedef S32 (*iPodDataComClose)(S32 fd)
 * \param [in] fd - This is file descriptor that is gotten by #iPodDataComOpen<br>
 * \retval #IPOD_DATACOM_SUCCESS function success.<br>
 * \retval error : Value is negative. This function is failed.<br>
 * \par DESCRIPTION
 * This function is used for close the opened device that is indicated by fd.
 * \attention
 * <b> This function must guarantee that it is thread safe.</b>
 */
typedef S32 (*iPodDataComClose)(S32 fd);

/*!
 * \fn typedef S32 (*iPodDataComWrite)(S32 fd, U32 size, const U8 *buf, S32 flags)
 * Data is written until size byte.<br>
 * \param [in] fd - This is file descreptor that is opened by #iPodDataComOpen.<br>
 * \param [in] size - Maximum size which can write the data<br>
 * \param [out] *buf - data buffer <br>
 * \param [in] flags - If plugin want to do special behavior, this flags may be used.<br>
 * \retval size : The size is that this function could write the data.<br>
 * \retval error : Value is negative. This function is failed.<br>
 * \par DESCRIPTION
 * This function is used for write the data.<br>
 * If this function succeed, this function returns the size which this function could write to device.<br>
 * Otherwise, this function will return negative value.<br>
 * Currently flags parameter does not affect to this function.
 * \attention
 * <b> This function must guarantee that it is thread safe.</b>
 */
typedef S32 (*iPodDataComWrite)(S32 fd, U32 size, const U8 *buf, S32 flags);


/*!
 * \fn typedef int (*iPodDataComRead)(S32 fd, U32 size, U8 *buf, S32 flags)
 * \param [in] fd - This is file descreptor that is opened by #iPodDataComOpen.<br>
 * \param [in] size - Maximum size which can read the data<br>
 * \param [out] *buf - data buffer <br>
 * \param [in] flags - If plugin want to do special behavior, this flags may be used.<br>
 * \retval size : The size is that this function could read the data.<br>
 * \retval error : Value is negative. This function is failed.<br>
 * \par DESCRIPTION
 * This function is used for read the data.<br>
 * Data can be read until size byte.<br>
 * If this function succeed, this function returns the size which this function could read from device.<br>
 * Otherwise, this function will return negative value.<br>
 * Also this function does not return until data is filled.<br>
 * Currently flags parameter does not affect to this function. 
 * \warning
 * If this function is called when device does not have data, this funcction is stopped until receve the data.<br>
 * \attention
 * <b> This function must guarantee that it is thread safe.</b>
 */
typedef S32 (*iPodDataComRead)(S32 fd, U32 size, U8 *buf, S32 flags);


/*!
 * \fn typedef S32 (*iPodDataComIoctl)(S32 fd, S32 request, U8 *argp)
 * \param [in] fd - This is file descreptor that is opened by #iPodDataComOpen.<br>
 * \param [in] request - It may be requested command, requested operation(read/write) or size of next parameter. <br>
 * \param [in] argp - Plugin specific parameter.<br>
 * \retval ret : Plugin can set freely.<br>
 * \par DESCRIPTION
 * This function is used for special behovior.<br>
 * Plugin can use freely for some purpose.<br>
 * \attention
 * <b> This function must guarantee that it is thread safe.</b>
 */
typedef S32 (*iPodDataComIoctl)(S32 fd, S32 request, U8 *argp);



/*!
 * \fn typedef S32 (*iPodDataComGetProperty)(S32 fd, IPOD_DATACOM_PROPERTY *property)
 * \param [in] fd - This is file descreptor that is opened by #iPodDataComOpen.<br>
 * \param [out] property - Structure of connected device's property. <br>
 * \retval #IPOD_DATACOM_SUCCESS : function success.<br>
 * \retval error : Value is negative.
 * \par DESCRIPTION
 * This function is used for get the property of opened device.<br>
 * If this function succeed, this function returns #IPOD_DATACOM_SUCCESS.<br>
 * Otherwise, this function will return with negative value.<br>
 * \attention
 * <b> This function must guarantee that it is thread safe.</b>
 */
typedef S32 (*iPodDataComGetProperty)(S32 fd, IPOD_DATACOM_PROPERTY *property);

/*!
 * \fn typedef S32 (*iPodDataComAbort)(S32 fd);
 * \param [in] fd - This is file descreptor that is opened by #iPodDataComOpen.<br>
 * \retval #IPOD_DATACOM_SUCCESS : function success.<br>
 * \retval error : Value is negative.
 * \par DESCRIPTION
 * This function is used for abort the current running request and 
 * #iPodDataComRead and #iPodDataComWrite cannot use after #iPodDataComAbort completed.<br>
 * If this function suceed, this function returns #IPOD_DATACOM_SUCCESS.<br>
 * Otherwise, this function will return with negative value.<br>
 * also, the request which is aborted by this function will finish with error.<br>
 * \attention
 * <b> This function must guarantee that it is thread safe.</b>
 */
typedef S32 (*iPodDataComAbort)(S32 fd);


typedef struct
{
    /*! See #iPodDataComOpen */
    iPodDataComOpen open;
    /*! See #iPodDataComClose */
    iPodDataComClose close;
    /*! See #iPodDataComAbort */
    iPodDataComAbort abort;
    /*! See #iPodDataComWrite */
    iPodDataComWrite write;
    /*! See #iPodDataComRead */
    iPodDataComRead read;
    /*! See #iPodDataComIoctl */
    iPodDataComIoctl ioctl;
    /*! See #iPodDataComGetProperty */
    iPodDataComGetProperty property;
} IPOD_DATACOM_FUNC_TABLE;


S32 iPodBTComInit(IPOD_DATACOM_FUNC_TABLE* data_com_function, U32 num_devices);
S32 iPodBTComDeinit(IPOD_DATACOM_FUNC_TABLE* data_com_function);

S32 iPodiAP1BTOpenPlugin(const U8* device_name, S32 flags, S32 mode);
S32 iPodiAP1BTClosePlugin(S32 fd);
S32 iPodiAP1BTCloseMsgHandling(S32 fd);
S32 iPodiAP1BTSendMessage(S32 fd, U32 msgLenTotal, const U8 *iPod_msg, S32 flags);
S32 iPodiAP1BTReceiveMessage(S32 fd, U32 buffer_size, U8 *msgBuffer, S32 flags);
S32 iPodiAP1BTGetProperty(S32 fd, IPOD_DATACOM_PROPERTY *property);
S32 iPodiAP1BTDeviceIoCtl(S32 fd, S32 request, U8* argp);

#ifdef __cplusplus
}
#endif

#endif /* _BT_IAP1_PLUGIN_H */

