/*******************************************************************************
* FILE:          FC_PhoneBook_VerticalKeyBoard.h
* PROJECT:       GM-NextGen
* SW-COMPONENT:  VerticalSearchKeyboard
*-------------------------------------------------------------------------------
*
* DESCRIPTION:   Main include file for clients who wants to use the 
*                Vertical Keyboard functionality
*
*-------------------------------------------------------------------------------
*
* AUTHOR:  CM-AI-Benthin  (ben2hi)
*
* COPYRIGHT:    (c) 2012 Robert Bosch GmbH, Hildesheim
*
* HISTORY:
* - 22.08.2012: Initial version 
* - 16.10.2012: Performance optimization by comparing characters based on 
*               "SortPrio" retrieved from SortLib, instead of a direct comparison
*               of characters with the SortLib.
*******************************************************************************/

#ifndef FC_PHONEBOOK_VERTICALKEYBOARD_H
#define FC_PHONEBOOK_VERTICALKEYBOARD_H


#include "../Common/FC_PhoneBook_Global.h" //c++11 compatibility

/*------------------------------------------------------------------------------
  Vertical Keyboard Description:
  
  Overview:
  Starting point is the presentation of an alphabetic sorted list in an HMI screen.
  Beside this list a Vertical Keyboard with related start letters might be 
  presented. 
  - Letters in the Vertical Keyboard shall be highlighted when List Items
    have a first character which corresponds to a Vertical Keyboard character
    (or its "compliant" region).
  - Tapping on a highlighted character in the Vertical Keyboard shall result in
    a direct scrolling/movement to List Items which start with this character.

  VerticalSearchKeyboard requirements/implementation documentation:  
  - https://hi-dms.de.bosch.com/docushare/dsweb/Get/Document-457482/bn_SCD_VerticalKeyboard.docx
  
  VerticalSearchKeyboard library:
  - The client will provide an alphabetically sorted list of e.g. contact names or
    media items.
    The sorted list has been sorted by using the 'SortLib'. This is a mandatory
    pre-condition, as the 'VerticalSearchKeyboard' implementation makes also use of
    the 'SortLib'.
  - The 'VerticalSearchKeyboard' will return a keyboard letter list with start-/end
    indices of items in the list, which correspond to a keyboard letter.
  
  How to use the 'VerticalSearchKeyboard' library (example code):
  
  - An alphabetically sorted list of type 'QStringList' shall be available/provided
    from the client side. Let's assume it is named: 'qsItemList'.
  - The client needs to include the VerticalSearchKeyboard main header.
  
  - Client Code Example: The client code should look as follows:
  
    #include "FC_PhoneBook_VerticalKeyBoard.h"  // Include the Vertical KeyBoard main header
    ..
    
    VerticalSearchKeyboard oVkbd;        // Create a VerticalSearchKeyboard object

    QList<VerticalSearchKeyboardListItem>* poVkbdList;  // Create a VerticalSearchKeyboard list pointer

    poVkbdList = oVkbd.showKeyboard(qsItemList, u32Language);  // Call .showKeyboard(..)
    // Note: quint32 'u32Language' typically is set by the client by using the type
    //       'most_fi_tcl_e8_GMLnGWLanguageSelection' e.g. it might use
    //       'most_fi_tcl_e8_GMLnGWLanguageSelection::FI_EN_E8LANGUAGE_SELECTION_NA_ENGLISH'
    //       as u32Language value. That is o.k. and intended even if this 'type' is not a 'quint32'.
    //       VerticalSearchKeyboard does not know MOST FI types, but will re-interprete
    //       the given value according to its internal knowledge (definition)
    //       of this MOST FI type language.

    
                                         // Process 'VerticalSearchKeyboard' result
    // Now the client shall extract the data from 'poVkbdList' and process it,
    // typically this might mean to hand over it to a MOST FI message object 
    // and to send this to HMI ..  This details are dependent on client's needs
    // and therefore will not presented here in more detail ..

    // As last step the client has to delete the data object 'poVkbdList' as 
    // otherwise the client would produce a memory leak, as VerticalSearchKeyboard has
    // added memory (by calling oVkbd.showKeyboard(..)) in the meantime to this pointer:
    delete poVkbdList;                   // Delete VerticalSearchKeyboard list pointer

  - Traces for VerticalSearchKeyboard:
    The VerticalKeyboard has no ETG trace class, nor it has the capability to 
    use ETG traces (the current build environment settings cause build problems
    when trying to include ETG trace capabilities).
    But we can do traces to 'stderr' (catched by TTFIS) via trace class
    'TR_SET_CONFIG TR_CLASS_STDOUT 3' for developer purposes, when enabling 
    the trace capability via the setting '#define DEBUG_FPRINTF_VERT_KEYBOARD'.
    Note: This define '#define DEBUG_FPRINTF_VERT_KEYBOARD' shall not be set
          in oficially released version of the component 'VerticalSearchKeyboard'.
    
------------------------------------------------------------------------------*/

// -----------------------------------------------------------------------------
// Includes:
// -----------------------------------------------------------------------------
#include "qstringlist.h"              // Needed for type 'QStringList'


// -----------------------------------------------------------------------------
// Type definitions:
// -----------------------------------------------------------------------------

// Note: The following enumeration for 'LanguageType' shall follow the defintion 
//       of 'languages'in FCat GIS-361 "IGMLANGateway.LanguageSelection".
//       Current definitions reflect GIS-361 v2.4.7 (23.08.2012 01:15)
enum LanguageType {
     NA_ENGLISH  = 0UL,
     GERMAN      = 1UL,
     ITALIAN     = 2UL,
     SWEDISH     = 3UL,
     FRENCH      = 4UL,
     SPANISH     = 5UL,
     DUTCH       = 6UL,
     PORTUGUESE  = 7UL,
     NORWEGIAN   = 8UL,
     FINNISH     = 9UL,
     DANISH      = 10UL,
     GREEK       = 11UL,
     JAPANESE    = 12UL,
     ARABIC      = 13UL,
     STANDARD_CHINESE     = 14UL,
     POLISH      = 15UL,
     TURKISH     = 16UL,
     KOREAN      = 17UL,
     TRADITIONAL_CHINESE  = 18UL,
     UK_ENGLISH  = 19UL,
     HUNGARIAN   = 20UL,
     CZECH       = 21UL,
     SLOVAK      = 22UL,
     RUSSIAN     = 23UL,
     BRAZILIAN_PORTUGUESE = 24UL,
     THAI        = 25UL,
     BULGARIAN   = 26UL,
     ROMANIAN    = 27UL,
     SLOVENIAN   = 28UL,
     CROATIAN    = 29UL,
     UKRAINIAN   = 30UL,
     NA_FRENCH   = 31UL,
     NA_SPANISH  = 32UL,
     CANTONESE   = 33UL,
     UZBEK   = 34UL,
     SERBIAN   = 35UL,
     LITHUANIAN   = 36UL,
     ESTONIAN   = 37UL,
     LATVIAN   = 38UL,
     ALBANESE   = 39UL,
     CHINESE_MANDARIN_SIMPLIFIED   = 40UL,
     ENGLISH_AUS   = 41UL,
     ENGLISH_JPN   = 42UL,
     ENGLISH_PRC   = 43UL,
     ENGLISH_CANADIAN   = 44UL,
     ENGLISH_INDIA   = 45UL,
     FILIPINO   = 46UL,
     FLEMISH   = 47UL,
     FRENCH_CANADIAN   = 48UL,
     HEBREW   = 49UL,
     HINDI   = 50UL,
     INDONESIAN   = 51UL,
     MALAY   = 52UL,
     PERSIAN   = 53UL,
     SPANISH_LATIN   = 54UL,
     VIETNAMESE   = 55UL
  };


// Struct'tsKeyboardChar' is used by FC_PhoneBook_VerticalKeyBoard.cpp to represent a character of the Vertical Keyboard.
// It contains the visible representation of the character in Vertical Keybard and the corresponding "search" 
// chararacter, used by the search algorithm.
struct tsKeyboardChar
{
   QString  qsKbdChar;          // Visibile representation of character in Vertical Keyboard
   QString  qsSearchChar;       // Search Character used in algorithmic search  
   quint32  u32SearchCharPrio;  // Sorting priority of Search Char
    
   bool     bCharIsAvailable;   // TRUE, if a list item corresponds to the area of this Vertical Keyboard Character
   quint32  u32StartIndex;      // First index with respect to ItemList, which is related to this Search Character
   quint32  u32EndIndex;        // Last index with respect to ItemList, which is related to this Search Character
                                // Note: List indices start with 0.

   tsKeyboardChar()
   {
      qsKbdChar         = QString();
      qsSearchChar      = QString();
      u32SearchCharPrio = 0;
                           
      bCharIsAvailable = false;    
      u32StartIndex    = 0;       
      u32EndIndex      = 0;         
   }
};


// Struct 'VerticalSearchKeyboardListItem' is used to hand over the Vertical Keyboard result to the client
struct VerticalSearchKeyboardListItem
{

    QString  m_sSearchKeyboardString;
    bool     m_bStringAvailable;
    quint32  m_u32StringStartIndex;
    quint32  m_u32StringEndIndex;


    VerticalSearchKeyboardListItem()
    {
        m_sSearchKeyboardString = QString();
        
        m_bStringAvailable = false;
        m_u32StringStartIndex = 0;
        m_u32StringEndIndex   = 0;
    }
};


// -----------------------------------------------------------------------------
// Class definition:
// -----------------------------------------------------------------------------

class VerticalSearchKeyboard
{

public:
    QList<VerticalSearchKeyboardListItem>* showKeyboard(QStringList& rfoItemList, quint32 u32language);

private:

    bool bInitializeSearchKeyboardLists( QList<tsKeyboardChar> &rfoLatinSearchKbdList, QList<tsKeyboardChar> &rfoNonLatinSearchKbdList, quint32 u32language );
    
    void vParseItemList( QStringList &rfoItemList, QList<tsKeyboardChar> &rfoSearchKbdList );
    
    QList<VerticalSearchKeyboardListItem>* poGetVerticalKeyboardList( QList<tsKeyboardChar> &rfoLatinSearchKbdList, QList<tsKeyboardChar> &rfoNonLatinSearchKbdList, bool bNonLatinAvailable );
  
};



// -----------------------------------------------------------------------------
// Trace capabilities are available via 'sterr' only with this library 
// as ETG traces unfortunately cannot be used (would result in build errors)
// -----------------------------------------------------------------------------

// #define DEBUG_FPRINTF_VERT_KEYBOARD  // This define  SHALL BE COMMENTED OUT  in officially released SW versions.

#ifdef DEBUG_FPRINTF_VERT_KEYBOARD
   // Option 1: (without file name or line number)
   #define VKBD_FPRINTF(fmtstr, args...) ( fprintf(stderr, "\n VERT_KEYBOARD_DEBUG:" fmtstr "\n", ##args) )
   
   // Option 2: (with file name and line number)
   // #define VKBD_FILE ( strrchr(__FILE__, '/') ? strrchr(__FILE__, '/') + 1 : __FILE__ )
   // #define VKBD_FPRINTF(fmtstr, args...) ( fprintf(stderr, "\n VERT_KEYBOARD_DEBUG: [%s(%d)]: " fmtstr " \n", VKBD_FILE, __LINE__, ##args) )

   // Note: To see the trace output in TTFIS, activate trace class: TR_SET_CONFIG TR_CLASS_STDOUT 3

#else
   #define VKBD_FPRINTF(dummy...)    
#endif  // DEBUG_FPRINTF_VERT_KEYBOARD


#endif // FC_PHONEBOOK_VERTICALKEYBOARD_H
