/*******************************************************************************
*
* FILE:          FC_PhoneBook_VCard_Parser.cpp
*
* SW-COMPONENT:  FC_PhoneBook application
*
* PROJECT:
*
* DESCRIPTION:
*
* AUTHOR:
*
* COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
*
*******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/

#include "FC_PhoneBook_VCard_Parser.h"
#include "FC_PhoneBook_VCard_Utils.h"
#include "FC_PhoneBook_VCard_LineReader.h"

// We will not support any nesting according to the specs.
#define FC_PB_MAX_VERSIT_DOCUMENT_NESTING_DEPTH 20


QHash<QPair<FC_PhoneBook_VCard_Document::CardType,QString>, FC_PhoneBook_VCard_Property::Type>*
    FC_PhoneBook_VCard_Parser::m_ValueTypeMap = NULLPTR;

/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::FC_PhoneBook_VCard_Parser()
*
* DESCRIPTION: Constructor
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
FC_PhoneBook_VCard_Parser::FC_PhoneBook_VCard_Parser()
  : m_IoDevice(0),
  m_DocumentNestingLevel(0),
  m_DefaultCodec(QTextCodec::codecForName("UTF-8"))
{

}

/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::~FC_PhoneBook_VCard_Parser()
*
* DESCRIPTION: Destructor
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
FC_PhoneBook_VCard_Parser::~FC_PhoneBook_VCard_Parser()
{
	m_DefaultCodec= NULLPTR;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::valueTypeMap()
*
* DESCRIPTION:
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
QHash<QPair<FC_PhoneBook_VCard_Document::CardType,QString>, FC_PhoneBook_VCard_Property::Type>*
    FC_PhoneBook_VCard_Parser::valueTypeMap()
{
  if (!m_ValueTypeMap)
  {
    m_ValueTypeMap = new QHash<QPair<FC_PhoneBook_VCard_Document::CardType,QString>, FC_PhoneBook_VCard_Property::Type>();

    m_ValueTypeMap->insert(qMakePair(FC_PhoneBook_VCard_Document::VCard21, QString::fromLatin1("AGENT")),
                          FC_PhoneBook_VCard_Property::Document);
    m_ValueTypeMap->insert(qMakePair(FC_PhoneBook_VCard_Document::VCard30, QString::fromLatin1("AGENT")),
                          FC_PhoneBook_VCard_Property::Document);

    m_ValueTypeMap->insert(qMakePair(FC_PhoneBook_VCard_Document::VCard21, QString::fromLatin1("N")),
                          FC_PhoneBook_VCard_Property::Compound);
    m_ValueTypeMap->insert(qMakePair(FC_PhoneBook_VCard_Document::VCard30, QString::fromLatin1("N")),
                          FC_PhoneBook_VCard_Property::Compound);

    m_ValueTypeMap->insert(qMakePair(FC_PhoneBook_VCard_Document::VCard21, QString::fromLatin1("SOUND")),
                          FC_PhoneBook_VCard_Property::Compound); // -bn: Added for 'TYPE=X-IRMC-N' dispatching (Japanese phonetics)
    m_ValueTypeMap->insert(qMakePair(FC_PhoneBook_VCard_Document::VCard30, QString::fromLatin1("SOUND")),
                          FC_PhoneBook_VCard_Property::Compound);  
						  
    m_ValueTypeMap->insert(qMakePair(FC_PhoneBook_VCard_Document::VCard21, QString::fromLatin1("SORT-STRING")),  
                          FC_PhoneBook_VCard_Property::Compound); // -bn: Added for 'TYPE=X-IRMC-N' dispatching (Japanese phonetics)
    m_ValueTypeMap->insert(qMakePair(FC_PhoneBook_VCard_Document::VCard30, QString::fromLatin1("SORT-STRING")),
                          FC_PhoneBook_VCard_Property::Compound);
                         
    m_ValueTypeMap->insert(qMakePair(FC_PhoneBook_VCard_Document::VCard21, QString::fromLatin1("ADR")),
                          FC_PhoneBook_VCard_Property::Compound);
    m_ValueTypeMap->insert(qMakePair(FC_PhoneBook_VCard_Document::VCard30, QString::fromLatin1("ADR")),
                          FC_PhoneBook_VCard_Property::Compound);

    m_ValueTypeMap->insert(qMakePair(FC_PhoneBook_VCard_Document::VCard21, QString::fromLatin1("GEO")),
                          FC_PhoneBook_VCard_Property::Compound);
    m_ValueTypeMap->insert(qMakePair(FC_PhoneBook_VCard_Document::VCard30, QString::fromLatin1("GEO")),
                          FC_PhoneBook_VCard_Property::Compound);

    m_ValueTypeMap->insert(qMakePair(FC_PhoneBook_VCard_Document::VCard21, QString::fromLatin1("ORG")),
                          FC_PhoneBook_VCard_Property::Compound);
    m_ValueTypeMap->insert(qMakePair(FC_PhoneBook_VCard_Document::VCard30, QString::fromLatin1("ORG")),
                          FC_PhoneBook_VCard_Property::Compound);

    m_ValueTypeMap->insert(qMakePair(FC_PhoneBook_VCard_Document::VCard21, QString::fromLatin1("NICKNAME")),
                          FC_PhoneBook_VCard_Property::List);
    m_ValueTypeMap->insert(qMakePair(FC_PhoneBook_VCard_Document::VCard30, QString::fromLatin1("NICKNAME")),
                          FC_PhoneBook_VCard_Property::List);

    m_ValueTypeMap->insert(qMakePair(FC_PhoneBook_VCard_Document::VCard21, QString::fromLatin1("CATEGORIES")),
                          FC_PhoneBook_VCard_Property::List);
    m_ValueTypeMap->insert(qMakePair(FC_PhoneBook_VCard_Document::VCard30, QString::fromLatin1("CATEGORIES")),
                          FC_PhoneBook_VCard_Property::List);

    m_ValueTypeMap->insert(qMakePair(FC_PhoneBook_VCard_Document::VCard21, QString::fromLatin1("X-CHILDREN")),
                          FC_PhoneBook_VCard_Property::List);
    m_ValueTypeMap->insert(qMakePair(FC_PhoneBook_VCard_Document::VCard30, QString::fromLatin1("X-CHILDREN")),
                          FC_PhoneBook_VCard_Property::List);

    m_ValueTypeMap->insert(qMakePair(FC_PhoneBook_VCard_Document::VCard21, QString::fromLatin1("X-NICKNAME")),
                          FC_PhoneBook_VCard_Property::List);
    m_ValueTypeMap->insert(qMakePair(FC_PhoneBook_VCard_Document::VCard30, QString::fromLatin1("X-NICKNAME")),
                          FC_PhoneBook_VCard_Property::List);

  }
  return m_ValueTypeMap;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::parseData()
*
* DESCRIPTION: The actual data is read through this function
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
QList<FC_PhoneBook_VCard_Document> FC_PhoneBook_VCard_Parser::parseData(const QByteArray &inputData)
{
  m_InputBytes.reset(new QBuffer);
  m_InputBytes->setData(inputData);
  m_InputBytes->open(QIODevice::ReadOnly);
  m_IoDevice = m_InputBytes.data();

  m_Documents.clear();

  FC_PhoneBook_VCard_LineReader lineReader(m_IoDevice, m_DefaultCodec);

  while(!lineReader.atEnd())
  {
    FC_PhoneBook_VCard_Document document;
    int oldPos = lineReader.bytesProcessed();
    bool ok = parseDocument(&lineReader, &document);

    if (ok)
    {
      if (document.isEmpty())
        break;
      else
      {
        m_Documents.append(document);
      }
    }
    else
    {
      if (lineReader.bytesProcessed() == oldPos)
        break;
    }
  }

  return m_Documents;

}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::parseVersitDocument()
*
* DESCRIPTION: Parses a document.
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
bool FC_PhoneBook_VCard_Parser::parseDocument(FC_PhoneBook_VCard_LineReader* lineReader, FC_PhoneBook_VCard_Document* document)
{
  if (m_DocumentNestingLevel >= FC_PB_MAX_VERSIT_DOCUMENT_NESTING_DEPTH)
    return false;

  if (document->type() == FC_PhoneBook_VCard_Document::Invalid)
    document->setType(FC_PhoneBook_VCard_Document::VCard30);

  FC_PhoneBook_VCard_Property property;

  property = parseNextProperty(document->type(), lineReader);
  QString propertyValue = property.value().trimmed().toUpper();
  if (property.isEmpty())
  {
    document->clear();
    return true;
  }
  else if (property.name() == QLatin1String("BEGIN"))
  {
    if (propertyValue == QLatin1String("VCARD"))
    {
      document->setComponentType(propertyValue);
    }
    else
    {
      document->clear();
      return false;
    }
  }
  else
  {
    document->clear();
    return false;
  }

  return parseDocumentBody(lineReader, document);
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::parseVersitDocumentBody()
*
* DESCRIPTION:
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
bool FC_PhoneBook_VCard_Parser::parseDocumentBody(FC_PhoneBook_VCard_LineReader* lineReader, FC_PhoneBook_VCard_Document* document)
{
  m_DocumentNestingLevel++;
  bool parsingOk = true;
  while (true)
  {
    FC_PhoneBook_VCard_Property property = parseNextProperty(document->type(), lineReader);

    if (property.name() == QLatin1String("BEGIN"))
    {
      FC_PhoneBook_VCard_Document subDocument;
      subDocument.setType(document->type());
      subDocument.setComponentType(property.value().trimmed().toUpper());
      if (!parseDocumentBody(lineReader, &subDocument))
        break;
      document->addSubDocument(subDocument);
    }
    else if (property.name() == QLatin1String("VERSION"))
    {
      if (!setVersionFromProperty(document, property))
      {
        parsingOk = false;
        break;
      }
    }
    else if (property.name() == QLatin1String("END"))
    {
      break;
    }
    else if (property.name().isEmpty())
    {
      parsingOk = false;
      break;
    }
    else
    {
      document->addProperty(property);
    }
  }

  if (!parsingOk)
    document->clear();

  m_DocumentNestingLevel--;

  return parsingOk;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::parseNextVersitProperty()
*
* DESCRIPTION: Parses a property.
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
FC_PhoneBook_VCard_Property FC_PhoneBook_VCard_Parser::parseNextProperty(
    FC_PhoneBook_VCard_Document::CardType type,
    FC_PhoneBook_VCard_LineReader* lineReader)
{
  FC_PhoneBook_VCard_ByteArray line = lineReader->readLine();
  if (line.isEmpty())
    return FC_PhoneBook_VCard_Property();

  QPair<QStringList,QString> groupsAndName =
      extractPropertyGroupsAndName(&line, lineReader->codec());

  FC_PhoneBook_VCard_Property property;
  property.setGroups(groupsAndName.first);
  property.setName(groupsAndName.second);

  QPair<FC_PhoneBook_VCard_Document::CardType, QString> key =
      qMakePair(type, property.name());
  if (valueTypeMap()->contains(key))
    property.setType(valueTypeMap()->value(key));

  if (type == FC_PhoneBook_VCard_Document::VCard21)
    parseVCard21Property(&line, &property, lineReader);
  else if (type == FC_PhoneBook_VCard_Document::VCard30)
    parseVCard30Property(type, &line, &property, lineReader);

  return property;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::parseVCard21Property()
*
* DESCRIPTION: Parses the property vCard 2.1 syntax.
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_VCard_Parser::parseVCard21Property(FC_PhoneBook_VCard_ByteArray* line, FC_PhoneBook_VCard_Property* property,
                                                     FC_PhoneBook_VCard_LineReader* lineReader)
{
  property->setParameters(extractVCard21PropertyParams(line, lineReader->codec()));

  QByteArray value = line->toByteArray();
  if (property->getType() == FC_PhoneBook_VCard_Property::Document)
  {

    if (value == "BEGIN:VCARD")
    {
      lineReader->appendLine(value);
    }
    else if (value.isEmpty())
    {
    }
    else
    {
      property->clear();
      return;
    }
    FC_PhoneBook_VCard_Document subDocument(FC_PhoneBook_VCard_Document::VCard21);
    if (!parseDocument(lineReader, &subDocument))
    {
      property->clear();
    }
    else
    {
      property->setValue(QVariant::fromValue(subDocument));
    }
  }
  else
  {
    QTextCodec* codec;
    bool isBinary = unEncode(&value, property, lineReader);
    if (isBinary)
    {
      property->setValue(value);
      property->setType(FC_PhoneBook_VCard_Property::Binary);

      QByteArray baPropertyName  = property->name().toUtf8();
      const char* name = baPropertyName .data();

      if( strcmp (name,"PHOTO")!= 0)
      {
      property->setValue(decodeCharset(value, property, lineReader->codec(), &codec));
      splitStructuredValue(property, false);
      }
    }
    else
    {
      property->setValue(decodeCharset(value, property, lineReader->codec(), &codec));
      splitStructuredValue(property, false);
    }
  }
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::parseVCard30Property()
*
* DESCRIPTION: Parses the property vCard 3.0 syntax.
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_VCard_Parser::parseVCard30Property(FC_PhoneBook_VCard_Document::CardType type,
                                                     FC_PhoneBook_VCard_ByteArray* line, FC_PhoneBook_VCard_Property* property,
                                                     FC_PhoneBook_VCard_LineReader* lineReader)
{
  property->setParameters(extractVCard30PropertyParams(line, lineReader->codec()));
  bool isList;

  QByteArray value = line->toByteArray();

  QTextCodec* codec;

  if (property->getType() == FC_PhoneBook_VCard_Property::Document)
  {
    QString valueString(decodeCharset(value, property, lineReader->codec(), &codec));
    removeBackSlashEscaping(&valueString);

    QByteArray subDocumentValue(codec->fromUnicode(valueString));
    QBuffer subDocumentData(&subDocumentValue);
    subDocumentData.open(QIODevice::ReadOnly);
    (void)subDocumentData.seek((qint64)0);
    FC_PhoneBook_VCard_LineReader subDocumentLineReader(&subDocumentData, codec);

    FC_PhoneBook_VCard_Document subDocument(type);
    if (!parseDocument(&subDocumentLineReader, &subDocument))
    {
      property->clear();
    }
    else
    {
      property->setValue(QVariant::fromValue(subDocument));
    }
  }
  else
  {
    bool isBinary = unEncode(&value, property, lineReader);
    if (isBinary)
    {
      property->setValue(value);
      property->setType(FC_PhoneBook_VCard_Property::Binary);
      QByteArray baPropertyName  = property->name().toUtf8();
      const char* name = baPropertyName .data();
      
      if( strcmp (name,"PHOTO")!= 0)
      {

            property->setValue(decodeCharset(value, property, lineReader->codec(), &codec));
            isList = splitStructuredValue(property, true);

            if (isList)
            {
               QStringList list1 = property->value<QStringList> ();
               for (int i = 0; i < list1.length(); i++)
               {
                  removeBackSlashEscaping(&list1[i]);
               }
               property->setValue(list1);
            }
            else
            {
               QString prop_value = property->value();
               removeBackSlashEscaping(&prop_value);
               property->setValue(prop_value);
            }

      }
    }
    else
    {
      property->setValue(decodeCharset(value, property, lineReader->codec(), &codec));
      isList = splitStructuredValue(property, true);

      if (isList)
      {
        QStringList list = property->value<QStringList>();
        for (int i = 0; i < list.length(); i++)
        {
          removeBackSlashEscaping(&list[i]);
        }
        property->setValue(list);
      }
      else
      {
        QString prop_value = property->value();
        removeBackSlashEscaping(&prop_value);
        property->setValue(prop_value);
      }
    }
  }
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::setVersionFromProperty()
*
* DESCRIPTION: Sets version to document
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
bool FC_PhoneBook_VCard_Parser::setVersionFromProperty(FC_PhoneBook_VCard_Document* document, const FC_PhoneBook_VCard_Property& property) const
{
  QString value = property.value().trimmed();
  if (document->componentType() == QLatin1String("VCARD")
    && value == QLatin1String("2.1"))
    {
    document->setType(FC_PhoneBook_VCard_Document::VCard21);
  }
  else if (document->componentType() == QLatin1String("VCARD")
    && value == QLatin1String("3.0"))
    {
    document->setType(FC_PhoneBook_VCard_Document::VCard30);
  }
  else
  {
    return false;
  }

  return true;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::unencode()
*
* DESCRIPTION: Helper function to decode the value.
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
bool FC_PhoneBook_VCard_Parser::unEncode(QByteArray* value,
                                         FC_PhoneBook_VCard_Property* property,
                                         FC_PhoneBook_VCard_LineReader* lineReader) const
{
  QStringList encodingParameters = property->parameters().values(QLatin1String("ENCODING"));
  QStringList typeParameters = property->parameters().values(QLatin1String("TYPE"));
  if (encodingParameters.contains(QLatin1String("QUOTED-PRINTABLE"), Qt::CaseInsensitive))
  {

    while (value->endsWith('='))
    {
      value->chop(1);
      value->append(lineReader->readLine().toByteArray());
    }
    decodeQuotedPrintable(value);
    property->removeParameters(QLatin1String("ENCODING"));
    return false;
  }
  else if (encodingParameters.contains(QLatin1String("BASE64"), Qt::CaseInsensitive)
    || encodingParameters.contains(QLatin1String("B"), Qt::CaseInsensitive)
    || typeParameters.contains(QLatin1String("BASE64"), Qt::CaseInsensitive)
    || typeParameters.contains(QLatin1String("B"), Qt::CaseInsensitive))
    {
    *value = QByteArray::fromBase64(*value);
    property->removeParameters(QLatin1String("ENCODING"));
    return true;
  }
  return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::decodeCharset()
*
* DESCRIPTION:  Decodes according to the charset.
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
QString FC_PhoneBook_VCard_Parser::decodeCharset(const QByteArray& value,
                                                 FC_PhoneBook_VCard_Property* property,
                                                 QTextCodec* defaultCodec,
                                                 QTextCodec** codec) const
{
  const QString charset(QLatin1String("CHARSET"));
  if (property->parameters().contains(charset))
  {
    QString charsetValue = *property->parameters().find(charset);
    property->removeParameters(charset);
    *codec = QTextCodec::codecForName(charsetValue.toLatin1());
    if (*codec != NULL)
    {
      return (*codec)->toUnicode(value);
    }
    else
    {
      *codec = defaultCodec;
      return defaultCodec->toUnicode(value);
    }
  }
  *codec = defaultCodec;
  return defaultCodec->toUnicode(value);
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::decodeQuotedPrintable()
*
* DESCRIPTION: Decodes Quoted-Printable encoded (RFC 1521) characters.
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_VCard_Parser::decodeQuotedPrintable(QByteArray* text) const
{
  for (int i=0; i < text->length(); i++)
  {
    char current = text->at(i);
    if (current == '=' && i+2 < text->length())
    {
      char next = text->at(i+1);
      char nextAfterNext = text->at(i+2);
      if (((next >= 'a' && next <= 'f') ||
           (next >= 'A' && next <= 'F') ||
           (next >= '0' && next <= '9')) &&
          ((nextAfterNext >= 'a' && nextAfterNext <= 'f') ||
           (nextAfterNext >= 'A' && nextAfterNext <= 'F') ||
           (nextAfterNext >= '0' && nextAfterNext <= '9')))
      {
        bool ok;
        //To solve Gen4 warning
        int decodedChar(text->mid(i+1, 2).toInt(&ok,16));
        if (ok) {
          (*text)[i] = (char)decodedChar;
          text->remove(i+1, 2);
        }
      }
      else if (next == '\r' && nextAfterNext == '\n')
      {
        text->remove(i, 3);
      }
    }
  }
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::extractPropertyGroupsAndName()
*
* DESCRIPTION:  Reads the groups and the name of the property
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
QPair<QStringList,QString>FC_PhoneBook_VCard_Parser::extractPropertyGroupsAndName(
    FC_PhoneBook_VCard_ByteArray* line, QTextCodec *codec) const
{
  const QByteArray semicolon = FC_PhoneBook_VCard_Utils::encode(';', codec);
  const QByteArray colon = FC_PhoneBook_VCard_Utils::encode(':', codec);
  const QByteArray backslash = FC_PhoneBook_VCard_Utils::encode('\\', codec);
  QPair<QStringList,QString> groupsAndName;
  int length = 0;

  int separatorLength = semicolon.length();
  for (int i = 0; i < line->size() - separatorLength + 1; i++)
  {
    if ((containsAt(*line, semicolon, i) && !containsAt(*line, backslash, i-separatorLength))
      || containsAt(*line, colon, i))
      {
      length = i;
      break;
    }
  }
  if (length > 0)
  {
    QString trimmedGroupsAndName = codec->toUnicode(line->left(length)).trimmed();
    QStringList parts = trimmedGroupsAndName.split(QLatin1Char('.'));
    if (parts.count() > 1)
    {
      groupsAndName.second = parts.takeLast();
      groupsAndName.first = parts;
    }
    else
    {
      groupsAndName.second = trimmedGroupsAndName;
    }

    line->chopLeft(length);
  }

  return groupsAndName;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::extractVCard21PropertyParams()
*
* DESCRIPTION:  Reads the property parameters as a QMultiHash
*               The parameters without names are added as "TYPE" parameters.
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
QMultiHash<QString,QString> FC_PhoneBook_VCard_Parser::extractVCard21PropertyParams(
    FC_PhoneBook_VCard_ByteArray* line, QTextCodec *codec) const
{
  QMultiHash<QString,QString> result;
  QList<QByteArray> paramList = extractParams(line, codec);
  while (!paramList.isEmpty())
  {
    QByteArray param = paramList.takeLast();
    QString name = paramName(param, codec);
    QString value = paramValue(param, codec);
    result.insert(name,value);
  }

  return result;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::extractVCard30PropertyParams()
*
* DESCRIPTION:   Reads the property parameters as a QMultiHash
*                The parameters without names are added as "TYPE" parameters.
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
QMultiHash<QString,QString> FC_PhoneBook_VCard_Parser::extractVCard30PropertyParams(
    FC_PhoneBook_VCard_ByteArray* line, QTextCodec *codec) const
{
  QMultiHash<QString,QString> result;
  QList<QByteArray> paramList = extractParams(line, codec);
  while (!paramList.isEmpty())
  {
    QByteArray param = paramList.takeLast();
    QString name(paramName(param, codec));
    removeBackSlashEscaping(&name);
    QString values = paramValue(param, codec);
    QStringList valueList = splitValue(values, QLatin1Char(','), QString::SkipEmptyParts, true);
    foreach (QString value, valueList)
    {
      removeBackSlashEscaping(&value);
      result.insert(name, value);
    }
  }
  return result;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::extractParams()
*
* DESCRIPTION: Reads the parameters as delimited by semicolons
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
QList<QByteArray> FC_PhoneBook_VCard_Parser::extractParams(FC_PhoneBook_VCard_ByteArray* line, QTextCodec *codec) const
{
  const QByteArray colon = FC_PhoneBook_VCard_Utils::encode(':', codec);
  const QByteArray semicolon = FC_PhoneBook_VCard_Utils::encode(';', codec);
  QList<QByteArray> params;

  int colonIndex = line->indexOf(colon);
  if (colonIndex > 0)
  {
    QByteArray nameAndParamsString = line->left(colonIndex);
    params = extractParts(nameAndParamsString, semicolon, codec);

    line->chopLeft(colonIndex + colon.length());
  }
  else if (colonIndex == 0)
  {
    line->chopLeft(colon.length());
  }

  return params;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::extractParts()
*
* DESCRIPTION: Reads the parts separated by separator discarding the separators escaped with a backslash
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
QList<QByteArray> FC_PhoneBook_VCard_Parser::extractParts(
    const QByteArray& text, const QByteArray& separator, QTextCodec* codec) const
{
  QList<QByteArray> parts;
  int partStartIndex = 0;
  int textLength = text.length();
  int separatorLength = separator.length();
  const QByteArray backslash = FC_PhoneBook_VCard_Utils::encode('\\', codec);
  int backslashLength = backslash.length();

  for (int i=0; i < textLength-separatorLength+1; i++)
  {
    if (containsAt(text, separator, i)
      && (i < backslashLength
          || !containsAt(text, backslash, i-backslashLength)))
      {
      int length = i-partStartIndex;
      QByteArray part = extractPart(text,partStartIndex,length);
      parts.append(part);
      partStartIndex = i+separatorLength;
    }
  }

  QByteArray part = extractPart(text,partStartIndex);
  if (part.length() > 0)
    parts.append(part);
  return parts;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::extractPart()
*
* DESCRIPTION:  Extracts a substring
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
QByteArray FC_PhoneBook_VCard_Parser::extractPart(
    const QByteArray& text, int startPosition, int length) const
{
  QByteArray part;
  if (startPosition >= 0)
    part = text.mid(startPosition,length).trimmed();
  return part;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::paramName()
*
* DESCRIPTION: Reads the name of the parameter
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
QString FC_PhoneBook_VCard_Parser::paramName(const QByteArray& parameter, QTextCodec* codec) const
{
  if (parameter.trimmed().length() == 0)
    return QString();
  const QByteArray equals = FC_PhoneBook_VCard_Utils::encode('=', codec);
  int equalsIndex = parameter.indexOf(equals);
  if (equalsIndex > 0)
  {
    return codec->toUnicode(parameter.left(equalsIndex)).trimmed();
  }

  return QLatin1String("TYPE");
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::paramValue()
*
* DESCRIPTION: Reads the value of the parameter
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
QString FC_PhoneBook_VCard_Parser::paramValue(const QByteArray& parameter, QTextCodec* codec) const
{
  QByteArray value(parameter);
  const QByteArray equals = FC_PhoneBook_VCard_Utils::encode('=', codec);
  int equalsIndex = parameter.indexOf(equals);
  if (equalsIndex > 0)
  {
    int valueLength = parameter.length() - (equalsIndex + equals.length());
    value = parameter.right(valueLength).trimmed();
  }

  return codec->toUnicode(value);
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::containsAt()
*
* DESCRIPTION: Returns true if text contains ba at index
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/

/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::splitStructuredValue()
*
* DESCRIPTION: Here a known property is split into QStringList
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
bool FC_PhoneBook_VCard_Parser::splitStructuredValue(
    FC_PhoneBook_VCard_Property* property,
    bool hasEscapedBackslashes) const
{
  QVariant variant = property->variantValue();
  if (property->getType() == FC_PhoneBook_VCard_Property::Compound || property->getType() == FC_PhoneBook_VCard_Property::Binary )
  {
    variant.setValue(splitValue(variant.toString(), QLatin1Char(';'),
                                QString::KeepEmptyParts, hasEscapedBackslashes));
    property->setValue(variant);
    return true;
  }
  else if (property->getType() == FC_PhoneBook_VCard_Property::List)
  {
    variant.setValue(splitValue(variant.toString(), QLatin1Char(','),
                                QString::SkipEmptyParts, hasEscapedBackslashes));
    property->setValue(variant);
    return true;
  }
  return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::splitValue()
*
* DESCRIPTION: Splits string into substrings wherever sep occurs.
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
QStringList FC_PhoneBook_VCard_Parser::splitValue(const QString& string,
                                                  const QChar& sep,
                                                  QString::SplitBehavior behaviour,
                                                  bool hasEscapedBackslashes)
{
  QStringList list;
  bool isEscaped = false;
  int segmentStartIndex = 0;
  QString segment;
  for (int i = 0; i < string.length(); i++)
  {
    if (string.at(i) == QLatin1Char('\\'))
    {
      if (hasEscapedBackslashes)
        isEscaped = !isEscaped;
      else
        isEscaped = true;
    }
    else if (string.at(i) == sep)
    {
      if (isEscaped)
      {
        segment += string.midRef(segmentStartIndex, i-segmentStartIndex-1);
        segment += sep;
      }
      else
      {
        segment += string.midRef(segmentStartIndex, i - segmentStartIndex);
        if (behaviour == QString::KeepEmptyParts || !segment.isEmpty())
          list.append(segment);
        segment.clear();
      }
      segmentStartIndex = i+1;
      isEscaped = false;
    }
    else
    {
      isEscaped = false;
    }
  }

  segment += string.midRef(segmentStartIndex);
  if (behaviour == QString::KeepEmptyParts || !segment.isEmpty())
    list.append(segment);
  return list;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_Parser::removeBackSlashEscaping()
*
* DESCRIPTION:  Removes backslash escaping for line breaks (CRLFs), colons, semicolons, backslashes and commas
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_VCard_Parser::removeBackSlashEscaping(QString* text)
{
  if (!(text->startsWith(QLatin1Char('"')) && text->endsWith(QLatin1Char('"'))))
  {
    /* replaces \; with ;
                    \, with ,
                    \: with :
                    \\ with \
         */
    text->replace(QRegExp(QLatin1String("\\\\([;,:\\\\])")), QLatin1String("\\1"));
    // replaces \n with a CRLF
    text->replace(QLatin1String("\\n"), QLatin1String("\r\n"), Qt::CaseInsensitive);
  }
}
