/*******************************************************************************
*
* FILE:          FC_PhoneBook_VCard_LineReader.cpp
*
* SW-COMPONENT:  FC_PhoneBook application
*
* PROJECT:
*
* DESCRIPTION:
*
* AUTHOR:
*
* COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
*
*******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/
#include "FC_PhoneBook_VCard_LineReader.h"
#include "FC_PhoneBook_VCard_Utils.h"
#include "FC_PhoneBook_VCard_Parser.h"

/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_LineReader::FC_PhoneBook_VCard_LineReader()
*
* DESCRIPTION:   Constructor
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
FC_PhoneBook_VCard_LineReader::FC_PhoneBook_VCard_LineReader(QIODevice* device, QTextCodec *textcodec, int chunkSize)
  : m_Device(device),
  m_Codec(textcodec),
  m_ChunkSize(chunkSize),
  m_CrlfList(*FC_PhoneBook_VCard_Utils::newlineList(m_Codec)),
  m_Buffer(FC_PhoneBook_VCard_ByteArray(QByteArray())),
  m_BytesProcessed(0),
  m_SearchFrom(0),
  m_Colon(FC_PhoneBook_VCard_Utils::encode(':', m_Codec)),
  m_Equals(FC_PhoneBook_VCard_Utils::encode('=', m_Codec)),
  m_Space(FC_PhoneBook_VCard_Utils::encode(' ', m_Codec)),
  m_Tab(FC_PhoneBook_VCard_Utils::encode('\t', m_Codec)),
  m_LineFeed(FC_PhoneBook_VCard_Utils::encode('\n', m_Codec))
{
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_LineReader::readLine()
*
* DESCRIPTION: Do some error checking before reading a line.
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
FC_PhoneBook_VCard_ByteArray FC_PhoneBook_VCard_LineReader::readLine()
{
  if (!m_AppendedLines.isEmpty())
  {
    FC_PhoneBook_VCard_ByteArray retval(m_AppendedLines.pop());
    return retval;
  }

  readOneLine(&m_Buffer);


  while (true)
  {
    int prevStart = m_Buffer.mStart;
    int prevEnd = m_Buffer.mEnd;

    readOneLine(&m_Buffer);

    FC_PhoneBook_VCard_ByteArray prevLine(m_Buffer.mData, prevStart, prevEnd);
    if (m_Buffer.isEmpty()
      || m_Buffer.contains(m_Colon))
      //  || prevLine.endsWith(m_Equals)) // -bn: 07.2017: Commented out this condition as fix for ticket NCG3D-54824
        
       // 07.2017: Motivation to comment out the above mentioned condition:
       // - My understanding is that here an end of subsequent base64 encoded lines should be detected via condition
       //  "prevLine.endsWith(m_Equals)"
       // - But this condition "prevLine.endsWith(m_Equals)" is not sufficient for this purpose:
       //   (1) The base64 encoding could end with a line, which has not any "filler byte" with value '=',
       //   (2) Even if the current lines end with '=', the base64 encoding might continue in the next line with a
       //       second "filler byte" with value '='.
       //       At the moment, exactly this situation is not handled by our vCard Parser and resulted in ticket NCG3D-54824.
       // - Therefore I conclude that the condition "prevLine.endsWith(m_Equals)", to detect an end of a base64 encoding
       //   is not correct and therefore I have removed it especially based on (2).
       // Side Notes:
       // - As vCard encoding is not trivial, I hope that I have not missed here a different purpose for which the condition
       //   "prevLine.endsWith(m_Equals)" might have been added here originally.
       //   -- The use of the condition "prevLine.endsWith(m_Equals)" was in this file since its original implementation
       //      by MV and has never been changed since its first official release in 2009 as far as I know..
       // - Now in 07.2017 we have decided to change it for the reasons as described above ;-).

    {
      m_Buffer.setBounds(prevStart, prevEnd);
      break;
    }
    else
    {
      int crlfLen = m_Buffer.mStart-prevEnd;
      m_Buffer.mData.remove(prevEnd, crlfLen);
      m_Buffer.setBounds(prevStart, m_Buffer.mEnd - crlfLen);
    }
  }

  m_Buffer.dropOldData();
  m_BytesProcessed += m_Buffer.size();
  return m_Buffer;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_LineReader::readOneLine()
*
* DESCRIPTION: Reads a line and updates the cursor
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_VCard_LineReader::readOneLine(FC_PhoneBook_VCard_ByteArray* cursor)
{
  cursor->mStart = cursor->mEnd;
  m_SearchFrom = cursor->mStart;

  if (tryReadLine(cursor, false))
  {
    return;
  }

  while (!m_Device->atEnd())
  {
    QByteArray temp = m_Device->read((qint64)m_ChunkSize);
    if (!temp.isEmpty())
    {
      cursor->mData.append(temp);
      if (tryReadLine(cursor, false))
        return;
    }
    else
    {
      m_Device->waitForReadyRead(500);
    }
  }

  tryReadLine(cursor, true);
  return;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_LineReader::pushLine()
*
* DESCRIPTION:   appends a line to the the list of lines to be read.
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_VCard_LineReader::appendLine(const QByteArray& line)
{
  m_AppendedLines.push(line);
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_LineReader::odometer()
*
* DESCRIPTION: Tracks the number of byte processed by this class
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
int FC_PhoneBook_VCard_LineReader::bytesProcessed()
{
  return m_BytesProcessed;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_LineReader::atEnd()
*
* DESCRIPTION:  Check if we have read all lines or not.
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
bool FC_PhoneBook_VCard_LineReader::atEnd()
{
  return m_AppendedLines.isEmpty() && m_Device->atEnd() && m_Buffer.mEnd == m_Buffer.mData.size();
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_LineReader::codec()
*
* DESCRIPTION: Returns the codec;
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
QTextCodec* FC_PhoneBook_VCard_LineReader::codec()
{
  return m_Codec;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_VCard_LineReader::tryReadLine()
*
* DESCRIPTION: Helper function for readLine
*
*
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
bool FC_PhoneBook_VCard_LineReader::tryReadLine(FC_PhoneBook_VCard_ByteArray *cursor, bool blAtEnd)
{
  int crlfPos = -1;

  int spaceLength = m_Space.length();

  forever
  {
    foreach(const QByteArrayMatcher& crlf, m_CrlfList)
    {
      int crlfLength = crlf.pattern().length();
      crlfPos = crlf.indexIn(cursor->mData, m_SearchFrom);
      if (crlfPos == cursor->mStart)
      {
        cursor->mStart += crlfLength;
        m_SearchFrom = cursor->mStart;
        if(FC_PhoneBook_VCard_Parser::containsAt(cursor->mData, m_LineFeed, m_SearchFrom))
        {
            cursor->mStart += spaceLength;
            m_SearchFrom = cursor->mStart;
        }
        break;
      }
      else if (crlfPos > cursor->mStart)
      {

        if (FC_PhoneBook_VCard_Parser::containsAt(cursor->mData, m_Space, crlfPos + crlfLength)
          || FC_PhoneBook_VCard_Parser::containsAt(cursor->mData, m_Tab, crlfPos + crlfLength))
          {
          cursor->mData.remove(crlfPos, crlfLength + spaceLength);
          m_SearchFrom = crlfPos;
          break;
        }
        else if (!blAtEnd && crlfPos + crlfLength + spaceLength >= cursor->mData.size())
        {
          m_SearchFrom = crlfPos;
          return false;
        }
        else
        {
          cursor->mEnd = crlfPos;
          return true;
        }
      }
    }
    if (crlfPos == -1)
    {
      cursor->mEnd = cursor->mData.size();
      return false;
    }
  }
}
