/*******************************************************************************
*                                                                               
* FILE:          FC_PhoneBook_service_PhoneBook.cpp                 
*                                                                               
* SW-COMPONENT:  FC_PhoneBook application                                  
*                                                                               
* PROJECT:       Bosch                                                               
*                                                                               
* DESCRIPTION:   CCA service PhoneBook.                                     
*                                                                               
* AUTHOR:        Arun Choudhary, Omveer                                                             
*                                                                               
* COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim                          
*
*                                                                               
*******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/
#include "FC_PhoneBook_Debug.h"
#include "FC_PhoneBook_Device_Details.h"
#include "FC_PhoneBook_service_PhoneBook.h"
#include "Database/FC_PhoneBook_Database.h"
#include "Database/FC_PhoneBook_List.h"
#include "Database/FC_PhoneBook_CallHistory_List.h"
#include "Database/FC_PhoneBook_SQLite.h"
#include "FC_PhoneBook_main.h"
#include "FC_PhoneBook_clienthandler_BluetoothSetings.h"
#include "FC_PhoneBook_Dnl_Manager.h"

#include <sstream>
#include <string.h>
using namespace std;

#include <sys/stat.h>   // needed for  "stat(..)" // for "-PHBK-QUERY-ERR-" instrumentation

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"
#include "../Configuration/PhoneBook/Pbk_FeatureConfig.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONEBOOK_SERVICE
#include "trcGenProj/Header/FC_PhoneBook_service_PhoneBook.cpp.trc.h"
#endif

/******************************************************************************/
/*                                                                            */
/* DEFINES                                                                    */
/*                                                                            */
/******************************************************************************/

#include "midw_fi_if.h"
/******************************************************************************/
/*                                                                            */
/* CCA MESSAGE MAP                                                            */
/*                                                                            */
/******************************************************************************/

#define CHECK_VALID(index) (((index>0)&&(index<=FC_PB_TOTAL_NUMBER_OF_PB))?true:false)

fc_phonebook_tclService_PhoneBook* \
fc_phonebook_tclService_PhoneBook::m_poPhoneBookService = NULLPTR;

BEGIN_MSG_MAP(fc_phonebook_tclService_PhoneBook, ahl_tclBaseWork)

   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_CREATEPHONEBOOKLIST,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_CreatePhoneBookList_MethodStart )
   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_REQUESTPHONEBOOKLISTSLICE,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_RequestSlicePhoneBookList_MethodStart )

   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_RELEASEPHONEBOOKLIST,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_ReleasePhoneBookList_MethodStart )

   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_SEARCHKEYBOARDPHONEBOOKLIST,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_SearchKeyboardPhoneBookList_MethodStart )

   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_SEARCHPHONEBOOK,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_SearchPhoneBook_MethodStart )

   // CMG3G-6312
   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_SEARCHCONTACTS,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_SearchContacts_MethodStart )
   //End of CMG3G-6312

   ON_MESSAGE_SVCDATA(MOST_PHONBKFI_C_U16_ADDCONTACT,AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_AddContact_MethodStart)

   ON_MESSAGE_SVCDATA(MOST_PHONBKFI_C_U16_EDITCONTACT,AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_EditContact_MethodStart)

   ON_MESSAGE_SVCDATA(MOST_PHONBKFI_C_U16_DELETECONTACT,AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_DeleteContact_MethodStart)

   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_GETCONTACTDETAILS,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_GetContactDetails_MethodStart )

   ON_MESSAGE_SVCDATA(MOST_PHONBKFI_C_U16_CLEARMISSEDCALLINDICATOR,AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_ClearMissedCallIndicator_MethodStart)

   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_RELEASECALLHISTORYLIST, AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_ReleaseCallHistoryList_MethodStart )				// for releasecallhistorylist

   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_CREATECALLHISTORYLIST, AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_CreateCallHistoryList_MethodStart )					// for createcallhistory list

   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_REQUESTSLICECALLHISTORYLIST, AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_RequestSliceCallHistoryList_MethodStart )			// for requestslicecallhistory list

   ON_MESSAGE_SVCDATA(MOST_PHONBKFI_C_U16_CLEARPHONEBOOKDATA,AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_ClearPhoneBookData_MethodStart)

   ON_MESSAGE_SVCDATA(MOST_PHONBKFI_C_U16_CLEARVEHICLEPHONEBOOK,AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_ClearVehiclePhoneBook_MethodStart)

   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_GETDEVICECALLHISTORYSUPPORT, AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_GetDeviceCallHistorySupport_MethodStart ) 			// for getdevicecallhistorysupport

   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_STOREFAVORITE, AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_Store_Favorite_MethodStart ) 			// for store favorite

   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_ACTIVATEFAVORITE, AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_Activate_Favorite_MethodStart ) 			// For  ActivateFavorite

   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_DELETEFAVORITE, AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_Delete_Favorite_MethodStart ) 			// For DeleteFavorite

   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_GETFAVORITEINFO, AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_Get_FavoriteInfo_MethodStart ) 			// For GetFavoriteInfo

   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_RESETFAVORITES, AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_Reset_Favorites_MethodStart ) 			// For ResetFavorites

   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_GETCONTACTPHOTOINFO, AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_GetContactPhotoInfo_MethodStart )

   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_GETCONTACTPHOTO, AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_GetContactPhoto_MethodStart )

   ON_MESSAGE_SVCDATA(MOST_PHONBKFI_C_U16_AUTOCOMPLETEPHONENUMBERENTRY,AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_AutoCompletePhoneNumberEntry_MethodStart)

   //FIX SUZUKI-1113 API for numeric value to indicate number of calls against a CallHistory item in Phonebook
   //RequestSliceCallHistoryListExtended API Added
   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_REQUESTSLICECALLHISTORYLISTEXTENDED, AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_RequestSliceCallHistoryListExtended_MethodStart )			// for RequestSliceCallHistoryListExtended
   //End of fix

   //FIX SUZUKI-1114 API for numeric value to indicate number of dial-able phone numbers available to a contact item in Phonebook
   //RequestPhoneBookListSliceExtended API Added
   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_REQUESTPHONEBOOKLISTSLICEEXTENDED, AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_RequestPhoneBookListSliceExtended_MethodStart )			// for RequestPhoneBookListSliceExtended
   //End of fix

   // Doors ID: 20263: Manual Phonebook download
   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_STARTPBDOWNLOAD, AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_StartPbDownload_MethodStart )
   
   //FIX CMG3G-8367 IS2424_DownloadOnOff behaviour@FC_Phonebook
   //The user shall have possibility to switch the phonebook and call history list download On or Off individual for each paired phone device.
   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_SWITCHDOWNLOADONOFFSTATE, AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_SwitchDownloadOnOffState_MethodStart )
   //End of fix FIX CMG3G-8367
   
   ON_MESSAGE_SVCDATA(MOST_PHONBKFI_C_U16_CREATECONTACTLIST,AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_CreateContactList_MethodStart)       //CMG3G-8368  IS2426_CreateContactList@FC_Phonebook

   //FIX CMG3G-8372 IS2430_SetContactPhoto@FC_Phonebook
   //The user shall have possibility to set the photo for a specific contact in Vehicle phonebook of a specific device.
   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_SETCONTACTPHOTO, AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_SetContactPhoto_MethodStart)
   //End of fix FIX CMG3G-8372

   //FIX CMG3G-8570 IS2502_Cancel PB Download@FC_Phonebook
   //The user shall have possibility to cancel the ongoing phone book download not the call history download.
   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_CANCELPBDOWNLOAD, AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_CancelPbDownload_MethodStart)
   //End of fix FIX CMG3G-8570
   
   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_SETCONTACTPOSITION, AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_SetContactPosition_MethodStart)

   //Fix CMG3G-8365 IS2421_Configurable phone number type representation@FC_Phonebook
   ON_MESSAGE_SVCDATA(MOST_PHONBKFI_C_U16_ADDCONTACTEXTENDED,AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_AddContactExtended_MethodStart)

   ON_MESSAGE_SVCDATA(MOST_PHONBKFI_C_U16_EDITCONTACTEXTENDED,AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_EditContactExtended_MethodStart)

   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_GETCONTACTDETAILSEXTENDED,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_GetContactDetailsExtended_MethodStart )
   //End of Fix CMG3G-8365 IS2421_Configurable phone number type representation@FC_Phonebook

   //Fix CMG3G-9334 Delete all vehicle phonebook entries at once
   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_CLEARVEHICLEPHONEBOOKEXTENDED,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_ClearVehiclePhonebookExtended_MethodStart )
   //End of fix

   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_SETAUTOMATICCONTACTDOWNLOAD,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_SetAutomaticContactDownload_MethodStart )

   ON_MESSAGE_SVCDATA( MOST_PHONBKFI_C_U16_GETPOSITIONFORCONTACTHANDLE,
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART,
   vHandle_GetPositionForContactHandle_MethodStart )
END_MSG_MAP()

/******************************************************************************/
/*                                                                            */
/* METHODS                                                                    */
/*                                                                            */
/******************************************************************************/

/*******************************************************************************
*
* FUNCTION: fc_phonebook_tclService_PhoneBook::\
fc_phonebook_tclService_PhoneBook(fc_phonebook_tclApp* poMainAppl)
*
* DESCRIPTION: Constructor.
*
*              Create an object of the base class ahl_tclBaseOneThreadService
*              with a pointer to this application, the service identifier,
*              the application identifier and the service version as parameters.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
fc_phonebook_tclService_PhoneBook::
fc_phonebook_tclService_PhoneBook(fc_phonebook_tclApp* poMainAppl)
:ahl_tclBaseOneThreadService(
                             /* Application Pointer              */ poMainAppl,
                             /* ID of offered Service            */ CCA_C_U16_SRV_FB_PHONEBOOK,
                             /* MajorVersion of offered Service  */
                             MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION,
                             /* MinoreVersion of offered Service */
                             MOST_PHONBKFI_C_U16_SERVICE_MINORVERSION,
                             /* PatchVersion of offered Service  */
                             0)
{
   ETG_TRACE_USR4(("fc_phonebook_tclService_PhoneBook() entered."));
   // Here we disable the standard "automatic" service availability handling
   // within the CCX framework temporariliy for our server until our service
   // is really ready. Then we will enable it again.

   // service is made unavaliable till the loading from persistant memory is completed
   // vSetPrivateServiceAvailableAllowed(FALSE);
   m_bServerAvailable = FALSE;
   fc_phonebook_tclService_PhoneBook::m_poPhoneBookService = this;
   /*Initialize dbus client*/
   PhBK_CCA_Dbus_If *pCCA_DBusIf = NULLPTR;

   m_btStack =ALPS_EVOLUTION_BT_STACK; //CMG3G-14314 Fix

   //callback instance
   pCCA_DBusIf = pGetCCAPhoneBookDBusIf_Evo();
   register_PhBkcca_callbacks_Evo(pCCA_DBusIf);
   //dbus interface
   m_pDBus_CCA_If_pb = get_cca_dbus_interface_pb_Evo();
   //End of CMG3G-14314 Fix

   /*Instantiate database */
   m_pPhoneBookDB =  FC_PhoneBook_Database::Instance();

   /*Instantiate Bluetooth Settings-CMG3G-14313-Coverity Fix*/
   m_poBTSettings =  NULLPTR;

   //Instantiate FC_PhoneBook_MemoryUsage after DB
   m_pPhoneBookMemoryUsage = OSAL_NEW FC_PhoneBook_MemoryUsage();

   m_oPbk_DnlManager.vSetPhonebookServiceInstance(this);

   m_poFavoriteProvider = OSAL_NEW FC_Phonebook_tclFavorite_Provider();
   m_pPhoneBookDB->RegisterFavoritesInstance(m_poFavoriteProvider);
   m_poDownloadState = new FC_PhoneBook_DownloadState();//FIX CMG3G-3262 Download State property implementation

   // CMG3G-7983 - Indicate the Phonebook Download Progress
   m_oDownloadProgress.u32NumberOfContacts = 0;
   // CMG3G-11074 - u8DeviceHandle has been added to support two HFP support
   m_oDownloadProgress.u8DeviceHandle = 0;

   memset(m_sVehicleBtAddress, 0, LENGTH_OF_DEVICE_ADDRESS);
   m_u8ActivePhoneDeviceHandle = 0;

   m_u8PbkLang = FC_PBK_DEFAULT_LANG; //CRQ-243550-Vertical Keyboard Language Selection

   ETG_TRACE_USR4(("fc_phonebook_tclService_PhoneBook() exit."));
}

/*******************************************************************************
*
* FUNCTION: fc_phonebook_tclService_PhoneBook::
~fc_phonebook_tclService_PhoneBook(tVoid)
*
* DESCRIPTION: Destructor.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
fc_phonebook_tclService_PhoneBook::~fc_phonebook_tclService_PhoneBook(tVoid)
{
   ETG_TRACE_USR4(("~fc_phonebook_tclService_PhoneBook() entered."));
   /*Free Database instance*/
   FC_PhoneBook_Database::DeleteInstance();
   m_oMisssedCallIndicatorStatus.vDestroy();
   m_oDownloadOnOffListStatus.vDestroy();
   m_oProbablyLockedDevices.vDestroy();
   m_pDBus_CCA_If_pb = NULLPTR;
   m_pPhoneBookDB = NULLPTR;
   m_poBTSettings = NULLPTR;
   delete m_poFavoriteProvider;
   m_poFavoriteProvider= NULLPTR;
   OSAL_DELETE m_pPhoneBookMemoryUsage;
   //FIX CMG3G-3262 Download State property implementation
   delete m_poDownloadState;
   m_poDownloadState = NULLPTR;
   //End of fix
   m_oDevicePhoneBookFeatureSupportExtendedStatus.vDestroy();
   //Fix 159808 DownloadProgreeList property implementation
   m_oDownloadProgressListStatus.vDestroy();
   //End fix 159808
   m_oMissedCallIndicatorListStatus.vDestroy();
   m_oPreferredSortOrderListStatus.vDestroy(); //CMG3G-14241
   m_oPhonebookLanguage_Status.vDestroy(); //NCG3D-44771
   m_oMaxContactCountStatus.vDestroy(); //CRQ-563636
}

/*******************************************************************************
** FUNCTION:   pGetInstance
*******************************************************************************/
/* DESCRIPTION:
*   Get the instance of PhoneBook service
*  PARAMETERS:
*     IN:   None
*
*  RETURNVALUE: fc_phonebook_tclService_PhoneBook* : Pointer to TEL service
********************************************************************************/
fc_phonebook_tclService_PhoneBook* fc_phonebook_tclService_PhoneBook:: pGetInstance(fc_phonebook_tclApp* poMainAppl)
{
   if(m_poPhoneBookService == NULL)
   {
      if(poMainAppl)
      {
         m_poPhoneBookService = new fc_phonebook_tclService_PhoneBook(poMainAppl);
      }
   }
   return m_poPhoneBookService;
}

tVoid fc_phonebook_tclService_PhoneBook::vDestroyPhonebookServiceInstance()
{
   delete m_poPhoneBookService;
   m_poPhoneBookService = NULLPTR;
}
/*******************************************************************************
*
* FUNCTION: tVoid fc_phonebook_tclService_PhoneBook::vOnServiceAvailable()
*
* DESCRIPTION: This function is called by the CCA framework when the service
*              which is offered by this server has become available.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadService::vOnServiceAvailable().
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vOnServiceAvailable()
{
   ETG_TRACE_USR4(("vOnServiceAvailable() entered."));
   m_bServerAvailable = TRUE;
   // RTC 742479 start
   if (m_poBTSettings)
   {
	   ETG_TRACE_USR4(("vOnServiceAvailable() registering for BT properties."));
	   m_poBTSettings->vRegisterForBTProperties();
   }
   // RTC 742479 end
   vMakeFavoriteAvaialble(FC_PB_VEHICLE_PBOOK);
   m_pPhoneBookDB->ChangefavoriteActiveStatus(0);
}
/*******************************************************************************
*
* FUNCTION: tVoid fc_phonebook_tclService_PhoneBook::vOnServiceUnavailable()
*
* DESCRIPTION: This function is called by the CCA framework when the service
*              which is offered by this server has become unavailable.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadService::vOnServiceUnavailable().
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vOnServiceUnavailable()
{
   ETG_TRACE_USR4(("vOnServiceUnavailable() entered."));
}

/*******************************************************************************
*
* FUNCTION: tBool fc_phonebook_tclService_PhoneBook
*                 ::bStatusMessageFactory(tU16 u16FunctionId,
*                                         amt_tclServiceData& roOutMsg,
*                                         amt_tclServiceData* poInMsg)
*
* DESCRIPTION: This function is called by the CCA framework to request ANY
*              property which is offered by this service. For each property
*              accessed via parameter 'u16FunctionId' the user has to prepare
*              the corresponding FI data object which is then copied to the
*              referenced parameter 'roOutMsg'.
*
* PARAMETER: [IN] u16FunctionId = Function ID of the requested property.
*            [OUT] roOutMsg = Reference to the service data object to which the
*                             content of the prepared FI data object should be
*                             copied to.
*            [IN] poInMsg = Selector message which is used to select dedicated
*                           content to be copied to 'roOutMsg' instead of
*                           updating the entire FI data object.
*
* RETURNVALUE: TRUE = Status message for property successfully generated.
*              FALSE = Failed to generate status message for property.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadService::bStatusMessageFactory().
*******************************************************************************/
tBool fc_phonebook_tclService_PhoneBook
::bStatusMessageFactory(tU16 u16FunctionId,
                        amt_tclServiceData& roOutMsg,
                        amt_tclServiceData* poInMsg)
{
   (tVoid) poInMsg;  // These lines are added to avoid LINT warnings. Please
   (tVoid) roOutMsg; // remove as soon as variables are used.

   tBool bSuccess = FALSE;

   ETG_TRACE_USR2(("bStatusMessageFactory() entered. FID = 0x%4x.",u16FunctionId));
   switch(u16FunctionId)
   {

      case MOST_PHONBKFI_C_U16_DEVICEPHONEBOOKFEATURESUPPORT:
      {
         ETG_TRACE_USR2(("bStatusMessageFactory MOST_PHONBKFI_C_U16_DEVICEPHONEBOOKFEATURESUPPORT"));
         bSuccess = fi_tclVisitorMessage(m_oDevicePhoneBookFeatureSupportStatus,MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;
      case MOST_PHONBKFI_C_U16_DEVICEPHONEBOOKFEATURESUPPORTEXTENDED:
      {
         ETG_TRACE_USR2(("bStatusMessageFactory MOST_PHONBKFI_C_U16_DEVICEPHONEBOOKFEATURESUPPORTEXTENDED"));
         bSuccess = fi_tclVisitorMessage(m_oDevicePhoneBookFeatureSupportExtendedStatus,MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;
      case MOST_PHONBKFI_C_U16_MISSEDCALLINDICATOR:
      {
         ETG_TRACE_USR2(("bStatusMessageFactory MOST_PHONBKFI_C_U16_MISSEDCALLINDICATOR"));
         bSuccess = fi_tclVisitorMessage(m_oMisssedCallIndicatorStatus,MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;
      case MOST_PHONBKFI_C_U16_LISTCHANGE:
      {
         ETG_TRACE_USR2(("bStatusMessageFactory MOST_PHONBKFI_C_U16_LISTCHANGE"));
         bSuccess = fi_tclVisitorMessage(m_oListChangeStatus,MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;
      case MOST_PHONBKFI_C_U16_PREFERREDPHONEBOOKSORTORDER:
      {
         ETG_TRACE_USR2(("bStatusMessageFactory MOST_PHONBKFI_C_U16_PREFERREDPHONEBOOKSORTORDER"));
         bSuccess = fi_tclVisitorMessage(m_oPreferredPhoneBookSortOrderStatus,MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;
      case MOST_PHONBKFI_C_U16_FAVORITESCHANGED:
      {
         ETG_TRACE_USR2(("bStatusMessageFactory MOST_PHONBKFI_C_U16_FAVORITESCHANGED"));
         bSuccess = fi_tclVisitorMessage(m_oFavoritesChangedStatus,MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;
      case MOST_PHONBKFI_C_U16_DOWNLOADSTATE:
      {
         ETG_TRACE_USR2(("bStatusMessageFactory MOST_PHONBKFI_C_U16_DOWNLOADSTATE"));
         bSuccess = fi_tclVisitorMessage(m_oDownloadState,MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;
      case MOST_PHONBKFI_C_U16_DOWNLOADSTATEEXTENDED:
      {
         ETG_TRACE_USR2(("bStatusMessageFactory MOST_PHONBKFI_C_U16_DOWNLOADSTATEEXTENDED"));
         bSuccess = fi_tclVisitorMessage(m_oDownloadStateExtended,MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;
      case MOST_PHONBKFI_C_U16_DOWNLOADONOFFLIST:
      {
         ETG_TRACE_USR2(("bStatusMessageFactory MOST_PHONBKFI_C_U16_DOWNLOADONOFFLIST"));
         bSuccess = fi_tclVisitorMessage(m_oDownloadOnOffListStatus,MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;
      case MOST_PHONBKFI_C_U16_DOWNLOADPROGRESS:
      {
         ETG_TRACE_USR2(("bStatusMessageFactory MOST_PHONBKFI_C_U16_DOWNLOADPROGRESS"));
         bSuccess = fi_tclVisitorMessage(m_oDownloadProgress, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;
      case MOST_PHONBKFI_C_U16_DOWNLOADPROGRESSLIST:
      {
         ETG_TRACE_USR2(("bStatusMessageFactory MOST_PHONBKFI_C_U16_DOWNLOADPROGRESSLIST"));
         bSuccess = fi_tclVisitorMessage(m_oDownloadProgressListStatus, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;
      case MOST_PHONBKFI_C_U16_AUTOMATICCONTACTDOWNLOADLIST:
      {
         ETG_TRACE_USR2(("bStatusMessageFactory MOST_PHONBKFI_C_U16_AUTOMATICCONTACTDOWNLOADLIST"));
         bSuccess = fi_tclVisitorMessage(m_oAutomaticContactDownloadListStatus, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;
      case MOST_PHONBKFI_C_U16_PROBABLYLOCKEDDEVICES:
      {
         ETG_TRACE_USR2(("bStatusMessageFactory MOST_PHONBKFI_C_U16_PROBABLYLOCKEDDEVICES"));
         bSuccess = fi_tclVisitorMessage(m_oProbablyLockedDevices,MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;
      case MOST_PHONBKFI_C_U16_MISSEDCALLINDICATORLIST:
      {
         ETG_TRACE_USR2(("bStatusMessageFactory MOST_PHONBKFI_C_U16_MISSEDCALLINDICATORLIST"));
         bSuccess = fi_tclVisitorMessage(m_oMissedCallIndicatorListStatus, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;
      case MOST_PHONBKFI_C_U16_PREFERREDSORTORDERLIST:
      {
         ETG_TRACE_USR2(("bStatusMessageFactory MOST_PHONBKFI_C_U16_PREFERREDSORTORDERLIST"));
         bSuccess = fi_tclVisitorMessage(m_oPreferredSortOrderListStatus,MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;
      case MOST_PHONBKFI_C_U16_PHONEBOOKLANGUAGE:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_PHONBKFI_C_U16_PHONEBOOKLANGUAGE"));
         bSuccess = fi_tclVisitorMessage(m_oPhonebookLanguage_Status, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;
      case MOST_PHONBKFI_C_U16_MAXCONTACTCOUNT:
      {
         ETG_TRACE_USR4(("bStatusMessageFactory MOST_PHONBKFI_C_U16_MAXCONTACTCOUNT"));
         bSuccess = fi_tclVisitorMessage(m_oMaxContactCountStatus, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION).bHandOver(&roOutMsg);
      }
      break;
      default:
      {
         ETG_TRACE_ERR(( "bStatusMessageFactory: Unknown 'FID = %u'",u16FunctionId ));
      }
   }
   if (FALSE == bSuccess)
   {
      ETG_TRACE_ERR(("bStatusMessageFactory(). Creation of message with 'FID = %u' failed.", u16FunctionId));
   }

   return bSuccess;
}

/*******************************************************************************
*
* FUNCTION: tBool fc_phonebook_tclService_PhoneBook
*                 ::bProcessSet(amt_tclServiceData* poMessage,
*                               tBool& bPropertyChanged,
*                               tU16& u16ErrorCode)
*
* DESCRIPTION: This function is called by the CCA framework when it has 
*              received a message for a property with Opcode 'SET' or 'PURESET'
*              and there is no dedicated handler function defined in the
*              message map for this pair of FID and opcode. The user has to
*              set the application specific property to the requested value
*              and the CCA framework then cares about informing the requesting
*              client as well as other registered clients.
*
* PARAMETER: [IN] poMessage = Property to be set.
*            [OUT] bPropertyChanged = Property changed flag to be set to TRUE
*                                     if property has changed. Otherwise to be
*                                     set to FALSE (default).
*            [OUT] u16ErrorCode = Error code to be set if a CCA error occurs,
*                                 otherwise don't touch.
*
* RETURNVALUE: TRUE = Send 'STATUS' message to the requesting client if opcode
*                     was 'SET'. Send 'STATUS' message to other registered
*                     clients as well if [OUT] parameter 'bPropertyChanged'
*                     is set to TRUE.
*              FALSE = Send an error message to the requesting client.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadService::bProcessSet().
*******************************************************************************/
tBool fc_phonebook_tclService_PhoneBook
::bProcessSet(amt_tclServiceData* poMessage,
              tBool& bPropertyChanged,
              tU16& u16ErrorCode)
{
   bPropertyChanged = FALSE;
   tBool bSuccess = FALSE;

   tU16 u16FunctionId = poMessage->u16GetFunctionID();
   ETG_TRACE_USR2(("bProcessSet() entered. FID = 0x%4x.", u16FunctionId));

   switch(u16FunctionId)
   {
      case MOST_PHONBKFI_C_U16_PREFERREDPHONEBOOKSORTORDER:
      {
         fi_tclVisitorMessage oVisitorMsg(poMessage);
         ETG_TRACE_USR4(("bProcessSet::MOST_PHONBKFI_C_U16_PREFERREDPHONEBOOKSORTORDER entered"));
         // Create the (message related) FI data object
         most_phonbkfi_tclMsgPreferredPhoneBookSortOrderPureSet oPreferredPhoneBookSortOrderStatus_PureSet;

         // Unmarshal the FI visitor message to the given FI data object.
         // The parameters are copied individually from shared memory to the FI data
         // object.

         if (OSAL_ERROR
            != oVisitorMsg.s32GetData(oPreferredPhoneBookSortOrderStatus_PureSet, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION))
         {

            ETG_TRACE_USR1(("bProcessSet::Preferred Phone Book Sort Order:: %u", ETG_ENUM(TR_PBKPREFSORTTYPE, m_oPreferredPhoneBookSortOrderStatus.e8PreferredPhoneBookSortOrder.enType)));
            bPropertyChanged = TRUE;
            tU8 u8DeviceHandle = u8GetActivePhoneDeviceHandle();

            FC_PhoneBook_SQLite* poFC_PhoneBook_SQLite = FC_PhoneBook_SQLite::GetSQLiteInstance();
            if (poFC_PhoneBook_SQLite)
            {
               quint8 u8SortOrder = oPreferredPhoneBookSortOrderStatus_PureSet.e8PreferredPhoneBookSortOrder.enType;
               ETG_TRACE_USR2(("bProcessSet::Updating PreferredPhoneBookSortOrderStatus sort order in database = %u", u8SortOrder));

               //Updating Preferred sort order to the DB and clients.
               poFC_PhoneBook_SQLite->UpdatePreferredSortOrder(u8SortOrder);
               vPhoneBook_UpdatePreferredPBSortOrderStatus();

               if(0 != u8DeviceHandle)
               {
                  //Updating the Preferred sort order List to the DB and clients
                  tBool bPrefSortSuccess = FALSE;
                  bPrefSortSuccess = poFC_PhoneBook_SQLite->UpdatePreferredSortOrderForDeviceHandle(u8DeviceHandle, u8SortOrder);
                  if(bPrefSortSuccess)
                  {
                     vPhoneBook_UpdatePreferredSortOrderListStatus();
                  }
                  else
                  {
                     ETG_TRACE_ERR(( "Preferred Sort Order not updated successfully" ));
                  }
               }
            }
         }
         else
         {
            ETG_TRACE_ERR(( "PreferredPhoneBookSortOrderStatus: CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED" ));
            u16ErrorCode = CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
         }

         bSuccess = TRUE;
         // Always destroy the (message related) FI data object (before leaving
         // its creation scope)
         oPreferredPhoneBookSortOrderStatus_PureSet.vDestroy();
      }
      break;
      case MOST_PHONBKFI_C_U16_PREFERREDSORTORDERLIST:
      {
         ETG_TRACE_USR4(("bProcessSet::MOST_PHONBKFI_C_U16_PREFERREDSORTORDERLIST entered"));

         fi_tclVisitorMessage oVisitorMsg(poMessage);
         // Create the (message related) FI data object
         most_phonbkfi_tclMsgPreferredSortOrderListPureSet oPreferredSortOrderListPureSet;

         // Unmarshal the FI visitor message to the given FI data object.
         // The parameters are copied individually from shared memory to the FI data
         // object.
         if (OSAL_ERROR != oVisitorMsg.s32GetData(oPreferredSortOrderListPureSet, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION))
         {

            FC_PhoneBook_SQLite* poFC_PhoneBook_SQLite = FC_PhoneBook_SQLite::GetSQLiteInstance();
            if (poFC_PhoneBook_SQLite)
            {
               for (tU8 u8Index = 0; u8Index < (tU8) (oPreferredSortOrderListPureSet.oPreferredSortOrderList.oItems.size()); u8Index++)
               {
                  tU8 u8DeviceHandle = oPreferredSortOrderListPureSet.oPreferredSortOrderList.oItems[u8Index].u8DeviceHandle;
                  quint8 u8SortOrder =  oPreferredSortOrderListPureSet.oPreferredSortOrderList.oItems[u8Index].e8PreferredPhoneBookSortOrder.enType;
                  ETG_TRACE_USR2(("bProcessSet::Updating PreferredSortOrderListStatus  u8DeviceHandle = %u", u8DeviceHandle));
                  ETG_TRACE_USR2(("bProcessSet::Updating PreferredSortOrderListStatus  u8SortOrder = %u", u8SortOrder));

                  if (poFC_PhoneBook_SQLite->UpdatePreferredSortOrderForDeviceHandle(u8DeviceHandle, u8SortOrder))
                  {
                     bPropertyChanged = TRUE;
                  }
                  if(u8DeviceHandle == u8GetActivePhoneDeviceHandle())
                  {
                     ETG_TRACE_USR2(("bProcessSet::Updating PreferredPhoneBookSortOrderStatus sort order in database = %u", u8SortOrder));
                     poFC_PhoneBook_SQLite->UpdatePreferredSortOrder(u8SortOrder);
                     vPhoneBook_UpdatePreferredPBSortOrderStatus();
                  }
               }
            }
            else
            {
               ETG_TRACE_ERR(( "PreferredSortOrderListStatus: CCA_C_U16_ERROR_TEMPORARY_NOT_AVAILABLE" ));
               u16ErrorCode = CCA_C_U16_ERROR_TEMPORARY_NOT_AVAILABLE;
            }

            if(bPropertyChanged == TRUE)
            {
               vPhoneBook_UpdatePreferredSortOrderListStatus();
            }
         }
         else
         {
            ETG_TRACE_ERR(( "PreferredSortOrderListStatus: CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED" ));
            u16ErrorCode = CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
         }

         bSuccess = TRUE;
         // Always destroy the (message related) FI data object (before leaving
         // its creation scope)
         oPreferredSortOrderListPureSet.vDestroy();
      }
      break;
      //CRQ-243550-Vertical Keyboard Language Selection
      case MOST_PHONBKFI_C_U16_PHONEBOOKLANGUAGE:
      {
         ETG_TRACE_USR4(("bProcessSet MOST_MSGFI_C_U16_PHONEBOOKLANGUAGE"));

         fi_tclVisitorMessage oVisitorMsg(poMessage);
        //NCG3D-44771- Switch Alphabet option not working in Phonebook screen
         most_phonbkfi_tclMsgPhonebookLanguageSet oPhonebookLanguagePureSet;

         if (OSAL_ERROR != oVisitorMsg.s32GetData(oPhonebookLanguagePureSet, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) )
         {
            // Before we set the new language we get the old one for comparison and related handling.
            tU8 u8Language_old = (most_fi_tcl_e8_PbkLanguage::tenType)GetPhonebookLanguage();   // Needed for: Hotfix for CRQ 509734 Phonetic Sorting
            
            ETG_TRACE_USR4(("PHBK_YOMI: bProcessSet Setting PHONEBOOKLANGUAGE to = %u ", oPhonebookLanguagePureSet.e8Language.enType));
            vSetPhonebookLanguage(oPhonebookLanguagePureSet.e8Language.enType) ;
            m_oPhonebookLanguage_Status.e8Language.enType = (most_fi_tcl_e8_PbkLanguage::tenType)GetPhonebookLanguage();
            bPropertyChanged = TRUE;
            bSuccess = TRUE;
            
            // Hotfix for CRQ 509734 Phonetic Sorting - BGN
            FC_PhoneBook_SQLite* poFC_PhoneBook_SQLite = FC_PhoneBook_SQLite::GetSQLiteInstance();
            if (poFC_PhoneBook_SQLite != NULLPTR)
            {
               bool bSetSortingWithPhonetics = false;
               bool bRefreshOfContactListsNecessary = false;

               // Get new phonebook language
               tU8 u8Language_new = (most_fi_tcl_e8_PbkLanguage::tenType)GetPhonebookLanguage();
               
               // Check if language has changed from any (non Japanese) language to Japanese
               if( (u8Language_old != most_fi_tcl_e8_PbkLanguage::FI_EN_JAPANESE) && (u8Language_new == most_fi_tcl_e8_PbkLanguage::FI_EN_JAPANESE) )   
               {
                  bSetSortingWithPhonetics = true;
                  
                  // Phonebook Contact Lists need to be recreated to get the phonetic sorting active.
                  bRefreshOfContactListsNecessary = true;
               }
               // Check if language has changed from Japanese to any (non Japanese language)
               else if( (u8Language_old == most_fi_tcl_e8_PbkLanguage::FI_EN_JAPANESE) && (u8Language_new != most_fi_tcl_e8_PbkLanguage::FI_EN_JAPANESE) ) 
               {  
                  bSetSortingWithPhonetics = false;
                  // Phonebook Contact Lists need to be recreated to deactivate the phonetic sorting.
                  bRefreshOfContactListsNecessary = true;
               }
               else 
               {
				   // When we come here we do not need a refresh of contact lists, but we will set the variable 'bSetSortingWithPhonetics'
				   if(most_fi_tcl_e8_PbkLanguage::FI_EN_JAPANESE == u8Language_new)
				   {
						bSetSortingWithPhonetics = true;
				   }
               }

               ETG_TRACE_USR2(("bProcessSet::PhoneBookLanguage: Set 'bSetSortingWithPhonetics'= %d in 'FC_PhoneBook_SQLite' module. ", bSetSortingWithPhonetics));

               // Updating member variable 'm_bSortingWithPhonetics' in class 'FC_PhoneBook_SQLite' module.
               poFC_PhoneBook_SQLite->vSetSortingWithPhonetics(bSetSortingWithPhonetics);
               
               // RefreshContactLists if necessary
               if(bRefreshOfContactListsNecessary)
               { 
                  // TODO: Add code for refreshing the contact lists.
               }
            }
            else
            {
               ETG_TRACE_FATAL (( " PHBK_NEVER_EXPECTED: poFC_PhoneBook_SQLite == NULLPTR " ));
               ETG_TRACE_ERRMEM(( " PHBK_NEVER_EXPECTED: poFC_PhoneBook_SQLite == NULLPTR " ));
            }
            // Hotfix for CRQ 509734 Phonetic Sorting - END
            
         }
         else
         {
            ETG_TRACE_ERR(( "PhoneBookLanguageStatus: CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED" ));
            u16ErrorCode = CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
         }
         oPhonebookLanguagePureSet.vDestroy();
		 //end of fix NCG3D-44771
      }
      break;
      //end of fix
      default:
      { // Added to solve lint warnings.
         ETG_TRACE_USR4(("bProcessSet::Following FID not found: '%u'", u16FunctionId));
      }
      break;
   }

   return bSuccess;
}

/*******************************************************************************
*
* FUNCTION: FC_Device_Details::GetPhonebookLanguage()
*
* DESCRIPTION: Returns the current set phonebook language.
*
* PARAMETER: None
*
* RETURNVALUE: Phonebook Language.
*
*******************************************************************************/
tU8 fc_phonebook_tclService_PhoneBook::GetPhonebookLanguage()
{
   ETG_TRACE_USR4(( " GetPhonebookLanguage= %d", this->m_u8PbkLang ));
   return this->m_u8PbkLang;
}

/*******************************************************************************
*
* FUNCTION: FC_Device_Details::bSetPhonebookLanguage()
*
* DESCRIPTION: Sets the vertical keyboard Language.
*
* PARAMETER: Phonebook Language
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid  fc_phonebook_tclService_PhoneBook::vSetPhonebookLanguage(tU8 f_PbkLang)
{
   this->m_u8PbkLang = f_PbkLang;
   ETG_TRACE_USR4(( " PhonebookLanguage set to m_u8PbkLang= %d ", this->m_u8PbkLang ));
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_phonebook_tclService_PhoneBook::\
vOnLoopbackService(amt_tclServiceData* poMessage)
*
* DESCRIPTION: This function is called from the vOnLoopback() function of this
*              CCA application if a message from one of this applications
*              services is received (so called self or loopback messages).
*
* PARAMETER: [IN] u16ServiceID = Identifier of the service from where the
*                                message was sent.
*            [IN] poMessage = Message object.
*
* RETURNVALUE: -
*
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vOnLoopbackService( \
                                                            amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR4(("PhoneBook vOnLoopbackService entered"));

   if(this->m_bServerAvailable == FALSE)
   {
      ETG_TRACE_USR4(("PHONE BOOK IS NOT AVAILABLE"));
      return ;
   }
   gm_tclU32Message oVisitorMsg(poMessage);
   //HOTFIX 047
   tU16 u16MsgPosition =0;
   PhBk_Loopback_Data LB_Data = PhBk_Loopback_Data();

   tU16 u16FunctionId = poMessage->u16GetFunctionID();

   tU16 u16MsgContent = (tU16)(oVisitorMsg.u32GetDWord());

   ETG_TRACE_USR2(("PhoneBook u16FunctionId:: 0x%x ",u16FunctionId));
   ETG_TRACE_USR2(("u16MsgContent:: %d",u16MsgContent));
   ETG_TRACE_USR2(("PhoneBook LB_data_queue  size:: %d",(int)LB_data_queue.size()));

   if(PHONEBOOK_C_U16_IFID_DBUS == u16FunctionId)
   {
      u16MsgPosition = u16Get_Data_Position(u16MsgContent);
      if(PHONEBOOK_LOOPBACK_VECTOR_INVALID_POSITION == u16MsgPosition)
      {

         ETG_TRACE_ERR(("ERROR :: CALLBACK FUNCTION_ID DATA IS NOT PRESENT IN VECTOR QUEUE  "));
         return;
      }

      LB_Data= LB_data_queue[u16MsgPosition];
      vErase_LB_Data(u16MsgPosition);
   }

   switch(u16FunctionId)
   {
   case PHONEBOOK_C_U16_IFID_DBUS :
      {
         switch(u16MsgContent)
         {
         case FC_PB_LB_FID_CONNECT:
            {
               vDeviceConnected(LB_Data.ucDeviceHandle, LB_Data.ucDevAddr);
            }
            break;
         case FC_PB_LB_FID_DISCONNECT:
            {
               QMutexLocker lock(&g_Phonebook_DeviceMutex);
               FC_Device_Details
                     * pDeviceObject =
                           FC_Device_Details::deviceHandleToObjectInstance(LB_Data.ucDeviceHandle);
               if (pDeviceObject)
               {
                  vDeviceDisconnected(pDeviceObject->getConnectionID());
               }

               vMakeFavoriteUnAvaialble(LB_Data.ucDeviceHandle);
               m_pPhoneBookDB->ChangefavoriteActiveStatus(0);
            }
            break;
         case FC_PB_LB_FID_DEVICE_DELETED: //RB FIX GMMY15-248
            {
               if (CHECK_VALID(LB_Data.ucDeviceHandle))
               {
                  QMutexLocker lock(&g_Phonebook_DeviceMutex);
                  FC_Device_Details* pDeviceObject = FC_Device_Details::deviceHandleToObjectInstance(LB_Data.ucDeviceHandle);
                  if (pDeviceObject)
                  {
                     tU8 ucConnectionID = pDeviceObject->getConnectionID();
                     vDeviceDisconnected(ucConnectionID);
                     pDeviceObject->vSetMissedcallCount(0);
                     vUpdatePhonebookMissedCallIndicatorList(LB_Data.ucDeviceHandle);
                  }
                  else
                  {
                     ETG_TRACE_USR1(("NULL POINTER : pDeviceObject"));
                  }
                  vDeviceDeleted(LB_Data.ucDeviceHandle);
                  //CMG3G-14241 - Updating the preferred sort order after device deletion.
                  vPhoneBook_UpdatePreferredSortOrderListStatus();
                  m_poFavoriteProvider->RefreshFavoritesOnDeviceDeletion(LB_Data.ucDeviceHandle);
               }
            }
            break;
         case FC_PB_LB_FID_GET_CAPABILITIES_CB:
            {
               m_oPbk_DnlManager.vGetCapabilitiesCallback(&LB_Data);
            }
            break;

         case FC_PB_LB_FID_DNL_PREPARE_CB:
            {
               m_oPbk_DnlManager.vPrepareCallback(&LB_Data);
            }
            break;

         case FC_PB_LB_FID_DNL_START_CB:
            {
               m_oPbk_DnlManager.vDnlStartCallback(&LB_Data);
            }
            break;

         case FC_PB_LB_FID_DNL_COMPLETE_CB:
            {
               m_oPbk_DnlManager.vDnlCompleteCallback(&LB_Data);
            }
            break;

         case FC_PB_LB_FID_DATA_COMPLETE_SIG:
            {
               m_oPbk_DnlManager.vDataCompleteCallback(&LB_Data);
            }
            break;

         case FC_PB_LB_FID_CONNECTION_LOST_SIG:
            {
               m_oPbk_DnlManager.vConnectionLost(&LB_Data);
               FC_Device_Details* pDeviceObject = FC_Device_Details::deviceHandleToObjectInstance(LB_Data.ucDeviceHandle);
               if (pDeviceObject)
               {
                  vClearMissedCalls(LB_Data.ucDeviceHandle, pDeviceObject);//244511- Chery Bug Fix - Clearing the Missed call list upon device disconnection
                  pDeviceObject->vSetMissedcallCount(0);
                  vUpdatePhonebookMissedCallIndicatorList(LB_Data.ucDeviceHandle);
               }
            }
            break;

         case FC_PB_LB_FID_DNL_STOP_CB:
            {
               vDnlStopCallback(&LB_Data);
            }
            break;

         case FC_PB_LB_FID_INCOMING_CALL_SIG:
            {
               m_oPbk_DnlManager.vIncomingCall(&LB_Data);
            }
            break;

         case FC_PB_LB_FID_CALL_STATUS_SIG:
            {
               m_oPbk_DnlManager.vCallStatus(&LB_Data);
            }
            break;

         case FC_PB_LB_FID_EVO_CONNECT:
            {
               QMutexLocker lock(&g_Phonebook_DeviceMutex);
               FC_Device_Details
                     * pDeviceObject =
                           FC_Device_Details::deviceHandleToObjectInstance(LB_Data.ucDeviceHandle);
               if (pDeviceObject && m_pDBus_CCA_If_pb)
               {
                  m_pDBus_CCA_If_pb->vPhoneBook_onDeviceConnect(pDeviceObject);
               }
            }
            break;

            case FC_PB_LB_FID_EVO_DISCONNECT:
            {
               QMutexLocker lock(&g_Phonebook_DeviceMutex);
               FC_Device_Details
                     * pDeviceObject =
                           FC_Device_Details::deviceHandleToObjectInstance(LB_Data.ucDeviceHandle);
               if (pDeviceObject && m_pDBus_CCA_If_pb)
               {
                  m_pDBus_CCA_If_pb->vPhoneBook_onDeviceDisconnect(pDeviceObject->getConnectionID());
               }
            }
            break;

            case FC_PB_LB_FID_RESUME_SUSPENDED_DWNLD:
            {
               QMutexLocker lock(&g_Phonebook_DeviceMutex);
               FC_Device_Details
                     * pDeviceObject =
                           FC_Device_Details::deviceHandleToObjectInstance(LB_Data.ucDeviceHandle);
               if (pDeviceObject && m_pDBus_CCA_If_pb)
               {
                  pDeviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS);
                  ETG_TRACE_USR4(("Download State changed - DOWNLOAD_ST_PBK_PROGRESS"));
                  m_pDBus_CCA_If_pb->vPhoneBook_StartDlPB(pDeviceObject, pDeviceObject->PBDLCommand);
               }
            }
            break;

         default:
            {
               ETG_TRACE_ERR(("vOnLoopbackService(). Unknown u16MsgContent %u.",u16MsgContent));
            }
            break;
         }
      }
      break;
   case PHONEBOOK_C_U16_IFID_TIMER:
      m_oPbk_DnlManager.PhoneBook_vTimerExpired(u16MsgContent);
      break;
   default:
      {
         ETG_TRACE_ERR(("vOnLoopbackService(). Unknown iFID %u.", u16FunctionId));
      }
      break;
   }


   ETG_TRACE_USR2(("vOnLoopbackService() exit. iFID = 0x%4x", u16FunctionId));
   ETG_TRACE_USR2(("u16MsgContent:: %u",u16MsgContent));
}
//FIX CMG3G-8367 IS2424_DownloadOnOff behaviour@FC_Phonebook
// The user shall have possibility to switch the phonebook and call history list download On or Off individual for each paired phone device.
/*******************************************************************************
 *
 * FUNCTION: tVoid fc_phonebook_tclService_PhoneBook::vDnlStopCallback(PhBk_Loopback_Data* poLB_Data)
 *
 * DESCRIPTION: This function is called On Download abort callback.
 *
 * PARAMETER: Loop back data
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vDnlStopCallback(PhBk_Loopback_Data* poLB_Data)
{
   ETG_TRACE_USR4(("vDnlStopCallback called"));
   ETG_TRACE_COMP((" -PBDL.S- PbDlStop has returned: bSuccess= %d ", (tU8) !poLB_Data->blFailed));
   tBool bDnlStopResult = FALSE;
   tU8 u8DeviceHandle = VALUE_ZERO;
   tU8 u8ConnectionID = VALUE_ZERO;

   FC_Device_Details* pDeviceObject = FC_Device_Details::deviceAddressToObjectInstance(poLB_Data->ucDevAddr);
   if (pDeviceObject)
   {
      u8DeviceHandle = pDeviceObject->getDeviceHandle();
      u8ConnectionID = pDeviceObject->getConnectionID();
      if (FALSE == poLB_Data->blFailed)
      {
         if (m_poDownloadState)
         {
            if (most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_INFO
                  == m_poDownloadState->u8GetPBDownloadState()
                  || most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_PHOTO
                        == m_poDownloadState->u8GetPBDownloadState())
            {
               ETG_TRACE_USR4(("Setting DownloadState as FI_EN_E8PBDS_COMPLETE/FI_EN_E8PBDS_ERROR - Pbdl Abort"));

               m_poDownloadState->vSetPBDownloadState(most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended::FI_EN_E8PBDS_ERROR);

               vPhoneBook_UpdateDownloadState();
            }
            if (most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_IN_PROCESS
                  == m_poDownloadState->u8GetRecentCallListDownloadState())
            {
               ETG_TRACE_USR4(("Setting DownloadState as FI_EN_E8RCDS_COMPLETE/FI_EN_E8RCDS_ERROR - Pbdl Abort"));

               m_poDownloadState->vSetRecentCallListDownloadState(most_fi_tcl_e8_PhonBkRecentCallListDownloadStateExtended::FI_EN_E8RCDS_ERROR);

               vPhoneBook_UpdateDownloadState();
            }
         }
         else
         {
            ETG_TRACE_ERR((" m_poDownloadState is NULL!!"));
         }

         //FIX CMG3G-8570 IS2502_Cancel PB Download@FC_Phonebook
         if (TRUE == pDeviceObject->bIsCancelPBRequested())
         {
            //Setting internal download state as IDLE on cancellation of ongoing manual download so that subsequent manual downloads and call history downloads can be triggered.
            pDeviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_IDLE);
            ETG_TRACE_USR4(( "Download State changed - DOWNLOAD_ST_IDLE" ));

            //Handle DB on download abort
            FC_PhoneBook_List* poFC_PhoneBook_List =
                  FC_PhoneBook_List::pGetInstance();
            if (poFC_PhoneBook_List != NULL)
            {
               poFC_PhoneBook_List->DownloadAborted(pDeviceObject->getDeviceHandle());
            }

            if (FALSE
                  == m_oPbk_DnlManager.bIsCHDownloadComplete(pDeviceObject))
            {
               //To-Do - Initiate only the required call history types instead of requesting CombinedCallHistory
               ETG_TRACE_USR4(("Re-Initiate call history download on cancel pb download"));
               if (m_pDBus_CCA_If_pb)
               {
                  m_pDBus_CCA_If_pb->PhoneBook_InitiateDownload(pDeviceObject, CombinedCallHistory);
               }
            }

            pDeviceObject->setCancelPBDownload(FALSE);
         }

         FC_PhoneBook_SQLite* poFC_PhoneBook_SQLite = FC_PhoneBook_SQLite::GetSQLiteInstance();
         if (PB_DL_IN_TRANSITION_TO_OFF == poFC_PhoneBook_SQLite->u16GetDownloadOnOffStateForDevicehandle(u8DeviceHandle))
         {
            bDnlStopResult = m_pPhoneBookDB->getCallHistoryListInstance()->bClearCHRecords(u8ConnectionID);
            bDnlStopResult = bDnlStopResult && m_pPhoneBookDB->getCurrentListInstance()->bClearPBRecords(u8DeviceHandle);
            if (TRUE == bDnlStopResult)
            {
               FC_PhoneBook_SQLite::GetSQLiteInstance()->UpdateSwitchDownloadOnOffState(u8DeviceHandle, PB_DL_OFF);
               vPhoneBook_UpdateClientsDownloadOnOffList();
               m_oMisssedCallIndicatorStatus.u32NumMissedCalls = VALUE_ZERO;
               vPhoneBook_UpdateClientsAboutMissedCallCount(u8DeviceHandle);
               pDeviceObject->vSetMissedcallCount(0);
               vUpdatePhonebookMissedCallIndicatorList(u8DeviceHandle);

            }
            else
            {
               ETG_TRACE_USR2(("Download abort failed due to delete phonebook data"));
            }
         }
      }
      else
      {
         if ((m_oPbk_DnlManager.PhoneBook_blIsRetryActive(u8ConnectionID, FC_PhoneBook_RetryDetails::RETRY_ANY_MESSAGE))
               && (!m_oPbk_DnlManager.PhoneBook_blIsRetryActive(u8ConnectionID, FC_PhoneBook_RetryDetails::RETRY_STOP_PBDL)))
         {
            ETG_TRACE_USR4(("Retrying already active for another message"));
         }
         else
         {
            ETG_TRACE_USR2(("Download abort callback failed"));
            if (m_oPbk_DnlManager.u8GetRetryCount() < NO_OF_RETRIES)
            {
               ETG_TRACE_USR2((" Retrying for Stop/cancel download... "));
               m_oPbk_DnlManager.PhoneBook_vStartRetry(FC_PhoneBook_RetryDetails::RETRY_STOP_PBDL, u8ConnectionID, FC_PhoneBook_RetryDetails::RETRY_TYPE_INCREMENTAL);
            }
            else
            {
               ETG_TRACE_USR2((" Download abort failed on max retries hence stopping retry "));

               m_oPbk_DnlManager.PhoneBook_vStopRetry(FC_PhoneBook_RetryDetails::RETRY_STOP_PBDL, u8ConnectionID);

               //Update download state as ERROR.
               if (m_poDownloadState)
               {
                  if (most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_INFO
                        == m_poDownloadState->u8GetPBDownloadState()
                        || most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_PHOTO
                              == m_poDownloadState->u8GetPBDownloadState())
                  {
                     ETG_TRACE_USR4(("Setting DownloadState as FI_EN_E8PBDS_COMPLETE/FI_EN_E8PBDS_ERROR - Pbdl Abort failed"));

                     m_poDownloadState->vSetPBDownloadState(most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended::FI_EN_E8PBDS_ERROR);

                     vPhoneBook_UpdateDownloadState();
                  }
                  else if (most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_IN_PROCESS
                        == m_poDownloadState->u8GetRecentCallListDownloadState())
                  {
                     ETG_TRACE_USR4(("Setting DownloadState as FI_EN_E8RCDS_COMPLETE/FI_EN_E8RCDS_ERROR - Pbdl Abort failed"));

                     m_poDownloadState->vSetRecentCallListDownloadState(most_fi_tcl_e8_PhonBkRecentCallListDownloadStateExtended::FI_EN_E8RCDS_ERROR);

                     vPhoneBook_UpdateDownloadState();
                  }
               }
               else
               {
                  ETG_TRACE_ERR((" m_poDownloadState is NULL!!"));
               }
            }
         }
      }
   }
   else
   {
      ETG_TRACE_ERR((" Device details NULL!!"));
   }
}
//End of fix FIX CMG3G-8367
/*******************************************************************************
*
* FUNCTION: tVoid fc_phonebook_tclService_PhoneBook::
vProcessTimer(tU16 u16TimerId)
*
* DESCRIPTION: This function is called from the vOnTimer() function of this
*              CCA application on the expiration of a previously via function
*              bStartTimer() started timer.
*
* PARAMETER: [IN] u16TimerId = Identifier of the expired timer.
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vProcessTimer(tU16 u16TimerId)
{
   ETG_TRACE_USR2(("vProcessTimer() entered. TimerID = %u.",u16TimerId));
}

/* Internal methods */
/*******************************************************************************
*
* FUNCTION: ail_tenCommunicationError vStartTimer
*
* DESCRIPTION:Just a wrapper function to start timer of the application
*
* PARAMETER: [IN] nTimerID
*            [IN] nTimeUntilFirstTimerTick
*            [IN] nTimerTickInterval
*
* RETURNVALUE: None.
*
*******************************************************************************/
tBool fc_phonebook_tclService_PhoneBook::bStartTimer( \
                                                     tU16 nTimerID,\
                                                     OSAL_tMSecond nTimeUntilFirstTimerTick,\
                                                     OSAL_tMSecond nTimerTickInterval)
{
   ETG_TRACE_USR4(("bStartTimer entered"));
   (tVoid)nTimerID;
   (tVoid)nTimeUntilFirstTimerTick;
   (tVoid)nTimerTickInterval;

   tBool bTrueOrFalse = TRUE;

   bTrueOrFalse = _poMainAppl->bStartTimer(nTimerID,
      nTimeUntilFirstTimerTick,
      nTimerTickInterval);

   if (FALSE == bTrueOrFalse)
   {
      //     FATAL_M_ASSERT_ALWAYS();

      ETG_TRACE_ERR(("Start of Timer id - %d, failed", nTimerID));
   }

   return bTrueOrFalse;
}


/*******************************************************************************
*
* FUNCTION: ail_tenCommunicationError vStopTimer
*
* DESCRIPTION:Just a wrapper function to stop timer of the application
*
* PARAMETER: [IN] nTimerID

*
* RETURNVALUE: None.
*
*******************************************************************************/
tBool fc_phonebook_tclService_PhoneBook::bStopTimer(tU16 nTimerID)
{
   ETG_TRACE_USR4(("bStopTimer entered"));
   (tVoid)nTimerID;

   tBool bTrueOrFalse = TRUE;

   bTrueOrFalse = _poMainAppl->bStopTimer(nTimerID);

   if(FALSE == bTrueOrFalse)
   {
      //       FATAL_M_ASSERT_ALWAYS();
      //       return FALSE;
   }

   return bTrueOrFalse;
}


/*******************************************************************************
** FUNCTION:   vHandle_CreatePhoneBookList_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for CreatePhoneBookList method
*   
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::\
vHandle_CreatePhoneBookList_MethodStart(\
                                        amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_CreatePhoneBookList_MethodStart()  entered."));

   tBool result = false;
   /* Extract the Call instance and other required info from poMessage */
   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );

   // Create the (message related) FI data object
   most_phonbkfi_tclMsgCreatePhoneBookListMethodStart \
      oCreatePhoneBookListMethodStart_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.
   if ( oVisitorMsg.s32GetData(oCreatePhoneBookListMethodStart_MS, \
      MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION)
      != OSAL_ERROR)
   {
      ETG_TRACE_USR2(("DEVICE HANDLE: %u. ",oCreatePhoneBookListMethodStart_MS.u8DeviceHandle));
      ETG_TRACE_USR1((" Phone Book Sort Type:: %u",ETG_ENUM(TR_PBKSORTTYPE,oCreatePhoneBookListMethodStart_MS.e8PhoneBookSortType.enType)));

      FC_Device_Details *p_checkObject;
      p_checkObject =  FC_Device_Details::deviceHandleToObjectInstance(oCreatePhoneBookListMethodStart_MS.u8DeviceHandle);
      if( oCreatePhoneBookListMethodStart_MS.u8DeviceHandle == FC_PB_VEHICLE_PBOOK)
      {
         ETG_TRACE_USR4((" Contact List:Vehicle Address Book   "));
         result = true;
      }
      else
      {
         ETG_TRACE_USR4(("Contact List : Check device object  "));

         if( p_checkObject != NULL)
         {
            ETG_TRACE_USR2(("Contact List : Device object exists for device handle = %u ", oCreatePhoneBookListMethodStart_MS.u8DeviceHandle));
            result = true;
         }
         else
         {

            ETG_TRACE_ERR((" Contact List : Device object do not exists for device handle = %u ", oCreatePhoneBookListMethodStart_MS.u8DeviceHandle));
         }
      }

      if(result == true)
      {

         most_phonbkfi_tclMsgCreatePhoneBookListMethodResult \
            oCreatePhoneBookList_MR;

         tBool dbStatus = FALSE;

         if (m_pPhoneBookDB)
         {
            dbStatus
                  = m_pPhoneBookDB->CreatePhoneBookList(&oCreatePhoneBookListMethodStart_MS, &oCreatePhoneBookList_MR, poMessage->u16GetRegisterID());//FIX GMMY 15-1578 RegisterID is passed to identify the requested client
            //Fix : CMG3G-12880
            if((p_checkObject) && (TRUE == p_checkObject->bIsDeviceProbablyLocked()))
            {
               oCreatePhoneBookList_MR.u16ListLength = 0;
            }
            //End Fix : CMG3G-12880
         }

         // Get the coresponding Call Instance

         if(dbStatus != TRUE)
         {
            //DB Returned error
            amt_tclServiceDataError oErrorMsg(*poMessage, most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC);

            // Post message
            ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

            if (enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_ERR(("vHandle_CreatePhoneBookList_MethodStart: 'ERROR', enPostMessage() failed"));
            }
         }
         else
         {
            fi_tclVisitorMessage oMRMsg(oCreatePhoneBookList_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
            // Here the message related header information is added and by this the
            // creation of the message is completed.
            vInitServiceData( oMRMsg,
               poMessage->u16GetSourceAppID(), // ServiceDataMsg
               poMessage->u16GetRegisterID(),  // RegisterID
               poMessage->u16GetCmdCounter(),  // CmdCounter
               poMessage->u16GetFunctionID(),  // Function ID
               AMT_C_U8_CCAMSG_OPCODE_METHODRESULT       // Opcode
               );

            if (oMRMsg.bIsValid())
            {
               // Post message
               ail_tenCommunicationError enResult =  _poMainAppl->enPostMessage(&oMRMsg, TRUE);

               if (enResult != AIL_EN_N_NO_ERROR)
               {
                  ETG_TRACE_ERR(("vHandle_CreatePhoneBookList_MethodStart: MethodResult', enPostMessage() failed"));
                  // We can come here e.g. if the client has changed his application
                  // state to OFF in the meantime. Therefore we dont throw an assert.
                  // NORMAL_M_ASSERT_ALWAYS();
               }
               else
               {   // Speech Support
                  ETG_TRACE_USR4(("Actions to be performed after posting create list result"));

                  if (m_pPhoneBookDB)
                  {
                     m_pPhoneBookDB->MethodResultPosted(poMessage->u16GetFunctionID(), &oCreatePhoneBookListMethodStart_MS, &oCreatePhoneBookList_MR);
                  }
               }
            }else
            {
               most_phonbkfi_tclMsgCreatePhoneBookListError createPBError;
               vHandleMethodError(createPBError,poMessage);
               createPBError.vDestroy();
            }
            oCreatePhoneBookList_MR.vDestroy();
         }
      }
      else
      {
         amt_tclServiceDataError oErrorMsg(*poMessage, CCA_C_U16_ERROR_PARAMETER_OUT_OF_RANGE);
         // Post message
         ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

         if (enResult != AIL_EN_N_NO_ERROR)
         {
            ETG_TRACE_ERR(("vHandle_CreatePhoneBookList_MethodStart: 'ERROR', enPostMessage() failed"));
         }
         else
         {
            ETG_TRACE_USR4(("vHandle_CreatePhoneBookList_MethodStart: 'SUCCESS', enPostMessage()"));
         }
      }
   }
   else
   {
      vHandleGetDataError(poMessage,"vHandle_CreatePhoneBookList_MethodStart");
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)
   oCreatePhoneBookListMethodStart_MS.vDestroy();

}



/*******************************************************************************
** FUNCTION:   vHandle_CreateContactList_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for CreateContactListkList method
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/

tVoid fc_phonebook_tclService_PhoneBook::\
vHandle_CreateContactList_MethodStart(\
                                      amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_CreateContactList_MethodStart()  entered."));

   tBool result = false;
   /* Extract the Call instance and other required info from poMessage */
   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );

   // Create the (message related) FI data object
   most_phonbkfi_tclMsgCreateContactListMethodStart \
      oCreateContactListMethodStart_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.
   if ( oVisitorMsg.s32GetData(oCreateContactListMethodStart_MS, \
      MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION)
      != OSAL_ERROR)
   {
      ETG_TRACE_USR2(("DEVICE HANDLE: %u. ",oCreateContactListMethodStart_MS.u8DeviceHandle));
      ETG_TRACE_USR1((" Phone Book ContactList Type:: %u",ETG_ENUM(TR_PBKContactListTYPE,oCreateContactListMethodStart_MS.e8PhoneBkContactListType.enType)));
      ETG_TRACE_USR1((" Phone Book ContactList Sort Type:: %u",ETG_ENUM(TR_PBKContacListSORTTYPE,oCreateContactListMethodStart_MS.e8PhoneBkContactSortType.enType)));

      FC_Device_Details *p_checkObject =  FC_Device_Details::deviceHandleToObjectInstance(oCreateContactListMethodStart_MS.u8DeviceHandle);
      if( p_checkObject != NULL)
      {
         ETG_TRACE_USR2(("Contact List : Device object exists for device handle = %u ", oCreateContactListMethodStart_MS.u8DeviceHandle));
         result = true;
      }
      else if( ( (VALUE_TWO <= oCreateContactListMethodStart_MS.u8DeviceHandle) && 
	             (FC_PB_TOTAL_NUMBER_OF_PB >= oCreateContactListMethodStart_MS.u8DeviceHandle ) && 
				 (VALUE_ZERO != oCreateContactListMethodStart_MS.e8PhoneBkContactListType.enType) ) ||
               ( (VALUE_ONE == oCreateContactListMethodStart_MS.u8DeviceHandle ) && (VALUE_ONE == oCreateContactListMethodStart_MS.e8PhoneBkContactListType.enType) ) )
      {  //276745-This condition satisfies combination of (DEVICEHANDLE =1 (i.eVPBtable) and LISTTYPE_FAVORITE) or a combination of (DEVICEHANDLE= 2-11 (DH2-DH11) and LISTYPE_VEHICEL (VPB2-VPB11) )
         //Fix NCG3D-68310
         //Irrespective of device object, create contact list can be called and list change updates can be send
         ETG_TRACE_USR2(("Device object does not exist but access to Vehicle(Quick Dial) is allowed"));
         result = true;
         //End fix NCG3D-68310
      }
      else
      {
         ETG_TRACE_ERR((" Contact List : Device object do not exists for device handle = %u ", oCreateContactListMethodStart_MS.u8DeviceHandle));
      }

      if(result == true)
      {
         most_phonbkfi_tclMsgCreateContactListMethodResult \
            oCreateContactList_MR;

         tBool dbStatus = FALSE;
         if (m_pPhoneBookDB)
         {
            dbStatus
                  = m_pPhoneBookDB->CreateContactList(&oCreateContactListMethodStart_MS, &oCreateContactList_MR, poMessage->u16GetRegisterID());

            //Fix : NCG3D-159329 (overrules older ticket CMG3G-12880)
            // User shall be able to access the Quick-Dial Lists in the situation that the phone is "probably" locked.
            if (p_checkObject != NULL)
            { 
               if( (VALUE_TWO != oCreateContactListMethodStart_MS.e8PhoneBkContactListType.enType) && (TRUE == p_checkObject->bIsDeviceProbablyLocked()) )
               {
               // The requested list is NOT a vehicle list, i.e. it is a favorite or contact list AND the phone is probably locked.
               // Therfore we report an empty list (ListLength = 0), so that HMI cannot access the list.
               // (Note: For a vehicle list we would allow the access [as long the related phone device is at least HFP connected (p_checkObject != NULL)].)
               oCreateContactList_MR.u16ListLength = 0;
               }
            }
            //End Fix : NCG3D-159329 (overrules older ticket CMG3G-12880)
         }
         if(dbStatus != TRUE)
         {
            //DB Returned error
            amt_tclServiceDataError oErrorMsg(*poMessage, most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC);

            // Post message
            ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

            if (enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_ERR((" vHandle_CreateContactList_MethodStart: 'ERROR', enPostMessage() failed"));
            }
         }
         else
         {
            // Get the coresponding Call Instance
            fi_tclVisitorMessage oMRMsg(oCreateContactList_MR,\
               MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
            // Here the message related header information is added and by this the
            // creation of the message is completed.
            vInitServiceData( oMRMsg,
               poMessage->u16GetSourceAppID(), // ServiceDataMsg
               poMessage->u16GetRegisterID(),  // RegisterID
               poMessage->u16GetCmdCounter(),  // CmdCounter
               poMessage->u16GetFunctionID(),  // Function ID
               AMT_C_U8_CCAMSG_OPCODE_METHODRESULT       // Opcode
               );

            if (oMRMsg.bIsValid())
            {
               // Post message
               ail_tenCommunicationError enResult =  _poMainAppl->enPostMessage(&oMRMsg, TRUE);

               if (enResult != AIL_EN_N_NO_ERROR)
               {
                  ETG_TRACE_ERR(("vHandle_CreateContactList_MethodStart: MethodResult', enPostMessage() failed"));
                  // We can come here e.g. if the client has changed his application
                  // state to OFF in the meantime. Therefore we dont throw an assert.
                  // NORMAL_M_ASSERT_ALWAYS();
               }
               else
               {   // Speech Support
                  ETG_TRACE_USR4(("Actions to be performed after posting create list result"));
                  if (m_pPhoneBookDB)
                  {
                     m_pPhoneBookDB->MethodResultPosted(poMessage->u16GetFunctionID(), &oCreateContactListMethodStart_MS, &oCreateContactList_MR);
                  }
               }
            }else
            {
               most_phonbkfi_tclMsgCreateContactListError createPBError;
               vHandleMethodError(createPBError,poMessage);
               createPBError.vDestroy();
            }
            oCreateContactList_MR.vDestroy();
         }
      }
      else
      {
         amt_tclServiceDataError oErrorMsg(*poMessage, CCA_C_U16_ERROR_PARAMETER_OUT_OF_RANGE);
         // Post message
         ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

         if (enResult != AIL_EN_N_NO_ERROR)
         {
            ETG_TRACE_ERR((" vHandle_CreateContactList_MethodError: Posting of ERROR failed"));
         }
         else
         {
            ETG_TRACE_USR4((" vHandle_CreateContactList_MethodError posted. "));
         }
      }
   }
   else
   {
      vHandleGetDataError(poMessage,"vHandle_CreateContactList_MethodStart");
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)
   oCreateContactListMethodStart_MS.vDestroy();

}


/*******************************************************************************
** FUNCTION:   vHandle_RequestSlicePhoneBookList_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for RequestSlicePhoneBookList method
*   
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::\
vHandle_RequestSlicePhoneBookList_MethodStart(\
                                              amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR4(("vHandle_RequestSlicePhoneBookList_MethodStart(). entered"));

   // Extract the Call instance and other required info from poMessage
   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );

   // Create the (message related) FI data object
   most_phonbkfi_tclMsgRequestPhoneBookListSliceMethodStart \
      oRequestSlicePhoneBookListMethodStart_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.
   if ( oVisitorMsg.s32GetData(oRequestSlicePhoneBookListMethodStart_MS, \
      MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {

      ETG_TRACE_USR2(("LIST HANDLE: %u",oRequestSlicePhoneBookListMethodStart_MS.u16ListHandle));
      ETG_TRACE_USR2(("WINDOW START: %d",oRequestSlicePhoneBookListMethodStart_MS.u32WindowStart));
      ETG_TRACE_USR2(("WINDOW SIZE: %u",oRequestSlicePhoneBookListMethodStart_MS.u16WindowSize));

      most_phonbkfi_tclMsgRequestPhoneBookListSliceMethodResult \
         oRequestSlicePhoneBookList_MR;

      //RequestSlicePhoneBookList call from Sqlite data base//
      tBool dbStatus = FALSE;

      if (m_pPhoneBookDB)
      {
         dbStatus
               = m_pPhoneBookDB->RequestSlicePhoneBookList(&oRequestSlicePhoneBookListMethodStart_MS, &oRequestSlicePhoneBookList_MR);
      }

      if(dbStatus != TRUE)
      {
         //DB Returned error

         amt_tclServiceDataError oErrorMsg(*poMessage, most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC);

         // Post message
         ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

         if (enResult != AIL_EN_N_NO_ERROR)
         {

            ETG_TRACE_ERR(("vHandle_RequestSlicePhoneBookList_MethodStart: 'ERROR', enPostMessage() failed"));
            // NORMAL_M_ASSERT_ALWAYS();
         }

         // NORMAL_M_ASSERT_ALWAYS();

      }
      else
      {
         fi_tclVisitorMessage oMRMsg(oRequestSlicePhoneBookList_MR,  MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
         vInitServiceData( oMRMsg,
            poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(),  // RegisterID
            poMessage->u16GetCmdCounter(),  // CmdCounter
            poMessage->u16GetFunctionID(),  // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT    // Opcode
            );

         if (oMRMsg.bIsValid())
         {

            // Post message
            ail_tenCommunicationError enResult =
               _poMainAppl->enPostMessage(&oMRMsg, TRUE);

            if (enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_ERR(( "vHandle_RequestSlicePhoneBookList_MethodStart:'MethodResult', enPostMessage() failed,'ail_tenCommunicationError' = %u",(tU32)enResult ));
               // We can come here e.g. if the client has changed his application
               // state to OFF in the meantime. Therefore we dont throw an assert.
               // NORMAL_M_ASSERT_ALWAYS();
            }
         }
         else
         {
            most_phonbkfi_tclMsgRequestPhoneBookListSliceError requestSlicePBError;
            vHandleMethodError(requestSlicePBError,poMessage);
            requestSlicePBError.vDestroy();
         }

      }

      oRequestSlicePhoneBookList_MR.vDestroy();

   }
   else
   {
      // If we come here, we assume the client has sent an erroneous message,
      // probably it has used a wrong FI_MAJOR_VERSION. Therefore we will
      // send back a an error message and throw a NORMAL_M_ASSERT_ALWAYS().

      vHandleGetDataError(poMessage,"vHandle_RequestSlicePhoneBookList_MethodStart");
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)

   oRequestSlicePhoneBookListMethodStart_MS.vDestroy();    


}

/*******************************************************************************
** FUNCTION:   vHandle_ReleasePhoneBookList_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for ReleasePhoneBookList method
*   
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::\
vHandle_ReleasePhoneBookList_MethodStart(amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR4(("vHandle_ReleasePhoneBookList_MethodStart(): entered"));

   /* Extract the Call instance and other required info from poMessage */
   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );

   // Create the (message related) FI data object
   most_phonbkfi_tclMsgReleasePhoneBookListMethodStart \
      oReleasePhoneBookListMethodStart_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.

   if ( oVisitorMsg.s32GetData(oReleasePhoneBookListMethodStart_MS, \
      MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION)
      != OSAL_ERROR)
   {

      ETG_TRACE_USR2(("list HANDLE: %u",oReleasePhoneBookListMethodStart_MS.u16ListHandle));

      most_phonbkfi_tclMsgReleasePhoneBookListMethodResult \
         oReleasePhoneBookList_MR;

      //ReleasePhoneBookList call from Sqlite data base//
      tBool dbStatus = FALSE;

      if (m_pPhoneBookDB)
      {
         dbStatus
               = m_pPhoneBookDB->ReleasePhoneBookList(&oReleasePhoneBookListMethodStart_MS, &oReleasePhoneBookList_MR, poMessage->u16GetRegisterID());//FIX GMMY 15-1578 RegisterID is passed to identify the requested client
      }

      // Get the coresponding Call Instance

      if(dbStatus != TRUE)
      {
         most_phonbkfi_tclMsgReleasePhoneBookListError releasePBError;
         vHandleMethodError(releasePBError,poMessage);
         releasePBError.vDestroy();
      }
      else
      {
         fi_tclVisitorMessage oMRMsg(oReleasePhoneBookList_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
         // Here the message related header information is added and by this the
         // creation of the message is completed.
         vInitServiceData( oMRMsg,
            poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(),  // RegisterID
            poMessage->u16GetCmdCounter(),  // CmdCounter
            poMessage->u16GetFunctionID(),  // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT     // Opcode
            );

         if (oMRMsg.bIsValid())
         {

            // Post message
            ail_tenCommunicationError enResult =
               _poMainAppl->enPostMessage(&oMRMsg, TRUE);

            if (enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_ERR(( "vHandle_ReleasePhoneBookList_MethodStart: 'MethodResult', enPostMessage() failed,'ail_tenCommunicationError' = %u",(tU32)enResult ));
               // We can come here e.g. if the client has changed his application
               // state to OFF in the meantime. Therefore we dont throw an assert.
               // NORMAL_M_ASSERT_ALWAYS();
            }

         }
         else
         {
            most_phonbkfi_tclMsgReleasePhoneBookListError releasePBError;
            vHandleMethodError(releasePBError,poMessage);
            releasePBError.vDestroy();
         }

      }
      oReleasePhoneBookList_MR.vDestroy();
   }
   else
   {
      // If we come here, we assume the client has sent an erroneous message,
      // probably it has used a wrong FI_MAJOR_VERSION. Therefore we will
      // send back a an error message and throw a NORMAL_M_ASSERT_ALWAYS().

      vHandleGetDataError(poMessage,"vHandle_ReleasePhoneBookList_MethodStart");
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)

   oReleasePhoneBookListMethodStart_MS.vDestroy();    



}

/*******************************************************************************
** FUNCTION:   vHandle_SearchKeyboardPhoneBookList_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for SearchKeyboardPhoneBookList method
*   
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::\
vHandle_SearchKeyboardPhoneBookList_MethodStart(\
                                                amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_SearchKeyboardPhoneBookList_MethodStart(): entered"));

   // Extract the Call instance and other required info from poMessage
   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );

   // Create the (message related) FI data object
   most_phonbkfi_tclMsgSearchKeyboardPhoneBookListMethodStart \
      oSearchKeyboardPhoneBookListMethodStart_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.

   if ( oVisitorMsg.s32GetData(oSearchKeyboardPhoneBookListMethodStart_MS, \
      MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION)
      != OSAL_ERROR)
   {

      ETG_TRACE_USR2(("LIST HANDLE: %u",oSearchKeyboardPhoneBookListMethodStart_MS.u16ListHandle));


      most_phonbkfi_tclMsgSearchKeyboardPhoneBookListMethodResult \
         oSearchKeyboardPhoneBookList_MR;

      tBool dbStatus = FALSE;

      if (m_pPhoneBookDB)
      {
         //SearchKeyboardPhoneBookList call from Sqlite data base//
         dbStatus
               = m_pPhoneBookDB->SearchKeyboardPhoneBookList(&oSearchKeyboardPhoneBookListMethodStart_MS, &oSearchKeyboardPhoneBookList_MR);
      }




      // Get the coresponding Call Instance
      if(dbStatus != TRUE)
      {
         //DB Returned error
         amt_tclServiceDataError oErrorMsg(*poMessage, most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC);

         // Post message
         ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

         if (enResult != AIL_EN_N_NO_ERROR)
         {

            ETG_TRACE_ERR(("vHandle_SearchKeyboardPhoneBookList_MethodStart: 'ERROR', enPostMessage() failed"));
            // NORMAL_M_ASSERT_ALWAYS();
         }

         // NORMAL_M_ASSERT_ALWAYS();
      }
      else
      {
         fi_tclVisitorMessage oMRMsg(oSearchKeyboardPhoneBookList_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
         // Here the message related header information is added and by this the
         // creation of the message is completed.
         vInitServiceData( oMRMsg,
            poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(),  // RegisterID
            poMessage->u16GetCmdCounter(),  // CmdCounter
            poMessage->u16GetFunctionID(),  // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT   // Opcode
            );

         if (oMRMsg.bIsValid())
         {

            // Post message
            ail_tenCommunicationError enResult =
               _poMainAppl->enPostMessage(&oMRMsg, TRUE);

            if (enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_ERR(("vHandle_SearchKeyboardPhoneBookList_MethodStart:'MethodResult', enPostMessage() failed,'ail_tenCommunicationError' = %u",(tU32)enResult ));
               // We can come here  e.g. if the client has changed his application
               // state to OFF in the meantime. Therefore we dont throw an assert.
               // NORMAL_M_ASSERT_ALWAYS();
            }
         }
         else
         {
            most_phonbkfi_tclMsgSearchKeyboardPhoneBookListError searchKBError;
            vHandleMethodError(searchKBError,poMessage);
            searchKBError.vDestroy();
         }

      }

      oSearchKeyboardPhoneBookList_MR.vDestroy();
   }
   else
   {
      // If we come here, we assume the client has sent an erroneous message,
      // probably it has used a wrong FI_MAJOR_VERSION. Therefore we will
      // send back a an error message and throw a NORMAL_M_ASSERT_ALWAYS().

      vHandleGetDataError(poMessage,"vHandle_SearchKeyboardPhoneBookList_MethodStart");
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)

   oSearchKeyboardPhoneBookListMethodStart_MS.vDestroy();


}

/*******************************************************************************
** FUNCTION:   vHandle_AddContact_MethodStart
*******************************************************************************/
/*  DESCRIPTION:
*  	 This is a handler function to add contact to Phone.
*   PARAMETERS:
*   	IN:   poMessage        : This is the received message from a client
*   	IN: Contact Details and DeviceHandle
*   OUT: Contact Handle and Status of Adding Contact Info to Database
*   RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_AddContact_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_AddContact_MethodStart(): entered"));

   bool ret = false;
   fi_tclVisitorMessage oVisitorMsg(poMessage);
   most_phonbkfi_tclMsgAddContactMethodStart oAddContact_MS;
   most_phonbkfi_tclMsgAddContactMethodResult oAddContactList_MR;

   if(oVisitorMsg.s32GetData(oAddContact_MS,MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {

      ETG_TRACE_USR2(("DEVICE HANDLE: %u",oAddContact_MS.u8DeviceHandle));

         // -bn: MY15 Feature: VPB Import: Code addition: BEGIN
         if(PHONEBOOK_C_U32_VEHICLE_CONTACT_IMPORT_COMPLETED == oAddContact_MS.oContactDetails.u32FirstNameTag)  // 0xFC000000
         {
            // This is a "dummy" function call of 'AddContact', which just indicates that "bulk" insertion of contacts to the Vechicle Contact list,
            // i.e. the import of Vehicle Contacts, is finished.
            // Note: No contact data will be added to the Phonebook DB for this "dummy" function call.
            ETG_TRACE_USR1((" Import of Vehicle contacts is finished. ")); 
            ret = true;
         }
         else
         {
            if( (m_pPhoneBookDB) && (FC_PhoneBook_SQLite::GetSQLiteInstance()->GetNoOfContactsInVPB(oAddContact_MS.u8DeviceHandle) < MAX_ENTRIES_VEHICLE_PHONEBOOK) && (VALUE_ZERO != oAddContact_MS.u8DeviceHandle) )
            {
            // Add contact with given details to the Phonebook DB.
               ret = m_pPhoneBookDB->AddContact(&oAddContact_MS,&oAddContactList_MR);
            }
         }
         // -bn: MY15 Feature: VPB Import: Code addition: END

      ETG_TRACE_USR1((" Contact Add Status:: %u",ETG_ENUM(TR_CONTACTADDSTATUS,oAddContactList_MR.e8ContactAddStatus.enType)));   //GMMY16-7695 Improper updation of Recent CH when AddContact fails.

      if( ret == true)
      {
         fi_tclVisitorMessage oMRMsg(oAddContactList_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
         vInitServiceData(oMRMsg,
            poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(),  // RegisterID
            poMessage->u16GetCmdCounter(),  // CmdCounter
            poMessage->u16GetFunctionID(),  // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT   // Opcode
            );

         if(oMRMsg.bIsValid())
         {
            // Post message
            ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);
            if(enResult != AIL_EN_N_NO_ERROR)
            {

               ETG_TRACE_ERR(("vHandle_AddContact_Result: 'Result', enPostMessage() failed, 'ail_tenCommunicationError' = %d",(tU32)enResult));
               // NORMAL_M_ASSERT_ALWAYS();
            }
            else if(most_fi_tcl_e8_PhonBkContactAddStatus::FI_EN_E8CONTACT_ADD_FAILURE != oAddContactList_MR.e8ContactAddStatus.enType)   //GMMY16-7695 (Improper updation of Recent CH when AddContact fails) MethodResultPosted is not called when AddContact fails.
            {
               if(m_pPhoneBookDB)
               {
               m_pPhoneBookDB->MethodResultPosted(poMessage->u16GetFunctionID(),
                  &oAddContact_MS, &oAddContactList_MR);
               }
            }
         }
         else
         {
            most_phonbkfi_tclMsgAddContactError oAddContact;
            vHandleMethodError(oAddContact,poMessage);
            oAddContact.vDestroy();
         }

      }
      else
      {
         amt_tclServiceDataError oErrorMsg(*poMessage, most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC);

         // Post message
         ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

         if (enResult != AIL_EN_N_NO_ERROR)
         {

            ETG_TRACE_ERR(("vHandle_AddContact_MethodStart: 'ERROR', enPostMessage() failed"));
         }
         // NORMAL_M_ASSERT_ALWAYS();
      }
      oAddContactList_MR.vDestroy();
   }
   else
   {
      vHandleGetDataError(poMessage,"vHandle_AddContact_MethodStart");
   }
   // Always destroy the(message related)FI data object (before leaving
   // its creation scope)
   oAddContact_MS.vDestroy();


}

/*******************************************************************************
** FUNCTION:   vHandle_EditContact_MethodStart
*******************************************************************************/
/*   DESCRIPTION:
*    This is a handler function to Edit Contact Information in database with reference to Contact Handle,Device Handle
*
*     PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*     IN : ContactHandle, DeviceHandle and Contact Information
*     OUT:  Contact handle and Record Edit Status (Success or Failure)
*     RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_EditContact_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_EditContact_MethodStart(): entered"));

   bool ret = false;
   fi_tclVisitorMessage oVisitorMsg(poMessage);
   most_phonbkfi_tclMsgEditContactMethodStart oEditContact_MS;
   most_phonbkfi_tclMsgEditContactMethodResult oEditContact_MR;
   if(oVisitorMsg.s32GetData(oEditContact_MS,MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {


      ETG_TRACE_USR2(("DEVICE HANDLE: %u",oEditContact_MS.u8DeviceHandle));

      ETG_TRACE_USR2(("CONTACT HANDLE: %d",oEditContact_MS.u32ContactHandle));

      
      if ((oEditContact_MS.u32ContactHandle & FC_PB_CONTACT_HANDLE_DPB_MARKER) != FC_PB_CONTACT_HANDLE_DPB_MARKER)
      {
         if( (m_pPhoneBookDB) && (VALUE_ZERO != oEditContact_MS.u8DeviceHandle) && (VALUE_ZERO != oEditContact_MS.u32ContactHandle))
         {
            ret = m_pPhoneBookDB->EditContact(&oEditContact_MS,&oEditContact_MR);
         }
      }

      ETG_TRACE_USR1((" Contact Edit Status:: %u",ETG_ENUM(TR_CONTACTEDITSTATUS,oEditContact_MR.e8ContactEditStatus.enType)));   //GMMY16-7695

      if(ret == true)
      {
         fi_tclVisitorMessage oMRMsg(oEditContact_MR,MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
         vInitServiceData(oMRMsg,
            poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(),  // RegisterID
            poMessage->u16GetCmdCounter(),  // CmdCounter
            poMessage->u16GetFunctionID(),  // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT   // Opcode
            );

         if(oMRMsg.bIsValid())
         {
            // Post message
            ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);
            if(enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_ERR(("vHandle_EditContact_MethodStart: 'ERROR', enPostMessage() failed"));
               // NORMAL_M_ASSERT_ALWAYS();
            }
            else if(most_fi_tcl_e8_PhonBkContactEditStatus::FI_EN_E8CONTACT_EDIT_FAILURE != oEditContact_MR.e8ContactEditStatus.enType)   //GMMY16-7695 MethodResultPosted should not be called when EditContact fails.
            {
               if(m_pPhoneBookDB)
               {
               m_pPhoneBookDB->MethodResultPosted(poMessage->u16GetFunctionID(),
                  &oEditContact_MS, &oEditContact_MR);
               }
            }

         }
         else
         {
            most_phonbkfi_tclMsgEditContactError oEditContact;
            vHandleMethodError(oEditContact,poMessage);
            oEditContact.vDestroy();
         }
      }
      else
      {
         amt_tclServiceDataError oErrorMsg(*poMessage, most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC);

         // Post message
         ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

         if (enResult != AIL_EN_N_NO_ERROR)
         {

            ETG_TRACE_USR1(("vHandle_EditContact_MethodStart: 'ERROR', enPostMessage() failed"));

         }

         // NORMAL_M_ASSERT_ALWAYS();

      }        
      oEditContact_MR.vDestroy();
   }
   else
   {
      vHandleGetDataError(poMessage,"vHandle_EditContact_MethodStart");
   }
   // Always destroy the(message related)FI data object (before leaving
   // its creation scope)
   oEditContact_MS.vDestroy();

}
/*******************************************************************************
** FUNCTION:   vHandle_SearchPhoneBook_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for SearchPhoneBook method
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/

tVoid fc_phonebook_tclService_PhoneBook::vHandle_SearchPhoneBook_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_SearchPhoneBook_MethodStart(): entered"));

   // Extract the Call instance and other required info from poMessage
   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );
   ail_tenCommunicationError enResult = AIL_EN_N_NO_ERROR;
   tBool blNoError= true;
   tU16  u16ErrorCode = CCA_C_U16_ERROR_UNSPECIFIED;

   // Create the (message related) FI data object
   most_phonbkfi_tclMsgSearchPhoneBookMethodStart \
      oSearchPhoneBookMethodStart_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.

   if ( oVisitorMsg.s32GetData(oSearchPhoneBookMethodStart_MS, \
      MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION)
      != OSAL_ERROR)
   {
      ETG_TRACE_USR2(("Phone Number: %s",oSearchPhoneBookMethodStart_MS.sPhoneNumber.szValue));
      ETG_TRACE_USR2(("Email Address: %s",oSearchPhoneBookMethodStart_MS.sEmailAddress.szValue));

      most_phonbkfi_tclMsgSearchPhoneBookMethodResult \
         oSearchPhoneBook_MR;

      ETG_TRACE_USR1((" Contact Detail Type:: %u",ETG_ENUM(TR_CONTDETAILTYPE,oSearchPhoneBook_MR.e8ContactDetailType.enType)));
      if(FC_Device_Details::deviceHandleToObjectInstance(oSearchPhoneBookMethodStart_MS.u8DeviceHandle))  //To check if the device handle is valid
      {
         //If the device is connected, then it is valid
         tBool dbStatus = false;

         if(m_pPhoneBookDB)
         {
            dbStatus
               = m_pPhoneBookDB->SearchPhoneBook(&oSearchPhoneBookMethodStart_MS,
                     &oSearchPhoneBook_MR);

         }

         oSearchPhoneBook_MR.u8DeviceHandle= oSearchPhoneBookMethodStart_MS.u8DeviceHandle;
         if(!dbStatus)
         {
            oSearchPhoneBook_MR.u32ContactHandle= 0/*NULL*/;
            // Solving Compiler warning  converting to non-pointer type 'tU32' from NULL
            oSearchPhoneBook_MR.sFirstName.vDestroy();
            oSearchPhoneBook_MR.sLastName.vDestroy();
            oSearchPhoneBook_MR.e8ContactDetailType.enType = most_fi_tcl_e8_PhonBkContactDetailType::FI_EN_E8NONE;
         }

         fi_tclVisitorMessage oMRMsg(oSearchPhoneBook_MR, \
            MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);

         // Here the message related header information is added and by this the
         // creation of the message is completed.

         vInitServiceData( oMRMsg,
            poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(),  // RegisterID
            poMessage->u16GetCmdCounter(),  // CmdCounter
            poMessage->u16GetFunctionID(),  // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT    // Opcode
            );

         if (oMRMsg.bIsValid())
         {
            // Post message
            enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

            if (enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_ERR(( "vHandle_SearchPhoneBook_MethodStart: 'Result', enPostMessage() failed"));
               // NORMAL_M_ASSERT_ALWAYS();
            }
         }
         else
         {
            most_phonbkfi_tclMsgSearchPhoneBookError searchPBerror;
            vHandleMethodError(searchPBerror,poMessage);
            searchPBerror.vDestroy();
         }
      }
      else //Device handle passed is incorrect. No connected device with that device handle
      {
         ETG_TRACE_USR4(("Device handle passed is incorrect. No connected device with that device handle"));

         blNoError= false;
         u16ErrorCode= CCA_C_U16_ERROR_PARAMETER_OUT_OF_RANGE;
      }
      oSearchPhoneBook_MR.vDestroy();
   }
   else
   {
      ETG_TRACE_USR4(("Unable to Un Marshal Incoming message "));

      // If we come here, we assume the client has sent an erroneous message,
      // probably it has used a wrong FI_MAJOR_VERSION. Therefore we will
      // send back a an error message and throw a NORMAL_M_ASSERT_ALWAYS().
      vHandleGetDataError(poMessage,"vHandle_SearchPhoneBook_MethodStart");
   }

   if(!blNoError)
   {
      amt_tclServiceDataError oErrorMsg(*poMessage,u16ErrorCode);
      enResult = _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_USR4(("  vHandle_SearchPhoneBook_MethodStart(): 'ERROR', enPostMessage() failed "));
      }
      else
      {
         ETG_TRACE_USR4(("  vHandle_SearchPhoneBook_MethodStart(): 'SUCCESS', enPostMessage() SUCCESS  "));
         // NORMAL_M_ASSERT_ALWAYS();
      }
   }
   oSearchPhoneBookMethodStart_MS.vDestroy();
}

/*******************************************************************************
** FUNCTION:   vHandle_SearchContacts_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for SearchContacts method
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_SearchContacts_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_SearchContacts_MethodStart(): entered"));

   // Extract the Call instance and other required info from poMessage
   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );

   // Create the (message related) FI data object
   most_phonbkfi_tclMsgSearchContactsMethodStart oSearchContactsMethodStart_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data object.

   if ( oVisitorMsg.s32GetData(oSearchContactsMethodStart_MS, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION)!= OSAL_ERROR)
   {
      tBool bError= FALSE;
      most_phonbkfi_tclMsgSearchContactsError oSearchContactsError;
      tU8 u8DeviceHandleListSize = (tU8)(oSearchContactsMethodStart_MS.oDeviceHandleList.oDeviceHandles.size());
      if ((0 != u8DeviceHandleListSize) && (oSearchContactsMethodStart_MS.sSearchString.szValue))
      {
         tU32 u32SrcLen = (tU32)OSAL_u32StringLength(oSearchContactsMethodStart_MS.sSearchString.szValue);
         ETG_TRACE_USR4((" u32SrcLen: %u", u32SrcLen));
         if(0 != u32SrcLen)
         {
            ETG_TRACE_USR2(("Search String: %s",oSearchContactsMethodStart_MS.sSearchString.szValue));

            most_phonbkfi_tclMsgSearchContactsMethodResult oSearchContacts_MR;

            tBool bRet = FALSE;
            if(m_pPhoneBookDB)
            {
               bRet = m_pPhoneBookDB->SearchContacts(&oSearchContactsMethodStart_MS, &oSearchContacts_MR);
            }

            if(TRUE == bRet)
            {
               fi_tclVisitorMessage oMRMsg(oSearchContacts_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);

               // Here the message related header information is added and by this the
               // creation of the message is completed.

               vInitServiceData( oMRMsg,
                  poMessage->u16GetSourceAppID(), // ServiceDataMsg
                  poMessage->u16GetRegisterID(),  // RegisterID
                  poMessage->u16GetCmdCounter(),  // CmdCounter
                  poMessage->u16GetFunctionID(),  // Function ID
                  AMT_C_U8_CCAMSG_OPCODE_METHODRESULT    // Opcode
                  );

               if (oMRMsg.bIsValid())
               {
                  // Post message
                  ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

                  if (enResult != AIL_EN_N_NO_ERROR)
                  {
                     ETG_TRACE_ERR(( "vHandle_SearchContacts_MethodStart: 'Result', enPostMessage() failed"));
                     // NORMAL_M_ASSERT_ALWAYS();
                  }
               }
               else
               {
                  ETG_TRACE_ERR(( " vHandle_SearchContacts_MethodStart: oMRMsg is invalid"));
                  bError= TRUE;
               }

            }
            else
            {
               //SearchContacts Fails
               ETG_TRACE_ERR(( " vHandle_SearchContacts_MethodStart: 'ERROR' SearchContacts fails "));
               bError= TRUE;
            }
            oSearchContacts_MR.vDestroy();
         }
         else
         {
            ETG_TRACE_ERR((" sSearchString is empty!!!"));
            bError= TRUE;
         }

      }
      else // DeviceHandleList or SearchString can't be empty.
      {
         ETG_TRACE_ERR(("Either DeviceHandleList or SearchString is empty."));
         bError= TRUE;
      }

      if(bError)
      {
         vHandleMethodError(oSearchContactsError,poMessage);
         // NORMAL_M_ASSERT_ALWAYS();
      }
      oSearchContactsError.vDestroy();
   }
   else
   {
      ETG_TRACE_ERR(("Unable to Un Marshal the Incoming message "));

      // If we come here, we assume the client has sent an erroneous message,
      // probably it has used a wrong FI_MAJOR_VERSION. Therefore we will
      // send back an error message and throw a NORMAL_M_ASSERT_ALWAYS().
      vHandleGetDataError(poMessage,"vHandle_SearchContacts_MethodStart");
   }
   oSearchContactsMethodStart_MS.vDestroy();
}

/*******************************************************************************
** FUNCTION:   vHandle_ClearPhoneBookData_MethodStart
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function to Delete All Contacts and Call History in Phone
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*     IN :
*     OUT:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_ClearPhoneBookData_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_ClearPhoneBookData_MethodStart(): entered"));

   bool ret = false;
   fi_tclVisitorMessage oVisitorMsg(poMessage);
   most_phonbkfi_tclMsgClearPhoneBookDataMethodStart oClearPhoneBookData_MS;
   if(oVisitorMsg.s32GetData(oClearPhoneBookData_MS,MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      if(m_pPhoneBookDB)
      {
      ret = m_pPhoneBookDB->ClearPhoneBookData();//FIX GMMY16-23858
      }
      if(ret == false)
      {
         //TODO This doesnt have MethodResult to report the status to HMI

         ETG_TRACE_ERR(("Error in Deleting Records/Deleting List"));
         most_phonbkfi_tclMsgClearPhoneBookDataError clearPB;
         vHandleMethodError(clearPB,poMessage);
         clearPB.vDestroy();

      }
   }
   else
   {
      vHandleGetDataError(poMessage,"vHandle_ClearPhoneBookData_MethodStart");
   }
   // Always destroy the(message related)FI data object (before leaving
   // its creation scope)
   oClearPhoneBookData_MS.vDestroy();

}

tVoid fc_phonebook_tclService_PhoneBook::vHandle_ClearVehiclePhoneBook_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_ClearVehiclePhoneBook_MethodStart(): entered"));

   bool ret = false;
   fi_tclVisitorMessage oVisitorMsg(poMessage);
   most_phonbkfi_tclMsgClearVehiclePhoneBookMethodStart oClearPhoneBookData_MS;
   if(oVisitorMsg.s32GetData(oClearPhoneBookData_MS,MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      if(m_pPhoneBookDB)
      {
      ret = m_pPhoneBookDB->ClearVehiclePhoneBook();
      }
      if(ret == false)
      {
         //TODO This doesnt have MethodResult to report the status to HMI

         ETG_TRACE_ERR(("Error in Deleting Records/Deleting List"));
         most_phonbkfi_tclMsgClearVehiclePhoneBookError clearVechilePB;
         vHandleMethodError(clearVechilePB,poMessage);
         clearVechilePB.vDestroy();
      }
   }
   else
   {
      vHandleGetDataError(poMessage,"vHandle_ClearVehiclePhoneBook_MethodStart");
   }
   // Always destroy the(message related)FI data object (before leaving
   // its creation scope)
   oClearPhoneBookData_MS.vDestroy();

}




/*******************************************************************************
** FUNCTION:   vHandle_DeleteContact_MethodStart
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function to Delete contact with reference to Contact Handle and device handle
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*     IN : Contact handle, Device Handle
*     OUT:  Contact handle and Record Delete Status (Success or Failure)
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_DeleteContact_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_DeleteContact_MethodStart(): entered"));

   bool ret = false;
   bool bValue = false;
   fi_tclVisitorMessage oVisitorMsg(poMessage);
   most_phonbkfi_tclMsgDeleteContactMethodStart oDeleteContact_MS;
   most_phonbkfi_tclMsgDeleteContactMethodResult oDeleteContact_MR;
   if (oVisitorMsg.s32GetData(oDeleteContact_MS, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION)
         != OSAL_ERROR)
   {
      ETG_TRACE_USR2(("DEVICE HANDLE:%u", oDeleteContact_MS.u8DeviceHandle));

      ETG_TRACE_USR2(("CONTACT HANDLE:%d", oDeleteContact_MS.u32ContactHandle));

      if ((oDeleteContact_MS.u32ContactHandle & FC_PB_CONTACT_HANDLE_DPB_MARKER)
            != FC_PB_CONTACT_HANDLE_DPB_MARKER)
      {
         //Fix : CMG3G-12880
         FC_Device_Details
               * pDeviceObject =
                     FC_Device_Details::deviceHandleToObjectInstance(oDeleteContact_MS.u8DeviceHandle);
         if (pDeviceObject)
         {
            bValue = true;
            ETG_TRACE_USR2(("DepDeviceObject is not NULL"));
         }
         else
         {
            ETG_TRACE_USR2(("DepDeviceObject is NULL"));
         }
         //End fix : CMG3G-12880

         //Fix NCG3D-68310
         //Once HFP is connected, user can be able to delete contacts in vehicle phonebook irrespective of device object creation.
         if (true == bValue)
         {
            //Fix RTC 159338
            if (m_pPhoneBookDB)
            {
               ret
                     = m_pPhoneBookDB->DeleteContact(&oDeleteContact_MS, &oDeleteContact_MR);
               // vRemoveFavorite(oDeleteContact_MS.u32ContactHandle);
            }
            //End fix RTC 159338
         }
         //End fix NCG3D-68310
         //End fix : CMG3G-12880
      }
      ETG_TRACE_USR1((" Contact Delete Status:: %u", ETG_ENUM(TR_CONTACTDELETESTATUS, oDeleteContact_MR.e8ContactDeleteStatus.enType))); //GMMY16-7695

      if (ret == true)
      {
         fi_tclVisitorMessage
               oMRMsg(oDeleteContact_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
         vInitServiceData(oMRMsg, poMessage->u16GetSourceAppID(), // ServiceDataMsg
         poMessage->u16GetRegisterID(), // RegisterID
         poMessage->u16GetCmdCounter(), // CmdCounter
         poMessage->u16GetFunctionID(), // Function ID
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
         );
         if (oMRMsg.bIsValid())
         {
            // Post message
            ail_tenCommunicationError enResult =
                  _poMainAppl->enPostMessage(&oMRMsg, TRUE);
            if (enResult != AIL_EN_N_NO_ERROR)
            {

               ETG_TRACE_ERR(("vHandle_DeleteContact_Result: 'Result', enPostMessage() failed, 'ail_tenCommunicationError' = %d", (tU32) enResult));

               // NORMAL_M_ASSERT_ALWAYS();
            }
            else if (most_fi_tcl_e8_PhonBkContactDeleteStatus::FI_EN_E8CONTACT_DELETE_FAILURE
                  != oDeleteContact_MR.e8ContactDeleteStatus.enType) //GMMY16-7695 MethodResultPosted should not be called when DeleteContact fails.
            {
               if (m_pPhoneBookDB)
               {
                  m_pPhoneBookDB->MethodResultPosted(poMessage->u16GetFunctionID(), &oDeleteContact_MS, &oDeleteContact_MR);
               }
            }

         }
         else
         {
            most_phonbkfi_tclMsgDeleteContactError deleteContactError;
            vHandleMethodError(deleteContactError, poMessage);
            deleteContactError.vDestroy();
         }
      }
      else
      {
         //Inserting Fails
         amt_tclServiceDataError
               oErrorMsg(*poMessage, most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC);

         // Post message
         ail_tenCommunicationError enResult =
               _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

         if (enResult != AIL_EN_N_NO_ERROR)
         {

            ETG_TRACE_ERR(("vHandle_DeleteContact_MethodStart: 'ERROR', enPostMessage() failed"));
         }

         // NORMAL_M_ASSERT_ALWAYS();
      }

      oDeleteContact_MR.vDestroy();
   }
   else
   {
      vHandleGetDataError(poMessage, "vHandle_DeleteContact_MethodStart");
   }
   // Always destroy the(message related)FI data object (before leaving
   // its creation scope)
   oDeleteContact_MS.vDestroy();

}


/*******************************************************************************
** FUNCTION:   vHandle_GetContactDetails_MethodStart
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for To Get contact Details for the Contact Handle
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_GetContactDetails_MethodStart(amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR4(("vHandle_GetContactDetails_MethodStart(): entered"));

   tBool result = FALSE;
   fi_tclVisitorMessage oVisitorMsg(poMessage);
   most_phonbkfi_tclMsgGetContactDetailsMethodStart oGetContactDetails_MS;
   most_phonbkfi_tclMsgGetContactDetailsMethodResult oGetContactDetails_MR;


   if(oVisitorMsg.s32GetData( oGetContactDetails_MS, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {


      ETG_TRACE_USR2(("Contact Handle: %d", oGetContactDetails_MS.u32ContactHandle));
      //ETG_TRACE_USR2(("Contact Filter: %u", oGetContactDetails_MS.e8ContactDetailFilter.enType));
      ETG_TRACE_USR1((" Contact Detail Filter:: %u",ETG_ENUM(TR_CONTACTDETAILFILTER,oGetContactDetails_MS.e8ContactDetailFilter.enType)));

      if((m_pPhoneBookDB) && (VALUE_ZERO != oGetContactDetails_MS.u32ContactHandle))
      {
         result = m_pPhoneBookDB->GetContactDetails(&oGetContactDetails_MS,&oGetContactDetails_MR);
      }

      if(result == TRUE)
      {
         fi_tclVisitorMessage oMRMsg(oGetContactDetails_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
         //Inserting Success
         vInitServiceData(oMRMsg,
            poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(),  // RegisterID
            poMessage->u16GetCmdCounter(),  // CmdCounter
            poMessage->u16GetFunctionID(),  // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT   // Opcode
            );

         if(oMRMsg.bIsValid())
         {
            // Post message
            ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);
            if(enResult != AIL_EN_N_NO_ERROR)
            {

               ETG_TRACE_ERR(("vHandle_GetContactDetails_MethodStart: 'Result', enPostMessage() failed, 'ail_tenCommunicationError' = %d",(tU32)enResult));
               // NORMAL_M_ASSERT_ALWAYS();
            }
         }
         else
         {
            most_phonbkfi_tclMsgGetContactDetailsError getContactDetailsError;
            vHandleMethodError(getContactDetailsError,poMessage);
            getContactDetailsError.vDestroy();
         }
      }
      else
      {
         //Inserting Fails
         amt_tclServiceDataError oErrorMsg(*poMessage, most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC);

         // Post message
         ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

         if (enResult != AIL_EN_N_NO_ERROR)
         {

            ETG_TRACE_ERR(("vHandle_GetContactDetails_MethodStart: 'ERROR', enPostMessage() failed"));
         }

         // NORMAL_M_ASSERT_ALWAYS();
      }


      oGetContactDetails_MR.vDestroy();
   }
   else
   {
      vHandleGetDataError(poMessage,"vHandle_GetContactDetails_MethodStart");
   }
   // Always destroy the(message related)FI data object (before leaving
   // its creation scope)
   oGetContactDetails_MS.vDestroy();


}
/*******************************************************************************
** FUNCTION:   vSetBTSettingsClientHandlerInstance(..)
*******************************************************************************/
/* DESCRIPTION: This function registers the pointer of the instance of the client handler
*               of Bluetooth Settings
*
*  PARAMETERS:
*     IN:   poBTSettings - Instance of the BT settings client handler
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vSetBTSettingsClientHandlerInstance(fc_phonebook_tclClientHandler_BluetoothSetings* poBTSettings)
{
   ETG_TRACE_USR4(("vSetBTSettingsClientHandlerInstance(): entered"));
   m_poBTSettings= poBTSettings;
}


/*******************************************************************************
** FUNCTION:   ResetMissedCall List Once Blue tooth Device is Connected(..)
*******************************************************************************/
/* DESCRIPTION:
*   	This is a handler function to Clear the missed calls once Connected
*
*  PARAMETERS:
*    	 IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vClearMissedCalls(tU8 u8DeviceHandle,FC_Device_Details * poDeviceObject)
{
   (tVoid)u8DeviceHandle;

   ETG_TRACE_USR4(("vClearMissedCalls(): entered"));
   if(poDeviceObject)
   {
      m_aNoOfMissedCalls[poDeviceObject->getDeviceHandle()] = 0;
      vPhoneBook_UpdateClientsAboutMissedCallCount(poDeviceObject->getDeviceHandle());
   }
   else
   {

      ETG_TRACE_ERR(("vClearMissedCalls: Device object is NULL"));
   }


}

/*******************************************************************************
*
* FUNCTION: tVoid vHandle_ClearMissedCallIndicator_MethodStart
*
* DESCRIPTION: This function is called by the democlient to clear the missed call when user viewed the missed call
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_ClearMissedCallIndicator_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_ClearMissedCallIndicator_MethodStart(): entered"));
   /* Extract the Call instance and other required info from poMessage */
   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);
   // Create the (message related) FI data object
   most_phonbkfi_tclMsgClearMissedCallIndicatorMethodStart oClearMissedCallIndicator_MS;
   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.
   if(oVisitorMsg.s32GetData(oClearMissedCallIndicator_MS,MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {


      ETG_TRACE_USR2(("	DEVICE HANDLE: %u",oClearMissedCallIndicator_MS.u8DeviceHandle));



      m_aNoOfMissedCalls[oClearMissedCallIndicator_MS.u8DeviceHandle] = 0;


      FC_Device_Details *deviceObject = FC_Device_Details::deviceHandleToObjectInstance(oClearMissedCallIndicator_MS.u8DeviceHandle);
      if(deviceObject)
      {
         deviceObject->vSetMissedcallCount(0);
         deviceObject->vSwapMissedCallDetails();
      }
      else
      {
         ETG_TRACE_ERR(("Device Object is NULL"));
      }


      most_phonbkfi_tclMsgClearMissedCallIndicatorMethodResult oClearMissedCallIndicator_MR;
      oClearMissedCallIndicator_MR.u8DeviceHandle = oClearMissedCallIndicator_MS.u8DeviceHandle;
      fi_tclVisitorMessage oMRMsg(oClearMissedCallIndicator_MR,MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);

      vInitServiceData(oMRMsg,
         poMessage->u16GetSourceAppID(), // ServiceDataMsg
         poMessage->u16GetRegisterID(),  // RegisterID
         poMessage->u16GetCmdCounter(),  // CmdCounter
         poMessage->u16GetFunctionID(),  // Function ID
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT);

      if(oMRMsg.bIsValid())
      {
         ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);
         if(enResult != AIL_EN_N_NO_ERROR)
         {
            ETG_TRACE_ERR(("vHandle_ClearMissedCall_MethodStart: 'MethodResult', enPostMessage() failed, "
               "'ail_tenCommunicationError'"));

            // NORMAL_M_ASSERT_ALWAYS();
         }
      }
      else
      {
         most_phonbkfi_tclMsgClearMissedCallIndicatorError ClearMissedCallIndicatorError;
         vHandleMethodError(ClearMissedCallIndicatorError,poMessage);
         ClearMissedCallIndicatorError.vDestroy();
      }

      oClearMissedCallIndicator_MR.vDestroy();
      vPhoneBook_UpdateClientsAboutMissedCallCount(oClearMissedCallIndicator_MS.u8DeviceHandle);
      vUpdatePhonebookMissedCallIndicatorList(oClearMissedCallIndicator_MS.u8DeviceHandle);
   }
   else
   {
      // If we come here, we assume the client has sent an erroneous message,
      // probably it has used a wrong FI_MAJOR_VERSION. Therefore we will
      // send back a an error message and throw a NORMAL_M_ASSERT_ALWAYS().
      vHandleGetDataError(poMessage,"vHandle_ClearMissedCallIndicator_MethodStart");
   }
   // Always destroy the(message related)FI data object (before leaving
   // its creation scope)
   oClearMissedCallIndicator_MS.vDestroy();


}
/*******************************************************************************
*
* FUNCTION: tVoid vMissedCallIndicatorUpdate
*
* DESCRIPTION: Updates number of missed calls to client with reference to DeviceHandle
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vMissedCallIndicatorUpdate(tU32 cMissedCallCnt,FC_Device_Details * poDeviceObject)
{
   ETG_TRACE_USR4(("vMissedCallIndicatorUpdate: entered"));

   if((poDeviceObject) &&
      (poDeviceObject->getDeviceCapabilities() & (MissedCallHistory)))
   {
      m_aNoOfMissedCalls[poDeviceObject->getDeviceHandle()] =  (tULong)cMissedCallCnt;

      if ((poDeviceObject->getSourceType()) == TRUE)
      {
         ETG_TRACE_USR4(("Update Missed Call Count to clients"));
         vPhoneBook_UpdateClientsAboutMissedCallCount(poDeviceObject->getDeviceHandle());
      }
      else
      {
         ETG_TRACE_USR4(("Device is not an outgoing source"));
      }
   }
   else
   {

      ETG_TRACE_ERR(("vMissedCallIndicatorUpdate: Device object is NULL"));
   }
}


/*******************************************************************************
** FUNCTION:   vPhoneBook_PBDownloadComplete(..)
*******************************************************************************/
/* DESCRIPTION:
*  
*  PARAMETERS:
*     IN:          : 
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vPhoneBook_PBDownloadComplete(tU8 u8DeviceHandle)
{
   ETG_TRACE_COMP((" -PBDL.S- vPhoneBook_PBDownloadComplete entered"));

   FC_Device_Details *deviceObject=
      FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);

   if(deviceObject && m_pPhoneBookDB)
   {
      m_pPhoneBookDB->DownloadCompleted(deviceObject->getDeviceHandle());
   }
   else
   {
      ETG_TRACE_ERR(("Device Object is NULL"));
   }
}


/*******************************************************************************
** FUNCTION:   vHandle_RequestSliceCallHistoryList_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for RequestSliceCallHistoryList method
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_RequestSliceCallHistoryList_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("MethodStart: vHandle_RequestSliceCallHistoryList_MethodStart entered  "));
   /* Extract the Call instance and other required info from poMessage */
   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );

   // Create the (message related) FI data object
   most_phonbkfi_tclMsgRequestSliceCallHistoryListMethodStart oRequestSliceCallHistoryList_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.

   if ( oVisitorMsg.s32GetData(oRequestSliceCallHistoryList_MS, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {


      ETG_TRACE_USR2(("	LIST HANDLE: %u", oRequestSliceCallHistoryList_MS.u16ListHandle));

      ETG_TRACE_USR2(("	WINDOW START: %d", oRequestSliceCallHistoryList_MS.u32WindowStart));

      ETG_TRACE_USR2(("	WINDOW SIZE: %u",  oRequestSliceCallHistoryList_MS.u16WindowSize));



      most_phonbkfi_tclMsgRequestSliceCallHistoryListMethodResult oRequestSliceCallHistoryList_MR;

      //RequestSlicePhoneBookList call from Sqlite data base//
      tBool dbStatus = FALSE;

      if(m_pPhoneBookDB)
      {
         dbStatus = m_pPhoneBookDB->RequestSliceCallHistoryList( &oRequestSliceCallHistoryList_MS , &oRequestSliceCallHistoryList_MR,poMessage->u16GetRegisterID());//FIX NCG3D-55494 RegisterID is passed to identify the requested client
      }

      //Get the coresponding Call Instance

      if(dbStatus != TRUE)
      {
         //DB Returned error
         ETG_TRACE_USR4(("Error message posted successfully."));
         most_phonbkfi_tclMsgRequestSliceCallHistoryListError hisotryListError;
         vHandleMethodError(hisotryListError,poMessage);
         hisotryListError.vDestroy();
      }
      else
      {
         fi_tclVisitorMessage oMRMsg(oRequestSliceCallHistoryList_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
         vInitServiceData( oMRMsg,
            poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(),  // RegisterID
            poMessage->u16GetCmdCounter(),  // CmdCounter
            poMessage->u16GetFunctionID(),  // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT    // Opcode
            );

         if (oMRMsg.bIsValid())
         {

            // Post message
            ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

            if (enResult != AIL_EN_N_NO_ERROR)
            {
               // ETG_TRACE_ERR(( "
               //	vHandle_RequestSlicePhoneBookList_MethodStart:
               //	'MethodResult', enPostMessage() failed,
               //	'ail_tenCommunicationError' = %u",
               //          (tU32)enResult ));

               // We can come here e.g. if the client has changed his application
               // state to OFF in the meantime. Therefore we dont throw an assert.
               // NORMAL_M_ASSERT_ALWAYS();
            }
         }
         else
         {
            most_phonbkfi_tclMsgRequestSliceCallHistoryListError hisotryListError;
            vHandleMethodError(hisotryListError,poMessage);
            hisotryListError.vDestroy();
         }
      }
      oRequestSliceCallHistoryList_MR.vDestroy();
   }
   else
   {
      // If we come here, we assume the client has sent an erroneous message,
      // probably it has used a wrong FI_MAJOR_VERSION. Therefore we will
      // send back a an error message and throw a NORMAL_M_ASSERT_ALWAYS().

      vHandleGetDataError(poMessage,"vHandle_RequestSliceCallHistoryList_MethodStart");
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)

   oRequestSliceCallHistoryList_MS.vDestroy();




}


bool fc_phonebook_tclService_PhoneBook::GetExistingDeviceCapability(FC_PhoneBook_FeatureSupport_Detail& fs, tU8 u8DeviceHandle)
{
   ETG_TRACE_USR4(("GetExistingDeviceCapability entered  "));
   bool returnVal =  false;
   FC_Device_Details* pDeviceObject=
      FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);


   FC_PhoneBook_SQLite* poFC_PhoneBook_SQLite =
      FC_PhoneBook_SQLite::GetSQLiteInstance();

   if((poFC_PhoneBook_SQLite!=NULL) && (pDeviceObject != NULL))
   {
      quint16 u16ErrCode=0;
      fs.m_address = fs.m_address+ pDeviceObject->getDeviceAddress();
      fs.m_devicehandle = u8DeviceHandle;
      ETG_TRACE_USR4(("CALLING...GetFeatureSupportforDevice"));
      returnVal=poFC_PhoneBook_SQLite->GetFeatureSupportforDevice(fs.m_address,fs.m_devicehandle ,fs,u16ErrCode);
      if(returnVal==true)
      {
         ETG_TRACE_USR4(("DEVICE FOUND ... FEATURE SUPPORT"));
      }
   }
   return returnVal;
}


/*******************************************************************************
** FUNCTION:   vHandle_CreateCallHistoryList_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for CreateCallHistoryList method
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_CreateCallHistoryList_MethodStart( amt_tclServiceData* poMessage)
{
   tBool dbStatus = FALSE;
   ETG_TRACE_USR4(("	MethodStart:CreateCallHistory : entered"));
   // Extract the Call instance and other required info from poMessage
   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );

   // Create the (message related) FI data object
   most_phonbkfi_tclMsgCreateCallHistoryListMethodStart oCreateCallHistoryList_MS;    

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data object.
   if ( oVisitorMsg.s32GetData(oCreateCallHistoryList_MS, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      most_phonbkfi_tclMsgCreateCallHistoryListMethodResult oCreateCallHistoryList_MR;
      FC_Device_Details
            *p_checkObject =
                  FC_Device_Details::deviceHandleToObjectInstance(oCreateCallHistoryList_MS.u8DeviceHandle);

      ETG_TRACE_USR2(("	DEVICE HANDLE: %u", oCreateCallHistoryList_MS.u8DeviceHandle));

      ETG_TRACE_USR1((" Call History Type:: %u",ETG_ENUM(TR_PBKCHTYPE,oCreateCallHistoryList_MS.e8CallHistoryType.enType)));
      ETG_TRACE_USR1((" Call History Sort Type:: %u",ETG_ENUM(TR_CALLHISTORYSORTTYPE,oCreateCallHistoryList_MS.e8CallHistorySortType.enType)));

      if( oCreateCallHistoryList_MS.u8DeviceHandle != FC_PB_VEHICLE_PBOOK)
      {
         ETG_TRACE_USR4(("Contact List : Check device object "));

         if( p_checkObject != NULL)
         {
            ETG_TRACE_USR2(("Contact List : Device object exists for device handle = %u ", oCreateCallHistoryList_MS.u8DeviceHandle));
            tU8 deviceCapabilities = p_checkObject->getDeviceCapabilities();

            // Check for the Combined call history type
            if (oCreateCallHistoryList_MS.e8CallHistoryType.enType
                  == most_fi_tcl_e8_PhonBkCallHistoryType::FI_EN_E8CCH)
            {
               ETG_TRACE_USR4((" Call History Type : FC_PhoneBook_SQLite::callhisttype_combined "));

               if ( ( ((deviceCapabilities & MissedCallHistory) == MissedCallHistory)
                     && ((deviceCapabilities & IncomingCallHistory) == IncomingCallHistory)
                     && ((deviceCapabilities & OutgoingCallHistory) == OutgoingCallHistory) )
                  || (p_checkObject->Device_DateTime_Support.CHDateTimeSupport == true) )
               {
                  if (m_pPhoneBookDB)
                  {
                     // Call CreateCallHistoryList handler function from FC_PhoneBook_Database
                     dbStatus
                           = m_pPhoneBookDB->CreateCallHistoryList(&oCreateCallHistoryList_MS, &oCreateCallHistoryList_MR, poMessage->u16GetRegisterID());//FIX GMMY 15-1578 RegisterID is passed to identify the requested client
                  }
               }
               else
               {
                  ETG_TRACE_USR4(("Both DateTime and Combined call history not supported "));

                  FC_PhoneBook_FeatureSupport_Detail fs;
                  if (GetExistingDeviceCapability(fs, oCreateCallHistoryList_MS.u8DeviceHandle))
                  {
                     if (fs.m_ichsupport && fs.m_mchsupport && fs.m_ochsupport)
                     {
                        if (m_pPhoneBookDB)
                        {
                           dbStatus
                                 = m_pPhoneBookDB->CreateCallHistoryList(&oCreateCallHistoryList_MS, &oCreateCallHistoryList_MR, poMessage->u16GetRegisterID());//FIX GMMY 15-1578 RegisterID is passed to identify the requested client
                        }
                     }
                     else
                     {
                        ETG_TRACE_USR4(("Device not known Both DateTime and Combined call history not supported and Device is new"));
                     }

                  }
                  else
                  {
                     ETG_TRACE_USR4(("Both DateTime and Combined call history not supported and Device is new"));
                  }
               }
            }
            else
            {
               ETG_TRACE_USR4((" Call History Type : INcoming, outgoing, missed "));

               if (m_pPhoneBookDB)
               {
                  // Call CreateCallHistoryList handler function from FC_PhoneBook_Database
                  dbStatus
                        = m_pPhoneBookDB->CreateCallHistoryList(&oCreateCallHistoryList_MS, &oCreateCallHistoryList_MR, poMessage->u16GetRegisterID());//FIX GMMY 15-1578 RegisterID is passed to identify the requested client
               }
            }

            // Get the corresponding Call Instance

            if (dbStatus != TRUE)
            {
               amt_tclServiceDataError
                     oErrorMsg(*poMessage, most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC);
               // Post message
               ail_tenCommunicationError enResult =
                     _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

               if (enResult != AIL_EN_N_NO_ERROR)
               {
                  ETG_TRACE_ERR(("vHandle_CreateCallHistoryList_MethodStart: 'ERROR', enPostMessage() failed "));
               }

               // NORMAL_M_ASSERT_ALWAYS();
            }
            else
            {
               fi_tclVisitorMessage
                     oMRMsg(oCreateCallHistoryList_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
               // Here the message related header information is added and by this the
               // creation of the message is completed.
               vInitServiceData(oMRMsg, poMessage->u16GetSourceAppID(), // ServiceDataMsg
               poMessage->u16GetRegisterID(), // RegisterID
               poMessage->u16GetCmdCounter(), // CmdCounter
               poMessage->u16GetFunctionID(), // Function ID
               AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
               );

               if (oMRMsg.bIsValid())
               {
                  // Post message
                  ail_tenCommunicationError enResult =
                        _poMainAppl->enPostMessage(&oMRMsg, TRUE);

                  if (enResult != AIL_EN_N_NO_ERROR)
                  {
                     ETG_TRACE_ERR(( "vHandle_CreateCallHistoryList_MethodStart: 'MethodResult', enPostMessage() failed "
                                 "'ail_tenCommunicationError'= %u", (tU32)enResult ));

                     // We can come here e.g. if the client has changed his application
                     // state to OFF in the meantime. Therefore we dont throw an assert.
                  }
               }
               else
               {
                  most_phonbkfi_tclMsgCreateCallHistoryListError
                        createCallHistoryError;
                  vHandleMethodError(createCallHistoryError, poMessage);
                  createCallHistoryError.vDestroy();
               }
            }

         }
         else
         {
            ETG_TRACE_USR2(("Contact List : Device object do not exists for device handle = %u ", oCreateCallHistoryList_MS.u8DeviceHandle));

            amt_tclServiceDataError oErrorMsg(*poMessage, CCA_C_U16_ERROR_PARAMETER_OUT_OF_RANGE);
            // Post message
            ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

            if (enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_ERR(("vHandle_CreateCallHistoryList_MethodStart: 'ERROR', enPostMessage() failed"));
            }
            else
            {
               ETG_TRACE_USR4(("vHandle_CreateCallHistoryList_MethodStart: 'SUCCESS', enPostMessage()"));
            }
            // NORMAL_M_ASSERT_ALWAYS();
         }
      }
      oCreateCallHistoryList_MR.vDestroy();
   }
   else
   {
      // If we come here, we assume the client has sent an erroneous message,
      // probably it has used a wrong FI_MAJOR_VERSION. Therefore we will
      // send back a an error message and throw a NORMAL_M_ASSERT_ALWAYS().

      vHandleGetDataError(poMessage,"vHandle_CreateCallHistoryList_MethodStart");
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)
   oCreateCallHistoryList_MS.vDestroy();
}


/*******************************************************************************
** FUNCTION:   vHandle_ReleaseCallHistoryList_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for ReleaseCallHistoryList method
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::\
vHandle_ReleaseCallHistoryList_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("MethodStart:ReleasePhoneBookList entered"));
   // Extract the Call instance and other required info from poMessage */
   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );

   // Create the (message related) FI data object
   most_phonbkfi_tclMsgReleaseCallHistoryListMethodStart \
      oReleaseCallHistoryListMethodStart_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.

   if ( oVisitorMsg.s32GetData(oReleaseCallHistoryListMethodStart_MS, \
      MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION)
      != OSAL_ERROR)
   {



      ETG_TRACE_USR2(("list HANDLE: %u",oReleaseCallHistoryListMethodStart_MS.u16ListHandle));


      most_phonbkfi_tclMsgReleaseCallHistoryListMethodResult \
         ooReleaseCallHistoryList_MR;

      //ReleasePhoneBookList call from Sqlite data base//
      tBool dbStatus = FALSE;

      if(m_pPhoneBookDB)
      {
      dbStatus = m_pPhoneBookDB->ReleaseCallHistoryList( &oReleaseCallHistoryListMethodStart_MS, \
         &ooReleaseCallHistoryList_MR, \
         poMessage->u16GetRegisterID());//FIX GMMY 15-1578 RegisterID is passed to identify the requested client
      }

      if(dbStatus != TRUE)
      {
         most_phonbkfi_tclMsgReleaseCallHistoryListError releaseCallHistoryError;
         vHandleMethodError(releaseCallHistoryError,poMessage);
         releaseCallHistoryError.vDestroy();
      }
      else
      {
         fi_tclVisitorMessage oMRMsg(ooReleaseCallHistoryList_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
         // Here the message related header information is added and by this the
         // creation of the message is completed.
         vInitServiceData( oMRMsg,
            poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(),  // RegisterID
            poMessage->u16GetCmdCounter(),  // CmdCounter
            poMessage->u16GetFunctionID(),  // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT     // Opcode
            );

         if (oMRMsg.bIsValid())
         {

            // Post message
            ail_tenCommunicationError enResult =
               _poMainAppl->enPostMessage(&oMRMsg, TRUE);

            if (enResult != AIL_EN_N_NO_ERROR)
            {
               //                 ETG_TRACE_ERR(( "vHandle_ReleasePhoneBookList_MethodStart:
               //				 'MethodResult', enPostMessage() failed,
               //				 'ail_tenCommunicationError' = %u",
               //                                    (tU32)enResult ));*/
               // We can come here e.g. if the client has changed his application
               // state to OFF in the meantime. Therefore we dont throw an assert.
               // NORMAL_M_ASSERT_ALWAYS();
            }

         }
         else
         {
            most_phonbkfi_tclMsgReleaseCallHistoryListError releaseCallHistoryError;
            vHandleMethodError(releaseCallHistoryError,poMessage);
            releaseCallHistoryError.vDestroy();
         }

      }
      ooReleaseCallHistoryList_MR.vDestroy();
   }
   else
   {
      // If we come here, we assume the client has sent an erroneous message,
      // probably it has used a wrong FI_MAJOR_VERSION. Therefore we will
      // send back a an error message and throw a NORMAL_M_ASSERT_ALWAYS().

      vHandleGetDataError(poMessage,"vHandle_ReleaseCallHistoryList_MethodStart");
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)

   oReleaseCallHistoryListMethodStart_MS.vDestroy();




}

/*******************************************************************************
** FUNCTION:   vHandle_GetDeviceCallHistorySupport_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for GetDeviceCallHistorySupport method
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_GetDeviceCallHistorySupport_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("MethodStart: GetDeviceCallHistorySupport_MS : entered"));

   // Extract the Device Handle and other required info from poMessage
   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );

   // Create the (message related) FI data object
   most_phonbkfi_tclMsgGetDeviceCallHistorySupportMethodStart oGetDeviceCallHistorySupport_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.
   if ( oVisitorMsg.s32GetData(oGetDeviceCallHistorySupport_MS, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      ETG_TRACE_USR2((" Device Handle: %u", oGetDeviceCallHistorySupport_MS.u8DeviceHandle));

      FC_Device_Details* p_checkObject = FC_Device_Details::deviceHandleToObjectInstance((INDEX)oGetDeviceCallHistorySupport_MS.u8DeviceHandle);
      if(p_checkObject != NULL)
      {
         most_phonbkfi_tclMsgGetDeviceCallHistorySupportMethodResult oGetDeviceCallHistorySupport_MR;

         tU8 deviceCapabilities = p_checkObject->getDeviceCapabilities();
         if ((deviceCapabilities & MissedCallHistory) == MissedCallHistory)
         {
            oGetDeviceCallHistorySupport_MR.oCallHistoryTypeSupport.bMCHSupport
                  = TRUE;
         }
         if ((deviceCapabilities & IncomingCallHistory) == IncomingCallHistory)
         {
            oGetDeviceCallHistorySupport_MR.oCallHistoryTypeSupport.bICHSupport
                  = TRUE;
         }
         if ((deviceCapabilities & OutgoingCallHistory) == OutgoingCallHistory)
         {
            oGetDeviceCallHistorySupport_MR.oCallHistoryTypeSupport.bOCHSupport
                  = TRUE;
         }
         if ((deviceCapabilities & CombinedCallHistory) == CombinedCallHistory)
         {
            oGetDeviceCallHistorySupport_MR.oCallHistoryTypeSupport.bCCHSupport
                  = TRUE;
         }

         oGetDeviceCallHistorySupport_MR.bCallHistoryDateTimeSupport
               = p_checkObject->Device_DateTime_Support.VcardsHaveDatetime;
         oGetDeviceCallHistorySupport_MR.u8DeviceHandle
               = oGetDeviceCallHistorySupport_MS.u8DeviceHandle;

         //Get the coresponding Call Instance
         fi_tclVisitorMessage oMRMsg(oGetDeviceCallHistorySupport_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);

         vInitServiceData( oMRMsg,
            poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(),  // RegisterID
            poMessage->u16GetCmdCounter(),  // CmdCounter
            poMessage->u16GetFunctionID(),  // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT    // Opcode
            );

         if (oMRMsg.bIsValid())
         {
            // Post message
            ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

            if (enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_ERR(("oGetDeviceCallHistorySupport_MS: 'Result', enPostMessage() failed, 'ail_tenCommunicationError' = %d",(tU32)enResult));
            }
         }
         else
         {
            most_phonbkfi_tclMsgGetDeviceCallHistorySupportError CallHistoryError;
            vHandleMethodError(CallHistoryError,poMessage);
            CallHistoryError.vDestroy();
         }
         oGetDeviceCallHistorySupport_MR.vDestroy();
      }
      else
      {
         ETG_TRACE_USR2(("Contact List : Device object do not exists for device handle = %u ", oGetDeviceCallHistorySupport_MS.u8DeviceHandle));
         amt_tclServiceDataError oErrorMsg(*poMessage, CCA_C_U16_ERROR_PARAMETER_OUT_OF_RANGE);
         // Post message
         ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

         if (enResult != AIL_EN_N_NO_ERROR)
         {
            ETG_TRACE_ERR(("vHandle_GetDeviceCallHistorySupport_MethodStart: 'ERROR', enPostMessage() failed"));
         }
         else
         {
            ETG_TRACE_USR4(("vHandle_GetDeviceCallHistorySupport_MethodStart: 'SUCCESS', enPostMessage()"));
         }
      }
   }
   else
   {
      // If we come here, we assume the client has sent an erroneous message,
      // probably it has used a wrong FI_MAJOR_VERSION. Therefore we will
      // send back a an error message and throw a NORMAL_M_ASSERT_ALWAYS().
      vHandleGetDataError(poMessage,"vHandle_GetDeviceCallHistorySupport_MethodStart");
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)
   oGetDeviceCallHistorySupport_MS.vDestroy();
}


/*******************************************************************************
** FUNCTION:   vPhoneBook_UpdateListChange(..)
*******************************************************************************/
/* DESCRIPTION:
*  Update clients on change of any List data
*  PARAMETERS:
*     IN:   List change information
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vPhoneBook_UpdateListChange
(most_phonbkfi_tclMsgListChangeStatus& oListChangeStatus)
{
   m_oListChangeStatus= oListChangeStatus;

   ETG_TRACE_USR4(("Updating clients about list change entered"));


   ail_tenCommunicationError enResult = eUpdateClients(MOST_PHONBKFI_C_U16_LISTCHANGE);
   if(AIL_EN_N_NO_ERROR != enResult)
   {

      ETG_TRACE_ERR(("ERROR fc_phonebook_tclService_PhoneBook:- Sending of list "
         "change update failed "));
   }
}


tVoid fc_phonebook_tclService_PhoneBook::vHandleGetDataError(
   amt_tclServiceData* poMessage, tCString poHandlerStr)
{
   ETG_TRACE_USR4(("vHandleGetDataError Entered"));
   // Create AMT error message and set ErrorCode
   amt_tclServiceDataError oErrorMsg(*poMessage,
      CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED);
   // Post message
   ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oErrorMsg,
      TRUE);

   if (enResult != AIL_EN_N_NO_ERROR)
   {
      ETG_TRACE_ERR(("ail_tenCommunicationError = %u", enResult));

      // We can come here e.g. if the client has changed his application
      // state to OFF in the meantime. Therefore we don't throw an assert.
      // NORMAL_M_ASSERT_ALWAYS();
   }
   else
   {
      ETG_TRACE_FATAL(("%s: Received message invalid.", poHandlerStr));
      // We never expect to come here.
      // NORMAL_M_ASSERT_ALWAYS();
   }
}


tVoid fc_phonebook_tclService_PhoneBook::vPhoneBook_UpdatePreferredPhoneBookSortOrderStatus
(most_phonbkfi_tclMsgPreferredPhoneBookSortOrderStatus& oPreferredPhoneBookSortOrderStatus)
{
   m_oPreferredPhoneBookSortOrderStatus = oPreferredPhoneBookSortOrderStatus;


   ETG_TRACE_USR4(("Updating clients about PreferredPhoneBookSortOrderStatus entered"));

   eUpdateClients(MOST_PHONBKFI_C_U16_PREFERREDPHONEBOOKSORTORDER);

}


/*******************************************************************************
** FUNCTION:   vPhoneBook_UpdatePreferredPBSortOrderStatus(..)
*******************************************************************************/
/* DESCRIPTION:
*  Update clients on change of PreferredPBSortOrder
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vPhoneBook_UpdatePreferredPBSortOrderStatus()
{
   ETG_TRACE_USR4(("vPhoneBook_UpdatePreferredPBSortOrderStatus entered"));

   quint8 u8SortOrder = 0;
   FC_PhoneBook_SQLite* poFC_PhoneBook_SQLite = FC_PhoneBook_SQLite::GetSQLiteInstance();
   if (poFC_PhoneBook_SQLite != NULL)
   {
      u8SortOrder = poFC_PhoneBook_SQLite->GetPreferredSortOrder();
      ETG_TRACE_USR1((" PreferredSortOrder from database = %u", u8SortOrder));

      if (u8SortOrder == 0)
      {
         m_oPreferredPhoneBookSortOrderStatus.e8PreferredPhoneBookSortOrder.enType
               = most_fi_tcl_e8_PhonBkPreferredPhoneBookSortOrder::FI_EN_E8PREFERRED_SORT_ORDER_FIRSTNAME;
      }
      else
      {
         m_oPreferredPhoneBookSortOrderStatus.e8PreferredPhoneBookSortOrder.enType
               = most_fi_tcl_e8_PhonBkPreferredPhoneBookSortOrder::FI_EN_E8PREFERRED_SORT_ORDER_LASTNAME;
      }
      // Note: The direct re-assignment of "m_oPreferredPhoneBookSortOrderStatus.e8PreferredPhoneBookSortOrder.enType" causes no memory leak as the enum is a basic data type.


      ail_tenCommunicationError enResult = eUpdateClients(MOST_PHONBKFI_C_U16_PREFERREDPHONEBOOKSORTORDER);
      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR((" vPhoneBook_UpdatePreferredPBSortOrderStatus : 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' "));
      }
      else
      {
         ETG_TRACE_USR4((" vPhoneBook_UpdatePreferredPBSortOrderStatus: Updated Clients()"));
      }
   }
}

//CMG3G-14241
/*******************************************************************************
** FUNCTION:   vPhoneBook_UpdatePreferredSortOrderListStatus(..)
*******************************************************************************/
/* DESCRIPTION:
*  Update clients on change of PreferredSortOrderList
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vPhoneBook_UpdatePreferredSortOrderListStatus()
{
   ETG_TRACE_USR4(("Function vPhoneBook_UpdatePreferredSortOrderListStatus entered"));

   if (FC_PhoneBook_SQLite::GetSQLiteInstance())
   {
      m_oPreferredSortOrderListStatus.oPreferredSortOrderList.oItems.clear();

      QList < quint8 > DeviceHandleList;
      QList < quint8 > PreferredSortOrderList;
      FC_PhoneBook_SQLite::GetSQLiteInstance()->bGetPreferredSortOrderList(DeviceHandleList, PreferredSortOrderList);

      for (qint8 ListIndex = VALUE_ZERO; ListIndex < DeviceHandleList.size(); ListIndex++)
      {
         most_fi_tcl_PhonBkPreferredSortOrderListItem oItem;

         oItem.u8DeviceHandle = DeviceHandleList.at(ListIndex);
         ETG_TRACE_USR2(("vPhoneBook_UpdatePreferredSortOrderListStatus DeviceHandle : %u", DeviceHandleList.at(ListIndex)));

         oItem.e8PreferredPhoneBookSortOrder.enType = (most_fi_tcl_e8_PhonBkPreferredPhoneBookSortOrder::tenType) PreferredSortOrderList.at(ListIndex);
         ETG_TRACE_USR2(("vPhoneBook_UpdatePreferredSortOrderListStatus PreferredSortOrder : %u", PreferredSortOrderList.at(ListIndex)));

         m_oPreferredSortOrderListStatus.oPreferredSortOrderList.oItems.push_back(oItem);
      }

      ail_tenCommunicationError enResult = eUpdateClients(MOST_PHONBKFI_C_U16_PREFERREDSORTORDERLIST);
      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR((" vPhoneBook_UpdatePreferredSortOrderListStatus : 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' "));
      }
      else
      {
         ETG_TRACE_USR4((" vPhoneBook_UpdatePreferredSortOrderListStatus: Updated Clients()"));
      }
   }
   else
   {
      ETG_TRACE_ERR((" SQLite instance is NULL!!!"));
   }
}
//End of CMG3G-14241

//CRQ-563636
/*******************************************************************************
** FUNCTION:   vPhoneBook_UpdateMaxContactCount(..)
*******************************************************************************/
/* DESCRIPTION:
*  Update clients on change of Max Contact count to clients
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vPhoneBook_UpdateMaxContactCount(unsigned int MaxContactsCount)
{
   ETG_TRACE_USR4(("Function vPhoneBook_UpdateMaxContactCount entered :%d",MaxContactsCount));

   m_oMaxContactCountStatus.u32MaxContactsCount = MaxContactsCount;

   ail_tenCommunicationError enResult = eUpdateClients(MOST_PHONBKFI_C_U16_MAXCONTACTCOUNT);
   if (AIL_EN_N_NO_ERROR != enResult)
   {
      ETG_TRACE_ERR((" vPhoneBook_UpdateMaxContactCount : 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' "));
   }
   else
   {
      ETG_TRACE_USR4((" vPhoneBook_UpdateMaxContactCount: Updated Clients()"));
   }
}
//End of CRQ-563636

/*******************************************************************************
** FUNCTION:   vDeviceDisconnected(..)
*******************************************************************************/
/* DESCRIPTION: This function is triggered when a device is disconnected
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vDeviceDisconnected(
   tU8 u8ConnectionID)
{
   ETG_TRACE_USR4(("Function vDeviceDisconnected entered with connection ID %d",u8ConnectionID));
   FC_Device_Details* pDeviceObject=
      FC_Device_Details::getDeviceObject((INDEX)(u8ConnectionID-1));

   if(pDeviceObject && m_pPhoneBookDB)
   {
      m_pPhoneBookDB->DeviceDisconnected(pDeviceObject);
      tU8 u8DeviceHandle = pDeviceObject->getDeviceHandle();
      //Erase entry from m_oDevicePhoneBookFeatureSupportExtendedStatus
      for (tU8 u8Index = 0; u8Index
            < m_oDevicePhoneBookFeatureSupportExtendedStatus.oDevicePhoneBookFeatureSupportList.oItems.size(); u8Index++)
      {
         if (u8DeviceHandle
               == m_oDevicePhoneBookFeatureSupportExtendedStatus.oDevicePhoneBookFeatureSupportList.oItems[u8Index].u8DeviceHandle)
         {
            m_oDevicePhoneBookFeatureSupportExtendedStatus.oDevicePhoneBookFeatureSupportList.oItems.erase(m_oDevicePhoneBookFeatureSupportExtendedStatus.oDevicePhoneBookFeatureSupportList.oItems.begin()
                  + u8Index);
            ETG_TRACE_USR4((" Erased entry for device handle: %d", pDeviceObject->getDeviceHandle()));
         }
      }

      //Update device phonebook feature support list property to clients
      ail_tenCommunicationError
            enResultExtended =
                  eUpdateClients(MOST_PHONBKFI_C_U16_DEVICEPHONEBOOKFEATURESUPPORTEXTENDED);

      if (AIL_EN_N_NO_ERROR != enResultExtended)
      {
         ETG_TRACE_ERR((" DevicePhonebookFeatureSupportExtended Status: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' = %d", (tU32) enResultExtended));
      }
      else
      {
         ETG_TRACE_USR4((" DevicePhonebookFeatureSupportExtended status sent"));
      }

      //fix for CMG3G-11146
      //Removing the disconnected device handle from the downloadStateExtendedStream
      for (tU8 u8Index = 0; u8Index
            < m_oDownloadStateExtended.oDownloadStateExtendedStream.oItems.size(); u8Index++)
      {
         if (u8DeviceHandle
               == m_oDownloadStateExtended.oDownloadStateExtendedStream.oItems[u8Index].u8DeviceHandle)
         {
            m_oDownloadStateExtended.oDownloadStateExtendedStream.oItems.erase(m_oDownloadStateExtended.oDownloadStateExtendedStream.oItems.begin()
                  + u8Index);
            ETG_TRACE_USR4((" Erased the entry from downloadExtrendedStream for the disconnected device handle %d", pDeviceObject->getDeviceHandle()));
            break;
         }
      }

      //Update download state extended property to Client
      enResultExtended = eUpdateClients(
         MOST_PHONBKFI_C_U16_DOWNLOADSTATEEXTENDED);
      if (AIL_EN_N_NO_ERROR != enResultExtended)
      {
         ETG_TRACE_ERR((" ERROR fc_phonebook_tclService_PhoneBook:- Sending of Download State Extended update failed "));
      }
      else
      {
         ETG_TRACE_USR4((" Sending of Download State Extended Property Successfull"));
      }

      //Removing the disconnected device handle from the downloadStateStream
      for (tU8 u8Index = 0; u8Index
            < m_oDownloadState.oDownloadStateStream.oItems.size(); u8Index++)
      {
         if (u8DeviceHandle
               == m_oDownloadState.oDownloadStateStream.oItems[u8Index].u8DeviceHandle)
         {
            m_oDownloadState.oDownloadStateStream.oItems.erase(m_oDownloadState.oDownloadStateStream.oItems.begin()
                  + u8Index);
            ETG_TRACE_USR4((" Erased the entry from downloadStream for the disconnected device handle %d", pDeviceObject->getDeviceHandle()));
            break;
         }
      }

      //Update Download state property to clients.
      ail_tenCommunicationError enResult =
            eUpdateClients(MOST_PHONBKFI_C_U16_DOWNLOADSTATE);
      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR((" Sending of Download State update failed "));
      }
      else
      {
         ETG_TRACE_USR4((" Sending of Download State Property Successfull"));
      }
      //End of fix CMG3G-11146

      //Fix 159808
      //Removing the disconnected device handle from the DownloadProgressList stream
      for (tU8 u8Index = 0; u8Index < (tU8)(m_oDownloadProgressListStatus.oDownloadProgressList.oItems.size()); u8Index++)
      {
         if (u8DeviceHandle
               == m_oDownloadProgressListStatus.oDownloadProgressList.oItems[u8Index].u8DeviceHandle)
         {
            m_oDownloadProgressListStatus.oDownloadProgressList.oItems.erase(m_oDownloadProgressListStatus.oDownloadProgressList.oItems.begin() + u8Index);
            ETG_TRACE_USR4((" Erased the entry from DownloadProgressList for the disconnected device handle %d", pDeviceObject->getDeviceHandle()));
            break;
         }
      }

      //Update DownloadProgressList property to clients.
      enResult = eUpdateClients(MOST_PHONBKFI_C_U16_DOWNLOADPROGRESSLIST);
      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR((" Sending of Download progress list update failed "));
      }
      else
      {
         ETG_TRACE_USR4((" Sending of Download progress list Property Successfull"));
      }
      //End of fix 159808


      //Fix CMG3G-12350
      for (tU8 u8Index = 0; u8Index < (tU8) (m_oMissedCallIndicatorListStatus.oMissedCallIndicatorList.oItems.size()); u8Index++)
      {
         if (u8DeviceHandle == m_oMissedCallIndicatorListStatus.oMissedCallIndicatorList.oItems[u8Index].u8DeviceHandle)
         {
            ETG_TRACE_USR4(("Removing disconnected device Item"));

            m_oMissedCallIndicatorListStatus.oMissedCallIndicatorList.oItems.erase(m_oMissedCallIndicatorListStatus.oMissedCallIndicatorList.oItems.begin() + u8Index);

            //Update MissedCallIndicatorListStatus property to clients.
            enResult = eUpdateClients(MOST_PHONBKFI_C_U16_MISSEDCALLINDICATORLIST);
            if (AIL_EN_N_NO_ERROR != enResult)
            {
               ETG_TRACE_ERR((" Sending of MissedCallIndicatorListStatus update failed "));
            }
            else
            {
               ETG_TRACE_USR4((" Sending of MissedCallIndicatorListStatus Property Successfull"));
            }
            break;
         }
      }
      //End of Fix CMG3G-12350
      vClearMissedCalls(u8DeviceHandle, pDeviceObject); //244511- Chery Bug Fix - Clearing the Missed call list upon device disconnection


      //Fix : CMG3G-12880
      //Remove the devicehandle from probably locked devices list
      vRemoveFromDeviceLockedList(u8DeviceHandle);
      //End of fix CMG3G-12880

      //Fix : NCG3D-65982
      //Remove the devicehandle from suspended download list
      vRemoveFromSuspendedDownloadList(u8DeviceHandle);
      //End of fix NCG3D-65982

      m_oPbk_DnlManager.PhoneBook_vStopRetry(FC_PhoneBook_RetryDetails::RETRY_ANY_MESSAGE, u8ConnectionID);
      m_oPbk_DnlManager.vDeviceDisconnected(u8ConnectionID);
      FC_Device_Details::deleteDeviceObject((INDEX) (u8ConnectionID - 1));
   }
   else
   {
      ETG_TRACE_ERR(("vDeviceDisconnected: Device object is NULL"));
   }
}


/*******************************************************************************
** FUNCTION:   vDeviceDeleted(..)
*******************************************************************************/
/* DESCRIPTION: This function is triggered when a paired device is deleted( unparied)
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vDeviceDeleted(tU8 u8DeviceHandle)
{
   ETG_TRACE_USR2(("Function vDeviceDeleted entered with DeviceHandle %d",u8DeviceHandle));

   if(m_pPhoneBookDB)
   {
      m_pPhoneBookDB->DeviceDeleted(u8DeviceHandle);
   }

   if(m_pPhoneBookMemoryUsage)
   {        
      m_pPhoneBookMemoryUsage->CalculateMemoryUsageFromDB();
   }
}


/*******************************************************************************
** FUNCTION:   ptrGetDatabaseInstance()
*******************************************************************************/
/* DESCRIPTION:
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
FC_PhoneBook_Database* fc_phonebook_tclService_PhoneBook::poGetDatabaseInstance()
{

   ETG_TRACE_USR4(("Function poGetDatabaseInstance entered"));
   return m_pPhoneBookDB;
}


/*******************************************************************************
** FUNCTION:   fc_phonebook_tclService_PhoneBook::vHandle_Store_Favorite_MethodStart()
*
*  DESCRIPTION: Store favorite method start
*
*  PARAMETERS:
*
*  RETURNVALUE: None
********************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_Store_Favorite_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4((" MethodStart: vHandle_Store_Favorite_MethodStart entered"));

   tBool result = false;
   tU16 u16ErrorCode=0;
   tU8 u8ErrorInfo=0;

   fi_tclVisitorMessage oVisitorMsg( poMessage );
   ail_tenCommunicationError enResult = AIL_EN_N_NO_ERROR;

   // Create the (message related) FI data object
   most_phonbkfi_tclMsgStoreFavoriteMethodStart  oStoreFavoriteMethodStart_MS;
   most_phonbkfi_tclMsgStoreFavoriteMethodResult oStoreFavoriteMethodStart_MR;

   if ( oVisitorMsg.s32GetData(oStoreFavoriteMethodStart_MS, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR )
   {
      if(m_pPhoneBookDB)
      {
         result = m_pPhoneBookDB->StoreFavorite(&oStoreFavoriteMethodStart_MS,
                                                &oStoreFavoriteMethodStart_MR, u16ErrorCode, u8ErrorInfo);   // Fix GMMY16-13959
      }

      if(result != false)
      {
         fi_tclVisitorMessage oMRMsg(oStoreFavoriteMethodStart_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);

         vInitServiceData( oMRMsg,
            poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(),  // RegisterID
            poMessage->u16GetCmdCounter(),  // CmdCounter
            poMessage->u16GetFunctionID(),  // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT       // Opcode
            );

         if(!oMRMsg.bIsValid())
         {
            result = false;
         }
         if (result != false)
         {
            enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

            if (enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_ERR(("1vHandle_Store_Favorite_MethodStart: 'ERROR', enPostMessage() failed"));
               // NORMAL_M_ASSERT_ALWAYS();
            }
            else
            {
               ETG_TRACE_USR4(("1vHandle_Store_Favorite_MethodStart: 'SUCCESS', enPostMessage() SUCCESS"));
               if(m_pPhoneBookDB)
               {
                  m_pPhoneBookDB->MethodResultPosted(poMessage->u16GetFunctionID(),
                                                     &oStoreFavoriteMethodStart_MS, &oStoreFavoriteMethodStart_MR);
               }
            }
         }
         else
         {
            // Fix GMMY16-13959
            vPostStoreFavoriteError(poMessage, most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);
         }
      }
      else
      {
         // Fix GMMY16-13959
         vPostStoreFavoriteError(poMessage, u16ErrorCode, u8ErrorInfo);
      }
      oStoreFavoriteMethodStart_MR.vDestroy();
   }
   else
   {
      vHandleGetDataError(poMessage,"vHandle_Store_Favorite_MethodStart");
   }
   oStoreFavoriteMethodStart_MS.vDestroy();
}


tVoid fc_phonebook_tclService_PhoneBook::vMakeFavoriteAvaialble(quint8 u8DeviceHandle)
{
   if(m_pPhoneBookDB)
   {
      ETG_TRACE_USR4(("vMakeFavoriteAvaialble DeviceHandle == %u",u8DeviceHandle));
      m_pPhoneBookDB->MakeFavoritesAvailable(u8DeviceHandle);
   }
}


tVoid fc_phonebook_tclService_PhoneBook::vMakeFavoriteUnAvaialble(quint8 u8DeviceHandle)
{
   if(m_pPhoneBookDB)
   {
      m_pPhoneBookDB->MakeFavoritesUnAvailable(u8DeviceHandle);
   }
}


/*******************************************************************************
** FUNCTION:   fc_phonebook_tclService_PhoneBook::vHandle_Activate_Favorite_MethodStart()
*
*  DESCRIPTION: Active favorite method start
*
*  PARAMETERS:
*
*  RETURNVALUE: None
********************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_Activate_Favorite_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("MethodStart: vHandle_Activate_Favorite_MethodStart: entered"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);
   bool blFavError = false;
   tU8 u8ErrorPosition = 0;

   // Create the (message related) FI data object
   most_phonbkfi_tclMsgActivateFavoriteMethodStart oActivateFavoriteMethodStart_MS;

   if (oVisitorMsg.s32GetData(oActivateFavoriteMethodStart_MS, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION)
      != OSAL_ERROR)
   {

      if (m_oActivateFavMap.find(oActivateFavoriteMethodStart_MS.u16FavoriteID)
         == m_oActivateFavMap.end())
      {
         IncomingMessage ActivateFavoriteMsg;

         ActivateFavoriteMsg.u16SourceAppID = poMessage->u16GetSourceAppID();
         ActivateFavoriteMsg.u16RegisterID  = poMessage->u16GetRegisterID();
         ActivateFavoriteMsg.u16CmdCounter  = poMessage->u16GetCmdCounter();
         ActivateFavoriteMsg.u16FunctionID  = poMessage->u16GetFunctionID();

         m_oActivateFavMap[oActivateFavoriteMethodStart_MS.u16FavoriteID] = ActivateFavoriteMsg;
         // Fix GMMY16-13959
         if (TRUE == bCheckAudioChannel(oActivateFavoriteMethodStart_MS.e8AudioChannel.enType))   //Fix for CMG3GB-1093
         {
            if (oActivateFavoriteMethodStart_MS.e8VideoChannel.enType
               == most_fi_tcl_e8_FavPrvVideoChannel::FI_EN_E8LC_NONE)   //Fix for CMG3GB-1093
            {
               if (m_poFavoriteProvider)
               {
                  m_poFavoriteProvider->Activate_Favorite(&oActivateFavoriteMethodStart_MS);
               }
            }

            else
            {
               ETG_TRACE_USR4(("Activate_Favorite Error Parameter e8VideoChannel is Wrong"));
               blFavError = true;
               u8ErrorPosition = PARAM_ERR_POSITION_4;
            }
         }
         else
         {
            ETG_TRACE_USR4(("Activate_Favorite Error Parameter e8AudioChannel is Wrong"));
            blFavError = true;
            u8ErrorPosition = PARAM_ERR_POSITION_3;
         }

         if(true == blFavError)
         {
            vPostActivateFavoriteError(most_fi_tcl_e8_ErrorCode::FI_EN_PARAMETERWRONG_OUTOFRANGE, oActivateFavoriteMethodStart_MS.u16FavoriteID, u8ErrorPosition);
         }
      }
      else
      {
         ETG_TRACE_USR4(("Activation of favorite id -%d, already in progress", oActivateFavoriteMethodStart_MS.u16FavoriteID));
         vPostActivateFavoriteError(most_fi_tcl_e8_ErrorCode::FI_EN_PARAMETERNOTAVAILABLE, oActivateFavoriteMethodStart_MS.u16FavoriteID, PARAM_ERR_POSITION_2);
      }

   }
   else
   {
      vHandleGetDataError(poMessage, "vHandle_Activate_Favorite_MethodStart");
   }
   oActivateFavoriteMethodStart_MS.vDestroy();
}


/*******************************************************************************
** FUNCTION:   fc_phonebook_tclService_PhoneBook::vHandle_Delete_Favorite_MethodStart()
*
*  DESCRIPTION: delete favorite method start
*
*  PARAMETERS:
*
*  RETURNVALUE: None
********************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_Delete_Favorite_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("	MethodStart: vHandle_Delete_Favorite_MethodStart entered  "));

   tBool result = false;
   quint16 u16ErrCode =0;

   fi_tclVisitorMessage oVisitorMsg( poMessage );
   ail_tenCommunicationError enResult = AIL_EN_N_NO_ERROR;

   // Create the (message related) FI data object
   most_phonbkfi_tclMsgDeleteFavoriteMethodStart oDeleteFavoriteMethodStart_MS;
   most_phonbkfi_tclMsgDeleteFavoriteMethodResult oDeleteFavoriteMethodStart_MR;

   if ( oVisitorMsg.s32GetData(oDeleteFavoriteMethodStart_MS, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) \
      != OSAL_ERROR)
   {


      //TODO:
      // Implementation of Delete_Favorite_Method
      //HOTFIX 047 
      if(m_poFavoriteProvider)
         result = m_poFavoriteProvider->Delete_Favorite(&oDeleteFavoriteMethodStart_MS,&oDeleteFavoriteMethodStart_MR, u16ErrCode);

      if(result != false)
      {
         /* Get the coresponding Call Instance */
         fi_tclVisitorMessage oMRMsg(oDeleteFavoriteMethodStart_MR,\
            MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
         vInitServiceData( oMRMsg,
            poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(),  // RegisterID
            poMessage->u16GetCmdCounter(),  // CmdCounter
            poMessage->u16GetFunctionID(),  // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT       // Opcode
            );

         if(!oMRMsg.bIsValid())
         {
            result = false;
         }
         if (result != false)
         {
            enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);
            if (enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_USR4((" vHandle_Delete_Favorite_MethodStart: 'ERROR', enPostMessage() failed "));

               // NORMAL_M_ASSERT_ALWAYS();
            }
            else
            {
               ETG_TRACE_USR4((" vHandle_Delete_Favorite_MethodStart: 'SUCCESS', enPostMessage() SUCCESS  "));

               //HOTFIX 047
               if(m_pPhoneBookDB)
                  m_pPhoneBookDB->MethodResultPosted(poMessage->u16GetFunctionID(),
                  &oDeleteFavoriteMethodStart_MS, &oDeleteFavoriteMethodStart_MR);
            }
         }
         else
         {
            most_phonbkfi_tclMsgDeleteFavoriteError deleteFavError;
            vHandleMethodError(deleteFavError,poMessage);
            deleteFavError.vDestroy();
         }
      }
      else
      {
         // Create AMT error message and set ErrorCode
         amt_tclServiceDataError oErrorMsg(*poMessage,u16ErrCode);
         enResult = _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

         if (enResult != AIL_EN_N_NO_ERROR)
         {
            ETG_TRACE_USR4(("  vHandle_Delete_Favorite_MethodStart: 'ERROR', enPostMessage() failed "));

         }
         else
         {
            ETG_TRACE_USR4(("  vHandle_Delete_Favorite_MethodStart: 'SUCCESS', enPostMessage() SUCCESS  "));

            // NORMAL_M_ASSERT_ALWAYS();
         }
         oDeleteFavoriteMethodStart_MR.vDestroy();
      }
   }    else
   {
      vHandleGetDataError(poMessage,"vHandle_Delete_Favorite_MethodStart");
   }
   oDeleteFavoriteMethodStart_MS.vDestroy();



}

/*******************************************************************************
** FUNCTION:   fc_phonebook_tclService_PhoneBook::vHandle_Get_FavoriteInfo_MethodStart()
*
*  DESCRIPTION: Get favorite info method start
*
*  PARAMETERS:
*
*  RETURNVALUE: None
********************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_Get_FavoriteInfo_MethodStart(amt_tclServiceData* poMessage)
{
   tBool result = false;
   tU16 u16ErrorCode = 0;   //Fix GMMY16-13959
   tU8 u8ErrorInfo = 0;   //Fix GMMY16-13959

   ETG_TRACE_USR4(("MethodStart: vHandle_Get_FavoriteInfo_MethodStart entered"));

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);
   ail_tenCommunicationError enResult = AIL_EN_N_NO_ERROR;

   // Create the (message related) FI data object
   most_phonbkfi_tclMsgGetFavoriteInfoMethodStart oGetFavoriteInfoMethodStart_MS;
   most_phonbkfi_tclMsgGetFavoriteInfoMethodResult oGetFavoriteInfoMethodStart_MR;

   if (oVisitorMsg.s32GetData(oGetFavoriteInfoMethodStart_MS, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION)
      != OSAL_ERROR)
   {
      //Fix GMMY16-13959
      if (m_pPhoneBookDB)
         result
         = m_pPhoneBookDB->GetFavoriteInfo(&oGetFavoriteInfoMethodStart_MS, &oGetFavoriteInfoMethodStart_MR, u16ErrorCode, u8ErrorInfo);

      if (result != false)
      {
         /* Get the coresponding Call Instance */
         fi_tclVisitorMessage
            oMRMsg(oGetFavoriteInfoMethodStart_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
         vInitServiceData(oMRMsg, poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(), // RegisterID
            poMessage->u16GetCmdCounter(), // CmdCounter
            poMessage->u16GetFunctionID(), // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
            );

         if (!oMRMsg.bIsValid())
         {
            result = false;
         }
         if (result != false)
         {
            enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

            if (enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_USR4(("  vHandle_Get_FavoriteInfo_MethodStart: 'ERROR', enPostMessage() failed "));

               // NORMAL_M_ASSERT_ALWAYS();
            }
            else
            {
               ETG_TRACE_USR4(("  vHandle_Get_FavoriteInfo_MethodStart: 'SUCCESS', enPostMessage() SUCCESS  "));

            }
         }
         else
         {
            vPostGetFavoriteInfoError(poMessage, most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE);   //Fix GMMY16-13959
         }
         oGetFavoriteInfoMethodStart_MR.vDestroy();
      }
      else
      {
         vPostGetFavoriteInfoError(poMessage, u16ErrorCode, u8ErrorInfo);
      }

   }
   else
   {
      vHandleGetDataError(poMessage, "vHandle_Get_FavoriteInfo_MethodStart");
   }

   oGetFavoriteInfoMethodStart_MS.vDestroy();
}

/*******************************************************************************
** FUNCTION:   fc_phonebook_tclService_PhoneBook::vHandle_Reset_Favorites_MethodStart()
*
*  DESCRIPTION: Reset favorite method start
*
*  PARAMETERS:
*
*  RETURNVALUE: None
********************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_Reset_Favorites_MethodStart(amt_tclServiceData* poMessage)
{
   tBool result = false;

   ETG_TRACE_USR4(("MethodStart: vHandle_Reset_Favorites_MethodStart entered "));

   fi_tclVisitorMessage oVisitorMsg( poMessage );
   ail_tenCommunicationError enResult = AIL_EN_N_NO_ERROR;

   // Create the (message related) FI data object
   most_phonbkfi_tclMsgResetFavoritesMethodStart oResetFavoriteMethodStart_MS;
   most_phonbkfi_tclMsgResetFavoritesMethodResult oResetFavoriteMethodStart_MR;

   if ( oVisitorMsg.s32GetData(oResetFavoriteMethodStart_MS, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) \
      != OSAL_ERROR)
   {


      if(m_poFavoriteProvider)
         result = m_poFavoriteProvider->Reset_Favorites();

      if(result != false)
      {
         fi_tclVisitorMessage oMRMsg(oResetFavoriteMethodStart_MR,\
            MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);

         // creation of the message is completed.
         vInitServiceData( oMRMsg,
            poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(),  // RegisterID
            poMessage->u16GetCmdCounter(),  // CmdCounter
            poMessage->u16GetFunctionID(),  // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT       // Opcode
            );

         if(!oMRMsg.bIsValid())
         {
            result = false;
         }
         if (result != false)
         {
            enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

            if (enResult != AIL_EN_N_NO_ERROR)
            {

               ETG_TRACE_ERR(("vHandle_Reset_Favorites_MethodStart: 'ERROR', enPostMessage() failed"));
               // NORMAL_M_ASSERT_ALWAYS();
            }
            else
            {

               ETG_TRACE_USR4(("vHandle_Reset_Favorites_MethodStart: 'SUCCESS', enPostMessage() SUCCESS"));
            }
         }
         else
         {
            most_phonbkfi_tclMsgResetFavoritesError resetFavError;
            vHandleMethodError(resetFavError,poMessage);
            resetFavError.vDestroy();
         }
      }
      else
      {
         // Create AMT error message and set ErrorCode
         amt_tclServiceDataError oErrorMsg(*poMessage,CCA_C_U16_ERROR_INTERNAL_FAILURE);
         enResult = _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

         if (enResult != AIL_EN_N_NO_ERROR)
         {
            ETG_TRACE_ERR((" vHandle_Reset_Favorites_MethodStart: 'ERROR', enPostErrMessage() failed "));

         }
         else
         {
            ETG_TRACE_USR4(("  vHandle_Reset_Favorites_MethodStart: 'SUCCESS', enPostErrMessage() SUCCESS  "));

         }
         // NORMAL_M_ASSERT_ALWAYS();
      }
      oResetFavoriteMethodStart_MR.vDestroy();
   }
   else
   {
      vHandleGetDataError(poMessage,"vHandle_Reset_Favorites_MethodStart");
   }
   oResetFavoriteMethodStart_MS.vDestroy();



}

/*******************************************************************************
** FUNCTION:   vHandle_GetContactPhotoInfo_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION: Handler function for the GetContactPhotoInfo Method Start
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::\
vHandle_GetContactPhotoInfo_MethodStart(amt_tclServiceData* poMessage)
{


   ETG_TRACE_USR4(("Function vHandle_GetContactPhotoInfo_MethodStart entered"));
   tBool blNoError= true;
   tU16  u16ErrorCode;

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );

   // Create the (message related) FI data object
   most_phonbkfi_tclMsgGetContactPhotoInfoMethodStart \
      oGetContactPhotoInfoMethodStart_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.

   if ( oVisitorMsg.s32GetData(oGetContactPhotoInfoMethodStart_MS, \
      MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION)
      != OSAL_ERROR)
   {
      most_phonbkfi_tclMsgGetContactPhotoInfoMethodResult \
         oGetContactPhotoInfo_MR;

      tU32 u32ContactHandle= oGetContactPhotoInfoMethodStart_MS.u32ContactHandle;

      ETG_TRACE_USR4(("GetContactPhotoInfo Method Start.ContactHandle- %d", u32ContactHandle));

      if( u32ContactHandle == 0) //Contact Handle- 0 is invalid
      {
         u16ErrorCode= CCA_C_U16_ERROR_PARAMETER_OUT_OF_RANGE;
         blNoError= false;
      }
      else
      {
         //HOTFIX 047
         if(m_pPhoneBookDB)
            blNoError= m_pPhoneBookDB->GetContactPhotoInfo(
            &oGetContactPhotoInfoMethodStart_MS, &oGetContactPhotoInfo_MR,
            u16ErrorCode);

         if(blNoError)
         {
            oGetContactPhotoInfo_MR.u32ContactHandle= u32ContactHandle;

            fi_tclVisitorMessage oMRMsg(oGetContactPhotoInfo_MR, \
               MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);

            // Here the message related header information is added and by this the
            // creation of the message is completed.

            vInitServiceData( oMRMsg,
               poMessage->u16GetSourceAppID(), // ServiceDataMsg
               poMessage->u16GetRegisterID(),  // RegisterID
               poMessage->u16GetCmdCounter(),  // CmdCounter
               poMessage->u16GetFunctionID(),  // Function ID
               AMT_C_U8_CCAMSG_OPCODE_METHODRESULT    // Opcode
               );

            if (oMRMsg.bIsValid())
            {

               // Post message
               ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

               if (enResult != AIL_EN_N_NO_ERROR)
               {
                  NORMAL_M_ASSERT_ALWAYS();
               }
            }
            else
            {
               most_phonbkfi_tclMsgGetContactPhotoInfoError contactPhotoInfoError;
               vHandleMethodError(contactPhotoInfoError,poMessage);
               contactPhotoInfoError.vDestroy();
            }
         }
         else
         {
            most_phonbkfi_tclMsgGetContactPhotoInfoError contactPhotoInfoError;
            vHandleMethodError(contactPhotoInfoError,poMessage);
            contactPhotoInfoError.vDestroy();
         }
      }
      oGetContactPhotoInfo_MR.vDestroy();
   }
   else
   {
      // If we come here, we assume the client has sent an erroneous message,
      // probably it has used a wrong FI_MAJOR_VERSION. Therefore we will
      // send back a an error message and throw a NORMAL_M_ASSERT_ALWAYS().
      blNoError= false;
      u16ErrorCode= CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
   }

   if(!blNoError)
   {
      vHandleGetDataError(poMessage,"vHandle_GetContactPhotoInfo_MethodStart");
   }
   oGetContactPhotoInfoMethodStart_MS.vDestroy();


}

/*******************************************************************************
** FUNCTION:   vHandle_GetContactPhoto_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION: Handler function for the GetContactPhoto Method Start
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::\
vHandle_GetContactPhoto_MethodStart(amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR4(("Function vHandle_GetContactPhoto_MethodStart entered"));
   tBool blNoError= true;
   tU16  u16ErrorCode;

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );

   // Create the (message related) FI data object
   most_phonbkfi_tclMsgGetContactPhotoMethodStart \
      oGetContactPhotoMethodStart_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.

   if ( oVisitorMsg.s32GetData(oGetContactPhotoMethodStart_MS, \
      MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION)
      != OSAL_ERROR)
   {

      most_phonbkfi_tclMsgGetContactPhotoMethodResult \
         oGetContactPhoto_MR;

      tU32 u32ContactHandle= oGetContactPhotoMethodStart_MS.u32ContactHandle;


      ETG_TRACE_USR4(("GetContactPhoto Method Start.ContactHandle- %d", u32ContactHandle));

      if( u32ContactHandle == 0) //Contact Handle- 0 is invalid
      {
         u16ErrorCode= CCA_C_U16_ERROR_PARAMETER_OUT_OF_RANGE;
         blNoError= false;
      }
      else
      {
         if(m_pPhoneBookDB)
            blNoError= m_pPhoneBookDB->GetContactPhoto(
            &oGetContactPhotoMethodStart_MS, &oGetContactPhoto_MR,
            u16ErrorCode);

         if(blNoError)
         {
            oGetContactPhoto_MR.u32ContactHandle= u32ContactHandle;

            fi_tclVisitorMessage oMRMsg(oGetContactPhoto_MR, \
               MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);

            // Here the message related header information is added and by this the
            // creation of the message is completed.

            vInitServiceData( oMRMsg,
               poMessage->u16GetSourceAppID(), // ServiceDataMsg
               poMessage->u16GetRegisterID(),  // RegisterID
               poMessage->u16GetCmdCounter(),  // CmdCounter
               poMessage->u16GetFunctionID(),  // Function ID
               AMT_C_U8_CCAMSG_OPCODE_METHODRESULT    // Opcode
               );

            if (oMRMsg.bIsValid())
            {

               // Post message
               ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

               if (enResult != AIL_EN_N_NO_ERROR)
               {
                  NORMAL_M_ASSERT_ALWAYS();
               }
            }
            else
            {
               most_phonbkfi_tclMsgGetContactPhotoError contactPhotoError;
               vHandleMethodError(contactPhotoError,poMessage);
               contactPhotoError.vDestroy();
            }
         }
         else
         {
            most_phonbkfi_tclMsgGetContactPhotoError contactPhotoError;
            vHandleMethodError(contactPhotoError,poMessage);
            contactPhotoError.vDestroy();
         }
      }
      oGetContactPhoto_MR.vDestroy();
   }

   else
   {
      // If we come here, we assume the client has sent an erroneous message,
      // probably it has used a wrong FI_MAJOR_VERSION. Therefore we will
      // send back a an error message and throw a NORMAL_M_ASSERT_ALWAYS().
      blNoError= false;
      u16ErrorCode= CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
   }

   if(!blNoError)
   {
      vHandleGetDataError(poMessage,"vHandle_GetContactPhoto_MethodStart");
   }
   oGetContactPhotoMethodStart_MS.vDestroy();



}

/*******************************************************************************
** FUNCTION:   vHandle_AutoCompletePhoneNumberEntry_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*  This is a handler function for AutoCompletePhoneNumberEntry method
*  On receiving the phone numbers(Phone numbers Digits vary from 1-n digits) this method query database of the call history list for the matched numbers and sends matched record results.
*  PARAMETERS:
*  IN:poMessage:This is the received message from a client
*  RETURNVALUE:None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_AutoCompletePhoneNumberEntry_MethodStart(amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR4(("Function vHandle_AutoCompletePhoneNumberEntry_MethodStart entered"));

   bool ret = false;
   tU8 ConnectionID = 0;
   tU8 DeviceHandle = u8GetActivePhoneDeviceHandle();

   fi_tclVisitorMessage oVisitorMsg(poMessage);
   most_phonbkfi_tclMsgAutocompletePhoneNumberEntryMethodStart oAutoCompletePhoneNumberEntry_MS;
   most_phonbkfi_tclMsgAutocompletePhoneNumberEntryMethodResult oAutoCompletePhoneNumberEntry_MR;

   FC_Device_Details *deviceObject = FC_Device_Details::deviceHandleToObjectInstance(DeviceHandle);
   if(deviceObject == NULL)
   {
      ETG_TRACE_USR4((" Device object not found for given device handle"));
   }
   else
   {
      ConnectionID = deviceObject->getConnectionID();
      ret = true;
   }

   ETG_TRACE_USR2(("ConnectionID is : %d",ConnectionID));


   if(oVisitorMsg.s32GetData(oAutoCompletePhoneNumberEntry_MS,MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      if(ret == true)
      {
         //Fix : CMG3G-12880
         if ((deviceObject) && (FALSE
               == deviceObject->bIsDeviceProbablyLocked()))
         {
#ifdef CONNECTIVITY_AUTOCOMPLETE_PB_SEARCH			
            ETG_TRACE_USR2(( "PhoneNumber to Search in PhoneBook Records: %s", oAutoCompletePhoneNumberEntry_MS.sPhoneNumberDigits.szValue));
            if(m_pPhoneBookDB)
            {
               ret = m_pPhoneBookDB->SearchPhoneNumberEntriesInPB(&oAutoCompletePhoneNumberEntry_MS,&oAutoCompletePhoneNumberEntry_MR,DeviceHandle);
            }
#else
            ETG_TRACE_USR2(("PhoneNumber to Search in CallHistory Records: %s", oAutoCompletePhoneNumberEntry_MS.sPhoneNumberDigits.szValue));
            if (NULL != m_pPhoneBookDB)
            {
               ret
                     = m_pPhoneBookDB->SearchPhoneNumberEntries(&oAutoCompletePhoneNumberEntry_MS, &oAutoCompletePhoneNumberEntry_MR, ConnectionID);
            }
#endif
         }
         else
         {
            ETG_TRACE_USR2(("Device is locked hence mocking the list"));
            ret = true;
         }
         //End fix : CMG3G-12880
      }

      if(ret == true)
      {
         //Search Of phonenumber entries success
         fi_tclVisitorMessage oMRMsg(oAutoCompletePhoneNumberEntry_MR,MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
         vInitServiceData(oMRMsg,
            poMessage->u16GetSourceAppID(), 	// ServiceDataMsg
            poMessage->u16GetRegisterID(),  	// RegisterID
            poMessage->u16GetCmdCounter(),  	// CmdCounter,
            poMessage->u16GetFunctionID(), 	// Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT   // Opcode
            );

         if(oMRMsg.bIsValid())
         {
            // Post message
            ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);
            if(enResult != AIL_EN_N_NO_ERROR)
            {

               ETG_TRACE_USR4(( "vHandle_AutoCompletePhoneNumberEntry_MethodResult: 'Result', enPostMessage() failed"));

            }
            oAutoCompletePhoneNumberEntry_MR.vDestroy();
         }
         else
         {
            most_phonbkfi_tclMsgAutocompletePhoneNumberEntryError autoCompleteError;
            vHandleMethodError(autoCompleteError,poMessage);
            autoCompleteError.vDestroy();
         }
      }
      else
      {
         amt_tclServiceDataError oErrorMsg(*poMessage, most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC);
         // Post message
         ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oErrorMsg, TRUE);
         if (enResult != AIL_EN_N_NO_ERROR)
         {

            ETG_TRACE_USR4(( "vHandle_AutoCompletePhoneNumberEntry_MethodStart: 'ERROR', enPostMessage() failed "));

         }
      }
   }
   else
   {
      vHandleGetDataError(poMessage,"vHandle_AutoCompletePhoneNumberEntry_MethodStart");
   }
   oAutoCompletePhoneNumberEntry_MS.vDestroy();



}
//FIX CMG3G-8367 IS2424_DownloadOnOff behaviour@FC_Phonebook
// The user shall have possibility to switch the phonebook and call history list download On or Off individual for each paired phone device.
/*******************************************************************************
 ** FUNCTION:   vHandle_SwitchDownloadOnOffState_MethodStart(..)
 *******************************************************************************/
/* DESCRIPTION:
 *  This is a handler function for SwitchDownloadOnOffState method
 *
 *  PARAMETERS:
 *  IN:poMessage:This is the received message from a client
 *  RETURNVALUE:None
 *******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_SwitchDownloadOnOffState_MethodStart(amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR4(("Function vHandle_SwitchDownloadOnOffState_MethodStart entered"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);

   most_phonbkfi_tclMsgSwitchDownloadOnOffStateMethodStart oSwitchDownloadOnOffState_MS;

   if (oVisitorMsg.s32GetData(oSwitchDownloadOnOffState_MS, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      ETG_TRACE_USR2(("Device Handle: %d", oSwitchDownloadOnOffState_MS.u8DeviceHandle));
      ETG_TRACE_USR2(("DownloadOnOff State : %d", oSwitchDownloadOnOffState_MS.u8DownloadOnOffState));

      tBool bSuccess = false, bError = false;

      tU8 u8DeviceHandle = oSwitchDownloadOnOffState_MS.u8DeviceHandle;

      bSuccess = bIsParametersValid(u8DeviceHandle, oSwitchDownloadOnOffState_MS.u8DownloadOnOffState);

      if (TRUE == bSuccess)
      {
         FC_Device_Details* pDeviceObject = FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);
         if (PB_DL_OFF == oSwitchDownloadOnOffState_MS.u8DownloadOnOffState)
         {
            bSuccess = FC_PhoneBook_SQLite::GetSQLiteInstance()->UpdateSwitchDownloadOnOffState(u8DeviceHandle, PB_DL_IN_TRANSITION_TO_OFF);
            vPhoneBook_UpdateClientsDownloadOnOffList();

            if (pDeviceObject)
            {
               pDeviceObject->setDownloadOnOffStatus(PB_DL_IN_TRANSITION_TO_OFF);
               tU8 u8ConnectionID = pDeviceObject->getConnectionID();
               if (bOnSwitchDownloadStateOff(u8DeviceHandle, u8ConnectionID))
               {
                  bSuccess = FC_PhoneBook_SQLite::GetSQLiteInstance()->UpdateSwitchDownloadOnOffState(u8DeviceHandle, PB_DL_OFF);
                  vPhoneBook_UpdateClientsDownloadOnOffList();
                  pDeviceObject->setDownloadOnOffStatus(PB_DL_OFF);
                  m_oMisssedCallIndicatorStatus.u32NumMissedCalls = VALUE_ZERO;
                  vPhoneBook_UpdateClientsAboutMissedCallCount(u8DeviceHandle);
               }
            }
            else
            {
               bSuccess = m_pPhoneBookDB->getCurrentListInstance()->bClearPBRecords(u8DeviceHandle);
               bSuccess = bSuccess && FC_PhoneBook_SQLite::GetSQLiteInstance()->UpdateSwitchDownloadOnOffState(u8DeviceHandle, PB_DL_OFF);
               vPhoneBook_UpdateClientsDownloadOnOffList();
            }
         }
         else if (PB_DL_ON == oSwitchDownloadOnOffState_MS.u8DownloadOnOffState)
         {
            bSuccess = FC_PhoneBook_SQLite::GetSQLiteInstance()->UpdateSwitchDownloadOnOffState(u8DeviceHandle, PB_DL_IN_TRANSITION_TO_ON);
            vPhoneBook_UpdateClientsDownloadOnOffList();

               bSuccess = FC_PhoneBook_SQLite::GetSQLiteInstance()->UpdateSwitchDownloadOnOffState(u8DeviceHandle, PB_DL_ON);
               vPhoneBook_UpdateClientsDownloadOnOffList();
               if (pDeviceObject)
                  pDeviceObject->setDownloadOnOffStatus(PB_DL_ON);
         }

         if (TRUE == bSuccess)
         {
            most_phonbkfi_tclMsgSwitchDownloadOnOffStateMethodResult oSwitchDownloadOnOffState_MR;
            oSwitchDownloadOnOffState_MR.u8DeviceHandle = oSwitchDownloadOnOffState_MS.u8DeviceHandle;
            oSwitchDownloadOnOffState_MR.u8DownloadOnOffState = oSwitchDownloadOnOffState_MS.u8DownloadOnOffState;

            fi_tclVisitorMessage oMRMsg(oSwitchDownloadOnOffState_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);

            vInitServiceData(oMRMsg, poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(), // RegisterID
            poMessage->u16GetCmdCounter(), // CmdCounter
            poMessage->u16GetFunctionID(), // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
            );

            if (oMRMsg.bIsValid())
            {
               ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);
               if (enResult != AIL_EN_N_NO_ERROR)
               {
                  bError = true;
                  ETG_TRACE_ERR(("vHandle_SwitchDownloadOnOffState_MethodStart: 'Result', enPostMessage() failed, 'ail_tenCommunicationError' = %d", (tU32) enResult));
               }
            }
            else
            {
               ETG_TRACE_ERR(("Function vHandle_SwitchDownloadOnOffState_MethodStart: oMRMsg is invalid"));
               bError = true;
            }

            oSwitchDownloadOnOffState_MR.vDestroy();
         }
         else
         {
            ETG_TRACE_ERR(("vHandle_SwitchDownloadOnOffState_MethodStart: Updating the query failed"));
            bError = true;
         }
      }
      else
      {
         ETG_TRACE_ERR(("vHandle_SwitchDownloadOnOffState_MethodStart: Parameters are invalid"));
         bError = true;
      }

      if (TRUE == bError)
      {
         most_phonbkfi_tclMsgSwitchDownloadOnOffStateError oSwitchDownloadOnOffStateError;
         vHandleMethodError(oSwitchDownloadOnOffStateError, poMessage);
         oSwitchDownloadOnOffStateError.vDestroy();
      }

   }
   else
   {
      ETG_TRACE_ERR(("vHandle_SwitchDownloadOnOffState_MethodStart: GetMesage Invalid"));
      vHandleGetDataError(poMessage, "vHandle_SwitchDownloadOnOffState_MethodStart");
   }
   // Always destroy the(message related)FI data object (before leaving
   // its creation scope)
   oSwitchDownloadOnOffState_MS.vDestroy();

}

/*******************************************************************************
 ** FUNCTION:   bOnSwitchDownloadStateOff(..)
 *******************************************************************************/
/* DESCRIPTION: This function is triggered when SwitchDownloadState is Off for particular device handle
 *
 *  PARAMETERS:
 *     IN: - DeviceHandle,ConnectionID
 *  RETURNVALUE: bool - indicating success or failure of this function
 *******************************************************************************/
bool fc_phonebook_tclService_PhoneBook::bOnSwitchDownloadStateOff(tU8 f_u8DeviceHandle, tU8 f_u8ConnectionID)
{
   ETG_TRACE_USR4(("vOnSwitchDownloadStateOff entered"));
   tBool l_bDnlOffResult = FALSE;

   if (m_poDownloadState)
   {
      if (most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_INFO
            == m_poDownloadState->u8GetPBDownloadState()
            || most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_PHOTO
                  == m_poDownloadState->u8GetPBDownloadState()
            || most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_IN_PROCESS
                  == m_poDownloadState->u8GetRecentCallListDownloadState())
      {
         if (m_pDBus_CCA_If_pb)
         {
            m_pDBus_CCA_If_pb->vPhoneBook_DnlStop(f_u8ConnectionID);
         }
      }
      else
      {
         ETG_TRACE_USR2(("No ongoing download"));
         if(m_pPhoneBookDB)
         {
            l_bDnlOffResult
                  = m_pPhoneBookDB->getCallHistoryListInstance()->bClearCHRecords(f_u8ConnectionID);
            l_bDnlOffResult
                  = l_bDnlOffResult
                        && m_pPhoneBookDB->getCurrentListInstance()->bClearPBRecords(f_u8DeviceHandle);
         }
         else
         {
            ETG_TRACE_ERR((" m_pPhoneBookDB is NULL!!"));
         }
      }
   }
   else
   {
      ETG_TRACE_ERR((" m_poDownloadState is NULL!!"));
   }

   return l_bDnlOffResult;
}


/*******************************************************************************
 ** FUNCTION:   bIsParametersValid(..)
 *******************************************************************************/
/* DESCRIPTION: This function is triggered when Checking Consistency For SwitchDownloadStateOff Method parameters
 *
 *  PARAMETERS:
 *     IN: Devicehandle, DownloadOnOffState
 *  RETURNVALUE: bool - indicating success or failure of this function
 *******************************************************************************/
bool fc_phonebook_tclService_PhoneBook::bIsParametersValid(tU8 f_u8DeviceHandle, tU8 f_u8DownloadOnOffState)
{
   tBool bIsValid = FALSE;

   if (((f_u8DeviceHandle > FC_PB_VEHICLE_PBOOK) && (f_u8DeviceHandle <= FC_PB_DEVICE_HANDLE11)) && ((f_u8DownloadOnOffState == PB_DL_ON) || (f_u8DownloadOnOffState == PB_DL_OFF)))
   {
      bIsValid = TRUE;
   }
   else
   {
      ETG_TRACE_USR2(("DeviceHandle or Download state is not valid"));
   }

   return bIsValid;
}
//End of fix FIX CMG3G-8367


#ifdef BUGZID_206322 // Workaround for device disconection status
/*******************************************************************************
** FUNCTION:   vDeviceConnected(..)
*******************************************************************************/
/* DESCRIPTION: This function is triggered when a device is disconnected
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vDeviceConnected(tU8 u8DeviceHandle, const tChar *deviceAddress)
{
   ETG_TRACE_USR4(("vDeviceConnected entered with device handle: %d", u8DeviceHandle));

   if (deviceAddress && ((u8DeviceHandle > FC_PB_VEHICLE_PBOOK)
         && (u8DeviceHandle <= FC_PB_DEVICE_HANDLE11)))
   {
      if (!FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle))
      {
         tU8 u8Index = 0;
         for(u8Index= 0; u8Index < NO_OF_CONNECTED_DEVICE; u8Index++)
         {
            if(!FC_Device_Details::getDeviceObject((INDEX)u8Index))
            {
               FC_Device_Details::createDeviceObject((INDEX)u8Index);
               FC_Device_Details* pDeviceObject=
                  FC_Device_Details::getDeviceObject((INDEX)u8Index);
               if(pDeviceObject)
               {
                  ETG_TRACE_USR2(("Connection ID - %d assigned for the newly connected "
                     "device", u8Index+1));

                  //Set device specific details
                  pDeviceObject->setConnectionID((INDEX)(u8Index+1));
                  pDeviceObject->setDeviceHandle(u8DeviceHandle);
                  pDeviceObject->setDeviceAddress(deviceAddress);
                  pDeviceObject->setSourceType(true);

                  tU16 u16DownloadOnOffStatus = FC_PhoneBook_SQLite::GetSQLiteInstance()->u16GetDownloadOnOffStateForDevicehandle(u8DeviceHandle);
                  pDeviceObject->setDownloadOnOffStatus(u16DownloadOnOffStatus);

                  vClearMissedCalls(u8DeviceHandle, pDeviceObject);
                  vUpdatePhonebookMissedCallIndicatorList(u8DeviceHandle);

                  //FIX CMG3G-3262 Download State property implementation
                  if(m_poDownloadState!= NULL)
                  {
                     if(INVALID_VALUE == m_oPbk_DnlManager.bCheckIfAnyDownloadIsInProgress())
                     {
                        ETG_TRACE_USR4(("Setting DownloadState to most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_NOT_STARTED and most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_NOT_STARTED on device connection"));
                        m_poDownloadState->vSetPBDownloadState(most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_NOT_STARTED);
                        m_poDownloadState->vSetRecentCallListDownloadState(most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_NOT_STARTED);
                        m_poDownloadState->vSetDeviceHandle(u8DeviceHandle);
                        //Update the Client
                        vPhoneBook_UpdateDownloadState();
                     }
                     else
                     {
                        ETG_TRACE_USR4(("PB/CH download is in progress for other device"));
                     }
                  }
                  else
                  {
                     ETG_TRACE_USR4(("m_poDownloadState is NULL"));
                  }

                  //End of fix

                  m_pPhoneBookDB->DeviceConnected(pDeviceObject);
                  m_oPbk_DnlManager.vNewDeviceConnected(u8DeviceHandle);
                  vMakeFavoriteAvaialble(u8DeviceHandle);
                  break;
               }
               else
               {

                  ETG_TRACE_ERR(("Device Object is NULL"));
               }
            }
         }

         if(u8Index == NO_OF_CONNECTED_DEVICE)
         {
            ETG_TRACE_ERR((" FATAL ERROR - No space for newly connected device "));
         }
      }
      else
      {
         ETG_TRACE_ERR(("Device object already exists for the device handle - %d",
               u8DeviceHandle));
      }
   }
   else
   {
      ETG_TRACE_ERR((" Invalid device handle or device address... "));
   }
}
#endif


/*************************************************************************
*
* FUNCTION: fc_phonebook_tclService_PhoneBook::vPrepareLoopBackMsg
*
* DESCRIPTION: This function prepares and posts a loopback message
*
* PARAMETER:
*
* RETURNVALUE:
*
***************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vPrepareandsendLoopBackMsg(tU32 u32Internal_FunctionID)
{
   // Create a 'generic message' (with tU32 content), intended to be sent
   // to our own server (loopback message)


   ETG_TRACE_USR4(("fc_phonebook_tclService_PhoneBook::vPrepareandsendLoopBackMsg :: prepare and send loopback msg entered"));

   gm_tclU32Message oServiceReadyMsg(
      _poMainAppl->u16GetAppId(),      // Source AppID
      _poMainAppl->u16GetAppId(),      // Target AppID
      0,                               // RegisterID
      0,                               // CmdCounter
      CCA_C_U16_SRV_FB_PHONEBOOK,      // ServiceID
      PHONEBOOK_C_U16_IFID_DBUS,    // Function ID
      AMT_C_U8_CCAMSG_OPCODE_STATUS    // Opcode
      );

   oServiceReadyMsg.vSetDWord(u32Internal_FunctionID);

   // Post message
   ail_tenCommunicationError enResult =
      _poMainAppl->enPostMessage(&oServiceReadyMsg, TRUE);

   if (enResult != AIL_EN_N_NO_ERROR)
   {
      ETG_TRACE_ERR(( "vProcessService: 'ServiceReady' Opcode STATUS: enPostMessage() failed, 'ail_tenCommunicationError' = %u",
         (tU32)enResult ));
      // NORMAL_M_ASSERT_ALWAYS();
   }


   ETG_TRACE_USR4((" fc_phonebook_tclService_PhoneBook::vPrepareandsendLoopBackMsg :: prepare and send loopback msg END"));


}


/*************************************************************************
*
* FUNCTION:fc_phonebook_tclService_PhoneBook::vClearAutocompleteListsOnOSChange()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:
*
***************************************************************************/

tVoid fc_phonebook_tclService_PhoneBook::vClearAutocompleteListsOnOSChange()
{


   ETG_TRACE_USR4(("clearAutocompleteListsOnOutgoingChange entered "));

   //HOTFIX 047
   if(m_pPhoneBookDB)
      m_pPhoneBookDB->clearAutocompleteListsOnOSChange();



}
/*******************************************************************************
** FUNCTION:   vPhoneBook_DownloadInitiated(..)
*******************************************************************************/
/* DESCRIPTION: This function is triggered when download is initiated for a
*               given device
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vPhoneBook_DownloadInitiated(FC_Device_Details* pDeviceObject)
{

   ETG_TRACE_USR4(("Download Initiated entered"));
   //HOTFIX 047
   if(m_pPhoneBookDB)
      m_pPhoneBookDB->PhoneBookDownloadInitiated(pDeviceObject);
}

/*************************************************************************
*
* FUNCTION:fc_phonebook_tclService_PhoneBook::vOnApplicationClose()
*
* DESCRIPTION: This function is triggered just before the application is closed
*
* PARAMETER:
*
* RETURNVALUE:
*
***************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vOnApplicationClose()
{

   ETG_TRACE_USR4(("vOnApplicationClose entered"));

   if(m_pPhoneBookDB)
   {
      m_pPhoneBookDB->getCallHistoryListInstance()->ClearCallHistoryRecords();
      m_pPhoneBookDB->ClearFavoritesAvailableAndActiveStatus();
   }
   else
   {

      ETG_TRACE_ERR(("Data base instance failed in application close"));
   }
}
/*******************************************************************************************
*
* FUNCTION:fc_phonebook_tclService_PhoneBook::vCallHistoryUpdated()
*
* DESCRIPTION: called on update of call history due to call
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vCallHistoryUpdated(tU8 u8ConnectionID,
                                                             tU8 u8CallHistoryDnlCmd)
{

   ETG_TRACE_USR4((" Call History updated and Download Command - %d", u8CallHistoryDnlCmd));
   if(m_pPhoneBookDB)
   {
      m_pPhoneBookDB->CallHistoryUpdated(u8ConnectionID, u8CallHistoryDnlCmd);
   }
   else
   {

      ETG_TRACE_FATAL((" NULL POINTER:m_pPhoneBookDB "));
   }


}
/*******************************************************************************************
*
* FUNCTION:fc_phonebook_tclService_PhoneBook::vUpdatePhoneBookFeatureSupport()
*
* DESCRIPTION: triggers MOST_PHONBKFI_C_U16_DEVICEPHONEBOOKFEATURESUPPORT and
*              MOST_PHONBKFI_C_U16_DEVICEPHONEBOOKFEATURESUPPORTEXTENDED property update to
*              the clients
*
* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vUpdatePhoneBookFeatureSupport(FC_Device_Details * poDeviceObject, tBool bUpdateClients)//FIX CMG3GB-696 If PbDlGetCapabilities is success, we hold the update and we shall update it in the PbDlPrepare success/failure response.
{
   ETG_TRACE_USR4(("Device Phonebook Feature support entered"));
   if (poDeviceObject)
   {
      poDeviceObject->setCapInProgress(FALSE);

      tU8 u8DeviceHandle = poDeviceObject->getDeviceHandle();
      tBool bFound = FALSE;
      tU8 u8Index = 0;

      for (u8Index = 0; u8Index
            < m_oDevicePhoneBookFeatureSupportExtendedStatus.oDevicePhoneBookFeatureSupportList.oItems.size(); u8Index++)
      {
         if (u8DeviceHandle
               == m_oDevicePhoneBookFeatureSupportExtendedStatus.oDevicePhoneBookFeatureSupportList.oItems[u8Index].u8DeviceHandle)
         {
            bFound = TRUE;
            break;
         }
      }

      most_fi_tcl_DevicePhoneBookFeatureSupportListItem
            oDevicePhoneBookFeatureSupportListItem;

      tU8 u8DeviceCapabilities = poDeviceObject->getDeviceCapabilities();
      if (((u8DeviceCapabilities & PBInSIM) == PBInSIM)
            || ((u8DeviceCapabilities & PBInLocal) == PBInLocal))
      {
         m_oDevicePhoneBookFeatureSupportStatus.bPhoneBookSupport = TRUE;
         oDevicePhoneBookFeatureSupportListItem.bPhoneBookSupport = TRUE;
         // sortList view support
         m_oDevicePhoneBookFeatureSupportStatus.bContactListSortView
               = poDeviceObject->getSortListViewSupport();
         oDevicePhoneBookFeatureSupportListItem.bContactListSortView
               = poDeviceObject->getSortListViewSupport();
      }
      else
      {
         ETG_TRACE_USR4(("NO : Phonebook support "));
         m_oDevicePhoneBookFeatureSupportStatus.bPhoneBookSupport = FALSE;
      }

      if ((u8DeviceCapabilities & MissedCallHistory) == MissedCallHistory)
      {
         m_oDevicePhoneBookFeatureSupportStatus.bMCHListSupport = TRUE;
         oDevicePhoneBookFeatureSupportListItem.bMCHListSupport = TRUE;
      }
      else
      {
         ETG_TRACE_USR4(("NO : Missed call history support"));
         m_oDevicePhoneBookFeatureSupportStatus.bMCHListSupport = FALSE;
      }
      if ((u8DeviceCapabilities & IncomingCallHistory) == IncomingCallHistory)
      {
         m_oDevicePhoneBookFeatureSupportStatus.bICHListSupport = TRUE;
         oDevicePhoneBookFeatureSupportListItem.bICHListSupport = TRUE;
      }
      else
      {
         ETG_TRACE_USR4(("NO : Incoming call history support"));
         m_oDevicePhoneBookFeatureSupportStatus.bICHListSupport = FALSE;
      }
      if ((u8DeviceCapabilities & OutgoingCallHistory) == OutgoingCallHistory)
      {
         m_oDevicePhoneBookFeatureSupportStatus.bOCHListSupport = TRUE;
         oDevicePhoneBookFeatureSupportListItem.bOCHListSupport = TRUE;
      }
      else
      {
         ETG_TRACE_USR4(("NO : Outgoing call history support"));
         m_oDevicePhoneBookFeatureSupportStatus.bOCHListSupport = FALSE;
      }

      if ((u8DeviceCapabilities & CombinedCallHistory) == CombinedCallHistory)
      {
         m_oDevicePhoneBookFeatureSupportStatus.bCCHListSupport = TRUE;
         oDevicePhoneBookFeatureSupportListItem.bCCHListSupport = TRUE;
      }
      else
      {
         ETG_TRACE_USR4(("NO : combined call history support"));
         m_oDevicePhoneBookFeatureSupportStatus.bCCHListSupport = FALSE;
      }

      m_oDevicePhoneBookFeatureSupportStatus.u8DeviceHandle = u8DeviceHandle;
      oDevicePhoneBookFeatureSupportListItem.u8DeviceHandle = u8DeviceHandle;

      if (TRUE == bFound)
      {
         ETG_TRACE_USR4((" Updating entry for device handle: %d", u8DeviceHandle));
         m_oDevicePhoneBookFeatureSupportExtendedStatus.oDevicePhoneBookFeatureSupportList.oItems[u8Index]
               = oDevicePhoneBookFeatureSupportListItem;
      }
      else
      {
         ETG_TRACE_USR4((" Adding new entry for device handle: %d", u8DeviceHandle));
         m_oDevicePhoneBookFeatureSupportExtendedStatus.oDevicePhoneBookFeatureSupportList.oItems.push_back(oDevicePhoneBookFeatureSupportListItem);
      }

      //Now update the DataBase for the Feature Support.
      FC_PhoneBook_SQLite* poFC_PhoneBook_SQLite =
            FC_PhoneBook_SQLite::GetSQLiteInstance();
      if (poFC_PhoneBook_SQLite)
      {
         FC_PhoneBook_FeatureSupport_Detail fs;
         fs.m_address = fs.m_address + poDeviceObject->getDeviceAddress();
         fs.m_devicehandle
               = m_oDevicePhoneBookFeatureSupportStatus.u8DeviceHandle;
         fs.m_cchsupport
               = m_oDevicePhoneBookFeatureSupportStatus.bCCHListSupport;
         fs.m_conatctSortSupport
               = m_oDevicePhoneBookFeatureSupportStatus.bContactListSortView;
         fs.m_ichsupport
               = m_oDevicePhoneBookFeatureSupportStatus.bICHListSupport;
         fs.m_mchsupport
               = m_oDevicePhoneBookFeatureSupportStatus.bMCHListSupport;
         fs.m_ochsupport
               = m_oDevicePhoneBookFeatureSupportStatus.bOCHListSupport;
         fs.m_pbSupport
               = m_oDevicePhoneBookFeatureSupportStatus.bPhoneBookSupport;
         fs.m_LastConnectionTimeElapsed
               = poDeviceObject->m_LastConnectionTimeElapsed;
         fs.m_DlCompletedOnce
               = poFC_PhoneBook_SQLite->bGetDlCompletedOnceValue(fs.m_devicehandle);
         fs.m_AutomaticContactDL
               = poFC_PhoneBook_SQLite->bGetAutomaticContactDLValue(fs.m_devicehandle);
         fs.m_SwitchDownloadOnOffState
               = poFC_PhoneBook_SQLite->u16GetDownloadOnOffStateForDevicehandle(fs.m_devicehandle);
         fs.m_PreferredSortOrder
                        = poFC_PhoneBook_SQLite->u8GetPreferredSortOrderForDevicehandle(fs.m_devicehandle);

         poFC_PhoneBook_SQLite->UpdateFeatureSupportRecord(fs);
      }
      else
      {
         ETG_TRACE_ERR((" poFC_PhoneBook_SQLite is NULL!! "));
      }

      /*FIX CMG3GB-696 Phonebook support value is received as 1 even if user has given cancel for access to contact and call list (after call operation)
       If PbDlGetCapabilities is success, we hold the update and we shall update it in the PbDlPrepare success/failure response.
       If PbDlGetCapabilities fails, update will be sent then itself.
       Also sending updates of Phone book support during call history update.*/
      if (TRUE == bUpdateClients)
      {
         ail_tenCommunicationError
               enResult =
                     eUpdateClients(MOST_PHONBKFI_C_U16_DEVICEPHONEBOOKFEATURESUPPORT);

         if (AIL_EN_N_NO_ERROR != enResult)
         {
            ETG_TRACE_ERR((" DevicePhoneBookFeatureSupport Status: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' = %d", (tU32) enResult));
         }
         else
         {
            ETG_TRACE_USR4((" DevicePhonebookFeatureSupport status sent"));
         }

         enResult
               = eUpdateClients(MOST_PHONBKFI_C_U16_DEVICEPHONEBOOKFEATURESUPPORTEXTENDED);

         if (AIL_EN_N_NO_ERROR != enResult)
         {
            ETG_TRACE_ERR((" DevicePhonebookFeatureSupportExtended Status: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' = %d", (tU32) enResult));
         }
         else
         {
            ETG_TRACE_USR4((" DevicePhonebookFeatureSupportExtended status sent"));
         }
      }
      //End of fix CMG3GB-696
   }
   else
   {
      ETG_TRACE_ERR((" Device Object is NULL!! "));
   }
}

// CMG3G-7983 - Indicate the Phonebook Download Progress
/***********************************************************************************************
*
* FUNCTION:fc_phonebook_tclService_PhoneBook::vUpdatePhonebookDownloadProgress()
*
* DESCRIPTION: Updates the clients about the current phonebook download progress information
*
* PARAMETER: IN:
*
* RETURNVALUE:
*
 ***********************************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vUpdatePhonebookDownloadProgress(tU8 u8DeviceHandle, tU32 u32ContactsDownloaded)
{
   ETG_TRACE_USR4((" vUpdatePhonebookDownloadProgress Entered. u8DeviceHandle: %d u32ContactsDownloaded: %d", u8DeviceHandle, u32ContactsDownloaded));

   m_oDownloadProgress.u32NumberOfContacts = u32ContactsDownloaded;
   // CMG3G-11074 - u8DeviceHandle has been added as extra parameter to support two HFP support
   m_oDownloadProgress.u8DeviceHandle = u8DeviceHandle;
   
   tBool bDnlStateCompleteOrLimitReached = FALSE;

   if(m_poDownloadState)
   {
      if (most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended::FI_EN_E8PBDS_COMPLETE
                  == m_poDownloadState->u8GetPBDownloadState()
                  || most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended::FI_EN_E8PBDS_LIMIT_REACHED
                        == m_poDownloadState->u8GetPBDownloadState())
      {
         bDnlStateCompleteOrLimitReached = TRUE;
      }

      if ((TRUE == bDnlStateCompleteOrLimitReached) || 0 == (u32ContactsDownloaded
            % PBK_DOWNLOAD_PROGRESS_UPDATE_COUNT_REACHED))
      {
         ail_tenCommunicationError enResult =
               eUpdateClients(MOST_PHONBKFI_C_U16_DOWNLOADPROGRESS);

         if (AIL_EN_N_NO_ERROR != enResult)
         {
            ETG_TRACE_ERR((" PhonebookDownloadProgress Status: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' = %d", (tU32) enResult));
         }
         else
         {
            ETG_TRACE_COMP((" -PBDL.S- PhonebookDownloadProgress updated - Count: %d", u32ContactsDownloaded));
            if (TRUE == bDnlStateCompleteOrLimitReached)
            {
               // CMG3G-11074 - After PhoneBook download has completed
               // assigning u32NumberOfContacts and u8DeviceHandle values as zero, which indicates "No PhoneBook download is in progress"
               m_oDownloadProgress.u32NumberOfContacts = VALUE_ZERO;
               m_oDownloadProgress.u8DeviceHandle = VALUE_ZERO;
            }
         }
         vUpdatePhonebookDownloadProgressList(u8DeviceHandle, bDnlStateCompleteOrLimitReached, u32ContactsDownloaded);
      }
   }
   else
   {
      ETG_TRACE_ERR((" m_poDownloadState is NULL!!"));
   }
}

// Fix CMG3G-12350 - Missed call indicator (ST 2.1)
/***********************************************************************************************
*
* FUNCTION:fc_phonebook_tclService_PhoneBook::vUpdatePhonebookMissedCallIndicatorList()
*
* DESCRIPTION: Updates the clients about the current missed call information of all connected devices
*
*
* PARAMETER: IN: tU8 u8DeviceHandle, tBool bMissedCallIndication
*
* RETURNVALUE: None
*
 ***********************************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vUpdatePhonebookMissedCallIndicatorList(tU8 u8DeviceHandle)
{
   ETG_TRACE_USR4((" vUpdatePhonebookMissedCallIndicatorList Entered. u8DeviceHandle: %u", u8DeviceHandle));
   tU8 u8Index = 0;
   tBool bFound = FALSE;


   for (u8Index = 0; u8Index < (tU8) (m_oMissedCallIndicatorListStatus.oMissedCallIndicatorList.oItems.size()); u8Index++)
   {
      if (u8DeviceHandle == m_oMissedCallIndicatorListStatus.oMissedCallIndicatorList.oItems[u8Index].u8DeviceHandle)
      {
         bFound = TRUE;
         break;
      }
   }

   FC_Device_Details* pDeviceObject = FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);
   if(pDeviceObject != NULL)
   {

      most_fi_tcl_PhonBkMissedCallIndicatorListItem oMissedCallIndicatorListItem;
      oMissedCallIndicatorListItem.u8DeviceHandle = u8DeviceHandle;
      oMissedCallIndicatorListItem.u32NumMissedCalls = pDeviceObject->GetMissedcallCount();


      if (bFound == TRUE)
      {
         ETG_TRACE_USR4((" -PBDL.S- MissedCallIndicatorList Update for device handle: %u", u8DeviceHandle));
         m_oMissedCallIndicatorListStatus.oMissedCallIndicatorList.oItems[u8Index] = oMissedCallIndicatorListItem;
      }
      else
      {
         ETG_TRACE_USR4((" -PBDL.S- MissedCallIndicatorList Added for device handle: %u", u8DeviceHandle));
         m_oMissedCallIndicatorListStatus.oMissedCallIndicatorList.oItems.push_back(oMissedCallIndicatorListItem);
      }

      ail_tenCommunicationError enResult = eUpdateClients(MOST_PHONBKFI_C_U16_MISSEDCALLINDICATORLIST);

      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR((" MissedCallIndicatorList Status: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' = %d", (tU32) enResult));
      }
      else
      {
         ETG_TRACE_COMP((" -PBDL.S- MissedCallIndicatorList updated successfully"));
      }

   }
   else
   {
      ETG_TRACE_USR4(("Device handle passed is incorrect. No connected device with that device handle"));
   }
}

// Fix 159808 - DownloadProgress API to return percentage of contacts downloaded
/***********************************************************************************************
*
* FUNCTION:fc_phonebook_tclService_PhoneBook::vUpdatePhonebookDownloadProgressList()
*
* DESCRIPTION: Updates the clients about the current phonebook download progress information of all connected devices
*              interms of Number of contacts downloaded and percentage of contacts downloaded
*
* PARAMETER: IN: tU8 u8DeviceHandle, tBool bDnlStateCompleteOrLimitReached, tU32 u32ContactsDownloaded
*
* RETURNVALUE: None
*
 ***********************************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vUpdatePhonebookDownloadProgressList(tU8 u8DeviceHandle, tBool bDnlStateCompleteOrLimitReached, tU32 u32ContactsDownloaded)
{
   ETG_TRACE_USR4((" vUpdatePhonebookDownloadProgressList Entered. u8DeviceHandle: %u u32ContactsDownloaded: %u", u8DeviceHandle, u32ContactsDownloaded));
   
   tBool bDnlStateCompleteOrLmtReached = bDnlStateCompleteOrLimitReached;

   FC_Device_Details* pDeviceObject = FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);
   if(pDeviceObject != NULL)
   {
      tBool bFound = FALSE;
      tU8 u8Index = 0;
      for (u8Index = 0; u8Index < (tU8)(m_oDownloadProgressListStatus.oDownloadProgressList.oItems.size()); u8Index++)
      {
         if (u8DeviceHandle
               == m_oDownloadProgressListStatus.oDownloadProgressList.oItems[u8Index].u8DeviceHandle)
         {
            bFound = TRUE;
            break;
         }
      }

      most_fi_tcl_PhonBkDownloadProgressListItem oDownloadProgressListItem;
      oDownloadProgressListItem.u8DeviceHandle = u8DeviceHandle;
      oDownloadProgressListItem.u32NumberOfContacts = u32ContactsDownloaded;
      oDownloadProgressListItem.u32ContactsReportedByPhone = pDeviceObject->m_u32DeviceContactsCount;

      //Limiting max number of contacts per device for each project as follows
      //Max number of contacts For IVI - 4000, Others - 0xFFFFFFFF (Unlimited)
      //Based on the project configurations, m_u32DeviceContactsCount is either (PB.size() + SIM.size()) or PB.size()
      tU32 u32LimitDeviceContactsCount = 0;
      u32LimitDeviceContactsCount
            = (pDeviceObject->m_u32DeviceContactsCount
                  <= PBDL_MAX_CONTACTS_PER_PHONE_TO_STORE) ? pDeviceObject->m_u32DeviceContactsCount : PBDL_MAX_CONTACTS_PER_PHONE_TO_STORE;

      ETG_TRACE_USR4((" Device contacts count :%u", pDeviceObject->m_u32DeviceContactsCount));
      //To avoid divide by zero exception (u32LimitDeviceContactsCount != 0)
      if(u32LimitDeviceContactsCount != VALUE_ZERO)
      {
         //There may be a mismatch between the number of contacts proveded by device and number of contacts actually downloaded
         //by FC_Phonebook (Since duplicate contacts will be removed). Hence, if we know download has completed
         //for PB, assign percentage value as 100.
         if ((TRUE == bDnlStateCompleteOrLmtReached))
         {
            oDownloadProgressListItem.u8Percentage = PBK_DOWNLOAD_PROGRESS_UPDATE_COUNT_REACHED;
         }
         else
         {
            oDownloadProgressListItem.u8Percentage = (tU8)((u32ContactsDownloaded * 100) / u32LimitDeviceContactsCount);
         }
      }
      else
      {
         //Fix RTC 176167
         //As per the design document "Pecrentage_of_contacts_downloaded_In_PhoneBook_F-Block_V2.0.docx"
         //If the download is aborted for a specific device, the percentage value would be reset for that specific device handle
         if((VALUE_ZERO == u32ContactsDownloaded) && (TRUE == pDeviceObject->bIsCancelPBRequested()))
         {
            // If download aborted
            oDownloadProgressListItem.u8Percentage = VALUE_ZERO;
         }
         else
         {
            //If device itself has zero contacts, logically percentage of contacts downloaded is 100 percentage
            oDownloadProgressListItem.u8Percentage = PBK_DOWNLOAD_PROGRESS_UPDATE_COUNT_REACHED;
         }
         oDownloadProgressListItem.u32ContactsReportedByPhone = VALUE_ZERO;
         //End fix RTC 176167
      }

      if(bFound == TRUE)
      {
         m_oDownloadProgressListStatus.oDownloadProgressList.oItems[u8Index]
               = oDownloadProgressListItem;
      }
      else
      {
         ETG_TRACE_USR4((" Adding new entry for device handle: %u", u8DeviceHandle));
         m_oDownloadProgressListStatus.oDownloadProgressList.oItems.push_back(oDownloadProgressListItem);
      }

      if(m_poDownloadState)
      {
         if ((TRUE == bDnlStateCompleteOrLmtReached) || 0
               == (u32ContactsDownloaded
                     % PBK_DOWNLOAD_PROGRESS_UPDATE_COUNT_REACHED))
         {
            ail_tenCommunicationError enResult =
                  eUpdateClients(MOST_PHONBKFI_C_U16_DOWNLOADPROGRESSLIST);

            if (AIL_EN_N_NO_ERROR != enResult)
            {
               ETG_TRACE_ERR((" PhonebookDownloadProgressList Status: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' = %d", (tU32) enResult));
            }
            else
            {
               ETG_TRACE_COMP((" -PBDL.S- PhonebookDownloadProgressList updated - Count: %u, Percentage : %d, Number of contacts reported by device :%u",
                     u32ContactsDownloaded, oDownloadProgressListItem.u8Percentage, oDownloadProgressListItem.u32ContactsReportedByPhone));
               if (TRUE == bDnlStateCompleteOrLmtReached)
               {
                  //After PhoneBook download has completed assigning u32NumberOfContacts, u8Percentage and u32ContactsReportedByPhone
                  //values as zero, which indicates "No PhoneBook download is in progress" for that particular device handle
                  m_oDownloadProgressListStatus.oDownloadProgressList.oItems[u8Index].u32NumberOfContacts
                        = VALUE_ZERO;
                  m_oDownloadProgressListStatus.oDownloadProgressList.oItems[u8Index].u8Percentage
                        = VALUE_ZERO;
                  m_oDownloadProgressListStatus.oDownloadProgressList.oItems[u8Index].u32ContactsReportedByPhone
                        = VALUE_ZERO;
               }
            }
         }
      }
      else
      {
         ETG_TRACE_ERR((" m_poDownloadState is NULL!!"));
      }
   }
   else
   {
      ETG_TRACE_USR4(("Device handle passed is incorrect. No connected device with that device handle"));
   }
}


/***********************************************************************************************
 *
 * FUNCTION:fc_phonebook_tclService_PhoneBook::vSetVehicleBtAddress
 *
 * DESCRIPTION: Stores the Vehicle BT address.
 *
 * PARAMETER: IN:
 *
 * RETURNVALUE:
 *
 ***********************************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vSetVehicleBtAddress(char* Address)
{
   ETG_TRACE_USR2(("vSetVehicleBtAddress - %s", Address));
   memset(m_sVehicleBtAddress, 0, LENGTH_OF_DEVICE_ADDRESS);
   vStringCopy(m_sVehicleBtAddress, Address, LENGTH_OF_DEVICE_ADDRESS);//CMG3G-14313-Coverity Fix
}

/***********************************************************************************************
 *
 * FUNCTION:fc_phonebook_tclService_PhoneBook::vGetVehicleBtAddress
 *
 * DESCRIPTION: Get the Vehicle BT address.
 *
 * PARAMETER: IN:
 *
 * RETURNVALUE:
 *
 ***********************************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vGetVehicleBtAddress(char* Address)
{
   ETG_TRACE_USR2(("vGetVehicleBtAddress - %s", m_sVehicleBtAddress));
   memset(Address, 0, LENGTH_OF_DEVICE_ADDRESS);
   vStringCopy(Address, m_sVehicleBtAddress, LENGTH_OF_DEVICE_ADDRESS);//CMG3G-14313-Coverity Fix
}

/***********************************************************************************************
 *
 * FUNCTION:fc_phonebook_tclService_PhoneBook::vSetActivePhoneDeviceHandle
 *
 * DESCRIPTION: Stores the Active phone device handle.
 *
 * PARAMETER: IN:
 *
 * RETURNVALUE:
 *
 ***********************************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vSetActivePhoneDeviceHandle(tU8 deviceHandle)
{
   ETG_TRACE_USR2(("vSetActivePhoneDeviceHandle - %d", deviceHandle));
   this->m_u8ActivePhoneDeviceHandle = deviceHandle;
}

/***********************************************************************************************
 *
 * FUNCTION:fc_phonebook_tclService_PhoneBook::u8GetActivePhoneDeviceHandle
 *
 * DESCRIPTION: Get the Active phone device handle.
 *
 * PARAMETER: IN:
 *
 * RETURNVALUE:
 *
 ***********************************************************************************************/
tU8 fc_phonebook_tclService_PhoneBook::u8GetActivePhoneDeviceHandle()
{
   ETG_TRACE_USR4(( " u8GetActivePhoneDeviceHandle = %d", m_u8ActivePhoneDeviceHandle ));
   return this->m_u8ActivePhoneDeviceHandle;
}

/*******************************************************************************
*
* FUNCTION: vPush_Back_LB_Data
*
* DESCRIPTION: push data to the loopback queue
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vPush_Back_LB_Data(PhBk_Loopback_Data &LBData)
{

   ETG_TRACE_USR4(("Function vPush_Back_LB_Data entered"));

   g_mutex_lock(&p_mutex_loopback);
   LB_data_queue.push_back(LBData);
   g_mutex_unlock(&p_mutex_loopback);
   ETG_TRACE_USR4(("EXIT vPush_Back_LB_Data  "));
}

/*******************************************************************************
*
* FUNCTION: vErase_LB_Data
*
* DESCRIPTION: erase data from loopback queue
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vErase_LB_Data(tU16 u16Position)
{

   ETG_TRACE_USR4(("Function vErase_LB_Data entered"));

   g_mutex_lock(&p_mutex_loopback);
   LB_data_queue.erase(LB_data_queue.begin()+u16Position);
   g_mutex_unlock(&p_mutex_loopback);
}


/*******************************************************************************
*
* FUNCTION: u16Get_Data_Position
*
* DESCRIPTION: get the data position in loopback data queue , corresponding to CB
*              function id
*
* PARAMETER:
*
* RETURNVALUE: None.
*
********************************************************************************/
tU16 fc_phonebook_tclService_PhoneBook::u16Get_Data_Position(tU16 u16CB_functionID)
{

   ETG_TRACE_USR4(("Function u16Get_Data_Position entered"));

   tU16 index1;
   tU16 position = PHONEBOOK_LOOPBACK_VECTOR_INVALID_POSITION ;


   for(index1=0;index1< LB_data_queue.size();index1++)
   {
      if(u16CB_functionID == LB_data_queue[index1].u16FunctionID )
      {
         position = index1 ;
         break;
      }
   }

   ETG_TRACE_USR2(("Position of the function id in the loop back que found: %d",position));

   return position;
}

/*******************************************************************************
** FUNCTION:   vPhoneBook_UpdateFavoritesChanged(..)
*******************************************************************************/
/* DESCRIPTION:
*  Update clients on change of any favorite change
*  PARAMETERS:
*     IN:   favorite change information
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vPhoneBook_UpdateFavoritesChanged
(most_phonbkfi_tclMsgFavoritesChangedStatus& oFavoritesChangedStatus)
{
   m_oFavoritesChangedStatus= oFavoritesChangedStatus;

   ETG_TRACE_USR4(("Updating clients about favorites change entered"));




   //ETG_TRACE_USR2((" FavoritesProvdier:- Favorites change update triggered for "
   //"ListChangeType        - %u ", m_oFavoritesChangedStatus.e8ListChangeType.enType));
   ETG_TRACE_USR1((" List Change Type:: %u",ETG_ENUM(TR_LISTCHANGETYPE,m_oFavoritesChangedStatus.e8ListChangeType.enType)));
   ETG_TRACE_USR2(("FavoritesProvdier:- Favorites change update triggered for "
      "ListLength        - %u ", m_oFavoritesChangedStatus.u32ListSize));
   ETG_TRACE_USR2(("FavoritesProvdier:- Favorites change update triggered for "
      "NumOfItemsChanged        - %u", m_oFavoritesChangedStatus.u32NumItems));

   for(tU16 ListIndex= 0; ListIndex <
      m_oFavoritesChangedStatus.oChangedFavorites.u16Items.size(); ListIndex++)
   {

      ETG_TRACE_USR2(("FavoritesProvdier:- Favorites change update triggered for "
         "FavoritesID        - %d", m_oFavoritesChangedStatus.oChangedFavorites.u16Items[ListIndex]));

   }

   ail_tenCommunicationError enResult = eUpdateClients(MOST_PHONBKFI_C_U16_FAVORITESCHANGED);
   if(AIL_EN_N_NO_ERROR != enResult)
   {

      ETG_TRACE_ERR(("ERROR fc_phonebook_tclService_PhoneBook:- Sending of favorite "
         "change update failed "));
   }
   else
   {
      ETG_TRACE_USR4(( "Sending of favorites change update success"));

   }
}

/*******************************************************************************
** FUNCTION:   poGetFavoritesProviderInstance(..)
*******************************************************************************/
/* DESCRIPTION:
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
FC_Phonebook_tclFavorite_Provider*
fc_phonebook_tclService_PhoneBook::poGetFavoritesProviderInstance()
{
   ETG_TRACE_USR4(("Function poGetFavoritesProviderInstance entered"));

   return m_poFavoriteProvider;
}


/*******************************************************************************
** FUNCTION:   vPostActivateFavoriteMethodResult(..)
*******************************************************************************/
/* DESCRIPTION:
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vPostActivateFavoriteMethodResult(
   tU16 u16FavoriteID,	tU32 u32ActivationHandle)
{
   ETG_TRACE_USR4(("Function vPostActivateFavoriteMethodResult entered"));


   if(m_oActivateFavMap.find(u16FavoriteID) != m_oActivateFavMap.end())
   {
      most_phonbkfi_tclMsgActivateFavoriteMethodResult
         oActivateFavoriteMethodStart_MR;
      oActivateFavoriteMethodStart_MR.u16FavoriteID= u16FavoriteID;
      oActivateFavoriteMethodStart_MR.u32ActivationHandle=
         u32ActivationHandle;

      IncomingMessage ActivateFavoriteMsg= m_oActivateFavMap[u16FavoriteID];
      m_oActivateFavMap.erase(u16FavoriteID);

      fi_tclVisitorMessage oMRMsg(oActivateFavoriteMethodStart_MR,\
         MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
      oActivateFavoriteMethodStart_MR.vDestroy();

      vInitServiceData( oMRMsg,
         ActivateFavoriteMsg.u16SourceAppID , // ServiceDataMsg
         ActivateFavoriteMsg.u16RegisterID,     // RegisterID
         ActivateFavoriteMsg.u16CmdCounter,     // CmdCounter,
         ActivateFavoriteMsg.u16FunctionID,	  // Function ID
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT       // Opcode
         );

      if(!oMRMsg.bIsValid())
      {
         ETG_TRACE_ERR(("Packing of Activate favorite method result failed"));

         most_phonbkfi_tclMsgActivateFavoriteError msgActivateFavoriteError;
         msgActivateFavoriteError.e8ErrorCode.enType
            = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

         fi_tclVisitorMessage oERRMsg(msgActivateFavoriteError,
            MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);

         vInitServiceData( oERRMsg,
            ActivateFavoriteMsg.u16SourceAppID , // ServiceDataMsg
            ActivateFavoriteMsg.u16RegisterID,     // RegisterID
            ActivateFavoriteMsg.u16CmdCounter,     // CmdCounter,
            ActivateFavoriteMsg.u16FunctionID,	  // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT       // Opcode
            );

         if (oERRMsg.bIsValid())
         {
            // Post message
            ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oERRMsg, TRUE);

            if (enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_ERR(("vHandleMethodError: enPostMessage() failed"));
               NORMAL_M_ASSERT_ALWAYS();
            }
            else
            {
               ETG_TRACE_USR4(("enPostMessage : Message Posted Successfully"));
            }
         }
      }
      else
      {
         ail_tenCommunicationError enResult= AIL_EN_N_NO_ERROR;

         enResult= _poMainAppl->enPostMessage(&oMRMsg, TRUE);

         if (enResult != AIL_EN_N_NO_ERROR)
         {

            ETG_TRACE_ERR(("vHandle_Activate_Favorite_MethodStart: 'ERROR', enPostMessage() failed"));
            // NORMAL_M_ASSERT_ALWAYS();
         }
         else
         {

            ETG_TRACE_USR4(("vHandle_Activate_Favorite_MethodStart: enPostMessage() SUCCESS"));
         }
      }        
   }
   else
   {

      ETG_TRACE_ERR(("Favorite ID- %d not available in the received id list", u16FavoriteID));
   }



}

/*******************************************************************************
** FUNCTION:   vPostActivateFavoriteError(..)
*******************************************************************************/
/* DESCRIPTION:
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vPostActivateFavoriteError(most_fi_tcl_e8_ErrorCode::tenType e8ErrorCode,
                                                                    tU16 u16FavoriteID, tU8 u8ErrorInfo)
{

   ETG_TRACE_USR4(("Function vPostActivateFavoriteError entered"));


   ETG_TRACE_USR2(("Favorite id - %d", u16FavoriteID));
   ETG_TRACE_ERR(("Error code is %d", e8ErrorCode));
   ETG_TRACE_ERR(("Error Info is %d", u8ErrorInfo));

   most_phonbkfi_tclMsgActivateFavoriteError oActivateFavoriteError_ER;

   oActivateFavoriteError_ER.e8ErrorCode.enType = e8ErrorCode;   //Fix GMMY16-13959

   //Set error info  Fix GMMY16-13959
   if (NIL_ERROR_INFO != u8ErrorInfo)
      oActivateFavoriteError_ER.oErrorInfo.vSetData(&u8ErrorInfo, sizeof(u8ErrorInfo));



   if (m_oActivateFavMap.find(u16FavoriteID) != m_oActivateFavMap.end()) {

      IncomingMessage ActivateFavoriteMsg = m_oActivateFavMap[u16FavoriteID];
      m_oActivateFavMap.erase(u16FavoriteID);
      fi_tclVisitorMessage oMRMsg(oActivateFavoriteError_ER, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);

      vInitServiceData(oMRMsg, ActivateFavoriteMsg.u16SourceAppID, // ServiceDataMsg
         ActivateFavoriteMsg.u16RegisterID, // RegisterID
         ActivateFavoriteMsg.u16CmdCounter, // CmdCounter,
         ActivateFavoriteMsg.u16FunctionID, // Function ID
         AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
         );

      if (oMRMsg.bIsValid()) {

         ETG_TRACE_USR2(("Posting ActivateFavorite Error"));
         // Post message
         ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(
            &oMRMsg, TRUE);

         if (enResult != AIL_EN_N_NO_ERROR) {

            ETG_TRACE_ERR(("vPostActivateFavoriteError: enPostMessage() failed"));
         }
      }
   }
   else
   {
      ETG_TRACE_USR2(("Posting ActivateFavorite Error FAILED"));
   }
   oActivateFavoriteError_ER.vDestroy();

}
// Fix GMMY16-13959
/*******************************************************************************
** FUNCTION:   vPostStoreFavoriteError(..)
*******************************************************************************/
/* DESCRIPTION:
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vPostStoreFavoriteError(amt_tclServiceData* poMessage, tU16 u16ErrorCode, tU8 u8ErrorInfo)
{

   ETG_TRACE_USR4(("Function vPostStoreFavoriteError entered"));

   ETG_TRACE_ERR(("Error code is %d", u16ErrorCode));
   ETG_TRACE_ERR(("Error Info is %d", u8ErrorInfo));

   most_phonbkfi_tclMsgStoreFavoriteError oStoreFavoriteError_ER;

   oStoreFavoriteError_ER.e8ErrorCode.enType = (most_fi_tcl_e8_ErrorCode::tenType) u16ErrorCode;

   //Set error info
   if (NIL_ERROR_INFO != u8ErrorInfo)
      oStoreFavoriteError_ER.oErrorInfo.vSetData(&u8ErrorInfo, sizeof(u8ErrorInfo));

   fi_tclVisitorMessage
      oMRMsg(oStoreFavoriteError_ER, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oMRMsg, poMessage->u16GetSourceAppID(), // ServiceDataMsg
      poMessage->u16GetRegisterID(), // RegisterID
      poMessage->u16GetCmdCounter(), // CmdCounter
      poMessage->u16GetFunctionID(), // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
      );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ETG_TRACE_USR2(("Posting StoreFavorite Error"));
      ail_tenCommunicationError enResult =
         _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {

         ETG_TRACE_ERR(("vPostStoreFavoriteError: enPostMessage() failed"));
      }
   }
   oStoreFavoriteError_ER.vDestroy();

}

/*******************************************************************************
** FUNCTION:   vPostGetFavoriteInfoError(..)
*******************************************************************************/
/* DESCRIPTION:
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vPostGetFavoriteInfoError(amt_tclServiceData* poMessage, tU16 u16ErrorCode, tU8 u8ErrorInfo)
{

   ETG_TRACE_USR4(("Function vPostGetFavoriteInfoError entered"));

   ETG_TRACE_ERR(("Error code is %d", u16ErrorCode));
   ETG_TRACE_ERR(("Error Info is %d", u8ErrorInfo));

   most_phonbkfi_tclMsgGetFavoriteInfoError oGetFavoriteInfoError_ER;

   oGetFavoriteInfoError_ER.e8ErrorCode.enType = (most_fi_tcl_e8_ErrorCode::tenType) u16ErrorCode;

   //Set error info
   if (NIL_ERROR_INFO != u8ErrorInfo)
      oGetFavoriteInfoError_ER.oErrorInfo.vSetData(&u8ErrorInfo, sizeof(u8ErrorInfo));

   fi_tclVisitorMessage
      oMRMsg(oGetFavoriteInfoError_ER, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oMRMsg, poMessage->u16GetSourceAppID(), // ServiceDataMsg
      poMessage->u16GetRegisterID(), // RegisterID
      poMessage->u16GetCmdCounter(), // CmdCounter
      poMessage->u16GetFunctionID(), // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
      );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ETG_TRACE_USR2(("Posting GetFavoriteInfo Error"));
      ail_tenCommunicationError enResult =
         _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {

         ETG_TRACE_ERR(("vPostGetFavoriteInfoError: enPostMessage() failed"));
      }
   }
   oGetFavoriteInfoError_ER.vDestroy();

}

// End of Fix GMMY16-13959

/*******************************************************************************
** FUNCTION:   vPhoneBook_UpdateMissedCallCount(..)
*******************************************************************************/
/* DESCRIPTION:
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vPhoneBook_UpdateClientsAboutMissedCallCount(tU8 u8DeviceHandle)
{
   ETG_TRACE_USR2(("vPhoneBook_UpdateClientsAboutMissedCallCount entered"));

   if (u8DeviceHandle == m_u8ActivePhoneDeviceHandle)
   {
      m_oMisssedCallIndicatorStatus.u32NumMissedCalls = (tU32) (m_aNoOfMissedCalls[u8DeviceHandle]);

      ETG_TRACE_USR2(("oMissedCallIndicatorStatus.u32NumMissedCalls = %d",m_oMisssedCallIndicatorStatus.u32NumMissedCalls));

      ail_tenCommunicationError enResult = eUpdateClients(MOST_PHONBKFI_C_U16_MISSEDCALLINDICATOR);
      if (AIL_EN_N_NO_ERROR != enResult)
      {

         ETG_TRACE_ERR(("Missed Call Indicator Status: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' "));
      }
   }
}

//FIX CMG3G-8367 IS2424_DownloadOnOff behaviour@FC_Phonebook
// The user shall have possibility to switch the phonebook and call history list download On or Off individual for each paired phone device.
/*******************************************************************************
 ** FUNCTION:   vPhoneBook_UpdateClientsDownloadOnOffList(..)
 *******************************************************************************/
/* DESCRIPTION: Gets the DownloadOnOffStateList from Database and update it to clients
 *  PARAMETERS:
 *     IN: None
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vPhoneBook_UpdateClientsDownloadOnOffList()
{
   ETG_TRACE_USR4(("Function vPhoneBook_UpdateClientsDownloadOnOffList entered"));

   if (FC_PhoneBook_SQLite::GetSQLiteInstance())
   {
      m_oDownloadOnOffListStatus.oDownloadOnOffStateList.oDownloadOnOffStateListItem.clear();

      QList < quint8 > DeviceHandleList;
      QList < quint16 > DownloadOnOffStateList;
      FC_PhoneBook_SQLite::GetSQLiteInstance()->GetDownloadOnOffStateList(DeviceHandleList, DownloadOnOffStateList);

      for (qint8 ListIndex = VALUE_ZERO; ListIndex < DeviceHandleList.size(); ListIndex++)
      {
         most_fi_tcl_PhonBkDownloadOnOffStateListItem oItem;
         oItem.DeviceHandle = DeviceHandleList.at(ListIndex);
         ETG_TRACE_USR2(("vPhoneBook_UpdateClientsDownloadOnOffList triggered for "
            "Device handle id received for property update        - %d", DeviceHandleList.at(ListIndex)));

         oItem.DownloadOnOffState.enType
               = (most_fi_tcl_e8_PhonBkSwitchDownloadOnOffState::tenType) DownloadOnOffStateList.at(ListIndex);
         ETG_TRACE_USR2(("vPhoneBook_UpdateClientsDownloadOnOffList triggered for "
            "DownloadOnOffState  id received for property update        - %d", DownloadOnOffStateList.at(ListIndex)));

         m_oDownloadOnOffListStatus.oDownloadOnOffStateList.oDownloadOnOffStateListItem.push_back(oItem);
      }

      ail_tenCommunicationError enResult =
            eUpdateClients(MOST_PHONBKFI_C_U16_DOWNLOADONOFFLIST);
      if (AIL_EN_N_NO_ERROR != enResult)
      {

         ETG_TRACE_ERR(("DownloadOnOffList Status: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' "));
      }
      else
      {
         ETG_TRACE_USR4(("DownloadOnOffList Status: Updated Clients()"));
      }
   }
   else
   {
      ETG_TRACE_ERR((" SQLite instance is NULL!!!"));
   }
}
//End of fix FIX CMG3G-8367


//Fix CMG3G-9889
/*******************************************************************************
 ** FUNCTION:   vPhoneBook_UpdateClientsAutomaticContactDownloadList(..)
 *******************************************************************************/
/* DESCRIPTION: Gets the AutomaticContactDownloadList from Database and update it to clients
 *  PARAMETERS:
 *     IN: None
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vPhoneBook_UpdateClientsAutomaticContactDownloadList()
{
   ETG_TRACE_USR4(("Function vPhoneBook_UpdateClientsAutomaticContactDownloadList entered"));

   if (FC_PhoneBook_SQLite::GetSQLiteInstance())
   {
      m_oAutomaticContactDownloadListStatus.oAutomaticContactDownloadList.oAutomaticContactDownloadListItem.clear();

      QList < quint8 > DeviceHandleList;
      QList<bool> AutomaticDLList;
      FC_PhoneBook_SQLite::GetSQLiteInstance()->bGetAutomaticContactDLList(DeviceHandleList, AutomaticDLList);

      for (qint8 ListIndex = VALUE_ZERO; ListIndex < DeviceHandleList.size(); ListIndex++)
      {
         most_fi_tcl_AutomaticContactDownloadListItem oItem;

         oItem.DeviceHandle = DeviceHandleList.at(ListIndex);
         ETG_TRACE_USR2(("vPhoneBook_UpdateClientsAutomaticContactDownloadList triggered for "
            "Device handle id received for property update        - %d", DeviceHandleList.at(ListIndex)));

         oItem.bAutomaticDL = AutomaticDLList.at(ListIndex);
         ETG_TRACE_USR2(("vPhoneBook_UpdateClientsAutomaticContactDownloadList triggered for "
            "bAutomaticDLValue  id received for property update        - %d", AutomaticDLList.at(ListIndex)));

         m_oAutomaticContactDownloadListStatus.oAutomaticContactDownloadList.oAutomaticContactDownloadListItem.push_back(oItem);
      }

      ail_tenCommunicationError enResult =
            eUpdateClients(MOST_PHONBKFI_C_U16_AUTOMATICCONTACTDOWNLOADLIST);
      if (AIL_EN_N_NO_ERROR != enResult)
      {
         ETG_TRACE_ERR(("AutomaticContactDownloadList Status: 'ERROR', UpdateClients() failed, 'ail_tenCommunicationError' "));
      }
      else
      {
         ETG_TRACE_USR4(("AutomaticContactDownloadList Status: Updated Clients()"));
      }
   }
   else
   {
      ETG_TRACE_ERR((" SQLite instance is NULL!!!"));
   }
}
/*******************************************************************************
*
* FUNCTION:  fc_phonebook_tclService_PhoneBook::OnLoadSettings
*
* DESCRIPTION:
**
*
* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE:
*
*******************************************************************************/
void fc_phonebook_tclService_PhoneBook::OnLoadSettings()
{
   ETG_TRACE_USR4((" OnLoadSettings function entered "));


   // -PHBK-QUERY-ERR- Instrumentation: BGN - for GMMY17-13044 "database disk image is malformed"
   //
   // At this code position we expect that we have no database connection open, so that
   // we can operate via file operations on the database without any interference regarding
   // any thread, who might do database operations.
   //
   // Here we check if the "database re-create trigger" file 'pb.db.recreate' exists in the database 
   // folder. If so, we just delete/rename the database. [It will then be re-created from scratch in 
   // method FC_PhoneBook_SQLite::InitializeDatabase()]
   int         iResult = 0;
   struct stat oStatBuf;
   
   iResult = stat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_RECREATE_TRIGGER_FILE_NAME, &oStatBuf);
   if( iResult == 0 )
   {
      //Fix NCG3D-67384
      //The "database re-create trigger" file exists.So database table schema check needs to be done.
      FC_PhoneBook_SQLite* poFC_PhoneBook_SQLite = FC_PhoneBook_SQLite::GetSQLiteInstance();
      if(poFC_PhoneBook_SQLite!=NULL)
      {
         poFC_PhoneBook_SQLite->SetCheckTableSchema(TRUE);
      }

      // The "database re-create trigger" file exists.
      //End fix NCG3D-67384
      // Delete the current (probably corrupted) database 'phonebook.db' 
      if( 0 != remove(FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME) )
      {
         iResult = -1;
         ETG_TRACE_FATAL (( " -PHBK-QUERY-ERR-: Database RE-CREATION has FAILED: DB file CANNOT be removed. "  ));
         ETG_TRACE_ERRMEM(( " -PHBK-QUERY-ERR-: Database RE-CREATION has FAILED: DB file CANNOT be removed. "  ));
      }

      // Remove the DB journal file(s), if they exist, also.
      int iRes = 0;  // Temporary result variable
      // Remove the WAL journal file, if it exists.
      if( 0 == stat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_FILE_NAME, &oStatBuf) )
      {
         ETG_TRACE_FATAL (( " -PHBK.ENCRYPTION-SDC-: INFO: The WAL journal file exists and will be deleted. [file= '%s'] ", FC_PB_DB_WAL_FILE_NAME ));
         ETG_TRACE_ERRMEM(( " -PHBK.ENCRYPTION-SDC-: INFO: The WAL journal file exists and will be deleted. [file= '%s'] ", FC_PB_DB_WAL_FILE_NAME ));

         iRes = remove(FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_FILE_NAME);
         if (iRes != 0)
         {
            iResult = -1;
            ETG_TRACE_FATAL (( " -PHBK.ENCRYPTION-SDC-: ERROR: Existing WAL journal file CANNOT be deleted. " ));
            ETG_TRACE_ERRMEM(( " -PHBK.ENCRYPTION-SDC-: ERROR: Existing WAL journal file CANNOT be deleted. " ));
         }
         else
         {
            ETG_TRACE_COMP  (( " -PHBK.ENCRYPTION-SDC-: SUCCESS: iRes= %d at remove('%s')", iRes, FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_FILE_NAME ));
         }
      }
      
      // Remove the SHM (SHared Memory) journal file, if it exists.
      if( 0 == stat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHM_FILE_NAME, &oStatBuf) )
      {
         ETG_TRACE_FATAL (( " -PHBK.ENCRYPTION-SDC-: Info: The SHM journal file exists and will be deleted. [file= '%s'] ", FC_PB_DB_SHM_FILE_NAME ));
         ETG_TRACE_ERRMEM(( " -PHBK.ENCRYPTION-SDC-: Info: The SHM journal file exists and will be deleted. [file= '%s'] ", FC_PB_DB_SHM_FILE_NAME ));

         iRes = remove(FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHM_FILE_NAME);
         if (iRes != 0)
         {
            iResult = -1;
            ETG_TRACE_FATAL (( " -PHBK.ENCRYPTION-SDC-: ERROR: Existing SHM journal file CANNOT be deleted. "  ));
            ETG_TRACE_ERRMEM(( " -PHBK.ENCRYPTION-SDC-: ERROR: Existing SHM journal file CANNOT be deleted. "  ));
         }
         else
         {
            ETG_TRACE_COMP  (( " -PHBK.ENCRYPTION-SDC-: SUCCESS: iRes= %d at remove('%s')", iRes, FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHM_FILE_NAME ));
         }
      }

      // Remove the DB "rollback" journal file, if it exists.
      if( 0 == stat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_JOURNAL_FILE_NAME, &oStatBuf) )
      {
         ETG_TRACE_FATAL (( " -PHBK-QUERY-ERR-: Info: The DB journal file exists and will be deleted. [file= '%s'] ", FC_PB_DB_JOURNAL_FILE_NAME ));
         ETG_TRACE_ERRMEM(( " -PHBK-QUERY-ERR-: Info: The DB journal file exists and will be deleted. [file= '%s'] ", FC_PB_DB_JOURNAL_FILE_NAME ));

         if( 0 != remove(FC_PB_DB_FILE_PATH "/" FC_PB_DB_JOURNAL_FILE_NAME) )
         {
            iResult = -1;
            ETG_TRACE_FATAL (( " -PHBK-QUERY-ERR-: ERROR: Existing DB journal file CANNOT be deleted. "  ));
            ETG_TRACE_ERRMEM(( " -PHBK-QUERY-ERR-: ERROR: Existing DB journal file CANNOT be deleted. "  ));
         }
      }

      // As final step the "database re-create trigger" file is removed.
      if( 0 != remove(FC_PB_DB_FILE_PATH "/" FC_PB_DB_RECREATE_TRIGGER_FILE_NAME) )
      {
         iResult = -1;
         ETG_TRACE_FATAL (( " -PHBK-QUERY-ERR-: Database RE-CREATION is INCOMPLETE: The 'recreate trigger' file CANNOT be deleted. "  ));
         ETG_TRACE_ERRMEM(( " -PHBK-QUERY-ERR-: Database RE-CREATION is INCOMPLETE: The 'recreate trigger' file CANNOT be deleted. "  ));
      }

      if(0 == iResult)
      {
         ETG_TRACE_FATAL (( " -PHBK-QUERY-ERR-: Database RE-CREATION: All affected files have been deleted with SUCCESS. "  ));
         ETG_TRACE_ERRMEM(( " -PHBK-QUERY-ERR-: Database RE-CREATION: All affected files have been deleted with SUCCESS. "  ));
      }
   }
   // -PHBK-QUERY-ERR- Instrumentation: END - for GMMY17-13044 "database disk image is malformed"

   
   // The following re-initalization of the member variables (which are FI data objects)
   // is needed when we are in a application state transition from OFF to NORMAL.
   // This re-initalization for FI data objects is a little bit tricky and therefore the 
   // steps are commented in more detail:

   // Calling .vDestroy() deletes the complex data structures (like strings or streams) 
   // in the FI data object. This step is necessary here to avoid memory leaks.
   // Note: .vDestroy() will not reset the basic data types (e.g. int, bool) to their
   // default values (which they get assigned when the FI data object is constructed).
   m_oDevicePhoneBookFeatureSupportStatus.vDestroy();
   m_oDevicePhoneBookFeatureSupportExtendedStatus.vDestroy();
   m_oMisssedCallIndicatorStatus.vDestroy();
   m_oListChangeStatus.vDestroy();
   m_oFavoritesChangedStatus.vDestroy();
   m_oDownloadState.vDestroy(); //FIX CMG3G-3262 Download State property implementation
   m_oDownloadOnOffListStatus.vDestroy();
   m_oDownloadStateExtended.vDestroy();//FIX CMG3GB-1754 new interface DownloadStateExtended required
   m_oAutomaticContactDownloadListStatus.vDestroy();
   m_oProbablyLockedDevices.vDestroy();
   m_oDownloadProgressListStatus.vDestroy(); //Fix RTC 159808
   m_oMissedCallIndicatorListStatus.vDestroy(); //CMG3G-12350
   m_oPreferredSortOrderListStatus.vDestroy();////CMG3G-14241
   m_oPhonebookLanguage_Status.vDestroy(); //NCG3D-44771
   m_oMaxContactCountStatus.vDestroy(); //CRQ-563636



   // Create temporary FI data objects with their default values for all data elements.
   // Note: No data will be filled to the temporary FI data objects in their life cycle (scope).
   // Therefore we do not need call .vDestroy() on them before leaving their scope.
   most_phonbkfi_tclMsgDevicePhoneBookFeatureSupportStatus           oDevicePhoneBookFeatureSupportStatus;
   most_phonbkfi_tclMsgDevicePhoneBookFeatureSupportExtendedStatus   oDevicePhoneBookFeatureSupportExtendedStatus;
   most_phonbkfi_tclMsgMissedCallIndicatorStatus                     oMisssedCallIndicatorStatus;
   most_phonbkfi_tclMsgListChangeStatus                              oListChangeStatus;
   most_phonbkfi_tclMsgFavoritesChangedStatus                        oFavoritesChangedStatus;
   most_phonbkfi_tclMsgDownloadStateStatus                           oDownloadState; //FIX CMG3G-3262 Download State property implementation
   most_phonbkfi_tclMsgDownloadOnOffListStatus                       oDownloadOnOffListStatus;
   most_phonbkfi_tclMsgDownloadStateExtendedStatus                   oDownloadStateExtended;//FIX CMG3GB-1754 new interface DownloadStateExtended required
   most_phonbkfi_tclMsgAutomaticContactDownloadListStatus            oAutomaticDownloadListStatus;
   most_phonbkfi_tclMsgProbablyLockedDevicesStatus                   oProbablyLockedDevices; //CMG3G-12880
   most_phonbkfi_tclMsgDownloadProgressListStatus                    oDownloadProgressListStatus; //Fix RTC 159808
   most_phonbkfi_tclMsgMissedCallIndicatorListStatus                 oMissedCallIndicatorListStatus;
   most_phonbkfi_tclMsgPreferredSortOrderListStatus                  oPreferredSortOrderListStatus; //CMG3G-14241
   most_phonbkfi_tclMsgPhonebookLanguageStatus                       oPhonebookLanguageStatus; //NCG3D-44771
   most_phonbkfi_tclMsgMaxContactCountStatus                         oMaxContactCountStatus; //CRQ-563636

   // The following assignment will do a shallow copy of all data elements of the FI data objects.
   // Afterwards our member variables are initialized with their default values again.
   m_oDevicePhoneBookFeatureSupportStatus = oDevicePhoneBookFeatureSupportStatus;
   m_oDevicePhoneBookFeatureSupportExtendedStatus = oDevicePhoneBookFeatureSupportExtendedStatus;
   m_oMisssedCallIndicatorStatus = oMisssedCallIndicatorStatus;
   m_oListChangeStatus = oListChangeStatus;
   m_oFavoritesChangedStatus = oFavoritesChangedStatus;
   m_oDownloadState = oDownloadState; //FIX CMG3G-3262 Download State property implementation
   m_oDownloadOnOffListStatus = oDownloadOnOffListStatus;
   m_oDownloadStateExtended = oDownloadStateExtended;//FIX CMG3GB-1754 new interface DownloadStateExtended required
   m_oAutomaticContactDownloadListStatus = oAutomaticDownloadListStatus;
   m_oProbablyLockedDevices = oProbablyLockedDevices; //CMG3G-12880
   m_oDownloadProgressListStatus = oDownloadProgressListStatus; //FIX RTC 159808
   m_oMissedCallIndicatorListStatus = oMissedCallIndicatorListStatus; //CMG3G-12350
   m_oPreferredSortOrderListStatus = oPreferredSortOrderListStatus; //CMG3G-14241
   m_oPhonebookLanguage_Status = oPhonebookLanguageStatus; //NCG3D-44771
   m_oMaxContactCountStatus = oMaxContactCountStatus; //CRQ-563636

   if(m_pPhoneBookDB)
   {
      m_pPhoneBookDB->
         getCurrentListInstance()->InitializeDatabase();
      m_oFavoritesChangedStatus.e8ListChangeType.enType=
         most_fi_tcl_e8_FavPrvListChangeType::FI_EN_E8LCH_CONTENT_CHANGED;
      m_oFavoritesChangedStatus.u32ListSize=
         m_pPhoneBookDB->GetTotalFavoritesCount();
   }
   else
   {
      ETG_TRACE_ERRMEM(("m_pPhoneBookDB IS NULL"));
   }

   if (NULL != m_pDBus_CCA_If_pb)
   {
      if(E_FAILURE == m_pDBus_CCA_If_pb->vPhoneBook_Load_MaxContactCount())
      {
         ETG_TRACE_ERR(("ERROR: Failed to load MaxContactCount.txt"));
      }
   }

   if(m_pPhoneBookMemoryUsage)
   {
      m_pPhoneBookMemoryUsage->CalculateMemoryUsageFromDB();
   }
   else
   {
      ETG_TRACE_ERRMEM(( " m_pPhoneBookMemoryUsage is NULL!!!" ));
   }


   vPhoneBook_UpdatePreferredPBSortOrderStatus();

   //CMG3G-14241
   vPhoneBook_UpdatePreferredSortOrderListStatus();

   //FIX CMG3G-8367 IS2424_DownloadOnOff behaviour@FC_Phonebook
   vPhoneBook_UpdateClientsDownloadOnOffList();
   // Changes of CMG3G-9889
   vPhoneBook_UpdateClientsAutomaticContactDownloadList();

   ETG_TRACE_USR4((" OnLoadSettings function exits "));
}


tVoid fc_phonebook_tclService_PhoneBook::vOnNewAppStateNotify(
   tU32 u32OldAppState, tU32 u32AppState)
{
   ETG_TRACE_COMP(( " vOnNewAppStateNotify Entered, OldAppState= %d, NewAppState= %d ", ETG_ENUM(ail_u32CCAState, u32OldAppState), 
                    ETG_ENUM(ail_u32CCAState, u32AppState) ));
   (void) u32OldAppState;
   (void) u32AppState;

#if 0 // FIX PSARCCB-9855 (06/2016):
   // The following section, which calls 'vOnSaveSettings' a second time when we are in transition to OFF has been found, 
   // when working on "FIX PSARCCB-9855 (06/2016)". I don't see that this second call might help or might be needed.
   // I have commented out this section completly.
   
   if (AMT_C_U32_STATE_OFF == u32AppState)
   {
      ETG_TRACE_USR4(("Changed to OFF state"));
      // OnSaveSettings();
      //Fix for GMMY15-11113--Reset Tickets on phonebook
      fc_phonebook_tclApp::m_poMainAppInstance->vOnSaveSettings();
   }
#endif

}

tVoid fc_phonebook_tclService_PhoneBook::vHandleMethodError(
   most_phonbkfi_tclMsgDefaultError& defaultError,amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandleMethodError : Called"));

   defaultError.e8ErrorCode.enType
      = most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC;

   fi_tclVisitorMessage oMRMsg(defaultError,
      MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData(oMRMsg, poMessage->u16GetSourceAppID(), // ServiceDataMsg
      poMessage->u16GetRegisterID(), // RegisterID
      poMessage->u16GetCmdCounter(), // CmdCounter,
      poMessage->u16GetFunctionID(), // Function ID
      AMT_C_U8_CCAMSG_OPCODE_ERROR // Opcode
      );

   if (oMRMsg.bIsValid())
   {
      // Post message
      ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

      if (enResult != AIL_EN_N_NO_ERROR)
      {
         ETG_TRACE_ERR(("vHandleMethodError: enPostMessage() failed"));
         NORMAL_M_ASSERT_ALWAYS();
      }
      else
      {
         ETG_TRACE_USR4(("enPostMessage : Message Posted Successfully"));
      }
   }
   else
   {
      vHandleGetDataError(poMessage, "vHandleMethodError");
   }
}

//FIX CMG3G-3262 Download State property implementation
/*******************************************************************************
** FUNCTION:   vPhoneBook_UpdateDownloadState(..)
*******************************************************************************/
/* DESCRIPTION:
*  Update clients if there is any change in DownloadState
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vPhoneBook_UpdateDownloadState()
{
   ETG_TRACE_USR4((" vPhoneBook_UpdateDownloadState entered"));

   //Send update only if there is any change in the previous value.
   tBool bUpdateRequired = TRUE;

   if(m_poDownloadState != NULL)
   {
      ETG_TRACE_USR4((" DOWNLOAD STATE EXTENDED PROPERTY UPDATE..."));
      most_fi_tcl_PhonBkDownloadStateExtendedStreamItem oExtendedStreamItem;
      tU8 u8DownloadStateExtendedStreamSize = (tU8)(m_oDownloadStateExtended.oDownloadStateExtendedStream.oItems.size());

      //Check whether update is to be sent or not. Send only if there is any change in the previous value.
      for (tU8 u8Index = 0; u8Index < u8DownloadStateExtendedStreamSize; u8Index++)
      {
         if (m_oDownloadStateExtended.oDownloadStateExtendedStream.oItems[u8Index].u8DeviceHandle
               == m_poDownloadState->getDeviceHandle())
         {
            if ((m_oDownloadStateExtended.oDownloadStateExtendedStream.oItems[u8Index].e8RecentCallListDownloadState.enType
                  == (most_fi_tcl_e8_PhonBkRecentCallListDownloadStateExtended::tenType) m_poDownloadState->u8GetRecentCallListDownloadState())
                  && (m_oDownloadStateExtended.oDownloadStateExtendedStream.oItems[u8Index].e8PhoneBookDownloadState.enType
                        == (most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended::tenType) m_poDownloadState->u8GetPBDownloadState()))
            {
               //Fix RTC 176167
               //Phonebook Download state extended update will be send though if previous state is same i.e., only in ERROR case
               //Reason: If previous download state is error and user triggers manual download or sync contacts on HMI
               //actual download sequence will start and if error occurs we need to update the same to HMI, so that
               //manual download popup will be discarded
               if (most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended::FI_EN_E8PBDS_ERROR
                     == (most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended::tenType) m_poDownloadState->u8GetPBDownloadState())
               {
                  bUpdateRequired = TRUE;
               }
               else
               {
                  bUpdateRequired = FALSE;
               }
               //End fix RTC 176167
            }
         }
      }

      if(TRUE == bUpdateRequired)
      {
         ETG_TRACE_USR4((" bUpdateRequired is TRUE"));
         ETG_TRACE_USR4((" Updating stream items for DownloadStateExtended."));
         oExtendedStreamItem.e8RecentCallListDownloadState.enType = (most_fi_tcl_e8_PhonBkRecentCallListDownloadStateExtended::tenType) m_poDownloadState->u8GetRecentCallListDownloadState();
         oExtendedStreamItem.e8PhoneBookDownloadState.enType = (most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended::tenType) m_poDownloadState->u8GetPBDownloadState();
         oExtendedStreamItem.u8DeviceHandle = m_poDownloadState->getDeviceHandle();

         //Fix for CMG3G-11146
         //Update member variables so that they are available when clients do a 'GET' on the property.
         ETG_TRACE_USR4((" Updating member variable m_oDownloadStateExtended"));
         bUpdateRequired = FALSE;
         tU8 u8Index = 0;
         for (u8Index = 0; u8Index < u8DownloadStateExtendedStreamSize; u8Index++)
         {
            if (m_poDownloadState->getDeviceHandle()
                  == m_oDownloadStateExtended.oDownloadStateExtendedStream.oItems[u8Index].u8DeviceHandle)
            {
               bUpdateRequired = TRUE;
               break;
            }
         }

         if(TRUE == bUpdateRequired)
         {
            //If the same device handle is existed in downloadStateExtendedtream, then update the previous values
            m_oDownloadStateExtended.oDownloadStateExtendedStream.oItems[u8Index] = oExtendedStreamItem;
         }
         else
         {
            //Inserting the new device handle into downloadStateExtendedtream for the first time
            m_oDownloadStateExtended.oDownloadStateExtendedStream.oItems.push_back(oExtendedStreamItem);
         }
         //End of fix CMG3G-11146

         //Update download state extended property to clients.
         ETG_TRACE_USR4((" Updating clients about DownloadState Change."));
         ail_tenCommunicationError enResultExtended = eUpdateClients(
            MOST_PHONBKFI_C_U16_DOWNLOADSTATEEXTENDED);
         if (AIL_EN_N_NO_ERROR != enResultExtended)
         {
            ETG_TRACE_ERR((" ERROR fc_phonebook_tclService_PhoneBook:- Sending of Download State Extended update failed "));
         }
         else
         {
            ETG_TRACE_USR4((" Sending of Download State Extended Property Successfull"));
         }
      }
      else
      {
         ETG_TRACE_USR4((" bUpdateRequired is FALSE"));
      }

      ETG_TRACE_USR4((" DOWNLOAD STATE PROPERTY UPDATE..."));
      most_fi_tcl_PhonBkDownloadStateStreamItem oStreamItem;

      tU8 u8DownloadStateStreamSize = (tU8)(m_oDownloadState.oDownloadStateStream.oItems.size());
      //Check whether update is to be sent or not. Send only if there is any change in the previous value.
      bUpdateRequired = TRUE; //Reset variable

      for (tU8 u8Index = 0; u8Index < u8DownloadStateStreamSize; u8Index++)
      {
         if (m_oDownloadState.oDownloadStateStream.oItems[u8DownloadStateStreamSize].u8DeviceHandle
               == m_poDownloadState->getDeviceHandle())
         {
            if ((m_oDownloadState.oDownloadStateStream.oItems[u8DownloadStateStreamSize].e8RecentCallListDownloadState.enType
                  == (most_fi_tcl_e8_PhonBkRecentCallListDownloadState::tenType) m_poDownloadState->u8GetRecentCallListDownloadState())
                  && (m_oDownloadState.oDownloadStateStream.oItems[u8DownloadStateStreamSize].e8PhoneBookDownloadState.enType
                        == (most_fi_tcl_e8_PhonBkPhoneBookDownloadState::tenType) m_poDownloadState->u8GetPBDownloadState()))
            {
               bUpdateRequired = FALSE;
            }
         }
      }

      if(TRUE == bUpdateRequired)
      {
         ETG_TRACE_USR4((" bUpdateRequired is TRUE"));
         ETG_TRACE_USR4((" Updating stream items for DownloadState."));
         if (most_fi_tcl_e8_PhonBkRecentCallListDownloadStateExtended::FI_EN_E8RCDS_ERROR
               == m_poDownloadState->u8GetRecentCallListDownloadState())
         {
            oStreamItem.e8RecentCallListDownloadState.enType
                  = most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_COMPLETE;
         }
         else
         {
            oStreamItem.e8RecentCallListDownloadState.enType
                  = (most_fi_tcl_e8_PhonBkRecentCallListDownloadState::tenType) m_poDownloadState->u8GetRecentCallListDownloadState();
         }

         if (most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended::FI_EN_E8PBDS_ERROR
               == m_poDownloadState->u8GetPBDownloadState()
               || (most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended::FI_EN_E8PBDS_LIMIT_REACHED
                     == m_poDownloadState->u8GetPBDownloadState()))
         {
            oStreamItem.e8PhoneBookDownloadState.enType
                  = most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_COMPLETE;
         }

         else
         {
            oStreamItem.e8PhoneBookDownloadState.enType
                  = (most_fi_tcl_e8_PhonBkPhoneBookDownloadState::tenType) m_poDownloadState->u8GetPBDownloadState();
         }
         oStreamItem.u8DeviceHandle = m_poDownloadState->getDeviceHandle();

         //Fix for CMG3G-11146
         //Update member variables so that they are available when clients do a 'GET' on the property.
         ETG_TRACE_USR4((" Updating member variable m_oDownloadState"));
         bUpdateRequired = FALSE;
         tU8 u8Index = 0;
         for (u8Index = 0; u8Index < u8DownloadStateStreamSize; u8Index++)
         {
            if (m_poDownloadState->getDeviceHandle()
                  == m_oDownloadState.oDownloadStateStream.oItems[u8Index].u8DeviceHandle)
            {
               bUpdateRequired = TRUE;
               break;
            }
         }

         if(TRUE == bUpdateRequired)
         {
            //If the same device handle is existed in downloadStateStream, then update the previous values
            m_oDownloadState.oDownloadStateStream.oItems[u8Index] = oStreamItem;
         }
         else
         {
            //Inserting the new device handle into downloadStateStream for the first time
            m_oDownloadState.oDownloadStateStream.oItems.push_back(oStreamItem);
         }
         //End of fix CMG3G-11146

         //Update Download state property to clients.
         ETG_TRACE_USR4((" Updating clients about DownloadState Change."));
         ail_tenCommunicationError enResult = eUpdateClients(
            MOST_PHONBKFI_C_U16_DOWNLOADSTATE);
         if (AIL_EN_N_NO_ERROR != enResult)
         {
            ETG_TRACE_ERR((" Sending of Download State update failed "));
         }
         else
         {
            ETG_TRACE_USR4((" Sending of Download State Property Successfull"));
         }
      }
      else
      {
         ETG_TRACE_USR4((" bUpdateRequired is FALSE"));
      }
   }
   else
   {
      ETG_TRACE_ERR((" m_poDownloadState is NULL"));
   }
}
/*******************************************************************************
** FUNCTION:   Constructor(..)
*******************************************************************************/
/* DESCRIPTION: Assigns default values for download state as 
*               most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_NOT_STARTED, most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_NOT_STARTED 
*               and device handle as zero
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
FC_PhoneBook_DownloadState::FC_PhoneBook_DownloadState()
{
   ETG_TRACE_USR4(("DownloadState constructor entered"));
   u8PB_DownloadState = most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_NOT_STARTED;
   u8RecentCallList_DownloadState = most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_NOT_STARTED;
   u8DeviceHandle = 0;
}


/*******************************************************************************
** FUNCTION:   Destructor(..)
*******************************************************************************/
/* DESCRIPTION: Assigns default values for download state as 
*               most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_NOT_STARTED, most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_NOT_STARTED 
*               and device handle as zero
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
FC_PhoneBook_DownloadState::~FC_PhoneBook_DownloadState()
{
   ETG_TRACE_USR4(("DownloadState destructor entered"));
   u8PB_DownloadState = most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_NOT_STARTED;
   u8RecentCallList_DownloadState = most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_NOT_STARTED;
   u8DeviceHandle = 0;
}


/*******************************************************************************
** FUNCTION:   vSetPBDownloadState(..)
*******************************************************************************/
/* DESCRIPTION: Sets u8PB_DownloadState with new value Download_State
*  PARAMETERS: 
*     IN: Download_State: New value for u8PB_DownloadState
*  RETURNVALUE: None
*******************************************************************************/
tVoid FC_PhoneBook_DownloadState::vSetPBDownloadState(tU8 u8Download_State)
{
   ETG_TRACE_USR4(("vSetPBDownloadState entered with Download state as :%d",u8Download_State));
   u8PB_DownloadState = u8Download_State;
}


/*******************************************************************************
** FUNCTION:   vSetRecentCallListDownloadState(..)
*******************************************************************************/
/* DESCRIPTION: Sets u8RecentCallList_DownloadState with new value Download_State
*  PARAMETERS: 
*     IN: Download_State: New value for u8RecentCallList_DownloadState
*  RETURNVALUE: None
*******************************************************************************/
tVoid FC_PhoneBook_DownloadState::vSetRecentCallListDownloadState(tU8 u8Download_State)
{
   ETG_TRACE_USR4(("vSetRecentCallListDownloadState entered  with Download state as :%d",u8Download_State));
   u8RecentCallList_DownloadState = u8Download_State;
}


/*******************************************************************************
** FUNCTION:   vSetDeviceHandle(..)
*******************************************************************************/
/* DESCRIPTION: Sets device handle with new value u8SetDeviceHandle
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid FC_PhoneBook_DownloadState::vSetDeviceHandle(tU8 u8SetDeviceHandle)
{
   ETG_TRACE_USR4(("vSetDeviceHandle entered with u8SetDeviceHandle :%d",u8SetDeviceHandle));
   u8DeviceHandle = u8SetDeviceHandle;
}

//End of fix CMG3G-3262
//FIX SUZUKI-1114 API for numeric value to indicate number of dial-able phone numbers available to a contact item in Phonebook
/*******************************************************************************
** FUNCTION:   vHandle_RequestPhoneBookListSliceExtended_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for vHandle_RequestPhoneBookListSliceExtended_MethodStart
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::\
vHandle_RequestPhoneBookListSliceExtended_MethodStart(\
                                                      amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR4(("vHandle_RequestPhoneBookListSliceExtended_MethodStart(). entered"));

   // Extract the Call instance and other required info from poMessage
   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );

   // Create the (message related) FI data object
   most_phonbkfi_tclMsgRequestPhoneBookListSliceExtendedMethodStart \
      oRequestSlicePhoneBookListExtendedMethodStart_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.

   if ( oVisitorMsg.s32GetData(oRequestSlicePhoneBookListExtendedMethodStart_MS, \
      MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {

      ETG_TRACE_USR2(("LIST HANDLE: %u",oRequestSlicePhoneBookListExtendedMethodStart_MS.u16ListHandle));
      ETG_TRACE_USR2(("WINDOW START: %d",oRequestSlicePhoneBookListExtendedMethodStart_MS.u32WindowStart));
      ETG_TRACE_USR2(("WINDOW SIZE: %u",oRequestSlicePhoneBookListExtendedMethodStart_MS.u16WindowSize));

      most_phonbkfi_tclMsgRequestPhoneBookListSliceExtendedMethodResult \
         oRequestSlicePhoneBookListExtended_MR;

      //RequestSlicePhoneBookListExtended call from Sqlite data base//
      tBool dbStatus = FALSE;
      if(m_pPhoneBookDB)
      {
         dbStatus = \
               m_pPhoneBookDB->RequestSlicePhoneBookListExtended( &oRequestSlicePhoneBookListExtendedMethodStart_MS ,
                                                            &oRequestSlicePhoneBookListExtended_MR);
      }
      // Get the coresponding Call Instance

      if(dbStatus != TRUE)
      {
         //DB Returned error

         amt_tclServiceDataError oErrorMsg(*poMessage, most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC);

         // Post message
         ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

         if (enResult != AIL_EN_N_NO_ERROR)
         {

            ETG_TRACE_ERR(("vHandle_RequestPhoneBookListSliceExtended_MethodStart: 'ERROR', enPostMessage() failed"));
            // NORMAL_M_ASSERT_ALWAYS();
         }

         // NORMAL_M_ASSERT_ALWAYS();

      }
      else
      {
         fi_tclVisitorMessage oMRMsg(oRequestSlicePhoneBookListExtended_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
         vInitServiceData( oMRMsg,
            poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(),  // RegisterID
            poMessage->u16GetCmdCounter(),  // CmdCounter
            poMessage->u16GetFunctionID(),  // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT    // Opcode
            );

         if (oMRMsg.bIsValid())
         {

            // Post message
            ail_tenCommunicationError enResult =
               _poMainAppl->enPostMessage(&oMRMsg, TRUE);

            if (enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_ERR(( "vHandle_RequestPhoneBookListSliceExtended_MethodStart:'MethodResult', enPostMessage() failed,'ail_tenCommunicationError' = %u",(tU32)enResult ));
               // We can come here e.g. if the client has changed his application
               // state to OFF in the meantime. Therefore we dont throw an assert.
               // NORMAL_M_ASSERT_ALWAYS();
            }
         }
         else
         {
            most_phonbkfi_tclMsgRequestPhoneBookListSliceExtendedError requestSlicePBErrorExtended;
            vHandleMethodError(requestSlicePBErrorExtended,poMessage);
            requestSlicePBErrorExtended.vDestroy();
         }

      }

      oRequestSlicePhoneBookListExtended_MR.vDestroy();

   }
   else
   {
      // If we come here, we assume the client has sent an erroneous message,
      // probably it has used a wrong FI_MAJOR_VERSION. Therefore we will
      // send back a an error message and throw a NORMAL_M_ASSERT_ALWAYS().

      vHandleGetDataError(poMessage,"vHandle_RequestPhoneBookListSliceExtended_MethodStart");
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)

   oRequestSlicePhoneBookListExtendedMethodStart_MS.vDestroy();
}
//End of fix
//FIX SUZUKI-1113 API for numeric value to indicate number of calls against a CallHistory item in Phonebook
/*******************************************************************************
** FUNCTION:   vHandle_RequestSliceCallHistoryListExtended_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for RequestSliceCallHistoryListExtended method
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_RequestSliceCallHistoryListExtended_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("MethodStart: vHandle_RequestSliceCallHistoryListExtended_MethodStart entered  "));
   /* Extract the Call instance and other required info from poMessage */
   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg( poMessage );

   // Create the (message related) FI data object
   most_phonbkfi_tclMsgRequestSliceCallHistoryListExtendedMethodStart oRequestSliceCallHistoryListExtended_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.

   if ( oVisitorMsg.s32GetData(oRequestSliceCallHistoryListExtended_MS, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {


      ETG_TRACE_USR2(("	LIST HANDLE: %u", oRequestSliceCallHistoryListExtended_MS.u16ListHandle));

      ETG_TRACE_USR2(("	WINDOW START: %d", oRequestSliceCallHistoryListExtended_MS.u32WindowStart));

      ETG_TRACE_USR2(("	WINDOW SIZE: %u",  oRequestSliceCallHistoryListExtended_MS.u16WindowSize));



      most_phonbkfi_tclMsgRequestSliceCallHistoryListExtendedMethodResult oRequestSliceCallHistoryListExtended_MR;

      //RequestSliceCallHistoryListExtended call from Sqlite data base//
      tBool dbStatus = FALSE;
      if(m_pPhoneBookDB)
      {
         dbStatus = m_pPhoneBookDB->RequestSliceCallHistoryListExtended( &oRequestSliceCallHistoryListExtended_MS , &oRequestSliceCallHistoryListExtended_MR,poMessage->u16GetRegisterID());//FIX NCG3D-55494 RegisterID is passed to identify the requested client);
      }

      //Get the coresponding Call Instance

      if(dbStatus != TRUE)
      {
         ETG_TRACE_USR4(("Error message posted successfully."));
         most_phonbkfi_tclMsgRequestSliceCallHistoryListExtendedError callHistoryListErrorExtended;
         vHandleMethodError(callHistoryListErrorExtended,poMessage);
         callHistoryListErrorExtended.vDestroy();
      }
      else
      {
         fi_tclVisitorMessage oMRMsg(oRequestSliceCallHistoryListExtended_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
         vInitServiceData( oMRMsg,
            poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(),  // RegisterID
            poMessage->u16GetCmdCounter(),  // CmdCounter
            poMessage->u16GetFunctionID(),  // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT    // Opcode
            );

         if (oMRMsg.bIsValid())
         {

            // Post message
            ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);

            if (enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_USR4(("Method result posted successfully."));
            }
         }
         else
         {
            most_phonbkfi_tclMsgRequestSliceCallHistoryListExtendedError callHistoryListErrorExtended;
            vHandleMethodError(callHistoryListErrorExtended,poMessage);
            callHistoryListErrorExtended.vDestroy();
         }
      }
      oRequestSliceCallHistoryListExtended_MR.vDestroy();
   }
   else
   {
      // If we come here, we assume the client has sent an erroneous message,
      // probably it has used a wrong FI_MAJOR_VERSION.
      // send back a an error message and throw a NORMAL_M_ASSERT_ALWAYS().

      vHandleGetDataError(poMessage,"vHandle_RequestSliceCallHistoryListExtended_MethodStart");
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)

   oRequestSliceCallHistoryListExtended_MS.vDestroy();

}
//End of fix

/*******************************************************************************
 ** FUNCTION:   vHandle_StartPbDownload_MethodStart(..)
 *******************************************************************************/
/* DESCRIPTION:
 *
 *   Doors ID: 20263: Manual Phonebook download
 *
 *   This is a handler function for StartPbDownload method
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 *******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_StartPbDownload_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("MethodStart: vHandle_StartPbDownload_MethodStart entered  "));

   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // Create the (message related) FI data object
   most_phonbkfi_tclMsgStartPbDownloadMethodStart oStartPbDownload_MS;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.

   if (oVisitorMsg.s32GetData(oStartPbDownload_MS, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      tBool bSuccess = false;

      tU8 u8DeviceHandle = oStartPbDownload_MS.u8DeviceHandle;
      ETG_TRACE_USR2(("vHandle_StartPbDownload_MethodStart: Device Handle: %d", u8DeviceHandle));
      
      QMutexLocker lock(&g_Phonebook_DeviceMutex);
      FC_Device_Details* pDeviceObject = FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);
      //If device is not in probably locked and download suspended, then only initiate manual download
      if ( pDeviceObject && (FALSE == pDeviceObject->bIsDeviceProbablyLocked() ) && (FALSE == pDeviceObject->bIsDeviceDownloadSuspended()) )
      {
         pDeviceObject->setCancelPBDownload(FALSE);//Reset as another manual download has been triggered by user.

         //Fix Bug 277614 - Check whether PBAP session is created or not
         if (bCheckSessionPathAvailable(pDeviceObject))
         {
            ETG_TRACE_USR4(("vHandle_StartPbDownload_MethodStart: Device is connected"));
            FC_PhoneBook_SQLite* poFC_PhoneBook_SQLite = FC_PhoneBook_SQLite::GetSQLiteInstance();

            if (poFC_PhoneBook_SQLite)
            {
               tU16 l_U16DownloadOnOffState = poFC_PhoneBook_SQLite->u16GetDownloadOnOffStateForDevicehandle(u8DeviceHandle);
               ETG_TRACE_USR4(("vHandle_StartPbDownload_MethodStart: DownloadOnOffState: %d ", l_U16DownloadOnOffState));

               if (PB_DL_ON == l_U16DownloadOnOffState)
               {
                  if (FC_Device_Details::DOWNLOAD_ST_IDLE
                        == pDeviceObject->GetDownloadState()
                        || FC_Device_Details::DOWNLOAD_ST_NOT_STARTED
                              == pDeviceObject->GetDownloadState()) // No Download is in progress
                  {
                     // Start Fresh download
                     // TBC: What other properties has to be reset before starting the download
                     // TODO: Modify the existing download code flow to incorporate Manual PB download.
                     ETG_TRACE_COMP((" -PBDL.S- Starting Manual Download of contacts... "));

                     pDeviceObject->setPhBkPrepareCallBackStatus(FALSE);
                     pDeviceObject->setCapInProgress(FALSE);
                     pDeviceObject->RetryCnt = 0;
                     m_oPbk_DnlManager.vStartDnlTime();
                     vClearMissedCalls(u8DeviceHandle, pDeviceObject);

                     ETG_TRACE_USR4((" Download State changed - DOWNLOAD_ST_PBK_PROGRESS"));
                     pDeviceObject->SetDownloadState(FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS);

                     poFC_PhoneBook_SQLite->bUpdateDlCompletedOnceValue(u8DeviceHandle, false);
                     //Fix NCG3D-4839 Call History is also downloaded when Manual phonebook download is requested
                     //Only contacts need to be downloaded as part of Manual PB download.
                     if (m_pDBus_CCA_If_pb)
                     {
                        m_pDBus_CCA_If_pb->vPhoneBook_StartDlPB(pDeviceObject, PBInLocal);
                     }
                     //End of fix
                     vPhoneBook_DownloadInitiated(pDeviceObject);
                  }
                  else if ((FC_Device_Details::DOWNLOAD_ST_CH_UPDATE
                        == pDeviceObject->GetDownloadState()
                        || FC_Device_Details::DOWNLOAD_ST_CH_PROGRESS
                              == pDeviceObject->GetDownloadState())
                        && (pDeviceObject->m_u8PendingManualPBCommandList.isEmpty())) // Currently CH Update is in progress
                  {
                     ETG_TRACE_USR4((" vHandle_StartPbDownload_MethodStart: Currently CH download is in progress"));

                     tBool bPBDownloadInArray = FALSE;
                     for (tU8 u8Itr = VALUE_ZERO; u8Itr < MAX_PBDL_TYPE_SUPPORTED; u8Itr++)
                     {
                        if ((pDeviceObject->u8DownloadTypeList[u8Itr]
                              == DownLoadPBSim)
                              || (pDeviceObject->u8DownloadTypeList[u8Itr]
                                    == DownLoadPBLocal)
                              || (pDeviceObject->u8DownloadTypeList[u8Itr]
                                    == DownloadPBSpeedDial)
                              || (pDeviceObject->u8DownloadTypeList[u8Itr]
                                    == DownloadPBFavorites))
                        {
                           ETG_TRACE_USR4((" Phonebook contacts download already in download array..."));
                           bPBDownloadInArray = TRUE;
                           break;
                        }
                     }

                     //If there is no pending contacts download in download array, then add it to pending list.
                     if (FALSE == bPBDownloadInArray)
                     {
                        ETG_TRACE_USR4((" Adding manual PB download to pending download array..."));
                        poFC_PhoneBook_SQLite->bUpdateDlCompletedOnceValue(u8DeviceHandle, false);
                        m_oPbk_DnlManager.vFillPendingManualPBCommandList(pDeviceObject, PBInLocal);
                     }
                  }

                  // When the DownloadState suggest that either any download is in progress or is there any CH update happening already, directly post the MethodResult

                  most_phonbkfi_tclMsgStartPbDownloadMethodResult
                        oStartPbDownload_MR;
                  oStartPbDownload_MR.u8DeviceHandle = u8DeviceHandle;

                  fi_tclVisitorMessage
                        oMRMsg(oStartPbDownload_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);

                  vInitServiceData(oMRMsg, poMessage->u16GetSourceAppID(), // ServiceDataMsg
                  poMessage->u16GetRegisterID(), // RegisterID
                  poMessage->u16GetCmdCounter(), // CmdCounter
                  poMessage->u16GetFunctionID(), // Function ID
                  AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
                  );

                  if (oMRMsg.bIsValid())
                  {
                     // Post message
                     ail_tenCommunicationError enResult =
                           _poMainAppl->enPostMessage(&oMRMsg, TRUE);

                     if (enResult != AIL_EN_N_NO_ERROR)
                     {
                        ETG_TRACE_USR4(("Method result posting ERROR"));
                     }
                     else
                     {
                        ETG_TRACE_USR4(("vHandle_StartPbDownload_MethodStart: MethodResult Posted Successfully"));
                        bSuccess = true;
                     }
                  }
                  else
                  {
                     ETG_TRACE_USR4(("Method result instance creation failed"));
                  }

                  oStartPbDownload_MR.vDestroy();
               }
               else
               {
                  ETG_TRACE_ERR(("vHandle_StartPbDownload_MethodStart: DownloadOnOff state is OFF. "));
               }
            }
            else
            {
               ETG_TRACE_ERR(("vHandle_StartPbDownload_MethodStart: SQLite object is NULL. "));
            }
         }
         else
         {
            ETG_TRACE_USR4(("Session path is null"));
            if (m_poDownloadState)
            {
               ETG_TRACE_USR4(("Setting DownloadState as FI_EN_E8PBDS_ERROR"));

               m_poDownloadState->vSetPBDownloadState(most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended::FI_EN_E8PBDS_ERROR);

               vPhoneBook_UpdateDownloadState();
            }
            else
            {
               ETG_TRACE_USR4(("m_poDownloadState is null"));
            }
         }
      }
      else
      {
         ETG_TRACE_ERR(("vHandle_StartPbDownload_MethodStart: Device details is NULL or Device is in probably locked state or Device is download is supsended"));
      }

      if (FALSE == bSuccess)
      {
         most_phonbkfi_tclMsgStartPbDownloadError oStartPbDownloadError;
         vHandleMethodError(oStartPbDownloadError, poMessage);
         oStartPbDownloadError.vDestroy();
      }
   }
   else
   {
      // If we come here, we assume the client has sent an erroneous message,
      // probably it has used a wrong FI_MAJOR_VERSION.
      vHandleGetDataError(poMessage, "vHandle_StartPbDownload_MethodStart");
   }

   // Always destroy the (message related) FI data object (before leaving
   // its creation scope)
   oStartPbDownload_MS.vDestroy();
}

//Fix for CMG3GB-1093
/*******************************************************************************
** FUNCTION:   bCheckAudioChannel(..)
*******************************************************************************/
/* DESCRIPTION: Checks if the AudioChannel parameter is valid
*  PARAMETERS:
*     IN:
*  RETURNVALUE: tBool bResult
*******************************************************************************/
tBool fc_phonebook_tclService_PhoneBook::bCheckAudioChannel(most_fi_tcl_e8_FavPrvAudioChannel::tenType e8AudioChannel)
{
   ETG_TRACE_USR4(("bCheckAudioChannel entered with e8AudioChannel :%d", e8AudioChannel));

   tBool bResult = FALSE;
   if ((e8AudioChannel
      == most_fi_tcl_e8_FavPrvAudioChannel::FI_EN_E8LC_NONE)
      || (e8AudioChannel
      == most_fi_tcl_e8_FavPrvAudioChannel::FI_EN_E8LC_MAIN_AUDIO)
      || (e8AudioChannel
      == most_fi_tcl_e8_FavPrvAudioChannel::FI_EN_E8LC_PHONE)
      || (e8AudioChannel
      == most_fi_tcl_e8_FavPrvAudioChannel::FI_EN_E8LC_REAR_AUDIO)
      || (e8AudioChannel
      == most_fi_tcl_e8_FavPrvAudioChannel::FI_EN_E8LC_REAR_PHONE)
      || (e8AudioChannel
      == most_fi_tcl_e8_FavPrvAudioChannel::FI_EN_E8LC_HEADPHONE_1)
      || (e8AudioChannel
      == most_fi_tcl_e8_FavPrvAudioChannel::FI_EN_E8LC_HEADPHONE_2)
      || (e8AudioChannel
      == most_fi_tcl_e8_FavPrvAudioChannel::FI_EN_E8LC_HEADPHONE_3)
      || (e8AudioChannel
      == most_fi_tcl_e8_FavPrvAudioChannel::FI_EN_E8LC_HEADPHONE_4))
   {
      bResult = TRUE;
   }
   ETG_TRACE_USR4(("bCheckAudioChannel bResult :%d", bResult));
   return bResult;
}
// End of Fix CMG3GB-1093

//FIX SUZUKI-20814 - Reset observed on varying the voltage (5.7V to 14V)
/*******************************************************************************
 ** FUNCTION:   vHandleProxyDereferencing(..)
 *******************************************************************************/
/* DESCRIPTION: Disconnect signals and Dereference Proxy on ignition OFF for
 *              current outgoing source, if any.
 *  PARAMETERS:
 *     IN:       None.
 *     OUT:      None.
 *  RETURNVALUE: None.
 *******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandleProxyDereferencing(tVoid)
{
   ETG_TRACE_USR4((" vHandleProxyDereferencing() entered"));
   for (tU8 i = 0; i < NO_OF_CONNECTED_DEVICE; i++)
   {
       //Unref the available proxies.
       vClearEvoProxyConnMapping(i);
       //Remove the object paths.
       vClearEvoObjectPathMapping(i);
   }
}
//End of fix


//FIX CMG3G-8372 IS2430_SetContactPhoto@FC_Phonebook
//The user shall have possibility to set the photo for a specific contact in Vehicle phonebook of a specific device.
/*******************************************************************************
** FUNCTION:   vHandle_SetContactPhoto_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION: Handler function for the SetContactPhoto Method Start
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::\
vHandle_SetContactPhoto_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4((" Function vHandle_SetContactPhoto_MethodStart entered"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);

   most_phonbkfi_tclMsgSetContactPhotoMethodStart oSetContactPhoto_MS;

   if (oVisitorMsg.s32GetData(oSetContactPhoto_MS, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      ETG_TRACE_USR2((" Contact Handle: %d", oSetContactPhoto_MS.u32ContactHandle));

      tBool bSuccess = false, bError = false;

      tU32 ContactHandle = oSetContactPhoto_MS.u32ContactHandle;

      tU32 PhotoStreamSize = (tU32)(oSetContactPhoto_MS.oVehicleContactPhoto.u8Items.size());
      ETG_TRACE_USR2((" Photo size: %d", PhotoStreamSize));

      if ((VALUE_ZERO == ContactHandle) || ((ContactHandle & FC_PB_CONTACT_HANDLE_DPB_MARKER) == FC_PB_CONTACT_HANDLE_DPB_MARKER)) //Contact Handle is invalid
      {
         ETG_TRACE_USR2((" Invalid Contact Handle: %d", oSetContactPhoto_MS.u32ContactHandle));
         bError = true;
      }

      if (FALSE == bError)
      {
         QByteArray PhotoData = QByteArray();
         for (tU32 u32Itr = VALUE_ZERO; u32Itr < PhotoStreamSize; u32Itr++)
         {
            tU8 photoItem = oSetContactPhoto_MS.oVehicleContactPhoto.u8Items[u32Itr];
            PhotoData.push_back((char)photoItem);
         }

         //Set Photo in DB
         ETG_TRACE_USR2((" Setting contact photo..."));
         bSuccess = FC_PhoneBook_SQLite::GetSQLiteInstance()->SetContactPhoto(ContactHandle, PhotoData);

         if (TRUE == bSuccess)
         {
            most_phonbkfi_tclMsgSetContactPhotoMethodResult oSetContactPhoto_MR;
            oSetContactPhoto_MR.u32ContactHandle = oSetContactPhoto_MS.u32ContactHandle;

            fi_tclVisitorMessage oMRMsg(oSetContactPhoto_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);

            vInitServiceData(oMRMsg, poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(), // RegisterID
            poMessage->u16GetCmdCounter(), // CmdCounter
            poMessage->u16GetFunctionID(), // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
            );

            if (oMRMsg.bIsValid())
            {
               ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);
               if (enResult != AIL_EN_N_NO_ERROR)
               {
                  bError = true;
                  ETG_TRACE_ERR((" vHandle_SetContactPhoto_MethodStart: 'Result', enPostMessage() failed, 'ail_tenCommunicationError' = %d", (tU32) enResult));
               }
               else
               {
                  ETG_TRACE_USR4((" List to be updated with new phone book lists after setting photo "));
                  if(m_pPhoneBookDB)
                  {
                     m_pPhoneBookDB->MethodResultPosted(poMessage->u16GetFunctionID(), &oSetContactPhoto_MS, &oSetContactPhoto_MR);
                  }
               }
            }
            else
            {
               ETG_TRACE_ERR((" Function vHandle_SetContactPhoto_MethodStart: oMRMsg is invalid"));
               bError = true;
            }

            oSetContactPhoto_MR.vDestroy();
         }
         else
         {
            ETG_TRACE_ERR((" vHandle_SetContactPhoto_MethodStart: Updating the database table failed"));
            bError = true;
         }
      }
      else
      {
         ETG_TRACE_ERR((" vHandle_SetContactPhoto_MethodStart: Parameters are invalid"));
         bError = true;
      }

      if (TRUE == bError)
      {
         most_phonbkfi_tclMsgSetContactPhotoError oSetContactPhotoError;
         vHandleMethodError(oSetContactPhotoError, poMessage);
         oSetContactPhotoError.vDestroy();
      }
   }
   else
   {
      ETG_TRACE_ERR((" vHandle_SetContactPhoto_MethodStart: GetMesage Invalid"));
      vHandleGetDataError(poMessage, "vHandle_SetContactPhoto_MethodStart");
   }
   // Always destroy the(message related)FI data object (before leaving
   // its creation scope)
   oSetContactPhoto_MS.vDestroy();
}
//End of fix FIX CMG3G-8372


//FIX CMG3G-8570 IS2502_Cancel PB Download@FC_Phonebook
//The user shall have possibility to cancel the ongoing phone book download not the call history download.
/*******************************************************************************
 ** FUNCTION:   vHandle_CancelPbDownload_MethodStart(..)
 *******************************************************************************/
/* DESCRIPTION:
 *  This is a handler function for CancelPbDownload method
 *
 *  PARAMETERS:
 *  IN:poMessage:This is the received message from a client
 *  RETURNVALUE:None
 *******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_CancelPbDownload_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("Function vHandle_CancelPbDownload_MethodStart entered"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);
   most_phonbkfi_tclMsgCancelPbDownloadMethodStart oCancelPbDownload_MS;

   if (oVisitorMsg.s32GetData(oCancelPbDownload_MS, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION)
         != OSAL_ERROR)
   {
      ETG_TRACE_USR2(("Device Handle: %d", oCancelPbDownload_MS.u8DeviceHandle));

      tBool bError = false;
      tU8 u8DeviceHandle = oCancelPbDownload_MS.u8DeviceHandle;

      if (((u8DeviceHandle > FC_PB_VEHICLE_PBOOK) && (u8DeviceHandle
            <= FC_PB_DEVICE_HANDLE11)))
      {
         FC_Device_Details* pDeviceObject =
               FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);
         if (pDeviceObject)
         {
            if (FALSE == pDeviceObject->bIsCancelPBRequested())
            {
               pDeviceObject->setCancelPBDownload(TRUE);

               if (m_poDownloadState)
               {
                  if (most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_INFO
                        == m_poDownloadState->u8GetPBDownloadState()
                        || most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_CONTACT_PHOTO
                              == m_poDownloadState->u8GetPBDownloadState())
                  {
                     ETG_TRACE_USR4((" Calling Pbdl stop of phonebook contacts on cancel download "));
                     if (m_pDBus_CCA_If_pb)
                     {
                        m_pDBus_CCA_If_pb->vPhoneBook_DnlStop(pDeviceObject->getConnectionID());
                     }
                  }
               }

               //Print array before deletion
               m_oPbk_DnlManager.PhoneBook_PrintAllDownloadTypes(pDeviceObject);

               for (tU8 u8Itr = VALUE_ZERO; u8Itr < MAX_PBDL_TYPE_SUPPORTED; u8Itr++)
               {
                  if ((pDeviceObject->u8DownloadTypeList[u8Itr]
                        == DownLoadPBSim)
                        || (pDeviceObject->u8DownloadTypeList[u8Itr]
                              == DownLoadPBLocal)
                        || (pDeviceObject->u8DownloadTypeList[u8Itr]
                              == DownloadPBSpeedDial)
                        || (pDeviceObject->u8DownloadTypeList[u8Itr]
                              == DownloadPBFavorites))
                  {
                     ETG_TRACE_USR4(("Phonebook contacts download pending on cancel download - resetting..."));
                     pDeviceObject->u8DownloadTypeList[u8Itr]
                           = PHONEBOOK_DOWNLOAD_ARRAY_RESET_VALUE;
                  }
               }

               //Print array after deletion
               m_oPbk_DnlManager.PhoneBook_PrintAllDownloadTypes(pDeviceObject);
            }
            else
            {
               ETG_TRACE_USR4((" Cancel PB request already in progress..."));
            }

            most_phonbkfi_tclMsgCancelPbDownloadMethodResult
                  oCancelPbDownload_MR;
            oCancelPbDownload_MR.u8DeviceHandle
                  = oCancelPbDownload_MS.u8DeviceHandle;
            fi_tclVisitorMessage
                  oMRMsg(oCancelPbDownload_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);

            vInitServiceData(oMRMsg, poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(), // RegisterID
            poMessage->u16GetCmdCounter(), // CmdCounter
            poMessage->u16GetFunctionID(), // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
            );

            if (oMRMsg.bIsValid())
            {
               ail_tenCommunicationError enResult =
                     _poMainAppl->enPostMessage(&oMRMsg, TRUE);
               if (enResult != AIL_EN_N_NO_ERROR)
               {
                  bError = true;
                  ETG_TRACE_ERR(("vHandle_CancelPbDownload_MethodStart: 'Result', enPostMessage() failed, 'ail_tenCommunicationError' = %d", (tU32) enResult));
               }
            }
            else
            {
               ETG_TRACE_ERR(("Function vHandle_CancelPbDownload_MethodStart: oMRMsg is invalid"));
               bError = true;
            }
            oCancelPbDownload_MR.vDestroy();
         }
         else
         {
            bError = true;
         }
      }
      else
      {
         ETG_TRACE_USR2(("DeviceHandle is not valid"));
         bError = true;
      }

      if (true == bError)
      {
         most_phonbkfi_tclMsgCancelPbDownloadError oCancelPbDownloadError;
         vHandleMethodError(oCancelPbDownloadError, poMessage);
         oCancelPbDownloadError.vDestroy();
      }
   }
   else
   {
      ETG_TRACE_ERR(("vHandle_CancelPbDownload_MethodStart: GetMessage Invalid"));
      vHandleGetDataError(poMessage, "vHandle_CancelPbDownload_MethodStart");
   }
   // Always destroy the(message related)FI data object (before leaving
   // its creation scope)
   oCancelPbDownload_MS.vDestroy();

   ETG_TRACE_USR4(("Function vHandle_CancelPbDownload_MethodStart exit"));

}
//End of fix FIX CMG3G-857l


//FIX CMG3G-8571 IS2430_SetContactPosition@FC_Phonebook
/*******************************************************************************
 ** FUNCTION:   vHandle_SetContactPosition_MethodStart(..)
 *******************************************************************************/
/* DESCRIPTION:
 *  This is a handler function for SetContactPositon method
 *
 *  PARAMETERS:
 *  IN:poMessage:This is the received message from a client
 *  RETURNVALUE:None
 *******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::\
vHandle_SetContactPosition_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("Function vHandle_SetContactPosition_MethodStart entered"));
   fi_tclVisitorMessage oVisitorMsg(poMessage);
   most_phonbkfi_tclMsgSetContactPositionMethodStart oSetContactPosition_MS;
   tU16 u16ErrorCode = VALUE_ZERO;
   if (oVisitorMsg.s32GetData(oSetContactPosition_MS, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      ETG_TRACE_USR2(("Contact Handle: %u", oSetContactPosition_MS.u32ContactHandle));
      ETG_TRACE_USR2(("Contact Position: %u", oSetContactPosition_MS.u32ContactPosition));
      tBool bSuccess = false, bError = false;
      tU32 u32ContactHandle = oSetContactPosition_MS.u32ContactHandle;
      tU32 u32ContactPosition = oSetContactPosition_MS.u32ContactPosition;
      if ((VALUE_ZERO == u32ContactHandle) || ((u32ContactHandle & FC_PB_CONTACT_HANDLE_DPB_MARKER) == FC_PB_CONTACT_HANDLE_DPB_MARKER) || (VALUE_ZERO ==u32ContactPosition)) //Contact Handle- 0 is invalid
      {
         ETG_TRACE_USR2(("Invalid Contact Handle: %u", oSetContactPosition_MS.u32ContactHandle));
         u16ErrorCode = CCA_C_U16_ERROR_PARAMETER_OUT_OF_RANGE;
         bError = true;
      }
      if (FALSE == bError)
      {
         bSuccess = FC_PhoneBook_SQLite::GetSQLiteInstance()->SetContactPosition(u32ContactHandle,u32ContactPosition, u16ErrorCode);
         if (TRUE == bSuccess)
         {
            most_phonbkfi_tclMsgSetContactPositionMethodResult oSetContactPosition_MR;
            oSetContactPosition_MR.u32ContactHandle = oSetContactPosition_MS.u32ContactHandle;
            fi_tclVisitorMessage oMRMsg(oSetContactPosition_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
            vInitServiceData(oMRMsg, poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(), // RegisterID
            poMessage->u16GetCmdCounter(), // CmdCounter
            poMessage->u16GetFunctionID(), // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
            );
            if (oMRMsg.bIsValid())
            {
               ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);
               if (enResult != AIL_EN_N_NO_ERROR)
               {
                  bError = true;
                  ETG_TRACE_ERR(("vHandle_SetContactPosition_MethodStart: 'Result', enPostMessage() failed, 'ail_tenCommunicationError' = %d", (tU32) enResult));
               }
               else
               {
                  ETG_TRACE_USR4(("List to be updated  with new phone book lists after setting position "));
                  if(m_pPhoneBookDB)
                  {
                     m_pPhoneBookDB->MethodResultPosted(poMessage->u16GetFunctionID(), &oSetContactPosition_MS, &oSetContactPosition_MR);
                  }
               }
            }
            else
            {
               ETG_TRACE_ERR(("Function vHandle_SetContactPosition_MethodStart: oMRMsg is invalid"));
               bError = true;
            }
            oSetContactPosition_MR.vDestroy();
         }

         else
         {
            ETG_TRACE_ERR(("vHandle_SetContactPosition_MethodStart: Updating the database table failed"));
            bError = true;
         }
      }
      else
      {
         ETG_TRACE_ERR(("vHandle_SetContactPosition_MethodStart: Parameters are invalid"));
         bError = true;
      }
      if (TRUE == bError)
      {
         most_phonbkfi_tclMsgSetContactPositionError oSetContactPositionError;
         vHandleMethodError(oSetContactPositionError, poMessage);
         oSetContactPositionError.vDestroy();
      }
   }
   else
   {
      ETG_TRACE_ERR(("vHandle_SetContactPosition_MethodStart: GetMesage Invalid"));
      u16ErrorCode = CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
      vHandleGetDataError(poMessage, "vHandle_SetContactPosition_MethodStart");
   }
   oSetContactPosition_MS.vDestroy();
}
//End of fix FIX CMG3G-8571


//Fix CMG3G-8365 IS2421_Configurable phone number type representation@FC_Phonebook
/*******************************************************************************
** FUNCTION:   vHandle_AddContactExtended_MethodStart
*******************************************************************************/
/*  DESCRIPTION:
*      This is a handler function to add contact to Phone.
*   PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*     IN: Contact Details and DeviceHandle
*   OUT: Contact Handle and Status of Adding Contact Info to Database
*   RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_AddContactExtended_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_AddContactExtended_MethodStart(): entered"));
   bool bValue = true;
   bool ret = false;
   fi_tclVisitorMessage oVisitorMsg(poMessage);
   most_phonbkfi_tclMsgAddContactExtendedMethodStart oAddContactExtended_MS;
   most_phonbkfi_tclMsgAddContactExtendedMethodResult
         oAddContactListExtended_MR;

   if (oVisitorMsg.s32GetData(oAddContactExtended_MS, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION)
         != OSAL_ERROR)
   {
      ETG_TRACE_USR2(("DEVICE HANDLE: %u", oAddContactExtended_MS.u8DeviceHandle));
      FC_Device_Details
            *pDeviceObject =
                  FC_Device_Details::deviceHandleToObjectInstance(oAddContactExtended_MS.u8DeviceHandle);

      if (pDeviceObject) //To check if the device handle is valid
      {
         ETG_TRACE_USR2(("Device Handle is Valid"));

      }
      else
      {
         bValue = false;
         ETG_TRACE_USR2(("DepDeviceObject is NULL"));
      }
      //End fix : CMG3G-12880

      //Fix NCG3D-68310
      //Once HFP is connected, user can be able to add contacts into vehicle phonebook irrespective of device object creation.
      if (bValue == true)
      {
         if ((m_pPhoneBookDB)
               && (FC_PhoneBook_SQLite::GetSQLiteInstance()->GetNoOfContactsInVPB(oAddContactExtended_MS.u8DeviceHandle)
                     < MAX_ENTRIES_VEHICLE_PHONEBOOK) && (VALUE_ZERO
               != oAddContactExtended_MS.u8DeviceHandle))
         {
            ret
                  = m_pPhoneBookDB->AddContactExtended(&oAddContactExtended_MS, &oAddContactListExtended_MR);
         }
      }
      //End fix NCG3D-68310
      if (ret == true)
      {
         fi_tclVisitorMessage
               oMRMsg(oAddContactListExtended_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
         vInitServiceData(oMRMsg, poMessage->u16GetSourceAppID(), // ServiceDataMsg
         poMessage->u16GetRegisterID(), // RegisterID
         poMessage->u16GetCmdCounter(), // CmdCounter
         poMessage->u16GetFunctionID(), // Function ID
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
         );

         if (oMRMsg.bIsValid())
         {
            // Post message
            ail_tenCommunicationError enResult =
                  _poMainAppl->enPostMessage(&oMRMsg, TRUE);
            if (enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_ERR(("vHandle_AddContactExtended_Result: 'Result', enPostMessage() failed, 'ail_tenCommunicationError' = %d", (tU32) enResult));
               // NORMAL_M_ASSERT_ALWAYS();
            }
            else if (most_fi_tcl_e8_PhonBkContactAddStatus::FI_EN_E8CONTACT_ADD_FAILURE
                  != oAddContactListExtended_MR.e8ContactAddStatusExtended.enType) //GMMY16-7695 (Improper updation of Recent CH when AddContact fails) MethodResultPosted is not called when AddContact fails.
            {
               if (m_pPhoneBookDB)
               {
                  m_pPhoneBookDB->MethodResultPosted(poMessage->u16GetFunctionID(), &oAddContactExtended_MS, &oAddContactListExtended_MR);
               }
            }
         }
         else
         {
            most_phonbkfi_tclMsgAddContactExtendedError oAddContactExtended;
            vHandleMethodError(oAddContactExtended, poMessage);
            oAddContactExtended.vDestroy();
         }
      }
      else
      {
         amt_tclServiceDataError
               oErrorMsg(*poMessage, most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC);

         // Post message
         ail_tenCommunicationError enResult =
               _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

         if (enResult != AIL_EN_N_NO_ERROR)
         {

            ETG_TRACE_ERR(("vHandle_AddContactExtended_MethodStart: 'ERROR', enPostMessage() failed"));
         }
         // NORMAL_M_ASSERT_ALWAYS();
      }
      oAddContactListExtended_MR.vDestroy();
   }
   else
   {
      vHandleGetDataError(poMessage, "vHandle_AddContactExtended_MethodStart");
   }
   // Always destroy the(message related)FI data object (before leaving
   // its creation scope)
   oAddContactExtended_MS.vDestroy();
}
/*******************************************************************************
** FUNCTION:   vHandle_EditContactExtended_MethodStart
*******************************************************************************/
/*   DESCRIPTION:
*    This is a handler function to Edit Contact Information in database with reference to Contact Handle,Device Handle
*
*     PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*     IN : ContactHandle, DeviceHandle and Contact Information
*     OUT:  Contact handle and Record Edit Status (Success or Failure) Extended
*     RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_EditContactExtended_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_EditContactExtended_MethodStart(): entered"));

   bool ret = false;
   bool bValue = true;
   fi_tclVisitorMessage oVisitorMsg(poMessage);
   most_phonbkfi_tclMsgEditContactExtendedMethodStart oEditContactExtended_MS;
   most_phonbkfi_tclMsgEditContactExtendedMethodResult oEditContactExtended_MR;

   if(oVisitorMsg.s32GetData(oEditContactExtended_MS,MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {

      ETG_TRACE_USR2(("DEVICE HANDLE: %u",oEditContactExtended_MS.u8DeviceHandle));

      ETG_TRACE_USR2(("CONTACT HANDLE: %d",oEditContactExtended_MS.u32ContactHandle));

      FC_Device_Details *pDeviceObject = FC_Device_Details::deviceHandleToObjectInstance(oEditContactExtended_MS.u8DeviceHandle);
      if(pDeviceObject)  //To check if the device handle is valid
      {

         ETG_TRACE_USR2(("Device Handle is Valid"));
      }
      else
      {
         bValue = false;
         ETG_TRACE_USR2(("DepDeviceObject is NULL"));
      }

      //Fix NCG3D-68310
      //Once HFP is connected, user can be able to edit contacts in vehicle phonebook irrespective of device object creation.
      if(true == bValue)
      {
         if ((oEditContactExtended_MS.u32ContactHandle
               & FC_PB_CONTACT_HANDLE_DPB_MARKER)
               != FC_PB_CONTACT_HANDLE_DPB_MARKER)
         {
            if ((m_pPhoneBookDB) && (VALUE_ZERO
                  != oEditContactExtended_MS.u8DeviceHandle) && (VALUE_ZERO
                  != oEditContactExtended_MS.u32ContactHandle))
            {
               ret
                     = m_pPhoneBookDB->EditContactExtended(&oEditContactExtended_MS, &oEditContactExtended_MR);
            }
         }
      }
      //End fix NCG3D-68310
      //End fix : CMG3G - 12880
      ETG_TRACE_USR1((" Contact Edit Extended Status:: %u",ETG_ENUM(TR_CONTACTEDITSTATUS,oEditContactExtended_MR.e8ContactEditStatusExtended.enType)));   //GMMY16-7695

      if(ret == true)
      {
         fi_tclVisitorMessage oMRMsg(oEditContactExtended_MR,MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
         vInitServiceData(oMRMsg,
            poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(),  // RegisterID
            poMessage->u16GetCmdCounter(),  // CmdCounter
            poMessage->u16GetFunctionID(),  // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT   // Opcode
            );

         if(oMRMsg.bIsValid())
         {
            // Post message
            ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);
            if(enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_ERR(("vHandle_EditContactExtended_MethodStart: 'ERROR', enPostMessage() failed"));
               // NORMAL_M_ASSERT_ALWAYS();
            }
            else if(most_fi_tcl_e8_PhonBkContactEditStatus::FI_EN_E8CONTACT_EDIT_FAILURE != oEditContactExtended_MR.e8ContactEditStatusExtended.enType)   //GMMY16-7695 MethodResultPosted should not be called when EditContact fails.
            {
               if(m_pPhoneBookDB)
               {
               m_pPhoneBookDB->MethodResultPosted(poMessage->u16GetFunctionID(),
                  &oEditContactExtended_MS, &oEditContactExtended_MR);
               }
            }

         }
         else
         {
            most_phonbkfi_tclMsgEditContactExtendedError oEditContactExtended;
            vHandleMethodError(oEditContactExtended,poMessage);
            oEditContactExtended.vDestroy();
         }
      }
      else
      {
         amt_tclServiceDataError oErrorMsg(*poMessage, most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC);

         // Post message
         ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

         if (enResult != AIL_EN_N_NO_ERROR)
         {

            ETG_TRACE_USR1(("vHandle_EditContactExtended_MethodStart: 'ERROR', enPostMessage() failed"));

         }

         // NORMAL_M_ASSERT_ALWAYS();
      }
      oEditContactExtended_MR.vDestroy();
   }
   else
   {
      vHandleGetDataError(poMessage,"vHandle_EditContactExtended_MethodStart");
   }
   // Always destroy the(message related)FI data object (before leaving
   // its creation scope)
   oEditContactExtended_MS.vDestroy();

}


/*******************************************************************************
** FUNCTION:   vHandle_GetContactDetailsExtended_MethodStart
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for To Get contact Details for the Contact Handle
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_GetContactDetailsExtended_MethodStart(amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR4(("vHandle_GetContactDetailsExtended_MethodStart(): entered"));

   bool ret = false;
   fi_tclVisitorMessage oVisitorMsg(poMessage);
   most_phonbkfi_tclMsgGetContactDetailsExtendedMethodStart oGetContactDetailsExtended_MS;
   most_phonbkfi_tclMsgGetContactDetailsExtendedMethodResult oGetContactDetailsExtended_MR;

   if(oVisitorMsg.s32GetData( oGetContactDetailsExtended_MS, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      ETG_TRACE_USR2(("Contact Handle: %d", oGetContactDetailsExtended_MS.u32ContactHandle));

      if((m_pPhoneBookDB) && (VALUE_ZERO != oGetContactDetailsExtended_MS.u32ContactHandle))
      {
         ret = m_pPhoneBookDB->GetContactDetailsExtended(&oGetContactDetailsExtended_MS,&oGetContactDetailsExtended_MR);
      }

      if(ret == TRUE)
      {
         fi_tclVisitorMessage oMRMsg(oGetContactDetailsExtended_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
         //Inserting Success
         vInitServiceData(oMRMsg,
            poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(),  // RegisterID
            poMessage->u16GetCmdCounter(),  // CmdCounter
            poMessage->u16GetFunctionID(),  // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT   // Opcode
            );

         if(oMRMsg.bIsValid())
         {
            // Post message
            ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);
            if(enResult != AIL_EN_N_NO_ERROR)
            {

               ETG_TRACE_ERR(("vHandle_GetContactDetailsExtended_MethodStart: 'Result', enPostMessage() failed, 'ail_tenCommunicationError' = %d",(tU32)enResult));
               // NORMAL_M_ASSERT_ALWAYS();
            }
         }
         else
         {
            most_phonbkfi_tclMsgGetContactDetailsExtendedError getContactDetailsExtendedError;
            vHandleMethodError(getContactDetailsExtendedError,poMessage);
            getContactDetailsExtendedError.vDestroy();
         }
      }
      else
      {
         //Inserting Fails
         amt_tclServiceDataError oErrorMsg(*poMessage, most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC);

         // Post message
         ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

         if (enResult != AIL_EN_N_NO_ERROR)
         {

            ETG_TRACE_ERR(("vHandle_GetContactDetailsExtended_MethodStart: 'ERROR', enPostMessage() failed"));
         }

         // NORMAL_M_ASSERT_ALWAYS();
      }


      oGetContactDetailsExtended_MR.vDestroy();
   }
   else
   {
      vHandleGetDataError(poMessage,"vHandle_GetContactDetailsExtended_MethodStart");
   }
   // Always destroy the(message related)FI data object (before leaving
   // its creation scope)
   oGetContactDetailsExtended_MS.vDestroy();
}
//End of Fix CMG3G-8365 IS2421_Configurable phone number type representation@FC_Phonebook


//Fix CMG3G-9334 Delete all vehicle phonebook entries at once
/*******************************************************************************
** FUNCTION:   vHandle_ClearVehiclePhonebookExtended_MethodStart
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for to clear vehicle phonebook entries.
*   This method clears all vehicle phonebook entries from a specific vehicle phonebook. [VPBx]
*   It can also clear all the entries from all the vehicle phonebooks at once [1-11] (For reset to factory default.)
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_ClearVehiclePhonebookExtended_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4((" vHandle_ClearVehiclePhonebookExtended_MethodStart(): entered"));

   bool bRet = false;
   fi_tclVisitorMessage oVisitorMsg(poMessage);
   most_phonbkfi_tclMsgClearVehiclePhonebookExtendedMethodStart oClearVehiclePhonebookExtended_MS;

   if(oVisitorMsg.s32GetData( oClearVehiclePhonebookExtended_MS, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      ETG_TRACE_USR2((" Device Handle: %d", oClearVehiclePhonebookExtended_MS.u8DeviceHandle));

      if (oClearVehiclePhonebookExtended_MS.u8DeviceHandle
            <= FC_PB_DEVICE_HANDLE11)//ZERO is also acceptable
      {
         ETG_TRACE_USR2((" Device Handle is Valid. Clearing Vehicle Phonebook... "));
         FC_PhoneBook_List* poFC_PhoneBook_List =
               FC_PhoneBook_List::pGetInstance();
         if (poFC_PhoneBook_List != NULL)
         {
            bRet
                  = poFC_PhoneBook_List->bClearVehiclePhonebookExtended(oClearVehiclePhonebookExtended_MS.u8DeviceHandle);
         }
      }
      else
      {
         ETG_TRACE_ERR((" Device Handle out of range!! "));
      }

      if(bRet == TRUE)
      {
         most_phonbkfi_tclMsgClearVehiclePhonebookExtendedMethodResult oClearVehiclePhonebookExtended_MR;

         //Update method result parameters
         oClearVehiclePhonebookExtended_MR.u8DeviceHandle
               = oClearVehiclePhonebookExtended_MS.u8DeviceHandle;

         fi_tclVisitorMessage oMRMsg(oClearVehiclePhonebookExtended_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
         //Inserting Success
         vInitServiceData(oMRMsg,
            poMessage->u16GetSourceAppID(), // ServiceDataMsg
            poMessage->u16GetRegisterID(),  // RegisterID
            poMessage->u16GetCmdCounter(),  // CmdCounter
            poMessage->u16GetFunctionID(),  // Function ID
            AMT_C_U8_CCAMSG_OPCODE_METHODRESULT   // Opcode
            );

         if(oMRMsg.bIsValid())
         {
            // Post message
            ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);
            if(enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_ERR(("vHandle_ClearVehiclePhonebookExtended_MethodStart: 'Result', enPostMessage() failed, 'ail_tenCommunicationError' = %d",(tU32)enResult));
            }
         }
         else
         {
            most_phonbkfi_tclMsgClearVehiclePhonebookExtendedError oClearVehiclePhonebookExtendedError;
            vHandleMethodError(oClearVehiclePhonebookExtendedError,poMessage);
            oClearVehiclePhonebookExtendedError.vDestroy();
         }

         oClearVehiclePhonebookExtended_MR.vDestroy();
      }
      else
      {
         amt_tclServiceDataError oErrorMsg(*poMessage, most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC);

         // Post message
         ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oErrorMsg, TRUE);
         if (enResult != AIL_EN_N_NO_ERROR)
         {
            ETG_TRACE_ERR(("vHandle_ClearVehiclePhonebookExtended_MethodStart: 'ERROR', enPostMessage() failed"));
         }
      }
   }
   else
   {
      vHandleGetDataError(poMessage,"vHandle_ClearVehiclePhonebookExtended_MethodStart");
   }
   // Always destroy the(message related)FI data object (before leaving
   // its creation scope)
   oClearVehiclePhonebookExtended_MS.vDestroy();
}
//End of fix CMG3G-9334

//Fix CMG3G-9889 Setting Automatic phonebook download on/off needs to be device specific
/*******************************************************************************
** FUNCTION:   vHandle_SetAutomaticContactDownload_MethodStart
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for Setting the automaticcontactdownload from the HMI.
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_SetAutomaticContactDownload_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4((" vHandle_SetAutomaticContactDownload_MethodStart(): entered"));
   bool bRet = false;
   fi_tclVisitorMessage oVisitorMsg(poMessage);
   most_phonbkfi_tclMsgSetAutomaticContactDownloadMethodStart oSetAutomaticContactDownload_MS;
   if(oVisitorMsg.s32GetData( oSetAutomaticContactDownload_MS, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION) != OSAL_ERROR)
   {
      ETG_TRACE_USR2((" Device Handle: %d", oSetAutomaticContactDownload_MS.u8DeviceHandle));
      ETG_TRACE_USR2((" bAutomaticDL : %d", oSetAutomaticContactDownload_MS.bAutomaticDL ));
      if (oSetAutomaticContactDownload_MS.u8DeviceHandle
            <= FC_PB_DEVICE_HANDLE11)//ZERO is also acceptable
      {
         ETG_TRACE_USR2((" Device Handle is Valid "));



         if (oSetAutomaticContactDownload_MS.u8DeviceHandle
               == FC_PB_VEHICLE_PBOOK)
         {
            bRet = true;
         }
         else
         {
            FC_Device_Details *pDeviceObject;
            pDeviceObject =  FC_Device_Details::deviceHandleToObjectInstance(oSetAutomaticContactDownload_MS.u8DeviceHandle);
            //Fix for NCG3D-20843
            //If device handle is not of vehicle phonebook type
            if (pDeviceObject != NULL || oSetAutomaticContactDownload_MS.u8DeviceHandle == 0)
            {
               ETG_TRACE_USR2(("Device object exists for device handle = %u ", oSetAutomaticContactDownload_MS.u8DeviceHandle));
               bRet = FC_PhoneBook_SQLite::GetSQLiteInstance()->bUpdateAutomaticContactDownload(oSetAutomaticContactDownload_MS.u8DeviceHandle, oSetAutomaticContactDownload_MS.bAutomaticDL);
               vPhoneBook_UpdateClientsAutomaticContactDownloadList();
            }
            else
            {
               ETG_TRACE_ERR((" Contact List : Device object does not exists for device handle = %u ", oSetAutomaticContactDownload_MS.u8DeviceHandle));
            }
            //End of fix NCG3D-20843
         }
      }
      else
      {
         ETG_TRACE_ERR((" Device Handle out of range!! "));
      }

      if(bRet == TRUE)
      {
         most_phonbkfi_tclMsgSetAutomaticContactDownloadMethodResult oSetAutomaticContactDownload_MR;

         oSetAutomaticContactDownload_MR.u8DeviceHandle
         = oSetAutomaticContactDownload_MS.u8DeviceHandle;
         oSetAutomaticContactDownload_MR.bAutomaticDL = oSetAutomaticContactDownload_MS.bAutomaticDL;

         fi_tclVisitorMessage oMRMsg(oSetAutomaticContactDownload_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
         vInitServiceData(oMRMsg,
               poMessage->u16GetSourceAppID(), // ServiceDataMsg
               poMessage->u16GetRegisterID(),  // RegisterID
               poMessage->u16GetCmdCounter(),  // CmdCounter
               poMessage->u16GetFunctionID(),  // Function ID
               AMT_C_U8_CCAMSG_OPCODE_METHODRESULT   // Opcode
         );
         if(oMRMsg.bIsValid())
         {
            ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMRMsg, TRUE);
            if(enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_ERR(("vHandle_SetAutomaticContactDownload_MethodStart: 'Result', enPostMessage() failed, 'ail_tenCommunicationError' = %d",(tU32)enResult));
            }
         }
         else
         {
            most_phonbkfi_tclMsgSetAutomaticContactDownloadError oSetAutomaticContactDownloadError;
            vHandleMethodError(oSetAutomaticContactDownloadError,poMessage);
            oSetAutomaticContactDownloadError.vDestroy();
         }
         oSetAutomaticContactDownload_MR.vDestroy();
      }
      else
      {
         amt_tclServiceDataError oErrorMsg(*poMessage, most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC);
         ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oErrorMsg, TRUE);
         if (enResult != AIL_EN_N_NO_ERROR)
         {
            ETG_TRACE_ERR(("vHandle_SetAutomaticContactDownload_MethodStart: 'ERROR', enPostMessage() failed"));
         }
      }
   }
   else
   {
      vHandleGetDataError(poMessage,"vHandle_SetAutomaticContactDownload_MethodStart");
   }
   oSetAutomaticContactDownload_MS.vDestroy();
}


/*******************************************************************************
** FUNCTION:   vHandle_GetPositionForContactHandle_MethodStart
*******************************************************************************/
/* DESCRIPTION:
*   This method can be called for a dedicated contact list (identified by the u16ListHandle)
*   to find out at which position in this list a contact (identified via its u32ContactHandle) is located.
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
*******************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vHandle_GetPositionForContactHandle_MethodStart(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4((" vHandle_GetPositionForContactHandle_MethodStart(): entered"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);
   most_phonbkfi_tclMsgGetPositionForContactHandleMethodStart
         oGetPositionForContactHandle_MS;

   if (oVisitorMsg.s32GetData(oGetPositionForContactHandle_MS, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION)
         != OSAL_ERROR)
   {
      ETG_TRACE_USR2((" List Handle: 0x%x", oGetPositionForContactHandle_MS.u16ListHandle));
      ETG_TRACE_USR2((" Contact Handle: %u", oGetPositionForContactHandle_MS.u32ContactHandle));

      tBool bResult = FALSE;
      most_phonbkfi_tclMsgGetPositionForContactHandleMethodResult
            oGetPositionForContactHandle_MR;

      if (m_pPhoneBookDB)
      {
         bResult
               = m_pPhoneBookDB->GetPositionForContactHandle(&oGetPositionForContactHandle_MS, &oGetPositionForContactHandle_MR);
      }
      else
      {
         ETG_TRACE_ERR((" m_pPhoneBookDB is NULL!!!"));
      }

      if (TRUE == bResult)
      {
         oGetPositionForContactHandle_MR.u16ListHandle
               = oGetPositionForContactHandle_MS.u16ListHandle;
         oGetPositionForContactHandle_MR.u32ContactHandle
               = oGetPositionForContactHandle_MS.u32ContactHandle;

         fi_tclVisitorMessage
               oMRMsg(oGetPositionForContactHandle_MR, MOST_PHONBKFI_C_U16_SERVICE_MAJORVERSION);
         vInitServiceData(oMRMsg, poMessage->u16GetSourceAppID(), // ServiceDataMsg
         poMessage->u16GetRegisterID(), // RegisterID
         poMessage->u16GetCmdCounter(), // CmdCounter
         poMessage->u16GetFunctionID(), // Function ID
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT // Opcode
         );

         if (oMRMsg.bIsValid())
         {
            ail_tenCommunicationError enResult =
                  _poMainAppl->enPostMessage(&oMRMsg, TRUE);
            if (enResult != AIL_EN_N_NO_ERROR)
            {
               ETG_TRACE_ERR((" vHandle_GetPositionForContactHandle_MethodStart: 'Result', enPostMessage() failed, 'ail_tenCommunicationError' = %d", (tU32) enResult));
            }
         }
         else
         {
            most_phonbkfi_tclMsgGetPositionForContactHandleError
                  oGetPositionForContactHandleError;
            vHandleMethodError(oGetPositionForContactHandleError, poMessage);
            oGetPositionForContactHandleError.vDestroy();
         }
         oGetPositionForContactHandle_MR.vDestroy();
      }
      else
      {
         amt_tclServiceDataError
               oErrorMsg(*poMessage, most_fi_tcl_e8_ErrorCode::FI_EN_FUNCTIONSPECIFIC);
         ail_tenCommunicationError enResult =
               _poMainAppl->enPostMessage(&oErrorMsg, TRUE);
         if (enResult != AIL_EN_N_NO_ERROR)
         {
            ETG_TRACE_ERR((" vHandle_GetPositionForContactHandle_MethodStart: 'ERROR', enPostMessage() failed"));
         }
      }
   }
   else
   {
      vHandleGetDataError(poMessage, "vHandle_GetPositionForContactHandle_MethodStart");
   }
   oGetPositionForContactHandle_MS.vDestroy();
}

//Fix : CMG3G-12880
/******************************************************************************
 ** FUNCTION:   vCheckDeviceInLockedState(..)
 ******************************************************************************
 * DESCRIPTION: Verify, whether given devicehandle is in device locked list
 * PARAMETERS:
 *     IN :  u8DeviceHandle - handle of the device to be checked whether locked
 *
 *     OUT:  u8Position - returns the position of device handle in the locked
 *           list.
 *           When the device is not found in the list, INVALID_VALUE is retuned.
 *
 * RETURNVALUE: None
 *****************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vCheckDeviceInLockedState(tU8 u8DeviceHandle, tU8 &u8Position)
{
   u8Position = INVALID_VALUE;

   for (tU8 u8Index = 0; u8Index
         < m_oProbablyLockedDevices.oProbablyLockedDevices.oDeviceHandles.size(); u8Index++)
   {
      if (u8DeviceHandle
            == m_oProbablyLockedDevices.oProbablyLockedDevices.oDeviceHandles[u8Index])
      {
         u8Position = u8Index;
         break;
      }
   }
}

/******************************************************************************
 ** FUNCTION:   vAddToDeviceLockedList(..)
 ******************************************************************************
 * DESCRIPTION: Verify, whether given devicehandle is in device locked list
 *              If already available in the list do nothing, else add to list
 * PARAMETERS:
 *     IN :  u8DeviceHandle - handle of the device to be added to locked device
 *           list
 *
 *     OUT:  NONE
 *
 * RETURNVALUE: None
 *****************************************************************************/

tVoid fc_phonebook_tclService_PhoneBook::vAddToDeviceLockedList(tU8 u8DeviceHandle)
{
   tU8 u8Position = INVALID_VALUE;

   vCheckDeviceInLockedState(u8DeviceHandle, u8Position);

   if (INVALID_VALUE == u8Position)
   {
      ETG_TRACE_USR4((" Device handle is not in probably device locked list, hence adding it "));

      m_oProbablyLockedDevices.oProbablyLockedDevices.oDeviceHandles.push_back(u8DeviceHandle);

      FC_Device_Details* pDeviceObject =
            FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);

      if (pDeviceObject)
      {
         pDeviceObject->vSetDeviceProbablyLocked(TRUE);
      }

      vUpdateProbablyLockedDeviceList();
   }
}

/******************************************************************************
 ** FUNCTION:   vRemoveFromDeviceLockedList(..)
 ******************************************************************************
 * DESCRIPTION: Verify, whether given devicehandle is in device locked list
 *              If not available in list do nothing, else remove from list
 * PARAMETERS:
 *     IN :  u8DeviceHandle - handle of the device to be removed from
 *           locked device list
 *
 *     OUT:  NONE
 *
 * RETURNVALUE: None
 *****************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vRemoveFromDeviceLockedList(tU8 u8DeviceHandle)
{
   tU8 u8Position = INVALID_VALUE;

   vCheckDeviceInLockedState(u8DeviceHandle, u8Position);

   if (INVALID_VALUE != u8Position)
   {
      m_oProbablyLockedDevices.oProbablyLockedDevices.oDeviceHandles.erase(m_oProbablyLockedDevices.oProbablyLockedDevices.oDeviceHandles.begin()
            + u8Position);

      FC_Device_Details* pDeviceObject =
            FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);

      if (pDeviceObject)
      {
         pDeviceObject->vSetDeviceProbablyLocked(FALSE);
      }

      vUpdateProbablyLockedDeviceList();
   }
   else
      ETG_TRACE_USR4((" Device handle is not in probably device locked list, do nothing"));

}

/******************************************************************************
 ** FUNCTION:   vUpdateProbablyLockedDeviceList(..)
 ******************************************************************************
 * DESCRIPTION: Update Clients on probably locked device list
 *  PARAMETERS:
 *     IN :  NONE
 *     OUT:  NONE
 *
 * RETURNVALUE: None
 *****************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vUpdateProbablyLockedDeviceList()
{
   ail_tenCommunicationError enResult =
         eUpdateClients(MOST_PHONBKFI_C_U16_PROBABLYLOCKEDDEVICES);

   if (AIL_EN_N_NO_ERROR != enResult)
   {
      ETG_TRACE_ERR(("ProbablyLockedDevices Status: 'ERROR', UpdateClients() failed"));
   }
   else
   {
      ETG_TRACE_USR4(("ProbablyLockedDevices Status: Updated Clients()"));
   }

}
//End fix : CMG3G-12800


//Fix : NCG3D-65982
/******************************************************************************
 ** FUNCTION:   vCheckDeviceInDownlodSuspendedState(..)
 ******************************************************************************
 * DESCRIPTION: Verify, whether given devicehandle is in suspended download list
 * PARAMETERS:
 *     IN :  u8DeviceHandle - handle of the device to be checked whether download suspended
 *
 *     OUT:  u8Position - returns the position of device handle in the download suspended list
 *           When the device is not found in the list, INVALID_VALUE is retuned.
 *
 * RETURNVALUE: None
 *****************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vCheckDeviceInDownlodSuspendedState(tU8 u8DeviceHandle, tU8 &u8Position)
{
   u8Position = INVALID_VALUE;

   for (tU8 u8Index = 0; u8Index
         < m_poBTSettings->m_u8SuspendedDeviceDownloadList.size(); u8Index++)
   {
      if (u8DeviceHandle
            == m_poBTSettings->m_u8SuspendedDeviceDownloadList[u8Index])
      {
         u8Position = u8Index;
         break;
      }
   }
}

/******************************************************************************
 ** FUNCTION:   vAddToSuspendedDownloadList(..)
 ******************************************************************************
 * DESCRIPTION: Verify, whether given devicehandle is in suspended download list
 *              If already available in the list do nothing, else add to list
 * PARAMETERS:
 *     IN :  u8DeviceHandle - handle of the device to be added to suspended device list
 *
 *     OUT:  NONE
 *
 * RETURNVALUE: None
 *****************************************************************************/

tVoid fc_phonebook_tclService_PhoneBook::vAddToSuspendedDownloadList(tU8 u8DeviceHandle)
{
   tU8 u8Position = INVALID_VALUE;

   vCheckDeviceInDownlodSuspendedState(u8DeviceHandle, u8Position);

   if (INVALID_VALUE == u8Position)
   {
      ETG_TRACE_USR4((" Device handle is not in suspended downloaded list, hence adding it "));

      m_poBTSettings->m_u8SuspendedDeviceDownloadList.push_back(u8DeviceHandle);

      FC_Device_Details* pDeviceObject =
            FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);

      if (pDeviceObject)
      {
         pDeviceObject->vSetDeviceDownloadSuspended(TRUE);
      }
   }
}

/******************************************************************************
 ** FUNCTION:   vRemoveFromSuspendedDownloadList(..)
 ******************************************************************************
 * DESCRIPTION: Verify, whether given devicehandle is in suspended download list
 *              If not available in list do nothing, else remove from list
 * PARAMETERS:
 *     IN :  u8DeviceHandle - handle of the device to be removed from suspended download list
 *
 *     OUT:  NONE
 *
 * RETURNVALUE: None
 *****************************************************************************/
tVoid fc_phonebook_tclService_PhoneBook::vRemoveFromSuspendedDownloadList(tU8 u8DeviceHandle)
{
   tU8 u8Position = INVALID_VALUE;

   vCheckDeviceInDownlodSuspendedState(u8DeviceHandle, u8Position);

   if (INVALID_VALUE != u8Position)
   {
      m_poBTSettings->m_u8SuspendedDeviceDownloadList.erase(m_poBTSettings->m_u8SuspendedDeviceDownloadList.begin() + u8Position);

      FC_Device_Details* pDeviceObject =
            FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);

      if (pDeviceObject)
      {
         pDeviceObject->vSetDeviceDownloadSuspended(FALSE);
      }
   }
   else
      ETG_TRACE_USR4((" Device handle is not in download suspended list, do nothing"));
}
//End fix NCG3D-65982
