/*******************************************************************************
*
* FILE:          FC_PhoneBook_clienthandler_Phone.h
*
* SW-COMPONENT:  FC_PhoneBook application
*
* PROJECT:
*
* DESCRIPTION:   CCA client-handler for the service Telephone
*
* AUTHOR:        Balasubramani
*
* COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
*
*******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/
// Include message framework interface (AMT, msgfw)
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE

//Include FI interface of used service
#define MOST_FI_S_IMPORT_INTERFACE_MOST_TELFI_TYPES
#define MOST_FI_S_IMPORT_INTERFACE_MOST_TELFI_FUNCTIONIDS
#define MOST_FI_S_IMPORT_INTERFACE_MOST_TELFI_ERRORCODES
#define MOST_FI_S_IMPORT_INTERFACE_MOST_TELFI_SERVICEINFO

#include "conn_most_fi_if.h"

#include "FC_PhoneBook_Debug.h"
#include "FC_PhoneBook_main.h"
#include "FC_PhoneBook_clienthandler_Phone.h"
#include "FC_PhoneBook_FavoritesProvider.h"
#include "FC_PhoneBook_Device_Details.h"
#include "FC_PhoneBook_service_PhoneBook.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONEBOOK_APPLICATION

#include "trcGenProj/Header/FC_PhoneBook_clienthandler_Phone.cpp.trc.h"
#endif

/******************************************************************************/
/*                                                                            */
/* DEFINES                                                                    */
/*                                                                            */
/******************************************************************************/


/******************************************************************************/
/*                                                                            */
/* CCA MESSAGE MAP                                                            */
/*                                                                            */
/******************************************************************************/

BEGIN_MSG_MAP(fc_phonebook_tclClientHandler_Phone, ahl_tclBaseWork)

   // Add your ON_MESSAGE_SVCDATA() macros here to define which corresponding
   // method should be called on receiving a specific message.
   ON_MESSAGE_SVCDATA( MOST_TELFI_C_U16_CALLSTATUSNOTICEEXTENDED, \
   AMT_C_U8_CCAMSG_OPCODE_STATUS , \
   vHandle_CallStatusNoticeExtended_Status )

   ON_MESSAGE_SVCDATA( MOST_TELFI_C_U16_DIAL, \
   AMT_C_U8_CCAMSG_OPCODE_METHODRESULT , \
   vHandle_Dial_MethodResult )
   ON_MESSAGE_SVCDATA(MOST_TELFI_C_U16_DIAL,
   AMT_C_U8_CCAMSG_OPCODE_ERROR,
   vHandle_Dial_Error)
   ON_MESSAGE_SVCDATA( MOST_TELFI_C_U16_SENDDTMF, \
   AMT_C_U8_CCAMSG_OPCODE_METHODRESULT , \
   vHandle_SendDTMF_MethodResult )
   ON_MESSAGE_SVCDATA(MOST_TELFI_C_U16_SENDDTMF,
   AMT_C_U8_CCAMSG_OPCODE_ERROR,
   vHandle_SendDTMF_Error)
   ON_MESSAGE_SVCDATA( MOST_TELFI_C_U16_ACTIVEPHONEDEVICE, \
   AMT_C_U8_CCAMSG_OPCODE_STATUS , \
   vHandle_ActivePhoneDevice_Status )

   //FIX GMNGA-56482 Favorite not highlighted when Call active
   //We need to identify the phone number corresponding to the call instance received in call status.
   //On identification, if the number is stored as favourite, it should be highlighted.
   ON_MESSAGE_SVCDATA( MOST_TELFI_C_U16_QUERYTELEPHONENUMBER, \
   AMT_C_U8_CCAMSG_OPCODE_METHODRESULT , \
   vHandle_QueryTelephoneNumber_MethodResult )

   ON_MESSAGE_SVCDATA(MOST_TELFI_C_U16_QUERYTELEPHONENUMBER,
   AMT_C_U8_CCAMSG_OPCODE_ERROR,
   vHandle_QueryTelephoneNumber_Error)

   //Fix for GMMY16-9895 Pressing the favorite button that was used to place an outgoing call does not end the call
   ON_MESSAGE_SVCDATA( MOST_TELFI_C_U16_HANGUPCALL, \
   AMT_C_U8_CCAMSG_OPCODE_METHODRESULT , \
   vHandle_HangUpCall_MethodResult )
   ON_MESSAGE_SVCDATA(MOST_TELFI_C_U16_HANGUPCALL,
   AMT_C_U8_CCAMSG_OPCODE_ERROR,
   vHandle_HangUpCall_Error)
   ON_MESSAGE_SVCDATA( MOST_TELFI_C_U16_CANCELOUTGOINGCALL, \
   AMT_C_U8_CCAMSG_OPCODE_METHODRESULT , \
   vHandle_CancelOutgoingCall_MethodResult )
   ON_MESSAGE_SVCDATA(MOST_TELFI_C_U16_CANCELOUTGOINGCALL,
   AMT_C_U8_CCAMSG_OPCODE_ERROR,
   vHandle_CancelOutgoingCall_Error)
   ON_MESSAGE_SVCDATA( MOST_TELFI_C_U16_IGNORECALL, \
   AMT_C_U8_CCAMSG_OPCODE_METHODRESULT , \
   vHandle_IgnoreCall_MethodResult )
   ON_MESSAGE_SVCDATA(MOST_TELFI_C_U16_IGNORECALL,
   AMT_C_U8_CCAMSG_OPCODE_ERROR,
   vHandle_IgnoreCall_Error)
   // End of fix for GMMY16-9895

END_MSG_MAP()

/******************************************************************************/
/*                                                                            */
/* METHODS                                                                    */
/*                                                                            */
/******************************************************************************/
/*******************************************************************************
*
* FUNCTION: fc_phonebook_tclClientHandler_Phone::
*     fc_phonebook_tclClientHandler_Phone(fc_phonebook_tclApp* poMainAppl)
*
* DESCRIPTION: Constructor.
*
*              Create an object of the base class
*              ahl_tclBaseOneThreadClientHandler with a pointer to this
*              application, the to be used service identifier and the service
*              version as parameters.
*
* PARAMETER: [IN] poMainAppl = Pointer to the object of this application.
*
* RETURNVALUE: None.
*
*******************************************************************************/
fc_phonebook_tclClientHandler_Phone::
fc_phonebook_tclClientHandler_Phone(fc_phonebook_tclApp* poMainAppl)
: ahl_tclBaseOneThreadClientHandler(
                                    // Application Pointer
                                    poMainAppl,
                                    // ID of used Service
                                    CCA_C_U16_SRV_FB_TELEPHONE,
                                    // MajorVersion of used Service
                                    MOST_TELFI_C_U16_SERVICE_MAJORVERSION,
                                    // MinorVersion of used Service
                                    MOST_TELFI_C_U16_SERVICE_MINORVERSION )
{

   ETG_TRACE_USR4(("fc_phonebook_tclClientHandler_Phone() entered."));

   m_poFavoritesProvider = NULLPTR;
   m_CurrentCallInstance = 0xff;
   m_u8ActivecallCount = 0;

   m_au8CallDirection[0] = CALLDIRECTION_UNKNOWN;
   m_au8CallDirection[1] = CALLDIRECTION_UNKNOWN;
   m_au8CallDirection[2] = CALLDIRECTION_UNKNOWN;
   m_au8CallDirection[3] = CALLDIRECTION_UNKNOWN;

   vAddAutoRegisterForProperty(MOST_TELFI_C_U16_CALLSTATUSNOTICEEXTENDED);
   vAddAutoRegisterForProperty(MOST_TELFI_C_U16_ACTIVEPHONEDEVICE);
}

/*******************************************************************************
*
* FUNCTION: fc_phonebook_tclClientHandler_Phone::
*             ~fc_phonebook_tclClientHandler_Phone(tVoid)
*
* DESCRIPTION: Destructor.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
fc_phonebook_tclClientHandler_Phone::
~fc_phonebook_tclClientHandler_Phone(tVoid)
{
   ETG_TRACE_USR4(("~fc_phonebook_tclClientHandler_Phone() entered."));

   m_oCallStatusNoticeExtendedStatus.vDestroy();
   m_poFavoritesProvider = NULLPTR;
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_phonebook_tclClientHandler_Phone::
vOnServiceAvailable()
*
* DESCRIPTION: This function is called by the CCA framework when the service
*              this client-handler has registered for has become available.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadClientHandler::vOnServiceAvailable().
*******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::vOnServiceAvailable()
{
   ETG_TRACE_USR4(("vOnServiceAvailable() entered. AppID = %u.",
      u16GetServerAppID()));
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_phonebook_tclClientHandler_Phone::
vOnServiceUnavailable()
*
* DESCRIPTION: This function is called by the CCA framework when the service
*              this client-handler has registered for has become unavailable.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadClientHandler::vOnServiceUnavailable().
*******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::vOnServiceUnavailable()
{
   ETG_TRACE_USR4(("vOnServiceUnavailable() entered. AppID = %u.",
      u16GetServerAppID()));
}


/*******************************************************************************
** FUNCTION:   vHandle_CallStatusNoticeExtended_Status(..)
*******************************************************************************/
/* DESCRIPTION:
*   This is a handler function for Call Status extended property status
*
*  PARAMETERS:
*     IN:   poMessage        : This is the received message from a client
*  RETURNVALUE: None
******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::vHandle_CallStatusNoticeExtended_Status(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("vHandle_CallStatusNoticeExtended_Status entered"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);
   
   tU8 u8ActivecallCount = 0;
   most_telfi_tclMsgCallStatusNoticeExtendedStatus oCallStatusNoticeExtendedStatus;
   tU8 u8Index = 0;

   if (oVisitorMsg.s32GetData(oCallStatusNoticeExtendedStatus, MOST_TELFI_C_U16_SERVICE_MAJORVERSION)
         != OSAL_ERROR)
   {
      //Iterator to received call status 
	  // 241829- Replace bpstl:: by std::
      std::vector<most_fi_tcl_TelCallStatusNoticeExtendedStreamItem,
            std::allocator<most_fi_tcl_TelCallStatusNoticeExtendedStreamItem> >::iterator
            ItrNew =
                  oCallStatusNoticeExtendedStatus.oCallStatusNoticeExtendedStream.oItems.begin();

      //If EVO stack is present, call history update is to be triggered once any call has ended.
      if (ALPS_EVOLUTION_BT_STACK == fc_phonebook_tclService_PhoneBook::pGetInstance()->getBtStack())
      {
         //Iterator to old call status
		 // 241829- Replace bpstl:: by std::
         std::vector<most_fi_tcl_TelCallStatusNoticeExtendedStreamItem,
               std::allocator<most_fi_tcl_TelCallStatusNoticeExtendedStreamItem> >::iterator
               ItrOld =
                     m_oCallStatusNoticeExtendedStatus.oCallStatusNoticeExtendedStream.oItems.begin();

        while (ItrOld
              != m_oCallStatusNoticeExtendedStatus.oCallStatusNoticeExtendedStream.oItems.end() && ItrNew
              != oCallStatusNoticeExtendedStatus.oCallStatusNoticeExtendedStream.oItems.end())
        {
           ETG_TRACE_USR4((" New call status details..."));
           ETG_TRACE_USR4((" ItrNew->u16CallInstance:: %d", ItrNew->u16CallInstance));
           ETG_TRACE_USR4((" ItrNew->e8CallStatus.enType:: %d", ETG_ENUM(TR_CALLSTATUS, ItrNew->e8CallStatus.enType)));

           ETG_TRACE_USR4((" Old call status details..."));
           ETG_TRACE_USR4((" ItrOld->u16CallInstance:: %d", ItrOld->u16CallInstance));
           ETG_TRACE_USR4((" ItrOld->e8CallStatus.enType:: %d", ETG_ENUM(TR_CALLSTATUS, ItrOld->e8CallStatus.enType)));

           if (ItrNew->e8CallStatus.enType
                 == most_fi_tcl_e8_TelCallStatus::FI_EN_E8IDLE
                 && ItrOld->e8CallStatus.enType
                       != most_fi_tcl_e8_TelCallStatus::FI_EN_E8IDLE)
           {
              //Update call history
              ETG_TRACE_USR4((" Any call has ended in device... "));
              ETG_TRACE_USR4((" Device handle: %d", ItrNew->u8DeviceHandle));

              tBool bDownloadMCH = FALSE;
              if (ItrOld->e8CallStatus.enType
                    == most_fi_tcl_e8_TelCallStatus::FI_EN_E8RINGTONE)
              {
                 bDownloadMCH = TRUE;
              }
              FC_Device_Details
                    *deviceObject =
                          FC_Device_Details::deviceHandleToObjectInstance(ItrNew->u8DeviceHandle);
              if (deviceObject)
              {
                 fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.vDownloadCallHistory(deviceObject, m_au8CallDirection[u8Index], bDownloadMCH);
              }
              m_au8CallDirection[u8Index] = CALLDIRECTION_UNKNOWN;
           }
           ItrNew++;
           ItrOld++;
           u8Index++;
        }

        //Store the received value in member variable
        ETG_TRACE_USR4((" Storing call status details..."));
        m_oCallStatusNoticeExtendedStatus.oCallStatusNoticeExtendedStream.oItems.clear();
        for (tU8 i = 0; i
              < oCallStatusNoticeExtendedStatus.oCallStatusNoticeExtendedStream.oItems.size(); i++)
        {
           m_oCallStatusNoticeExtendedStatus.oCallStatusNoticeExtendedStream.oItems.push_back(oCallStatusNoticeExtendedStatus.oCallStatusNoticeExtendedStream.oItems[i]);
   
           if (oCallStatusNoticeExtendedStatus.oCallStatusNoticeExtendedStream.oItems[i].e8CallStatus.enType
                 == most_fi_tcl_e8_TelCallStatus::FI_EN_E8DIALING)
           {
              ETG_TRACE_USR4((" Setting direction as CALLDIRECTION_OUTGOING..."));
              //Fix RTC 169773
              m_au8CallDirection[i] = CALLDIRECTION_OUTGOING;
              //End Fix 169773
           }
           else if (oCallStatusNoticeExtendedStatus.oCallStatusNoticeExtendedStream.oItems[i].e8CallStatus.enType
                 == most_fi_tcl_e8_TelCallStatus::FI_EN_E8RINGTONE)
           {
              ETG_TRACE_USR4((" Setting direction as CALLDIRECTION_INCOMING..."));
              //Fix RTC 169773
              m_au8CallDirection[i] = CALLDIRECTION_INCOMING;
              //End Fix 169773
           }
        }
      } //End of fix for EVO stack call history update
      else
      {
         if (m_poFavoritesProvider != NULL)
         {
            ETG_TRACE_USR1(("oCallStatusNoticeExtendedStatus.oCallStatusNoticeExtendedStream.oItems.Size()== %u", (tUInt)oCallStatusNoticeExtendedStatus.oCallStatusNoticeExtendedStream.oItems.size()));
            while (ItrNew
                  != oCallStatusNoticeExtendedStatus.oCallStatusNoticeExtendedStream.oItems.end())
            {
               m_poFavoritesProvider->CallStatusNotice(ItrNew->u16CallInstance, ItrNew->e8CallStatus.enType);

               if (ItrNew->e8CallStatus.enType
                     == most_fi_tcl_e8_TelCallStatus::FI_EN_E8ACTIVE)
               {
                  u8ActivecallCount++;
               }
               ItrNew++;
            }
            m_u8ActivecallCount = u8ActivecallCount;
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("Version mismatch"));
      m_u8ActivecallCount = 0;
   }
   oCallStatusNoticeExtendedStatus.vDestroy();
}


/*******************************************************************************
** FUNCTION:   vTrigger_DialMethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This function triggers a Dial function method start
*
*  PARAMETERS:
*     IN: PhoneNumber to be dialled
*  RETURNVALUE: None
******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::
vTrigger_Dial_MethodStart(const char* PhoneNumber)
{

   ETG_TRACE_USR3(("Function vTrigger_DialMethodStart entered"));

   m_CurrentCallInstance = 0x00FF;  // GMMY15-10219: (Re)set the current FavDial CallInstance to an invalid one.
   most_telfi_tclMsgDialMethodStart DialMethodStart;
   DialMethodStart.sTelephoneNumber.bSet(PhoneNumber);
   DialMethodStart.e8EchoCancellationNoiseReductionSetting.enType=
      most_fi_tcl_e8_TelEchoCancellationNoiseReductionSetting::FI_EN_E8ECNR_VOICE;

   ETG_TRACE_USR1(("PhoneNumber to be dialled - %s",
      DialMethodStart.sTelephoneNumber.szValue));
   ETG_TRACE_USR1((" EchoCancellationNoiseReductionSetting:: %u",ETG_ENUM(TR_ECHOCANCELLATIONNOISEREDUCTION,DialMethodStart.e8EchoCancellationNoiseReductionSetting.enType)));
   fi_tclVisitorMessage oMsg(DialMethodStart, MOST_TELFI_C_U16_SERVICE_MAJORVERSION);

   DialMethodStart.vDestroy();

   vInitServiceData( oMsg,      // ServiceDataMsg
      0,                                     // CmdCounter
      MOST_TELFI_C_U16_DIAL, // Function ID
      AMT_C_U8_CCAMSG_OPCODE_METHODSTART     // Opcode
      );

   // Post message
   ail_tenCommunicationError enResult1 =
      _poMainAppl->enPostMessage(&oMsg, TRUE);

   if (enResult1 != AIL_EN_N_NO_ERROR)
   {
      // Error situation:

      ETG_TRACE_ERR(("Failed to post message for dial method start"));
   }
   else
   {

      ETG_TRACE_USR4(("Dial method start message posted successfully"));
   }



}

/*******************************************************************************
** FUNCTION:   vHandle_Dial_MethodResult(..)
*******************************************************************************/
/* DESCRIPTION:
*   Method result handler function for dial
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::
vHandle_Dial_MethodResult(amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR4(("Function vHandle_Dial_MethodResult entered"));

   fi_tclVisitorMessage oVisitorMsg( poMessage );

   most_telfi_tclMsgDialMethodResult oDialMethodResult;

   if ( oVisitorMsg.s32GetData(oDialMethodResult, \
      MOST_TELFI_C_U16_SERVICE_MAJORVERSION) 	!= OSAL_ERROR)
   {

      ETG_TRACE_USR1(("Call Instance- %d", oDialMethodResult.u16CallInstance));


      ETG_TRACE_USR1((" Call Status:: %u",ETG_ENUM(TR_CALLSTATUS,oDialMethodResult.e8CallStatus.enType)));
      m_CurrentCallInstance = oDialMethodResult.u16CallInstance;
   }
   else
   {

      ETG_TRACE_ERR(("Version mismatch"));

      if(m_poFavoritesProvider != NULL)
      {
         m_poFavoritesProvider->SendActivateFavoriteMError(CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED);   // Fix for GMMY16-9895
      }
   }
   oDialMethodResult.vDestroy();



}

/*******************************************************************************
** FUNCTION:   vRegisterFavoritesProviderInstance(..)
*******************************************************************************/
/* DESCRIPTION:
*
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::vRegisterFavoritesProviderInstance(
   FC_Phonebook_tclFavorite_Provider* pFavoritesProvider)
{
   m_poFavoritesProvider= pFavoritesProvider;
}

/*******************************************************************************
** FUNCTION:   vHandle_Dial_Error(..)
*******************************************************************************/
/* DESCRIPTION:
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::vHandle_Dial_Error(
   amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR3(("Function vHandle_Dial_Error entered"));

   fi_tclVisitorMessage oVisitorMsg( poMessage );
   tU8 u8ErrorCode;

   most_telfi_tclMsgDialError oDialError;

   if (oVisitorMsg.s32GetData(oDialError, MOST_TELFI_C_U16_SERVICE_MINORVERSION)
      != OSAL_ERROR)
   {
      u8ErrorCode= (tU8)oDialError.e8ErrorCode.enType;


      ETG_TRACE_USR4(("Error Code received - %d", u8ErrorCode));
   }
   else
   {
      u8ErrorCode= CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
   }

   if( m_poFavoritesProvider != NULL)
   {
      m_poFavoritesProvider->SendActivateFavoriteMError(u8ErrorCode);   // Fix for GMMY16-9895
   }
   else
   {
      ETG_TRACE_FATAL((" m_poFavoritesProvider Pointer NULL"));
   }

   oDialError.vDestroy();

}


/*******************************************************************************
** FUNCTION:   vTrigger_SendDTMFMethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This function triggers a SendDTMF function method start
*
*******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::
vTrigger_SendDTMFMethodStart(tU16 u16CallInstance, const char* sDTMFChar)
{

   ETG_TRACE_USR4(("Function vTrigger_SendDTMFMethodStart entered"));

   most_telfi_tclMsgSendDTMFMethodStart SendDTMFMethodStart;
   SendDTMFMethodStart.sDTMFChar.bSet(sDTMFChar);
   SendDTMFMethodStart.u16CallInstance = u16CallInstance;


   ETG_TRACE_USR1(("Call Instance  - %u",
      SendDTMFMethodStart.u16CallInstance));

   ETG_TRACE_USR1(("sDTMFChar   - %s",
      SendDTMFMethodStart.sDTMFChar.szValue));

   fi_tclVisitorMessage oMsg(SendDTMFMethodStart, MOST_TELFI_C_U16_SERVICE_MAJORVERSION);

   SendDTMFMethodStart.vDestroy();

   vInitServiceData( oMsg,      // ServiceDataMsg
      0,                                     // CmdCounter
      MOST_TELFI_C_U16_SENDDTMF, // Function ID
      AMT_C_U8_CCAMSG_OPCODE_METHODSTART     // Opcode
      );

   // Post message
   ail_tenCommunicationError enResult1 =
      _poMainAppl->enPostMessage(&oMsg, TRUE);

   if (enResult1 != AIL_EN_N_NO_ERROR)
   {
      // Error situation:

      ETG_TRACE_ERR(("Failed to post message for SendDTMF method start"));
   }
   else
   {

      ETG_TRACE_USR4(("SendDTMF method start message posted successfully"));
   }



}

/*******************************************************************************
** FUNCTION:   vHandle_SendDTMF_MethodResult(..)
*******************************************************************************/
/* DESCRIPTION:
*   Method result handler function for SendDTMF
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::
vHandle_SendDTMF_MethodResult(amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR4(("Function vHandle_SendDTMF_MethodResult entered"));

   fi_tclVisitorMessage oVisitorMsg( poMessage );

   most_telfi_tclMsgSendDTMFMethodResult oSendDTMFMethodResult;

   if ( oVisitorMsg.s32GetData(oSendDTMFMethodResult, \
      MOST_TELFI_C_U16_SERVICE_MAJORVERSION) 	!= OSAL_ERROR)
   {

      ETG_TRACE_USR1(("Call Instance- %d", oSendDTMFMethodResult.u16CallInstance));


      ETG_TRACE_USR1(("Call Status- %d", (tU16)oSendDTMFMethodResult.bCommandStatus));
      //HOTFIX 047
      if(m_poFavoritesProvider)
         m_poFavoritesProvider->InActivateFavoriteAfterDTMF(m_poFavoritesProvider->FavoriteIdqueue.dequeueCCAMessage());
   }
   else
   {

      ETG_TRACE_ERR(("Version mismatch"));


   }
   oSendDTMFMethodResult.vDestroy();



}

/*******************************************************************************
** FUNCTION:   vHandle_SendDTMF_Error(..)
*******************************************************************************/
/* DESCRIPTION:
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::vHandle_SendDTMF_Error(
   amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR4(("Function vHandle_SendDTMF_Error entered"));

   fi_tclVisitorMessage oVisitorMsg( poMessage );
   tU8 u8ErrorCode;

   most_telfi_tclMsgSendDTMFError oSendDTMFError;

   if (oVisitorMsg.s32GetData(oSendDTMFError, MOST_TELFI_C_U16_SERVICE_MINORVERSION)
      != OSAL_ERROR)
   {
      u8ErrorCode= (tU8)oSendDTMFError.e8ErrorCode.enType;
      //HOTFIX 047
      if(m_poFavoritesProvider)
         m_poFavoritesProvider->InActivateFavoriteAfterDTMF(m_poFavoritesProvider->FavoriteIdqueue.dequeueCCAMessage());
      ETG_TRACE_ERR(("Error Code received - %d", u8ErrorCode));

   }
   else
   {
      u8ErrorCode= CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
   }


   oSendDTMFError.vDestroy();

   ETG_TRACE_USR4(("Function vHandle_Dial_Error exited"));

}

/*******************************************************************************
** FUNCTION:  vHandle_ActivePhoneDevice_Status(..)
*******************************************************************************/
/* DESCRIPTION:
   Get the DeviceHandle of AcitePhoneDevice from Phone.
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::
vHandle_ActivePhoneDevice_Status(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4((" MethodResult: vHandle_ActivePhoneDevice_Status  entered "));

   tU8 u8ActivePhoneDeviceHandle;
   most_telfi_tclMsgActivePhoneDeviceStatus oActivePhoneDeviceStatus;

   fi_tclVisitorMessage oVisitorMsg(poMessage);

   if (oVisitorMsg.s32GetData(oActivePhoneDeviceStatus, MOST_TELFI_C_U16_SERVICE_MAJORVERSION)
         != OSAL_ERROR)
   {
      u8ActivePhoneDeviceHandle = oActivePhoneDeviceStatus.u8DeviceHandle;
      ETG_TRACE_USR1((" Active phone device handle to be updated = %d", u8ActivePhoneDeviceHandle));
      if (fc_phonebook_tclService_PhoneBook::pGetInstance()->u8GetActivePhoneDeviceHandle()
            != u8ActivePhoneDeviceHandle)
      {
         fc_phonebook_tclService_PhoneBook::pGetInstance()->vClearAutocompleteListsOnOSChange();
      }
      fc_phonebook_tclService_PhoneBook::pGetInstance()->vSetActivePhoneDeviceHandle(u8ActivePhoneDeviceHandle);
   }
   else
   {
      ETG_TRACE_FATAL((" Error - in s32GetData."));
   }
   oActivePhoneDeviceStatus.vDestroy();
}


/*******************************************************************************
** FUNCTION:  vHandle_QueryTelephoneNumber_MethodResult(..)
*******************************************************************************/
/* DESCRIPTION:
*   Method result handler function for query telephone number
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::
vHandle_QueryTelephoneNumber_MethodResult(amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR4(("Function vHandle_QueryTelephoneNumber_MethodResult entered"));


   fi_tclVisitorMessage oVisitorMsg( poMessage );

   most_telfi_tclMsgQueryTelephoneNumberMethodResult oQueryTelephoneNumberMethodResult;

   if ( oVisitorMsg.s32GetData(oQueryTelephoneNumberMethodResult, 
      MOST_TELFI_C_U16_SERVICE_MAJORVERSION) 	!= OSAL_ERROR)
   {

      ETG_TRACE_USR1(("Call Instance- %d", oQueryTelephoneNumberMethodResult.u16CallInstance));

      if( m_poFavoritesProvider != NULL)
      {
         QString sTelephoneNumber(oQueryTelephoneNumberMethodResult.sTelephoneNumber.szValue);
         quint16 u16CallInstance = oQueryTelephoneNumberMethodResult.u16CallInstance;
         ETG_TRACE_USR1(("Telephone Number- %s", sTelephoneNumber.toUtf8().constData()));
         m_poFavoritesProvider->QueryTelephoneNumberMethodResult(sTelephoneNumber,u16CallInstance);
      }
      else
      {
         ETG_TRACE_FATAL(("m_poFavoritesProvider Pointer Null"));
      }

   }
   else
   {

      ETG_TRACE_ERR(("Version mismatch"));

   }
   oQueryTelephoneNumberMethodResult.vDestroy();	
}
/*******************************************************************************
** FUNCTION:   vTrigger_QueryTelephoneNumber_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This function triggers a QueryTelephoneNumber function method start
*
*  PARAMETERS:
*     IN: Call instance for which number is to be retrieved
*  RETURNVALUE: None
******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::
vTrigger_QueryTelephoneNumber_MethodStart(tU16 u16CallInstance)
{

   ETG_TRACE_USR3(("Function vTrigger_QueryTelephoneNumber_MethodStart entered"));

   most_telfi_tclMsgQueryTelephoneNumberMethodStart QueryTelephoneNumberMethodStart; 
   QueryTelephoneNumberMethodStart.u16CallInstance = u16CallInstance;

   ETG_TRACE_USR1(("Call Instance - %d",
      QueryTelephoneNumberMethodStart.u16CallInstance));

   fi_tclVisitorMessage oMsg(QueryTelephoneNumberMethodStart, MOST_TELFI_C_U16_SERVICE_MAJORVERSION);

   vInitServiceData( oMsg,      // ServiceDataMsg
      0,                                     // CmdCounter
      MOST_TELFI_C_U16_QUERYTELEPHONENUMBER, // Function ID 
      AMT_C_U8_CCAMSG_OPCODE_METHODSTART     // Opcode
      );

   // Post message
   ail_tenCommunicationError enResult1 =
      _poMainAppl->enPostMessage(&oMsg, TRUE);

   if (enResult1 != AIL_EN_N_NO_ERROR)
   {
      // Error situation:

      ETG_TRACE_ERR(("Failed to post message for Query Telephone Number method start"));
   }
   else
   {

      ETG_TRACE_USR4(("Query Telephone Number method start message posted successfully"));
   }	
   QueryTelephoneNumberMethodStart.vDestroy();
}
/*******************************************************************************
** FUNCTION:   vHandle_QueryTelephoneNumber_Error(..)
*******************************************************************************/
/* DESCRIPTION:
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::vHandle_QueryTelephoneNumber_Error(
   amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3(("Function vHandle_QueryTelephoneNumber_Error entered"));

   fi_tclVisitorMessage oVisitorMsg( poMessage );
   tU8 u8ErrorCode;

   most_telfi_tclMsgQueryTelephoneNumberError oQueryTelephoneNumberError;

   if (oVisitorMsg.s32GetData(oQueryTelephoneNumberError, MOST_TELFI_C_U16_SERVICE_MINORVERSION)
      != OSAL_ERROR)
   {
      u8ErrorCode= (tU8)oQueryTelephoneNumberError.e8ErrorCode.enType;


      ETG_TRACE_USR4(("Error Code received - %d", u8ErrorCode));
   }
   else
   {
      u8ErrorCode= CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
   }	
   oQueryTelephoneNumberError.vDestroy();

   ETG_TRACE_USR4(("Function vHandle_QueryTelephoneNumber_Error exited"));
}

// Fix for GMMY16-9895 Pressing the favorite button that was used to place an outgoing call does not end the call

/*******************************************************************************
** FUNCTION:   vTrigger_Hangup_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This function triggers the Hangup method start
*
*  PARAMETERS:
*     IN: CallInstance to be hungup
*  RETURNVALUE: None
******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::vTrigger_Hangup_MethodStart(tU16 u16CallInstance)
{
   ETG_TRACE_USR3(("Function vTrigger_Hangup_MethodStart entered"));

   m_CurrentCallInstance = 0x00FF;
   most_telfi_tclMsgHangUpCallMethodStart HangUpCallMethodStart;
   HangUpCallMethodStart.u16CallInstance = u16CallInstance;

   ETG_TRACE_USR1(("Hangup Call Instance  - %u", HangUpCallMethodStart.u16CallInstance));

   fi_tclVisitorMessage
      oMsg(HangUpCallMethodStart, MOST_TELFI_C_U16_SERVICE_MAJORVERSION);

   HangUpCallMethodStart.vDestroy();

   vInitServiceData(oMsg, // ServiceDataMsg
      0, // CmdCounter
      MOST_TELFI_C_U16_HANGUPCALL, // Function ID
      AMT_C_U8_CCAMSG_OPCODE_METHODSTART // Opcode
      );

   // Post message
   ail_tenCommunicationError enResult1 =
      _poMainAppl->enPostMessage(&oMsg, TRUE);

   if (enResult1 != AIL_EN_N_NO_ERROR)
   {
      // Error situation:
      ETG_TRACE_ERR(("Failed to post message for hangup method start"));
   }
   else
   {
      ETG_TRACE_USR4(("Hangup method start message posted successfully"));
   }
}

/*******************************************************************************
** FUNCTION:   vHandle_HangUpCall_MethodResult(..)
*******************************************************************************/
/* DESCRIPTION:
*   Method result handler function for HangUpCall
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::vHandle_HangUpCall_MethodResult(amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR4(("Function vHandle_HangUpCall_MethodResult entered"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);

   most_telfi_tclMsgHangUpCallMethodResult oHangUpCall_MR;

   if (oVisitorMsg.s32GetData(oHangUpCall_MR, MOST_TELFI_C_U16_SERVICE_MAJORVERSION)
      != OSAL_ERROR)
   {
      ETG_TRACE_USR1(("Call Instance- %d, Call Status:: %u", oHangUpCall_MR.u16CallInstance, ETG_ENUM(TR_CALLSTATUS, oHangUpCall_MR.e8CallStatus.enType) ));
      m_CurrentCallInstance = oHangUpCall_MR.u16CallInstance;
   }
   else
   {
      ETG_TRACE_ERR(("Version mismatch"));
      if (m_poFavoritesProvider != NULL)
      {
         m_poFavoritesProvider->SendActivateFavoriteMError(CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED);
      }
   }
   oHangUpCall_MR.vDestroy();
}

/*******************************************************************************
** FUNCTION:   vHandle_HangUpCall_Error(..)
*******************************************************************************/
/* DESCRIPTION:
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::vHandle_HangUpCall_Error(amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR3(("Function vHandle_HangUpCall_Error entered"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);
   tU8 u8ErrorCode;

   most_telfi_tclMsgHangUpCallError oHangUpCallError;

   if (oVisitorMsg.s32GetData(oHangUpCallError, MOST_TELFI_C_U16_SERVICE_MINORVERSION)
      != OSAL_ERROR)
   {
      u8ErrorCode = (tU8) oHangUpCallError.e8ErrorCode.enType;

      ETG_TRACE_USR4(("Error Code received - %d", u8ErrorCode));
   }
   else
   {
      u8ErrorCode = CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
   }

   if (m_poFavoritesProvider != NULL)
   {
      m_poFavoritesProvider->SendActivateFavoriteMError(u8ErrorCode);

   }
   else
   {

      ETG_TRACE_FATAL((" m_poFavoritesProvider Pointer NULL"));
   }

   oHangUpCallError.vDestroy();

}

/*******************************************************************************
** FUNCTION:   vTrigger_CancelOutgoingCall_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This function triggers the CancelOutgoingCall method start
*
*  PARAMETERS:
*     IN: CallInstance of the outgoing call to be cancelled
*  RETURNVALUE: None
******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::vTrigger_CancelOutgoingCall_MethodStart(tU16 u16CallInstance)
{
   ETG_TRACE_USR3(("Function vTrigger_CancelOutgoingCall_MethodStart entered for Call Instance: %u", u16CallInstance));

   m_CurrentCallInstance = 0x00FF;
   most_telfi_tclMsgCancelOutgoingCallMethodStart CancelOutgoingCall_MS;
   CancelOutgoingCall_MS.u16CallInstance = u16CallInstance;

   ETG_TRACE_USR1(("CancelOutgoingCall Call Instance  - %u", CancelOutgoingCall_MS.u16CallInstance));

   fi_tclVisitorMessage
      oMsg(CancelOutgoingCall_MS, MOST_TELFI_C_U16_SERVICE_MAJORVERSION);

   CancelOutgoingCall_MS.vDestroy();

   vInitServiceData(oMsg, // ServiceDataMsg
      0, // CmdCounter
      MOST_TELFI_C_U16_CANCELOUTGOINGCALL, // Function ID
      AMT_C_U8_CCAMSG_OPCODE_METHODSTART // Opcode
      );

   // Post message
   ail_tenCommunicationError enResult1 =
      _poMainAppl->enPostMessage(&oMsg, TRUE);

   if (enResult1 != AIL_EN_N_NO_ERROR)
   {
      // Error situation:
      ETG_TRACE_ERR(("Failed to post message for CancelOutgoingCall method start"));
   }
   else
   {
      ETG_TRACE_USR4(("CancelOutgoingCall method start message posted successfully"));
   }
}

/*******************************************************************************
** FUNCTION:   vHandle_CancelOutgoingCall_MethodResult(..)
*******************************************************************************/
/* DESCRIPTION:
*   Method result handler function for CancelOutgoingCall
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::vHandle_CancelOutgoingCall_MethodResult(amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR4(("Function vHandle_CancelOutgoingCall_MethodResult entered"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);

   most_telfi_tclMsgCancelOutgoingCallMethodResult oCancelOutgoingCall_MR;

   if (oVisitorMsg.s32GetData(oCancelOutgoingCall_MR, MOST_TELFI_C_U16_SERVICE_MAJORVERSION)
      != OSAL_ERROR)
   {
      ETG_TRACE_USR1(("Call Instance- %d, Call Status:: %u", oCancelOutgoingCall_MR.u16CallInstance, ETG_ENUM(TR_CALLSTATUS, oCancelOutgoingCall_MR.e8CallStatus.enType) ));
      m_CurrentCallInstance = oCancelOutgoingCall_MR.u16CallInstance;
   }
   else
   {
      ETG_TRACE_ERR(("Version mismatch"));
      if (m_poFavoritesProvider != NULL)
      {
         m_poFavoritesProvider->SendActivateFavoriteMError(CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED);
      }
   }
   oCancelOutgoingCall_MR.vDestroy();
}

/*******************************************************************************
** FUNCTION:   vHandle_CancelOutgoingCall_Error(..)
*******************************************************************************/
/* DESCRIPTION:
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::vHandle_CancelOutgoingCall_Error(amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR3(("Function vHandle_CancelOutgoingCall_Error entered"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);
   tU8 u8ErrorCode;

   most_telfi_tclMsgCancelOutgoingCallError oCancelOutgoingCall_MError;

   if (oVisitorMsg.s32GetData(oCancelOutgoingCall_MError, MOST_TELFI_C_U16_SERVICE_MINORVERSION)
      != OSAL_ERROR)
   {
      u8ErrorCode = (tU8) oCancelOutgoingCall_MError.e8ErrorCode.enType;

      ETG_TRACE_USR4(("Error Code received - %d", u8ErrorCode));
   }
   else
   {
      u8ErrorCode = CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
   }

   if (m_poFavoritesProvider != NULL)
   {
      m_poFavoritesProvider->SendActivateFavoriteMError(u8ErrorCode);

   }
   else
   {

      ETG_TRACE_FATAL((" m_poFavoritesProvider Pointer NULL"));
   }

   oCancelOutgoingCall_MError.vDestroy();

}

/*******************************************************************************
** FUNCTION:   vTrigger_IgnoreCall_MethodStart(..)
*******************************************************************************/
/* DESCRIPTION:
*   This function triggers the IgnoreCall method start
*
*  PARAMETERS:
*     IN: CallInstance of the incoming call to be ignored
*  RETURNVALUE: None
******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::vTrigger_IgnoreCall_MethodStart(tU16 u16CallInstance)
{
   ETG_TRACE_USR3(("Function vTrigger_IgnoreCall_MethodStart entered for Call Instance: %u", u16CallInstance));

   m_CurrentCallInstance = 0x00FF;
   most_telfi_tclMsgIgnoreCallMethodStart IgnoreCall_MS;
   IgnoreCall_MS.u16CallInstance = u16CallInstance;

   ETG_TRACE_USR1(("IgnoreCall Call Instance - %u", IgnoreCall_MS.u16CallInstance));

   fi_tclVisitorMessage
      oMsg(IgnoreCall_MS, MOST_TELFI_C_U16_SERVICE_MAJORVERSION);

   IgnoreCall_MS.vDestroy();

   vInitServiceData(oMsg, // ServiceDataMsg
      0, // CmdCounter
      MOST_TELFI_C_U16_IGNORECALL, // Function ID
      AMT_C_U8_CCAMSG_OPCODE_METHODSTART // Opcode
      );

   // Post message
   ail_tenCommunicationError enResult1 =
      _poMainAppl->enPostMessage(&oMsg, TRUE);

   if (enResult1 != AIL_EN_N_NO_ERROR)
   {
      // Error situation:
      ETG_TRACE_ERR(("Failed to post message for IgnoreCall method start"));
   }
   else
   {
      ETG_TRACE_USR4(("IgnoreCall method start message posted successfully"));
   }
}

/*******************************************************************************
** FUNCTION:   vHandle_IgnoreCall_MethodResult(..)
*******************************************************************************/
/* DESCRIPTION:
*   Method result handler function for IgnoreCall
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::vHandle_IgnoreCall_MethodResult(amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR4(("Function vHandle_IgnoreCall_MethodResult entered"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);

   most_telfi_tclMsgIgnoreCallMethodResult oIgnoreCall_MR;

   if (oVisitorMsg.s32GetData(oIgnoreCall_MR, MOST_TELFI_C_U16_SERVICE_MAJORVERSION)
      != OSAL_ERROR)
   {
      ETG_TRACE_USR1(("Call Instance- %d, Call Status:: %u", oIgnoreCall_MR.u16CallInstance, ETG_ENUM(TR_CALLSTATUS, oIgnoreCall_MR.e8CallStatus.enType) ));
      m_CurrentCallInstance = oIgnoreCall_MR.u16CallInstance;
   }
   else
   {
      ETG_TRACE_ERR(("Version mismatch"));
      if (m_poFavoritesProvider != NULL)
      {
         m_poFavoritesProvider->SendActivateFavoriteMError(CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED);
      }
   }
   oIgnoreCall_MR.vDestroy();
}

/*******************************************************************************
** FUNCTION:   vHandle_IgnoreCall_Error(..)
*******************************************************************************/
/* DESCRIPTION:
*
*  PARAMETERS:
*     IN:
*  RETURNVALUE: None
******************************************************************************/
tVoid fc_phonebook_tclClientHandler_Phone::vHandle_IgnoreCall_Error(amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR3(("Function vHandle_IgnoreCall_Error entered"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);
   tU8 u8ErrorCode;

   most_telfi_tclMsgIgnoreCallError oIgnoreCall_MError;

   if (oVisitorMsg.s32GetData(oIgnoreCall_MError, MOST_TELFI_C_U16_SERVICE_MINORVERSION)
      != OSAL_ERROR)
   {
      u8ErrorCode = (tU8) oIgnoreCall_MError.e8ErrorCode.enType;

      ETG_TRACE_USR4(("Error Code received - %d", u8ErrorCode));
   }
   else
   {
      u8ErrorCode = CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED;
   }

   if (m_poFavoritesProvider != NULL)
   {
      m_poFavoritesProvider->SendActivateFavoriteMError(u8ErrorCode);

   }
   else
   {

      ETG_TRACE_FATAL((" m_poFavoritesProvider Pointer NULL"));
   }

   oIgnoreCall_MError.vDestroy();

}
// End of fix for GMMY16-9895
