/************************************************************************
* FILE:         Phonebook_DefSetHandler.h
* PROJECT:      
* SW-COMPONENT: 
*----------------------------------------------------------------------
*
* DESCRIPTION: Example implementation of a system set listener.
* This handler sets the demodata value used for IncreaseDecrease demo
* back to default on a DefSet request.
* Take this class as a template for your own listener classes and
* adjust the parts marked with
* // +++ comment +++
*
* - ccademo_diaghandler.x              => // +++ Comment +++
* - ccademo_IncreaseDecreaseHandler.x  => // +++ Comment +++
* - Phonebook_DefSetHandler.x            => // +++ Comment +++
* - ccademo_main.x                     => // +++diaglib+++ comment
* - ccademo_msgtypes.h                 => // +++diaglib+++ comment
*              
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2008 Robert Bosch GmbH, Hildesheim
* HISTORY:      
* Date      | Author                 | Modification
* 02.10.08  | CM-AI/PJ-FO45 Evers    | Inital version
* 06.06.11  | TMS Plischke           | Demoserver for GMG2
*
*************************************************************************/
// +++ Remove feature switch when not needed. Used to turn diaglib support on/off +++
// +++ Specify what part of diaglib you need in this file +++
#include "../Configuration/PhoneBook/Pbk_FeatureConfig.h"

#define DIAGLIB_INCLUDE_SYSTEM_SET
#include "diaglib_if.h"
#include "FC_PhoneBook_Debug.h"
#include "FC_PhoneBook_service_PhoneBook.h"
#include "FC_PhoneBook_main.h"
#include "FC_PhoneBook_clienthandler_DefSetHandler.h"
#include "Database/FC_PhoneBook_Database.h"

/* Needed for Trace */
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONEBOOK_APPLICATION
#include "trcGenProj/Header/FC_PhoneBook_clienthandler_DefSetHandler.cpp.trc.h"
#endif


Phonebook_DefSetHandler::Phonebook_DefSetHandler(diaglib::tclServiceDiaglib* pDiaglibService): mpDiaglibService(pDiaglibService)
{
   if(mpDiaglibService != NULL)
   {
      ETG_TRACE_COMP_THR(("--- Phonebook_DefSetHandler::Phonebook_DefSetHandler => vRegisterListener(SYSSETID_PHON_BOOK) "));
      mpDiaglibService->poGetSysSet()->vRegisterListener(SYSSETID_PHON_BOOK, this);
      ETG_TRACE_COMP_THR(("--- Phonebook_DefSetHandler::Phonebook_DefSetHandler => vRegisterListener(SYSSETID_ALL_COMP) "));
      mpDiaglibService->poGetSysSet()->vRegisterListener(SYSSETID_ALL_COMP, this);
      ETG_TRACE_COMP_THR(("--- Phonebook_DefSetHandler::Phonebook_DefSetHandler => vRegisterListener(SYSSETID_GROUP_HMI_CLEAR_ALL_PRIVATE_DATA) "));
      mpDiaglibService->poGetSysSet()->vRegisterListener(SYSSETID_GROUP_HMI_CLEAR_ALL_PRIVATE_DATA, this);
   }
}

Phonebook_DefSetHandler::~Phonebook_DefSetHandler()
{
   mpDiaglibService = NULLPTR;
}

tU32 Phonebook_DefSetHandler::vOnSystemSet   ( 
   tU32 u32SystemSetID,
   diaglib::tenSystemSetType u32SystemSetType,
   diaglib::tContext MsgContext
   )
{
   ETG_TRACE_USR3_THR(("--> Phonebook_DefSetHandler::vOnSystemSet entered with u32SystemSetID: %d and u32SystemSetType: %d",u32SystemSetID,u32SystemSetType));
   tU32 u32Return = diaglib::U32_DIAGLIB_RETURN_OK;
   diaglib::tclParameterVector oTmpVec;
   diaglib::tenSystemSetResult oSysSetResult = diaglib::EN_SYSTEMSET_OK; // default response - success
   tBool bResetPrivateData = FALSE;
   tBool bResetTotalContactCount = FALSE;

   // +++ Set values back to default, then send positive response +++

   switch(u32SystemSetID)
   {
   case SYSSETID_ALL_COMP:
   case SYSSETID_PHON_BOOK: // single Group
      {
         switch(u32SystemSetType)
         {
            case diaglib::EN_TYPE_DEFAULT_TEF:
            {
               ETG_TRACE_COMP_THR(("--- Phonebook_DefSetHandler::vOnSystemSet => EN_TYPE_DEFAULT_TEF"));
               bResetPrivateData = TRUE;
               break;
            }
            // RTC- 681736 - SMART
            case diaglib::EN_TYPE_DEFAULT_CUSTOMER:
            {
               ETG_TRACE_USR4(("--- Phone_DefSetHandler::vOnSystemSet => EN_TYPE_DEFAULT_CUSTOMER"));
               bResetPrivateData = TRUE;
               bResetTotalContactCount = TRUE;
               break;
            }// case EN_TYPE_DEFAULT_CUSTOMER:

            case diaglib::EN_TYPE_DEFAULT_HMI:
            {
               ETG_TRACE_COMP_THR(("--- Phonebook_DefSetHandler::vOnSystemSet => EN_TYPE_DEFAULT_HMI"));
               bResetPrivateData = TRUE;
               break;
            }

            case diaglib::EN_TYPE_CODING: // Customer EOL
            {
               ETG_TRACE_COMP_THR(("--- Phonebook_DefSetHandler::vOnSystemSet => EN_TYPE_CODING"));
               bResetPrivateData = TRUE;
               break;
            }

            default:
            {
               ETG_TRACE_ERR_THR(("!!! Phonebook_DefSetHandler::vOnSystemSet => ERROR: not supported Systemset type"));
               oSysSetResult = diaglib::EN_SYSTEMSET_NOT_OK;

               diaglib::trParameter paramError;
               paramError.enType         = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
               paramError.u8Value        = diaglib::EN_ERROR_INCOMPATIBLE_PARAMETER_SIGNATURE;
               oTmpVec.push_back(paramError);
               break;
            }
         }
         break;
      }// case SYSSETID_ALL_COMP and case SYSSETID_PHON_BOOK

   case SYSSETID_GROUP_HMI_CLEAR_ALL_PRIVATE_DATA:   
      {
         ETG_TRACE_COMP_THR(("--- Phonebook_DefSetHandler::vOnSystemSet => SYSSETID_GROUP_HMI_CLEAR_ALL_PRIVATE_DATA"));
         bResetPrivateData = TRUE;
         break;
      } 

   default:
      {
         ETG_TRACE_ERR_THR(("!!! Phonebook_DefSetHandler::vOnSystemSet => ERROR: not supported Systemset ID"));
         oSysSetResult = diaglib::EN_SYSTEMSET_NOT_OK;

         diaglib::trParameter paramError;
         paramError.enType         = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
         paramError.u8Value        = diaglib::EN_ERROR_ID_NOT_SUPPORTED;
         oTmpVec.push_back(paramError);
         break;
      }
   }// switch(u32SystemSetID)

   if(bResetPrivateData)
   {
      bool bTriggerDbRecreation = false;  // -PHBK-QUERY-ERR- Instrumentation: for GMMY17-13044 "database disk image is malformed"
   
      FC_PhoneBook_SQLite* poFC_PhoneBook_SQLite = FC_PhoneBook_SQLite::GetSQLiteInstance();
      fc_phonebook_tclService_PhoneBook* poPhoneBookService = fc_phonebook_tclService_PhoneBook::pGetInstance();
      quint8 u8Sortorder = most_fi_tcl_e8_PhonBkPreferredPhoneBookSortOrder::FI_EN_E8PREFERRED_SORT_ORDER_FIRSTNAME;
      tBool bAutoDnlValue = TRUE;
      ETG_TRACE_USR4(("Phonebook_DefSetHandler::vOnSystemSet bAutoDnlValue =%u ",bAutoDnlValue ));

      if(poFC_PhoneBook_SQLite!=NULL)
      {
         if( false == poFC_PhoneBook_SQLite->UpdatePreferredSortOrder(u8Sortorder) )
         {
            bTriggerDbRecreation = true;  // -PHBK-QUERY-ERR- Instrumentation: for GMMY17-13044 "database disk image is malformed"
         }
         quint8 u8DeviceHandle = 0;
         poFC_PhoneBook_SQLite->bUpdateAutomaticContactDownload(u8DeviceHandle,bAutoDnlValue);   //CMG3G-8366
      }

      if(poPhoneBookService != NULL)
      {
         //Set preferred sort order
         most_phonbkfi_tclMsgPreferredPhoneBookSortOrderStatus PreferredPhoneBookSortOrderStatus;
         PreferredPhoneBookSortOrderStatus.e8PreferredPhoneBookSortOrder.enType = (u8Sortorder == 0)? (most_fi_tcl_e8_PhonBkPreferredPhoneBookSortOrder::FI_EN_E8PREFERRED_SORT_ORDER_FIRSTNAME) : (most_fi_tcl_e8_PhonBkPreferredPhoneBookSortOrder::FI_EN_E8PREFERRED_SORT_ORDER_LASTNAME);
         poPhoneBookService->vPhoneBook_UpdatePreferredPhoneBookSortOrderStatus(PreferredPhoneBookSortOrderStatus);

         //Clear favorites, contacts and call history
         ETG_TRACE_USR4(("Clearing favorites, contacts and call history"));
         if( false == poPhoneBookService->m_pPhoneBookDB->ClearPhoneBookData() ) //FIX GMMY16-23858
         {
            bTriggerDbRecreation = true;  // -PHBK-QUERY-ERR- Instrumentation: for GMMY17-13044 "database disk image is malformed"
         }
      }

      if(bResetTotalContactCount)
      {
         updateMaxContactCount(DEFAULT_CONTACT_COUNT);
      }

      if(bTriggerDbRecreation)  // -PHBK-QUERY-ERR- Instrumentation: for GMMY17-13044 "database disk image is malformed"
      {
         if(poFC_PhoneBook_SQLite!=NULL)
         {
            poFC_PhoneBook_SQLite->vSetDbRecreationTrigger();
            ETG_TRACE_FATAL (( " -PHBK-QUERY-ERR-: Database RE-CREATION trigger has been set in DefSet handler!! "  ));
            ETG_TRACE_ERRMEM(( " -PHBK-QUERY-ERR-: Database RE-CREATION trigger has been set in DefSet handler!! "  ));
         }
         else
         {
            ETG_TRACE_FATAL (( " -PHBK-QUERY-ERR-: poFC_PhoneBook_SQLite == NULL "  ));
            ETG_TRACE_ERRMEM(( " -PHBK-QUERY-ERR-: poFC_PhoneBook_SQLite == NULL "  ));         
         }
      }
   }
   // send Data back to Server
   if(mpDiaglibService)
   {
      if(TRUE == mpDiaglibService->poGetSysSet()->bSendSystemSetResult(  oSysSetResult, 
         oTmpVec, 
         MsgContext) )
      {
         ETG_TRACE_USR4_THR(("--- Phonebook_DefSetHandler::vOnSystemSet => bSendSystemSetResult successfull"));
      }
      else
      {
         ETG_TRACE_ERR_THR(("!!! Phonebook_DefSetHandler::vOnSystemSet => ERROR : bSendSystemSetResult failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR_THR(("!!! Phonebook_DefSetHandler::vOnSystemSet => ERROR : mpDiaglibService == OSAL_NULL"));
   }

   ETG_TRACE_USR3_THR(("<-- Phonebook_DefSetHandler::vOnSystemSet(%x)",u32Return));
   return u32Return;
}

tU32 Phonebook_DefSetHandler::vOnSystemSetFinished   ( 
   tU32 u32SystemSetID, 
   diaglib::tenSystemSetType u32SystemSetType,
   diaglib::tContext MsgContext
   )
{
   ETG_TRACE_USR3_THR(("--> Phonebook_DefSetHandler::vOnSystemSetFinished"));
   // +++ Reallow access to values that needed default setting, then send positive response +++
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32SystemSetID );
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32SystemSetType );
   /*
   Respond does not contain any data and is just an acknowledgement.
   */

   if(mpDiaglibService)
   {
      if(TRUE == mpDiaglibService->poGetSysSet()->bAcknowledgeSystemSetFinished(MsgContext))
      {
         // +++ Trace info +++
      }
      else
      {
         // +++ Trace error +++
      }
   }
   else
   {
      // +++ Trace error +++
   }

   ETG_TRACE_USR3_THR(("<-- Phonebook_DefSetHandler::vOnSystemSetFinished"));
   return diaglib::U32_DIAGLIB_RETURN_OK;
}

tU32 Phonebook_DefSetHandler::vOnSystemSetPrepare   ( 
   tU32 u32SystemSetID, 
   diaglib::tenSystemSetType u32SystemSetType,
   diaglib::tContext MsgContext
   )
{
   ETG_TRACE_USR3_THR(("--> Phonebook_DefSetHandler::vOnSystemSetPrepare"));
   // +++ Stop access to values that need default setting, then send positive response +++
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32SystemSetID );
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32SystemSetType );
   tBool bResponse = TRUE;
   /*
   Response can be one of the following:
   EN_SYSTEMSET_OK            -> Empty error vector

   Actually the diaglib already checks if ID is supported, so we don't need
   to do it here again.
   */
   diaglib::tclParameterVector oTmpVec;


   if(mpDiaglibService)
   {
      if(TRUE == mpDiaglibService->poGetSysSet()->bSendSystemSetResult( (bResponse ? diaglib::EN_SYSTEMSET_OK : diaglib::EN_SYSTEMSET_NOT_OK), 
         oTmpVec, 
         MsgContext) )
      {
         // +++ Trace info +++
      }
      else
      {
         // +++ Trace error +++
      }
   }
   else
   {
      // +++ Trace error +++
   }
   ETG_TRACE_USR3_THR(("<-- Phonebook_DefSetHandler::vOnSystemSetPrepare"));
   return diaglib::U32_DIAGLIB_RETURN_OK;
}

tU32 Phonebook_DefSetHandler::vOnSystemSetCheck ( 
   tU32 u32SystemSetID, 
   diaglib::tenSystemSetType u32SystemSetType,
   diaglib::tContext MsgContext
   )
{
   ETG_TRACE_USR3_THR(("--> Phonebook_DefSetHandler::vOnSystemSetCheck"));
   // +++ Check if all values are set to default. +++
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32SystemSetID );
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32SystemSetType );
   tBool bResponse = TRUE;
   /*
   Response can be one of the following:
   EN_SYSTEMSET_OK            -> Empty error vector
   EN_SYSTEMSET_NOT_OK        -> List of wrong values

   diaglib already checks if ID is supported, so we don't need
   to do it here again.
   */
   diaglib::tclParameterVector oTmpVec;

   if(mpDiaglibService)
   {
      if(TRUE == mpDiaglibService->poGetSysSet()->bSendSystemSetCheckResult( (bResponse ? diaglib::EN_SYSTEMSET_OK : diaglib::EN_SYSTEMSET_NOT_OK), 
         oTmpVec, 
         MsgContext) )
      {
         // +++ Trace info +++
      }
      else
      {
         // +++ Trace error +++
      }
   }
   else
   {
      // +++ Trace error +++
   }
   ETG_TRACE_USR3_THR(("<-- Phonebook_DefSetHandler::vOnSystemSetCheck"));
   return diaglib::U32_DIAGLIB_RETURN_OK;
}
