/*******************************************************************************
*                                                                               
* FILE:          FC_PhoneBook_clienthandler_BluetoothSetings.cpp           
*                                                                               
* SW-COMPONENT:  FC_PhoneBook application                                  
*                                                                               
* PROJECT:       Bosch                                                              
*                                                                               
* DESCRIPTION:   CCA client-handler for the service BluetoothSetings.              
*                                                                               
* AUTHOR:        Arun Choudhary                                                              
*                                                                               
* COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim                          
*                                                                               
*******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/
#include "FC_PhoneBook_Debug.h"
#include "FC_PhoneBook_clienthandler_BluetoothSetings.h"
#include "FC_PhoneBook_service_PhoneBook.h"
#include "Database/FC_PhoneBook_List.h"
#include "FC_PhoneBook_Device_Details.h"
#include "Database/FC_PhoneBook_SQLite.h"
#include "DbusClient/FC_PhoneBook_EvoDbusClient.h"

                             // Include message framework interface (AMT, msgfw)
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE

                                          //Include FI interface of used service

#define MOST_FI_S_IMPORT_INTERFACE_MOST_BTSETFI_TYPES
#define MOST_FI_S_IMPORT_INTERFACE_MOST_BTSETFI_FUNCTIONIDS
#define MOST_FI_S_IMPORT_INTERFACE_MOST_BTSETFI_ERRORCODES
#define MOST_FI_S_IMPORT_INTERFACE_MOST_BTSETFI_SERVICEINFO

#include "conn_most_fi_if.h"


//#include "midw_fi_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONEBOOK_APPLICATION
#include "trcGenProj/Header/FC_PhoneBook_clienthandler_BluetoothSetings.cpp.trc.h"
#endif

/******************************************************************************/
/*                                                                            */
/* DEFINES                                                                    */
/*                                                                            */
/******************************************************************************/
//Key: Device Handle, Value: Device Address
std::map<tU8, std::string> fc_phonebook_tclClientHandler_BluetoothSetings::ConnnectedDevicesMap; // 241829- Replace bpstl:: by std::

/******************************************************************************/
/*                                                                            */
/* CCA MESSAGE MAP                                                            */
/*                                                                            */
/******************************************************************************/

BEGIN_MSG_MAP(fc_phonebook_tclClientHandler_BluetoothSetings, ahl_tclBaseWork)

   // Add your ON_MESSAGE_SVCDATA() macros here to define which corresponding
   // method should be called on receiving a specific message.

ON_MESSAGE_SVCDATA( MOST_BTSETFI_C_U16_DEVICELISTEXTENDED,
      AMT_C_U8_CCAMSG_OPCODE_STATUS ,
      vHandle_DeviceListExtended_Status )

ON_MESSAGE_SVCDATA( MOST_BTSETFI_C_U16_VEHICLEBTADDRESS,
      AMT_C_U8_CCAMSG_OPCODE_STATUS ,
      vHandle_VehicleBtAddress_Status )

ON_MESSAGE_SVCDATA( MOST_BTSETFI_C_U16_CONNECTBLUETOOTHPROFILE, \
      AMT_C_U8_CCAMSG_OPCODE_METHODRESULT , \
      vHandle_ConnectBluetoothProfile_MethodResult )

ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_CONNECTBLUETOOTHPROFILE,
      AMT_C_U8_CCAMSG_OPCODE_ERROR,
      vHandle_ConnectBluetoothProfile_Error)

ON_MESSAGE_SVCDATA( MOST_BTSETFI_C_U16_DISCONNECTBLUETOOTHPROFILE,
      AMT_C_U8_CCAMSG_OPCODE_METHODRESULT ,
      vHandle_DisconnectBluetoothProfile_MethodResult )

ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_DISCONNECTBLUETOOTHPROFILE,
      AMT_C_U8_CCAMSG_OPCODE_ERROR,
      vHandle_DisconnectBluetoothProfile_Error)

END_MSG_MAP()

/******************************************************************************/
/*                                                                            */
/* METHODS                                                                    */
/*                                                                            */
/******************************************************************************/

/*******************************************************************************
*
* FUNCTION: fc_phonebook_tclClientHandler_BluetoothSetings::
*     fc_phonebook_tclClientHandler_BluetoothSetings(fc_phonebook_tclApp* poMainAppl)
*
* DESCRIPTION: Constructor.
*
*              Create an object of the base class
*              ahl_tclBaseOneThreadClientHandler with a pointer to this
*              application, the to be used service identifier and the service
*              version as parameters.
*
* PARAMETER: [IN] poMainAppl = Pointer to the object of this application.
*
* RETURNVALUE: None.
*
*******************************************************************************/
fc_phonebook_tclClientHandler_BluetoothSetings::
 fc_phonebook_tclClientHandler_BluetoothSetings(fc_phonebook_tclApp* poMainAppl)
    : ahl_tclBaseOneThreadClientHandler(
          // Application Pointer
          poMainAppl,
          // ID of used Service
          CCA_C_U16_SRV_FB_BLUETOOTHSETTINGS,
          // MajorVersion of used Service
          MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION,
          // MinorVersion of used Service
          MOST_BTSETFI_C_U16_SERVICE_MINORVERSION )
{
   ETG_TRACE_USR4(("fc_phonebook_tclClientHandler_BluetoothSetings() entered."));
   m_poPhoneBookService = NULLPTR;
   m_u8DeviceConnectInProgress = 0;
}

/*******************************************************************************
*
* FUNCTION: fc_phonebook_tclClientHandler_BluetoothSetings::
*             ~fc_phonebook_tclClientHandler_BluetoothSetings(tVoid)
*
* DESCRIPTION: Destructor.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
fc_phonebook_tclClientHandler_BluetoothSetings::
  ~fc_phonebook_tclClientHandler_BluetoothSetings(tVoid)
{
  ETG_TRACE_USR4(("~fc_phonebook_tclClientHandler_BluetoothSetings() entered."));
  
  m_poPhoneBookService = NULLPTR;
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_phonebook_tclClientHandler_BluetoothSetings::
                  vOnServiceAvailable()
*
* DESCRIPTION: This function is called by the CCA framework when the service
*              this client-handler has registered for has become available.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadClientHandler::vOnServiceAvailable().
*******************************************************************************/
tVoid fc_phonebook_tclClientHandler_BluetoothSetings::vOnServiceAvailable()
{
  ETG_TRACE_USR1(("vOnServiceAvailable() entered. AppID = %u.",
                  u16GetServerAppID()));
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_phonebook_tclClientHandler_StreamRouter::
                       vOnServiceUnavailable()
*
* DESCRIPTION: This function is called by the CCA framework when the service
*              this client-handler has registered for has become unavailable.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadClientHandler::vOnServiceUnavailable().
*******************************************************************************/
tVoid fc_phonebook_tclClientHandler_BluetoothSetings::vOnServiceUnavailable()
{
  ETG_TRACE_USR1(("vOnServiceUnavailable() entered. AppID = %u.",
                  u16GetServerAppID()));
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_phonebook_tclClientHandler_BluetoothSetings::
                  vProcessTimer(tU16 u16TimerId)
*
* DESCRIPTION: This function is called from the superior application of this
*              client-handler in case of the expiration of a timer.
*
* PARAMETER: [IN] u16TimerId = Identifier of the expired timer.
*
* RETURNVALUE: -
*
*******************************************************************************/
tVoid fc_phonebook_tclClientHandler_BluetoothSetings:: \
  						  vProcessTimer(tU16 u16TimerId)
{
  ETG_TRACE_USR1(("vProcessTimer() entered. TimerID = %u.",u16TimerId));

  if(TRUE == bIfServiceAvailable())
  {
    // Create message for used service.
  }
}


/*******************************************************************************
*
* FUNCTION: tVoid fc_phonebook_tclClientHandler_BluetoothSetings::
                  vRegisterForBTProperties()
*
* DESCRIPTION: This function is called once phonebook service available to register for BT properties.
*
* PARAMETER: None.
*
* RETURNVALUE: -
*
*******************************************************************************/
tVoid fc_phonebook_tclClientHandler_BluetoothSetings:: \
						vRegisterForBTProperties()
{
  ETG_TRACE_USR1(("vRegisterForBTProperties() entered."));

  vAddAutoRegisterForProperty( MOST_BTSETFI_C_U16_VEHICLEBTADDRESS);
  vAddAutoRegisterForProperty( MOST_BTSETFI_C_U16_DEVICELISTEXTENDED);
}


/*******************************************************************************
 ** FUNCTION:   vHandle_DeviceListExtended_Status(..)
*******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for DeviceListExtended property status
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 ******************************************************************************/
tVoid fc_phonebook_tclClientHandler_BluetoothSetings::vHandle_DeviceListExtended_Status(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4((" Property Status:DeviceListExtendedStatus in PhoneBook entered "));

   // Extract required info from poMessage
   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // Create the (message related) FI data object
   most_btsetfi_tclMsgDeviceListExtendedStatus oDeviceListExtendedStatus;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.
   if (oVisitorMsg.s32GetData(oDeviceListExtendedStatus, MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION)
         != OSAL_ERROR)
   {
      //FIX SUZUKI-22811 Phone settings (Ringtone, First/last name, paired device list) to be persistent even after SW update
      //FC_Phonebook has to validate a connected device reported
      //Get all existing handles and addresses from DB.
      std::map<tU8, std::string> DeviceHandlesAndDeviceAddr; // 241829- Replace bpstl:: by std::
      std::map<tU8, std::string>::iterator iter; // 241829- Replace bpstl:: by std::
      FC_PhoneBook_SQLite::GetSQLiteInstance()->vGetDevicesInDB(DeviceHandlesAndDeviceAddr);

      tU8 u8DeviceListExtendedSize =
            (tU8) oDeviceListExtendedStatus.oDeviceListExtendedResult.oItems.size();
      if (0 != u8DeviceListExtendedSize)
      {
         for (tU8 u8DeviceListExtendedIndex = 0; u8DeviceListExtendedIndex < u8DeviceListExtendedSize; u8DeviceListExtendedIndex++)
         {
            most_fi_tcl_BTSetDeviceListExtendedResultItem DeviceListExtendedItem =
                              oDeviceListExtendedStatus.oDeviceListExtendedResult.oItems[u8DeviceListExtendedIndex];

            if (!vIsDeviceHandleValid(DeviceListExtendedItem.u8DeviceHandle))
            {
               ETG_TRACE_ERR(("vHandle_DeviceList_Status:: DeviceHandle (%d) is invalid",DeviceListExtendedItem.u8DeviceHandle));
               continue;
            }

            ETG_TRACE_USR1((" B_BTDeviceListItem[%u]::u8DeviceHandle :%u", u8DeviceListExtendedIndex, DeviceListExtendedItem.u8DeviceHandle));
            ETG_TRACE_USR1((" B_BTDeviceListItem[%u]::sDeviceName.szValue :%s", u8DeviceListExtendedIndex, DeviceListExtendedItem.sDeviceName.szValue));
            ETG_TRACE_USR1((" B_BTDeviceListItem[%u]::oDeviceProfileStatus.bHFP :%u", u8DeviceListExtendedIndex, DeviceListExtendedItem.oDeviceProfileStatus.bHFP));
            ETG_TRACE_USR1((" B_BTDeviceListItem[%u]::oDeviceProfileStatus.bPBAP :%u", u8DeviceListExtendedIndex, DeviceListExtendedItem.oDeviceProfileStatus.bPBAP));

            //Convert received device address to 'dev_aa_bb_cc_dd_ee_ff' format of DB
            char achDeviceAddr[LENGTH_OF_DEVICE_ADDRESS];
            memset(achDeviceAddr, '\0', LENGTH_OF_DEVICE_ADDRESS);
            memcpy(achDeviceAddr, "dev", 3);

            for (tU8 u8Index = 0; u8Index < 6; u8Index++)
            {
               achDeviceAddr[3 + u8Index * 3] = '_';
               achDeviceAddr[4 + u8Index * 3]
                     = DeviceListExtendedItem.sDeviceAddress.szValue[0
                           + u8Index * 2];
               achDeviceAddr[5 + u8Index * 3]
                     = DeviceListExtendedItem.sDeviceAddress.szValue[1
                           + u8Index * 2];
            }
            achDeviceAddr[LENGTH_OF_DEVICE_ADDRESS-1] = '\0';
            ETG_TRACE_USR1((" B_BTDeviceListItem[%u]::Formatted device address : %s", u8DeviceListExtendedIndex, achDeviceAddr));

            //On every notification of the property update from the FC_Bluetooth,
            //FC_Phonebook has to validate its table with (DeviceHandle,Device Address) pair.
            //If any mismatch then it has to delete the Phonebook database tables.
            //Also, clear DB when DeviceListExtendedStatus is empty.

            //Validate against entries in DB. On validation success delete entry from DeviceHandlesAndDeviceAddr map.
            iter = DeviceHandlesAndDeviceAddr.find(DeviceListExtendedItem.u8DeviceHandle);
            if (iter != DeviceHandlesAndDeviceAddr.end())
            {
               ETG_TRACE_USR4((" Device handle found in DB... "));
               if (VALUE_ZERO == u8StringCompare(iter->second.c_str(), achDeviceAddr))
               {
                  ETG_TRACE_USR4((" Device validated against same address... "));
                  DeviceHandlesAndDeviceAddr.erase(iter);
               }
               else
               {
                  ETG_TRACE_USR4((" Device validated against different address!!! "));
               }
            }

            //Handle device connection/disconnection
            iter
                  = ConnnectedDevicesMap.find(DeviceListExtendedItem.u8DeviceHandle);
            if (iter != ConnnectedDevicesMap.end()) // Entry found in ConnnectedDevicesMap
            {
               if (FALSE == DeviceListExtendedItem.oDeviceProfileStatus.bHFP)
               {
                  //Device which was already connected got disconnected
                  ETG_TRACE_COMP((" -PBDL.S- Phone Device (HFP) got DISCONNECTED, DeviceHandle= %d, DeviceName= '%s' ", DeviceListExtendedItem.u8DeviceHandle, DeviceListExtendedItem.sDeviceName.szValue));
                  vDeviceDisconnected(DeviceListExtendedItem.u8DeviceHandle);
                  ConnnectedDevicesMap.erase(iter);
               }
            }
            else // Entry not found in ConnnectedDevicesMap
            {
               if (TRUE == DeviceListExtendedItem.oDeviceProfileStatus.bHFP)
               {
                  ETG_TRACE_COMP((" -PBDL.S- Phone Device got newly CONNECTED, DeviceHandle= %d, DeviceName= '%s' ", DeviceListExtendedItem.u8DeviceHandle, DeviceListExtendedItem.sDeviceName.szValue));
                  ConnnectedDevicesMap.insert(std::pair<tU8, std::string>(DeviceListExtendedItem.u8DeviceHandle, achDeviceAddr)); // 241829- Replace bpstl:: by std::

                  //Process new device connection if no other connection in progress.
                  if (0 == u8GetDeviceConnectInProgress())
                  {
                     //Fix : CMG3G-12880
#ifdef HANDLE_PROBABLY_LOCKED_SCENARIO
                     FC_Device_Details *pDeviceObject = NULL;
                     for(tU8 u8Index= 0; u8Index < NO_OF_CONNECTED_DEVICE; u8Index++)
                     {
                        pDeviceObject = FC_Device_Details::getDeviceObject((INDEX)u8Index);
                        if(pDeviceObject && TRUE == pDeviceObject->bIsDeviceProbablyLocked())
                        {
                           ETG_TRACE_USR4((" Stopping retry for download sequence as a new hfp device got connected. "));
                           m_poPhoneBookService->m_oPbk_DnlManager.PhoneBook_vStopRetry(FC_PhoneBook_RetryDetails::RETRY_DOWNLOAD_SEQUENCE, pDeviceObject->getConnectionID());
                           break;
                        }
                     }
#endif
                     vSetDeviceConnectInProgress(DeviceListExtendedItem.u8DeviceHandle);
                     vPostConnectLoopback(DeviceListExtendedItem.u8DeviceHandle, achDeviceAddr);
                  }
                  else
                  {
                     ETG_TRACE_USR4((" Device connection in progress for another device..."));
                  }
               }
            }
         }
      }
      else
      {
         ETG_TRACE_USR4((" Property Status:Device List Empty  in PhoneBook  "));
         vSetDeviceConnectInProgress(0); //Reset device handle to ZERO to denote no device connection is in progress
         ConnnectedDevicesMap.clear();
      }

      //Delete those entries in DB, but not in property update, if any.
      FC_Phonebook_tclFavorite_Provider* lPoFavorite_Provider;
      lPoFavorite_Provider = FC_Phonebook_tclFavorite_Provider::GetInstance();

      iter = DeviceHandlesAndDeviceAddr.begin();
      while (iter != DeviceHandlesAndDeviceAddr.end())
      {
         //Reset favorites
         if (lPoFavorite_Provider)
         {
            lPoFavorite_Provider->Reset_Favorites(true, iter->first);
         }
         ETG_TRACE_USR4((" Calling device deleted for device handle: %d", iter->first));
         vDeviceDeleted(iter->first);
         iter++;
      }
   }
   
   oDeviceListExtendedStatus.vDestroy();
}


/*******************************************************************************
 ** FUNCTION:   vUpdate_ClientHandler_PhoneBook
*******************************************************************************/
/* DESCRIPTION:
 *
 *  PARAMETERS:
 *     IN:   poMessage        :
 *  RETURNVALUE:
*******************************************************************************/
tVoid fc_phonebook_tclClientHandler_BluetoothSetings::vUpdate_ClientHandler_PhoneBook(fc_phonebook_tclService_PhoneBook *& p_clientHandler_PhoneBook)
{
   m_poPhoneBookService = p_clientHandler_PhoneBook;
}


/*******************************************************************************
 ** FUNCTION:   vDeviceDisconnected
*******************************************************************************/
/* DESCRIPTION:
 *
 *  PARAMETERS:
 *     IN:   poMessage:
 *  RETURNVALUE:
 *******************************************************************************/
tVoid fc_phonebook_tclClientHandler_BluetoothSetings::vDeviceDisconnected(tU8 u8DeviceHandle)
{
   ETG_TRACE_USR4(("vDeviceDisconnected() entered."));

   ETG_TRACE_USR4(("Posting loopback message for FID - FC_PB_LB_FID_DISCONNECT"));

   PhBk_Loopback_Data Disconnection_LB_Data;
   Disconnection_LB_Data.u16FunctionID = (guint16)FC_PB_LB_FID_DISCONNECT;
   Disconnection_LB_Data.ucDeviceHandle = u8DeviceHandle;
   if (m_poPhoneBookService != NULL)
   {
      m_poPhoneBookService->vPush_Back_LB_Data(Disconnection_LB_Data);
      m_poPhoneBookService->vPrepareandsendLoopBackMsg(FC_PB_LB_FID_DISCONNECT);
   }
   else
   {
      ETG_TRACE_FATAL(("NULL POINTER : m_poPhoneBookService"));
   }
}

/*******************************************************************************
 ** FUNCTION:   vDeviceDeleted
*******************************************************************************/
/* DESCRIPTION:
 *
 *  PARAMETERS:
 *     IN:   poMessage:
 *  RETURNVALUE:
*******************************************************************************/
tVoid fc_phonebook_tclClientHandler_BluetoothSetings::vDeviceDeleted(tU8 u8DeviceHandle)
{
   ETG_TRACE_USR4(("vDeviceDeleted() entered."));

   PhBk_Loopback_Data Device_Deleted_LB_Data;
   if ((m_poPhoneBookService != NULL) && (vIsDeviceHandleValid(u8DeviceHandle)))
   {
      Device_Deleted_LB_Data.u16FunctionID
            = (guint16)FC_PB_LB_FID_DEVICE_DELETED;
      Device_Deleted_LB_Data.ucDeviceHandle = u8DeviceHandle;
      m_poPhoneBookService->vPush_Back_LB_Data(Device_Deleted_LB_Data);
      m_poPhoneBookService->vPrepareandsendLoopBackMsg(FC_PB_LB_FID_DEVICE_DELETED);
      ETG_TRACE_USR4(("Posting loopback message for FID - FC_PB_LB_FID_DEVICE_DELETED"));
   }
   else
   {
      ETG_TRACE_FATAL(("NULL POINTER : m_poPhoneBookService"));
   }
}


/*******************************************************************************
 ** FUNCTION:   vPostConnectLoopback
*******************************************************************************/
/* DESCRIPTION:
 *
 *  PARAMETERS:
 *  IN:   u8DeviceHandle  : Device for which connect loopback is to be posted.
 *  IN:   deviceAddress  : Device address for which connect loopback is to be posted.
 *  RETURNVALUE: None.
*******************************************************************************/
tVoid fc_phonebook_tclClientHandler_BluetoothSetings::vPostConnectLoopback(tU8 u8DeviceHandle, const tChar *deviceAddress)
{
   ETG_TRACE_USR4((" vPostConnectLoopback  entered with device handle: %d", u8DeviceHandle));

   if (!m_poPhoneBookService)
   {
      ETG_TRACE_ERR((" m_poPhoneBookService  is NULL! "));
      return;
   }

   PhBk_Loopback_Data NewConnection_LB_Data;

   memset(NewConnection_LB_Data.ucDevAddr, 0, LENGTH_OF_DEVICE_ADDRESS);
   memcpy(NewConnection_LB_Data.ucDevAddr, deviceAddress, LENGTH_OF_DEVICE_ADDRESS);
   ETG_TRACE_USR1((" Device address: %s", NewConnection_LB_Data.ucDevAddr));

   NewConnection_LB_Data.u16FunctionID = (guint16)FC_PB_LB_FID_CONNECT;
   NewConnection_LB_Data.ucDeviceHandle = u8DeviceHandle;

   ETG_TRACE_USR4(("Posting loopback message for FID - FC_PB_LB_FID_CONNECT"));

   m_poPhoneBookService->vPush_Back_LB_Data(NewConnection_LB_Data);
   m_poPhoneBookService->vPrepareandsendLoopBackMsg(FC_PB_LB_FID_CONNECT);
}


/*******************************************************************************
 ** FUNCTION:   vHandle_VehicleBtAddress_Status
*******************************************************************************/
/* DESCRIPTION: Get the VehicleBtAddress_Status from BT Settings.
 *
 *  PARAMETERS:
 *  IN:   poMessage        :
 *  RETURNVALUE:
*******************************************************************************/
tVoid fc_phonebook_tclClientHandler_BluetoothSetings::vHandle_VehicleBtAddress_Status(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4((" MethodResult: vHandle_VehicleBtAddress_Status  entered "));

   char ucVehicleBtAddress[LENGTH_OF_DEVICE_ADDRESS];
   most_btsetfi_tclMsgVehicleBTAddressStatus oVehicleBtAddress;

   fi_tclVisitorMessage oVisitorMsg(poMessage);

   if (oVisitorMsg.s32GetData(oVehicleBtAddress, MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION)
         != OSAL_ERROR)
   {
      //FIXME: Lenth of the address needs to be checked
      memset(ucVehicleBtAddress, 0, LENGTH_OF_DEVICE_ADDRESS);
      //copying vehicle BT address
      vStringCopy(ucVehicleBtAddress, oVehicleBtAddress.sVehicleBTAddress.szValue, LENGTH_OF_DEVICE_ADDRESS);

      if ((m_poPhoneBookService != NULL))
      {
         ETG_TRACE_USR1((" Local BT Device address to be updated = %s",ucVehicleBtAddress));
         m_poPhoneBookService->vSetVehicleBtAddress(ucVehicleBtAddress);
      }
      else
      {

         ETG_TRACE_USR1(("m_poPhoneBookService instance is not available"));
      }
   }
   else
   {
      ETG_TRACE_FATAL((" Error - in s32GetData."));
   }
   oVehicleBtAddress.vDestroy();
}


/*******************************************************************************
 ** FUNCTION:   vIsDeviceHandleValid
*******************************************************************************/
/* DESCRIPTION: Check if device handle is valid.
 *
 *  PARAMETERS:
 *  IN:  u8DeviceHandle: Device handle to be checked
 *  RETURNVALUE: If device handle is valid or not.
*******************************************************************************/
tBool fc_phonebook_tclClientHandler_BluetoothSetings::vIsDeviceHandleValid(tU8 u8DeviceHandle)
{
   if (u8DeviceHandle <= MAXIMUM_DEVICE_HANDLE)
   {
      return true;
   }
   return false;
}


/*******************************************************************************
 ** FUNCTION:   vIsDeviceConnected
*******************************************************************************/
/* DESCRIPTION: Check if device handle is connected.
 *
 *  PARAMETERS:
 *  IN:   u8DeviceHandle: Device handle to be checked
 *  RETURNVALUE: If device handle is connected or not.
*******************************************************************************/
tBool fc_phonebook_tclClientHandler_BluetoothSetings::vIsDeviceConnected(tU8 u8DeviceHandle)
{
   ETG_TRACE_USR4((" vIsDeviceConnected() entered for device handle: %d", u8DeviceHandle));

   tBool bConnectionStatus = false;
   if (u8DeviceHandle == FC_PB_VEHICLE_PBOOK)
   {
      bConnectionStatus = true;
   }
   else
   {
      std::map<tU8, std::string>::iterator iter = ConnnectedDevicesMap.find(u8DeviceHandle); // 241829- Replace bpstl:: by std::
      if (iter != ConnnectedDevicesMap.end()) // Entry found in ConnnectedDevicesMap
      {
         bConnectionStatus = true;
      }
   }

   return bConnectionStatus;
}


/*******************************************************************************
 ** FUNCTION:   vSetDeviceConnectInProgress
*******************************************************************************/
/* DESCRIPTION: Set the DeviceConnectInProgress value in BT Settings.
 *
 *  PARAMETERS:
 *  IN:   bValue        : Value to be set
 *  RETURNVALUE:
*******************************************************************************/
tVoid fc_phonebook_tclClientHandler_BluetoothSetings::vSetDeviceConnectInProgress(tU8 u8DeviceHandle)
{
   ETG_TRACE_USR4((" vSetDeviceConnectInProgress with : %d", u8DeviceHandle));
   m_u8DeviceConnectInProgress = u8DeviceHandle;
}


/*******************************************************************************
 ** FUNCTION:   u8GetDeviceConnectInProgress
*******************************************************************************/
/* DESCRIPTION: Get the DeviceConnectInProgress value from BT Settings.
 *
 *  PARAMETERS:
 *  IN:
 *  RETURNVALUE: Value of DeviceConnectInProgress
*******************************************************************************/
tU8 fc_phonebook_tclClientHandler_BluetoothSetings::u8GetDeviceConnectInProgress(tVoid)
{
   ETG_TRACE_USR4((" u8GetDeviceConnectInProgress : %d", m_u8DeviceConnectInProgress));
   return m_u8DeviceConnectInProgress;
}


/*******************************************************************************
 ** FUNCTION:   vTrigger_ConnectBluetoothProfile_MethodStart(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This function triggers a ConnectBluetoothProfile function method start
 *
 *  PARAMETERS:
 *     IN: Device handle
 *  RETURNVALUE: None
 ******************************************************************************/
tVoid fc_phonebook_tclClientHandler_BluetoothSetings::vTrigger_ConnectBluetoothProfile_MethodStart(tU8 u8DeviceHandle)
{
   ETG_TRACE_USR3(("Function vTrigger_ConnectBluetoothProfile_MethodStart entered with device handle: %d", u8DeviceHandle));

   most_btsetfi_tclMsgConnectBluetoothProfileMethodStart
         ConnectBluetoothProfileMethodStart;
   ConnectBluetoothProfileMethodStart.u8DeviceHandle = u8DeviceHandle;
   ConnectBluetoothProfileMethodStart.oServiceType.bPBAP = TRUE; // Phonebook Access Profile is to be connected.

   fi_tclVisitorMessage
         oMsg(ConnectBluetoothProfileMethodStart, MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION);

   ConnectBluetoothProfileMethodStart.vDestroy();
   vInitServiceData(oMsg, // ServiceDataMsg
   (tU16) u8DeviceHandle, // CmdCounter (deviceHandle will be used in Error Result)
   MOST_BTSETFI_C_U16_CONNECTBLUETOOTHPROFILE, // Function ID
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART // Opcode
   );

   // Post message
   ail_tenCommunicationError enResult1 =
         _poMainAppl->enPostMessage(&oMsg, TRUE);

   if (enResult1 != AIL_EN_N_NO_ERROR)
   {
      ETG_TRACE_ERR((" Failed to post message for ConnectBluetoothProfile method start"));
   }
   else
   {
      ETG_TRACE_USR4((" ConnectBluetoothProfile method start message posted successfully"));
   }
}


/*******************************************************************************
 ** FUNCTION:   vHandle_ConnectBluetoothProfile_MethodResult
 *******************************************************************************/
/* DESCRIPTION: Handle ConnectBluetoothProfile method result.
 *
 *  PARAMETERS:
 *  IN:
 *  RETURNVALUE:
 *******************************************************************************/
tVoid fc_phonebook_tclClientHandler_BluetoothSetings::vHandle_ConnectBluetoothProfile_MethodResult(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4((" Function vHandle_ConnectBluetoothProfile_MethodResult entered"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);
   most_btsetfi_tclMsgConnectBluetoothProfileMethodResult
         oConnectBluetoothProfileMethodResult;

   if (oVisitorMsg.s32GetData(oConnectBluetoothProfileMethodResult, MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION)
         != OSAL_ERROR)
   {
      ETG_TRACE_USR1((" Device handle: %d", oConnectBluetoothProfileMethodResult.u8DeviceHandle));
      ETG_TRACE_USR1((" oProfileConnectionStatus.bPBAP: %d", oConnectBluetoothProfileMethodResult.oProfileConnectionStatus.bPBAP));

      if (TRUE
            == oConnectBluetoothProfileMethodResult.oProfileConnectionStatus.bPBAP)
      {
         ETG_TRACE_USR4((" Device PBAP connected"));
         //Note: Further DBus handling and download initiation will be triggered via Interfaces-Added call back.

#ifdef RETRY_PBAP_MAP_VIA_BM_APPL
         //Stop retry if active, on successful PBAP connection
         FC_Device_Details
               *deviceObject =
                     FC_Device_Details::deviceHandleToObjectInstance(oConnectBluetoothProfileMethodResult.u8DeviceHandle);
         if (deviceObject)
         {
            fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.PhoneBook_vStopRetry(FC_PhoneBook_RetryDetails::RETRY_PBAP_CONNECT_VIA_BM_APPL, deviceObject->getConnectionID());
         }
#endif
      }
      else
      {
         ETG_TRACE_USR4((" Device PBAP not connected"));

#ifdef RETRY_PBAP_MAP_VIA_BM_APPL
         vHandleRetryForPBAPConnect(oConnectBluetoothProfileMethodResult.u8DeviceHandle);
#endif
      }
   }
   else
   {
      ETG_TRACE_ERR(("Version mismatch"));
   }
   oConnectBluetoothProfileMethodResult.vDestroy();
}


/*******************************************************************************
 ** FUNCTION:   vHandle_ConnectBluetoothProfile_Error
 *******************************************************************************/
/* DESCRIPTION: Handle ConnectBluetoothProfile method error.
 *
 *  PARAMETERS:
 *  IN:
 *  RETURNVALUE:
 *******************************************************************************/
tVoid fc_phonebook_tclClientHandler_BluetoothSetings::vHandle_ConnectBluetoothProfile_Error(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3((" Function vHandle_ConnectBluetoothProfile_Error entered"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);
   most_btsetfi_tclMsgConnectBluetoothProfileError
         oConnectBluetoothProfileError;

   if (oVisitorMsg.s32GetData(oConnectBluetoothProfileError, MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION)
         != OSAL_ERROR)
   {
      ETG_TRACE_USR4((" Error Code received: %u", oConnectBluetoothProfileError.e8ErrorCode.enType));

#ifdef RETRY_PBAP_MAP_VIA_BM_APPL
      vHandleRetryForPBAPConnect((tU8)poMessage->u16GetCmdCounter());
#endif
   }
   else
   {
      ETG_TRACE_ERR((" s32GetData Error! "));
   }

   oConnectBluetoothProfileError.vDestroy();
}


/*******************************************************************************
 ** FUNCTION:   vTrigger_DisconnectBluetoothProfile_MethodStart(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This function triggers a DisconnectBluetoothProfile function method start
 *
 *  PARAMETERS:
 *     IN: Device handle
 *  RETURNVALUE: None
 ******************************************************************************/
tVoid fc_phonebook_tclClientHandler_BluetoothSetings::vTrigger_DisconnectBluetoothProfile_MethodStart(tU8 u8DeviceHandle)
{
   ETG_TRACE_USR3(("Function vTrigger_DisconnectBluetoothProfile_MethodStart entered with device handle: %d", u8DeviceHandle));

   most_btsetfi_tclMsgDisconnectBluetoothProfileMethodStart
         oDisconnectBluetoothProfileMethodStart;
   oDisconnectBluetoothProfileMethodStart.u8DeviceHandle = u8DeviceHandle;
   oDisconnectBluetoothProfileMethodStart.oServiceType.bPBAP = TRUE; // Phonebook Access Profile is to be connected.

   fi_tclVisitorMessage
         oMsg(oDisconnectBluetoothProfileMethodStart, MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION);

   oDisconnectBluetoothProfileMethodStart.vDestroy();

   vInitServiceData(oMsg, // ServiceDataMsg
   0, // CmdCounter
   MOST_BTSETFI_C_U16_DISCONNECTBLUETOOTHPROFILE, // Function ID
   AMT_C_U8_CCAMSG_OPCODE_METHODSTART // Opcode
   );

   // Post message
   ail_tenCommunicationError enResult1 =
         _poMainAppl->enPostMessage(&oMsg, TRUE);

   if (enResult1 != AIL_EN_N_NO_ERROR)
   {
      ETG_TRACE_ERR((" Failed to post message for DisconnectBluetoothProfile method start"));
   }
   else
   {
      ETG_TRACE_USR4((" DisconnectBluetoothProfile method start message posted successfully"));
   }
}


/*******************************************************************************
 ** FUNCTION:   vHandle_DisconnectBluetoothProfile_MethodResult
 *******************************************************************************/
/* DESCRIPTION: Handle DisconnectBluetoothProfile method result.
 *
 *  PARAMETERS:
 *  IN:
 *  RETURNVALUE:
 *******************************************************************************/
tVoid fc_phonebook_tclClientHandler_BluetoothSetings::vHandle_DisconnectBluetoothProfile_MethodResult(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("Function vHandle_DisconnectBluetoothProfile_MethodResult entered"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);
   most_btsetfi_tclMsgConnectBluetoothProfileMethodResult
         oDisconnectBluetoothProfileMethodResult;

   if (oVisitorMsg.s32GetData(oDisconnectBluetoothProfileMethodResult, MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION)
         != OSAL_ERROR)
   {
      ETG_TRACE_USR1((" Device handle: %d", oDisconnectBluetoothProfileMethodResult.u8DeviceHandle));
      ETG_TRACE_USR1((" oProfileDisconnectionStatus.bPBAP: %d", oDisconnectBluetoothProfileMethodResult.oProfileConnectionStatus.bPBAP));

      if (TRUE
            == oDisconnectBluetoothProfileMethodResult.oProfileConnectionStatus.bPBAP)
      {
         ETG_TRACE_USR1((" Device PBAP disconnected"));
         //Note: Further DBus handling and download abort will be triggered via Interfaces-Removed call back.
      }
   }
   else
   {
      ETG_TRACE_ERR(("Version mismatch"));
   }
   oDisconnectBluetoothProfileMethodResult.vDestroy();
}


/*******************************************************************************
 ** FUNCTION:   vHandle_DisconnectBluetoothProfile_Error
 *******************************************************************************/
/* DESCRIPTION: Handle DisconnectBluetoothProfile method error.
 *
 *  PARAMETERS:
 *  IN:
 *  RETURNVALUE:
 *******************************************************************************/
tVoid fc_phonebook_tclClientHandler_BluetoothSetings::vHandle_DisconnectBluetoothProfile_Error(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR3(("Function vHandle_DisconnectBluetoothProfile_Error entered"));

   fi_tclVisitorMessage oVisitorMsg(poMessage);
   most_btsetfi_tclMsgDisconnectBluetoothProfileError
         oDisconnectBluetoothProfileError;

   if (oVisitorMsg.s32GetData(oDisconnectBluetoothProfileError, MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION)
         != OSAL_ERROR)
   {
      ETG_TRACE_USR4((" Error Code received: %u", oDisconnectBluetoothProfileError.e8ErrorCode.enType));
   }
   else
   {
      ETG_TRACE_ERR((" s32GetData Error! "));
   }

   oDisconnectBluetoothProfileError.vDestroy();
}


/*******************************************************************************
 ** FUNCTION:   vCheckForNewlyConnectedDevicesOrRetries
 *******************************************************************************/
/* DESCRIPTION: Check for Newly Connected Devices or pending entire download retries if any. If yes trigger handling.
 *
 *  PARAMETERS:
 *  IN: None.
 *  RETURNVALUE: None.
 *******************************************************************************/
tVoid fc_phonebook_tclClientHandler_BluetoothSetings::vCheckForNewlyConnectedDevicesOrRetries()
{
   ETG_TRACE_USR4((" Function vCheckForNewlyConnectedDevicesOrRetries entered"));

   ETG_TRACE_COMP((" -PBDL.S- Check for newly connected or suspended download devices if any... "));

   //NCG3D-65982
   //Check if download is suspended for any of the devices
   for (tU8 u8Index = 0; u8Index < m_u8SuspendedDeviceDownloadList.size(); u8Index++)
   {
      FC_Device_Details *pDeviceObject = FC_Device_Details::deviceHandleToObjectInstance(m_u8SuspendedDeviceDownloadList[u8Index]);

      if ((pDeviceObject) && (TRUE
            == pDeviceObject->bIsDeviceDownloadSuspended()))
      {
         //Resume the download for suspended device by posting loopback messasge
         vPostResumeSuspendedDwnldLoopback(m_u8SuspendedDeviceDownloadList[u8Index]);
      }
   }
   //End fix NCG3D-65982

   std::map<tU8, std::string>::iterator iter = ConnnectedDevicesMap.begin(); // 241829- Replace bpstl:: by std::
   while (iter != ConnnectedDevicesMap.end())
   {
      FC_Device_Details *pDeviceObject = FC_Device_Details::deviceHandleToObjectInstance(iter->first);
      if (! pDeviceObject)
      {
         vSetDeviceConnectInProgress(iter->first);
         m_poPhoneBookService->vDeviceConnected(iter->first, iter->second.c_str());
         break;
      }
      iter++;
   }

   //Fix : CMG3G-12880
#ifdef HANDLE_PROBABLY_LOCKED_SCENARIO
   ETG_TRACE_COMP((" -PBDL.S- Check for pending entire download retries if any... "));
   if(INVALID_VALUE == m_poPhoneBookService->m_oPbk_DnlManager.bCheckIfAnyDownloadIsInProgress())
   {
      //Currently no device connection is in progress. Hence start retry for locked device or device with suspended retry, if any.
      FC_Device_Details *pDeviceObject = NULL;
      for(tU8 u8Index= 0; u8Index < NO_OF_CONNECTED_DEVICE; u8Index++)
      {
         pDeviceObject = FC_Device_Details::getDeviceObject((INDEX)u8Index);
         if ( pDeviceObject && (TRUE == pDeviceObject->bIsDeviceProbablyLocked()) )
         {
            ETG_TRACE_USR4((" Starting retry for Entire download sequence for device handle: %d ", pDeviceObject->getDeviceHandle()));
            m_poPhoneBookService->m_oPbk_DnlManager.PhoneBook_vStartRetry(FC_PhoneBook_RetryDetails::RETRY_DOWNLOAD_SEQUENCE, pDeviceObject->getConnectionID(), FC_PhoneBook_RetryDetails::RETRY_TYPE_CONSTANT);
         }
      }
   }
#endif
   //End fix : CMG3G-12800
   
}

   
   
/*******************************************************************************
 ** FUNCTION:   vHandleRetryForPBAPConnect(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   This function handles the retry for PBAP connection
 *
 *  PARAMETERS:
 *     IN: u8DeviceHandle: Device handle for which retry sit o be done
 *  RETURNVALUE: None
 ******************************************************************************/
tVoid fc_phonebook_tclClientHandler_BluetoothSetings::vHandleRetryForPBAPConnect(tU8 u8DeviceHandle)
{
   ETG_TRACE_USR4((" Function vHandleRetryForPBAPConnect entered"));

   FC_Device_Details *deviceObject =
         FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);
   if (deviceObject)
   {
      tU8 u8ConnectionID = deviceObject->getConnectionID();
      if (fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.u8GetRetryCount()
            < NO_OF_RETRIES_EVO)
      {
         ETG_TRACE_USR4((" Starting retry for Device PBAP connection sequence. "));
         fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.PhoneBook_vStartRetry(FC_PhoneBook_RetryDetails::RETRY_PBAP_CONNECT_VIA_BM_APPL, u8ConnectionID, FC_PhoneBook_RetryDetails::RETRY_TYPE_CONSTANT);
      }
      else
      {
         ETG_TRACE_USR4((" Max No. of retries reached. Hence stopping retry. "));
         fc_phonebook_tclService_PhoneBook::pGetInstance()->m_oPbk_DnlManager.PhoneBook_vStopRetry(FC_PhoneBook_RetryDetails::RETRY_PBAP_CONNECT_VIA_BM_APPL, u8ConnectionID);

         //Updating feature support to ignore loading animation in SMART
         deviceObject->setDeviceCapabilities(PBAPNotSupported);
         (fc_phonebook_tclService_PhoneBook::pGetInstance())->vUpdatePhoneBookFeatureSupport(deviceObject);

         //477511 - [PSA] A pop up timeout  (in Connected Phone about  accessing contacts) not able to verify via putty
         (fc_phonebook_tclService_PhoneBook::pGetInstance())->pGetDownloadStateInstance()->vSetPBDownloadState(most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended::FI_EN_E8PBDS_ERROR);
         (fc_phonebook_tclService_PhoneBook::pGetInstance())->vPhoneBook_UpdateDownloadState();
         //end of fix- 477511
         //NCG3D-68310
         //Check if any device session is to be created, after number of retries have been reached for current device PBAP session
         vSetDeviceConnectInProgress(0);
         vCheckForNewlyConnectedDevicesOrRetries();
         //End fix NCG3D-68310
      }
   }
}

//Fix NCG3D-65982
/*******************************************************************************
 ** FUNCTION:   vPostResumeSuspendedDwnldLoopback
*******************************************************************************/
/* DESCRIPTION:
 *
 *  PARAMETERS:
 *  IN:   u8DeviceHandle  :Loopback data of suspended device for which download has to be resumed
 *  RETURNVALUE: None.
*******************************************************************************/
tVoid fc_phonebook_tclClientHandler_BluetoothSetings::vPostResumeSuspendedDwnldLoopback(tU8 u8DeviceHandle)
{
   ETG_TRACE_USR4((" vPostResumeSuspendedDwnldLoopback  entered with device handle: %d", u8DeviceHandle));

   if (!m_poPhoneBookService)
   {
      ETG_TRACE_ERR((" Phonebook service instance is NULL! "));
      return;
   }

   FC_Device_Details *deviceObject =
         FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);
   if(deviceObject)
   {

      ETG_TRACE_USR4(("Setting DownloadState to most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_NOT_STARTED and most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_NOT_STARTED on device connection"));
      m_poPhoneBookService->m_poDownloadState->vSetPBDownloadState(most_fi_tcl_e8_PhonBkPhoneBookDownloadState::FI_EN_E8PBDS_NOT_STARTED);
      m_poPhoneBookService->m_poDownloadState->vSetRecentCallListDownloadState(most_fi_tcl_e8_PhonBkRecentCallListDownloadState::FI_EN_E8RCDS_NOT_STARTED);
      m_poPhoneBookService->m_poDownloadState->vSetDeviceHandle(u8DeviceHandle);
      //Update the Client
      m_poPhoneBookService->vPhoneBook_UpdateDownloadState();

      FC_PhoneBook_List* poFC_PhoneBook_List =
            FC_PhoneBook_List::pGetInstance();

      if(poFC_PhoneBook_List)
      {
         poFC_PhoneBook_List->PhoneBookDownloadInitiated(deviceObject);
      }

      PhBk_Loopback_Data Resume_Download_LB_Data;

      memset((void*) &Resume_Download_LB_Data, 0, sizeof(PhBk_Loopback_Data));

      Resume_Download_LB_Data.u16FunctionID = (guint16)FC_PB_LB_FID_RESUME_SUSPENDED_DWNLD;
      Resume_Download_LB_Data.ucDeviceHandle = u8DeviceHandle;

      ETG_TRACE_USR4(("Posting loopback message for FID - FC_PB_LB_FID_RESUME_SUSPENDED_DWNLD"));

      m_poPhoneBookService->vPush_Back_LB_Data(Resume_Download_LB_Data);
      m_poPhoneBookService->vPrepareandsendLoopBackMsg(FC_PB_LB_FID_RESUME_SUSPENDED_DWNLD);
   }
   else
   {
      ETG_TRACE_ERR((" Device object does not exist "));
   }
}
//End fix NCG3D-65982
