/*******************************************************************************
*
* FILE:          FC_PhoneBook_FavoriteProvider.cpp
*
* SW-COMPONENT:  FC_PhoneBook application
*
* PROJECT:       Bosch
*
* DESCRIPTION:   Favorite Provider PhoneBook.
*
* AUTHOR:        Abhishek Kumar Dhiman
*
* COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
*
*******************************************************************************/
/*******************************************************************************
*                              INCLUDES
*******************************************************************************/
#include "FC_PhoneBook_Debug.h"
#include "FC_PhoneBook_FavoritesProvider.h"
#include "Database/FC_PhoneBook_SQLite.h"
#include "FC_PhoneBook_Device_Details.h"
#include "FC_PhoneBook_service_PhoneBook.h"
#include "FC_PhoneBook_clienthandler_Phone.h"
#include "FC_PhoneBook_clienthandler_BluetoothSetings.h"
#define UTFUTIL_S_IMPORT_INTERFACE_GENERIC
#include "utf_if.h"


#define ET_TRACE_INFO_ON
#include "etrace_if.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONEBOOK_SERVICE
#include "trcGenProj/Header/FC_PhoneBook_FavoritesProvider.cpp.trc.h"
#endif

/*******************************************************************************
*                                     DEFINES
*******************************************************************************/

/*******************************************************************************/
#define INVALID_FAV_TYPE (most_fi_tcl_e8_FavPrvFavoriteType::FI_EN_E8FT_BLANK)

#define FC_PB_MIN_FAV_ID   0x0100     //  As FBlock PhoneBook has no Default Favorites, the valid range for
//  Custom Favorites FAV_IDs is 0x0100 to 0xFFFF, see GIS-370-200.

#define FC_PB_MAX_FAV_ID   0xFFF0     //  We restrict the maximum used FAV_ID slightly from 0xFFFF to 0xFFF0 
//  to avoid tU16 wrap around situation for "++" operation.


#define FC_PB_GET_TAG_VALUE(objectID)      (quint8)(objectID >> 24)
#define FC_PB_GET_CONTACTHANDLE(objectID)  (objectID & (quint32)0x00FFFFFF)
#define PBOOK_GETFAVINFO_STRINGLENGTH 40
#define PBOOK_GETFAVINFO_MAX_LEN_BYTE 121
#define UTFUTIL_S_IMPORT_INTERFACE_GENERIC

FC_Phonebook_tclFavorite_Provider* FC_Phonebook_tclFavorite_Provider::m_self= NULL;

/******************************************************************************/
/*                        CCA MESSAGE MAP                                     */
/******************************************************************************/

/******************************************************************************/
/*                             START CODE                                     */

/******************************************************************************/
/*                                                                            */
/*                              METHODS                                       */
/*                                                                            */
/******************************************************************************/

/*******************************************************************************
*
* FUNCTION: fc_phonebook_Favorite_Provider::fc_phonebook_Favorite_Provider()
*
* DESCRIPTION: Constructor.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
FC_Phonebook_tclFavorite_Provider::FC_Phonebook_tclFavorite_Provider()
{

   ETG_TRACE_USR4(("fc_phonebook_Favorite_Provider() entered."));

   m_self= this;
   m_favSqlite= NULLPTR;
   m_poservice_phonebook= fc_phonebook_tclService_PhoneBook::pGetInstance();
   m_poPhoneClientHandler= NULLPTR;//CMG3G-14313-Coverity Fix
   m_favSqlite = FC_PhoneBook_SQLite::GetSQLiteInstance();
   if(!m_favSqlite)
   {

      ETG_TRACE_ERR(("Failed to get SQlite instance for favorites "));
      // TODO: Its Fatal error condition. M_FATAL_ASSERT_ALWAYS??
   }
}

/*******************************************************************************
*
* FUNCTION: fc_phonebook_Favorite_Provider::~fc_phonebook_Favorite_Provider(tVoid)
*
* DESCRIPTION: Destructor.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
FC_Phonebook_tclFavorite_Provider::~FC_Phonebook_tclFavorite_Provider(tVoid)
{

   ETG_TRACE_USR4(("~fc_phonebook_Favorite_Provider() entered."));

   m_poservice_phonebook = NULLPTR;
   m_favSqlite = NULLPTR;
   m_poPhoneClientHandler = NULLPTR;
}

/*******************************************************************************
** FUNCTION:   m_poPhoneBookFavoriteProvider::Store_Favorite()
*
*  DESCRIPTION: store favorite method
*
*  PARAMETERS:
*
*  RETURNVALUE: true on success
********************************************************************************/
bool FC_Phonebook_tclFavorite_Provider:: Store_Favorite(const most_phonbkfi_tclMsgStoreFavoriteMethodStart* methodStart,
                                                        most_phonbkfi_tclMsgStoreFavoriteMethodResult *methodResult, quint16& u16ErrorCode, tU8& u8ErrorInfo)
{


   ETG_TRACE_USR4(("Store_Favorite entered."));

   //TODO:- Error codes to be checked again.

   bool blRet = true;
   FC_PhoneBook_Fav_Detail fav;
   most_fi_tcl_e8_FavPrvFavoriteType::tenType favType;
   quint32 u32ObjectID= methodStart->u32ObjectID;
   quint16 u16FavoriteID;


   ETG_TRACE_USR2(("Object id received- %d", u32ObjectID));

   favType = getFavoriteType(FC_PB_GET_TAG_VALUE(u32ObjectID));

   if(favType != INVALID_FAV_TYPE)
   {
      if(generateUniqueFavoritesID(u16FavoriteID))
      {
         if(fetchFavoriteDetailsFromContact(u32ObjectID,fav))
         {
            fav.favoriteID= u16FavoriteID;
            if(!m_favSqlite->InsertFavRecord(fav ))
            {

               ETG_TRACE_ERR((" Adding of favorite to database failed "));
               blRet = false;
            }
            else
            {
               methodResult->u16FavoriteID= u16FavoriteID;
               m_UsedFavoriteIDList<<u16FavoriteID;
            }
         }
         else
         {

            ETG_TRACE_ERR(("Getting details of favorite failed. ContactHandle- %d", fav.contacthandle));
            blRet = false;
         }
      }
      else
      {
         ETG_TRACE_ERR(("Failed to generate unique favorite id"));
         blRet= false;
      }
      if(false == blRet)
      {
         u16ErrorCode= most_fi_tcl_e8_ErrorCode::FI_EN_PARAMETERNOTAVAILABLE;
         u8ErrorInfo=PARAM_ERR_POSITION_2;
      }
   }
   else
   {

      ETG_TRACE_ERR(("Invalid favorite type passed"));
      u16ErrorCode= most_fi_tcl_e8_ErrorCode::FI_EN_PARAMETERWRONG_OUTOFRANGE;   //Fix GMMY16-13959
      u8ErrorInfo=PARAM_ERR_POSITION_2;   //Fix GMMY16-13959
      blRet= false;
   }

   return blRet;
}


bool FC_Phonebook_tclFavorite_Provider:: InActivateFavoriteAfterDTMF(tU16 u16FavoriteID)
{
   if(    (m_favSqlite != NULL) 
      && (m_favSqlite->UpdateFavoritesActiveStatus(u16FavoriteID, 0x00)) )  // -bn: formatting improved and 0x01 replaced by 0x00
   {
      ETG_TRACE_USR2(("Active status set for favorite id- %d",
         u16FavoriteID));

      quint32 u32ListChangeType=
         (quint32)most_fi_tcl_e8_FavPrvListChangeType::FI_EN_E8LCH_ITEMS_CHANGED;
      quint32 u32ListSize= m_favSqlite->GetFavoritesCount();
      quint32 u32NumOfItems= 1;
      QList<quint16> FavoritesIDList;
      FavoritesIDList<<u16FavoriteID;

      sendFavoritesChangedPropertyUpdate(u32ListChangeType, u32ListSize,
         u32NumOfItems, FavoritesIDList);
      return true;
   }
   else
   {
      ETG_TRACE_ERR(("Failed to set active status for favorite id- %d",
         u16FavoriteID));
      return false;
   }
}

/*******************************************************************************
** FUNCTION:   m_poPhoneBookFavoriteProvider::Activate_Favorite()
*
*  DESCRIPTION: active favorite method
*
*  PARAMETERS:
*
*  RETURNVALUE: true on success
********************************************************************************/
bool FC_Phonebook_tclFavorite_Provider::Activate_Favorite(const most_phonbkfi_tclMsgActivateFavoriteMethodStart* methodStart)
{

   ETG_TRACE_USR2((" Activate_Favorite entered and Favorite_id received - %d", methodStart->u16FavoriteID));
   bool result = false;
   FC_PhoneBook_Fav_Detail favoriteData;
   most_fi_tcl_e8_FavPrvFavoriteType::tenType favType;
   quint16 u16ErrorCode = CCA_C_U16_ERROR_UNSPECIFIED;

   tU16 u16FavoriteID = methodStart->u16FavoriteID;
   tU16 u16CallInstance = m_CallInstanceFavIDMap.key(u16FavoriteID);   // Fix for GMMY16-9895
   ETG_TRACE_USR4((" Activate_Favorite entered and Favorite_id received for call instance: %d", u16CallInstance));   // Fix for GMMY16-9895

   if ((m_favSqlite != NULL)
      && (m_favSqlite->GetFavRecord(u16FavoriteID, favoriteData, u16ErrorCode)))
   {

      if (!fc_phonebook_tclClientHandler_BluetoothSetings::vIsDeviceConnected(favoriteData.devicehandle))
      {
         RefreshFavoritesOnDeviceDisconnection(favoriteData.devicehandle);
         m_poservice_phonebook->vPostActivateFavoriteError(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, u16FavoriteID, 2);   //Fix GMMY16-13959
         return result;
      }

      favType = getFavoriteType(favoriteData.tagvalue);

      if (favType
         != (most_fi_tcl_e8_FavPrvFavoriteType::tenType) INVALID_FAV_TYPE)
      {
         switch (favType)
         {
            case most_fi_tcl_e8_FavPrvFavoriteType::FI_EN_E8FT_CONTACT_NAME:
            {
               ETG_TRACE_USR4(("Favorite type is : FI_EN_E8FT_CONTACT_NAME"));
               m_poservice_phonebook->vPostActivateFavoriteMethodResult(u16FavoriteID, favoriteData.contacthandle);
               result = true;
            }
            break;

            case most_fi_tcl_e8_FavPrvFavoriteType::FI_EN_E8FT_PHONE_NUMBER:
            {
               ETG_TRACE_USR4(("Favorite type is : FI_EN_E8FT_PHONE_NUMBER. Active Call Count: %u", m_poPhoneClientHandler->m_u8ActivecallCount));

               // Fix for GMMY16-9895 Pressing the favorite button that was used to place an outgoing call does not end the call
               if ((TRUE == favoriteData.activestatus)
                     && (most_fi_tcl_e8_TelCallStatus::FI_EN_E8DIALING
                           == m_CallInstanceStateMap[u16CallInstance])) // If the Favorite is previously active and the call status is DIALING, trigger Cancel Outgoing Call
               {
                  ETG_TRACE_USR4(("Calling vTrigger_CancelOutgoingCall_MethodStart for Call Instance: %u", u16CallInstance));
                  m_ActivatedFavoriteIDList << u16FavoriteID;
                  m_poPhoneClientHandler->vTrigger_CancelOutgoingCall_MethodStart(u16CallInstance);
               }
               else if ((TRUE == favoriteData.activestatus)
                     && (most_fi_tcl_e8_TelCallStatus::FI_EN_E8RINGTONE
                           == m_CallInstanceStateMap[u16CallInstance])) // If the Favorite is previously active and the call status is RINGTONE, trigger Ignore Call
               {
                  ETG_TRACE_USR4(("Calling vTrigger_IgnoreCall_MethodStart for Call Instance: %u", u16CallInstance));
                  m_ActivatedFavoriteIDList << u16FavoriteID;
                  m_poPhoneClientHandler->vTrigger_IgnoreCall_MethodStart(u16CallInstance);
               }
               // End of Fix for GMMY16-9895
               else
               {
                  if (m_poPhoneClientHandler->m_u8ActivecallCount >= 1) // Fix for GMMY15-10219: More than one call Active
                  {
                     // Fix for GMMY16-9895 Pressing the favorite button that was used to place an outgoing call does not end the call
                     // When the call count is >=1 and if the favorite activestatus is true, trigger Hangup MethodStart
                     if (TRUE == favoriteData.activestatus)
                     {
                        if(most_fi_tcl_e8_TelCallStatus::FI_EN_E8ACTIVE == m_CallInstanceStateMap[u16CallInstance] ||
                           most_fi_tcl_e8_TelCallStatus::FI_EN_E8ON_HOLD == m_CallInstanceStateMap[u16CallInstance])
                        {
                           ETG_TRACE_USR4(("Calling vTrigger_Hangup_MethodStart for Call Instance: %u", u16CallInstance));
                           m_ActivatedFavoriteIDList << u16FavoriteID;
                           m_poPhoneClientHandler->vTrigger_Hangup_MethodStart(u16CallInstance);
                        }
                     }
                     // End of Fix GMMY16-9895
                     else
                     {
                        ETG_TRACE_USR4(("Calling vTrigger_SendDTMFMethodStart for Call Instance: %u", u16CallInstance));
                        FavoriteIdqueue.enqueueCCAMessage(u16FavoriteID);

                        m_poPhoneClientHandler->vTrigger_SendDTMFMethodStart(u16CallInstance, favoriteData.phonenumber.toUtf8().data());

                        if ((m_favSqlite != NULL) && (m_favSqlite->UpdateFavoritesActiveStatus(u16FavoriteID, 0x01)))
                        {
                           ETG_TRACE_USR2(("Active status set for favorite id- %d", u16FavoriteID));

                           quint32 u32ListChangeType = (quint32) most_fi_tcl_e8_FavPrvListChangeType::FI_EN_E8LCH_ITEMS_CHANGED;
                           quint32 u32ListSize = m_favSqlite->GetFavoritesCount();
                           quint32 u32NumOfItems = 1;
                           QList < quint16 > FavoritesIDList;
                           FavoritesIDList << u16FavoriteID;

                           sendFavoritesChangedPropertyUpdate(u32ListChangeType, u32ListSize, u32NumOfItems, FavoritesIDList);
                        }
                        else
                        {
                           ETG_TRACE_ERR(("Failed to set active status for favorite id- %d", u16FavoriteID));
                        }
                        //Fix for GMMY15-10219:ActivateFavoriteMethodResult Posted
                        m_poservice_phonebook->vPostActivateFavoriteMethodResult(u16FavoriteID, (quint32) u16CallInstance);
                     }
                  }
                  else
                  {
                     ETG_TRACE_USR4(("Calling vTrigger_Dial_MethodStart"));
                     m_ActivatedFavoriteIDList << u16FavoriteID; // Fix for GMMY15-10219
                     m_poPhoneClientHandler->vTrigger_Dial_MethodStart(favoriteData.phonenumber.toUtf8().data()); // CLONE TICKET:GMMY15-1009 FIXPHONE_NUMBER Favorite with leading '+' in phone number is dialed by Phonebook without '+'
                  }
               }
               result = true;
            }
            break;

         default:
            ETG_TRACE_ERR(("Invalid favorite type"));
         }
      }
      else
      {
         ETG_TRACE_USR4((" Activate favorite  failed: INVALID FAVORITE TYPE"));
         m_poservice_phonebook->vPostActivateFavoriteError(most_fi_tcl_e8_ErrorCode::FI_EN_PARAMETERWRONG_OUTOFRANGE, u16FavoriteID, PARAM_ERR_POSITION_2);   //Fix GMMY16-13959
      }
   }
   else
   {
      ETG_TRACE_USR2((" Activate favorite  failed: Internal Error code -%d", u16ErrorCode));
      m_poservice_phonebook->vPostActivateFavoriteError(most_fi_tcl_e8_ErrorCode::FI_EN_PARAMETERNOTAVAILABLE, u16FavoriteID, PARAM_ERR_POSITION_2);   //Fix GMMY16-13959
   }
   return result;
}

/*******************************************************************************
** FUNCTION:   m_poPhoneBookFavoriteProvider::Delete_Favorite()
*
*  DESCRIPTION: delete favorite method
*
*  PARAMETERS:
*
*  RETURNVALUE: true on success
********************************************************************************/
bool FC_Phonebook_tclFavorite_Provider::Delete_Favorite(const most_phonbkfi_tclMsgDeleteFavoriteMethodStart* methodStart,
                                                        most_phonbkfi_tclMsgDeleteFavoriteMethodResult *methodResult , quint16& u16ErrorCode)
{
   bool result = false;

   ETG_TRACE_USR2((" Delete_Favorite entered andFavorite_id received - %d", methodStart->u16FavoriteID));

   if(( m_favSqlite != NULL) && (m_favSqlite->DeleteFavRecord( methodStart->u16FavoriteID )))
   {
      result = true;

      methodResult->u16FavoriteID = methodStart->u16FavoriteID;
      m_UsedFavoriteIDList.removeOne(methodStart->u16FavoriteID);


      ETG_TRACE_USR4((" Delete details fetched successfully "));
   }
   else
   {

      ETG_TRACE_ERR(("Delete favorite  failed. Internal Error"));
      u16ErrorCode= CCA_C_U16_ERROR_INTERNAL_FAILURE;
   }




   return result;
}

/*******************************************************************************
** FUNCTION:   m_poPhoneBookFavoriteProvider::Get_FavoriteInfo()
*
*  DESCRIPTION: Get favorite info method
*
*  PARAMETERS:
*
*  RETURNVALUE: true on success
********************************************************************************/
bool FC_Phonebook_tclFavorite_Provider::GetFavoriteInfo(
   const most_phonbkfi_tclMsgGetFavoriteInfoMethodStart* methodStart,
   most_phonbkfi_tclMsgGetFavoriteInfoMethodResult *methodResult,tU16& u16ErrorCode, tU8& u8ErrorInfo)   //Fix GMMY16-13959
{
   bool blRet = false;

   ETG_TRACE_USR2((" GetFavoriteInfo entered  andFavorite_id received - %d", methodStart->u16FavoriteID));

   QList<FC_PhoneBook_Fav_Detail> favList;

   QString DescriptionText, LabelText;
   quint16 u16IntErrCode = FC_PB_ERROR_CODE_UNKNOWN;

   if(( m_favSqlite != NULL) && ( m_favSqlite->GetFavInfoForFavorite( methodStart->u16FavoriteID, favList, u16IntErrCode)))
   {
      ETG_TRACE_USR2(( " Total No. of favorites = %d ",favList.size()));

      for ( int favIndex = 0; favIndex < favList.size(); favIndex++)
      {
         most_fi_tcl_FavPrvFavoriteInfoItem favoriteInfoItem;
         FC_PhoneBook_Fav_Detail fav;

         fav = favList.at( favIndex);

         favoriteInfoItem.bActive 						= fav.activestatus;
         if(!fc_phonebook_tclClientHandler_BluetoothSetings::vIsDeviceConnected(fav.devicehandle))
         {
            favoriteInfoItem.bAvailable						= 0;
         }
         else
         {
            favoriteInfoItem.bAvailable						= fav.availablestatus;
         }

         //Correction for deviceHandle 0 and 1
         if((m_poservice_phonebook->m_bServerAvailable) &&
            ( (fav.devicehandle == 0) || (fav.devicehandle == 1) ))
         {
            favoriteInfoItem.bAvailable                        = 1;

         }

         favoriteInfoItem.e8FavoriteType.enType			= getFavoriteType(fav.tagvalue);

         fillLabelAndDescriptionText(&fav, LabelText, DescriptionText);

         tChar acLabelText[PBOOK_GETFAVINFO_MAX_LEN_BYTE] = { 0 };
         tChar acDescriptionText[PBOOK_GETFAVINFO_MAX_LEN_BYTE] = { 0 };

         (void) UTF8_szSaveStringNCopy( (tString) acLabelText /*tString szDest*/,
            (tString) LabelText.toUtf8().data(),
            PBOOK_GETFAVINFO_MAX_LEN_BYTE /*tU32 u32BufferLength*/,
            PBOOK_GETFAVINFO_STRINGLENGTH /*tU32 u32CntChar*/ );
         favoriteInfoItem.sLabelText.bSet(acLabelText);

         (void) UTF8_szSaveStringNCopy( (tString) acDescriptionText /*tString szDest*/,
            (tString) DescriptionText.toUtf8().data(),
            PBOOK_GETFAVINFO_MAX_LEN_BYTE /*tU32 u32BufferLength*/,
            PBOOK_GETFAVINFO_STRINGLENGTH /*tU32 u32CntChar*/ );
         favoriteInfoItem.sDescriptorText.bSet(acDescriptionText);
         favoriteInfoItem.u16FavoriteID					= fav.favoriteID;

         methodResult->oFavoriteInfo.oItems.push_back(favoriteInfoItem);
         methodResult->u32NumItems++;
      }


      ETG_TRACE_USR2(( "Favorite details fetched successfully and GetFavoriteInfo NumItems = %u", methodResult->u32NumItems));
      blRet = true;
   }
   else
   {

      ETG_TRACE_ERR(("Fetching favorite detail failed. Internal ErrorCode- %d", u16IntErrCode));

      if (FC_PB_ERROR_DATABASE_NOT_OPEN == u16IntErrCode)
      {
         u16ErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE; //Fix GMMY16-13959
         u8ErrorInfo = NIL_ERROR_INFO; //Fix GMMY16-13959
      }
      else
      {
         u16ErrorCode = most_fi_tcl_e8_ErrorCode::FI_EN_PARAMETERNOTAVAILABLE; //Fix GMMY16-13959
         u8ErrorInfo = PARAM_ERR_POSITION_2; //Fix GMMY16-13959
	  }
   }

   return blRet;
}

/*******************************************************************************
** FUNCTION:   m_poPhoneBookFavoriteProvider::Reset_Favorites()
*
*  DESCRIPTION: Reset favorite method
*
*  PARAMETERS:
*
*  RETURNVALUE: true on success
********************************************************************************/
bool FC_Phonebook_tclFavorite_Provider::Reset_Favorites(bool fResetDeviceSpecificFav, tU8 fU8DeviceHandle)
{
   ETG_TRACE_USR4(( "Reset_Favorites entered"));

   m_UsedFavoriteIDList.clear();
   RefreshFavoritesOnClearPhoneBook(fResetDeviceSpecificFav,fU8DeviceHandle);

   return true;
}

/*******************************************************************************
** FUNCTION:   getFavoriteType()
*
*  DESCRIPTION:

*  PARAMETERS:
*
*  RETURNVALUE: true on success
********************************************************************************/
most_fi_tcl_e8_FavPrvFavoriteType::tenType FC_Phonebook_tclFavorite_Provider::
getFavoriteType(quint8 favType)
{

   ETG_TRACE_USR4(( "getFavoriteType entered"));
   most_fi_tcl_e8_FavPrvFavoriteType::tenType returnVal;

   switch((FC_Phonebook_tclFavorite_Provider::favoriteTag)favType)
   {
   case PrefNumTag:
   case CellNum1Tag:
   case CellNum2Tag:
   case HomeNum1Tag:
   case HomeNum2Tag:
   case WorkNum1Tag:
   case WorkNum2Tag:
   case OtherNumTag:
      returnVal =  most_fi_tcl_e8_FavPrvFavoriteType::FI_EN_E8FT_PHONE_NUMBER;
      break;
   case ContactNameTag:
      returnVal =  most_fi_tcl_e8_FavPrvFavoriteType::FI_EN_E8FT_CONTACT_NAME;
      break;
   default :
      returnVal = (most_fi_tcl_e8_FavPrvFavoriteType::tenType)INVALID_FAV_TYPE;

   }
   return returnVal;
}
/*******************************************************************************
** FUNCTION:   fillLabelDescriptionText()
*
*  DESCRIPTION:

*  PARAMETERS:
*
*  RETURNVALUE: true on success
********************************************************************************/
void FC_Phonebook_tclFavorite_Provider::fillLabelAndDescriptionText(
   const FC_PhoneBook_Fav_Detail* FavDetails, QString& LabelText, QString& DescriptionText)
{
   ETG_TRACE_USR4(( "fillLabelAndDescriptionText entered"));
   quint8 favType = FavDetails->tagvalue;
   DescriptionText.clear();
   LabelText = FavDetails->labeltext;

   switch(favType)
   {
   case FC_Phonebook_tclFavorite_Provider :: PrefNumTag:
   case FC_Phonebook_tclFavorite_Provider :: CellNum1Tag:
   case FC_Phonebook_tclFavorite_Provider :: CellNum2Tag:
   case FC_Phonebook_tclFavorite_Provider :: HomeNum1Tag:
   case FC_Phonebook_tclFavorite_Provider :: HomeNum2Tag:
   case FC_Phonebook_tclFavorite_Provider :: WorkNum1Tag:
   case FC_Phonebook_tclFavorite_Provider :: WorkNum2Tag:
   case FC_Phonebook_tclFavorite_Provider :: OtherNumTag:
      DescriptionText = "Call " + FavDetails->phonenumber;
      break;

   case FC_Phonebook_tclFavorite_Provider :: ContactNameTag:
      DescriptionText = FavDetails->labeltext;
      break;

   default :

      ETG_TRACE_ERR(("Invalid tag value found and fillLabelAndDescriptionText: Invalid tag value found"));
   }
}

/*******************************************************************************
** FUNCTION:   generateUniqueFavoritesID()
*
*  DESCRIPTION: Generates a unique favorites ID

*  PARAMETERS:
*
*  RETURNVALUE: true on success
********************************************************************************/
bool FC_Phonebook_tclFavorite_Provider::generateUniqueFavoritesID(quint16& u16FavID)
{
   ETG_TRACE_USR4(( "generateUniqueFavoritesID entered"));
   bool blRet= false;
   quint16 u16GenFavID;

   u16FavID= OSAL_NULL;
   if(m_UsedFavoriteIDList.size() > OSAL_NULL)
   {
      QList<quint16>::const_iterator itr=
         m_UsedFavoriteIDList.constEnd();
      u16GenFavID= *(--itr);


      ETG_TRACE_USR2(("Last inserted favorite ID fetched- %d", u16GenFavID));

      u16GenFavID++;

      if( (u16GenFavID >= FC_PB_MIN_FAV_ID) && (u16GenFavID <= FC_PB_MAX_FAV_ID) )
      {
         if(!m_UsedFavoriteIDList.contains(u16GenFavID))
         {
            blRet= true;
         }
         else
         {

            ETG_TRACE_USR2(("Favorite id - %d already exists", u16GenFavID));
         }
      }

      if(!blRet)
      {
         ETG_TRACE_USR4(("Scanning for a unique favorite id"));


         for(u16GenFavID = FC_PB_MIN_FAV_ID; u16GenFavID <= FC_PB_MAX_FAV_ID; u16GenFavID++)
         {
            if(!m_UsedFavoriteIDList.contains(u16GenFavID))
            {
               blRet= true;
               break;
            }
         }
      }
   }
   else
   {
      ETG_TRACE_USR4(("Favorite id list is empty"));


      u16GenFavID = FC_PB_MIN_FAV_ID;
      blRet= true;
   }

   if(blRet)
   {
      u16FavID= u16GenFavID;
      ETG_TRACE_USR4(("Favorites ID successfully generated- %d", u16FavID));

   }
   else
   {

      ETG_TRACE_ERR(("Failed to generate unique favorite id"));
   }

   return blRet;
}

/*******************************************************************************
** FUNCTION:   fetchFavoriteDetailsFromContact
*
*  DESCRIPTION: populates the favorite details to be stored from the database using
*               contact handle

*  PARAMETERS:
*
*  RETURNVALUE: true on success
********************************************************************************/
bool FC_Phonebook_tclFavorite_Provider::fetchFavoriteDetailsFromContact(quint32 u32ObjectId,
                                                                        FC_PhoneBook_Fav_Detail& FavDetails)
{

   ETG_TRACE_USR4(( "fetchFavoriteDetailsFromContactentered"));
   FC_PhoneBook_Contact_Detail contact;

   FavDetails.contacthandle = FC_PB_GET_CONTACTHANDLE(u32ObjectId);
   FavDetails.tagvalue = FC_PB_GET_TAG_VALUE(u32ObjectId);

   if((m_favSqlite != NULL) && ( true == m_favSqlite->GetContactDetails(FavDetails.contacthandle, contact)))
   {

      ETG_TRACE_USR4(("fetchFavoriteDetailsFromContact: m_sqlite.GetContactDetais Success  "));
   }
   else
   {

      ETG_TRACE_ERR(("fetchFavoriteDetailsFromContact: m_sqlite.GetContactDetails Failed "));
      return false;
   }

   FavDetails.availablestatus= isFavoriteAvailable(FavDetails.contacthandle,
      FavDetails.devicehandle);
   if(1 == m_favSqlite->GetPreferredSortOrder())   // GMMY17-1233 LabelText is constructed based on the current sort order
   {
      FavDetails.labeltext = contact.lastName +" "+ contact.firstName;
   }
   else
   {
      FavDetails.labeltext = contact.firstName +" "+ contact.lastName;
   }
   FavDetails.activestatus= OSAL_NULL;

   retreivePhoneNumberFromContact(contact, FavDetails.tagvalue, FavDetails.phonenumber);

   return true;
}

/*******************************************************************************
** FUNCTION:   isFavoriteAvailable
*
*  DESCRIPTION: Verifies whether the given contact handle is available

*  PARAMETERS: IN- Contact handle
*              OUT- Device Handle
*
*  RETURNVALUE: available or not
********************************************************************************/
bool FC_Phonebook_tclFavorite_Provider::isFavoriteAvailable(quint32 u32ContactHandle,
                                                            quint8& u8DeviceHandle)
{
   ETG_TRACE_USR2(("isFavoriteAvailable entered with ContactHandle %d",u32ContactHandle));
   ETG_TRACE_USR2(("DeviceHandle %d",u8DeviceHandle));
   bool blRet= false;

   if (m_favSqlite != NULL)
   {
      u8DeviceHandle = m_favSqlite->GetDeviceHandleForContacthandle(u32ContactHandle);

      if(OSAL_NULL != u8DeviceHandle)
      {
         if(FC_PB_VEHICLE_PBOOK == u8DeviceHandle)
         {
            blRet= true;
         }
         else
         {
            FC_Device_Details* DevDetails=
               FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);
            if(DevDetails)
            {
               blRet= true;
            }
         }
      }
   }
   return blRet;
}

/*******************************************************************************
** FUNCTION:   ResetAllFavoritesAsUnavailable
*
*  DESCRIPTION: This function clears the available status of all favorite entries
*  except for those related to vehicle phonebook. The active status of all entries
*  are cleared
*
*  PARAMETERS: IN-
*              OUT-
*
*  RETURNVALUE:
********************************************************************************/
void FC_Phonebook_tclFavorite_Provider::ResetAllFavoritesAsUnavailable()
{

   ETG_TRACE_USR4(("Function ResetAllFavoritesAsUnavailable entered"));

   if(m_favSqlite != NULL)
   {
      m_favSqlite->MakeAllFavoritesUnavailable();
      m_favSqlite->MakeAllFavoritesInactive();
   }
   else
   {

      ETG_TRACE_FATAL(("Database pointer is NULL"));
      //todo: fatal error
   }


}

/*******************************************************************************
** FUNCTION:   TriggerFavoritesChangeOnStore
*
*  DESCRIPTION: This function triggers favorites change property update after
*  storage of new favorite
*
*  PARAMETERS: IN- favorite id of the stored favorite
*              OUT-
*
*  RETURNVALUE:
********************************************************************************/
void FC_Phonebook_tclFavorite_Provider::triggerFavoritesChangeOnStore(quint16 u16FavoriteID)
{

   ETG_TRACE_USR4(("Function TriggerFavoritesChangeOnStore entered with FavoriteID :%d ",u16FavoriteID));

   if (m_favSqlite != NULL)
   {
      quint32 u32ListChangeType=
         (quint32)most_fi_tcl_e8_FavPrvListChangeType::FI_EN_E8LCH_ITEMS_ADDED;
      quint32 u32ListSize= m_favSqlite->GetFavoritesCount();
      quint32 u32NumOfItems= 1;
      QList<quint16> FavoritesIDList;
      FavoritesIDList<<u16FavoriteID;

      sendFavoritesChangedPropertyUpdate(u32ListChangeType, u32ListSize,
         u32NumOfItems, FavoritesIDList);
   }



}

void FC_Phonebook_tclFavorite_Provider::MakeFavoritesAvailable(quint8 DeviceHandle)
{
   ETG_TRACE_USR4(("Function MakeFavoritesAvailable entered with DeviceHandle : %d",DeviceHandle));
   if (m_favSqlite != NULL)
   {
      m_favSqlite->MakeFavoritesAvailable(DeviceHandle);
   }
}
void FC_Phonebook_tclFavorite_Provider::MakeFavoritesUnAvailable(quint8 DeviceHandle)
{
   ETG_TRACE_USR4(("Function MakeFavoritesUnAvailable entered with DeviceHandle : %d",DeviceHandle));
   if (m_favSqlite != NULL)
   {
      m_favSqlite->MakeFavoritesUnAvailable(DeviceHandle);
   }
}
//void FC_Phonebook_tclFavorite_Provider::RemoveFavorite(quint32 ContactHandle)
//{
//    if (m_favSqlite != NULL)
//    {
//        m_favSqlite->RemoveFavorite(ContactHandle);
//    }
//}

void FC_Phonebook_tclFavorite_Provider::triggerFavoritesChange(tU8 DeviceHandle)
{

   ETG_TRACE_USR4(("Function TriggerFavoritesChange entered"));

   if (m_favSqlite != NULL)
   {
      quint32 u32ListChangeType=
         (quint32)most_fi_tcl_e8_FavPrvListChangeType::FI_EN_E8LCH_ITEMS_CHANGED;


      quint32 u32ListSize= m_favSqlite->GetFavoritesCount();
      quint32 u32NumOfItems= m_favSqlite->GetFavoritesCount(DeviceHandle);
      QList<quint16> FavoritesIDList;
      m_favSqlite->GetAllFavoritesIDForDevice(FavoritesIDList, DeviceHandle);

      sendFavoritesChangedPropertyUpdate(u32ListChangeType, u32ListSize,
         u32NumOfItems, FavoritesIDList);
   }



}

void FC_Phonebook_tclFavorite_Provider::triggerFavoritesChange(tU32 ContactHandle)
{

   ETG_TRACE_USR4(("Function TriggerFavoritesChange entered with ContactHandle :%d ",ContactHandle));

   if (m_favSqlite != NULL)
   {
      quint32 u32ListChangeType=
         (quint32)most_fi_tcl_e8_FavPrvListChangeType::FI_EN_E8LCH_ITEMS_CHANGED;


      quint32 u32ListSize= m_favSqlite->GetFavoritesCount();
      quint32 u32NumOfItems= 1;
      QList<quint16> FavoritesIDList;
      m_favSqlite->GetFavoritesIDForContact(FavoritesIDList, ContactHandle);

      sendFavoritesChangedPropertyUpdate(u32ListChangeType, u32ListSize,
         u32NumOfItems, FavoritesIDList);
   }



}


/*******************************************************************************
** FUNCTION:   TriggerFavoritesChangeOnDelete
*
*  DESCRIPTION: This function triggers favorites change property update after
*  storage of deletion of favorite
*
*  PARAMETERS: IN- favorite id of the stored favorite
*              OUT-
*
*  RETURNVALUE:
********************************************************************************/
void FC_Phonebook_tclFavorite_Provider::triggerFavoritesChangeOnDelete(quint16 u16FavoriteID)
{

   ETG_TRACE_USR4(("Function TriggerFavoritesChangeOnDelete entered with FavoriteID : %d ",u16FavoriteID));

   if( m_favSqlite != NULL)
   {
      quint32 u32ListChangeType=
         (quint32)most_fi_tcl_e8_FavPrvListChangeType::FI_EN_E8LCH_ITEMS_REMOVED;
      quint32 u32ListSize= m_favSqlite->GetFavoritesCount();
      quint32 u32NumOfItems= 1;
      QList<quint16> FavoritesIDList;
      FavoritesIDList<<u16FavoriteID;

      sendFavoritesChangedPropertyUpdate(u32ListChangeType, u32ListSize,
         u32NumOfItems, FavoritesIDList);

   }


}

/*******************************************************************************
** FUNCTION:   MethodResultPosted
*
*  DESCRIPTION: This function is called after posting messages for storefavorite,
*  deletefavorite, resetfavorite
*
*  PARAMETERS: IN-
*              OUT-
*
*  RETURNVALUE:
********************************************************************************/
void FC_Phonebook_tclFavorite_Provider::MethodResultPosted(const quint16 u16FuntionID,
                                                           const most_phonbkfi_tclMsgBaseMessage* MethodStart,
                                                           const most_phonbkfi_tclMsgBaseMessage* MethodResult)
{

   ETG_TRACE_USR4(("FC_Phonebook_tclFavorite_Provider::MethodResultPosted entered"));

   switch(u16FuntionID)
   {
   case MOST_PHONBKFI_C_U16_STOREFAVORITE:
      {
         const most_phonbkfi_tclMsgStoreFavoriteMethodResult *MethodResultSpecific
            = static_cast<const most_phonbkfi_tclMsgStoreFavoriteMethodResult*>(MethodResult);

         triggerFavoritesChangeOnStore(MethodResultSpecific->u16FavoriteID);
      }
      break;
   case MOST_PHONBKFI_C_U16_DELETEFAVORITE:
      {
         const most_phonbkfi_tclMsgDeleteFavoriteMethodStart *MethodStartSpecific
            = static_cast<const most_phonbkfi_tclMsgDeleteFavoriteMethodStart*>(MethodStart);

         triggerFavoritesChangeOnDelete(MethodStartSpecific->u16FavoriteID);
      }
      break;
   default:
      {
         ETG_TRACE_USR4(("Not Valid u16FuntionID "));
      }
   }
}

bool FC_Phonebook_tclFavorite_Provider::RefreshFavoritesOnClearPhoneBook(bool fResetDeviceSpecificFav, tU8 fU8DeviceHandle)
{
   ETG_TRACE_USR4(("Function RefreshFavoritesOnClearPhoneBook entered"));

   bool bretval = FALSE;

   if (m_favSqlite != NULL)
   {
      quint32 u32ListChangeType=
         (quint32)most_fi_tcl_e8_FavPrvListChangeType::FI_EN_E8LCH_CONTENT_CHANGED;

      //Get list size
      quint32 u32ListSize = 0;
      if(fResetDeviceSpecificFav == false)
      {
         //Get count of all favorites
         u32ListSize = m_favSqlite->GetFavoritesCount();
      }
      else
      {  // Get the device specific count of favorites
         u32ListSize = m_favSqlite->GetFavoritesCount(fU8DeviceHandle);
      }

      //Send favorite changed property update
      QList<quint16> FavoritesIDList;
      quint32 u32NumOfItems= 0;
      m_favSqlite->GetAllFavoritesIDForDevice(FavoritesIDList, fU8DeviceHandle);
      sendFavoritesChangedPropertyUpdate(u32ListChangeType, u32ListSize,
         u32NumOfItems, FavoritesIDList);

      //Delete favorites from DB
      bretval = m_favSqlite->DeleteFavRecord(DELETE_ALL_FAVORITE,fResetDeviceSpecificFav,fU8DeviceHandle);
   }

   return bretval;
}


void FC_Phonebook_tclFavorite_Provider::RefreshFavoritesOnClearVehiclePhoneBook()
{

   ETG_TRACE_USR4(("Function RefreshFavoritesOnClearVehiclePhoneBook entered"));

   if (m_favSqlite != NULL)
   {
      quint32 u32ListChangeType=
         (quint32)most_fi_tcl_e8_FavPrvListChangeType::FI_EN_E8LCH_ITEMS_REMOVED;


      quint32 u32ListSize= m_favSqlite->GetFavoritesCount();

      QList<quint16> FavoritesIDList;
      m_favSqlite->GetAllFavoritesIDForDevice(FavoritesIDList,1);
      quint32 u32NumOfItems = (quint32)FavoritesIDList.size();
      sendFavoritesChangedPropertyUpdate(u32ListChangeType, u32ListSize,
         u32NumOfItems, FavoritesIDList);
      quint16 favoriteid;
      foreach(favoriteid, FavoritesIDList)
      {
         (void)m_favSqlite->DeleteFavRecord(favoriteid);
      }

   }
}
/*******************************************************************************
** FUNCTION:   RefreshFavoritesOnDeleteContact
*
*  DESCRIPTION: This function is triggered after deletion of a contact in vehicle
*  phonebook to update any associated favorite. It also triggers favorite change
*  property in case of any update
*
*  PARAMETERS: IN- contact handle of the deleted contact
*              OUT-
*
*  RETURNVALUE:
********************************************************************************/
void FC_Phonebook_tclFavorite_Provider::RefreshFavoritesOnDeleteContact(quint32 u32ContactHandle)
{
   ETG_TRACE_USR4(("Function RefreshFavoritesOnDeleteContact entered with ContactHandle = %u",u32ContactHandle));

   if (m_favSqlite != NULL)
   {
      quint32 u32ListChangeType=
         (quint32)most_fi_tcl_e8_FavPrvListChangeType::FI_EN_E8LCH_ITEMS_REMOVED;

      quint32 u32ListSize= m_favSqlite->GetFavoritesCount();

      QList<quint16> FavoritesIDList;
      m_favSqlite->GetFavoritesIDForContact(FavoritesIDList, u32ContactHandle);
      quint32 u32NumOfItems= (quint32)FavoritesIDList.size();
      if(u32NumOfItems > 0)
      {
         sendFavoritesChangedPropertyUpdate(u32ListChangeType, u32ListSize,
            u32NumOfItems, FavoritesIDList);

         (void)m_favSqlite->RemoveFavorite(u32ContactHandle);
      }
   }
}

/*******************************************************************************
** FUNCTION:   SendFavoritesChangedPropertyUpdate
*
*  DESCRIPTION: Triggers a favorites change property update
*
*  PARAMETERS: IN-
*              OUT-
*
*  RETURNVALUE:
********************************************************************************/
void FC_Phonebook_tclFavorite_Provider::sendFavoritesChangedPropertyUpdate(quint32 u32FavoritesChangeType, quint32 u32ListLength,
                                                                           quint32 u32NumOfItems, QList<quint16> FavoritesIDList)
{

   ETG_TRACE_USR4(("Function SendFavoritesChangedPropertyUpdate entered"));

   most_phonbkfi_tclMsgFavoritesChangedStatus FavoritesChangedStatus;



   ETG_TRACE_USR2((" FavoritesProvdier:- Favorites change update triggered for "
      "ListChangeType        - %u ", u32FavoritesChangeType));
   ETG_TRACE_USR2(("FavoritesProvdier:- Favorites change update triggered for "
      "ListLength        - %u ", u32ListLength));
   ETG_TRACE_USR2(("FavoritesProvdier:- Favorites change update triggered for "
      "NumOfItemsChanged        - %u", u32NumOfItems));

   FavoritesChangedStatus.e8ListChangeType.enType=
      (most_fi_tcl_e8_FavPrvListChangeType::tenType)u32FavoritesChangeType;
   FavoritesChangedStatus.u32ListSize= u32ListLength;
   FavoritesChangedStatus.u32NumItems= u32NumOfItems;

   ETG_TRACE_USR1((" ListChangeType:: %u",ETG_ENUM(TR_LISTCHANGETYPE,FavoritesChangedStatus.e8ListChangeType.enType)));
   if(1 == u32NumOfItems)
   {
      if(FavoritesIDList.size() > 0)
      {
         FavoritesChangedStatus.oChangedFavorites.u16Items.push_back(FavoritesIDList.at(0));
         ETG_TRACE_USR2(("FavoritesProvdier:- Favorites change update triggered for "
            "FavoritesID        - %d", FavoritesIDList.at(0)));
      }
      else
      {
         ETG_TRACE_USR4(("FavoritesProvdier:- Favorites List is empty"));
      }
   }
   else
   {
      for(qint16 ListIndex= 0; ListIndex < FavoritesIDList.size(); ListIndex++)
      {
         FavoritesChangedStatus.oChangedFavorites.u16Items.push_back(FavoritesIDList.at(ListIndex));

         ETG_TRACE_USR2(("FavoritesProvider:- Favorites change update triggered for "
            "FavoritesID        - %d", FavoritesIDList.at(ListIndex)));

      }
   }

   m_poservice_phonebook->vPhoneBook_UpdateFavoritesChanged(FavoritesChangedStatus);




}

/*******************************************************************************
** FUNCTION:   RefreshFavoritesOnEditContact
*
*  DESCRIPTION: This function is triggered after editing a contact in vehicle
*  phonebook to update any associated favorite. It also triggers favorite change
*  property in case of any update
*
*  PARAMETERS: IN- contact handle of the edited contact
*              OUT-
*
*  RETURNVALUE:
********************************************************************************/
void FC_Phonebook_tclFavorite_Provider::RefreshFavoritesOnEditContact(
   FC_PhoneBook_Contact_Detail& ContactDetail)
{

   ETG_TRACE_USR4(("Function RefreshFavoritesOnEditContact entered"));

   QList<FC_PhoneBook_Fav_Detail> FavInfoList;

   if(( m_favSqlite != NULL) && (m_favSqlite->GetFavoriteRecordsOfContactHandle(
      ContactDetail.contactHandle, FavInfoList)))
   {
      if(OSAL_NULL != FavInfoList.size())
      {
         QList<quint16> DeletedFavoritesIDList;

         QList<FC_PhoneBook_Fav_Detail>::const_iterator itr=
            FavInfoList.constBegin();
         while(itr != FavInfoList.constEnd())
         {
            most_fi_tcl_e8_FavPrvFavoriteType::tenType enFavType=
               getFavoriteType(itr->tagvalue);
            QString PhoneNumber;
            retreivePhoneNumberFromContact(ContactDetail,
               itr->tagvalue, PhoneNumber);

            if(most_fi_tcl_e8_FavPrvFavoriteType::FI_EN_E8FT_PHONE_NUMBER
               == enFavType)
            {
               if(PhoneNumber != itr->phonenumber)
               {

                  ETG_TRACE_USR2(("Phonenumbers doesn't match for favoriteID- %d.Therefore making favorite as unavailable"
                     , itr->favoriteID));
                  DeletedFavoritesIDList<<itr->favoriteID;
               }
               else
               {

                  ETG_TRACE_USR2(("Phone numbers match for favoriteID- %d", itr->favoriteID));
               }
            }
            else if(most_fi_tcl_e8_FavPrvFavoriteType::FI_EN_E8FT_CONTACT_NAME
               == enFavType)
            {
               FC_PhoneBook_Fav_Detail ModifiedFavDetail;
               ModifiedFavDetail.contacthandle= itr->contacthandle;
               ModifiedFavDetail.devicehandle= itr->devicehandle;
               ModifiedFavDetail.favoriteID= itr->favoriteID;
               if(1 == m_favSqlite->GetPreferredSortOrder())   // GMMY17-1233 LabelText is constructed based on the current sort order
               {
                  ModifiedFavDetail.labeltext= ContactDetail.lastName +
                                    " " + ContactDetail.firstName;
               }
               else
               {
                  ModifiedFavDetail.labeltext= ContactDetail.firstName +
                                    " " + ContactDetail.lastName;
               }
               ModifiedFavDetail.phonenumber= PhoneNumber;
               ModifiedFavDetail.tagvalue= itr->tagvalue;
               ModifiedFavDetail.availablestatus= 1;
               ModifiedFavDetail.activestatus= 0;


               ETG_TRACE_USR2(("Updating favorites id- %d", ModifiedFavDetail.favoriteID));

               m_favSqlite->UpdateFavoritesRecord(ModifiedFavDetail);
            }
            else
            {

               ETG_TRACE_ERR((" Invalid favorites type- %d ", (quint8)enFavType));
            }

            itr++;

         }


         ETG_TRACE_USR2(("Total favorites associated with contact handle- %d", FavInfoList.size()));

         if(DeletedFavoritesIDList.size()>0)
         {
            if(m_favSqlite->UpdateFavoritesAvailableStatus(DeletedFavoritesIDList, OSAL_NULL))
            {
               quint32 u32ListChangeType=
                  (quint32)most_fi_tcl_e8_FavPrvListChangeType::FI_EN_E8LCH_ITEMS_CHANGED;
               quint32 u32ListSize= m_favSqlite->GetFavoritesCount();
               quint32 u32NumOfItems= (quint32)DeletedFavoritesIDList.size();

               sendFavoritesChangedPropertyUpdate(u32ListChangeType, u32ListSize,
                  u32NumOfItems, DeletedFavoritesIDList);


               ETG_TRACE_USR2(("Favorites updated successfully as unavailable"));

            }
            else
            {

               ETG_TRACE_ERR(("Failed to clear available status favorites associated with "
                  "deleted contact handle- %u", ContactDetail.contactHandle));
            }
         }
         else
         {

            ETG_TRACE_USR2(("No favorites to make as unavailable"));
         }
      }
      else
      {

         ETG_TRACE_USR2(("No favorites associated with contact handle - %u",ContactDetail.contactHandle));
      }
   }
   else
   {

      ETG_TRACE_ERR(("Fetching favorites associated with contact handle- %u failed", ContactDetail.contactHandle));
   }



}

/*******************************************************************************
** FUNCTION:   retreivePhoneNumberFromContact
*
*  DESCRIPTION:
*
*  PARAMETERS: IN-
*              OUT-
*
*  RETURNVALUE:
********************************************************************************/
void FC_Phonebook_tclFavorite_Provider::retreivePhoneNumberFromContact(
   const FC_PhoneBook_Contact_Detail& ContactDetail, quint8 u8TagValue, QString& PhoneNumber)
{
   ETG_TRACE_USR4(("retreivePhoneNumberFromContact entered"));
   PhoneNumber.clear();

   switch((FC_Phonebook_tclFavorite_Provider::favoriteTag)u8TagValue)
   {
   case FC_Phonebook_tclFavorite_Provider :: PrefNumTag:
      {
         PhoneNumber = ContactDetail.PhoneNum1;
      }
      break;
   case FC_Phonebook_tclFavorite_Provider :: CellNum1Tag:
      {
         PhoneNumber = ContactDetail.PhoneNum2;
      }
      break;
   case FC_Phonebook_tclFavorite_Provider :: CellNum2Tag:
      {
         PhoneNumber = ContactDetail.PhoneNum3;
      }
      break;
   case FC_Phonebook_tclFavorite_Provider :: HomeNum1Tag:
      {
         PhoneNumber = ContactDetail.PhoneNum4;
      }
      break;
   case FC_Phonebook_tclFavorite_Provider :: HomeNum2Tag:
      {
         PhoneNumber = ContactDetail.PhoneNum5;
      }
      break;
   case FC_Phonebook_tclFavorite_Provider :: WorkNum1Tag:
      {
         PhoneNumber = ContactDetail.PhoneNum6;
      }
      break;
   case FC_Phonebook_tclFavorite_Provider :: WorkNum2Tag:
      {
         PhoneNumber = ContactDetail.PhoneNum7;
      }
      break;
   case FC_Phonebook_tclFavorite_Provider :: OtherNumTag:
      {
         PhoneNumber = ContactDetail.PhoneNum8;
      }
      break;
   case FC_Phonebook_tclFavorite_Provider :: ContactNameTag:
      {

         ETG_TRACE_USR2(("No number available as it is a contact name tag"));
      }
      break;

   default :

      ETG_TRACE_ERR((" Tag value - %d is invalid", u8TagValue));
   }


   ETG_TRACE_USR4(("PhoneNumber retrieved is %s", PhoneNumber.toLatin1().constData()));
}

/*******************************************************************************
** FUNCTION:   GetInstance
*
*  DESCRIPTION:
*
*  PARAMETERS: IN-
*              OUT-
*
*  RETURNVALUE:
********************************************************************************/
FC_Phonebook_tclFavorite_Provider* FC_Phonebook_tclFavorite_Provider::GetInstance()
{
   return m_self;
}

/*******************************************************************************
** FUNCTION:   RefreshFavoritesOnDownloadComplete
*
*  DESCRIPTION:
*
*  PARAMETERS: IN-
*              OUT-
*
*  RETURNVALUE:
********************************************************************************/
void FC_Phonebook_tclFavorite_Provider::RefreshFavoritesOnDownloadComplete(
   quint8 u8DeviceHandle)
{


   ETG_TRACE_USR2(("Function RefreshFavoritesOnDownloadComplete entered with DeviceHandle %d",u8DeviceHandle));

   QList<FC_PhoneBook_Fav_Detail> FavInfoList;
   if(( m_favSqlite != NULL) && (m_favSqlite->GetFavoriteRecordsOfDeviceHandle(u8DeviceHandle,
      FavInfoList)))
   {
      if(FavInfoList.size() > OSAL_NULL)
      {
         FC_PhoneBook_Contact_Detail ContactDetail;
         QList<FC_PhoneBook_Fav_Detail>::const_iterator itr=
            FavInfoList.constBegin();
         quint32 u32ContactHandle;
         QList<quint16> FavoritesIDList;

         while(itr != FavInfoList.constEnd())
         {
            u32ContactHandle= itr->contacthandle;
            if(m_favSqlite->GetContactDetails(u32ContactHandle,
               ContactDetail))
            {
               QString PhoneNumber;
               retreivePhoneNumberFromContact(ContactDetail, itr->tagvalue,
                  PhoneNumber);
               if ((itr->phonenumber == PhoneNumber)
                     && (((itr->labeltext) == (ContactDetail.firstName + " " + ContactDetail.lastName))
                           || ((itr->labeltext) == (ContactDetail.lastName + " " + ContactDetail.firstName))))   // GMMY17-1233 LabelText can be stored based on any of the sort order
               {
                  FavoritesIDList << itr->favoriteID;

                  ETG_TRACE_USR2(("Favorite id - %d is now available", itr->favoriteID));
               }
               else
               {

                  ETG_TRACE_USR2(("Favorite id - %d will remain unavailable", itr->favoriteID));
               }

            }
            else
            {

               ETG_TRACE_ERR(("Failed to fetch contact details for "
                  "contact handle- %d", u32ContactHandle));
            }
            itr++;
         }

         if(FavoritesIDList.size() > OSAL_NULL)
         {
            if(m_favSqlite->UpdateFavoritesAvailableStatus(FavoritesIDList, 1))
            {

               ETG_TRACE_USR2(("Favorites updated as available successfully"));

               quint32 u32ListChangeType=
                  (quint32)most_fi_tcl_e8_FavPrvListChangeType::FI_EN_E8LCH_ITEMS_CHANGED;
               quint32 u32ListSize= m_favSqlite->GetFavoritesCount();
               quint32 u32NumOfItems= (quint32)FavoritesIDList.size();

               sendFavoritesChangedPropertyUpdate(u32ListChangeType, u32ListSize,
                  u32NumOfItems, FavoritesIDList);

            }
            else
            {

               ETG_TRACE_ERR(("Failed to update favorites as available"));
            }
         }
      }
      else
      {

         ETG_TRACE_USR2(("No favorites associated with device handle- %d", u8DeviceHandle));
      }
   }
   else
   {

      ETG_TRACE_ERR(("Failed to get the favorite records for "
         "the device handle - %d", u8DeviceHandle));
   }

}

/*******************************************************************************
** FUNCTION:   RefreshFavoritesOnDeviceDisconnection
*
*  DESCRIPTION:
*
*  PARAMETERS: IN-
*              OUT-
*
*  RETURNVALUE:
********************************************************************************/
void FC_Phonebook_tclFavorite_Provider::RefreshFavoritesOnDeviceDisconnection(
   quint8 u8DeviceHandle)
{

   ETG_TRACE_USR2(("Function RefreshFavoritesOnDeviceDisconnection entered DeviceHandle %d ",u8DeviceHandle));

   QList<FC_PhoneBook_Fav_Detail> FavInfoList;
   if((m_favSqlite != NULL) && (m_favSqlite->GetFavoriteRecordsOfDeviceHandle(u8DeviceHandle,
      FavInfoList)))
   {
      if(FavInfoList.size() > OSAL_NULL)
      {
         QList<FC_PhoneBook_Fav_Detail>::const_iterator itr=
            FavInfoList.constBegin();
         QList<quint16> FavoritesIDList;

         while(itr != FavInfoList.constEnd())
         {
            FavoritesIDList<<itr->favoriteID;


            ETG_TRACE_USR2(("Favorite id - %d is now unavailable", itr->favoriteID));

            itr++;
         }

         if(FavoritesIDList.size() > OSAL_NULL)
         {
            if(m_favSqlite->UpdateFavoritesAvailableStatus(FavoritesIDList, OSAL_NULL))
            {

               ETG_TRACE_USR2(("Favorites updated as unavailable successfully"));

               quint32 u32ListChangeType=
                  (quint32)most_fi_tcl_e8_FavPrvListChangeType::FI_EN_E8LCH_ITEMS_CHANGED;
               quint32 u32ListSize= m_favSqlite->GetFavoritesCount();
               quint32 u32NumOfItems= (quint32)FavoritesIDList.size();

               sendFavoritesChangedPropertyUpdate(u32ListChangeType, u32ListSize,
                  u32NumOfItems, FavoritesIDList);

            }
            else
            {

               ETG_TRACE_ERR(("Failed to update favorites as unavailable"));
            }
         }
      }
      else
      {

         ETG_TRACE_USR2(("No favorites associated with device handle- %d", u8DeviceHandle));
      }
   }
   else
   {

      ETG_TRACE_ERR(("Failed to get the favorite records for "
         "the device handle - %d", u8DeviceHandle));
   }

}
// ---: RB fix proposal for GMNGA-53014 and GMMY15-248
/*******************************************************************************
** FUNCTION:   RefreshFavoritesOnDeviceDeletion
*
*  DESCRIPTION: Delete favorites in FAV table for the deleted (unpaired) BT device
*
*  PARAMETERS: IN-  u8DeviceHandle
*              OUT- none
*
*  RETURNVALUE:
********************************************************************************/
void FC_Phonebook_tclFavorite_Provider::RefreshFavoritesOnDeviceDeletion(quint8 u8DeviceHandle)  // ---:
{
   ETG_TRACE_USR4(( "--> FC_Phonebook_tclFavorite_Provider::RefreshFavoritesOnDeviceDeletion " ));
   ETG_TRACE_USR4(( " Refresh Favorites as BT phone got deleted (unpaired), DeviceHandle= %d ", u8DeviceHandle ));
   if (m_favSqlite != NULL)
   {
      QList<quint16> FavoritesIDList;
      m_favSqlite->GetAllFavoritesIDForDevice(FavoritesIDList, u8DeviceHandle);
      (void)m_favSqlite->DeleteAllFavRecordsOfDevice(u8DeviceHandle);
      quint32 u32ListChangeType = (quint32) most_fi_tcl_e8_FavPrvListChangeType::FI_EN_E8LCH_ITEMS_REMOVED;
      quint32 u32ListSize       = m_favSqlite->GetFavoritesCount();
      quint32 u32NumOfItems     = (quint32)FavoritesIDList.size();
      sendFavoritesChangedPropertyUpdate(u32ListChangeType, u32ListSize,
         u32NumOfItems, FavoritesIDList);
   }
   ETG_TRACE_USR4(( "<-- FC_Phonebook_tclFavorite_Provider::RefreshFavoritesOnDeviceDeletion " ));
}

/*******************************************************************************
** FUNCTION:   RefreshFavoritesOnDeviceDisconnection
*
*  DESCRIPTION:
*
*  PARAMETERS: IN-
*              OUT-
*
*  RETURNVALUE:
********************************************************************************/
quint32 FC_Phonebook_tclFavorite_Provider::GetTotalFavoritesCount()
{
   ETG_TRACE_USR4(("GetTotalFavoritesCount entered"));

   if( m_favSqlite != NULL)
   {
      return m_favSqlite->GetFavoritesCount();
   }
   else
   {

      ETG_TRACE_FATAL(("NULL POINTER: m_favSqlite"));
      return 0;
   }
}

/*******************************************************************************
** FUNCTION:   RegisterPhoneClientHandlerInstance
*
*  DESCRIPTION:
*
*  PARAMETERS: IN-
*              OUT-
*
*  RETURNVALUE:
********************************************************************************/
void FC_Phonebook_tclFavorite_Provider::RegisterPhoneClientHandlerInstance(
   fc_phonebook_tclClientHandler_Phone* pPhoneClientHandler)
{
   m_poPhoneClientHandler= pPhoneClientHandler;
   m_poPhoneClientHandler->vRegisterFavoritesProviderInstance(this);
}

void FC_Phonebook_tclFavorite_Provider::ChangefavoriteActiveStatus(quint8 u8Status)
{
   ETG_TRACE_USR2(("ChangefavoriteActiveStatus Entered with Status %d",u8Status));
   if((m_favSqlite != NULL ) && (m_favSqlite->UpdateAllFavoritesActiveStatus(u8Status)))
   {


      quint32 u32ListChangeType=
         (quint32)most_fi_tcl_e8_FavPrvListChangeType::FI_EN_E8LCH_ITEMS_CHANGED;
      quint32 u32ListSize= m_favSqlite->GetFavoritesCount();
      quint32 u32NumOfItems= u32ListSize;
      QList<quint16> FavoritesIDList;
      // FavoritesIDList<<u16FavoriteID;
      m_favSqlite->GetAllFavoritesID(FavoritesIDList);
      sendFavoritesChangedPropertyUpdate(u32ListChangeType, u32ListSize,
         u32NumOfItems, FavoritesIDList);
   }
   else
   {
      ETG_TRACE_ERR(("Failed to set active status"));
   }

}
//Fix for GMMY16-9895 Pressing the favorite button that was used to place an outgoing call does not end the call
/*******************************************************************************
** FUNCTION:   SendActivateFavoriteMR
*
*  DESCRIPTION:
*
*  PARAMETERS: IN-
*              OUT-
*
*  RETURNVALUE:
********************************************************************************/
void FC_Phonebook_tclFavorite_Provider::SendActivateFavoriteMR(quint16 u16CallInstance, most_fi_tcl_e8_TelCallStatus::tenType enCallStatus)
{
   ETG_TRACE_USR4(("Function SendActivateFavoriteMR entered for Call Instance - %d", u16CallInstance));

   quint16 u16FavoriteID;
   if (!m_ActivatedFavoriteIDList.empty())
   {
      u16FavoriteID = m_ActivatedFavoriteIDList.first();

      ETG_TRACE_USR2(("This call state is corresponding to favorite id - %d", u16FavoriteID));

      if (enCallStatus == most_fi_tcl_e8_TelCallStatus::FI_EN_E8DIALING) // -bn: improved formatting/readability
      {
         ETG_TRACE_USR4(("Call Status is DIALING"));
         m_CallInstanceFavIDMap.insert(u16CallInstance, u16FavoriteID);

         if (m_poservice_phonebook != NULL)
         {
            m_poservice_phonebook->vPostActivateFavoriteMethodResult(u16FavoriteID, (quint32) u16CallInstance);
            ETG_TRACE_USR4(("m_poservice_phonebook->vPostActivateFavoriteMethodResult Posted"));
         }

         m_ActivatedFavoriteIDList.removeOne(u16FavoriteID); // ---: Bosch Correction for GMNGA-54547: (added this line)
      }
      else if (enCallStatus == most_fi_tcl_e8_TelCallStatus::FI_EN_E8IDLE)
      {
         ETG_TRACE_USR4(("Call Status is IDLE"));

         if (m_poservice_phonebook != NULL)
         {
            m_poservice_phonebook->vPostActivateFavoriteMethodResult(u16FavoriteID, (quint32) u16CallInstance);
            ETG_TRACE_USR4(("m_poservice_phonebook->vPostActivateFavoriteMethodResult Posted"));
         }
         m_ActivatedFavoriteIDList.removeOne(u16FavoriteID);
      }
   }
   else
   {
      ETG_TRACE_USR4(("m_ActivatedFavoriteIDList is empty."));
   }
}

/*******************************************************************************
** FUNCTION:   SendActivateFavoriteMError
*
*  DESCRIPTION:
*
*  PARAMETERS: IN-
*              OUT-
*
*  RETURNVALUE:
********************************************************************************/
void FC_Phonebook_tclFavorite_Provider::SendActivateFavoriteMError(quint16 u16ErrorCode)
{
   ETG_TRACE_USR4(("Function SendActivateFavoriteMError entered with ErrorCode %d ",u16ErrorCode));

   if(!m_ActivatedFavoriteIDList.empty())
   {
      quint16 u16FavoriteID= m_ActivatedFavoriteIDList.first();

      ETG_TRACE_USR2(("Favorite associated with this SendActivateFavorite Method Error  - %d", u16FavoriteID));

      if(m_poservice_phonebook != NULL)
      {
         m_poservice_phonebook->vPostActivateFavoriteError(most_fi_tcl_e8_ErrorCode::FI_EN_NOTAVAILABLE, u16FavoriteID, 0);   //Fix GMMY16-13959
      }
      m_ActivatedFavoriteIDList.removeOne(u16FavoriteID);
      ETG_TRACE_USR4(("ActivateFavorite_Error triggered"));
   }
   else
   {
      ETG_TRACE_USR4(("No data in Activated favorites"));
   }
}

/*******************************************************************************
** FUNCTION:   CallStatusNotice
*
*  DESCRIPTION:
*
*  PARAMETERS: IN-
*              OUT-
*
*  RETURNVALUE:
********************************************************************************/
void FC_Phonebook_tclFavorite_Provider::CallStatusNotice(quint16 u16CallInstance, most_fi_tcl_e8_TelCallStatus::tenType enCallStatus)
{
   ETG_TRACE_USR4(("Function CallStatusNotice entered for Call Instance - %d", u16CallInstance));
   tU8 u8Status = 0;
   tBool bUpdateRequired = FALSE;

   if (u16CallInstance == m_poPhoneClientHandler->m_CurrentCallInstance)
   {
      SendActivateFavoriteMR(u16CallInstance, enCallStatus);
   }

   //Updating Favorite Status to active or inactive
   
   if (TRUE == m_CallInstanceFavIDMap.contains(u16CallInstance)) //CMG3G-14313-Coverity Fix //Fix for GMMY15-10219: 'm_CallInstanceFavIDMap' contains all the ACTIVE Favorites.
   {
      quint16 u16FavoriteID = m_CallInstanceFavIDMap[u16CallInstance];
      if ((most_fi_tcl_e8_TelCallStatus::FI_EN_E8IDLE == enCallStatus) || (most_fi_tcl_e8_TelCallStatus::FI_EN_E8DISCONNECTING == enCallStatus) )
      {
         ETG_TRACE_USR4(("CallStatus::FI_EN_E8IDLE"));
         m_CallInstanceFavIDMap.remove(u16CallInstance);
         u8Status = 0;
         bUpdateRequired = TRUE;
      }
      else if (most_fi_tcl_e8_TelCallStatus::FI_EN_E8DIALING == enCallStatus)
      {
         ETG_TRACE_USR4(("CallStatus::FI_EN_E8DIALING"));
         u8Status = 1;
         bUpdateRequired = TRUE;
      }

      if ((m_favSqlite != NULL) && (TRUE == bUpdateRequired))
      {
         if (m_favSqlite->UpdateFavoritesActiveStatus(u16FavoriteID, u8Status))
         {
            ETG_TRACE_ERR(("Status updated for favorite id- %d", u16FavoriteID));

            quint32 u32ListChangeType = (quint32) most_fi_tcl_e8_FavPrvListChangeType::FI_EN_E8LCH_ITEMS_CHANGED;
            quint32 u32ListSize = m_favSqlite->GetFavoritesCount();
            quint32 u32NumOfItems = 1;
            QList < quint16 > FavoritesIDList;
            FavoritesIDList << u16FavoriteID;

            sendFavoritesChangedPropertyUpdate(u32ListChangeType, u32ListSize, u32NumOfItems, FavoritesIDList);
         }
         else
         {
            ETG_TRACE_ERR(("Failed to update active status of favorite id- %d", u16FavoriteID));
         }
      }
   }
   else
   {
      ETG_TRACE_USR4(("Status update not for activated call instance"));
      //FIX GMNGA-56482 If the dialed number from HMI/device is stored as favourite, it should be highlighted.
      if (m_poPhoneClientHandler)
      {
         if (most_fi_tcl_e8_TelCallStatus::FI_EN_E8IDLE != enCallStatus
            && m_CallInstanceStateMap[u16CallInstance] != (tU8) enCallStatus)
         {
            m_poPhoneClientHandler->vTrigger_QueryTelephoneNumber_MethodStart(u16CallInstance);

         }
      }
   }
   m_CallInstanceStateMap[u16CallInstance] = (tU8) enCallStatus;
}
//End of fix GMMY16-9895

/*******************************************************************************
** FUNCTION:   UpdateFavoriteIDList
*
*  DESCRIPTION:
*
*  PARAMETERS: IN-
*              OUT-
*
*  RETURNVALUE:
********************************************************************************/
void FC_Phonebook_tclFavorite_Provider::UpdateFavoriteIDList()
{
   ETG_TRACE_USR4(("UpdateFavoriteIDList entered"));
   if(!m_favSqlite)
   {

      ETG_TRACE_ERR(("SQlite instance is NULL"));
   }
   else
   {
      m_UsedFavoriteIDList.clear();
      m_favSqlite->GetAllFavoritesID(m_UsedFavoriteIDList);
   }
}

tVoid favoritidequeue::enqueueCCAMessage(tU16 service_data)
{ 
   ETG_TRACE_USR4(("enqueueCCAMessage entered"));

   mutexCCAMsgService.lock();

   ccaMessageQBuffer.enqueue(service_data);

   mutexCCAMsgService.unlock();

}
tU16 favoritidequeue::dequeueCCAMessage()
{
   tU16 data = 0;
   mutexCCAMsgService.lock();



   if (false == ccaMessageQBuffer.isEmpty())
   {
      data = ccaMessageQBuffer.dequeue();

   }



   mutexCCAMsgService.unlock();
   return data;
}
/*******************************************************************************
** FUNCTION:   QueryTelephoneNumberMethodResult
*
*  DESCRIPTION:
*
*  PARAMETERS: IN-
*              OUT-
*
*  RETURNVALUE:
********************************************************************************/
void FC_Phonebook_tclFavorite_Provider::QueryTelephoneNumberMethodResult(QString sTelephoneNumber, quint16 u16CallInstance)
{
   ETG_TRACE_USR4(("FavoritesProvdier:- QueryTelephoneNumberMethodResult entered"));

   if (m_favSqlite != NULL)
   {
      QList<quint16> FavoritesIDList;

      m_favSqlite->GetFavouritesIDForPhoneNumber(FavoritesIDList, sTelephoneNumber);

      if(FavoritesIDList.size()!=0)
      {
         for(qint16 ListIndex= 0; ListIndex < FavoritesIDList.size(); ListIndex++)
         {
            if(m_favSqlite->UpdateFavoritesActiveStatus(FavoritesIDList.at(ListIndex),0x01))
            {
               ETG_TRACE_USR2(("Active status set for favorite id- %d",
                  FavoritesIDList.at(ListIndex)));

               m_CallInstanceFavIDMap[u16CallInstance]= FavoritesIDList.at(ListIndex);
               ETG_TRACE_USR2(("u16CallInstance- %d",u16CallInstance));
            }
         }
         quint32 u32NumOfItems = (quint32)FavoritesIDList.size();
         quint32 u32ListChangeType= (quint32)most_fi_tcl_e8_FavPrvListChangeType::FI_EN_E8LCH_ITEMS_CHANGED;
         quint32 u32ListSize= m_favSqlite->GetFavoritesCount();

         sendFavoritesChangedPropertyUpdate(u32ListChangeType, u32ListSize,
            u32NumOfItems, FavoritesIDList);
      }
      else
      {
         ETG_TRACE_USR2(("No favorites stored for the phone number in call status"));
      } 
   }
   else
   {
      ETG_TRACE_ERR(("Failed to set active status of favorite id"));
   }
}
