/*******************************************************************************
*
* FILE:          FC_Phonebook_Device_Details.cpp
*
* SW-COMPONENT:  FC_PhoneBook application
*
* PROJECT:
*
* DESCRIPTION:   Handles the Device Data.
*              Every connected device will be represented by one instance this class
*
* AUTHOR:       Abhishek Kumar Dhiman
*
* COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
*
*******************************************************************************/

/*******************************************************************************
*                      INCLUDES
*******************************************************************************/
#include <glib.h>

#include "FC_PhoneBook_Debug.h"
#include "FC_PhoneBook_Device_Details.h"
#include "FC_PhoneBook_main.h"
#include "DbusClient/FC_PhoneBook_EvoDbusClient.h" //CMG3G-14314

#define ETRACE_S_IMPORT_INTERFACE_GENERIC

#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONEBOOK_APPLICATION
#include "trcGenProj/Header/FC_PhoneBook_Device_Details.cpp.trc.h"
#endif
/*******************************************************************************
*                      DEFINES
*******************************************************************************/

/*******************************************************************************
*                      START CODE
*******************************************************************************/
FC_Device_Details *FC_Device_Details::arrayConnectedDevice[] = {NULLPTR,NULLPTR};

//CLONE TICKET FIX GMMY15-1531 - Avoids parallel access to FC_Device_Details
QMutex FC_Device_Details::m_DeviceListMutex;


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::FC_Device_Details()
*
* DESCRIPTION: Constructor
*
* PARAMETER: default parameters
*
* RETURNVALUE: None.
*
*******************************************************************************/
FC_Device_Details::FC_Device_Details(tU8 tU8Values)
{
   ETG_TRACE_USR4(("* FC_Device_Details constructor "));

   m_connectionID = tU8Values;
   m_deviceCapabilities = tU8Values;
   m_deviceHandle = tU8Values;
   m_sourceType = tU8Values;
   m_enPBKDownloadState = DOWNLOAD_ST_NOT_STARTED;

   Device_DateTime_Support.CHDateTimeSupport = TRUE;
   Device_DateTime_Support.DateTimesupportUpdated = FALSE;
   Device_DateTime_Support.VcardsHaveDatetime = FALSE;
   Device_DateTime_Support.VcardsHaveDatetimeupdated = FALSE;

   m_sortListViewSupport = FALSE;
   memset(u8DownloadTypeList, 0xFF, sizeof(u8DownloadTypeList));
   RetryCnt = tU8Values;
   m_newMissedcall = tU8Values;
   PBDLCommand = tU8Values;
   m_blFirstTimeDownload= false;
   setCapInProgress(false);
   memset(m_deviceAddress,0,LENGTH_OF_DEVICE_ADDRESS);
   setPhBkPrepareCallBackStatus(false);

   m_bIsPendingRecentCallDownloading = false;
   m_bIsPendingManualPBDownloading = false;
   m_countDownload = 0;
   downloadType = 0;
   m_bMEDownloadStatus = false;
   m_bSMDownloadStatus = false;  // -bn-PBDL.S: added

   m_sRecentMissedCallDate = EMPTY_STRING;
   m_sRecentMissedCallTime = EMPTY_STRING;

   m_sFirstMissedCallDate = EMPTY_STRING;
   m_sFirstMissedCallTime = EMPTY_STRING;

   m_MissedcallCount = 0; //CMG3G-12350

   m_bDlCompletedOnceValue = false;
   m_LastConnectionTimeElapsed = 0;

   //CMG3G-14313-coverity fix
   m_u32ContributingVCardCount=0;
   m_MissedcallCount=0;
   //end of fix

   m_u32DeviceContactsCount = 0; //CMG3G-8363
   m_u32DeviceSimContactsCount = 0; //RTC 159808
   m_u32InidvidualTypeContactsCount = 0; //RTC 159808
   m_u32DeviceIntPbContactsCount = 0; //CMG3G-12876
   m_u32DeviceFavContactsCount = 0; //CMG3G-12876
   m_u32DeviceSpdContactsCount = 0; //CMG3G-12876
   m_u32VCardsAvailableForDwnld = 0;
   m_downloadOnOffStatus = 0;

   m_NewContactsToBeAddedRAMList.clear();  // -bn: 2016.01: DL speed improvement by collecting contacts before insertion to database
   m_PhotosToBeUpdatedRAMList.clear();     // -bn: 2016.01: DL speed improvement by collecting contacts before photo update in database

   m_bIsCancelPbDownload = false; //FIX CMG3G-8372
   m_u32PhotosUpdatedInCurrentDwnld = 0;

   m_u8FAVSPDDwnldSupport = 0; //CMG3G-11867

   //Fix CMG3G-12880
   m_bDeviceProbablyLocked = FALSE;
   m_bSuspiciousGetSize = FALSE;
   //End fix CMG3G-12880

   //Fix NCG3D-65982
   m_bDeviceDownloadSuspended = FALSE;
   //End fix NCG3D-65982
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::~FC_Device_Details()
*
* DESCRIPTION: Destructor
*
* PARAMETER: None
*
* RETURNVALUE: None.
*
*******************************************************************************/
FC_Device_Details::~FC_Device_Details()
{
   m_u8PendingManualPBCommandList.clear();
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::getConnectionID()
*
* DESCRIPTION: Returns the connectionID for the deviceObject.
*
* PARAMETER: None
*
* RETURNVALUE: ConnectionID.
*
*******************************************************************************/
tU8  FC_Device_Details::getConnectionID()
{
   ETG_TRACE_USR4(( " getConnectionID= %d ", m_connectionID ));
   return this->m_connectionID;
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::setConnectionID()
*
* DESCRIPTION: Set the connectionID for the deviceObject.
*
* PARAMETER: ConnectionID
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid  FC_Device_Details::setConnectionID(tU8 connectionId)
{
   ETG_TRACE_USR2((" setConnectionID: old connectionID= %d, newconnectionID= %d ", m_connectionID, connectionId ));
   this->m_connectionID = connectionId;
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::getDeviceHandle()
*
* DESCRIPTION: Returns the DeviceHandle for the deviceObject.
*
* PARAMETER: None
*
* RETURNVALUE: deviceHandle.
*
*******************************************************************************/
tU8  FC_Device_Details::getDeviceHandle()
{
   ETG_TRACE_USR4(( " getDeviceHandle= %d", m_deviceHandle ));
   return this->m_deviceHandle;
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::setDeviceHandle()
*
* DESCRIPTION: Set the DeviceHandle for the deviceObject.
*
* PARAMETER: deviceHandle
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid  FC_Device_Details::setDeviceHandle(tU8 deviceHandle)
{
   ETG_TRACE_USR4(( " setDeviceHandle: deviceHandle= %d ", deviceHandle ));
   this->m_deviceHandle = deviceHandle;
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::getSourceType()
*
* DESCRIPTION: Returns the sourceType for the deviceObject.
*
* PARAMETER: None
*
* RETURNVALUE: sourceType.
*
*******************************************************************************/
tU8  FC_Device_Details::getSourceType()
{
   ETG_TRACE_USR4(( " getSourceType= %d ", m_sourceType ));
   return this->m_sourceType;
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::setSourceType()
*
* DESCRIPTION: Set the sourceType for the deviceObject.
*
* PARAMETER: sourceType
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid  FC_Device_Details::setSourceType(tU8 sourceType)
{
   ETG_TRACE_USR4(( " setSourceType: sourceType= %d ", sourceType ));
   this->m_sourceType = sourceType;
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::getDeviceCapabilities()
*
* DESCRIPTION: Returns the DeviceCapabilities for the deviceObject.
*
* PARAMETER: None
*
* RETURNVALUE: DeviceCapabilities.
*
*******************************************************************************/
tU8  FC_Device_Details::getDeviceCapabilities()
{
   ETG_TRACE_USR4(( " getDeviceCapabilities= 0x%X ", m_deviceCapabilities ));
   return this->m_deviceCapabilities;
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::setDeviceCapabilities()
*
* DESCRIPTION: Set the DeviceCapabilities for the deviceObject.
*
* PARAMETER: DeviceCapabilities
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid  FC_Device_Details::setDeviceCapabilities(tU8 deviceCapabilities)
{
   ETG_TRACE_USR4(("setDeviceCapabilities: deviceCapabilities= 0x%X ", deviceCapabilities ));
   this->m_deviceCapabilities = deviceCapabilities;
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::getDeviceAddress()
*
* DESCRIPTION: Returns the deviceAddress for the deviceObject.
*
* PARAMETER: None
*
* RETURNVALUE: deviceAddress.
*
*******************************************************************************/
tChar*  FC_Device_Details::getDeviceAddress()
{
   ETG_TRACE_USR4(("getDeviceAddress entered"));
   return this->m_deviceAddress;
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::setDeviceAddress()
*
* DESCRIPTION: Set the deviceAddress for the deviceObject.
*
* PARAMETER: deviceAddress
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid  FC_Device_Details::setDeviceAddress(const tChar * bluetoothDeviceAddress)
{
   vStringCopy(this->m_deviceAddress, bluetoothDeviceAddress, LENGTH_OF_DEVICE_ADDRESS);  //CMG3G-14313-coverity fix- Using vStringCopy instead of strncpy
   ETG_TRACE_USR4(( " setDeviceAddress: address= '%s' ", m_deviceAddress));
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::getDeviceObject()
*
* DESCRIPTION: Get the deviceObject.
*
* PARAMETER: Index in the array of connected devices
*
* RETURNVALUE: deviceObject.
*
*******************************************************************************/
FC_Device_Details * FC_Device_Details::getDeviceObject(INDEX getIndex)
{
   ETG_TRACE_USR4(("getDeviceObject entered"));
   return FC_Device_Details::arrayConnectedDevice[getIndex];
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::createDeviceObject()
*
* DESCRIPTION: Creates a new deviceObject.
*
* PARAMETER: Index in the array of connected devices
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid FC_Device_Details::createDeviceObject(INDEX createIndex)
{
   ETG_TRACE_COMP((" -PBDL.S- Creating device object for index: %d", createIndex));
   FC_Device_Details::arrayConnectedDevice[createIndex]
         = new FC_Device_Details();
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::deleteDeviceObject()
*
* DESCRIPTION: Deletes a deviceObject.
*
* PARAMETER: Index in the array of connected devices
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid FC_Device_Details::deleteDeviceObject(INDEX deleteIndex)
{
   ETG_TRACE_COMP((" -PBDL.S- Deleting device object for index: %d", deleteIndex));
   QMutexLocker lock(&m_DeviceListMutex);
   delete FC_Device_Details::arrayConnectedDevice[deleteIndex];
   FC_Device_Details::arrayConnectedDevice[deleteIndex] = NULLPTR;
}


/*******************************************************************************
** FUNCTION:   FC_Device_Details::deviceAddressToInstance()
*******************************************************************************/
/* DESCRIPTION: DeviceAddress to Object Instance no. Mapping
*
*  PARAMETERS:
*     IN:   poMessage        :
*  RETURNVALUE:
*******************************************************************************/
FC_Device_Details* FC_Device_Details::deviceAddressToObjectInstance(const tChar *deviceAddress)
{
   ETG_TRACE_USR4(("deviceAddressToObjectInstance entered"));
   FC_Device_Details* objectInstanceNo = NULL;

   //CLONE TICKET FIX GMMY15-1531 -  - Avoids parallel access to FC_Device_Details
   QMutexLocker lock(&m_DeviceListMutex);

   if(deviceAddress)
   {
      if(((FC_Device_Details::getDeviceObject(FIRST) != NULL) && (!strncmp(FC_Device_Details::getDeviceObject(FIRST)->getDeviceAddress(),deviceAddress,LENGTH_OF_DEVICE_ADDRESS))))
      {
         objectInstanceNo = FC_Device_Details::getDeviceObject(FIRST);
      }
      else if(((FC_Device_Details::getDeviceObject(SECOND) != NULL) && (!strncmp(FC_Device_Details::getDeviceObject(SECOND)->getDeviceAddress(),deviceAddress,LENGTH_OF_DEVICE_ADDRESS))))
      {
         objectInstanceNo = FC_Device_Details::getDeviceObject(SECOND);
      }
      else
      {
         return objectInstanceNo;
      }
   }

   return objectInstanceNo;
}


/*******************************************************************************
** FUNCTION:  FC_Device_Details::deviceHandleToObjectInstance()
*******************************************************************************/
/* DESCRIPTION: DeviceHandle to Object Instance no. Mapping
*
*  PARAMETERS:
*     IN: 
*  RETURNVALUE:
*******************************************************************************/
FC_Device_Details* FC_Device_Details::deviceHandleToObjectInstance(tU8 devicehandle)
{

   ETG_TRACE_USR4(("deviceHandleToObjectInstance entered"));
   FC_Device_Details* objectInstanceNo = NULL;
   //   ETG_TRACE_USR2(("deviceHandleToObjectInstance-dev handle%d", devicehandle));
   if(((FC_Device_Details::getDeviceObject(FIRST) != NULL) && (FC_Device_Details::getDeviceObject(FIRST)->getDeviceHandle() == devicehandle)))
   {
      objectInstanceNo = FC_Device_Details::getDeviceObject(FIRST);
   }
   else if(((FC_Device_Details::getDeviceObject(SECOND) != NULL) && (FC_Device_Details::getDeviceObject(SECOND)->getDeviceHandle() == devicehandle)))
   {
      objectInstanceNo = FC_Device_Details::getDeviceObject(SECOND);
   }

   return objectInstanceNo;
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::SetDownloadState()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE: .
*
*******************************************************************************/
tVoid FC_Device_Details::SetDownloadState(enDownloadStates enNewDnlState)
{
   ETG_TRACE_USR4(( " SetDownloadState:  NewDnlState= %d ", enNewDnlState ));
   m_enPBKDownloadState = enNewDnlState;
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::GetDownloadState()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE: .
*
*******************************************************************************/
FC_Device_Details::enDownloadStates FC_Device_Details::GetDownloadState()
{
   ETG_TRACE_USR4((" GetDownloadState: %d", ETG_CENUM(enDownloadStates, m_enPBKDownloadState) ));
   return m_enPBKDownloadState;
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::getSortListViewSupport()
*
* DESCRIPTION: Returns the SortListViewSupport for the deviceObject.
*
* PARAMETER: None
*
* RETURNVALUE: m_sortListViewSupport.
*
*******************************************************************************/
tBool  FC_Device_Details::getSortListViewSupport()
{
   ETG_TRACE_USR4(( "getSortListViewSupport= %d ", this->m_sortListViewSupport ));
   return this->m_sortListViewSupport;
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::setSortListViewSupport()
*
* DESCRIPTION: Set the SortListViewSupport for the deviceObject.
*
* PARAMETER: b_sortListViewSupport
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid  FC_Device_Details::setSortListViewSupport(tBool b_sortListViewSupport)
{
   ETG_TRACE_USR4((" setSortListViewSupport to value= %d ", b_sortListViewSupport ));
   this->m_sortListViewSupport = b_sortListViewSupport;
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::blIsCCHSupported()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE: None.
*
*******************************************************************************/
tBool  FC_Device_Details::blIsCCHSupported()
{
   tBool bCCHSupport = (  (m_deviceCapabilities & CombinedCallHistory)
      && Device_DateTime_Support.CHDateTimeSupport  );

   ETG_TRACE_COMP(( " -PBDL.S.FS- blIsCCHSupported= %d  [m_deviceCapabilities= 0x%X, Device_DateTime_Support.CHDateTimeSupport= %d] ",
      bCCHSupport, m_deviceCapabilities, Device_DateTime_Support.CHDateTimeSupport ));

   return bCCHSupport;
}


//****************************************************************************

/*******************************************************************************
*
* FUNCTION: FC_Device_Details::setFirstTimeDownload()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE: None.
*
*******************************************************************************/
void FC_Device_Details::setFirstTimeDownload(tBool blValue)
{
   ETG_TRACE_USR4(( "setFirstTimeDownload: bValue= %d ", blValue ));
   m_blFirstTimeDownload = blValue;
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::blIsItFirstTimeDownload()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE: None.
*
*******************************************************************************/
tBool FC_Device_Details::blIsItFirstTimeDownload()
{
   ETG_TRACE_USR4(( " blIsItFirstTimeDownload= %d ", m_blFirstTimeDownload ));
   return m_blFirstTimeDownload;
}


//FIX GMMY15-8739 Contacts are Displayed on the HMI, even though PBAP request was not accepted on the phone device.
/*******************************************************************************
*
* FUNCTION: FC_Device_Details::getPhBkPrepareCallBackStatus()
*
* DESCRIPTION: Returns the Status of PbDlPrepare for the deviceObject.
*
* PARAMETER: None
*
* RETURNVALUE: Status 0 -> PHBK Prepare is received and device doesnot prepare for PbDl Download
*                  1 -> PHBK Prepare is received and device prepares for PbDl Download
*
*******************************************************************************/
tBool FC_Device_Details::getPhBkPrepareCallBackStatus()
{
   ETG_TRACE_USR4(( " getPhBkPrepareCallBackStatus= %d", m_bPrepareCBStatus ));
   return(m_bPrepareCBStatus);
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::setPhBkPrepareCallBackStatus()
*
* DESCRIPTION: Sets the Status of PbDlPrepare for the deviceObject.
*
* PARAMETER: Status 0 -> PHBK Prepare is received and device doesnot prepare for PbDl Download
1 -> PHBK Prepare is received and device prepares for PbDl Download         
*
* RETURNVALUE: None                  
*
*******************************************************************************/
tVoid FC_Device_Details::setPhBkPrepareCallBackStatus(tBool setPrepCBStatus)
{
   ETG_TRACE_USR2(( " setPhBkPrepareCallBackStatus: Current value of m_bPrepareCBStatus= %d, New value= %d", m_bPrepareCBStatus, setPrepCBStatus ));
   m_bPrepareCBStatus = setPrepCBStatus;
}
//End of fix

//FIX CMG3G-8367 IS2424_DownloadOnOff behaviour@FC_Phonebook
// The user shall have possibility to switch the phonebook and call history list download On or Off individual for each paired phone device.
/*******************************************************************************
 *
 * FUNCTION: FC_Device_Details::getDownloadOnOffStatus()
 *
 * DESCRIPTION: Returns the DownloadOnOffStatus for the device object.
 *
 * PARAMETER: None
 *
 * RETURNVALUE: DownloadOnOffStatus
 *
 *******************************************************************************/
tU16 FC_Device_Details::getDownloadOnOffStatus()
{
   ETG_TRACE_USR4((" getDownloadOnOffStatus entered in FC_Device_Details"));
   return this->m_downloadOnOffStatus;
}

/*******************************************************************************
 *
 * FUNCTION: FC_Device_Details::setDownloadOnOffStatus()
 *
 * DESCRIPTION: Set the DownloadOnOffStatus for the device object.
 *
 * PARAMETER: DownloadOnOffStatus
 *
 * RETURNVALUE: None
 *
 *******************************************************************************/
tVoid FC_Device_Details::setDownloadOnOffStatus(tU16 downloadOnOffStatus)
{
   ETG_TRACE_USR4((" setDeviceHandle: DownloadOnOffStatus= %d ", downloadOnOffStatus));
   this->m_downloadOnOffStatus = downloadOnOffStatus;
}
//End of fix FIX CMG3G-8367


// -bn: 2016.01: DL speed improvement by collecting contacts before insertion to database
/***************************************************************************************
*
* FUNCTION:  FC_PhoneBook_SQLite::FindExactMatchingContact_inRAMList
*
* DESCRIPTION: Finds an exact matching contact in RAM list for the given contact
*
*    Note: This method will be called when parsing vCards from a connected 'Phone device'.
*          The decision for a match is based on relevant APL1 attributes.
*          In addition the photo (APL2 attribute) is checked if it matches also and the related
*          return values 'bPhotoMatches' and 'u32photoSizeOld' are set accordingly.
*          (Hint: This method is not intended to be used for 'Vechicle Contacts'
*           as these contacts have additional attributes like 'category' and 'notes' ..)
*
* PARAMETER: IN   ContactDetail: Contact to be searched in the RAM list
*            OUT  bPhotoMatches: true if Photos match also, false otherwise
*            OUT  u32photoSizeOld: Photo Size of matching contact in RAM list
*
* RETURNVALUE: true if a matching contact (based on APL1 attributes) has been found 
*              in RAM list, false otherwise
*
*****************************************************************************************/
bool FC_Device_Details::FindExactMatchingContact_inRAMList(const FC_PhoneBook_Contact_Detail& ContactDetail,
                                                           bool& bPhotoMatches, quint32& u32photoSizeOld)
{
   bool bContactFound = false;
   QTime DebugTime;
   DebugTime.start();

   qint32 u32Idx = 0;
   while( (u32Idx < m_NewContactsToBeAddedRAMList.size()) && (bContactFound == false) )
   {
      FC_PhoneBook_Contact_Detail contact = m_NewContactsToBeAddedRAMList[u32Idx];

      if(contact.crcAPL1 == ContactDetail.crcAPL1)
      {
         if( true == contact.bSubsetCompare(ContactDetail) )
         {
            ETG_TRACE_USR4(( " Contacts are equal regarding to 'relevant' APL1 attributes. (Note: PHOTO details are not checked so far.) " ));
            bContactFound   = true;
            u32photoSizeOld = contact.photoSize;

            ETG_TRACE_USR4(( " RAM contact.photoDlSize= %d,  Downloaded ContactDetail.photoDlSize [byte]= %5u ", contact.photoDlSize, ContactDetail.photoDlSize ));
            ETG_TRACE_USR4(( " RAM contact.photoDlCRC= %d,  Downloaded ContactDetail.photoDlCRC= %u ", contact.photoDlCRC,  ContactDetail.photoDlCRC ));
            if(contact.photoDlSize == ContactDetail.photoDlSize)
            {
               ETG_TRACE_USR4(( " PHOTOs are equal (based on their size), now check their CRC hash " ));

               // Now check equivalence of 'PhotoDlCRC'
               if( contact.photoDlCRC == ContactDetail.photoDlCRC)
               {
                  bPhotoMatches = true;
                  break;
               }
               else
               {
                  ETG_TRACE_USR4(( " PHOTOs are NOT equal (based on their CRC hash)" ));
                  bPhotoMatches = false;
                  break;
               }
            }
            else
            {
               ETG_TRACE_USR4(( " PHOTOs are NOT equal (based on their sizes)" ));
               bPhotoMatches = false;
               break;
            }
         }
      }

      u32Idx++;
   }

   ETG_TRACE_USR2(( " -PBDL.Perf- 'FindExactMatchingContact': exited after %d ms, [RAM] ", DebugTime.elapsed() ));
   return bContactFound;
}


/*******************************************************************************
 *
 * FUNCTION: FC_Device_Details::bIsCancelPBRequested()
 *
 * DESCRIPTION: Get m_bIsCancelPbDownload.
 *
 * PARAMETER: None
 *
 * RETURNVALUE: m_bIsCancelPbDownload
 *
 *******************************************************************************/
//get/set Cancel contacts download
tBool FC_Device_Details::bIsCancelPBRequested()
{
   ETG_TRACE_USR4((" bIsCancelPBRequested m_bIsCancelPbDownload: %d", m_bIsCancelPbDownload));
   return m_bIsCancelPbDownload;
}


/*******************************************************************************
 *
 * FUNCTION: FC_Device_Details::setCancelPBDownload()
 *
 * DESCRIPTION: Set m_bIsCancelPbDownload
 *
 * PARAMETER: None
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid FC_Device_Details::setCancelPBDownload(tBool bIsCancelPbDownload)
{
   ETG_TRACE_USR4((" setCancelPBDownload: bIsCancelPbDownload= %d ", bIsCancelPbDownload));
   m_bIsCancelPbDownload = bIsCancelPbDownload;
}

/*******************************************************************************
 *
 * FUNCTION: FC_Device_Details::setDeviceProbablyLocked()
 *
 * DESCRIPTION: Sets whether device is locked or not
 *
 * PARAMETER: [IN] tBool bDeviceLockStatus
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid FC_Device_Details::vSetDeviceProbablyLocked(tBool bDeviceLockStatus)
{
   ETG_TRACE_USR4((" vSetDeviceProbablyLocked: bDeviceLockStatus = %d ", bDeviceLockStatus));
   m_bDeviceProbablyLocked = bDeviceLockStatus;
}

/*******************************************************************************
 *
 * FUNCTION: FC_Device_Details::bIsDeviceProbablyLocked()
 *
 * DESCRIPTION: Get device locked status
 *
 * PARAMETER: None
 *
 * RETURNVALUE: m_bDeviceProbablyLocked
 *
 *******************************************************************************/
tBool FC_Device_Details::bIsDeviceProbablyLocked()
{
   ETG_TRACE_USR4((" bIsDeviceProbablyLocked : m_bDeviceProbablyLocked = %d", m_bDeviceProbablyLocked));
   return m_bDeviceProbablyLocked;
}

/*******************************************************************************
 *
 * FUNCTION: FC_Device_Details::setSuspiciousGetSize()
 *
 * DESCRIPTION:
 *
 * PARAMETER: [IN] tBool bSuspiciousGetSizeStatus
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid FC_Device_Details::vSetSuspiciousGetSize(tBool bSuspiciousGetSizeStatus)
{
   ETG_TRACE_USR4((" vSetSuspiciousGetSize: bSuspiciousGetSizeStatus = %d ", bSuspiciousGetSizeStatus));
   m_bSuspiciousGetSize = bSuspiciousGetSizeStatus;
}

/*******************************************************************************
 *
 * FUNCTION: FC_Device_Details::bGetSuspiciousGetSize()
 *
 * DESCRIPTION: Get suspicious getsize status
 *
 * PARAMETER: None
 *
 * RETURNVALUE: m_bDeviceProbablyLocked
 *
 *******************************************************************************/
tBool FC_Device_Details::bGetSuspiciousGetSize()
{
   ETG_TRACE_USR4((" bGetSuspiciousGetSize : m_bSuspiciousGetSize = %d", m_bSuspiciousGetSize));
   return m_bSuspiciousGetSize;
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::GetMissedcallCount()
*
* DESCRIPTION: Returns the Missed call count for the deviceObject.
*
* PARAMETER: None
*
* RETURNVALUE: deviceHandle.
*
*******************************************************************************/
tU32  FC_Device_Details::GetMissedcallCount()
{
   ETG_TRACE_USR4(( " GetMissedcallCount= %d", this->m_MissedcallCount ));
   return this->m_MissedcallCount;
}


/*******************************************************************************
*
* FUNCTION: FC_Device_Details::vSetMissedcallCount()
*
* DESCRIPTION: Set the Missed call count for the deviceObject.
*
* PARAMETER: deviceHandle
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid  FC_Device_Details::vSetMissedcallCount(tU32 missedcallCount)
{
   ETG_TRACE_USR4(( " vSetMissedcallCount: missedcallCount= %d ", missedcallCount ));
   this->m_MissedcallCount = missedcallCount;
}

/*******************************************************************************
*
* FUNCTION: FC_Device_Details::vSwapMissedCallDetails()
*
* DESCRIPTION: Swaps the missed call details from Recent to First.
*
* PARAMETER: deviceHandle
*
* RETURNVALUE: None.
*
*******************************************************************************/
tVoid FC_Device_Details::vSwapMissedCallDetails()
{
   ETG_TRACE_USR4(( " vSwapMissedCallDetails entered "));

   this->m_sFirstMissedCallDate = this->m_sRecentMissedCallDate;
   this->m_sFirstMissedCallTime = this->m_sRecentMissedCallTime;

}

//Fix NCG3D-65982
/*******************************************************************************
 *
 * FUNCTION: FC_Device_Details::vSetDeviceDownloadSuspended()
 *
 * DESCRIPTION: Sets whether device download is suspended or not
 *
 * PARAMETER: [IN] tBool bDownloadSuspendedStatus
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
tVoid FC_Device_Details::vSetDeviceDownloadSuspended(tBool bDownloadSuspendedStatus)
{
   ETG_TRACE_USR4((" vSetDeviceDownloadSuspended: bDownloadSuspendedStatus = %d ", bDownloadSuspendedStatus));
   m_bDeviceDownloadSuspended = bDownloadSuspendedStatus;
}

/*******************************************************************************
 *
 * FUNCTION: FC_Device_Details::bIsDeviceDownloadSuspended()
 *
 * DESCRIPTION: Get device download suspended status
 *
 * PARAMETER: None
 *
 * RETURNVALUE: m_bDeviceDownloadSuspended
 *
 *******************************************************************************/
tBool FC_Device_Details::bIsDeviceDownloadSuspended()
{
   ETG_TRACE_USR4((" bIsDeviceDownloadSuspended : m_bDeviceDownloadSuspended = %d", m_bDeviceDownloadSuspended));
   return m_bDeviceDownloadSuspended;
}
//End fix 65982
