/*******************************************************************************
*
* FILE:          FC_PhoneBook_SQLite.h
*
* SW-COMPONENT:  FC_PhoneBook application
*
* PROJECT:
*
* DESCRIPTION:   Handles the sqlite database using QtSql module. This is the
*                real class where all the Qt/QSql work is done.
* AUTHOR:
*
* COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
*
*******************************************************************************/

#ifndef _FC_PHONEBOOK_SQLITE_
#define _FC_PHONEBOOK_SQLITE_

#include <QtCore/QtCore>
#include <QtSql/QtSql>
#define SYSTEM_S_IMPORT_INTERFACE_STRING
#define SYSTEM_S_IMPORT_INTERFACE_MAP

#include "FC_PhoneBook_Usage.h"
#include "../../Configuration/PhoneBook/Pbk_FeatureConfig.h"
#include "../Common/FC_PhoneBook_Global.h"

#define FC_PB_ERROR_INTERNAL_FAILURE          (quint16)0x01
#define FC_PB_ERROR_CONTACT_HANDLE_INVALID    (quint16)0x02
#define FC_PB_ERROR_DATABASE_NOT_OPEN         (quint16)0x05
#define FC_PB_ERROR_DEVICE_NOT_IN_LIST    	  (quint16)0x03
#define FC_PB_ERROR_CODE_UNKNOWN              (quint16)0x00

#define FC_PB_CNT_HANDLE_GEN_RETRY_COUNT      (quint16)50
#define FC_PB_PHONEBOOK_CNT_HANDLE_OFFSET     (quint32)500  //First 500 contact handles reserved for call history
#define FC_PB_DUMMY_TABLE_INDEX               (quint8)20


#define FC_PB_TOTAL_NUMBER_OF_PB      11
#define FC_PB_TOTAL_NUMBER_OF_CH      2
#define FC_PB_TOTAL_DEVICE_FEATURE_SUPPORT 1
#define FC_PB_TOTAL_NUMBER_OF_VPB      10 //Maximum Number of paired devices allowed

#define FC_PB_DNL_CALL_HISTORY_TYPES             (4)

#define FC_PB_DEVICE_TMP        0x00   // CMG3G-7702
#define FC_PB_VEHICLE_PBOOK     0x01
#define FC_PB_DEVICE_HANDLE2    0x02
#define FC_PB_DEVICE_HANDLE3    0x03
#define FC_PB_DEVICE_HANDLE4    0x04
#define FC_PB_DEVICE_HANDLE5    0x05
#define FC_PB_DEVICE_HANDLE6    0x06
#define FC_PB_DEVICE_HANDLE7    0x07
#define FC_PB_DEVICE_HANDLE8    0x08
#define FC_PB_DEVICE_HANDLE9    0x09
#define FC_PB_DEVICE_HANDLE10   0x0A
#define FC_PB_DEVICE_HANDLE11   0x0B
#define FC_PB_CALLHIST_DEVICE0  0x0C
#define FC_PB_CALLHIST_DEVICE1  0x0D
#define FC_PB_FAVORITES         0x0E
#define FC_PB_FS         		0x0F

#define VALUE_ZERO (0)
#define VALUE_ONE (1)
#define VALUE_TWO (2)


#define DELETE_ALL_FAVORITE 			0

#define MAX_AUTO_COMPLETE_LISTS 2
#define MAX_SIZE_SEARCH_RESULT_LIST  30


#define FC_VPB_DEVICE_HANDLE2    0x02
#define FC_VPB_DEVICE_HANDLE3    0x03
#define FC_VPB_DEVICE_HANDLE4    0x04
#define FC_VPB_DEVICE_HANDLE5    0x05
#define FC_VPB_DEVICE_HANDLE6    0x06
#define FC_VPB_DEVICE_HANDLE7    0x07
#define FC_VPB_DEVICE_HANDLE8    0x08
#define FC_VPB_DEVICE_HANDLE9    0x09
#define FC_VPB_DEVICE_HANDLE10   0x0A
#define FC_VPB_DEVICE_HANDLE11   0x0B

#define FC_PB_CONTACT_HANDLE_DPB_MARKER       (quint32)0x00800000 //BT Phone Contact list (24th bit = 1)

class most_fi_tcl_e8_PhonBkContactListType;
class FC_PhoneBook_Contact_Detail_Short
{
public:
   quint32 contactHandle;
   QString firstName;
   QString lastName;
   QString sortName;   // Hotfix for CRQ 509734 Phonetic Sorting
   QString poiName;
   quint8 phoneNumberCount; //FIX SUZUKI-1114 API for numeric value to indicate number of dial-able phone numbers available to a contact item in Phonebook
   bool bQuickDial;
   bool bQuickRoute;
   bool bQuickText;

   FC_PhoneBook_Contact_Detail_Short()
      : contactHandle(0),
      firstName(""),
      lastName(""),
      sortName(""),   // Hotfix for CRQ 509734 Phonetic Sorting
      poiName(""),
      phoneNumberCount(0), //FIX SUZUKI-1114 API for numeric value to indicate number of dial-able phone numbers available to a contact item in Phonebook
      bQuickDial(false),
      bQuickRoute(false),
      bQuickText(false)
   {
   }
};

struct PbkAdditionalContactInfo
{
   QString PhoneNum1;
   quint8 NumberType1;
   QString PhoneNum2;
   quint8 NumberType2;
   QString PhoneNum3;
   quint8 NumberType3;
   QString PhoneNum4;
   quint8 NumberType4;
   QString PhoneNum5;
   quint8 NumberType5;
   QString PhoneNum6;
   quint8 NumberType6;
   QString PhoneNum7;
   quint8 NumberType7;
   QString PhoneNum8;
   quint8 NumberType8;
   QString EmailAdd1;
   QString EmailAdd2;
   QString EmailAdd3;
   quint8 u8EmailAddrCount;
   quint8 u8AddressesCount;
   bool bPhotoAvailable;
   quint8 u8ContactType;
};

class FC_PhoneBook_Contact_Detail
{
public:
   quint32 contactHandle;                 // DB value   (DB = Phonebook DataBase)
   quint32 u32contactPosition;            // DB value
   QString firstName;                     // DB value
   QString lastName;                      // DB value
   QString Phonetics;                     // DB value   //292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
   QString SortFirstName;                 // DB value   // NEW
   QString SortLastName;                  // DB value   // NEW
   QString poiName;                       // DB value

   QString PhoneNum1;                     // DB value
   QString PhoneNum2;                     // DB value
   QString PhoneNum3;                     // DB value
   QString PhoneNum4;                     // DB value
   QString PhoneNum5;                     // DB value
   QString PhoneNum6;                     // DB value
   QString PhoneNum7;                     // DB value
   QString PhoneNum8;                     // DB value

   quint8 NumberType1;                    // DB value
   quint8 NumberType2;                    // DB value
   quint8 NumberType3;                    // DB value
   quint8 NumberType4;                    // DB value
   quint8 NumberType5;                    // DB value
   quint8 NumberType6;                    // DB value
   quint8 NumberType7;                    // DB value
   quint8 NumberType8;                    // DB value

   QString PPPhoneNum1;                   // DB value
   QString PPPhoneNum2;                   // DB value
   QString PPPhoneNum3;                   // DB value
   QString PPPhoneNum4;                   // DB value
   QString PPPhoneNum5;                   // DB value
   QString PPPhoneNum6;                   // DB value
   QString PPPhoneNum7;                   // DB value
   QString PPPhoneNum8;                   // DB value

   QString NumberSubstr1;                 // DB value   // NEW
   QString NumberSubstr2;                 // DB value   // NEW
   QString NumberSubstr3;                 // DB value   // NEW
   QString NumberSubstr4;                 // DB value   // NEW
   QString NumberSubstr5;                 // DB value   // NEW
   QString NumberSubstr6;                 // DB value   // NEW
   QString NumberSubstr7;                 // DB value   // NEW
   QString NumberSubstr8;                 // DB value   // NEW

   QString emailAdd1;                     // DB value
   QString emailAdd2;                     // DB value
   QString emailAdd3;                     // DB value
   QString homeAdd;                       // DB value
   QString workAdd;                       // DB value
   QString otherAdd;                      // DB value
   QString category;                      // DB value
   qint8   contactType;                   // DB value
   QString notes;                         // DB value

   tBool   homeGeoCodeLatLongValidity;    // DB value
   qint32  homeGeoCodeLatitude;           // DB value
   qint32  homeGeocodeLongitude;          // DB value
   tBool   workGeoCodeLatLongValidity;    // DB value
   qint32  workGeoCodeLatitude;           // DB value
   qint32  workGeocodeLongitude;          // DB value
   tBool   otherGeoCodeLatLongValidity;   // DB value
   qint32  otherGeoCodeLatitude;          // DB value
   qint32  otherGeocodeLongitude;         // DB value

   quint32 photoSize;                     // DB value   // Photo Size in bytes of Photo stored in DB
   quint32 photoDlSize;                   // DB value   // -FastPhotoHandling- : Photo Download Size in bytes, i.e. size of Photo as it is received in vCard
   QByteArray photo;                      // DB value   
   QString photoType;                     // DB value   // -FastPhotoHandling- : Photo encoding type, (e.g. JPG, GIF or PNG) of Photo stored in DB
   quint16 photoCRC;                      // DB value   // -FastPhotoHandling- : CRC-Hash value on the Photo stored in DB
   quint16 photoDlCRC;                    // DB value   // -FastPhotoHandling- : CRC-Hash value on the DL Photo ("Download Photo"), i.e. CRC from the Photo as it is received in vCard
   quint32 APL1Size;                      // DB value   // -FastContactSearch- : Size in bytes of APL1 contact attributes in DB
   quint16 crcAPL1;                       // DB value   // -FastContactSearch- : CRC-Hash value on "relevant" APL1 contact attributes

   bool bQuickDial;
   bool bQuickRoute;
   bool bQuickText;

   quint8 phoneNumberCount;    //FIX SUZUKI-1114 API for numeric value to indicate number of dial-able phone numbers available to a contact item in Phonebook
   quint8 otherType;           // Number type assigned for the other number
   bool bPreferredPhonetics;   // Indicates if the "Phonetics" information of a contact stems from X-PHONETIC-FIRST-NAME / X-PHONETIC-LAST-NAME (which is preferred)

#ifdef VALID_VCARD_FORMAT_PHONENUMBER
   bool isValid(FC_PhoneBook_Contact_Detail& contact);
#endif
#ifdef VALID_VCARD_FORMAT_ADDRESS_PHONENUMBER
   bool isValidVcardFormatPhoneNumberAddress(FC_PhoneBook_Contact_Detail& contact);//Coverity Fix-Pass By reference
#endif

   FC_PhoneBook_Contact_Detail() :
      contactHandle(0),
      u32contactPosition(0),
      firstName(""),
      lastName(""),
      Phonetics(""),
      SortFirstName(""),
      SortLastName(""),
      poiName(""),

      PhoneNum1(""),
      PhoneNum2(""),
      PhoneNum3(""),
      PhoneNum4(""),
      PhoneNum5(""),
      PhoneNum6(""),
      PhoneNum7(""),
      PhoneNum8(""),

      NumberType1(0),
      NumberType2(0),
      NumberType3(0),
      NumberType4(0),
      NumberType5(0),
      NumberType6(0),
      NumberType7(0),
      NumberType8(0),

      PPPhoneNum1(""),
      PPPhoneNum2(""),
      PPPhoneNum3(""),
      PPPhoneNum4(""),
      PPPhoneNum5(""),
      PPPhoneNum6(""),
      PPPhoneNum7(""),
      PPPhoneNum8(""),

      NumberSubstr1(""),
      NumberSubstr2(""),
      NumberSubstr3(""),
      NumberSubstr4(""),
      NumberSubstr5(""),
      NumberSubstr6(""),
      NumberSubstr7(""),
      NumberSubstr8(""),

      emailAdd1(""),
      emailAdd2(""),
      emailAdd3(""),
      homeAdd(""),
      workAdd(""),
      otherAdd(""),
      category(""),
      contactType(0),
      notes(""),

      homeGeoCodeLatLongValidity(false),
      homeGeoCodeLatitude(0),
      homeGeocodeLongitude(0),
      workGeoCodeLatLongValidity(false),
      workGeoCodeLatitude(0),
      workGeocodeLongitude(0),
      otherGeoCodeLatLongValidity(false),
      otherGeoCodeLatitude(0),
      otherGeocodeLongitude(0),

      photoSize(0),    // Photo Size in bytes of Photo stored in DB
      photoDlSize(0),  // -FastPhotoHandling- : Photo Download Size in bytes, i.e. size of Photo as it is received in vCard
      photo(""),
      photoType(""),   // -FastPhotoHandling- : Photo encoding type, (e.g. "JPG", "GIF" or "PNG") of Photo stored in DB
      photoCRC(0),     // -FastPhotoHandling- : CRC-Hash value on the Photo stored in DB
      photoDlCRC(0),   // -FastPhotoHandling- : CRC-Hash value on the DL Photo ("Download Photo"), i.e. CRC from the Photo as it is received in vCard
      APL1Size(0),     // -FastContactSearch- : Size in bytes of APL1 contact attributes in DB
      crcAPL1(0),      // -FastContactSearch- : CRC-Hash value on "relevant" APL1 contact attributes

      bQuickDial(false),
      bQuickRoute(false),
      bQuickText(false),

      phoneNumberCount(0), //FIX SUZUKI-1114 API for numeric value to indicate number of dial-able phone numbers available to a contact item in Phonebook
      otherType(0),
      bPreferredPhonetics(false) //292459 - Japanese Yomi Phonetic Transcription
   {
   }

   bool bSubsetCompare(const FC_PhoneBook_Contact_Detail& contact);

   unsigned GetContactLength()
   { 
      return (unsigned)(4   // ContactHandle length (1x tU32)
         + firstName.length() + lastName.length() + poiName.length() 
         + PhoneNum1.length() + PhoneNum2.length() + PhoneNum3.length() + PhoneNum4.length() + PhoneNum5.length() + PhoneNum6.length()
         + PhoneNum7.length() + PhoneNum8.length()
         + emailAdd1.length() + emailAdd2.length() + emailAdd3.length() 
         + homeAdd.length() + workAdd.length() + otherAdd.length() + category.length() + notes.length() 
         + 27   // geo-coordinates length (3x [1x tU8 + 1x tU32 + tU32] = 3x [9 byte] = 27 
         + photoType.length()
         + 2       // -FastPhotoHandling- : 'photoDlCRC'
         + 4    // -FastContactSearch- : 'APL1Size' (tU32)
         + 2    // -FastContactSearch- : 'crcAPL1' (tU16)
      );
   }

};


#define FC_PB_TOTAL_EMAILID           3
#define PHONENUMBER_NUM_INDEX_IN_PB   5
#define PHONENUMBER_NUM_INDEX_IN_VPB   6
#define FC_PHONEBOOK_EMAIL_ID1_INDEX  29

class FC_PhoneBook_Callhistory_Detail
{
public:
   quint32 contactHandle;
   quint8  callType;
   QString firstName;
   QString lastName;
   QString phoneNumber;
   QString PPphoneNumber;
   quint8  numberType;
   QString dateStamp;
   QString timeStamp;
   quint8  callCount; //FIX SUZUKI-1113 API for numeric value to indicate number of calls against a CallHistory item in Phonebook

   FC_PhoneBook_Callhistory_Detail() :
      contactHandle(0), 
      callType(0),
      firstName(""),
      lastName(""),
      phoneNumber(""),
      PPphoneNumber(""),
      numberType(9),  // TODO: Check 9
      dateStamp(""),
      timeStamp(""),
      callCount(0)
   {
   }
};


struct FC_PhoneBook_Search_Result
{
   quint32 contactHandle;
   QString firstname;
   QString lastname;
   quint8  contactdetailtype;

};

struct AutoCompletePhoneNumberEntryResult
{
   QString FirstName;
   QString LastName;
   QString PhoneNumber;
};

struct SearchContacts_Result
{
   QString ReportedFirstName;
   QString ReportedLastName;
   quint32 ContactHandle;
   QString PhoneNum1;
   quint8 NumberType1;
   QString PhoneNum2;
   quint8 NumberType2;
   QString PhoneNum3;
   quint8 NumberType3;
   QString PhoneNum4;
   quint8 NumberType4;
   QString PhoneNum5;
   quint8 NumberType5;
   QString PhoneNum6;
   quint8 NumberType6;
   QString PhoneNum7;
   quint8 NumberType7;
   QString PhoneNum8;
   quint8 NumberType8;
   QString EmailAdd1;
   QString EmailAdd2;
   QString EmailAdd3;
   quint8 u8EmailAddrCount;
   quint8 u8AddressesCount;
   qint8 u8phoneNumberCount;
   bool bPhotoAvailable;

};

/******************************************************************************/
/*                                                                            */
/* CLASS DEFINITION                                                           */
/*                                                                            */
/******************************************************************************/
class FC_PhoneBook_Fav_Detail
{
public:
   quint16 favoriteID;
   quint8  devicehandle;
   quint32 contacthandle;
   quint8  tagvalue;
   QString labeltext;
   QString phonenumber;
   quint8  activestatus;
   quint8  availablestatus;

   FC_PhoneBook_Fav_Detail():favoriteID(0),devicehandle(0),
      contacthandle(0), tagvalue(0), activestatus(0), availablestatus(0)
   {
   }

   ~FC_PhoneBook_Fav_Detail()
   {
   }
};

class FC_PhoneBook_FeatureSupport_Detail
{
public:
   QString m_address;
   quint8  m_devicehandle;
   quint8  m_cchsupport;
   quint8  m_ichsupport;
   quint8  m_ochsupport;
   quint8  m_mchsupport;
   quint8  m_conatctSortSupport;
   quint8  m_pbSupport;
   quint32 m_LastConnectionTimeElapsed;
   bool    m_DlCompletedOnce;
   quint16  m_SwitchDownloadOnOffState;
   bool    m_AutomaticContactDL;
   quint8  m_PreferredSortOrder;

   FC_PhoneBook_FeatureSupport_Detail():m_address(""),m_devicehandle(0),
      m_cchsupport(0), m_ichsupport(0), m_ochsupport(0), m_mchsupport(0), m_conatctSortSupport(0),m_pbSupport(0),m_LastConnectionTimeElapsed(0),m_DlCompletedOnce(false),m_SwitchDownloadOnOffState(0),m_AutomaticContactDL(true),m_PreferredSortOrder(0)
   {
   }

   ~FC_PhoneBook_FeatureSupport_Detail()
   {
   }
};

class FC_PhoneBook_SQLite
{
public:
   quint8 u8GetLRUDeviceHandle();
   bool bGetDlCompletedOnceValue(quint8 u8DeviceHandle);
   bool bUpdateDlCompletedOnceValue(quint8 u8DeviceHandle, bool bValue);
   bool bUpdateDeviceLastConnectionTimeElapsed(
      quint8 f_u8DeviceHandle);

   enum SortType
   {
      LastName,
      FirstName,
      Address_LastName,
      Address_FirstName,
      Number_LastName,
      Number_FirstName,
      Email_LastName,
      Email_FirstName,
      DateTime,
      Position,
      Invalid
   };

   enum SearchType
   {
      PhoneNumber,
      EmailID         // CMG3G-5150 LINT Fix -Info 726: Extraneous comma ignored
   };

   enum enContactDetailType
   {
      None=0,
      PrefferedNum,
      CellNumber,
      HomeNumber,
      WorkNumber,
      OtherNumber,
      EmailAdd1      =  9,  // Note: The 'EmailAdd1' could start at any value, which does not interfere with other enumerations in this file.
      EmailAdd2      = 10,
      EmailAdd3      = 11
   };

   enum FC_PhoneBook_enCallHistoryType
   {
      callhisttype_combined=0,
      callhisttype_missed,
      callhisttype_incoming,
      callhisttype_outgoing,
      callhisttype_invalid
   };

   enum enListType
   {
      ListType_PHONEBOOK=0,
      ListType_FAVORITE,
      ListType_VEHICLE,
      ListType_UNKNOWN,
      ListType_ALL
   };

#define TOTAL_CONTACT_DETAIL_TYPES 11

   bool InsertRecords( QList<FC_PhoneBook_Contact_Detail>& contacts,
      quint8 devicehandle, QList<quint32>& ContactHandleList );
   bool InsertCallHistoryRecords(QList<FC_PhoneBook_Callhistory_Detail>& Records, quint8 ConnectionID);
   void GetRecords(tU8 Listtype,SortType sortType, quint8 deviceHandle, QList<FC_PhoneBook_Contact_Detail_Short>& list);//List is passed as reference
   void GetCallHistoryRecords( SortType enSorttype, quint8 u8CallType,
      quint8 u8ConnectionID, QList<FC_PhoneBook_Callhistory_Detail>& callhistoryList, bool bDeviceDateTimeSupport );
   // Fix for CMG3GB-621
   QList<FC_PhoneBook_Callhistory_Detail> GetAllCallHistoryRecords(quint8 u8ConnectionID);
   // End of fix
   bool GetMatchingRecord(FC_PhoneBook_Search_Result& searchresult, const char* searchkey,
      SearchType searchtype, quint8 devicehandle);
  /* bool GetMatchingRecord(QList<FC_PhoneBook_Search_Result> &searchresult,
      const char* searchkey, SearchType searchtype, quint8 devicehandle);*/
   bool InsertSingleRecord(FC_PhoneBook_Contact_Detail& contact, quint8 devicehandle, quint32& contacthandle, bool bIsVehiclePbk);//CMG3G-14313-Coverity Fix-Pass by Reference
//   bool UpdateContactRecord(FC_PhoneBook_Contact_Detail& contact, quint32 contacthandle, bool bIsPhotoDiscarded);//CMG3G-14313-Coverity Fix-Pass by Reference   // TODO: -bn: I have not found where this method is called (on Connectivity stabi branch) and therefore commented it out. Is it needed on feature branch?
   bool UpdateRecord(FC_PhoneBook_Contact_Detail& contact, quint32 contacthandle);//CMG3G-14313-Coverity Fix-Pass by Reference
   bool UpdateCallHistoryRecord(FC_PhoneBook_Callhistory_Detail contact, quint32 contacthandle,
      quint8 ConnectionID);
   bool DeleteRecord(quint32 contacthandle,quint8 tableindex);

   bool SearchPhoneNumberEntries(QString PhNumberToSearch,quint8 Index,quint8 ConnectionID);

   bool GetContactDetails(quint32 contacthandle ,FC_PhoneBook_Contact_Detail &contact);
   bool DeleteAllContactsAndCallHistoryFromDB();//FIX GMMY16-23858
   bool DeletePhoneBookContacts(quint8 deviceHandle);
   bool DeleteVehiclePhoneBook(tU8 u8DeviceHandle);
   bool DeleteRecordsOfDevice(quint8 u8TableIndex);
   bool DeleteCallHistoryRecord(quint8 u8ConnectionID, quint32 u32ContactHandle);
   bool GetCallHistoryRecordDetails(quint32 contacthandle , quint8 u8ConnectionID,
      QList<FC_PhoneBook_Callhistory_Detail>& RecordList);
   bool InsertCallHistoryRecords(FC_PhoneBook_Callhistory_Detail& Record,
      quint8 ConnectionID, quint32& ContactHandle);
   bool FindExactMatchingContact(const FC_PhoneBook_Contact_Detail& ContactDetail, quint8 u8DeviceHandle, 
      quint32& u32ContactHandle, bool& bPicStatus, quint32& u32photoSizeOld);
   bool DeleteCallHistoryRecordsOfCallType(quint8 u8ConnectionID,
      quint8 u8CallType= callhisttype_invalid);
   bool GetContactHandles(quint8 u8DeviceHandle,
      QList<quint32>& u32ContactHandleList);
   bool bGetVPBContactHandles(quint8 u8DeviceHandle,
      QList<quint32>& u32ContactHandleList);
   bool DeleteContactHandles(quint8 u8DeviceHandle,
      QList<quint32> ContactHandleList);
   bool InsertTmpToDeviceTable(quint8 u8DeviceHandle);   //CMG3G-7702
   bool DropTableFromDB(quint8 u8TableIndex);            //CMG3G-7702
   bool DropVPBTableFromDB(quint8 u8TableIndex);
   bool DeleteContentsOfTable(quint8 u8TableIndex);      //CMG3G-7702
   bool DeleteCallHistoryRecords(quint8 u8ConnectionID);
   bool GetPhoto( quint32 u32ContactHandle, QByteArray& PhotoData, QString& qsPhotoType, quint16& PhotoDlCRC, quint16& u16ErrorCode );
   quint8 GetDeviceHandleForContacthandle(quint32 u32ContactHandle);

   QList<AutoCompletePhoneNumberEntryResult> AutoCompletePhNoSearchRecords[MAX_AUTO_COMPLETE_LISTS];
   QString SearchedPhoneNumberDigit[MAX_AUTO_COMPLETE_LISTS];
   bool InsertFavRecord(FC_PhoneBook_Fav_Detail fav);
   bool GetAllFavoritesID(QList<quint16>& u16FavoriteIdList);
   bool GetAllFavoritesIDForDevice(QList<quint16>& u16FavoriteIdList, quint8 DeviceHandle);
   bool GetFavoritesIDForContact(QList<quint16>& u16FavoriteIdList, quint32 ContactHandle);
   bool GetFavRecord(quint16 u16FavoriteId, FC_PhoneBook_Fav_Detail& fav, quint16& u16ErrCode);
   //FIX GMNGA-56482 Favorite not highlighted when Call active
   bool GetFavouritesIDForPhoneNumber(QList<quint16>& u16FavoriteIdList, QString sTelephoneNumber);
   bool DeleteFavRecord(quint32 u16FavoriteId, int fResetDeviceSpecificFav = false, quint8 fU8DeviceHandle = 0xff); 
   bool DeleteAllFavRecordsOfDevice(quint8 u8DeviceHandle);  // ---: RB fix proposal for GMNGA-53014 and GMMY15-248
   bool UpdateFeatureSupportRecord(FC_PhoneBook_FeatureSupport_Detail fs);
   bool GetFeatureSupportforDevice(QString device_address, quint8 device_Handle, FC_PhoneBook_FeatureSupport_Detail& fs, quint16& u16ErrCode);
   bool DeletetFeatureSupportRecord(QString device_address,quint8 device_Handle);
   bool UpdateSortListView(quint8 u8DeviceHandle);

   void SpeechUpdateCheckSumTable(quint8 u8DeviceHandle,bool bUpdateVPBx = false);
   bool MakeAllFavoritesUnavailable();
   bool MakeAllFavoritesInactive();
   quint32 GetFavoritesCount();
   quint32 GetFavoritesCount(quint8 DeviceHandle);
   bool GetFavoriteRecordsOfContactHandle(quint32 u32ContactHandle,
      QList<FC_PhoneBook_Fav_Detail>& FavInfoList);
   bool UpdateFavoritesAvailableStatus(QList<quint16> FavoritesIDList,
      quint8 AvailStatus);
   bool UpdateFavoritesRecord(FC_PhoneBook_Fav_Detail FavDetailList);
   bool GetFavoriteRecordsOfDeviceHandle(quint8 u8DeviceHandle,
      QList<FC_PhoneBook_Fav_Detail>& FavInfoList);
   bool UpdateFavoritesActiveStatus(quint16 FavoritesID, quint8 ActiveStatus);
   bool UpdateAllFavoritesActiveStatus(quint8 u8Status);

   quint32 GetNoOfContactsInVAB(quint8 u8DeviceHandle);
   quint32 GetNoOfContactsInVPB(quint8 u8DeviceHandle);
   quint32 GetNoOfContactsInCHtable(quint8 u8ConnectionID); // -SP,31.05.2019 :: Callhistory Efficient Resolving

   void IncNoOfContactsVAB(quint8 u8DeviceHandle);
   bool DecNoOfContactsVAB(quint8 u8DeviceHandle);
   void InitializeDatabase();
   void CloseDatabase();
   bool GetFavInfoForFavorite( quint16 u16FavoriteId, QList<FC_PhoneBook_Fav_Detail> & , quint16& u16ErrCode);
   // MV-BugzID: 207609
   bool GetMatchingRecordBasedOnName(
      FC_PhoneBook_Search_Result& searchresultlist,
      const char* firstname, const char* lastname, const char* number, quint8 devicehandle);

   bool GetMatchingRecordBasedOnNumberfromCHtable(
      FC_PhoneBook_Search_Result& MatchingRecord,
      const char* number, quint8 u8ConnectionID); // -SP,31.05.2019 :: Callhistory Efficient Resolving

   quint16 GetTotalRecords(quint8 devicehandle);
   bool UpdatePreferredSortOrder(quint8 u8SortOrder);
   quint8 GetPreferredSortOrder();
   quint8 GetSortOrderFromPreferredSortOrderTable();
   bool UpdateAutomaticContactDownloadValue(bool bValue);   // CMG3G-8366
   bool bGetAutomaticContactDLValue(quint8 DeviceHandle);   // CMG3G-8366

   bool GetLeastRecentlyUsedDevice(QString &device_address, quint8 device_Handle, quint8 sSelectRecord = 1);
   bool MakeFavoritesUnAvailable(quint8 DeviceHandle);
   bool RemoveFavorite(quint32 ContactHandle);
   bool MakeFavoritesAvailable(quint8 DeviceHandle);

   bool SearchPhoneNumberEntriesInPB(QString PhNumberToSearch,quint8 Index,quint8 devicehandle);
   bool SearchContacts(QList<quint8> DeviceHandleList,QString sSearchString,FC_PhoneBook_SQLite::SortType enSearchSortType,
         quint16 u16FieldSelector, quint16 u16MaximumReportedMatches, QList<SearchContacts_Result>& SearchResultList);

   //CLONE TICKET FIX - GMMY15-1545 - Singleton creation
   static FC_PhoneBook_SQLite* GetSQLiteInstance();
   static void vDestroyInstance();
   //Fix for GMMY15-11511
   void clearSearchPhoneDigitList();

   // -PHBK-QUERY-ERR- Instrumentation: BGN - for GMMY17-13044 "database disk image is malformed"
   
   #define PB_CHECK_QUERY_TIMESTAMP_BYTE_LEN 50  // Max string length in byte for string of the following type:
                  // "up-time= [  17.318] sec, "  , i.e. 25 bytes needed, with spare space of 25, defined as 50

   void vCheckQueryError(QSqlQuery &query, quint32 u32LineNr = 0, QString qsFileName = "file_name", QString qsFunctionName = "function_name");
   void vSetDbRecreationTrigger();

   #define PB_CHECK_QUERY_ERROR(query) vCheckQueryError(query, __LINE__, __FILE__, __FUNCTION__);
   // -PHBK-QUERY-ERR- Instrumentation: END - for GMMY17-13044 "database disk image is malformed"

   //FIX SUZUKI-22811 Phone settings (Ringtone, First/last name, paired device list) to be persistent even after SW update
   //FC_Phonebook has to validate a connected device reported
   void vGetDevicesInDB(std::map<tU8, std::string>& DeviceHandlesAndDeviceAddr);
   void getAdditionalContactInfo(quint32 u32Wstart,unsigned WEnd,
   QList<PbkAdditionalContactInfo>& PbkcontactList, QList<FC_PhoneBook_Contact_Detail_Short>& contactList, quint8 Devicehandle ,tU8 u8ListType);
   //FIX CMG3G-8367 IS2424_DownloadOnOff behaviour@FC_Phonebook
   // The user shall have possibility to switch the phonebook and call history list download On or Off individual for each paired phone device.
   bool UpdateSwitchDownloadOnOffState(quint8 DeviceHandle, quint8 SwitchDownloadOnOffState);
   bool GetDownloadOnOffStateList(QList<quint8>& DeviceHandleList, QList<quint16>& DownloadOnOffStateListdd);
   quint16 u16GetDownloadOnOffStateForDevicehandle(quint8 u8DeviceHandle);
   bool bDeletePBContacts(quint8 u8DeviceHandle);
   //End of fix FIX CMG3G-8367	
   bool CreateDeviceTable(quint8 u8DeviceHandle);  //CMG3G-7702
   bool CreateDeviceVpbTable(quint8 u8DeviceHandle);
   //FIX CMG3G-8372 IS2430_SetContactPhoto@FC_Phonebook
   bool SetContactPhoto(quint32 u32ContactHandle, QByteArray PhotoData);
   bool SetContactPosition(quint32 u32ContactHandle, quint32 u8ContactPosition, tU16& u16ErrorCode);
   bool bCalculateUsedMemoryFromOnePhoneBookTable(quint8 u8DeviceHandle, quint32 &u32TotalContactMemory, quint32 &u32TotalPhotoMemory);
   bool CalculateUsedMemoryFromPhoneBookTables(quint32 &u32TotalContactMemory, quint32 &u32TotalPhotoMemory);
   bool bUpdatePhotosOfExistingRecords(quint8 u8DeviceHandle, QList<FC_PhoneBook_Contact_Detail> &oPhotosToBeUpdatedRAMList);
   bool bUpdateAutomaticContactDownload(quint8 u8DeviceHandle, bool bAutomaticDL);

   bool RemoveContactHandlesFromMasterTable(
      QList<quint32> ContactHandleList, quint8 u8TableIndex);
   bool bGetAutomaticContactDLList(QList<quint8>& DeviceHandleList,QList<
         bool>& AutomaticDLList);

   //CMG3G-14241
   bool UpdatePreferredSortOrderForDeviceHandle(tU8 u8DeviceHandle, quint8 u8SortOrder);
   bool bGetPreferredSortOrderList(QList<quint8>& DeviceHandleList, QList<quint8>& PreferredSortOrderList);
   quint8 u8GetPreferredSortOrderForDevicehandle(quint8 u8DeviceHandle);
   //End of CMG3G-14241
   
   void vSetSortingWithPhonetics(bool bSetSortingWithPhonetics);   // Hotfix for CRQ 509734 Phonetic Sorting

private:
   //CLONE TICKET FIX - GMMY15-1545 - Singleton creation
   static FC_PhoneBook_SQLite* m_poSelf;
   FC_PhoneBook_SQLite();
   FC_PhoneBook_SQLite(const FC_PhoneBook_SQLite&);
   ~FC_PhoneBook_SQLite();

   bool CheckTableSchema();
   bool CreateCallHistoryTable(quint8 index);
   bool CreateFeatureSupportTable();
   bool CreateMasterTable();
   bool CheckMasterTable();
   bool CheckSettingsTable();
   bool CheckPreferredSortOrderTable();

   bool CheckDeviceTable(quint8 index);
   bool CheckDeviceVpbTable(quint8 index);
   bool CheckCallHistoryTable(quint8 index);
   bool CheckFeatureSupportTable();
   bool CheckSpeechContactView();
   bool CheckSpeechPhChksumView();

   QSqlDatabase Database();

   QString GetSqlString(const SortType& enSorttype, const quint8& u8CallType,
      const quint8& u8TableIndex, bool bDeviceDateTimeSupport );
   bool InsertIntoMasterTable(QSqlDatabase& db,
      const quint8 TableIndex, quint32& ContactHandle);
   bool InsertDataIntoMasterTable(QSqlDatabase& db, const quint8 TableIndex, quint32& ContactHandle); // CMG3G-7702
   bool UpdateMasterTable(const quint8 TableIndex, quint32& ContactHandle);
   bool GetNewUniqueContactHandle(quint32& u32ContactHandle);
   bool GetLastInsertedContactHandle(quint32& u32ContactHandle);
   bool GetContactHandleForCallHistory(quint8 u8ConnectionID, quint8 u8CallType, 
      quint32& u32ContactHandle);


   // Speech Support
   bool SpeechCreateCheckSumTable();
   bool SpeechCreateViews();
   bool SpeechPopulateCheckSumTable();
   bool SpeechVerifyCheckSumTable();

   bool CreateFavoritesTable();
   bool CheckFavoritesTable();
   bool CreateSettingsTable(quint8 u8SortOrder);

   quint32 u32LastCallHistoryCntHandle[FC_PB_TOTAL_NUMBER_OF_CH][FC_PB_DNL_CALL_HISTORY_TYPES];
//   void FillEmptyStringToEmptyFields(FC_PhoneBook_Contact_Detail &contact);

   //FIX GMMY15-5669 Synchronisation problem between contact addresses in phonebook, HMI and navigation component.
   //The 24th bit of the ContactHandle shall be reserved to carry the following information.
   //if the Contact is from the Vehicle Contact list (24th bit = 0) or from any BT Phone Contact list (24th bit = 1).
   void MarkContactHandle(const quint8 TableIndex, quint32& u32ContactHandle);

   bool bAdjustWalJournalingNamesAndLinks();
   bool bAdjustWalJournalingAccessRights();
   
#ifdef FC_PB_DB_USE_DUMMY_RECORDS
   void insertDummyRecords();
#endif
   // QMutex m_mutex_master;
   QMutex m_mutex;
   quint32 u32LastContactHandleUsed;
   quint32 m_noOfContactsVehicleAddBook[12];
   bool m_bIsVehiclePbk;
   bool m_bCheckTableSchema;
   bool m_bSortingWithPhonetics;   // Hotfix for CRQ 509734 Phonetic Sorting
   
public:
   bool OpenDatabase();
   //Fix NCG3D-67384
   bool isCheckTableSchemaEnabled();
   void SetCheckTableSchema(bool bCheckTableSchema);
   //End fix NCG3D-67384
   
   void SetWalCheckPoint(void);
};

#endif//_FC_PHONEBOOK_SQLITE_
