/*******************************************************************************
*
* FILE:          FC_PhoneBook_SQLite.cpp
*
* SW-COMPONENT:  FC_PhoneBook application
*
* PROJECT:
*
* DESCRIPTION:   Handles the sqlite database using QtSql module
*
* AUTHOR:
*
* COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
*
*******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/

#include "../FC_PhoneBook_Debug.h"
#include "../FC_PhoneBook_main.h"
#include "FC_PhoneBook_SQLite.h"
#include "FC_PhoneBook_List.h"

#define SORTLIB_S_IMPORT_INTERFACE_GENERIC
#include "sortlib_if.h"

// Include Sqlite headers
#include "sqlite3.h"


#ifdef DATABASE_ENCRYPTION_VIA_SEE_AND_SDC
   // Include SDC (Secure Data Container Lib) related header files.
   #include "sdc_perm.h"             // needed for type "sdc_permissions_t"
   #include "sdc_session.h"          // needed for "sdc_open_session(..)"
   #include "sdc_keystore_keys.h"    // needed for "sdc_generate_random_storage_key(sdc_key_id_t *kid, ..)" and enumerations like 'SDC_CREATE_KEY_FIXED_ID'
   //#include "sdc.h"                // not needed
   //#include <sdc_op_conv.h>        // not needed
   //#include <sdc_random.h>         // not needed
#endif


#include <sys/stat.h>   // needed for "chmod(..)", "stat(..)",
                        // and enums like: S_IRUSR, S_IWUSR, S_IRGRP, ..

#include <sys/statvfs.h>   // needed for "statvfs(..)" [e.g. to check the free size of the partition, where a dedicated file resides.]

#include <pwd.h>
#include <grp.h>
#include <unistd.h>    // needed for "chown(..)" or 'syncfs(fd)'

// For "-PHBK-QUERY-ERR-" instrumentation:
#include <sys/time.h>  // for 'gettimeofday' or 'clock_gettime'
// #define PB_CHECK_QUERY_EXEC__TESTMODE  // ONLY for test of: "-PHBK-QUERY-ERR". This complier switch shall not be active in released code. It shall be active only in developer builds for testing.


#define ETRACE_S_IMPORT_INTERFACE_GENERIC

#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONEBOOK_SQLITE
#include "trcGenProj/Header/FC_PhoneBook_SQLite.cpp.trc.h"
#endif

/******************************************************************************/
/*                                                                            */
/* DEFINES                                                                    */
/*                                                                            */
/******************************************************************************/

// Defines for WAL journaling
#define FC_PB_DB_WAL_LINK_NAME         "phonebook.wal"
#define FC_PB_DB_SHM_LINK_NAME         "phonebook.shm"

#define FC_PB_DB_LONG_WAL_FILE_NAME    "phonebook.db-wal"
#define FC_PB_DB_LONG_SHM_FILE_NAME    "phonebook.db-shm"

#define FC_PB_DB_SHORT_WAL_FILE_NAME   "phonebook.wal"
#define FC_PB_DB_SHORT_SHM_FILE_NAME   "phonebook.shm"

// Defines for database tables and columns
#define FC_PB_MASTER_TABLE_NAME  "PhoneBook_Master"

#define FC_PB_TABLE_NAME_TMP         "PhoneBook_Tmp"     // CMG3G-7702
#define FC_PB_TABLE_NAME_VPB         "PhoneBook_VPB"
#define FC_PB_TABLE_NAME_DH2         "PhoneBook_DH2"
#define FC_PB_TABLE_NAME_DH3         "PhoneBook_DH3"
#define FC_PB_TABLE_NAME_DH4         "PhoneBook_DH4"
#define FC_PB_TABLE_NAME_DH5         "PhoneBook_DH5"
#define FC_PB_TABLE_NAME_DH6         "PhoneBook_DH6"
#define FC_PB_TABLE_NAME_DH7         "PhoneBook_DH7"
#define FC_PB_TABLE_NAME_DH8         "PhoneBook_DH8"
#define FC_PB_TABLE_NAME_DH9         "PhoneBook_DH9"
#define FC_PB_TABLE_NAME_DH10        "PhoneBook_DH10"
#define FC_PB_TABLE_NAME_DH11        "PhoneBook_DH11"

#define FC_PB_TABLE_NAME_VPB2         "PhoneBook_VPB2"
#define FC_PB_TABLE_NAME_VPB3         "PhoneBook_VPB3"
#define FC_PB_TABLE_NAME_VPB4         "PhoneBook_VPB4"
#define FC_PB_TABLE_NAME_VPB5         "PhoneBook_VPB5"
#define FC_PB_TABLE_NAME_VPB6         "PhoneBook_VPB6"
#define FC_PB_TABLE_NAME_VPB7         "PhoneBook_VPB7"
#define FC_PB_TABLE_NAME_VPB8         "PhoneBook_VPB8"
#define FC_PB_TABLE_NAME_VPB9         "PhoneBook_VPB9"
#define FC_PB_TABLE_NAME_VPB10        "PhoneBook_VPB10"
#define FC_PB_TABLE_NAME_VPB11        "PhoneBook_VPB11"
#define FC_PB_TABLE_CHECKSUM         "PhoneBook_CheckSum"
#define FC_PB_TABLE_NAME_FAV         "PhoneBook_FAV"

#define FC_PB_CHKSUM_COL_TABLE_INDEX  "TableIndex"
#define FC_PB_CHKSUM_COL_TABLE_CHKSUM "CheckSum"


#define FC_PB_CH_TABLE_NAME_CON_ID0   "CallHist_DEVICE0"
#define FC_PB_CH_TABLE_NAME_CON_ID1   "CallHist_DEVICE1"
#define FC_PB_TABLE_NAME_FS          "PhoneBook_FeatureSupport"

#define FC_PB_MASTER_COL_CONTACT_HANDLE "ContactHandle"
#define FC_PB_MASTER_COL_TABLE_INDEX    "TableIndex"

#define UNION           " union "

#define FC_PB_COL_CONTACT_HANDLE "ContactHandle"
#define FC_PB_COL_VPB_CONTACT_POSITION  "ContactPosition" //CMG3G-8571
#define FC_PB_COL_FIRST_NAME     "FirstName"
#define FC_PB_COL_LAST_NAME      "LastName"
#define FC_PB_COL_PHONETICS      "Phonetics" //292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
#define FC_PB_COL_SORTFIRSTNAME  "SortFirstName"   // NEW
#define FC_PB_COL_SORTLASTNAME   "SortLastName"    // NEW
#define FC_PB_COL_POI_NAME       "PoiName"
#define FC_PB_COL_PHONENUM1      "PhoneNumber1"
#define FC_PB_COL_NUMPRETTY1     "PPPhoneNumber1"
#define FC_PB_COL_NUMTYPE1       "NumberType1"
#define FC_PB_COL_NUMSUBSTR1     "NumberSubstr1"   // NEW
#define FC_PB_COL_PHONENUM2      "PhoneNumber2"
#define FC_PB_COL_NUMPRETTY2     "PPPhoneNumber2"
#define FC_PB_COL_NUMTYPE2       "NumberType2"
#define FC_PB_COL_NUMSUBSTR2     "NumberSubstr2"   // NEW
#define FC_PB_COL_PHONENUM3      "PhoneNumber3"
#define FC_PB_COL_NUMPRETTY3     "PPPhoneNumber3"
#define FC_PB_COL_NUMTYPE3       "NumberType3"
#define FC_PB_COL_NUMSUBSTR3     "NumberSubstr3"   // NEW
#define FC_PB_COL_PHONENUM4      "PhoneNumber4"
#define FC_PB_COL_NUMPRETTY4     "PPPhoneNumber4"
#define FC_PB_COL_NUMTYPE4       "NumberType4"
#define FC_PB_COL_NUMSUBSTR4     "NumberSubstr4"   // NEW
#define FC_PB_COL_PHONENUM5      "PhoneNumber5"
#define FC_PB_COL_NUMPRETTY5     "PPPhoneNumber5"
#define FC_PB_COL_NUMTYPE5       "NumberType5"
#define FC_PB_COL_NUMSUBSTR5     "NumberSubstr5"   // NEW
#define FC_PB_COL_PHONENUM6      "PhoneNumber6"
#define FC_PB_COL_NUMPRETTY6     "PPPhoneNumber6"
#define FC_PB_COL_NUMTYPE6       "NumberType6"
#define FC_PB_COL_NUMSUBSTR6     "NumberSubstr6"   // NEW
#define FC_PB_COL_PHONENUM7      "PhoneNumber7"
#define FC_PB_COL_NUMPRETTY7     "PPPhoneNumber7"
#define FC_PB_COL_NUMTYPE7       "NumberType7"
#define FC_PB_COL_NUMSUBSTR7     "NumberSubstr7"   // NEW
#define FC_PB_COL_PHONENUM8      "PhoneNumber8"
#define FC_PB_COL_NUMPRETTY8     "PPPhoneNumber8"
#define FC_PB_COL_NUMTYPE8       "NumberType8"
#define FC_PB_COL_NUMSUBSTR8     "NumberSubstr8"   // NEW
#define FC_PB_COL_EMAIL_ADD1     "EmailAdd1"
#define FC_PB_COL_EMAIL_ADD2     "EmailAdd2"
#define FC_PB_COL_EMAIL_ADD3     "EmailAdd3"
#define FC_PB_COL_HOME_ADD       "HomeAdd"
#define FC_PB_COL_WORK_ADD       "WorkAdd"
#define FC_PB_COL_OTHER_ADD      "OtherAdd"
#define FC_PB_COL_CATEGORY       "Category"
#define FC_PB_COL_CONTACT_TYPE   "ContactType"
#define FC_PB_COL_NOTES          "Notes"
#define FC_PB_COL_HGCLATLONGVALIDITY   "HomeGeoCodeLatLongValidity"
#define FC_PB_COL_HGCLATITUDE          "HomeGeoCodeLatitude"
#define FC_PB_COL_HGCLONGITUDE         "HomeGeocodeLongitude"
#define FC_PB_COL_WGCLATLONGVALIDITY   "WorkGeoCodeLatLongValidity"
#define FC_PB_COL_WGCLATITUDE          "WorkGeoCodeLatitude"
#define FC_PB_COL_WGCLONGITUDE         "WorkGeocodeLongitude"
#define FC_PB_COL_OGCLATLONGVALIDITY   "OtherGeoCodeLatLongValidity"
#define FC_PB_COL_OGCLATITUDE          "OtherGeoCodeLatitude"
#define FC_PB_COL_OGCLONGITUDE         "OtherGeocodeLongitude"
#define FC_PB_COL_PHOTO_DL_SIZE        "PhotoDlSize"
#define FC_PB_COL_PHOTO                "Photo"
#define FC_PB_COL_PHOTO_TYPE           "PhotoType"     // -FastPhotoHandling-
#define FC_PB_COL_PHOTO_CRC            "PhotoCRC"      // -FastPhotoHandling-
#define FC_PB_COL_PHOTO_DL_CRC         "PhotoDlCRC"    // -FastPhotoHandling-
#define FC_PB_COL_PHOTO_SIZE           "PhotoSize"     // -FastPhotoHandling-
#define FC_PB_COL_APL1_SIZE            "APL1Size" // -FastContactSearch-
#define FC_PB_COL_CRC_APL1             "CrcAPL1"  // -FastContactSearch-

#define FC_PB_COL_CH_CONTACT_HANDLE    "ContactHandle"
#define FC_PB_COL_CH_CALL_TYPE         "CallType"
#define FC_PB_COL_CH_FIRST_NAME        "FirstName"
#define FC_PB_COL_CH_LAST_NAME         "LastName"
#define FC_PB_COL_CH_PHONE_NUM         "PhoneNumber"
#define FC_PB_COL_CH_NUMBER_TYPE       "NumberType"
#define FC_PB_COL_CH_CALL_DATE         "CallDate"
#define FC_PB_COL_CH_CALL_TIME         "CallTime"
#define FC_PB_COL_CH_PP_PHONE_NUM       "PPPhoneNumber"

#define FC_PB_COL_FAV_FAVORITE_ID       "FavoriteID"
#define FC_PB_COL_FAV_DEVICE_HANDLE     "DeviceHandle"
#define FC_PB_COL_FAV_CONTACT_HANDLE    "ContactHandle"
#define FC_PB_COL_FAV_TAG_VALUE         "TagValue"
#define FC_PB_COL_FAV_LABEL_TEXT        "LabelText"
#define FC_PB_COL_FAV_PHONE_NUMBER      "PhoneNumber"
#define FC_PB_COL_FAV_ACTIVE_STATUS     "Active"
#define FC_PB_COL_FAV_AVALIABLE_STATUS  "Available"

//Feature Support Table Details
#define FC_PB_FS_COL_DEVICE_ADDRESS    "Address"
#define FC_PB_FS_COL_DEVICE_HANDLE     "Handle"
#define FC_PB_FS_COL_PB_SUPPORT        "PhoneBookSupport"
#define FC_PB_FS_COL_PB_CONTACTS_SORT  "ContactListSortView"
#define FC_PB_FS_COL_PB_MISSED_CALL    "MCHListSupport"
#define FC_PB_FS_COL_PB_INCOMING_CALL  "ICHListSupport"
#define FC_PB_FS_COL_PB_OUTGOING_CALL  "OCHListSupport"
#define FC_PB_FS_COL_PB_COMBINED_CALL  "CCHListSupport"

#define FC_PB_FS_COL_PB_LAST_CONNECTION_TIME_ELAPSED          "LastConnectionTimeElapsed"
#define FC_PB_FS_COL_PB_DL_COMPLETED_ONCE    "DlCompletedOnce"
#define FC_PB_FS_COL_PB_DOWNLOAD_ONOFF   "DownloadOnOff" //FIX CMG3G-8367 IS2424_DownloadOnOff behaviour@FC_Phonebook
#define FC_PB_FS_COL_PB_AUTOMATIC_CONTACT_DL   "AutomaticDL" //CMG3G-9889

#define FC_PB_DB_CONNECTION_NAME "fc_phonebook_database_connection"
//#define BLANK_NAME ""  // BAD

#define FC_PB_SETTINGS_TABLE_NAME                  "Settings"
#define FC_PB_PREFERREDSORTORDER_COL_SORTORDER     "PreferredSortOrder"

#define FC_PB_PREFERREDSORTORDER_TABLE_NAME                  "PreferredSortOrderTable"

#define FC_PB_CONTACT_TABLE_NAME     "Contact"
#define FC_PB_CHKSUM_TABLE_NAME   "Ph_Checksum"

//Field selectors for SearchContacts
#define PB_FieldSelector_FirstName 0x0001
#define PB_FieldSelector_LastName 0x0002
#define PB_FieldSelector_PoiName 0x0004
#define PB_FieldSelector_Address 0x0008
#define PB_FieldSelector_Email 0x0010
#define PB_FieldSelector_Number 0x0020

/* Device handle zero is reserved*/
const QString fc_pb_aTableNames[FC_PB_TOTAL_NUMBER_OF_PB + FC_PB_TOTAL_NUMBER_OF_CH + 1]={
   FC_PB_TABLE_NAME_TMP,   // CMG3G-7702
   FC_PB_TABLE_NAME_VPB,
   FC_PB_TABLE_NAME_DH2,
   FC_PB_TABLE_NAME_DH3,
   FC_PB_TABLE_NAME_DH4,
   FC_PB_TABLE_NAME_DH5,
   FC_PB_TABLE_NAME_DH6,
   FC_PB_TABLE_NAME_DH7,
   FC_PB_TABLE_NAME_DH8,
   FC_PB_TABLE_NAME_DH9,
   FC_PB_TABLE_NAME_DH10,
   FC_PB_TABLE_NAME_DH11,
   FC_PB_CH_TABLE_NAME_CON_ID0,
   FC_PB_CH_TABLE_NAME_CON_ID1
};

const QString fc_vpb_aTableNames[FC_PB_TOTAL_NUMBER_OF_VPB+2]={
   NULLPTR,
   NULLPTR,
   FC_PB_TABLE_NAME_VPB2,
   FC_PB_TABLE_NAME_VPB3,
   FC_PB_TABLE_NAME_VPB4,
   FC_PB_TABLE_NAME_VPB5,
   FC_PB_TABLE_NAME_VPB6,
   FC_PB_TABLE_NAME_VPB7,
   FC_PB_TABLE_NAME_VPB8,
   FC_PB_TABLE_NAME_VPB9,
   FC_PB_TABLE_NAME_VPB10,
   FC_PB_TABLE_NAME_VPB11
};
#define MAP_TO_CH_INDEX(index) (quint8)(index+FC_PB_TOTAL_NUMBER_OF_PB)

#define OSAL_NULL (0)

#define MINIMUM_LENGTH_FOR_TEL_NUMBER 2

#define CHECK_VALID_SORT_ORDER(index) (((index == 0)||(index == 1))?true:false)
#define CHECK_VALID(index) (((index>0)&&(index<=FC_PB_TOTAL_NUMBER_OF_PB))?true:false)
#define CHECK_CH_VALID(index) (((index>0)&&(index<=FC_PB_TOTAL_NUMBER_OF_CH))?true:false)
#define CHECK_VALID_CALL_HIST(index) (((index == FC_PB_CALLHIST_DEVICE0 )||(index == FC_PB_CALLHIST_DEVICE1))?true:false)
#define CHECK_VALID_VPBx(index) (((index>=FC_PB_DEVICE_HANDLE2)&&(index<=FC_PB_DEVICE_HANDLE11))?true:false)

#define FC_PB_BASE_CNT_HANDLE_FOR_CH(ConnectionID, CallType) \
   ( (MAX_NO_OF_CCH_ENTRIES * 3 * (ConnectionID-1)) + ((CallType - 1)*MAX_NO_OF_CCH_ENTRIES) )
// Contact handles are overlapping for call history entries while inseting into the master table.
// Macro Logic changed from (CallType)* FC_PB_MAX_CALL_HISTORY_ENTRIES_PER_TYPE to (CallType)* MAX_NO_OF_CCH_ENTRIES for solving the issue PSARCCB-8150
// Offset for the second connected device corrected to 3 * MAX_NO_OF_CCH_ENTRIES [3: OCH, ICH, MCH]
//For example: For PSA - MCH starts from 0 to 119, OCH from 120 to 239 and ICH from 240 to 359.
//Hence even if there are 118 MCH and 1 OCH while downloading combined call histroy they will not overlap.

#define CHUNK_DELETE             5000   // Fix SUZUKI-20814



//FIX GMMY15-5669 Synchronisation problem between contact addresses in phonebook, HMI and navigation component.
//The 24th bit of the ContactHandle shall be reserved to carry the following information.
//if the Contact is from the Vehicle Contact list (24th bit = 0) or from any BT Phone Contact list (24th bit = 1).
//Hence maximum limit for contact handle is 7FFFFF
#define FC_PB_CONTACT_HANDLE_LIMIT            (quint32)0x7FFFFF
//End of FIX GMMY15-5669
#define FC_PB_NO_OF_MAX_PHONENUMS_PER_CONTACT 8 //Fix CMG3GB-2829

FC_PhoneBook_SQLite* FC_PhoneBook_SQLite::m_poSelf = NULL;

/******************************************************************************/
/*                                                                            */
/* METHODS                                                                    */
/*                                                                            */
/******************************************************************************/


//CRQ CMG3G-2888
//CMG3G-2971:Enhanced vcard handling@FC_Phonebook
#ifdef VALID_VCARD_FORMAT_PHONENUMBER
/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_Contact_Detail::isValid()
*
* DESCRIPTION: Validation check
*
*
* PARAMETER: None.
*
* RETURNVALUE: None
*
*******************************************************************************/
bool FC_PhoneBook_Contact_Detail::isValid(FC_PhoneBook_Contact_Detail& contact)//CMG3G-14313-FC_PhoneBook_Contact_Detail-Passed as a reference-Coverity Fix
{
   ETG_TRACE_USR4(("isValid entered"));
   // Check if the contact contains at least a valid Tel No.
   if ((    ( contact.PhoneNum1.length() >= MINIMUM_LENGTH_FOR_TEL_NUMBER) \
      || ( contact.PhoneNum2.length() >= MINIMUM_LENGTH_FOR_TEL_NUMBER) \
      || ( contact.PhoneNum3.length() >= MINIMUM_LENGTH_FOR_TEL_NUMBER) \
      || ( contact.PhoneNum4.length() >= MINIMUM_LENGTH_FOR_TEL_NUMBER) \
      || ( contact.PhoneNum5.length() >= MINIMUM_LENGTH_FOR_TEL_NUMBER) \
      || ( contact.PhoneNum6.length() >= MINIMUM_LENGTH_FOR_TEL_NUMBER) \
      || ( contact.PhoneNum7.length()  >= MINIMUM_LENGTH_FOR_TEL_NUMBER) \
      || ( contact.PhoneNum8.length() >= MINIMUM_LENGTH_FOR_TEL_NUMBER)))

      //FIX GMMY15-5704 This function identifies valid contacts in VCards stream received from DBus.
      //The following condition marks contacts with no firstname and no lastname as invalid.
      //This condition needs to be removed to allow contacts with only phone numbers to be displayed in HMI.
      //&&  ((!contact.firstName.isEmpty()) || (!contact.lastName.isEmpty()
   {
      return true;
   }
   else
   {
      ETG_TRACE_USR4((" InValid contact"));
      return false;
   }

}
#endif


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::FC_PhoneBook_SQLite()
*
* DESCRIPTION: Constructor
*
*
* PARAMETER: None.
*
* RETURNVALUE: None
*
*******************************************************************************/
FC_PhoneBook_SQLite::FC_PhoneBook_SQLite()
{
   ETG_TRACE_USR4(("FC_PhoneBook_SQLite() constructor entered "));

   m_bSortingWithPhonetics = false;   // Hotfix for CRQ 509734 Phonetic Sorting

   //Fix NCG3D-67384
   m_bCheckTableSchema = FALSE;
   //End fix NCG3D-67384
#ifdef FC_PB_DB_USE_DUMMY_RECORDS
   bool newfile = !QFile::exists(FC_PB_DB_FILE_PATH);
#endif

   u32LastContactHandleUsed= OSAL_NULL;
   for(quint8 uIndex= VALUE_ZERO; uIndex <= FC_VPB_DEVICE_HANDLE11; uIndex++)
   {
      m_noOfContactsVehicleAddBook[uIndex] = VALUE_ZERO;
   }
   m_bIsVehiclePbk = false;
   memset(u32LastCallHistoryCntHandle, 0x00, sizeof(u32LastCallHistoryCntHandle));
   if(!QFile::exists(FC_PB_DB_FILE_PATH))
   {
      QDir dirpath;
      if(dirpath.mkpath(FC_PB_DB_FILE_PATH))
      {
         ETG_TRACE_USR4(("New path created - %s", FC_PB_DB_FILE_PATH ));
      }
      else
      {
         ETG_TRACE_ERR(("New path creation failed- %s", FC_PB_DB_FILE_PATH ));
      }
   }
   else
   {

      ETG_TRACE_USR4(("db directory exists "));
   }

#ifdef FC_PB_DB_USE_DUMMY_RECORDS
   if ( newfile )
      insertDummyRecords();
#endif

   return;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::~FC_PhoneBook_SQLite()
*
* DESCRIPTION: Destructor
*
*
* PARAMETER: None.
*
* RETURNVALUE: None
*
*******************************************************************************/
FC_PhoneBook_SQLite::~FC_PhoneBook_SQLite()
{
   //Fix for GMM15-11113-Reset Tickets on phonebook
   CloseDatabase();
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::Database()
*
* DESCRIPTION:
*              Helper function for QSqlQuery to get the current database.
*              We should never store the QSqlDatabase as it will cause
*              warnings while closing it.
* PARAMETER: None
*
* RETURNVALUE: QSqlDatabase for the current connection
*
*******************************************************************************/
inline QSqlDatabase FC_PhoneBook_SQLite::Database()
{
   return QSqlDatabase::database( FC_PB_DB_CONNECTION_NAME, false );
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetSQLiteInstance()
*
* DESCRIPTION:
*              Returns pointer to current Instance
*
* PARAMETER: None
*
* RETURNVALUE: QSqlDatabase for the current connection
*
*******************************************************************************/
FC_PhoneBook_SQLite* FC_PhoneBook_SQLite::GetSQLiteInstance()
{
   //CLONE TICKET FIX - GMMY15-1545 - Singleton creation
   if(m_poSelf == NULL)
   {
      m_poSelf = new FC_PhoneBook_SQLite;
   }
   return m_poSelf;
}

/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::vDestroyInstance()
*
* DESCRIPTION:
*             Deletes SQLite Instance
*
* PARAMETER: None
*
* RETURNVALUE: None
*
*******************************************************************************/
tVoid FC_PhoneBook_SQLite::vDestroyInstance()
{
   delete m_poSelf;
   m_poSelf = NULLPTR;
}
/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::OpenDatabase()
*
* DESCRIPTION:
*              Initializes the database connection
*
*
* PARAMETER: None.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::OpenDatabase()
{
   ETG_TRACE_USR4(("OpenDatabase entered"));

   int rc;
   bool result = true;
   QVariant qVariant;

   // Print Qt and SQLite version strings
   ETG_TRACE_COMP(( "QT_VERSION_STR= '%s' ", QT_VERSION_STR ));
   ETG_TRACE_COMP(( "qVersion()    = '%s' ", qVersion() ));

   ETG_TRACE_COMP(( "SQLITE_VERSION      = '%s' ", SQLITE_VERSION ));
   ETG_TRACE_COMP(( "sqlite3_libversion()= '%s' ", sqlite3_libversion() ));


   // The following SQLite "sqlite3_initialize()" statement is needed only when we use an SQLite version, which has
   // been compiled with compile option "SQLITE_OMIT_AUTOINIT" as in this case we would get a SIGSEGV reset, when not
   // using the "initalize" statement.
   // - The compile option "SQLITE_OMIT_AUTOINIT" typically is set, when we experiment with some "NDS SQLite" library,
   //   provided by Naviagation team for test.
   // - This compile option is not relevant (activated) when we use a standard SQLite provided by ESO.
   // - The latter case is the standard one, so that the following line is commented out by default.
   //
   // sqlite3_initialize();  // Only needed when the SQLite library, we are using, has been compiled with compile option "SQLITE_OMIT_AUTOINIT".


#ifdef DATABASE_ENCRYPTION_VIA_SEE_AND_SDC
   // ============= Key-Id availibility check: BGN =========================================================
   // The code in this '#ifdef clause' will check if an Key-ID/AES-Key pair is available in the
   // SDC (Secure Data Container) or try to create it.
   // - If an AES Key has to be created (typically only in a virgin startup scenario), we do so
   //   and if this was successfull we will delete an already existing database or journal file
   //   to avoid any inconsistencies.

   int         iDBRes = 0;        // Relevant, if we need to delete the database (DB) file or the journal file(s)
   sdc_error_t sdcRes = SDC_OK;   // Relevant for all SDC related methods


   // SQLite: Specify the activation key for the SEE (SQLite Encryption Extension).
   // Note: Unless activated, none of the SEE routines will work.
   sqlite3_activate_see("2ca6e9e8d47d21bc");  // This "key" is BOSCH (CMD, ESO, ESN) specific. We do not the use the default one, which is public in in Internet ("7bb07b8d471d642e").


   // SDC (Secure Data Container): Check if the SDC keystore has already a <Key-ID, AES-Key> "pair" for the phonebook database, otherwise create such "pair".
   #define SDC_PHBK_KEY_ID           8101  // Note: Key-ID range for Connectivity is 8101 - 8200. We have chosen the first one in range.
   #define SKEY_LENGTH_BYTES           32  // 32 byte = 256 bit
   #define AID_PHBK_SHARE_GROUP_ID  10114  // group 'aid_phbk_share' has ID= 10114

   sdc_permissions_t *poPermissions = NULL;
   sdc_session_t     *poSession     = NULL;
   sdc_key_id_t       keyId         = SDC_PHBK_KEY_ID;   // sdc_key_id_t == unit32  (see /usr/include/adit-components/sdc.h)

   // As first step we open an SDC session
   sdcRes = sdc_open_session(&poSession);
   if (sdcRes != SDC_OK)
   {
      ETG_TRACE_FATAL (( " -PBDB.SECURITY-SDC-: sdc_open_session(.): ERROR:   sdcRes= %d, result_string= '%s' ", sdcRes, sdc_get_error_string(sdcRes) ));
      ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-SDC-: sdc_open_session(.): ERROR:   sdcRes= %d, result_string= '%s' ", sdcRes, sdc_get_error_string(sdcRes) ));
   }
   else
   {
      ETG_TRACE_COMP  (( " -PBDB.SECURITY-SDC-: sdc_open_session(.): SUCCESS: sdcRes= %d, result_string= '%s' ", sdcRes, sdc_get_error_string(sdcRes) ));

      // Probe the key store, if the phonebook related <Key-ID, AES-Key> is already available/known by the SDC.
      sdcRes = sdc_session_load_storage_key(poSession, keyId);

      if (sdcRes == SDC_KID_NOT_AVAILABLE)
      {
         // The phonebook Key-ID is not known by the SDC keystore, so we seem to be in a "virgin" startup of the device.
         // We will try to create the needed <Key-ID, AES-Key> "pair" in the SDC keystore for the phonebook database

         ETG_TRACE_FATAL (( " -PBDB.SECURITY-SDC-: Phonebook Key-ID is not known by the SDC keystore. We will try to create it. " ));
         ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-SDC-: Phonebook Key-ID is not known by the SDC keystore. We will try to create it. " ));

         sdcRes = SDC_OK;
         // Create a new object of SDC permissions. This has to be freed later.
         sdcRes = sdc_permissions_alloc(&poPermissions);
         if (sdcRes != SDC_OK)
         {
            ETG_TRACE_FATAL (( " -PBDB.SECURITY-SDC-: sdc_permissions_alloc(.): ERROR:   sdcRes= %d, result_string= '%s' ", sdcRes, sdc_get_error_string(sdcRes) ));
            ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-SDC-: sdc_permissions_alloc(.): ERROR:   sdcRes= %d, result_string= '%s' ", sdcRes, sdc_get_error_string(sdcRes) ));
         }
         else
         {
            ETG_TRACE_COMP  (( " -PBDB.SECURITY-SDC-: sdc_permissions_alloc(.): SUCCESS: sdcRes= %d, result_string= '%s' ", sdcRes, sdc_get_error_string(sdcRes) ));

            // Variants to set the permissions for the Phonebook Key-ID at SDC (Secure Data Container).

            // Set the Key-ID permissions in the SDC to the current user ('aid_phbk'). This variant is NOT used as we need to grant access to a dedicate group also.
            // sdcRes = sdc_set_default_permissions_current_gid(poPermissions);

            // Set the Key-ID permissions in the SDC to the current user ('aid_phbk') and group ('aid_phbk_share'), so that the Speech SDS
            // transcription service also can access the phonebook database.
            sdcRes = sdc_set_default_permissions_and_gid(poPermissions, AID_PHBK_SHARE_GROUP_ID);

            if (sdcRes != SDC_OK)
            {
               ETG_TRACE_FATAL (( " -PBDB.SECURITY-SDC-: sdc_set_default_permissions_current_gid(.): ERROR:   sdcRes= %d, result_string= '%s' ", sdcRes, sdc_get_error_string(sdcRes) ));
               ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-SDC-: sdc_set_default_permissions_current_gid(.): ERROR:   sdcRes= %d, result_string= '%s' ", sdcRes, sdc_get_error_string(sdcRes) ));
            }
            else
            {
               ETG_TRACE_COMP  (( " -PBDB.SECURITY-SDC-: sdc_set_default_permissions_current_gid(.): SUCCESS: sdcRes= %d, result_string= '%s' ", sdcRes, sdc_get_error_string(sdcRes) ));

               // Here we create the <Key-ID, AES-Key> "pair" for the phonebook database
               sdcRes = sdc_generate_random_storage_key (&keyId, SDC_CREATE_KEY_FIXED_ID, SKEY_LENGTH_BYTES, SDC_PERSISTENT_STORAGE_KEY, poPermissions);
               if (sdcRes != SDC_OK)
               {
                  ETG_TRACE_FATAL (( " -PBDB.SECURITY-SDC-: sdc_generate_random_storage_key(..): ERROR:   sdcRes= %d, result_string= '%s' ", sdcRes, sdc_get_error_string(sdcRes) ));
                  ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-SDC-: sdc_generate_random_storage_key(..): ERROR:   sdcRes= %d, result_string= '%s' ", sdcRes, sdc_get_error_string(sdcRes) ));
               }
               else
               {
                  ETG_TRACE_FATAL (( " -PBDB.SECURITY-SDC-: sdc_generate_random_storage_key(..): SUCCESS: sdcRes= %d, result_string= '%s' ", sdcRes, sdc_get_error_string(sdcRes) ));
                  ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-SDC-: sdc_generate_random_storage_key(..): SUCCESS: sdcRes= %d, result_string= '%s' ", sdcRes, sdc_get_error_string(sdcRes) ));

                  // Here we check a 2nd time, if the phonebook related <Key-ID, AES-Key> is now available / known by the SDC.
                  sdcRes = sdc_session_load_storage_key(poSession, keyId);
                  if (sdcRes != SDC_OK)
                  {
                     ETG_TRACE_FATAL (( " -PBDB.SECURITY-SDC-: sdc_session_load_storage_key(..): ERROR: [2nd attempt]: sdcRes= %d, result_string= '%s' ", sdcRes, sdc_get_error_string(sdcRes) ));
                     ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-SDC-: sdc_session_load_storage_key(..): ERROR: [2nd attempt]: sdcRes= %d, result_string= '%s' ", sdcRes, sdc_get_error_string(sdcRes) ));
                  }
                  else
                  {
                     ETG_TRACE_COMP  (( " -PBDB.SECURITY-SDC-: sdc_session_load_storage_key(..): SUCCESS: [2nd attempt]: sdcRes= %d, result_string= '%s' ", sdcRes, sdc_get_error_string(sdcRes) ));
                     // When we come here we have created a new pair <Key-ID, AES-Key> successfully, but this means that any current database will not be accessible
                     // with any old <Key-ID, AES-Key> information. Therefore we will remove the current database.

                     int iRes = 0;  // Temporary result variable

                     // Remove the current database, if it exists
                     struct stat oStatBuf;
                     if( 0 == stat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME, &oStatBuf) )
                     {
                        ETG_TRACE_FATAL (( " -PBDB.SECURITY-SDC-: INFO: The database file exists and will be deleted. "  ));
                        ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-SDC-: INFO: The database file exists and will be deleted. "  ));

                        iRes = remove(FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME);
                        if (iRes != 0)
                        {
                           iDBRes = iRes;
                           ETG_TRACE_FATAL (( " -PBDB.SECURITY-SDC-: ERROR:   iRes= %d at remove('%s')", iRes, FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME ));
                           ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-SDC-: ERROR:   iRes= %d at remove('%s')", iRes, FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME ));
                        }
                        else
                        {
                           ETG_TRACE_COMP  (( " -PBDB.SECURITY-SDC-: SUCCESS: iRes= %d at remove('%s')", iRes, FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME ));
                        }
                     }

                     // Remove the DB journal file(s), if they exist, also.
#ifdef DATABASE_WAL_JOURNALING
                     // Remove the WAL journal file, if it exists.
                     if( 0 == stat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_FILE_NAME, &oStatBuf) )
                     {
                        ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-: INFO: The WAL journal file exists and will be deleted. "  ));
                        ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-: INFO: The WAL journal file exists and will be deleted. "  ));

                        iRes = remove(FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_FILE_NAME);
                        if (iRes != 0)
                        {
                           iDBRes = iRes;
                           ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-: ERROR: Existing WAL journal file CANNOT be deleted. "  ));
                           ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-: ERROR: Existing WAL journal file CANNOT be deleted. "  ));
                        }
                        else
                        {
                           ETG_TRACE_COMP  (( " -PBDB.SECURITY-WAL-: SUCCESS: iRes= %d at remove('%s')", iRes, FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_FILE_NAME ));
                        }
                     }

                     // Remove the SHM (SHared Memory) journal file, if it exists.
                     if( 0 == stat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHM_FILE_NAME, &oStatBuf) )
                     {
                        ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-: INFO: The SHM journal file exists and will be deleted. "  ));
                        ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-: INFO: The SHM journal file exists and will be deleted. "  ));

                        iRes = remove(FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHM_FILE_NAME);
                        if (iRes != 0)
                        {
                           iDBRes = iRes;
                           ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-: ERROR: Existing SHM journal file CANNOT be deleted. "  ));
                           ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-: ERROR: Existing SHM journal file CANNOT be deleted. "  ));
                        }
                        else
                        {
                           ETG_TRACE_COMP  (( " -PBDB.SECURITY-WAL-: SUCCESS: iRes= %d at remove('%s')", iRes, FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHM_FILE_NAME ));
                        }
                     }
#else
                     // Remove the DB "rollback" journal file, if it exists.
                     if( 0 == stat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_JOURNAL_FILE_NAME, &oStatBuf) )
                     {
                        ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-: INFO: The rollback journal file exists and will be deleted. "  ));
                        ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-: INFO: The rollback journal file exists and will be deleted. "  ));

                        iRes = remove(FC_PB_DB_FILE_PATH "/" FC_PB_DB_JOURNAL_FILE_NAME);
                        if (iRes != 0)
                        {
                           iDBRes = iRes;
                           ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-: ERROR: Existing rollback journal file CANNOT be deleted. "  ));
                           ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-: ERROR: Existing rollback journal file CANNOT be deleted. "  ));
                        }
                        else
                        {
                           ETG_TRACE_COMP  (( " -PBDB.SECURITY-WAL-: SUCCESS: iRes= %d at remove('%s')", iRes, FC_PB_DB_FILE_PATH "/" FC_PB_DB_JOURNAL_FILE_NAME ));
                        }
                     }
#endif
                  }
               }
            }
            // Free the object of SDC permissions.
            sdc_permissions_free(poPermissions);
         }  // end of: if (sdcRes != SDC_OK)   related to:   sdcRes = sdc_permissions_alloc(&poPermissions);
      }
      else
      {
         if (sdcRes == SDC_OK)
         {
            ETG_TRACE_COMP  (( " -PBDB.SECURITY-SDC-: sdc_session_load_storage_key(..): SUCCESS: sdcRes= %d, result_string= '%s' ", sdcRes, sdc_get_error_string(sdcRes) ));
         }
         else
         {
            ETG_TRACE_FATAL (( " -PBDB.SECURITY-SDC-: sdc_session_load_storage_key(..): ERROR:   sdcRes= %d, result_string= '%s' ", sdcRes, sdc_get_error_string(sdcRes) ));
            ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-SDC-: sdc_session_load_storage_key(..): ERROR:   sdcRes= %d, result_string= '%s' ", sdcRes, sdc_get_error_string(sdcRes) ));
         }
      }

      // Close the open SDC session
      sdc_close_session(poSession);
   }

   // When we come here and 'sdcRes == SDC_OK', then the SDC storage could be accessed and a Key-ID/AES-Key is now available.
   // When 'iDBRes  == 0', then the deletion of the database file and journal files (if that was needed) has been performed
   // successfully or this was not needed.
   // I.e. both conditions need to be fullfuilled to continue with the opening of the phonebook data base.
   // Otherwise we will set the 'result' to false here:
   if ( (sdcRes != SDC_OK) || (iDBRes  != 0) )
   {
      result = false;
   }
   // ============= Key-Id availibility check: END =========================================================

#endif   // end of: #ifdef DATABASE_ENCRYPTION_VIA_SEE_AND_SDC


   // Security: BGN: Check if database file exists and which settings are active for "permissions", "owner" and "group" (multi-user target)
   struct stat   oStatBuf;
   QSqlDatabase  db;

   int iResult = 0;

   if (result == true)
   {
      // Security: BGN: Check if database file exists and which settings are active for "permissions", "owner" and "group" (multi-user target)

      // Retrieve current "owner", "group" and "permissions" of database file
      iResult = stat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME, &oStatBuf);
      if( iResult != 0 )
      {
         ETG_TRACE_ERR(( " -PBDB.SECURITY- Startup: 'stat(file, &oStatBuf)' has failed with error=%d, the file probably does not exist. [file='%s'] ",
            iResult, FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME ));
      }
      else
      {
         // Print out "owner", "group" and "permissions" of database file
         ETG_TRACE_USR1(( " -PBDB.SECURITY- Startup: 'stat' database file: owner_number= '%d' ", oStatBuf.st_uid ));
         ETG_TRACE_USR1(( " -PBDB.SECURITY- Startup: 'stat' database file: group_number= '%d' ", oStatBuf.st_gid ));
         ETG_TRACE_USR1(( " -PBDB.SECURITY- Startup: 'stat' database file:  permissions= '%d' (octal) ",
               ((oStatBuf.st_mode>>6)&0x07)*100 + ((oStatBuf.st_mode>>3)&0x07)*10 + (oStatBuf.st_mode&0x07) ));
      }
      // Security: END: Check if database file exists and which settings are active for "permissions", "owner" and "group" (multi-user target)

      // ---------------------------------------
      // This here is the main part to establish the database connection.
      // Note: When the database file does not exist, it will be created by "db.open()" here as empty file (0 byte)

      // Create database connection
      db = QSqlDatabase::addDatabase( "QSQLITE", FC_PB_DB_CONNECTION_NAME );

#if defined(DATABASE_WAL_JOURNALING) && defined(DATABASE_WAL_SHORT_NAMES)
      // If WAL journaling and WAL short names are activated, we use the URI path variant to open the database
      db.setConnectOptions(QStringLiteral("QSQLITE_OPEN_URI"));
      // Construct path to database with URI details
      QString sqzDbPath = QStringLiteral("file:") + QStringLiteral(FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME) + QStringLiteral("?8_3_names=1");
#else
      // Otherwise, construct path to database in the classical way (no URI details)
      QString sqzDbPath = QStringLiteral(FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME);
#endif
      ETG_TRACE_COMP(( " -PBDB.SECURITY- DbPath= '%s' ", sqzDbPath.toLatin1().constData() ))

      // Set database path
      db.setDatabaseName(sqzDbPath);

      // Open database
      if ( !db.open() )
      {
         ETG_TRACE_ERR(( " -PBDB.SECURITY- Startup: ERROR - Cannot open database "));
         result = false;
      }
      // ---------------------------------------
   }

   if (result == true)
   {
      // Security: BGN: Check and (if necessary) adapt database file to the wanted "permissions", "owner" and "group" settings (multi-user target),
      // which are as follows: permission= 660 ("-rw-rw----"), user= aid_phbk, group= aid_phbk_share. (Note: Write access for group has been added for WAL journaling.)
      struct passwd *poWantedPasswd = NULL;
      struct group  *poWantedGroup  = NULL;

      // Retrieve current "owner", "group" and "permissions" of database file - after 'db.open()'
      iResult = 0;
      iResult = stat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME, &oStatBuf);
      if( iResult != 0 )
      {
         // We never expect to come here .. (as "db.open()" should create a new database if there exists currently no one)
         ETG_TRACE_FATAL (( " -PBDB.SECURITY- After 'db.open()': 'stat(file, &oStatBuf)' has FAILED with error=%d, the file probably does not exist. [file='%s'] ",
                            iResult, FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME ));
         ETG_TRACE_ERRMEM(( " -PBDB.SECURITY- After 'db.open()': 'stat(file, &oStatBuf)' has FAILED with error=%d, the file probably does not exist. [file='%s'] ",
                            iResult, FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME ));

         // Set result to false
         result = false;
      }
      else
      {
         // Check if current "permissions" of the database file matches to the wanted ones, if not try to adapt accordingly
         if( (oStatBuf.st_mode & 0x1FF) != (S_IRUSR | S_IWUSR | S_IRGRP  | S_IWGRP) )   // Added group write access (for WAL journaling)
         {
            // Set "-rw-rw----" access for the database file
            ETG_TRACE_ERR(( " -PBDB.SECURITY- After Security Check: Correcting the database 'permissions' to '-rw-rw----' " ));
            iResult = chmod(FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME, S_IRUSR | S_IWUSR | S_IRGRP  | S_IWGRP);   // Added group write access (for WAL journaling)
            if( iResult != 0 )
            {
               ETG_TRACE_FATAL (( " -PBDB.SECURITY- After Security Check: 'chmod(file, -rw-rw----)' has FAILED with error=%d, [file='%s'] ",
                                  iResult, FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME ));
               ETG_TRACE_ERRMEM(( " -PBDB.SECURITY- After Security Check: 'chmod(file, -rw-rw----)' has FAILED with error=%d, [file='%s'] ",
                                  iResult, FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME ));

               // Set result to false
               result = false;
            }
         }

         // Check if current "owner" and "group" matches to the wanted ones, if not try to adapt accordingly
         poWantedPasswd = getpwnam(FC_PB_USER_NAME);
         poWantedGroup  = getgrnam(FC_PB_GROUP_NAME);

         if( (poWantedPasswd) && (poWantedGroup) )
         {
            // Check if current "owner" and "group" for database are correct. If this not o.k., do the needed adaptations.
            if( (poWantedPasswd->pw_uid != oStatBuf.st_uid) || (poWantedGroup->gr_gid != oStatBuf.st_gid) )
            {
               // Change "owner" and "group" settings if current "owner/group" settings are not correct.
               ETG_TRACE_ERR(( " -PBDB.SECURITY- After Security Check: Correcting the database 'owner' and 'group' settings. " ));
               iResult = chown(FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME, poWantedPasswd->pw_uid, poWantedGroup->gr_gid);
               if( iResult != 0 )
               {
                  ETG_TRACE_FATAL (( " -PBDB.SECURITY- After Security Check: ERROR: 'chown(file, %d, %d)' has FAILED with error=%d, [file='%s'] ",
                                     poWantedPasswd->pw_uid, poWantedGroup->gr_gid, iResult, FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME ));
                  ETG_TRACE_ERRMEM(( " -PBDB.SECURITY- After Security Check: ERROR: 'chown(file, %d, %d)' has FAILED with error=%d, [file='%s'] ",
                                     poWantedPasswd->pw_uid, poWantedGroup->gr_gid, iResult, FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME ));

                  // Set result to false
                  result = false;
               }
            }
         }
         else
         {
            ETG_TRACE_FATAL (( " -PBDB.SECURITY- After Security Check: ERROR: Cannot resolve the 'wanted' user-name='%s' ", FC_PB_USER_NAME ));
            ETG_TRACE_FATAL (( " -PBDB.SECURITY- After Security Check: ERROR:   or the 'wanted' group-name='%s' ",  FC_PB_GROUP_NAME ));
            ETG_TRACE_ERRMEM(( " -PBDB.SECURITY- After Security Check: ERROR: Cannot resolve the 'wanted' user-name='%s' ", FC_PB_USER_NAME ));
            ETG_TRACE_ERRMEM(( " -PBDB.SECURITY- After Security Check: ERROR:   or the 'wanted' group-name='%s' ",  FC_PB_GROUP_NAME ));

            // Set result to false
            result = false;
         }

         // Print out "owner", "group" and "permissions" of database file (after security check) and related changes might have been done.
         iResult = 0;
         iResult = stat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME, &oStatBuf);
         if( iResult != 0 )
         {
            ETG_TRACE_FATAL (( " -PBDB.SECURITY- After Security Check: ERROR: 'stat(file, &oStatBuf)' has FAILED with error=%d, the file probably does not exist. [file='%s'] ",
                               iResult, FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME ));
            ETG_TRACE_ERRMEM(( " -PBDB.SECURITY- After Security Check: ERROR: 'stat(file, &oStatBuf)' has FAILED with error=%d, the file probably does not exist. [file='%s'] ",
                               iResult, FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME ));
         }
         else
         {
            ETG_TRACE_USR1(( " -PBDB.SECURITY- After Security Check:  'stat' database file: owner_number= '%d' ", oStatBuf.st_uid ));
            ETG_TRACE_USR1(( " -PBDB.SECURITY- After Security Check:  'stat' database file: group_number= '%d' ", oStatBuf.st_gid ));
            ETG_TRACE_USR1(( " -PBDB.SECURITY- After Security Check:  'stat' database file:  permissions= '%d' (octal) ",
                             ((oStatBuf.st_mode>>6)&0x07)*100 + ((oStatBuf.st_mode>>3)&0x07)*10 + (oStatBuf.st_mode&0x07) ));
         }
      }
      // Security: END: Check and (if necessary) adapt database file to the wanted "permissions", "owner" and "group" settings (multi-user target)
   }


   // STATUS: When we come here the phonebook database file is available (can be empty, 0 bytes) and it has correct user and group settings
   //         and the database has been opened successfully via Qt [otherwise the 'result' is false].

   sqlite3 *handle = NULL;
   if (result == true)
   {
      // Check if can get a valid (not NULL) database handle

      // Get the Database Driver Handle
      qVariant = db.driver()->handle();

      if (qVariant.isValid() && u8StringCompare(qVariant.typeName(), "sqlite3*") == 0)
      {
         // Get Database Handle
         handle = *static_cast<sqlite3 **> (qVariant.data());

         if (handle)
         {
            // Database handle is not NULL, i.e. we assume that it is valid
            ETG_TRACE_USR4(( "Able to get Sqlite3 driver handle"));

            // Until here we have success, so we set result to true.
            result = true;
         }
         else
         {
            ETG_TRACE_FATAL (( " Sqlite database handle is NULL. " ));
            ETG_TRACE_ERRMEM(( " Sqlite database handle is NULL. " ));
            result = false;
         }
      }
      else
      {
         ETG_TRACE_FATAL (( " Unable to get Sqlite3 driver handle. " ));
         ETG_TRACE_ERRMEM(( " Unable to get Sqlite3 driver handle. " ));
         result = false;
      }
   }


   // STATUS: If we come here we have valid database handle [otherwise the 'result' is false].


#ifdef DATABASE_ENCRYPTION_VIA_SEE_AND_SDC
   if (result == true)
   {
      // ======================= Check if DB is encrypted: BGN ==========================================

      // When we come here, we have passed the Key-Id availibility check at the begin of the method and
      // now can check if the database is encrypted or not.

      bool bIsDbEncrypted = false;
      sqlite3_stmt* pStmt = NULL;

      // Execute (simple) "read" query in 'sqlite_master' table to check if database is encrypted.
      iResult = sqlite3_prepare_v2(handle, "SELECT name FROM sqlite_master WHERE type = 'table' LIMIT 1;", -1, &pStmt, 0);

      int iResult2 = 0;
      if( (SQLITE_OK != iResult) || (SQLITE_ROW != (iResult2 = sqlite3_step(pStmt))) )
      {
         // Either the 'prepare'result was NOT OK or we have not been able to retrieve an 'SQLITE_ROW' via the 'step' command,
         // so we assume the DB is encrypted.
         bIsDbEncrypted = true;
      }
      else
      {
         // The 'prepare' and the 'step' command have been successful, so we assume the DB is NOT encrypted.
         bIsDbEncrypted = false;
      }

      // Finalize query (which has checked if the DB is encrypted).
      //(Note: according to SQLite docu: Invoking sqlite3_finalize() on a NULL pointer is a harmless no-op)
      sqlite3_finalize(pStmt);

      ETG_TRACE_COMP(( " -PBDB.SECURITY-SDC-: Check if DB is encrypted: bResult=%d ", bIsDbEncrypted ));
      // ======================= Check if DB is encrypted: END ==========================================

      // ======================= If DB is encrypted, use 'sqlite3_key_v2(..)': BGN ======================
      if(bIsDbEncrypted)
      {  // Database is assumed to be encrypted.
         // -  This means we need to close the database, open it again and then use the sqlite3_key_v2(..) statement.
         //    This is necessary as the "key" statement shall be the first command after DB has been opened.

         // Close the DB (I have re-used the code from method 'CloseDatabase()', but without the mutex part.)
         bool bContains = QSqlDatabase::contains(FC_PB_DB_CONNECTION_NAME);
         ETG_TRACE_COMP((" Check if a database connection exists: bResult= %d", bContains));
         if(bContains)
         {
            if(Database().isValid())
            {
               if(Database().isOpen() == TRUE)
               {
                  Database().close();
               }
               QSqlDatabase::removeDatabase(FC_PB_DB_CONNECTION_NAME);
            }
         }

         // Reopen the database as the database seems to be encrypted and we intend to activate the encryption key
         // ---------------------------------------
         ETG_TRACE_COMP(( " Reopen the database. " ));

         // Create database connection
         db = QSqlDatabase::addDatabase( "QSQLITE", FC_PB_DB_CONNECTION_NAME );

#if defined(DATABASE_WAL_JOURNALING) && defined(DATABASE_WAL_SHORT_NAMES)
         // If WAL journaling and WAL short names are activated, we use the URI path variant to open the database
         db.setConnectOptions(QStringLiteral("QSQLITE_OPEN_URI"));
         // Construct path to database with URI details
         QString sqzDbPath = QStringLiteral("file:") + QStringLiteral(FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME) + QStringLiteral("?8_3_names=1");
#else
         // Otherwise, consruct path to database in the classical way (no URI details)
         QString sqzDbPath = QStringLiteral(FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME);
#endif
         ETG_TRACE_COMP(( " -PBDB.SECURITY- DbPath= '%s' ", sqzDbPath.toLatin1().constData() ))

         // Set database path
         db.setDatabaseName(sqzDbPath);

         // Open database
         if ( !db.open() )
         {
            ETG_TRACE_FATAL (( " -PBDB.SECURITY- Startup: ERROR - Cannot reopen database "));
            ETG_TRACE_ERRMEM(( " -PBDB.SECURITY- Startup: ERROR - Cannot reopen database "));
            return false;
         }
         // ---------------------------------------

         // Get the "fresh" Database Driver Handle (from the reopened database)
         qVariant = db.driver()->handle();

         if (qVariant.isValid() && u8StringCompare(qVariant.typeName(), "sqlite3*") == 0)
         {
            // Get Database Handle
            handle = *static_cast<sqlite3 **> (qVariant.data());

            if (handle)
            {
               // Until here we have success, so we set 'result' to true.
               result = true;

               // Activate encryption for the opened phonebook database by setting the related Key-ID (via SDC)
               int iRes = sqlite3_key_v2(
                                handle,            // sqlite3 *db,          [database handle]
                                NULL,              // const char *zDbName,  [use "main" or equivalently NULL (main database connection) here]
                                (void *) &keyId,   // const void *pKey,     [Key-ID (or AES-Key)]
                                0);                // int nKey              [key length]
               // Notes:
               // - 'nKey' == 0 indicates that a 'Key-ID' (instead of an real 'AES-Key') is used for parameter 'pKey'.
               // - 'Key-ID' ranges (for ADIT, ESO, CMD, Project, Customer or 'Dynamic' components) are defined at:
               //   https://inside-docupedia.bosch.com/confluence/display/gen3generic/SDC+Key+Id+Range

               if (iRes == SQLITE_OK)
               {
                  result = true;
                  ETG_TRACE_COMP  (( " -PBDB.SECURITY-SDC-: SUCCESS: sqlite3_key_v2(..): iRes= %d ", iRes ));
               }
               else
               {
                  result = false;
                  ETG_TRACE_FATAL (( " -PBDB.SECURITY-SDC-: ERROR:   sqlite3_key_v2(..): iRes= %d ", iRes ));
                  ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-SDC-: ERROR:   sqlite3_key_v2(..): iRes= %d ", iRes ));
               }

// -----
               if(result == true)
               {
                  // Check if the database is really encrypted, i.e. if we now (in encrypted mode) can successfully execute a query.

                  // First we need to check if the phonebook database file has a size > 0. A size of 0 would indicate that we have
                  // an empty database and on an empty database file any query would fail.
                  struct stat oStatBuf;
                  iResult = stat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_FILE_NAME, &oStatBuf);
                  if( (0 == iResult) && (oStatBuf.st_size > 0) )
                  {
                     // The stat command was successfull and the file size is greater than 0, so that we can assume that we have a
                     // database, which contains at least the table structure.

                     // Execute (simple) "read" query in 'sqlite_master' table to check if database is encrypted.
                     pStmt = NULL;
                     int iRes = sqlite3_prepare_v2(handle, "SELECT name FROM sqlite_master WHERE type = 'table' LIMIT 1;", -1, &pStmt, 0);

                     int iRes2 = 0;
                     if( (SQLITE_OK != iRes) || (SQLITE_ROW != (iRes2 = sqlite3_step(pStmt))) )
                     {
                        // Either the 'prepare'result was NOT OK or we have not been able to retrieve an 'SQLITE_ROW' via the 'step' command,
                        // This is a severe issue as the assumption that the database is encrypted could not be confirmed.
                        result = false;
                        ETG_TRACE_FATAL (( " -PBDB.SECURITY-SDC-: FATAL ERROR: DB encryption status not clear. sqlite3_prepare_v2(..): iRes= %d, iRes2= %d ", iRes, iRes2 ));
                        ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-SDC-: FATAL ERROR: DB encryption status not clear. sqlite3_prepare_v2(..): iRes= %d, iRes2= %d ", iRes, iRes2 ));

                        // As we have dedected a severe DB encryption issue and we cannot work with this DB, we trigger a database 're-creation'.
                        // Note: The removal of the "corrupted" database and the creation of a new one will be done in next power cycle (or to be more precice: in next call of 'vOnLoadSettings')

                        // Set the database "recreation trigger file" to the FFS
                        vSetDbRecreationTrigger();

                        NORMAL_M_ASSERT_ALWAYS();    // With the normal assert we can see from the callstack from which method we have been called.
                     }
                     else
                     {
                        // The 'prepare' and the 'step' command have been successful, so we have confirmed that the DB is encrypted.
                        // We can continue to operate with the encrypted database.
                        ETG_TRACE_FATAL (( " -PBDB.SECURITY-SDC-: INFO: We have reconfirmed that the DB is encrypted. "  ));
                     }

                     // Finalize query (which has checked if the DB is encrypted).
                     //(Note: according to SQLite docu: Invoking sqlite3_finalize() on a NULL pointer is a harmless no-op)
                     sqlite3_finalize(pStmt);
                  }
               }
// -----
               // The following steps make sense only, if the database is really operable, i.e. if 'result == true'
               if(result == true)
               {
                  // Set SQLite memory 'page_size' (e.g. to 4096 byte)
                  char *pzErrMsg = NULL;
                  // --
                  int iPageSize = 4096;
                  QString sql = "PRAGMA page_size=" + QString::number(iPageSize);
                  // --
                  iRes = sqlite3_exec(handle, sql.toLatin1().constData(), /*callback*/ NULL, 0, &pzErrMsg);
                  if (iRes == SQLITE_OK)
                  {
                     ETG_TRACE_COMP  (( " -PBDB.PRAGMA-: Set 'PRAGMA page_size=%d': iRes= %d ", iPageSize, iRes ));
                  }
                  else
                  {
                     if(pzErrMsg)
                     {
                        ETG_TRACE_FATAL (( " -PBDB.PRAGMA-: Set 'PRAGMA page_size = %d:' ERROR: ErrMsg= %s ", iPageSize, pzErrMsg ));
                        ETG_TRACE_ERRMEM(( " -PBDB.PRAGMA-: Set 'PRAGMA page_size = %d:' ERROR: ErrMsg= %s ", iPageSize, pzErrMsg ));
                     }
                     else
                     {
                        ETG_TRACE_FATAL (( " -PBDB.PRAGMA-: Set 'PRAGMA page_size = %d:' ERROR ", iPageSize ));
                        ETG_TRACE_ERRMEM(( " -PBDB.PRAGMA-: Set 'PRAGMA page_size = %d:' ERROR ", iPageSize ));
                     }
                     result = false;
                  }
                  if(pzErrMsg != NULL)
                  {
                     sqlite3_free(pzErrMsg);
                  }
               }
            }
            else
            {
               ETG_TRACE_FATAL (( " Sqlite database handle is NULL. " ));
               ETG_TRACE_ERRMEM(( " Sqlite database handle is NULL. " ));
               result = false;
            }
         }
         else
         {
            ETG_TRACE_FATAL (( " Unable to get Sqlite3 driver handle. " ));
            ETG_TRACE_ERRMEM(( " Unable to get Sqlite3 driver handle. " ));
            result = false;
         }
      }
      else
      {  // When we come here the database is assumed to be NOT encrypted.
         // We have two options here:
         // - Let the system run with an unencrypted database (debug mode),
         // - or alternatively we could close it and return false as ENCRYPTION is activated,
         //   but the current DB is not ecrypted.

         // We have deciced to let the system run with an unencrypted DB here as this situation
         // is only possible, if we have decrypted the phonebook database with an external tool
         // for debugging.
         // In this situation no action is needed here as we have already a valid database handle,
         // which we have achieved before we have entered the "encryption check".
      }
      // ======================= If DB is encrypted, use 'sqlite3_key_v2(..)': END ======================
   }
#endif  // end of: #ifdef DATABASE_ENCRYPTION_VIA_SEE_AND_SDC


   // STATUS: If we come here we have a valid database, which is open and the database handle is valid
   //         [otherwise the 'result' is false].


   if( (result == true) && handle )
   {
      // Set 'locking_mode' to 'NORMAL' (i.e. overriding a potential SQLite build option 'locking_mode' = 'EXCLUSIVE')
      // [Motivation: When this application (FC_Phonebook) has been built with an SQLite version,
      // which has been built with option 'SQLITE_DEFAULT_LOCKING_MODE=1', then we will revert this now by the following pragma statement.
      char *pzErrMsg = NULL;
      int iRes = sqlite3_exec(handle, "PRAGMA locking_mode=NORMAL", /*callback*/ NULL, 0, &pzErrMsg);
      if (iRes == SQLITE_OK)
      {
         ETG_TRACE_COMP  (( " -PBDB.PRAGMA-: Set 'PRAGMA locking_mode=NORMAL': SUCCESS: iRes= %d ", iRes ));
      }
      else
      {
         ETG_TRACE_FATAL (( " -PBDB.PRAGMA-: Set 'PRAGMA locking_mode=NORMAL': ERROR: ErrMsg= %s ", pzErrMsg ));
         ETG_TRACE_ERRMEM(( " -PBDB.PRAGMA-: Set 'PRAGMA locking_mode=NORMAL': ERROR: ErrMsg= %s ", pzErrMsg ));
         result = false;
      }
      if(pzErrMsg != NULL)
      {
         sqlite3_free(pzErrMsg);
      }

#ifdef DATABASE_WAL_JOURNALING
      // Set 'journaling mode' to 'WAL' (Write Ahead Logging) which allows concurrent 'read' and 'write' by different processes.
      // [Motivation: Improve concurrent DB access from Phonebook and Speech Transcription Service]
      pzErrMsg = NULL;
      iRes = sqlite3_exec(handle, "PRAGMA journal_mode=WAL", /*callback*/ NULL, 0, &pzErrMsg);
      if (iRes == SQLITE_OK)
      {
         ETG_TRACE_COMP  (( " -PBDB.PRAGMA-: Set 'PRAGMA journal_mode=WAL': SUCCESS: iRes= %d ", iRes ));
      }
      else
      {
         ETG_TRACE_FATAL (( " -PBDB.PRAGMA-: Set 'PRAGMA journal_mode=WAL' ERROR: ErrMsg= %s ", pzErrMsg ));
         ETG_TRACE_ERRMEM(( " -PBDB.PRAGMA-: Set 'PRAGMA journal_mode=WAL' ERROR: ErrMsg= %s ", pzErrMsg ));
         result = false;
      }
      if(pzErrMsg != NULL)
      {
         sqlite3_free(pzErrMsg);
      }


#if 0  // 1= For TEST of smaller number of pages for auto checkpointing; 0= Default number of pages for checkpointing
      // Set 'wal_autocheckpoint' to N (pages) e.g. 100 for test (Default from SQLite is 1000)
      int iPages = 100;
      iRes = sqlite3_wal_autocheckpoint(handle, iPages);

      if (iRes == SQLITE_OK)
      {
         ETG_TRACE_COMP  (( " -PHBK.SECURITy-WAL-: sqlite3_wal_autocheckpoint(handle, iPages=%d): SUCCESS: iRes= %d ", iPages, iRes ));
      }
      else
      {
         ETG_TRACE_FATAL (( " -PHBK.SECURITy-WAL-: sqlite3_wal_autocheckpoint(., iPages=%d): ERROR: ErrCode= %d, ErrMsg= %s ", iPages, sqlite3_errcode(handle), sqlite3_errmsg(handle) ));
         ETG_TRACE_ERRMEM(( " -PHBK.SECURITy-WAL--: sqlite3_wal_autocheckpoint(., iPages=%d): ERROR: ErrCode= %d, ErrMsg= %s ", iPages, sqlite3_errcode(handle), sqlite3_errmsg(handle) ));
         result = false;
      }
#endif

      // Set SQLite 'synchronous' (mode) to 'NORMAL' via 'pragma' statement.
      // When synchronous is NORMAL (1), the SQLite database engine will still sync at the most critical moments, but less often than in FULL mode.
      // ...
      // The synchronous=NORMAL setting is a good choice for most applications running in WAL mode.
      // See also: https://www.sqlite.org/pragma.html#pragma_synchronous
      pzErrMsg = NULL;
      iRes = sqlite3_exec(handle, "PRAGMA synchronous=NORMAL", /*callback*/ NULL, 0, &pzErrMsg);
      if (iRes == SQLITE_OK)
      {
         ETG_TRACE_COMP  (( " -PBDB.PRAGMA-: Set 'PRAGMA synchronous=NORMAL': SUCCESS: iRes= %d ", iRes ));
      }
      else
      {
         ETG_TRACE_FATAL (( " -PBDB.PRAGMA-: Set 'PRAGMA synchronous=NORMAL' ERROR: ErrMsg= %s ", pzErrMsg ));
         ETG_TRACE_ERRMEM(( " -PBDB.PRAGMA-: Set 'PRAGMA synchronous=NORMAL' ERROR: ErrMsg= %s ", pzErrMsg ));
         result = false;
      }
      if(pzErrMsg != NULL)
      {
         sqlite3_free(pzErrMsg);
      }
#endif

      // Register CM sorting library as collation at SQLite
      rc = sortlib_register_sort_function( handle );

      if (rc != SQLITE_OK)
      {
         ETG_TRACE_FATAL (( " Error while registering the CM sorting function. " ));
         ETG_TRACE_ERRMEM(( " Error while registering the CM sorting function. " ));
         result = false;
      }
      else
      {
         ETG_TRACE_COMP(( " The CM Sorting Library (sorting collation) has been registered successfully. " ));
         result = true;
      }

#ifdef ENABLE_SQLITE_ICU_EXTENSION
      //CMG3G-12949
      // SQLite only understands upper/lower case for ASCII characters by default. The LIKE operator is case sensitive
      // by default for unicode characters that are beyond the ASCII range. For example, the expression 'a' LIKE 'A' is TRUE
      // but 'æ' LIKE 'Æ' is FALSE. The ICU extension to SQLite includes an enhanced version of the LIKE operator
      // that does case folding across all unicode characters.
      int iEnableExtensions =  sqlite3_enable_load_extension(
               handle,  //sqlite3 *db,
               1        // int onoff (1= ON, 0= Off)
               );
      if(iEnableExtensions == SQLITE_OK)
      {
         ETG_TRACE_FATAL(( " Sqlite 'load_extensions' have been switched ON . " ));
      }
      else
      {
         ETG_TRACE_FATAL (( " FATAL: Sqlite cannot switch ON 'load_extensions'. " ));
         ETG_TRACE_ERRMEM(( " FATAL: Sqlite cannot switch ON 'load_extensions'. " ));
      }
      char *pzErrMsg2 = NULL;
      int iIcuLoadResult = sqlite3_load_extension(
                                   handle,                // sqlite3 *db          /* Load the extension into this database connection */
                                   "libSqliteIcu.so.0",     // const char *zFile,   /* Name of the shared library containing extension */
                                   NULLPTR,                     // const char *zProc,   /* Entry point.  Derived from zFile if 0 */
                                   &pzErrMsg2             // char **pzErrMsg2     /* Put error message here if not 0 */
      );
      if(iIcuLoadResult == SQLITE_OK)
      {
         ETG_TRACE_FATAL(( " Sqlite extension 'libSqliteIcu.so.0' was loaded successfully. " ));
      }
      else
      {
         ETG_TRACE_FATAL (( " FATAL: Sqlite extension 'libSqliteIcu.so.0' could not be loaded. " ));
         ETG_TRACE_FATAL (( " FATAL: error text= '%s'. ", pzErrMsg2 ));
         ETG_TRACE_ERRMEM(( " FATAL: Sqlite extension 'libSqliteIcu.so.0' could not be loaded. " ));
         ETG_TRACE_ERRMEM(( " FATAL: error text= '%s'. ", pzErrMsg2 ));
      }

      int iDisableExtensions = sqlite3_enable_load_extension(
               handle,  // sqlite3 *db,
               0        // int onoff (1= ON, 0= Off)
               );
      if(iDisableExtensions == SQLITE_OK)
      {
         ETG_TRACE_FATAL(( " Sqlite 'load_extensions' have been switched OFF . " ));
      }
      else
      {
         ETG_TRACE_FATAL (( " FATAL: Sqlite cannot switch OFF 'load_extensions'. " ));
         ETG_TRACE_ERRMEM(( " FATAL: Sqlite cannot switch OFF 'load_extensions'. " ));
      }

      if(pzErrMsg2 != NULL)
      {
         sqlite3_free(pzErrMsg2);
      }
#endif

      if(db.driver()->hasFeature(QSqlDriver::Transactions))
      {
         ETG_TRACE_COMP(( "Database driver supports Transaction" ));
      }
      else
      {
         ETG_TRACE_ERR(( "ERROR: INFO: Database driver doesn't support Transaction" ));
      }

   }   // end of: if( (result == true) && handle )


   if (result == false)
   {
      // When we come here we have NOT been able to open the database successfully or to get a valid database handle.
      // Therefore we try to close the database in dedicated steps, so that we do not have to consider the dedicted
      // stage, which might have been already reached.

      // Close the DB (I have re-used the code from method 'CloseDatabase()', but without the mutex part.)
      bool bContains = QSqlDatabase::contains(FC_PB_DB_CONNECTION_NAME);
      ETG_TRACE_USR2((" Check if a database connection exists: bResult= %d", bContains));
      if(bContains)
      {
         if(Database().isValid())
         {
            if(Database().isOpen() == TRUE)
            {
               Database().close();
            }
            QSqlDatabase::removeDatabase(FC_PB_DB_CONNECTION_NAME);
         }
      }
   }

   return result;
}


/*******************************************************************************
*
* FUNCTION: bool FC_PhoneBook_SQLite::bAdjustWalJournalingNamesAndLinks()
*
* DESCRIPTION:
*           - Internal function to check, set or repair WAL and SHM file names and related symbolic links, based on the compiler switches:
*              'DATABASE_WAL_JOURNALING'
*              'DATABASE_WAL_SHORT_NAMES'
*           - The challenge here is, that the dynamic partition might have still the status of the last SW version (in case of USB stick SW update),
*             but the journaling scheme (CLASSICAL or WAL) or the WAL SHORT/LONG name scheme has changed with the new SW version.
*           - This function shall do the needed modifications ("repair actions") in such a "transition" scenario. Otherwise this function shall
*             do nothing and return as fast as possible.
*             If the related "repair" actions cannot be excuted completely, this function will return 'false'.
*
*           Note:
*           - Both "transition directions", i.e. from "WAL LONG NAME" to 'DATABASE_WAL_SHORT_NAMES'
*             and from 'DATABASE_WAL_SHORT_NAMES' to "WAL LONG NAME" are fully supported.
*
* PARAMETER: IN:  none
*
* RETURNVALUE: bool - Indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::bAdjustWalJournalingNamesAndLinks()
{
   bool bRes = true;
   struct stat oStatBuf;

#ifdef DATABASE_WAL_SHORT_NAMES

   // Case: "DATABASE_WAL_SHORT_NAMES"
   // (1.S) Check if now old/obsolete links ["phonebook.wal -> phonebook.db-wal" and "phonebook.shm -> phonebook.db-shm"] exist and if this is the case,
   //       remove them.
   // (2.S) Check if the "LONG NAME" related wal and shm files [phonebook.db-wal and phonebook.db-shm] exist and if this is the case, rename them to
   //       the "SHORT NAME" variants [phonebook.wal and phonebook.shm].

   //------------------------
   // WAL
   //------------------------
   // (1.S) Check if a now obsolete/old symbolic link to WAL file exists ["phonebook.wal -> phonebook.db-wal"]
   //       Note: We are using lstat (instead of stat) in combination with "S_ISLNK(.)" here to check for the link.
   if( (0 == lstat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_LINK_NAME, &oStatBuf)) && (S_ISLNK(oStatBuf.st_mode) == 1) )
   {
      ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: INFO: Obsolete/old symlink to WAL file NEEDS to be REMOVED. " ));
      ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: INFO: Obsolete/old symlink to WAL file NEEDS to be REMOVED. " ));

      // We need to remove the symbolic link "phonebook.wal -> phonebook.db-wal"
      // Note: The function 'remove(.)' does not follow an existing symlink, i.e. the symlink is removed, not its target.
      if( 0 == remove( FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_LINK_NAME) )
      {
         ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: SUCCESS: Obsolete/old symlink to WAL file has been removed. [link= '../%s'] ", FC_PB_DB_WAL_LINK_NAME ));
         ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: SUCCESS: Obsolete/old symlink to WAL file has been removed. [link= '../%s'] ", FC_PB_DB_WAL_LINK_NAME ));
      }
      else
      {
         ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: ERROR: Cannot remove obsolete/old symlink to WAL file. [link= '../%s'] ", FC_PB_DB_WAL_LINK_NAME ));
         ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: ERROR: Cannot remove obsolete/old symlink to WAL file. [link= '../%s'] ", FC_PB_DB_WAL_LINK_NAME ));
         bRes = false;
      }
   }

   // (2.S) Check if a related "old" (LONG NAME) WAL file still exists and therefore has to be renamed. If needed, try to do so.
   if( 0 == stat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_LONG_WAL_FILE_NAME, &oStatBuf) )
   {
      ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: INFO: Obsolete/old WAL file exists (and has to be renamed), [file= '%s'] ", FC_PB_DB_FILE_PATH "/" FC_PB_DB_LONG_WAL_FILE_NAME ));
      ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: INFO: Obsolete/old WAL file exists (and has to be renamed), [file= '%s'] ", FC_PB_DB_FILE_PATH "/" FC_PB_DB_LONG_WAL_FILE_NAME ));

      // Rename 'phonebook.db-wal' to 'phonebook.wal'
      if( 0 == rename(FC_PB_DB_FILE_PATH "/" FC_PB_DB_LONG_WAL_FILE_NAME, FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHORT_WAL_FILE_NAME) )
      {
         ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: SUCCESS: Obsolete/old WAL file has been renamed successfully. " ));
         ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: SUCCESS: Obsolete/old WAL file has been renamed successfully. " ));
      }
      else
      {
         ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: ERROR: Cannot rename obsolete/old WAL file to '%s'. ", FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHORT_WAL_FILE_NAME ));
         ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: ERROR: Cannot rename Obsolete/old WAL file to '%s'. ", FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHORT_WAL_FILE_NAME ));
         bRes = false;
      }
   }

   //------------------------
   // SHM
   //------------------------
   // (1.S) Check if a now obsolete/old symbolic link to SHM file exists ["phonebook.shm -> phonebook.db-shm"]
   //       Note: We are using lstat (instead of stat) in combination with "S_ISLNK(.)" here to check for the link.
   if( (0 == lstat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHM_LINK_NAME, &oStatBuf)) && (S_ISLNK(oStatBuf.st_mode) == 1) )
   {
      ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: INFO: Obsolete/old symlink to SHM file NEEDS to be REMOVED. " ));
      ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: INFO: Obsolete/old symlink to SHM file NEEDS to be REMOVED. " ));

      // We need to remove the symbolic link "phonebook.shm -> phonebook.db-shm"
      // Note: The function 'remove(.)' does not follow an existing symlink, i.e. the symlink is removed, not its target.
      if( 0 == remove( FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHM_LINK_NAME) )
      {
         ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: SUCCESS: Obsolete/old symlink to SHM file has been removed. [link= '../%s'] ", FC_PB_DB_SHM_LINK_NAME ));
         ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: SUCCESS: Obsolete/old symlink to SHM file has been removed. [link= '../%s'] ", FC_PB_DB_SHM_LINK_NAME ));
      }
      else
      {
         ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: ERROR: Cannot remove obsolete/old symlink to SHM file. [link= '../%s'] ", FC_PB_DB_SHM_LINK_NAME ));
         ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: ERROR: Cannot remove obsolete/old symlink to SHM file. [link= '../%s'] ", FC_PB_DB_SHM_LINK_NAME ));
         bRes = false;
      }
   }

   // (2.S) Check if a related "old" (LONG NAME) SHM file still exists and therefore has to be renamed. If needed, try to do so.
   if( 0 == stat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_LONG_SHM_FILE_NAME, &oStatBuf) )
   {
      ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: INFO: Obsolete/old SHM file exists (and has to be renamed), [file= '%s'] ", FC_PB_DB_FILE_PATH "/" FC_PB_DB_LONG_SHM_FILE_NAME ));
      ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: INFO: Obsolete/old SHM file exists (and has to be renamed), [file= '%s'] ", FC_PB_DB_FILE_PATH "/" FC_PB_DB_LONG_SHM_FILE_NAME ));

      // Rename 'phonebook.db-shm' to 'phonebook.shm'
      if( 0 == rename(FC_PB_DB_FILE_PATH "/" FC_PB_DB_LONG_SHM_FILE_NAME, FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHORT_SHM_FILE_NAME) )
      {
         ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: SUCCESS: Obsolete/old SHM file has been renamed successfully. " ));
         ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: SUCCESS: Obsolete/old SHM file has been renamed successfully. " ));
      }
      else
      {
         ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: ERROR: Cannot rename obsolete/old SHM file to '%s'. ", FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHORT_SHM_FILE_NAME ));
         ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: ERROR: Cannot rename obsolete/old SHM file to '%s'. ", FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHORT_SHM_FILE_NAME ));
         bRes = false;
      }
   }

#else

   // Case: "DATABASE WAL LONG NAMES"
   // (1.L) Check if a now obsolete/old (SHORT NAME related) WAL or SHM file exists as regular file and therefore have to be renamed. If needed, try to do so.
   // (2.L) Check if the symbolic links from "short" name (used by Speech SDS TS) to "long" name (used by FC_Phonebook in case of "DATABASE WAL LONG NAMES")
   //       for the WAL and SHM file exist, i.e. ["phonebook.wal -> phonebook.db-wal" and "phonebook.shm -> phonebook.db-shm"]. If these links do not exist,
   //       they will be created.

   //------------------------
   // WAL
   //------------------------
   // (1.L) Check if a now obsolete/old (SHORT NAME related) WAL file exists as regular file and therefore has to be renamed. If needed, try to do so.
   //       Note: We are using lstat (instead of stat) in combination with "S_ISREG(.)" here to check for a regular file.
   if( (0 == lstat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHORT_WAL_FILE_NAME, &oStatBuf))  && (S_ISREG(oStatBuf.st_mode) == 1) )
   {
      ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: INFO: Obsolete/old WAL file exists (and has to be renamed), [file= '%s'] ", FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHORT_WAL_FILE_NAME ));
      ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: INFO: Obsolete/old WAL file exists (and has to be renamed), [file= '%s'] ", FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHORT_WAL_FILE_NAME ));

      // Rename 'phonebook.wal' to 'phonebook.db-wal'
      if( 0 == rename(FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHORT_WAL_FILE_NAME, FC_PB_DB_FILE_PATH "/" FC_PB_DB_LONG_WAL_FILE_NAME) )
      {
         ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: SUCCESS: The WAL file has been renamed successfully. " ));
         ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: SUCCESS: The WAL file has been renamed successfully. " ));
      }
      else
      {
         ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: ERROR: Could not rename WAL file to '%s'. ", FC_PB_DB_FILE_PATH "/" FC_PB_DB_LONG_WAL_FILE_NAME ));
         ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: ERROR: Could not rename WAL file to '%s'. ", FC_PB_DB_FILE_PATH "/" FC_PB_DB_LONG_WAL_FILE_NAME ));
         bRes = false;
      }
   }

   // (2.L) Check if the needed WAL link ["phonebook.wal -> phonebook.db-wal"] exists and if this is NOT the case, create it.
   //       Note: We are using lstat (instead of stat) in combination with "S_ISLNK(.)" here to check for the symbolic link.
   if( (0 == lstat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_LINK_NAME, &oStatBuf)) && (S_ISLNK(oStatBuf.st_mode) == 1) )
   {
      ETG_TRACE_COMP(( " -PBDB.SECURITY-WAL-Names_Links-: INFO: The needed symlink to WAL file exists correctly. " ));
   }
   else
   {
      // We need to create the symbolic link "phonebook.shm -> phonebook.db-shm"
      ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: INFO: The WAL related link NEEDS to be created. " ));
      ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: INFO: The WAL related link NEEDS to be created. " ));

      // We need to create the symbolic link "ln -s phonebook.db-shm phonebook.shm"
      // Note: A link can be created even if the file does not exist.
      if( 0 == symlink(FC_PB_DB_LONG_WAL_FILE_NAME, FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_LINK_NAME) )
      {
         ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: SUCCESS: The WAL related link has been created successfully. " ));
         ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: SUCCESS: The WAL related link has been created successfully. " ));
      }
      else
      {
         ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: ERROR: The WAL related link CANNOT be created. " ));
         ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: ERROR: The WAL related link CANNOT be created. " ));
         bRes = false;
      }
   }

   //------------------------
   // SHM
   //------------------------
   // (1.L) Check if a now obsolete/old (SHORT NAME related) SHM file exists as regular file and therefore has to be renamed. If needed, try to do so.
   //       Note: We are using lstat (instead of stat) in combination with "S_ISREG(.)" here to check for a regular file.
   if( (0 == lstat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHORT_SHM_FILE_NAME, &oStatBuf))  && (S_ISREG(oStatBuf.st_mode) == 1) )
   {
      ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: INFO: Obsolete/old SHM file exists (and has to be renamed), [file= '%s'] ", FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHORT_SHM_FILE_NAME ));
      ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: INFO: Obsolete/old SHM file exists (and has to be renamed), [file= '%s'] ", FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHORT_SHM_FILE_NAME ));

      // Rename 'phonebook.shm' to 'phonebook.db-shm'
      if( 0 == rename(FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHORT_SHM_FILE_NAME, FC_PB_DB_FILE_PATH "/" FC_PB_DB_LONG_SHM_FILE_NAME) )
      {
         ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: SUCCESS: The SHM file has been renamed successfully. " ));
         ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: SUCCESS: The SHM file has been renamed successfully. " ));
      }
      else
      {
         ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: ERROR: Could not rename SHM file to '%s'. ", FC_PB_DB_FILE_PATH "/" FC_PB_DB_LONG_SHM_FILE_NAME ));
         ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: ERROR: Could not rename SHM file to '%s'. ", FC_PB_DB_FILE_PATH "/" FC_PB_DB_LONG_SHM_FILE_NAME ));
         bRes = false;
      }
   }

   // (2.L) Check if the needed SHM link ["phonebook.shm -> phonebook.db-shm"] exists and if this is NOT the case, create it.
   //       Note: We are using lstat (instead of stat) in combination with "S_ISLNK(.)" here to check for the link.
   if( (0 == lstat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHM_LINK_NAME, &oStatBuf)) && (S_ISLNK(oStatBuf.st_mode) == 1) )
   {
      ETG_TRACE_COMP(( " -PBDB.SECURITY-WAL-Names_Links-: INFO: The needed symlink to SHM file exists correctly. " ));
   }
   else
   {
      // We need to create the symbolic link "phonebook.shm -> phonebook.db-shm"
      ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: INFO: The SHM related link NEEDS to be created. " ));
      ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: INFO: The SHM related link NEEDS to be created. " ));

      // We need to create the symbolic link "ln -s phonebook.db-shm phonebook.shm"
      // Note: A link can be created even if the file does not exist.
      if( 0 == symlink(FC_PB_DB_LONG_SHM_FILE_NAME, FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHM_LINK_NAME) )
      {
         ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: SUCCESS: The SHM related link has been created successfully. " ));
         ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: SUCCESS: The SHM related link has been created successfully. " ));
      }
      else
      {
         ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Names_Links-: ERROR: The SHM related link CANNOT be created. " ));
         ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Names_Links-: ERROR: The SHM related link CANNOT be created. " ));
         bRes = false;
      }
   }

#endif

   return bRes;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::bAdjustWalJournalingAccessRights()
*
* DESCRIPTION:
*           - Internal function to validate the existence and access right
*             settings the WAL and SHM files, related to WAL journaling.
*           - This method is intended to be called at the end of the method
*             'InitializeDatabase', which is called in 'vOnLoadSettings', i.e.
*             before the service of FC_Phonebook becomes available.
*           - It shall ensure that the SDS TS service, which is running as user
*             'pj_speech' and is member of group 'aid_phbk_share' gets
*             gets "read/write" access to the WAL and SHM journaling files.
*           - The WAL and SHM files should already exist at this time as
*             phonebook itself has made several queries during the "database
*             consistency check" in method 'InitializeDatabase', but when
*             the files are created, SQLite unfortunately create them with
*             group 'aid_phbk' instead of 'aid_phbk_share'. This has to be
*             corrected as otherwise the speech user 'pj_speech' cannot access
*             them.
*
*
* PARAMETER: IN:  none
*
* RETURNVALUE: bool - Indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::bAdjustWalJournalingAccessRights()
{
   bool bResult = true;

   int iRes = 0;                      // Local/temporary result of internal methods.
   struct stat oStatBuf;

   struct passwd *poWantedPasswd = NULL;
   struct group  *poWantedGroup  = NULL;


   //------------------------
   // WAL
   //------------------------
   // Check if the WAL file itself exists
   if( 0 == stat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_FILE_NAME, &oStatBuf) )
   {
      ETG_TRACE_COMP (( " -PBDB.SECURITY-WAL-Access_Rights-: INFO: The WAL file exists. "  ));

      // Check, if current "permissions" of the WAL file matches to the wanted ones, if not try to adapt accordingly
      if( (oStatBuf.st_mode & 0x1FF) != (S_IRUSR | S_IWUSR | S_IRGRP  | S_IWGRP) )   // Added group write access (for WAL journaling)
      {
         // Print current (not corrected) access settings of the WAL file
         ETG_TRACE_USR1(( " -PBDB.SECURITY-WAL-Access_Rights-: INFO: 'stat' WAL file:  [incorrect] permissions= '%d' (octal) ",
                          ((oStatBuf.st_mode>>6)&0x07)*100 + ((oStatBuf.st_mode>>3)&0x07)*10 + (oStatBuf.st_mode&0x07) ));

         // Set "-rw-rw----" access for the WAL file
         ETG_TRACE_USR1(( " -PBDB.SECURITY-WAL-Access_Rights-: INFO: Correcting the WAL file 'permissions' to '-rw-rw----' " ));
         iRes = chmod(FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_FILE_NAME, S_IRUSR | S_IWUSR | S_IRGRP  | S_IWGRP);   // group write access is needed for WAL journaling
         if( iRes != 0 )
         {
            ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Access_Rights-: ERROR: 'chmod(file, -rw-rw----)' has FAILED with error= %d, [file= '%s'] ",
                               iRes, FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_FILE_NAME ));
            ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Access_Rights-: ERROR: 'chmod(file, -rw-rw----)' has FAILED with error= %d, [file= '%s'] ",
                               iRes, FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_FILE_NAME ));
            // Set result to false
            bResult = false;
         }

         // Check access settings of the WAL after the correction
         if( 0 == stat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_FILE_NAME, &oStatBuf) )
         {
            // Print current access settings of the WAL file
            ETG_TRACE_USR1(( " -PBDB.SECURITY-WAL-Access_Rights-: INFO: 'stat' WAL file:  permissions= '%d' (octal) ",
                             ((oStatBuf.st_mode>>6)&0x07)*100 + ((oStatBuf.st_mode>>3)&0x07)*10 + (oStatBuf.st_mode&0x07) ));
         }
      }

      // Check if current "group" of the file matches 'aid_phbk_share', if not try to adapt it accordingly.
      poWantedPasswd = getpwnam(FC_PB_USER_NAME);
      poWantedGroup  = getgrnam(FC_PB_GROUP_NAME);

      if( (poWantedPasswd) && (poWantedGroup))
      {
         // Check if current "group" for the file is correct. If this not o.k., do the needed adaptations.
         if( poWantedGroup->gr_gid != oStatBuf.st_gid )
         {
            // Adapt "group" (and "owner") settings as current "group" setting is NOT correct.
            ETG_TRACE_COMP(( " -PBDB.SECURITY-WAL-Access_Rights-: INFO: The WAL file 'group' setting has to be corrected. " ));
            iRes = chown(FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_FILE_NAME, poWantedPasswd->pw_uid, poWantedGroup->gr_gid);
            if(iRes != 0)
            {
               ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Access_Rights-: ERROR: 'chown(file, %d, %d)' has FAILED with error= %d, [file= '%s'] ",
                                  poWantedPasswd->pw_uid, poWantedGroup->gr_gid, iRes, FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_FILE_NAME ));
               ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Access_Rights-: ERROR: 'chown(file, %d, %d)' has FAILED with error=%d, [file= '%s'] ",
                                  poWantedPasswd->pw_uid, poWantedGroup->gr_gid, iRes, FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_FILE_NAME ));
               // Set result to false
               bResult = false;
            }
            else
            {
               ETG_TRACE_COMP(( " -PBDB.SECURITY-WAL-Access_Rights-: SUCCESS: The WAL file 'group' setting has been corrected. " ));
            }
         }
      }
      else
      {
         ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Access_Rights-: ERROR: Cannot resolve the 'wanted' group-name= '%s' ", FC_PB_GROUP_NAME ));
         ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Access_Rights-: ERROR: Cannot resolve the 'wanted' group-name= '%s' ", FC_PB_GROUP_NAME ));
         bResult = false;
      }
   }
   else
   {
      ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Access_Rights-: ERROR: The WAL file DOES NOT exist, [file= '%s'] ", FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_FILE_NAME ));
      ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Access_Rights-: ERROR: The WAL file DOES NOT exist, [file= '%s'] ", FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_FILE_NAME ));
      bResult = false;
   }


   //------------------------
   // SHM
   //------------------------
   // Check if the SHM file itself exists
   if( 0 == stat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHM_FILE_NAME, &oStatBuf) )
   {
      ETG_TRACE_COMP(( " -PBDB.SECURITY-WAL-Access_Rights-: INFO: The SHM file exists. " ));

      // Check if current "group" of the file matches 'aid_phbk_share', if not try to adapt it accordingly.
      poWantedPasswd = getpwnam(FC_PB_USER_NAME);
      poWantedGroup  = getgrnam(FC_PB_GROUP_NAME);

      if( (poWantedPasswd) && (poWantedGroup) )
      {
         // Check if current "group" for the file is correct. If this not o.k., do the needed adaptations.
         if( poWantedGroup->gr_gid != oStatBuf.st_gid )
         {
            // Adapt "group" (and "owner") settings as current "group" setting is NOT correct.
            ETG_TRACE_COMP(( " -PBDB.SECURITY-WAL-Access_Rights-: INFO: The SHM file 'group' setting has to be corrected. " ));
            iRes = chown(FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHM_FILE_NAME, poWantedPasswd->pw_uid, poWantedGroup->gr_gid);
            if(iRes != 0)
            {
               ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Access_Rights-: After Security Check: ERROR: 'chown(file, %d, %d)' has FAILED with error= %d, [file= '%s'] ",
                                  poWantedPasswd->pw_uid, poWantedGroup->gr_gid, iRes, FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHM_FILE_NAME ));
               ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Access_Rights-: After Security Check: ERROR: 'chown(file, %d, %d)' has FAILED with error= %d, [file= '%s'] ",
                                  poWantedPasswd->pw_uid, poWantedGroup->gr_gid, iRes, FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHM_FILE_NAME ));

               // Set result to false
               bResult = false;
            }
            else
            {
               ETG_TRACE_COMP(( " -PBDB.SECURITY-WAL-Access_Rights-: SUCCESS: The SHM file 'group' setting has been corrected. " ));
            }
         }
      }
      else
      {
         ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Access_Rights-: After Security Check: ERROR: Cannot resolve the 'wanted' group-name= '%s' ", FC_PB_GROUP_NAME ));
         ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Access_Rights-: After Security Check: ERROR: Cannot resolve the 'wanted' group-name= '%s' ", FC_PB_GROUP_NAME ));
         bResult = false;
      }
   }
   else
   {
      ETG_TRACE_FATAL (( " -PBDB.SECURITY-WAL-Access_Rights-: ERROR: The SHM file DOES NOT exist, [file= '%s'] ", FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHM_FILE_NAME ));
      ETG_TRACE_ERRMEM(( " -PBDB.SECURITY-WAL-Access_Rights-: ERROR: The SHM file DOES NOT exist, [file= '%s'] ", FC_PB_DB_FILE_PATH "/" FC_PB_DB_SHM_FILE_NAME ));
      bResult = false;
   }

   return bResult;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::CheckTableSchema()
*
* DESCRIPTION:
*             Internal function to validate the sql table schema
*
*
* PARAMETER: None.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::CheckTableSchema()
{
   ETG_TRACE_USR4(("CheckTableSchema entered"));
   bool blRet = false;
   QMutexLocker lock(&m_mutex);//GMMY17-17143
   QSqlDatabase db = Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   db.transaction(); //Start transaction
   QSqlQuery query(db);

   QTime TotalTime, IndividualTime;
   TotalTime.start();

   if(!query.exec( "SELECT name FROM sqlite_master WHERE type = \"table\"" ) )
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query failed in CheckTableSchema"));
      return false;
   }

   QStringList list_table;
   while (query.next())
      list_table << query.value( 0 ).toString();

   if(!query.exec( "SELECT name FROM sqlite_master WHERE type = \"view\"" ) )
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query failed in CheckTableSchema"));
      return false;
   }

   QStringList list_views;
   while (query.next())
      list_views << query.value( 0 ).toString();

   // -----------------------Handling Master Table--------------------------------------------

   IndividualTime.start();
   if( list_table.contains(FC_PB_MASTER_TABLE_NAME) )
   {
      blRet= CheckMasterTable();
      if(!blRet)
      {
         if ( !query.exec( "DROP TABLE " FC_PB_MASTER_TABLE_NAME ))
         {
            PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

            ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query failed while deleting Master table"));
            return false;
         }
      }
   }
   else
   {
      blRet= false;
   }

   if(!blRet)
   {
      blRet= CreateMasterTable();
	  //CMG3G-14313-Coverity Fix
      if (blRet)
      {
         ETG_TRACE_USR4(("MasterTable Created Successfully"));
      }
	  //end of Fix
   }

   ETG_TRACE_USR2(("Time elapsed in Master table verification - %d ms", IndividualTime.elapsed()));

   // -----------------------Handling Settings Table--------------------------------------------

   IndividualTime.start();
   if( list_table.contains(FC_PB_SETTINGS_TABLE_NAME) )
   {
      blRet= CheckSettingsTable();
      if(!blRet)
      {
         if ( !query.exec( "DROP TABLE " FC_PB_SETTINGS_TABLE_NAME ))
         {
            PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

            ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query failed while deleting Settings table"));
            return false;
         }
      }
   }
   else
   {
      blRet= false;
   }

   if(!blRet)
   {
      //Check if old table PreferredPhonebookSortOrder is available in existing DB. If yes retrieve the previous user setting.
      quint8 u8SortOrder = 0;
      if (list_table.contains(FC_PB_PREFERREDSORTORDER_TABLE_NAME))
      {
         blRet = CheckPreferredSortOrderTable();
         if (blRet)
         {
            //Table exists and is valid. Hence fetch the old user setting for Preferred sort order.
            u8SortOrder = GetSortOrderFromPreferredSortOrderTable();

            //Delete the table as it is no longer required.
            if ( !query.exec( "DROP TABLE " FC_PB_PREFERREDSORTORDER_TABLE_NAME ))
            {
               PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

               ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query failed while deleting PreferredSortOrderTable"));
            }
         }
      }

      blRet= CreateSettingsTable(u8SortOrder);
   }

   ETG_TRACE_USR2(("Time elapsed in Settings table verification - %d ms", IndividualTime.elapsed()));

   // -----------------------Handling Device Tables--------------------------------------------

   IndividualTime.start();
   if(blRet)
   {
      ETG_TRACE_USR4((" Master Table created "));

      for(quint8 uIndex= FC_PB_VEHICLE_PBOOK; uIndex <= FC_PB_DEVICE_HANDLE11; uIndex++)
      {
         if ( list_table.contains( fc_pb_aTableNames[uIndex] ) )
         {
            blRet= CheckDeviceTable(uIndex);
            if(!blRet)
            {
               if ( !DropTableFromDB(uIndex) )
               {
                  return false;
               }
            }
         }
         else
         {
            blRet= false;
         }

         if(!blRet)
         {
            blRet= CreateDeviceTable(uIndex);
         }

         if(!blRet)
         {
            ETG_TRACE_ERR(("FC_PhoneBook_SQLite:- ERROR - Error while creating device table having uIndex - %u", uIndex));
            break;
         }
      }
      ETG_TRACE_USR4((" Vehicle phone book Table created "));
      for(quint8 uIndex= FC_VPB_DEVICE_HANDLE2; uIndex <= FC_VPB_DEVICE_HANDLE11; uIndex++)
      {
         if ( list_table.contains( fc_vpb_aTableNames[uIndex] ) )
         {
            blRet= CheckDeviceVpbTable(uIndex);
            if(!blRet)
            {
               if ( !DropVPBTableFromDB(uIndex) )
               {
                  return false;
               }
            }
         }
         else
         {
            blRet= false;
         }
         if(!blRet)
         {
            blRet= CreateDeviceVpbTable(uIndex);
         }
         if(!blRet)
         {
            ETG_TRACE_ERR(("FC_PhoneBook_SQLite:- ERROR - Error while creating device table having uIndex - %u", uIndex));
            break;
         }
      }
#ifdef DATABASE_ENCRYPTION_VIA_SEE_AND_SDC
      // The following generation of INDEX for "ContactHandle" in the phonebook tables, we currently restrict
      // to the case of database encryption. Otherwise we would boost the performance also for the not encrypted
      // operation and we would have to argue why we will loose a significant part of this when the encrypted
      // mode gets acitvated. On a long term we intend to switch to the encrypted mode operation for all projects
      // anyway.

      // Create indices of the following type
      // "CREATE INDEX IF NOT EXISTS index_DHx_ContactHandle on PhoneBook_DHx (ContactHandle)"
      // for x= 2.. 11 (i.e. the possible phone devices)
      for(quint8 uIndex= FC_PB_DEVICE_HANDLE2; uIndex <= FC_PB_DEVICE_HANDLE11; uIndex++)
      {
         // Protoype for query: CREATE INDEX IF NOT EXISTS index_DHx_ContactHandle ON PhoneBook_DHx (ContactHandle)"
         const QString sql2 = "CREATE INDEX IF NOT EXISTS index_DH" + QString::number(uIndex) + "_" FC_PB_COL_CONTACT_HANDLE
               " ON " + fc_pb_aTableNames[uIndex] + "(" FC_PB_COL_CONTACT_HANDLE ")";

         if( !query.exec(sql2) )
         {
            PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

            ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query to create index has failed:"
                            " 'index_DH%d_ContactHandle' ", uIndex ));
            return false;
         }
      }
#endif

      // Efficient resolving of CallHistory entries (regarding the contact data in tables PhoneBook_DHx)
      // Create indices of the following type:
      // "CREATE INDEX IF NOT EXISTS index_DHx_NumberSubstrX ON PhoneBook_DHx (NumberSubstrX)"
      // for x= 2.. 11 (i.e. the possible phone devices) and for X= 1.. 8 (i.e. max numbers per contact)
      // Note: Some projects use less than 8 numbers. The related definition is done via PBDL_MAX_PHONE_NUMBERS_PER_CONTACT_COUNT,
      //       but we currently do not exploit this optimization potential here as also the related queries run over all 8 numbers.

      QTime DebugTime;
      DebugTime.start();

      QString sDeviceHandle = "";
      QString sNum = "";

      for(quint8 u8DeviceHandle= FC_PB_DEVICE_HANDLE2; u8DeviceHandle <= FC_PB_DEVICE_HANDLE11; u8DeviceHandle++)
      {
         // Create DB indices on NumberSubstrX
         QString sql;
         sDeviceHandle = QString::number(u8DeviceHandle);
         for(int iNum =1; iNum <= 8; iNum++)   // iNum is used for the X in String 'NumberSubstrX', X= 1,.. 8
         {
            sNum = QString::number(iNum);

            // Create database indixes on the columns 'NumberSubstrX'
            // Example for table 'PhoneBook_DH2' for on column 'NumberSubstr1' (which is related to PhoneNumber1):
            // CREATE INDEX IF NOT EXISTS index_DHx_NumberSubstr1 ON PhoneBook_DH2 (NumberSubstr1)
            sql = "CREATE INDEX IF NOT EXISTS index_DH" + sDeviceHandle + "_NumberSubstr" + sNum
                  + " ON PhoneBook_DH" + sDeviceHandle + " (NumberSubstr" + sNum + ")";

            ETG_TRACE_USR3((" Create DB-Index on 'NumberSubstrX': sql= '%s' ", sql.toLatin1().constData() ));

            if (query.exec(sql) == false)
            {
               PB_CHECK_QUERY_ERROR(query);
               ETG_TRACE_ERR((" Creation of DB-Index on 'NumberSubstrX' failed. DeviceHandle= %d, iNum= %d ", u8DeviceHandle, iNum));
            }
         }

         // Create DB indices on FirstName and LastName
         // Example:
         // CREATE INDEX IF NOT EXISTS index_DH2_FirstName ON PhoneBook_DH2 (FirstName)

         sql = "CREATE INDEX IF NOT EXISTS index_DH" + sDeviceHandle + "_LastName ON PhoneBook_DH" + sDeviceHandle + " (FirstName)";
         ETG_TRACE_USR3((" Create DB-Index on FirstName: sql= '%s' ", sql.toLatin1().constData() ));

         if (query.exec(sql) == false)
         {
            PB_CHECK_QUERY_ERROR(query);
            ETG_TRACE_ERR((" Creation of DB-Index on FirstName failed. DeviceHandle= %d ", u8DeviceHandle ));
         }

         sql = "CREATE INDEX IF NOT EXISTS index_DH" + sDeviceHandle + "_FirstName ON PhoneBook_DH" + sDeviceHandle + " (LastName)";
         ETG_TRACE_USR3((" Create DB-Index on FirstName: sql= '%s' ", sql.toLatin1().constData() ));

         if (query.exec(sql) == false)
         {
            PB_CHECK_QUERY_ERROR(query);
            ETG_TRACE_ERR((" Creation of DB-Index on LastName failed. DeviceHandle= %d ", u8DeviceHandle ));
         }

         // Create DB indices on photoSize
         // Example:
         // CREATE INDEX IF NOT EXISTS index_DHx ON photoSize
         sql = "CREATE INDEX IF NOT EXISTS index_DH" + sDeviceHandle + "_PhotoSize ON PhoneBook_DH" + sDeviceHandle + " (PhotoSize)" ;
         ETG_TRACE_USR3((" Create DB-Index on photoSize: sql= '%s' ", sql.toLatin1().constData() ));

         if (query.exec(sql) == false)
         {
        	 PB_CHECK_QUERY_ERROR(query);
        	 ETG_TRACE_ERR((" Creation of DB-Index on photoSize failed. DeviceHandle= %d ", u8DeviceHandle ));
         }

         // Create DB indices on APL1Size
         // Example:
         // CREATE INDEX IF NOT EXISTS index_DHx ON APL1Size
         sql = "CREATE INDEX IF NOT EXISTS index_DH" + sDeviceHandle + "_APL1Size ON PhoneBook_DH" + sDeviceHandle + " (APL1Size)";
         ETG_TRACE_USR3((" Create DB-Index on APL1Size: sql= '%s' ", sql.toLatin1().constData() ));

         if (query.exec(sql) == false)
         {
        	 PB_CHECK_QUERY_ERROR(query);
        	 ETG_TRACE_ERR((" Creation of DB-Index on APL1Size failed. DeviceHandle= %d ", u8DeviceHandle ));
         }

      }
      ETG_TRACE_USR4((" Database Indices created for NumberSubstrX, Names, PhotoSize,APL1Size in %d ms", DebugTime.elapsed()));
   }

   ETG_TRACE_USR2(("Time elapsed for device tables verification - %d ms", IndividualTime.elapsed()));



   // -----------------------Handling Callhistory Tables--------------------------------------------

   IndividualTime.start();
   if(blRet)
   {
      for(quint8 uIndex=1; uIndex <= FC_PB_TOTAL_NUMBER_OF_CH; uIndex++)
      {
         if ( list_table.contains( fc_pb_aTableNames[MAP_TO_CH_INDEX(uIndex)] ) )
         {
            blRet= CheckCallHistoryTable(uIndex);
            if(!blRet)
            {
               QString sql= "DROP TABLE " + fc_pb_aTableNames[MAP_TO_CH_INDEX(uIndex)];
               if ( !query.exec(sql) )
               {
                  PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

                  ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query failed while deleting table %s", fc_pb_aTableNames[MAP_TO_CH_INDEX(uIndex)].toLatin1().constData()));
                  return false;
               }
            }
         }
         else
         {
            blRet= false;
         }

         if(!blRet)
         {
            blRet= CreateCallHistoryTable(uIndex);
         }

         if(!blRet)
         {

            ETG_TRACE_USR2(("FC_PhoneBook_SQLite:- ERROR - Error while creating call history table having uIndex - %u", uIndex));
            break;
         }
      }
   }

   ETG_TRACE_USR2(("Time elapsed for call history tables verification - %d ms", IndividualTime.elapsed()));

   IndividualTime.start();

   if(blRet)
   {
      ETG_TRACE_USR4(("CallHistory Tables created"));
      if ( list_table.contains( FC_PB_TABLE_CHECKSUM ) )
      {
         blRet= SpeechVerifyCheckSumTable();
         if(!blRet)
         {
            QString sql= "DROP TABLE " FC_PB_TABLE_CHECKSUM;
            if ( !query.exec(sql) )
            {
               PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

               ETG_TRACE_COMP(("FC_PhoneBook_SQLite :- ERROR - SQL query failed while deleting table - PhoneBook_CheckSum"));
               return false;
            }
         }
      }
      else
      {
         blRet= false;
      }

      if(!blRet)
      {
         blRet= SpeechCreateCheckSumTable();
      }

      if(!blRet)
      {
         ETG_TRACE_USR4(("FC_PhoneBook_SQLite:- ERROR - Error while creating checksum table"));
      }
   }

   ETG_TRACE_USR2(("Time elapsed for checksum table verification - %d ms", IndividualTime.elapsed()));

   // -----------------------Handling Favorites Tables--------------------------------------------

   IndividualTime.start();

   if(blRet)
   {
      if ( list_table.contains( FC_PB_TABLE_NAME_FAV ) )
      {
         blRet= CheckFavoritesTable();
         if(!blRet)
         {
            QString sql= "DROP TABLE " FC_PB_TABLE_NAME_FAV;
            if ( !query.exec(sql) )
            {
               PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

               ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query failed while deleting the favourites table"));
               return false;
            }
         }
      }
      else
      {
         blRet= false;
      }
      if(!blRet)
      {
         blRet= CreateFavoritesTable();
      }
   }

   if(!blRet)
   {
      ETG_TRACE_USR4(("FC_PhoneBook_SQLite :- ERROR - SQL query failed while creating the favourites table"));
   }
   ETG_TRACE_USR2(("Time elapsed for favorites table verification - %d ms", IndividualTime.elapsed()));

   // -----------------------Handling Feature Support  Tables--------------------------------------------

   IndividualTime.start();
   if(blRet)
   {
      ETG_TRACE_USR4(("Checking Handling Feature Support  Tables"));
      if ( list_table.contains(  (const char*)FC_PB_TABLE_NAME_FS ) )
      {
         blRet= CheckFeatureSupportTable();
         if(!blRet)
         {
            QString sql= "DROP TABLE " FC_PB_TABLE_NAME_FS;
            if ( !query.exec(sql) )
            {
               PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

               ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query failed while deleting table - \
                               %s", (const char*)FC_PB_TABLE_NAME_FS));
               return false;
            }
         }
      }
      else
      {
         blRet= false;
      }

      if(!blRet)
      {
         blRet= CreateFeatureSupportTable();
      }

      if(!blRet)
      {
         ETG_TRACE_ERR(("FC_PhoneBook_SQLite:- ERROR - Error while creating Feature Support table "));
      }
   }

   ETG_TRACE_USR2(("Time elapsed for Feature table verification - %d ms", IndividualTime.elapsed()));

   // -----------------------Speech(Contact) views--------------------------------------------

   IndividualTime.start();
   if(blRet)
   {
      ETG_TRACE_USR4(("Checking Speech(Contact) view"));
      if ( list_views.contains(  (const char*)FC_PB_CONTACT_TABLE_NAME ) )
      {
         blRet= CheckSpeechContactView();
         if(!blRet)
         {
            QString sql= "DROP VIEW " FC_PB_CONTACT_TABLE_NAME;
            if ( !query.exec(sql) )
            {
               PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

               ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query failed while deleting table - \
                               %s", (const char*)FC_PB_CONTACT_TABLE_NAME));
               return false;
            }
         }
      }
      else
      {
         blRet= false;
      }

      if(!blRet)
      {
         // Speech Support
         blRet = SpeechCreateViews();
      }

      if(!blRet)
      {
         ETG_TRACE_ERR(("FC_PhoneBook_SQLite:- ERROR - Error while creating Speech tables "));
      }
   }

   ETG_TRACE_USR2(("Time elapsed for Speech view(Contact) verification - %d ms", IndividualTime.elapsed()));

// -----------------------Speech(Ph_Checksum) view--------------------------------------------
   IndividualTime.start();
   if(blRet)
   {
      ETG_TRACE_USR4(("Checking Speech(Ph_Checksum) view"));
      if ( list_views.contains(  (const char*)FC_PB_CHKSUM_TABLE_NAME ) )
      {
         blRet= CheckSpeechPhChksumView();
         if(!blRet)
         {
            QString sql= "DROP VIEW " FC_PB_CHKSUM_TABLE_NAME;
            if ( !query.exec(sql) )
            {
               PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

               ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query failed while deleting table - \
                               %s", (const char*)FC_PB_CHKSUM_TABLE_NAME));
               return false;
            }
         }
      }
      else
      {
         blRet= false;
      }

      if(!blRet)
      {
         // Speech Support
         blRet = SpeechCreateViews();
      }

      if(!blRet)
      {
         ETG_TRACE_ERR(("FC_PhoneBook_SQLite:- ERROR - Error while creating Speech tables "));
      }
   }

   ETG_TRACE_USR2(("Time elapsed for Speech View(Ph_Checksum) table verification - %d ms", IndividualTime.elapsed()));

   if (!blRet)
   {
      db.rollback();
      ETG_TRACE_USR4((" Database rolled back "));
   }
   else
   {
      if (!db.commit())
      {
         db.rollback();
         ETG_TRACE_USR4((" Database rolled back since commit failed "));
      }
      else
      {
         ETG_TRACE_USR4((" Database committed "));
      }
   }

   ETG_TRACE_USR2((" Time elapsed for complete database verification and commit - %d ms ", TotalTime.elapsed()));
   return blRet;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::CreateMasterTable()
*
* DESCRIPTION:
*              QtSql specific function to create Master table
*
*
* PARAMETER: None.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::CreateMasterTable()
{
   ETG_TRACE_USR4(("CreateMasterTable entered"));
   bool bRetValue = false;

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);

      const QString sql = "CREATE TABLE IF NOT EXISTS " FC_PB_MASTER_TABLE_NAME " "
         "( "
         FC_PB_MASTER_COL_CONTACT_HANDLE" INTEGER UNIQUE ,"
         FC_PB_MASTER_COL_TABLE_INDEX" INTEGER "
         ");";
      if(!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query failed in Creating Master Table"));
      }
      else
      {
         bRetValue = true;
      }
   }

   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::CreateSettingsTable()
*
* DESCRIPTION:
*              QtSql specific function to create settings table
*
*
* PARAMETER: None.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::CreateSettingsTable(quint8 u8SortOrder)
{
   ETG_TRACE_USR4(("CreateSettingsTable entered with u8SortOrder: %d", u8SortOrder));

   QSqlDatabase db = Database();
   if (!db.isOpen())
   {

      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);

   const QString sql =
         "CREATE TABLE IF NOT EXISTS " FC_PB_SETTINGS_TABLE_NAME " "
         "( "
         FC_PB_PREFERREDSORTORDER_COL_SORTORDER" INTEGER "

         ");";
   ETG_TRACE_USR4((" SQL query used to CreateSettingsTable: '%s' ", sql.toLatin1().constData()));

   if (!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(("FC_PhoneBook_SQLite :- ERROR - SQL query failed in Creating Settings Table"));
      return false;
   }
   query.clear();

   QString
         sql2 =
               "INSERT INTO " FC_PB_SETTINGS_TABLE_NAME " ( " FC_PB_PREFERREDSORTORDER_COL_SORTORDER " ) VALUES ("
                     + QString::number(u8SortOrder) +  ")";
   ETG_TRACE_USR4((" SQL query used in CreateSetingsTable: '%s' ", sql2.toLatin1().constData()));

   if (!query.exec(sql2))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(("FC_PhoneBook_SQLite :- ERROR - SQL query failed in Insert Records in CreateSetingsTable"));
      return false;
   }
   return true;
}

/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::UpdatePreferredSortOrder()
*
* DESCRIPTION:
*              QtSql specific function to update the sort order
*
*
* PARAMETER: None.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::UpdatePreferredSortOrder(quint8 u8SortOrder)
{
   ETG_TRACE_USR2(("UpdatePreferredSortOrder entered with SortOrder %d ", u8SortOrder));

   QMutexLocker lock(&m_mutex);
   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);

   //CLONE TICKET FIX - GMMY15-622, GMMY15-1528 - Phonebook sort will be Last Name First Name for Chinese HMI.
   QString sql= "UPDATE "  FC_PB_SETTINGS_TABLE_NAME  " SET "  FC_PB_PREFERREDSORTORDER_COL_SORTORDER  " = " + QString::number(u8SortOrder);

   ETG_TRACE_USR4(( " Query used to update PreferredSortOrder: '%s' ", sql.toLatin1().constData() ));

   if (!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Update record failed for Sort Order"));
      return false;
   }

#ifdef PB_CHECK_QUERY_EXEC__TESTMODE
   PB_CHECK_QUERY_ERROR(query); // This is only for TEST (Error Injection) of: -PHBK-QUERY-ERR- Instrumentation - for GMMY17-13044 "database disk image is malformed"
#endif

   ETG_TRACE_USR4(( " 'PreferredSortOrder' has been updated successfully in phonebook database" ));
   return true;

}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetPreferredSortOrder()
*
* DESCRIPTION:
*              QtSql specific function to GetPreferredSortOrder
*
*
* PARAMETER: None.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
quint8 FC_PhoneBook_SQLite::GetPreferredSortOrder()
{
   ETG_TRACE_USR4(("FC_PhoneBook_SQLite::GetPreferredSortOrder entered"));
   QMutexLocker lock(&m_mutex);//GMMY17-17143
   quint8 u8SortOrder= 0;
   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR((" Database is not open"));
   }
   else
   {
      QSqlQuery query(db);

      QString sql = " SELECT " FC_PB_PREFERREDSORTORDER_COL_SORTORDER " FROM "  FC_PB_SETTINGS_TABLE_NAME;

      ETG_TRACE_USR2((" SQL query used to GetPreferredSortOrder- %s ",sql.toLatin1().constData()));

      if(!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" GetPreferredSortOrder Failed"));
      }
      else
      {
         query.next();
         if(query.isValid() == true)
         {
            u8SortOrder = (query.value(0)).value<quint8>();
            ETG_TRACE_USR2((" PreferredSortOrder = %d" , u8SortOrder));
         }
         else
         {
            ETG_TRACE_ERR((" Sql query failed"));
         }
      }
   }

   return u8SortOrder;
}

/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::bGetAutomaticContactDLValue()
*
* DESCRIPTION:
*              Function to get AutomaticDL value for the corresponding devicehandle
*
*
* PARAMETER: None.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::bGetAutomaticContactDLValue(quint8 devicehandle)
{
   ETG_TRACE_USR4(("FC_PhoneBook_SQLite::GetAutomaticContactDownloadValue entered"));
   bool bValue = false;
   QMutexLocker lock(&m_mutex);//GMMY17-17143
   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);
      QString sql = " SELECT " FC_PB_FS_COL_PB_AUTOMATIC_CONTACT_DL " FROM "  FC_PB_TABLE_NAME_FS " WHERE " FC_PB_FS_COL_DEVICE_HANDLE " = " + QString::number(devicehandle) ;
      ETG_TRACE_USR2((" SQL query used to GetAutomaticContactDownloadValue- %s ",sql.toLatin1().constData()));
      if(!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime
         ETG_TRACE_ERR(("GetAutomaticContactDownloadValue Failed"));
      }
      else
      {
         query.next();
         if(query.isValid() == TRUE)
         {
            bValue = query.value(0).toBool();
            ETG_TRACE_USR2(("GetAutomaticContactDownloadValue = %d" , bValue));
         }
         else
         {
            ETG_TRACE_ERR(("No entry found for device in feature table, returning default value TRUE"));
            bValue = true;
         }
      }
   }
   return bValue;
}

/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::CreateDeviceTable(..)
*
* DESCRIPTION:
*              QtSql specific function to create a table
*
*
* PARAMETER: None.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::CreateDeviceTable(quint8 u8DeviceHandle)
{
   ETG_TRACE_USR2(( " CreateDeviceTable entered with u8DeviceHandle= %d ", u8DeviceHandle ));

   // CMG3G-7702 Removing Validiy check for DeviceHandle since CreateDeviceTable fn is called internally only for valid cases.

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(( " Database is not open " ));
      return false;
   }
   QSqlQuery query(db);

   const QString sql = "CREATE TABLE IF NOT EXISTS " + fc_pb_aTableNames[u8DeviceHandle] + " "
      "( "
      FC_PB_COL_CONTACT_HANDLE" INTEGER,"
      FC_PB_COL_FIRST_NAME" TEXT,"
      FC_PB_COL_LAST_NAME" TEXT,"
      FC_PB_COL_PHONETICS" TEXT," //292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
      FC_PB_COL_POI_NAME" TEXT,"
      FC_PB_COL_PHONENUM1 " TEXT,"
      FC_PB_COL_NUMTYPE1 " INTEGER,"
      FC_PB_COL_NUMPRETTY1 " TEXT,"
      FC_PB_COL_PHONENUM2 " TEXT,"
      FC_PB_COL_NUMTYPE2 " INTEGER,"
      FC_PB_COL_NUMPRETTY2 " TEXT,"
      FC_PB_COL_PHONENUM3 " TEXT,"
      FC_PB_COL_NUMTYPE3 " INTEGER,"
      FC_PB_COL_NUMPRETTY3 " TEXT,"
      FC_PB_COL_PHONENUM4 " TEXT,"
      FC_PB_COL_NUMTYPE4 " INTEGER,"
      FC_PB_COL_NUMPRETTY4 " TEXT,"
      FC_PB_COL_PHONENUM5 " TEXT,"
      FC_PB_COL_NUMTYPE5 " INTEGER,"
      FC_PB_COL_NUMPRETTY5 " TEXT,"
      FC_PB_COL_PHONENUM6 " TEXT,"
      FC_PB_COL_NUMTYPE6 " INTEGER,"
      FC_PB_COL_NUMPRETTY6 " TEXT,"
      FC_PB_COL_PHONENUM7 " TEXT,"
      FC_PB_COL_NUMTYPE7 " INTEGER,"
      FC_PB_COL_NUMPRETTY7 " TEXT,"
      FC_PB_COL_PHONENUM8 " TEXT,"
      FC_PB_COL_NUMTYPE8 " INTEGER,"
      FC_PB_COL_NUMPRETTY8 " TEXT,"
      FC_PB_COL_EMAIL_ADD1" TEXT,"
      FC_PB_COL_EMAIL_ADD2" TEXT,"
      FC_PB_COL_EMAIL_ADD3" TEXT,"
      FC_PB_COL_HOME_ADD" TEXT,"
      FC_PB_COL_WORK_ADD" TEXT,"
      FC_PB_COL_OTHER_ADD" TEXT,"
      FC_PB_COL_CATEGORY" TEXT,"
      FC_PB_COL_CONTACT_TYPE" INTEGER,"
      FC_PB_COL_NOTES" TEXT,"
      FC_PB_COL_HGCLATLONGVALIDITY" INTEGER,"
      FC_PB_COL_HGCLATITUDE" INTEGER,"
      FC_PB_COL_HGCLONGITUDE" INTEGER,"
      FC_PB_COL_WGCLATLONGVALIDITY" INTEGER,"
      FC_PB_COL_WGCLATITUDE" INTEGER,"
      FC_PB_COL_WGCLONGITUDE" INTEGER,"
      FC_PB_COL_OGCLATLONGVALIDITY" INTEGER,"
      FC_PB_COL_OGCLATITUDE" INTEGER,"
      FC_PB_COL_OGCLONGITUDE" INTEGER,"
      FC_PB_COL_PHOTO_DL_SIZE " INTEGER,"
      FC_PB_COL_PHOTO         " BLOB,"
      FC_PB_COL_PHOTO_TYPE    " TEXT,"        // -FastPhotoHandling-
      FC_PB_COL_PHOTO_DL_CRC  " INTEGER,"     // -FastPhotoHandling-
      FC_PB_COL_PHOTO_SIZE    " INTEGER,"     // -FastPhotoHandling-
      FC_PB_COL_PHOTO_CRC     " INTEGER,"     // -FastPhotoHandling-
      FC_PB_COL_APL1_SIZE     " INTEGER,"  // -FastContactSearch-
      FC_PB_COL_CRC_APL1      " INTEGER,"   // -FastContactSearch-
      FC_PB_COL_SORTFIRSTNAME " TEXT,"    // Note: At the moment I just add the new columns at the end of each row.
      FC_PB_COL_SORTLASTNAME  " TEXT,"    //       So I will not interfere with any 'SELECT * from ..' query.
      FC_PB_COL_NUMSUBSTR1    " TEXT,"    //       After the 'SELECT * from ..' issue has been corrected, we might
      FC_PB_COL_NUMSUBSTR2    " TEXT,"    //       move these columns to their "native" (logically correct) position.
      FC_PB_COL_NUMSUBSTR3    " TEXT,"    //       Or we keep it as it is now.
      FC_PB_COL_NUMSUBSTR4    " TEXT,"
      FC_PB_COL_NUMSUBSTR5    " TEXT,"
      FC_PB_COL_NUMSUBSTR6    " TEXT,"
      FC_PB_COL_NUMSUBSTR7    " TEXT,"
      FC_PB_COL_NUMSUBSTR8    " TEXT"
      ");";

   if ( !query.exec(sql) )
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(( " FC_PhoneBook_SQLite :- ERROR - SQL query failed in creating table - %s ",
                      fc_pb_aTableNames[u8DeviceHandle].toLatin1().constData() ));
      return false;
   }

   // -FastContactSearch- : -BGN: Create index 'index_DHx_CrcAPL1(CrcAPL1)' for
   //     Phone devices (DH = 2 ..), [but not for DH= 1 (Vehicle Contacts), as we have
   //     no need for a FastContactSearch for Vehicle Contacts, but have a feature for "bulk"
   //     import. An index would slow down this bulk import.]
   if(u8DeviceHandle != 1)
   {
      query.clear();

      // Protoype for query: CREATE INDEX IF NOT EXISTS index_DH2_CrcApl1 ON PhoneBook_DH2(CrcAPL1)
      const QString sql2 = "CREATE INDEX IF NOT EXISTS index_DH" + QString::number(u8DeviceHandle) + "_" "CrcAPL1"
            " ON " + fc_pb_aTableNames[u8DeviceHandle] + "(" FC_PB_COL_CRC_APL1 ")";

      if( !query.exec(sql2) )
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query to create index has failed:"
                         " 'index_DH%d' ", u8DeviceHandle ));
         return false;
      }
   }
   // -FastContactSearch- : -END: Create index 'index_DHx_CrcAPL1(CrcAPL1)'

   return true;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::CreateDeviceVpbTable(..)
*
* DESCRIPTION:
*              QtSql specific function to create a table
*
*
* PARAMETER: None.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::CreateDeviceVpbTable(quint8 u8DeviceHandle)
{
   ETG_TRACE_USR2(( " CreateDeviceVpbTable entered with u8DeviceHandle= %d ", u8DeviceHandle ));
   bool bRetValue = false;

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(( " Database is not open " ));
   }
   else
   {
      QSqlQuery query(db);

      const QString sql = "CREATE TABLE IF NOT EXISTS " + fc_vpb_aTableNames[u8DeviceHandle] + " "
           "( "
           FC_PB_COL_CONTACT_HANDLE" INTEGER,"
           FC_PB_COL_VPB_CONTACT_POSITION" INTEGER,"
           FC_PB_COL_FIRST_NAME" TEXT,"
           FC_PB_COL_LAST_NAME" TEXT,"
           FC_PB_COL_PHONETICS" TEXT," //292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
           FC_PB_COL_POI_NAME" TEXT,"
           FC_PB_COL_PHONENUM1 " TEXT,"
           FC_PB_COL_NUMTYPE1 " INTEGER,"
           FC_PB_COL_NUMPRETTY1 " TEXT,"
           FC_PB_COL_PHONENUM2 " TEXT,"
           FC_PB_COL_NUMTYPE2 " INTEGER,"
           FC_PB_COL_NUMPRETTY2 " TEXT,"
           FC_PB_COL_PHONENUM3 " TEXT,"
           FC_PB_COL_NUMTYPE3 " INTEGER,"
           FC_PB_COL_NUMPRETTY3 " TEXT,"
           FC_PB_COL_PHONENUM4 " TEXT,"
           FC_PB_COL_NUMTYPE4 " INTEGER,"
           FC_PB_COL_NUMPRETTY4 " TEXT,"
           FC_PB_COL_PHONENUM5 " TEXT,"
           FC_PB_COL_NUMTYPE5 " INTEGER,"
           FC_PB_COL_NUMPRETTY5 " TEXT,"
           FC_PB_COL_PHONENUM6 " TEXT,"
           FC_PB_COL_NUMTYPE6 " INTEGER,"
           FC_PB_COL_NUMPRETTY6 " TEXT,"
           FC_PB_COL_PHONENUM7 " TEXT,"
           FC_PB_COL_NUMTYPE7 " INTEGER,"
           FC_PB_COL_NUMPRETTY7 " TEXT,"
           FC_PB_COL_PHONENUM8 " TEXT,"
           FC_PB_COL_NUMTYPE8 " INTEGER,"
           FC_PB_COL_NUMPRETTY8 " TEXT,"
           FC_PB_COL_EMAIL_ADD1" TEXT,"
           FC_PB_COL_EMAIL_ADD2" TEXT,"
           FC_PB_COL_EMAIL_ADD3" TEXT,"
           FC_PB_COL_HOME_ADD" TEXT,"
           FC_PB_COL_WORK_ADD" TEXT,"
           FC_PB_COL_OTHER_ADD" TEXT,"
           FC_PB_COL_CATEGORY" TEXT,"
           FC_PB_COL_CONTACT_TYPE" INTEGER,"
           FC_PB_COL_NOTES" TEXT,"
           FC_PB_COL_HGCLATLONGVALIDITY" INTEGER,"
           FC_PB_COL_HGCLATITUDE" INTEGER,"
           FC_PB_COL_HGCLONGITUDE" INTEGER,"
           FC_PB_COL_WGCLATLONGVALIDITY" INTEGER,"
           FC_PB_COL_WGCLATITUDE" INTEGER,"
           FC_PB_COL_WGCLONGITUDE" INTEGER,"
           FC_PB_COL_OGCLATLONGVALIDITY" INTEGER,"
           FC_PB_COL_OGCLATITUDE" INTEGER,"
           FC_PB_COL_OGCLONGITUDE" INTEGER,"
           FC_PB_COL_PHOTO_DL_SIZE " INTEGER,"
           FC_PB_COL_PHOTO         " BLOB,"
           FC_PB_COL_PHOTO_TYPE    " TEXT,"        // -FastPhotoHandling-
           FC_PB_COL_PHOTO_DL_CRC  " INTEGER,"     // -FastPhotoHandling-
           FC_PB_COL_PHOTO_SIZE    " INTEGER,"     // -FastPhotoHandling-
           FC_PB_COL_PHOTO_CRC     " INTEGER,"     // -FastPhotoHandling-
           FC_PB_COL_APL1_SIZE     " INTEGER,"  // -FastContactSearch-
           FC_PB_COL_CRC_APL1      " INTEGER,"  // -FastContactSearch-
           FC_PB_COL_SORTFIRSTNAME " TEXT,"    // Note: At the moment I just add the new columns at the end of each row.
           FC_PB_COL_SORTLASTNAME  " TEXT,"    //       So I will not interfere with any 'SELECT * from ..' query.
           FC_PB_COL_NUMSUBSTR1    " TEXT,"    //       After the 'SELECT * from ..' issue has been corrected, we might
           FC_PB_COL_NUMSUBSTR2    " TEXT,"    //       move these columns to their "native" (logically correot) position.
           FC_PB_COL_NUMSUBSTR3    " TEXT,"    //       Or we keep it as it is now.
           FC_PB_COL_NUMSUBSTR4    " TEXT,"
           FC_PB_COL_NUMSUBSTR5    " TEXT,"
           FC_PB_COL_NUMSUBSTR6    " TEXT,"
           FC_PB_COL_NUMSUBSTR7    " TEXT,"
           FC_PB_COL_NUMSUBSTR8    " TEXT"
           ");";

      if ( !query.exec(sql) )
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(( " FC_PhoneBook_SQLite :- ERROR - SQL query failed in creating table - %s ",
               fc_vpb_aTableNames[u8DeviceHandle].toLatin1().constData() ));
      }
      else
      {
         bRetValue = true;
      }
   }
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::CreateCallhistoryTable()
*
* DESCRIPTION:
*              QtSql specific function to create a call history table
*
*
* PARAMETER: None.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::CreateCallHistoryTable(quint8 uIndex)
{
   ETG_TRACE_USR2(("CreateCallHistoryTable entered with uIndex %d ", uIndex));
   bool bRetValue = false;
   if(!CHECK_CH_VALID(uIndex))
   {
      ETG_TRACE_USR4(("FC_PhoneBook_SQLite: Call history table id is invalid"));
   }
   else
   {
      QSqlDatabase db= Database();
      if(!db.isOpen())
      {
         ETG_TRACE_ERR(("Database is not open"));
      }
      else
      {
         QSqlQuery query(db);

         const QString sql = "CREATE TABLE IF NOT EXISTS " + fc_pb_aTableNames[MAP_TO_CH_INDEX(uIndex)] + " "
            "( "
             FC_PB_COL_CH_CONTACT_HANDLE" INTEGER,"
             FC_PB_COL_CH_CALL_TYPE" INTEGER,"
             FC_PB_COL_CH_FIRST_NAME" TEXT,"
             FC_PB_COL_CH_LAST_NAME" TEXT,"
             FC_PB_COL_CH_PHONE_NUM" TEXT,"
             FC_PB_COL_CH_PP_PHONE_NUM" TEXT,"
             FC_PB_COL_CH_NUMBER_TYPE" TEXT,"
             FC_PB_COL_CH_CALL_DATE" TEXT,"
             FC_PB_COL_CH_CALL_TIME" TEXT"
            ");";

         if ( !query.exec(sql) )
         {
            PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

            ETG_TRACE_ERR(("FC_PhoneBook_SQLite :- ERROR - SQL query failed in creating call history table - %s \
                           ", fc_pb_aTableNames[MAP_TO_CH_INDEX(uIndex)].toLatin1().constData()));
         }
         else
            bRetValue = true;
      }
   }
   return bRetValue;
}


bool FC_PhoneBook_SQLite::CreateFeatureSupportTable()
{
   ETG_TRACE_USR4(("CreateFeatureSupportTable entered "));
   bool bRetValue = false;

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);

      const QString sql = "CREATE TABLE IF NOT EXISTS " FC_PB_TABLE_NAME_FS " "
      "( "
      FC_PB_FS_COL_DEVICE_ADDRESS" TEXT UNIQUE,"
      FC_PB_FS_COL_DEVICE_HANDLE"  INTEGER UNIQUE,"
      FC_PB_FS_COL_PB_SUPPORT" INTEGER,"
      FC_PB_FS_COL_PB_CONTACTS_SORT" INTEGER,"
      FC_PB_FS_COL_PB_MISSED_CALL" INTEGER,"
      FC_PB_FS_COL_PB_INCOMING_CALL" INTEGER,"
      FC_PB_FS_COL_PB_OUTGOING_CALL" INTEGER,"
      FC_PB_FS_COL_PB_COMBINED_CALL" INTEGER,"
      FC_PB_FS_COL_PB_LAST_CONNECTION_TIME_ELAPSED" INTEGER,"
      FC_PB_FS_COL_PB_DL_COMPLETED_ONCE" BOOLEAN,"
      FC_PB_FS_COL_PB_DOWNLOAD_ONOFF" INTEGER,"
      FC_PB_FS_COL_PB_AUTOMATIC_CONTACT_DL" BOOLEAN,"
      FC_PB_PREFERREDSORTORDER_COL_SORTORDER" INTEGER"
      ");";

      ETG_TRACE_USR2(("Create FeatureTable Query =%s",sql.toLatin1().constData()));

      if ( !query.exec(sql) )
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("FC_PhoneBook_SQLite :- ERROR - SQL query failed in creating feature support- %s \
                        ", (const char*)FC_PB_TABLE_NAME_FS ));
      }
      else
      {
         ETG_TRACE_USR4(("CreateFeatureSupportTable Tables Created"));
         bRetValue = true;
      }
      query.clear();
   }
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::CheckMasterTable()
*
* DESCRIPTION:
*              QtSql specific function to validate Master table
*
*
* PARAMETER: None.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::CheckMasterTable()
{
   ETG_TRACE_USR4(("CheckMasterTable entered "));
   bool bRetValue = false;

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);

      QStringList list_columns;

      if(!query.exec("PRAGMA table_info(" FC_PB_MASTER_TABLE_NAME ")"))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("Error:- SQL query failed while validating Master Table"));
      }
      else
      {
         while(query.next())
         {
            list_columns << query.value(1).toString();
         }

         if(   list_columns.contains(FC_PB_MASTER_COL_CONTACT_HANDLE)
            && list_columns.contains(FC_PB_MASTER_COL_TABLE_INDEX) )
         {
            ETG_TRACE_USR4((" CheckMasterTable: MasterTable contains the correct columns "));
            bRetValue = true;
         }
         else
         {
            ETG_TRACE_USR4((" CheckMasterTable: MasterTable doesn't contain the correct columns "));
         }
      }
   }

   return bRetValue;
}



/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::CheckSettingsTable()
*
* DESCRIPTION:
*              QtSql specific function to validate Master table
*
*
* PARAMETER: None.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::CheckSettingsTable()
{
   ETG_TRACE_USR4(("CheckSettingsTable entered "));
   bool bRetValue = false;

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);


      QStringList list_columns;

      if(!query.exec("PRAGMA table_info(" FC_PB_SETTINGS_TABLE_NAME ")"))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("Error:- SQL query failed while validating Settings Table"));
      }
      else
      {
         while(query.next())
            list_columns << query.value(1).toString();
         if( list_columns.contains(FC_PB_PREFERREDSORTORDER_COL_SORTORDER))
         {
            ETG_TRACE_USR4((" CheckSettingsTable: SettingsTable contains the correct columns "));
            bRetValue = true;
         }
         else
         {
            ETG_TRACE_USR4((" CheckSettingsTable: SettingsTable doesn't contain the correct columns "));
         }
      }
   }

   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::CheckDeviceTable(quint8 uIndex)
*
* DESCRIPTION:
*              QtSql specific function to validate device table
*
*
* PARAMETER: None.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::CheckDeviceTable(quint8 uIndex)
{
   ETG_TRACE_USR2(("CheckDeviceTable entered  with uIndex %d ", uIndex));
   bool bRetValue = false;

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);

      QStringList list_columns;

      QString sql= "PRAGMA table_info(" + fc_pb_aTableNames[uIndex] + ")";

      if ( !query.exec(sql) )
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_USR2(("FC_PhoneBook_SQLite :- ERROR - SQL query failed while validating table - \
                         %s", fc_pb_aTableNames[uIndex].data()));
      }
      else
      {
         while (query.next())
         {
            list_columns << query.value( 1 ).toString();
         }

         if ( list_columns.contains( FC_PB_COL_CONTACT_HANDLE) &&
            list_columns.contains( FC_PB_COL_FIRST_NAME ) &&
            list_columns.contains( FC_PB_COL_LAST_NAME ) &&
            list_columns.contains( FC_PB_COL_PHONETICS ) && //292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
            list_columns.contains( FC_PB_COL_SORTFIRSTNAME ) &&   // NEW
            list_columns.contains( FC_PB_COL_SORTLASTNAME ) &&    // NEW
            list_columns.contains( FC_PB_COL_POI_NAME ) &&
            list_columns.contains( FC_PB_COL_PHONENUM1 ) &&
            list_columns.contains( FC_PB_COL_NUMTYPE1 ) &&
            list_columns.contains( FC_PB_COL_NUMPRETTY1 ) &&
            list_columns.contains( FC_PB_COL_NUMSUBSTR1 ) &&   // NEW
            list_columns.contains( FC_PB_COL_PHONENUM2 ) &&
            list_columns.contains( FC_PB_COL_NUMTYPE2 ) &&
            list_columns.contains( FC_PB_COL_NUMPRETTY2 ) &&
            list_columns.contains( FC_PB_COL_NUMSUBSTR2 ) &&   // NEW
            list_columns.contains( FC_PB_COL_PHONENUM3 ) &&
            list_columns.contains( FC_PB_COL_NUMTYPE3 ) &&
            list_columns.contains( FC_PB_COL_NUMPRETTY3 ) &&
            list_columns.contains( FC_PB_COL_NUMSUBSTR3 ) &&   // NEW
            list_columns.contains( FC_PB_COL_PHONENUM4 ) &&
            list_columns.contains( FC_PB_COL_NUMTYPE4 ) &&
            list_columns.contains( FC_PB_COL_NUMPRETTY4 ) &&
            list_columns.contains( FC_PB_COL_NUMSUBSTR4 ) &&   // NEW
            list_columns.contains( FC_PB_COL_PHONENUM5 ) &&
            list_columns.contains( FC_PB_COL_NUMTYPE5 ) &&
            list_columns.contains( FC_PB_COL_NUMPRETTY5 ) &&
            list_columns.contains( FC_PB_COL_NUMSUBSTR5 ) &&   // NEW
            list_columns.contains( FC_PB_COL_PHONENUM6 ) &&
            list_columns.contains( FC_PB_COL_NUMTYPE6 ) &&
            list_columns.contains( FC_PB_COL_NUMPRETTY6 ) &&
            list_columns.contains( FC_PB_COL_NUMSUBSTR6 ) &&   // NEW
            list_columns.contains( FC_PB_COL_PHONENUM7 ) &&
            list_columns.contains( FC_PB_COL_NUMTYPE7 ) &&
            list_columns.contains( FC_PB_COL_NUMPRETTY7 ) &&
            list_columns.contains( FC_PB_COL_NUMSUBSTR7 ) &&   // NEW
            list_columns.contains( FC_PB_COL_PHONENUM8 ) &&
            list_columns.contains( FC_PB_COL_NUMTYPE8 ) &&
            list_columns.contains( FC_PB_COL_NUMPRETTY8 ) &&
            list_columns.contains( FC_PB_COL_NUMSUBSTR8 ) &&   // NEW
            list_columns.contains( FC_PB_COL_EMAIL_ADD1 ) &&
            list_columns.contains( FC_PB_COL_EMAIL_ADD2 ) &&
            list_columns.contains( FC_PB_COL_EMAIL_ADD3 ) &&
            list_columns.contains( FC_PB_COL_HOME_ADD ) &&
            list_columns.contains( FC_PB_COL_WORK_ADD ) &&
            list_columns.contains( FC_PB_COL_OTHER_ADD ) &&
            list_columns.contains( FC_PB_COL_CATEGORY ) &&
            list_columns.contains( FC_PB_COL_CONTACT_TYPE ) &&
            list_columns.contains( FC_PB_COL_NOTES ) &&
            list_columns.contains( FC_PB_COL_HGCLATLONGVALIDITY) &&
            list_columns.contains( FC_PB_COL_HGCLATITUDE) &&
            list_columns.contains( FC_PB_COL_HGCLONGITUDE) &&
            list_columns.contains( FC_PB_COL_WGCLATLONGVALIDITY) &&
            list_columns.contains( FC_PB_COL_WGCLATITUDE) &&
            list_columns.contains( FC_PB_COL_WGCLONGITUDE) &&
            list_columns.contains( FC_PB_COL_OGCLATLONGVALIDITY) &&
            list_columns.contains( FC_PB_COL_OGCLATITUDE) &&
            list_columns.contains( FC_PB_COL_OGCLONGITUDE)&&
            list_columns.contains( FC_PB_COL_PHOTO_DL_SIZE ) &&
            list_columns.contains( FC_PB_COL_PHOTO ) &&
            list_columns.contains( FC_PB_COL_PHOTO_TYPE ) &&     // -FastPhotoHandling-
            list_columns.contains( FC_PB_COL_PHOTO_DL_CRC ) &&   // -FastPhotoHandling-
            list_columns.contains( FC_PB_COL_PHOTO_SIZE ) &&     // -FastPhotoHandling-
            list_columns.contains( FC_PB_COL_PHOTO_CRC ) &&      // -FastPhotoHandling-
            list_columns.contains( FC_PB_COL_APL1_SIZE ) &&   // -FastContactSearch-
            list_columns.contains( FC_PB_COL_CRC_APL1 ) )     // -FastContactSearch-
         {
            bRetValue= true;
         }
      }
   }
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::CheckDeviceVpbTable(quint8 uIndex)
*
* DESCRIPTION:
*              QtSql specific function to validate device table
*
*
* PARAMETER: None.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::CheckDeviceVpbTable(quint8 uIndex)
{
   ETG_TRACE_USR2(("CheckDeviceVpbTable entered  with uIndex %d ", uIndex));
   bool bRetValue= false;

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);


      QStringList list_columns;

      QString sql= "PRAGMA table_info(" + fc_vpb_aTableNames[uIndex] + ")";
      if (!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("FC_PhoneBook_SQLite :- ERROR - SQL query failed while validating Vpbtable - \
                         %s", fc_vpb_aTableNames[uIndex].data()));
      }
      else
      {
         while (query.next())
            list_columns << query.value( 1 ).toString();
         if ( list_columns.contains( FC_PB_COL_CONTACT_HANDLE) &&
            list_columns.contains( FC_PB_COL_VPB_CONTACT_POSITION ) &&
            list_columns.contains( FC_PB_COL_FIRST_NAME ) &&
            list_columns.contains( FC_PB_COL_LAST_NAME ) &&
            list_columns.contains( FC_PB_COL_PHONETICS ) && //292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
            list_columns.contains( FC_PB_COL_SORTFIRSTNAME ) &&   // NEW
            list_columns.contains( FC_PB_COL_SORTLASTNAME ) &&    // NEW
            list_columns.contains( FC_PB_COL_POI_NAME ) &&
            list_columns.contains( FC_PB_COL_PHONENUM1 ) &&
            list_columns.contains( FC_PB_COL_NUMPRETTY1 ) &&
            list_columns.contains( FC_PB_COL_NUMSUBSTR1 ) &&    // NEW
            list_columns.contains( FC_PB_COL_NUMTYPE1 ) &&
            list_columns.contains( FC_PB_COL_PHONENUM2 ) &&
            list_columns.contains( FC_PB_COL_NUMPRETTY2 ) &&
            list_columns.contains( FC_PB_COL_NUMSUBSTR2 ) &&    // NEW
            list_columns.contains( FC_PB_COL_NUMTYPE2 ) &&
            list_columns.contains( FC_PB_COL_PHONENUM3 ) &&
            list_columns.contains( FC_PB_COL_NUMPRETTY3 ) &&
            list_columns.contains( FC_PB_COL_NUMSUBSTR3 ) &&    // NEW
            list_columns.contains( FC_PB_COL_NUMTYPE3 ) &&
            list_columns.contains( FC_PB_COL_PHONENUM4 ) &&
            list_columns.contains( FC_PB_COL_NUMPRETTY4 ) &&
            list_columns.contains( FC_PB_COL_NUMSUBSTR4 ) &&    // NEW
            list_columns.contains( FC_PB_COL_NUMTYPE4 ) &&
            list_columns.contains( FC_PB_COL_PHONENUM5 ) &&
            list_columns.contains( FC_PB_COL_NUMPRETTY5 ) &&
            list_columns.contains( FC_PB_COL_NUMSUBSTR5 ) &&    // NEW
            list_columns.contains( FC_PB_COL_NUMTYPE5 ) &&
            list_columns.contains( FC_PB_COL_PHONENUM6 ) &&
            list_columns.contains( FC_PB_COL_NUMPRETTY6 ) &&
            list_columns.contains( FC_PB_COL_NUMSUBSTR6 ) &&    // NEW
            list_columns.contains( FC_PB_COL_NUMTYPE6 ) &&
            list_columns.contains( FC_PB_COL_PHONENUM7 ) &&
            list_columns.contains( FC_PB_COL_NUMPRETTY7 ) &&
            list_columns.contains( FC_PB_COL_NUMSUBSTR7 ) &&    // NEW
            list_columns.contains( FC_PB_COL_NUMTYPE7 ) &&
            list_columns.contains( FC_PB_COL_PHONENUM8 ) &&
            list_columns.contains( FC_PB_COL_NUMPRETTY8 ) &&
            list_columns.contains( FC_PB_COL_NUMSUBSTR8 ) &&    // NEW
            list_columns.contains( FC_PB_COL_NUMTYPE8 ) &&
            list_columns.contains( FC_PB_COL_EMAIL_ADD1 ) &&
            list_columns.contains( FC_PB_COL_EMAIL_ADD2 ) &&
            list_columns.contains( FC_PB_COL_EMAIL_ADD3 ) &&
            list_columns.contains( FC_PB_COL_HOME_ADD ) &&
            list_columns.contains( FC_PB_COL_WORK_ADD ) &&
            list_columns.contains( FC_PB_COL_OTHER_ADD ) &&
            list_columns.contains( FC_PB_COL_CATEGORY ) &&
            list_columns.contains( FC_PB_COL_CONTACT_TYPE ) &&
            list_columns.contains( FC_PB_COL_NOTES ) &&
            list_columns.contains( FC_PB_COL_HGCLATLONGVALIDITY) &&
            list_columns.contains( FC_PB_COL_HGCLATITUDE) &&
            list_columns.contains( FC_PB_COL_HGCLONGITUDE) &&
            list_columns.contains( FC_PB_COL_WGCLATLONGVALIDITY) &&
            list_columns.contains( FC_PB_COL_WGCLATITUDE) &&
            list_columns.contains( FC_PB_COL_WGCLONGITUDE) &&
            list_columns.contains( FC_PB_COL_OGCLATLONGVALIDITY) &&
            list_columns.contains( FC_PB_COL_OGCLATITUDE) &&
            list_columns.contains( FC_PB_COL_OGCLONGITUDE)&&
            list_columns.contains( FC_PB_COL_PHOTO_DL_SIZE ) &&
            list_columns.contains( FC_PB_COL_PHOTO ) &&
            list_columns.contains( FC_PB_COL_PHOTO_TYPE ) &&     // -FastPhotoHandling-
            list_columns.contains( FC_PB_COL_PHOTO_DL_CRC ) &&   // -FastPhotoHandling-
            list_columns.contains( FC_PB_COL_PHOTO_SIZE ) &&     // -FastPhotoHandling-
            list_columns.contains( FC_PB_COL_PHOTO_CRC ) &&      // -FastPhotoHandling-
            list_columns.contains( FC_PB_COL_APL1_SIZE ) &&   // -FastContactSearch-
            list_columns.contains( FC_PB_COL_CRC_APL1 ) )     // -FastContactSearch-
         {
            bRetValue= true;
         }
      }
   }
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::CheckCallHistoryTable(quint8 uIndex)
*
* DESCRIPTION:
*              QtSql specific function to validate Master table
*
*
* PARAMETER: None.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::CheckCallHistoryTable(quint8 uIndex)
{
   ETG_TRACE_USR2(("CheckCallHistoryTable entered with uIndex %d ", uIndex));

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {

      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);

   bool blRet= false;
   QStringList list_columns;

   if(!CHECK_CH_VALID(uIndex))
   {
      ETG_TRACE_USR4(("FC_PhoneBook_SQLite: Call history table id is invalid"));
      return false;
   }

   QString sql= "PRAGMA table_info(" + fc_pb_aTableNames[MAP_TO_CH_INDEX(uIndex)] + ")";

   if ( !query.exec(sql) )
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query failed while validating table - \
                       %s", fc_pb_aTableNames[MAP_TO_CH_INDEX(uIndex)].data()));
      return false;
   }

   while (query.next())
   {
      list_columns << query.value( 1 ).toString();
   }

   if ( list_columns.contains( FC_PB_COL_CH_CONTACT_HANDLE) &&
      list_columns.contains( FC_PB_COL_CH_CALL_TYPE) &&
      list_columns.contains( FC_PB_COL_CH_FIRST_NAME ) &&
      list_columns.contains( FC_PB_COL_CH_LAST_NAME ) &&
      list_columns.contains( FC_PB_COL_CH_PHONE_NUM ) &&
      list_columns.contains( FC_PB_COL_CH_PP_PHONE_NUM )&&
      list_columns.contains( FC_PB_COL_CH_NUMBER_TYPE ) &&
      list_columns.contains( FC_PB_COL_CH_CALL_DATE ) &&
      list_columns.contains( FC_PB_COL_CH_CALL_TIME ))
   {
      blRet= true;
   }

   return blRet;
}

bool FC_PhoneBook_SQLite::CheckFeatureSupportTable()
{
   ETG_TRACE_USR4(("CheckFeatureSupportTable entered "));
   bool bRetValue= false;

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);


      QStringList list_columns;

      QString sql= "PRAGMA table_info(" FC_PB_TABLE_NAME_FS ")";

      ETG_TRACE_USR2(("Check FeatureTable Querry =%s",sql.toLatin1().constData()));
      if ( !query.exec(sql) )
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query failed while validating table - \
                          %s", (const char*)FC_PB_TABLE_NAME_FS));
      }
      else
      {
         while (query.next())
         {
            list_columns << query.value( 1 ).toString();
         }

         if (list_columns.contains(FC_PB_FS_COL_DEVICE_ADDRESS)
               && list_columns.contains(FC_PB_FS_COL_DEVICE_HANDLE)
               && list_columns.contains(FC_PB_FS_COL_PB_SUPPORT)
               && list_columns.contains(FC_PB_FS_COL_PB_CONTACTS_SORT)
               && list_columns.contains(FC_PB_FS_COL_PB_MISSED_CALL)
               && list_columns.contains(FC_PB_FS_COL_PB_INCOMING_CALL)
               && list_columns.contains(FC_PB_FS_COL_PB_OUTGOING_CALL)
               && list_columns.contains(FC_PB_FS_COL_PB_COMBINED_CALL)
               && list_columns.contains(FC_PB_FS_COL_PB_LAST_CONNECTION_TIME_ELAPSED)
               && list_columns.contains(FC_PB_FS_COL_PB_DL_COMPLETED_ONCE)
               && list_columns.contains(FC_PB_FS_COL_PB_DOWNLOAD_ONOFF)
               && list_columns.contains(FC_PB_FS_COL_PB_AUTOMATIC_CONTACT_DL)
               && list_columns.contains(FC_PB_PREFERREDSORTORDER_COL_SORTORDER))
         {
            bRetValue= true;
            ETG_TRACE_USR4(("Feature Support Table is CORRECT"));
         }
         else
         {
            ETG_TRACE_USR4(("Feature Support Table is -----INCORRECT"));
         }
      }
   }
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::CheckSpeechContactView()
*
* DESCRIPTION:
*              QtSql specific function to validate Speech Contact view
*
*
* PARAMETER: None.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::CheckSpeechContactView()
{
   ETG_TRACE_USR2(("CheckSpeechContactView entered "));
   bool bRetValue = false;

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);

      QStringList list_columns;

      QString sql= "PRAGMA table_info(" FC_PB_CONTACT_TABLE_NAME ")";

      if ( !query.exec(sql) )
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query failed while validating table - \
                          %s", FC_PB_CONTACT_TABLE_NAME));
      }
      else
      {
         while (query.next())
         {
            list_columns << query.value( 1 ).toString();
         }

         if ( list_columns.contains( "DeviceID") &&
            list_columns.contains( "Firstname") &&
            list_columns.contains( "Lastname" ) &&
            list_columns.contains( "Phonetics" ) && //292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
            list_columns.contains( "ContactID" ) &&
            list_columns.contains( "Category" ))
         {
            bRetValue = true;
         }
         else
         {
            ETG_TRACE_ERR((" Validation failed for Speech Contact view!!!"));
         }
      }
   }

   return bRetValue;
}

/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::CheckSpeechPhChksumView()
*
* DESCRIPTION:
*              QtSql specific function to validate Speech Checksum view
*
*
* PARAMETER: None.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::CheckSpeechPhChksumView()
{
   ETG_TRACE_USR2(("CheckSpeechPhChksumView entered "));
   bool bRetValue= false;

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);

      QStringList list_columns;

      QString sql= "PRAGMA table_info(" FC_PB_CHKSUM_TABLE_NAME ")";

      if ( !query.exec(sql) )
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query failed while validating table - \
                          %s", FC_PB_CHKSUM_TABLE_NAME));
      }
      else
      {
         while (query.next())
         {
            list_columns << query.value( 1 ).toString();
         }

         if ( list_columns.contains( "DeviceID") &&
            list_columns.contains( "CheckSum"))
         {
            bRetValue = true;
         }
         else
         {
            ETG_TRACE_ERR((" Validation failed for Speech Ph_Checksum view!!!"));
         }
      }
   }

   return bRetValue;
}


#ifdef FC_PB_DB_USE_DUMMY_RECORDS
/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::insertDummyRecords
*
* DESCRIPTION:
*              Dummy api for inserting records
*
*
* PARAMETER:  None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
void FC_PhoneBook_SQLite::insertDummyRecords()
{
   ETG_TRACE_USR4(("insertDummyRecords entered"));
   QList<FC_PhoneBook_Contact_Detail> list;
   FC_PhoneBook_Contact_Detail contact;

   contact.firstName = "FirstName_1";
   contact.lastName = "LastName_1";
   contact.PhoneNum2 = "1-0987654321";
   contact.PhoneNum3 = "1-0123456789";
   contact.emailAdd1 = "FirstName_1@LastName_1.com";
   contact.homeAdd = "home address";

   list << contact;

   contact.firstName = "a_FirstName_2";
   contact.lastName = "a_LastName_2";
   contact.PhoneNum2 = "2-0987654321";
   contact.PhoneNum3 = "2-0123456789";
   contact.emailAdd1 = "a_FirstName_2@LastName_2.com";
   contact.homeAdd = "a_home address";

   list << contact;

   contact.firstName = "FirstName_3";
   contact.lastName = "LastName_3";
   contact.PhoneNum2 = "3-0987654321";
   contact.PhoneNum3 = "3-0123456789";
   contact.emailAdd1 = "FirstName_3@LastName_3.com";
   contact.homeAdd = "home address";

   list << contact;

   contact.firstName = "FirstName_4";
   contact.lastName = "LastName_4";
   contact.PhoneNum2 = "4-0987654321";
   contact.PhoneNum3 = "4-0123456789";
   contact.emailAdd1 = "FirstName_4@LastName_4.com";
   contact.homeAdd = "home address";

   list << contact;


   contact.firstName = "b_FirstName_5";
   contact.lastName = "b_LastName_5";
   contact.PhoneNum2 = "5-0987654321";
   contact.PhoneNum3 = "5-0123456789";
   contact.emailAdd1 = "b_FirstName_5@LastName_5.com";
   contact.homeAdd = "b_home address";

   list << contact;


   contact.firstName = "FirstName_6";
   contact.lastName = "LastName_6";
   contact.PhoneNum2 = "6-0987654321";
   contact.PhoneNum3 = "6-0123456789";
   contact.emailAdd1 = "FirstName_6@LastName_6.com";
   contact.homeAdd = "home address";

   list << contact;


   contact.firstName = "FirstName_7";
   contact.lastName = "LastName_7";
   contact.PhoneNum2 = "7-0987654321";
   contact.PhoneNum3 = "7-0123456789";
   contact.emailAdd1 = "FirstName_7@LastName_7.com";
   contact.homeAdd = "home address";

   list << contact;

   InsertRecords(list);
}
#endif


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::InsertRecords(
*
* DESCRIPTION:
*              Qt specific function to insert the records in database
*
*
* PARAMETER:  [IN] records - The list of Qt records
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::InsertRecords( QList<FC_PhoneBook_Contact_Detail>& contacts,
                                        quint8 devicehandle, QList<quint32>& ContactHandleList)
{
   ETG_TRACE_USR4(("InsertRecords entered"));

   QMutexLocker lock(&m_mutex);
   quint32 contacthandle;

   QTime DebugTime;
   DebugTime.start();

   bool blRet= true;

   if(!CHECK_VALID(devicehandle))
   {
      if(devicehandle != 0)   // CMG3G-7702
      {
         ETG_TRACE_ERR((" FC_PhoneBook_SQLite: device handle is invalid  "));
         return false;
      }
   }

   QSqlDatabase db= Database();

   //Fix for GMMY15-10814
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   db.transaction();  // Start transaction
   QSqlQuery query(db);
   QString sql;

   if(devicehandle != FC_PB_VEHICLE_PBOOK )  // TODO: Check why we distinguish Vehicle Contacts and Phone Contacts here
   {
      sql = "INSERT INTO " + fc_pb_aTableNames[devicehandle] + " ( "
         FC_PB_COL_CONTACT_HANDLE ", "
         FC_PB_COL_FIRST_NAME     ", "
         FC_PB_COL_LAST_NAME      ", "
         FC_PB_COL_PHONETICS      ", "//292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
         FC_PB_COL_SORTFIRSTNAME  ", "   // NEW
         FC_PB_COL_SORTLASTNAME   ", "   // NEW
         FC_PB_COL_POI_NAME       ", "
         FC_PB_COL_PHONENUM1      ", "
         FC_PB_COL_NUMTYPE1       ", "
         FC_PB_COL_NUMPRETTY1      ", "
         FC_PB_COL_NUMSUBSTR1     ", "   // NEW
         FC_PB_COL_PHONENUM2       ", "
         FC_PB_COL_NUMTYPE2       ", "
         FC_PB_COL_NUMPRETTY2      ", "
         FC_PB_COL_NUMSUBSTR2     ", "   // NEW
         FC_PB_COL_PHONENUM3        ", "
         FC_PB_COL_NUMTYPE3       ", "
         FC_PB_COL_NUMPRETTY3      ", "
         FC_PB_COL_NUMSUBSTR3     ", "   // NEW
         FC_PB_COL_PHONENUM4       ", "
         FC_PB_COL_NUMTYPE4       ", "
         FC_PB_COL_NUMPRETTY4      ", "
         FC_PB_COL_NUMSUBSTR4     ", "   // NEW
         FC_PB_COL_PHONENUM5       ", "
         FC_PB_COL_NUMTYPE5       ", "
         FC_PB_COL_NUMPRETTY5     ", "
         FC_PB_COL_NUMSUBSTR5     ", "   // NEW
         FC_PB_COL_PHONENUM6       ", "
         FC_PB_COL_NUMTYPE6       ", "
         FC_PB_COL_NUMPRETTY6      ", "
         FC_PB_COL_NUMSUBSTR6     ", "   // NEW
         FC_PB_COL_PHONENUM7      ", "
         FC_PB_COL_NUMTYPE7       ", "
         FC_PB_COL_NUMPRETTY7      ", "
         FC_PB_COL_NUMSUBSTR7     ", "   // NEW
         FC_PB_COL_PHONENUM8      ", "
         FC_PB_COL_NUMTYPE8       ", "
         FC_PB_COL_NUMPRETTY8      ", "
         FC_PB_COL_NUMSUBSTR8     ", "   // NEW
         FC_PB_COL_EMAIL_ADD1     ", "
         FC_PB_COL_EMAIL_ADD2     ", "
         FC_PB_COL_EMAIL_ADD3     ", "
         FC_PB_COL_HOME_ADD       ", "
         FC_PB_COL_WORK_ADD       ", "
         FC_PB_COL_OTHER_ADD      ", "
         FC_PB_COL_CATEGORY       ", "
         FC_PB_COL_CONTACT_TYPE   ", "
         FC_PB_COL_NOTES          ", "
         FC_PB_COL_PHOTO_DL_SIZE  ", "
         FC_PB_COL_PHOTO          ", "
         FC_PB_COL_PHOTO_TYPE     ", "      // -FastPhotoHandling-
         FC_PB_COL_PHOTO_DL_CRC   ", "      // -FastPhotoHandling-
         FC_PB_COL_PHOTO_SIZE     ", "      // -FastPhotoHandling-
         FC_PB_COL_PHOTO_CRC      ", "      // -FastPhotoHandling-
         FC_PB_COL_APL1_SIZE      ", "   // -FastContactSearch-
         FC_PB_COL_CRC_APL1       " )"   // -FastContactSearch-
      // "VALUES ( ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?,?);";
         "VALUES ( ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?,?, ?,?, ?,?,?,?,?,?,?,?);";
   }
   else
   {
      sql = "INSERT INTO " + fc_pb_aTableNames[devicehandle] + " ( "
         FC_PB_COL_CONTACT_HANDLE", "
         FC_PB_COL_FIRST_NAME", "
         FC_PB_COL_LAST_NAME", "
         FC_PB_COL_PHONETICS      ", "//292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
         FC_PB_COL_SORTFIRSTNAME  ", "   // NEW
         FC_PB_COL_SORTLASTNAME   ", "   // NEW
         FC_PB_COL_POI_NAME", "
         FC_PB_COL_PHONENUM1", "
         FC_PB_COL_NUMTYPE1 ", "
         FC_PB_COL_NUMPRETTY1", "
         FC_PB_COL_NUMSUBSTR1     ", "   // NEW
         FC_PB_COL_PHONENUM2", "
         FC_PB_COL_NUMTYPE2 ", "
         FC_PB_COL_NUMPRETTY2", "
         FC_PB_COL_NUMSUBSTR2     ", "   // NEW
         FC_PB_COL_PHONENUM3", "
         FC_PB_COL_NUMTYPE3 ", "
         FC_PB_COL_NUMPRETTY3", "
         FC_PB_COL_NUMSUBSTR3     ", "   // NEW
         FC_PB_COL_PHONENUM4", "
         FC_PB_COL_NUMTYPE4 ", "
         FC_PB_COL_NUMPRETTY4", "
         FC_PB_COL_NUMSUBSTR4     ", "   // NEW
         FC_PB_COL_PHONENUM5", "
         FC_PB_COL_NUMTYPE5 ", "
         FC_PB_COL_NUMPRETTY5", "
         FC_PB_COL_NUMSUBSTR5     ", "   // NEW
         FC_PB_COL_PHONENUM6", "
         FC_PB_COL_NUMTYPE6 ", "
         FC_PB_COL_NUMPRETTY6", "
         FC_PB_COL_NUMSUBSTR6     ", "   // NEW
         FC_PB_COL_PHONENUM7", "
         FC_PB_COL_NUMTYPE7 ", "
         FC_PB_COL_NUMPRETTY7", "
         FC_PB_COL_NUMSUBSTR7     ", "   // NEW
         FC_PB_COL_PHONENUM8", "
         FC_PB_COL_NUMTYPE8 ", "
         FC_PB_COL_NUMPRETTY8", "
         FC_PB_COL_NUMSUBSTR8     ", "   // NEW
         FC_PB_COL_EMAIL_ADD1", "
         FC_PB_COL_EMAIL_ADD2", "
         FC_PB_COL_EMAIL_ADD3", "
         FC_PB_COL_HOME_ADD", "
         FC_PB_COL_WORK_ADD", "
         FC_PB_COL_OTHER_ADD", "
         FC_PB_COL_CATEGORY", "
         FC_PB_COL_CONTACT_TYPE","
         FC_PB_COL_NOTES","
         FC_PB_COL_HGCLATLONGVALIDITY","
         FC_PB_COL_HGCLATITUDE","
         FC_PB_COL_HGCLONGITUDE","
         FC_PB_COL_WGCLATLONGVALIDITY","
         FC_PB_COL_WGCLATITUDE","
         FC_PB_COL_WGCLONGITUDE","
         FC_PB_COL_OGCLATLONGVALIDITY","
         FC_PB_COL_OGCLATITUDE","
         FC_PB_COL_OGCLONGITUDE","
         FC_PB_COL_PHOTO_SIZE", "
         FC_PB_COL_PHOTO") "
       //"VALUES ( ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);";
         "VALUES ( ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?,?, ?,?,?,?,?,?,?,?);";
   }

   int counter = 0;
   int size = contacts.size();
   FC_PhoneBook_Contact_Detail* contact;

   ETG_TRACE_USR2(( " InsertRecord: List size= %u", size ));

   for ( counter = 0; counter < size; counter++ )
   {
      contact = &contacts[counter];

      contacthandle = contact->contactHandle; //If a valid contact handle exists, the same will be
      //used. If not, then new contact handle will be generated

      if(!InsertIntoMasterTable(db, devicehandle, contacthandle))
      {
         blRet= false;
         break;
      }
      ContactHandleList << contacthandle;

      //Fix for GMMY15-10814
      if(db.isOpen())
      {
         query.prepare(sql);
      }
      else
      {
         ETG_TRACE_ERR(("Database is not open"));
         return false;
      }

//      FillEmptyStringToEmptyFields(*contact);   // BAD

      //Fix for GMMY15-10814
      if(db.isOpen())
      {
         if(devicehandle != FC_PB_VEHICLE_PBOOK)
         {
            query.addBindValue( contacthandle );
            query.addBindValue( contact->firstName );
            query.addBindValue( contact->lastName );
            query.addBindValue( contact->Phonetics); //292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
            query.addBindValue( contact->SortFirstName);    // NEW
            query.addBindValue( contact->SortLastName);     // NEW
            query.addBindValue( contact->poiName );
            query.addBindValue( contact->PhoneNum1);
            query.addBindValue( contact->NumberType1);
            query.addBindValue( contact->PPPhoneNum1);
            query.addBindValue( contact->NumberSubstr1 );   // NEW
            query.addBindValue( contact->PhoneNum2 );
            query.addBindValue( contact->NumberType2);
            query.addBindValue( contact->PPPhoneNum2 );
            query.addBindValue( contact->NumberSubstr2 );   // NEW
            query.addBindValue( contact->PhoneNum3 );
            query.addBindValue( contact->NumberType3);
            query.addBindValue( contact->PPPhoneNum3 );
            query.addBindValue( contact->NumberSubstr3 );   // NEW
            query.addBindValue( contact->PhoneNum4 );
            query.addBindValue( contact->NumberType4);
            query.addBindValue( contact->PPPhoneNum4 );
            query.addBindValue( contact->NumberSubstr4 );   // NEW
            query.addBindValue( contact->PhoneNum5);
            query.addBindValue( contact->NumberType5);
            query.addBindValue( contact->PPPhoneNum5 );
            query.addBindValue( contact->NumberSubstr5 );   // NEW
            query.addBindValue( contact->PhoneNum6 );
            query.addBindValue( contact->NumberType6);
            query.addBindValue( contact->PPPhoneNum6 );
            query.addBindValue( contact->NumberSubstr6 );   // NEW
            query.addBindValue( contact->PhoneNum7 );
            query.addBindValue( contact->NumberType7);
            query.addBindValue( contact->PPPhoneNum7 );
            query.addBindValue( contact->NumberSubstr7 );   // NEW
            query.addBindValue( contact->PhoneNum8 );
            query.addBindValue( contact->NumberType8);
            query.addBindValue( contact->PPPhoneNum8 );
            query.addBindValue( contact->NumberSubstr8 );   // NEW
            query.addBindValue( contact->emailAdd1 );
            query.addBindValue( contact->emailAdd2 );
            query.addBindValue( contact->emailAdd3 );
            query.addBindValue( contact->homeAdd );
            query.addBindValue( contact->workAdd );
            query.addBindValue( contact->otherAdd );
            query.addBindValue( contact->category );
            query.addBindValue( contact->contactType );
            query.addBindValue( contact->notes );
            //165100-RTC Fix: Contact photo Disappears after edit and manual sync
            if ( 0 == contact->photoDlSize)
            {
               query.addBindValue(0); // contact->photoDLSize  // -FastPhotoHandling- Set Photo details to "0" as Jambit delivers PHOTO even if not requested (in 1st pass download)
               query.addBindValue(QByteArray().constData()); // contact->photo    // -FastPhotoHandling-
               query.addBindValue(""); // contact->photoType    // -FastPhotoHandling-
               query.addBindValue(0); // contact->photoDlCRC   // -FastPhotoHandling-
               query.addBindValue(0); // contact->photoSize    // -FastPhotoHandling-
               query.addBindValue(0); // contact->photoCRC     // -FastPhotoHandling-
            }
            else
            {
               query.addBindValue(contact->photoDlSize);
               query.addBindValue(contact->photo);
               query.addBindValue(contact->photoType);
               query.addBindValue(contact->photoDlCRC);
               query.addBindValue(contact->photoSize);
               query.addBindValue(contact->photoCRC);
            }
            //end of fix
            query.addBindValue( contact->APL1Size );  // -FastContactSearch-
            query.addBindValue( contact->crcAPL1 );   // -FastContactSearch-
         }
         else
         {       // NEW -bn: 20.07.2019: TODO: ISSUE: The following bindings do not fit to the related query. At least the Pretty-Printed Numbers are not bound here. To be checked and fixed.
            query.addBindValue( contacthandle );
            query.addBindValue( contact->firstName );
            query.addBindValue( contact->lastName );
            query.addBindValue( contact->Phonetics); //292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
            query.addBindValue( contact->SortFirstName);    // NEW
            query.addBindValue( contact->SortLastName);     // NEW
            query.addBindValue( contact->poiName );
            query.addBindValue( contact->PhoneNum1 );
            query.addBindValue( contact->NumberType1);
            query.addBindValue( contact->NumberSubstr1 );   // NEW
            query.addBindValue( contact->PhoneNum2 );
            query.addBindValue( contact->NumberType2);
            query.addBindValue( contact->NumberSubstr2 );   // NEW
            query.addBindValue( contact->PhoneNum3 );
            query.addBindValue( contact->NumberType3);
            query.addBindValue( contact->NumberSubstr3 );   // NEW
            query.addBindValue( contact->PhoneNum4 );
            query.addBindValue( contact->NumberType4);
            query.addBindValue( contact->NumberSubstr4 );   // NEW
            query.addBindValue( contact->PhoneNum5 );
            query.addBindValue( contact->NumberType5);
            query.addBindValue( contact->NumberSubstr5 );   // NEW
            query.addBindValue( contact->PhoneNum6 );
            query.addBindValue( contact->NumberType6);
            query.addBindValue( contact->NumberSubstr6 );   // NEW
            query.addBindValue( contact->PhoneNum7 );
            query.addBindValue( contact->NumberType7);
            query.addBindValue( contact->NumberSubstr7 );   // NEW
            query.addBindValue( contact->PhoneNum8 );
            query.addBindValue( contact->NumberType8);
            query.addBindValue( contact->NumberSubstr8 );   // NEW
            query.addBindValue( contact->emailAdd1 );
            query.addBindValue( contact->emailAdd2 );
            query.addBindValue( contact->emailAdd3 );
            query.addBindValue( contact->homeAdd );
            query.addBindValue( contact->workAdd );
            query.addBindValue( contact->otherAdd );
            query.addBindValue( contact->category );
            query.addBindValue( contact->contactType );
            query.addBindValue( contact->notes );
            query.addBindValue( contact->homeGeoCodeLatLongValidity );
            query.addBindValue( contact->homeGeoCodeLatitude );
            query.addBindValue( contact->homeGeocodeLongitude );
            query.addBindValue( contact->workGeoCodeLatLongValidity );
            query.addBindValue( contact->workGeoCodeLatitude );
            query.addBindValue( contact->workGeocodeLongitude );
            query.addBindValue( contact->otherGeoCodeLatLongValidity );
            query.addBindValue( contact->otherGeoCodeLatitude );
            query.addBindValue( contact->otherGeocodeLongitude );
            query.addBindValue( 0 );                         // contact->photoSize   // Vehicle Contacts have no Photo
            query.addBindValue( QByteArray().constData() );  // contact->photo       // Vehicle Contacts have no Photo
         }

         if(!query.exec())
         {
            PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

            ETG_TRACE_ERR(("FC_PhoneBook_SQLite :- ERROR - SQL query failed in InsertRecords"));
            blRet= false;
            break;
         }

         query.clear();
      }
      else
      {
         ETG_TRACE_ERR(("Database is not open"));
         return false;
      }
   }

   //Fix for GMMY15-10814
   if(db.isOpen())
   {
      if(!blRet)
      {
         db.rollback(); //Rollback
         ETG_TRACE_USR4(("Database rolled back"));
      }
      else
      {
         if(!db.commit())
         {
            db.rollback();
            ETG_TRACE_USR4(("Database rolled back since commit failed"));
         }
         else
         {
            ETG_TRACE_USR2(("Database committed successfully andTotal time elapsed in inserting - %d ms", DebugTime.elapsed()));
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   return blRet;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::InsertCallHistoryRecords(
*
* DESCRIPTION:
*              Qt specific function to insert records related to call history in database
*
*
* PARAMETER:  [IN] records - The list of Qt records. Received as variable since fields not provided by device will be updated with null strings.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::InsertCallHistoryRecords( QList<FC_PhoneBook_Callhistory_Detail>& Records, quint8 ConnectionID)
{
   ETG_TRACE_USR4((" InsertCallHistoryRecords entered with connection id :%u", ConnectionID));

   QMutexLocker lock(&m_mutex);
   bool blRet= true;

   QTime DebugTime;
   DebugTime.start();

   if(!CHECK_CH_VALID(ConnectionID))
   {

      ETG_TRACE_USR4(("FC_PhoneBook_SQLite: Connection ID is invalid"));
      return false;
   }

   QSqlDatabase db= Database();

   //Fix for GMMY15-10814
   if(db.isOpen())
   {
      db.transaction(); //Transaction started
   }
   else
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);

   QString sql = "INSERT INTO " + fc_pb_aTableNames[MAP_TO_CH_INDEX(ConnectionID)] + " ( "
      FC_PB_COL_CH_CONTACT_HANDLE", "
      FC_PB_COL_CH_CALL_TYPE", "
      FC_PB_COL_CH_FIRST_NAME", "
      FC_PB_COL_CH_LAST_NAME", "
      FC_PB_COL_CH_PHONE_NUM", "
      FC_PB_COL_CH_PP_PHONE_NUM","
      FC_PB_COL_CH_NUMBER_TYPE", "
      FC_PB_COL_CH_CALL_DATE", "
      FC_PB_COL_CH_CALL_TIME") VALUES ( ?, ?, ?, ?, ?, ?, ?, ?, ? );";
      int counter, size = Records.size();
      FC_PhoneBook_Callhistory_Detail* ch_record;


   ETG_TRACE_USR2(("InsertCallHistoryRecords: List size %u", size));

   quint32 contactHandle;

   for ( counter = 0; counter < size; ++counter )
   {
      ch_record = &Records[counter];
      contactHandle= ch_record->contactHandle;

      /* Generate contact handle only if it is not provided */
      if(!contactHandle)
      {
         if(GetContactHandleForCallHistory(ConnectionID, ch_record->callType, contactHandle))
         {
            if(!InsertIntoMasterTable( db, MAP_TO_CH_INDEX(ConnectionID), contactHandle))
            {
               blRet= false;
               break;
            }
         }
         else
         {
            blRet= false;
            ETG_TRACE_USR4(("Failed to generate contact handle"));
            break;
         }
      }

      //FIX GMMY15-10514 	GMMY15-10515 Null pointer exception
      if(!db.isOpen())
      {
         ETG_TRACE_ERR(("Database is not open"));
         return false;
      }
      //End of fix

      query.prepare( sql );

/*
      //updating fields in call history record with null string, if it is not provided by device.
      //For instance, Sony Ericsson W810C will not provide CallDate and CallTime.
      //We assign default value as null string to such fields in order to avoid malfunction.
      if ( 0 == ch_record->firstName.simplified().length())
      {
         ch_record->firstName = BLANK_NAME;
      }
      if ( 0 == ch_record->lastName.simplified().length())
      {
         ch_record->lastName = BLANK_NAME;
      }

      if ( 0 == ch_record->PPphoneNumber.simplified().length())
      {
         ch_record->PPphoneNumber = BLANK_NAME;
      }

      if ( 0 == ch_record->phoneNumber.simplified().length())
      {
         ch_record->phoneNumber = BLANK_NAME;
      }

      if ( 0 == ch_record->dateStamp.simplified().length())
      {
         ch_record->dateStamp = BLANK_NAME;
      }
      if ( 0 == ch_record->timeStamp.simplified().length())
      {
         ch_record->timeStamp = BLANK_NAME;
      }
      */

      //Fix for GMMY15-10814
      if(db.isOpen())
      {
         query.addBindValue( contactHandle );
         query.addBindValue( ch_record->callType );
         query.addBindValue( ch_record->firstName );
         query.addBindValue( ch_record->lastName );
         query.addBindValue( ch_record->phoneNumber );
         query.addBindValue( ch_record->PPphoneNumber );
         query.addBindValue( ch_record->numberType );
         query.addBindValue( ch_record->dateStamp );
         query.addBindValue( ch_record->timeStamp );

         if (!query.exec())
         {
            PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

            ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query failed in InsertCallHistoryRecords"));
            blRet= false;
            break;
         }
      }
      else
      {
         ETG_TRACE_ERR(("Database is not open"));
         return false;
      }
   }

   //Fix for GMMY15-10814
   if(db.isOpen())
   {
      if(!blRet)
      {
         db.rollback(); //Rollback
         ETG_TRACE_USR4(("Database rolled back"));
      }
      else
      {
         if(!db.commit())
         {
            db.rollback();
            ETG_TRACE_USR4(("Database rolled back since commit failed"));
         }
         else
         {
            ETG_TRACE_USR4(("Total time elapsed in inserting - %d ms", DebugTime.elapsed()));
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }
   return blRet;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetRecords(..)
*
* DESCRIPTION:
*           According to the given 'ListType','sortType' and 'deviceHandle', a sorted list
*           of all Contacts (short form) is retrieved from the database.
*
* PARAMETER:
*   [IN]  -  sortType
*   [IN]  -  deviceHandle
*   [OUT] -  list (of "Short" Contact Details: First-/LastName, Quick Flags)
*
* RETURNVALUE: None.
*
*******************************************************************************/
void FC_PhoneBook_SQLite::GetRecords( tU8 u8ListType ,SortType sortType, quint8 deviceHandle, QList<FC_PhoneBook_Contact_Detail_Short>& list)
{
   ETG_TRACE_USR4((" GetRecords entered with ListType: %d, deviceHandle: %d and sortType: %d", ETG_CENUM(FC_PhoneBook_SQLite::enListType, u8ListType), deviceHandle, ETG_CENUM(FC_PhoneBook_SQLite::SortType, sortType) ));

   QMutexLocker lock(&m_mutex);
   // Preparation of SQL query which requests the relevant information for a "slice" list directly from the database:
   // - inclusive sorting according to GIS-329 Sorting Requirements,
   // - use of appropriate 'substitution detail' in case of empty First and LastName (depending on sortType),
   // - find Quick Dial-, Route- and Email-Flag.
   // The query will return the following columns:
   //      ContactHandle, PoiName, SortName, ReportedFirstName, ReportedLastName, QuickDialFlag, QuickRouteFlag, QuickEmailFlag


   // Set the maxiumum character size for "reported" FirstName or LastName according to GIS-361: FBlock Phonebook,
   // IPhoneBook.RequestPhoneBookListSlice requirements:
   #define FC_PB_REPORTED_NAME_MAX_LEN " 50 "

   // Definition of helper strings, which are used in the queries.

   QString sql_case_substitute_check_sortType__FirstName_Sort = "";   // Hotfix for CRQ 509734 Phonetic Sorting
   if(m_bSortingWithPhonetics)   // Hotfix for CRQ 509734 Phonetic Sorting
   {
      sql_case_substitute_check_sortType__FirstName_Sort =   // Needed for SortName / FirstName sorting types
      "  CASE                                        "
      "     WHEN   SortFirstName != ''  THEN   SortFirstName || ' ' || SortLastName "  // Hotfix for CRQ 509734 Phonetic Sorting
      "     WHEN   SortLastName  != ''  THEN   SortLastName "
      "     WHEN   FirstName     != ''  THEN   FirstName || ' ' || LastName "
      "     ELSE CASE                                "
      "     WHEN   LastName      != ''  THEN   LastName  "
      "     ELSE CASE                                "
      "     WHEN   PoiName       != ''  THEN   PoiName   ";
   }
   else
   {
      sql_case_substitute_check_sortType__FirstName_Sort =   // Needed for SortName / FirstName sorting types
      "  CASE                                        "
      "     WHEN   FirstName != ''  THEN   FirstName || ' ' || LastName "
      "     ELSE CASE                                "
      "     WHEN   LastName  != ''  THEN   LastName  "
      "     ELSE CASE                                "
      "     WHEN   PoiName   != ''  THEN   PoiName   ";
   }

   QString sql_case_substitute_check_sortType__LastName_Sort = "";   // Hotfix for CRQ 509734 Phonetic Sorting
   if(m_bSortingWithPhonetics)   // Hotfix for CRQ 509734 Phonetic Sorting
   {
      sql_case_substitute_check_sortType__LastName_Sort =   // Needed for SortName / LastName sorting types
      "  CASE                                        "
      "     WHEN   SortLastName  != ''  THEN   SortLastName || ' ' || SortFirstName "  // Hotfix for CRQ 509734 Phonetic Sorting
      "     WHEN   SortFirstName != ''  THEN   SortFirstName "
      "     WHEN   LastName      != ''  THEN   LastName || ' ' || FirstName "
      "     ELSE CASE                                "
      "     WHEN   FirstName     != ''  THEN   FirstName "
      "     ELSE CASE                                "
      "     WHEN   PoiName       != ''  THEN   PoiName   ";
   }
   else
   {
      sql_case_substitute_check_sortType__LastName_Sort =   // Needed for SortName / LastName sorting types
      "  CASE                                        "
      "     WHEN   LastName  != ''  THEN   LastName || ' ' || FirstName "
      "     ELSE CASE                                "
      "     WHEN   FirstName != ''  THEN   FirstName "
      "     ELSE CASE                                "
      "     WHEN   PoiName   != ''  THEN   PoiName   ";
   }

   QString sql_case_substitute_check_sortType__FirstName_View =  // Needed for ReportedFirstName / FirstName sorting types
      "  CASE                                        "
      "     WHEN   FirstName != ''  THEN   FirstName "
      "     ELSE CASE                                "
      "     WHEN   LastName  != ''  THEN   LastName  "
      "     ELSE CASE                                "
      "     WHEN   PoiName   != ''  THEN   PoiName   ";
   QString sql_case_substitute_check_sortType__LastName_View =   // Needed for ReportedLastName / LastName sorting types
      "  CASE                                        "
      "     WHEN   LastName  != ''  THEN   LastName  "
      "     ELSE CASE                                "
      "     WHEN   FirstName != ''  THEN   FirstName "
      "     ELSE CASE                                "
      "     WHEN   PoiName   != ''  THEN   PoiName   ";
   QString sql_case_last_name_already_used_as_first_name_sortType_FirstName =  // Needed for ReportedFirstName / FirstName sorting types
      " CASE "
      "    WHEN   FirstName != '' "
      "       THEN LastName "
      "       ELSE '' "
      "    END ";
   QString sql_case_first_name_already_used_as_last_name_sortType_LastName =   // Needed for ReportedLastName / LastName sorting types
      " CASE "
      "    WHEN   LastName != '' "
      "       THEN FirstName "
      "       ELSE '' "
      "    END ";
   QString sql_case_substitute_check_Number =
      "     ELSE CASE                                "
      "     WHEN   PhoneNumber1   != ''  THEN   PhoneNumber1   "
      "     ELSE CASE                                "
      "     WHEN   PhoneNumber2  != ''  THEN   PhoneNumber2  "
      "     ELSE CASE                                "
      "     WHEN   PhoneNumber3  != ''  THEN   PhoneNumber3  "
      "     ELSE CASE                                "
      "     WHEN   PhoneNumber4  != ''  THEN   PhoneNumber4  "
      "     ELSE CASE                                "
      "     WHEN   PhoneNumber5  != ''  THEN   PhoneNumber5  "
      "     ELSE CASE                                "
      "     WHEN   PhoneNumber6  != ''  THEN   PhoneNumber6  "
      "     ELSE CASE                                "
      "     WHEN   PhoneNumber7  != ''  THEN   PhoneNumber7  "
      "     ELSE CASE                                "
      "     WHEN   PhoneNumber8  != ''  THEN   PhoneNumber8  ";
   QString sql_case_substitute_check_Address =
      "     ELSE CASE                                "
      "     WHEN   HomeAdd   != ''  THEN   HomeAdd   "
      "     ELSE CASE                                "
      "     WHEN   WorkAdd   != ''  THEN   WorkAdd   "
      "     ELSE CASE                                "
      "     WHEN   OtherAdd  != ''  THEN   OtherAdd  ";
   QString sql_case_substitute_check_Email =
      "     ELSE CASE                                "
      "     WHEN   EmailAdd1 != ''  THEN   EmailAdd1 "
      "     ELSE CASE                                "
      "     WHEN   EmailAdd2 != ''  THEN   EmailAdd2 "
      "     ELSE CASE                                "
      "     WHEN   EmailAdd3 != ''  THEN   EmailAdd3 ";
   QString sql_case_substitute_check_Category_Notes_ENDs =
      "     ELSE CASE                                "
      "     WHEN   Category  != ''  THEN   Category  "
      "     ELSE CASE                                "
      "     WHEN   Notes     != ''  THEN   Notes     "
      "  "
      "     END END END                     "
      "     END END END END END END END END "
      "     END END END                     "
      "     END END END                     "
      "     END END                         ";
   QString sql_case_check_Quick_Dial_Route_Email_flags =
      "  CASE              "
      "     WHEN ((NumberType1 == 1) + (NumberType2 == 1)+ (NumberType3 == 1)+ (NumberType4 == 1)+ (NumberType5 == 1)+ (NumberType6 == 1) + (NumberType7 == 1)+ (NumberType8 == 1))  THEN 1 "
      "     ELSE CASE      "
      "     WHEN (   (PhoneNumber1 != '') + (PhoneNumber2 != '') + (PhoneNumber3 != '') "
      "            + (PhoneNumber5 != '') + (PhoneNumber6 != '') + (PhoneNumber7 != '') + (PhoneNumber8 != '') )  == 1 "
      "        THEN  1     "
      "        ELSE  0     "
      "     END END        "
      "  AS QuickDialFlag, "
      "  "
      "  CASE                  "
      "     WHEN (   (HomeAdd != '') + (WorkAdd != '') + (OtherAdd != '') )  == 1 "
      "        THEN 1       "
      "        ELSE 0       "
      "     END             "
      "  AS QuickRouteFlag, "
      "  "
      "  CASE                  "
      "     WHEN (   (EmailAdd1  != '')  +  (EmailAdd2 != '') + (EmailAdd3 != '') )  == 1 "
      "        THEN 1      "
      "        ELSE 0      "
      "     END            "
      "  AS QuickEmailFlag ";

   QString sql;

   if(u8ListType == most_fi_tcl_e8_PhonBkContactListType::FI_EN_PHONEBOOK || u8ListType == most_fi_tcl_e8_PhonBkContactListType::FI_EN_FAVORITE)
   {
      switch (sortType)
      {
         case FirstName:
         case Number_FirstName:
         {
            sql =
                  " SELECT  ContactHandle, PoiName, "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__FirstName_Sort
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Category_Notes_ENDs +
                  " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS SortName, "
                  "  "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__FirstName_View
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Category_Notes_ENDs +
                  " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS ReportedFirstName, "
                  "  "
                  + sql_case_last_name_already_used_as_first_name_sortType_FirstName +
                  " AS ReportedLastName, "
                  "  "
                  + sql_case_check_Quick_Dial_Route_Email_flags;
            //NCG3D-148499: Favorite Contact Name empty on Incoming Screen
            if(u8ListType == most_fi_tcl_e8_PhonBkContactListType::FI_EN_FAVORITE)
            {
               // For ListType = 'FAVORITES' we extract only the FAV entries. HMI uses this ListType to show (the small number) of contacts, which a tagged as FAVORITE
               sql += "  FROM " + fc_pb_aTableNames[deviceHandle] + " WHERE "  FC_PB_COL_CONTACT_TYPE " = " + QString::number(u8ListType) + " ORDER BY SortName COLLATE " + SORTLIB_COLLATE_NAME;
            }
            else
            {
               // For ListType = 'PHONEBOOK' we use all contacts, i.e. also the FAVORITES as these are small subset of all contacts.
               sql += "  FROM " + fc_pb_aTableNames[deviceHandle] + " ORDER BY SortName COLLATE " + SORTLIB_COLLATE_NAME;
            }
         }
         break;
         case LastName:
         case Number_LastName:
         {
            sql =
                  " SELECT  ContactHandle, PoiName, "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__LastName_Sort
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Category_Notes_ENDs +
                  " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS SortName, "
                  "  "
                  + sql_case_first_name_already_used_as_last_name_sortType_LastName +
                  " AS ReportedFirstName, "
                  "  "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__LastName_View
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Category_Notes_ENDs +
                  " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS ReportedLastName, "
                  "  "
                  + sql_case_check_Quick_Dial_Route_Email_flags;
             //NCG3D-148499: Favorite Contact Name empty on Incoming Screen
            if(u8ListType == most_fi_tcl_e8_PhonBkContactListType::FI_EN_FAVORITE)
            {
               // For ListType = 'FAVORITES' we extract only the FAV entries. HMI uses this ListType to show (the small number) of contacts, which a tagged as FAVORITE
               sql += "  FROM " + fc_pb_aTableNames[deviceHandle] + " WHERE "  FC_PB_COL_CONTACT_TYPE " = " + QString::number(u8ListType) + " ORDER BY SortName COLLATE " + SORTLIB_COLLATE_NAME;
            }
            else
            {
               // For ListType = 'PHONEBOOK' we use all contacts, i.e. also the FAVORITES as these are small subset of all contacts.
               sql += "  FROM " + fc_pb_aTableNames[deviceHandle] + " ORDER BY SortName COLLATE " + SORTLIB_COLLATE_NAME;
            }
         }
         break;
         case Address_FirstName:
         {
            sql =
                  " SELECT  ContactHandle, PoiName, "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__FirstName_Sort
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Category_Notes_ENDs +
                  " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS SortName, "
                  "  "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__FirstName_View
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Category_Notes_ENDs +
                  " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS ReportedFirstName, "
                  "  "
                  + sql_case_last_name_already_used_as_first_name_sortType_FirstName +
                  " AS ReportedLastName, "
                  "  "
                  + sql_case_check_Quick_Dial_Route_Email_flags;
            //NCG3D-148499: Favorite Contact Name empty on Incoming Screen
            if(u8ListType == most_fi_tcl_e8_PhonBkContactListType::FI_EN_FAVORITE)
            {
               // For ListType = 'FAVORITES' we extract only the FAV entries. HMI uses this ListType to show (the small number) of contacts, which a tagged as FAVORITE
               sql += "  FROM " + fc_pb_aTableNames[deviceHandle] + " WHERE "  FC_PB_COL_CONTACT_TYPE " = " + QString::number(u8ListType) + " ORDER BY SortName COLLATE " + SORTLIB_COLLATE_NAME;
            }
            else
            {
               // For ListType = 'PHONEBOOK' we use all contacts, i.e. also the FAVORITES as these are small subset of all contacts.
               sql += "  FROM " + fc_pb_aTableNames[deviceHandle] + " ORDER BY SortName COLLATE " + SORTLIB_COLLATE_NAME;
            }
         }
         break;
         case Address_LastName:
         {
            sql =
                  " SELECT  ContactHandle, PoiName, "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__LastName_Sort
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Category_Notes_ENDs +
                  " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS SortName, "
                  "  "
                  + sql_case_first_name_already_used_as_last_name_sortType_LastName +
                  " AS ReportedFirstName, "
                  "  "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__LastName_View
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Category_Notes_ENDs +
                  " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS ReportedLastName, "
                  "  "
                  + sql_case_check_Quick_Dial_Route_Email_flags;
             //NCG3D-148499: Favorite Contact Name empty on Incoming Screen
            if(u8ListType == most_fi_tcl_e8_PhonBkContactListType::FI_EN_FAVORITE)
            {
               // For ListType = 'FAVORITES' we extract only the FAV entries. HMI uses this ListType to show (the small number) of contacts, which a tagged as FAVORITE
               sql += "  FROM " + fc_pb_aTableNames[deviceHandle] + " WHERE "  FC_PB_COL_CONTACT_TYPE " = " + QString::number(u8ListType) + " ORDER BY SortName COLLATE " + SORTLIB_COLLATE_NAME;
            }
            else
            {
               // For ListType = 'PHONEBOOK' we use all contacts, i.e. also the FAVORITES as these are small subset of all contacts.
               sql += "  FROM " + fc_pb_aTableNames[deviceHandle] + " ORDER BY SortName COLLATE " + SORTLIB_COLLATE_NAME;
            }
         }
         break;
         case Email_FirstName:
         {
            sql =
                  " SELECT  ContactHandle, PoiName, "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__FirstName_Sort
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Category_Notes_ENDs +
                  " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS SortName, "
                  "  "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__FirstName_View
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Category_Notes_ENDs +
                  " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS ReportedFirstName, "
                  "  "
                  + sql_case_last_name_already_used_as_first_name_sortType_FirstName +
                  " AS ReportedLastName, "
                  "  "
                  + sql_case_check_Quick_Dial_Route_Email_flags;
             //NCG3D-148499: Favorite Contact Name empty on Incoming Screen
            if(u8ListType == most_fi_tcl_e8_PhonBkContactListType::FI_EN_FAVORITE)
            {
               // For ListType = 'FAVORITES' we extract only the FAV entries. HMI uses this ListType to show (the small number) of contacts, which a tagged as FAVORITE
               sql += "  FROM " + fc_pb_aTableNames[deviceHandle] + " WHERE "  FC_PB_COL_CONTACT_TYPE " = " + QString::number(u8ListType) + " ORDER BY SortName COLLATE " + SORTLIB_COLLATE_NAME;
            }
            else
            {
               // For ListType = 'PHONEBOOK' we use all contacts, i.e. also the FAVORITES as these are small subset of all contacts.
               sql += "  FROM " + fc_pb_aTableNames[deviceHandle] + " ORDER BY SortName COLLATE " + SORTLIB_COLLATE_NAME;
            }
         }
         break;
         case Email_LastName:
         {
            sql =
                  " SELECT  ContactHandle, PoiName, "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__LastName_Sort
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Category_Notes_ENDs +
                  " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS SortName, "
                  "  "
                  + sql_case_first_name_already_used_as_last_name_sortType_LastName +
                  " AS ReportedFirstName, "
                  "  "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__LastName_View
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Category_Notes_ENDs +
                  " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS ReportedLastName, "
                  "  "
                  + sql_case_check_Quick_Dial_Route_Email_flags;
             //NCG3D-148499: Favorite Contact Name empty on Incoming Screen
            if(u8ListType == most_fi_tcl_e8_PhonBkContactListType::FI_EN_FAVORITE)
            {
               // For ListType = 'FAVORITES' we extract only the FAV entries. HMI uses this ListType to show (the small number) of contacts, which a tagged as FAVORITE
               sql += "  FROM " + fc_pb_aTableNames[deviceHandle] + " WHERE "  FC_PB_COL_CONTACT_TYPE " = " + QString::number(u8ListType) + " ORDER BY SortName COLLATE " + SORTLIB_COLLATE_NAME;
            }
            else
            {
               // For ListType = 'PHONEBOOK' we use all contacts, i.e. also the FAVORITES as these are small subset of all contacts.
               sql += "  FROM " + fc_pb_aTableNames[deviceHandle] + " ORDER BY SortName COLLATE " + SORTLIB_COLLATE_NAME;
            }
         }
         break;
         case Position:  // -bn: 2019.07.02 To my understanding Position sorting makes no sense for PHONEBOOK or FAVORITES at all.
         {               //                 (Note: Currently HMI requests the FAV list with Position sorting, which will result in a wrong sorting.)
            sql
            = " SELECT  ContactHandle, PoiName, "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__LastName_Sort
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Category_Notes_ENDs
                  + " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS SortName, "
                  "  "
                  + sql_case_first_name_already_used_as_last_name_sortType_LastName
                  + " AS ReportedFirstName, "
                  "  "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__LastName_View
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Category_Notes_ENDs
                  + " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS ReportedLastName, "
                  "  " + sql_case_check_Quick_Dial_Route_Email_flags;
             //NCG3D-148499: Favorite Contact Name empty on Incoming Screen
            if(u8ListType == most_fi_tcl_e8_PhonBkContactListType::FI_EN_FAVORITE)
            {
               // For ListType = 'FAVORITES' we extract only the FAV entries. HMI uses this ListType to show (the small number) of contacts, which a tagged as FAVORITE
               sql += "  FROM " + fc_pb_aTableNames[deviceHandle] + " WHERE "  FC_PB_COL_CONTACT_TYPE " = " + QString::number(u8ListType) + " ORDER BY ROWID ";
            }
            else
            {
               // For ListType = 'PHONEBOOK' we use all contacts, i.e. also the FAVORITES as these are small subset of all contacts.
               sql += "  FROM " + fc_pb_aTableNames[deviceHandle] + " ORDER BY ROWID ";
            }
         }
         break;
         default:
         {
            ETG_TRACE_ERR((" Sort type not supported!"));
            return;
         }
      } // end of: 'switch(sortType)'
   }    // end of: 'switch(Listtype :Phonebook/Vehilce)'
   else if (u8ListType == most_fi_tcl_e8_PhonBkContactListType::FI_EN_VEHICLE)
   {
      switch (sortType)
      {
         case FirstName:
         case Number_FirstName:
         {
            sql
            = " SELECT  ContactHandle, PoiName, "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__FirstName_Sort
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Category_Notes_ENDs
                  + " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS SortName, "
                  "  "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__FirstName_View
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Category_Notes_ENDs
                  + " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS ReportedFirstName, "
                  "  "
                  + sql_case_last_name_already_used_as_first_name_sortType_FirstName
                  + " AS ReportedLastName, "
                  "  " + sql_case_check_Quick_Dial_Route_Email_flags
                  + "  FROM " + fc_vpb_aTableNames[deviceHandle]
                  + " ORDER BY SortName COLLATE " + SORTLIB_COLLATE_NAME;
         }
         break;
         case LastName:
         case Number_LastName:
         {
            sql
            = " SELECT  ContactHandle, PoiName, "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__LastName_Sort
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Category_Notes_ENDs
                  + " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS SortName, "
                  "  "
                  + sql_case_first_name_already_used_as_last_name_sortType_LastName
                  + " AS ReportedFirstName, "
                  "  "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__LastName_View
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Category_Notes_ENDs
                  + " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS ReportedLastName, "
                  "  " + sql_case_check_Quick_Dial_Route_Email_flags
                  + "  FROM " + fc_vpb_aTableNames[deviceHandle]
                  + " ORDER BY SortName COLLATE " + SORTLIB_COLLATE_NAME;
         }
         break;
         case Address_FirstName:
         {
            sql
            = " SELECT  ContactHandle, PoiName, "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__FirstName_Sort
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Category_Notes_ENDs
                  + " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS SortName, "
                  "  "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__FirstName_View
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Category_Notes_ENDs
                  + " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS ReportedFirstName, "
                  "  "
                  + sql_case_last_name_already_used_as_first_name_sortType_FirstName
                  + " AS ReportedLastName, "
                  "  " + sql_case_check_Quick_Dial_Route_Email_flags
                  + "  FROM " + fc_vpb_aTableNames[deviceHandle]
                  + " ORDER BY SortName COLLATE " + SORTLIB_COLLATE_NAME;
         }
         break;
         case Address_LastName:
         {
            sql
            = " SELECT  ContactHandle, PoiName, "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__LastName_Sort
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Category_Notes_ENDs
                  + " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS SortName, "
                  "  "
                  + sql_case_first_name_already_used_as_last_name_sortType_LastName
                  + " AS ReportedFirstName, "
                  "  "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__LastName_View
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Category_Notes_ENDs
                  + " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS ReportedLastName, "
                  "  " + sql_case_check_Quick_Dial_Route_Email_flags
                  + "  FROM " + fc_vpb_aTableNames[deviceHandle]
                  + " ORDER BY SortName COLLATE " + SORTLIB_COLLATE_NAME;
         }
         break;
         case Email_FirstName:
         {
            sql
            = " SELECT  ContactHandle, PoiName, "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__FirstName_Sort
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Category_Notes_ENDs
                  + " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS SortName, "
                  "  "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__FirstName_View
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Category_Notes_ENDs
                  + " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS ReportedFirstName, "
                  "  "
                  + sql_case_last_name_already_used_as_first_name_sortType_FirstName
                  + " AS ReportedLastName, "
                  "  " + sql_case_check_Quick_Dial_Route_Email_flags
                  + "  FROM " + fc_vpb_aTableNames[deviceHandle]
                  + " ORDER BY SortName COLLATE " + SORTLIB_COLLATE_NAME;
         }
         break;
         case Email_LastName:
         {
            sql
            = " SELECT  ContactHandle, PoiName, "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__LastName_Sort
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Category_Notes_ENDs
                  + " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS SortName, "
                  "  "
                  + sql_case_first_name_already_used_as_last_name_sortType_LastName
                  + " AS ReportedFirstName, "
                  "  "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__LastName_View
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Category_Notes_ENDs
                  + " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS ReportedLastName, "
                  "  " + sql_case_check_Quick_Dial_Route_Email_flags
                  + "  FROM " + fc_vpb_aTableNames[deviceHandle]
                  + " ORDER BY SortName COLLATE " + SORTLIB_COLLATE_NAME;
         }
         break;
         case Position:
         {
            sql
            = " SELECT  ContactHandle, PoiName, "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__LastName_Sort
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Category_Notes_ENDs
                  + " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS SortName, "
                  "  "
                  + sql_case_first_name_already_used_as_last_name_sortType_LastName
                  + " AS ReportedFirstName, "
                  "  "
                  " SUBSTR( "
                  + sql_case_substitute_check_sortType__LastName_View
                  + sql_case_substitute_check_Email
                  + sql_case_substitute_check_Number
                  + sql_case_substitute_check_Address
                  + sql_case_substitute_check_Category_Notes_ENDs
                  + " , 1, " + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                  " AS ReportedLastName, "
                  "  " + sql_case_check_Quick_Dial_Route_Email_flags
                  + "  FROM " + fc_vpb_aTableNames[deviceHandle]
                  + " ORDER BY " FC_PB_COL_VPB_CONTACT_POSITION " ";

         }
         break;
         default:
         {
            ETG_TRACE_ERR((" Sort type not supported!"));
            return;
         }
      }
   }

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(( " Database is not open " ));
   }
   else
   {
      QSqlQuery query(db);
      if ( !query.exec(sql) )
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(( " FC_PhoneBook_SQLite : - ERROR - SQL query failed in GetContactRecords " ));
      }
      else
      {
         int counter = 0;
         while (query.next())
         {
            counter = 0;
            FC_PhoneBook_Contact_Detail_Short contact;
            contact.contactHandle = query.value( counter++ ).toUInt();      // ContactHandle
            contact.poiName       = query.value( counter++ ).toString();    // PoiName
            // counter++;                 // skip the SortName   // Hotfix for CRQ 509734 Phonetic Sorting (Commented out this line.)
            contact.sortName      = query.value( counter++ ).toString();    // SortName  // Hotfix for CRQ 509734 Phonetic Sorting
            contact.firstName     = query.value( counter++ ).toString();    // ReportedFirstName
            contact.lastName      = query.value( counter++ ).toString();    // ReportedLastName
            contact.bQuickDial    = query.value( counter++ ).toUInt();       // QuickDialFlag
            contact.bQuickRoute   = query.value( counter++ ).toUInt();       // QuickRouteFlag
            contact.bQuickText    = query.value( counter++ ).toUInt();       // QuickEmailFlag
            list << contact;
         }
         if ( list.isEmpty() )
         {
            ETG_TRACE_USR4(("FC_PhoneBook_SQLite :- WARNING - Empty list of contacts returned"));
         }
      }
   }
}


/*******************************************************************************
* FUNCTION: FC_PhoneBook_SQLite::GetTotalRecords(
*
* DESCRIPTION:
*              Retrieve Total records in database
*
*
* PARAMETER:  [IN] DeviceHandle
*             [OUT] - None
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
quint16 FC_PhoneBook_SQLite::GetTotalRecords(quint8 devicehandle)
{
   ETG_TRACE_USR2(("FC_PhoneBook_SQLite::GetTotalRecords entered device Handle=%d",devicehandle));
   QMutexLocker lock(&m_mutex);//GMMY17-17143
   quint16 u16Count= 0;
   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);

      QString sql = " SELECT COUNT (" FC_PB_COL_CONTACT_HANDLE ") FROM " + fc_pb_aTableNames[devicehandle];

      ETG_TRACE_USR2(( " SQL query used to find Total Number of Records- %s ", sql.toLatin1().constData() ));

      if(!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("Finding Total Records Failed"));
         u16Count = 0;
      }
      else
      {
         query.next();
         if(query.isValid() == true)
         {
            u16Count = (query.value(0)).value<quint16>();
            ETG_TRACE_USR2(("Total No. of Records = %d" , u16Count));
         }
         else
         {
            ETG_TRACE_ERR(("Sql query failed"));
         }
      }
   }

   return u16Count;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetCallHistoryRecords()
*
* DESCRIPTION:
*             This is a Qt specific function to get the list of all call history related
*             records in the current database for a particular device.
* PARAMETER:
*
* RETURNVALUE: A list of Qt format contacts.
*
*******************************************************************************/
void FC_PhoneBook_SQLite::GetCallHistoryRecords( SortType enSorttype, quint8 u8CallType,
                                                quint8 u8ConnectionID, QList<FC_PhoneBook_Callhistory_Detail> &list, bool bDeviceDateTimeSupport )//Passing callhistoryList as reference
{
   ETG_TRACE_USR4(("GetCallHistoryRecords entered"));

   QMutexLocker lock(&m_mutex);
   QString sql = GetSqlString( enSorttype, u8CallType, u8ConnectionID, bDeviceDateTimeSupport );

   QSqlDatabase db= Database();

   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return;
   }

   QSqlQuery query(db);
   FC_PhoneBook_Callhistory_Detail record;

   if (!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR( ( "FC_PhoneBook_SQLite :- ERROR - SQL query failed in GetCallHistoryRecords "));
   }
   else
   {
      int counter = 0;

      if ( u8CallType == FC_PhoneBook_SQLite::callhisttype_combined)
      {
         // Restrict CCH list count to 50.
         for( int NoOfCchEntry = 0; ((db.isOpen() && query.next()) && (NoOfCchEntry < MAX_NO_OF_CCH_ENTRIES)); NoOfCchEntry++)
         {
            counter = 0;
            record.contactHandle = query.value( counter++ ).toUInt();
            record.callType      = (query.value(counter++)).value<quint8>();
            record.firstName     = query.value( counter++ ).toString();
            record.lastName      = query.value( counter++ ).toString();
            record.phoneNumber   = query.value( counter++ ).toString();
            record.PPphoneNumber = query.value( counter++ ).toString();
            record.numberType    = (query.value(counter++)).value<quint8>();
            record.dateStamp     = query.value( counter++ ).toString();
            record.timeStamp     = query.value( counter++ ).toString();
            record.callCount     = (query.value(counter++)).value<quint8>();

            list << record;
         }
      }
      else
      {
         while (db.isOpen() && query.next())
         {
            counter = 0;

            record.contactHandle = query.value( counter++ ).toUInt();
            record.callType      = (query.value(counter++)).value<quint8>();
            record.firstName     = query.value( counter++ ).toString();
            record.lastName      = query.value( counter++ ).toString();
            record.phoneNumber   = query.value( counter++ ).toString();
            record.PPphoneNumber = query.value( counter++ ).toString();
            record.numberType    = (query.value(counter++)).value<quint8>();
            record.dateStamp     = query.value( counter++ ).toString();
            record.timeStamp     = query.value( counter++ ).toString();
            record.callCount     = (query.value(counter++)).value<quint8>();

            list << record;
         }
      }
      if ( list.isEmpty() )
      {
         ETG_TRACE_USR4( ( "FC_PhoneBook_SQLite :- WARNING - Empty call history list "));
      }
   }

}


//Fix for CMG3GB-621 Call count is not incremented after consecutive incoming/outgoing/missed calls from same number
/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetAllCallHistoryRecords()
*
* DESCRIPTION:
*             This is a Qt specific function to get the list of all call history related
*             records in the current database for a particular device.
* PARAMETER:
*
* RETURNVALUE: A list of Qt format contacts.
*
*******************************************************************************/
QList<FC_PhoneBook_Callhistory_Detail> FC_PhoneBook_SQLite::GetAllCallHistoryRecords(quint8 u8ConnectionID)
{
   ETG_TRACE_USR4(("GetAllCallHistoryRecords entered"));

   QMutexLocker lock(&m_mutex);
   QString sql = "SELECT * FROM " + fc_pb_aTableNames[MAP_TO_CH_INDEX(u8ConnectionID)];
   QList < FC_PhoneBook_Callhistory_Detail > list;

   QSqlDatabase db = Database();

   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);
      FC_PhoneBook_Callhistory_Detail record;

      ETG_TRACE_USR2((" SQL query used: %s ",sql.toLatin1().constData()));

      if (!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("FC_PhoneBook_SQLite :- ERROR - SQL query failed in GetAllCallHistoryRecords "));
      }
      else
      {
         int counter = 0;
         while (db.isOpen() && query.next())
         {
            counter = 0;

            record.contactHandle = query.value(counter++).toUInt();
            record.callType = (query.value(counter++)).value<quint8>();
            record.firstName = query.value(counter++).toString();
            record.lastName = query.value(counter++).toString();
            record.phoneNumber = query.value(counter++).toString();
            record.PPphoneNumber = query.value(counter++).toString();
            record.numberType = (query.value(counter++)).value<quint8>();
            record.dateStamp = query.value(counter++).toString();
            record.timeStamp = query.value(counter++).toString();

            list << record;
         }

         if (list.isEmpty())
         {
            ETG_TRACE_USR4(("FC_PhoneBook_SQLite :- WARNING - Empty call history list "));
         }
      }
   }
   return list;
}
// End of Fix CMG3GB-621


// BAD
/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::FillEmptyStringToEmptyFields()
*
* DESCRIPTION:

* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/

/*
void FC_PhoneBook_SQLite::FillEmptyStringToEmptyFields(FC_PhoneBook_Contact_Detail &contact)   // NEW -bn: 20.07.2019: This method would become obsolete, when the constructor of class FC_PhoneBook_Contact_Detail would initialize all member variables correctly.
{
   ETG_TRACE_USR4(("FillEmptyStringToEmptyFields entered"));

   if ( 0 == contact.PPPhoneNum1.simplified().length() )
   {
      contact.PPPhoneNum1 = BLANK_NAME;
   }
   if ( 0 == contact.PPPhoneNum2.simplified().length() )
   {
      contact.PPPhoneNum2 = BLANK_NAME;
   }
   if ( 0 == contact.PPPhoneNum3.simplified().length() )
   {
      contact.PPPhoneNum3 = BLANK_NAME;
   }
   if ( 0 == contact.PPPhoneNum4.simplified().length() )
   {
      contact.PPPhoneNum4 = BLANK_NAME;
   }
   if ( 0 == contact.PPPhoneNum5.simplified().length() )
   {
      contact.PPPhoneNum5 = BLANK_NAME;
   }
   if ( 0 == contact.PPPhoneNum6.simplified().length() )
   {
      contact.PPPhoneNum6 = BLANK_NAME;
   }
   if ( 0 == contact.PPPhoneNum7.simplified().length() )
   {
      contact.PPPhoneNum7 = BLANK_NAME;
   }
   if ( 0 == contact.PPPhoneNum8.simplified().length() )
   {
      contact.PPPhoneNum8 = BLANK_NAME;
   }
   if ( 0 == contact.firstName.simplified().length() )
   {
      contact.firstName = BLANK_NAME;
   }
   if ( 0 == contact.lastName.simplified().length() )
   {
      contact.lastName = BLANK_NAME;
   } //292459 - Japanese Yomi Phonetics Field
   if ( 0 == contact.Phonetics.simplified().length() )
   {
      contact.Phonetics = BLANK_NAME;
   }//end of 292459 - Japanese Yomi Phonetics Field
   if ( 0 == contact.homeAdd.simplified().length() )
   {
      contact.homeAdd = BLANK_NAME;
   }
   if ( 0 == contact.workAdd.simplified().length() )
   {
      contact.workAdd = BLANK_NAME;
   }
   if ( 0 == contact.otherAdd.simplified().length() )
   {
      contact.otherAdd = BLANK_NAME;
   }
   if ( 0 == contact.poiName.simplified().length() )
   {
      contact.poiName = BLANK_NAME;
   }
   if ( 0 == contact.PhoneNum1.simplified().length() )
   {
      contact.PhoneNum1 = BLANK_NAME;
   }
   if ( 0 == contact.PhoneNum2.simplified().length() )
   {
      contact.PhoneNum2 = BLANK_NAME;
   }
   if ( 0 == contact.PhoneNum3.simplified().length() )
   {
      contact.PhoneNum3 = BLANK_NAME;
   }
   if ( 0 == contact.PhoneNum4.simplified().length() )
   {
      contact.PhoneNum4 = BLANK_NAME;
   }
   if ( 0 == contact.PhoneNum5.simplified().length() )
   {
      contact.PhoneNum5 = BLANK_NAME;
   }
   if ( 0 == contact.PhoneNum6.simplified().length() )
   {
      contact.PhoneNum6 = BLANK_NAME;
   }
   if ( 0 == contact.PhoneNum7.simplified().length() )
   {
      contact.PhoneNum7 = BLANK_NAME;
   }
   if ( 0 == contact.PhoneNum8.simplified().length() )
   {
      contact.PhoneNum8 = BLANK_NAME;
   }
   if ( 0 == contact.emailAdd1.simplified().length() )
   {
      contact.emailAdd1 = BLANK_NAME;
   }
   if ( 0 == contact.emailAdd2.simplified().length() )
   {
      contact.emailAdd2 = BLANK_NAME;
   }
   if ( 0 == contact.emailAdd3.simplified().length() )
   {
      contact.emailAdd3 = BLANK_NAME;
   }
   if ( 0 == contact.category.simplified().length() )
   {
      contact.category = BLANK_NAME;
   }
   if ( 0 == contact.notes.simplified().length() )
   {
      contact.notes = BLANK_NAME;
   }
}
*/


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::InsertSingleRecord(
*
* DESCRIPTION:
*              Inserts a single record into database
*
*
* PARAMETER:  [IN] records - record to be added
*             [OUT] - Contacthandle of the record
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::InsertSingleRecord(FC_PhoneBook_Contact_Detail& contact, quint8 devicehandle, quint32& contacthandle,bool bIsVehiclePbk)//CMG3G-14313-FC_PhoneBook_Contact_Detail-Pass by Reference-Coverity Fix
{
   ETG_TRACE_USR4(("InsertSingleRecord entered"));

   QMutexLocker lock(&m_mutex);
   bool blRet= true;
   m_bIsVehiclePbk = bIsVehiclePbk;

   if(!CHECK_VALID(devicehandle))
   {
      if(devicehandle != 0)   // CMG3G-7702
      {
         ETG_TRACE_ERR((" FC_PhoneBook_SQLite: device handle is invalid  "));
         return false;
      }
   }

   QSqlDatabase db= Database();

   //Fix for GMMY15-10814
   if(db.isOpen())
   {
      db.transaction();
   }
   else
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);
   QString sql;

   if (!InsertIntoMasterTable(db, devicehandle, contacthandle))
   {
      blRet = false;
   }
   else
   {
      if (((contacthandle & FC_PB_CONTACT_HANDLE_DPB_MARKER)
            == FC_PB_CONTACT_HANDLE_DPB_MARKER) || (devicehandle
            == FC_PB_VEHICLE_PBOOK))
      {
         sql
               = "INSERT INTO " + fc_pb_aTableNames[devicehandle] + " ( "

               FC_PB_COL_CONTACT_HANDLE", "
               FC_PB_COL_FIRST_NAME", "
               FC_PB_COL_LAST_NAME", "
               FC_PB_COL_PHONETICS", "   //292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
               FC_PB_COL_SORTFIRSTNAME ", "   // NEW
               FC_PB_COL_SORTLASTNAME  ", "   // NEW
               FC_PB_COL_POI_NAME", "
               FC_PB_COL_PHONENUM1", "
               FC_PB_COL_NUMTYPE1", "
               FC_PB_COL_NUMPRETTY1 ", "
               FC_PB_COL_NUMSUBSTR1 ", "   // NEW
               FC_PB_COL_PHONENUM2", "
               FC_PB_COL_NUMTYPE2", "
               FC_PB_COL_NUMPRETTY2 ", "
               FC_PB_COL_NUMSUBSTR2 ", "   // NEW
               FC_PB_COL_PHONENUM3", "
               FC_PB_COL_NUMTYPE3", "
               FC_PB_COL_NUMPRETTY3 ", "
               FC_PB_COL_NUMSUBSTR3 ", "   // NEW
               FC_PB_COL_PHONENUM4 ", "
               FC_PB_COL_NUMTYPE4", "
               FC_PB_COL_NUMPRETTY4 ", "
               FC_PB_COL_NUMSUBSTR4 ", "   // NEW
               FC_PB_COL_PHONENUM5 ", "
               FC_PB_COL_NUMTYPE5", "
               FC_PB_COL_NUMPRETTY5 ", "
               FC_PB_COL_NUMSUBSTR5 ", "   // NEW
               FC_PB_COL_PHONENUM6 ", "
               FC_PB_COL_NUMTYPE6", "
               FC_PB_COL_NUMPRETTY6 ", "
               FC_PB_COL_NUMSUBSTR6 ", "   // NEW
               FC_PB_COL_PHONENUM7 ", "
               FC_PB_COL_NUMTYPE7", "
               FC_PB_COL_NUMPRETTY7 ", "
               FC_PB_COL_NUMSUBSTR7 ", "   // NEW
               FC_PB_COL_PHONENUM8 ", "
               FC_PB_COL_NUMTYPE8", "
               FC_PB_COL_NUMPRETTY8 ", "
               FC_PB_COL_NUMSUBSTR8 ", "   // NEW
               FC_PB_COL_EMAIL_ADD1", "
               FC_PB_COL_EMAIL_ADD2", "
               FC_PB_COL_EMAIL_ADD3", "
               FC_PB_COL_HOME_ADD", "
               FC_PB_COL_WORK_ADD", "
               FC_PB_COL_OTHER_ADD", "
               FC_PB_COL_CATEGORY", "
               FC_PB_COL_CONTACT_TYPE ", "
               FC_PB_COL_NOTES", "
               FC_PB_COL_HGCLATLONGVALIDITY","
               FC_PB_COL_HGCLATITUDE","
               FC_PB_COL_HGCLONGITUDE","
               FC_PB_COL_WGCLATLONGVALIDITY","
               FC_PB_COL_WGCLATITUDE","
               FC_PB_COL_WGCLONGITUDE","
               FC_PB_COL_OGCLATLONGVALIDITY","
               FC_PB_COL_OGCLATITUDE","
               FC_PB_COL_OGCLONGITUDE","
               FC_PB_COL_PHOTO_DL_SIZE", "
               FC_PB_COL_PHOTO", "
               FC_PB_COL_PHOTO_TYPE", " // -FastPhotoHandling-
               FC_PB_COL_PHOTO_DL_CRC", " // -FastPhotoHandling-
               FC_PB_COL_PHOTO_SIZE", " // -FastPhotoHandling-
               FC_PB_COL_PHOTO_CRC", " // -FastPhotoHandling-
               FC_PB_COL_APL1_SIZE", " // -FastContactSearch-
               FC_PB_COL_CRC_APL1") " // -FastContactSearch-
             //"VALUES ( ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);";
               "VALUES ( ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?,?, ?,?,?,?,?,?,?,?);";
      }
      else
      {
         sql
               = "INSERT INTO " + fc_vpb_aTableNames[devicehandle] + " ( "
               FC_PB_COL_CONTACT_HANDLE", "
               FC_PB_COL_VPB_CONTACT_POSITION ", "
               FC_PB_COL_FIRST_NAME", "
               FC_PB_COL_LAST_NAME", "
               FC_PB_COL_PHONETICS", "   //292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
               FC_PB_COL_SORTFIRSTNAME ", "   // NEW
               FC_PB_COL_SORTLASTNAME  ", "   // NEW
               FC_PB_COL_POI_NAME", "
               FC_PB_COL_PHONENUM1", "
               FC_PB_COL_NUMTYPE1", "
               FC_PB_COL_NUMPRETTY1 ", "
               FC_PB_COL_NUMSUBSTR1 ", "   // NEW
               FC_PB_COL_PHONENUM2", "
               FC_PB_COL_NUMTYPE2", "
               FC_PB_COL_NUMPRETTY2 ", "
               FC_PB_COL_NUMSUBSTR2 ", "   // NEW
               FC_PB_COL_PHONENUM3", "
               FC_PB_COL_NUMTYPE3", "
               FC_PB_COL_NUMPRETTY3 ", "
               FC_PB_COL_NUMSUBSTR3 ", "   // NEW
               FC_PB_COL_PHONENUM4 ", "
               FC_PB_COL_NUMTYPE4", "
               FC_PB_COL_NUMPRETTY4 ", "
               FC_PB_COL_NUMSUBSTR4 ", "   // NEW
               FC_PB_COL_PHONENUM5 ", "
               FC_PB_COL_NUMTYPE5", "
               FC_PB_COL_NUMPRETTY5 ", "
               FC_PB_COL_NUMSUBSTR5 ", "   // NEW
               FC_PB_COL_PHONENUM6 ", "
               FC_PB_COL_NUMTYPE6", "
               FC_PB_COL_NUMPRETTY6 ", "
               FC_PB_COL_NUMSUBSTR6 ", "   // NEW
               FC_PB_COL_PHONENUM7 ", "
               FC_PB_COL_NUMTYPE7", "
               FC_PB_COL_NUMPRETTY7 ", "
               FC_PB_COL_NUMSUBSTR7 ", "   // NEW
               FC_PB_COL_PHONENUM8 ", "
               FC_PB_COL_NUMTYPE8", "
               FC_PB_COL_NUMPRETTY8 ", "
               FC_PB_COL_NUMSUBSTR8 ", "   // NEW
               FC_PB_COL_EMAIL_ADD1", "
               FC_PB_COL_EMAIL_ADD2", "
               FC_PB_COL_EMAIL_ADD3", "
               FC_PB_COL_HOME_ADD", "
               FC_PB_COL_WORK_ADD", "
               FC_PB_COL_OTHER_ADD", "
               FC_PB_COL_CATEGORY", "
               FC_PB_COL_CONTACT_TYPE ", "
               FC_PB_COL_NOTES", "
               FC_PB_COL_HGCLATLONGVALIDITY","
               FC_PB_COL_HGCLATITUDE","
               FC_PB_COL_HGCLONGITUDE","
               FC_PB_COL_WGCLATLONGVALIDITY","
               FC_PB_COL_WGCLATITUDE","
               FC_PB_COL_WGCLONGITUDE","
               FC_PB_COL_OGCLATLONGVALIDITY","
               FC_PB_COL_OGCLATITUDE","
               FC_PB_COL_OGCLONGITUDE","
               FC_PB_COL_PHOTO_DL_SIZE", "
               FC_PB_COL_PHOTO", "
               FC_PB_COL_PHOTO_TYPE", " // -FastPhotoHandling-
               FC_PB_COL_PHOTO_DL_CRC", " // -FastPhotoHandling-
               FC_PB_COL_PHOTO_SIZE", " // -FastPhotoHandling-
               FC_PB_COL_PHOTO_CRC", " // -FastPhotoHandling-
               FC_PB_COL_APL1_SIZE", " // -FastContactSearch-
               FC_PB_COL_CRC_APL1") " // -FastContactSearch-
             //"VALUES ( ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);";
               "VALUES ( ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?,?, ?,?,?,?,?,?,?,?);";
      }

      //Fix for GMMY15-10814
      if (db.isOpen())
      {
         query.prepare(sql);
      }
      else
      {
         ETG_TRACE_ERR(("Database is not open"));
         return false;
      }

//      FillEmptyStringToEmptyFields(contact);   // BAD

      //Fix for GMMY15-10814
      if (db.isOpen())
      {
         if (((contacthandle & FC_PB_CONTACT_HANDLE_DPB_MARKER)
               == FC_PB_CONTACT_HANDLE_DPB_MARKER) || (devicehandle
               == FC_PB_VEHICLE_PBOOK))
         {
            query.addBindValue(contacthandle);
            query.addBindValue(contact.firstName);
            query.addBindValue(contact.lastName);
            query.addBindValue(contact.Phonetics); //292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
            query.addBindValue( contact.SortFirstName);    // NEW
            query.addBindValue( contact.SortLastName);     // NEW
            query.addBindValue(contact.poiName);
            query.addBindValue(contact.PhoneNum1);
            query.addBindValue(contact.NumberType1);
            query.addBindValue(contact.PPPhoneNum1);
            query.addBindValue( contact.NumberSubstr1 );   // NEW
            query.addBindValue(contact.PhoneNum2);
            query.addBindValue(contact.NumberType2);
            query.addBindValue(contact.PPPhoneNum2);
            query.addBindValue( contact.NumberSubstr2 );   // NEW
            query.addBindValue(contact.PhoneNum3);
            query.addBindValue(contact.NumberType3);
            query.addBindValue(contact.PPPhoneNum3);
            query.addBindValue( contact.NumberSubstr3 );   // NEW
            query.addBindValue(contact.PhoneNum4);
            query.addBindValue(contact.NumberType4);
            query.addBindValue(contact.PPPhoneNum4);
            query.addBindValue( contact.NumberSubstr4 );   // NEW
            query.addBindValue(contact.PhoneNum5);
            query.addBindValue(contact.NumberType5);
            query.addBindValue(contact.PPPhoneNum5);
            query.addBindValue( contact.NumberSubstr5 );   // NEW
            query.addBindValue(contact.PhoneNum6);
            query.addBindValue(contact.NumberType6);
            query.addBindValue(contact.PPPhoneNum6);
            query.addBindValue( contact.NumberSubstr6 );   // NEW
            query.addBindValue(contact.PhoneNum7);
            query.addBindValue(contact.NumberType7);
            query.addBindValue(contact.PPPhoneNum7);
            query.addBindValue( contact.NumberSubstr7 );   // NEW
            query.addBindValue(contact.PhoneNum8);
            query.addBindValue(contact.NumberType8);
            query.addBindValue(contact.PPPhoneNum8);
            query.addBindValue( contact.NumberSubstr8 );   // NEW
            query.addBindValue(contact.emailAdd1);
            query.addBindValue(contact.emailAdd2);
            query.addBindValue(contact.emailAdd3);
            query.addBindValue(contact.homeAdd);
            query.addBindValue(contact.workAdd);
            query.addBindValue(contact.otherAdd);
            query.addBindValue(contact.category);
            query.addBindValue(contact.contactType);
            query.addBindValue(contact.notes);
            query.addBindValue(contact.homeGeoCodeLatLongValidity);
            query.addBindValue(contact.homeGeoCodeLatitude);
            query.addBindValue(contact.homeGeocodeLongitude);
            query.addBindValue(contact.workGeoCodeLatLongValidity);
            query.addBindValue(contact.workGeoCodeLatitude);
            query.addBindValue(contact.workGeocodeLongitude);
            query.addBindValue(contact.otherGeoCodeLatLongValidity);
            query.addBindValue(contact.otherGeoCodeLatitude);
            query.addBindValue(contact.otherGeocodeLongitude);
            query.addBindValue(contact.photo.size());
            query.addBindValue(contact.photo);
            query.addBindValue(contact.photoType); // -FastPhotoHandling-
            query.addBindValue(contact.photoDlCRC); // -FastPhotoHandling-
            query.addBindValue(contact.photoSize); // -FastPhotoHandling-
            query.addBindValue(contact.photoCRC); // -FastPhotoHandling-
            query.addBindValue(contact.APL1Size); // -FastContactSearch-
            query.addBindValue(contact.crcAPL1); // -FastContactSearch-
         }
         else
         {
            query.addBindValue(contacthandle);
            query.addBindValue(contact.u32contactPosition);
            query.addBindValue(contact.firstName);
            query.addBindValue(contact.lastName);
            query.addBindValue(contact.Phonetics); //292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
            query.addBindValue( contact.SortFirstName);    // NEW
            query.addBindValue( contact.SortLastName);     // NEW
            query.addBindValue(contact.poiName);
            query.addBindValue(contact.PhoneNum1);
            query.addBindValue(contact.NumberType1);
            query.addBindValue(contact.PPPhoneNum1);
            query.addBindValue(contact.NumberSubstr1);   // NEW
            query.addBindValue(contact.PhoneNum2);
            query.addBindValue(contact.NumberType2);
            query.addBindValue(contact.PPPhoneNum2);
            query.addBindValue(contact.NumberSubstr2);   // NEW
            query.addBindValue(contact.PhoneNum3);
            query.addBindValue(contact.NumberType3);
            query.addBindValue(contact.PPPhoneNum3);
            query.addBindValue(contact.NumberSubstr3);   // NEW
            query.addBindValue(contact.PhoneNum4);
            query.addBindValue(contact.NumberType4);
            query.addBindValue(contact.PPPhoneNum4);
            query.addBindValue(contact.NumberSubstr4);   // NEW
            query.addBindValue(contact.PhoneNum5);
            query.addBindValue(contact.NumberType5);
            query.addBindValue(contact.PPPhoneNum5);
            query.addBindValue(contact.NumberSubstr5);   // NEW
            query.addBindValue(contact.PhoneNum6);
            query.addBindValue(contact.NumberType6);
            query.addBindValue(contact.PPPhoneNum6);
            query.addBindValue(contact.NumberSubstr6);   // NEW
            query.addBindValue(contact.PhoneNum7);
            query.addBindValue(contact.NumberType7);
            query.addBindValue(contact.PPPhoneNum7);
            query.addBindValue(contact.NumberSubstr7);   // NEW
            query.addBindValue(contact.PhoneNum8);
            query.addBindValue(contact.NumberType8);
            query.addBindValue(contact.PPPhoneNum8);
            query.addBindValue(contact.NumberSubstr8);   // NEW
            query.addBindValue(contact.emailAdd1);
            query.addBindValue(contact.emailAdd2);
            query.addBindValue(contact.emailAdd3);
            query.addBindValue(contact.homeAdd);
            query.addBindValue(contact.workAdd);
            query.addBindValue(contact.otherAdd);
            query.addBindValue(contact.category);
            query.addBindValue(contact.contactType);
            query.addBindValue(contact.notes);
            query.addBindValue(contact.homeGeoCodeLatLongValidity);
            query.addBindValue(contact.homeGeoCodeLatitude);
            query.addBindValue(contact.homeGeocodeLongitude);
            query.addBindValue(contact.workGeoCodeLatLongValidity);
            query.addBindValue(contact.workGeoCodeLatitude);
            query.addBindValue(contact.workGeocodeLongitude);
            query.addBindValue(contact.otherGeoCodeLatLongValidity);
            query.addBindValue(contact.otherGeoCodeLatitude);
            query.addBindValue(contact.otherGeocodeLongitude);
            query.addBindValue(contact.photo.size());
            query.addBindValue(contact.photo);
            query.addBindValue(contact.photoType); // -FastPhotoHandling-
            query.addBindValue(contact.photoDlCRC); // -FastPhotoHandling-
            query.addBindValue(contact.photoSize); // -FastPhotoHandling-
            query.addBindValue(contact.photoCRC); // -FastPhotoHandling-
            query.addBindValue(contact.APL1Size); // -FastContactSearch-
            query.addBindValue(contact.crcAPL1); // -FastContactSearch-
         }

         if (!query.exec())
         {
            PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

            ETG_TRACE_ERR(("FC_PhoneBook_SQLite :- ERROR - SQL query failed in InsertSingleRecord "));
            blRet = false;
         }
      }
      else
      {
         ETG_TRACE_ERR(("Database is not open"));
         return false;
      }
   }

   //Fix for GMMY15-10814
   if(db.isOpen())
   {
      if(!blRet)
      {
         db.rollback(); //Rollback
         ETG_TRACE_USR4(("Database rolled back"));
      }
      else
      {
         if(!db.commit())
         {
            db.rollback();
            ETG_TRACE_USR4(("Database rolled back since commit failed"));
         }
         else
         {
            ETG_TRACE_USR4(("Database committed successfully"));
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   return blRet;
}


/*******************************************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetMatchingRecord()
*
* DESCRIPTION: Searches for a contact which matches a given 'searchkey', 'searchtype' (PhoneNumber or EmailID)
*              in the Contactlist for a given DeviceHandle (DHx tables, VPBx table or VPB)
* PARAMETER:
*
* RETURNVALUE: success or failure of search.
*
********************************************************************************************************/
bool FC_PhoneBook_SQLite::GetMatchingRecord(FC_PhoneBook_Search_Result& searchresult,
                                            const char* searchkey, SearchType searchtype, quint8 devicehandle)
{
   ETG_TRACE_USR4(("GetMatchingRecord entered"));
   QMutexLocker lock(&m_mutex);

   QString searchstring(searchkey);
   QString sql;
   QString sql_1;
   QString sql_2;
   QString sql_3;
   bool blContactFound = false;

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }
   QSqlQuery query(db);

   if(PhoneNumber == searchtype)
   {

      sql_1 = "SELECT ContactHandle, FirstName, LastName, "
                     "PhoneNumber1, NumberType1, PhoneNumber2, NumberType2, PhoneNumber3, NumberType3, PhoneNumber4, NumberType4, "
                     "PhoneNumber5, NumberType5, PhoneNumber6, NumberType6, PhoneNumber7, NumberType7, PhoneNumber8, NumberType8 ";

      if(devicehandle > FC_PB_TOTAL_NUMBER_OF_PB)
      {
         sql_2 = "FROM " + fc_vpb_aTableNames[devicehandle - FC_PB_TOTAL_NUMBER_OF_VPB];
      }
      else
      {
         sql_2 = "FROM "+ fc_pb_aTableNames[devicehandle];
      }

         sql_3 = " WHERE ( "   // NEW
           "(( " FC_PB_COL_NUMSUBSTR1 " = '" + searchstring + "' ) OR "
            "( " FC_PB_COL_NUMSUBSTR2 " = '" + searchstring + "' ) OR "
            "( " FC_PB_COL_NUMSUBSTR3 " = '" + searchstring + "' ) OR "
            "( " FC_PB_COL_NUMSUBSTR4 " = '" + searchstring + "' ) OR "
            "( " FC_PB_COL_NUMSUBSTR5 " = '" + searchstring + "' ) OR "
            "( " FC_PB_COL_NUMSUBSTR6 " = '" + searchstring + "' ) OR "
            "( " FC_PB_COL_NUMSUBSTR7 " = '" + searchstring + "' ) OR "
            "( " FC_PB_COL_NUMSUBSTR8 " = '" + searchstring + "' )) AND "
            "((" FC_PB_COL_FIRST_NAME " != '') OR (" FC_PB_COL_LAST_NAME " != '')) ) LIMIT 1";

      // --- -bn: The following traces should be commented out for release:
      ETG_TRACE_USR4(( "sql_1= '%s' ", sql_1.left(230).toLatin1().constData() ));
      ETG_TRACE_USR4(( "     = '%s' ", sql_1.mid(230,230).toLatin1().constData() ));
      ETG_TRACE_USR4(( "sql_2= '%s' ", sql_2.toLatin1().constData() ));
      ETG_TRACE_USR4(( "sql_3= '%s' ", sql_3.left(230).toLatin1().constData() ));
      ETG_TRACE_USR4(( "     = '%s' ", sql_3.mid(230,230).toLatin1().constData() ));

      // Construct whole query from the created parts
      sql = sql_1 + sql_2 + sql_3;

      if(query.exec(sql))
      {
         if(query.next())
         {
            searchresult.contactHandle = query.value(0).toUInt();   // query.value(0) ->contains contacthandle
            searchresult.firstname     = query.value(1).toString(); // query.value(1) ->contains firstname
            searchresult.lastname      = query.value(2).toString(); // query.value(2) ->contains lastname

            ETG_TRACE_USR4(( "GetMatchingRecord: ContactHandle= %d, LastName= '%s'", searchresult.contactHandle, searchresult.lastname.toLatin1().constData() ));

            quint8 u8Index = 3;  // On this value of u8Index, the PhoneNumber1 starts in the query result
            // We traverse through the PhoneNumberX / NumberTypeX, (X=1,.. PBDL_MAX_PHONE_NUMBERS_PER_CONTACT_COUNT) of the contact to find
            // the NumberType of that Number which matches to our 'searchstring'
            searchresult.contactdetailtype = 0;  // Initialization with invalid NumberType
            for( quint8 u8NumCount = 1; u8NumCount <= PBDL_MAX_PHONE_NUMBERS_PER_CONTACT_COUNT; u8NumCount++)
            {
               // Note: PBDL_MAX_PHONE_NUMBERS_PER_CONTACT_COUNT is project specific, e.g. in RN-AIVI it has a value of 5   // NEW  Comment changed.

               ETG_TRACE_USR4((" FindNumberTypeForTheNumber:PhoneNumber Index= %u ", u8Index));
               QString sNumber = query.value(u8Index).toString();

               if( (searchstring.size() >= FC_PHONEBOOK_LIST_SEARCH_NUM_LEN )?
                  sNumber.contains(searchstring):(searchstring == sNumber))
               {
                  u8Index++;

                  quint8 u8NumberType = (query.value(u8Index)).value<quint8>();
                  ETG_TRACE_USR4(( " Found PhoneNumberX, X= %u, NumberType= %u, N ", u8NumCount, u8NumberType ));
                  searchresult.contactdetailtype = u8NumberType;
                  break;
               }
               //u8Index += 2u;
               u8Index = (quint8)(u8Index + 2);   // TODO: Original code was: u8Index += 2; Sushmitha got a compiler warning in Gen-4 build. We need to check how to get rid of this in a readable way.
            }
            blContactFound= true;
         }
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime
      }
   }
   else if(EmailID == searchtype)
   {
      if(devicehandle > FC_PB_TOTAL_NUMBER_OF_PB)
      {
            sql = "SELECT ContactHandle, FirstName, LastName, EmailAdd1, EmailAdd2, EmailAdd3  FROM " + fc_vpb_aTableNames[devicehandle - FC_PB_TOTAL_NUMBER_OF_VPB] + " WHERE ("; //NCG3D-139336
      }
      else
      {
            sql = "SELECT ContactHandle, FirstName, LastName, EmailAdd1, EmailAdd2, EmailAdd3  FROM " + fc_pb_aTableNames[devicehandle] + " WHERE ("; //NCG3D-139336
      }

      sql +=
         "( ( " FC_PB_COL_EMAIL_ADD1 " LIKE '" +searchstring+"' ) OR "
           "( " FC_PB_COL_EMAIL_ADD2 " LIKE '" +searchstring+"' ) OR "
           "( " FC_PB_COL_EMAIL_ADD3 " LIKE '" +searchstring+"' ) )";

      ETG_TRACE_USR2(("  SQL string for searchNumber = %s  ", (sql.toLatin1()).constData()));

      if ( query.exec(sql) )
      {
         if(query.next())
         {
            searchresult.contactHandle = query.value(0).toUInt();   // query.value(0) ->contains contacthandle
            searchresult.firstname     = query.value(1).toString(); // query.value(1) ->contains firstname
            searchresult.lastname      = query.value(2).toString(); // query.value(2) ->contains lastname

            ETG_TRACE_USR4(( "FC_PhoneBook_SQLite :- Entry for the requested EmailID found"));

            searchresult.contactdetailtype = 0;  // Initialization with invalid NumberType
            for(quint8 u8Index= 0; u8Index < FC_PB_TOTAL_EMAILID; u8Index++ )
            {
               QString emailaddress = query.value(3+u8Index).toString(); //EmailAdd1 starts at index 3 in the above SELECT query
               if(searchstring.contains(emailaddress))
               {
                  quint8 u8EmailIDType = static_cast<quint8>(FC_PhoneBook_SQLite::EmailAdd1 + u8Index);
                  searchresult.contactdetailtype = u8EmailIDType;
                  break;
               }
            }
            blContactFound = true;
         }
      }
   }

   return blContactFound;
}


/*****************************************************************************************
*
* FUNCTION:  FC_PhoneBook_SQLite::GetSqlString
*
* DESCRIPTION: Forms a query string for the call history
*
*
*
* PARAMETER:

* RETURNVALUE:
*
*******************************************************************************************/
QString FC_PhoneBook_SQLite::GetSqlString( const SortType& enSorttype, const quint8& u8CallType,
                                          const quint8& u8TableIndex, bool bDeviceDateTimeSupport )
{
   ETG_TRACE_USR4(("GetSqlString entered"));

   QString sql="";

#ifdef LIMIT_CCH_ENTRIES   // CMG3G-6311 IS1929_PB entry concatenation@FC_Phonebook
   if (callhisttype_combined == u8CallType) //Combined Call history
   {
      sql += "SELECT * FROM (";
   }
#endif

#ifdef REQUESTSLICE_EXTENDED_CH
   sql += "SELECT " FC_PB_COL_CH_CONTACT_HANDLE "," FC_PB_COL_CH_CALL_TYPE "," FC_PB_COL_CH_FIRST_NAME "," FC_PB_COL_CH_LAST_NAME "," FC_PB_COL_CH_PP_PHONE_NUM "," FC_PB_COL_CH_PHONE_NUM

      "," FC_PB_COL_CH_NUMBER_TYPE "," FC_PB_COL_CH_CALL_DATE
      ",MAX(" FC_PB_COL_CH_CALL_TIME ") AS " FC_PB_COL_CH_CALL_TIME
      ",COUNT(*) AS FC_PB_CH_CALL_COUNT FROM "+fc_pb_aTableNames[MAP_TO_CH_INDEX(u8TableIndex)];
#else
   sql += "SELECT * FROM " + fc_pb_aTableNames[MAP_TO_CH_INDEX(u8TableIndex)];
#endif

   if(callhisttype_combined != u8CallType)//Separate Call history
   {
#ifdef REQUESTSLICE_EXTENDED_CH
      sql+= " WHERE " FC_PB_COL_CH_CALL_TYPE " = "+QString::number(u8CallType)+
         " GROUP BY " FC_PB_COL_CH_PHONE_NUM "," FC_PB_COL_CH_CALL_DATE
         " ORDER BY ";
#else
      sql+= " WHERE " FC_PB_COL_CH_CALL_TYPE " = " + QString::number(u8CallType) + " ORDER BY ";
#endif
   }
   else //Combined Call history
   {
#ifdef REQUESTSLICE_EXTENDED_CH
      sql+= " GROUP BY " FC_PB_COL_CH_PHONE_NUM "," FC_PB_COL_CH_CALL_DATE "," FC_PB_COL_CH_CALL_TYPE " ORDER BY ";
#else
      sql+= " ORDER BY ";
#endif
   }

   switch ( enSorttype )
   {
   default:
   case DateTime:
      if( (callhisttype_combined != u8CallType) && (FALSE == bDeviceDateTimeSupport) )
      {  // Separate Call History Lists and NO DateTimeSupport (i.e. a "few" vCards don't have Date/Time information)
         sql += " rowid";
         break;
      }
      else
      {
         sql += FC_PB_COL_CH_CALL_DATE" COLLATE " SORTLIB_COLLATE_NAME " DESC ," FC_PB_COL_CH_CALL_TIME " COLLATE " SORTLIB_COLLATE_NAME " DESC , rowid";
         break;
      }


   case LastName:
      sql += " CASE WHEN " FC_PB_COL_LAST_NAME " = '' THEN " FC_PB_COL_FIRST_NAME " COLLATE " SORTLIB_COLLATE_NAME
         " ELSE " FC_PB_COL_LAST_NAME " COLLATE " SORTLIB_COLLATE_NAME " END ," FC_PB_COL_FIRST_NAME" COLLATE " SORTLIB_COLLATE_NAME " , "
         FC_PB_COL_CH_CALL_DATE" COLLATE " SORTLIB_COLLATE_NAME " DESC ," FC_PB_COL_CH_CALL_TIME" COLLATE " SORTLIB_COLLATE_NAME " DESC , rowid";
      break;

   case FirstName:
      sql +=  " CASE WHEN " FC_PB_COL_FIRST_NAME " = '' THEN " FC_PB_COL_LAST_NAME " COLLATE " SORTLIB_COLLATE_NAME
         " ELSE " FC_PB_COL_FIRST_NAME " COLLATE " SORTLIB_COLLATE_NAME " END ," FC_PB_COL_LAST_NAME" COLLATE " SORTLIB_COLLATE_NAME " , "
         FC_PB_COL_CH_CALL_DATE" COLLATE " SORTLIB_COLLATE_NAME " DESC ," FC_PB_COL_CH_CALL_TIME" COLLATE " SORTLIB_COLLATE_NAME " DESC , rowid";
      break;
   }

#ifdef LIMIT_CCH_ENTRIES   // CMG3G-6311 IS1929_PB entry concatenation@FC_Phonebook
   if(callhisttype_combined == u8CallType)   //Combined Call history
   {
      sql += ") LIMIT 30";
   }
#endif

   ETG_TRACE_USR2((" FC_PhoneBook_SQLite: String framed in GetSqlStringForCallHistory is %s", sql.toLatin1().constData()));

   return sql;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_SQLite::InsertIntoMasterTable
*
* DESCRIPTION: Creates a contact handle for the contact and maps it to the device table
*
*
*
* PARAMETER:

* RETURNVALUE: Result whether success or failure
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::InsertIntoMasterTable(QSqlDatabase& db,
                                                const quint8 TableIndex, quint32& ContactHandle)
{
   ETG_TRACE_USR4(("Entering function- FC_PhoneBook_SQLite::InsertIntoMasterTable"));

   bool bRetValue = true;

   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      bRetValue = false;
   }
   else
   {
      if(ContactHandle == OSAL_NULL)
      {
         ETG_TRACE_USR4(("Generating New contact handle"));
         ContactHandle= u32LastContactHandleUsed;
         if(++ContactHandle > FC_PB_CONTACT_HANDLE_LIMIT)
         {
            bRetValue = GetNewUniqueContactHandle(ContactHandle);
         }
      }

      if(bRetValue)
      {
         //FIX GMMY15-5669 Synchronisation problem between contact addresses in phonebook, HMI and navigation component.
         //The 24th bit of the ContactHandle shall be reserved to carry the following information.
         //If the Contact is from the Vehicle Contact list (24th bit = 0) or from any BT Phone Contact list (24th bit = 1).

         if(TableIndex <= FC_PB_DEVICE_HANDLE11) //CMG3G-7702 Including TableIndex = 0.
         {
            u32LastContactHandleUsed= ContactHandle; //Only contact handles used for phonebook
            MarkContactHandle(TableIndex,ContactHandle);
         }

         if (TableIndex != FC_PB_DEVICE_TMP)
         {
            bRetValue = InsertDataIntoMasterTable(db, TableIndex, ContactHandle);   // CMG3G-7702
         }
      }
      else
      {
         ETG_TRACE_ERR(("Failed to generate new unique contact handle"));
         bRetValue = false;
      }
   }

   ETG_TRACE_USR4(("Exiting function- FC_PhoneBook_SQLite::InsertIntoMasterTable"));
   return bRetValue;
}


//CMG3G-7702
/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_SQLite::InsertDataIntoMasterTable
*
* DESCRIPTION: Inserts the Contact Handle and the corresponding Device Handle into the MasterTable
*
*
*
* PARAMETER:

* RETURNVALUE: Result whether success or failure
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::InsertDataIntoMasterTable(QSqlDatabase& db,
                                                const quint8 TableIndex, quint32& ContactHandle)
{
   bool bRetValue = false;
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      ETG_TRACE_USR2(("InsertDataIntoMasterTable:: Contact Handle to be inserted -%u", ContactHandle));
      //NCG3D-101744 -Unique constraint failed, Unable to fetch row, Parameter count mismatch error
      QSqlQuery query(db);

      if(query.prepare("INSERT INTO " FC_PB_MASTER_TABLE_NAME " (" FC_PB_MASTER_COL_CONTACT_HANDLE "," FC_PB_MASTER_COL_TABLE_INDEX ") VALUES (?,?)"))
      {
         ETG_TRACE_USR2(("FC_PhoneBook_SQLite::InsertDataIntoMasterTable:- Query is succesfully prepared"));
         query.addBindValue(ContactHandle);
         query.addBindValue(TableIndex);

         if(!query.exec())
         {
            ETG_TRACE_ERR(("FC_PhoneBook_SQLite::InsertDataIntoMasterTable:- Query execution failed"));
            PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime
            bRetValue= false;
         }
         else
         {
            ETG_TRACE_USR2(("FC_PhoneBook_SQLite::InsertDataIntoMasterTable:- Query is successfully executed"));
            bRetValue = true;
         }
      }
      else
      {
         ETG_TRACE_ERR(("FC_PhoneBook_SQLite::InsertDataIntoMasterTable:- Query prepare failed"));
         bRetValue= false;
      }

	  //end of fix NCG3D-101744
   }
   return bRetValue;
}
//End of CMG3G-7702


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::UpdateRecord(   -FastContactSearch- TODO: Check if this method is called for Vehicle Contacts only.
*                     Note: -bn: Also to be checked: If CHECK_VALID_CALL_HIST(tableindex) is true the Vehicle Contacts are updated.
*                           I wonder if this makes sense, maybe this specific code part is never entered?
* DESCRIPTION:
*              Updates a single record in database
*
*
* PARAMETER:  [IN]  - contact handle to be edited and data
*             [OUT] -
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::UpdateRecord(FC_PhoneBook_Contact_Detail& contact, quint32 contacthandle)//CMG3G_14313-FC_PhoneBook_Contact_Detail-Pass by Reference-Coverity Fix
{
   ETG_TRACE_USR2(( " UpdateRecord entered with ContactHandle= %d ", contacthandle ));

   QMutexLocker lock(&m_mutex);

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(( " Database is not open " ));
      return false;
   }
   QSqlQuery query(db);

   QString sql = "SELECT " FC_PB_MASTER_COL_TABLE_INDEX " FROM "
      FC_PB_MASTER_TABLE_NAME " WHERE " FC_PB_MASTER_COL_CONTACT_HANDLE " = "+QString::number(contacthandle);


   ETG_TRACE_USR2(( " FC_PhoneBook_SQLite: String used for master table in update record - %s ",
                    sql.toLatin1().constData() ));

   if(!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(( " FC_PhoneBook_SQLite: Update record failed in master table " ));
      return false;
   }
  //CMG3G_14313-Coverity Fix
   quint8 tableindex =0;
   if(query.next())
   {
      tableindex= (query.value(0)).value<quint8>();
      ETG_TRACE_USR2(( " FC_PHoneBook_SQLite: Update table with DeviceHandle= %d", tableindex ));
   }
   //end of Fix

   if( !(CHECK_VALID(tableindex)) )
   {
      ETG_TRACE_ERR(( " FC_PhoneBook_SQLite: DeviceHandle is invalid " ));

      return false;
   }
   else
   {
      sql.clear();
      if (tableindex == FC_PB_VEHICLE_PBOOK )
      {
      sql= "UPDATE " + fc_pb_aTableNames[FC_PB_VEHICLE_PBOOK] + " SET "
         FC_PB_COL_FIRST_NAME" = ?" ","
         FC_PB_COL_LAST_NAME" = ?" ","
         FC_PB_COL_PHONETICS" = ?" ","   //292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
         FC_PB_COL_SORTFIRSTNAME" = ?" ","   // NEW
         FC_PB_COL_SORTLASTNAME" = ?" ","    // NEW
         FC_PB_COL_POI_NAME" = ?" ","
         FC_PB_COL_PHONENUM1" = ?" ","
         FC_PB_COL_PHONENUM2" = ?" ","
         FC_PB_COL_PHONENUM3" = ?" ","
         FC_PB_COL_PHONENUM4" = ?" ","
         FC_PB_COL_PHONENUM5" = ?" ","
         FC_PB_COL_PHONENUM6" = ?" ","
         FC_PB_COL_PHONENUM7" = ?" ","
         FC_PB_COL_PHONENUM8" = ?" ","
         FC_PB_COL_NUMPRETTY1" = ?" ","
         FC_PB_COL_NUMPRETTY2" = ?" ","
         FC_PB_COL_NUMPRETTY3" = ?" ","
         FC_PB_COL_NUMPRETTY4" = ?" ","
         FC_PB_COL_NUMPRETTY5" = ?" ","
         FC_PB_COL_NUMPRETTY6" = ?" ","
         FC_PB_COL_NUMPRETTY7" = ?" ","
         FC_PB_COL_NUMPRETTY8" = ?" ","
         FC_PB_COL_NUMSUBSTR1" = ?" ","   // NEW
         FC_PB_COL_NUMSUBSTR2" = ?" ","   // NEW
         FC_PB_COL_NUMSUBSTR3" = ?" ","   // NEW
         FC_PB_COL_NUMSUBSTR4" = ?" ","   // NEW
         FC_PB_COL_NUMSUBSTR5" = ?" ","   // NEW
         FC_PB_COL_NUMSUBSTR6" = ?" ","   // NEW
         FC_PB_COL_NUMSUBSTR7" = ?" ","   // NEW
         FC_PB_COL_NUMSUBSTR8" = ?" ","   // NEW
         FC_PB_COL_NUMTYPE1" = ?" ","
         FC_PB_COL_NUMTYPE2" = ?" ","
         FC_PB_COL_NUMTYPE3" = ?" ","
         FC_PB_COL_NUMTYPE4" = ?" ","
         FC_PB_COL_NUMTYPE5" = ?" ","
         FC_PB_COL_NUMTYPE6" = ?" ","
         FC_PB_COL_NUMTYPE7" = ?" ","
         FC_PB_COL_NUMTYPE8" = ?" ","
         FC_PB_COL_EMAIL_ADD1" = ?" ","
         FC_PB_COL_EMAIL_ADD2" = ?" ","
         FC_PB_COL_EMAIL_ADD3" = ?" ","
         FC_PB_COL_HOME_ADD" = ? " ","
         FC_PB_COL_WORK_ADD" = ?" ","
         FC_PB_COL_OTHER_ADD" = ?" ","
         FC_PB_COL_CATEGORY" = ?" ","
         FC_PB_COL_NOTES" = ?" ","
         FC_PB_COL_HGCLATLONGVALIDITY" = ?" ","
         FC_PB_COL_HGCLATITUDE" = ?" ","
         FC_PB_COL_HGCLONGITUDE" = ?" ","
         FC_PB_COL_WGCLATLONGVALIDITY" = ?" ","
         FC_PB_COL_WGCLATITUDE" = ?" ","
         FC_PB_COL_WGCLONGITUDE" = ?" ","
         FC_PB_COL_OGCLATLONGVALIDITY" = ?" ","
         FC_PB_COL_OGCLATITUDE" = ?" ","
         FC_PB_COL_OGCLONGITUDE" = ?"
         " WHERE " FC_PB_COL_CONTACT_HANDLE " = " + QString::number(contacthandle);
      }
      else
      {   // NEW -bn: 20.07.2019: TODO: ISSUE: The following query is nearly the same as above. It just works on a different table 'fc_vpb_aTableNames'.
          //                                    This differenc can be coded more compact. And just note that the 2 queries use the same binding below. This a second hint, that the queries can be compacted.
         sql= "UPDATE " + fc_vpb_aTableNames[tableindex] + " SET "
            FC_PB_COL_FIRST_NAME" = ?" ","
            FC_PB_COL_LAST_NAME" = ?" ","
            FC_PB_COL_PHONETICS" = ?" ","   //292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
            FC_PB_COL_SORTFIRSTNAME" = ?" ","   // NEW
            FC_PB_COL_SORTLASTNAME" = ?" ","    // NEW
            FC_PB_COL_POI_NAME" = ?" ","
            FC_PB_COL_PHONENUM1" = ?" ","
            FC_PB_COL_PHONENUM2" = ?" ","
            FC_PB_COL_PHONENUM3" = ?" ","
            FC_PB_COL_PHONENUM4" = ?" ","
            FC_PB_COL_PHONENUM5" = ?" ","
            FC_PB_COL_PHONENUM6" = ?" ","
            FC_PB_COL_PHONENUM7" = ?" ","
            FC_PB_COL_PHONENUM8" = ?" ","
            FC_PB_COL_NUMPRETTY1" = ?" ","
            FC_PB_COL_NUMPRETTY2" = ?" ","
            FC_PB_COL_NUMPRETTY3" = ?" ","
            FC_PB_COL_NUMPRETTY4" = ?" ","
            FC_PB_COL_NUMPRETTY5" = ?" ","
            FC_PB_COL_NUMPRETTY6" = ?" ","
            FC_PB_COL_NUMPRETTY7" = ?" ","
            FC_PB_COL_NUMPRETTY8" = ?" ","
            FC_PB_COL_NUMSUBSTR1" = ?" ","   // NEW
            FC_PB_COL_NUMSUBSTR2" = ?" ","   // NEW
            FC_PB_COL_NUMSUBSTR3" = ?" ","   // NEW
            FC_PB_COL_NUMSUBSTR4" = ?" ","   // NEW
            FC_PB_COL_NUMSUBSTR5" = ?" ","   // NEW
            FC_PB_COL_NUMSUBSTR6" = ?" ","   // NEW
            FC_PB_COL_NUMSUBSTR7" = ?" ","   // NEW
            FC_PB_COL_NUMSUBSTR8" = ?" ","   // NEW
            FC_PB_COL_NUMTYPE1" = ?" ","
            FC_PB_COL_NUMTYPE2" = ?" ","
            FC_PB_COL_NUMTYPE3" = ?" ","
            FC_PB_COL_NUMTYPE4" = ?" ","
            FC_PB_COL_NUMTYPE5" = ?" ","
            FC_PB_COL_NUMTYPE6" = ?" ","
            FC_PB_COL_NUMTYPE7" = ?" ","
            FC_PB_COL_NUMTYPE8" = ?" ","
            FC_PB_COL_EMAIL_ADD1" = ?" ","
            FC_PB_COL_EMAIL_ADD2" = ?" ","
            FC_PB_COL_EMAIL_ADD3" = ?" ","
            FC_PB_COL_HOME_ADD" = ? " ","
            FC_PB_COL_WORK_ADD" = ?" ","
            FC_PB_COL_OTHER_ADD" = ?" ","
            FC_PB_COL_CATEGORY" = ?" ","
            FC_PB_COL_NOTES" = ?" ","
            FC_PB_COL_HGCLATLONGVALIDITY" = ?" ","
            FC_PB_COL_HGCLATITUDE" = ?" ","
            FC_PB_COL_HGCLONGITUDE" = ?" ","
            FC_PB_COL_WGCLATLONGVALIDITY" = ?" ","
            FC_PB_COL_WGCLATITUDE" = ?" ","
            FC_PB_COL_WGCLONGITUDE" = ?" ","
            FC_PB_COL_OGCLATLONGVALIDITY" = ?" ","
            FC_PB_COL_OGCLATITUDE" = ?" ","
            FC_PB_COL_OGCLONGITUDE" = ?"
            " WHERE " FC_PB_COL_CONTACT_HANDLE " = " + QString::number(contacthandle);
      }
      query.prepare(sql);

//      FillEmptyStringToEmptyFields(contact);   // BAD

      query.addBindValue(contact.firstName);
      query.addBindValue(contact.lastName);
      query.addBindValue(contact.Phonetics); //292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
      query.addBindValue(contact.SortFirstName);   // NEW
      query.addBindValue(contact.SortLastName);    // NEW
      query.addBindValue(contact.poiName);
      query.addBindValue(contact.PhoneNum1);
      query.addBindValue(contact.PhoneNum2);
      query.addBindValue(contact.PhoneNum3);
      query.addBindValue(contact.PhoneNum4);
      query.addBindValue(contact.PhoneNum5);
      query.addBindValue(contact.PhoneNum6);
      query.addBindValue(contact.PhoneNum7);
      query.addBindValue(contact.PhoneNum8);
      query.addBindValue(contact.PPPhoneNum1);
      query.addBindValue(contact.PPPhoneNum2);
      query.addBindValue(contact.PPPhoneNum3);
      query.addBindValue(contact.PPPhoneNum4);
      query.addBindValue(contact.PPPhoneNum5);
      query.addBindValue(contact.PPPhoneNum6);
      query.addBindValue(contact.PPPhoneNum7);
      query.addBindValue(contact.PPPhoneNum8);
      query.addBindValue(contact.NumberSubstr1);   // NEW
      query.addBindValue(contact.NumberSubstr2);   // NEW
      query.addBindValue(contact.NumberSubstr3);   // NEW
      query.addBindValue(contact.NumberSubstr4);   // NEW
      query.addBindValue(contact.NumberSubstr5);   // NEW
      query.addBindValue(contact.NumberSubstr6);   // NEW
      query.addBindValue(contact.NumberSubstr7);   // NEW
      query.addBindValue(contact.NumberSubstr8);   // NEW
      query.addBindValue(contact.NumberType1);
      query.addBindValue(contact.NumberType2);
      query.addBindValue(contact.NumberType3);
      query.addBindValue(contact.NumberType4);
      query.addBindValue(contact.NumberType5);
      query.addBindValue(contact.NumberType6);
      query.addBindValue(contact.NumberType7);
      query.addBindValue(contact.NumberType8);
      query.addBindValue(contact.emailAdd1);
      query.addBindValue(contact.emailAdd2);
      query.addBindValue(contact.emailAdd3);
      query.addBindValue(contact.homeAdd);
      query.addBindValue(contact.workAdd);
      query.addBindValue(contact.otherAdd);
      query.addBindValue(contact.category);
      query.addBindValue(contact.notes);
      query.addBindValue( contact.homeGeoCodeLatLongValidity );
      query.addBindValue( contact.homeGeoCodeLatitude );
      query.addBindValue( contact.homeGeocodeLongitude );
      query.addBindValue( contact.workGeoCodeLatLongValidity );
      query.addBindValue( contact.workGeoCodeLatitude );
      query.addBindValue( contact.workGeocodeLongitude );
      query.addBindValue( contact.otherGeoCodeLatLongValidity );
      query.addBindValue( contact.otherGeoCodeLatitude );
      query.addBindValue( contact.otherGeocodeLongitude );


      ETG_TRACE_USR2((" FC_PhoneBook_SQLite: String used for master table in update record - %s",sql.toLatin1().constData()));
      if (!query.exec())
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Update record failed in device table"));
         return false;
      }
   }

   ETG_TRACE_USR4((" FC_PhoneBook_SQLite: Record updated successfully"));
   return true;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::UpdateContactRecord(   -FastContactSearch- TODO: Check if this is used only for Phone Contacts
*
* DESCRIPTION:
*              Updates a single contact in database, which stems from a
*              Phone Device (i.e. not from Vehicle Contacts)
*
*
* PARAMETER:
*          [IN]  - contact (new contact details)
*          [IN]  - contacthandle
*          [IN]  - bIsPhotoDiscarded (flag, which indicates if the photo shall be
*                                     discarded or updated also)
*           Note: -bn: if 'bIsPhotoDiscarded' is true, an existing photo gets deleted.
*                 (I am unsure, if this behaviour was or is really intended.)
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
/*  TODO: -bn: I have not found where this method is called (on Connectivity stabi branch) and therefore commented it out. Is it needed on feature branch?
bool FC_PhoneBook_SQLite::UpdateContactRecord(FC_PhoneBook_Contact_Detail& contact, quint32 contacthandle, bool bIsPhotoDiscarded)//CMG3G-14313-FC_PhoneBook_Contact_Detail-Pass by Reference-Coverity Fix
{
   ETG_TRACE_USR4(("UpdateContactRecord entered"));
   bool bRetValue = false;
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);

      QString sql = "SELECT " FC_PB_MASTER_COL_TABLE_INDEX " FROM "
         FC_PB_MASTER_TABLE_NAME " WHERE " FC_PB_MASTER_COL_CONTACT_HANDLE " = "+QString::number(contacthandle);


      ETG_TRACE_USR2((" FC_PhoneBook_SQLite: String used for master table in update record - %s ",
         sql.toLatin1().constData()));

      if(!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Update record failed in master table"));
      }
      else
      {
         //CMG3G-14313-Coverity fix
         quint8 tableindex =0;
         if(query.next())
         {
             tableindex= (query.value(0)).value<quint8>();
             ETG_TRACE_USR2(("FC_PHoneBook_SQLite: Update table with Device handle %d", tableindex));
         }
         //end of fix
         ETG_TRACE_USR4(("updating record"));
         sql.clear();
         sql= "UPDATE " + fc_pb_aTableNames[tableindex] + " SET "   // NEW -bn: 20.07.2019: TODO: Check the following: I have doubts on the following query as it uses only a subset of member variables of 'FC_PhoneBook_Contact_Detail'
            FC_PB_COL_FIRST_NAME" = ?" ","
            FC_PB_COL_LAST_NAME" = ?" ","
            FC_PB_COL_PHONETICS" = ?" ","   //292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
            FC_PB_COL_SORTFIRSTNAME" = ?" ","   // NEW
            FC_PB_COL_SORTLASTNAME" = ?" ","    // NEW
            FC_PB_COL_POI_NAME" = ?" ","
            FC_PB_COL_PHONENUM1" = ?" ","
            FC_PB_COL_PHONENUM2" = ?" ","
            FC_PB_COL_PHONENUM3" = ?" ","
            FC_PB_COL_PHONENUM4" = ?" ","
            FC_PB_COL_PHONENUM5" = ?" ","
            FC_PB_COL_PHONENUM6" = ?" ","
            FC_PB_COL_PHONENUM7" = ?" ","
            FC_PB_COL_PHONENUM8" = ?" ","
            FC_PB_COL_NUMSUBSTR1" = ?" ","   // NEW
            FC_PB_COL_NUMSUBSTR2" = ?" ","   // NEW
            FC_PB_COL_NUMSUBSTR3" = ?" ","   // NEW
            FC_PB_COL_NUMSUBSTR4" = ?" ","   // NEW
            FC_PB_COL_NUMSUBSTR5" = ?" ","   // NEW
            FC_PB_COL_NUMSUBSTR6" = ?" ","   // NEW
            FC_PB_COL_NUMSUBSTR7" = ?" ","   // NEW
            FC_PB_COL_NUMSUBSTR8" = ?" ","   // NEW
            FC_PB_COL_EMAIL_ADD1" = ?" ","
            FC_PB_COL_EMAIL_ADD2" = ?" ","
            FC_PB_COL_EMAIL_ADD3" = ?" ","
            FC_PB_COL_HOME_ADD" = ? " ","
            FC_PB_COL_WORK_ADD" = ?" ","
            FC_PB_COL_OTHER_ADD" = ?" ","
            FC_PB_COL_CATEGORY" = ?" ","
            FC_PB_COL_NOTES" = ?" ","
            FC_PB_COL_HGCLATLONGVALIDITY" = ?" ","
            FC_PB_COL_HGCLATITUDE" = ?" ","
            FC_PB_COL_HGCLONGITUDE" = ?" ","
            FC_PB_COL_WGCLATLONGVALIDITY" = ?" ","
            FC_PB_COL_WGCLATITUDE" = ?" ","
            FC_PB_COL_WGCLONGITUDE" = ?" ","
            FC_PB_COL_OGCLATLONGVALIDITY" = ?" ","
            FC_PB_COL_OGCLATITUDE" = ?" ","
            FC_PB_COL_OGCLONGITUDE" = ?" ","
            FC_PB_COL_PHOTO_DL_SIZE" = ?" ","
            FC_PB_COL_PHOTO" = ?" ","
            FC_PB_COL_PHOTO_TYPE" = ?" ","     // -FastPhotoHandling-
            FC_PB_COL_PHOTO_DL_CRC" = ?" ","   // -FastPhotoHandling-
            FC_PB_COL_PHOTO_SIZE" = ?" ","     // -FastPhotoHandling-
            FC_PB_COL_PHOTO_CRC" = ?" ","      // -FastPhotoHandling-
            FC_PB_COL_APL1_SIZE" = ?" ","  // -FastContactSearch-
            FC_PB_COL_CRC_APL1" = ?"       // -FastContactSearch-
            " WHERE " FC_PB_COL_CONTACT_HANDLE " = " + QString::number(contacthandle);

         query.prepare(sql);

//         FillEmptyStringToEmptyFields(contact);   // BAD

         if(bIsPhotoDiscarded == true)
         {
            contact.photo     = "";   //.clear();
            contact.photoSize = 0;
            contact.photoType = "";
            contact.photoCRC  = 0;
         }

         query.addBindValue(contact.firstName);
         query.addBindValue(contact.lastName);
         query.addBindValue(contact.Phonetics); //292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
         query.addBindValue(contact.SortFirstName);   // NEW
         query.addBindValue(contact.SortLastName);    // NEW
         query.addBindValue(contact.poiName);
         query.addBindValue(contact.PhoneNum1);
         query.addBindValue(contact.PhoneNum2);
         query.addBindValue(contact.PhoneNum3);
         query.addBindValue(contact.PhoneNum4);
         query.addBindValue(contact.PhoneNum5);
         query.addBindValue(contact.PhoneNum6);
         query.addBindValue(contact.PhoneNum7);
         query.addBindValue(contact.PhoneNum8);
         query.addBindValue(contact.NumberSubstr1);   // NEW
         query.addBindValue(contact.NumberSubstr2);   // NEW
         query.addBindValue(contact.NumberSubstr3);   // NEW
         query.addBindValue(contact.NumberSubstr4);   // NEW
         query.addBindValue(contact.NumberSubstr5);   // NEW
         query.addBindValue(contact.NumberSubstr6);   // NEW
         query.addBindValue(contact.NumberSubstr7);   // NEW
         query.addBindValue(contact.NumberSubstr8);   // NEW
         query.addBindValue(contact.emailAdd1);
         query.addBindValue(contact.emailAdd2);
         query.addBindValue(contact.emailAdd3);
         query.addBindValue(contact.homeAdd);
         query.addBindValue(contact.workAdd);
         query.addBindValue(contact.otherAdd);
         query.addBindValue(contact.category);
         query.addBindValue(contact.notes);
         query.addBindValue( contact.homeGeoCodeLatLongValidity );
         query.addBindValue( contact.homeGeoCodeLatitude );
         query.addBindValue( contact.homeGeocodeLongitude );
         query.addBindValue( contact.workGeoCodeLatLongValidity );
         query.addBindValue( contact.workGeoCodeLatitude );
         query.addBindValue( contact.workGeocodeLongitude );
         query.addBindValue( contact.otherGeoCodeLatLongValidity );
         query.addBindValue( contact.otherGeoCodeLatitude );
         query.addBindValue( contact.otherGeocodeLongitude );
         query.addBindValue( contact.photoDlSize);
         query.addBindValue( contact.photo);
         query.addBindValue( contact.photoType);    // -FastPhotoHandling-
         query.addBindValue( contact.photoDlCRC);   // -FastPhotoHandling-
         query.addBindValue( contact.photoSize );   // -FastPhotoHandling-
         query.addBindValue( contact.photoCRC );    // -FastPhotoHandling-
         query.addBindValue( contact.APL1Size);  // -FastContactSearch-
         query.addBindValue( contact.crcAPL1);   // -FastContactSearch-

         ETG_TRACE_USR2((" FC_PhoneBook_SQLite: String used for master table in update record - %s",sql.toLatin1().constData()));
         if (!query.exec())
         {
            PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

            ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Update record failed in device table"));
         }
         else
         {
            bRetValue = true;
         }
      }
   }
   ETG_TRACE_USR4((" FC_PhoneBook_SQLite: Record updated successfully"));
   return bRetValue;
}
*/


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::DeleteAllContactsAndCallHistoryFromDB(
*
* DESCRIPTION:
*              Delete a All records in database
*
*
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::DeleteAllContactsAndCallHistoryFromDB()//FIX GMMY16-23858
{
   ETG_TRACE_USR4(("DeleteAllContactsAndCallHistoryFromDB entered"));

   int i = 0;
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);

   QString sql;
   for(i = FC_PB_VEHICLE_PBOOK ; i <= FC_PB_DEVICE_HANDLE11; i++)
   {
      sql = "DELETE  FROM "+ fc_pb_aTableNames[i];
      ETG_TRACE_USR2((" FC_PhoneBook_SQLite: Delete All Records in PhoneBook Contacts - %s",sql.toLatin1().constData()));

      //Fix for GMMY15-10814
      if(db.isOpen())
      {
         if(!query.exec(sql))
         {
            PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

            ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Delete record failed in All table"));
            return false;
         }
      }
      else
      {
         ETG_TRACE_ERR(("Database is not open"));
         return false;
      }
   }

   for(i = 1 ; i <= FC_PB_TOTAL_NUMBER_OF_CH; i++)
   {
      sql = "DELETE  FROM "+ fc_pb_aTableNames[MAP_TO_CH_INDEX(i)];

      ETG_TRACE_USR2((" FC_PhoneBook_SQLite: Delete All Records in CallHistory - %s ",sql.toLatin1().constData()));

      //Fix for GMMY15-10814
      if(db.isOpen())
      {
         if(!query.exec(sql))
         {
            PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

            ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Delete CallHistory failed in All table"));
            return false;
         }
      }
      else
      {
         ETG_TRACE_ERR(("Database is not open"));
         return false;
      }
   }

   sql = "DELETE  FROM " FC_PB_MASTER_TABLE_NAME;
   ETG_TRACE_USR2((" FC_PhoneBook_SQLite: Delete All Records in Master Table - %s ",sql.toLatin1().constData()));

   //Fix for GMMY15-10814
   if(db.isOpen())
   {
      if(!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Failed in Deleting All Records in Master Table"));
         return false;
      }
   }
   else
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   return true;
}


bool FC_PhoneBook_SQLite::DeletePhoneBookContacts(quint8 DeviceHandle)
{
   ETG_TRACE_USR2((" DeletePhoneBookContacts entered with DeviceHandle %d ", DeviceHandle));

   QMutexLocker lock(&m_mutex);

   QSqlDatabase db= Database();

   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);
   QString sql;

   sql = "DELETE  FROM "+ fc_pb_aTableNames[DeviceHandle];
   ETG_TRACE_USR2((" FC_PhoneBook_SQLite: Delete All Records in PhoneBook Contacts - %s",sql.toLatin1().constData()));

   if(!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Delete record failed in All table"));
      return false;
   }

   sql = "DELETE  FROM " FC_PB_MASTER_TABLE_NAME " WHERE TableIndex = " + QString::number(DeviceHandle);
   ETG_TRACE_USR2((" FC_PhoneBook_SQLite: Delete All Records in Master Table - %s ",sql.toLatin1().constData()));

   //Fix for GMMY15-10814
   if(db.isOpen())
   {
      if(!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Failed in Deleting All Records in Master Table"));
         return false;
      }

      //TODO: Currently it will delete current outgoing source also in case it exceeds 10 MB.
      // Delete the Feature Support for the Given Device.
      sql= "DELETE FROM  " FC_PB_TABLE_NAME_FS " WHERE " FC_PB_FS_COL_DEVICE_HANDLE " = " + QString::number(DeviceHandle);

      if(!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("FC_PhoneBook_SQLite: Deleting of Device Feature Support entry in table FC_PB_TABLE_NAME_FS"));
         return false;
      }
      else
      {
         ETG_TRACE_USR4(("FC_PhoneBook_SQLite:  Deleting of Device Feature Support entry in table FC_PB_TABLE_NAME_FS is successful"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   return true;
}


bool FC_PhoneBook_SQLite::DeleteVehiclePhoneBook(tU8 u8DeviceHandle)
{
   ETG_TRACE_USR4(("DeleteVehiclePhoneBook function entered with device handle: %d", u8DeviceHandle));

   QMutexLocker lock(&m_mutex);

   QSqlDatabase db = Database();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   if (false == CHECK_VALID_VPBx(u8DeviceHandle) && FC_PB_VEHICLE_PBOOK
         != u8DeviceHandle)
   {
      ETG_TRACE_ERR((" Not a valid VPBx handle %d", u8DeviceHandle));
      return false;
   }

   QSqlQuery query(db);

   QString sql = "DELETE  FROM ";

   if (FC_PB_VEHICLE_PBOOK == u8DeviceHandle)
   {
      sql += FC_PB_TABLE_NAME_VPB;
   }
   else
   {
      sql += fc_vpb_aTableNames[u8DeviceHandle];
   }

   ETG_TRACE_USR2((" FC_PhoneBook_SQLite: Delete All Records in Vehicle PhoneBook  - %s ", sql.toLatin1().constData()));
   if (!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Delete record failed in Vehicle Phonebook"));
      return false;
   }

   sql.clear();
   if (FC_PB_VEHICLE_PBOOK == u8DeviceHandle)
   {
      sql = "DELETE  FROM " FC_PB_MASTER_TABLE_NAME " WHERE TableIndex = 1";

      ETG_TRACE_USR2(("FC_PhoneBook_SQLite: Delete All vehicle phonebook Records in Master Table - %s ", sql.toLatin1().constData()));
      if (!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Failed in Deleting All Records in Master Table"));
         return false;
      }
   }

   m_noOfContactsVehicleAddBook[u8DeviceHandle] = 0;

   ETG_TRACE_USR4(("DeleteVehiclePhoneBook function exited"));
   return true;
}

// -bn: TODO: Method name confuses me as the comments focus just on Vehicle address book tables. Discuss with Suthmitha.
/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::DeleteRecord(
*
* DESCRIPTION:
*              Delete a single record in Vehicle address book table
*
*
* PARAMETER:  [IN]  - contact handle to be deleted from Vehicle address book table
*             [OUT] -
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::DeleteRecord(quint32 contacthandle, quint8 tableindex)
{
   ETG_TRACE_USR4((" DeleteRecord entered with contacthandle %d ", contacthandle));

   QMutexLocker lock(&m_mutex);

   QSqlDatabase db= Database();

   quint8 u8currentposition = VALUE_ZERO;
   quint32 noofvehiclecontacts = GetNoOfContactsInVPB(tableindex);
   if(!db.isOpen())
   {

      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);

   QString sql;

   if (tableindex == FC_PB_VEHICLE_PBOOK)
   {
      ETG_TRACE_USR4((" FC_PhoneBook_SQLite: Marked bit is set to one, hence writing query to delete VPB "));
      sql = "SELECT ContactHandle FROM " + fc_pb_aTableNames[tableindex] + " WHERE " FC_PB_COL_CONTACT_HANDLE " = " + QString::number(contacthandle); //NCG3D-139336
   }
   else
   {
      ETG_TRACE_USR4((" FC_PhoneBook_SQLite: Marked bit is set to zero, hence writing query to delete VPB 2 to 11"));
      sql = "SELECT ContactHandle FROM " + fc_vpb_aTableNames[tableindex] + " WHERE " FC_PB_COL_CONTACT_HANDLE " = " + QString::number(contacthandle); //NCG3D-139336
   }

   ETG_TRACE_USR2((" FC_PhoneBook_SQLite: String used for master table in delete record - %s ",sql.toLatin1().constData()));

   if(!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Delete record failed in master table"));
      return false;
   }

   if(query.next())
   {
      ETG_TRACE_USR4((" FC_PhoneBook_SQLite: Contact found having the contact handle"));
   }
   else
   {
      ETG_TRACE_USR4((" FC_PhoneBook_SQLite: No contact found having the contact handle"));
      return false;
   }

   if (tableindex!= FC_PB_VEHICLE_PBOOK)
   {
      sql = "SELECT " FC_PB_COL_VPB_CONTACT_POSITION " FROM "
            + fc_vpb_aTableNames[tableindex]
            + " WHERE " FC_PB_COL_CONTACT_HANDLE " = "
            + QString::number(contacthandle);
      if (!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Contact position failed"));
         return false;
      }
      if (query.next())
      {
         u8currentposition = (query.value(VALUE_ZERO)).value<quint8>();
         ETG_TRACE_USR4(("current position of the requested contact handle - %d   ", u8currentposition));
         ETG_TRACE_USR4((" FC_PhoneBook_SQLite: Contact found having the contact handle"));
      }
   }
   sql.clear();

   //Delete the Contact from the Vehicle address book Table
   if (tableindex == FC_PB_VEHICLE_PBOOK)
   {
   sql= "DELETE FROM " FC_PB_TABLE_NAME_VPB " WHERE " FC_PB_COL_CONTACT_HANDLE " = " + QString::number(contacthandle);
   }
   else
   {
      sql= "DELETE FROM " +fc_vpb_aTableNames[tableindex] + " WHERE " FC_PB_COL_CONTACT_HANDLE " = " + QString::number(contacthandle);
   }

   ETG_TRACE_USR2((" FC_PhoneBook_SQLite: String used for Vehicle address book table in delete record - %s ",sql.toLatin1().constData()));
   if(!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_USR4((" FC_PhoneBook_SQLite: Delete record failed in Vehicle address book table"));
      return false;
   }

   ETG_TRACE_USR4((" FC_PhoneBook_SQLite: Record deleted from Vehicle address book successfully"));

   //Delete the Contents from Master Table
   sql= "DELETE FROM " FC_PB_MASTER_TABLE_NAME " WHERE " FC_PB_MASTER_COL_CONTACT_HANDLE " = "+QString::number(contacthandle);

   ETG_TRACE_USR2((" FC_PhoneBook_SQLite: String used for master table in delete record - %s ",sql.toLatin1().constData()));
   if(!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Delete record failed in Master table"));
      return false;
   }

   ETG_TRACE_USR4((" FC_PhoneBook_SQLite: Record deleted successfully in Master table"));

   sql.clear();
   if (tableindex!= FC_PB_VEHICLE_PBOOK)
   {
      sql = "UPDATE " + fc_vpb_aTableNames[tableindex]
                  + " SET " FC_PB_COL_VPB_CONTACT_POSITION " = " FC_PB_COL_VPB_CONTACT_POSITION
                  " -1 WHERE (( " FC_PB_COL_VPB_CONTACT_POSITION " >= "
                  + QString::number(u8currentposition + 1)
                  + ") AND (" FC_PB_COL_VPB_CONTACT_POSITION " <= "
                  + QString::number(noofvehiclecontacts) + "))";
      ETG_TRACE_USR4(("SQL string to be executed::: %s", sql.toUtf8().constData()));
      if (!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Failed to update the position"));
         return false;
      }

   }

   return true;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::DeleteRecordsOfDevice
*
* DESCRIPTION:
*              Deletes all the phonebook contacts of a device in database
*
*
* PARAMETER:  [IN]  - device handle of the device
*             [OUT] -
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::DeleteRecordsOfDevice(quint8 u8TableIndex)
{
   ETG_TRACE_USR2((" DeleteRecordsOfDevice entered with TableIndex %d ", u8TableIndex));

   QMutexLocker lock(&m_mutex);

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);

   if(!CHECK_VALID(u8TableIndex))
   {
      ETG_TRACE_USR4((" FC_PhoneBook_SQLite: device handle is invalid in function"
         "DeleteRecordsOfDevice "));
      return false;
   }

   QString sql= "DELETE FROM " + fc_pb_aTableNames[u8TableIndex];

   if(!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Deleting of all the records of a table has failed"));
      return false;
   }
   else
   {
      ETG_TRACE_USR4((" FC_PhoneBook_SQLite: Delete of all the records of a device was successful "));
      sql.clear();
      sql= "DELETE FROM " FC_PB_MASTER_TABLE_NAME " WHERE " FC_PB_MASTER_COL_TABLE_INDEX " = "
         + QString::number(u8TableIndex);

      if(!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(( " FC_PhoneBook_SQLite: Deleting of all contacts of a device in master table has failed " ));
         return false;
      }
      else
      {
         ETG_TRACE_USR4(( " FC_PhoneBook_SQLite: Deleting of all contacts of a device in master table was successful " ));
      }

      QString sqlvpb = "DELETE FROM " + fc_vpb_aTableNames[u8TableIndex];

      if(!query.exec(sqlvpb))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(( " FC_PhoneBook_SQLite: Deleting of all contacts of a VPBdevice  has failed " ));
         return false;
      }
      else
      {
         ETG_TRACE_USR4(( " FC_PhoneBook_SQLite: Deleting of all contacts of a VPBdevice in master table was successful " ));
      }
   }

   // Delete the Feature Support for the Given Device.
   QString sql_fs = "DELETE FROM  " FC_PB_TABLE_NAME_FS " WHERE " FC_PB_FS_COL_DEVICE_HANDLE " = " + QString::number(u8TableIndex);

   if(!query.exec(sql_fs))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(("FC_PhoneBook_SQLite: Deleting of Device Feature Support entry in table FC_PB_TABLE_NAME_FS"));
      return false;
   }
   else
   {
      ETG_TRACE_USR4(("FC_PhoneBook_SQLite:  Deleting of Device Feature Support entry in table FC_PB_TABLE_NAME_FS was successful"));
   }

   ETG_TRACE_USR2(("Query used in DeleteRecordsOfDevice to delete Fetaure Support entries- %s", sql_fs.toLatin1().constData()));

   SpeechUpdateCheckSumTable(u8TableIndex);
   return true;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetContactDetails(
*
* DESCRIPTION:
*              Get the contact details for a given contact handle
*
*
* PARAMETER:  [IN]  - contact handle
*             [OUT] -
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::GetContactDetails(quint32 contacthandle, FC_PhoneBook_Contact_Detail &contact)
{
   ETG_TRACE_USR4((" FC_PhoneBook_SQLite: GetContactDetails method STARTS "));

   QMutexLocker lock(&m_mutex);

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {

      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);
   bool IsSkipPosition = false;

   QString sql = "SELECT " FC_PB_MASTER_COL_TABLE_INDEX " FROM "
      FC_PB_MASTER_TABLE_NAME " WHERE " FC_PB_MASTER_COL_CONTACT_HANDLE " = "+QString::number(contacthandle);

   ETG_TRACE_USR2(("FC_PhoneBook_SQLite: String used for master table in GetContactDetails - %s ",
      sql.toLatin1().constData()));

   if(!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR((" FC_PhoneBook_SQLite: GetContactDetails failed in master table"));
      return false;
   }

   if(!query.next())
   {
      ETG_TRACE_USR4((" FC_PhoneBook_SQLite: Get contact details - No entry found for the contact handle"));
      return false;
   }

   quint8 tableindex= (query.value(0)).value<quint8>();

   ETG_TRACE_USR2((" FC_PHoneBook_SQLite: Get contact details from the Device handle %d", tableindex));

   if(!CHECK_VALID(tableindex))
   {
      ETG_TRACE_USR4((" FC_PhoneBook_SQLite: device handle is invalid  "));
      return false;
   }

   sql.clear();

   // Get the contact details for the devicehandle(Table) and contacthandle
   //FIX CMG3G-8369 IS2427_GetContactDetailsExtended@FC_Phonebook
   if ( ((contacthandle & FC_PB_CONTACT_HANDLE_DPB_MARKER) == FC_PB_CONTACT_HANDLE_DPB_MARKER) || (tableindex == FC_PB_VEHICLE_PBOOK) )
   {
      ETG_TRACE_USR4((" FC_PhoneBook_SQLite: Marked bit is set to one, hence writing query for device phonebook to get details"));
      sql = "SELECT * FROM " + fc_pb_aTableNames[tableindex] + " WHERE " FC_PB_COL_CONTACT_HANDLE " = " + QString::number(contacthandle);
   }
   else
   {
      ETG_TRACE_USR4((" FC_PhoneBook_SQLite: Marked bit is set to zero, hence writing query for vehicle phonebook to get details"));
      sql = "SELECT * FROM " + fc_vpb_aTableNames[tableindex] + " WHERE " FC_PB_COL_CONTACT_HANDLE " = " + QString::number(contacthandle);
      IsSkipPosition = true;
   }

   ETG_TRACE_USR2(( " FC_PhoneBook_SQLite: String used in device contact details - %s ", sql.toLatin1().constData() ));

   if(!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR((" FC_PhoneBook_SQLite: GetContactDetails Query has FAILED "));
      return false;
   }

   FC_PhoneBook_List* poFC_PhoneBook_List = FC_PhoneBook_List::pGetInstance();
   if (poFC_PhoneBook_List != NULL)
   {
      // Fill the contact details into contact structure
      if (query.next())
      {
         int counter = 0;

         contact.contactHandle = query.value(counter++).toUInt();
         if (true == IsSkipPosition)
         {
            counter++;
         }
         contact.firstName = query.value(counter++).toString();
         ETG_TRACE_USR4(("firstName %s", contact.firstName.toLatin1().constData()));
         contact.lastName = query.value(counter++).toString();
         ETG_TRACE_USR4(("lastName %s", contact.lastName.toLatin1().constData()));
         contact.Phonetics = query.value(counter++).toString();
         ETG_TRACE_USR4(("Phonetics %s", contact.Phonetics.toLatin1().constData())); //292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
//         counter++;  // To skip SortFirstName  // NEW , but commented out. TODO: Activate (uncomment), when the related column has been moved to here
//         counter++;  // To skip SortLastName   // NEW , but commented out. TODO: Activate (uncomment), when the related column has been moved to here
         contact.poiName = query.value(counter++).toString();
         ETG_TRACE_USR4(("poiName %s", contact.poiName.toLatin1().constData()));
         //Fix for CMG3GB-2829 : Getcontactdetailsextended is not filling the values from phonenmuber 1 to 8
         //Number 8 used in for loop (condition) to iterate number of phonenumbers in database, If it changes in future
         //need to set the value accordingly
         for (int i = 0; i < FC_PB_NO_OF_MAX_PHONENUMS_PER_CONTACT; i++)
         {
            if (query.value(counter++).toString().isEmpty() != TRUE)
            {
               poFC_PhoneBook_List->ExtractingOfFirstNPhoneNumbers((query.value(counter)).value<quint8>(), &contact, query.value(counter
                     + 1).toString());
            }
            counter = counter + 2; //Skipping the fields NumberType and PPPhoneNum to go to next consecutive PhoneNumber field
//          counter = counter + 3;   // NEW  //Skipping the fields NumberType, PPPhoneNum and NumberSubtr to go to next consecutive PhoneNumber field. TODO: Activate (uncomment), when the related column has been moved to here
         }
         //End of Fix CMG3GB-2829
         ETG_TRACE_USR4(("PPPhoneNum1 %s", contact.PPPhoneNum1.toLatin1().constData()));
         ETG_TRACE_USR4(("NumberType1 %d", contact.NumberType1));
         ETG_TRACE_USR4(("PPPhoneNum2 %s", contact.PPPhoneNum2.toLatin1().constData()));
         ETG_TRACE_USR4(("NumberType2 %d", contact.NumberType2));
         ETG_TRACE_USR4(("PPPhoneNum3 %s", contact.PPPhoneNum3.toLatin1().constData()));
         ETG_TRACE_USR4(("NumberType3 %d", contact.NumberType3));
         ETG_TRACE_USR4(("PPPhoneNum4 %s", contact.PPPhoneNum4.toLatin1().constData()));
         ETG_TRACE_USR4(("NumberType4 %d", contact.NumberType4));
         ETG_TRACE_USR4(("PPPhoneNum5 %s", contact.PPPhoneNum5.toLatin1().constData()));
         ETG_TRACE_USR4(("NumberType5 %d", contact.NumberType5));
         ETG_TRACE_USR4(("PPPhoneNum6 %s", contact.PPPhoneNum6.toLatin1().constData()));
         ETG_TRACE_USR4(("NumberType6 %d", contact.NumberType6));
         ETG_TRACE_USR4(("PPPhoneNum7 %s", contact.PPPhoneNum7.toLatin1().constData()));
         ETG_TRACE_USR4(("NumberType7 %d", contact.NumberType7));
         ETG_TRACE_USR4(("PPPhoneNum8 %s", contact.PPPhoneNum8.toLatin1().constData()));
         ETG_TRACE_USR4(("NumberType8 %d", contact.NumberType8));
         contact.emailAdd1 = query.value(counter++).toString();
         contact.emailAdd2 = query.value(counter++).toString();
         contact.emailAdd3 = query.value(counter++).toString();
         contact.homeAdd = query.value(counter++).toString();
         contact.workAdd = query.value(counter++).toString();
         contact.otherAdd = query.value(counter++).toString();
         contact.category = query.value(counter++).toString();
         counter++; // To skip ContactType
         contact.notes = query.value(counter++).toString();
         contact.homeGeoCodeLatLongValidity = query.value(counter++).toBool();
         contact.homeGeoCodeLatitude = query.value(counter++).toInt();
         contact.homeGeocodeLongitude = query.value(counter++).toInt();
         contact.workGeoCodeLatLongValidity = query.value(counter++).toBool();
         contact.workGeoCodeLatitude = query.value(counter++).toInt();
         contact.workGeocodeLongitude = query.value(counter++).toInt();
         contact.otherGeoCodeLatLongValidity = query.value(counter++).toBool();
         contact.otherGeoCodeLatitude = query.value(counter++).toInt();
         contact.otherGeocodeLongitude = query.value(counter++).toInt();
         contact.photoDlSize = query.value(counter++).toUInt();
         contact.photo = QByteArray(); // Set photo to an empty photo
         //Fix CMG3GB-2535
         counter++; //To skip the photo
         contact.photoType = query.value(counter++).toString(); // -FastPhotoHandling- TODO: Check if this assignment is really needed.
         contact.photoDlCRC = (query.value(counter++)).value<quint16>(); // -FastPhotoHandling- TODO: Check if this assignment is really needed.
         contact.photoSize = query.value(counter++).toUInt(); // Needed to decide if contact has photo
         contact.APL1Size = query.value(counter++).toUInt(); // -FastContactSearch- TODO: Check if this assignment is really needed.
         contact.crcAPL1 = (query.value(counter++)).value<quint16>(); // -FastContactSearch- TODO: Check if this assignment is really needed.

         ETG_TRACE_USR4((" FC_PhoneBook_SQLite: GetContactDetails contacts retrieved from table "));
      }
      else // MV-BugzID: 207903
      {
         ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Get contact details - No entry found in device table!!"));
         return false;
      }
   }

   ETG_TRACE_USR4((" FC_PhoneBook_SQLite: GetContactDetails method ENDS"));

   return true;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetCallHistoryRecordDetails(
*
* DESCRIPTION:
*              Get the contact details for a given contact handle
*
*
* PARAMETER:  [IN]  - contact handle
*             [OUT] -
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::GetCallHistoryRecordDetails(quint32 contacthandle , quint8 u8ConnectionID,
                                                      QList<FC_PhoneBook_Callhistory_Detail>& RecordList)
{
   ETG_TRACE_USR4((" GetCallHistoryRecordDetails entered "));

   QMutexLocker lock(&m_mutex);
   QSqlDatabase db= Database();
   if(!db.isOpen())
   {

      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);

   if(!CHECK_CH_VALID(u8ConnectionID))
   {

      ETG_TRACE_USR4((" FC_PhoneBook_SQLite: ConnectionID is invalid "));
      return false;
   }

   // Get the contact details for the contacthandle

   QString sql= "SELECT * FROM " + fc_pb_aTableNames[MAP_TO_CH_INDEX(u8ConnectionID)] +
      " WHERE " FC_PB_COL_CH_CONTACT_HANDLE " = " + QString::number(contacthandle);


   ETG_TRACE_USR2(( " FC_PhoneBook_SQLite: String used in device contact details - %s",
                    sql.toLatin1().constData() ));


   if(!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR((" FC_PhoneBook_SQLite: GetContactDetails Query Failed  for call history "));
      return false;
   }


   RecordList.clear();
   //Fix for GMMY15-10814
   while(db.isOpen() && query.next())
   {
      int counter=0;
      FC_PhoneBook_Callhistory_Detail contact;
      contact.contactHandle = query.value( counter++ ).toUInt();
      contact.callType      = (query.value(counter++)).value<quint8>();
      contact.firstName     = query.value( counter++ ).toString();
      contact.lastName      = query.value( counter++ ).toString();
      contact.phoneNumber   = query.value( counter++ ).toString();
      contact.numberType    = (query.value(counter++)).value<quint8>();
      contact.dateStamp     = query.value( counter++ ).toString();
      contact.timeStamp     = query.value( counter++ ).toString();

      RecordList<<contact;

      ETG_TRACE_USR4((" FC_PhoneBook_SQLite: GetContactDetails contacts retrieved for call history "));
   }

   if(RecordList.empty())
   {
      ETG_TRACE_USR4((" FC_PhoneBook_SQLite: GetContactDetails contacts retrieved for call history "));
      return false;
   }

   return true;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::DeleteCallHistoryRecord
*
* DESCRIPTION:
*
*
*
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::DeleteCallHistoryRecord(quint8 u8ConnectionID,
                                                  quint32 u32ContactHandle)
{
   ETG_TRACE_USR4((" DeleteCallHistoryRecord entered "));

   QMutexLocker lock(&m_mutex);
   QSqlDatabase db= Database();
   if(!db.isOpen())
   {

      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);

   bool blRet= false;

   if(!CHECK_CH_VALID(u8ConnectionID))
   {
      ETG_TRACE_USR4((" FC_PhoneBook_SQLite: ConnectionID is invalid  "));
      return false;
   }

   QString sql= "DELETE FROM " + fc_pb_aTableNames[MAP_TO_CH_INDEX(u8ConnectionID)] +
      " WHERE " FC_PB_COL_CH_CONTACT_HANDLE " = "+QString::number(u32ContactHandle);

   if(query.exec(sql))
   {
      sql.clear();
      sql= "DELETE FROM " FC_PB_MASTER_TABLE_NAME " WHERE " FC_PB_COL_CH_CONTACT_HANDLE
         " = " + QString::number(u32ContactHandle) + " AND " FC_PB_MASTER_COL_TABLE_INDEX
         " = " + QString::number(MAP_TO_CH_INDEX(u8ConnectionID));
      if(query.exec(sql))
      {
         ETG_TRACE_USR4((" FC_PhoneBook_SQLite:- Call history record deleted successfully"));
         blRet= true;
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" FC_PhoneBook_SQLite:- Error - failed in deleting call history"
            "record from the master table"));
      }
   }
   else
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR((" FC_PhoneBook_SQLite:- Error - failed in deleting call history"
         "record from the table"));
   }

   return blRet;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::UpdateCallHistoryRecord(
*
* DESCRIPTION:
*              Updates a single call history record in database
*
*
* PARAMETER:  [IN]  - contact handle and phone number
*                     for which details are to be updated
*             [OUT] -
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::UpdateCallHistoryRecord(FC_PhoneBook_Callhistory_Detail contact,
                                                  quint32 contacthandle, quint8 ConnectionID)
{
   ETG_TRACE_USR4(("FC_PhoneBook_SQLite::UpdateCallHistoryRecord entered "));

   QMutexLocker lock(&m_mutex);
   QSqlDatabase db= Database();

   /*FIX for GMMY15-11621 Adding check to verify if db is open at the time of calling prepare.
   Removing redundant character 'space' from query.
   Binding values only on prepare success.*/
   bool bPrepareQuery = FALSE;

   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);

   if(!CHECK_CH_VALID(ConnectionID))
   {
      ETG_TRACE_USR4((" FC_PhoneBook_SQLite: ConnectionID is invalid  "));
      return false;
   }

   /* Fix for GMMY15-10944 */
   QString sql= "UPDATE " + fc_pb_aTableNames[MAP_TO_CH_INDEX(ConnectionID)] + " SET "
      FC_PB_COL_CH_CONTACT_HANDLE " = ? ,"
      FC_PB_COL_CH_FIRST_NAME " = ? ,"
      FC_PB_COL_CH_LAST_NAME " = ? ,"
      FC_PB_COL_CH_NUMBER_TYPE " = ?"
      " WHERE " FC_PB_COL_CONTACT_HANDLE " = " + QString::number(contacthandle) +
      " AND " FC_PB_COL_CH_PHONE_NUM "== '"+contact.phoneNumber+"' ";//FIX GMMY15-11882 Update the record for the specific phone number and contact handle

   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   bPrepareQuery = query.prepare(sql);
/* // BAD
   //updating fields in call history record with null string, if it is not provided by device.
   //For instance, Sony Ericsson W810C will not provide CallDate and CallTime.
   //We assign default value as null string to such fields in order to avoid malfunction.
   if ( 0 == contact.firstName.simplified().length())
   {
      contact.firstName = BLANK_NAME;
   }
   if ( 0 == contact.lastName.simplified().length())
   {
      contact.lastName = BLANK_NAME;
   }
*/

   if(TRUE == bPrepareQuery)
   {
      /* Fix for GMMY15-10944 */
      query.addBindValue(contact.contactHandle);
      query.addBindValue(contact.firstName);
      query.addBindValue(contact.lastName);
      query.addBindValue(contact.numberType);

      ETG_TRACE_USR2((" FC_PhoneBook_SQLite: String used for update call history record= '%s' ",
         sql.toLatin1().constData()));

      if(!db.isOpen())
      {
         ETG_TRACE_ERR(("Database is not open"));
         return false;
      }
      //End of fix GMMY15-11621
      if (!query.exec())
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Update record failed for call history"));
         return false;
      }

      ETG_TRACE_USR4((" FC_PhoneBook_SQLite: Record updated successfully"));
      return true;
   }
   else
   {
      ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Query prepare failed"));
      return false;
   }
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::InsertCallHistoryRecords(
*
* DESCRIPTION:
*              overloaded function to insert a single record related to call history in database
*
*
* PARAMETER:  [IN] records - The call history record
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::InsertCallHistoryRecords( FC_PhoneBook_Callhistory_Detail& Record,
                                                   quint8 ConnectionID, quint32& ContactHandle)
{
   ETG_TRACE_USR4((" InsertCallHistoryRecords entered "));

   QMutexLocker lock(&m_mutex);
   bool blRet= true;

   QTime DebugTime;
   DebugTime.start();

   if(!CHECK_CH_VALID(ConnectionID))
   {

      ETG_TRACE_USR4((" FC_PhoneBook_SQLite: Connection ID is invalid  "));
      return false;
   }

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {

      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   db.transaction();
   QSqlQuery query(db);

   QString sql = "INSERT INTO " + fc_pb_aTableNames[MAP_TO_CH_INDEX(ConnectionID)] + " ( "
      FC_PB_COL_CH_CONTACT_HANDLE", "
      FC_PB_COL_CH_CALL_TYPE", "
      FC_PB_COL_CH_FIRST_NAME", "
      FC_PB_COL_CH_LAST_NAME", "
      FC_PB_COL_CH_PHONE_NUM", "
      FC_PB_COL_CH_NUMBER_TYPE", "
      FC_PB_COL_CH_CALL_DATE", "
      FC_PB_COL_CH_CALL_TIME") VALUES ( ?, ?, ?, ?, ?, ?, ?, ?);";

   /* Generate contact handle only if it is not provided */
   if(!Record.contactHandle)
   {
      if(GetContactHandleForCallHistory(ConnectionID, Record.callType, ContactHandle))
      {
         if(!InsertIntoMasterTable( db, MAP_TO_CH_INDEX(ConnectionID), ContactHandle))
         {
            blRet= false;
         }
      }
      else
      {
         blRet= false;
      }
   }
   else
   {
      ContactHandle= Record.contactHandle;
   }

   if(blRet)
   {
      //Fix for GMMY15-10814
      if(db.isOpen())
      {
         query.prepare( sql );
      }
      else
      {
         ETG_TRACE_ERR(("Database is not open"));
         return false;
      }
/* BAD
      //updating fields in call history record with null string, if it is not provided by device.
      //For instance, Sony Ericsson W810C will not provide CallDate and CallTime.
      //We assign default value as null string to such fields in order to avoid malfunction.
      if ( 0 == Record.firstName.simplified().length())
      {
         Record.firstName = BLANK_NAME;
      }
      if ( 0 == Record.lastName.simplified().length())
      {
         Record.lastName = BLANK_NAME;
      }
      if ( 0 == Record.phoneNumber.simplified().length())
      {
         Record.phoneNumber = BLANK_NAME;
      }
      if ( 0 == Record.dateStamp.simplified().length())
      {
         Record.dateStamp = BLANK_NAME;
      }
      if ( 0 == Record.timeStamp.simplified().length())
      {
         Record.timeStamp = BLANK_NAME;
      }
*/
      //Fix for GMMY15-10814
      if(db.isOpen())
      {
         query.addBindValue( ContactHandle );
         query.addBindValue( Record.callType );
         query.addBindValue( Record.firstName );
         query.addBindValue( Record.lastName );
         query.addBindValue( Record.phoneNumber );
         query.addBindValue( Record.numberType );
         query.addBindValue( Record.dateStamp );
         query.addBindValue( Record.timeStamp );

         if (!query.exec())
         {
            PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

            ETG_TRACE_ERR( ( "FC_PhoneBook_SQLite :- ERROR - SQL query failed in InsertCallHistoryRecords "));
            blRet= false;
         }
      }
      else
      {
         ETG_TRACE_ERR(("Database is not open"));
         return false;
      }
   }

   //Fix for GMMY15-10814
   if(db.isOpen())
   {
      if(!blRet)
      {
         db.rollback(); //Rollback
         ETG_TRACE_USR4(("Database rolled back"));
      }
      else
      {
         if(!db.commit())
         {
            db.rollback();
            ETG_TRACE_USR4(("Database rolled back since commit failed"));
         }
         else
         {
            ETG_TRACE_USR2(("Database committed successfully andTotal time elapsed in inserting - %d ms", DebugTime.elapsed()));
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   return blRet;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::DeleteCallHistoryRecordsOfCallType
* DESCRIPTION:
*
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE: bool - indicating success or failure of this function
*******************************************************************************/
bool FC_PhoneBook_SQLite::DeleteCallHistoryRecordsOfCallType(quint8 u8ConnectionID,
                                                             quint8 u8CallType)
{
   ETG_TRACE_USR4((" DeleteCallHistoryRecordsOfCallType entered with call type: %d", u8CallType));

   QMutexLocker lock(&m_mutex);
   QSqlDatabase db= Database();
   if(!db.isOpen())
   {

      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }
   QSqlQuery query(db);

   bool blRet= false;

   if(!CHECK_CH_VALID(u8ConnectionID))
   {
      ETG_TRACE_USR4((" FC_PhoneBook_SQLite: ConnectionID is invalid "));
      return false;
   }

   QString sql= "SELECT " FC_PB_COL_CONTACT_HANDLE " FROM "+
      fc_pb_aTableNames[MAP_TO_CH_INDEX(u8ConnectionID)] +" WHERE "
      FC_PB_COL_CH_CALL_TYPE " = " + QString::number(u8CallType);

   if(query.exec(sql))
   {
      QList<quint32> ContactHandleList;
      while(query.next())
      {
         quint32 u32ContactHandle= query.value(0).toUInt();
         ContactHandleList<< u32ContactHandle;
      }

      if(ContactHandleList.size()>0)
      {
         ETG_TRACE_USR2(("FC_PhoneBook_SQLite:: %d Contact Handles found for deleting",
                         ContactHandleList.size()));

         sql.clear();
         sql= "DELETE FROM " + fc_pb_aTableNames[MAP_TO_CH_INDEX(u8ConnectionID)] +
            " WHERE " FC_PB_COL_CH_CALL_TYPE " = "+QString::number(u8CallType);

         if(query.exec(sql))
         {
            if(RemoveContactHandlesFromMasterTable(ContactHandleList,
               MAP_TO_CH_INDEX(u8ConnectionID)))
            {
               ETG_TRACE_USR4((" FC_PhoneBook_SQLite:- Call history record deleted successfully"));
               blRet= true;
            }
            else
            {
               ETG_TRACE_USR4((" FC_PhoneBook_SQLite:- Error - failed in deleting call history"
                               "record from the master table"));
            }

            u32LastCallHistoryCntHandle[u8ConnectionID-1][u8CallType]= OSAL_NULL;

            ETG_TRACE_USR2(("Last contact handle used cleared for connection id- %d, "
               "Calltype- %d", u8ConnectionID, u8CallType));
         }
         else
         {
            PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

            ETG_TRACE_USR4((" FC_PhoneBook_SQLite:- Error - failed in deleting call history"
                            "record from the table"));
         }
      }
      else
      {
         blRet= true;
         ETG_TRACE_USR4(("No Contact handles found for deleting"));
      }
   }
   else
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(("FC_PhoneBook_SQLite:- Error - failed to get contact handles to be"
         "deleted"));
   }

   return blRet;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::RemoveContactHandlesFromMasterTable
* DESCRIPTION: Removes the list of contact handles from master table
*
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE: bool - indicating success or failure of this function
*******************************************************************************/
bool FC_PhoneBook_SQLite::RemoveContactHandlesFromMasterTable(
   QList<quint32> ContactHandleList, quint8 u8TableIndex)
{
   ETG_TRACE_USR4((" RemoveContactHandlesFromMasterTable entered "));

   quint16 u16ListSize= (quint16)ContactHandleList.size();
   QSqlDatabase db= Database();
   if(!db.isOpen())
   {

      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);

   bool blRet= true;

   QString sql= "DELETE FROM " FC_PB_MASTER_TABLE_NAME " WHERE "
      FC_PB_MASTER_COL_TABLE_INDEX " = "
      + QString::number(u8TableIndex) + " AND (";

   for(quint16 u16Index= 0; u16Index < u16ListSize; u16Index++)
   {
      sql+= "(" FC_PB_MASTER_COL_CONTACT_HANDLE " = " + QString::number(
         ContactHandleList.at(u16Index)) + ")";

      if(((( u16Index + 1) % CHUNK_DELETE ) == 0) || ( u16Index ==  (u16ListSize -1) ))
      {
         sql+= ")";

         ETG_TRACE_USR2(("FC_PhoneBook_SQLite:: Sql query used in delete contact handles -"
            "%s", sql.toLatin1().constData()));

         if(query.exec(sql))
         {
            ETG_TRACE_USR4(("FC_PhoneBook_SQLite:- Contact Handles removed successfully"
               " from master table"));

            sql.clear();
            sql= "DELETE FROM " FC_PB_MASTER_TABLE_NAME " WHERE "
               FC_PB_MASTER_COL_TABLE_INDEX " = "
               + QString::number(u8TableIndex) + " AND (";
         }
         else
         {
            PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

            blRet = false;
            ETG_TRACE_ERR(("FC_PhoneBook_SQLite:- Error - failed in removing "
               "contact handles from the master table"));
            break;
         }
      }
      else
      {
         sql+= " OR ";
      }
   }

   return blRet;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_Contact_Detail::bSubsetCompare(const FC_PhoneBook_Contact_Detail &contact)
*
* DESCRIPTION: Compares the given contact with another contact regarding a defined
*              subset of contact attributes, e.g. firstName, lastName, ..
*              The used subset correspond mainly to the APL1 attributes of a vCard.
*
*              Note: Attributes like ContactHandle, photoDlSize, Photo, .. are not
*                    compared/checked.
*
* PARAMETER: IN: contact
*
* RETURNVALUE: Indicates if the contacts are "matching" (true) or not (false).
*
*******************************************************************************/
bool FC_PhoneBook_Contact_Detail::bSubsetCompare(const FC_PhoneBook_Contact_Detail& contact)
{
   bool blRet = false;

   if((!firstName.compare(contact.firstName))&&
      (!lastName.compare(contact.lastName))&&
      (!Phonetics.compare(contact.Phonetics))&& //292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
      (!poiName.compare(contact.poiName))&&
      (!PhoneNum1.compare(contact.PhoneNum1))&&
      (!PhoneNum2.compare(contact.PhoneNum2))&&
      (!PhoneNum3.compare(contact.PhoneNum3))&&
      (!PhoneNum4.compare(contact.PhoneNum4))&&
      (!PhoneNum5.compare(contact.PhoneNum5))&&
      (!PhoneNum6.compare(contact.PhoneNum6))&&
      (!PhoneNum7.compare(contact.PhoneNum7))&&
      (!PhoneNum8.compare(contact.PhoneNum8))&&
      (NumberType1==contact.NumberType1)&&
      (NumberType2==contact.NumberType2)&&
      (NumberType3==contact.NumberType3)&&
      (NumberType4==contact.NumberType4)&&
      (NumberType5==contact.NumberType5)&&
      (NumberType6==contact.NumberType6)&&
      (NumberType7==contact.NumberType7)&&
      (NumberType8==contact.NumberType8)&&
      (!emailAdd1.compare(contact.emailAdd1))&&
      (!emailAdd2.compare(contact.emailAdd2))&&
      (!emailAdd3.compare(contact.emailAdd3))&&
      (!homeAdd.compare(contact.homeAdd))&&
      (!workAdd.compare(contact.workAdd))&&
      (!otherAdd.compare(contact.otherAdd)))
//      (!category.compare(contact.category))&&
//      (!notes.compare(contact.notes)))
   {
      blRet= true;
      ETG_TRACE_USR4((" Contacts are 'matching' (based on subset check). "));
   }
   else
   {
      ETG_TRACE_USR4((" Contacts are NOT 'matching' (based on subset check). " ));
   }

   return blRet;
}


/***************************************************************************************
*
* FUNCTION:  FC_PhoneBook_SQLite::FindExactMatchingContact
*
* DESCRIPTION: Finds a exact matching contact from the database for the given devicehandle
*
*    Note: This method will be called when parsing vCards from a connected 'Phone device'.
*          The database query to check if a contact is already in the database, is
*          optimized with respect to the contact attributes provided by a phone device.
*          (Hint: Therefore this method is not intended to be used for 'Vechicle Contacts'
*           as these contacts have additional attributes like 'category' and 'notes' ..)
*
* PARAMETER: IN   ContactDetail: Contact to be searched in the database
*            IN   u8DeviceHandle of connected phone
*            OUT  ContactHandle of the matching contact
*            OUT  bPhotoMatches: true if Photos also match, false otherwies
*            OUT  u32photoSizeOld: Photo Size of matching contact in DB
*
* RETURNVALUE: true if a matching contact (based on APL1 attributes) hs been found,
*              false otherwise
*
*****************************************************************************************/
bool FC_PhoneBook_SQLite::FindExactMatchingContact(
   const FC_PhoneBook_Contact_Detail& ContactDetail, quint8 u8DeviceHandle,
   quint32& u32ContactHandle, bool& bPicStatus, quint32& u32photoSizeOld)
{
   ETG_TRACE_USR4(( " FindExactMatchingContact entered " ));

   bool blContactFound = false;

   QTime DebugTime;
   DebugTime.start();

   u32ContactHandle = 0;
   bPicStatus       = false;
   u32photoSizeOld  = 0;

   if(!CHECK_VALID(u8DeviceHandle))
   {
      if (u8DeviceHandle != FC_PB_DEVICE_TMP)   // CMG3G-7702 Incorporating changes for PhoneBook_Tmp table.
      {
         ETG_TRACE_USR4((" FC_PhoneBook_SQLite: device handle is invalid "));
         return false;
      }
   }

   QMutexLocker lock(&m_mutex);
   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);
   QString sql;

   sql = "SELECT "                              // -FastContactSearch- : Query reworked
         FC_PB_COL_CONTACT_HANDLE ", "

         FC_PB_COL_FIRST_NAME     ", "
         FC_PB_COL_LAST_NAME      ", "
         FC_PB_COL_PHONETICS      ", "//292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
         FC_PB_COL_POI_NAME       ", "
         FC_PB_COL_PHONENUM1       ", "
         FC_PB_COL_PHONENUM2      ", "
         FC_PB_COL_PHONENUM3      ", "
         FC_PB_COL_PHONENUM4      ", "
         FC_PB_COL_PHONENUM5      ", "
         FC_PB_COL_PHONENUM6      ", "
         FC_PB_COL_PHONENUM7      ", "
         FC_PB_COL_PHONENUM8      ", "
         FC_PB_COL_NUMTYPE1       ", "
         FC_PB_COL_NUMTYPE2       ", "
         FC_PB_COL_NUMTYPE3       ", "
         FC_PB_COL_NUMTYPE4       ", "
         FC_PB_COL_NUMTYPE5       ", "
         FC_PB_COL_NUMTYPE6       ", "
         FC_PB_COL_NUMTYPE7       ", "
         FC_PB_COL_NUMTYPE8       ", "
         FC_PB_COL_EMAIL_ADD1     ", "
         FC_PB_COL_EMAIL_ADD2     ", "
         FC_PB_COL_EMAIL_ADD3     ", "
         FC_PB_COL_HOME_ADD       ", "
         FC_PB_COL_WORK_ADD       ", "
         FC_PB_COL_OTHER_ADD      ", "

         FC_PB_COL_PHOTO_DL_SIZE  ", "
         // FC_PB_COL_PHOTO_TYPE  ", "    // -FastPhotoHandling-    // 'photoType' NOT needed in this context
         FC_PB_COL_PHOTO_DL_CRC   ", "    // -FastPhotoHandling-
         FC_PB_COL_PHOTO_SIZE     ", "    // -FastPhotoHandling-
         // FC_PB_COL_PHOTO_CRC   ", "    // -FastPhotoHandling-    // 'photoCRC' NOT needed in this context
         // FC_PB_COL_APL1_SIZE   ", "  // -FastContactSearch-   // 'APL1Size' NOT needed in this context
         FC_PB_COL_CRC_APL1       " "   // -FastContactSearch-
         "FROM " + fc_pb_aTableNames[u8DeviceHandle] +
         " WHERE " FC_PB_COL_CRC_APL1 " = " + QString::number(ContactDetail.crcAPL1);

   //Fix for GMMY15-10814
   if(db.isOpen() && query.exec(sql))
   {
      while(db.isOpen() && query.next())
      {
         FC_PhoneBook_Contact_Detail contact;
         quint16 counter= 0;

         contact.contactHandle = query.value( counter++ ).toUInt();

         contact.firstName     = query.value( counter++ ).toString();
         contact.lastName      = query.value( counter++ ).toString();
         contact.Phonetics      = query.value( counter++ ).toString();//292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
         contact.poiName       = query.value( counter++ ).toString();
         contact.PhoneNum1       = query.value( counter++ ).toString();
         contact.PhoneNum2      = query.value( counter++ ).toString();
         contact.PhoneNum3      = query.value( counter++ ).toString();
         contact.PhoneNum4      = query.value( counter++ ).toString();
         contact.PhoneNum5      = query.value( counter++ ).toString();
         contact.PhoneNum6      = query.value( counter++ ).toString();
         contact.PhoneNum7      = query.value( counter++ ).toString();
         contact.PhoneNum8      = query.value( counter++ ).toString();
         contact.NumberType1    = (query.value(counter++)).value<quint8>();
         contact.NumberType2    = (query.value(counter++)).value<quint8>();
         contact.NumberType3    = (query.value(counter++)).value<quint8>();
         contact.NumberType4    = (query.value(counter++)).value<quint8>();
         contact.NumberType5    = (query.value(counter++)).value<quint8>();
         contact.NumberType6    = (query.value(counter++)).value<quint8>();
         contact.NumberType7    = (query.value(counter++)).value<quint8>();
         contact.NumberType8    = (query.value(counter++)).value<quint8>();
         contact.emailAdd1     = query.value( counter++ ).toString();
         contact.emailAdd2     = query.value( counter++ ).toString();
         contact.emailAdd3     = query.value( counter++ ).toString();
         contact.homeAdd       = query.value( counter++ ).toString();
         contact.workAdd       = query.value( counter++ ).toString();
         contact.otherAdd      = query.value( counter++ ).toString();

         contact.photoDlSize   = query.value( counter++ ).toUInt();
         // contact.photoType  = query.value( counter++ ).toString();  // -FastPhotoHandling-   // 'photoType' NOT needed in this context
         contact.photoDlCRC    = (query.value(counter++)).value<quint16>();    // -FastPhotoHandling-
         contact.photoSize     = query.value( counter++ ).toUInt();    // -FastPhotoHandling-   // 'photoSize' is a needed "return" value.
         // contact.photoCRC   = query.value( counter++ ).toUInt();    // -FastPhotoHandling-   // 'photoCRC' NOT needed in this context
         // contact.APL1Size   = query.value( counter++ ).toUInt();  // -FastContactSearch-   // 'APL1Size' NOT needed in this context
         // contact.crcAPL1    = query.value( counter++ ).toUInt();  // -FastContactSearch-   // 'crcAPL1' NOT needed in this context

         if( true == contact.bSubsetCompare(ContactDetail) )
         {
            ETG_TRACE_USR4(( " Contacts are equal regarding to 'relevant' APL1 attributes. (Note: PHOTO details are not checked so far.) " ));
            blContactFound   = true;
            u32ContactHandle = contact.contactHandle;
            u32photoSizeOld  = contact.photoSize;

            ETG_TRACE_USR4(( " DB contact.photoDlSize= %5u, Downloaded ContactDetail.photoDlSize [byte]= %5u ", contact.photoDlSize, ContactDetail.photoDlSize ));
            ETG_TRACE_USR4(( " DB contact.photoDlCRC=  %u,  Downloaded ContactDetail.photoDlCRC= %u ", contact.photoDlCRC,  ContactDetail.photoDlCRC ));
            if(contact.photoDlSize == ContactDetail.photoDlSize)
            {
               ETG_TRACE_USR4(( " PHOTOs are equal (based on their size), now check their CRC hash " ));

               // Now check equivalence of 'PhotoDlCRC'
               if( contact.photoDlCRC == ContactDetail.photoDlCRC)
               {
                  bPicStatus = true;
                  break;
               }
               else
               {
                  ETG_TRACE_USR4(( " PHOTOs are NOT equal (based on their CRC hashes)" ));
                  bPicStatus = false;
                  break;
               }
            }
            else
            {
               ETG_TRACE_USR4(( " PHOTOs are NOT equal (based on their sizes)" ));
               bPicStatus = false;
               break;
            }
         }
      }
   }
   else
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(("FC_PhoneBook_SQLite:: Sql query failed"));
      blContactFound = false;
   }

   if(!blContactFound)
   {
      ETG_TRACE_USR4(( "No contact found with same 'relevant' APL1 attributes. " ));
   }
   else
   {
      ETG_TRACE_USR2((" Matching contact found, ContactHandle= %d ", u32ContactHandle ));
   }

   ETG_TRACE_USR2(( " -PBDL.Perf- 'FindExactMatchingContact': exited after %d ms, [DH= %u] ", DebugTime.elapsed(), u8DeviceHandle ));
   return blContactFound;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::DeleteContactHandles
* DESCRIPTION: Deletes all the entries corresponding to the given contact handle
* list for a given device from table PhoneBook_DHx and table PhoneBook_Master
*
* PARAMETER:  [IN]  - u8DeviceHandle, ContactHandleList
*             [OUT] - none
*
* RETURNVALUE: bool - indicating success or failure of this function
*******************************************************************************/
bool FC_PhoneBook_SQLite::DeleteContactHandles(quint8 u8DeviceHandle,
                                               QList<quint32> ContactHandleList)
{
   ETG_TRACE_USR4(("DeleteContactHandles entered"));

   if(!CHECK_VALID(u8DeviceHandle))
   {
      ETG_TRACE_USR4(( " FC_PhoneBook_SQLite: device handle is invalid in function 'DeleteContactHandles' " ));
      return false;
   }

   QMutexLocker lock(&m_mutex);
   QSqlDatabase db = Database();

   if(db.isOpen())
   {
      // Start transaction as we execute queries in 2 tables (and in chunks)
      db.transaction();
   }
   else
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);
   bool blRet = false;
   bool bChunkIsComplete = false;
   quint16 u16ListSize= (quint16)(ContactHandleList.size());
   //Fix: SUZUKI-20814: Reducing the query execution time so that the entry thread is not blocked during low voltage scenario.
   QString qsChunkINList = " IN (";

   ETG_TRACE_USR2(( " ContactHandleList size = %d ", u16ListSize ));

// Deactivating the following code snippet which can be used in future for time calculations.
#if 0  // Measure Deletion Time
      QTime DebugTime;
      DebugTime.start();

      struct timespec time1     = {0, 0};
      struct timespec time2     = {0, 0};
      struct timespec diff_time = {0, 0};

      clock_gettime(CLOCK_MONOTONIC, &time1);
#endif

   for(quint16 u16Index = 0; u16Index < u16ListSize; u16Index++)
   {
      // Check if query chunk has enough contact handles to exeute query
      bChunkIsComplete = (((u16Index + 1) % CHUNK_DELETE) == 0) || (u16Index == (u16ListSize - 1));

      if( bChunkIsComplete == false )
      {  // Append contact handle with a "," to query chunk "IN list"
         qsChunkINList += QString::number(ContactHandleList.at(u16Index)) + ",";
      }
      else
      {  // Append contact handle to query chunk, finalize "IN list" with ")" and execute query
         qsChunkINList += QString::number(ContactHandleList.at(u16Index)) + ")";

         QString sql1 = "DELETE FROM " + fc_pb_aTableNames[u8DeviceHandle] +
                        " WHERE " FC_PB_COL_CONTACT_HANDLE + qsChunkINList;

         QString sql2 = "DELETE FROM " FC_PB_MASTER_TABLE_NAME " WHERE "
                        FC_PB_MASTER_COL_TABLE_INDEX " = " + QString::number(u8DeviceHandle) +
                        " AND " FC_PB_COL_CONTACT_HANDLE + qsChunkINList;

         //Fix for GMMY15-10814
         if(db.isOpen())
         {
            if(query.exec(sql1))     // Delete contacts from related Device Table
            {
               blRet = true;
               if(query.exec(sql2))  // Delete contact handle entries from Master Table
               {
                  blRet = true;
               }
               else
               {
                  PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

                  blRet = false;
                  ETG_TRACE_ERR(( " Failed in removing contact handles from Master Table " ));
                  break;
               }

               ETG_TRACE_USR2(( " Chunk of contacts removed successfully, DeviceHandle= %d ",
                                u8DeviceHandle ));
               // Re-intializes chunk "IN list"
               qsChunkINList = " IN (";
            }
            else
            {
               PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

               blRet = false;
               ETG_TRACE_ERR(( " Failed in removing contact handles from Device Table " ));
               break;
            }
         }
         else
         {
            blRet = false;
            ETG_TRACE_ERR(("Database is not open"));

            break;
         }
      }
   }

   if(!blRet)
   {
      db.rollback();
      ETG_TRACE_ERR((" Database rolled back " ));
   }
   else
   {
      if(!db.commit())
      {
         db.rollback();
         ETG_TRACE_ERR(( " Database rolled back since commit failed " ));
      }
      else
      {
         ETG_TRACE_USR1(( " -PBDL.S- Database committed, number of deleted contacts= %d",
                          u16ListSize ));
      }
   }

// Deactivating the following code snippet which can be used in future for time calculations.
#if 0  // Measure Deletion Time
      ETG_TRACE_COMP(( " Total time elapsed for deletion (QTime measurement): %d ms ", DebugTime.elapsed() ));
      // Note: The QTime measurement is not reliable, if the system's clock setting
      // has been changed since the last time start(); the result then is undefined.

      clock_gettime(CLOCK_MONOTONIC, &time2);

      if( (time2.tv_nsec - time1.tv_nsec) < 0 )
      {
         diff_time.tv_sec  = time2.tv_sec  - time1.tv_sec-1;
         diff_time.tv_nsec = 1000000000 + time2.tv_nsec - time1.tv_nsec;
      } else {
         diff_time.tv_sec  = time2.tv_sec  - time1.tv_sec;
         diff_time.tv_nsec = time2.tv_nsec - time1.tv_nsec;
      }

      ETG_TRACE_COMP(( " Contact deletion : Number of contacts= %d, used Time[msec]= %5d ",
                       u16ListSize, diff_time.tv_sec *1000 + diff_time.tv_nsec / 1000000 ));
#endif

   return blRet;
}
// End of FIX SUZUKI-20814


// CMG3G-7702
/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::InsertTmpToDeviceTable
* DESCRIPTION: Inserts all the entries from PhoneBook_Tmp to the corresponding
*              device table of the connected device.
*
* PARAMETER:  [IN]  - u8DeviceHandle
*             [OUT] - none
*
* RETURNVALUE: bool - indicating success or failure of this function
*******************************************************************************/
bool FC_PhoneBook_SQLite::InsertTmpToDeviceTable(quint8 u8DeviceHandle)
{
   ETG_TRACE_USR4((" InsertTmpToDeviceTable entered "));

   QMutexLocker lock(&m_mutex);
   QSqlDatabase db = Database();

   if (db.isOpen())
   {
      // Start transaction for safer updations of data between multiple tables
      db.transaction();
   }
   else
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);
   bool blRet = false;

   QString sql1 = "INSERT INTO " + fc_pb_aTableNames[u8DeviceHandle]
         + " SELECT * FROM " + fc_pb_aTableNames[FC_PB_DEVICE_TMP];

   QString
         sql2 =
               "INSERT INTO " FC_PB_MASTER_TABLE_NAME "(" FC_PB_COL_CONTACT_HANDLE ") SELECT "
               FC_PB_COL_CONTACT_HANDLE " FROM "
                     + fc_pb_aTableNames[FC_PB_DEVICE_TMP];

   QString
         sql3 =
               "UPDATE " FC_PB_MASTER_TABLE_NAME " SET " FC_PB_MASTER_COL_TABLE_INDEX " = "
                     + QString::number(u8DeviceHandle)
                     + " WHERE " FC_PB_MASTER_COL_TABLE_INDEX " IS NULL";

   if (db.isOpen())
   {
      if (query.exec(sql1)) // Insert records in Phonebook_Tmp table to the connected Device Table
      {
         blRet = true;
         if (query.exec(sql2)) // Insert ContactHandles into Master Table
         {
            blRet = true;
            if (query.exec(sql3)) // Insert DeviceHandles for the added Contacthandles into Master Table
            {
               blRet = true;
            }
            else
            {
               PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

               blRet = false;
               ETG_TRACE_ERR((" Failed in inserting DeviceHandles for ContactHandles into Master Table "));
            }
            ETG_TRACE_USR2((" Insertion of records from Tmp to Device table done successfully"));
         }
         else
         {
            PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

            blRet = false;
            ETG_TRACE_ERR((" Failed in inserting ContactHandles into Master Table "));
         }
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         blRet = false;
         ETG_TRACE_ERR((" Failed in insertion of records from Tmp to Device table"));
      }
   }
   else
   {
      blRet = false;
      ETG_TRACE_ERR(("Database is not open"));
   }

   if (!blRet)
   {
      db.rollback();
      ETG_TRACE_ERR((" Database rolled back "));
   }
   else
   {
      if (!db.commit())
      {
         db.rollback();
         ETG_TRACE_ERR((" Database rolled back since commit failed "));
      }
      else
      {
         ETG_TRACE_USR1((" -PBDL.S- Database committed, Contacts in Tmp inserted into Device Table"));
         blRet = true;
      }
   }

   return blRet;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::DropTableFromDB
* DESCRIPTION: Drops the table indicated by the table index from the PhoneBook DB
*
* PARAMETER:  [IN]  - u8TableIndex
*             [OUT] - none
*
* RETURNVALUE: bool - indicating success or failure of this function
*******************************************************************************/
bool FC_PhoneBook_SQLite::DropTableFromDB(quint8 u8TableIndex)
{
   ETG_TRACE_USR4((" DropTableFromDB entered for TableIndex: %d", u8TableIndex));
   bool bRetValue = false;

   QSqlDatabase db = Database();

   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);


      QString sql = "DROP TABLE IF EXISTS " + fc_pb_aTableNames[u8TableIndex];
      ETG_TRACE_USR4(( " Query used to DropTableFromDB: '%s' ", sql.toLatin1().constData() ));


      if (query.exec(sql)) // Drop Table
      {
         bRetValue = true;
         ETG_TRACE_USR4((" DropTableFromDB SUCCESS for TableIndex: %d", u8TableIndex));
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" Failed in Dropping the Table "));
      }
   }

   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::DropVPBTableFromDB
* DESCRIPTION: Drops the table indicated by the table index from the Vehicle PhoneBook DB
*
* PARAMETER:  [IN]  - u8TableIndex
*             [OUT] - none
*
* RETURNVALUE: bool - indicating success or failure of this function
*******************************************************************************/
bool FC_PhoneBook_SQLite::DropVPBTableFromDB(quint8 u8TableIndex)
{
   ETG_TRACE_USR4((" DropTableFromDB entered for TableIndex: %d", u8TableIndex));
   bool bRetValue = false;

   QSqlDatabase db = Database();

   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);
      QString sql = "DROP TABLE " + fc_vpb_aTableNames[u8TableIndex];
      ETG_TRACE_USR4(( " Query used to DropTableFromDB: '%s' ", sql.toLatin1().constData() ));

      if (query.exec(sql)) // Drop Table
      {
         bRetValue = true;
         ETG_TRACE_USR4((" DropTableFromDB SUCCESS for TableIndex: %d", u8TableIndex));
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" Failed in Dropping the Table "));
      }
   }

   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::DeleteContentsOfTable
* DESCRIPTION: Deletes the contents of the table indicated by the table index
*
* PARAMETER:  [IN]  - u8TableIndex
*             [OUT] - none
*
* RETURNVALUE: bool - indicating success or failure of this function
*******************************************************************************/
bool FC_PhoneBook_SQLite::DeleteContentsOfTable(quint8 u8TableIndex)
{
   ETG_TRACE_USR4((" DeleteContentsOfTable entered for TableIndex: %d", u8TableIndex));
   bool bRetValue = false;
   QMutexLocker lock(&m_mutex);
   QSqlDatabase db = Database();

   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);


      QString sql = "DELETE FROM " + fc_pb_aTableNames[u8TableIndex];

      if (db.isOpen())
      {
         if (query.exec(sql)) // Delete Contents from Table
         {
            bRetValue = true;
            ETG_TRACE_USR4((" DeleteContentsOfTable SUCCESS for TableIndex: %d", u8TableIndex));
         }
         else
         {
            PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

            ETG_TRACE_ERR((" Failed in Deleting the contents of the Table "));
         }
      }
      else
      {
         ETG_TRACE_ERR(("Database is not open"));
      }
   }
   return bRetValue;
}
// End of CMG3G-7702


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetContactHandles
* DESCRIPTION: Returns all contact handles associated with a device
*
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE: bool - indicating success or failure of this function
*******************************************************************************/
bool FC_PhoneBook_SQLite::GetContactHandles(quint8 u8DeviceHandle,
                                            QList<quint32>& u32ContactHandleList)
{
   ETG_TRACE_USR4(("GetContactHandles entered"));

   QMutexLocker lock(&m_mutex);
   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);

   bool blRet= false;

   if(!CHECK_VALID(u8DeviceHandle))
   {
      ETG_TRACE_ERR(("FC_PhoneBook_SQLite: device handle is invalid"));
      return false;
   }

   QString sql = "SELECT " FC_PB_COL_CONTACT_HANDLE " FROM "
      + fc_pb_aTableNames[u8DeviceHandle];

   if (query.exec(sql))
   {
      blRet= true;
      u32ContactHandleList.clear();
      while (query.next())
      {
         //Fix for GMMY15-10814
         if(db.isOpen())
         {
            u32ContactHandleList << query.value(0).toUInt();
         }
         else
         {
            ETG_TRACE_ERR(("Database is not open"));
            return false;
         }
      }

      ETG_TRACE_COMP(( " -PBDL.S- DeviceHandle= %d, NumberOfContacts= %d ", u8DeviceHandle, u32ContactHandleList.size() ));
   }
   else
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query failed"
         " in GetContactHandles"));
   }

   return blRet;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::DeleteCallHistoryRecords
* DESCRIPTION: Deletes all call history entries corresponding to the given connection ID
*
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE: bool - indicating success or failure of this function
*******************************************************************************/
bool FC_PhoneBook_SQLite::DeleteCallHistoryRecords(quint8 u8ConnectionID)
{
   ETG_TRACE_USR4(("DeleteCallHistoryRecords entered"));

   QMutexLocker lock(&m_mutex);
   QSqlDatabase db= Database();
   if(!db.isOpen())
   {

      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }
   QSqlQuery query(db);

   bool blRet= false;

   if(!CHECK_CH_VALID(u8ConnectionID))
   {
      ETG_TRACE_ERR((" FC_PhoneBook_SQLite: ConnectionID is invalid "));
      return false;
   }

   QString sql= "DELETE FROM " + fc_pb_aTableNames[MAP_TO_CH_INDEX(u8ConnectionID)];

   if(query.exec(sql))
   {
      sql.clear();
      sql= "DELETE FROM " FC_PB_MASTER_TABLE_NAME " WHERE " FC_PB_MASTER_COL_TABLE_INDEX
         " = " + QString::number(MAP_TO_CH_INDEX(u8ConnectionID));
      if(query.exec(sql))
      {
         ETG_TRACE_USR4((" FC_PhoneBook_SQLite:- Call history record deleted successfully"));
         blRet= true;
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" FC_PhoneBook_SQLite:- Error - failed in deleting call history"
            "record from the master table"));
      }

      for(quint8 u8Index= OSAL_NULL; u8Index < FC_PB_DNL_CALL_HISTORY_TYPES; u8Index++)
      {
         u32LastCallHistoryCntHandle[u8ConnectionID-1][u8Index]= OSAL_NULL;
      }
   }
   else
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR((" FC_PhoneBook_SQLite:- Error - failed in deleting call history"
         "record from the table"));
   }

   return blRet;

}
/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_SQLite::UpdateMasterTable
*
* DESCRIPTION: Update's fields in the mastertable
*
*
*
* PARAMETER:

* RETURNVALUE: Result whether success or failure
*
*******************************************************************************/

bool FC_PhoneBook_SQLite::UpdateMasterTable(const quint8 TableIndex, quint32& ContactHandle)
{
   ETG_TRACE_USR4(("UpdateMasterTable entered"));
   QMutexLocker lock(&m_mutex);//GMMY17-17143
   bool bRetValue = false;
   QString sql;
   QSqlDatabase db= Database();

   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);

      if(ContactHandle == 0)
      {
         ETG_TRACE_ERR((" Contact handle is null   "));
      }
      else
      {
         sql.clear();

         sql= "UPDATE " FC_PB_MASTER_TABLE_NAME  " SET "
            FC_PB_MASTER_COL_TABLE_INDEX" = " +QString::number(TableIndex)+ " WHERE "
            FC_PB_MASTER_COL_CONTACT_HANDLE" = " + QString::number(ContactHandle);

         if(!query.exec(sql))
         {
            PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

            ETG_TRACE_ERR((" Error in UpdateMasterTable:- Updating a record in master table failed  "));
         }
         else
            bRetValue = true;
      }
   }

   return bRetValue;
}



/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetPhoto
* DESCRIPTION: Provides the photo associated with the contact handle
*              passed
*
* PARAMETER:  [IN]  - Contact Handle
*             [OUT] - Photo
*             [OUT] - qsPhotoType
*             [OUT] - u16ErrorCode
*
* RETURNVALUE: bool - indicating success or failure of this function
*******************************************************************************/
bool FC_PhoneBook_SQLite::GetPhoto( quint32 u32ContactHandle, QByteArray& PhotoData,
                                    QString& qsPhotoType, quint16& PhotoDlCRC, quint16& u16ErrorCode )
{
   ETG_TRACE_USR4(( " GetPhoto entered " ));

   QMutexLocker lock(&m_mutex);
   QSqlDatabase db = Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(( " Database is not open " ));
      return false;
   }
   QSqlQuery query(db);

   bool blRet = true;

   QString sql = "SELECT " FC_PB_MASTER_COL_TABLE_INDEX " FROM "
      FC_PB_MASTER_TABLE_NAME " WHERE " FC_PB_MASTER_COL_CONTACT_HANDLE " = "+QString::number(u32ContactHandle);

   ETG_TRACE_USR2((" FC_PhoneBook_SQLite: String used for master table in GetPhoto - %s ",
      sql.toLatin1().constData()));

   if(!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR((" FC_PhoneBook_SQLite: GetPhoto failed in master table"));
      blRet = false;
      u16ErrorCode = FC_PB_ERROR_INTERNAL_FAILURE;
   }
   else
   {
      if(!query.next())
      {
         ETG_TRACE_ERR(( " FC_PhoneBook_SQLite: GetPhoto - No entry found for the contact handle" ));
         blRet = false;
         u16ErrorCode= FC_PB_ERROR_CONTACT_HANDLE_INVALID;
      }
      else
      {
         quint8 tableindex = (query.value(0)).value<quint8>();

         if(!CHECK_VALID(tableindex))
         {
            ETG_TRACE_ERR(( " FC_PhoneBook_SQLite: Table Index - %d is invalid  ", tableindex ));
            blRet= false;
            u16ErrorCode= FC_PB_ERROR_INTERNAL_FAILURE;
         }
         else
         {
            sql.clear();
            //FIX CMG3G-8372 IS2430_SetContactPhoto@FC_Phonebook
            if( ((u32ContactHandle & FC_PB_CONTACT_HANDLE_DPB_MARKER) == FC_PB_CONTACT_HANDLE_DPB_MARKER) || (tableindex == FC_PB_VEHICLE_PBOOK) )
            {
               ETG_TRACE_USR4((" FC_PhoneBook_SQLite: Marked bit is set to one, hence writing query for device phonebook"));
               // Get Photo and PhotoType for ContactHandle (from table of given DeviceHandle)
               sql = "SELECT " FC_PB_COL_PHOTO ", "
               FC_PB_COL_PHOTO_TYPE ", "
               FC_PB_COL_PHOTO_DL_CRC " FROM " + fc_pb_aTableNames[tableindex] + " WHERE " FC_PB_COL_CONTACT_HANDLE " = " + QString::number(u32ContactHandle);
            }
            else
            {
               ETG_TRACE_USR4(( " FC_PhoneBook_SQLite: Marked bit is set to zero, hence writing query for vehicle phonebook" ));
               // Get Photo and PhotoType for ContactHandle (from table of given DeviceHandle)
               sql = "SELECT " FC_PB_COL_PHOTO ", "
               FC_PB_COL_PHOTO_TYPE ", "
               FC_PB_COL_PHOTO_DL_CRC " FROM " + fc_vpb_aTableNames[tableindex] + " WHERE " FC_PB_COL_CONTACT_HANDLE " = " + QString::number(u32ContactHandle);
            }


            ETG_TRACE_USR2(( " FC_PhoneBook_SQLite: String used in GetPhoto - %s ",sql.toLatin1().constData() ));

            if(!query.exec(sql))
            {
               PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

               ETG_TRACE_ERR(( " FC_PhoneBook_SQLite: GetPhoto Query Failed  in device table " ));
               blRet = false;
               u16ErrorCode = FC_PB_ERROR_INTERNAL_FAILURE;
            }
            else
            {
               if(query.next())
               {
                  quint8 u8Counter = 0;

                  PhotoData   = query.value(u8Counter++).toByteArray();
                  qsPhotoType = query.value(u8Counter++).toString();
                  PhotoDlCRC  = (query.value(u8Counter++)).value<quint16>();

                  ETG_TRACE_USR4(( " Photo retrieved: PhotoDlCRC= %d, PhotoType= '%s' ",
                                   PhotoDlCRC, qsPhotoType.toUtf8().constData() ));
               }
            }
         }  // end of: if(!CHECK_VALID(tableindex))

      }
   }

   return blRet;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetDeviceHandleForContacthandle
* DESCRIPTION: GetDeviceHandle For Contacthandle
*
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE:
*******************************************************************************/
quint8 FC_PhoneBook_SQLite::GetDeviceHandleForContacthandle(quint32 u32ContactHandle)
{
   ETG_TRACE_USR4(( " GetDeviceHandleForContacthandle entered with ContactHandle %d ", u32ContactHandle ));
   QMutexLocker lock(&m_mutex);//GMMY17-17143
   quint8 deviceHandle;
   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }
   QSqlQuery query(db);


   ETG_TRACE_USR4(("FC_PhoneBook_SQLite:GetDeviceHandleForContacthandle: entered"));

   QString sql = "SELECT " FC_PB_MASTER_COL_TABLE_INDEX " FROM "
      FC_PB_MASTER_TABLE_NAME " WHERE " FC_PB_MASTER_COL_CONTACT_HANDLE " = "+QString::number(u32ContactHandle);


   ETG_TRACE_USR2(("FC_PhoneBook_SQLite: String used for master table in Get Contact details - %s ",
      sql.toLatin1().constData()));

   if(!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_USR4(("FC_PhoneBook_SQLite: Get contact details failed in master table"));
      return false;
   }

   if(!query.next())
   {
      ETG_TRACE_USR4(("FC_PhoneBook_SQLite: Get contact details - No entry found for the contact handle"));
      return false;
   }

   quint8 tableindex= (query.value(VALUE_ZERO)).value<quint8>();

   if(!CHECK_VALID(tableindex))
   {
      ETG_TRACE_USR4(("FC_PhoneBook_SQLite: device handle is invalid "));
      return false;
   }

   deviceHandle = tableindex;

   ETG_TRACE_USR4(("FC_PhoneBook_SQLite:GetDeviceHandleForContacthandle:   exit "));

   return deviceHandle;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::SearchPhoneNumberDigits(
*
* DESCRIPTION:
*              Search for Phone Number from Call History Database
*
*
* PARAMETER:  [IN]  - PhoneNumberdigits to search
*             [OUT] -
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::SearchPhoneNumberEntries(QString PhNumberToSearch,quint8 Index,quint8 ConnectionID)
{
   ETG_TRACE_USR4(("SearchPhoneNumberEntries entered with Index %d", Index));

   QMutexLocker lock(&m_mutex);
   QSqlDatabase db= Database();
   QVector <QString>alreadyAdded;
   if(!db.isOpen())
   {

      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);

   AutoCompletePhoneNumberEntryResult PhNoContactsInfo;
   //Which Table to refer to get the Callhistory records
   SortType enSorttype = DateTime;

   if(!CHECK_CH_VALID(ConnectionID))
   {
      ETG_TRACE_USR4(( "FC_PhoneBook_SQLite: ConnectionID is invalid "));
      return false;
   }

   // Get the contact details for the given phone number
   QString sql = "SELECT * FROM "+ fc_pb_aTableNames[MAP_TO_CH_INDEX(ConnectionID)] +" WHERE "
      "" FC_PB_COL_CH_PHONE_NUM " LIKE '"+PhNumberToSearch+"%'";
   sql+= " ORDER BY " ;

   switch(enSorttype)
   {
   default:
   case DateTime:
      {
         sql += FC_PB_COL_CH_CALL_DATE" COLLATE NOCASE DESC, " FC_PB_COL_CH_CALL_TIME" COLLATE NOCASE DESC";
         break;
      }
   }

   ETG_TRACE_USR2(( "FC_PhoneBook_SQLite: Query to Gett Phone Nos - %s ",sql.toUtf8().constData()));

   if(!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_USR4(( "FC_PhoneBook_SQLite: SearchPhoneNumberDigits Query Failed from retrieving matched records from call history table"));
      return false;
   }
   else
   {
      ETG_TRACE_USR4(( "FC_PhoneBook_SQLite: AutoCompletePhoneNumberEntry Details Retrieved from Call History Database"));
   }

   AutoCompletePhNoSearchRecords[Index].clear();
   SearchedPhoneNumberDigit[Index] = PhNumberToSearch;
   while(query.next())
   {
      int counter = 2;
      PhNoContactsInfo.FirstName     = query.value(counter++).toString();
      PhNoContactsInfo.LastName      = query.value(counter++).toString();
#ifdef PB_PHONENUMBERS_IN_PRETTY_PRINT
      counter++;
#endif
      PhNoContactsInfo.PhoneNumber   = query.value(counter++).toString();
      if(alreadyAdded.contains(PhNoContactsInfo.PhoneNumber) == false)
      {
         ETG_TRACE_USR2(("NUMBER Not ALREADY ADDED IN SERACH LIST= %s",PhNoContactsInfo.PhoneNumber.toLatin1().constData()));
         AutoCompletePhNoSearchRecords[Index] << PhNoContactsInfo;
         ETG_TRACE_USR4(("PUSHED DATA"));
         alreadyAdded.push_back(PhNoContactsInfo.PhoneNumber);
      }
      else
      {
         ETG_TRACE_USR2(("NUMBER ALREADY ADDED IN SERACH LIST= %s",PhNoContactsInfo.PhoneNumber.toLatin1().constData()));
      }
   }

   ETG_TRACE_USR2(( "Size = %d",AutoCompletePhNoSearchRecords[Index].size()));
   ETG_TRACE_USR2(( "PhNoContactsInfo.PhoneNumberToSearch = %s",SearchedPhoneNumberDigit[Index].toUtf8().constData()));

   if(AutoCompletePhNoSearchRecords[Index].empty())
   {
      ETG_TRACE_USR4(( "FC_PhoneBook_SQLite: No matching entry found in Call History Records"));
      return false;
   }

   return true;
}

/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::SpeechCreateCheckSumTable
*
* DESCRIPTION: Creates the checksum table required for the Speech
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE:
*
******************************************************************************/
bool FC_PhoneBook_SQLite::SpeechCreateCheckSumTable()
{
   ETG_TRACE_USR4(("SpeechCreateCheckSumTable entered "));
   bool bRetValue = false;
   const QString sql = "CREATE TABLE " FC_PB_TABLE_CHECKSUM
      " ( "
      FC_PB_CHKSUM_COL_TABLE_INDEX" INTEGER ,"
      FC_PB_CHKSUM_COL_TABLE_CHKSUM" INTEGER "
      " )";

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {

      QSqlQuery query(db);

      if(!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query failed in Creating CheckSum Table"));
      }
      else
      {
         ETG_TRACE_USR4((" CheckSum Table created successfully "));
         bRetValue = SpeechPopulateCheckSumTable();
      }
   }
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::SpeechVerifyCheckSumTable
*
* DESCRIPTION: Verifies the table schema of checksum table
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE:
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::SpeechVerifyCheckSumTable()
{
   ETG_TRACE_USR4(("SpeechVerifyCheckSumTable entered "));
   bool bRetValue = false;

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {

      QSqlQuery query(db);


      QStringList list_columns;

      if(!query.exec("PRAGMA table_info(" FC_PB_TABLE_CHECKSUM ")"))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("Error:- SQL query failed while validating checksum Table"));
      }
      else
      {
         while(query.next())
            list_columns << query.value(1).toString();
         if( list_columns.contains(FC_PB_CHKSUM_COL_TABLE_INDEX) &&
            list_columns.contains(FC_PB_CHKSUM_COL_TABLE_CHKSUM))
         {
            bRetValue = true;
         }
         else
         {
            ETG_TRACE_ERR(("Checksum table doesn't contain valid table columns"));
         }
      }
   }

   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::SpeechCreateViews
*
* DESCRIPTION: Creates the views required for the Speech
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE:
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::SpeechCreateViews()
{
   ETG_TRACE_USR4(("SpeechCreateViews entered "));
   bool bRetValue = false;

   QSqlQuery query(Database());

   //292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
   QString sql = "CREATE VIEW IF NOT EXISTS Contact AS  SELECT CAST(DeviceHandle AS INTEGER) AS DeviceID, FirstName AS Firstname,"
         " LastName AS Lastname,Phonetics AS Phonetics, PhoneBook_Master.ContactHandle AS ContactID,CAST(Category AS INTEGER) AS Category FROM ("
         " SELECT 2 AS DeviceHandle, ContactHandle, FirstName, LastName, Phonetics, Category FROM PhoneBook_DH2 UNION ALL "
         " SELECT 3 AS DeviceHandle, ContactHandle, FirstName, LastName, Phonetics, Category FROM PhoneBook_DH3 UNION ALL "
         " SELECT 4 AS DeviceHandle, ContactHandle, FirstName, LastName, Phonetics, Category  FROM PhoneBook_DH4 UNION ALL "
         " SELECT 5 AS DeviceHandle, ContactHandle, FirstName, LastName, Phonetics, Category  FROM PhoneBook_DH5 UNION ALL "
         " SELECT 6 AS DeviceHandle, ContactHandle, FirstName, LastName, Phonetics, Category  FROM PhoneBook_DH6 UNION ALL "
         " SELECT 7 AS DeviceHandle, ContactHandle, FirstName, LastName, Phonetics, Category  FROM PhoneBook_DH7 UNION ALL "
         " SELECT 8 AS DeviceHandle, ContactHandle, FirstName, LastName, Phonetics, Category  FROM PhoneBook_DH8 UNION ALL "
         " SELECT 9 AS DeviceHandle, ContactHandle, FirstName, LastName, Phonetics, Category  FROM PhoneBook_DH9 UNION ALL "
         " SELECT 10 AS DeviceHandle, ContactHandle, FirstName, LastName, Phonetics, Category  FROM PhoneBook_DH10 UNION ALL "
         " SELECT 11 AS DeviceHandle, ContactHandle, FirstName, LastName, Phonetics, Category  FROM PhoneBook_DH11 UNION ALL "
         " SELECT 1 AS DeviceHandle, ContactHandle, FirstName, LastName, Phonetics, Category  FROM PhoneBook_VPB UNION ALL "
         " SELECT 1002 AS DeviceHandle, ContactHandle, FirstName, LastName, Phonetics, Category  FROM PhoneBook_VPB2 UNION ALL "
         " SELECT 1003 AS DeviceHandle, ContactHandle, FirstName, LastName, Phonetics, Category  FROM PhoneBook_VPB3 UNION ALL "
         " SELECT 1004 AS DeviceHandle, ContactHandle, FirstName, LastName, Phonetics, Category  FROM PhoneBook_VPB4 UNION ALL "
         " SELECT 1005 AS DeviceHandle, ContactHandle, FirstName, LastName, Phonetics, Category  FROM PhoneBook_VPB5 UNION ALL "
         " SELECT 1006 AS DeviceHandle, ContactHandle, FirstName, LastName, Phonetics, Category  FROM PhoneBook_VPB6 UNION ALL "
         " SELECT 1007 AS DeviceHandle, ContactHandle, FirstName, LastName, Phonetics, Category  FROM PhoneBook_VPB7 UNION ALL "
         " SELECT 1008 AS DeviceHandle, ContactHandle, FirstName, LastName, Phonetics, Category  FROM PhoneBook_VPB8 UNION ALL "
         " SELECT 1009 AS DeviceHandle, ContactHandle, FirstName, LastName, Phonetics, Category  FROM PhoneBook_VPB9 UNION ALL "
         " SELECT 1010 AS DeviceHandle, ContactHandle, FirstName, LastName, Phonetics, Category  FROM PhoneBook_VPB10 UNION ALL "
         " SELECT 1011 AS DeviceHandle, ContactHandle, FirstName, LastName, Phonetics, Category FROM PhoneBook_VPB11) AS all_contacts "
         " JOIN PhoneBook_Master ON all_contacts.ContactHandle = PhoneBook_Master.ContactHandle";
   //end of fix- 292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
   if(!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query failed in Creating view \"Contact\" for speech"));
   }

   else
   {
      ETG_TRACE_USR4(("View \"Contact\" created successfully for speech"));


      sql.clear();
      sql= "CREATE VIEW IF NOT EXISTS Ph_Checksum AS SELECT CASE WHEN ((12 <= TableIndex ) AND (TableIndex <= 21)) THEN (TableIndex -10 + 1000) ELSE TableIndex END  AS DeviceID, CheckSum FROM PhoneBook_CheckSum";

      if(!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query failed in Creating view \"Ph_Checksum\" for speech"));
      }
      else
      {
         ETG_TRACE_USR4(("View \"Ph_Checksum\" created successfully for speech"));
         bRetValue = true;
      }
   }

   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::SpeechPopulateCheckSumTable
*
* DESCRIPTION: Adds default values to the checksum table
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE:
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::SpeechPopulateCheckSumTable()
{
   ETG_TRACE_USR4(("SpeechPopulateCheckSumTable and Populating the checksum table with the initial values"));

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {

      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);

   bool blRet= false;
   const QString sql= "INSERT INTO " FC_PB_TABLE_CHECKSUM " (" FC_PB_CHKSUM_COL_TABLE_INDEX
      "," FC_PB_CHKSUM_COL_TABLE_CHKSUM ") VALUES(?,?)";

   for(quint8 u8TableIndex= FC_PB_VEHICLE_PBOOK; u8TableIndex <= (FC_PB_DEVICE_HANDLE11 + FC_PB_TOTAL_NUMBER_OF_VPB);
       u8TableIndex++)
   {
      if(query.prepare(sql))
      {
         query.addBindValue(u8TableIndex);
         query.addBindValue(OSAL_NULL);
         if(!query.exec())
         {
            PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

            ETG_TRACE_ERR(("FC_PhoneBook_SQLite::SpeechPopulateCheckSumTable:-"
                           "Failed to add value to CheckSum table "));
            blRet= false;
            break;
         }
         else
         {
            blRet= true;
         }
      }
      else
      {
         ETG_TRACE_ERR(("FC_PhoneBook_SQLite::SpeechPopulateCheckSumTable:- Query prepare failed"));
         blRet= false;
         break;
      }
  	}

   return blRet;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::SpeechUpdateCheckSumTable
*
* DESCRIPTION: modifies the checksum for the given device handle
* PARAMETER:  [IN]  - devicehandle , bUpdateVPBx
*             [OUT] -
*
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_SQLite::SpeechUpdateCheckSumTable(quint8 u8DeviceHandle,bool bUpdateVPBx)
{
   ETG_TRACE_USR4(("SpeechUpdateCheckSumTable with DeviceHandle %d ", u8DeviceHandle));

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return;
   }
   QSqlQuery query(db);

   QString sql;
   quint32 u32CheckSum;
   if (true == bUpdateVPBx)
   {
      u8DeviceHandle = (quint8)(u8DeviceHandle + FC_PB_TOTAL_NUMBER_OF_VPB);
   }

   sql = "SELECT " FC_PB_CHKSUM_COL_TABLE_CHKSUM " FROM " FC_PB_TABLE_CHECKSUM
         " WHERE " FC_PB_CHKSUM_COL_TABLE_INDEX " = " +QString::number(u8DeviceHandle);

   if(query.exec(sql))
   {
      if(query.next())
      {
         u32CheckSum= query.value(OSAL_NULL).toUInt();
         sql.clear();
         sql = "UPDATE " FC_PB_TABLE_CHECKSUM " SET " FC_PB_CHKSUM_COL_TABLE_CHKSUM
               " = " +QString::number(++u32CheckSum)+ " WHERE " FC_PB_CHKSUM_COL_TABLE_INDEX
               " = " +QString::number(u8DeviceHandle);

         ETG_TRACE_USR2((" Sql query used - %s", sql.toLatin1().constData()));
         if(query.exec(sql))
         {
            ETG_TRACE_COMP((" -PBDL.S- New checksum= %d for DeviceHandle= %d", u32CheckSum, u8DeviceHandle));
         }
         else
         {
            PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

            ETG_TRACE_ERR(("1Failed to update checksum for tableindex- %d", u8DeviceHandle));
         }
      }
      else
      {
         ETG_TRACE_ERR(("1Failed to fetch checksum for tableindex- %d", u8DeviceHandle));
      }
   }
   else
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(("2Failed to fetch checksum for tableindex- %d", u8DeviceHandle));
   }
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::CreateFavoritesTable()
*
* DESCRIPTION:
*              QtSql specific function to create a favorites table
*
*
* PARAMETER: None.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::CreateFavoritesTable()
{
   ETG_TRACE_USR4(("CreateFavoritesTable Entered"));
   bool bRetValue = false;

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {

      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {

      QSqlQuery query(db);

      const QString sql = "CREATE TABLE IF NOT EXISTS " FC_PB_TABLE_NAME_FAV
         " ( "
         FC_PB_COL_FAV_FAVORITE_ID" INTEGER UNIQUE, "
         FC_PB_COL_FAV_DEVICE_HANDLE" INTEGER, "
         FC_PB_COL_FAV_CONTACT_HANDLE" INTEGER, "
         FC_PB_COL_FAV_TAG_VALUE" INTEGER, "
         FC_PB_COL_FAV_LABEL_TEXT" TEXT, "
         FC_PB_COL_FAV_PHONE_NUMBER" TEXT, "
         FC_PB_COL_FAV_ACTIVE_STATUS" INTEGER, "
         FC_PB_COL_FAV_AVALIABLE_STATUS" INTEGER"
         " )";

      if ( !query.exec(sql) )
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("FC_PhoneBook_SQLite :- ERROR - SQL query failed in creating favorites table"));
      }
      else
         bRetValue = true;
   }

   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::CheckFavoritesTable(quint8 uIndex)
*
* DESCRIPTION:
*              QtSql specific function to validate favourites table
*
*
* PARAMETER: None.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::CheckFavoritesTable()
{
   ETG_TRACE_USR4(("CheckFavoritesTable Entered"));
   bool bRetValue = false;

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {

      QSqlQuery query(db);

      QStringList list_columns;

      QString sql= "PRAGMA table_info( " FC_PB_TABLE_NAME_FAV " )";

      if ( !query.exec(sql) )
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(( "FC_PhoneBook_SQLite :- ERROR - SQL query failed while validating favorites table"));
      }
      else
      {

         while (query.next())
            list_columns << query.value( 1 ).toString();

         if ( list_columns.contains( FC_PB_COL_FAV_FAVORITE_ID) &&
            list_columns.contains( FC_PB_COL_FAV_DEVICE_HANDLE) &&
            list_columns.contains( FC_PB_COL_FAV_CONTACT_HANDLE ) &&
            list_columns.contains( FC_PB_COL_FAV_TAG_VALUE ) &&
            list_columns.contains( FC_PB_COL_FAV_LABEL_TEXT ) &&
            list_columns.contains( FC_PB_COL_FAV_PHONE_NUMBER ) &&
            list_columns.contains( FC_PB_COL_FAV_ACTIVE_STATUS ) &&
            list_columns.contains( FC_PB_COL_FAV_AVALIABLE_STATUS ))
         {
            bRetValue = true;
         }
      }
   }
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::UpdateFavRecord(
*
* DESCRIPTION:
*              Updates a single Favorite record database
*
*
* PARAMETER:  [IN]  - Details of the favorite to be stored
*             [OUT] -
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::InsertFavRecord(FC_PhoneBook_Fav_Detail fav)
{
   ETG_TRACE_USR4(("InsertFavRecord entered."));
   bool bRetValue = false;
   QMutexLocker lock(&m_mutex);
   QString sql;
   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);

      sql = "INSERT INTO " FC_PB_TABLE_NAME_FAV " ( "
         FC_PB_COL_FAV_FAVORITE_ID", "
         FC_PB_COL_FAV_DEVICE_HANDLE", "
         FC_PB_COL_FAV_CONTACT_HANDLE", "
         FC_PB_COL_FAV_TAG_VALUE", "
         FC_PB_COL_FAV_LABEL_TEXT", "
         FC_PB_COL_FAV_PHONE_NUMBER", "
         FC_PB_COL_FAV_ACTIVE_STATUS", "
         FC_PB_COL_FAV_AVALIABLE_STATUS" ) VALUES (?,?,?,?,?,?,?,?)";

      ETG_TRACE_USR2(("String used for favorite record insert - %s",
         sql.toLatin1().constData()));

      query.prepare(sql);
      if((fav.labeltext.contains(" ")) && (fav.labeltext.size() == 1))
      {
         fav.labeltext = fav.phonenumber;
      }
      query.addBindValue(fav.favoriteID);
      query.addBindValue(fav.devicehandle);
      query.addBindValue(fav.contacthandle);
      query.addBindValue(fav.tagvalue);
      query.addBindValue(fav.labeltext);
      query.addBindValue(fav.phonenumber);
      query.addBindValue(fav.activestatus);
      //query.addBindValue(fav.availablestatus);
      query.addBindValue(1);

      if ( !query.exec() )
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("ERROR - SQL query failed in InsertFavRecord "));
      }
      else
      {
         ETG_TRACE_USR4(("Favorites inserted successfully"));
         bRetValue = true;
      }
   }
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetAllFavoritesID
*
* DESCRIPTION: Fetches all favorite ids used
*
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::GetAllFavoritesID(QList<quint16>& u16FavoriteIdList)
{
   ETG_TRACE_USR4(("GetAllFavoritesID entered."));

   bool bRetValue = false;

   QMutexLocker lock(&m_mutex);
   QString sql;
   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);

      u16FavoriteIdList.clear();
      sql= "SELECT " FC_PB_COL_FAV_FAVORITE_ID " FROM " FC_PB_TABLE_NAME_FAV;

      if(query.exec(sql))
      {
         while(query.next())
         {
            u16FavoriteIdList << (query.value(OSAL_NULL)).value<quint16>();
         }
         bRetValue = true;
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("ERROR - SQL query failed in GetAllFavoritesID "));
      }
   }

   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetAllFavoritesIDForDevice
*
* DESCRIPTION: Fetches all favorite ids used
*
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::GetAllFavoritesIDForDevice(QList<quint16>& u16FavoriteIdList, quint8 DeviceHandle)
{
   ETG_TRACE_USR4(("GetAllFavoritesIDForDevice entered."));

   bool bRetValue = false;

   QMutexLocker lock(&m_mutex);
   QString sql;

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {

      QSqlQuery query(db);

      u16FavoriteIdList.clear();
      sql= "SELECT " FC_PB_COL_FAV_FAVORITE_ID " FROM " FC_PB_TABLE_NAME_FAV "  WHERE DeviceHandle = " + QString::number(DeviceHandle);

      if(query.exec(sql))
      {
         while(query.next())
         {
            u16FavoriteIdList<<(query.value(OSAL_NULL)).value<quint16>();
         }
         bRetValue = true;
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("ERROR - SQL query failed in GetAllFavoritesIDForDevice "));
      }
   }

   return bRetValue;
}


bool FC_PhoneBook_SQLite::GetFavoritesIDForContact(QList<quint16>& u16FavoriteIdList, quint32 ContactHandle)
{
   ETG_TRACE_USR4(("GetFavoritesIDForContact entered."));

   bool bRetValue = false;

   QMutexLocker lock(&m_mutex);
   QString sql;
   QSqlDatabase db= Database();

   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {

      QSqlQuery query(db);

      u16FavoriteIdList.clear();
      sql= "SELECT " FC_PB_COL_FAV_FAVORITE_ID " FROM " FC_PB_TABLE_NAME_FAV "  WHERE ContactHandle = " + QString::number(ContactHandle);


      if(query.exec(sql))
      {
         while(query.next())
            u16FavoriteIdList<<(query.value(OSAL_NULL)).value<quint16>();
         bRetValue = true;
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("ERROR - SQL query failed in GetFavoritesIDForContact "));
      }
   }

   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetFavRecord
*
* DESCRIPTION: fetch a favorite record from the database
*
* PARAMETER:  [IN]  - favorite id
*             [OUT] - favorite details
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::GetFavRecord(
                                       quint16 u16FavoriteId, FC_PhoneBook_Fav_Detail& fav, quint16& u16ErrCode)
{
   ETG_TRACE_USR4(("GetFavRecord entered."));

   bool bRetValue = false;
   QMutexLocker lock(&m_mutex);
   QSqlDatabase db= Database();
   if(!db.isOpen())
   {

      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {

      QSqlQuery query(db);

      QString sql= "SELECT * FROM " FC_PB_TABLE_NAME_FAV " WHERE "
         FC_PB_COL_FAV_FAVORITE_ID " = " + QString::number(u16FavoriteId);

      ETG_TRACE_USR2(("Query used in GetFavRecord- %s", sql.toLatin1().constData()));

      if(query.exec(sql))
      {
         if(query.next())
         {
            quint8 u8Counter= 0;
            fav.favoriteID=      (query.value(u8Counter++)).value<quint16>();
            fav.devicehandle=    (query.value(u8Counter++)).value<quint8>();
            fav.contacthandle=   query.value(u8Counter++).toUInt();
            fav.tagvalue=        (query.value(u8Counter++)).value<quint8>();
            fav.labeltext=       query.value(u8Counter++).toString();
            fav.phonenumber=     query.value(u8Counter++).toString();
            fav.activestatus=    (query.value(u8Counter++)).value<quint8>();
            fav.availablestatus= (query.value(u8Counter++)).value<quint8>();

            bRetValue = true;
         }
         else
         {
            ETG_TRACE_ERR(("Fetching favorite detail failed. Invalid FavoriteID"));
            u16ErrCode= FC_PB_ERROR_CONTACT_HANDLE_INVALID;
         }
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("Fetching favorite detail failed. Query failed"));
         u16ErrCode= FC_PB_ERROR_INTERNAL_FAILURE;
      }
   }

   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::DeleteFavRecord
*
* DESCRIPTION:
*             delete favorite method
*
*
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE: bool -
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::DeleteFavRecord(quint32 u16FavoriteId, int fResetDeviceSpecificFav, tU8 fU8DeviceHandle)
{
   ETG_TRACE_USR2(("DeleteFavRecord entered with FavoriteId %d ",u16FavoriteId));
   bool bRetValue = false;
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);
      QString sql;

      if(fResetDeviceSpecificFav == true)
      {
         sql= "DELETE FROM " FC_PB_TABLE_NAME_FAV "  WHERE DeviceHandle = " + QString::number(fU8DeviceHandle);
      }
      else
      {
         if( u16FavoriteId != DELETE_ALL_FAVORITE)
         {
            sql= "DELETE FROM " FC_PB_TABLE_NAME_FAV " WHERE " FC_PB_COL_FAV_FAVORITE_ID " = " + QString::number(u16FavoriteId);
         }
         else
         {
            sql = "DELETE  FROM " FC_PB_TABLE_NAME_FAV;
         }
      }

      ETG_TRACE_USR2((" FC_PhoneBook_SQLite:String used for delete favorite record  - %s ",sql.toLatin1().constData()));

      if(!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Delete record failed "));
      }
      else
      {
         ETG_TRACE_USR4((" FC_PhoneBook_SQLite: Record deleted  successfully"));
         bRetValue = true;
      }
   }
   return bRetValue;
}


// ---: RB fix proposal for GMNGA-53014 and GMMY15-248
/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::DeleteAllFavRecordsOfDevice
*
* DESCRIPTION:
*             delete favorite method
*
*
* PARAMETER:  [IN]  -
*          [OUT] -
*
* RETURNVALUE: bool -
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::DeleteAllFavRecordsOfDevice(quint8 u8DeviceHandle)
{
   bool bRetValue = false;
   QMutexLocker lock(&m_mutex);
   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {

      QSqlQuery query(db);
      QString sql;
      sql= "DELETE FROM " FC_PB_TABLE_NAME_FAV " WHERE " FC_PB_COL_FAV_DEVICE_HANDLE " = " + QString::number(u8DeviceHandle);
      ETG_TRACE_USR4(( " Query used to delete all favorite records of one device: '%s' ", sql.toLatin1().constData() ));
      if(!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" Deletion of all FAV records for DeviceHandle= %d has failed ", u8DeviceHandle ));
      }
      else
      {
         ETG_TRACE_USR4((" All FAV records deleted successfully for DeviceHandle= %d ", u8DeviceHandle));
         bRetValue = true;
      }
   }
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::UpdateFavRecord(
*
* DESCRIPTION:
*              Updates a single Favorite record database
*
*
* PARAMETER:  [IN]  - Details of the favorite to be stored
*             [OUT] -
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::UpdateFeatureSupportRecord(FC_PhoneBook_FeatureSupport_Detail fs)
{
   ETG_TRACE_USR4(("UpdateFeatureSupportRecord entered."));
   bool bRetValue = false;
   DeletetFeatureSupportRecord(fs.m_address,fs.m_devicehandle);
   QMutexLocker lock(&m_mutex);
   QString sql;
   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);

      ETG_TRACE_USR2(("Device Not Inserted before so inserting now for the Device for =%d ",fs.m_devicehandle));

      sql = "INSERT INTO " FC_PB_TABLE_NAME_FS " ( "
         FC_PB_FS_COL_DEVICE_ADDRESS", "
         FC_PB_FS_COL_DEVICE_HANDLE", "
         FC_PB_FS_COL_PB_SUPPORT", "
         FC_PB_FS_COL_PB_CONTACTS_SORT", "
         FC_PB_FS_COL_PB_MISSED_CALL", "
         FC_PB_FS_COL_PB_INCOMING_CALL", "
         FC_PB_FS_COL_PB_OUTGOING_CALL ", "
         FC_PB_FS_COL_PB_COMBINED_CALL", "
         FC_PB_FS_COL_PB_LAST_CONNECTION_TIME_ELAPSED", "
         FC_PB_FS_COL_PB_DL_COMPLETED_ONCE", "
         FC_PB_FS_COL_PB_DOWNLOAD_ONOFF", "
         FC_PB_FS_COL_PB_AUTOMATIC_CONTACT_DL", "
         FC_PB_PREFERREDSORTORDER_COL_SORTORDER") "
         "VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?);";

      ETG_TRACE_USR2(("String used for FeatureSupport record insert - %s",
         sql.toLatin1().constData()));

      query.prepare(sql);

      query.addBindValue(fs.m_address);
      query.addBindValue(fs.m_devicehandle);
      query.addBindValue(fs.m_pbSupport);
      query.addBindValue(fs.m_conatctSortSupport);
      query.addBindValue(fs.m_mchsupport);
      query.addBindValue(fs.m_ichsupport);
      query.addBindValue(fs.m_ochsupport);
      query.addBindValue(fs.m_cchsupport);
      query.addBindValue(fs.m_LastConnectionTimeElapsed);
      query.addBindValue(fs.m_DlCompletedOnce);
      query.addBindValue(fs.m_SwitchDownloadOnOffState);
      query.addBindValue(fs.m_AutomaticContactDL);
      query.addBindValue(fs.m_PreferredSortOrder);

      if ( !query.exec() )
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("ERROR - SQL query failed in UpdateFeatureSupportRecord For Insertion"));
      }
      else
      {
         bRetValue = true;
      }
   }
   ETG_TRACE_USR4(("UpdateFeatureSupportRecord EXIT."));
   return bRetValue;
}


bool FC_PhoneBook_SQLite::bUpdateDeviceLastConnectionTimeElapsed(
   quint8 f_u8DeviceHandle)
{
   ETG_TRACE_USR4(("UpdateDeviceLastConnectionTimeElapsed entered"));
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db = Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QString SelectSql;
   QString UpdateSql;

   QSqlQuery SelectQuery(db);
   QSqlQuery UpdateQuery(db);

   SelectSql =
      "SELECT " FC_PB_FS_COL_DEVICE_HANDLE " , " FC_PB_FS_COL_PB_LAST_CONNECTION_TIME_ELAPSED " FROM " FC_PB_TABLE_NAME_FS;

   ETG_TRACE_USR2(("Select Query::%s", SelectSql.toLatin1().constData()));

   if (!SelectQuery.exec(SelectSql))
   {
      PB_CHECK_QUERY_ERROR(SelectQuery); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime
      return false;
   }

   quint32 l_u32LastConnectionTimeElapsed;
   quint8 l_u8DeviceHandle;

   //Fix for GMMY15-10814
   while(db.isOpen() && SelectQuery.next())
   {
      l_u8DeviceHandle = (SelectQuery.value(0)).value<quint8>();
      l_u32LastConnectionTimeElapsed = SelectQuery.value(1).toUInt();

      if(l_u8DeviceHandle == f_u8DeviceHandle)
      {
         ETG_TRACE_USR3(("Setting device handle to zero::%u",l_u8DeviceHandle));
         l_u32LastConnectionTimeElapsed = 0; //Lint
      }
      else
      {
         ETG_TRACE_USR2(("Incrementing device handle::%u",l_u8DeviceHandle));
         l_u32LastConnectionTimeElapsed++;
      }

      UpdateSql = "UPDATE " FC_PB_TABLE_NAME_FS " SET "
         FC_PB_FS_COL_PB_LAST_CONNECTION_TIME_ELAPSED " = " +
         QString::number(l_u32LastConnectionTimeElapsed) +  " WHERE "
         FC_PB_FS_COL_DEVICE_HANDLE" = " + QString::number(l_u8DeviceHandle);

      ETG_TRACE_USR2(("Update Query::%s", UpdateSql.toLatin1().constData()));

      if (!UpdateQuery.exec(UpdateSql))
      {
         PB_CHECK_QUERY_ERROR(UpdateQuery); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime
         return false;
      }

      ETG_TRACE_USR4(("Success"));
   }

   return true;
}


quint8 FC_PhoneBook_SQLite::u8GetLRUDeviceHandle()
{
   ETG_TRACE_USR4(("u8GetLRUDeviceHandle"));
   QMutexLocker lock(&m_mutex);
   QSqlDatabase db = Database();

   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return 0;
   }

   QString SelectSql;
   QSqlQuery SelectQuery(db);

   SelectSql =
      "SELECT " FC_PB_FS_COL_DEVICE_HANDLE " , " FC_PB_FS_COL_PB_LAST_CONNECTION_TIME_ELAPSED " FROM " FC_PB_TABLE_NAME_FS;

   ETG_TRACE_USR2(("Select Query::%s", SelectSql.toLatin1().constData()));

   if (!SelectQuery.exec(SelectSql))
   {
      PB_CHECK_QUERY_ERROR(SelectQuery); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime
      return 0;
   }

   quint32 l_u32MaxLastConnectionTimeElapsed = 0;
   quint8 l_u8LRUDeviceHandle = 0;
   quint8 l_u8DeviceHandle;
   quint32 l_u32LastConnectionTimeElapsed;

   while(db.isOpen() && SelectQuery.next())
   {
      l_u8DeviceHandle = (SelectQuery.value(0)).value<quint8>();
      l_u32LastConnectionTimeElapsed = SelectQuery.value(1).toUInt();

      if(l_u32LastConnectionTimeElapsed >= l_u32MaxLastConnectionTimeElapsed)
      {
         l_u32MaxLastConnectionTimeElapsed = l_u32LastConnectionTimeElapsed;
         l_u8LRUDeviceHandle = l_u8DeviceHandle;
      }
   }

   ETG_TRACE_USR2(("LRU Device is::%u",l_u8LRUDeviceHandle));
   return l_u8LRUDeviceHandle;
}


bool FC_PhoneBook_SQLite::bGetDlCompletedOnceValue(quint8 u8DeviceHandle)
{
   ETG_TRACE_USR4(("bGetDlCompletedOnceValue DeviceHandle: %d", u8DeviceHandle));

   bool bDlCompletedOnce_Value = FALSE;
   QMutexLocker lock(&m_mutex);
   QSqlDatabase db = Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {

      QSqlQuery SelectQuery(db);
      QString sql= "SELECT " FC_PB_FS_COL_PB_DL_COMPLETED_ONCE " FROM " FC_PB_TABLE_NAME_FS " WHERE "
         FC_PB_FS_COL_DEVICE_HANDLE" == " + QString::number(u8DeviceHandle);

      if(!SelectQuery.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(SelectQuery); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("ERROR - SQL query failed in bGetDlCompletedOnceValue"));
      }
      else
      {
         if(!SelectQuery.next())
         {
            ETG_TRACE_USR4((" FC_PhoneBook_SQLite: No entry found for DlCompletedOnce in bGetDlCompletedOnceValue "));
         }
         else
         {
            bDlCompletedOnce_Value = SelectQuery.value(0).toBool();

            ETG_TRACE_USR4(("bGetDlCompletedOnceValue bDlCompletedOnce_Value:%u", bDlCompletedOnce_Value));
         }
      }
   }
   return bDlCompletedOnce_Value;
}


bool FC_PhoneBook_SQLite::bUpdateDlCompletedOnceValue(quint8 u8DeviceHandle, bool bValue)
{
   ETG_TRACE_USR4((" bUpdateDlCompletedOnceValue entered : DeviceHandle= %d, bValue= %u ", u8DeviceHandle, bValue));
   bool bRetValue = false;
   QMutexLocker lock(&m_mutex);
   QString sql;

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {

      QSqlQuery query(db);

      sql = "UPDATE " FC_PB_TABLE_NAME_FS " SET "
            FC_PB_FS_COL_PB_DL_COMPLETED_ONCE " = " + QString::number(bValue) + " WHERE "
            FC_PB_FS_COL_DEVICE_HANDLE " == " + QString::number(u8DeviceHandle);

      query.prepare(sql);
      if (!query.exec())
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("ERROR - SQL query failed in bUpdateDlCompletedOnceValue For Updation "));
      }
      else
         bRetValue = true;
   }
   ETG_TRACE_USR4(("bUpdateDlCompletedOnceValue EXIT."));
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetFeatureSupportforDevice
*
* DESCRIPTION: fetch a FeatureSupportforDevice from the database
*
* PARAMETER:  [IN]  - device_address
*             [OUT] - FC_PhoneBook_FeatureSupport_Detail
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::GetFeatureSupportforDevice(QString device_address, quint8 device_Handle,
                                                     FC_PhoneBook_FeatureSupport_Detail& fs, quint16& u16ErrCode)
{
   ETG_TRACE_USR2(( " GetFeatureSupportforDevice entered for DeviceHandle= %d, Address= '%s' ",
                    device_Handle, device_address.toLatin1().constData()));

   bool bRetValue= false;
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db= Database();
   if(db.isOpen())
   {
      QSqlQuery query(db);

      QString sql = "SELECT * FROM " FC_PB_TABLE_NAME_FS " WHERE "
      FC_PB_FS_COL_DEVICE_HANDLE" = " + QString::number(device_Handle);

      ETG_TRACE_USR2((" Query used in GetFeatureSupportforDevice: '%s' ", sql.toLatin1().constData()));

      if (query.exec(sql))
      {
         if (query.next())
         {
            quint8 u8Counter = 0;
            fs.m_address = query.value(u8Counter++).toString();
            ETG_TRACE_USR2(("Fetching GetFeatureSupportforDevice. Device FOUND=%s", fs.m_address.toLatin1().constData()));
            fs.m_devicehandle = (query.value(u8Counter++)).value<quint8>();
            ETG_TRACE_USR2(("Fetching GetFeatureSupportforDevice. Device FOUND Device Handle =%d", fs.m_devicehandle));
            fs.m_pbSupport = (query.value(u8Counter++)).value<quint8>();
            ETG_TRACE_USR2(("Fetching GetFeatureSupportforDevice. Device PB SUPPORT =%d", fs.m_pbSupport));
            fs.m_conatctSortSupport = (query.value(u8Counter++)).value<quint8>();
            ETG_TRACE_USR2(("Fetching GetFeatureSupportforDevice. Device CONTACT SUPPORT =%d", fs.m_conatctSortSupport));
            fs.m_mchsupport = (query.value(u8Counter++)).value<quint8>();
            ETG_TRACE_USR2(("Fetching GetFeatureSupportforDevice. Device MCH SUPPORT =%d", fs.m_mchsupport));
            fs.m_ichsupport = (query.value(u8Counter++)).value<quint8>();
            ETG_TRACE_USR2(("Fetching GetFeatureSupportforDevice. Device ICH SUPPORT =%d", fs.m_ichsupport));
            fs.m_ochsupport = (query.value(u8Counter++)).value<quint8>();
            ETG_TRACE_USR2(("Fetching GetFeatureSupportforDevice. Device OCH SUPPORT =%d", fs.m_ochsupport));
            fs.m_cchsupport = (query.value(u8Counter++)).value<quint8>();
            ETG_TRACE_USR2(("Fetching GetFeatureSupportforDevice. Device CCH SUPPORT =%d", fs.m_cchsupport));
            fs.m_LastConnectionTimeElapsed=query.value(u8Counter++).toUInt();
            ETG_TRACE_USR2(("Fetching GetFeatureSupportforDevice. Device LastConnectionTimeElapsed =%d",fs.m_LastConnectionTimeElapsed));
            fs.m_DlCompletedOnce = query.value(u8Counter++).toUInt();
            ETG_TRACE_USR2(( " Fetching GetFeatureSupportforDevice. DlCompletedOnce= %d", fs.m_DlCompletedOnce ));
            fs.m_SwitchDownloadOnOffState = (query.value(u8Counter++)).value<quint8>();
            ETG_TRACE_USR2(( " Fetching GetFeatureSupportforDevice. SwitchDownloadOnOffState= %d", fs.m_SwitchDownloadOnOffState ));
            fs.m_AutomaticContactDL = query.value(u8Counter++).toUInt();
            ETG_TRACE_USR2(( " Fetching GetFeatureSupportforDevice. AutomaticContactDL= %d", fs.m_AutomaticContactDL ));
            fs.m_PreferredSortOrder = query.value(u8Counter++).value<quint8>();
            ETG_TRACE_USR2(( " Fetching GetFeatureSupportforDevice. PreferredSortOrder= %d", fs.m_PreferredSortOrder ));

            bRetValue = true;
         }
         else
         {
            ETG_TRACE_ERR((" GetFeatureSupportforDevice(..). Query failed for DeviceHandle= %d ", device_Handle));
            u16ErrCode = FC_PB_ERROR_DEVICE_NOT_IN_LIST;
         }
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" Fetching FeatureSupportforDevice. Query failed "));
         u16ErrCode = FC_PB_ERROR_INTERNAL_FAILURE;
      }
      if (bRetValue == true)
      {
         ETG_TRACE_USR2(("Fetching FeatureSupportforDevice. SUCCESS for DeviceHandle= %d ", device_Handle));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   ETG_TRACE_USR4(( " GetFeatureSupportforDevice EXIT. " ));
   return bRetValue;
}


bool FC_PhoneBook_SQLite::bCalculateUsedMemoryFromOnePhoneBookTable(quint8 u8DeviceHandle, quint32 &u32TotalContactMemory, quint32 &u32TotalPhotoMemory)
{
   ETG_TRACE_USR4(( " bCalculateUsedMemoryFromOnePhoneBookTable entered. " ));

   QTime DebugTime;
   DebugTime.start();

   bool bRet= true;
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db = Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(( " Database is not open " ));
      return false;
   }

   // Initialize the given memory variables
   u32TotalContactMemory = 0;
   u32TotalPhotoMemory  = 0;

   QSqlQuery query(db);
   QString sql = "";

   quint8 u8Index = u8DeviceHandle;
   if( (u8Index>=2) && (u8Index<=FC_PB_TOTAL_NUMBER_OF_PB) )
   {
      // Example for used query: SELECT SUM(APL1Size), SUM(PhotoSize) FROM PhoneBook_DH2
      sql = "SELECT SUM(APL1Size), SUM(PhotoSize) FROM " + fc_pb_aTableNames[u8Index];

      ETG_TRACE_USR2(( " Query= '%s' ", sql.toLatin1().constData() ));

      if(query.exec(sql))
      {
         while(query.next())
         {
            quint8 u8Counter = 0;
            u32TotalContactMemory = query.value(u8Counter++).toUInt();  // contact
            u32TotalPhotoMemory  = query.value(u8Counter++).toUInt();  // photo

            ETG_TRACE_USR2((" Fetched Memory ussage [bytes]: TableIndex= %u, Contact= %u, Photo= %u ",
                              u8Index, u32TotalContactMemory, u32TotalPhotoMemory ));
         }
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(( " ERROR: Query returned with error " ));
         bRet = false;
      }
   }
   else
   {
      ETG_TRACE_ERR(( " ERROR: DeviceHandle is out of range. " ));
      bRet = false;
   }

   ETG_TRACE_USR2(( " -PBDL.Perf- 'bCalculateUsedMemoryFromOnePhoneBookTable' exited after %d ms ", DebugTime.elapsed() ));
   return bRet;
}


bool FC_PhoneBook_SQLite::CalculateUsedMemoryFromPhoneBookTables(quint32 &u32TotalContactMemory, quint32 &u32TotalPhotoMemory)
{
   ETG_TRACE_USR4(( " CalculateUsedMemoryFromPhoneBookTables entered. " ));

   QTime DebugTime;
   DebugTime.start();

   bool bRet= false;
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db = Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(( " Database is not open " ));
   }
   else
   {
      // Initialize the given memory variables
      quint32 u32ContactMemory = 0;
      quint32 u32PhotoMemory  = 0;

      u32TotalContactMemory = 0;
      u32TotalPhotoMemory  = 0;

      QSqlQuery query(db);
      QString sql = "";

      for(quint8 u8Index = 2; u8Index <= 11; u8Index++)
      {
         // Example for used query: SELECT SUM(APL1Size), SUM(PhotoSize) FROM PhoneBook_DH2
         sql = "SELECT SUM(APL1Size), SUM(PhotoSize) FROM " + fc_pb_aTableNames[u8Index];

         ETG_TRACE_USR2(( " Query= '%s' ", sql.toLatin1().constData() ));

         if(query.exec(sql))
         {
            while(query.next())
            {
               quint8 u8Counter = 0;
               u32ContactMemory = query.value(u8Counter++).toUInt();  // contact
               u32PhotoMemory  = query.value(u8Counter++).toUInt();  // photo

               u32TotalContactMemory += u32ContactMemory;
               u32TotalPhotoMemory  += u32PhotoMemory;

               ETG_TRACE_USR2((" Fetched Memory usage [bytes]: TableIndex= %u, Contact= %u, Photo= %u ", u8Index, u32ContactMemory, u32PhotoMemory ));
            }
            bRet = true;
         }
         else
         {
            PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

            ETG_TRACE_ERR(( " ERROR: Query returned with error " ));
         }
      }

      ETG_TRACE_USR2(( " -PBDL.Perf- 'CalculateUsedMemoryFromPhoneBookTables' exited after %d ms ", DebugTime.elapsed() ));
   }
   return bRet;
}


/*******************************************************************************
 *
 * FUNCTION: FC_PhoneBook_SQLite::bUpdatePhotosOfExistingRecords(..)
 *
 * DESCRIPTION: Updates the Contact Photo and related photo details (size, CRC,..)
 *              of existing contacts in table of the given DeviceHandle.
 *
 *
 * PARAMETER: IN:  u8DeviceHandle
 *                 oPhotosToBeUpdatedRAMList (Contact list with updated photo details)
 *
 * RETURNVALUE: bool - returns the success or failure of the function
 ********************************************************************************/
bool FC_PhoneBook_SQLite::bUpdatePhotosOfExistingRecords(quint8 u8DeviceHandle,
                             QList<FC_PhoneBook_Contact_Detail> &oPhotosToBeUpdatedRAMList)
{
   ETG_TRACE_USR4(( " bUpdatePhotosOfExistingRecords entered " ));

   QMutexLocker lock(&m_mutex);
   bool bRetValue = true;

   QTime UpdatPhotoTime;
   UpdatPhotoTime.start();

   QSqlDatabase db = Database();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      bRetValue = false;
   }
   else
   {
      db.transaction();  // Start transaction
      QSqlQuery query(db);

      quint32 u32ListSize = (quint32)oPhotosToBeUpdatedRAMList.size();

      // Query example:
      // UPDATE Phonebook_DH2 SET PhotoDlSize = 42, Photo = 0, PhotoType = 'JPG', PhotoDlCRC = 44, PhotoSize = 43, PhotoCRC = 44 WHERE ContactHandle = 8391502

      QString sql = "UPDATE " + fc_pb_aTableNames[u8DeviceHandle] + " SET "
                        FC_PB_COL_PHOTO_DL_SIZE " = ?, "
                        FC_PB_COL_PHOTO         " = ?, "
                        FC_PB_COL_PHOTO_TYPE    " = ?, "
                        FC_PB_COL_PHOTO_DL_CRC  " = ?, "
                        FC_PB_COL_PHOTO_SIZE    " = ?, "
                        FC_PB_COL_PHOTO_CRC     " = ?  "
                    "WHERE ContactHandle = ? ";
      query.prepare(sql);

      ETG_TRACE_USR2(( " Query used for updating photo= '%s' ", sql.toLatin1().constData() ));

      FC_PhoneBook_Contact_Detail *poContact;
      for(quint32 u32Idx = 0; u32Idx < u32ListSize; u32Idx++)
      {
         poContact = &oPhotosToBeUpdatedRAMList[(int)u32Idx];
         ETG_TRACE_USR2(( " Contact Index= %u, photoDlSize= %d, photoType= '%s' ",
            u32Idx, poContact->photoDlSize, (poContact->photoType).toLatin1().constData() ));
         ETG_TRACE_USR2(( " ContactHandle= %u, photoDlCRC= %d, photoSize= %d, photoCRC= %d ",
            poContact->contactHandle, poContact->photoDlCRC, poContact->photoSize, poContact->photoCRC ));

         query.addBindValue(poContact->photoDlSize);
         query.addBindValue(poContact->photo);
         query.addBindValue(poContact->photoType);
         query.addBindValue(poContact->photoDlCRC);
         query.addBindValue(poContact->photoSize);
         query.addBindValue(poContact->photoCRC);
         query.addBindValue(poContact->contactHandle);

         if (!query.exec())
         {
            PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

            ETG_TRACE_ERR(( " ERROR: Query failed " ));
            bRetValue = false;
         }
      }

      if(false == bRetValue)
      {
         db.rollback(); // Rollback
         ETG_TRACE_ERR(( " ERROR: Updating photos: Database rolled back due error in at least one query " ));
      }
      else
      {
         if(!db.commit())
         {
            db.rollback();
            ETG_TRACE_ERR(( " ERROR: Updating photos: Database rolled back since commit failed " ));
            bRetValue = false;
         }
         else
         {
            ETG_TRACE_USR2(( " Updating photos: Database committed successfully  " ));
         }
      }
   }
   ETG_TRACE_USR2(( " -PBDL.Perf- bUpdatePhotosOfExistingRecords EXIT, processing time= %d ms ", UpdatPhotoTime.elapsed() ));
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::DeletetFeatureSupportRecord
*
* DESCRIPTION:
*             DeletetFeatureSupportRecord
*
*
* PARAMETER:  [IN]  - device_address
*             [OUT] -
*
* RETURNVALUE: bool -
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::DeletetFeatureSupportRecord(QString device_address, quint8 device_Handle)
{
   ETG_TRACE_USR2(( " DeletetFeatureSupportRecord entered for Device Address= '%s' ", device_address.toLatin1().constData() ));
   bool bRetValue = false;
   QMutexLocker lock(&m_mutex);
   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {

      QSqlQuery query(db);
      QString sql;

      sql= "DELETE FROM " FC_PB_TABLE_NAME_FS " WHERE " FC_PB_FS_COL_DEVICE_HANDLE " = " + QString::number(device_Handle);

      ETG_TRACE_USR2((" FC_PhoneBook_SQLite:String used for DeletetFeatureSupportRecord  - %s ",sql.toLatin1().constData()));

      if(!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_USR4((" FC_PhoneBook_SQLite: DeletetFeatureSupportRecord failed "));
      }
      else
      {
         ETG_TRACE_USR4((" FC_PhoneBook_SQLite: DeletetFeatureSupport Record deleted  successfully"));
         bRetValue = true;
      }
   }
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::UpdateSortListView
*
* DESCRIPTION:
*             check for update list sort view
*
*
* PARAMETER:  [IN]  - u8DeviceHandle
*             [OUT] -
*
* RETURNVALUE: bool -
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::UpdateSortListView(quint8 u8DeviceHandle)
{
   ETG_TRACE_USR2(("UpdatedSortListView entered with DeviceHandle %d ", u8DeviceHandle));
   QMutexLocker lock(&m_mutex);//GMMY17-17143
   bool b_retResult = false;
   quint16 u16Count = 0;

   QSqlDatabase db= Database();

   //Fix for GMMY15-10814
   if(db.isOpen())
   {
      QSqlQuery query(db);
      QString sql;

      if(!CHECK_VALID(u8DeviceHandle))
      {
         ETG_TRACE_ERR((" FC_PhoneBook_SQLite: device handle is invalid "));
         return b_retResult;
      }

      // check for separate FN & LN in database
      sql = "SELECT COUNT ( " FC_PB_COL_FIRST_NAME " ) FROM "+ fc_pb_aTableNames[u8DeviceHandle];

      if(query.exec(sql))
      {
         ETG_TRACE_USR4(("FC_PhoneBook_SQLite:  Get first name count successfully"));
         if(query.next())
         {
            u16Count= (query.value(0)).value<quint16>();
            ETG_TRACE_USR2((" FC_PhoneBook_SQLite: first name count  u16Count =%d",u16Count));
            if(u16Count != 0)
            {
               b_retResult = true;
            }
         }
         else
         {
            ETG_TRACE_ERR(("FC_PhoneBook_SQLite:  Get first name count query.next failed"));
         }
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("FC_PhoneBook_SQLite:  Get first name count failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Database is not open"));
   }

   ETG_TRACE_USR4(("FC_PhoneBook_SQLite::UpdateSortListView Exit"));
   return b_retResult;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetNewUniqueContactHandle
*
* DESCRIPTION: Generates a unique contact handle within the contact handle limit
*
*
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE:
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::GetNewUniqueContactHandle(quint32& u32ContactHandle)
{
   ETG_TRACE_USR4(("GetNewUniqueContactHandle entered with ContactHandle %d ", u32ContactHandle));

   bool blFound= false;
   u32ContactHandle= 1;

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);
   QString sql = "SELECT " FC_PB_MASTER_COL_CONTACT_HANDLE " FROM " FC_PB_MASTER_TABLE_NAME " ORDER BY " FC_PB_MASTER_COL_CONTACT_HANDLE "";

   //Fix for GMMY15-1582 :: Database: Check the ContactHandle generation in case of wrap-around
   if(db.isOpen()) //Fix for GMMY15-10814
   {
      if(query.exec(sql))
      {
         while (db.isOpen() && query.next())
         {
            //FIX GMMY16-782 GMMY16-1106 After contacts downloaded, recent calls list still only shows phone numbers (no names).
            //Correct the contact handle generated to suit to marking of 24th bit
            // Unmark the "masked" ContactHandle (stored in database) to obtain the "pure" ContactHandle
            //
            // The ContactHandle in the database carry in the 24th bit a marker bit, which indicate if the contact is a
            // VehicleContact (marker bit = 0) or if the contact stems from a BT phone device (marker bit = 1). This marker bit
            // has to be removed from the retrieved ContactHandle to obtain the "pure" unique ContactHandles (lower 23bit).
            // (FC_PB_CONTACT_HANDLE_LIMIT = 0x7FFFFF)
            if (u32ContactHandle == ((quint16) (query.value(0).toUInt()) & FC_PB_CONTACT_HANDLE_LIMIT))
            {
               u32ContactHandle++;
            }
            else//Take unused/deleted handle
            {
               blFound= true;
               ETG_TRACE_USR2(("Found a unique contact handle - %d", u32ContactHandle));
               break;
            }
         }
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" Sql query failed in getting the list of contact handles"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   return blFound;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetLastInsertedContactHandle
*
* DESCRIPTION: Fetches the last used contact handle from master table
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE:
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::GetLastInsertedContactHandle(quint32& u32ContactHandle)
{
   ETG_TRACE_USR4(("GetLastInsertedContactHandle entered "));

   bool blRet= false;
   QString sql;
   quint32 u32RowId;

   sql= "SELECT MAX(_ROWID_) FROM " FC_PB_MASTER_TABLE_NAME
      " WHERE ((" FC_PB_MASTER_COL_TABLE_INDEX " >= 1) AND ("
      FC_PB_MASTER_COL_TABLE_INDEX " < 12))";

   QMutexLocker lock(&m_mutex);//GMMY17-17143
   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);

   if(query.exec(sql))
   {
      if(query.next())
      {
         u32RowId= query.value(OSAL_NULL).toUInt();

         ETG_TRACE_USR2((" RowId of the last inserted ContactHandle in Master table- %d", u32RowId ));

         if(u32RowId > OSAL_NULL)
         {
            sql.clear();
            sql= "SELECT " FC_PB_MASTER_COL_CONTACT_HANDLE
               " FROM " FC_PB_MASTER_TABLE_NAME " WHERE _ROWID_ = " + QString::number(u32RowId);

            if(query.exec(sql))
            {
               if(query.next())
               {
                  // Unmark the "masked" ContactHandle (stored in database) to obtain the "pure" ContactHandle
                  //
                  // The ContactHandle in the database carry in the 24th bit a marker bit, which indicate if the contact is a
                  // VehicleContact (marker bit = 0) or if the contact stems from a BT phone device (marker bit = 1). This marker bit
                  // has to be removed from the retrieved ContactHandle to obtain the "pure" unique ContactHandles (lower 23bit).
                  // (FC_PB_CONTACT_HANDLE_LIMIT = 0x7FFFFF)
                  //
                  u32ContactHandle= query.value(OSAL_NULL).toUInt() & FC_PB_CONTACT_HANDLE_LIMIT;
                  blRet= true;
               }
               else
               {
                  ETG_TRACE_ERR(("1Failed to fetch contact handle from rowid- %d", u32RowId));
               }
            }
            else
            {
               PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

               ETG_TRACE_ERR(("2Failed to fetch contact handle from rowid- %d", u32RowId));
            }
         }
         else //There was no row other than the dummy inserted
         {
            blRet= true;
            u32ContactHandle= FC_PB_PHONEBOOK_CNT_HANDLE_OFFSET;
         }
      }
      else
      {
         ETG_TRACE_ERR(("1Failed to get the RowId of the last inserted row "
            "for the phonebook contact"));
      }
   }
   else
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(("2Failed to get the RowId of the last inserted row "
         "for the phonebook contact"));
   }

   if(!blRet)
   {
      ETG_TRACE_USR2(("Generating a Random Contact Handle"));
      blRet= GetNewUniqueContactHandle(u32ContactHandle);
      u32ContactHandle--; //This number will be incremented once while inserting for a
      // new contact
   }

   return blRet;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetContactHandleForCallHistory
*
* DESCRIPTION: Generates contact handle for call history entry
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE:
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::GetContactHandleForCallHistory(quint8 u8ConnectionID, quint8 u8CallType,
                                                         quint32& u32ContactHandle)
{
   ETG_TRACE_USR4((" GetContactHandleForCallHistory entered with connection id :%u and calltype : %u", u8ConnectionID, u8CallType));

   bool bRetValue = false;
   u32ContactHandle= 0;

   if(CHECK_CH_VALID(u8ConnectionID))
   {
      if(u8CallType < FC_PB_DNL_CALL_HISTORY_TYPES)
      {
         u32ContactHandle = (quint32)FC_PB_BASE_CNT_HANDLE_FOR_CH(u8ConnectionID, u8CallType)
            + ++u32LastCallHistoryCntHandle[u8ConnectionID-1][u8CallType];

         ETG_TRACE_USR2((" Contact handle generated for the call history entry- %u", u32ContactHandle));
         bRetValue = true;
      }
      else
      {
         ETG_TRACE_ERR((" ERROR :- Invalid call type"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("ERROR :- Invalid Connection ID"));
   }

   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::MakeAllFavoritesUnavailable
*
* DESCRIPTION: Available status of all favorites entries are cleared.
*              Vehilce phonebook related favorites should not be made unavailable unless
*              contact deleted
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE:
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::MakeAllFavoritesUnavailable()
{
   ETG_TRACE_USR4(("MakeAllFavoritesUnavailable entered."));

   bool bRetValue = false;
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {

      QSqlQuery query(db);

      QString sql= "UPDATE " FC_PB_TABLE_NAME_FAV " SET "
         FC_PB_COL_FAV_AVALIABLE_STATUS" = 0 WHERE "
         FC_PB_COL_FAV_DEVICE_HANDLE" != " + QString::number(FC_PB_VEHICLE_PBOOK); //Clear all entries except for vehicle phonebook

      ETG_TRACE_USR3(("sql query used - %s", sql.toLatin1().constData()));

      if(query.exec(sql))
      {
         ETG_TRACE_USR4(("Favorites updated successfully"));
         bRetValue = true;
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("Failed to update favorites"));
      }
   }
   ETG_TRACE_USR4(("MakeAllFavoritesUnavailable exited."));
   return bRetValue;

}


bool FC_PhoneBook_SQLite::MakeFavoritesUnAvailable(quint8 DeviceHandle)
{
   ETG_TRACE_USR4(("MakeAllFavoritesUnavailable entered."));

   bool bRetValue = false;
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db= Database();

   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {

      QSqlQuery query(db);

      QString sql= "UPDATE " FC_PB_TABLE_NAME_FAV " SET "
         FC_PB_COL_FAV_AVALIABLE_STATUS " = 0 WHERE DeviceHandle = " + QString::number(DeviceHandle); //Clear all entries except for vehicle phonebook

      ETG_TRACE_USR2(("sql query used - %s", sql.toLatin1().constData()));

      if(query.exec(sql))
      {
         ETG_TRACE_USR2(("Favorites updated successfully"));
         bRetValue = true;
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("Failed to update favorites"));
      }
   }
   ETG_TRACE_USR4(("MakeAllFavoritesUnavailable exited."));
   return bRetValue;
}


bool FC_PhoneBook_SQLite::RemoveFavorite(quint32 ContactHandle)
{
   ETG_TRACE_USR4(("RemoveFavorite entered."));

   bool bRetValue = false;
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);

      QString sql= "DELETE FROM " FC_PB_TABLE_NAME_FAV " WHERE ContactHandle = " + QString::number(ContactHandle);

      ETG_TRACE_USR2(("sql query used - %s", sql.toLatin1().constData()));

      if(query.exec(sql))
      {
         ETG_TRACE_USR2(("Favorites updated successfully"));
         bRetValue = true;
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("Failed to update favorites"));
      }
   }

   return bRetValue;

}


bool FC_PhoneBook_SQLite::MakeFavoritesAvailable(quint8 DeviceHandle)
{
   ETG_TRACE_USR4(("MakeFavoritesAvailable entered."));

   bool bRetValue = false;
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);

      QString sql= "UPDATE " FC_PB_TABLE_NAME_FAV " SET "
         FC_PB_COL_FAV_AVALIABLE_STATUS " = 1 WHERE DeviceHandle = " + QString::number(DeviceHandle) ; //Clear all entries except for vehicle phonebook

      ETG_TRACE_USR4(("sql query used - %s", sql.toLatin1().constData()));

      if(query.exec(sql))
      {
         ETG_TRACE_USR2(("Favorites updated successfully"));
         bRetValue = true;
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("Failed to update favorites"));
      }
   }

   ETG_TRACE_USR4(("MakeFavoritesAvailable exited."));
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::MakeAllFavoritesInactive
*
* DESCRIPTION: Active status of all favorites entries are cleared.
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE:
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::MakeAllFavoritesInactive()
{
   ETG_TRACE_USR4(("MakeAllFavoritesInactive entered."));

   bool u32Count = false;
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {

      QSqlQuery query(db);

      QString sql= "UPDATE " FC_PB_TABLE_NAME_FAV " SET "
         FC_PB_COL_FAV_ACTIVE_STATUS" = 0";

      ETG_TRACE_USR2(("sql query used - %s", sql.toLatin1().constData()));

      if(query.exec(sql))
      {
         ETG_TRACE_USR2(("Favorites updated successfully"));
         u32Count = true;
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("Failed to update favorites"));
      }
   }

   ETG_TRACE_USR4(("MakeAllFavoritesInactive exited."));
   return u32Count;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetFavoritesCount
*
* DESCRIPTION: returns the total number of favorites in database
* RETURNVALUE:
*
*******************************************************************************/
quint32 FC_PhoneBook_SQLite::GetFavoritesCount()
{
   ETG_TRACE_USR4(("GetFavoritesCount entered."));
   QMutexLocker lock(&m_mutex); //GMMY17-17143

   quint32 u32Count= 0;
   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {

      QSqlQuery query(db);

      QString sql= "SELECT COUNT ( " FC_PB_COL_FAV_FAVORITE_ID " )"
         "FROM " FC_PB_TABLE_NAME_FAV;

      if(query.exec(sql))
      {
         if(query.next())
         {
            u32Count= query.value(OSAL_NULL).toUInt();
            ETG_TRACE_USR2((" Total no of favorite ids - %d", u32Count));
         }
         else
         {
            ETG_TRACE_ERR(("1Failed to get the count of favorite ids"));
         }
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("2Failed to get the count of favorite ids"));
      }
   }
   ETG_TRACE_USR4(("GetFavoritesCount exited."));

   return u32Count;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetFavoritesCount
*
* DESCRIPTION: returns the total number of favorites in database
* RETURNVALUE:
*
*******************************************************************************/

quint32 FC_PhoneBook_SQLite::GetFavoritesCount(quint8 DeviceHandle)
{
   ETG_TRACE_USR4(("GetFavoritesCount entered with DeviceHandle %d",DeviceHandle));
   QMutexLocker lock(&m_mutex); //GMMY17-17143

   quint32 bRetValue = 0;
   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {

      QSqlQuery query(db);

      QString sql= "SELECT COUNT ( " FC_PB_COL_FAV_FAVORITE_ID " )"
         "FROM " FC_PB_TABLE_NAME_FAV "  WHERE DeviceHandle = " + QString::number(DeviceHandle);

      if(query.exec(sql))
      {
         if(query.next())
         {
            bRetValue = query.value(OSAL_NULL).toUInt();
            ETG_TRACE_USR2((" Total no of favorite ids - %d", bRetValue));
         }
         else
         {
            ETG_TRACE_ERR(("1Failed to get the count of favorite ids"));
         }
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("2Failed to get the count of favorite ids"));
      }

      ETG_TRACE_USR4(("GetFavoritesCount exited."));
   }
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetFavoriteRecordsOfContactHandle
*
* DESCRIPTION:
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE:
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::GetFavoriteRecordsOfContactHandle(quint32 u32ContactHandle,
                                                            QList<FC_PhoneBook_Fav_Detail>& FavInfoList)
{
   ETG_TRACE_USR4(("GetFavoriteRecordsOfContactHandle entered."));

   bool bRetValue = false;
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {

      QSqlQuery query(db);

      QString sql= "SELECT * FROM " FC_PB_TABLE_NAME_FAV " WHERE "
         FC_PB_COL_FAV_CONTACT_HANDLE " = " + QString::number(u32ContactHandle);

      ETG_TRACE_USR2(("Query used in GetFavoriteRecordsOfContactHandle- %s", sql.toLatin1().constData()));

      FavInfoList.clear();
      if(query.exec(sql))
      {
         bRetValue = true;
         FC_PhoneBook_Fav_Detail fav;
         while(query.next())
         {
            quint8 u8Counter= 0;
            fav.favoriteID=      (query.value(u8Counter++)).value<quint16>();
            fav.devicehandle=    (query.value(u8Counter++)).value<quint8>();
            fav.contacthandle=   query.value(u8Counter++).toUInt();
            fav.tagvalue=        (query.value(u8Counter++)).value<quint8>();
            fav.labeltext=       query.value(u8Counter++).toString();
            fav.phonenumber=     query.value(u8Counter++).toString();
            fav.activestatus=    (query.value(u8Counter++)).value<quint8>();
            fav.availablestatus= (query.value(u8Counter++)).value<quint8>();
            FavInfoList<<fav;
         }
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("GetFavoriteRecordsOfContactHandle:- Fetching favorite detail "
            "using contact handle failed. Query failed"));
      }
   }

   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::UpdateFavoritesAvailableStatus
*
* DESCRIPTION:
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE:
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::UpdateFavoritesAvailableStatus(
   QList<quint16> FavoritesIDList, quint8 AvailStatus)
{
   ETG_TRACE_USR4(("Function UpdateFavoritesAvailableStatus entered."));

   bool bRetValue = false;
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {

      QSqlQuery query(db);
      quint16 ListSize= (quint16)(FavoritesIDList.size());

      ETG_TRACE_USR2(("%d favorites to be updated with available status as -%d",
         FavoritesIDList.size(), AvailStatus));

      QString sql= "UPDATE " FC_PB_TABLE_NAME_FAV " SET "
         FC_PB_COL_FAV_AVALIABLE_STATUS " = "+ QString::number(AvailStatus) +
         " WHERE (";

      for(quint16 ListIndex= 0; ListIndex < ListSize; ListIndex++)
      {
         sql+= "(" FC_PB_COL_FAV_FAVORITE_ID " = " + QString::number(
            FavoritesIDList.at(ListIndex)) + ")";

         if(ListIndex != (ListSize -1))
         {
            sql+= " OR ";
         }
         else
         {
            sql+= ")";
         }
      }

      if(query.exec(sql))
      {
         ETG_TRACE_USR4(("Favorites updated successfully"));
         bRetValue = true;
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("Failed to update favorites"));
      }

      ETG_TRACE_USR4(("Function UpdateFavoritesAvailableStatus exited"));
   }
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::UpdateFavoritesRecord
*
* DESCRIPTION:
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE:
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::UpdateFavoritesRecord(FC_PhoneBook_Fav_Detail FavDetail)
{
   ETG_TRACE_USR4(("Function UpdateFavoritesRecord entered."));

   bool bRetValue = false;
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {

      QSqlQuery query(db);

      QString sql= "UPDATE " FC_PB_TABLE_NAME_FAV " SET "
         FC_PB_COL_FAV_DEVICE_HANDLE" = ?,"
         FC_PB_COL_FAV_CONTACT_HANDLE" = ?,"
         FC_PB_COL_FAV_TAG_VALUE" = ?,"
         FC_PB_COL_FAV_LABEL_TEXT" = ?,"
         FC_PB_COL_FAV_PHONE_NUMBER" = ?,"
         FC_PB_COL_FAV_ACTIVE_STATUS" = ?,"
         FC_PB_COL_FAV_AVALIABLE_STATUS" = ?"
         " WHERE " FC_PB_COL_FAV_FAVORITE_ID " = " + QString::number(FavDetail.favoriteID);

      query.prepare(sql);
      query.addBindValue(FavDetail.devicehandle);
      query.addBindValue(FavDetail.contacthandle);
      query.addBindValue(FavDetail.tagvalue);
      query.addBindValue(FavDetail.labeltext);
      query.addBindValue(FavDetail.phonenumber);
      query.addBindValue(FavDetail.activestatus);
      query.addBindValue(FavDetail.availablestatus);

      ETG_TRACE_USR2(("String used for favorites update record - %s ",sql.toLatin1().constData()));

      if (!query.exec())
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("Update record failed in favorites"));
      }
      else
      {
         bRetValue = true;
      }

      ETG_TRACE_USR4(("Function UpdateFavoritesRecord exited."));
   }
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetFavoriteRecordsOfDeviceHandle
*
* DESCRIPTION:
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE:
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::GetFavoriteRecordsOfDeviceHandle(quint8 u8DeviceHandle,
                                                           QList<FC_PhoneBook_Fav_Detail>& FavInfoList)
{
   ETG_TRACE_USR4(("GetFavoriteRecordsOfDeviceHandle entered."));

   bool bRetValue = false;
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {

      QSqlQuery query(db);

      QString sql= "SELECT * FROM " FC_PB_TABLE_NAME_FAV " WHERE "
         FC_PB_COL_FAV_DEVICE_HANDLE" = " + QString::number(u8DeviceHandle);

      ETG_TRACE_USR2(("Query used in GetFavoriteRecordsOfDeviceHandle- %s", sql.toLatin1().constData()));

      FavInfoList.clear();
      if(query.exec(sql))
      {
         bRetValue = true;
         FC_PhoneBook_Fav_Detail fav;
         while(query.next())
         {
            quint8 u8Counter= 0;
            fav.favoriteID=      (query.value(u8Counter++)).value<quint16>();
            fav.devicehandle=    (query.value(u8Counter++)).value<quint8>();
            fav.contacthandle=   query.value(u8Counter++).toUInt();
            fav.tagvalue=        (query.value(u8Counter++)).value<quint8>();
            fav.labeltext=       query.value(u8Counter++).toString();
            fav.phonenumber=     query.value(u8Counter++).toString();
            fav.activestatus=    (query.value(u8Counter++)).value<quint8>();
            fav.availablestatus= (query.value(u8Counter++)).value<quint8>();
            FavInfoList<<fav;
         }
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("GetFavoriteRecordsOfContactHandle:- Fetching favorite detail "
            "using device handle failed. Query failed"));
      }

      ETG_TRACE_USR4(("Function GetFavoriteRecordsOfDeviceHandle exited."));
   }
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::UpdateFavoritesActiveStatus
*
* DESCRIPTION:
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE:
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::UpdateFavoritesActiveStatus(quint16 FavoritesID,
                                                      quint8 ActiveStatus)
{
   ETG_TRACE_USR4(("Function UpdateFavoritesActiveStatus entered."));

   bool bRetValue = false;
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);

      QString sql= "UPDATE " FC_PB_TABLE_NAME_FAV " SET "
         FC_PB_COL_FAV_ACTIVE_STATUS" = "+ QString::number(ActiveStatus) +
         " WHERE " FC_PB_COL_FAV_FAVORITE_ID " = "+ QString::number(FavoritesID);

      ETG_TRACE_USR2(("Sql query used - %s", sql.toLatin1().constData()));

      if(query.exec(sql))
      {
         ETG_TRACE_USR4(("Favorites updated successfully"));
         bRetValue = true;
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("Failed to update favorites"));
      }

      ETG_TRACE_USR4(("Function UpdateFavoritesActiveStatus exited"));
   }
   return bRetValue;
}


bool FC_PhoneBook_SQLite::UpdateAllFavoritesActiveStatus(quint8 u8Status)
{
   ETG_TRACE_USR4(("Function UpdateAllFavoritesActiveStatus entered."));

   bool bRetValue = false;
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);

      QString sql= "UPDATE " FC_PB_TABLE_NAME_FAV " SET "
         FC_PB_COL_FAV_ACTIVE_STATUS" = "+ QString::number(u8Status);


      ETG_TRACE_USR4(("Sql query used - %s", sql.toLatin1().constData()));

      if(query.exec(sql))
      {
         ETG_TRACE_USR2(("Favorites updated successfully"));
         bRetValue = true;
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("Failed to update favorites"));
      }

      ETG_TRACE_USR4(("Function UpdateAllFavoritesActiveStatus exited"));
   }
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetNoOfContactsInVAB
*
* DESCRIPTION:
*
*
* PARAMETER:  [IN]  - None
*             [OUT] -
*
* RETURNVALUE: bool -
*
*******************************************************************************/
//Gen4 Warning : Return type should be quint32
quint32 FC_PhoneBook_SQLite::GetNoOfContactsInVAB(quint8 u8DeviceHandle)
{
   return m_noOfContactsVehicleAddBook[u8DeviceHandle];
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::IncNoOfContactsVAB
*
* DESCRIPTION:
*             check for update list sort view
*
*
* PARAMETER:  [IN]  - u8DeviceHandle
*             [OUT] -
*
* RETURNVALUE: bool -
*
*******************************************************************************/
void FC_PhoneBook_SQLite::IncNoOfContactsVAB(quint8 u8DeviceHandle)
{
   m_noOfContactsVehicleAddBook[u8DeviceHandle]++;

   ETG_TRACE_USR4(("In IncNoOfContactsVAB: VAB count=%u", m_noOfContactsVehicleAddBook[u8DeviceHandle]));
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::DecNoOfContactsVAB
*
* DESCRIPTION:
*
*
* PARAMETER:  [IN]  - None
*             [OUT] -
*
* RETURNVALUE: bool -
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::DecNoOfContactsVAB(quint8 u8DeviceHandle)
{
   if( m_noOfContactsVehicleAddBook[u8DeviceHandle] != 0)
   {
      m_noOfContactsVehicleAddBook[u8DeviceHandle]--;
      return true;
   }
   return false;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::InitializeDatabase
*
* DESCRIPTION: Initializes database
*
* PARAMETER:  [IN]  - None
*             [OUT] -
*
* RETURNVALUE: bool -
*
*******************************************************************************/
void FC_PhoneBook_SQLite::InitializeDatabase()
{
   ETG_TRACE_USR4(("InitializeDatabase entered"));

   QTime InitialiseTime;
   InitialiseTime.start();

#ifdef DATABASE_WAL_JOURNALING
         // In the early stage of using WAL journaling, FC_Phonebook used long file names for the WAL and SHM files
         // and established symbolic links to "bridge" the 8.3 file naming, used by the NDS SQLite (which is used
         // by the Speech Transcription service) to the long names.
         //    lrwxrwxrwx    1 10113    10113         16 Nov  1 02:22 phonebook.shm -> phonebook.db-shm
         //    lrwxrwxrwx    1 10113    10113         16 Nov  1 05:18 phonebook.wal -> phonebook.db-wal

         // In the meantime the ESO SQLite has been build with build option "SQLITE_ENABLE_8_3_NAMES=1", which by
         // default means "long file names", but allows to switch via URI query parameter "?8_3_names=1" to short
         // names, when opening the database. We now follow this approach.

         // As we already have targets in the "field" which have started with the "long file names" and get SW updates
         // via USB stick (i.e. the dynamic partition remains preserved), we have to check the current situation
         // regarding the now obsolete usage of "long file names" and if needed, to do the related "repair" actions:
         //   -- Removal of the obsolete links, if they exist.
         //   -- Renaming of existing WAL and SHM files with long names to the short name variants.
         //      Note: This second aspect probably will not be observed as the WAL and SHM files should not exist
         //            when we come here. (Could only happen when we come here and the journal is "hot".)

         // Notes:
         // - The real repair actions should happen only once. Afterwards the check for the obsolete links or
         //   "wrong" journaling file names should indicate that all alignments are in place.
         // - The following method bAdjustWalJournalingNamesAndLinks() can handle both directions, i.e.
         //   a transition from LONG to SHORT names as motivated above, but also a transition from SHORT to LONG
         //   names, which is helpfull for test purposes.

         if( bAdjustWalJournalingNamesAndLinks() == false )
         {
            // When we come here, we have a serious issue, so we set the database recreation trigger to get the
            // situation healed with the next power cycle (or next transition to NORMAL).

            // Set the database "recreation trigger file" to the FFS
            vSetDbRecreationTrigger();

            NORMAL_M_ASSERT_ALWAYS();   // With the normal assert we can see from the callstack from which method we have been called.
         }
#endif

   if(!OpenDatabase())
   {
      ETG_TRACE_FATAL(( "FC_PhoneBook_SQLite :- ERROR - Cannot initialize database "));
      // M_FATAL_ASSERT_ALWAYS() ???
   }
   else
   {
      //Fix NCG3D-67384
      bool isTableSchemaChecked = TRUE;

      if(TRUE == isCheckTableSchemaEnabled())
      {
         isTableSchemaChecked = CheckTableSchema();
         SetCheckTableSchema(FALSE);
      }
      //End fix NCG3D-67384

      if(isTableSchemaChecked)
      {
         memset(u32LastCallHistoryCntHandle, 0x00, sizeof(u32LastCallHistoryCntHandle));
         GetLastInsertedContactHandle(u32LastContactHandleUsed);

         ETG_TRACE_USR2(( " ::InitializeDatabase(): Last Inserted Contact Handle= %d ", u32LastContactHandleUsed ));

         for(quint8 uIndex= FC_PB_VEHICLE_PBOOK; uIndex <= FC_VPB_DEVICE_HANDLE11; uIndex++)
         {
            m_noOfContactsVehicleAddBook[uIndex]= GetNoOfContactsInVPB(uIndex);
         }

#ifdef DATABASE_WAL_JOURNALING
         // Here we are close to the end of the method 'InitializeDatabase()', which is called in 'vOnLoadSettings()'
         // This is a good code position to check the WAL and SHM journaling files regarding their access rights
         // and group settings and do the needed corrections.
         (bool) bAdjustWalJournalingAccessRights();
#endif

      }
      else
      {
         ETG_TRACE_ERR(("Validating table schema failed"));
      }
   }

   ETG_TRACE_COMP(( " -PBDL.Perf- 'InitializeDatabase' exited after %d ms ", InitialiseTime.elapsed() ));
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::CloseDatabase
*
* DESCRIPTION: Disconnect database
*
* PARAMETER:  [IN]  - None
*             [OUT] -
*
* RETURNVALUE: bool -
*
*******************************************************************************/
void FC_PhoneBook_SQLite::CloseDatabase()
{
   ETG_TRACE_COMP(( "CloseDatabase entered " ));

   //FIX GMMY15-9645 SHUTDOWN_AFTER_NO_STATE_REQUEST state change failed of CCA_C_U16_APP_PHONEBOOK (NORMAL -> OFF)
   //To reduce time taken in vonsavesettings, mutex lock is not acquired.,
   //
   // FIX PSARCCB-9855 (06/2016):
   // Reactivated the mutex 'QMutexLocker lock(&m_mutex);' with changed concept.
   // New concept is to "enforce" the 'close database' action. For this purpose we have added two global variables
   // ['g_iSkipDataSignals' and 'g_iRequestSchedYield']. 'g_iSkipDataSignals' is set to 1 in 'fc_phonebook_tclApp::vOnSaveSettings()'
   // and reset to 0 in 'FC_PhoneBook_List::InitializeDatabase()'


   // Start time measurement to achieve mutex
   struct timespec time1     = {0, 0};
   struct timespec time2     = {0, 0};
   struct timespec diff_time = {0, 0};
   clock_gettime(CLOCK_MONOTONIC, &time1);

   // -- Try to achieve relevant mutex: BGN
   QMutexLocker lock(&m_mutex);  // Note: The method 'CloseDatabase' is also called in destructor of 'FC_PhoneBook_SQLite', but this is not deadlock relevant.
   // -- Try to achieve relevant mutex: END

   // Stop time measurement to achieve mutex
   clock_gettime(CLOCK_MONOTONIC, &time2);
   if( (time2.tv_nsec - time1.tv_nsec) < 0 )
   {
      diff_time.tv_sec  = time2.tv_sec  - time1.tv_sec-1;
      diff_time.tv_nsec = 1000000000 + time2.tv_nsec - time1.tv_nsec;
   } else {
      diff_time.tv_sec  = time2.tv_sec  - time1.tv_sec;
      diff_time.tv_nsec = time2.tv_nsec - time1.tv_nsec;
   }

   ETG_TRACE_COMP(( " FC_PhoneBook_SQLite::CloseDatabase(): achieve mutex 'm_mutex', used Time[msec]= %5ld ",
                    diff_time.tv_sec *1000 + diff_time.tv_nsec / 1000000 ));


   // Bugfix - GMMY15-8335 - Reset on RemoveDatabase. Check whether the DB is valid before
   // removing the connection.
   //Bugfix-8020-10472-Check whether the DB is valid and open then close the DB before removing the connection.
   //Fix for GMM15-11113--Reset on Phonebook
   bool bContains = QSqlDatabase::contains( FC_PB_DB_CONNECTION_NAME);
   ETG_TRACE_USR2((" Database connections:: bContains : %d",bContains));
   if(bContains)
   {
      ETG_TRACE_USR3(("  Database is already opened then closed the Database "));
      if(Database().isValid())
      {
         if(Database().isOpen() == TRUE)
         {
            Database().close();
         }
         QSqlDatabase::removeDatabase(FC_PB_DB_CONNECTION_NAME);
      }
   }

   // Check if the WAL has been removed after having closed the database or if it still exists, as Speech TS might still have the database open.
   struct stat oStatBuf;
   if( 0 == stat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_FILE_NAME, &oStatBuf) )
   {
      ETG_TRACE_COMP(( " -PBDB.SECURITY-WAL-: INFO: The WAL file still exists after having closed the database. Probably Speech TS still has the DB open. "  ));
   }

   ETG_TRACE_USR4(("CloseDatabase exited"));
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetFavInfoForFavorite
*
* DESCRIPTION: fetch a favorite record from the database
*
* PARAMETER:  [IN]  - favorite id
*             [OUT] - favorite details
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::GetFavInfoForFavorite(
   quint16 u16FavoriteId, QList<FC_PhoneBook_Fav_Detail> & favList, quint16& u16ErrCode)
{
   ETG_TRACE_USR4(("GetFavRecord entered."));

   bool blRet = false;
   QMutexLocker lock(&m_mutex);
   QSqlDatabase db= Database();
   quint8 u8Counter= 0;

   if(!db.isOpen())
   {

      ETG_TRACE_ERR(("Database is not open"));
      u16ErrCode = FC_PB_ERROR_DATABASE_NOT_OPEN;
      blRet = false;
   }
   else
   {
      QSqlQuery query(db);
      QString sql;

      if ( u16FavoriteId == 0)
      {
         sql= "SELECT * FROM " FC_PB_TABLE_NAME_FAV;
      }
      else
      {
         sql= "SELECT * FROM " FC_PB_TABLE_NAME_FAV " WHERE "
            FC_PB_COL_FAV_FAVORITE_ID" = " + QString::number(u16FavoriteId);
      }

      ETG_TRACE_USR2(("Query used in GetFavRecord- %s", sql.toLatin1().constData()));

      if(query.exec(sql))
      {
         if(query.next() || ( u16FavoriteId == 0))
         {
            query.previous();

            while (query.next())
            {
               FC_PhoneBook_Fav_Detail fav;
               u8Counter= 0;

               fav.favoriteID =      (query.value(u8Counter++)).value<quint16>();
               fav.devicehandle =    (query.value(u8Counter++)).value<quint8>();
               fav.contacthandle =   query.value(u8Counter++).toUInt();
               fav.tagvalue =        (query.value(u8Counter++)).value<quint8>();
               fav.labeltext =       query.value(u8Counter++).toString();
               fav.phonenumber =     query.value(u8Counter++).toString();
               fav.activestatus =    (query.value(u8Counter++)).value<quint8>();
               fav.availablestatus = (query.value(u8Counter++)).value<quint8>();

               favList << fav;
            }
            blRet = true;
         }
         else
         {
            // Invalid Favorite ID case
            ETG_TRACE_ERR(("Fetching favorite detail failed. Invalid FavoriteID"));
            u16ErrCode= FC_PB_ERROR_CONTACT_HANDLE_INVALID;
         }
      }
      else
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("Fetching favorite detail failed. Query failed"));
         blRet = true; // Internal error is considered as method result with empty details
      }
   }

   return blRet;
}



/*******************************************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetMatchingRecordBasedOnName()
*
* DESCRIPTION:
*             Overloaded function searches for the list of phonebook contacts containing the searchkey
* PARAMETER:
*
* RETURNVALUE: success or failure of search.
*
********************************************************************************************************/
bool FC_PhoneBook_SQLite::GetMatchingRecordBasedOnName(
   FC_PhoneBook_Search_Result& MatchingRecord,
   const char* firstname, const char* lastname, const char* number, quint8 devicehandle)
{
   ETG_TRACE_USR4(("GetMatchingRecordBasedOnName entered"));

   QMutexLocker lock(&m_mutex);
   QString l_FirstName(firstname);
   QString l_LastName(lastname);
   QString Number(number);
   QString sql;
   bool blContactFound= false;

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);
   //276734, 296481- unrecognized token error fix
   sql = "SELECT ContactHandle, FirstName, LastName, PhoneNumber1, NumberType1, PhoneNumber2, NumberType2, PhoneNumber3, NumberType3,"
                                                   " PhoneNumber4, NumberType4, PhoneNumber5, NumberType5, PhoneNumber6, NumberType6,"
                                                   " PhoneNumber7, NumberType7, PhoneNumber8, NumberType8  FROM " + fc_pb_aTableNames[devicehandle]+ " WHERE ";
   if (!l_FirstName.isEmpty() && !l_LastName.isEmpty())
   {
      sql+="(" FC_PB_COL_FIRST_NAME  " = ? AND "  FC_PB_COL_LAST_NAME  "= ? );";
      //sql += "(" FC_PB_COL_FIRST_NAME "," FC_PB_COL_LAST_NAME ") VALUES (?, ?) ;";
      query.prepare(sql);
      query.addBindValue(l_FirstName);
      query.addBindValue(l_LastName);

   }
   else if (!l_FirstName.isEmpty())
   {
      sql += "(" FC_PB_COL_FIRST_NAME " = ? );";
      //sql += "(" FC_PB_COL_FIRST_NAME " ) VALUES (?); ";
      query.prepare(sql);
      query.addBindValue(l_FirstName);

   }
   else
   {
      sql += "(" FC_PB_COL_LAST_NAME " = ? );";
      //sql += "(" FC_PB_COL_LAST_NAME " ) VALUES (?);";
      query.prepare(sql);
      query.addBindValue(l_LastName);

   }
   //end of fix-[276734, 296481- unrecognized token error fix]
   ETG_TRACE_USR2(("SQL string for search Name = %s", (sql.toLatin1()).constData()));
   if (query.exec())
   {
      if( query.next() )
      {
         FC_PhoneBook_Search_Result searchresult;
         searchresult.contactHandle = query.value(0).toUInt();
         searchresult.firstname     = query.value(1).toString();
         searchresult.lastname      = query.value(2).toString();
		 quint8 u8Index = 3;  // On this value of u8Index, the PhoneNumber1 starts in the query result
         // We traverse through the PhoneNumberX / NumberTypeX, (X=1,.. PBDL_MAX_PHONE_NUMBERS_PER_CONTACT_COUNT) of the contact to find
         // the NumberType of that Number which matches to our 'searchstring'
         searchresult.contactdetailtype = 0;  // Initialization with invalid NumberType
         for( quint8 u8NumCount = 1; u8NumCount <= PBDL_MAX_PHONE_NUMBERS_PER_CONTACT_COUNT; u8NumCount++)
         {
            //PBDL_MAX_PHONE_NUMBERS_PER_CONTACT_COUNT = 5 in RN-AIVI
            ETG_TRACE_USR4((" FindNumberTypeForTheNumber:PhoneNumber Index= %u ", u8Index));
            QString sNumber = query.value(u8Index).toString();

            if( (Number.size() >= FC_PHONEBOOK_LIST_SEARCH_NUM_LEN )?
                 sNumber.contains(Number):(Number == sNumber))
            {
                u8Index++;
                quint8 u8NumberType = (query.value(u8Index)).value<quint8>();
                ETG_TRACE_USR4(( " Found PhoneNumberX, X= %u, NumberType= %u, N ", u8NumCount, u8NumberType ));
                searchresult.contactdetailtype = u8NumberType;
				MatchingRecord = searchresult;
                blContactFound = true;
                break;
            }
			else
            {
               ETG_TRACE_ERR(("FC_PhoneBook_SQLite:- Error - Contact Detail Type not found"));
            }
            u8Index = (quint8)(u8Index + 2);
        }
      }
   }
   else
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(("ERROR - SQL query failed in GetMatchingRecordBasedOnName "));
   }

   return blContactFound;
}


/*******************************************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetMatchingRecordBasedOnNumberfromCHtable()
*
* DESCRIPTION:
*             Overloaded function searches for the list of phonebook contacts containing the searchkey
* PARAMETER:
*
* RETURNVALUE: success or failure of search.
*
********************************************************************************************************/
bool FC_PhoneBook_SQLite::GetMatchingRecordBasedOnNumberfromCHtable(
   FC_PhoneBook_Search_Result& MatchingRecord,
   const char* number, quint8 u8ConnectionID)
{
   ETG_TRACE_USR4(("GetMatchingRecordBasedOnNumberfromCHtable entered"));

   QMutexLocker lock(&m_mutex);

   QTime DebugTime;
   DebugTime.start();

   QString Number(number);
   QString sql;
   bool blContactFound= false;

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);

   sql = "SELECT ContactHandle, FirstName, LastName, NumberType FROM "
                  + fc_pb_aTableNames[MAP_TO_CH_INDEX(u8ConnectionID)] + " WHERE "
                  " ( " FC_PB_COL_CH_PHONE_NUM " == '"+Number+"' ) " ;

   ETG_TRACE_USR2(("SQL string for search Number = %s", (sql.toLatin1()).constData()));

   if (query.exec(sql))
   {
      ETG_TRACE_USR2(("Executing query"));
      if (query.next())
      {
         MatchingRecord.contactHandle =     query.value(0).toUInt();
         MatchingRecord.firstname =         query.value(1).toString();
         MatchingRecord.lastname =          query.value(2).toString();
         MatchingRecord.contactdetailtype = (quint8)query.value(3).toUInt();

         blContactFound = true;
      }
   }
   else
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(("ERROR - SQL query failed in GetMatchingRecordBasedOnName "));   // -bn: Trace text wrong. It should use 'GetMatchingRecordBasedOnNumberfromCHtable'
   }

   ETG_TRACE_USR2(( " -PBDL.Perf- 'GetMatchingRecordBasedOnNumberfromCHtable' exited after %d ms ", DebugTime.elapsed() ));
   return blContactFound;
}

//FIX GMNGA-56482 Favorite not highlighted when Call active
/*******************************************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetFavouritesIDForPhoneNumber()
*
* DESCRIPTION:
*             Function that searches for the list of favourite ids containing the phone number
* PARAMETER:
*
* RETURNVALUE: success or failure of search.
*
********************************************************************************************************/

bool FC_PhoneBook_SQLite::GetFavouritesIDForPhoneNumber(QList<quint16>& u16FavoriteIdList, QString sTelephoneNumber)
{
   ETG_TRACE_USR4(("GetAllFavoritesIDForDevice entered."));
   QMutexLocker lock(&m_mutex);//GMMY17-17143
   bool blRet= false;
   QString sql;
   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query(db);

   u16FavoriteIdList.clear();
   sql= "SELECT " FC_PB_COL_FAV_PHONE_NUMBER "," FC_PB_COL_FAV_FAVORITE_ID " FROM " FC_PB_TABLE_NAME_FAV "";

   if(query.exec(sql))
   {
      while(query.next())
      {
         //if(sTelephoneNumber.length() >= (quint16)FC_PHONEBOOK_LIST_SEARCH_NUM_LEN && query.value(OSAL_NULL).toString()!='\0')//All search string ends with NULL. Hence endsWith alone will return true for all numbers with length > 10 and no number is stored in DB.
         if(sTelephoneNumber.length() >= (quint16)FC_PHONEBOOK_LIST_SEARCH_NUM_LEN && (FALSE == (query.value(OSAL_NULL).toString().isNull())))//(bug fix:GMGNA-58120)Lint Error:58-Bad Type.
         {
            //if(sTelephoneNumber.endsWith(query.value(OSAL_NULL).toString()))
            QString searchString(query.value(OSAL_NULL).toString()); // Change to incorporate differences in +91 and 0 prefixes
            if( sTelephoneNumber.endsWith(searchString.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN)))  // Change to incorporate differences in +91 and 0 prefixes
            {
               u16FavoriteIdList<<(query.value(1)).value<quint16>();
               ETG_TRACE_USR4(("Inserted"));
            }
         }
         else
         {
            if(sTelephoneNumber == (query.value(OSAL_NULL).toString()) )
            {
               u16FavoriteIdList<<(query.value(1)).value<quint16>();
               ETG_TRACE_USR4(("Inserted"));
            }
         }
      }
      blRet= true;
   }
   else
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(("ERROR - SQL query failed in GetAllFavoritesIDForDevice "));
   }

   return blRet;
}


//FIX GMMY15-5669 Synchronisation problem between contact addresses in phonebook, HMI and navigation component.
/*******************************************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::MarkContactHandle()
*
* DESCRIPTION:
*             The 24th bit of the ContactHandle shall be reserved to carry the following information.
If the Contact is from the Vehicle Contact list (24th bit = 0) or from any BT Phone Contact list (24th bit = 1).
This function does this marking.
* PARAMETER:
*
* RETURNVALUE: NONE
*
********************************************************************************************************/
void FC_PhoneBook_SQLite::MarkContactHandle(const quint8 TableIndex, quint32& u32ContactHandle)
{
   ETG_TRACE_USR4(("Function MarkContactHandle entered with Contact Handle: %d, TableIndex: %d", u32ContactHandle, TableIndex));

   if(TRUE != m_bIsVehiclePbk)
   {
      //If contact is to be inserted to device PB, then set Marker bit to One
      ETG_TRACE_USR4(("Adding FC_PB_CONTACT_HANDLE_DPB_MARKER"));
      u32ContactHandle = u32ContactHandle | FC_PB_CONTACT_HANDLE_DPB_MARKER;//BT Phone Contact list (24th bit = 1)
   }
   //else if contact is to be inserted to VPB, then no need to update Marker bit as it is already Zero
   ETG_TRACE_USR4(("Marked Contact Handle %d", u32ContactHandle));
   m_bIsVehiclePbk = false;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::SearchPhoneNumberEntriesInPB(
*
* DESCRIPTION:
*              Search for Phone Number from Phonebook Database
*
*
* PARAMETER:  [IN]  - PhoneNumberdigits to search
*             [OUT] -
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::SearchPhoneNumberEntriesInPB(QString PhNumberToSearch,quint8 Index,quint8 devicehandle)
{
   ETG_TRACE_USR4(("SearchPhoneNumberEntriesInPB entered with Index %d",Index));

   QMutexLocker lock(&m_mutex);
   QSqlDatabase db= Database();

   AutoCompletePhoneNumberEntryResult PhNoContactsInfo;

   if(!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   if(!CHECK_VALID(devicehandle))
   {
      ETG_TRACE_USR4(( "FC_PhoneBook_SQLite: devicehandle is invalid "));
      return false;
   }

   QSqlQuery query(db);
   // Get the contact details for the given phone number
   // JAC-Fix-CMG3GB-90(Name Should show last name and First name(matches contact name) in the KEYPAD view while user entering a number)
#ifdef PB_PHONENUMBERS_IN_PRETTY_PRINT
   QString sql = "SELECT * FROM (SELECT (" FC_PB_COL_FIRST_NAME " ),(" FC_PB_COL_LAST_NAME "),(" FC_PB_COL_NUMPRETTY1 ") FROM "+ fc_pb_aTableNames[devicehandle] +" WHERE " FC_PB_COL_PHONENUM1 " LIKE '"+PhNumberToSearch+"%'" UNION
      "SELECT  ( " FC_PB_COL_FIRST_NAME " ),(" FC_PB_COL_LAST_NAME "),(" FC_PB_COL_NUMPRETTY2 ") FROM "+ fc_pb_aTableNames[devicehandle] +" WHERE " FC_PB_COL_PHONENUM2 " LIKE '"+PhNumberToSearch+"%'" UNION
      "SELECT  ( " FC_PB_COL_FIRST_NAME " ),(" FC_PB_COL_LAST_NAME "),(" FC_PB_COL_NUMPRETTY3 ") FROM "+ fc_pb_aTableNames[devicehandle] +" WHERE " FC_PB_COL_PHONENUM3 " LIKE '"+PhNumberToSearch+"%'" UNION
      "SELECT  ( " FC_PB_COL_FIRST_NAME " ),(" FC_PB_COL_LAST_NAME "),(" FC_PB_COL_NUMPRETTY4 ") FROM "+ fc_pb_aTableNames[devicehandle] +" WHERE " FC_PB_COL_PHONENUM4 " LIKE '"+PhNumberToSearch+"%'" UNION
      "SELECT  ( " FC_PB_COL_FIRST_NAME " ),(" FC_PB_COL_LAST_NAME "),(" FC_PB_COL_NUMPRETTY5 ") FROM "+ fc_pb_aTableNames[devicehandle] +" WHERE " FC_PB_COL_PHONENUM5 " LIKE '"+PhNumberToSearch+"%'" UNION
      "SELECT  ( " FC_PB_COL_FIRST_NAME " ),(" FC_PB_COL_LAST_NAME "),(" FC_PB_COL_NUMPRETTY6 ") FROM "+ fc_pb_aTableNames[devicehandle] +" WHERE " FC_PB_COL_PHONENUM6 " LIKE '"+PhNumberToSearch+"%'" UNION
      "SELECT  ( " FC_PB_COL_FIRST_NAME " ),(" FC_PB_COL_LAST_NAME "),(" FC_PB_COL_NUMPRETTY7 ") FROM "+ fc_pb_aTableNames[devicehandle] +" WHERE " FC_PB_COL_PHONENUM7 " LIKE '"+PhNumberToSearch+"%'" UNION
      "SELECT  ( " FC_PB_COL_FIRST_NAME " ),(" FC_PB_COL_LAST_NAME "),(" FC_PB_COL_NUMPRETTY8 ") FROM "+ fc_pb_aTableNames[devicehandle] +" WHERE " FC_PB_COL_PHONENUM8 " LIKE '"+PhNumberToSearch+"%')" ;
#else
   QString sql = "SELECT * FROM (SELECT (" FC_PB_COL_FIRST_NAME " ),(" FC_PB_COL_LAST_NAME "),(" FC_PB_COL_PHONENUM1 ") FROM "+ fc_pb_aTableNames[devicehandle] +" WHERE " FC_PB_COL_PHONENUM1 " LIKE '"+PhNumberToSearch+"%'" UNION
      "SELECT  ( " FC_PB_COL_FIRST_NAME " ),(" FC_PB_COL_LAST_NAME "),(" FC_PB_COL_PHONENUM2 ") FROM "+ fc_pb_aTableNames[devicehandle] +" WHERE " FC_PB_COL_PHONENUM2 " LIKE '"+PhNumberToSearch+"%'" UNION
      "SELECT  ( " FC_PB_COL_FIRST_NAME " ),(" FC_PB_COL_LAST_NAME "),(" FC_PB_COL_PHONENUM3 ") FROM "+ fc_pb_aTableNames[devicehandle] +" WHERE " FC_PB_COL_PHONENUM3 " LIKE '"+PhNumberToSearch+"%'" UNION
      "SELECT  ( " FC_PB_COL_FIRST_NAME " ),(" FC_PB_COL_LAST_NAME "),(" FC_PB_COL_PHONENUM4 ") FROM "+ fc_pb_aTableNames[devicehandle] +" WHERE " FC_PB_COL_PHONENUM4 " LIKE '"+PhNumberToSearch+"%'" UNION
      "SELECT  ( " FC_PB_COL_FIRST_NAME " ),(" FC_PB_COL_LAST_NAME "),(" FC_PB_COL_PHONENUM5 ") FROM "+ fc_pb_aTableNames[devicehandle] +" WHERE " FC_PB_COL_PHONENUM5 " LIKE '"+PhNumberToSearch+"%'" UNION
      "SELECT  ( " FC_PB_COL_FIRST_NAME " ),(" FC_PB_COL_LAST_NAME "),(" FC_PB_COL_PHONENUM6 ") FROM "+ fc_pb_aTableNames[devicehandle] +" WHERE " FC_PB_COL_PHONENUM6 " LIKE '"+PhNumberToSearch+"%'" UNION
      "SELECT  ( " FC_PB_COL_FIRST_NAME " ),(" FC_PB_COL_LAST_NAME "),(" FC_PB_COL_PHONENUM7 ") FROM "+ fc_pb_aTableNames[devicehandle] +" WHERE " FC_PB_COL_PHONENUM7 " LIKE '"+PhNumberToSearch+"%'" UNION
      "SELECT  ( " FC_PB_COL_FIRST_NAME " ),(" FC_PB_COL_LAST_NAME "),(" FC_PB_COL_PHONENUM8 ") FROM "+ fc_pb_aTableNames[devicehandle] +" WHERE " FC_PB_COL_PHONENUM8 " LIKE '"+PhNumberToSearch+"%')" ;
#endif
   sql += " ORDER BY ";

   sql += " CASE WHEN " FC_PB_COL_LAST_NAME " = '' THEN " FC_PB_COL_FIRST_NAME " COLLATE " SORTLIB_COLLATE_NAME
      " ELSE " FC_PB_COL_LAST_NAME " COLLATE " SORTLIB_COLLATE_NAME " END ," FC_PB_COL_FIRST_NAME" COLLATE " SORTLIB_COLLATE_NAME;

   sql += " LIMIT " + QString::number(MAX_SIZE_SEARCH_RESULT_LIST);

   ETG_TRACE_USR2(( "FC_PhoneBook_SQLite: Query to Get Phone Nos - %s ",sql.toUtf8().constData()));

   if(!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(( "FC_PhoneBook_SQLite: SearchPhoneNumberDigits Query Failed from retrieving matched records from Phonebook table"));
      return false;
   }
   else
   {
      ETG_TRACE_USR4(( "FC_PhoneBook_SQLite: AutoCompletePhoneNumberEntry Details Retrieved from Phonebook Database"));
   }

   AutoCompletePhNoSearchRecords[Index].clear();
   SearchedPhoneNumberDigit[Index] = PhNumberToSearch;

   while(query.next())
   {
      int counter = 0;
      PhNoContactsInfo.FirstName     = query.value(counter++).toString();
      PhNoContactsInfo.LastName      = query.value(counter++).toString();
      PhNoContactsInfo.PhoneNumber   = query.value(counter++).toString();

      AutoCompletePhNoSearchRecords[Index].append(PhNoContactsInfo);

      ETG_TRACE_USR4(("Phone number added in Search List: %s", PhNoContactsInfo.PhoneNumber.toLatin1().constData()));
      ETG_TRACE_USR4(("LastName added in Search List: %s", PhNoContactsInfo.LastName.toLatin1().constData()));
      ETG_TRACE_USR4(("FirstName added in Search List: %s", PhNoContactsInfo.FirstName.toLatin1().constData()));
   }

   ETG_TRACE_USR2(( "Size = %d",AutoCompletePhNoSearchRecords[Index].size()));
   ETG_TRACE_USR2(( "PhNoContactsInfo.PhoneNumberToSearch = %s",SearchedPhoneNumberDigit[Index].toUtf8().constData()));

   if(AutoCompletePhNoSearchRecords[Index].empty())
   {
      ETG_TRACE_USR4(( "FC_PhoneBook_SQLite: No matching entry found in Phonebook Records"));
      return false;
   }
   return true;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::SearchContacts(
*
* DESCRIPTION:
*              Search for matching Contacts from Phonebook Database
*
*
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::SearchContacts(QList<quint8> DeviceHandleList, QString sSearchString, FC_PhoneBook_SQLite::SortType enSearchSortType, tU16 u16FieldSelector, tU16 u16MaximumReportedMatches, QList<
      SearchContacts_Result>& SearchResultList)
{
   ETG_TRACE_USR4(("FC_PhoneBook_SQLite::SearchContacts entered."));

   QMutexLocker lock(&m_mutex);

   QSqlDatabase db = Database();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR((" Database is not open "));
      return FALSE;
   }

   QSqlQuery query(db);

   QTime TimeToQuery;
   TimeToQuery.start();

   QList < QString > FieldSelectors;
   if (PB_FieldSelector_FirstName == (u16FieldSelector
         & PB_FieldSelector_FirstName))
   {
      FieldSelectors << FC_PB_COL_FIRST_NAME;
   }
   if (PB_FieldSelector_LastName == (u16FieldSelector
         & PB_FieldSelector_LastName))
   {
      FieldSelectors << FC_PB_COL_LAST_NAME;
   }
   if (PB_FieldSelector_PoiName == (u16FieldSelector & PB_FieldSelector_PoiName))
   {
      FieldSelectors << FC_PB_COL_POI_NAME;
   }
   if (PB_FieldSelector_Address == (u16FieldSelector & PB_FieldSelector_Address))
   {
      FieldSelectors << FC_PB_COL_HOME_ADD;
      FieldSelectors << FC_PB_COL_WORK_ADD;
      FieldSelectors << FC_PB_COL_OTHER_ADD;
   }
   if (PB_FieldSelector_Email == (u16FieldSelector & PB_FieldSelector_Email))
   {
      FieldSelectors << FC_PB_COL_EMAIL_ADD1;
      FieldSelectors << FC_PB_COL_EMAIL_ADD2;
      FieldSelectors << FC_PB_COL_EMAIL_ADD3;
   }
   if (PB_FieldSelector_Number == (u16FieldSelector & PB_FieldSelector_Number))
   {
      FieldSelectors << FC_PB_COL_PHONENUM1;
      FieldSelectors << FC_PB_COL_PHONENUM2;
      FieldSelectors << FC_PB_COL_PHONENUM3;
      FieldSelectors << FC_PB_COL_PHONENUM4;
      FieldSelectors << FC_PB_COL_PHONENUM5;
      FieldSelectors << FC_PB_COL_PHONENUM6;
      FieldSelectors << FC_PB_COL_PHONENUM7;
      FieldSelectors << FC_PB_COL_PHONENUM8;
   }

   QString sql_where_clause_substitute = " WHERE ";
   for (tU8 u8Itr = 0; u8Itr < FieldSelectors.size(); u8Itr++)
     {
        sql_where_clause_substitute += FieldSelectors[u8Itr] + " LIKE '%"
              + sSearchString + "%' ";
        if (u8Itr != FieldSelectors.size()-1)
        {
           sql_where_clause_substitute += " OR ";
        }
     }

     ETG_TRACE_USR4((" SQL String used for searching sql_where_clause_substitute: %s", sql_where_clause_substitute.toUtf8().constData()));

     // Set the maxiumum character size for "reported" FirstName or LastName according to GIS-361: FBlock Phonebook,
     // IPhoneBook.RequestPhoneBookListSlice requirements:

#define FC_PB_REPORTED_NAME_MAX_LEN " 50 "

   // Definition of helper strings, which are used in the queries.
   QString sql_case_substitute_check_sortType__FirstName_Sort = // Needed for SortName / FirstName sorting types
         "  CASE                       "
            "     WHEN   FirstName != ''  THEN   FirstName || ' ' || LastName "
            "     ELSE CASE                                "
            "     WHEN   LastName  != ''  THEN   LastName  "
            "     ELSE CASE                                "
            "     WHEN   PoiName   != ''  THEN   PoiName   ";

   QString sql_case_substitute_check_sortType__LastName_Sort = // Needed for SortName / LastName sorting types
         "  CASE                       "
            "     WHEN   LastName  != ''  THEN   LastName || ' ' || FirstName "
            "     ELSE CASE                                "
            "     WHEN   FirstName != ''  THEN   FirstName "
            "     ELSE CASE                                "
            "     WHEN   PoiName   != ''  THEN   PoiName   ";

   QString sql_case_substitute_check_sortType__FirstName_View = // Needed for ReportedFirstName / FirstName sorting types
         "  CASE                                        "
            "     WHEN   FirstName != ''  THEN   FirstName "
            "     ELSE CASE                                "
            "     WHEN   LastName  != ''  THEN   LastName  "
            "     ELSE CASE                                "
            "     WHEN   PoiName   != ''  THEN   PoiName   ";

   QString sql_case_substitute_check_sortType__LastName_View = // Needed for ReportedLastName / LastName sorting types
         "  CASE                                        "
            "     WHEN   LastName  != ''  THEN   LastName  "
            "     ELSE CASE                                "
            "     WHEN   FirstName != ''  THEN   FirstName "
            "     ELSE CASE                                "
            "     WHEN   PoiName   != ''  THEN   PoiName   ";

   QString sql_case_last_name_already_used_as_first_name_sortType_FirstName = // Needed for ReportedFirstName / FirstName sorting types
         " CASE "
            "    WHEN   FirstName != '' "
            "       THEN LastName "
            "       ELSE '' "
            "    END ";

   QString sql_case_first_name_already_used_as_last_name_sortType_LastName = // Needed for ReportedLastName / LastName sorting types
         " CASE "
            "    WHEN   LastName != '' "
            "       THEN FirstName "
            "       ELSE '' "
            "    END ";
#ifdef PB_PHONENUMBERS_IN_PRETTY_PRINT

   QString sql_case_substitute_check_Number =
         "     ELSE CASE                                "
            "     WHEN   PPPhoneNumber1   != ''  THEN   PPPhoneNumber1   "
            "     ELSE CASE                                "
            "     WHEN   PPPhoneNumber2  != ''  THEN   PPPhoneNumber2  "
            "     ELSE CASE                                "
            "     WHEN   PPPhoneNumber3  != ''  THEN   PPPhoneNumber3  "
            "     ELSE CASE                                "
            "     WHEN   PPPhoneNumber4  != ''  THEN   PPPhoneNumber4  "
            "     ELSE CASE                                "
            "     WHEN   PPPhoneNumber5  != ''  THEN   PPPhoneNumber5  "
            "     ELSE CASE                                "
            "     WHEN   PPPhoneNumber6  != ''  THEN   PPPhoneNumber6  "
            "     ELSE CASE                                "
            "     WHEN   PPPhoneNumber7  != ''  THEN   PPPhoneNumber7  "
            "     ELSE CASE                                "
            "     WHEN   PPPhoneNumber8  != ''  THEN   PPPhoneNumber8  ";
#else
   QString sql_case_substitute_check_Number =
         "     ELSE CASE                                "
            "     WHEN   PhoneNumber1   != ''  THEN   PhoneNumber1   "
            "     ELSE CASE                                "
            "     WHEN   PhoneNumber2  != ''  THEN   PhoneNumber2  "
            "     ELSE CASE                                "
            "     WHEN   PhoneNumber3  != ''  THEN   PhoneNumber3  "
            "     ELSE CASE                                "
            "     WHEN   PhoneNumber4  != ''  THEN   PhoneNumber4  "
            "     ELSE CASE                                "
            "     WHEN   PhoneNumber5  != ''  THEN   PhoneNumber5  "
            "     ELSE CASE                                "
            "     WHEN   PhoneNumber6  != ''  THEN   PhoneNumber6  "
            "     ELSE CASE                                "
            "     WHEN   PhoneNumber7  != ''  THEN   PhoneNumber7  "
            "     ELSE CASE                                "
            "     WHEN   PhoneNumber8  != ''  THEN   PhoneNumber8  ";
#endif
   QString sql_case_substitute_check_Address =
         "     ELSE CASE                                "
            "     WHEN   HomeAdd   != ''  THEN   HomeAdd   "
            "     ELSE CASE                                "
            "     WHEN   WorkAdd   != ''  THEN   WorkAdd   "
            "     ELSE CASE                                "
            "     WHEN   OtherAdd  != ''  THEN   OtherAdd  ";

   QString sql_case_substitute_check_Email =
         "     ELSE CASE                                "
            "     WHEN   EmailAdd1 != ''  THEN   EmailAdd1 "
            "     ELSE CASE                                "
            "     WHEN   EmailAdd2 != ''  THEN   EmailAdd2 "
            "     ELSE CASE                                "
            "     WHEN   EmailAdd3 != ''  THEN   EmailAdd3 ";

   QString sql_case_substitute_check_Category_Notes_ENDs =
         "     ELSE CASE                                "
            "     WHEN   Category  != ''  THEN   Category  "
            "     ELSE CASE                                "
            "     WHEN   Notes     != ''  THEN   Notes     "
            "  "
            "     END END END                     "
            "     END END END END END END END END "
            "     END END END                     "
            "     END END END                     "
            "     END END                         ";


   ETG_TRACE_USR4((" Entered new query for search contact"));

   QString sql_case_substitute_get_EmailAddressCountForSearchContact =
         "( (" FC_PB_COL_EMAIL_ADD1 " != '') + (" FC_PB_COL_EMAIL_ADD2 " != '') + (" FC_PB_COL_EMAIL_ADD3 " != '') ) "
         " AS EmailCount";
   //CRQ 428861 - Fetch the addresses count from the database for ContactMatching
   QString sql_case_substitute_get_AddressesCountForSearchContact =
         ", ( (" FC_PB_COL_HOME_ADD " != '') + (" FC_PB_COL_WORK_ADD " != '') + (" FC_PB_COL_OTHER_ADD " != '') ) "
         " AS AddressesCount";


   QString sql_case_substitute_get_PhotoAvailableForSearchContact =
          " , CASE"
             " WHEN " FC_PB_COL_PHOTO_DL_SIZE "!=0  THEN 1"
             " ELSE 0"
             " END"
             " AS bPhotoAvailable";

#ifdef PB_PHONENUMBERS_IN_PRETTY_PRINT
   QString sql_Additional_Contactinfo = " , " FC_PB_COL_NUMPRETTY1 " , " FC_PB_COL_NUMPRETTY2 " , " FC_PB_COL_NUMPRETTY3 " , " FC_PB_COL_NUMPRETTY4 " , " FC_PB_COL_NUMPRETTY5 " , " FC_PB_COL_NUMPRETTY6 " , " FC_PB_COL_NUMPRETTY7
   " , " FC_PB_COL_NUMPRETTY8;
#else
   QString
         sql_Additional_Contactinfo =
               " , " FC_PB_COL_PHONENUM1 " , " FC_PB_COL_PHONENUM2 " , " FC_PB_COL_PHONENUM3 " , " FC_PB_COL_PHONENUM4 " , " FC_PB_COL_PHONENUM5 " , " FC_PB_COL_PHONENUM6 " , " FC_PB_COL_PHONENUM7
               " , " FC_PB_COL_PHONENUM8;
#endif
   QString
         sql_case_NumberType =
               " , " FC_PB_COL_NUMTYPE1 " , " FC_PB_COL_NUMTYPE2 " ," FC_PB_COL_NUMTYPE3 " , " FC_PB_COL_NUMTYPE4 " , " FC_PB_COL_NUMTYPE5 " , " FC_PB_COL_NUMTYPE6 " , " FC_PB_COL_NUMTYPE7 " , " FC_PB_COL_NUMTYPE8;
   //CRQ 428861 - Fetch the addresses count from the database for ContactMatching
   QString sql_Additional_Contactinfo_substitute = sql_Additional_Contactinfo + " , " FC_PB_COL_EMAIL_ADD1 " , " FC_PB_COL_EMAIL_ADD2 " , " FC_PB_COL_EMAIL_ADD3 " , "
         + sql_case_substitute_get_EmailAddressCountForSearchContact
         + sql_case_substitute_get_AddressesCountForSearchContact
         + sql_case_substitute_get_PhotoAvailableForSearchContact
         + sql_case_NumberType;

   //CMG3G-13535 Fix
   QString sql = " ";
   for(tU8 u8Index = 0; u8Index < DeviceHandleList.size(); u8Index++)
   {
      ETG_TRACE_USR4(("Forming SQL query for devicehandle: %d", DeviceHandleList[u8Index] ));
      switch (enSearchSortType)
      {
         case FirstName:
         case Number_FirstName:
         {
            // Construct whole query string for sortType FirstName
            sql
                  += " SELECT SUBSTR( "
                        + sql_case_substitute_check_sortType__FirstName_Sort
                        + sql_case_substitute_check_Number
                        + sql_case_substitute_check_Address
                        + sql_case_substitute_check_Email
                        + sql_case_substitute_check_Category_Notes_ENDs + " , 1, "
                        + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                     " AS SortName, "
                     "  "
                     " SUBSTR( "
                        + sql_case_substitute_check_sortType__FirstName_View
                        + sql_case_substitute_check_Number
                        + sql_case_substitute_check_Address
                        + sql_case_substitute_check_Email
                        + sql_case_substitute_check_Category_Notes_ENDs + " , 1, "
                        + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                     " AS ReportedFirstName, "
                     "  "
                        + sql_case_last_name_already_used_as_first_name_sortType_FirstName
                        + " AS ReportedLastName, ContactHandle " +sql_Additional_Contactinfo_substitute+
                           " "
                           " FROM " ;
         }
         break;

         case LastName:
         case Number_LastName:
         {
            // Construct whole query string for sortType LastName
            sql
                  += " SELECT SUBSTR( "
                        + sql_case_substitute_check_sortType__LastName_Sort
                        + sql_case_substitute_check_Number
                        + sql_case_substitute_check_Address
                        + sql_case_substitute_check_Email
                        + sql_case_substitute_check_Category_Notes_ENDs + " , 1, "
                        + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                     " AS SortName, "
                     "  "
                        + sql_case_first_name_already_used_as_last_name_sortType_LastName
                        + " AS ReportedFirstName, "
                           "  "
                           " SUBSTR( "
                        + sql_case_substitute_check_sortType__LastName_View
                        + sql_case_substitute_check_Number
                        + sql_case_substitute_check_Address
                        + sql_case_substitute_check_Email
                        + sql_case_substitute_check_Category_Notes_ENDs + " , 1, "
                        + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                     " AS ReportedLastName, ContactHandle" +sql_Additional_Contactinfo_substitute+
                     " "
                     "  FROM " ;
         }
         break;

         case Address_FirstName:
         {
            // Construct whole query string for sortType Address_FirstName
            sql
                  += " SELECT SUBSTR( "
                        + sql_case_substitute_check_sortType__FirstName_Sort
                        + sql_case_substitute_check_Address
                        + sql_case_substitute_check_Number
                        + sql_case_substitute_check_Email
                        + sql_case_substitute_check_Category_Notes_ENDs + " , 1, "
                        + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                     " AS SortName, "
                     "  "
                     " SUBSTR( "
                        + sql_case_substitute_check_sortType__FirstName_View
                        + sql_case_substitute_check_Address
                        + sql_case_substitute_check_Number
                        + sql_case_substitute_check_Email
                        + sql_case_substitute_check_Category_Notes_ENDs + " , 1, "
                        + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                     " AS ReportedFirstName, "
                     "  "
                        + sql_case_last_name_already_used_as_first_name_sortType_FirstName
                        + " AS ReportedLastName, ContactHandle" +sql_Additional_Contactinfo_substitute+
                           "  "
                           "  FROM " ;
         }
         break;

         case Address_LastName:
         {
            // Construct whole query string for sortType LastName
            sql
                  += " SELECT  SUBSTR( "
                        + sql_case_substitute_check_sortType__LastName_Sort
                        + sql_case_substitute_check_Address
                        + sql_case_substitute_check_Number
                        + sql_case_substitute_check_Email
                        + sql_case_substitute_check_Category_Notes_ENDs + " , 1, "
                        + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                     " AS SortName, "
                     "  "
                        + sql_case_first_name_already_used_as_last_name_sortType_LastName
                        + " AS ReportedFirstName, "
                           "  "
                           " SUBSTR( "
                        + sql_case_substitute_check_sortType__LastName_View
                        + sql_case_substitute_check_Address
                        + sql_case_substitute_check_Number
                        + sql_case_substitute_check_Email
                        + sql_case_substitute_check_Category_Notes_ENDs + " , 1, "
                        + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                     " AS ReportedLastName, ContactHandle " + sql_Additional_Contactinfo_substitute+
                     "  "
                     "  FROM " ;
         }
         break;

         case Email_FirstName:
         {
            // Construct whole query string for sortType Address_FirstName
            sql
                  += " SELECT  SUBSTR( "
                        + sql_case_substitute_check_sortType__FirstName_Sort
                        + sql_case_substitute_check_Email
                        + sql_case_substitute_check_Number
                        + sql_case_substitute_check_Address
                        + sql_case_substitute_check_Category_Notes_ENDs + " , 1, "
                        + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                     " AS SortName, "
                     "  "
                     " SUBSTR( "
                        + sql_case_substitute_check_sortType__FirstName_View
                        + sql_case_substitute_check_Email
                        + sql_case_substitute_check_Number
                        + sql_case_substitute_check_Address
                        + sql_case_substitute_check_Category_Notes_ENDs + " , 1, "
                        + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                     " AS ReportedFirstName, "
                     "  "
                        + sql_case_last_name_already_used_as_first_name_sortType_FirstName
                        + " AS ReportedLastName, ContactHandle " + sql_Additional_Contactinfo_substitute+
                           "  "
                           "  FROM " ;
         }
         break;

         case Email_LastName:
         {
            // Construct whole query string for sortType LastName
            sql
                  += " SELECT SUBSTR( "
                        + sql_case_substitute_check_sortType__LastName_Sort
                        + sql_case_substitute_check_Email
                        + sql_case_substitute_check_Number
                        + sql_case_substitute_check_Address
                        + sql_case_substitute_check_Category_Notes_ENDs + " , 1, "
                        + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                     " AS SortName, "
                     "  "
                        + sql_case_first_name_already_used_as_last_name_sortType_LastName
                        + " AS ReportedFirstName, "
                           "  "
                           " SUBSTR( "
                        + sql_case_substitute_check_sortType__LastName_View
                        + sql_case_substitute_check_Email
                        + sql_case_substitute_check_Number
                        + sql_case_substitute_check_Address
                        + sql_case_substitute_check_Category_Notes_ENDs + " , 1, "
                        + FC_PB_REPORTED_NAME_MAX_LEN + " ) "
                     " AS ReportedLastName, ContactHandle " + sql_Additional_Contactinfo_substitute+
                     "  "
                     "  FROM " ;
         }
         break;
         default:
         {
            ETG_TRACE_ERR((" Sort type not supported!"));
            return false;
         }
      } // end of: 'switch(sortType)'

     //Query for searching on multiple tables simulataneously
     sql += fc_pb_aTableNames[DeviceHandleList[u8Index]] + sql_where_clause_substitute; //Appending search table to the substring till where clause

     //Using UNION to search in multiple tables in parallel
     if (u8Index != DeviceHandleList.size()-1)
     {
        ETG_TRACE_USR4(("Appending UNION"));
        sql += " UNION ";
     }
     //end of UNION statement
   }

   u16MaximumReportedMatches = (tU16)(u16MaximumReportedMatches + 1);//For MatchLimitExceeded Setting

   //ORDER BY SORTNAME and then LIMIT SEARCH TO u16MaximumReportedMatches
   sql = sql + " ORDER BY SortName COLLATE " + SORTLIB_COLLATE_NAME + " LIMIT " + QString::number(u16MaximumReportedMatches);
   //end of sql query (End of CMG3G-13535 Fix)

   if (!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR((" FC_PhoneBook_SQLite : - ERROR - SQL query failed in SearchContacts "));
      return FALSE;
   }

   int counter = 0;
   while (query.next())
   {
      counter = 0;

      SearchContacts_Result contact;

      // query result columns are: SortName, ReportedFirstName, ReportedLastName, ContactHandle
      counter++; // skip the SortName
      contact.ReportedFirstName = query.value(counter++).toString(); // ReportedFirstName
      contact.ReportedLastName = query.value(counter++).toString(); // ReportedLastName
      contact.ContactHandle = query.value(counter++).toUInt(); // ContactHandle
      contact.PhoneNum1 = query.value(counter++).toString();
      contact.PhoneNum2 = query.value(counter++).toString();
      contact.PhoneNum3 = query.value(counter++).toString();
      contact.PhoneNum4 = query.value(counter++).toString();
      contact.PhoneNum5 = query.value(counter++).toString();
      contact.PhoneNum6 = query.value(counter++).toString();
      contact.PhoneNum7 = query.value(counter++).toString();
      contact.PhoneNum8 = query.value(counter++).toString();
      contact.EmailAdd1 = query.value(counter++).toString();
      contact.EmailAdd2 = query.value(counter++).toString();
      contact.EmailAdd3 = query.value(counter++).toString();
      contact.u8EmailAddrCount = (query.value(counter++)).value<quint8>();
      contact.u8AddressesCount = (query.value(counter++)).value<quint8>(); //CRQ 428861 - Fetch the addresses count from the database for ContactMatching
      contact.bPhotoAvailable = query.value(counter++).toBool();
      contact.NumberType1 = (query.value(counter++)).value<quint8>();
      contact.NumberType2 = (query.value(counter++)).value<quint8>();
      contact.NumberType3 = (query.value(counter++)).value<quint8>();
      contact.NumberType4 = (query.value(counter++)).value<quint8>();
      contact.NumberType5 = (query.value(counter++)).value<quint8>();
      contact.NumberType6 = (query.value(counter++)).value<quint8>();
      contact.NumberType7 = (query.value(counter++)).value<quint8>();
      contact.NumberType8 = (query.value(counter++)).value<quint8>();

      SearchResultList << contact;
   }

   ETG_TRACE_USR2(("Time elapsed in Search Contacts Query - %d ms", TimeToQuery.elapsed()));

   if (SearchResultList.isEmpty())
   {
      ETG_TRACE_USR4(("FC_PhoneBook_SQLite :- WARNING - Empty list of contacts returned"));
   }

   return TRUE;
}


//Fix for GMMY15-11511-Autocompleted numbers are retained even though a different phone is paired and connected.
/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::clearSearchPhoneDigitList(
*
* DESCRIPTION:
*
*
*
* PARAMETER:  [IN]  -
*             [OUT] -
*
* RETURNVALUE:
*
*******************************************************************************/

void FC_PhoneBook_SQLite::clearSearchPhoneDigitList()
{
   ETG_TRACE_USR4(("FC_PhoneBook_SQLite:clearSearchPhoneDigitList: entered"));
   quint32 RecIndex = 0;
   for(RecIndex = 0; RecIndex < MAX_AUTO_COMPLETE_LISTS; RecIndex++)
   {
      SearchedPhoneNumberDigit[RecIndex].clear();
   }
}
//End of fix


//CRQ CMG3G-2888
//CMG3G-2971:Enhanced vcard handling
#ifdef VALID_VCARD_FORMAT_ADDRESS_PHONENUMBER
/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_Contact_Detail::isValidVcardFormatPhoneNumberAddress()
*
* DESCRIPTION: Validation check
*
*
* PARAMETER: None.
*
* RETURNVALUE: None
********************************************************************************/
bool FC_PhoneBook_Contact_Detail::isValidVcardFormatPhoneNumberAddress(FC_PhoneBook_Contact_Detail& contact)//CMG3G-14313-FC_PhoneBook_Contact_Detail-Passed as a reference-Coverity Fix
{
   bool bRetValue = false;
   ETG_TRACE_USR4(("isValidVcardFormatPhoneNumberAddress entered"));
   //Contacts with 1 valid phone number OR not empty  address are stored
   if((contact.PhoneNum1.length() >= MINIMUM_LENGTH_FOR_TEL_NUMBER) \
      || (contact.PhoneNum2.length() >= MINIMUM_LENGTH_FOR_TEL_NUMBER) \
      || (contact.PhoneNum3.length() >= MINIMUM_LENGTH_FOR_TEL_NUMBER) \
      || (contact.PhoneNum4.length() >= MINIMUM_LENGTH_FOR_TEL_NUMBER) \
      || (contact.PhoneNum5.length() >= MINIMUM_LENGTH_FOR_TEL_NUMBER) \
      || (contact.PhoneNum6.length() >= MINIMUM_LENGTH_FOR_TEL_NUMBER) \
      || (contact.PhoneNum7.length() >= MINIMUM_LENGTH_FOR_TEL_NUMBER) \
      || (contact.PhoneNum8.length() >= MINIMUM_LENGTH_FOR_TEL_NUMBER) \
      || ((!contact.homeAdd.isEmpty()) || (!contact.workAdd.isEmpty()) || (!contact.otherAdd.isEmpty())))
   {
      bRetValue = true;
   }

   else
   {
      ETG_TRACE_USR4((" InValid contact"));
   }
   return bRetValue;
}
#endif
//End fix CMG3G-2971


// CMG3G-7853: Phonebook list slice result to have the number and E-Mail details
/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::getAdditionalContactInfo()
*
* DESCRIPTION: This is used to fetch the Phonenumber, phonenumber type.
*
* PARAMETER: None.
*
* RETURNVALUE: None
********************************************************************************/
void FC_PhoneBook_SQLite::getAdditionalContactInfo(quint32 windowstart, unsigned end,
      QList<PbkAdditionalContactInfo>& PbkcontactList,
      QList<FC_PhoneBook_Contact_Detail_Short>& contactList, quint8 Devicehandle, tU8 u8ListType)
{
   ETG_TRACE_USR4(("FC_PhoneBook_SQLite::getAdditionalContactInfo entered"));

   ETG_TRACE_USR4(("End Value::: %d", end));
   ETG_TRACE_USR4(("ListType::: %d", u8ListType));

   FC_PhoneBook_Contact_Detail_Short contact;
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db = Database();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR((" Database is not open "));
      return;
   }

   QSqlQuery query(db);

   QString sql_case_substitute_get_EmailAddress =
         "( (EmailAdd1 != '') + (EmailAdd2 != '') + (EmailAdd3 != '') ) "
         " AS EmailCount";
   //CRQ 300933 - Fetch the addresses count from the database
   QString sql_case_substitute_get_Addresses =
         "( (HomeAdd != '') + (WorkAdd != '') + (OtherAdd != '') ) "
         " AS AddressesCount";

   QString sqlcontacthandlelist = "(";
   for (quint32 i = windowstart; i<end; i++)
   {
      sqlcontacthandlelist += QString::number((contactList.at(i)).contactHandle);
      if (i != end - 1)
      {
        sqlcontacthandlelist += " , ";
      }
   }
   sqlcontacthandlelist += ")";

   ETG_TRACE_USR4(("SQL1::: %s", sql_case_substitute_get_EmailAddress.toUtf8().constData()));
   ETG_TRACE_USR4(("SQL2::: %s", sql_case_substitute_get_Addresses.toUtf8().constData()));

#ifdef PB_PHONENUMBERS_IN_PRETTY_PRINT
   QString  sql = "SELECT  " FC_PB_COL_NUMPRETTY1 " , " FC_PB_COL_NUMPRETTY2 " , " FC_PB_COL_NUMPRETTY3 " , " FC_PB_COL_NUMPRETTY4 " , " FC_PB_COL_NUMPRETTY5 " , " FC_PB_COL_NUMPRETTY6 " , " FC_PB_COL_NUMPRETTY7
                  " , " FC_PB_COL_NUMPRETTY8 " , ";

#else

  QString sql = "SELECT  " FC_PB_COL_PHONENUM1 " , " FC_PB_COL_PHONENUM2 " , " FC_PB_COL_PHONENUM3 " , " FC_PB_COL_PHONENUM4 " , " FC_PB_COL_PHONENUM5 " , " FC_PB_COL_PHONENUM6 " , " FC_PB_COL_PHONENUM7
                  " , " FC_PB_COL_PHONENUM8 " , ";

#endif
   sql += FC_PB_COL_NUMTYPE1 "," FC_PB_COL_NUMTYPE2 "," FC_PB_COL_NUMTYPE3 "," FC_PB_COL_NUMTYPE4 "," FC_PB_COL_NUMTYPE5 "," FC_PB_COL_NUMTYPE6 "," FC_PB_COL_NUMTYPE7 "," FC_PB_COL_NUMTYPE8 "," FC_PB_COL_CONTACT_TYPE " , " FC_PB_COL_EMAIL_ADD1 " , " FC_PB_COL_EMAIL_ADD2 " , " FC_PB_COL_EMAIL_ADD3 " , "
         "CASE"
         " WHEN " FC_PB_COL_PHOTO_DL_SIZE "!=0  THEN 1"
         " ELSE 0"
         " END"
         " AS bPhotoAvailable , " + sql_case_substitute_get_EmailAddress + " , " + sql_case_substitute_get_Addresses;
   if ((u8ListType == ListType_PHONEBOOK) || (u8ListType == ListType_FAVORITE))
   {
    sql+= " FROM " + fc_pb_aTableNames[Devicehandle];
   }

  else if (u8ListType == ListType_VEHICLE)
   {
     sql +=   " FROM " + fc_vpb_aTableNames[Devicehandle];
   }

  else
  {
     ETG_TRACE_USR4(("Invalid list type"));
  }

     sql +=  " WHERE " FC_PB_COL_CONTACT_HANDLE " IN "
         + sqlcontacthandlelist + "ORDER BY ";

   for (quint32 i = windowstart; i < end; i++)
   {
      sql += "( " FC_PB_COL_CONTACT_HANDLE " = "
            + QString::number((contactList.at(i)).contactHandle) + ") DESC";
      if (i != end - 1)
      {
         sql += " , ";
      }
   }

   ETG_TRACE_USR4(("SQL string to be executed= '%s' ", sql.toUtf8().constData()));


   if (!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR((" FC_PhoneBook_SQLite : - ERROR - SQL query failed in getAdditionalContactInfo "));
   }
   else
   {
      tU8 counter = 0;

      while (query.next())
      {
         counter = 0;
         PbkAdditionalContactInfo Pbkcontact;

         Pbkcontact.PhoneNum1 = query.value(counter++).toString();
         ETG_TRACE_USR4(("PhoneNum1 %s", Pbkcontact.PhoneNum1.toLatin1().constData()));
         Pbkcontact.PhoneNum2 = query.value(counter++).toString();
         ETG_TRACE_USR4(("PhoneNum2 %s", Pbkcontact.PhoneNum2.toLatin1().constData()));
         Pbkcontact.PhoneNum3 = query.value(counter++).toString();
         ETG_TRACE_USR4(("PhoneNum3 %s", Pbkcontact.PhoneNum3.toLatin1().constData()));
         Pbkcontact.PhoneNum4 = query.value(counter++).toString();
         ETG_TRACE_USR4(("PhoneNum4 %s", Pbkcontact.PhoneNum4.toLatin1().constData()));
         Pbkcontact.PhoneNum5 = query.value(counter++).toString();
         ETG_TRACE_USR4(("PhoneNum5 %s", Pbkcontact.PhoneNum5.toLatin1().constData()));
         Pbkcontact.PhoneNum6 = query.value(counter++).toString();
         ETG_TRACE_USR4(("PhoneNum6 %s", Pbkcontact.PhoneNum6.toLatin1().constData()));
         Pbkcontact.PhoneNum7 = query.value(counter++).toString();
         ETG_TRACE_USR4(("PhoneNum7 %s", Pbkcontact.PhoneNum7.toLatin1().constData()));
         Pbkcontact.PhoneNum8 = query.value(counter++).toString();
         ETG_TRACE_USR4(("PhoneNum8 %s", Pbkcontact.PhoneNum8.toLatin1().constData()));
         Pbkcontact.NumberType1 = (query.value(counter++)).value<quint8>();
         ETG_TRACE_USR4(("NumberType1 %d", Pbkcontact.NumberType1));
         Pbkcontact.NumberType2 = (query.value(counter++)).value<quint8>();
         ETG_TRACE_USR4(("NumberType2 %d", Pbkcontact.NumberType2));
         Pbkcontact.NumberType3 = (query.value(counter++)).value<quint8>();
         ETG_TRACE_USR4(("NumberType3 %d", Pbkcontact.NumberType3));
         Pbkcontact.NumberType4 = (query.value(counter++)).value<quint8>();
         ETG_TRACE_USR4(("NumberType4 %d", Pbkcontact.NumberType4));
         Pbkcontact.NumberType5 = (query.value(counter++)).value<quint8>();
         ETG_TRACE_USR4(("NumberType5 %d", Pbkcontact.NumberType5));
         Pbkcontact.NumberType6 = (query.value(counter++)).value<quint8>();
         ETG_TRACE_USR4(("NumberType6 %d", Pbkcontact.NumberType6));
         Pbkcontact.NumberType7 = (query.value(counter++)).value<quint8>();
         ETG_TRACE_USR4(("NumberType7 %d", Pbkcontact.NumberType7));
         Pbkcontact.NumberType8 = (query.value(counter++)).value<quint8>();
         ETG_TRACE_USR4(("NumberType8 %d", Pbkcontact.NumberType8));
         Pbkcontact.u8ContactType = (query.value(counter++)).value<quint8>();
         ETG_TRACE_USR4(("u8ContactType %d", Pbkcontact.u8ContactType));
         Pbkcontact.EmailAdd1 = query.value(counter++).toString();
         ETG_TRACE_USR4(("Emailaddress1 %s", Pbkcontact.EmailAdd1.toLatin1().constData()));
         Pbkcontact.EmailAdd2 = query.value(counter++).toString();
         ETG_TRACE_USR4(("Emailaddress2 %s", Pbkcontact.EmailAdd2.toLatin1().constData()));
         Pbkcontact.EmailAdd3 = query.value(counter++).toString();
         ETG_TRACE_USR4(("Emailaddress3 %s", Pbkcontact.EmailAdd3.toLatin1().constData()));
         Pbkcontact.bPhotoAvailable = query.value(counter++).toBool();
         ETG_TRACE_USR4(("bPhotoAvailable %d", Pbkcontact.bPhotoAvailable));
         Pbkcontact.u8EmailAddrCount = (query.value(counter++)).value<quint8>();
         ETG_TRACE_USR4(("u8EmailAddrCount %d", Pbkcontact.u8EmailAddrCount));
         Pbkcontact.u8AddressesCount = (query.value(counter++)).value<quint8>();
         ETG_TRACE_USR4(("u8AddressesCount %d", Pbkcontact.u8AddressesCount));

         PbkcontactList << Pbkcontact;
      }
   }

   if (PbkcontactList.isEmpty())
   {
      ETG_TRACE_USR4(("FC_PhoneBook_SQLite :- WARNING - Empty list of PbkcontactList returned"));
   }
}


//FIX SUZUKI-22811 Phone settings (Ringtone, First/last name, paired device list) to be persistent even after SW update
/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::vGetDevicesInDB
*
* DESCRIPTION: Get all existing handles and addresses from DB.
*
* PARAMETER: [IN]   - None.
*            [OUT]  - DeviceHandlesAndDeviceAddr
* RETURNVALUE: None
********************************************************************************/
void FC_PhoneBook_SQLite::vGetDevicesInDB(std::map<tU8, std::string>& DeviceHandlesAndDeviceAddr) // 241829- Replace bpstl:: by std::
{
   ETG_TRACE_USR4((" vGetDevicesInDB() entered. "));

   // FIX PSARCCB-9855 (06/2016):  // Added mutex
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db = Database();
   if (db.isOpen())
   {
      QSqlQuery query(db);
      QString sql = "SELECT " FC_PB_FS_COL_DEVICE_HANDLE " , " FC_PB_FS_COL_DEVICE_ADDRESS " FROM " FC_PB_TABLE_NAME_FS;
      if (!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" vGetDevicesInDB FAILED!!! "));
      }
      else
      {
         ETG_TRACE_USR4((" vGetDevicesInDB SUCCESS! "));
         while(query.next())
         {
            DeviceHandlesAndDeviceAddr.insert(std::pair<tU8, std::string>((query.value(0)).value<tU8>(), query.value(1).toString().toStdString().c_str())); // 241829- Replace bpstl:: by std::
            ETG_TRACE_USR4(( " Device Handle inserted to DeviceHandlesAndDeviceAddr: %d ", query.value( 0 ).toUInt() ));
            ETG_TRACE_USR4(( " Device Address inserted to DeviceHandlesAndDeviceAddr: %s ", query.value( 1 ).toString().toStdString().c_str() ));
         }
      }
   }
   else
   {
      ETG_TRACE_ERR((" Database is not open!!! "));
   }
}
//End of fix


//FIX CMG3G-8367 IS2424_DownloadOnOff behaviour@FC_Phonebook
// The user shall have possibility to switch the phonebook and call history list download On or Off individual for each paired phone device.
/*******************************************************************************
 *
 * FUNCTION: FC_PhoneBook_SQLite::UpdateSwitchDownloadOnOffState()
 *
 * DESCRIPTION: Updates the DownloadOnOffState of particular device handle
 *
 *
 * PARAMETER: IN - DeviceHandle, SwitchDownloadOnOffState
 *
 * RETURNVALUE: bool - returns the success or failure of the function
 ********************************************************************************/
bool FC_PhoneBook_SQLite::UpdateSwitchDownloadOnOffState(quint8 DeviceHandle, quint8 SwitchDownloadOnOffState)
{
   ETG_TRACE_USR4(("UpdateSwitchDownloadOnOffState entered."));
   QMutexLocker lock(&m_mutex);
   QString sql;
   QSqlDatabase db = Database();
   bool bRetValue = false;
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {

      QSqlQuery query(db);

      sql = "UPDATE " FC_PB_TABLE_NAME_FS " SET "
      FC_PB_FS_COL_PB_DOWNLOAD_ONOFF" = " + QString::number(SwitchDownloadOnOffState) + " WHERE "
      FC_PB_FS_COL_DEVICE_HANDLE" = " + QString::number(DeviceHandle);

      ETG_TRACE_USR2(("UpdateSwitchDownloadOnOffState %s", sql.toLatin1().constData()));

      if (!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR(("ERROR - SQL query failed in UpdateSwitchDownloadOnOffState For Updation "));
      }
      else
      {
         ETG_TRACE_USR4(("UpdateSwitchDownloadOnOffState EXIT."));
         bRetValue = true;
      }
   }
   return bRetValue;
}
//End of fix FIX CMG3G-8367


//FIX CMG3G-8367 IS2424_DownloadOnOff behaviour@FC_Phonebook
// The user shall have possibility to switch the phonebook and call history list download On or Off individual for each paired phone device.
/*******************************************************************************
 *
 * FUNCTION: FC_PhoneBook_SQLite::GetDownloadOnOffStateList(
 *
 * DESCRIPTION:
 *              Updates DownloadOnOffState List fetched from database
 *
 *
 * PARAMETER:  [IN]  - Map object containing device handle and Download on off state of all the device handles
 *             [OUT] -
 *
 * RETURNVALUE: bool - indicating success or failure of this function
 *
 *******************************************************************************/
bool FC_PhoneBook_SQLite::GetDownloadOnOffStateList(QList<quint8>& DeviceHandleList, QList<
      quint16>& DownloadOnOffStateList)
{
   ETG_TRACE_USR4(("GetDownloadOnOffStateList entered"));
   bool bRetValue = false;
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db = Database();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {

      QSqlQuery SelectQuery(db);

      QString SelectSql = "SELECT " FC_PB_FS_COL_DEVICE_HANDLE " , " FC_PB_FS_COL_PB_DOWNLOAD_ONOFF " FROM " FC_PB_TABLE_NAME_FS;

      ETG_TRACE_USR2(("Select Query::%s", SelectSql.toLatin1().constData()));

      if (!SelectQuery.exec(SelectSql))
      {
         PB_CHECK_QUERY_ERROR(SelectQuery); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime
      }
      else
      {
         while (db.isOpen() && SelectQuery.next())
         {
            DeviceHandleList << (SelectQuery.value(0)).value<quint8>();
            DownloadOnOffStateList << (SelectQuery.value(1)).value<quint16>();
         }
         bRetValue = true;
      }
   }

   return bRetValue;
}
//End of fix FIX CMG3G-8367


//FIX CMG3G-8367 IS2424_DownloadOnOff behaviour@FC_Phonebook
// The user shall have possibility to switch the phonebook and call history list download On or Off individual for each paired phone device.
/*******************************************************************************
 *
 * FUNCTION: FC_PhoneBook_SQLite::u16GetDownloadOnOffStateForDevicehandle
 * DESCRIPTION: To Get the DownloadOnOffState For Devicehandle
 *
 * PARAMETER: [IN]  - Devicehandle
 *            [OUT] -
 *
 * RETURNVALUE: Download on off status of the particular device handle
 *******************************************************************************/
quint16 FC_PhoneBook_SQLite::u16GetDownloadOnOffStateForDevicehandle(quint8 u8DeviceHandle)
{
   ETG_TRACE_USR4(("u16GetDownloadOnOffStateForDevicehandle entered with DeviceHandle %d ", u8DeviceHandle));
   quint16 u16DownloadOnOffState = VALUE_ZERO;
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db = Database();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);

      QString sql = "SELECT " FC_PB_FS_COL_PB_DOWNLOAD_ONOFF " FROM "
      FC_PB_TABLE_NAME_FS " WHERE " FC_PB_FS_COL_DEVICE_HANDLE " = " + QString::number(u8DeviceHandle);

      ETG_TRACE_USR2(("FC_PhoneBook_SQLite: String used for master table in u16GetDownloadOnOffStateForDevicehandle - %s ", sql.toLatin1().constData()));

      if (!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_USR4(("FC_PhoneBook_SQLite: u16GetDownloadOnOffStateForDevicehandle query failed"));
      }
      else
      {
         if (!query.next())
         {
            ETG_TRACE_USR4(("FC_PhoneBook_SQLite: u16GetDownloadOnOffStateForDevicehandle - No entry found for the device handle"));
         }
         else
         {
            u16DownloadOnOffState = (query.value(VALUE_ZERO)).value<quint16>();

            ETG_TRACE_USR4(("FC_PhoneBook_SQLite:u16GetDownloadOnOffStateForDevicehandle : u16DownloadOnOffState value  %d ", u16DownloadOnOffState));
         }
      }
   }

   return u16DownloadOnOffState;
}


/*******************************************************************************
 *
 * FUNCTION: FC_PhoneBook_SQLite::bDeletePBContacts
 *
 * DESCRIPTION:
 *              Deletes all the phonebook contacts of a device in database
 *
 *
 * PARAMETER:  [IN]  - device handle of the device
 *          [OUT] -
 *
 * RETURNVALUE: bool - indicating success or failure of this function
 *
 *******************************************************************************/
bool FC_PhoneBook_SQLite::bDeletePBContacts(quint8 u8DeviceHandle)
{
   ETG_TRACE_USR2((" bDeletePBContacts entered with DeviceHandle %d ", u8DeviceHandle));
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db = Database();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }
   QSqlQuery query(db);

   if (!CHECK_VALID(u8DeviceHandle))
   {
      ETG_TRACE_USR4((" FC_PhoneBook_SQLite: device handle is invalid in function"
            "bDeletePBContacts "));
      return false;
   }

   QString sql = "DELETE FROM " + fc_pb_aTableNames[u8DeviceHandle];

   if (!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Deleting of all the records of a table has failed"));
      return false;
   }
   else
   {
      ETG_TRACE_USR4((" FC_PhoneBook_SQLite: Delete of all the records of a device was successful "));
      sql.clear();
      sql = "DELETE FROM " FC_PB_MASTER_TABLE_NAME " WHERE " FC_PB_MASTER_COL_TABLE_INDEX " = " + QString::number(u8DeviceHandle);

      if (!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Deleting of all contacts of a device in master table has failed "));
         return false;
      }
      else
      {
         ETG_TRACE_USR4((" FC_PhoneBook_SQLite: Deleting of all contacts of a device in master table was successful "));
      }
   }

   SpeechUpdateCheckSumTable(u8DeviceHandle);
   return true;
}
//End of fix FIX CMG3G-8367


//FIX CMG3G-8372 IS2430_SetContactPhoto@FC_Phonebook
//The user shall have possibility to set the photo for a specific contact in Vehicle phonebook of a specific device.
/*******************************************************************************
 *
 * FUNCTION: FC_PhoneBook_SQLite::SetContactPhoto()
 *
 * DESCRIPTION: Updates the Contact Photo in Vehicle phonebook of a specific device.
 *
 *
 * PARAMETER: IN - ContactHandle, PhotoData
 *
 * RETURNVALUE: bool - returns the success or failure of the function
 ********************************************************************************/
bool FC_PhoneBook_SQLite::SetContactPhoto(quint32 u32ContactHandle, QByteArray PhotoData)
{
   ETG_TRACE_USR4((" SetContactPhoto entered  "));

   QMutexLocker lock(&m_mutex);

   QSqlDatabase db = Database();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   QSqlQuery query1(db);
   QSqlQuery query2(db);

   QString sql = "SELECT " FC_PB_MASTER_COL_TABLE_INDEX " FROM "
   FC_PB_MASTER_TABLE_NAME " WHERE " FC_PB_MASTER_COL_CONTACT_HANDLE " = " + QString::number(u32ContactHandle);

   if (!query1.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query1); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(("ERROR - SQL query failed in selecting master table "));
      return false;
   }

   if (!query1.next())
   {
      ETG_TRACE_ERR((" FC_PhoneBook_SQLite: SetPhoto - No entry found for the contact handle"));
      return false;
   }

   quint8 tableindex = (query1.value(0)).value<quint8>();
   ETG_TRACE_USR4((" SetPhoto tableindex: %d  ", tableindex));

   quint32 PhotoSize = (quint32)PhotoData.size();
   quint16 PhotoChkSum = qChecksum(PhotoData.constData(), PhotoSize);
   quint32 PhotoDlSize = PhotoSize;
   quint16 PhotoDlCRC = PhotoChkSum;
   QString photoType = " ";

   QString sqlpd = "UPDATE " + fc_vpb_aTableNames[tableindex] + " SET "
   FC_PB_COL_PHOTO_DL_SIZE " = ?, "
   FC_PB_COL_PHOTO " = ?, "
   FC_PB_COL_PHOTO_TYPE " = ?, "
   FC_PB_COL_PHOTO_DL_CRC " = ?, "
   FC_PB_COL_PHOTO_SIZE " = ?, "
   FC_PB_COL_PHOTO_CRC " = ?  "
   "WHERE ContactHandle = ? ";

   query2.prepare(sqlpd);

   query2.addBindValue(PhotoDlSize);
   query2.addBindValue(PhotoData);
   query2.addBindValue(photoType);
   query2.addBindValue(PhotoDlCRC);
   query2.addBindValue(PhotoSize);
   query2.addBindValue(PhotoChkSum);
   query2.addBindValue(u32ContactHandle);

   if (!query2.exec())
   {
      PB_CHECK_QUERY_ERROR(query2); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(("ERROR - SQL query failed  in SetContactPhoto  Update  "));
      return false;
   }
   else
   {
      ETG_TRACE_USR4((" Photo updated successfully."));
   }

   return true;
}
//End of fix FIX CMG3G-8372


//FIX CMG3G-8371 IS2430_SetContactPosition@FC_Phonebook
/*******************************************************************************
 *
 * FUNCTION: FC_PhoneBook_SQLite::SetContactPosition()
 *
 * DESCRIPTION: Updates the Contact position in Vehicle phone book of a specific device.
 *
 *
 * PARAMETER: IN - ContactHandle
 *
 * RETURNVALUE: bool - returns the success or failure of the function
 ********************************************************************************/
bool FC_PhoneBook_SQLite::SetContactPosition(quint32 u32ContactHandle, quint32 u32ContactPosition, quint16& u16ErrorCode)
{
   ETG_TRACE_USR4(("SetContactPosition entered  "));
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db = Database();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      u16ErrorCode = FC_PB_ERROR_DATABASE_NOT_OPEN;
      return false;
   }

   quint32 u32destinationposition  = u32ContactPosition;

   ETG_TRACE_USR4(("destination position - %d   ", u32destinationposition));
   ETG_TRACE_USR2(("Contact Handle: %d",u32ContactHandle));

   QString sql;
   QSqlQuery query(db);

   sql = "SELECT " FC_PB_MASTER_COL_TABLE_INDEX " FROM "
      FC_PB_MASTER_TABLE_NAME " WHERE " FC_PB_MASTER_COL_CONTACT_HANDLE" = " + QString::number(u32ContactHandle);

   if (!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(("ERROR - SQL query failed in selecting master table "));
      u16ErrorCode = FC_PB_ERROR_INTERNAL_FAILURE;
      return false;
   }

   if (!query.next())
   {
      ETG_TRACE_ERR((" FC_PhoneBook_SQLite: SetContactPosition - No entry found for the contact handle"));
      u16ErrorCode = FC_PB_ERROR_CONTACT_HANDLE_INVALID;
      return false;
   }

   quint8 tableindex = (query.value(VALUE_ZERO)).value<quint8>();
   sql.clear();
   query.clear();

   quint32 u32NoOfContactInVAB = GetNoOfContactsInVPB(tableindex);

   if(u32destinationposition > u32NoOfContactInVAB)
   {
      u32destinationposition = u32NoOfContactInVAB;
   }

   if (!CHECK_VALID(tableindex))
   {
      ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Table Index - %d is invalid  ", tableindex));
      u16ErrorCode = FC_PB_ERROR_INTERNAL_FAILURE;
      return false;
   }
   else
   {
      ETG_TRACE_USR4((" Set contact position table index - %d  ", tableindex));
   }

   sql = "SELECT " FC_PB_COL_VPB_CONTACT_POSITION " FROM "
      + fc_vpb_aTableNames[tableindex]+ " WHERE " FC_PB_COL_CONTACT_HANDLE " = " + QString::number(u32ContactHandle);


   if (!query.exec(sql))
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(("ERROR - SQL query failed in selecting the contact current position "));
      u16ErrorCode = FC_PB_ERROR_INTERNAL_FAILURE;
      return false;
   }

   if (!query.next())
   {
      ETG_TRACE_ERR((" FC_PhoneBook_SQLite: SetContactPosition - No entry found for the contact handle"));
      u16ErrorCode = FC_PB_ERROR_CONTACT_HANDLE_INVALID;
      return false;
   }

   quint32 u32currentposition = query.value(VALUE_ZERO).toUInt();
   ETG_TRACE_USR4(("current position of the requested contact handle - %d   ", u32currentposition));

   sql.clear();
   query.clear();

   QString sqlpos;

   if (u32currentposition != u32destinationposition)
   {
      if (u32destinationposition < u32currentposition)
      {
         sqlpos = "UPDATE " + fc_vpb_aTableNames[tableindex]
                  + " SET " FC_PB_COL_VPB_CONTACT_POSITION " = " FC_PB_COL_VPB_CONTACT_POSITION
                  " +1 WHERE (( " FC_PB_COL_VPB_CONTACT_POSITION " >= "
                  + QString::number(u32destinationposition)
                  + ") AND (" FC_PB_COL_VPB_CONTACT_POSITION " <= "
                  + QString::number(u32currentposition) + "))";

         ETG_TRACE_USR4(("SQL string to be executed::: %s", sqlpos.toUtf8().constData()));
      }

      if (u32destinationposition > u32currentposition)
      {

         sqlpos = "UPDATE " + fc_vpb_aTableNames[tableindex]
                  + " SET " FC_PB_COL_VPB_CONTACT_POSITION " = " FC_PB_COL_VPB_CONTACT_POSITION
                  " -1 WHERE (( " FC_PB_COL_VPB_CONTACT_POSITION " <= "
                  + QString::number(u32destinationposition)
                  + ") AND (" FC_PB_COL_VPB_CONTACT_POSITION " >= "
                  + QString::number(u32currentposition) + "))";

         ETG_TRACE_USR4(("SQL string to be executed::: %s", sqlpos.toUtf8().constData()));
      }

      if (!query.exec(sqlpos))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Failed to set the position"));
         return false;
      }

      QString sqlcontactposset = "UPDATE " + fc_vpb_aTableNames[tableindex]
                                 + " SET "
                                 FC_PB_COL_VPB_CONTACT_POSITION " = "
                                 + QString::number(u32destinationposition)
                                 + " WHERE " FC_PB_COL_CONTACT_HANDLE " = "
                                 + QString::number(u32ContactHandle);

      ETG_TRACE_USR4(("SQL string to be executed::: %s", sqlcontactposset.toUtf8().constData()));

      if (!query.exec(sqlcontactposset))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" FC_PhoneBook_SQLite: Failed to set the position"));
         return false;
      }
   }
   else
   {
      ETG_TRACE_USR4(("Contact already is in requested position"));
      return true;
   }

   return true;
}
//End of fix FIX CMG3G-8371


/*******************************************************************************
 *
 * FUNCTION: FC_PhoneBook_SQLite::GetNoOfContactsInVPB()
 *
 * DESCRIPTION: Retreives the No Of Contacts In VPB .
 *
 *
 * PARAMETER: IN - tableindex
 *
 * RETURNVALUE: quint32 - returns the No Of Contacts In VPB
 ********************************************************************************/
quint32 FC_PhoneBook_SQLite::GetNoOfContactsInVPB(quint8 tableindex)
{
   ETG_TRACE_USR4(("GetNoOfContactsInVPB entered with index: %d", tableindex));

   QSqlDatabase db = Database();
   quint32 u32Count = 0;
   QSqlQuery query(db);

   QString sql;

   if(tableindex == FC_PB_VEHICLE_PBOOK)
   {
      sql = "SELECT COUNT ( " FC_PB_COL_CONTACT_HANDLE " )"
            "FROM "+fc_pb_aTableNames[tableindex];
   }
   else
   {
      sql = "SELECT COUNT ( " FC_PB_COL_CONTACT_HANDLE " )"
            "FROM "+fc_vpb_aTableNames[tableindex];
   }

   if(query.exec(sql))
   {
      if(query.next())
      {
         u32Count= query.value(OSAL_NULL).toUInt();

         ETG_TRACE_USR2((" Total no of Contact Handles are - %d", u32Count));
      }
      else
      {
         ETG_TRACE_ERR(("Failed to get the count of ContactHandles"));
      }
   }
   else
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(("FC_PhoneBook_SQLite :- ERROR - SQL query failed"
         " in bGetVPBContactHandles"));

   }
   return u32Count;
}
//End of fix FIX CMG3G-8371


/*******************************************************************************
 *
 * FUNCTION: FC_PhoneBook_SQLite::bGetVPBContactHandles()
 *
 * DESCRIPTION: Retreives the contact handles from VPB.
 *
 *
 * PARAMETER: IN - tableindex
 *
 * RETURNVALUE: bool - returns the success or failure of the function
 ********************************************************************************/
bool FC_PhoneBook_SQLite::bGetVPBContactHandles(quint8 u8DeviceHandle, QList<
      quint32>& u32ContactHandleList)
{
   ETG_TRACE_USR4((" bGetVPBContactHandles entered with device handle: %d", u8DeviceHandle));

   QMutexLocker lock(&m_mutex);
   QSqlDatabase db = Database();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
      return false;
   }

   if (false == CHECK_VALID(u8DeviceHandle) && 0 != u8DeviceHandle)
   {
      ETG_TRACE_ERR((" Not a valid handle!!"));
      return false;
   }

   QSqlQuery query(db);
   bool blRet = false;

   QString sql = "SELECT " FC_PB_COL_CONTACT_HANDLE " FROM ";

   if (0 == u8DeviceHandle)//When Device handle is set to ZERO, this interface should clear all the entries from the all the vehicle phonebooks [VPB1 to VPB11].
   {
      sql
            += "(SELECT " FC_PB_COL_CONTACT_HANDLE " FROM " FC_PB_TABLE_NAME_VPB " UNION ";

      for (int index = FC_VPB_DEVICE_HANDLE2; index <= FC_VPB_DEVICE_HANDLE11; index++)
      {
         sql += "SELECT " FC_PB_COL_CONTACT_HANDLE " FROM "
               + fc_vpb_aTableNames[index];
         if (index != FC_VPB_DEVICE_HANDLE11)
         {
            sql += " UNION ";
         }
      }
      sql += ")";
   }
   else if (FC_PB_VEHICLE_PBOOK == u8DeviceHandle)
   {
      sql += FC_PB_TABLE_NAME_VPB;
   }
   else
   {
      sql += fc_vpb_aTableNames[u8DeviceHandle];
   }

   ETG_TRACE_USR4(("SQL string to be executed: %s", sql.toLatin1().constData()));

   if (query.exec(sql))
   {
      blRet = true;
      u32ContactHandleList.clear();
      while (query.next())
      {
         u32ContactHandleList << query.value(0).toUInt();
      }
      ETG_TRACE_USR4((" u32ContactHandleList size: %d", u32ContactHandleList.size()));
   }
   else
   {
      PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

      ETG_TRACE_ERR(("FC_PhoneBook_SQLite :- ERROR - SQL query failed"
         " in bGetVPBContactHandles"));
   }

   return blRet;
}


// -PHBK-QUERY-ERR- Instrumentation: BGN - for GMMY17-13044 "database disk image is malformed"
/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::vCheckQueryError
*
* DESCRIPTION: In case that a query has failed, this method should be called to
*              check the returned SQLite error in more detail.
*              Based on the severity of the related error code, we will decide
*              here if the database shall be DELETED, so that it can be recreated
*              at the next start-up.
*
* HINT:        This method is called from
*              - FC_PhoneBook_SQLite::vCheckQueryError(..)
*              if required.
*
* PARAMETER:   [IN]  - None
*              [OUT] - None
*
* RETURNVALUE: void
*
*******************************************************************************/
void FC_PhoneBook_SQLite::vCheckQueryError(QSqlQuery &query, quint32 u32LineNr, QString qsFileName, QString qsFunctionName)
{
   // Strip path name from 'qsFileName'
   int iLastIndex = qsFileName.lastIndexOf("/");  // returns -1, if "/" is not found, otherwise the position
   qsFileName = qsFileName.mid(iLastIndex +1);    // (iLastIndex +1) to skip the "/" resp. to start with position 0

   QSqlError qsqlError = query.lastError();
   // Hints regarding QSqlError:
   // QSqlError::type()    Returns the error type, or -1 if the type cannot be determined.
   // QSqlError::number()  Returns the database-specific error number, or -1 if it cannot be determined.
   // QSqlError::text()    This is a convenience function that returns databaseText() and driverText() concatenated into a single string.
   // Note: Note that the last error for this query is reset when exec() is called.

   ETG_TRACE_ERR (( " -PHBK-QUERY-ERR-: lastError: QSqlError.type = %d, .number= %d, .text= '%s' ",
                            qsqlError.type(), qsqlError.number(), qsqlError.text().toLatin1().constData() ));
   bool bFatalTrace  = false;
   bool bErrmemEntry = false;
   bool bDbRecreate  = false;

   if(qsqlError.type() != QSqlError::NoError)  // Note: QSqlError::NoError = 0
   {  // Query error has occurred.

      // Based on available the error details, the resulting measures will be taken appropriately:
      switch( qsqlError.number() )
      {
      case -1:
         {
            if(qsqlError.type() == QSqlError::ConnectionError)
            {
               // QSqlError.type = 1, .number= -1, .text= 'Driver not loaded Driver not loaded' has been
               // observed in the past (GMMY16-26189, GMMY16-27225), when Phonebook is already in OFF, which
               // means the database is closed and the database connection is removed. Then the
               // dbus thread (via loopback) issues a database access and we will get this situation.
               // It is not a critical error it is more ore less a "use-case".
               //
               // Therefore we will generate a FATAL Trace here only.
               bFatalTrace  = true;
            }
            else
            {
               // This error combination [(qsqlError.number() == -1) && (qsqlError.type() != QSqlError::ConnectionError)]
               // is not known so far. We will do a FATAL trace and ERRMEM entry.
               bFatalTrace  = true;
               bErrmemEntry = true;
            }

            break;
         }
      case SQLITE_CORRUPT:  // = 11
         {
            // This error code is related to GMMY17-8233: QSqlError.type == 2, .number == 11 (SQLITE_CORRUPT),
            // .text= 'malformed database schema (table_devices) - unrecognised token: "." Unable to execute statement'
            // A malformed database schema typically is created by:
            //  --  a memory writer (by our component or a kernel module),
            //  --  or a HW defect in the eMMC circuit
            // as the file system itself (as used by SQlite) "can be assumed" to be transaction safe.
            //
            // Anyhow, this error case is a most severe error, indicating that the database got corrupted.
            // Therefore the database shall be recreated with next transition to NORMAL.
            bFatalTrace  = true;
            bErrmemEntry = true;
            bDbRecreate  = true;

            break;
         }
      case SQLITE_NOTADB:  // = 26
         {
            // SQLITE_NOTADB = 26 [file is encrypted or is not a database]
            // We have added this severe error code here, on recommendation from Navigation team in the context
            // phonebook encryption.
            //
            // This error might occur, when we have caused an encryption mismatch, e.g. when working by accident on
            // an encrypted database with wrong or no key-ID or have an issue with the SDC keystore.

            //
            // At the moment we understand this error (when it is returned on a "standard" query during normal operation)
            // as severe error, which requires that the database gets recreated with the next transition to NORMAL.
            bFatalTrace  = true;
            bErrmemEntry = true;
            bDbRecreate  = true;

            break;
         }
      default:
         {
            // Any error in this section is not known so far. So we are interested, if such errors occurs.
            // We will do an FATAL trace and an ERMMEM entry (but do not trigger the database to be recreated).
            bFatalTrace  = true;
            bErrmemEntry = true;

            break;
         }
      }
   }

#ifdef PB_CHECK_QUERY_EXEC__TESTMODE
   // - If 'PB_CHECK_QUERY_EXEC__TESTMODE is defined' then a special Testmode is active. This shall be
   //   done only for developer builds for dedicated testing.
   // - If this dedicated switch is not defined: Normal operation mode (product code delivery)
   //
   // In "Testmode active" we simulate a severe error (e.g. corrupt database). The database recreation
   // trigger will be set (with each call of this method [vCheckQueryError(..)]), independent if a severe
   // error has occurred in a query or not.
   //
   // Note: The "Testmode active" requires in addition that the method 'vCheckQueryError' is called after
   // any query, even if this query has returned successful. We have done such an instrumentation in
   // method 'FC_PhoneBook_SQLite::UpdatePreferredSortOrderTable(()' (also guarded by the define
   // "PB_CHECK_QUERY_EXEC__TESTMODE").




   // Here we simulate an fatal database error
   ETG_TRACE_FATAL((  " -PHBK-QUERY-ERR-: TESTMODE is active. Now we simulate a fatal DB error. " ));
   ETG_TRACE_ERRMEM(( " -PHBK-QUERY-ERR-: TESTMODE is active. Now we simulate a fatal DB error. " ));
   bFatalTrace  = true;
   bErrmemEntry = true;
   bDbRecreate  = true;
#endif

   // Get time via 'clock_gettime(CLOCK_MONOTONIC,.)'
   struct timespec ts;  // for clock_gettime(CLOCK_MONOTONIC,.)
   char sz_index_ts[PB_CHECK_QUERY_TIMESTAMP_BYTE_LEN +1] = {0};

   clock_gettime(CLOCK_MONOTONIC, &ts);

   snprintf(sz_index_ts, MIN(sizeof(sz_index_ts), PB_CHECK_QUERY_TIMESTAMP_BYTE_LEN), "up-time= [%7lu.%03lu] sec, ", ts.tv_sec, ts.tv_nsec/1000000);
   sz_index_ts[PB_CHECK_QUERY_TIMESTAMP_BYTE_LEN] = 0;  // Take care that string is terminated with \0

   // Generate "Main Trace line": ['up-time'] + '<::function_name> ' + '[Filename.cpp (LineNr)] - '
   QString qsMainTraceLine = "-PHBK-QUERY-ERR-MAIN-: " + QString(sz_index_ts) + "<::" + qsFunctionName + "> "
                             + "[" + qsFileName + " (" + QString::number(u32LineNr) + ")] -";

   ETG_TRACE_FATAL (( " %s ", qsMainTraceLine.toLatin1().constData() ));
   if(bErrmemEntry == true)
   {
      ETG_TRACE_ERRMEM (( " %s ", qsMainTraceLine.toLatin1().constData() ));
   }

   if(bFatalTrace == true)
   {
      ETG_TRACE_FATAL (( " -PHBK-QUERY-ERR-: lastError: QSqlError.type = %d, .number= %d, .text= '%s' ",
                         qsqlError.type(), qsqlError.number(), qsqlError.text().toLatin1().constData() ));
   }
   if(bErrmemEntry == true)
   {
      ETG_TRACE_ERRMEM(( " -PHBK-QUERY-ERR-: lastError: QSqlError.type = %d, .number= %d, .text= '%s' ",
                         qsqlError.type(), qsqlError.number(), qsqlError.text().toLatin1().constData() ));
   }

   if(bDbRecreate)
   {
      // Set the database "recreation trigger file" to the FFS
      vSetDbRecreationTrigger();

      NORMAL_M_ASSERT_ALWAYS();    // With the normal assert we can see from the callstack from which method we have been called
      // FATAL_M_ASSERT_ALWAYS();  // This option with the fatal assert would issue a direct reset and could be used to avoid
                                   // bad user experience with a corrupted database.
   }
}
// -PHBK-QUERY-ERR- Instrumentation: END - for GMMY17-13044 "database disk image is malformed"


// -PHBK-QUERY-ERR- Instrumentation: BGN - for GMMY17-13044 "database disk image is malformed"
/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::vSetDbRecreationTrigger
*
* DESCRIPTION: Sets DB recreation trigger in FFS.
*              The used sequence of operations guarantees that the recreation
*              trigger is written into the FFS and the file system is synced
*              when the method is left.
*              This allows to issue directly afterwards a FATAL_M_ASSERT_ALWAYS()
*              if wanted (e.g. to avoid bad user experience with a corrupted
*              database in the currently active power cycle).
*
* HINT:        This method could be called from
*              - FC_PhoneBook_SQLite::vCheckQueryError(..)
*              - Phonebook_DefSetHandler::vOnSystemSet(..)
*
* PARAMETER:   [IN]  - None
*              [OUT] - None
*
* RETURNVALUE: void
*
*******************************************************************************/
void FC_PhoneBook_SQLite::vSetDbRecreationTrigger()
{
   bool bRes = false;

   // First we check if the database "recreation trigger" file exists already. If it already exists, we should NOT create it again.
   // Background: A repeated creation of this file is "expensive" as we need several 'syncfs' operations, which are rather time
   // consuming and would block the file system for other clients during such operation. This should be avoided, if not really needed.
   struct stat oStatBuf;
   if( 0 != stat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_RECREATE_TRIGGER_FILE_NAME, &oStatBuf) )
   {
      // The database "recreation trigger" does NOT exist and has to be created.

      FILE *fp_DbRecreateTriggerFile = NULL;
      fp_DbRecreateTriggerFile = fopen(FC_PB_DB_FILE_PATH "/" FC_PB_DB_RECREATE_TRIGGER_FILE_NAME, "w");

      if(fp_DbRecreateTriggerFile != NULL)
      {
         ETG_TRACE_FATAL(( " -PHBK-QUERY-ERR-: Create RE-CREATION trigger 'pb.db.recreate.trigger'. " ));
         // The following fflush(.) is not be necessary here, as we have not added any content to the file, but we do it for generality.
         if( 0 == fflush(fp_DbRecreateTriggerFile) )
         {  // fflush SUCCESS

            // Get file descriptor 'fd' from an open FILE* object 'fp'
            int fd = fileno(fp_DbRecreateTriggerFile);
            if(-1 != fd)
            {  // fileno(.) SUCCESS

               // Sync the file system (partition), in which the file descriptor resides, to the FFS (eMMC)
               if( 0 == syncfs(fd) )
               {  // syncfs(.) SUCCESS
                  bRes = true;
                  // The database recreation trigger file has been written successfully to the FFS.
                  ETG_TRACE_FATAL (( " -PHBK-QUERY-ERR-: Severe database error detected. Database RE-CREATION trigger has been set!! "  ));
                  ETG_TRACE_ERRMEM(( " -PHBK-QUERY-ERR-: Severe database error detected. Database RE-CREATION trigger has been set!! "  ));
               }
            }
         }
         if(false == bRes)
         {
            // We wanted to set the database recreation trigger file, but have failed on this. So we will do (as last resort) a related EM-Trace entry.
            ETG_TRACE_FATAL (( " -PHBK-QUERY-ERR-: Database RE-CREATION trigger 'pb.db.recreate.trigger' cannot be created. " ));
            ETG_TRACE_ERRMEM(( " -PHBK-QUERY-ERR-: Database RE-CREATION trigger 'pb.db.recreate.trigger' cannot be created. " ));
         }

         // Close FILE* - CMG3G-14313-Coverity Fix
         if( 0 != fclose(fp_DbRecreateTriggerFile) )
         {
            ETG_TRACE_ERR(( " Unable to close the file!!! "  ));
         }
         else
         {
            fp_DbRecreateTriggerFile = NULLPTR;
         }
         //End of Fix
      }
   }
   else
   {
      ETG_TRACE_FATAL(( " -PHBK-QUERY-ERR-: RE-CREATION trigger 'pb.db.recreate.trigger' already exists. " ));
   }
}
// -PHBK-QUERY-ERR- Instrumentation: END - for GMMY17-13044 "database disk image is malformed"


/*******************************************************************************
 *
 * FUNCTION: FC_PhoneBook_SQLite::bUpdateAutomaticContactDownload()
 *
 * DESCRIPTION: Update the bAutomaticDL
 *
 *
 * PARAMETER: IN -
 *
 * RETURNVALUE: bool - returns the success or failure of the function
 ********************************************************************************/
bool FC_PhoneBook_SQLite::bUpdateAutomaticContactDownload(quint8 u8DeviceHandle, bool bAutomaticDL)
{
   ETG_TRACE_USR4(("bUpdateAutomaticContactDownload entered with device handle: %d, AutomaticDL: %d", u8DeviceHandle,bAutomaticDL ));
   QMutexLocker lock(&m_mutex);
   QString sql;
   QSqlDatabase db = Database();
   bool bRetValue = false;
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);

      if(u8DeviceHandle ==0)
      {
         sql = "UPDATE " FC_PB_TABLE_NAME_FS " SET "
               FC_PB_FS_COL_PB_AUTOMATIC_CONTACT_DL " = " + QString::number(bAutomaticDL);

      }
      else
      {
           sql = "UPDATE " FC_PB_TABLE_NAME_FS " SET "
                 FC_PB_FS_COL_PB_AUTOMATIC_CONTACT_DL" = " + QString::number(bAutomaticDL) + " WHERE "
           FC_PB_FS_COL_DEVICE_HANDLE" = " + QString::number(u8DeviceHandle);
       }
      ETG_TRACE_USR2(("bUpdateAutomaticContactDownload %s", sql.toLatin1().constData()));
      if (!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query);
         ETG_TRACE_ERR(("ERROR - SQL query failed in bUpdateAutomaticContactDownload For Updation "));
      }
      else
         bRetValue = true;
   }
   ETG_TRACE_USR4(("bUpdateAutomaticContactDownload EXIT."));
   return bRetValue;
}

/*******************************************************************************
 *
 * FUNCTION: FC_PhoneBook_SQLite::bGetAutomaticContactDLList()
 *
 * DESCRIPTION: Get the device handles of all paired devices
 *
 *
 * PARAMETER: IN - QList<quint8> &DeviceHandleList , QList<bool> &AutomaticDLList
 *
 * RETURNVALUE: bool - returns the success or failure of the function
 ********************************************************************************/
bool FC_PhoneBook_SQLite::bGetAutomaticContactDLList(QList<quint8> &DeviceHandleList,QList<
      bool>& AutomaticDLList)
{
   bool bRetValue = false;
   ETG_TRACE_USR4(("bGetAutomaticContactDLList entered"));
   QMutexLocker lock(&m_mutex);
   QSqlDatabase db = Database();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);
      QString Sql = "SELECT " FC_PB_FS_COL_DEVICE_HANDLE " , " FC_PB_FS_COL_PB_AUTOMATIC_CONTACT_DL " FROM " FC_PB_TABLE_NAME_FS;
      ETG_TRACE_USR2(("Select Query::%s", Sql.toLatin1().constData()));
      if (!query.exec(Sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime
      }
      else
      {
         while (db.isOpen() && query.next())
         {
            DeviceHandleList << (query.value(0)).value<quint8>();
            AutomaticDLList << query.value(1).toUInt();
         }
         bRetValue = true;
      }
   }
   return bRetValue;
}

//End of fix FIX CMG3G-9889

//CMG3G-14241
/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::UpdatePreferredSortOrderForDeviceHandle()
*
* DESCRIPTION:
*              QtSql specific function to UpdatePreferredSortOrderForDeviceHandle
*
*
* PARAMETER: None.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::UpdatePreferredSortOrderForDeviceHandle(tU8 u8DeviceHandle,quint8 u8SortOrder)
{
   ETG_TRACE_USR4(("UpdatePreferredSortOrderForDeviceHandle entered with device handle: %d, u8SortOrder: %u", u8DeviceHandle,u8SortOrder ));
   QMutexLocker lock(&m_mutex);
   QString sql;
   QSqlDatabase db = Database();
   bool bRetValue = false;
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      if (CHECK_VALID_SORT_ORDER(u8SortOrder))
      {
         QSqlQuery query(db);

         //If device handle as 0, the particular sort order will be set for all devices
         if (0 == u8DeviceHandle)
         {
            sql = "UPDATE " FC_PB_TABLE_NAME_FS " SET " FC_PB_PREFERREDSORTORDER_COL_SORTORDER " = " + QString::number(u8SortOrder);

         }
         else
         {
            sql = "UPDATE " FC_PB_TABLE_NAME_FS " SET " FC_PB_PREFERREDSORTORDER_COL_SORTORDER" = " + QString::number(u8SortOrder)
                        + " WHERE " FC_PB_FS_COL_DEVICE_HANDLE" = "
                        + QString::number(u8DeviceHandle);
         }

         ETG_TRACE_USR2(("UpdatePreferredSortOrderForDeviceHandle %s", sql.toLatin1().constData()));

         if (!query.exec(sql))
         {
            PB_CHECK_QUERY_ERROR(query);
            ETG_TRACE_ERR(("ERROR - SQL query failed in UpdatePreferredSortOrderForDeviceHandle For Updation "));
         }
         else
            bRetValue = true;
      }
   }
   ETG_TRACE_USR4(("UpdatePreferredSortOrderForDeviceHandle EXIT."));

   return bRetValue;
}

/*******************************************************************************
 *
 * FUNCTION: FC_PhoneBook_SQLite::bGetPreferredSortOrderList()
 *
 * DESCRIPTION: Get the Preferred Sort order for all the paired devices.
 *
 *
 * PARAMETER: IN - QList<quint8> &DeviceHandleList , QList<quint8> &PreferredSortOrderList
 *
 * RETURNVALUE: bool - returns the success or failure of the function
 ********************************************************************************/
bool FC_PhoneBook_SQLite::bGetPreferredSortOrderList(QList<quint8> &DeviceHandleList,QList<
      quint8>& PreferredSortOrderList)
{
   bool bRetValue = false;
   ETG_TRACE_USR4(("bGetPreferredSortOrderList entered"));
   QMutexLocker lock(&m_mutex);
   QSqlDatabase db = Database();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);
      QString Sql = "SELECT " FC_PB_FS_COL_DEVICE_HANDLE " , " FC_PB_PREFERREDSORTORDER_COL_SORTORDER " FROM " FC_PB_TABLE_NAME_FS;
      ETG_TRACE_USR2(("Select Query::%s", Sql.toLatin1().constData()));
      if (!query.exec(Sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime
      }
      else
      {
         while (db.isOpen() && query.next())
         {
            DeviceHandleList << (query.value(0)).value<quint8>();
            PreferredSortOrderList << query.value(1).value<quint8>();
         }
         bRetValue = true;
      }
   }
   return bRetValue;
}

/*******************************************************************************
 *
 * FUNCTION: FC_PhoneBook_SQLite::u8GetPreferredSortOrderForDevicehandle
 * DESCRIPTION: To Get the PreferredSortOrder For Devicehandle
 *
 * PARAMETER: [IN]  - Devicehandle
 *            [OUT] -
 *
 * RETURNVALUE: Preferred sort order of the particular device handle
 *******************************************************************************/
quint8 FC_PhoneBook_SQLite::u8GetPreferredSortOrderForDevicehandle(quint8 u8DeviceHandle)
{
   ETG_TRACE_USR4(("u8GetPreferredSortOrderForDevicehandle entered with DeviceHandle %d ", u8DeviceHandle));
   quint8 u8PreferredSortOrder = VALUE_ZERO;
   QMutexLocker lock(&m_mutex);

   QSqlDatabase db = Database();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      QSqlQuery query(db);

      QString sql = "SELECT " FC_PB_PREFERREDSORTORDER_COL_SORTORDER " FROM "
      FC_PB_TABLE_NAME_FS " WHERE " FC_PB_FS_COL_DEVICE_HANDLE " = " + QString::number(u8DeviceHandle);

      ETG_TRACE_USR2(("FC_PhoneBook_SQLite: String used for u8GetPreferredSortOrderForDevicehandle - %s ", sql.toLatin1().constData()));

      if (!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_USR4(("FC_PhoneBook_SQLite: u8GetPreferredSortOrderForDevicehandle query failed"));
      }
      else
      {
         if (!query.next())
         {
            ETG_TRACE_USR4(("FC_PhoneBook_SQLite: u8GetPreferredSortOrderForDevicehandle - No entry found for the device handle"));
         }
         else
         {
            u8PreferredSortOrder = (query.value(VALUE_ZERO)).value<quint8>();

            ETG_TRACE_USR4(("FC_PhoneBook_SQLite:u8GetPreferredSortOrderForDevicehandle : u8PreferredSortOrder value  %d ", u8PreferredSortOrder));
         }
      }
   }

   return u8PreferredSortOrder;
}
//End of CMG3G-14241

/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::GetSortOrderFromPreferredSortOrderTable()
*
* DESCRIPTION:
*              QtSql specific function to GetPreferredSortOrder From PreferredSortOrderTable
*
*
* PARAMETER: None.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
quint8 FC_PhoneBook_SQLite::GetSortOrderFromPreferredSortOrderTable()
{
   ETG_TRACE_USR4(("FC_PhoneBook_SQLite::GetSortOrderFromPreferredSortOrderTable entered"));

   quint8 u8SortOrder= 0;

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR((" Database is not open"));
   }
   else
   {
      QSqlQuery query(db);
      QString sql = " SELECT " FC_PB_PREFERREDSORTORDER_COL_SORTORDER " FROM "  FC_PB_PREFERREDSORTORDER_TABLE_NAME;

      ETG_TRACE_USR2((" SQL query used to GetSortOrderFromPreferredSortOrderTable - %s ",sql.toLatin1().constData()));

      if(!query.exec(sql))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" GetSortOrderFromPreferredSortOrderTable Failed"));
      }
      else
      {
         query.next();
         if(query.isValid() == true)
         {
            u8SortOrder = (query.value(0)).value<quint8>();
            ETG_TRACE_USR2((" PreferredSortOrder = %d" , u8SortOrder));
         }
         else
         {
            ETG_TRACE_ERR(("Sql query failed"));
         }
      }
   }

   return u8SortOrder;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_SQLite::CheckPreferredSortOrderTable()
*
* DESCRIPTION:
*              QtSql specific function to validate PreferredSortOrderTable
*
*
* PARAMETER: None.
*
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_SQLite::CheckPreferredSortOrderTable()
{
   ETG_TRACE_USR4(("CheckPreferredSortOrderTable entered "));
   bool bRetValue = false;

   QSqlDatabase db= Database();
   if(!db.isOpen())
   {
      ETG_TRACE_ERR((" Database is not open"));
   }
   else
   {
      QSqlQuery query(db);


      QStringList list_columns;

      if(!query.exec("PRAGMA table_info(" FC_PB_PREFERREDSORTORDER_TABLE_NAME ")"))
      {
         PB_CHECK_QUERY_ERROR(query); //FIX CMG3G-8752 Check for Severe DB error("malformed database schema ") during runtime

         ETG_TRACE_ERR((" Error:- SQL query failed while validating PreferredSortOrderTable"));
      }
      else
      {
         while(query.next())
            list_columns << query.value(1).toString();
         if( list_columns.contains(FC_PB_PREFERREDSORTORDER_COL_SORTORDER))
         {
            ETG_TRACE_USR4((" CheckPreferredSortOrderTable: PreferredSortOrderTable contains the correct columns "));
            bRetValue = true;
         }
         else
         {
            ETG_TRACE_USR4((" CheckPreferredSortOrderTable: PreferredSortOrderTable doesn't contain the correct columns "));
         }
      }
   }

   return bRetValue;
}


// Hotfix for CRQ 509734 Phonetic Sorting
/*******************************************************************************
 *
 * FUNCTION: FC_PhoneBook_SQLite::vSetSortingWithPhonetics(bool bSortingWithPhonetics)
 *
 * DESCRIPTION: Sets the boolean member variable 'bSortingWithPhonetics'.
 *              - Typically set to true (by HMI) to achieve "Phonetic Sorting" for the
 *                JAPANESE market.
 *              - If the member variable 'bSortingWithPhonetics' is true, the FirstName
 *                and LastName sortiing schemes will behave in a way that the phonetic
 *                information in database colums 'Phonetics' will be used with first
 *                priority.
 *              - Affected methods are: 'GetRecords'.
 *                Maybe, later the method 'SearchContacts' should be adapted in the
 *                same way.
 *
 * PARAMETER: IN - bool bSortingWithPhonetics
 *
 * RETURNVALUE: void -
 ********************************************************************************/
void FC_PhoneBook_SQLite::vSetSortingWithPhonetics(bool bSortingWithPhonetics)
{
   ETG_TRACE_USR2(( "PHBK_YOMI: vSetSortingWithPhonetics entered, bSortingWithPhonetics= %d ", bSortingWithPhonetics ));

   QMutexLocker lock(&m_mutex);  // We use this mutex to avoid that the member variable gets changed while an (affected) query is running.

   m_bSortingWithPhonetics = bSortingWithPhonetics;

   ETG_TRACE_USR2(( "vSetSortingWithPhonetics leaving " ));
   return;
}


//Fix NCG3D-67384
/*******************************************************************************
 *
 * FUNCTION: FC_PhoneBook_SQLite::isCheckTableSchemaEnabled()
 *
 * DESCRIPTION: Checking the TableSchema flag value.
 *
 *
 * PARAMETER: IN -
 *
 * RETURNVALUE: bool - returns m_bCheckTableSchema
 ********************************************************************************/
bool FC_PhoneBook_SQLite::isCheckTableSchemaEnabled()
{
   ETG_TRACE_USR4((" -PHBK_isCheckTableSchemaFlagEnabled = %d", m_bCheckTableSchema ));
   return this->m_bCheckTableSchema;
}
/*******************************************************************************
 *
 * FUNCTION: FC_PhoneBook_SQLite::SetCheckTableSchemaFlag()
 *
 * DESCRIPTION: Setting the TableSchema flag value.
 *
 *
 * PARAMETER: IN - bool bCheckTableSchema
 *
 * RETURNVALUE:
 ********************************************************************************/
void FC_PhoneBook_SQLite::SetCheckTableSchema(bool bCheckTableSchema)
{
   this->m_bCheckTableSchema = bCheckTableSchema;
   ETG_TRACE_USR4((" -PHBK_SetCheckTableSchemaFlag = %d", m_bCheckTableSchema ));
}
//End fix NCG3D-67384


/*******************************************************************************
 *
 * FUNCTION: FC_PhoneBook_SQLite::SetWalCheckPoint()
 *
 * --- This function is EXPERIMENTAL not FINALIZED and NOT USED at the moment. ---
 *
 * I experimented with the manual WAL checkpointing aspects described
 * below, but deciced at some point of time that we will try to operate the
 * system without any manual checkpointing. As long we do not get negative
 * feedback from system test, we just rely on the automatic checkpointing,
 * i.e.: This function is NOT called from any code position at the moment.
 * On the other hand it contains several interesting code snippets, which
 * are worth to be kept as reference (for further experiments, if needed
 * later).
 *
 * DESCRIPTION: CheckpointAndTruncateWalJournal
 *              (1) This method checks the current size of the WAL file
 *                  and if it exceeds a relevant size, step (2) is executed.
 *                  (This part is currently not implemented. It could be an
 *                   later optimization.)
 *              (2) A WAL checkpoint with option "truncate" is executed.
 *                  The "truncate" option should result in WAL file of
 *                  size 0 and therfore helps to limit the WAL file size in the
 *                  flash file system.
 *
 *              Note: The automatic checkpointing by SQLite (e.g. after 1000
 *                    pages) is done wihtout "truncate" option, i.e. the WAL
 *                    file size remains unchanged even if the related data is
 *                    already finalized to the database file. This might result
 *                    in an unlimited increase of the WAL file (until the DB
 *                    gets closed). This method intends to mitigate this issue.
 *
 *              As a checkpointing can result in a "database busy" for other
 *              database clients (e.g. the Speech Transcription Service), we
 *              should be careful with the use this method.
 *              The currrent idea on this aspect is as follows:
 *              - This method is intended to be called, when we expect no further
 *                DB write action, e.g. when an ME/SM download has been finalized,
 *                before we inform our clients of a potentially changed "CheckSum".
 *              - This criterion might not be perfect, but hopefully is a good
 *                compromise to avoid an unlimited increasing WAL file while
 *                minimizing a "database busy" situation for clients.
 *
 * As mentioned above:
 * --- This function is EXPERIMENTAL not FINALIZED and NOT USED at the moment. ---
 *
 * PARAMETER: IN - none
 *
 * RETURNVALUE:
 ********************************************************************************/
void FC_PhoneBook_SQLite::SetWalCheckPoint(void)
{
   ETG_TRACE_USR4(( "SetWalCheckPoint entered " ));

   // Get own thread ID
   pthread_t thread;   // Note: pthread_t is: typedef unsigned long int pthread_t;
   thread = pthread_self();
   ETG_TRACE_USR2(( " SetWalCheckPoint: pthread_self()= 0x%x. ", thread ));

   QMutexLocker lock(&m_mutex);  // Use mutex as we do for the most queries also


   sqlite3 *handle = NULL;
   QVariant qVariant;
   int iSizeOfWalLogInFrames    = 0;
   int iNumOfCheckpointedFrames = 0;
   int iRes                     = 0;

   // Check free space on partition where our phonebook database (and the WAL files) resides (i.e. the partition: /var/opt/bosch/dynamic)
   {
      struct statvfs oStatVfsBuf;
      int iRes = 0;
      //fsblkcnt_t FreeAvailableBlocks = 0;   // free blocks for unprivileged users
      iRes = statvfs(FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_FILE_NAME /* const char *path */, &oStatVfsBuf /* struct statvfs *buf */);
      if(iRes == 0)
      {
         unsigned long ulBlocksizeByte = oStatVfsBuf.f_bsize;
         unsigned long ulDynamicPartitionFreeSize_kB = oStatVfsBuf.f_bavail / 1024 * ulBlocksizeByte;
         ETG_TRACE_COMP(( "statvfs: Free blocks for unprivileged users: f_bavail= %u, blocksize: f_bsize= %u ", oStatVfsBuf.f_bavail, ulBlocksizeByte ));
         ETG_TRACE_COMP(( "statvfs: Free memory in dynamic partition, freeMem= %u [kB] ", ulDynamicPartitionFreeSize_kB ));
      }
      else
      {
         ETG_TRACE_COMP(( "statvfs: Error: iRes= %d ", iRes ));
      }
   }

   // Check WAL file size before checkpointing
   struct stat oStatBuf;
   iRes = stat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_FILE_NAME, &oStatBuf);
   if (0 == iRes)
   {
      ETG_TRACE_COMP (( " sqlite3_wal_checkpoint_v2: Current WAL file size (before 'truncate' checkpointing), size[byte]= %d", oStatBuf.st_size ));
   }
   else
   {
      ETG_TRACE_FATAL(( " sqlite3_wal_checkpoint_v2: Current WAL file size (before 'truncate' checkpointing) cannot be determined. " ));
   }

   // Get DB handle and if this can be obtained, do a checkpointing
   QSqlDatabase db = Database();
   if (!db.isOpen())
   {
      ETG_TRACE_ERR(("Database is not open"));
   }
   else
   {
      // Get the Database Driver Handle
      qVariant = db.driver()->handle();

      if (qVariant.isValid() && u8StringCompare(qVariant.typeName(), "sqlite3*") == 0)
      {
         // Get Database Handle
         handle = *static_cast<sqlite3 **> (qVariant.data());

         if (handle)
         {
            // Database handle is not NULL, i.e. we assume that it is valid
            ETG_TRACE_USR4(( "Able to get Sqlite3 database handle"));

            // Until here we have success, so we set result to true.
         }
         else
         {
            ETG_TRACE_FATAL (( " Sqlite database handle is NULL. " ));
            ETG_TRACE_ERRMEM(( " Sqlite database handle is NULL. " ));
         }
      }
      else
      {
         ETG_TRACE_FATAL (( " Unable to get Sqlite3 database handle. " ));
         ETG_TRACE_ERRMEM(( " Unable to get Sqlite3 database handle. " ));
      }

      if (handle)
      {
         iRes = sqlite3_wal_checkpoint_v2(
                           handle,                       // sqlite3 *db,       /* Database handle */
                           NULL,                         // const char *zDb,   /* Name of attached database (or NULL) */
//                           SQLITE_CHECKPOINT_FULL,       // int eMode,         /* SQLITE_CHECKPOINT_* value */
                           SQLITE_CHECKPOINT_TRUNCATE,   // int eMode,         /* SQLITE_CHECKPOINT_* value */
                           &iSizeOfWalLogInFrames,       // int *pnLog,        /* OUT: Size of WAL log in frames */
                           &iNumOfCheckpointedFrames);   // int *pnCkpt);      /* OUT: Total number of frames checkpointed */

         ETG_TRACE_COMP(( " sqlite3_wal_checkpoint_v2: iRes= %d, iSizeOfWalLogInFrames= %d, iNumOfCheckpointedFrames= %d ",
                           iRes, iSizeOfWalLogInFrames, iNumOfCheckpointedFrames ));

         if(    (iRes != SQLITE_OK)
             || (iSizeOfWalLogInFrames == -1) || (iNumOfCheckpointedFrames == -1)
             || (iSizeOfWalLogInFrames != iNumOfCheckpointedFrames)
           )
         {
            ETG_TRACE_FATAL(( " sqlite3_wal_checkpoint_v2(..): ERROR: iRes= %d, errcode= %d, errmsg= '%s' ", iRes, sqlite3_errcode(handle), sqlite3_errmsg(handle) ));
         }

         // Check WAL file size after checkpointing
         iRes = stat(FC_PB_DB_FILE_PATH "/" FC_PB_DB_WAL_FILE_NAME, &oStatBuf);
         if (0 == iRes)
         {
            ETG_TRACE_COMP (( " sqlite3_wal_checkpoint_v2: Remaining WAL file size (after 'truncate' checkpointing), size[byte]= %d", oStatBuf.st_size ));
         }
         else
         {
            ETG_TRACE_FATAL(( " sqlite3_wal_checkpoint_v2: Remaining WAL file size (after 'truncate' checkpointing) cannot be determined. " ));
         }
      }
   }
}
