/*******************************************************************************
*
* FILE:          FC_PhoneBook_List.cpp
*
* SW-COMPONENT:  FC_PhoneBook application
*
* PROJECT:
*
* DESCRIPTION:
*
* AUTHOR:
*
* COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
*
*******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/

#include <QtGui/QImage>

#include "../VerticalKeyBoard/FC_PhoneBook_VerticalKeyBoard.h"

#include "../FC_PhoneBook_Debug.h"

#include "../DbusClient/FC_PhoneBook_EvoDbusClient.h" //CMG3G-14314
#include "FC_PhoneBook_List.h"
#include "FC_PhoneBook_CallHistory_List.h"
#include "FC_PhoneBook_ListChange_Handler.h"
#include "../VCard_Parser/FC_PhoneBook_VCard_Parser.h"
#include "../FC_PhoneBook_FavoritesProvider.h"
#include "../FC_PhoneBook_service_PhoneBook.h"
#include "../Database/FC_PhoneBook_Database.h"
#include "../FC_PhoneBook_main.h"
#include "FC_PhoneBook_Database.h"
#include <malloc.h>  // -bn: Added for 'malloc_trim()'

#define ETRACE_S_IMPORT_INTERFACE_GENERIC

#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONEBOOK_SQLITE
#include "trcGenProj/Header/FC_PhoneBook_List.cpp.trc.h"
#endif

/******************************************************************************/
/*                                                                            */
/* DEFINES                                                                    */
/*                                                                            */
/******************************************************************************/

#define FC_PB_DB_INT16_MAX 65535

#define CONTACT_TYPE_STANDARD 0
#define CONTACT_TYPE_FAVORITE 1

#define PHBK_SHIFT_CONTACTS_TO_TMP_THRESHOLD 100  // number of contacts to be collected in RAM before they are shifted to Tmp table in database
#define PHBK_SHIFT_PHOTOS_TO_DHx_THRESHOLD   30   // number of photos   to be collected in RAM before the photos are updated in the database

enum FC_PHONEBOOK_NUM_TYPES
{
   FC_PHONEBOOK_PRIO_TYPE_VOICE = 1,
   FC_PHONEBOOK_PRIO_TYPE_PAGER,
   FC_PHONEBOOK_PRIO_TYPE_FAX,
   FC_PHONEBOOK_PRIO_TYPE_NOTHING,
   FC_PHONEBOOK_PRIO_TYPE_MSG,
   FC_PHONEBOOK_PRIO_TYPE_OTHER
};

#define GENERATE_TAG(contacthandle, tag_value) (contacthandle | (tag_value << 24))

static int noofphonenos;
FC_PhoneBook_List* FC_PhoneBook_List::m_self = NULL;

//CLONE TICKET FIX - GMMY15-1545 - Singleton creation for FC_PhoneBook_List and FC_PhoneBook_CallHistory_List
FC_PhoneBook_List* FC_PhoneBook_List::pGetInstance()
{
   if(m_self == NULL)
   {
      m_self = new FC_PhoneBook_List;
   }

   return m_self;
}


tVoid FC_PhoneBook_List::vDestroyInstance()
{
   delete m_self;
   m_self = NULLPTR;
}


// Set the maxiumum size for "reported" ContactDetails according to GIS-361: FBlock Phonebook,
#define FC_PB_REPORTED_NUMBER_MAX_LEN		25
#define FC_PB_REPORTED_NAME_MAX_LEN			50
#define FC_PB_REPORTED_POI_MAIL_MAX_LEN		75
#define FC_PB_REPORTED_ADDRESS_MAX_LEN		300
#define FC_PB_REPORTED_WORK_ADDRESS_MAX_LEN 500
#define FC_PB_REPORTED_NOTE_MAX_LEN			1000
#define FC_PB_REPORTED_LAT_MIN_RANG			0xE0000000
#define FC_PB_REPORTED_LAT_MAX_RANG			0x1FFFFFFF
#define FC_PB_REPORTED_LONG_MIN_RANG		0xC0000000
#define FC_PB_REPORTED_LONG_MAX_RANG		0x3FFFFFFF


/******************************************************************************/
/*                                                                            */
/* METHODS                                                                    */
/*                                                                            */
/******************************************************************************/

/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_List::
*
* DESCRIPTION:
*
*
* PARAMETER: None.
*
* RETURNVALUE: None
*
*******************************************************************************/
FC_PhoneBook_List::FC_PhoneBook_List()
{
   m_sqlite = FC_PhoneBook_SQLite::GetSQLiteInstance();
   ETG_TRACE_USR4(("FC_PhoneBook_PhoneBook_List() entered."));
   m_poFavoritesProvider = FC_Phonebook_tclFavorite_Provider::GetInstance();

   //FIX CMG3GB-505 Upon disconnecting a call/call is established/entering numbers in dialpad, resets when doing handsfree operation was observed
   AutoCompletePhNoSearchIndex = 0;
   //End of fix



}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_List::
*
* DESCRIPTION:
*
*
* PARAMETER: None.
*
* RETURNVALUE: None
*
*******************************************************************************/
FC_PhoneBook_List::~FC_PhoneBook_List()
{
   FC_PhoneBook_SQLite::vDestroyInstance();
   m_sqlite = NULLPTR;
   qDeleteAll(m_hashPbList);
   m_poFavoritesProvider = NULLPTR;
   m_hashPbList.clear();
   //FIX GMMY15-1578 Clear Multihash contents
   m_multiHashPBClientRequests.clear();
   m_multiHashPBClientRequests.squeeze();
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::CreatePhoneBookList
*
* DESCRIPTION:
*              Creates a new contact info list from the database.
*
*
* PARAMETER: [IN] methodStart :- Input arguments like sort type etc are in this class
*            [OUT] methodResult :- new list handle and list length are in this class
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_List::CreatePhoneBookList(
   const most_phonbkfi_tclMsgCreatePhoneBookListMethodStart* methodStart,
   most_phonbkfi_tclMsgCreatePhoneBookListMethodResult* methodResult,
   tU32 u32RegisterID)
{
   ETG_TRACE_USR4 (( "CreatePhoneBookList entered "));
   FC_PhoneBook_SQLite::SortType sortType = getSortType(&methodStart->e8PhoneBookSortType);

   tU8 u8ListType = FC_PhoneBook_SQLite::ListType_PHONEBOOK;

   //FIX GMMY15-1578 Bosch Internal: Proper usage of SQLite database instead of RAM -- removing redundant objects which are created in RAM for different clients & sorted lists.
   //Compute the list handle from the parameters in methodstart
   tU16 u16ListHandle = (tU16)(methodStart->u8DeviceHandle * 16 + sortType);//This format is chosen for creating list handle; as DeviceHandle and sortType can be identified easily as separate digits on debugging via ETG traces.
   ETG_TRACE_USR4 (( " List handle generated is: 0x%04x",u16ListHandle));
   ETG_TRACE_USR4 (( " List type is: %d", ETG_CENUM(FC_PhoneBook_SQLite::enListType, u8ListType) ));

   FC_PhoneBook_List_Data* pPBListData;
   if(m_multiHashPBClientRequests.contains(u16ListHandle))
   {
      //Retrieve the old List from Hashtable
      ETG_TRACE_USR4 (( "Retrieving old List Handle as same list has been requested before "));
      pPBListData = m_hashPbList.value(u16ListHandle);
   }
   else
   {
      pPBListData = new FC_PhoneBook_List_Data;
      ETG_TRACE_USR4 (( "New list created in RAM as list was not requested before"));

      //It is a newly created list, add it to the hash tables
      if(pPBListData)
      {
         m_hashPbList.insert( u16ListHandle, pPBListData );
         ETG_TRACE_USR4 (( "m_hashPbList inserted with entry for Listhandle: 0x%04x",u16ListHandle));

         //Store values specific to the newly created list
         pPBListData->u8DeviceHandle= methodStart->u8DeviceHandle;
         pPBListData->sortType = sortType;
         pPBListData->listType = u8ListType;
      }
   }

   //Get all records from Database if the list is not yet created.
   //On subsequent changes to list via Addcontact/EditContact/DeleteContact of VPB or download of device PB,
   //it will be updated via GetRecords() invocation in UpdateList on Phonebook HashTable via actions like enAction_Contact_Added/enAction_Contact_Edited/enAction_Contact_Deleted/enAction_Content_Refreshed
   if( (pPBListData) && (!(pPBListData->contactList.size())) )
   {
      QTime DebugTime;
      DebugTime.start();
      m_sqlite->GetRecords(u8ListType, sortType, methodStart->u8DeviceHandle, pPBListData->contactList);  //List is passed as reference to reduce the amount of memory used
      ETG_TRACE_USR2((" -PBDL.Perf- 'CreatePhoneBookList.GetRecords': exited after %d ms. [DeviceHandle= %u ListType= %d, sortType= %d], ListLength= %d", 
                        DebugTime.elapsed(), methodStart->u8DeviceHandle, ETG_CENUM(FC_PhoneBook_SQLite::enListType, u8ListType), 
                        ETG_CENUM(FC_PhoneBook_SQLite::SortType, sortType), pPBListData->contactList.size() ));
   }

   //If it is not a duplicate request, it needs to be stored in the ClientRequests Hashtable
   if(!(m_multiHashPBClientRequests.contains(u16ListHandle, u32RegisterID)))
   {
      m_multiHashPBClientRequests.insert(u16ListHandle, u32RegisterID);
      ETG_TRACE_USR4(( " Added entry to multiHashPBClientRequests for RegisterID: %d ", u32RegisterID ));
   }

   //Assign the list handle to methodresult
   methodResult->u16ListHandle = u16ListHandle;

   if(pPBListData)
   {
      methodResult->u16ListLength = (quint16)(pPBListData->contactList.size());
      ETG_TRACE_USR4(( " Updated methodResult with ListLength: %d ", methodResult->u16ListLength ));

      // CMG3G-6838 To remove LINT Warning 429: Custodial Pointer
      tVoid*  pCustodialLose = pPBListData;
      (tVoid) pCustodialLose;
   }

   return true;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::CreateContactList
*
* DESCRIPTION:
*              Creates a new contact info list from the database.
*
*
* PARAMETER: [IN]  methodStart  :- Input arguments like sort type etc are in this class
*            [OUT] methodResult :- new list handle and list length are in this class
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_List::CreateContactList(
   const most_phonbkfi_tclMsgCreateContactListMethodStart* methodStart,
   most_phonbkfi_tclMsgCreateContactListMethodResult* methodResult,
   tU32 u32RegisterID)
{
   ETG_TRACE_USR4(( "CreateContactList entered " ));
   bool bRetValue = false;
   //Get the list type from method start

   //Get the sort type from method start
   FC_PhoneBook_SQLite::SortType sortType = getSortType(&methodStart->e8PhoneBkContactSortType);
   if(FC_PhoneBook_SQLite::Invalid != sortType)
   {
      const most_fi_tcl_e8_PhonBkContactListType* type = &methodStart->e8PhoneBkContactListType;
      tU8 u8listType = type->enType;
      tU16 u16ListHandle = (tU16)(u8listType * (16*16) + methodStart->u8DeviceHandle * 16 + sortType); //This format is chosen for creating list handle; as DeviceHandle and sortType can be identified easily as separate digits on debugging via ETG traces.
      ETG_TRACE_USR4(( " List handle generated is: 0x%04x ", u16ListHandle ));
      ETG_TRACE_USR4(( " List type is: %d ", ETG_CENUM(FC_PhoneBook_SQLite::enListType, u8listType)));

      FC_PhoneBook_List_Data* pPBListData;
      if(m_multiHashPBClientRequests.contains(u16ListHandle))
      {
         //Retrieve the old List from Hashtable
         ETG_TRACE_USR4(( " Retrieving old List Handle as same list has been requested before " ));
         pPBListData = m_hashPbList.value(u16ListHandle);
      }
      else
      {
         m_multiHashPBClientRequests.insert(u16ListHandle,u32RegisterID);
         ETG_TRACE_USR4(( " Added entry to multiHashPBClientRequests for RegisterID: %d ", u32RegisterID ));
         pPBListData = new FC_PhoneBook_List_Data;
         ETG_TRACE_USR4(( " New list created in RAM as list was not requested before " ));

         //It is a newly created list, add it to the hash tables
         if(pPBListData)
         {
            m_hashPbList.insert( u16ListHandle, pPBListData );
            ETG_TRACE_USR4(( " m_hashPbList inserted with entry for Listhandle: 0x%04x ", u16ListHandle ));

            //Store values specific to the newly created list
            pPBListData->u8DeviceHandle= methodStart->u8DeviceHandle;
            pPBListData->sortType = sortType;
            pPBListData->listType = u8listType;
         }
      }

      //Get all records from Database if the list is not yet created.
      //On subsequent changes to list via Addcontact/EditContact/DeleteContact of VPB or download of device PB,
      //it will be updated via GetRecords() invocation in UpdateList on Phonebook HashTable via actions like enAction_Contact_Added/enAction_Contact_Edited/enAction_Contact_Deleted/enAction_Content_Refreshed
      if ((pPBListData) && ( !(pPBListData->contactList.size())) )
      {
         QTime DebugTime;
         DebugTime.start();
         m_sqlite->GetRecords(u8listType, sortType, methodStart->u8DeviceHandle, pPBListData->contactList);//List is passed as reference to reduce the amount of memory used
      //If it is not a duplicate request, it needs to be stored in the ClientRequests Hashtable
      ETG_TRACE_USR2((" -PBDL.Perf- 'CreateContactList.GetRecords': exited after %d ms. [DeviceHandle= %u ListType= %d, sortType= %d], ListLength= %d", 
                        DebugTime.elapsed(), methodStart->u8DeviceHandle, ETG_CENUM(FC_PhoneBook_SQLite::enListType, u8listType), 
                        ETG_CENUM(FC_PhoneBook_SQLite::SortType, sortType), pPBListData->contactList.size() ));
      }

      //Assign the list handle to methodresult
      methodResult->u16ListHandle = u16ListHandle;

      if(pPBListData)
      {
         methodResult->u16ListLength = (quint16)(pPBListData->contactList.size());
         ETG_TRACE_USR4 (( "Updated methodResult with ListLength: %d",methodResult->u16ListLength));

        // CMG3G-6838 To remove LINT Warning 429: Custodial Pointer
         tVoid*  pCustodialLose = pPBListData ;
         (tVoid) pCustodialLose ;
      }
      bRetValue = true;
   }
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::RequestSlicePhoneBookList
*
* DESCRIPTION:
*              Creates a subset of the list pointed by listhandle
*
*
* PARAMETER: [IN] methodStart FC_PB_DB_NUM_0:- Input arguments like listhandles are in this class
*            [OUT] methodResult :- The acutal sub list is passed in this class
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_List::RequestSlicePhoneBookList(
   const most_phonbkfi_tclMsgRequestPhoneBookListSliceMethodStart * methodStart,
   most_phonbkfi_tclMsgRequestPhoneBookListSliceMethodResult* methodResult )

{
   ETG_TRACE_USR4 (( "RequestSlicePhoneBookList entered "));
   QTime DebugTime;
   DebugTime.start();
   bool bRetValue = false;
   if (false == m_hashPbList.contains(methodStart->u16ListHandle))
   {
      ETG_TRACE_USR4 (( "FC_PhoneBook_List :- WARNING - empty list returned in RequestSlice "));
      return bRetValue;
   }
   else
   {
      FC_PhoneBook_List_Data* data = m_hashPbList.value( methodStart->u16ListHandle, NULLPTR );
      //CMG#G-14313-Coverity Fix-NULL check for data
      if(!data)
      {
           ETG_TRACE_ERR (( "FC_PhoneBook_List :- Data/HashList is NULL "));
           return false;
      }
      //end of Fix
      unsigned listLen = (unsigned)data->contactList.size();

      if ( (methodStart->u16WindowSize < 1) ||
         (methodStart->u32WindowStart >= listLen) )
      {
         ETG_TRACE_USR4(("WARNING - negative number requested in windowStart or WindowSize, or zero windowSize, returning empty list"));
      }
      else
      {
         unsigned end = methodStart->u32WindowStart + methodStart->u16WindowSize;

         if ( end > listLen )
            end = listLen;

         ETG_TRACE_USR4(( "TOTAL Items = %u",listLen));
         ETG_TRACE_USR4(( "START = %u",methodStart->u32WindowStart));
         ETG_TRACE_USR4(( "END = %u",end-1));
         ETG_TRACE_USR4(( "SIZE = %u",methodStart->u16WindowSize));

         //Fix CMG3G-12880
         FC_Device_Details
               * pDeviceObject =
                     FC_Device_Details::deviceHandleToObjectInstance(data->u8DeviceHandle);
         if (pDeviceObject)
         {
            //If device is not locked, continue as normal
            if (FALSE == pDeviceObject->bIsDeviceProbablyLocked())
            {
               for (unsigned counter = methodStart->u32WindowStart; counter < end; ++counter)
               {
                  most_fi_tcl_PhonBkPhoneBookListSliceResultItem item;

                  Convert(&data->contactList.at((int) counter), &item);

                  methodResult->oPhoneBookListSliceResult.oItems.push_back(item);

                  //TODO:
                  //Should be call vDestroy() on this or not?
                  //Update:- calling vDestory does not work, as it also modifies the copy
                  //in the vector.
                  //item.vDestroy();
               }
            }
            else
            {
               //Fix : CMG3G-12880 Mock phonebook list as empty since device is locked. so returning true
               ETG_TRACE_USR4(("Device is locked"));
            }
            bRetValue = true;
         }
         else
         {
            //Fix RTC 159338
            //In TML, if we try to add contact in quick dial with the device handle one (though the current device handle is not one),
            //the contact will be added into quick dial list. But, through TML, it has requested for interface "RequestSlicePhoneBook"
            //in a sequence once contact has been added to quick dial successfully. So, FC_Phonebook is throwing an errot to TML test adapter
            //like "Device object doesn't exist for device handle". To overcome that a check has been added (1 == data->u8DeviceHandle)
            if(1 == data->u8DeviceHandle)
            {
               for (unsigned counter = methodStart->u32WindowStart; counter < end; ++counter)
               {
                  most_fi_tcl_PhonBkPhoneBookListSliceResultItem item;

                  Convert(&data->contactList.at((int) counter), &item);

                  methodResult->oPhoneBookListSliceResult.oItems.push_back(item);
               }
               bRetValue = true;
            }
            else
            {
               ETG_TRACE_USR4((" Device object not found for given device handle"));
            }
            //End fix RTC 159338
         }
      }
      ETG_TRACE_USR2(( " -PBDL.Perf- 'RequestSlicePhoneBookList' exited after %d ms. SliceList-Size= %d ", DebugTime.elapsed(), listLen ));
   }
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::ReleasePhoneBookList
*
* DESCRIPTION:
*              Free the memory occupied by the list pointed by the listhandle.
*180
*
* PARAMETER: [IN] methodStart :- Input arguments like listhandles are in this class
*            [OUT] methodResult :- Error if any
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_List::ReleasePhoneBookList(
   const most_phonbkfi_tclMsgReleasePhoneBookListMethodStart* methodStart,
   most_phonbkfi_tclMsgReleasePhoneBookListMethodResult* methodResult,
   tU32 u32RegisterID)//FIX GMMY15-1578 RegisterID is used to identify the requested client
{
   ETG_TRACE_USR4 (( "ReleasePhoneBookList entered "));

   bool bRetValue = false;

   //FIX GMMY15-1578 Bosch Internal: Proper usage of SQLite database instead of RAM -- removing redundant objects which are created in RAM for different clients & sorted lists.
   ETG_TRACE_USR4(( "Request from Client with RegisterID: %d",u32RegisterID));
   if( m_multiHashPBClientRequests.contains(methodStart->u16ListHandle,u32RegisterID) )
   {
      ETG_TRACE_USR4(( "Client has requested for the list."));
      if( m_multiHashPBClientRequests.count(methodStart->u16ListHandle) > 1 )
      {
         //Same list requested more than once. So we don't release the list from RAM
         ETG_TRACE_USR4(( "Multiple clients have requested for the same list."));
      }
      else
      {
         //Only this client has requested the list before so release list from RAM
         ETG_TRACE_USR4(( "Deleting the list and pointer to list. "));
         FC_PhoneBook_List_Data* PBListData = m_hashPbList.take(methodStart->u16ListHandle);//Deleting entry from PB hash
         if(PBListData)
         {
            PBListData->contactList.clear();//Deleting PB list of FC_PhoneBook_Contact_Detail inside structure FC_Phonebook_List_Data
            delete PBListData;//Deleting structure FC_PhoneBook_List_Data
            m_hashPbList.squeeze();//Deallocates excess memory in RAM
         }
      }
      //For all valid requests, we delete the entry corresponding to the request from the particular client from ClientRequests HashList
      m_multiHashPBClientRequests.remove(methodStart->u16ListHandle,u32RegisterID);
      m_multiHashPBClientRequests.squeeze();//Deallocates excess memory in RAM

      //Fill values in methodResult
      malloc_trim(0);  // -bn: This is a trigger to return freed memory to the OS

      methodResult->u16ListHandle = methodStart->u16ListHandle;
      bRetValue = true;
   }
   else
   {
      //List was not requested before by the client
      ETG_TRACE_USR4(( "Client has not requested for the list."));
      bRetValue = false;
   }
   //End of FIX GMMY15-1578

   return bRetValue;

}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::SearchKeyboardPhoneBookList
*
* DESCRIPTION:
*              Returns the keyboard list
*
*
* PARAMETER: [IN] methodStart :- Input arguments like listhandles are in this class
*            [OUT] methodResult :- Error if any
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_List::SearchKeyboardPhoneBookList(
   const most_phonbkfi_tclMsgSearchKeyboardPhoneBookListMethodStart* methodStart,
   most_phonbkfi_tclMsgSearchKeyboardPhoneBookListMethodResult* methodResult )
{
   ETG_TRACE_USR4 (( "SearchKeyboardPhoneBookList entered "));

   if ( false == m_hashPbList.contains( methodStart->u16ListHandle))
   {
      ETG_TRACE_USR4(( "FC_PhoneBook_List :- WARNING - Could not find any matching listhandle to search"));
      return false;
   }

   FC_PhoneBook_List_Data* data = m_hashPbList.value( methodStart->u16ListHandle, NULLPTR );

   QStringList l;
   int i;

   if(data != NULL)
   {
      for(i=0;i<data->contactList.size();i++)
      {
         // We use the "Reported" First/LastName (depending on the SortingType) to evaluate the Vertical Keyboard.
         switch( data->sortType )
         {
            case FC_PhoneBook_SQLite::FirstName:
            case FC_PhoneBook_SQLite::Number_FirstName:
            case FC_PhoneBook_SQLite::Address_FirstName:
            case FC_PhoneBook_SQLite::Email_FirstName:
            {
               // l << data->contactList.at(i).firstName;
               l << data->contactList.at(i).sortName;   // Hotfix for CRQ 509734 Phonetic Sorting
            }
            break;

            case FC_PhoneBook_SQLite::LastName:
            case FC_PhoneBook_SQLite::Number_LastName:
            case FC_PhoneBook_SQLite::Address_LastName:
            case FC_PhoneBook_SQLite::Email_LastName:
            {
               // l << data->contactList.at(i).lastName;
               l << data->contactList.at(i).sortName;   // Hotfix for CRQ 509734 Phonetic Sorting
            }
            break;

            default:
            {
               ETG_TRACE_FATAL((" ::SearchKeyboardPhoneBookList: We never expect to come here. " ));
            }
         }
      }
   }

   VerticalSearchKeyboard k;

   QList<VerticalSearchKeyboardListItem>* lt;
   //English is the default language in all the projects except GM
   //lt = k.showKeyboard(l,NA_ENGLISH);
   //CRQ- 243550: HMI tells FC_PHONEBOOK the currently used system language so that the FC_PHONEBOOK can setup the vertical keyboard accordingly .
   tU8 u8PbkLanguage = FC_PBK_DEFAULT_LANG;
   if (fc_phonebook_tclService_PhoneBook::pGetInstance())
   {
      ETG_TRACE_USR4 (( "FC_Phonebook_List::Calling GetPhonebookLanguage" ));
      u8PbkLanguage = fc_phonebook_tclService_PhoneBook::pGetInstance()->GetPhonebookLanguage();
      ETG_TRACE_USR4 (( "FC_Phonebook_List::Phonebook Language = %u ", u8PbkLanguage ));
   }

   lt = k.showKeyboard(l,u8PbkLanguage);

   int j;
   for(j=0;j<lt->size();j++)
   {
      most_fi_tcl_PhonBkSearchKeyboardListItem o;

      QString s((*lt)[j].m_sSearchKeyboardString);

      o.sSearchKeyboardLetter.bSet(s.toUtf8().constData());
      o.bLetterAvailable = (*lt)[j].m_bStringAvailable;
      o.u32LetterStartIndex = (*lt)[j].m_u32StringStartIndex;
      o.u32LetterEndIndex = (*lt)[j].m_u32StringEndIndex;

      methodResult->oSearchKeyboardList.oItems.push_back(o);

   }
   delete lt;
   methodResult->u16ListHandle = methodStart->u16ListHandle;

   return true;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::SearchPhoneBook
*
* DESCRIPTION:
*
*
*
* PARAMETER: [IN] methodStart :- Input arguments like PhoneNumber/Email address are in this class
*            [OUT] methodResult :- Contact handle, First Name, Last Name and Contact details
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_List::SearchPhoneBook( const most_phonbkfi_tclMsgSearchPhoneBookMethodStart* methodStart,
                                         most_phonbkfi_tclMsgSearchPhoneBookMethodResult* methodResult)
{
   // TODO
   // Search in Vehicle address book
   // To check the version ID to be passed with u32GetSize()
   ETG_TRACE_USR4(( "SearchPhoneBook entered "));

   QTime DebugTime;
   DebugTime.start();
   
   tString sgSearchString;
   FC_PhoneBook_SQLite::SearchType u8SearchType;
   FC_PhoneBook_Search_Result Search_Result;

   sgSearchString = methodStart->sPhoneNumber.szValue;
   GString* str= g_string_new(sgSearchString);
   tU16 len= (tU16)(str->len);

   QByteArray batempsearchstring;


   if (len)
   {
      ETG_TRACE_USR2((" Phone no. to be searched = %s", str->str));
      ETG_TRACE_USR2(("Length of the Phone No. = %d", len));

      QString sStrippedSearchString;
      sStrippedSearchString = StripTelephoneNumber(QString::fromUtf8(sgSearchString));
      batempsearchstring  = sStrippedSearchString.toUtf8();
      sgSearchString = batempsearchstring.data();
      len = (tU16)(sStrippedSearchString.length());
      ETG_TRACE_USR4((" len = %d", len ));
      ETG_TRACE_USR2((" Final sgSearchString = %s", sgSearchString));


      if (len > FC_PHONEBOOK_LIST_SEARCH_NUM_LEN)
      {
         sgSearchString += len - FC_PHONEBOOK_LIST_SEARCH_NUM_LEN;

      }

      u8SearchType = FC_PhoneBook_SQLite::PhoneNumber;

   }
   else
   {
      g_string_free(str, true);
      sgSearchString= methodStart->sEmailAddress.szValue;
      str= g_string_new(sgSearchString);
      len= (tU16)(str->len);

      ETG_TRACE_USR2((" EmailID to be searched = %s", str->str ));

      if(len)
      {
         u8SearchType= FC_PhoneBook_SQLite::EmailID;
      }
      else
      {
         ETG_TRACE_USR4(("FC_PhoneBook_List :- Empty search key received "));
         g_string_free(str, true);
         return false;
      }
   }

   g_string_free(str, true);

   ETG_TRACE_USR2(("Search Type = %u", u8SearchType ));
   ETG_TRACE_USR2(("Search Key = %s", sgSearchString ));

   bool ret= false;

   //Fix : CMG3G-12880
   FC_Device_Details * pDeviceObject = FC_Device_Details::deviceHandleToObjectInstance(methodStart->u8DeviceHandle);
   if (pDeviceObject)
   {
      //If device is not locked contiue as before
      if (false == pDeviceObject->bIsDeviceProbablyLocked())
      {
         if ((m_sqlite != NULL))
         {
            if( m_sqlite->GetMatchingRecord(Search_Result, sgSearchString, u8SearchType, methodStart->u8DeviceHandle) )
            {
                ETG_TRACE_USR4(("searching in DeviceTable"));
                ret = true;
            }
            else if (m_sqlite->GetMatchingRecord(Search_Result,sgSearchString, u8SearchType, FC_PHONEBOOK_VEHICLE_DEVICEHANDLE))
            {
                ETG_TRACE_USR4(("searching in VPB"));
                ret = true;
            }
            else if (m_sqlite->GetMatchingRecord(Search_Result,sgSearchString, u8SearchType, (quint8)(methodStart->u8DeviceHandle + FC_PB_TOTAL_NUMBER_OF_VPB)))
            {
                ETG_TRACE_USR4(("searching in VPBx"));
                ret = true;
            }
            else
            {
                ETG_TRACE_USR4(("contact not found"));
                ret = false;
            }

            if(ret == true)
            {
               QByteArray ba;
               methodResult->u32ContactHandle = Search_Result.contactHandle;
               ba = Search_Result.firstname.toUtf8();
               methodResult->sFirstName.bSet(ba.constData());
               ba = Search_Result.lastname.toUtf8();
               methodResult->sLastName.bSet(ba.constData());
               MapContactDetailTypeToFItype(Search_Result.contactdetailtype);
               methodResult->e8ContactDetailType.enType = (most_fi_tcl_e8_PhonBkContactDetailType::tenType) Search_Result.contactdetailtype;
            }
         }
      }
      else
      {
         ETG_TRACE_USR4(("Device is locked, hence mocking the search result"));
         methodResult->sFirstName.bSet("");
         methodResult->sLastName.bSet("");
         ret = true;
      }
   }
   else
   {
      ETG_TRACE_USR4((" Device object not found for given device handle"));
   }
   
   ETG_TRACE_USR2(( " -PBDL.Perf- 'SearchPhonebook' exited after %d ms ", DebugTime.elapsed() ));

   return ret;
}

/*******************************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::MapContactDetailTypeToFItype
*
* DESCRIPTION:  Mapping of the 'contactdetailtype' which is used within FC_PhoneBook_SQLite
*               to the 'phonenumber/email type' (T_e8_PhonBkContactDetailType), which is
*               relevant for FI method 'SearchPhonebook::MethodResult'
*
*
*
*
*********************************************************************************************/

void FC_PhoneBook_List::MapContactDetailTypeToFItype(quint8& u8NumberType)
{

   ETG_TRACE_USR4(( "FC_PhoneBook_List::MapContactDetailTypeToSearchPhonetype entered with numbertype %d",u8NumberType));

   switch(u8NumberType)
      {
      case FC_PhoneBook_SQLite::PrefferedNum:
         u8NumberType= most_fi_tcl_e8_PhonBkContactDetailType::FI_EN_E8PREFERRED_NUMBER;
         break;

      case FC_PhoneBook_SQLite::CellNumber:
         u8NumberType= most_fi_tcl_e8_PhonBkContactDetailType::FI_EN_E8CELL_NUMBER_1;
         break;

      case FC_PhoneBook_SQLite::HomeNumber:
         u8NumberType= most_fi_tcl_e8_PhonBkContactDetailType::FI_EN_E8HOME_NUMBER_1;
         break;

      case FC_PhoneBook_SQLite::WorkNumber:
         u8NumberType= most_fi_tcl_e8_PhonBkContactDetailType::FI_EN_E8WORK_NUMBER_1;
         break;

      case FC_PhoneBook_SQLite::OtherNumber:
         u8NumberType=  most_fi_tcl_e8_PhonBkContactDetailType::FI_EN_E8OTHER_NUMBER;
         break;

	  case FC_PhoneBook_SQLite::EmailAdd1:
         u8NumberType=  most_fi_tcl_e8_PhonBkContactDetailType::FI_EN_E8EMAIL_ADDRESS_1;
         break;

	  case FC_PhoneBook_SQLite::EmailAdd2:
         u8NumberType=  most_fi_tcl_e8_PhonBkContactDetailType::FI_EN_E8EMAIL_ADDRESS_2;
         break;

	  case FC_PhoneBook_SQLite::EmailAdd3:
         u8NumberType=  most_fi_tcl_e8_PhonBkContactDetailType::FI_EN_E8EMAIL_ADDRESS_3;
         break;
      default:
         u8NumberType= most_fi_tcl_e8_PhonBkContactDetailType::FI_EN_E8NONE;
      }

}

/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::SearchContacts
*
* DESCRIPTION:
*
*
*
* PARAMETER: [IN] methodStart :- Input arguments like DeviceHandleList, SearchString, FieldSelector, SearchSortType, MaxReportedMatches are in this class
*            [OUT] methodResult :- Contains the arguments like Contact handle, First Name, Last Name, NumberOfMatches, MatchLimitExceeded
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_List::SearchContacts( const most_phonbkfi_tclMsgSearchContactsMethodStart* methodStart,
                                        most_phonbkfi_tclMsgSearchContactsMethodResult* methodResult)
{
   ETG_TRACE_USR4(( "FC_PhoneBook_List::SearchContacts entered "));

   if( m_sqlite == NULL )
   {
      return FALSE;
   }
   //Validating SearchString
   QString sSearchString = methodStart->sSearchString.szValue;
   ETG_TRACE_USR4(( "SearchContacts - sSearchString: %s", sSearchString.toUtf8().constData()));
   if (false == bValidateSearchString(sSearchString))
   {
      ETG_TRACE_ERR(( "SearchContacts - Invalid SearchString"));
      return FALSE;
   }

   //Limiting the Reported Matches
   tU16 u16MaximumReportedMatches = methodStart->u16MaxReportedMatches;
   ETG_TRACE_USR4(( "SearchContacts - Received u16MaximumReportedMatches: %u", u16MaximumReportedMatches));
   if (u16MaximumReportedMatches > MAXIMUM_REPORTED_MATCHES)
   {
       u16MaximumReportedMatches = MAXIMUM_REPORTED_MATCHES;
       ETG_TRACE_USR4(( "SearchContacts - Limiting the Maximum Reported Matches to the internal limit : %u", u16MaximumReportedMatches));
   }

   //Validating SearchSortType
   FC_PhoneBook_SQLite::SortType enSearchSortType;
   enSearchSortType = getSortType(&methodStart->e8SearchSortType);
   ETG_TRACE_USR4(( "SearchContacts - enSearchSortType: %u", ETG_ENUM(TR_PBKSORTTYPE,enSearchSortType)));
   if(enSearchSortType > FC_PhoneBook_SQLite::Email_FirstName)
   {
      ETG_TRACE_ERR(( "SearchContacts - Invalid SearchSortType"));
      return FALSE;
   }

   //Validating FieldSelector
   tU16 u16FieldSelector = methodStart->u16FieldSelector;
   ETG_TRACE_USR4(( "SearchContacts - u16FieldSelector: %u", u16FieldSelector));
   if((0 == u16FieldSelector) || (u16FieldSelector > 0x003F))   // The Maximum Field selection combinations defined now is (0011 1111)b i.e.0x3F
   {
      ETG_TRACE_USR4(( "SearchContacts - Invalid FieldSelector"));
      return FALSE;
   }

   //Populating  and Validating the DeviceHandleList -CMG3G-13535 fix
   QList < tU8 > DeviceHandleList;
   for (tU8 u8Itr=0; u8Itr< methodStart->oDeviceHandleList.oDeviceHandles.size(); u8Itr++)
   {
       DeviceHandleList.append(methodStart->oDeviceHandleList.oDeviceHandles[u8Itr]);
       ETG_TRACE_USR4(( "SearchContacts - DeviceHandleList[%u]: %u", u8Itr, DeviceHandleList[u8Itr]));
   }
   if(false == bValidateDeviceHandles(DeviceHandleList))
   {
      ETG_TRACE_ERR(( "SearchContacts - Invalid DeviceHandleList"));
      return FALSE;
   }
   //end of CMG3G-13535 fix

   QList<SearchContacts_Result> SearchResultList;

   if(!m_sqlite->SearchContacts(DeviceHandleList,sSearchString,enSearchSortType,u16FieldSelector,u16MaximumReportedMatches,SearchResultList))
   {
      ETG_TRACE_USR4((" Search for Contacts containing the given SearchString Failed"));
      return FALSE;
   }

   //Populate Method result
   tU16 u16ListSize = (tU16)(SearchResultList.size());

   if (u16ListSize > u16MaximumReportedMatches)
   {
      methodResult->bMatchLimitExceeded = TRUE;
      methodResult->u16NumberOfMatches = (tU16)(u16ListSize-1);
   }
   else
   {
      methodResult->bMatchLimitExceeded = FALSE;
      methodResult->u16NumberOfMatches = u16ListSize;
   }
   ETG_TRACE_USR4((" bMatchLimitExceeded: %d", methodResult->bMatchLimitExceeded));
   ETG_TRACE_USR4((" u16NumberOfMatches: %d", methodResult->u16NumberOfMatches));

   for (quint16 u16Index = 0; u16Index < methodResult->u16NumberOfMatches; u16Index++)
   {
      QByteArray ba;
      most_fi_tcl_PhonBkContactMatchListItem items;

      ba = SearchResultList.at(u16Index).ReportedFirstName.toUtf8();
      items.sReportedFirstName.bSet(ba.constData());
      ba = SearchResultList.at(u16Index).ReportedLastName.toUtf8();
      items.sReportedLastName.bSet(ba.constData());
      items.u32ContactHandle = SearchResultList.at(u16Index).ContactHandle;

      ETG_TRACE_USR4((" Entered additional info search contacts"));

      items.bPhotoAvailable = SearchResultList.at(u16Index).bPhotoAvailable;
      ETG_TRACE_USR4((" items.bPhotoAvailable: %d", items.bPhotoAvailable));

      items.u8EmailCount = SearchResultList.at(u16Index).u8EmailAddrCount;
      ETG_TRACE_USR4((" items.u8EmailCount: %d", items.u8EmailCount));

      //428861- Additional parameter added to SearchContactResultList for ContactMatching
      items.u8AddressesCount = SearchResultList.at(u16Index).u8AddressesCount;
      ETG_TRACE_USR4((" items.u8AddressesCount: %d", items.u8AddressesCount));
      //end of fix
      QList<PbkPhoneNoItem> qSearchPhoneNoItemList;

      if( SearchResultList.at(u16Index).PhoneNum1 != "")
      {
         PbkPhoneNoItem oSearchPhNoItem;
         oSearchPhNoItem.qsPhoneNumber = SearchResultList.at(u16Index).PhoneNum1;
         oSearchPhNoItem.enPhoneNumberType =  static_cast<tenPhonBkNumberType>(SearchResultList.at(u16Index).NumberType1);
         ETG_TRACE_USR4(( "oSearchPhNoItem.enPhoneNumberType: %d", oSearchPhNoItem.enPhoneNumberType));
         qSearchPhoneNoItemList << oSearchPhNoItem;
      }

      if(SearchResultList.at(u16Index).PhoneNum2 != "")
      {
         PbkPhoneNoItem oSearchPhNoItem;
         oSearchPhNoItem.qsPhoneNumber = SearchResultList.at(u16Index).PhoneNum2;
         oSearchPhNoItem.enPhoneNumberType =  static_cast<tenPhonBkNumberType>(SearchResultList.at(u16Index).NumberType2);
         ETG_TRACE_USR4(( "oSearchPhNoItem.enPhoneNumberType: %d", oSearchPhNoItem.enPhoneNumberType));
         qSearchPhoneNoItemList << oSearchPhNoItem;
      }

      if(SearchResultList.at(u16Index).PhoneNum3 != "")
      {
         PbkPhoneNoItem oSearchPhNoItem;
         oSearchPhNoItem.qsPhoneNumber = SearchResultList.at(u16Index).PhoneNum3;
         oSearchPhNoItem.enPhoneNumberType =  static_cast<tenPhonBkNumberType>(SearchResultList.at(u16Index).NumberType3);
         ETG_TRACE_USR4(( "oSearchPhNoItem.enPhoneNumberType: %d", oSearchPhNoItem.enPhoneNumberType));
         qSearchPhoneNoItemList << oSearchPhNoItem;
      }

      if(SearchResultList.at(u16Index).PhoneNum4 != "")
      {
         PbkPhoneNoItem oSearchPhNoItem;
         oSearchPhNoItem.qsPhoneNumber = SearchResultList.at(u16Index).PhoneNum4;
         oSearchPhNoItem.enPhoneNumberType =  static_cast<tenPhonBkNumberType>(SearchResultList.at(u16Index).NumberType4);
         ETG_TRACE_USR4(( "oSearchPhNoItem.enPhoneNumberType: %d", oSearchPhNoItem.enPhoneNumberType));
         qSearchPhoneNoItemList << oSearchPhNoItem;
      }

      if(SearchResultList.at(u16Index).PhoneNum5 != "")
      {
         PbkPhoneNoItem oSearchPhNoItem;
         oSearchPhNoItem.qsPhoneNumber = SearchResultList.at(u16Index).PhoneNum5;
         oSearchPhNoItem.enPhoneNumberType =  static_cast<tenPhonBkNumberType>(SearchResultList.at(u16Index).NumberType5);
         ETG_TRACE_USR4(( "oSearchPhNoItem.enPhoneNumberType: %d", oSearchPhNoItem.enPhoneNumberType));
         qSearchPhoneNoItemList << oSearchPhNoItem;
      }

      if(SearchResultList.at(u16Index).PhoneNum6 != "")
      {
         PbkPhoneNoItem oSearchPhNoItem;
         oSearchPhNoItem.qsPhoneNumber = SearchResultList.at(u16Index).PhoneNum6;
         oSearchPhNoItem.enPhoneNumberType =  static_cast<tenPhonBkNumberType>(SearchResultList.at(u16Index).NumberType6);
         ETG_TRACE_USR4(( "oSearchPhNoItem.enPhoneNumberType: %d", oSearchPhNoItem.enPhoneNumberType));
         qSearchPhoneNoItemList << oSearchPhNoItem;
      }

      if(SearchResultList.at(u16Index).PhoneNum7 != "")
      {
         PbkPhoneNoItem oSearchPhNoItem;
         oSearchPhNoItem.qsPhoneNumber = SearchResultList.at(u16Index).PhoneNum7;
         oSearchPhNoItem.enPhoneNumberType =  static_cast<tenPhonBkNumberType>(SearchResultList.at(u16Index).NumberType7);
         ETG_TRACE_USR4(( "oSearchPhNoItem.enPhoneNumberType: %d", oSearchPhNoItem.enPhoneNumberType));
         qSearchPhoneNoItemList << oSearchPhNoItem;
      }

      if(SearchResultList.at(u16Index).PhoneNum8 != "")
      {
         PbkPhoneNoItem oSearchPhNoItem;
         oSearchPhNoItem.qsPhoneNumber = SearchResultList.at(u16Index).PhoneNum8;
         oSearchPhNoItem.enPhoneNumberType =  static_cast<tenPhonBkNumberType>(SearchResultList.at(u16Index).NumberType8);
         ETG_TRACE_USR4(( "oSearchPhNoItem.enPhoneNumberType: %d", oSearchPhNoItem.enPhoneNumberType));
         qSearchPhoneNoItemList << oSearchPhNoItem;
      }

      tU8 u8SearchItemListsize = (tU8)(qSearchPhoneNoItemList.size());
      ETG_TRACE_USR2(( "qSearchPhoneNoItemListSize = %d", u8SearchItemListsize));

      items.u8PhoneNumberCount = u8SearchItemListsize;
      ETG_TRACE_USR2(( " u8PhoneNumberCount = %d", items.u8PhoneNumberCount ));

      if (u8SearchItemListsize > 0)
      {
         for(tU8 u8SearchPhNoIndex = 0; u8SearchPhNoIndex < u8SearchItemListsize; u8SearchPhNoIndex++)
         {
            PbkPhoneNoItem oSearchPhNoItem = qSearchPhoneNoItemList.at(u8SearchPhNoIndex);
            QByteArray qbPhoneNo = oSearchPhNoItem.qsPhoneNumber.toUtf8();
            ETG_TRACE_USR2(( "oSearchPhoneNumber%d = %s", u8SearchPhNoIndex+1, qbPhoneNo.constData() ));

            switch (u8SearchPhNoIndex)
            {
               case 0:
               {
                  items.oPhoneNumber1.sNumber.bSet(qbPhoneNo.constData() );
                  items.oPhoneNumber1.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)oSearchPhNoItem.enPhoneNumberType;
                  ETG_TRACE_USR2(( " oSearchPhoneNumber1type = %d", items.oPhoneNumber1.e8Type.enType ));
               }
               break;

               case 1:
               {
                  items.oPhoneNumber2.sNumber.bSet(qbPhoneNo.constData() );
                  items.oPhoneNumber2.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)oSearchPhNoItem.enPhoneNumberType;
                  ETG_TRACE_USR2(( " oSearchPhoneNumber2type = %d", items.oPhoneNumber2.e8Type.enType ));
               }
               break;

               case 2:
               {
                  items.oPhoneNumber3.sNumber.bSet(qbPhoneNo.constData() );
                  items.oPhoneNumber3.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)oSearchPhNoItem.enPhoneNumberType;
                  ETG_TRACE_USR2(( " oSearchPhoneNumber3type = %d", items.oPhoneNumber3.e8Type.enType ));
               }
               break;

               case 3:
               {
                  items.oPhoneNumber4.sNumber.bSet(qbPhoneNo.constData() );
                  items.oPhoneNumber4.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)oSearchPhNoItem.enPhoneNumberType;
                  ETG_TRACE_USR2(( " oSearchPhoneNumber4type = %d", items.oPhoneNumber4.e8Type.enType ));
               }
               break;

               case 4:
               {
                  items.oPhoneNumber5.sNumber.bSet(qbPhoneNo.constData() );
                  items.oPhoneNumber5.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)oSearchPhNoItem.enPhoneNumberType;
                  ETG_TRACE_USR2(( " oSearchPhoneNumber5type = %d", items.oPhoneNumber5.e8Type.enType ));
               }
               break;

               case 5:
               {
                  items.oPhoneNumber6.sNumber.bSet(qbPhoneNo.constData() );
                  items.oPhoneNumber6.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)oSearchPhNoItem.enPhoneNumberType;
                  ETG_TRACE_USR2(( " oSearchPhoneNumber6type = %d", items.oPhoneNumber6.e8Type.enType ));
               }
               break;

               case 6:
               {
                  items.oPhoneNumber7.sNumber.bSet(qbPhoneNo.constData());
                  items.oPhoneNumber7.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)oSearchPhNoItem.enPhoneNumberType;
                  ETG_TRACE_USR2(( " oSearchPhoneNumber7type = %d", items.oPhoneNumber7.e8Type.enType ));
               }
               break;

               case 7:
               {
                  items.oPhoneNumber8.sNumber.bSet(qbPhoneNo.constData());
                  items.oPhoneNumber8.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)oSearchPhNoItem.enPhoneNumberType;
                  ETG_TRACE_USR2(( " oSearchPhoneNumber8type = %d", items.oPhoneNumber8.e8Type.enType ));
               }
               break;

               default:
               ETG_TRACE_USR2(( " oSearchPhoneNumbertype : Default value"));
            }
         }
      }

      // Remapping of EmailAddress
      QList<QString> qSearchEmailAddrList;

      if(SearchResultList.at(u16Index).EmailAdd1 != "")
      {
         qSearchEmailAddrList << SearchResultList.at(u16Index).EmailAdd1;
      }

      if(SearchResultList.at(u16Index).EmailAdd2 != "")
      {
         qSearchEmailAddrList << SearchResultList.at(u16Index).EmailAdd2;
      }

      if(SearchResultList.at(u16Index).EmailAdd3 != "")
      {
         qSearchEmailAddrList << SearchResultList.at(u16Index).EmailAdd3;
      }

      for(tU8 u8EmailCount = 0; u8EmailCount< SearchResultList.at(u16Index).u8EmailAddrCount; u8EmailCount++)
      {
         QByteArray qbEmailaddr = qSearchEmailAddrList.at(u8EmailCount).toUtf8();

         ETG_TRACE_USR2(( "oEmailAddr%d = %s", u8EmailCount, qbEmailaddr.constData() ));
         switch (u8EmailCount)
         {
            case 0:
            {
               items.oEmailAddr1.bSet(qbEmailaddr.constData());
            }
            break;
            case 1:
            {
               items.oEmailAddr2.bSet(qbEmailaddr.constData());
            }
            break;
            case 2:
            {
               items.oEmailAddr3.bSet(qbEmailaddr.constData());
            }
            break;
            default:
            ETG_TRACE_USR2(( " oEmailAddr%d : unknown value", u8EmailCount));
         }
      }


      methodResult->oContactMatchList.oItems.push_back(items);
   }

   return TRUE;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_List::Convert()
*
* DESCRIPTION:
*              Converts the Qt format structure into the MOST format structure
*
*
* PARAMETER: [IN] contact :- The Qt format contact info
*            [OUT] item :- MOST format contact info
* RETURNVALUE: None.
*
*******************************************************************************/
void FC_PhoneBook_List::Convert( const FC_PhoneBook_Contact_Detail_Short* contact,
                                 most_fi_tcl_PhonBkPhoneBookListSliceResultItem* item )
{
   item->u32ContactHandle = contact->contactHandle;

   // Set the quick options
   item->bQuickDial  = contact->bQuickDial;
   item->bQuickRoute = contact->bQuickRoute;
   item->bQuickText  = contact->bQuickText;

   QByteArray ba = contact->firstName.toUtf8();
   item->sFirstName.bSet( ba.constData() );

   ETG_TRACE_USR2(( " FIRST NAME = %s", ba.constData() ));

   ba = contact->lastName.toUtf8();
   item->sLastName.bSet( ba.constData() );

   ETG_TRACE_USR2(( " LAST NAME = %s", ba.constData() ));

   ba = contact->poiName.toUtf8();
   item->sPOIName.bSet( ba.constData() );

   item->u32ContactNameTag = GENERATE_TAG(contact->contactHandle, CONTACTNAME_TAG);
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_List::getSortType()
*
* DESCRIPTION:
*
*
*
* PARAMETER:
*
* RETURNVALUE: None.
*
*******************************************************************************/
FC_PhoneBook_SQLite::SortType FC_PhoneBook_List::getSortType( const most_fi_tcl_e8_PhonBkPhoneBookSortType* type )
{
   tU8 val = type->enType;
   FC_PhoneBook_SQLite::SortType eRetValue;
   switch ( val )
   {
   default:
   case most_fi_tcl_e8_PhonBkPhoneBookSortType::FI_EN_E8PB_LIST_SORT_LASTNAME:

      ETG_TRACE_USR4(("SortType == LastName"));
      eRetValue = FC_PhoneBook_SQLite::LastName;
      break;
   case most_fi_tcl_e8_PhonBkPhoneBookSortType::FI_EN_E8PB_LIST_SORT_FIRSTNAME:

      ETG_TRACE_USR4(("SortType == FirstName"));
      eRetValue = FC_PhoneBook_SQLite::FirstName;
      break;
   case most_fi_tcl_e8_PhonBkPhoneBookSortType::FI_EN_E8PB_LIST_SORT_ADDRESS_LASTNAME:

      ETG_TRACE_USR4(("SortType == Address_LastName"));
      eRetValue = FC_PhoneBook_SQLite::Address_LastName;
      break;
   case most_fi_tcl_e8_PhonBkPhoneBookSortType::FI_EN_E8PB_LIST_SORT_ADDRESS_FIRSTNAME:

      ETG_TRACE_USR4(("SortType == Address_FirstName"));
      eRetValue = FC_PhoneBook_SQLite::Address_FirstName;
      break;
   case most_fi_tcl_e8_PhonBkPhoneBookSortType::FI_EN_E8PB_LIST_SORT_NUMBER_LASTNAME:

      ETG_TRACE_USR4(("SortType == Number_LastName"));
      eRetValue = FC_PhoneBook_SQLite::Number_LastName;
      break;
   case most_fi_tcl_e8_PhonBkPhoneBookSortType::FI_EN_E8PB_LIST_SORT_NUMBER_FIRSTNAME:

      ETG_TRACE_USR4(("SortType == Number_FirstName"));
      eRetValue = FC_PhoneBook_SQLite::Number_FirstName;
      break;
   case most_fi_tcl_e8_PhonBkPhoneBookSortType::FI_EN_E8PB_LIST_SORT_EMAIL_LASTNAME:

      ETG_TRACE_USR4(("SortType == Email_LastName"));
      eRetValue = FC_PhoneBook_SQLite::Email_LastName;
      break;
   case most_fi_tcl_e8_PhonBkPhoneBookSortType::FI_EN_E8PB_LIST_SORT_EMAIL_FIRSTNAME:

      ETG_TRACE_USR4(("SortType == Email_FirstName"));
      eRetValue = FC_PhoneBook_SQLite::Email_FirstName;
      break;
   }
   return eRetValue;
}

/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_List::getPhoneBkContactSortType()
*
* DESCRIPTION:
*
*
*
* PARAMETER:
*
* RETURNVALUE: None.
*
*******************************************************************************/
FC_PhoneBook_SQLite::SortType FC_PhoneBook_List::getSortType( const most_fi_tcl_e8_PhonBkContactSortType* type )
{
   tU8 val = type->enType;

   switch (val)
   {
      case most_fi_tcl_e8_PhonBkContactSortType::FI_EN_E8PB_LIST_SORT_LASTNAME:

         ETG_TRACE_USR4(("SortType == LastName"));
         return FC_PhoneBook_SQLite::LastName;

      case most_fi_tcl_e8_PhonBkContactSortType::FI_EN_E8PB_LIST_SORT_FIRSTNAME:

         ETG_TRACE_USR4(("SortType == FirstName"));
         return FC_PhoneBook_SQLite::FirstName;

      case most_fi_tcl_e8_PhonBkContactSortType::FI_EN_E8PB_LIST_SORT_ADDRESS_LASTNAME:

         ETG_TRACE_USR4(("SortType == Address_LastName"));
         return FC_PhoneBook_SQLite::Address_LastName;

      case most_fi_tcl_e8_PhonBkContactSortType::FI_EN_E8PB_LIST_SORT_ADDRESS_FIRSTNAME:

         ETG_TRACE_USR4(("SortType == Address_FirstName"));
         return FC_PhoneBook_SQLite::Address_FirstName;

      case most_fi_tcl_e8_PhonBkContactSortType::FI_EN_E8PB_LIST_SORT_NUMBER_LASTNAME:

         ETG_TRACE_USR4(("SortType == Number_LastName"));
         return FC_PhoneBook_SQLite::Number_LastName;

      case most_fi_tcl_e8_PhonBkContactSortType::FI_EN_E8PB_LIST_SORT_NUMBER_FIRSTNAME:

         ETG_TRACE_USR4(("SortType == Number_FirstName"));
         return FC_PhoneBook_SQLite::Number_FirstName;

      case most_fi_tcl_e8_PhonBkContactSortType::FI_EN_E8PB_LIST_SORT_EMAIL_LASTNAME:

         ETG_TRACE_USR4(("SortType == Email_LastName"));
         return FC_PhoneBook_SQLite::Email_LastName;

      case most_fi_tcl_e8_PhonBkContactSortType::FI_EN_E8PB_LIST_SORT_EMAIL_FIRSTNAME:

         ETG_TRACE_USR4(("SortType == Email_FirstName"));
         return FC_PhoneBook_SQLite::Email_FirstName;

     case most_fi_tcl_e8_PhonBkContactSortType::FI_EN_E8PB_LIST_SORT_CATEGORY_LASTNAME:

       ETG_TRACE_USR4(("SortType == LastName for Category_LastName"));
         return FC_PhoneBook_SQLite::LastName;

     case most_fi_tcl_e8_PhonBkContactSortType::FI_EN_E8PB_LIST_SORT_CATEGORY_FIRSTNAME:

         ETG_TRACE_USR4(("SortType == FirstName for Category_FirstName "));
         return FC_PhoneBook_SQLite::FirstName;

      case most_fi_tcl_e8_PhonBkContactSortType::FI_EN_E8PB_LIST_SORT_POSITION:

         ETG_TRACE_USR4(("SortType == Position"));
         return FC_PhoneBook_SQLite::Position;

     default:

       ETG_TRACE_ERR(("SortType Invalid!!"));
         return FC_PhoneBook_SQLite::Invalid;
   }

}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_List::getSortType()
*
* DESCRIPTION: Overloaded getSortType() function for getting the sort type for SearchContacts()
*
*
*
* PARAMETER:
*
* RETURNVALUE: None.
*
*******************************************************************************/
FC_PhoneBook_SQLite::SortType FC_PhoneBook_List::getSortType( const most_fi_tcl_e8_PhonBkSearchSortType* type )
{
   tU8 val = type->enType;

   switch ( val )
   {
   default:
   case most_fi_tcl_e8_PhonBkSearchSortType::FI_EN_E8PB_SEARCH_SORT_LASTNAME:

      ETG_TRACE_USR4(("SearchSortType == LastName"));
      return FC_PhoneBook_SQLite::LastName;
   case most_fi_tcl_e8_PhonBkSearchSortType::FI_EN_E8PB_SEARCH_SORT_FIRSTNAME:

      ETG_TRACE_USR4(("SearchSortType == FirstName"));
      return FC_PhoneBook_SQLite::FirstName;
   case most_fi_tcl_e8_PhonBkSearchSortType::FI_EN_E8PB_SEARCH_SORT_ADDRESS_LASTNAME:

      ETG_TRACE_USR4(("SearchSortType == Address_LastName"));
      return FC_PhoneBook_SQLite::Address_LastName;
   case most_fi_tcl_e8_PhonBkSearchSortType::FI_EN_E8PB_SEARCH_SORT_ADDRESS_FIRSTNAME:

      ETG_TRACE_USR4(("SearchSortType == Address_LastName"));
      return FC_PhoneBook_SQLite::Address_FirstName;
   case most_fi_tcl_e8_PhonBkSearchSortType::FI_EN_E8PB_SEARCH_SORT_NUMBER_LASTNAME:

      ETG_TRACE_USR4(("SearchSortType == Number_LastName"));
      return FC_PhoneBook_SQLite::Number_LastName;
   case most_fi_tcl_e8_PhonBkSearchSortType::FI_EN_E8PB_SEARCH_SORT_NUMBER_FIRSTNAME:

      ETG_TRACE_USR4(("SearchSortType == Number_FirstName"));
      return FC_PhoneBook_SQLite::Number_FirstName;
   case most_fi_tcl_e8_PhonBkSearchSortType::FI_EN_E8PB_SEARCH_SORT_EMAIL_LASTNAME:

      ETG_TRACE_USR4(("SearchSortType == Email_LastName"));
      return FC_PhoneBook_SQLite::Email_LastName;
   case most_fi_tcl_e8_PhonBkSearchSortType::FI_EN_E8PB_SEARCH_SORT_EMAIL_FIRSTNAME:

      ETG_TRACE_USR4(("SearchSortType == Email_FirstName"));
      return FC_PhoneBook_SQLite::Email_FirstName;
   }
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_List::
*
* DESCRIPTION:
*
*
* PARAMETER: None.
*
* RETURNVALUE: None
*
*******************************************************************************/
quint32 FC_PhoneBook_List::GetListSize(QList<FC_PhoneBook_Contact_Detail>& PBContactList)
{
   ETG_TRACE_USR4(( " GetListSize entered" ));
   quint32 MemorySizeInByte = 0;
   quint16 u16ListSize= (quint16)(PBContactList.size());
   for(quint16 u16Index= 0; u16Index < u16ListSize; u16Index++)
   {
      FC_PhoneBook_Contact_Detail ContactDetail= PBContactList.at(u16Index);
      MemorySizeInByte += ContactDetail.GetContactLength();
   }

   return MemorySizeInByte;
}


bool FC_PhoneBook_List::bCheckForContactMemory(FC_PhoneBook_MemoryUsage* poPbMemUsage, quint32 u32ContactMemoryToBeAdded)
{
   ETG_TRACE_USR4(( " bCheckForContactMemory entered" ));

   bool bMemoryAvailable = false;

   // Check if we have enough Contact (APL1) memory. If not we try to drop older phonebooks
   bool bContactsRemoved = true;
   while( (false == bMemoryAvailable) && (true == bContactsRemoved) )
   {
      bMemoryAvailable = poPbMemUsage->bCheckMemoryPoolForContacts(u32ContactMemoryToBeAdded);
      if(false == bMemoryAvailable)
      {
         // The memory limit is exceeded. We will drop a complete contact set (inclusive photos) of a phone,
         // which has not been connected for some time, i.e. (LRU Least Recent Used) phone device.
         bContactsRemoved = poPbMemUsage->bRemoveLeastRecentlyUsedPhonebook();
      }
   }

   return bMemoryAvailable;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::AddNewVCardData
*
* DESCRIPTION:
*              Add new data to ByteArray and start parsing if completed is true.
*
*
* PARAMETER:
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_List::AddNewVCardData(GByteArray* data, int phonebookType, unsigned char deviceHandle)
{

   ETG_TRACE_USR4(( " AddNewVCardData entered " ));

   QTime DebugTime;
   DebugTime.start();

   QByteArray ba((const char*) (data->data), (int)data->len);

   switch (phonebookType)
   {
   case DownLoadPBSim:
   case DownLoadPBLocal:
   case DownloadPBSpeedDial:
   case DownloadPBFavorites:
      {
         QList <FC_PhoneBook_Contact_Detail> list;

         if (true == ParseVCards(ba, phonebookType, deviceHandle, list))
         {
            if (!list.isEmpty())
            {
               FC_Device_Details *DevDetails = FC_Device_Details::deviceHandleToObjectInstance(deviceHandle);
               if (DevDetails)
               {
#ifdef TWO_PASS_DOWNLOAD
                  if (DevDetails->blIsItFirstTimeDownload() || DownLoadPBSim == phonebookType) //For SIM contacts, comparison would have already performed during parsing.
                  {
   #ifdef SUPERVISION_IN_FIRST_PASS
                     ETG_TRACE_USR4(("Inserting contacts after comparison in first pass"));
                     CompareAndInsertIntoDatabase(list, deviceHandle);
   #else
                     ETG_TRACE_USR4(("Inserting contacts without comparison"));
                     InsertContactsIntoDatabase(list, deviceHandle);
   #endif
                  }
                  else
                  {
#endif
                     ETG_TRACE_USR4(("Inserting contacts after comparison"));
                     CompareAndInsertIntoDatabase(list, deviceHandle);
#ifdef TWO_PASS_DOWNLOAD
                  }
#endif
               }
               else
               {
                  ETG_TRACE_ERR(("Device Object is NULL"));
               }
            }
            else
            {
               ETG_TRACE_USR4(("empty list"));
            }
         }
         else
         {
            ETG_TRACE_USR4(("no documents "));
         }
      }
      break;

   default:
      {
         ETG_TRACE_ERR(("phonebookType= %d is not a valid download type ", phonebookType));
      }
   }

   ETG_TRACE_USR2(( " -PBDL.Perf- 'AddNewVCardData' exited after %d ms ", DebugTime.elapsed() ));
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::ParseVCards
*
* DESCRIPTION:
*              Parses the vcard stream.
*
*
* PARAMETER:
* RETURNVALUE:
*
*******************************************************************************/
bool FC_PhoneBook_List::ParseVCards(QByteArray ba, int phonebookType, unsigned char deviceHandle,QList<FC_PhoneBook_Contact_Detail> & list)
{
   ETG_TRACE_USR4(( " ParseVCards entered " ));

   QTime ParseVCardsTime;
   ParseVCardsTime.start();

   FC_PhoneBook_VCard_Parser parser;

   QList<FC_PhoneBook_VCard_Document> docs = parser.parseData(ba);

   int docsSize = docs.size();

   if(0 == docsSize)
   {
      ETG_TRACE_ERR(( "ERROR: doc list empty on parsing vcards" ));
      return false;
   }

   for ( int docCounter = 0; docCounter < docsSize; ++docCounter )
   {
      noofphonenos = 0;
      const FC_PhoneBook_VCard_Document* doc;
      doc = &docs.at( docCounter );
      if ( !doc->isEmpty() ||
         doc->subDocuments().empty()) //As per the requirement, nested Vcards should be omited. GIS 327
      {
         QList<FC_PhoneBook_VCard_Property> props = doc->properties();
         int propSize = props.size();
         FC_PhoneBook_Contact_Detail contact;

         if (DownloadPBSpeedDial == phonebookType || DownloadPBFavorites
               == phonebookType)
         {
            contact.contactType
                  = (most_fi_tcl_e8_PhonBkContactType::tenType) CONTACT_TYPE_FAVORITE;
         }
         else
         {
            contact.contactType
                  = (most_fi_tcl_e8_PhonBkContactType::tenType) CONTACT_TYPE_STANDARD;
         }
         ETG_TRACE_USR4((" ContactType = %d", (most_fi_tcl_e8_PhonBkContactType::tenType) contact.contactType));

         for ( int propCounter = 0; propCounter < propSize; propCounter++ )
         {
            AddPropertyToPhoneBookContact( &(props.at( propCounter )), &contact, phonebookType );
         }

         contact.contactHandle = 0; // Contact Handle has to be generated by database, so initialize it with zero

         ETG_TRACE_USR2(("firstName = %s",contact.firstName.toUtf8().constData()));
         ETG_TRACE_USR2(("lastName = %s",contact.lastName.toUtf8().constData()));

         //CRQ-CMG3G-2888
         //Fix for CMG3G-2971:Enhanced vcard handling@FC_Phonebook
         bool bIsContactValid = false;
#ifdef VALID_VCARD_FORMAT_ADDRESS_PHONENUMBER
         if (  true == contact.isValidVcardFormatPhoneNumberAddress( contact))
         {
            bIsContactValid = true;

         }
#else
#ifdef VALID_VCARD_FORMAT_PHONENUMBER
         if ( true == contact.isValid( contact))
         {
            bIsContactValid = true;
         }

#else
         bIsContactValid = true;
#endif
#endif

         if( TRUE == bIsContactValid )
         {
            // -FastContactSearch- : -BGN: Calculate the CRC hash from a Contact
            //   Here we calculate a CRC hash on "relevant" (APL1 properties) of the received contact
            //   and add this CrcAPL1 value to the contact structure (FC_PhoneBook_Contact_Detail).
            //   This CRC later is used to search for contacts in the database which have same CRC and
            //   therefore are potentially "double" entries.
            //   We use as "relevant" contact properties those ones, which are currently used anyway to
            //   identify a "matching contact" in method 'FC_PhoneBook_SQLite::FindExactMatchingContact(..)'.

            // Calculate a CRC value from the "relevant" APL1 properties of a contact
            // - First the "relevant" APL1 properties are concatenated - separated by unique tag strings -
            //   to one long string, with the intention to calculate a CRC value from the data of this
            //   long string.
            // - The separators inserted during the concatenation are needed to detect a movement of e.g.
            //   a dedicated phone number from one field (e.g. cellNum1) to another field (e.g. homeNum2).
            // - The idea is that, due to the separators, such a "movement" results in a different CRC value
            //   and therefore will be detected as different contact.

            // Create "long" string with relevant APL1 attributes and related separators.
            QString qsContactDetails =
                 "FN" + contact.firstName
               + "LN" + contact.lastName
               + "Ph" + contact.Phonetics   // Hotfix for CRQ 509734 Phonetic Sorting
               + "SF" + contact.SortFirstName
               + "SL" + contact.SortLastName
               + "PN" + contact.poiName
               + "P1" + contact.PhoneNum1
               + "P2" + contact.PhoneNum2
               + "P3" + contact.PhoneNum3
               + "P4" + contact.PhoneNum4
               + "P5" + contact.PhoneNum5
               + "P6" + contact.PhoneNum6
               + "P7" + contact.PhoneNum7
               + "P8" + contact.PhoneNum8
               + "T1" + QString::number(contact.NumberType1)
               + "T2" + QString::number(contact.NumberType2)
               + "T3" + QString::number(contact.NumberType3)
               + "T4" + QString::number(contact.NumberType4)
               + "T5" + QString::number(contact.NumberType5)
               + "T6" + QString::number(contact.NumberType6)
               + "T7" + QString::number(contact.NumberType7)
               + "T8" + QString::number(contact.NumberType8)
               + "E1" + contact.emailAdd1
               + "E2" + contact.emailAdd2
               + "E3" + contact.emailAdd3
               + "HA" + contact.homeAdd
               + "WA" + contact.workAdd
               + "AA" + contact.otherAdd;

            // Calculate the CRC and assign the value to the (new) contact attribute 'contact.crcAPL1'.
            QByteArray qBA   = QByteArray( qsContactDetails.toUtf8().constData() );
            contact.APL1Size = (quint32)qBA.size();
            contact.crcAPL1  = qChecksum( qBA.constData(), (quint16)qBA.length() );  // -FastContactSearch- : TODO: Think about a
                                                                           // Qt independent implementation of the CRC calculation.
            // -FastContactSearch- : -END: Calculate the CRC hash from a Contact

            // For SIM contacts we will do a check, if the contact is already in the DB here, whereas for ME contacts this step is done later.
            // This "logic" is somehow historical, but we keep it for the moment as it is working. (Maybe it is worth to think about this
            // detail later again. Maybe there is further room for improvement or alignment, e.g. by doing the same for ME contacts here also).
            if (phonebookType == DownLoadPBSim)
            {
               quint32 u32ContactHandle = 0;
               // Create "dummy variables" ('bPhotoStatus', 'u32photoSizeOld') which is needed for calling "FindExactMatchingContact(..)",
               // but it will not be evaluated afterwards as SIM contacts have no photo anyway.
               bool bPhotoStatus = false;
               quint32 u32photoSizeOld = 0;

               // Check if the same contact already exists in the database.
               if( m_sqlite->FindExactMatchingContact(contact, deviceHandle,
                                                      u32ContactHandle, bPhotoStatus, u32photoSizeOld) )
               {
                  ETG_TRACE_USR2(( " FOUND THE SAME CONTACT IN DATABASE " ));
                  // Don't copy this contact to list

                  // Remove contact handle from 'Invalid Contact Handle List'
                  FC_Device_Details* DeviceDetails =
                     FC_Device_Details::deviceHandleToObjectInstance(deviceHandle);
                  if(DeviceDetails)
                  {
                     DeviceDetails->InvalidContactHandleList.removeOne(u32ContactHandle);
                  }
                  else
                  {
                     ETG_TRACE_ERR(("Device Object is NULL"));
                  }
               }
               else
               {
                  // Contact has not been found in database, therefore it is added to the 'list' (for insertion to the database)
                  ETG_TRACE_USR2(( " CONTACT NOT FOUND IN DATABASE " ));
                  list << contact;
               }
            }
            else
            {
               list << contact;
            }

            // CMG3G-7983 - Indicate the Phonebook Download Progress
         }
      }
   }

   ETG_TRACE_USR2(( " -PBDL.Perf- 'ParseVCards' exited after %d ms ", ParseVCardsTime.elapsed() ));
   return true;
}
//Fix for CMG3G-2971


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::AddPropertyToPhoneBookContact
*
* DESCRIPTION:
*              Parses the vcard stream.
*
*
* PARAMETER:
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_List::AddPropertyToPhoneBookContact(
   const FC_PhoneBook_VCard_Property* prop,
   FC_PhoneBook_Contact_Detail* contact,
   const int& phonebookType)
{
   ETG_TRACE_USR4(( " AddPropertyToPhoneBookContact entered" ));
   (tVoid)phonebookType;
   QString text;

   FC_PhoneBook_VCard_Property::Type propType;

   QMultiHash <QString,QString> full_m_Parameters = prop->parameters();


   propType = prop->getType();

   QString name = prop->name();
   QChar space(' ');

   if ( name == "FN" )
   {
      if ( propType == FC_PhoneBook_VCard_Property::List || propType == FC_PhoneBook_VCard_Property::Compound || propType == FC_PhoneBook_VCard_Property::Binary )
      {
         text = prop->value<QStringList>().join(" ").simplified();
      }
      else
      {
         text = prop->value().simplified();
      }

      if ( (!text.isEmpty()) && ((contact->firstName.isEmpty()) && (contact->lastName.isEmpty())) )
      {
#ifdef LIMIT_ATTRIBUTE_LENGTH //FIX NCG3D-11656 Limiting of characters not required for AIVI
         contact->lastName =  text.left(25);

         //FIX SUZUKI-18215 strange "?" showed in Phone screen for a long Chinese contact
         //The replacement character is found if any and then string is truncated at the replacement character, to make it a valid utf8 string.
         FC_PhoneBook_CallHistory_List::GetCurrentInstance()->vUtf8Validate(contact->lastName);
         //End of fix SUZUKI-18215
#else
         contact->lastName =  text;
#endif
      }

      ETG_TRACE_USR2(("lastName =%s",contact->lastName.toUtf8().constData()));

   }
   if ( name == "N" )
   {
      contact->lastName  = "";   //.clear();
      contact->firstName = "";   //.clear();
      
      ETG_TRACE_USR4(( " N: propType= %d", propType ));   // Maybe comment out later. I just wanted to understand which value the propType can have
     
      if ( propType == FC_PhoneBook_VCard_Property::List || propType == FC_PhoneBook_VCard_Property::Compound || propType == FC_PhoneBook_VCard_Property::Binary )
      {
         QStringList NameValue= prop->value<QStringList>();

         ETG_TRACE_USR4(("Vcard property N: is a string list. Values are: "));
         for(qint16 listIndex= 0; (listIndex< NameValue.size()&&
            (listIndex<= (qint16)FC_PHONEBOOK_PROP_N_FIRSTNAME_INDEX)); listIndex++)
         {

            ETG_TRACE_USR4(("NameValue[%d]- %s", listIndex, NameValue.at(listIndex).toUtf8().constData()));
            switch(listIndex)
            {
            case FC_PHONEBOOK_PROP_N_LASTNAME_INDEX:
#ifdef LIMIT_ATTRIBUTE_LENGTH //FIX NCG3D-11656 Limiting of characters not required for AIVI
               contact->lastName= NameValue.at(listIndex).left(25);

               //FIX SUZUKI-18215 strange "?" showed in Phone screen for a long Chinese contact
               //The replacement character is found if any and then string is truncated at the replacement character, to make it a valid Utf8 string.
               FC_PhoneBook_CallHistory_List::GetCurrentInstance()->vUtf8Validate(contact->lastName);
               //End of fix SUZUKI-18215
#else
               contact->lastName= NameValue.at(listIndex);
#endif
               break;
            case FC_PHONEBOOK_PROP_N_FIRSTNAME_INDEX:
#ifdef LIMIT_ATTRIBUTE_LENGTH //FIX NCG3D-11656 Limiting of characters not required for AIVI
               contact->firstName= NameValue.at(listIndex).left(25);

               //FIX SUZUKI-18215 strange "?" showed in Phone screen for a long Chinese contact
               //The replacement character is found if any and then string is truncated at the replacement character, to make it a valid Utf8 string.
               FC_PhoneBook_CallHistory_List::GetCurrentInstance()->vUtf8Validate(contact->firstName);
               //End of fix SUZUKI-18215
#else
               contact->firstName= NameValue.at(listIndex);
#endif
               break;
            default:
               {
                  ETG_TRACE_USR4(("Not Valid listIndex "));
               }
            }
         }
      }
      else
      {
         contact->lastName= prop->value().simplified();

         ETG_TRACE_ERR(("Vcard property N: is not a string list- %s", contact->lastName.toLatin1().constData()));
      }

      ETG_TRACE_USR2(("firstName =%s",contact->firstName.toUtf8().constData()));
      ETG_TRACE_USR2(("lastName =%s",contact->lastName.toUtf8().constData()));

   }
   else if ( name == "PHOTO" )
   {

      ETG_TRACE_USR4(("Parsing Photo Property"));

      QString numType =  prop->parameters().value("TYPE");

      if(numType == "")// If Type filed is missing in the Vcard.
      {
         numType = "JPEG"; // Take JPEG as default
      }

      if( (numType == "JPEG") ||
         (numType == "JPG")  ||
         (numType == "PNG")  ||
         (numType == "GIF")  ||
         (numType == "image/jpeg"   )  )
      {
         QByteArray ba= prop->variantValue().toByteArray();

#ifdef FC_PB_TEST_PHOTO_DUMP
         static int flag= 0;
         flag++;
         QString RawDataFile("/opt/bosch/RawImage");
         quint32 u32Len;

         RawDataFile  += QString::number(flag) + "." + numType;


         QFile file(RawDataFile);
         if(file.open(QIODevice::WriteOnly))
         {
            u32Len= file.write(ba, ba.size());
            file.close();
         }
         else
         {
            ETG_TRACE_USR4(("Failed to Open file for Raw Data"));
         }
#endif
         quint32 u32Size      = (quint32)( (ba.size() > 0) ? (ba.size()) : 0 );  // Not sure if this is needed, but it should not harm anyway

         contact->photoType   = numType;
         contact->photo       = ba;
         contact->photoDlSize = u32Size;
         contact->photoDlCRC  = qChecksum( ba.constData(), u32Size );
         contact->photoSize   = u32Size;
         contact->photoCRC    = contact->photoDlCRC;
      }
      else
      {
         ETG_TRACE_USR4((" Image type- %s not supported", numType.toLatin1().constData()));
      }
   }
   else if ( name == "ADR" )
   {
      QStringList list1;
      QString AddType =  prop->parameters().value("TYPE");
      ETG_TRACE_USR2(("Address filteration AddType == %s",AddType.toLatin1().constData()));
      if ( propType == FC_PhoneBook_VCard_Property::List || propType == FC_PhoneBook_VCard_Property::Compound)
      {
         QStringList list =  prop->value<QStringList>();

         // -bn: BEGIN, Change for Navigation: Jira Task GMMY15-5668; Sub-Task for FC_Phonebook: GMMY15-5669
         //     Rationale: - The resulting address string shall have 5 well defined sub elements, separated by comma,
         //                  which allow Navigation to make use of this structure for easy Route Guidance (RG).
         //                - The resulting address string shall be an acceptable compromise for a "pretty print"
         //                  representation of the address in HMI list entries.
         //     Details:
         //     - The 'Extended Address' and 'Post Office Address' and the 'Street' field will be joined to one
         //       new field: 'Street' + " " + 'Extended Address' + " " + 'Post Office Address'.
         //     - The other 'ADR' fields will be added then, separated by 'commas'. This "maintains" an inner
         //       structure within the new created address (needed by Navigation for efficient RG).
         //       -- 'Commas' which are in the original 'ADR' fields will be replaces by 'Spaces'.
         //       -- 'Spaces' at string start and end will be removed. Multiple consecutive 'Spaces' within the string
         //          will be reduced to one Space.

         // Create "Street" info from vCard fields: 'Street' [2] + 'Extended Address' [1] + 'Post Office Adddress' [0]
         int sTmpMaxIdx = 2;    // list[2] contains 'Street'
         if(list.size() < (sTmpMaxIdx +1))
         {
            // List size is smaller than 3, which means that the 'ADR' attribute in the vCard has not all mandatory 7 fields
            sTmpMaxIdx = list.size() -1;
         }
         QString qsStreet = "";
         for(int j = sTmpMaxIdx; j >= 1; j--)  // 01/2019: NCG3D-117515: Changed 'j >= 0' to 'j >= 1', i.e. removal of 'Post Office Adddress' [0]
         {
            qsStreet += " " + list[j];
         }

         // Remove commas in 'qsStreet' and replace them by spaces. Store the result in list1[0].
         list1.append( qsStreet.replace(QString(","),QString(" ")).simplified() );
         //  Note: The operation '.replace(QString(","),QString(" "))' can be applied on empty strings also.

         // Append the other 'ADR' fields to list1[.]
         for(int i = 3; i < list.size(); i++)  // 09/2019: A-IVI Bug 599530 [related to NCG3D-117515]: Corrected start value ('City' [3])
         {
            // Remove commas from ADR fields, replace them by spaces and append 'ADR' field to list.
            list1.append( list[i].replace(QString(","),QString(" ")).simplified() );
         }

         // Add "empty" strings to list1[.], if the 'ADR' property has less than 7 fields
         for(int i = list.size(); i < 7; i++)
         {
            list1.append( QString("") );
         }
         // -bn: END, Change for Navigation

         text = list1.join(",").simplified();  // The 'simplified' operation removes 'Spaces' at start and end of string,
         // and reduces "multiple" 'Spaces' within the string to single ones.

         ETG_TRACE_USR2(("Address filteration after join text == %s",text.toLatin1().constData()));
      }
      else
      {
         text = prop->value().simplified();
         ETG_TRACE_USR2(("Address retrieved: %s",text.toLatin1().constData()));
      }

      if ( AddType == "HOME" )
      {
         if (contact->homeAdd.isEmpty())
         {
            contact->homeAdd = text;
            ETG_TRACE_USR2(("contact->homeAdd == %s", contact->homeAdd.toLatin1().constData()));
         }
         else
         {
            ETG_TRACE_USR2((" contact->homeAdd already filled with: %s", contact->homeAdd.toLatin1().constData()));
         }
      }
      else if ( AddType == "WORK" )
      {
         if (contact->workAdd.isEmpty())
         {
            contact->workAdd = text;
            ETG_TRACE_USR2(("contact->workAdd == %s", contact->workAdd.toLatin1().constData()));
         }
         else
         {
            ETG_TRACE_USR2((" contact->workAdd already filled with: %s", contact->workAdd.toLatin1().constData()));
         }
      }
      else
      {
         if (contact->otherAdd.isEmpty())
         {
            contact->otherAdd = text;
            ETG_TRACE_USR2(("contact->otherAdd == %s", contact->otherAdd.toLatin1().constData()));
         }
         else
         {
            ETG_TRACE_USR2((" contact->otherAdd already filled with: %s", contact->otherAdd.toLatin1().constData()));
         }
      }
   }
   else if ( name == "TEL" )  // -bn: GMMY15-2277:
      //"TEL clause" significanlty reworked to fix the issue.
   {
      // -bn: Note: ETG traces are commented out in the following code part by intention,
      // as the implicite called function "bIsTraceActive(.)" (in each ETG trace)
      // would need computation time in the order of the implementation itself
      // (as nearly "every" code line is acompanied by a trace statement).
      // These traces should be seen as 'comment' or can be re-activated in a local
      // build for debugging purposes.

      text = prop->value().simplified();
      //ETG_TRACE_USR4(( " vCard-Prop TEL phone number: '%s'", text.toLatin1().constData() ));

      QString qsPhoneNumber;
#ifdef LIMIT_ATTRIBUTE_LENGTH //FIX NCG3D-11656 Limiting of characters not required for AIVI
      qsPhoneNumber = text.left(25);
#else
      qsPhoneNumber = text;
#endif
      //ETG_TRACE_USR4(( " vCard-Prop TEL phone number (reduced to 25 character): '%s'", qsPhoneNumber.toLatin1().constData() ));

      /* using qhash value for key TYPE */
      QList<QString> typeList = prop->parameters().values("TYPE");

      QStringList typeList2 = prop->parameters().values("TYPE");
      QString text2 = typeList2.join(";").simplified();
      //ETG_TRACE_USR4(( " vCard-Prop TEL parameters (';' separated): '%s'", text2.toLatin1().constData() ));

#ifdef PBDL_VCARD_PARSE_NUMBERS_STYLE_FIRST_N_WITH_INDIVIDUAL_TYPE

      quint8 u8numbertype;

      ETG_TRACE_USR4(( " Parsing the Vcards with the new style entered" ));
      if(noofphonenos <PBDL_MAX_PHONE_NUMBERS_PER_CONTACT_COUNT) // PBDL_MAX_PHONE_NUMBERS_PER_CONTACT_COUNT 5 for IVI , 4 for PSA and 8 for rest all projects
      {
         ETG_TRACE_USR4(( " noofphonenos are parsing %d",noofphonenos ));
         if(typeList.contains("PREF"))
         {
            u8numbertype = PHONE_NUMBER_PREFERRED;
            ETG_TRACE_USR4(( " Parsing the Vcards with type PHONE_NUMBER_PREFERRED %d",u8numbertype ));
            ExtractingOfFirstNPhoneNumbers(u8numbertype,contact,qsPhoneNumber);
         }

         if(typeList.contains("CELL"))
         {
            u8numbertype = PHONE_NUMBER_CELL;
            ETG_TRACE_USR4(( " Parsing the Vcards with type PHONE_NUMBER_CELL %d",u8numbertype ));
            ExtractingOfFirstNPhoneNumbers(u8numbertype,contact,qsPhoneNumber);
         }
         else if(typeList.contains("HOME"))
         {
            u8numbertype = PHONE_NUMBER_HOME;
            ETG_TRACE_USR4(( " Parsing the Vcards with type PHONE_NUMBER_HOME %d",u8numbertype ));
            ExtractingOfFirstNPhoneNumbers(u8numbertype,contact,qsPhoneNumber);

         }
         else if(typeList.contains("WORK"))
         {
            u8numbertype = PHONE_NUMBER_WORK;
            ETG_TRACE_USR4(( " Parsing the Vcards with type PHONE_NUMBER_WORK %d",u8numbertype ));
            ExtractingOfFirstNPhoneNumbers(u8numbertype,contact,qsPhoneNumber);
         }
         else // "OTHER" type
         {
            u8numbertype = PHONE_NUMBER_OTHER;
            ETG_TRACE_USR4(( " Parsing the Vcards with type PHONE_NUMBER_OTHER %d",u8numbertype ));
            ExtractingOfFirstNPhoneNumbers(u8numbertype,contact,qsPhoneNumber);
         } // end of: "OTHER" type
      }
      noofphonenos++;

#else

      // 'bPREFNumberSet' will be set to TRUE, if the Pref Number of a contact is set for this property (TEL number).
      // This boolean will be used, when it has to be decided if the 'Other' number has to be set also to the Pref Number or not.
      tBool bPREFNumberSet = FALSE;

      if (typeList.contains("PREF"))
      {
         //ETG_TRACE_USR4(( "   vCard-Prop TEL number has type PREF. " ));

         if (contact->PPPhoneNum1.isEmpty())
         {
            contact->PPPhoneNum1 = qsPhoneNumber;
            contact->PhoneNum1 = StripTelephoneNumber(qsPhoneNumber);
            contact->NumberSubstr1 = contact->PhoneNum1.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW 
            contact->NumberType1 = PHONE_NUMBER_PREFERRED;

            ETG_TRACE_USR4((" contact->PPPhoneNum1: '%s'", contact->PPPhoneNum1.toLatin1().constData()));
            ETG_TRACE_USR4((" contact->PhoneNum1: '%s'", contact->PhoneNum1.toLatin1().constData()));

            //ETG_TRACE_USR4(( "   vCard-Prop TEL; 'PrefNum' is SET to: '%s'", qsPhoneNumber.toLatin1().constData() ));
            bPREFNumberSet = TRUE;
         }
         else
         {
            //ETG_TRACE_USR4(( "   vCard-Prop TEL; 'PrefNum' has been set already and will NOT be changed.  " ));
         }
      }

      if (typeList.contains("CELL"))
      {
         //ETG_TRACE_USR4(( "   vCard-Prop TEL number has type CELL. " ));
         if (contact->PPPhoneNum2.isEmpty())
         {
            contact->PPPhoneNum2 = qsPhoneNumber;
            ETG_TRACE_USR4((" contact->PPPhoneNum2: %s", contact->PPPhoneNum2.toLatin1().constData()));
            contact->PhoneNum2 = StripTelephoneNumber(qsPhoneNumber);
            contact->NumberSubstr2 = contact->PhoneNum2.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW 
            contact->NumberType2 = PHONE_NUMBER_CELL;
            //ETG_TRACE_USR4(( "   vCard-Prop TEL; 'CellNum1' is SET to: '%s'.  ", qsPhoneNumber.toLatin1().constData() ));
         }
         else if (contact->PPPhoneNum3.isEmpty())
         {
            contact->PPPhoneNum3 = qsPhoneNumber;
            ETG_TRACE_USR4((" contact->PPPhoneNum3: %s", contact->PPPhoneNum3.toLatin1().constData()));
            contact->PhoneNum3 = StripTelephoneNumber(qsPhoneNumber);
            contact->NumberSubstr3 = contact->PhoneNum3.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
            contact->NumberType3 = PHONE_NUMBER_CELL;
            //ETG_TRACE_USR4(( "   vCard-Prop TEL; 'CellNum2' is SET to: '%s'.  ", qsPhoneNumber.toLatin1().constData() ));
         }
         else if (bPREFNumberSet)
         {
            //ETG_TRACE_USR4(( "   vCard-Prop TEL; 'CellNum2' is CHANGED (due to first 'PREF') to: '%s'.  ", qsPhoneNumber.toLatin1().constData() ));
            //ETG_TRACE_USR4(( "   vCard-Prop TEL; 'OtherNum' is SET to (old 'CellNum2'): '%s'.  ",  contact->homeNum2.toLatin1().constData() ));
            contact->PhoneNum8 = contact->PhoneNum3;
            contact->PPPhoneNum8 = contact->PPPhoneNum3;
            contact->NumberSubstr8 = contact->NumberSubstr3;   // NEW
            contact->PhoneNum3 = StripTelephoneNumber(qsPhoneNumber);
            contact->PPPhoneNum3 = qsPhoneNumber;
            contact->NumberSubstr3 = contact->PhoneNum3.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
            contact->NumberType8 = PHONE_NUMBER_CELL;
            contact->NumberType3 = PHONE_NUMBER_PREFERRED;
         }
      }
      else if (typeList.contains("HOME"))
      {
         //ETG_TRACE_USR4(( "   vCard-Prop TEL number has type HOME. " ));
         if (contact->PPPhoneNum4.isEmpty())
         {
            contact->PPPhoneNum4 = qsPhoneNumber;
            ETG_TRACE_USR4((" contact->PPPhoneNum4: %s", contact->PPPhoneNum4.toLatin1().constData()));
            contact->PhoneNum4 = StripTelephoneNumber(qsPhoneNumber);
            contact->NumberSubstr4 = contact->PhoneNum4.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
            contact->NumberType4 = PHONE_NUMBER_HOME;
            //ETG_TRACE_USR4(( "   vCard-Prop TEL; 'HomeNum1' is SET to: '%s'.  ", qsPhoneNumber.toLatin1().constData() ));
         }
         else if (contact->PPPhoneNum5.isEmpty())
         {
            contact->PPPhoneNum5 = qsPhoneNumber;
            ETG_TRACE_USR4((" contact->PPPhoneNum5: %s", contact->PPPhoneNum5.toLatin1().constData()));
            contact->PhoneNum5 = StripTelephoneNumber(qsPhoneNumber);
            contact->NumberSubstr5 = contact->PhoneNum5.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
            contact->NumberType5 = PHONE_NUMBER_HOME;
            //ETG_TRACE_USR4(( "   vCard-Prop TEL; 'HomeNum2' is SET to: '%s'.  ", qsPhoneNumber.toLatin1().constData() ));
         }
         else if (bPREFNumberSet)
         {
            //ETG_TRACE_USR4(( "   vCard-Prop TEL; 'HomeNum2' is CHANGED (due to first 'PREF') to: '%s'.  ", qsPhoneNumber.toLatin1().constData() ));
            //ETG_TRACE_USR4(( "   vCard-Prop TEL; 'OtherNum' is SET to (old 'HomeNum2'): '%s'.  ",  contact->homeNum2.toLatin1().constData() ));
            contact->PhoneNum8 = contact->PhoneNum5;
            contact->PPPhoneNum8 = contact->PPPhoneNum5;
            contact->NumberSubstr8 = contact->NumberSubstr5;   // NEW
            contact->PhoneNum5 = StripTelephoneNumber(qsPhoneNumber);
            contact->PPPhoneNum5 = qsPhoneNumber;
            contact->NumberSubstr5 = contact->PhoneNum5.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
            contact->NumberType8 = PHONE_NUMBER_HOME;
            contact->NumberType5 = PHONE_NUMBER_PREFERRED;

         }
      }
      else if (typeList.contains("WORK"))
      {
         //ETG_TRACE_USR4(( "   vCard-Prop TEL number has type WORK.  " ));
         if (contact->PPPhoneNum6.isEmpty())
         {
            contact->PPPhoneNum6 = qsPhoneNumber;
            ETG_TRACE_USR4((" contact->PPPhoneNum6: %s", contact->PPPhoneNum6.toLatin1().constData()));
            contact->PhoneNum6 = StripTelephoneNumber(qsPhoneNumber);
            contact->NumberSubstr6 = contact->PhoneNum6.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
            contact->NumberType6 = PHONE_NUMBER_WORK;
            //ETG_TRACE_USR4(( "   vCard-Prop TEL; 'WorkNum1' is SET to: '%s'.  ", qsPhoneNumber.toLatin1().constData() ));
         }
         else if (contact->PPPhoneNum7.isEmpty())
         {
            contact->PPPhoneNum7 = qsPhoneNumber;
            ETG_TRACE_USR4((" contact->PPPhoneNum7: %s", contact->PPPhoneNum7.toLatin1().constData()));
            contact->PhoneNum7 = StripTelephoneNumber(qsPhoneNumber);
            contact->NumberSubstr7 = contact->PhoneNum7.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
            contact->NumberType7 = PHONE_NUMBER_WORK;
            //ETG_TRACE_USR4(( "   vCard-Prop TEL; 'WorkNum2' is SET to: '%s'.  ", qsPhoneNumber.toLatin1().constData() ));
         }
         else if (bPREFNumberSet)
         {
            //ETG_TRACE_USR4(( "   vCard-Prop TEL; 'WorkNum2' is CHANGED (due to first 'PREF') to: '%s'.  ", qsPhoneNumber.toLatin1().constData() ));
            //ETG_TRACE_USR4(( "   vCard-Prop TEL; 'OtherNum' is SET to (old 'WorkNum2'): '%s'.  ",  contact->workNum2.toLatin1().constData() ));
            contact->PhoneNum8 = contact->PhoneNum7;
            contact->PPPhoneNum8 = contact->PPPhoneNum7;
            contact->NumberSubstr8 = contact->NumberSubstr7;   // NEW
            contact->PhoneNum7 = StripTelephoneNumber(qsPhoneNumber);
            contact->PPPhoneNum7 = qsPhoneNumber;
            contact->NumberSubstr7 = contact->PhoneNum7.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
            contact->NumberType8 = PHONE_NUMBER_WORK;
            contact->NumberType7 = PHONE_NUMBER_PREFERRED;

         }
      }

      else // "OTHER" type

      {
         if (bPREFNumberSet)
         {
            //ETG_TRACE_USR4(( "   vCard-Prop TEL number HAS been set as 'PrefNum', but has not type HOME, WORK or CELL. " ));
            contact->PPPhoneNum8 = qsPhoneNumber;
            ETG_TRACE_USR4((" contact->PPPhoneNum8: %s", contact->PPPhoneNum8.toLatin1().constData()));
            contact->PhoneNum8 = StripTelephoneNumber(qsPhoneNumber);
            contact->NumberSubstr8 = contact->PhoneNum8.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW   
            contact->NumberType8 = PHONE_NUMBER_PREFERRED;
            //ETG_TRACE_USR4(( "   vCard-Prop TEL; 'OtherNum' is (PREF) SET to: '%s'.  ", qsPhoneNumber.toLatin1().constData() ));

            // Set "otherType" to highest prio ("VOICE") so that 'otherNum' will not be changed by any later parsed property (phone number)
            contact->otherType = FC_PHONEBOOK_PRIO_TYPE_VOICE;
            //ETG_TRACE_USR4(( "   vCard-Prop TEL; 'OtherNum', \"numType\" = 'VOICE'.  " ));
         }
         else // bPREFNumberSet == FALSE

         {
            //ETG_TRACE_USR4(( "   vCard-Prop TEL number has NOT been set as 'PrefNum', and has not type HOME, WORK or CELL.  " ));
            QString qsNumType = prop->parameters().value("TYPE");
            quint8 u8NumType = MapTelTypeToInteger(qsNumType);
            //ETG_TRACE_USR4(( "   vCard-Prop TEL; MappedNumType = %d, numType is: '%s'.  ", u8NumType, qsNumType.toLatin1().constData() ));

            if (contact->PPPhoneNum8.isEmpty())
            {
               contact->PPPhoneNum8 = qsPhoneNumber;
               contact->PhoneNum8 = StripTelephoneNumber(qsPhoneNumber);
               contact->NumberSubstr8 = contact->PhoneNum8.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
               contact->otherType = u8NumType;
               contact->NumberType8 = PHONE_NUMBER_OTHER;
               //ETG_TRACE_USR4(( "   vCard-Prop TEL; 'OtherNum' is (empty) SET to: '%s'.  ", qsPhoneNumber.toLatin1().constData() ));
               //ETG_TRACE_USR4(( "   vCard-Prop TEL; 'OtherNum', \"numType\" = '%s'.  ", qsNumType.toLatin1().constData() ));
            }
            else
            {
               if (u8NumType < contact->otherType)
               {
                  contact->PPPhoneNum8 = qsPhoneNumber;
                  contact->PhoneNum8 = StripTelephoneNumber(qsPhoneNumber);
                  contact->NumberSubstr8 = contact->PhoneNum8.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
                  contact->otherType = u8NumType;
                  contact->NumberType8 = PHONE_NUMBER_OTHER;
                  //ETG_TRACE_USR4(( "   vCard-Prop TEL; 'OtherNum' is (prority) SET to: '%s'.  ", qsPhoneNumber.toLatin1().constData() ));
                  //ETG_TRACE_USR4(( "   vCard-Prop TEL; 'OtherNum', \"numType\" = '%s'.  ", qsNumType.toLatin1().constData() ));
               }
            }
         }
      } // end of: "OTHER" type

#endif

}
   else if ( name == "EMAIL" )
   {
      if ( propType == FC_PhoneBook_VCard_Property::List || propType == FC_PhoneBook_VCard_Property::Compound)
         text = prop->value<QStringList>().join(" ").simplified();
      else
         text = prop->value().simplified();

      ETG_TRACE_USR2(("Email address assigned - %s", text.toLatin1().constData()));
      if ( contact->emailAdd1.isEmpty() )
      {
#ifdef LIMIT_ATTRIBUTE_LENGTH //FIX NCG3D-11656 Limiting of characters not required for AIVI
         contact->emailAdd1 = text.left(75);
#else
         contact->emailAdd1 = text;
#endif
      }
      else if ( contact->emailAdd2.isEmpty() )
      {
#ifdef LIMIT_ATTRIBUTE_LENGTH //FIX NCG3D-11656 Limiting of characters not required for AIVI
         contact->emailAdd2 = text.left(75);
#else
         contact->emailAdd2 = text;
#endif
      }
      else if ( contact->emailAdd3.isEmpty() )
      {
#ifdef LIMIT_ATTRIBUTE_LENGTH //FIX NCG3D-11656 Limiting of characters not required for AIVI
         contact->emailAdd3 = text.left(75);
#else
         contact->emailAdd3 = text;
#endif
      }
   }
   //CRQ- CMG3G-2887:PhonebookSearching and sorting
   else if ( name == "ORG" )
   {
      if ((contact->lastName.isEmpty()) && (contact->firstName.isEmpty()))
      {
         if ( propType == FC_PhoneBook_VCard_Property::List || propType == FC_PhoneBook_VCard_Property::Compound)
            text = prop->value<QStringList>().join(" ").simplified();
         else
            text = prop->value().simplified();

#ifdef LIMIT_ATTRIBUTE_LENGTH //FIX NCG3D-11656 Limiting of characters not required for AIVI
         contact->lastName = text.left(25);

         //FIX SUZUKI-18215 strange "?" showed in Phone screen for a long Chinese contact
         //The replacement character is found if any and then string is truncated at the replacement character, to make it a valid utf8 string.
         FC_PhoneBook_CallHistory_List::GetCurrentInstance()->vUtf8Validate(contact->lastName);
         //End of fix SUZUKI-18215
#else
         contact->lastName = text;
#endif
      }
      ETG_TRACE_USR2((" LastName replaced with ORG name: %s",contact->lastName.toLatin1().constData()));
   }
   //292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
   else if (name == "X-PHONETIC-LAST-NAME" )
   {
      if(!contact->bPreferredPhonetics)
      {
         ETG_TRACE_USR3(( "PHBK_YOMI: X-PHONETIC-LAST-NAME: Preferred Phonetics not set, hence setting " ));
         contact->Phonetics = "";   //.clear();
         contact->bPreferredPhonetics = true;
      }
      if ( propType == FC_PhoneBook_VCard_Property::List || propType == FC_PhoneBook_VCard_Property::Compound || propType == FC_PhoneBook_VCard_Property::Binary )
      {
         // We limit the string to some reasonable value, to be sure that not something very long goes into the database
         text = prop->value<QStringList>().join(" ").left(25).simplified();
      }
      else
      {
         // We limit the string to some reasonable value, to be sure that not something very long goes into the database
         text = prop->value().left(25).simplified();
      }

      contact->SortLastName = text;
      ETG_TRACE_USR3(( "PHBK_YOMI: X-PHONETIC-LAST-NAME= '%s'", text.toUtf8().constData() ));

      if ( !contact->Phonetics.isEmpty()  )
      {
         ETG_TRACE_USR3(("PHBK_YOMI: X-PHONETIC-LAST-NAME: Phonetics string not empty, append text "));
         contact->Phonetics.append(" ");
         contact->Phonetics.append(text);
         contact->Phonetics = contact->Phonetics.simplified();
      }
      else  if ( (!text.isEmpty()) && (contact->Phonetics.isEmpty())  )
      {
         ETG_TRACE_USR3(( "PHBK_YOMI: X-PHONETIC-LAST-NAME: Phonetics string is empty, copying text to phonetics field " ));
         contact->Phonetics = text.simplified();
      }
      
      // Remove the Half-witdh Katakana VOICED and SEMI-VOICED Sound Marks from the SortLastName string
      uint codepoint1 = 0xFF9E;  // HALFWIDTH KATAKANA VOICED SOUND MARK 'ﾞ'
      uint codepoint2 = 0xFF9F;  // HALFWIDTH KATAKANA SEMI-VOICED SOUND MARK 'ﾟ'
      QString qsVoicedSoundMark =     QString::fromUcs4(&codepoint1, 1);
      QString qsSemiVoicedSoundMark = QString::fromUcs4(&codepoint2, 1);

      contact->SortLastName.replace(qsVoicedSoundMark,     QString(""));
      contact->SortLastName.replace(qsSemiVoicedSoundMark, QString(""));

      ETG_TRACE_USR3(( "PHBK_YOMI: X-PHONETIC-LAST-NAME: Phonetics= '%s'", contact->Phonetics.toUtf8().constData() ));
   }
   else if (name == "X-PHONETIC-FIRST-NAME")
   {
      if(!contact->bPreferredPhonetics)
      {
         ETG_TRACE_USR3(( "PHBK_YOMI: X-PHONETIC-FIRST-NAME: Preferred Phonetics not set, hence setting " ));
         contact->Phonetics = "";   //.clear();
         contact->bPreferredPhonetics = true;
      }
      if ( propType == FC_PhoneBook_VCard_Property::List || propType == FC_PhoneBook_VCard_Property::Compound || propType == FC_PhoneBook_VCard_Property::Binary )
      {
         // We limit the string to some reasonable value, to be sure that not something very long goes into the database
         text = prop->value<QStringList>().join(" ").left(25).simplified();
      }
      else
      {
         // We limit the string to some reasonable value, to be sure that not something very long goes into the database
         text = prop->value().left(25).simplified();
      }

      contact->SortFirstName = text;
      ETG_TRACE_USR3(( "PHBK_YOMI: X-PHONETIC-FIRST-NAME= '%s'", text.toUtf8().constData() ));
      
      if ( !contact->Phonetics.isEmpty()  )
      {
         ETG_TRACE_USR3(("PHBK_YOMI: X-PHONETIC-FIRST-NAME: Phonetics string not empty, append text "));
         text.append(" ");
         text.append(contact->Phonetics);
         contact->Phonetics = text.simplified();
      }
      else  if ( (!text.isEmpty()) && (contact->Phonetics.isEmpty())  )
      {
         ETG_TRACE_USR3(( "PHBK_YOMI: X-PHONETIC-FIRST-NAME: Phonetics string is empty, copying text to phonetics field " ));
         contact->Phonetics = text.simplified();
      }

      // Remove the Half-witdh Katakana VOICED and SEMI-VOICED Sound Marks from the SortFirstName string
      uint codepoint1 = 0xFF9E;  // HALFWIDTH KATAKANA VOICED SOUND MARK 'ﾞ'
      uint codepoint2 = 0xFF9F;  // HALFWIDTH KATAKANA SEMI-VOICED SOUND MARK 'ﾟ'
      QString qsVoicedSoundMark =     QString::fromUcs4(&codepoint1, 1);
      QString qsSemiVoicedSoundMark = QString::fromUcs4(&codepoint2, 1);

      contact->SortFirstName.replace(qsVoicedSoundMark,     QString(""));
      contact->SortFirstName.replace(qsSemiVoicedSoundMark, QString(""));
        
      ETG_TRACE_USR3(( "PHBK_YOMI: X-PHONETIC-FIRST-NAME: Phonetics= '%s'", contact->Phonetics.toUtf8().constData() ));
   }
   else if( ((name == "SORT-STRING") && (!contact->bPreferredPhonetics))  // SORT-STRING has 2nd priority. 'bPreferredPhonetics' indicates that X-PHONETIC-XXX (1st Priority) has already been received
         || ((name == "SOUND") && (contact->Phonetics.isEmpty())) ) // SOUND has 3rd priority. X-PHONETIC-XXX or SORT-STRING would overwrite 'Phonetics'
   {
      QString qsPhoneticFirstName = "";
      QString qsPhoneticLastName =  "";
      contact->Phonetics = "";   //.clear();

      /* Get the values for the key TYPE */
      QList<QString> typeList = prop->parameters().values("TYPE");

      // According to the vCard specifications version 2.1 and 3.0
      // - SORT-STRING contains text information to be used for sorting.
      // - SOUND contains digital sound content or an uri (which contains the sound content) for proper pronounciation. 
      // Both properties can be extended according to http://www.mcpc-jp.org/news/pdf/TR-015vCard_Guideline_V1.00.pdf via the parameter TYPE "X-IRMC-N".
      // This means: 
      // - We can trust SORT-STRING in general, that it contains a string for sorting
      // - If the parameter "X-IRMC-N" is available, we can trust the content of SOUND and SORT-STRING, that they contain a string for sorting.
      // - In case of SOUND, we trust its content (to be a string for sorting) only if the type is "X-IRMC-N".


      if( typeList.contains("X-IRMC-N") )
      {
         QStringList typeList2 = prop->parameters().values("TYPE");
         QString text2 = typeList2.join(";").simplified();
         ETG_TRACE_USR3(( "PHBK_YOMI: X-IRMC-N: vCard TYPE parameters (';' separated): '%s'", text2.toLatin1().constData() ));
         ETG_TRACE_USR3(( "PHBK_YOMI: X-IRMC-N: propType= %d", propType ));

         if ( propType == FC_PhoneBook_VCard_Property::List || propType == FC_PhoneBook_VCard_Property::Compound || propType == FC_PhoneBook_VCard_Property::Binary )
         {
            QStringList NameValueList = prop->value<QStringList>();

            ETG_TRACE_USR3(( "PHBK_YOMI: X-IRMC-N: Vcard property SORT-STRING/SOUND is a string list. Values are: " ));
            for(qint16 listIndex = 0; (listIndex < NameValueList.size()) && (listIndex <= (qint16)FC_PHONEBOOK_PROP_N_FIRSTNAME_INDEX); listIndex++)
            {
               ETG_TRACE_USR3(("PHBK_YOMI: X-IRMC-N: Name Value[%d]= '%s'", listIndex, NameValueList.at(listIndex).toUtf8().constData()));
               switch(listIndex)
               {
               case FC_PHONEBOOK_PROP_N_LASTNAME_INDEX:   // 0
                  // We limit the string to some reasonable value, to ensure that not something very long goes into the database
                  qsPhoneticLastName = NameValueList.at(listIndex).left(25).simplified();

                  //FIX SUZUKI-18215 strange "?" showed in Phone screen for a long Chinese contact
                  //The replacement character is found if any and then string is truncated at the replacement character, to make it a valid Utf8 string.
                  FC_PhoneBook_CallHistory_List::GetCurrentInstance()->vUtf8Validate(qsPhoneticLastName);
                  //End of fix SUZUKI-18215

                  break;
               case FC_PHONEBOOK_PROP_N_FIRSTNAME_INDEX:   // 1
                  // We limit the string to some reasonable value, to be sure that not something very long goes into the database.
                  qsPhoneticFirstName = NameValueList.at(listIndex).left(25).simplified();

                  //FIX SUZUKI-18215 strange "?" showed in Phone screen for a long Chinese contact
                  //The replacement character is found if any and then string is truncated at the replacement character, to make it a valid Utf8 string.
                  FC_PhoneBook_CallHistory_List::GetCurrentInstance()->vUtf8Validate(qsPhoneticFirstName);
                  //End of fix SUZUKI-18215

                  break;
               default:
                  {
                     ETG_TRACE_USR3(("PHBK_YOMI: X-IRMC-N: SORT-STRING/SOUND: Not Valid listIndex "));
                  }
               }
               // The defined order for Phonetics is: First FirstName then LastName
               contact->Phonetics = (qsPhoneticFirstName + " " + qsPhoneticLastName).simplified();
               
               contact->SortFirstName = qsPhoneticFirstName;
               contact->SortLastName =  qsPhoneticLastName;
            }
         }
      }
      else
      {
         // When we come here, the 'typeList' of the property (SORT-STRING or SOUND) does not contain the type "X-IRMC-N". 
         // Therfore we will only consider the property SORT-STRING here.
         // (As explained above, we do not trust SOUND here (as type 'X-IRMC-N' is not set) and therefore SOUND might not be a string for sorting.)
         if (name == "SORT-STRING")
         {
            // When we come here, we expect that we dispatch a SORT-STRING and that its 'value' is "plain", i.e. SORT-STRING= 'FirstName LastName'.

            // We limit the found value to a reasonable size, to ensure that not something very long goes into the database later.
            // Note: .simplified() removes leading and trailing whitespaces and also replace each sequence of internal whitespace by a single space
            QString qsSortStringSimplified = prop->value().left(50).simplified();

            ETG_TRACE_USR3(( "PHBK_YOMI: SORT-STRING (plain)= '%s'", qsSortStringSimplified.toUtf8().constData() ));

            // BGN: Extract "potential" Phonetic-FirstName and Phonetic-LastName sections from plain SORT-STRING.
            QString qsPotentialPhoneticFirstName = "";
            QString qsPotentialPhoneticLastName  = "";

            // Find first space in qsSortStringSimplified
            int iSpaceIndex = qsSortStringSimplified.indexOf(" ");
            if(iSpaceIndex > 0)  // We just check for '> 0' (and not for '>= 0') as the string is simplified and therfore cannot start with a space.
            {
               // First separating space is found. Note: Index counting starts with index 0 for the first character of the string.
               // The found iSpaceIndex represent directly the length of the string up the space without that space.
               // Extract now the 'PotentialPhoneticFirstName'
               qsPotentialPhoneticFirstName = qsSortStringSimplified.left(iSpaceIndex);
               ETG_TRACE_USR3(( "PHBK_YOMI: SORT-STRING (plain): qsPotentialPhoneticFirstName= '%s'", qsPotentialPhoneticFirstName.toUtf8().constData() ));
               
               // Extract 'PotentialPhoneticLastName' (starting after the space, i.e. with index (iSpaceIndex + 1)
               // Last index of string is (qsSortStringSimplified.length - 1)
               if( (qsSortStringSimplified.length() - 1) >=  (iSpaceIndex + 1) )  // Check if last index >= start index
               {
                  qsPotentialPhoneticLastName = qsSortStringSimplified.mid(iSpaceIndex + 1);  // .mid extracts all characters from the string, starting with index (iSpaceIndex + 1)
                  ETG_TRACE_USR3(( "PHBK_YOMI: SORT-STRING (plain): qsPotentialPhoneticLastName= '%s'", qsPotentialPhoneticLastName.toUtf8().constData() ));
               }
            }
            else
            {
               qsPotentialPhoneticLastName = qsSortStringSimplified;
            }
            // END: Extract "potential" Phonetic-FirstName and Phonetic-LastName sections from plain SORT-STRING. 

            // The defined order for Phonetics is: First FirstName then LastName
            contact->Phonetics = (qsPotentialPhoneticFirstName + " " + qsPotentialPhoneticLastName).simplified();
               
            contact->SortFirstName = qsPotentialPhoneticFirstName;
            contact->SortLastName =  qsPotentialPhoneticLastName;
         }
         else
         {
            ETG_TRACE_USR3(( "PHBK_YOMI: The SOUND information probably is NOT the phonetic description for the Name of the contact." ));
         }
      }

      // Removal of any Half-witdh Katakana VOICED and SEMI-VOICED Sound Marks from 'SortFirstName' and 'SortLastName' strings
      uint codepoint1 = 0xFF9E;  // HALFWIDTH KATAKANA VOICED SOUND MARK 'ﾞ'
      uint codepoint2 = 0xFF9F;  // HALFWIDTH KATAKANA SEMI-VOICED SOUND MARK 'ﾟ'
      QString qsVoicedSoundMark =     QString::fromUcs4(&codepoint1, 1);
      QString qsSemiVoicedSoundMark = QString::fromUcs4(&codepoint2, 1);
      
      contact->SortFirstName.replace(qsVoicedSoundMark,     QString(""));
      contact->SortFirstName.replace(qsSemiVoicedSoundMark, QString(""));

      contact->SortLastName.replace(qsVoicedSoundMark,     QString(""));
      contact->SortLastName.replace(qsSemiVoicedSoundMark, QString(""));
   }
   //end of fix- 292459 - Phonetics column added to recognize (YOMI field Phonetics)Hiragana & Katakana
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::ExtractingOfFirstNPhoneNumbers
*
* DESCRIPTION:
*              Extracting the first N Phone numbers.
*              N can be 5 for IVI,4 for PSA and 8 for rest all projects.
*
*
* PARAMETER:
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_List::ExtractingOfFirstNPhoneNumbers(
   quint8 u8Numbertype,
   FC_PhoneBook_Contact_Detail* contact,QString qsPhoneNumber
   )
{
   ETG_TRACE_USR4(( " ExtractingOfFirstNPhoneNumbers entered" ));

   if(contact->PPPhoneNum1.isEmpty())
   {
      contact->PPPhoneNum1 = qsPhoneNumber;
      ETG_TRACE_USR4(( " contact->PPPhoneNum1: %s", contact->PPPhoneNum1.toLatin1().constData() ));
      contact->PhoneNum1 = StripTelephoneNumber(qsPhoneNumber);
      contact->NumberSubstr1 = contact->PhoneNum1.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
      contact->NumberType1 = u8Numbertype;
   }
   else if(contact->PPPhoneNum2.isEmpty())
   {
      contact->PPPhoneNum2 = qsPhoneNumber;
      ETG_TRACE_USR4(( " contact->PPPhoneNum2: %s", contact->PPPhoneNum2.toLatin1().constData() ));
      contact->PhoneNum2 = StripTelephoneNumber(qsPhoneNumber);
      contact->NumberSubstr2 = contact->PhoneNum2.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
      contact->NumberType2 = u8Numbertype;
   }
   else if(contact->PPPhoneNum3.isEmpty())
   {
      contact->PPPhoneNum3 = qsPhoneNumber;
      ETG_TRACE_USR4(( " contact->PPPhoneNum3: %s", contact->PPPhoneNum3.toLatin1().constData() ));
      contact->PhoneNum3 = StripTelephoneNumber(qsPhoneNumber);
      contact->NumberSubstr3 = contact->PhoneNum3.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
      contact->NumberType3 = u8Numbertype;
   }
   else if(contact->PPPhoneNum4.isEmpty())
   {
      contact->PPPhoneNum4 = qsPhoneNumber;
      ETG_TRACE_USR4(( " contact->PPPhoneNum4: %s", contact->PPPhoneNum4.toLatin1().constData() ));
      contact->PhoneNum4 = StripTelephoneNumber(qsPhoneNumber);
      contact->NumberSubstr4 = contact->PhoneNum4.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
      contact->NumberType4 = u8Numbertype;
   }
   else if(contact->PPPhoneNum5.isEmpty())
   {
      contact->PPPhoneNum5 = qsPhoneNumber;
      ETG_TRACE_USR4(( " contact->PPPhoneNum5: %s", contact->PPPhoneNum5.toLatin1().constData() ));
      contact->PhoneNum5 = StripTelephoneNumber(qsPhoneNumber);
      contact->NumberSubstr5 = contact->PhoneNum5.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
      contact->NumberType5 = u8Numbertype;
   }
   else if(contact->PPPhoneNum6.isEmpty())
   {
      contact->PPPhoneNum6 = qsPhoneNumber;
      ETG_TRACE_USR4(( " contact->PPPhoneNum6: %s", contact->PPPhoneNum6.toLatin1().constData() ));
      contact->PhoneNum6 = StripTelephoneNumber(qsPhoneNumber);
      contact->NumberSubstr6 = contact->PhoneNum6.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
      contact->NumberType6 = u8Numbertype;
   }
   else if(contact->PPPhoneNum7.isEmpty())
   {
      contact->PPPhoneNum7 = qsPhoneNumber;
      ETG_TRACE_USR4(( " contact->PPPhoneNum7: %s", contact->PPPhoneNum7.toLatin1().constData() ));
      contact->PhoneNum7 = StripTelephoneNumber(qsPhoneNumber);
      contact->NumberSubstr7 = contact->PhoneNum7.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
      contact->NumberType7 = u8Numbertype;
   }
   else if(contact->PPPhoneNum8.isEmpty())
   {
      contact->PPPhoneNum8 = qsPhoneNumber;
      ETG_TRACE_USR4(( " contact->PPPhoneNum8: %s", contact->PPPhoneNum8.toLatin1().constData() ));
      contact->PhoneNum8 = StripTelephoneNumber(qsPhoneNumber);
      contact->NumberSubstr8 = contact->PhoneNum8.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
      contact->NumberType8 = u8Numbertype;
   }
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::StripTelephoneNumber
*
* DESCRIPTION:
*              Returns stripped phone number.
*
*
* PARAMETER:
* RETURNVALUE:
*
*******************************************************************************/
QString FC_PhoneBook_List::StripTelephoneNumber( QString sTelephoneNumber)
{
   //ETG_TRACE_USR4(( " UnStripped telephone number %s", sTelephoneNumber.toUtf8().constData()));
   sTelephoneNumber.replace(QRegExp("[^0-9*#+P]"), "");
   //ETG_TRACE_USR4(( " Stripped telephone number %s", sTelephoneNumber.toUtf8().constData()));
   return sTelephoneNumber;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::GetSortTypeForDevice
*
* DESCRIPTION:
*              Returns the sort order of a given device handle.
*
*
* PARAMETER:
* RETURNVALUE:
*
*******************************************************************************/
FC_PhoneBook_SQLite::SortType FC_PhoneBook_List::GetSortTypeForDevice(quint8 u8DeviceHandle)
{
   ETG_TRACE_USR4(( " GetSortTypeForDevice entered with DeviceHandle %d",u8DeviceHandle ));
   FC_PhoneBook_SQLite::SortType sorttype = FC_PhoneBook_SQLite::LastName;

   if(m_hashPbList.isEmpty())
   {
      ETG_TRACE_USR4(("FC_PhoneBook_List -  There are no existing lists. Using default sort order i.e LastName"));
      sorttype= FC_PhoneBook_SQLite::LastName;
   }
   else
   {
      int indexListData;
      bool blFound = FALSE;
      QList<FC_PhoneBook_List_Data*> pListData_Set;
      pListData_Set= m_hashPbList.values();
      for(indexListData=0; indexListData < pListData_Set.size(); indexListData++)
      {
         if(pListData_Set.at(indexListData)->u8DeviceHandle == u8DeviceHandle)
         {
            sorttype= (FC_PhoneBook_SQLite::SortType)pListData_Set.at(indexListData)->sortType;
            blFound= TRUE;
            break;
         }
      }
      if(FALSE == blFound)
      {
         ETG_TRACE_USR4(("FC_PhoneBook_List -  There are no lists corresponding to the "
            "outgoing source. Using default sort order i.e LastName"));
         sorttype= FC_PhoneBook_SQLite::LastName;
      }
   }
   return sorttype;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::bValidateSearchString
*
* DESCRIPTION:
*              Validates whether the search string.
*
*
* PARAMETER:
* RETURNVALUE: tBool - Indicates whether or not the search string is valid.
*
*******************************************************************************/
tBool FC_PhoneBook_List::bValidateSearchString(QString &sSearchString)
{
   ETG_TRACE_USR4((" bValidateSearchString entered with SearchString: %s", sSearchString.toUtf8().constData()));

   sSearchString.replace(QRegExp("[:;?!\\-(){}\\[\\]]"), "");
   ETG_TRACE_USR4((" Validated Search String : %s", sSearchString.toUtf8().constData()));

   return (sSearchString.size())? TRUE : FALSE;
}

/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::bValidateDeviceHandles
*
* DESCRIPTION:
*              Validates the device handles received in the device handle list.
*
*
* PARAMETER: [IN]QList - List of device handles received
* RETURNVALUE: tBool - Indicates whether or not the device handles are valid.
*
*******************************************************************************/
tBool FC_PhoneBook_List::bValidateDeviceHandles(QList<tU8>& DeviceHandleList)
{
   ETG_TRACE_USR4(( " bValidateDeviceHandles entered" ));
   QList<tU8> ValidDeviceHandles;
   tBool bRetValue = FALSE;
   for (tU8 u8Itr=0; u8Itr< DeviceHandleList.size(); u8Itr++)
   {
        ETG_TRACE_USR4(( "Processing DeviceHandle: %u ", DeviceHandleList[u8Itr]));
        if( 0 == DeviceHandleList[u8Itr] )
        {
           FC_Device_Details *pDeviceObject = NULL;
           for(tU8 u8Index= 0; u8Index < NO_OF_CONNECTED_DEVICE; u8Index++)
           {
              pDeviceObject = FC_Device_Details::getDeviceObject((INDEX)u8Index);
			  //CMG3G-14313-Coverity Fix
              if (pDeviceObject)
              {
                 if(FALSE == ValidDeviceHandles.contains(pDeviceObject->getDeviceHandle()))
                 {
                   ETG_TRACE_USR4(( " Appending valid device handle : %u to ValidDeviceHandles", pDeviceObject->getDeviceHandle()));
                   ValidDeviceHandles.append(pDeviceObject->getDeviceHandle());
                 }
                 else
                 {
                   ETG_TRACE_USR4((" DeviceHandle : %u already exists in ValidDeviceHandles ", pDeviceObject->getDeviceHandle() ));
                 }
              }//end of fix
           }
        }
        else  if ( (1 == DeviceHandleList[u8Itr]) || (FC_Device_Details::deviceHandleToObjectInstance(DeviceHandleList[u8Itr])) )
        {
           ETG_TRACE_USR4(( " Valid Devicehandles(1-11) Passed " ));
           if(FALSE == ValidDeviceHandles.contains(DeviceHandleList[u8Itr]) )
           {
              ETG_TRACE_USR4(( " Appending device handle : %u to ValidDeviceHandles", DeviceHandleList[u8Itr]));
              ValidDeviceHandles.append(DeviceHandleList[u8Itr]);
           }
           else
           {
              ETG_TRACE_USR4((" DeviceHandle : %u already exists in ValidDeviceHandles ", DeviceHandleList[u8Itr]));
           }
        }
        else
        {
            ETG_TRACE_ERR(("The DeviceHandle :%u is either not connected or is not a valid one.Hence,Not appending to DeviceHandleList.", DeviceHandleList[u8Itr]));
        }
     }

     if(ValidDeviceHandles.size() > 0)
     {
        DeviceHandleList.clear();
        DeviceHandleList = ValidDeviceHandles;
        bRetValue = TRUE;
     }
     return bRetValue;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::ClearPhoneBookData
*
* DESCRIPTION: Clears all contacts, call history and favorites from PB DB.
*
*
* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_List::ClearPhoneBookData()
{
   ETG_TRACE_USR4(( " ClearPhoneBookData entered" ));

   //FIX GMMY16-23858 FC_Phonebook sends an error even though the desired action is done
   //ClearPhoneBookData() was returning ERROR even though the desired action was performed.
   bool bRetValue =  false;

   //Delete all contacts and call history from DB
   if(!m_sqlite->DeleteAllContactsAndCallHistoryFromDB())
   {
      ETG_TRACE_ERR(("  Deleting of  contacts/call history FAILED! "));
   }
   else
   {
      ETG_TRACE_USR4(("  Deleting of  contacts/call history SUCCESS. "));

      //Send list change updates for PB
      ListChange_Handler pListChange_Handler;
      quint8 i;
      for(i = FC_PB_VEHICLE_PBOOK ; i <= FC_PB_DEVICE_HANDLE11; i++)
      {
         pListChange_Handler.UpdateList(&m_hashPbList, enAction_Content_Refreshed, i, 0);
      }

      //Send list change updates for CH
      FC_PhoneBook_CallHistory_List::GetCurrentInstance()->ClearcallHistory();

      //Delete all favorites from DB and send favorites changed update
      if(TRUE != m_poFavoritesProvider->RefreshFavoritesOnClearPhoneBook())
      {
         ETG_TRACE_ERR(("  Deleting of  favorites failed "));
      }
      else
         bRetValue = true;
   }
   //End of fix GMMY16-23858

   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::ClearVehiclePhoneBook
*
* DESCRIPTION: This method clears all vehicle phonebook entries from a specific vehicle phonebook. [VPBx]
*
* PARAMETER: [IN] methodStart :- devicehandle
*            [OUT]
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_List::ClearVehiclePhoneBook(tU8 u8DeviceHandle)
{
   ETG_TRACE_USR4((" ClearVehiclePhoneBook entered with device handle: %d", u8DeviceHandle));

   tBool bResult = FALSE;

   if (!m_sqlite)
   {
      ETG_TRACE_ERR((" m_sqlite is NULL!!! "));
      return FALSE;
   }

   //Get the contact handles in the VPB
   QList < quint32 > u32ContactHandleListVPB;
   m_sqlite->bGetVPBContactHandles(u8DeviceHandle, u32ContactHandleListVPB);

   if (0 != u32ContactHandleListVPB.size())//Deletion is to be done only if there exists any contacts.
   {
      bResult = m_sqlite->DeleteVehiclePhoneBook(u8DeviceHandle);

      if (TRUE == bResult)
      {
         bResult
               = (tBool)(bResult & m_sqlite->RemoveContactHandlesFromMasterTable(u32ContactHandleListVPB, u8DeviceHandle));

         ListChange_Handler oListChange_Handler;
         if (FC_PB_VEHICLE_PBOOK == u8DeviceHandle)
         {
            oListChange_Handler.UpdateList(&m_hashPbList, enAction_Content_Refreshed, FC_PB_VEHICLE_PBOOK, 0);

            //Remove favorites for VPB
            m_poFavoritesProvider->RefreshFavoritesOnClearVehiclePhoneBook();
         }
         else
         {
            oListChange_Handler.UpdateList(&m_hashPbList, enAction_Content_Refreshed, u8DeviceHandle, 0, FC_PhoneBook_SQLite::ListType_VEHICLE);
         }

         //Update call history
         FC_PhoneBook_CallHistory_List* pCallHistory =
               FC_PhoneBook_CallHistory_List::GetCurrentInstance();

         if (pCallHistory)
         {
            for (quint16 u16count = 0; u16count
                  < u32ContactHandleListVPB.size(); u16count++)
            {
               FC_PhoneBook_Contact_Detail Contact;

               //We don't have to delete the Entry from Call History but have to Update/Edit it
               Contact.contactHandle = u32ContactHandleListVPB.at(u16count);
               Contact.firstName = "";
               Contact.lastName = "";
               pCallHistory->RefreshCallHistoryOnChangeInContact(Contact);
            }
         }
      }
      else
      {
         ETG_TRACE_ERR((" Deletion of VehiclePhonebook Contacts in DB has failed!!! "));
      }
   }
   else
   {
      //If there are no entries, there is no need to do any operation, still return SUCCESS.
      bResult = TRUE;
   }

   return bResult;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::AddContact
*
* DESCRIPTION:
*
*
*
* PARAMETER: [IN] methodStart :- devicehandle and Contact details to be added
*            [OUT] methodResult :- Contact handle and status (success or failure)
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_List::AddContact( const most_phonbkfi_tclMsgAddContactMethodStart* methodStart,
                                   most_phonbkfi_tclMsgAddContactMethodResult* methodResult )
{
   ETG_TRACE_USR4(("AddContact  Entered  "));

   FC_PhoneBook_Contact_Detail Contact;
   quint32 ContactHandle = 0; //Contact Handle has to be generated by database.
   //So make it zero

   Contact.firstName= QString::fromUtf8(methodStart->oContactDetails.sFirstName.szValue);
   Contact.lastName= QString::fromUtf8(methodStart->oContactDetails.sLastName.szValue);
   Contact.poiName= QString::fromUtf8(methodStart->oContactDetails.sPOIName.szValue);
   Contact.PPPhoneNum1= QString::fromUtf8(methodStart->oContactDetails.sPreferredNumber.szValue);
   Contact.PhoneNum1= StripTelephoneNumber(methodStart->oContactDetails.sPreferredNumber.szValue);
   Contact.NumberSubstr1 = Contact.PhoneNum1.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
   if(Contact.PPPhoneNum1!="")
   {
      Contact.NumberType1 = PHONE_NUMBER_PREFERRED;
   }
   Contact.PPPhoneNum2= QString::fromUtf8(methodStart->oContactDetails.sCellNumber1.szValue);
   Contact.PhoneNum2= StripTelephoneNumber(methodStart->oContactDetails.sCellNumber1.szValue);
   Contact.NumberSubstr2 = Contact.PhoneNum2.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
   if(Contact.PPPhoneNum2!="")
   {
      Contact.NumberType2 = PHONE_NUMBER_CELL;
   }
   Contact.PPPhoneNum3= QString::fromUtf8(methodStart->oContactDetails.sCellNumber2.szValue);
   Contact.PhoneNum3= StripTelephoneNumber(methodStart->oContactDetails.sCellNumber2.szValue);
   Contact.NumberSubstr3 = Contact.PhoneNum3.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
   if(Contact.PPPhoneNum3!="")
   {
      Contact.NumberType3 = PHONE_NUMBER_CELL;
   }
   Contact.PPPhoneNum4= QString::fromUtf8(methodStart->oContactDetails.sHomeNumber1.szValue);
   Contact.PhoneNum4= StripTelephoneNumber(methodStart->oContactDetails.sHomeNumber1.szValue);
   Contact.NumberSubstr4 = Contact.PhoneNum4.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
   if(Contact.PPPhoneNum4!="")
   {
      Contact.NumberType4 = PHONE_NUMBER_HOME;
   }
   Contact.PPPhoneNum5= QString::fromUtf8(methodStart->oContactDetails.sHomeNumber2.szValue);
   Contact.PhoneNum5= StripTelephoneNumber(methodStart->oContactDetails.sHomeNumber2.szValue);
   Contact.NumberSubstr5 = Contact.PhoneNum5.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
   if(Contact.PPPhoneNum5!="")
   {
      Contact.NumberType5 = PHONE_NUMBER_HOME;
   }
   Contact.PPPhoneNum6= QString::fromUtf8(methodStart->oContactDetails.sWorkNumber1.szValue);
   Contact.PhoneNum6= StripTelephoneNumber(methodStart->oContactDetails.sWorkNumber1.szValue);
   Contact.NumberSubstr6 = Contact.PhoneNum6.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
   if(Contact.PPPhoneNum6!="")
   {
      Contact.NumberType6 = PHONE_NUMBER_WORK;
   }
   Contact.PPPhoneNum7= QString::fromUtf8(methodStart->oContactDetails.sWorkNumber2.szValue);
   Contact.PhoneNum7= StripTelephoneNumber(methodStart->oContactDetails.sWorkNumber2.szValue);
   Contact.NumberSubstr7 = Contact.PhoneNum7.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
   if(Contact.PPPhoneNum7!="")
   {
      Contact.NumberType7 = PHONE_NUMBER_WORK;
   }
   Contact.PPPhoneNum8= QString::fromUtf8(methodStart->oContactDetails.sOtherNumber.szValue);
   Contact.PhoneNum8= StripTelephoneNumber(methodStart->oContactDetails.sOtherNumber.szValue);
   Contact.NumberSubstr8 = Contact.PhoneNum8.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);   // NEW
   if(Contact.PPPhoneNum8!="")
   {
      Contact.NumberType8 = PHONE_NUMBER_OTHER;
   }


   Contact.emailAdd1= QString::fromUtf8(methodStart->oContactDetails.sEmailAddress1.szValue);
   Contact.emailAdd2= QString::fromUtf8(methodStart->oContactDetails.sEmailAddress2.szValue);
   Contact.emailAdd3= QString::fromUtf8(methodStart->oContactDetails.sEmailAddress3.szValue);
   Contact.homeAdd= QString::fromUtf8(methodStart->oContactDetails.sHomeAddress.szValue);
   Contact.workAdd= QString::fromUtf8(methodStart->oContactDetails.sWorkAddress.szValue);
   Contact.otherAdd= QString::fromUtf8(methodStart->oContactDetails.sOtherAddress.szValue);
   Contact.category= QString::fromUtf8(methodStart->oContactDetails.sCategory.szValue);
   Contact.notes= QString::fromUtf8(methodStart->oContactDetails.sNotes.szValue);
   Contact.homeGeoCodeLatLongValidity = methodStart->oContactDetails.oHomeAddressLatLong.bLatLongValidity;
   Contact.homeGeoCodeLatitude =  methodStart->oContactDetails.oHomeAddressLatLong.s32Latitude;
   Contact.homeGeocodeLongitude = methodStart->oContactDetails.oHomeAddressLatLong.s32Longitude;
   Contact.workGeoCodeLatLongValidity = methodStart->oContactDetails.oWorkAddressLatLong.bLatLongValidity;
   Contact.workGeoCodeLatitude =  methodStart->oContactDetails.oWorkAddressLatLong.s32Latitude;
   Contact.workGeocodeLongitude = methodStart->oContactDetails.oWorkAddressLatLong.s32Longitude;
   Contact.otherGeoCodeLatLongValidity = methodStart->oContactDetails.oOtherAddressLatLong.bLatLongValidity;
   Contact.otherGeoCodeLatitude = methodStart->oContactDetails.oOtherAddressLatLong.s32Latitude;
   Contact.otherGeocodeLongitude = methodStart->oContactDetails.oOtherAddressLatLong.s32Longitude;
   Contact.photoDlSize = 0;
   Contact.photo       = "";   //.clear();
   Contact.photoSize   = 0;
   Contact.photoType   = "";   //.clear();
   Contact.photoCRC    = 0;
   Contact.photoDlCRC  = 0;
   Contact.APL1Size    = 0;
   Contact.crcAPL1     = 0;

   if (m_sqlite->GetNoOfContactsInVAB(methodStart->u8DeviceHandle) >= MAX_ENTRIES_VEHICLE_PHONEBOOK )
   {
      ETG_TRACE_ERR(("  Adding of Contact to VAB failed: NO OF CONTACTS REACHED TO MAX LIMIT OF %d ", MAX_ENTRIES_VEHICLE_PHONEBOOK ));
      methodResult->u32ContactHandle = ContactHandle;
      methodResult->e8ContactAddStatus.enType = most_fi_tcl_e8_PhonBkContactAddStatus::FI_EN_E8CONTACT_ADD_FAILURE;
   }
   else
   {
      if(!m_sqlite->InsertSingleRecord(Contact, methodStart->u8DeviceHandle, ContactHandle,true))
      {
         ETG_TRACE_ERR(("   Adding of contact failed          "));
         methodResult->u32ContactHandle= ContactHandle;
         methodResult->e8ContactAddStatus.enType= most_fi_tcl_e8_PhonBkContactAddStatus::FI_EN_E8CONTACT_ADD_FAILURE;
      }
      else
      {
         m_sqlite->IncNoOfContactsVAB(methodStart->u8DeviceHandle);
         ETG_TRACE_USR2(("  Add contact success: total contacts in VAB = %d", m_sqlite->GetNoOfContactsInVAB(methodStart->u8DeviceHandle)));
         methodResult->u32ContactHandle= ContactHandle;
         methodResult->e8ContactAddStatus.enType= most_fi_tcl_e8_PhonBkContactAddStatus::FI_EN_E8CONTACT_ADD_SUCCESS;
      }

      ETG_TRACE_USR4(("   AddContact    Exit         "));
   }

   return true;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::DeleteContact
*
* DESCRIPTION:
*
*
*
* PARAMETER: [IN] methodStart :- Contact handle and Contact details to be added
*            [OUT] methodResult :- Contact handle and status (success or failure)
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_List::DeleteContact( const most_phonbkfi_tclMsgDeleteContactMethodStart* methodStart,
                                      most_phonbkfi_tclMsgDeleteContactMethodResult* methodResult )
{
   ETG_TRACE_USR4(( " DeleteContact entered" ));
   bool bRetValue = true;
   quint32 ContactHandle= methodStart->u32ContactHandle;
   if(!m_sqlite->DeleteRecord(ContactHandle,methodStart->u8DeviceHandle))
   {

      ETG_TRACE_ERR(("Deleting of contact failed"));
      methodResult->u32ContactHandle= ContactHandle;
      methodResult->e8ContactDeleteStatus.enType= most_fi_tcl_e8_PhonBkContactDeleteStatus::FI_EN_E8CONTACT_DELETE_FAILURE;
   }
   else
   {
      if(!m_sqlite->DecNoOfContactsVAB(methodStart->u8DeviceHandle))
      {
         ETG_TRACE_ERR(("VAB count is already 0"));
         bRetValue = false;
      }
      else
      {
         ETG_TRACE_USR2(("  delete contact success: total contacts in VAB = %d", m_sqlite->GetNoOfContactsInVAB(methodStart->u8DeviceHandle)));
         methodResult->u32ContactHandle= ContactHandle;
         methodResult->e8ContactDeleteStatus.enType= most_fi_tcl_e8_PhonBkContactDeleteStatus::FI_EN_E8CONTACT_DELETE_SUCCESS;
      }
   }
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::EditContact
*
* DESCRIPTION:
*
*
*
* PARAMETER: [IN] methodStart :- Contact handle and Contact details to be added
*            [OUT] methodResult :- Contact handle and status (success or failure)
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_List::EditContact( const most_phonbkfi_tclMsgEditContactMethodStart* methodStart,
                                    most_phonbkfi_tclMsgEditContactMethodResult* methodResult )
{
   ETG_TRACE_USR4(( " EditContact entered" ));
   //TODO: Take care of PhotoInfo

   FC_PhoneBook_Contact_Detail Contact;
   quint32 ContactHandle= methodStart->u32ContactHandle;

   Contact.contactHandle= ContactHandle;
   Contact.firstName= QString::fromUtf8(methodStart->oContactDetails.sFirstName.szValue);
   Contact.lastName= QString::fromUtf8(methodStart->oContactDetails.sLastName.szValue);
   Contact.poiName= QString::fromUtf8(methodStart->oContactDetails.sPOIName.szValue);
   Contact.PPPhoneNum1= QString::fromUtf8(methodStart->oContactDetails.sPreferredNumber.szValue);
   Contact.PhoneNum1= StripTelephoneNumber(methodStart->oContactDetails.sPreferredNumber.szValue);
      if(Contact.PPPhoneNum1!="")
      {
         Contact.NumberType1 = PHONE_NUMBER_PREFERRED;
      }
   Contact.PPPhoneNum2= QString::fromUtf8(methodStart->oContactDetails.sCellNumber1.szValue);
   Contact.PhoneNum2= StripTelephoneNumber(methodStart->oContactDetails.sCellNumber1.szValue);
      if(Contact.PPPhoneNum2!="")
      {
         Contact.NumberType2 = PHONE_NUMBER_CELL;
      }
   Contact.PPPhoneNum3= QString::fromUtf8(methodStart->oContactDetails.sCellNumber2.szValue);
   Contact.PhoneNum3= StripTelephoneNumber(methodStart->oContactDetails.sCellNumber2.szValue);
      if(Contact.PPPhoneNum3!="")
      {
         Contact.NumberType3 = PHONE_NUMBER_CELL;
      }
   Contact.PPPhoneNum4= QString::fromUtf8(methodStart->oContactDetails.sHomeNumber1.szValue);
   Contact.PhoneNum4= StripTelephoneNumber(methodStart->oContactDetails.sHomeNumber1.szValue);
      if(Contact.PPPhoneNum4!="")
      {
         Contact.NumberType4 = PHONE_NUMBER_HOME;
      }
   Contact.PPPhoneNum5= QString::fromUtf8(methodStart->oContactDetails.sHomeNumber2.szValue);
   Contact.PhoneNum5= StripTelephoneNumber(methodStart->oContactDetails.sHomeNumber2.szValue);
      if(Contact.PPPhoneNum5!="")
      {
         Contact.NumberType5 = PHONE_NUMBER_HOME;
      }
   Contact.PPPhoneNum6= QString::fromUtf8(methodStart->oContactDetails.sWorkNumber1.szValue);
   Contact.PhoneNum6= StripTelephoneNumber(methodStart->oContactDetails.sWorkNumber1.szValue);
      if(Contact.PPPhoneNum6!="")
      {
         Contact.NumberType6 = PHONE_NUMBER_WORK;
      }
   Contact.PPPhoneNum7= QString::fromUtf8(methodStart->oContactDetails.sWorkNumber2.szValue);
   Contact.PhoneNum7= StripTelephoneNumber(methodStart->oContactDetails.sWorkNumber2.szValue);
      if(Contact.PPPhoneNum7!="")
      {
         Contact.NumberType7 = PHONE_NUMBER_WORK;
      }
   Contact.PPPhoneNum8= QString::fromUtf8(methodStart->oContactDetails.sOtherNumber.szValue);
   Contact.PhoneNum8= StripTelephoneNumber(methodStart->oContactDetails.sOtherNumber.szValue);
      if(Contact.PPPhoneNum8!="")
      {
         Contact.NumberType8 = PHONE_NUMBER_OTHER;
      }

   Contact.NumberSubstr1 = Contact.PhoneNum1.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);
   Contact.NumberSubstr2 = Contact.PhoneNum2.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);
   Contact.NumberSubstr3 = Contact.PhoneNum3.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);
   Contact.NumberSubstr4 = Contact.PhoneNum4.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);
   Contact.NumberSubstr5 = Contact.PhoneNum5.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);
   Contact.NumberSubstr6 = Contact.PhoneNum6.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);
   Contact.NumberSubstr7 = Contact.PhoneNum7.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);
   Contact.NumberSubstr8 = Contact.PhoneNum8.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);

   Contact.emailAdd1= QString::fromUtf8(methodStart->oContactDetails.sEmailAddress1.szValue);
   Contact.emailAdd2= QString::fromUtf8(methodStart->oContactDetails.sEmailAddress2.szValue);
   Contact.emailAdd3= QString::fromUtf8(methodStart->oContactDetails.sEmailAddress3.szValue);
   Contact.homeAdd= QString::fromUtf8(methodStart->oContactDetails.sHomeAddress.szValue);
   Contact.workAdd= QString::fromUtf8(methodStart->oContactDetails.sWorkAddress.szValue);
   Contact.otherAdd= QString::fromUtf8(methodStart->oContactDetails.sOtherAddress.szValue);
   Contact.category= QString::fromUtf8(methodStart->oContactDetails.sCategory.szValue);
   Contact.notes= QString::fromUtf8(methodStart->oContactDetails.sNotes.szValue);
   Contact.homeGeoCodeLatLongValidity = methodStart->oContactDetails.oHomeAddressLatLong.bLatLongValidity;
   Contact.homeGeoCodeLatitude =  methodStart->oContactDetails.oHomeAddressLatLong.s32Latitude;
   Contact.homeGeocodeLongitude = methodStart->oContactDetails.oHomeAddressLatLong.s32Longitude;
   Contact.workGeoCodeLatLongValidity = methodStart->oContactDetails.oWorkAddressLatLong.bLatLongValidity;
   Contact.workGeoCodeLatitude =  methodStart->oContactDetails.oWorkAddressLatLong.s32Latitude;
   Contact.workGeocodeLongitude = methodStart->oContactDetails.oWorkAddressLatLong.s32Longitude;
   Contact.otherGeoCodeLatLongValidity = methodStart->oContactDetails.oOtherAddressLatLong.bLatLongValidity;
   Contact.otherGeoCodeLatitude = methodStart->oContactDetails.oOtherAddressLatLong.s32Latitude;
   Contact.otherGeocodeLongitude = methodStart->oContactDetails.oOtherAddressLatLong.s32Longitude;
   Contact.photoDlSize = 0;
   Contact.photo       = "";   //.clear();
   Contact.photoSize   = 0;
   Contact.photoType   = "";   //.clear();
   Contact.photoCRC    = 0;
   Contact.photoDlCRC  = 0;
   Contact.APL1Size    = 0;
   Contact.crcAPL1     = 0;

   if(!m_sqlite->UpdateRecord(Contact, ContactHandle))
   {
      ETG_TRACE_ERR((" Editing of contact failed "));
      methodResult->u32ContactHandle= ContactHandle;
      methodResult->e8ContactEditStatus.enType= most_fi_tcl_e8_PhonBkContactEditStatus::FI_EN_E8CONTACT_EDIT_FAILURE;
   }
   else
   {
      methodResult->u32ContactHandle= ContactHandle;
      methodResult->e8ContactEditStatus.enType= most_fi_tcl_e8_PhonBkContactEditStatus::FI_EN_E8CONTACT_EDIT_SUCCESS;
   }
   return true;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::GetContactDetails
*
* DESCRIPTION:
*
*
*
* PARAMETER: [IN] methodStart :-
*            [OUT] methodResult :-
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_List::GetContactDetails(
   const most_phonbkfi_tclMsgGetContactDetailsMethodStart* methodStart,
   most_phonbkfi_tclMsgGetContactDetailsMethodResult* methodResult )
{
   ETG_TRACE_USR4(( " GetContactDetails entered" ));
   //TODO: Take care of PhotoInfo

   FC_PhoneBook_Contact_Detail contact;
   quint32 ContactHandle = methodStart->u32ContactHandle;
   QByteArray tmpArray;
   QString sPhonenumber;
   quint8 u8Numbertype;

   //Get Details

   if( true == m_sqlite->GetContactDetails(ContactHandle, contact))
   {
      ETG_TRACE_USR4((" m_sqlite->GetContactDetais Success"));
   }
   else
   {
      ETG_TRACE_ERR((" m_sqlite->GetContactDetails Failed "));
      return false;     // MV-BugzID: 207903
   }

   switch( methodStart->e8ContactDetailFilter.enType )
   {
   case most_fi_tcl_e8_PhonBkContactDetailFilter::FI_EN_E8CDF_NONE:
      {
         tmpArray = contact.firstName.toUtf8();
         methodResult->oContactDetails.sFirstName.bSet( tmpArray.constData());
         methodResult->oContactDetails.u32FirstNameTag=
            GENERATE_TAG(ContactHandle, FIRSTNAME_TAG);

         tmpArray = contact.lastName.toUtf8();
         methodResult->oContactDetails.sLastName.bSet( tmpArray.constData());
         methodResult->oContactDetails.u32LastNameTag=
            GENERATE_TAG(ContactHandle, LASTNAME_TAG);

         tmpArray = contact.poiName.toUtf8();
         methodResult->oContactDetails.sPOIName.bSet(tmpArray.constData());
         methodResult->oContactDetails.u32POINameTag=
            GENERATE_TAG(ContactHandle, POINAME_TAG);

#ifdef PB_PHONENUMBERS_IN_PRETTY_PRINT

         if(contact.PPPhoneNum1!= "")
         {
            sPhonenumber = contact.PPPhoneNum1;
            u8Numbertype = contact.NumberType1;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

         if(contact.PPPhoneNum2!= "")
         {
            sPhonenumber = contact.PPPhoneNum2;
            u8Numbertype = contact.NumberType2;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

         if(contact.PPPhoneNum3!= "")
         {
            sPhonenumber = contact.PPPhoneNum3;
            u8Numbertype = contact.NumberType3;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

         if(contact.PPPhoneNum4!= "")
         {
            sPhonenumber = contact.PPPhoneNum4;
            u8Numbertype = contact.NumberType4;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

         if(contact.PPPhoneNum5!= "")
         {
            sPhonenumber = contact.PPPhoneNum5;
            u8Numbertype = contact.NumberType5;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

         if(contact.PPPhoneNum6!= "")
         {
            sPhonenumber = contact.PPPhoneNum6;
            u8Numbertype = contact.NumberType6;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

         if(contact.PPPhoneNum7!= "")
         {
            sPhonenumber = contact.PPPhoneNum7;
            u8Numbertype = contact.NumberType7;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }
         if(contact.PPPhoneNum8!= "")
         {
            sPhonenumber = contact.PPPhoneNum8;
            u8Numbertype = contact.NumberType8;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

#else
         if(contact.PhoneNum1!= "")
         {
            sPhonenumber = contact.PhoneNum1;
            u8Numbertype = contact.NumberType1;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }
         if(contact.PhoneNum2!= "")
         {
            sPhonenumber = contact.PhoneNum2;
            u8Numbertype = contact.NumberType2;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

         if(contact.PhoneNum3!= "")
         {
            sPhonenumber = contact.PhoneNum3;
            u8Numbertype = contact.NumberType3;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

         if(contact.PhoneNum4!= "")
         {
            sPhonenumber = contact.PhoneNum4;
            u8Numbertype = contact.NumberType4;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

         if(contact.PhoneNum5!= "")
         {
            sPhonenumber = contact.PhoneNum5;
            u8Numbertype = contact.NumberType5;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

         if(contact.PhoneNum6!= "")
         {
            sPhonenumber = contact.PhoneNum6;
            u8Numbertype = contact.NumberType6;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

         if(contact.PhoneNum7!= "")
         {
            sPhonenumber = contact.PhoneNum7;
            u8Numbertype = contact.NumberType7;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }
         if(contact.PhoneNum8!= "")
         {
            sPhonenumber = contact.PhoneNum8;
            u8Numbertype = contact.NumberType8;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }
#endif
        tmpArray =  contact.emailAdd1.toUtf8();
         methodResult->oContactDetails.sEmailAddress1.bSet(tmpArray.constData());
         methodResult->oContactDetails.u32EmailAdd1Tag=
            GENERATE_TAG(ContactHandle, EMAILADD1_TAG);

         tmpArray = contact.emailAdd2.toUtf8();
         methodResult->oContactDetails.sEmailAddress2.bSet(tmpArray.constData());
         methodResult->oContactDetails.u32EmailAdd2Tag=
            GENERATE_TAG(ContactHandle, EMAILADD2_TAG);

         tmpArray = contact.emailAdd3.toUtf8();
         methodResult->oContactDetails.sEmailAddress3.bSet(tmpArray.constData());
         methodResult->oContactDetails.u32EmailAdd3Tag=
            GENERATE_TAG(ContactHandle, EMAILADD3_TAG);

         tmpArray = contact.homeAdd.toUtf8();
         methodResult->oContactDetails.sHomeAddress.bSet(tmpArray.constData());
         methodResult->oContactDetails.u32HomeAddTag=
            GENERATE_TAG(ContactHandle, HOMEADD_TAG);

         tmpArray = contact.workAdd.toUtf8();
         methodResult->oContactDetails.sWorkAddress.bSet(tmpArray.constData());
         methodResult->oContactDetails.u32WorkAddTag=
            GENERATE_TAG(ContactHandle, WORKADD_TAG);

         tmpArray = contact.otherAdd.toUtf8();
         methodResult->oContactDetails.sOtherAddress.bSet(tmpArray.constData());
         methodResult->oContactDetails.u32OtherAddTag=
            GENERATE_TAG(ContactHandle, OTHERADD_TAG);

         tmpArray = contact.category.toUtf8();
         methodResult->oContactDetails.sCategory.bSet(tmpArray.constData());
         methodResult->oContactDetails.u32CategoryTag=
            GENERATE_TAG(ContactHandle, CATEGORY_TAG);

         tmpArray = contact.notes.toUtf8();
         methodResult->oContactDetails.sNotes.bSet(tmpArray.constData());
         methodResult->oContactDetails.u32NotesTag=
            GENERATE_TAG(ContactHandle, NOTES_TAG);

         methodResult->oContactDetails.bPhotoAvailable = (contact.photoSize > 0)?true:false;

         methodResult->oContactDetails.oHomeAddressLatLong.bLatLongValidity = contact.homeGeoCodeLatLongValidity;

         methodResult->oContactDetails.oHomeAddressLatLong.s32Latitude = contact.homeGeoCodeLatitude;

         methodResult->oContactDetails.oHomeAddressLatLong.s32Longitude = contact.homeGeocodeLongitude;

         methodResult->oContactDetails.oWorkAddressLatLong.bLatLongValidity = contact.workGeoCodeLatLongValidity;

         methodResult->oContactDetails.oWorkAddressLatLong.s32Latitude = contact.workGeoCodeLatitude;

         methodResult->oContactDetails.oWorkAddressLatLong.s32Longitude = contact.workGeocodeLongitude;

         methodResult->oContactDetails.oOtherAddressLatLong.bLatLongValidity = contact.otherGeoCodeLatLongValidity;

         methodResult->oContactDetails.oOtherAddressLatLong.s32Latitude = contact.otherGeoCodeLatitude;

         methodResult->oContactDetails.oOtherAddressLatLong.s32Longitude = contact.otherGeocodeLongitude;
      }
      break;

   case most_fi_tcl_e8_PhonBkContactDetailFilter::FI_EN_E8CDF_ADDRESS:
      {
         tmpArray = contact.homeAdd.toUtf8();
         methodResult->oContactDetails.sHomeAddress.bSet(tmpArray.constData());
         methodResult->oContactDetails.u32HomeAddTag=
            GENERATE_TAG(ContactHandle, HOMEADD_TAG);

         tmpArray = contact.workAdd.toUtf8();
         methodResult->oContactDetails.sWorkAddress.bSet(tmpArray.constData());
         methodResult->oContactDetails.u32WorkAddTag=
            GENERATE_TAG(ContactHandle, WORKADD_TAG);

         tmpArray = contact.otherAdd.toUtf8();
         methodResult->oContactDetails.sOtherAddress.bSet(tmpArray.constData());
         methodResult->oContactDetails.u32OtherAddTag=
            GENERATE_TAG(ContactHandle, OTHERADD_TAG);

         // FOR BUGZ : 208572
         methodResult->oContactDetails.oHomeAddressLatLong.bLatLongValidity = contact.homeGeoCodeLatLongValidity;

         methodResult->oContactDetails.oHomeAddressLatLong.s32Latitude = contact.homeGeoCodeLatitude;

         methodResult->oContactDetails.oHomeAddressLatLong.s32Longitude = contact.homeGeocodeLongitude;

         methodResult->oContactDetails.oWorkAddressLatLong.bLatLongValidity = contact.workGeoCodeLatLongValidity;

         methodResult->oContactDetails.oWorkAddressLatLong.s32Latitude = contact.workGeoCodeLatitude;

         methodResult->oContactDetails.oWorkAddressLatLong.s32Longitude = contact.workGeocodeLongitude;

         methodResult->oContactDetails.oOtherAddressLatLong.bLatLongValidity = contact.otherGeoCodeLatLongValidity;

         methodResult->oContactDetails.oOtherAddressLatLong.s32Latitude = contact.otherGeoCodeLatitude;

         methodResult->oContactDetails.oOtherAddressLatLong.s32Longitude = contact.otherGeocodeLongitude;
      }
      break;

   case most_fi_tcl_e8_PhonBkContactDetailFilter::FI_EN_E8CDF_TELEPHONE:
      {

#ifdef PB_PHONENUMBERS_IN_PRETTY_PRINT
         if(contact.PPPhoneNum1!= "")
         {
            sPhonenumber = contact.PPPhoneNum1;
            u8Numbertype = contact.NumberType1;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

         if(contact.PPPhoneNum2!= "")
         {
            sPhonenumber = contact.PPPhoneNum2;
            u8Numbertype = contact.NumberType2;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

         if(contact.PPPhoneNum3!= "")
         {
            sPhonenumber = contact.PPPhoneNum3;
            u8Numbertype = contact.NumberType3;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

         if(contact.PPPhoneNum4!= "")
         {
            sPhonenumber = contact.PPPhoneNum4;
            u8Numbertype = contact.NumberType4;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

         if(contact.PPPhoneNum5!= "")
         {
            sPhonenumber = contact.PPPhoneNum5;
            u8Numbertype = contact.NumberType5;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

         if(contact.PPPhoneNum6!= "")
         {
            sPhonenumber = contact.PPPhoneNum6;
            u8Numbertype = contact.NumberType6;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

         if(contact.PPPhoneNum7!= "")
         {
            sPhonenumber = contact.PPPhoneNum7;
            u8Numbertype = contact.NumberType7;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }
         if(contact.PPPhoneNum8!= "")
         {
            sPhonenumber = contact.PPPhoneNum8;
            u8Numbertype = contact.NumberType8;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

#else
         if(contact.PhoneNum1!= "")
         {
            sPhonenumber = contact.PhoneNum1;
            u8Numbertype = contact.NumberType1;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }
         if(contact.PhoneNum2!= "")
         {
            sPhonenumber = contact.PhoneNum2;
            u8Numbertype = contact.NumberType2;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

         if(contact.PhoneNum3!= "")
         {
            sPhonenumber = contact.PhoneNum3;
            u8Numbertype = contact.NumberType3;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

         if(contact.PhoneNum4!= "")
         {
            sPhonenumber = contact.PhoneNum4;
            u8Numbertype = contact.NumberType4;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

         if(contact.PhoneNum5!= "")
         {
            sPhonenumber = contact.PhoneNum5;
            u8Numbertype = contact.NumberType5;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

         if(contact.PhoneNum6!= "")
         {
            sPhonenumber = contact.PhoneNum6;
            u8Numbertype = contact.NumberType6;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }

         if(contact.PhoneNum7!= "")
         {
            sPhonenumber = contact.PhoneNum7;
            u8Numbertype = contact.NumberType7;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }
         if(contact.PhoneNum8!= "")
         {
            sPhonenumber = contact.PhoneNum8;
            u8Numbertype = contact.NumberType8;
            FindingNumbertypeOfNumber(ContactHandle,sPhonenumber,u8Numbertype ,methodResult);
         }
#endif

      }
      break;

   case most_fi_tcl_e8_PhonBkContactDetailFilter::FI_EN_E8CDF_EMAIL:
      {

         tmpArray =  contact.emailAdd1.toUtf8();
         methodResult->oContactDetails.sEmailAddress1.bSet(tmpArray.constData());
         methodResult->oContactDetails.u32EmailAdd1Tag=
            GENERATE_TAG(ContactHandle, EMAILADD1_TAG);

         tmpArray = contact.emailAdd2.toUtf8();
         methodResult->oContactDetails.sEmailAddress2.bSet(tmpArray.constData());
         methodResult->oContactDetails.u32EmailAdd2Tag=
            GENERATE_TAG(ContactHandle, EMAILADD2_TAG);

         tmpArray = contact.emailAdd3.toUtf8();
         methodResult->oContactDetails.sEmailAddress3.bSet(tmpArray.constData());
         methodResult->oContactDetails.u32EmailAdd3Tag=
            GENERATE_TAG(ContactHandle, EMAILADD3_TAG);

      }
      break;

   case most_fi_tcl_e8_PhonBkContactDetailFilter::FI_EN_E8CDF_NOTE:
      {

         tmpArray = contact.notes.toUtf8();
         methodResult->oContactDetails.sNotes.bSet(tmpArray.constData());
         methodResult->oContactDetails.u32NotesTag=
            GENERATE_TAG(ContactHandle, NOTES_TAG);

      }
      break;

   default:
      break;

   }

   ETG_TRACE_USR4((" GetContactdetails Ends          "));

   return true;

}

/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::FindingNumbertypeOfNumber
*
* DESCRIPTION: Finding out of NumberType of a Phonenumber
*
*
*
* PARAMETER: [IN] -[OUT]
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
void FC_PhoneBook_List::FindingNumbertypeOfNumber(quint32 ContactHandle, QString sPhonenumber, quint8 u8Numbertype,most_phonbkfi_tclMsgGetContactDetailsMethodResult* methodResult)
{
   ETG_TRACE_USR4((" FindingOfNumbertypeOfNumber entered " ));
   ETG_TRACE_USR4((" u8Numbertype and sPhonenumber are %d, %s",u8Numbertype, sPhonenumber.toLatin1().constData()));
   QByteArray tmpArray = sPhonenumber.toUtf8();
   switch(u8Numbertype)
   {
      case PHONE_NUMBER_PREFERRED:
      {

         if(!methodResult->oContactDetails.sPreferredNumber.szValue)
         {
         methodResult->oContactDetails.sPreferredNumber.bSet(tmpArray.constData());
         methodResult->oContactDetails.u32PrefNumTag=
            GENERATE_TAG(ContactHandle, PREFNUMBER_TAG);
         }

         ETG_TRACE_USR4(("PHONE_NUMBER_PREFERRED EXIT" ));
      }

      break;

      case PHONE_NUMBER_CELL:
      {

         if(!methodResult->oContactDetails.sCellNumber1.szValue)
         {
            methodResult->oContactDetails.sCellNumber1.bSet(tmpArray.constData());
            methodResult->oContactDetails.u32CellNum1Tag=
            GENERATE_TAG(ContactHandle, CELLNUMBER1_TAG);

         }
         else
         {
            methodResult->oContactDetails.sCellNumber2.bSet(tmpArray.constData());
            methodResult->oContactDetails.u32CellNum2Tag=
               GENERATE_TAG(ContactHandle, CELLNUMBER2_TAG);
         }
      }

      break;

      case PHONE_NUMBER_HOME:
      {


         if(!methodResult->oContactDetails.sHomeNumber1.szValue)
         {
            methodResult->oContactDetails.sHomeNumber1.bSet(tmpArray.constData());
            methodResult->oContactDetails.u32HomeNum1Tag=
            GENERATE_TAG(ContactHandle, HOMENUMBER1_TAG);
         }

         else
         {
               methodResult->oContactDetails.sHomeNumber2.bSet(tmpArray.constData());
               methodResult->oContactDetails.u32HomeNum2Tag=
                      GENERATE_TAG(ContactHandle, HOMENUMBER2_TAG);
         }
      }
      break;

      case PHONE_NUMBER_WORK:
      {


         if(!methodResult->oContactDetails.sWorkNumber1.szValue)
         {
            methodResult->oContactDetails.sWorkNumber1.bSet(tmpArray.constData());
            methodResult->oContactDetails.u32WorkNum1Tag=
                           GENERATE_TAG(ContactHandle, WORKNUMBER1_TAG);

         }
         else
         {
            methodResult->oContactDetails.sWorkNumber2.bSet(tmpArray.constData());
            methodResult->oContactDetails.u32WorkNum2Tag=
                     GENERATE_TAG(ContactHandle, WORKNUMBER2_TAG);
         }

      }
      break;

      case PHONE_NUMBER_OTHER:
      {

         methodResult->oContactDetails.sOtherNumber.bSet(tmpArray.constData());
         methodResult->oContactDetails.u32OtherNumTag=
            GENERATE_TAG(ContactHandle, OTHERNUMBER_TAG);
      }
      break;

      default:
         ETG_TRACE_USR4((" Invalid numbertype "));

   }

}
/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::DeleteRecordsOFDevice
*
* DESCRIPTION: Delete all phonebook contacts of a device
*
*
*
* PARAMETER: [IN] -[OUT]
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_List::DeleteRecordsOFDevice(quint8 u8Devicehandle)
{
   return m_sqlite->DeleteRecordsOfDevice(u8Devicehandle);
}

//FIX CMG3G-8367 IS2424_DownloadOnOff behaviour@FC_Phonebook
// The user shall have possibility to switch the phonebook and call history list download On or Off individual for each paired phone device.
/*******************************************************************************
 *
 * FUNCTION:  FC_PhoneBook_List::bClearPBRecords
 *
 * DESCRIPTION: This function deletes the contacts from database, frees related RAM content and triggers list change update.
 *
 *
 *
 * PARAMETER: [IN] - u8Devicehandle - Device handle for which the phonebook records should be deleted
 *            [OUT]
 * RETURNVALUE: - bRetValCH - Boolean return value of the deleting phonebook records in database
 *
 *******************************************************************************/
bool FC_PhoneBook_List::bClearPBRecords(quint8 u8Devicehandle)
{
   ETG_TRACE_USR4((" bClearPBRecords entered"));
   bool bRetValPB = m_sqlite->bDeletePBContacts(u8Devicehandle);
   if (bRetValPB)
   {
      ListChange_Handler pListChange_Handler;
      pListChange_Handler.UpdateList(&m_hashPbList, enAction_Content_Refreshed, u8Devicehandle, 0, FC_PhoneBook_SQLite::ListType_PHONEBOOK);
      //Sending list change for lists of type FAVORITE as they will also get cleared from DB.
      pListChange_Handler.UpdateList(&m_hashPbList, enAction_Content_Refreshed, u8Devicehandle, 0, FC_PhoneBook_SQLite::ListType_FAVORITE);
      ETG_TRACE_USR4(("PB records deleted successfully"));
   }
   else
   {
      ETG_TRACE_USR4(("PB records deletion FAILED"));
   }
   return bRetValPB;

}
//End of fix FIX CMG3G-8367
/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::MethodResultPosted
*
* DESCRIPTION: This function is called when a method result is posted successfully
* by service
*
*
*
* PARAMETER: [IN] -[OUT]
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_List::MethodResultPosted(const guint16 u16FuntionID,
                                           const most_phonbkfi_tclMsgBaseMessage* MethodStart,
                                           const most_phonbkfi_tclMsgBaseMessage* MethodResult)
{
   ETG_TRACE_USR4(( " MethodResultPosted entered" ));
   ListChange_Handler mListChangeHandler;
   FC_PhoneBook_CallHistory_List *poCallHistoryList=
      FC_PhoneBook_CallHistory_List::GetCurrentInstance();
   FC_PhoneBook_Contact_Detail Contact;

   switch(u16FuntionID)
   {
   case MOST_PHONBKFI_C_U16_ADDCONTACT:
      {
         ETG_TRACE_USR4(( " MethodResultPosted MOST_PHONBKFI_C_U16_ADDCONTACT" ));
         const most_phonbkfi_tclMsgAddContactMethodStart *MethodStartSpecific
            = static_cast<const most_phonbkfi_tclMsgAddContactMethodStart*>(MethodStart);
         const most_phonbkfi_tclMsgAddContactMethodResult *MethodResultSpecific
            = static_cast<const most_phonbkfi_tclMsgAddContactMethodResult*>(MethodResult);

         // -bn: MY15 Feature: VPB Import: Code addition: BEGIN
         // Code modification for "bulk" import of Vehicle Contact data, which is added via 'AddContact' per each contact.
         if(PHONEBOOK_C_U32_VEHICLE_CONTACT_IMPORT_SILENT_ADD_CONTACT == MethodStartSpecific->oContactDetails.u32FirstNameTag )  // 0xFA000000
         {
            // This is a "silent" 'AddContact' to the VPB, i.e. without 'ListChange' status to clients and also without update of RAM lists.
         }
         else if(PHONEBOOK_C_U32_VEHICLE_CONTACT_IMPORT_COMPLETED == MethodStartSpecific->oContactDetails.u32FirstNameTag )      // 0xFC000000
         {
            // VPB contact list in DB has changed as import has been completed.

            // Update the checksum table for VPB in DB.
            m_sqlite->SpeechUpdateCheckSumTable(MethodStartSpecific->u8DeviceHandle);

            // Send out 'ListChange.Status' for VPB to clients.
            mListChangeHandler.UpdateList(&m_hashPbList, enAction_Content_Refreshed,
               MethodStartSpecific->u8DeviceHandle, 0, FC_PhoneBook_SQLite::ListType_VEHICLE );

            // Update Call History list based on new available contact details in VPB.
            // TODO: Here we need a new implementation as we have added now many contacts to VPB.
            // Note: At the moment the Call History list is not updated by using the new contacts.
         }
         else  // Normal insertion of a single VPB contact.
         {
            Contact.contactHandle= MethodResultSpecific->u32ContactHandle;

            Contact.firstName= QString::fromUtf8(MethodStartSpecific->oContactDetails.sFirstName.szValue);
            Contact.firstName.truncate(FC_PB_REPORTED_NAME_MAX_LEN);
            ETG_TRACE_USR2(("Contact.firstName - %s", Contact.firstName.toUtf8().constData()));

            Contact.lastName= QString::fromUtf8(MethodStartSpecific->oContactDetails.sLastName.szValue);
            Contact.lastName.truncate(FC_PB_REPORTED_NAME_MAX_LEN);
            ETG_TRACE_USR2(("Contact.lastName - %s", Contact.lastName.toUtf8().constData()));

            Contact.poiName= QString::fromUtf8(MethodStartSpecific->oContactDetails.sPOIName.szValue);
            Contact.poiName.truncate(FC_PB_REPORTED_POI_MAIL_MAX_LEN);
            ETG_TRACE_USR2(("Contact.poiName - %s", Contact.poiName.toUtf8().constData()));

            if (MethodStartSpecific->oContactDetails.sPreferredNumber.szValue)
            {
               Contact.PPPhoneNum1 = QString::fromUtf8(MethodStartSpecific->oContactDetails.sPreferredNumber.szValue);
               Contact.PhoneNum1 = StripTelephoneNumber(MethodStartSpecific->oContactDetails.sPreferredNumber.szValue);
               Contact.NumberType1 = PHONE_NUMBER_PREFERRED;
            }

            if (MethodStartSpecific->oContactDetails.sCellNumber1.szValue)
            {
               Contact.PPPhoneNum2 = QString::fromUtf8(MethodStartSpecific->oContactDetails.sCellNumber1.szValue);
               Contact.PhoneNum2 = StripTelephoneNumber(MethodStartSpecific->oContactDetails.sCellNumber1.szValue);
               Contact.NumberType2 = PHONE_NUMBER_CELL;
            }

            if (MethodStartSpecific->oContactDetails.sCellNumber2.szValue)
            {
               Contact.PPPhoneNum3 = QString::fromUtf8(MethodStartSpecific->oContactDetails.sCellNumber2.szValue);
               Contact.PhoneNum3 = StripTelephoneNumber(MethodStartSpecific->oContactDetails.sCellNumber2.szValue);
               Contact.NumberType3 = PHONE_NUMBER_CELL;
            }

            if (MethodStartSpecific->oContactDetails.sHomeNumber1.szValue)
            {
               Contact.PPPhoneNum4 = QString::fromUtf8(MethodStartSpecific->oContactDetails.sHomeNumber1.szValue);
               Contact.PhoneNum4 = StripTelephoneNumber(MethodStartSpecific->oContactDetails.sHomeNumber1.szValue);
               Contact.NumberType4 = PHONE_NUMBER_HOME;
            }

            if (MethodStartSpecific->oContactDetails.sHomeNumber2.szValue)
            {
               Contact.PPPhoneNum5 = QString::fromUtf8(MethodStartSpecific->oContactDetails.sHomeNumber2.szValue);
               Contact.PhoneNum5 = StripTelephoneNumber(MethodStartSpecific->oContactDetails.sHomeNumber2.szValue);
               Contact.NumberType5 = PHONE_NUMBER_HOME;
            }

            if (MethodStartSpecific->oContactDetails.sWorkNumber1.szValue)
            {
               Contact.PPPhoneNum6 = QString::fromUtf8(MethodStartSpecific->oContactDetails.sWorkNumber1.szValue);
               Contact.PhoneNum6 = StripTelephoneNumber(MethodStartSpecific->oContactDetails.sWorkNumber1.szValue);
               Contact.NumberType6 = PHONE_NUMBER_WORK;
            }

            if (MethodStartSpecific->oContactDetails.sWorkNumber2.szValue)
            {
               Contact.PPPhoneNum7 = QString::fromUtf8(MethodStartSpecific->oContactDetails.sWorkNumber2.szValue);
               Contact.PhoneNum7 = StripTelephoneNumber(MethodStartSpecific->oContactDetails.sWorkNumber2.szValue);
               Contact.NumberType7 = PHONE_NUMBER_WORK;
            }

            if (MethodStartSpecific->oContactDetails.sOtherNumber.szValue)
            {
               Contact.PPPhoneNum8 = QString::fromUtf8(MethodStartSpecific->oContactDetails.sOtherNumber.szValue);
               Contact.PhoneNum8 = StripTelephoneNumber(MethodStartSpecific->oContactDetails.sOtherNumber.szValue);
               Contact.NumberType8 = PHONE_NUMBER_OTHER;
            }

            ETG_TRACE_USR2(("Contact.PhoneNum8 - %s", Contact.PhoneNum8.toUtf8().constData()));
            ETG_TRACE_USR2(("Contact.NumberType8 - %d", Contact.NumberType8));

            // -bn: There is no need to add the substring numbers 'Contact.NumberSubstr1' .. 'Contact.NumberSubstr8' to the
            //      'Contact' variable as they are not evaluated in the later method call 'poCallHistoryList->RefreshCallHistoryOnChangeInContact(Contact)'
            
            Contact.emailAdd1= QString::fromUtf8(MethodStartSpecific->oContactDetails.sEmailAddress1.szValue);
            Contact.emailAdd1.truncate(FC_PB_REPORTED_POI_MAIL_MAX_LEN);
            ETG_TRACE_USR2(("Contact.emailAdd1 - %s", Contact.emailAdd1.toUtf8().constData()));

            Contact.emailAdd2= QString::fromUtf8(MethodStartSpecific->oContactDetails.sEmailAddress2.szValue);
            Contact.emailAdd2.truncate(FC_PB_REPORTED_POI_MAIL_MAX_LEN);
            ETG_TRACE_USR2(("Contact.emailAdd2 - %s", Contact.emailAdd2.toUtf8().constData()));

            Contact.emailAdd3= QString::fromUtf8(MethodStartSpecific->oContactDetails.sEmailAddress3.szValue);
            Contact.emailAdd3.truncate(FC_PB_REPORTED_POI_MAIL_MAX_LEN);
            ETG_TRACE_USR2(("Contact.emailAdd3 - %s", Contact.emailAdd3.toUtf8().constData()));

            Contact.homeAdd= QString::fromUtf8(MethodStartSpecific->oContactDetails.sHomeAddress.szValue);
            Contact.homeAdd.truncate(FC_PB_REPORTED_ADDRESS_MAX_LEN);
            ETG_TRACE_USR2(("Contact.homeAdd - %s", Contact.homeAdd.toUtf8().constData()));

            Contact.workAdd= QString::fromUtf8(MethodStartSpecific->oContactDetails.sWorkAddress.szValue);
            Contact.workAdd.truncate(FC_PB_REPORTED_WORK_ADDRESS_MAX_LEN);
            ETG_TRACE_USR2(("Contact.workAdd - %s", Contact.workAdd.toUtf8().constData()));

            Contact.otherAdd= QString::fromUtf8(MethodStartSpecific->oContactDetails.sOtherAddress.szValue);
            Contact.otherAdd.truncate(FC_PB_REPORTED_ADDRESS_MAX_LEN);
            ETG_TRACE_USR2(("Contact.otherAdd - %s", Contact.otherAdd.toUtf8().constData()));

            Contact.category= QString::fromUtf8(MethodStartSpecific->oContactDetails.sCategory.szValue);
            Contact.category.truncate(FC_PB_REPORTED_NUMBER_MAX_LEN);
            ETG_TRACE_USR2(("Contact.category - %s", Contact.category.toUtf8().constData()));

            Contact.notes= QString::fromUtf8(MethodStartSpecific->oContactDetails.sNotes.szValue);
            Contact.notes.truncate(FC_PB_REPORTED_NOTE_MAX_LEN);
            ETG_TRACE_USR2(("Contact.notes - %s", Contact.notes.toUtf8().constData()));

            Contact.homeGeoCodeLatLongValidity=	MethodStartSpecific->oContactDetails.oHomeAddressLatLong.bLatLongValidity;
            ETG_TRACE_USR2(("Contact.homeGeoCodeLatLongValidity - %d", Contact.homeGeoCodeLatLongValidity));

            if(((unsigned)MethodStartSpecific->oContactDetails.oHomeAddressLatLong.s32Latitude >= FC_PB_REPORTED_LAT_MIN_RANG) && ((unsigned)MethodStartSpecific->oContactDetails.oHomeAddressLatLong.s32Latitude <= FC_PB_REPORTED_LAT_MAX_RANG))
            {
               Contact.homeGeoCodeLatitude= MethodStartSpecific->oContactDetails.oHomeAddressLatLong.s32Latitude;
            }
            ETG_TRACE_USR2(("Contact.homeGeoCodeLatitude - %d", Contact.homeGeoCodeLatitude));

            if(((unsigned)MethodStartSpecific->oContactDetails.oHomeAddressLatLong.s32Longitude >= FC_PB_REPORTED_LONG_MIN_RANG) && ((unsigned)MethodStartSpecific->oContactDetails.oHomeAddressLatLong.s32Longitude <= FC_PB_REPORTED_LONG_MAX_RANG))
            {
               Contact.homeGeocodeLongitude= MethodStartSpecific->oContactDetails.oHomeAddressLatLong.s32Longitude;
            }
            ETG_TRACE_USR2(("Contact.homeGeocodeLongitude - %d", Contact.homeGeocodeLongitude));

            Contact.workGeoCodeLatLongValidity=	MethodStartSpecific->oContactDetails.oWorkAddressLatLong.bLatLongValidity;
            ETG_TRACE_USR2(("Contact.workGeoCodeLatLongValidity - %d", Contact.workGeoCodeLatLongValidity));

            if(((unsigned)MethodStartSpecific->oContactDetails.oWorkAddressLatLong.s32Latitude >= FC_PB_REPORTED_LAT_MIN_RANG) && ((unsigned)MethodStartSpecific->oContactDetails.oWorkAddressLatLong.s32Latitude <= FC_PB_REPORTED_LAT_MAX_RANG))
            {
               Contact.workGeoCodeLatitude= MethodStartSpecific->oContactDetails.oWorkAddressLatLong.s32Latitude;
            }
            ETG_TRACE_USR2(("Contact.workGeoCodeLatitude - %d", Contact.workGeoCodeLatitude));

            if(((unsigned)MethodStartSpecific->oContactDetails.oWorkAddressLatLong.s32Longitude >= FC_PB_REPORTED_LONG_MIN_RANG) && ((unsigned)MethodStartSpecific->oContactDetails.oWorkAddressLatLong.s32Longitude <= FC_PB_REPORTED_LONG_MAX_RANG))
            {
               Contact.workGeocodeLongitude= MethodStartSpecific->oContactDetails.oWorkAddressLatLong.s32Longitude;
            }
            ETG_TRACE_USR2(("Contact.workGeocodeLongitude - %d", Contact.workGeocodeLongitude));

            Contact.otherGeoCodeLatLongValidity= MethodStartSpecific->oContactDetails.oOtherAddressLatLong.bLatLongValidity;
            ETG_TRACE_USR2(("Contact.otherGeoCodeLatLongValidity - %d", Contact.otherGeoCodeLatLongValidity));

            if(((unsigned)MethodStartSpecific->oContactDetails.oOtherAddressLatLong.s32Latitude >= FC_PB_REPORTED_LAT_MIN_RANG) && ((unsigned)MethodStartSpecific->oContactDetails.oOtherAddressLatLong.s32Latitude <= FC_PB_REPORTED_LAT_MAX_RANG))
            {
               Contact.otherGeoCodeLatitude= MethodStartSpecific->oContactDetails.oOtherAddressLatLong.s32Latitude;
            }
            ETG_TRACE_USR2(("Contact.otherGeoCodeLatitude - %d", Contact.otherGeoCodeLatitude));

            if(((unsigned)MethodStartSpecific->oContactDetails.oOtherAddressLatLong.s32Longitude >= FC_PB_REPORTED_LONG_MIN_RANG) && ((unsigned)MethodStartSpecific->oContactDetails.oOtherAddressLatLong.s32Longitude <= FC_PB_REPORTED_LONG_MAX_RANG))
            {
               Contact.otherGeocodeLongitude= MethodStartSpecific->oContactDetails.oOtherAddressLatLong.s32Longitude;
            }
            ETG_TRACE_USR2(("Contact.otherGeocodeLongitude - %d", Contact.otherGeocodeLongitude));


            // VPB contact has changed by user interaction via HMI. Update the checksum table, send 'ListChange' and refresh CallHistory.
            m_sqlite->SpeechUpdateCheckSumTable(MethodStartSpecific->u8DeviceHandle);

            mListChangeHandler.UpdateList(&m_hashPbList, enAction_Contact_Added,
               MethodStartSpecific->u8DeviceHandle,
               MethodResultSpecific->u32ContactHandle,
               FC_PhoneBook_SQLite::ListType_PHONEBOOK);

            poCallHistoryList->RefreshCallHistoryOnChangeInContact(Contact);//FIX GMMY15-11882 Replacing multiple similar functions with single one
         }
         // -bn: MY15 Feature: VPB Import: Code addition: END
      }
      break;

   case MOST_PHONBKFI_C_U16_DELETECONTACT:
      {
         ETG_TRACE_USR4(( " MethodResultPosted MOST_PHONBKFI_C_U16_DELETECONTACT" ));
         const most_phonbkfi_tclMsgDeleteContactMethodStart *MethodStartSpecific
            = static_cast<const most_phonbkfi_tclMsgDeleteContactMethodStart*>(MethodStart);

         // VPB contact has changed. Update the checksum table
         if (FC_PB_VEHICLE_PBOOK == MethodStartSpecific->u8DeviceHandle)
         {
            m_sqlite->SpeechUpdateCheckSumTable(MethodStartSpecific->u8DeviceHandle);
         }
         else
         {
            m_sqlite->SpeechUpdateCheckSumTable(MethodStartSpecific->u8DeviceHandle, true);
         }

         if(FC_PB_VEHICLE_PBOOK == MethodStartSpecific->u8DeviceHandle)
         {
            //For VPB with device handle = 1, the list type is ListType_PHONEBOOK
            mListChangeHandler.UpdateList(&m_hashPbList, enAction_Contact_Deleted,
               MethodStartSpecific->u8DeviceHandle, MethodStartSpecific->u32ContactHandle,
               FC_PhoneBook_SQLite::ListType_PHONEBOOK);
         }
         else
         {
            //For VPB 2 to 11, the list type is ListType_VEHICLE
            mListChangeHandler.UpdateList(&m_hashPbList, enAction_Contact_Deleted,
               MethodStartSpecific->u8DeviceHandle, MethodStartSpecific->u32ContactHandle,
               FC_PhoneBook_SQLite::ListType_VEHICLE);
         }

         //We don't have to delete the Entry from Call History but to Update/Edit
         Contact.contactHandle= MethodStartSpecific->u32ContactHandle;
         Contact.firstName= "";
         Contact.lastName= "";
         poCallHistoryList->RefreshCallHistoryOnChangeInContact(Contact);//FIX GMMY15-11882 Replacing multiple similar functions with single one

         m_poFavoritesProvider->RefreshFavoritesOnDeleteContact(MethodStartSpecific->u32ContactHandle);
      }
      break;

   case MOST_PHONBKFI_C_U16_EDITCONTACT:
      {
         ETG_TRACE_USR4(( " MethodResultPosted MOST_PHONBKFI_C_U16_EDITCONTACT" ));
         const most_phonbkfi_tclMsgEditContactMethodStart *MethodStartSpecific
            = static_cast<const most_phonbkfi_tclMsgEditContactMethodStart*>(MethodStart);

         Contact.contactHandle= MethodStartSpecific->u32ContactHandle;
         //Fix for GMMY15-11538:Garbled code is displayed in recent calls list (Chinese HMI)
         Contact.firstName= QString::fromUtf8(MethodStartSpecific->oContactDetails.sFirstName.szValue);
         Contact.lastName=  QString::fromUtf8(MethodStartSpecific->oContactDetails.sLastName.szValue);
         Contact.poiName=   QString::fromUtf8(MethodStartSpecific->oContactDetails.sPOIName.szValue);
         if (MethodStartSpecific->oContactDetails.sPreferredNumber.szValue)
         {
            Contact.PPPhoneNum1 = QString::fromUtf8(MethodStartSpecific->oContactDetails.sPreferredNumber.szValue);
            Contact.PhoneNum1 = StripTelephoneNumber(MethodStartSpecific->oContactDetails.sPreferredNumber.szValue);
            Contact.NumberType1 = PHONE_NUMBER_PREFERRED;
         }

         if (MethodStartSpecific->oContactDetails.sCellNumber1.szValue)
         {
            Contact.PPPhoneNum2 = QString::fromUtf8(MethodStartSpecific->oContactDetails.sCellNumber1.szValue);
            Contact.PhoneNum2 = StripTelephoneNumber(MethodStartSpecific->oContactDetails.sCellNumber1.szValue);
            Contact.NumberType2 = PHONE_NUMBER_CELL;
         }

         if (MethodStartSpecific->oContactDetails.sCellNumber2.szValue)
         {
            Contact.PPPhoneNum3 = QString::fromUtf8(MethodStartSpecific->oContactDetails.sCellNumber2.szValue);
            Contact.PhoneNum3  = StripTelephoneNumber(MethodStartSpecific->oContactDetails.sCellNumber2.szValue);
            Contact.NumberType3 = PHONE_NUMBER_CELL;
         }

         if (MethodStartSpecific->oContactDetails.sHomeNumber1.szValue)
         {
            Contact.PPPhoneNum4 = QString::fromUtf8(MethodStartSpecific->oContactDetails.sHomeNumber1.szValue);
            Contact.PhoneNum4 = StripTelephoneNumber(MethodStartSpecific->oContactDetails.sHomeNumber1.szValue);
            Contact.NumberType4 = PHONE_NUMBER_HOME;
         }

         if (MethodStartSpecific->oContactDetails.sHomeNumber2.szValue)
         {
            Contact.PPPhoneNum5 = QString::fromUtf8(MethodStartSpecific->oContactDetails.sHomeNumber2.szValue);
            Contact.PhoneNum5 = StripTelephoneNumber(MethodStartSpecific->oContactDetails.sHomeNumber2.szValue);
            Contact.NumberType5 = PHONE_NUMBER_HOME;
         }

         if (MethodStartSpecific->oContactDetails.sWorkNumber1.szValue)
         {
            Contact.PPPhoneNum6 = QString::fromUtf8(MethodStartSpecific->oContactDetails.sWorkNumber1.szValue);
            Contact.PhoneNum6 = StripTelephoneNumber(MethodStartSpecific->oContactDetails.sWorkNumber1.szValue);
            Contact.NumberType6 = PHONE_NUMBER_WORK;
         }

         if (MethodStartSpecific->oContactDetails.sWorkNumber2.szValue)
         {
            Contact.PPPhoneNum7 = QString::fromUtf8(MethodStartSpecific->oContactDetails.sWorkNumber2.szValue);
            Contact.PhoneNum7 = StripTelephoneNumber(MethodStartSpecific->oContactDetails.sWorkNumber2.szValue);
            Contact.NumberType7 = PHONE_NUMBER_WORK;
         }

         if (MethodStartSpecific->oContactDetails.sOtherNumber.szValue)
         {
            Contact.PPPhoneNum8 = QString::fromUtf8(MethodStartSpecific->oContactDetails.sOtherNumber.szValue);
            Contact.PhoneNum8 = StripTelephoneNumber(MethodStartSpecific->oContactDetails.sOtherNumber.szValue);
            Contact.NumberType8 = PHONE_NUMBER_OTHER;
         }

         Contact.emailAdd1= QString::fromUtf8(MethodStartSpecific->oContactDetails.sEmailAddress1.szValue);
         Contact.emailAdd2= QString::fromUtf8(MethodStartSpecific->oContactDetails.sEmailAddress2.szValue);
         Contact.emailAdd3= QString::fromUtf8(MethodStartSpecific->oContactDetails.sEmailAddress3.szValue);
         Contact.homeAdd=   QString::fromUtf8(MethodStartSpecific->oContactDetails.sHomeAddress.szValue);
         Contact.workAdd=   QString::fromUtf8(MethodStartSpecific->oContactDetails.sWorkAddress.szValue);
         Contact.otherAdd=  QString::fromUtf8(MethodStartSpecific->oContactDetails.sOtherAddress.szValue);
         Contact.category=  QString::fromUtf8(MethodStartSpecific->oContactDetails.sCategory.szValue);
         Contact.notes= MethodStartSpecific->oContactDetails.sNotes.szValue;
         Contact.homeGeoCodeLatLongValidity	=	MethodStartSpecific->oContactDetails.oHomeAddressLatLong.bLatLongValidity;
         Contact.homeGeoCodeLatitude			=	MethodStartSpecific->oContactDetails.oHomeAddressLatLong.s32Latitude;
         Contact.homeGeocodeLongitude		=	MethodStartSpecific->oContactDetails.oHomeAddressLatLong.s32Longitude;
         Contact.workGeoCodeLatLongValidity	=	MethodStartSpecific->oContactDetails.oWorkAddressLatLong.bLatLongValidity;
         Contact.workGeoCodeLatitude			=	MethodStartSpecific->oContactDetails.oWorkAddressLatLong.s32Latitude;
         Contact.workGeocodeLongitude		=	MethodStartSpecific->oContactDetails.oWorkAddressLatLong.s32Longitude;
         Contact.otherGeoCodeLatLongValidity	=	MethodStartSpecific->oContactDetails.oOtherAddressLatLong.bLatLongValidity;
         Contact.otherGeoCodeLatitude		=	MethodStartSpecific->oContactDetails.oOtherAddressLatLong.s32Latitude;
         Contact.otherGeocodeLongitude		=	MethodStartSpecific->oContactDetails.oOtherAddressLatLong.s32Longitude;
         // VPB contact has changed. Update the checksum table
         m_sqlite->SpeechUpdateCheckSumTable(MethodStartSpecific->u8DeviceHandle);
         mListChangeHandler.UpdateList(&m_hashPbList, enAction_Contact_Edited,
            MethodStartSpecific->u8DeviceHandle, MethodStartSpecific->u32ContactHandle,
            FC_PhoneBook_SQLite::ListType_PHONEBOOK);

         poCallHistoryList->RefreshCallHistoryOnChangeInContact(Contact);//FIX GMMY15-11882 Replacing multiple similar functions with single one
         m_poFavoritesProvider->RefreshFavoritesOnEditContact(Contact);
      }
      break;

   case MOST_PHONBKFI_C_U16_CREATEPHONEBOOKLIST:
      {
         ETG_TRACE_USR4(( " MethodResultPosted MOST_PHONBKFI_C_U16_CREATEPHONEBOOKLIST" ));
         const most_phonbkfi_tclMsgCreatePhoneBookListMethodStart *MethodStartSpecific
            = static_cast<const most_phonbkfi_tclMsgCreatePhoneBookListMethodStart*>(MethodStart);
         const most_phonbkfi_tclMsgCreatePhoneBookListMethodResult *MethodResultSpecific
            = static_cast<const most_phonbkfi_tclMsgCreatePhoneBookListMethodResult*>(MethodResult);

         quint8 u8DeviceHandle= MethodStartSpecific->u8DeviceHandle;
         //Dummy list change reqd for Vehicle Phonebook too
         if(FC_PB_VEHICLE_PBOOK == u8DeviceHandle)
         {
            ETG_TRACE_USR2(("Triggering dummy list change for list handle - %d", MethodResultSpecific->u16ListHandle));
            mListChangeHandler.TriggerDummyListChange(&m_hashPbList,
               MethodResultSpecific->u16ListHandle);
         }
         else
         {
            ETG_TRACE_USR4(("Phonebook list created for a device. Dummy list change required"));

            FC_Device_Details* DevDetails= FC_Device_Details::deviceHandleToObjectInstance(
               u8DeviceHandle);
            if(DevDetails)
            {
               if(FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS != DevDetails->GetDownloadState())
               {  //Send a list change only if the download is not in progress

                  ETG_TRACE_USR2(("Triggering dummy list change for list handle - %d", MethodResultSpecific->u16ListHandle));

                  mListChangeHandler.TriggerDummyListChange(&m_hashPbList,
                     MethodResultSpecific->u16ListHandle);
               }
               else
               {
                  ETG_TRACE_USR2(("No dummy list changes as download is in progess"));
               }
            }
            else
            {
               ETG_TRACE_ERR(("Device Object is NULL for device handle- %d", u8DeviceHandle));
            }
         }
      }
      break;
   case MOST_PHONBKFI_C_U16_CREATECONTACTLIST:
      {
         ETG_TRACE_USR4(( " MethodResultPosted MOST_PHONBKFI_C_U16_CREATECONTACTLIST" ));
         const most_phonbkfi_tclMsgCreateContactListMethodStart *MethodStartSpecific
            = static_cast<const most_phonbkfi_tclMsgCreateContactListMethodStart*>(MethodStart);
         const most_phonbkfi_tclMsgCreateContactListMethodResult *MethodResultSpecific
            = static_cast<const most_phonbkfi_tclMsgCreateContactListMethodResult*>(MethodResult);
         quint8 u8DeviceHandle= MethodStartSpecific->u8DeviceHandle;
         if(FC_PB_VEHICLE_PBOOK == u8DeviceHandle)
         {
            ETG_TRACE_USR2(("Triggering dummy list change for list handle - %d", MethodResultSpecific->u16ListHandle));
            mListChangeHandler.TriggerDummyListChange(&m_hashPbList,
               MethodResultSpecific->u16ListHandle);
         }
         else
         {
            ETG_TRACE_USR4(("Phonebook list created for a device. Dummy list change required"));
            FC_Device_Details* DevDetails= FC_Device_Details::deviceHandleToObjectInstance(
               u8DeviceHandle);
            if(DevDetails)
            {
               if(FC_Device_Details::DOWNLOAD_ST_PBK_PROGRESS != DevDetails->GetDownloadState())
               {  //Send a list change only if the download is not in progress
                  ETG_TRACE_USR2(("Triggering dummy list change for list handle - %d", MethodResultSpecific->u16ListHandle));

                  // Fix for NCG3D-159329: In "phone probably locked" state, the method TriggerDummyListChange(..) will NOT be 
                  // called anymore for LIST_TYPE_VEHICLE (Quick-Dial) Lists. Otherwise, in method TriggerDummyListChange(..) the 
                  // list size would be set zero and this is not wanted. 
                  // User shall be able to access the Quick-Dial Lists in the situation that the phone is "probably" locked.
                  if( (VALUE_TWO != MethodStartSpecific->e8PhoneBkContactListType.enType) && (TRUE == DevDetails->bIsDeviceProbablyLocked()) )
                  {
                     mListChangeHandler.TriggerDummyListChange(&m_hashPbList, MethodResultSpecific->u16ListHandle);
                  }
               }
               else
               {
                  ETG_TRACE_USR2(("No dummy list changes as download is in progess"));
               }
            }
            else
            {
               ETG_TRACE_ERR(("Device Object is NULL for device handle- %d", u8DeviceHandle));
            }
         }
      }
      break;
   case MOST_PHONBKFI_C_U16_SETCONTACTPOSITION:
      {
         ETG_TRACE_USR4((" MethodResultPosted MOST_PHONBKFI_C_U16_SETCONTACTPOSITION"));
         const most_phonbkfi_tclMsgSetContactPositionMethodResult
               *MethodResultSpecific =
                     static_cast<const most_phonbkfi_tclMsgSetContactPositionMethodResult*> (MethodResult);
         quint8 u8DeviceHandle= m_sqlite->GetDeviceHandleForContacthandle(MethodResultSpecific->u32ContactHandle);
         mListChangeHandler.UpdateList(&m_hashPbList, enAction_Content_Refreshed, u8DeviceHandle, MethodResultSpecific->u32ContactHandle, FC_PhoneBook_SQLite::ListType_VEHICLE);
      }
      break;
   case MOST_PHONBKFI_C_U16_SETCONTACTPHOTO:
      {
         ETG_TRACE_USR4((" MethodResultPosted MOST_PHONBKFI_C_U16_SETCONTACTPHOTO"));
         const most_phonbkfi_tclMsgSetContactPhotoMethodResult
               *MethodResultSpecific =
                     static_cast<const most_phonbkfi_tclMsgSetContactPhotoMethodResult*> (MethodResult);
         quint8 u8DeviceHandle= m_sqlite->GetDeviceHandleForContacthandle(MethodResultSpecific->u32ContactHandle);
         mListChangeHandler.UpdateList(&m_hashPbList, enAction_Content_Refreshed, u8DeviceHandle, MethodResultSpecific->u32ContactHandle, FC_PhoneBook_SQLite::ListType_VEHICLE);
      }
      break;
   case MOST_PHONBKFI_C_U16_ADDCONTACTEXTENDED:
         {
            ETG_TRACE_USR4(( " MethodResultPosted MOST_PHONBKFI_C_U16_ADDCONTACTEXTENDED" ));
            const most_phonbkfi_tclMsgAddContactExtendedMethodStart *MethodStartSpecific
               = static_cast<const most_phonbkfi_tclMsgAddContactExtendedMethodStart*>(MethodStart);
            const most_phonbkfi_tclMsgAddContactExtendedMethodResult *MethodResultSpecific
               = static_cast<const most_phonbkfi_tclMsgAddContactExtendedMethodResult*>(MethodResult);

               Contact.contactHandle= MethodResultSpecific->u32ContactHandle;

               Contact.firstName= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.sFirstName.szValue);
               Contact.firstName.truncate(FC_PB_REPORTED_NAME_MAX_LEN);
               ETG_TRACE_USR2(("Contact.firstName - %s", Contact.firstName.toUtf8().constData()));

               Contact.lastName= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.sLastName.szValue);
               Contact.lastName.truncate(FC_PB_REPORTED_NAME_MAX_LEN);
               ETG_TRACE_USR2(("Contact.lastName - %s", Contact.lastName.toUtf8().constData()));

               Contact.poiName= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.sPOIName.szValue);
               Contact.poiName.truncate(FC_PB_REPORTED_POI_MAIL_MAX_LEN);
               ETG_TRACE_USR2(("Contact.poiName - %s", Contact.poiName.toUtf8().constData()));

               Contact.PhoneNum1= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.oPhoneNumber1.sNumber.szValue);
               Contact.PhoneNum1.truncate(FC_PB_REPORTED_NUMBER_MAX_LEN);
               ETG_TRACE_USR2(("Contact.PhoneNum1 - %s", Contact.PhoneNum1.toUtf8().constData()));

               Contact.PhoneNum2= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.oPhoneNumber2.sNumber.szValue);
               Contact.PhoneNum2.truncate(FC_PB_REPORTED_NUMBER_MAX_LEN);
               ETG_TRACE_USR2(("Contact.PhoneNum2 - %s", Contact.PhoneNum2.toUtf8().constData()));

               Contact.PhoneNum3= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.oPhoneNumber3.sNumber.szValue);
               Contact.PhoneNum3.truncate(FC_PB_REPORTED_NUMBER_MAX_LEN);
               ETG_TRACE_USR2(("Contact.PhoneNum3 - %s", Contact.PhoneNum3.toUtf8().constData()));

               Contact.PhoneNum4= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.oPhoneNumber4.sNumber.szValue);
               Contact.PhoneNum4.truncate(FC_PB_REPORTED_NUMBER_MAX_LEN);
               ETG_TRACE_USR2(("Contact.PhoneNum4 - %s", Contact.PhoneNum4.toUtf8().constData()));

               Contact.PhoneNum5= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.oPhoneNumber5.sNumber.szValue);
               Contact.PhoneNum5.truncate(FC_PB_REPORTED_NUMBER_MAX_LEN);
               ETG_TRACE_USR2(("Contact.PhoneNum5 - %s", Contact.PhoneNum5.toUtf8().constData()));

               Contact.PhoneNum6= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.oPhoneNumber6.sNumber.szValue);
               Contact.PhoneNum6.truncate(FC_PB_REPORTED_NUMBER_MAX_LEN);
               ETG_TRACE_USR2(("Contact.PhoneNum6 - %s", Contact.PhoneNum6.toUtf8().constData()));

               Contact.PhoneNum7= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.oPhoneNumber7.sNumber.szValue);
               Contact.PhoneNum7.truncate(FC_PB_REPORTED_NUMBER_MAX_LEN);
               ETG_TRACE_USR2(("Contact.PhoneNum7 - %s", Contact.PhoneNum7.toUtf8().constData()));

               Contact.PhoneNum8= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.oPhoneNumber8.sNumber.szValue);
               Contact.PhoneNum8.truncate(FC_PB_REPORTED_NUMBER_MAX_LEN);
               ETG_TRACE_USR2(("Contact.PhoneNum8 - %s", Contact.PhoneNum8.toUtf8().constData()));

               Contact.emailAdd1= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.sEmailAddress1.szValue);
               Contact.emailAdd1.truncate(FC_PB_REPORTED_POI_MAIL_MAX_LEN);
               ETG_TRACE_USR2(("Contact.emailAdd1 - %s", Contact.emailAdd1.toUtf8().constData()));

               Contact.emailAdd2= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.sEmailAddress2.szValue);
               Contact.emailAdd2.truncate(FC_PB_REPORTED_POI_MAIL_MAX_LEN);
               ETG_TRACE_USR2(("Contact.emailAdd2 - %s", Contact.emailAdd2.toUtf8().constData()));

               Contact.emailAdd3= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.sEmailAddress3.szValue);
               Contact.emailAdd3.truncate(FC_PB_REPORTED_POI_MAIL_MAX_LEN);
               ETG_TRACE_USR2(("Contact.emailAdd3 - %s", Contact.emailAdd3.toUtf8().constData()));

               Contact.homeAdd= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.sHomeAddress.szValue);
               Contact.homeAdd.truncate(FC_PB_REPORTED_ADDRESS_MAX_LEN);
               ETG_TRACE_USR2(("Contact.homeAdd - %s", Contact.homeAdd.toUtf8().constData()));

               Contact.workAdd= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.sWorkAddress.szValue);
               Contact.workAdd.truncate(FC_PB_REPORTED_WORK_ADDRESS_MAX_LEN);
               ETG_TRACE_USR2(("Contact.workAdd - %s", Contact.workAdd.toUtf8().constData()));

               Contact.otherAdd= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.sOtherAddress.szValue);
               Contact.otherAdd.truncate(FC_PB_REPORTED_ADDRESS_MAX_LEN);
               ETG_TRACE_USR2(("Contact.otherAdd - %s", Contact.otherAdd.toUtf8().constData()));

               Contact.category= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.sCategory.szValue);
               Contact.category.truncate(FC_PB_REPORTED_NUMBER_MAX_LEN);
               ETG_TRACE_USR2(("Contact.category - %s", Contact.category.toUtf8().constData()));

               Contact.notes= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.sNotes.szValue);
               Contact.notes.truncate(FC_PB_REPORTED_NOTE_MAX_LEN);
               ETG_TRACE_USR2(("Contact.notes - %s", Contact.notes.toUtf8().constData()));

               Contact.homeGeoCodeLatLongValidity= MethodStartSpecific->oContactDetailsExtended.oHomeAddressLatLong.bLatLongValidity;
               ETG_TRACE_USR2(("Contact.homeGeoCodeLatLongValidity - %d", Contact.homeGeoCodeLatLongValidity));

               if(((unsigned)MethodStartSpecific->oContactDetailsExtended.oHomeAddressLatLong.s32Latitude >= FC_PB_REPORTED_LAT_MIN_RANG) && ((unsigned)MethodStartSpecific->oContactDetailsExtended.oHomeAddressLatLong.s32Latitude <= FC_PB_REPORTED_LAT_MAX_RANG))
               {
                  Contact.homeGeoCodeLatitude= MethodStartSpecific->oContactDetailsExtended.oHomeAddressLatLong.s32Latitude;
               }
               ETG_TRACE_USR2(("Contact.homeGeoCodeLatitude - %d", Contact.homeGeoCodeLatitude));

               if(((unsigned)MethodStartSpecific->oContactDetailsExtended.oHomeAddressLatLong.s32Longitude >= FC_PB_REPORTED_LONG_MIN_RANG) && ((unsigned)MethodStartSpecific->oContactDetailsExtended.oHomeAddressLatLong.s32Longitude <= FC_PB_REPORTED_LONG_MAX_RANG))
               {
                  Contact.homeGeocodeLongitude= MethodStartSpecific->oContactDetailsExtended.oHomeAddressLatLong.s32Longitude;
               }
               ETG_TRACE_USR2(("Contact.homeGeocodeLongitude - %d", Contact.homeGeocodeLongitude));

               Contact.workGeoCodeLatLongValidity= MethodStartSpecific->oContactDetailsExtended.oWorkAddressLatLong.bLatLongValidity;
               ETG_TRACE_USR2(("Contact.workGeoCodeLatLongValidity - %d", Contact.workGeoCodeLatLongValidity));

               if(((unsigned)MethodStartSpecific->oContactDetailsExtended.oWorkAddressLatLong.s32Latitude >= FC_PB_REPORTED_LAT_MIN_RANG) && ((unsigned)MethodStartSpecific->oContactDetailsExtended.oWorkAddressLatLong.s32Latitude <= FC_PB_REPORTED_LAT_MAX_RANG))
               {
                  Contact.workGeoCodeLatitude= MethodStartSpecific->oContactDetailsExtended.oWorkAddressLatLong.s32Latitude;
               }
               ETG_TRACE_USR2(("Contact.workGeoCodeLatitude - %d", Contact.workGeoCodeLatitude));

               if(((unsigned)MethodStartSpecific->oContactDetailsExtended.oWorkAddressLatLong.s32Longitude >= FC_PB_REPORTED_LONG_MIN_RANG) && ((unsigned)MethodStartSpecific->oContactDetailsExtended.oWorkAddressLatLong.s32Longitude <= FC_PB_REPORTED_LONG_MAX_RANG))
               {
                  Contact.workGeocodeLongitude= MethodStartSpecific->oContactDetailsExtended.oWorkAddressLatLong.s32Longitude;
               }
               ETG_TRACE_USR2(("Contact.workGeocodeLongitude - %d", Contact.workGeocodeLongitude));

               Contact.otherGeoCodeLatLongValidity= MethodStartSpecific->oContactDetailsExtended.oOtherAddressLatLong.bLatLongValidity;
               ETG_TRACE_USR2(("Contact.otherGeoCodeLatLongValidity - %d", Contact.otherGeoCodeLatLongValidity));

               if(((unsigned)MethodStartSpecific->oContactDetailsExtended.oOtherAddressLatLong.s32Latitude >= FC_PB_REPORTED_LAT_MIN_RANG) && ((unsigned)MethodStartSpecific->oContactDetailsExtended.oOtherAddressLatLong.s32Latitude <= FC_PB_REPORTED_LAT_MAX_RANG))
               {
                  Contact.otherGeoCodeLatitude= MethodStartSpecific->oContactDetailsExtended.oOtherAddressLatLong.s32Latitude;
               }
               ETG_TRACE_USR2(("Contact.otherGeoCodeLatitude - %d", Contact.otherGeoCodeLatitude));

               if(((unsigned)MethodStartSpecific->oContactDetailsExtended.oOtherAddressLatLong.s32Longitude >= FC_PB_REPORTED_LONG_MIN_RANG) && ((unsigned)MethodStartSpecific->oContactDetailsExtended.oOtherAddressLatLong.s32Longitude <= FC_PB_REPORTED_LONG_MAX_RANG))
               {
                  Contact.otherGeocodeLongitude= MethodStartSpecific->oContactDetailsExtended.oOtherAddressLatLong.s32Longitude;
               }
               ETG_TRACE_USR2(("Contact.otherGeocodeLongitude - %d", Contact.otherGeocodeLongitude));

               // VPB contact has changed by user interaction via HMI. Update the checksum table, send 'ListChange' and refresh CallHistory.
               m_sqlite->SpeechUpdateCheckSumTable(MethodStartSpecific->u8DeviceHandle,true);

               mListChangeHandler.UpdateList(&m_hashPbList, enAction_Contact_Added,
                  MethodStartSpecific->u8DeviceHandle,
                  MethodResultSpecific->u32ContactHandle,
                  FC_PhoneBook_SQLite::ListType_VEHICLE);

               poCallHistoryList->RefreshCallHistoryOnChangeInContact(Contact);//FIX GMMY15-11882 Replacing multiple similar functions with single one
            // -bn: MY15 Feature: VPB Import: Code addition: END
         }
         break;
   case MOST_PHONBKFI_C_U16_EDITCONTACTEXTENDED:
         {
            ETG_TRACE_USR4(( " MethodResultPosted MOST_PHONBKFI_C_U16_EDITCONTACTEXTENDED" ));
            const most_phonbkfi_tclMsgEditContactExtendedMethodStart *MethodStartSpecific
               = static_cast<const most_phonbkfi_tclMsgEditContactExtendedMethodStart*>(MethodStart);

            Contact.contactHandle= MethodStartSpecific->u32ContactHandle;
            //Fix for GMMY15-11538:Garbled code is displayed in recent calls list (Chinese HMI)
            Contact.firstName= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.sFirstName.szValue);
            Contact.lastName=  QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.sLastName.szValue);
            Contact.poiName=   QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.sPOIName.szValue);
            Contact.PhoneNum1=   QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.oPhoneNumber1.sNumber.szValue);
            Contact.PhoneNum2=  QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.oPhoneNumber2.sNumber.szValue);
            Contact.PhoneNum3=  QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.oPhoneNumber3.sNumber.szValue);
            Contact.PhoneNum4=  QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.oPhoneNumber4.sNumber.szValue);
            Contact.PhoneNum5=  QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.oPhoneNumber5.sNumber.szValue);
            Contact.PhoneNum6=  QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.oPhoneNumber6.sNumber.szValue);
            Contact.PhoneNum7=  QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.oPhoneNumber7.sNumber.szValue);
            Contact.PhoneNum8=  QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.oPhoneNumber8.sNumber.szValue);
            Contact.emailAdd1= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.sEmailAddress1.szValue);
            Contact.emailAdd2= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.sEmailAddress2.szValue);
            Contact.emailAdd3= QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.sEmailAddress3.szValue);
            Contact.homeAdd=   QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.sHomeAddress.szValue);
            Contact.workAdd=   QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.sWorkAddress.szValue);
            Contact.otherAdd=  QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.sOtherAddress.szValue);
            Contact.category=  QString::fromUtf8(MethodStartSpecific->oContactDetailsExtended.sCategory.szValue);
            Contact.notes= MethodStartSpecific->oContactDetailsExtended.sNotes.szValue;
            Contact.homeGeoCodeLatLongValidity  =  MethodStartSpecific->oContactDetailsExtended.oHomeAddressLatLong.bLatLongValidity;
            Contact.homeGeoCodeLatitude         =  MethodStartSpecific->oContactDetailsExtended.oHomeAddressLatLong.s32Latitude;
            Contact.homeGeocodeLongitude     =  MethodStartSpecific->oContactDetailsExtended.oHomeAddressLatLong.s32Longitude;
            Contact.workGeoCodeLatLongValidity  =  MethodStartSpecific->oContactDetailsExtended.oWorkAddressLatLong.bLatLongValidity;
            Contact.workGeoCodeLatitude         =  MethodStartSpecific->oContactDetailsExtended.oWorkAddressLatLong.s32Latitude;
            Contact.workGeocodeLongitude     =  MethodStartSpecific->oContactDetailsExtended.oWorkAddressLatLong.s32Longitude;
            Contact.otherGeoCodeLatLongValidity =  MethodStartSpecific->oContactDetailsExtended.oOtherAddressLatLong.bLatLongValidity;
            Contact.otherGeoCodeLatitude     =  MethodStartSpecific->oContactDetailsExtended.oOtherAddressLatLong.s32Latitude;
            Contact.otherGeocodeLongitude    =  MethodStartSpecific->oContactDetailsExtended.oOtherAddressLatLong.s32Longitude;
            // VPB contact has changed. Update the checksum table
            m_sqlite->SpeechUpdateCheckSumTable(MethodStartSpecific->u8DeviceHandle,true);
            mListChangeHandler.UpdateList(&m_hashPbList, enAction_Contact_Edited,
               MethodStartSpecific->u8DeviceHandle, MethodStartSpecific->u32ContactHandle,
               FC_PhoneBook_SQLite::ListType_VEHICLE);

            poCallHistoryList->RefreshCallHistoryOnChangeInContact(Contact);//FIX GMMY15-11882 Replacing multiple similar functions with single one
            m_poFavoritesProvider->RefreshFavoritesOnEditContact(Contact);
         }
         break;
   default:
      {
         ETG_TRACE_ERR(( " u16FuntionID= %d was not handled in 'switch(.) statement. ", u16FuntionID ));
      }

   }  // end of: switch(u16FuntionID)
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::DownloadCompleted
*
* DESCRIPTION: - This function is called when a BT phone gets disconnected or
*                an abnormal loss of the BT connection occurs.
*              - This function is called when an ME/SM download has completed
*                normally.
*
*              Dedicated actions:
*              - Check ME/SM Download capability/status and based on this:
*                   Delete 'invalidated' contacts, update favorites
*              - Check if contacts have been added or deleted and based on this:
*                   Update the Speech checksum in database.
*
*              - Update contact lists in RAM and send out ListChanges to clients
*              - Update SortView in database and deviceDetails
*
* PARAMETER: [IN]  - u8DeviceHandle
*            [OUT] - none
*
* RETURNVALUE: void
*
*******************************************************************************/
void FC_PhoneBook_List::DownloadCompleted(quint8 u8DeviceHandle)
{
   ETG_TRACE_USR4(( "DownloadCompleted entered" ));

   bool blListChanged = false;
   tS32 s32ContactsAddded  = 0;
   tS32 s32ContactsDeleted = 0;

   // Transfer new contacts [which have to be iserted] or photos [which have to be updated] from the related RAM lists
   // to the database, when relevant memory thresholds for these lists have been reached.
   vShiftRAMContentToDatabaseBasedOnMemThresholds(u8DeviceHandle, 1 /*PHBK_SHIFT_CONTACTS_TO_TMP_THRESHOLD*/, 1/* PHBK_SHIFT_PHOTOS_TO_DHx_THRESHOLD*/);

   FC_Device_Details* DeviceDetails = FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);
   if(DeviceDetails)
   {
      if(FALSE == DeviceDetails->bIsDeviceProbablyLocked())
      {
         ListChange_Handler oListChangeHandler;
         ETG_TRACE_USR4(( " Total no of photos updated to DB in current download pass: %u ", DeviceDetails->m_u32PhotosUpdatedInCurrentDwnld ));

         s32ContactsDeleted = DeviceDetails->InvalidContactHandleList.size();
         if(s32ContactsDeleted > 0)
         {
            ETG_TRACE_COMP((" -PBDL.S- Going to delete %d 'invalidated' contacts ", s32ContactsDeleted ));

            // Local variables for code compactness in next if clause
            tBool l_bMEsupport = (PBInLocal & DeviceDetails->getDeviceCapabilities()) > 0 ? true : false;
            tBool l_bMEStatus  = DeviceDetails->m_bMEDownloadStatus;

            ETG_TRACE_COMP((" -PBDL.S- ME-Support= %d, ME-Dl-Status= %d ", l_bMEsupport, l_bMEStatus));

            // Check if ME/SM was supported and if the related ME/SM downloads have completed with status= 1 (successfully)
            if( (l_bMEsupport && l_bMEStatus) || (l_bMEsupport == false) )
            {
               ETG_TRACE_COMP((" -PBDL.S- Download 'Support' and 'Status' for ME allows deletion of 'invalidated' contacts and we do so. "));

              // Delete contacts in database which correspond to 'invalidated' contact handles
              m_sqlite->DeleteContactHandles(u8DeviceHandle, DeviceDetails->InvalidContactHandleList);

              // Check favorites, if a deleted Contact would require an update of a related favorite
              for (tU32 iCount = 0; iCount
                    < (tU32) DeviceDetails->InvalidContactHandleList.size(); iCount++)
              {
                 m_poFavoritesProvider->RefreshFavoritesOnDeleteContact(DeviceDetails->InvalidContactHandleList[(int) iCount]);
              }
              blListChanged = true;
            }
            else
            {
               ETG_TRACE_COMP((" -PBDL.S- Download 'Support' and 'Status' for ME indicate that 'invalidated' contacts should not be deleted. "));
            }
         }

         s32ContactsAddded = DeviceDetails->NewContactsAdded.size();
         if(s32ContactsAddded > 0)
         {
            DeviceDetails->NewContactsAdded.clear();
            blListChanged = true;

            if( false == m_sqlite->InsertTmpToDeviceTable(u8DeviceHandle) )  // CMG3G-7702
            {
               ETG_TRACE_ERR(( " ERROR: Insertion of 'PhoneBook_Tmp' contacts to 'PhoneBook_DH%u' has failed. ", u8DeviceHandle ));
            }
         }

         // Clear PhoneBook_Tmp Table as contents in it are no longer required
         if ( false == m_sqlite->DeleteContentsOfTable(FC_PB_DEVICE_TMP) )
         {
            ETG_TRACE_ERR(( " ERROR: Clearing of table 'PhoneBook_Tmp' has failed. " ));
         }

#ifdef DATABASE_WAL_JOURNALING
#if 0  // Code section is currently deactivated as we think that the anyway active "autocheckpointing" is sufficient.
       // We just keep this section for reference.

         QTime WalCheckpointingTime;
         WalCheckpointingTime.start();
         // When we come here an ME (or SIM) list DL has completed. We try to "checkpoint" the WAL journaling here ("i.e. to sync it to the database").

         // Currently commented out as we think that the anyway active "autocheckpointing" is sufficient.
         // m_sqlite->SetWalCheckPoint();
         ETG_TRACE_USR1(( " -PHBK.ENCRYPTION-SDC-: 'm_sqlite->SetWalCheckPoint()' exited after %d ms ", WalCheckpointingTime.elapsed() ));
#endif
#endif

         if((blListChanged))
         {
            ETG_TRACE_COMP(( " -PBDL.S- Contact data has changed: [Contacts added= %d, deleted= %d] ; CheckSum will be updated",
               s32ContactsAddded, s32ContactsDeleted ));

            // Update speech checksum as Contact list has changed.
            m_sqlite->SpeechUpdateCheckSumTable(u8DeviceHandle);

            oListChangeHandler.UpdateList(&m_hashPbList, enAction_Content_Refreshed,
               u8DeviceHandle, 0, FC_PhoneBook_SQLite::ListType_PHONEBOOK);
            //Once phonebook download is completed, actual list changes for list type favorites should be sent to clients.
            oListChangeHandler.UpdateList(&m_hashPbList, enAction_Content_Refreshed,
               u8DeviceHandle, 0, FC_PhoneBook_SQLite::ListType_FAVORITE);
         }

         else
         {
            if(DeviceDetails->m_u32PhotosUpdatedInCurrentDwnld!=0)  // PSARCCB-8551 : Added condition check for updating list change to the clients for photo updation in the second pass
            {
               ETG_TRACE_COMP(( " -PBDL.S- Only Photo data has changed, CheckSum will NOT be updated" ));

               oListChangeHandler.UpdateList(&m_hashPbList, enAction_Content_Refreshed,
                  u8DeviceHandle, 0, FC_PhoneBook_SQLite::ListType_PHONEBOOK);
               //Once phonebook download is completed, actual list changes for list type favorites should be sent to clients.
               oListChangeHandler.UpdateList(&m_hashPbList, enAction_Content_Refreshed,
                  u8DeviceHandle, 0, FC_PhoneBook_SQLite::ListType_FAVORITE);
            }
            else
            {
            ETG_TRACE_COMP(( " -PBDL.S- No Contact data changed, CheckSum will NOT be updated" ));

            // Dummy list change required by speech
            oListChangeHandler.UpdateList(&m_hashPbList, enAction_Dummy, u8DeviceHandle, 0);
            }
         }

         // Update List SortView
         tBool b_result = m_sqlite->UpdateSortListView(DeviceDetails->getDeviceHandle());
         DeviceDetails->setSortListViewSupport(b_result);
      }
      else
      {
         ETG_TRACE_USR1((" -PBDL.S- Device is locked! "));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Device Object is NULL"));
   }
}


/***************************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::DeviceDisconnected
*
* DESCRIPTION: This function is called when a device is disconnected from IVI
*
* PARAMETER: [IN] -[OUT]
* RETURNVALUE:
*
****************************************************************************************/
void FC_PhoneBook_List::DeviceDisconnected(FC_Device_Details* DeviceDetails)
{
   ETG_TRACE_USR4(( " DeviceDisconnected entered" ));

   if(DeviceDetails)
   {
      FC_PhoneBook_CallHistory_List *poCallHistoryList=
         FC_PhoneBook_CallHistory_List::GetCurrentInstance();
      ListChange_Handler pListChange_Handler;
      pListChange_Handler.UpdateList(&m_hashPbList, enAction_Content_Cleared,
         DeviceDetails->getDeviceHandle(), 0);
      poCallHistoryList->RefreshCallHistoryOnDeviceDisconnect(
         DeviceDetails->getConnectionID());
      DeviceDetails->m_u8PendingManualPBCommandList.clear();
   }
   else
   {

      ETG_TRACE_ERR(("Device Object is NULL"));
   }
}


/***************************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::CompareAndInsertIntoDatabase(..)
*
* DESCRIPTION:
*    All contacts of the provided list are processed as follows:
*    Check if a provided Contact already exists in the database for the given DeviceHandle
*    or in the "lists" of contacts, which are already forseen to be added to the database
*    (Tmp table in database or RAM list).
*    If the contact is new and has to be added, it will be inserted to the RAM list.
*    (The related further actions are done, when the ammount of collected contacts has
*     reached a significant/relevant value. The next step would be to transfer the RAM
*     contacts to the Tmp table of the database.)
*    In addition we check if the photo of an existing contact has to be updated. If this
*    is needed we add the contact in a dedicated RAM list 'DeviceDetails->m_PhotosToBeUpdatedRAMList'.
*    (Also here the related processing is done, when a significant number of contacts
*     has accumulated in this dedicated RAM list.)
*
*    Note: This method will be called when parsing vCards from a connected 'Phone device'.
*          The database queries for checking, if a contact is already in the database,
*          are optimized with respect to the contact attributes provided by a phone device.
*          (Hint: Therefore this method is not intended to be used for 'Vechicle Contacts'
*          as these contacts have additional attributes like 'category' and 'notes' ..)
*
*
* PARAMETER: [IN]  PBContactList, u8DeviceHandle
*            [OUT] none
* RETURNVALUE: none
*
****************************************************************************************/
void FC_PhoneBook_List::CompareAndInsertIntoDatabase(
   QList<FC_PhoneBook_Contact_Detail>& PBContactList, quint8 u8DeviceHandle)
{
   ETG_TRACE_USR4(( " CompareAndInsertIntoDatabase entered " ));

   QTime DebugTime;
   DebugTime.start();
   quint16  s16ListSize = (quint16)(PBContactList.size());

   FC_Device_Details* DeviceDetails = FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);
   if(!DeviceDetails)
   {
      ETG_TRACE_FATAL(( " DeviceDetails object is NULL " ));
      return;
   }

   FC_PhoneBook_MemoryUsage *poPbMemUsage = fc_phonebook_tclService_PhoneBook::pGetInstance()->m_pPhoneBookMemoryUsage;
   // FIX GMMY15-13121 Nullpointerexception in "phonebook.out" in thread phonebook.out
   // Probably 'm_pPhoneBookMemoryUsage' (member of 'fc_phonebook_tclService_PhoneBook') was not existent, so
   // we add a check here. (-bn: Comment: I don't think that this explanation fits to the reset... To check this
   // I have added the else clause with FATAL and ERRMEM traces.. Let's see if we see such traces ..)
   if(!poPbMemUsage )
   {
      ETG_TRACE_FATAL((  "  PHBK_NEVER_EXPECTED:  poPbMemUsage == NULL " ));
      ETG_TRACE_ERRMEM(( "  PHBK_NEVER_EXPECTED:  poPbMemUsage == NULL " ));
      return;
   }

   ETG_TRACE_USR4(( " Size of received vcards:%u ", s16ListSize));
   // Processing loop over the received vCards
   for(quint16 u16Index = 0; u16Index < s16ListSize; u16Index++)
   {
      bool bRemoved = false; // Value becomes relevant after we have tried to remove the contact handle from 'InvalidContactHandleList'.
                             // true  = contact handle was still in this list, i.e. we have verified a known contact.
                             // false = contact handle has already been removed from the list and we have a double contact.
      bool bFound = false;   // Contact was not found as existing contact in database or as a contact, which is intended to be added,
                             // i.e. as contact in table 'Phonebook_Tmp' or in RAM list 'DeviceDetails->m_NewContactsToBeAddedRAMList'

      // The following variable initializations might be updated after any "FindExactMatchingContact(..)" operation.
      quint32 u32ContactHandle = 0;       // 0 = Invalid contact handle
      bool bPhotoMatches       = false;   // false = Contact photo does not match the already available one. (The "available"
                                          // photo can also be an empty one.)
      quint32 u32photoSizeOld  = 0;       // Old photo size in bytes

      // Check if we are allowed to add a new contact regarding to a given maximum number of contacts
      if(DeviceDetails->m_u32ContributingVCardCount < PBDL_MAX_CONTACTS_PER_PHONE_TO_STORE)
      {
         FC_PhoneBook_Contact_Detail ContactDetail = PBContactList.at(u16Index);

         // Check if contact is already in the database (A) or the temporary table (B) in the database or in the RAM list (C)
         //    (PhoneBook_DHx or Phonebook_Tmp or RAM list 'DeviceDetails->m_NewContactsToBeAddedRAMList')

         // (A): Check if contact is already in 'PhoneBook_DHx'
         if( true == m_sqlite->FindExactMatchingContact(ContactDetail, u8DeviceHandle,
                                                        u32ContactHandle, bPhotoMatches, u32photoSizeOld) )
         {
            bFound = true;
            ContactDetail.contactHandle = u32ContactHandle;
            bRemoved = DeviceDetails->InvalidContactHandleList.removeOne(u32ContactHandle);
         }

         if(bFound == false)
         {
            // (B): Check if contact is already in 'PhoneBook_Tmp'
            if( true == m_sqlite->FindExactMatchingContact(ContactDetail, FC_PB_DEVICE_TMP,
                                                           u32ContactHandle, bPhotoMatches, u32photoSizeOld) )
            {
               bFound = true;
               ContactDetail.contactHandle = u32ContactHandle;
            }
         }

         if(bFound == false)
         {
            // (C): Check if contact is already in RAM list 'DeviceDetails->m_NewContactsToBeAddedRAMList'
            if( true == DeviceDetails->FindExactMatchingContact_inRAMList(ContactDetail,
                                                                          bPhotoMatches, u32photoSizeOld) )
            {
               bFound = true;
            }
         }

         // Now we know via 'bFound' if the contact is already known. Via 'bRemoved' we know if the contact has
         // been verified the first time (i.e. it is no double and the photo needs to be checked in more detail).
         if(bFound && bRemoved)
         {
            ETG_TRACE_USR2(( " Existing Contact verified: photoSize= %d, old photoSize= %d ", ContactDetail.photoSize, u32photoSizeOld ));
            // The contact is already in the database table 'PhoneBook_DHx' and has been verified now the first time,
            // i.e. the contact no double and it contributes to the number of relevant contacts/vCards.
            DeviceDetails->m_u32ContributingVCardCount++;
            (fc_phonebook_tclService_PhoneBook::pGetInstance())->vUpdatePhonebookDownloadProgress(u8DeviceHandle, DeviceDetails->m_u32ContributingVCardCount);

            // We have to check in addition, if the photo has to be updated.
            if(false == bPhotoMatches)
            {
               // Photo has changed, so we have to check if we have enough free photo memory available
               if( true == poPbMemUsage->bCheckMemoryPoolForPhoto(ContactDetail.photoSize, u32photoSizeOld) )
               {
                  // We have enough free photo memory

                  // Check if the photoSize is "very big", and if so then transcode it to reduce memory of photo for storage in DB.
                  // Notes:
                  // - Related contact details (photoSize, photoCRC) might change ater this method has returned.
                  // - If the transcoded imgage size is bigger then the original size, then the original image is just kept.
                  vTranscodePhotoIfBig(ContactDetail);

                  // Add contact to the "photo update list"
                  DeviceDetails->m_PhotosToBeUpdatedRAMList  << ContactDetail;

                  // Update the photo memory used by all phones
                  poPbMemUsage->vUpdateMemoryPoolForPhoto(ContactDetail.photoSize, u32photoSizeOld);
               }
            }
         }
         else if(bFound)
         {
            ETG_TRACE_USR2(( " Double contact found: It will be ignored. " ));
            // [Currently] this contact is a double and will be ignored. No further action needed.

            // TODO: When we later consider PBAP 1.2 'fav' and 'spd' (speed dial) lists in the download sequence,
            //       then this logic here might have to be refinded in more detail.
         }
         else
         {
            ETG_TRACE_USR2(( " Contact is new: photoSize= %d ", ContactDetail.photoSize ));
            // The contact is new and contributes to the number of relevant contacts/vCards

            // Check if we have free Contact memory (or if we have to skip contacts from an "old" phone first,
            // which would be done automatically in method 'bCheckForContactMemory')
            if( true == bCheckForContactMemory(poPbMemUsage, ContactDetail.APL1Size) )
            {
               // Contact memory is available; we add the Contact memory size to the "memory pool", i.e.
               // this memory counts now to the used memory for all contacts by all phone devices
               poPbMemUsage->vUpdateMemoryPoolForContacts(ContactDetail.APL1Size);

               // Check if we can store photo also or if the related photo memory limit is exceeded.
               if( true == poPbMemUsage->bCheckMemoryPoolForPhoto(ContactDetail.photoSize, u32photoSizeOld) )
               {
                  // We have enough free photo memory

                  // Check if Photo size is "very big", and if so then transcode it to reduce memory of photo for storage in DB.
                  // Notes:
                  // - Related contact details (photoSize, photoCRC) might change ater this method has returned.
                  // - If the transcoded imgage size is bigger then the original size, then the original image is just kept.
                  vTranscodePhotoIfBig(ContactDetail);

                  // Photo can be inserted with the contact and we update the photo memory used by all phones
                  poPbMemUsage->vUpdateMemoryPoolForPhoto(ContactDetail.photoSize, u32photoSizeOld);
               }
               else
               {
                  // PHOTO storage not possible, therefore clear Photo details of contact
                  ContactDetail.photo = "";   //.clear();
                  ContactDetail.photoSize = 0;
                  ContactDetail.photoDlSize = 0;
                  ContactDetail.photoCRC = 0;
                  ContactDetail.photoDlCRC = 0;
                  ContactDetail.photoType = "";
               }

               // The contact now is inserted to the RAM list 'DeviceDetails->m_NewContactsToBeAddedRAMList'.
               // (And will be stored later to the database.)
               DeviceDetails->m_NewContactsToBeAddedRAMList  << ContactDetail;

               // The new contact contributes to the number of relevant contacts/vCards
               DeviceDetails->m_u32ContributingVCardCount++;
               (fc_phonebook_tclService_PhoneBook::pGetInstance())->vUpdatePhonebookDownloadProgress(u8DeviceHandle, DeviceDetails->m_u32ContributingVCardCount);
            }
            else
            {
               // We have to skip the contact as we have no contact memory available. This is not really
               // expeceted as it would mean that 1 phone needs more contact memory as totally available...
               ETG_TRACE_FATAL(( " STRANGE: Contact skipped as no contact memory is available." ));
            }
         }
      }
      else
      {
         ETG_TRACE_USR3(( " vCards ignored because: m_u32ContributingVCardCount >= PBDL_MAX_CONTACTS_PER_PHONE_TO_STORE " ));
         //Fix RTC 167747
         if( (fc_phonebook_tclService_PhoneBook::pGetInstance())->pGetDownloadStateInstance())
         {
            ETG_TRACE_USR4(("Setting DownloadState as most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended::FI_EN_E8PBDS_LIMIT_REACHED"));
            ((fc_phonebook_tclService_PhoneBook::pGetInstance())->pGetDownloadStateInstance())->vSetPBDownloadState(most_fi_tcl_e8_PhonBkPhoneBookDownloadStateExtended::FI_EN_E8PBDS_LIMIT_REACHED);
         }
         //End fix RTC 167747
         break;
      }  // end of: if(DeviceDetails->m_u32ContributingVCardCount < PBDL_MAX_CONTACTS_PER_PHONE_TO_STORE)
   }  // end of: for(quint16 u16Index = 0; u16Index < s16ListSize; u16Index++)

   // Transfer new contacts [which have to be iserted] or photos [which have to be updated] from the related RAM lists
   // to the database, when relevant memory thresholds for these lists have been reached.
   vShiftRAMContentToDatabaseBasedOnMemThresholds(u8DeviceHandle, PHBK_SHIFT_CONTACTS_TO_TMP_THRESHOLD, PHBK_SHIFT_PHOTOS_TO_DHx_THRESHOLD);
   ETG_TRACE_USR2(( " -PBDL.Perf- 'CompareAndInsertIntoDatabase' exited after %d ms ", DebugTime.elapsed() ));
}


void FC_PhoneBook_List::vShiftRAMContentToDatabaseBasedOnMemThresholds(quint8 u8DeviceHandle,
                                   quint32 u32ContactsNumberThreshold, quint32 u32PhotoNumberThreshold)
{
   ETG_TRACE_USR4(( " ShiftRAMContentToDatabaseBasedOnMemThresholds entered " ));

   QTime InsertTime;
   InsertTime.start();

   quint32 u32ContactsInserted = 0;

   FC_Device_Details* DeviceDetails = FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);
   if(!DeviceDetails)
   {
      ETG_TRACE_FATAL(( " DeviceDetails object is NULL " ));
      return;
   }

   // Check RAM contacts should be inserted to table Phonebook_Tmp in the database
   quint32 u32Size = (quint32)(DeviceDetails->m_NewContactsToBeAddedRAMList.size());
   if(u32Size >= u32ContactsNumberThreshold)
   {
      // Transfer new contacts from RAM list to the Tmp list in the database
      QList<quint32> u32ContactHandleList;
      if( m_sqlite->InsertRecords(DeviceDetails->m_NewContactsToBeAddedRAMList,
          0 /*DeviceHandle of Tmp List*/, u32ContactHandleList) )
      {
         ETG_TRACE_USR4(( " FC_PhoneBook_List::Contact inserted successfully " ));
         DeviceDetails->NewContactsAdded.append(u32ContactHandleList);
         u32ContactsInserted = u32Size;
      }

      // Clear Contact RAM list
      DeviceDetails->m_NewContactsToBeAddedRAMList.clear();
   }

   // Check if RAM photos shall be updated in table Phonebook_DHx in the database
   u32Size = (quint32)(DeviceDetails->m_PhotosToBeUpdatedRAMList.size());
   if(u32Size >= u32PhotoNumberThreshold)
   {
      // Update the photos of existing contacts, which have changed.
      if( true == m_sqlite->bUpdatePhotosOfExistingRecords(u8DeviceHandle, DeviceDetails->m_PhotosToBeUpdatedRAMList) )
      {
         ETG_TRACE_USR4(( " Number of photos updated= %u ", u32Size ));
      }
      else
      {
         ETG_TRACE_ERR(( " ERROR: Update of photos has failed. " ));
      }

      // Clear Photo RAM list
      DeviceDetails->m_PhotosToBeUpdatedRAMList.clear();
      DeviceDetails->m_u32PhotosUpdatedInCurrentDwnld = DeviceDetails->m_u32PhotosUpdatedInCurrentDwnld + u32Size;    //Changes for PSARCCB-8551
   }

   if( (u32ContactsInserted > 0) || (u32Size > 0) )
   {
      ETG_TRACE_USR2(( " -PBDL.Perf- Shift RAM Content to DB done: ContactsInserted= %u, PhotosUpdated = %u, Processing Time= %d ms ",
                       u32ContactsInserted, u32Size , InsertTime.elapsed() ));
   }
}


/***************************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::vTranscodePhotoIfBig( oContactDetail )
*
* DESCRIPTION: The Photo Size of a Contact (received via vCard) is checked against
*              a usefull threshold (bytes).
*              If the Photo Size is larger then the threshold the Image is transcoded
*              to JPEG, 240x240.
*              This is done to "minimize" the used memory in the database for the Photoe.
*
* PARAMETER: [IN] and [OUT]: oContactDetail
*
* RETURNVALUE: None
*
****************************************************************************************/
void FC_PhoneBook_List::vTranscodePhotoIfBig(FC_PhoneBook_Contact_Detail& ContactDetail)
{
   // Check if the the received PHOTO in vCard has a "usefull" size in bytes,
   // otherwise we will do a transcoding to JPEG, 240x240 pixel
   if( ContactDetail.photoDlSize >= FC_PHONEBOOK_PHOTO_SIZE_TRANSCODING_LIMIT_BYTES )
   {
      ETG_TRACE_USR2((" Photo is size is larger than threshold --> Image Transcoding "));

#if 0  // Measure Transcoding Time
      struct timespec time1      = {  0, 0};
      struct timespec time2      = {  0, 0};
      struct timespec diff_time  = {  0, 0};

      clock_gettime(CLOCK_MONOTONIC, &time1);
#endif
      QImage img = QImage::fromData(ContactDetail.photo, ContactDetail.photoType.toLatin1().constData());
      if (img.isNull())
      {
         ETG_TRACE_ERR((" Transcoded image is empty "));
      }
      else
      {
         QByteArray ba;
         ba.clear();

         QBuffer buffer(&ba);
         buffer.open(QIODevice::WriteOnly);

         if (img.save(&buffer, "JPEG")) // writes image into ba in JPEG format
         {
            //Image transcoded successfully
            QImage
                  TransImage =
                        QImage::fromData(ba, "JPEG").convertToFormat(QImage::Format_RGB888, Qt::AutoColor); //Format the JPEG image to 24 bits per pixel
            QImage
                  ScaledImage =
                        TransImage.scaled(240, 240, Qt::IgnoreAspectRatio, Qt::SmoothTransformation); //Scale the image to 240x240
            ba.clear();
            QBuffer TransBuf(&ba);
            TransBuf.open(QIODevice::WriteOnly);
            ScaledImage.save(&TransBuf, "JPEG");

            ETG_TRACE_USR4((" Image transcoded successfully. Transcocded size= %d, DL size= %d [bytes] ", (quint32) ba.size(), ContactDetail.photoDlSize));
            // If transcoded image is smaller than the original one, then assign transcoded image to contact;
            // otherwise let's keep the original image, even it has exceeded the threshold.
            if ((quint32) ba.size() < ContactDetail.photoDlSize)
            {
               ContactDetail.photo = ba;
               ContactDetail.photoType = "JPEG";
               ContactDetail.photoSize = (quint32) ba.size();
               ContactDetail.photoCRC = qChecksum(ba.constData(), ContactDetail.photoSize);
            }
         }
         else
         {
            ETG_TRACE_ERR((" Failed to transcode the image "));
            ContactDetail.photo = QByteArray(); // empty byte arrray
            ContactDetail.photoSize = 0;
            ContactDetail.photoCRC = 0;
         }
#if 0  // Measure Transcoding Time
         clock_gettime(CLOCK_MONOTONIC, &time2);

         if( (time2.tv_nsec - time1.tv_nsec) < 0 )
         {
            diff_time.tv_sec = time2.tv_sec - time1.tv_sec-1;
            diff_time.tv_nsec = 1000000000 + time2.tv_nsec - time1.tv_nsec;
         }
         else
         {
            diff_time.tv_sec = time2.tv_sec - time1.tv_sec;
            diff_time.tv_nsec = time2.tv_nsec - time1.tv_nsec;
         }

         ETG_TRACE_USR2(( " Image transcoding result[boolean]= %d, used Time[msec]= %5d ",
                     bResult, diff_time.tv_sec *1000 + diff_time.tv_nsec / 1000000 ));
#endif
      }
   }  // end of: if( ContactDetail.photoDlSize >= FC_PHONEBOOK_PHOTO_SIZE_TRANSCODING_LIMIT_BYTES )
}


/***************************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::PhoneBookDownloadInitiated
*
* DESCRIPTION: This function is called when the phonebook download is
* started for a device
*
* PARAMETER: [IN] -[OUT]
* RETURNVALUE:
*
****************************************************************************************/
void FC_PhoneBook_List::PhoneBookDownloadInitiated(FC_Device_Details* DeviceDetails)
{
   ETG_TRACE_USR4(( " PhoneBookDownloadInitiated entered " ));

   if(DeviceDetails)
   {
      DeviceDetails->m_bMEDownloadStatus = FALSE;
      DeviceDetails->m_bSMDownloadStatus = FALSE;


      DeviceDetails->NewContactsAdded.clear();
      DeviceDetails->InvalidContactHandleList.clear();

      DeviceDetails->m_NewContactsToBeAddedRAMList.clear();  // -bn: 2016.01: DL speed improvement by collecting contacts for db insertion
      DeviceDetails->m_PhotosToBeUpdatedRAMList.clear();     // -bn: 2016.01: DL speed improvement by collecting contacts before photo update in database

      DeviceDetails->m_u32PhotosUpdatedInCurrentDwnld = 0;  //Changes for PSARCCB-8551

      // Start of CMG3G-7702
      DeviceDetails->m_u32ContributingVCardCount = 0;
      if(m_sqlite->CreateDeviceTable(FC_PB_DEVICE_TMP))
      {
         if(m_sqlite->DeleteContentsOfTable(FC_PB_DEVICE_TMP))  // Delete contents of the Tmp Table.
         {
            ETG_TRACE_USR4(( " PhoneBookDownloadInitiated: Tmp table CREATED and CLEARED "));
         }
         else
         {
            ETG_TRACE_ERR((" PhoneBookDownloadInitiated:- ERROR - Error while Deleting Contents Of Tmp Table "));
         }
      }
      else
      {
         ETG_TRACE_ERR((" PhoneBookDownloadInitiated:- ERROR - Error while creating Tmp table "));
      }
      // End of CMG3G-7702

      if(m_sqlite->GetContactHandles(DeviceDetails->getDeviceHandle(),
         DeviceDetails->InvalidContactHandleList))
      {
         ETG_TRACE_USR4(("FC_PhoneBook_List::Fetched all contact handles"));
      }
      else
      {
         ETG_TRACE_USR4(("FC_PhoneBook_List::Failed in getting "
            "contact handles"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Device Object is NULL"));
   }
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::DeviceDeleted
*
* DESCRIPTION: This function is when a device is deleted( unpaired )
*
* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_List::DeviceDeleted(guint8 u8DeviceHandle)
{
   ETG_TRACE_USR4(( " DeviceDeleted entered" ));
   if(!m_sqlite->DeleteRecordsOfDevice(u8DeviceHandle))
   {
      ETG_TRACE_USR2(("Failed to delete contacts from database for "
         "device handle - %d",u8DeviceHandle));
   }
   else
   {
      ETG_TRACE_USR2(("Deletion of contacts from database for "
         "device handle - %d successful",u8DeviceHandle));
   }
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::MapTelTypeToInteger
*
* DESCRIPTION:
*
* PARAMETER:
* RETURNVALUE:
*
*******************************************************************************/
quint8 FC_PhoneBook_List::MapTelTypeToInteger(QString numType)
{
   ETG_TRACE_USR4(( " MapTelTypeToInteger entered" ));
   quint8 IntNumType;
   if(numType == "VOICE")
   {
      IntNumType= (quint8)FC_PHONEBOOK_PRIO_TYPE_VOICE;
   }
   else if(numType == "PAGER")
   {
      IntNumType= (quint8)FC_PHONEBOOK_PRIO_TYPE_PAGER;
   }
   else if(numType == "FAX")
   {
      IntNumType= (quint8)FC_PHONEBOOK_PRIO_TYPE_FAX;
   }
   else if(numType.isEmpty())
   {
      IntNumType= (quint8)FC_PHONEBOOK_PRIO_TYPE_NOTHING;
   }
   else if(numType == "MSG")
   {
      IntNumType= (quint8)FC_PHONEBOOK_PRIO_TYPE_MSG;
   }
   else
   {
      IntNumType= (quint8)FC_PHONEBOOK_PRIO_TYPE_OTHER;
   }

   return IntNumType;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::GetContactPhoto(..)
*
* DESCRIPTION:
*
* PARAMETER: [IN]  methodStart :-
*            [OUT] methodResult :-
*            [OUT] u16ErrorCode :-
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_List::GetContactPhoto( const most_phonbkfi_tclMsgGetContactPhotoMethodStart* methodStart,
                                         most_phonbkfi_tclMsgGetContactPhotoMethodResult* methodResult,
                                         tU16& u16ErrorCode )
{
   ETG_TRACE_USR4(( " GetContactPhoto entered " ));

   quint32    u32ContactHandle = methodStart->u32ContactHandle;
   QByteArray PhotoData   = QByteArray();
   QString    qsPhotoType = "";
   quint16    PhotoDlCRC  = 0;

   bool blRet = true;


   if( m_sqlite->GetPhoto(u32ContactHandle, PhotoData, qsPhotoType, PhotoDlCRC, u16ErrorCode) )
   {
      ETG_TRACE_USR4(( " m_sqlite->GetPhoto Success; PhotoDlCRC= %d, PhotoType= '%s' ",
                       PhotoDlCRC, qsPhotoType.toUtf8().constData() ));


      methodResult->u32ContactHandle = u32ContactHandle;

      qint32 s32PhotoSize = PhotoData.size();
      for(qint32 s32Index = 0; s32Index < s32PhotoSize; s32Index++)
      {
         methodResult->oContactPhoto.u8Items.push_back((unsigned char)PhotoData[s32Index]);
      }

   }
   else
   {
      ETG_TRACE_ERR(( " m_sqlite->GetPhoto Failed. ErrorCode= %u ", u16ErrorCode ));

      blRet= false;

      if(FC_PB_ERROR_INTERNAL_FAILURE == u16ErrorCode)
      {
         u16ErrorCode= CCA_C_U16_ERROR_INTERNAL_FAILURE;
      }
      else
      {
         u16ErrorCode= CCA_C_U16_ERROR_PARAMETER_OUT_OF_RANGE;
      }
   }

   return blRet;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::GetContactPhotoInfo
*
* DESCRIPTION:
*
* PARAMETER: [IN] methodStart :-
*            [OUT] methodResult :-
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_List::GetContactPhotoInfo(
   const most_phonbkfi_tclMsgGetContactPhotoInfoMethodStart* methodStart,
   most_phonbkfi_tclMsgGetContactPhotoInfoMethodResult* methodResult,
   tU16& u16ErrorCode)
{
   // TODO: Track if the specification changes on this requirement: The to be returned Image type is currently hardcoded to PNG.
   ETG_TRACE_USR4(( " GetContactPhotoInfo entered " ));

   quint32 u32ContactHandle = methodStart->u32ContactHandle;

   QByteArray PhotoData    = QByteArray();
   QString    qsPhotoType  = "";
   quint16    PhotoDlCRC   = 0;

   bool blRet= true;

   if( m_sqlite->GetPhoto(u32ContactHandle, PhotoData, qsPhotoType, PhotoDlCRC, u16ErrorCode) )
   {
      ETG_TRACE_USR4(( " m_sqlite->GetPhoto Success; PhotoDlCRC= %d, PhotoType= '%s' ",
                       PhotoDlCRC, qsPhotoType.toUtf8().constData() ));


      ETG_TRACE_USR2(( " m_sqlite->GetContactPhotoInfo Success. ImageSize[bytes]= %d ", (quint32)PhotoData.size()));

      methodResult->u32ContactHandle  = u32ContactHandle;
      methodResult->u32PhotoImageSize = (quint32)PhotoData.size();
      methodResult->sMIMEImageSubtype.bSet(qsPhotoType.toLatin1().constData());
   }
   else
   {
      ETG_TRACE_ERR(( " m_sqlite->GetContactPhotoInfo Failed. ErrorCode= %u ", u16ErrorCode ));
      blRet= false;

      if(FC_PB_ERROR_INTERNAL_FAILURE == u16ErrorCode)
      {
         u16ErrorCode= CCA_C_U16_ERROR_INTERNAL_FAILURE;
      }
      else
      {
         u16ErrorCode= CCA_C_U16_ERROR_PARAMETER_OUT_OF_RANGE;
      }
   }

   return blRet;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::DownloadAborted
*
* DESCRIPTION: This function is called when download is aborted
**
*
* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_List::DownloadAborted(tU8 u8DeviceHandle)
{
   ETG_TRACE_COMP(( " -PBDL.S- DownloadAborted entered with DeviceHandle %d ", u8DeviceHandle ));
   FC_Device_Details* DeviceDetails =
      FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);

   if(DeviceDetails)
   {
      ListChange_Handler oListChangeHandler;

      ETG_TRACE_COMP(( " -PBDL.S-    NewContactsAdded.size()= %d, InvalidContactHandleList.size()= %d ",
         DeviceDetails->NewContactsAdded.size(), DeviceDetails->InvalidContactHandleList.size() ));

      if(DeviceDetails->NewContactsAdded.size() > 0)
      {
         if( DeviceDetails->InvalidContactHandleList.size() > 0 )
         {
            // The list of to be 'invalidtated' contacts is not empty, i.e. we have not been able to confirm
            // so far all contacts, which have been in the database before we started the download.
            // Therefore we will delete all newly downloaded contacts as otherwise we have the risk to see
            // changed contacts "twice" in the database.

            ETG_TRACE_COMP(( " -PBDL.S-    DL aborted: Remove newly added contacts from DB [%d contacts] ",
               DeviceDetails->NewContactsAdded.size() ));
            m_sqlite->DeleteContactHandles(u8DeviceHandle, DeviceDetails->NewContactsAdded);

            // Note: Logically we should also recalculate the 'ContactMemory' and the 'APL2_Photo_Memory'
            // values for this device now. But we keep it as it is. This will be corrected automatically, if
            // a download completes normally in future.
         }
         else
         {
            // We keep the already added contacts as we have no contflict with potentially to be deleted contacts
            ETG_TRACE_COMP(( " -PBDL.S-    DL aborted: We keep the newly added contacts in DB [%d contacts]",
               DeviceDetails->NewContactsAdded.size() ));
         }

         // -bn: Regarding the following 'UpdateList' I have doubts, if it is needed (as the phone got disconnected),
         //      but I keep it for the moment as it would cost some time to check the details. TODO: Check topic
         oListChangeHandler.UpdateList(&m_hashPbList, enAction_Content_Refreshed,
            u8DeviceHandle, 0);

         DeviceDetails->NewContactsAdded.clear();
      }
      else
      {
         // -bn: Regarding the following 'UpdateList' I have doubts, if it is needed (as the phone got disconnected),
         //      but I keep it for the moment as it would cost some time to check the details. TODO: Check topic
         oListChangeHandler.UpdateList(&m_hashPbList, enAction_Dummy,
            u8DeviceHandle, 0);  //Dummy list change required by speech
      }

      DeviceDetails->InvalidContactHandleList.clear();
      m_sqlite->DropTableFromDB(FC_PB_DEVICE_TMP); //CMG3G-7702 DROP Tmp TABLE

      //Fix 159808 - Reset Download Progress values for the aborted device handle
      fc_phonebook_tclService_PhoneBook::pGetInstance()->vUpdatePhonebookDownloadProgress(u8DeviceHandle, VALUE_ZERO);
      //End of fix 159808

      //Fix NCG3D-65982
      fc_phonebook_tclService_PhoneBook::pGetInstance()->vRemoveFromSuspendedDownloadList(u8DeviceHandle);
      //End fix NCG3D-65982
   }
   else
   {
      ETG_TRACE_ERR(("Device Object is NULL"));
   }

}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::RegisterFavoritesProvider
*
* DESCRIPTION: This function is called when download is aborted
**
*
* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_List::RegisterFavoritesProvider(
   FC_Phonebook_tclFavorite_Provider* pFavoritesProvider)
{
   m_poFavoritesProvider= pFavoritesProvider;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::InitializeDatabase
*
* DESCRIPTION:
**
*
* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_List::InitializeDatabase()
{

   ETG_TRACE_USR4(("Initializing database entered"));

   for(quint8 u8Index= 0; u8Index < FC_PB_TOTAL_NUMBER_OF_CH; u8Index++)
   {
      FC_Device_Details *DevDetails=
         FC_Device_Details::getDeviceObject((INDEX)u8Index);
      if(DevDetails)
      {
         DevDetails->deleteDeviceObject((INDEX)u8Index);
      }
   }
   m_sqlite->InitializeDatabase();
   /* FIX PSARCCB-9855 (06/2016):
   // Reset the variable 'g_iSkipDataSignals' to allow PbDLDataSignals to be processed in D-Bus thread
   g_iSkipDataSignals = 0;

   // FIX PSARCCB-9855 (06/2016):
   // At this code position the variable 'g_iRequestSchedYield' should already be zero. The following
   // code line will just make sure that it is definitely zero afterwards (when the database has been opened [after transition to NORMAL))
   g_iRequestSchedYield = 0;
  */
   m_poFavoritesProvider->UpdateFavoriteIDList();
}


/***************************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::DeviceConnected
*
* DESCRIPTION: This function is called when a device is connected
*
* PARAMETER: [IN] -[OUT]
* RETURNVALUE:
*
****************************************************************************************/
void FC_PhoneBook_List::DeviceConnected(FC_Device_Details* DeviceDetails)
{
   ETG_TRACE_USR2(( " FC_PhoneBook_List::DeviceConnected entered, DeviceHandle= %d ", DeviceDetails->getDeviceHandle() ));

   quint16 u16NumOfEntries = m_sqlite->GetTotalRecords(DeviceDetails->getDeviceHandle());

   DeviceDetails->setFirstTimeDownload((u16NumOfEntries>0)?false:true);

   m_sqlite->bUpdateDeviceLastConnectionTimeElapsed(DeviceDetails->getDeviceHandle());

   //FIX CMG3GB-505 Upon disconnecting a call/call is established/entering numbers in dialpad, resets when doing handsfree operation was observed
   m_sqlite->clearSearchPhoneDigitList();
   //End of fix
}


/***************************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::InsertContactsIntoDatabase
*
* DESCRIPTION: Inserts the given contacts into database without comparison
*
*
* PARAMETER: [IN] -[OUT]
* RETURNVALUE:
*
****************************************************************************************/
void FC_PhoneBook_List::InsertContactsIntoDatabase(
   QList<FC_PhoneBook_Contact_Detail>& PBContactList, quint8 u8DeviceHandle)
{

   ETG_TRACE_USR4(("Function InsertContactsIntoDatabase entered"));
   tBool bIsSimAndFirstPassDwnld = FALSE;
   // Fix RTC 169518
   //Download progress update will be send here in first pass download for supervision
   //strategy disabled projects
   FC_Device_Details* DeviceDetails =
         FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);

   if (DeviceDetails)
   {
#ifndef PBDL_SKIP_SIM_LIST
      if (TRUE == DeviceDetails->blIsItFirstTimeDownload())
      {
         bIsSimAndFirstPassDwnld = TRUE;
         quint16 u16NumOfEntries = m_sqlite->GetTotalRecords(u8DeviceHandle);

         //For first time, total number of records in data base are zero
         if (VALUE_ZERO != u16NumOfEntries)
         {
            if (((u16NumOfEntries % PBK_DOWNLOAD_PROGRESS_UPDATE_COUNT_REACHED)
                  + PBContactList.size())
                  >= PBK_DOWNLOAD_PROGRESS_UPDATE_COUNT_REACHED)
            {
               quint32 u32ContactsContributed =
                     (PBK_DOWNLOAD_PROGRESS_UPDATE_COUNT_REACHED
                           - (u16NumOfEntries
                                 % PBK_DOWNLOAD_PROGRESS_UPDATE_COUNT_REACHED) + u16NumOfEntries);
               (fc_phonebook_tclService_PhoneBook::pGetInstance())->vUpdatePhonebookDownloadProgress(u8DeviceHandle, u32ContactsContributed);
            }
         }
      }
#endif

      //End fix RTC 169518
      QList < quint32 > u32ContactHandleList;

      if (m_sqlite->InsertRecords(PBContactList, u8DeviceHandle, u32ContactHandleList))
      {
         ETG_TRACE_USR4(("FC_PhoneBook_List::Contact inserted successfully"));

         // Fix RTC 159808
         // Updating the m_u32ContributingVCardCount value in first time download for projects that
         //has supervision strategy disabled. When Supervision strategy is enabled the code flow doesnt
         //come here instead the method - CompareAndInsertIntoDatabase() will be invoked
         if (TRUE == bIsSimAndFirstPassDwnld)
         {
            DeviceDetails->m_u32ContributingVCardCount
                  = (quint32)(DeviceDetails->m_u32ContributingVCardCount
                        + PBContactList.size());
         }
         //End of fix RTC 159808
         //DeviceDetails->NewContactsAdded = u32ContactHandleList;  // -bn-PBDL.S: This is not correct as only the contact handles of the last received DataSignal are stored here. Corrected in next line.
         DeviceDetails->NewContactsAdded.append(u32ContactHandleList);
      }
   }
   else
   {
      ETG_TRACE_ERR(("Device Object is NULL"));
   }
}


bool FC_PhoneBook_List::UpdateSortView(quint8 u8DeviceHandle)
{
   return m_sqlite->UpdateSortListView(u8DeviceHandle);
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::bUpdateDlCompletedOnceValue
*
* DESCRIPTION: Updates Download completed or not for a particular device handle
*
* PARAMETER: [IN] -  u8DeviceHandle: Handle of the device to be updated.
*            [IN] -  bValue: Value to be updated. Denotes if download is completed or not.
*            [OUT] - None
*
* RETURNVALUE: bool - Indicating SUCCESS/FAILURE of the operation.
*
*******************************************************************************/
bool FC_PhoneBook_List::bUpdateDlCompletedOnceValue(quint8 u8DeviceHandle, bool bValue)
{
   bool bRetValue = false;
   if(TRUE == m_sqlite->bUpdateDlCompletedOnceValue(u8DeviceHandle, bValue))
   {
      FC_Device_Details* DeviceDetails=
                  FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);
      if(DeviceDetails)
      {
         DeviceDetails->m_bDlCompletedOnceValue = bValue;
         bRetValue = true;
      }
      else
      {
         ETG_TRACE_ERR(("Device Object is NULL"));
      }
   }
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::SearchPhoneNumberEntriesInPB
*
* DESCRIPTION:
*
*
*
* PARAMETER: [IN] methodStart :- Phone Number to search
*            [OUT] methodResult :- Matched PhoneNumber with first name, last name from Phonebook
* RETURNVALUE: void
*
*******************************************************************************/
void FC_PhoneBook_List::SearchPhoneNumberEntriesInPB(const most_phonbkfi_tclMsgAutocompletePhoneNumberEntryMethodStart* methodStart,
                                                     most_phonbkfi_tclMsgAutocompletePhoneNumberEntryMethodResult* methodResult,guint8 u8DeviceHandle)
{
   ETG_TRACE_USR4(("FC_PhoneBook_List::SearchPhoneNumberEntriesInPB Entered "));
   quint32 RecIndex = 0;
   quint8 DataCache = 0;
   quint8 SearchIndex = 0;
   quint8 u8ListSize = 0;
   QString ExistingPhNo;

   tString PhoneNumberDigits = methodStart->sPhoneNumberDigits.szValue;
   QString PhNoToSearch(PhoneNumberDigits);
   QString SearchPhNo = StripTelephoneNumber(PhNoToSearch);

   ETG_TRACE_USR2(("PhoneNumberDigits to Search in PB = %s",PhoneNumberDigits));

   for(RecIndex = 0; RecIndex < MAX_AUTO_COMPLETE_LISTS; RecIndex++)
   {
      if(m_sqlite != NULL)
      {
         ExistingPhNo =  m_sqlite->SearchedPhoneNumberDigit[RecIndex].toUtf8().constData();
         ETG_TRACE_USR2(("EXISTING Phone Numbers before Match =%s",ExistingPhNo.toLatin1().constData()));
         if(ExistingPhNo.compare(SearchPhNo) == 0)
         {
            ETG_TRACE_USR2(("Matched Phone Number = %s",ExistingPhNo.toLatin1().constData()));
            DataCache = 1;
            break;
         }
      }
   }
   if(( m_sqlite != NULL) && (DataCache == 1))
   {
      ETG_TRACE_USR4(("DATA Matched  "));

      u8ListSize = (quint8)(m_sqlite->AutoCompletePhNoSearchRecords[RecIndex].size());
      if(u8ListSize != VALUE_ZERO)
      {
         methodResult->bAutocompleteMatch = TRUE;
      }
      else
      {
         methodResult->bAutocompleteMatch = FALSE;
      }
      methodResult->u8NumberOfMatches = u8ListSize;

      if(!m_sqlite->AutoCompletePhNoSearchRecords[RecIndex].empty())
      {
         for(quint8 u8Index= 0; u8Index < u8ListSize; u8Index++)
         {
            QByteArray ba;
            AutoCompletePhoneNumberEntryResult MatchedContacts;
            most_fi_tcl_PhonBkAutocompleteMultipleMatchListItem	items;

            MatchedContacts = m_sqlite->AutoCompletePhNoSearchRecords[RecIndex].at(u8Index);
            ba = MatchedContacts.FirstName.toUtf8();
            items.sFirstName.bSet(ba.constData());
            ba = MatchedContacts.LastName.toUtf8();
            items.sLastName.bSet(ba.constData());
            ba = MatchedContacts.PhoneNumber.toUtf8();
            items.sPhoneNumber.bSet(ba.constData());

            methodResult->oAutocompleteMultipleMatchList.oItems.push_back(items);
         }
         //FIX CMG3GB-505 Upon disconnecting a call/call is established/entering numbers in dialpad, resets when doing handsfree operation was observed
         methodResult->sAutocompleteMatchPhoneNumber.bSet(methodResult->oAutocompleteMultipleMatchList.oItems[0].sPhoneNumber.szValue);
         //End of fix
      }
   }
   else
   {
      ETG_TRACE_USR4(("DATA MISMatched  "));

      SearchIndex = (quint8)AutoCompletePhNoSearchIndex;
      if(( m_sqlite != NULL ) && (!m_sqlite->SearchPhoneNumberEntriesInPB(SearchPhNo,SearchIndex,u8DeviceHandle)))
      {
         ETG_TRACE_USR4((" Search for Phone Number Digits Failed "));

         methodResult->bAutocompleteMatch = FALSE;
         methodResult->u8NumberOfMatches = u8ListSize;
         //FIX CMG3GB-505 Upon disconnecting a call/call is established/entering numbers in dialpad, resets when doing handsfree operation was observed
         methodResult->sAutocompleteMatchPhoneNumber.bSet(NULLPTR);
         //End of fix
      }
      else
      {
         ETG_TRACE_USR4(("DATA MISMatched  else part"));
         if( m_sqlite != NULL)
         {
            u8ListSize = (quint8)(m_sqlite->AutoCompletePhNoSearchRecords[SearchIndex].size());
            methodResult->bAutocompleteMatch = TRUE;
            methodResult->u8NumberOfMatches = u8ListSize;
            for(quint8 u8Index= 0; u8Index < u8ListSize; u8Index++)
            {
               QByteArray ba;
               AutoCompletePhoneNumberEntryResult MatchedContacts;
               most_fi_tcl_PhonBkAutocompleteMultipleMatchListItem	items;

               MatchedContacts = m_sqlite->AutoCompletePhNoSearchRecords[SearchIndex].at(u8Index);

               ba = MatchedContacts.FirstName.toUtf8();
               items.sFirstName.bSet(ba.constData());
               ba = MatchedContacts.LastName.toUtf8();
               items.sLastName.bSet(ba.constData());
               ba = MatchedContacts.PhoneNumber.toUtf8();
               items.sPhoneNumber.bSet(ba.constData());

               methodResult->oAutocompleteMultipleMatchList.oItems.push_back(items);
            }
            //FIX CMG3GB-505 Upon disconnecting a call/call is established/entering numbers in dialpad, resets when doing handsfree operation was observed
            methodResult->sAutocompleteMatchPhoneNumber.bSet(methodResult->oAutocompleteMultipleMatchList.oItems[0].sPhoneNumber.szValue);
            //End of fix
         }
      }

      //FIX CMG3GB-505 Upon disconnecting a call/call is established/entering numbers in dialpad, resets when doing handsfree operation was observed
      AutoCompletePhNoSearchIndex = (AutoCompletePhNoSearchIndex==0)?1:0;//Toggle the values
      //End of fix
   }
}

//FIX SUZUKI-1114 API for numeric value to indicate number of dial-able phone numbers available to a contact item in Phonebook
/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::RequestSlicePhoneBookListExtended
*
* DESCRIPTION:
*              Creates a subset of the list pointed by listhandle
*
*
* PARAMETER: [IN] methodStart FC_PB_DB_NUM_0:- Input arguments like listhandles are in this class
*            [OUT] methodResult :- The acutal sub list is passed in this class (with PhoneNumberCount)
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_List::RequestSlicePhoneBookListExtended(
   const most_phonbkfi_tclMsgRequestPhoneBookListSliceExtendedMethodStart * methodStart,
   most_phonbkfi_tclMsgRequestPhoneBookListSliceExtendedMethodResult* methodResult )
{
   ETG_TRACE_USR4 (( "RequestSlicePhoneBookListExtended entered "));

   QTime DebugTime;
   DebugTime.start();

   methodResult->u16ListHandle = methodStart->u16ListHandle;
   if (false == m_hashPbList.contains(methodStart->u16ListHandle))
   {
      ETG_TRACE_USR4 (( "FC_PhoneBook_List :- WARNING - empty list returned in RequestSlice "));
      return false;
   }

   FC_PhoneBook_List_Data* data = m_hashPbList.value( methodStart->u16ListHandle, NULLPTR );
  //CMG3G-14313-Coverity Fix-NULL check for data
   if(!data)
   {
      ETG_TRACE_ERR (( "FC_PhoneBook_List :- Data/HashList is NULL "));
      return false;
   }
   //end of Fix
   unsigned listLen = (unsigned)data->contactList.size();

   if ( (methodStart->u16WindowSize < 1) ||
      (methodStart->u32WindowStart >= listLen) )
   {
      ETG_TRACE_USR4(("WARNING - negative number requested in windowStart or WindowSize, or zero windowSize, returning empty list"));
      return false;
   }

   unsigned end = methodStart->u32WindowStart + methodStart->u16WindowSize;

   if ( end > listLen )
      end = listLen;

   ETG_TRACE_USR4(( "TOTAL Items = %u",listLen));
   ETG_TRACE_USR4(( "START = %u",methodStart->u32WindowStart));
   ETG_TRACE_USR4(( "END = %u",end-1));
   ETG_TRACE_USR4(( "SIZE = %u",methodStart->u16WindowSize));
   ETG_TRACE_USR4 (( "List type is: %d", ETG_CENUM(FC_PhoneBook_SQLite::enListType, data->listType) ));

   //CMG3G-8447. Lint Fix
   if (!m_sqlite)
   {
      ETG_TRACE_USR4(("m_sqlite is NULL. Hence returning."));
      return false;
   }

   QList<PbkAdditionalContactInfo> PbkcontactList;

   //Fix : CMG3G-12880
   FC_Device_Details *pDeviceObject;
   pDeviceObject =  FC_Device_Details::deviceHandleToObjectInstance(data->u8DeviceHandle);

   if( pDeviceObject != NULL)
   {
      // Fix for NCG3D-159329: Deactivated the "bIsDeviceProbablyLocked() check" for LIST_TYPE_VEHICLE (Quick-Dial) Lists
      // User shall be able to access the Quick-Dial Lists in the situation that the phone is "probably" locked.
      if( (data->listType == FC_PhoneBook_SQLite::ListType_VEHICLE) || (FALSE == pDeviceObject->bIsDeviceProbablyLocked()) )
      {
         // CMG3G-7853: Phonebook list slice result to have the number and email details
         m_sqlite->getAdditionalContactInfo(methodStart->u32WindowStart, end, PbkcontactList, data->contactList, data->u8DeviceHandle, data->listType);
      }
      else
      {
         ETG_TRACE_USR2(("Device is locked hence mocking the phonebook list"));
         return true;
      }
   }
   else
   {
      ETG_TRACE_USR4((" Device object not found for given device handle"));
   }

   ETG_TRACE_USR4(( "Size of Phonebook contact slice List: %d", PbkcontactList.size() ));

   tU8 u8ContactCount = 0;
   if(PbkcontactList.size()!=0)
   {
      for (unsigned counter = methodStart->u32WindowStart; (counter < end) && (u8ContactCount<PbkcontactList.size()); ++counter )
      {
         most_fi_tcl_PhonBkPhoneBookListSliceResultExtendedItem item;

         ConvertExtended( &data->contactList.at((int)counter), &item, &PbkcontactList.at((int)u8ContactCount++));
         methodResult->oPhoneBookListSliceResultExtended.oItems.push_back( item );
      }

      ETG_TRACE_USR2(( " -PBDL.Perf- 'RequestSlicePhoneBookListExtended' exited after %d ms. SliceList-Size= %d ", DebugTime.elapsed(), PbkcontactList.size() ));
      return true;
   }
   else
   {
      ETG_TRACE_USR4(( "The PbkcontactList size is empty: %d", PbkcontactList.size() ));
      
      ETG_TRACE_USR2(( " -PBDL.Perf- 'RequestSlicePhoneBookListExtended' exited after %d ms. SliceList-Size= %d ", DebugTime.elapsed(), PbkcontactList.size() ));
      return false;
   }
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_List::ConvertExtended() for additional info Phonenumber,Emailaddress
*
* DESCRIPTION:
*              Converts the Qt format structure into the MOST format structure
*
*
* PARAMETER: [IN] contact :- The Qt format contact info
*            [OUT] item :- MOST format contact info
* RETURNVALUE: None.
*
*******************************************************************************/
void FC_PhoneBook_List::ConvertExtended(const FC_PhoneBook_Contact_Detail_Short* contact, most_fi_tcl_PhonBkPhoneBookListSliceResultExtendedItem* item, const PbkAdditionalContactInfo* Pbkcontact)
{
   ETG_TRACE_USR4(("ConvertExtended() entered."));

   item->u32ContactHandle = contact->contactHandle;

   // Set the quick options
   item->bQuickDial  = contact->bQuickDial;
   item->bQuickRoute = contact->bQuickRoute;
   item->bQuickText  = contact->bQuickText;

   QByteArray ba = contact->firstName.toUtf8();
   item->sFirstName.bSet( ba.constData() );
   ETG_TRACE_USR2(( " FIRST NAME = %s", ba.constData() ));

   ba = contact->lastName.toUtf8();
   item->sLastName.bSet( ba.constData() );
   ETG_TRACE_USR2(( " LAST NAME = %s", ba.constData() ));

   ba = contact->poiName.toUtf8();
   item->sPOIName.bSet(ba.constData());

   item->u32ContactNameTag
         = GENERATE_TAG(contact->contactHandle, CONTACTNAME_TAG);

   //Update contact type
   item->e8ContactType.enType
         = (most_fi_tcl_e8_PhonBkContactType::tenType) Pbkcontact->u8ContactType;
   ETG_TRACE_USR2((" ContactType = %d", (most_fi_tcl_e8_PhonBkContactType::tenType) item->e8ContactType.enType));

   // Re-Mapping of Phone numbers for the Method Result
   QList<PbkPhoneNoItem> qPhoneNoItemList;

   if(Pbkcontact->PhoneNum1 != "")
   {
      PbkPhoneNoItem oPhNoItem;
      oPhNoItem.qsPhoneNumber = Pbkcontact->PhoneNum1;
      oPhNoItem.enPhoneNumberType = static_cast<tenPhonBkNumberType>(Pbkcontact->NumberType1);
      ETG_TRACE_USR2(( " oPhNoItem.enPhoneNumberType1 = %d", oPhNoItem.enPhoneNumberType ));
      qPhoneNoItemList << oPhNoItem;
   }

   if(Pbkcontact->PhoneNum2 != "")
   {
      PbkPhoneNoItem oPhNoItem;
      oPhNoItem.qsPhoneNumber = Pbkcontact->PhoneNum2;
      oPhNoItem.enPhoneNumberType = static_cast<tenPhonBkNumberType>(Pbkcontact->NumberType2);
      ETG_TRACE_USR2(( " oPhNoItem.enPhoneNumberType2 = %d", oPhNoItem.enPhoneNumberType ));
      qPhoneNoItemList << oPhNoItem;
   }

   if(Pbkcontact->PhoneNum3 != "")
   {
      PbkPhoneNoItem oPhNoItem;
      oPhNoItem.qsPhoneNumber = Pbkcontact->PhoneNum3;
      oPhNoItem.enPhoneNumberType = static_cast<tenPhonBkNumberType>(Pbkcontact->NumberType3);
      ETG_TRACE_USR2(( " oPhNoItem.enPhoneNumberType3 = %d", oPhNoItem.enPhoneNumberType ));
      qPhoneNoItemList << oPhNoItem;
   }

   if(Pbkcontact->PhoneNum4 != "")
   {
      PbkPhoneNoItem oPhNoItem;
      oPhNoItem.qsPhoneNumber = Pbkcontact->PhoneNum4;
      oPhNoItem.enPhoneNumberType =  static_cast<tenPhonBkNumberType>(Pbkcontact->NumberType4);
      ETG_TRACE_USR2(( " oPhNoItem.enPhoneNumberType4 = %d", oPhNoItem.enPhoneNumberType ));
      qPhoneNoItemList << oPhNoItem;
   }

   if(Pbkcontact->PhoneNum5 != "")
   {
      PbkPhoneNoItem oPhNoItem;
      oPhNoItem.qsPhoneNumber = Pbkcontact->PhoneNum5;
      oPhNoItem.enPhoneNumberType =  static_cast<tenPhonBkNumberType>(Pbkcontact->NumberType5);
      ETG_TRACE_USR2(( " oPhNoItem.enPhoneNumberType5 = %d", oPhNoItem.enPhoneNumberType ));
      qPhoneNoItemList << oPhNoItem;
   }

   if(Pbkcontact->PhoneNum6 != "")
   {
      PbkPhoneNoItem oPhNoItem;
      oPhNoItem.qsPhoneNumber = Pbkcontact->PhoneNum6;
      oPhNoItem.enPhoneNumberType =  static_cast<tenPhonBkNumberType>(Pbkcontact->NumberType6);
      ETG_TRACE_USR2(( " oPhNoItem.enPhoneNumberType6 = %d", oPhNoItem.enPhoneNumberType ));
      qPhoneNoItemList << oPhNoItem;
   }

   if(Pbkcontact->PhoneNum7 != "")
   {
      PbkPhoneNoItem oPhNoItem;
      oPhNoItem.qsPhoneNumber = Pbkcontact->PhoneNum7;
      oPhNoItem.enPhoneNumberType =   static_cast<tenPhonBkNumberType>(Pbkcontact->NumberType7);
      ETG_TRACE_USR2(( " oPhNoItem.enPhoneNumberType7 = %d", oPhNoItem.enPhoneNumberType ));
      qPhoneNoItemList << oPhNoItem;
   }

   if(Pbkcontact->PhoneNum8 != "")
   {
      PbkPhoneNoItem oPhNoItem;
      oPhNoItem.qsPhoneNumber = Pbkcontact->PhoneNum8;
      oPhNoItem.enPhoneNumberType =  static_cast<tenPhonBkNumberType>(Pbkcontact->NumberType8);
      ETG_TRACE_USR2(( " oPhNoItem.enPhoneNumberType8 = %d", oPhNoItem.enPhoneNumberType ));
      qPhoneNoItemList << oPhNoItem;
   }

   tU8 qPhoneNoItemListSize = (tU8)(qPhoneNoItemList.size());
   ETG_TRACE_USR2(( "qPhoneNoItemListSize = %d", qPhoneNoItemListSize));

   if (qPhoneNoItemListSize > 0)
   {
      for(tU8 u8PhNoIndex = 0; u8PhNoIndex < qPhoneNoItemListSize; u8PhNoIndex++)
      {
         PbkPhoneNoItem oPhNoItem = qPhoneNoItemList.at(u8PhNoIndex);
         QByteArray qbPhoneNo = oPhNoItem.qsPhoneNumber.toUtf8();
         ETG_TRACE_USR2(( "oPhoneNumber%d = %s", u8PhNoIndex+1, qbPhoneNo.constData() ));

         switch (u8PhNoIndex)
         {
            case 0:
            {
               item->oPhoneNumber1.sNumber.bSet(qbPhoneNo.constData() );
               item->oPhoneNumber1.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)oPhNoItem.enPhoneNumberType;
               ETG_TRACE_USR2(( " oPhoneNumber1type = %d", item->oPhoneNumber1.e8Type.enType ));
            }
            break;

            case 1:
            {
               item->oPhoneNumber2.sNumber.bSet(qbPhoneNo.constData() );
               item->oPhoneNumber2.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)oPhNoItem.enPhoneNumberType;
               ETG_TRACE_USR2(( " oPhoneNumber2type = %d", item->oPhoneNumber2.e8Type.enType ));
            }
            break;

            case 2:
            {
               item->oPhoneNumber3.sNumber.bSet(qbPhoneNo.constData() );
               item->oPhoneNumber3.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)oPhNoItem.enPhoneNumberType;
               ETG_TRACE_USR2(( " oPhoneNumber3type = %d", item->oPhoneNumber3.e8Type.enType ));
            }
            break;

            case 3:
            {
               item->oPhoneNumber4.sNumber.bSet(qbPhoneNo.constData() );
               item->oPhoneNumber4.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)oPhNoItem.enPhoneNumberType;
               ETG_TRACE_USR2(( " oPhoneNumber4type = %d", item->oPhoneNumber4.e8Type.enType ));
            }
            break;

            case 4:
            {
               item->oPhoneNumber5.sNumber.bSet(qbPhoneNo.constData() );
               item->oPhoneNumber5.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)oPhNoItem.enPhoneNumberType;
               ETG_TRACE_USR2(( " oPhoneNumber5type = %d", item->oPhoneNumber5.e8Type.enType ));
            }
            break;

            case 5:
            {
               item->oPhoneNumber6.sNumber.bSet(qbPhoneNo.constData() );
               item->oPhoneNumber6.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)oPhNoItem.enPhoneNumberType;
               ETG_TRACE_USR2(( " oPhoneNumber6type = %d", item->oPhoneNumber6.e8Type.enType ));
            }
            break;

            case 6:
            {
               item->oPhoneNumber7.sNumber.bSet(qbPhoneNo.constData());
               item->oPhoneNumber7.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)oPhNoItem.enPhoneNumberType;
               ETG_TRACE_USR2(( " oPhoneNumber7type = %d", item->oPhoneNumber7.e8Type.enType ));
            }
            break;

            case 7:
            {
               item->oPhoneNumber8.sNumber.bSet(qbPhoneNo.constData());
               item->oPhoneNumber8.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)oPhNoItem.enPhoneNumberType;
               ETG_TRACE_USR2(( " oPhoneNumber8type = %d", item->oPhoneNumber8.e8Type.enType ));
            }
            break;

            default:
            ETG_TRACE_USR2(( " oPhoneNumbertype : Default value"));
         }
      }
   }

   item->u8PhoneNumberCount = qPhoneNoItemListSize;
   ETG_TRACE_USR2(( " Final u8PhoneNumberCount = %d", item->u8PhoneNumberCount ));

   // Re-Mapping for E-Mail addresses
   QList<QString> qEmailAddrList;

   if(Pbkcontact->EmailAdd1 != "")
   {
      qEmailAddrList << Pbkcontact->EmailAdd1;
   }

   if(Pbkcontact->EmailAdd2 != "")
   {
      qEmailAddrList << Pbkcontact->EmailAdd2;
   }

   if(Pbkcontact->EmailAdd3 != "")
   {
      qEmailAddrList << Pbkcontact->EmailAdd3;
   }

   for(tU8 u8EmailCount = 0; u8EmailCount < Pbkcontact->u8EmailAddrCount; u8EmailCount++)
   {
      QByteArray qbEmailaddr = qEmailAddrList.at(u8EmailCount).toUtf8();

      ETG_TRACE_USR2(( "oEmailAddr%d = %s", u8EmailCount, qbEmailaddr.constData() ));
      switch (u8EmailCount)
      {
         case 0:
         {
            item->oEmailAddr1.bSet(qbEmailaddr.constData());
         }
         break;
         case 1:
         {
            item->oEmailAddr2.bSet(qbEmailaddr.constData());
         }
         break;
         case 2:
         {
            item->oEmailAddr3.bSet(qbEmailaddr.constData());
         }
         break;
         default:
         ETG_TRACE_USR2(( " oEmailAddr%d : unknown value", u8EmailCount));
      }
   }

   ETG_TRACE_USR2(( " u8EmailCount = %d", Pbkcontact->u8EmailAddrCount ));
   item->u8EmailCount = Pbkcontact->u8EmailAddrCount;

   ETG_TRACE_USR2(( " u8AddressesCount = %d", Pbkcontact->u8AddressesCount ));
   item->u8AddressesCount = Pbkcontact->u8AddressesCount;

   ETG_TRACE_USR2(( " PhotoAvailable = %d", Pbkcontact->bPhotoAvailable ));
   item->bPhotoAvailable = Pbkcontact->bPhotoAvailable;

}


//FIX GMMY15-1578 Removing redundant objects which are created in RAM for different clients & sorted lists.
/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_List::vClearPBListAndHashListEntriesForListHandle()
*
* DESCRIPTION: Deletes entries from m_hashPbList and m_multiHashPBClientRequests
*              for a particular list handle.
*
* PARAMETER: [IN] u16ListHandle :- List handle whose corresponding entries are to be deleted
*                 from hash tables.
*            [OUT] None.
* RETURNVALUE: None.
*
*******************************************************************************/
void FC_PhoneBook_List::vClearPBListAndHashListEntriesForListHandle(tU16 u16ListHandle, tU8 u8DeviceHandle)
{
   ETG_TRACE_USR4(( " vClearPBListAndHashListEntriesForListHandle entered with list handle: %d", u16ListHandle));
   ETG_TRACE_USR4((" Device Handle: %d", u8DeviceHandle));

   FC_PhoneBook_List_Data* toDelete = NULL;

   ETG_TRACE_USR4((" Clearing the list and removing from hash... "));
   toDelete = m_hashPbList.take(u16ListHandle);
   if (toDelete)
   {
      toDelete->contactList.clear();
      delete toDelete;
      toDelete = NULLPTR;
   }
   m_hashPbList.squeeze();

   //Delete entries of all requests from this device which is stored in Multihash as it is no longer required
   m_multiHashPBClientRequests.remove(u16ListHandle);
   m_multiHashPBClientRequests.squeeze();
}
//End of fix GMMY15-1578


//Fix CMG3G-8365 IS2421_Configurable phone number type representation@FC_Phonebook
/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::AddContactExtended
*
* DESCRIPTION: Adds the contact in VPBx where x intended for 2 to 11
**
*
* PARAMETER: [IN] methodStart :-
*            [OUT] methodResult :-
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_List::AddContactExtended(
                                       const most_phonbkfi_tclMsgAddContactExtendedMethodStart* methodStart,
                                       most_phonbkfi_tclMsgAddContactExtendedMethodResult* methodResult)
{
   ETG_TRACE_USR4(( " AddContactExtended entered" ));

   quint32 ContactHandle = 0;

   if(m_sqlite)
   {
      if (m_sqlite->GetNoOfContactsInVPB(methodStart->u8DeviceHandle) >= MAX_ENTRIES_VEHICLE_PHONEBOOK )
      {
         ETG_TRACE_ERR(("Adding of Contact to VPB failed: NO OF CONTACTS REACHED TO MAX LIMIT OF %d ", MAX_ENTRIES_VEHICLE_PHONEBOOK));
         methodResult->u32ContactHandle = ContactHandle;
         methodResult->e8ContactAddStatusExtended.enType = most_fi_tcl_e8_PhonBkContactAddStatus::FI_EN_E8CONTACT_ADD_FAILURE;
         return false;
      }

      FC_PhoneBook_Contact_Detail Contact;

      Contact.firstName= QString::fromUtf8(methodStart->oContactDetailsExtended.sFirstName.szValue);
      Contact.lastName= QString::fromUtf8(methodStart->oContactDetailsExtended.sLastName.szValue);
      Contact.poiName= QString::fromUtf8(methodStart->oContactDetailsExtended.sPOIName.szValue);
      Contact.PPPhoneNum1= QString::fromUtf8(methodStart->oContactDetailsExtended.oPhoneNumber1.sNumber.szValue);
      Contact.PhoneNum1= StripTelephoneNumber(methodStart->oContactDetailsExtended.oPhoneNumber1.sNumber.szValue);
      Contact.PPPhoneNum2= QString::fromUtf8(methodStart->oContactDetailsExtended.oPhoneNumber2.sNumber.szValue);
      Contact.PhoneNum2= StripTelephoneNumber(methodStart->oContactDetailsExtended.oPhoneNumber2.sNumber.szValue);
      Contact.PPPhoneNum3= QString::fromUtf8(methodStart->oContactDetailsExtended.oPhoneNumber3.sNumber.szValue);
      Contact.PhoneNum3= StripTelephoneNumber(methodStart->oContactDetailsExtended.oPhoneNumber3.sNumber.szValue);
      Contact.PPPhoneNum4= QString::fromUtf8(methodStart->oContactDetailsExtended.oPhoneNumber4.sNumber.szValue);
      Contact.PhoneNum4= StripTelephoneNumber(methodStart->oContactDetailsExtended.oPhoneNumber4.sNumber.szValue);
      Contact.PPPhoneNum5= QString::fromUtf8(methodStart->oContactDetailsExtended.oPhoneNumber5.sNumber.szValue);
      Contact.PhoneNum5= StripTelephoneNumber(methodStart->oContactDetailsExtended.oPhoneNumber5.sNumber.szValue);
      Contact.PPPhoneNum6= QString::fromUtf8(methodStart->oContactDetailsExtended.oPhoneNumber6.sNumber.szValue);
      Contact.PhoneNum6= StripTelephoneNumber(methodStart->oContactDetailsExtended.oPhoneNumber6.sNumber.szValue);
      Contact.PPPhoneNum7= QString::fromUtf8(methodStart->oContactDetailsExtended.oPhoneNumber7.sNumber.szValue);
      Contact.PhoneNum7= StripTelephoneNumber(methodStart->oContactDetailsExtended.oPhoneNumber7.sNumber.szValue);
      Contact.PPPhoneNum8= QString::fromUtf8(methodStart->oContactDetailsExtended.oPhoneNumber8.sNumber.szValue);
      Contact.PhoneNum8= StripTelephoneNumber(methodStart->oContactDetailsExtended.oPhoneNumber8.sNumber.szValue);
      Contact.NumberType1= methodStart->oContactDetailsExtended.oPhoneNumber1.e8Type.enType;
      Contact.NumberType2= methodStart->oContactDetailsExtended.oPhoneNumber2.e8Type.enType;
      Contact.NumberType3= methodStart->oContactDetailsExtended.oPhoneNumber3.e8Type.enType;
      Contact.NumberType4= methodStart->oContactDetailsExtended.oPhoneNumber4.e8Type.enType;
      Contact.NumberType5= methodStart->oContactDetailsExtended.oPhoneNumber5.e8Type.enType;
      Contact.NumberType6= methodStart->oContactDetailsExtended.oPhoneNumber6.e8Type.enType;
      Contact.NumberType7= methodStart->oContactDetailsExtended.oPhoneNumber7.e8Type.enType;
      Contact.NumberType8= methodStart->oContactDetailsExtended.oPhoneNumber8.e8Type.enType;

      Contact.NumberSubstr1 = Contact.PhoneNum1.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);
      Contact.NumberSubstr2 = Contact.PhoneNum2.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);
      Contact.NumberSubstr3 = Contact.PhoneNum3.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);
      Contact.NumberSubstr4 = Contact.PhoneNum4.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);
      Contact.NumberSubstr5 = Contact.PhoneNum5.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);
      Contact.NumberSubstr6 = Contact.PhoneNum6.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);
      Contact.NumberSubstr7 = Contact.PhoneNum7.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);
      Contact.NumberSubstr8 = Contact.PhoneNum8.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);

      Contact.emailAdd1= QString::fromUtf8(methodStart->oContactDetailsExtended.sEmailAddress1.szValue);
      Contact.emailAdd2= QString::fromUtf8(methodStart->oContactDetailsExtended.sEmailAddress2.szValue);
      Contact.emailAdd3= QString::fromUtf8(methodStart->oContactDetailsExtended.sEmailAddress3.szValue);
      Contact.homeAdd= QString::fromUtf8(methodStart->oContactDetailsExtended.sHomeAddress.szValue);
      Contact.workAdd= QString::fromUtf8(methodStart->oContactDetailsExtended.sWorkAddress.szValue);
      Contact.otherAdd= QString::fromUtf8(methodStart->oContactDetailsExtended.sOtherAddress.szValue);
      Contact.category= QString::fromUtf8(methodStart->oContactDetailsExtended.sCategory.szValue);
      Contact.notes= QString::fromUtf8(methodStart->oContactDetailsExtended.sNotes.szValue);
      Contact.homeGeoCodeLatLongValidity = methodStart->oContactDetailsExtended.oHomeAddressLatLong.bLatLongValidity;
      Contact.homeGeoCodeLatitude =  methodStart->oContactDetailsExtended.oHomeAddressLatLong.s32Latitude;
      Contact.homeGeocodeLongitude = methodStart->oContactDetailsExtended.oHomeAddressLatLong.s32Longitude;
      Contact.workGeoCodeLatLongValidity = methodStart->oContactDetailsExtended.oWorkAddressLatLong.bLatLongValidity;
      Contact.workGeoCodeLatitude =  methodStart->oContactDetailsExtended.oWorkAddressLatLong.s32Latitude;
      Contact.workGeocodeLongitude = methodStart->oContactDetailsExtended.oWorkAddressLatLong.s32Longitude;
      Contact.otherGeoCodeLatLongValidity = methodStart->oContactDetailsExtended.oOtherAddressLatLong.bLatLongValidity;
      Contact.otherGeoCodeLatitude = methodStart->oContactDetailsExtended.oOtherAddressLatLong.s32Latitude;
      Contact.otherGeocodeLongitude = methodStart->oContactDetailsExtended.oOtherAddressLatLong.s32Longitude;
      Contact.photoDlSize = 0;
      Contact.photo       = "";   //.clear();
      Contact.photoSize   = 0;
      Contact.photoType   = "";   //.clear();
      Contact.photoCRC    = 0;
      Contact.photoDlCRC  = 0;
      Contact.APL1Size    = 0;
      Contact.crcAPL1     = 0;
      Contact.u32contactPosition = m_sqlite->GetNoOfContactsInVPB(methodStart->u8DeviceHandle);
      Contact.u32contactPosition = Contact.u32contactPosition + 1;
      ETG_TRACE_USR2(("  contactPosition = %d", Contact.u32contactPosition ));

      if(!m_sqlite->InsertSingleRecord(Contact, methodStart->u8DeviceHandle, ContactHandle,true))
      {
         ETG_TRACE_ERR(("   Adding of contact failed          "));
         methodResult->u32ContactHandle= ContactHandle;
         methodResult->e8ContactAddStatusExtended.enType= most_fi_tcl_e8_PhonBkContactAddStatus::FI_EN_E8CONTACT_ADD_FAILURE;
         return false;
      }
      else
      {
         m_sqlite->IncNoOfContactsVAB(methodStart->u8DeviceHandle);
         ETG_TRACE_USR2(("  Add contact extended success: total contacts in VPB = %d", m_sqlite->GetNoOfContactsInVAB(methodStart->u8DeviceHandle)));
         methodResult->u32ContactHandle= ContactHandle;
         methodResult->e8ContactAddStatusExtended.enType= most_fi_tcl_e8_PhonBkContactAddStatus::FI_EN_E8CONTACT_ADD_SUCCESS;
      }
   }
   else
   {
      ETG_TRACE_ERR((" m_sqlite is NULL!!!"));
      return FALSE;
   }

   return true;
}

/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::EditContactExtended
*
* DESCRIPTION:Edits the contact in VPBx where x intended for 2 to 11
**
*
* PARAMETER: [IN] methodStart :-
*            [OUT] methodResult :- Error if any
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_List::EditContactExtended(
                                        const most_phonbkfi_tclMsgEditContactExtendedMethodStart* methodStart,
                                        most_phonbkfi_tclMsgEditContactExtendedMethodResult* methodResult)
{
   ETG_TRACE_USR4(( " EditContactExtended entered" ));

   FC_PhoneBook_Contact_Detail Contact; //Object to store the data
   quint32 ContactHandle= methodStart->u32ContactHandle;

   Contact.contactHandle = ContactHandle;
   Contact.firstName = QString::fromUtf8(methodStart->oContactDetailsExtended.sFirstName.szValue);
   Contact.lastName = QString::fromUtf8(methodStart->oContactDetailsExtended.sLastName.szValue);
   Contact.poiName = QString::fromUtf8(methodStart->oContactDetailsExtended.sPOIName.szValue);
   Contact.PPPhoneNum1 = QString::fromUtf8(methodStart->oContactDetailsExtended.oPhoneNumber1.sNumber.szValue);
   Contact.PhoneNum1 = StripTelephoneNumber(methodStart->oContactDetailsExtended.oPhoneNumber1.sNumber.szValue);
   Contact.PPPhoneNum2 = QString::fromUtf8(methodStart->oContactDetailsExtended.oPhoneNumber2.sNumber.szValue);
   Contact.PhoneNum2 = StripTelephoneNumber(methodStart->oContactDetailsExtended.oPhoneNumber2.sNumber.szValue);
   Contact.PPPhoneNum3 = QString::fromUtf8(methodStart->oContactDetailsExtended.oPhoneNumber3.sNumber.szValue);
   Contact.PhoneNum3 = StripTelephoneNumber(methodStart->oContactDetailsExtended.oPhoneNumber3.sNumber.szValue);
   Contact.PPPhoneNum4 = QString::fromUtf8(methodStart->oContactDetailsExtended.oPhoneNumber4.sNumber.szValue);
   Contact.PhoneNum4 = StripTelephoneNumber(methodStart->oContactDetailsExtended.oPhoneNumber4.sNumber.szValue);
   Contact.PPPhoneNum5 = QString::fromUtf8(methodStart->oContactDetailsExtended.oPhoneNumber5.sNumber.szValue);
   Contact.PhoneNum5 = StripTelephoneNumber(methodStart->oContactDetailsExtended.oPhoneNumber5.sNumber.szValue);
   Contact.PPPhoneNum6 = QString::fromUtf8(methodStart->oContactDetailsExtended.oPhoneNumber6.sNumber.szValue);
   Contact.PhoneNum6 = StripTelephoneNumber(methodStart->oContactDetailsExtended.oPhoneNumber6.sNumber.szValue);
   Contact.PPPhoneNum7 = QString::fromUtf8(methodStart->oContactDetailsExtended.oPhoneNumber7.sNumber.szValue);
   Contact.PhoneNum7 = StripTelephoneNumber(methodStart->oContactDetailsExtended.oPhoneNumber7.sNumber.szValue);
   Contact.PPPhoneNum8 = QString::fromUtf8(methodStart->oContactDetailsExtended.oPhoneNumber8.sNumber.szValue);
   Contact.PhoneNum8 = StripTelephoneNumber(methodStart->oContactDetailsExtended.oPhoneNumber8.sNumber.szValue);

   Contact.NumberSubstr1 = Contact.PhoneNum1.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);
   Contact.NumberSubstr2 = Contact.PhoneNum2.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);
   Contact.NumberSubstr3 = Contact.PhoneNum3.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);
   Contact.NumberSubstr4 = Contact.PhoneNum4.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);
   Contact.NumberSubstr5 = Contact.PhoneNum5.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);
   Contact.NumberSubstr6 = Contact.PhoneNum6.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);
   Contact.NumberSubstr7 = Contact.PhoneNum7.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);
   Contact.NumberSubstr8 = Contact.PhoneNum8.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN);

   Contact.NumberType1= methodStart->oContactDetailsExtended.oPhoneNumber1.e8Type.enType;
   Contact.NumberType2= methodStart->oContactDetailsExtended.oPhoneNumber2.e8Type.enType;
   Contact.NumberType3= methodStart->oContactDetailsExtended.oPhoneNumber3.e8Type.enType;
   Contact.NumberType4= methodStart->oContactDetailsExtended.oPhoneNumber4.e8Type.enType;
   Contact.NumberType5= methodStart->oContactDetailsExtended.oPhoneNumber5.e8Type.enType;
   Contact.NumberType6= methodStart->oContactDetailsExtended.oPhoneNumber6.e8Type.enType;
   Contact.NumberType7= methodStart->oContactDetailsExtended.oPhoneNumber7.e8Type.enType;
   Contact.NumberType8= methodStart->oContactDetailsExtended.oPhoneNumber8.e8Type.enType;
   Contact.emailAdd1 = QString::fromUtf8(methodStart->oContactDetailsExtended.sEmailAddress1.szValue);
   Contact.emailAdd2 = QString::fromUtf8(methodStart->oContactDetailsExtended.sEmailAddress2.szValue);
   Contact.emailAdd3 = QString::fromUtf8(methodStart->oContactDetailsExtended.sEmailAddress3.szValue);
   Contact.homeAdd = QString::fromUtf8(methodStart->oContactDetailsExtended.sHomeAddress.szValue);
   Contact.workAdd = QString::fromUtf8(methodStart->oContactDetailsExtended.sWorkAddress.szValue);
   Contact.otherAdd = QString::fromUtf8(methodStart->oContactDetailsExtended.sOtherAddress.szValue);
   Contact.category = QString::fromUtf8(methodStart->oContactDetailsExtended.sCategory.szValue);
   Contact.notes = QString::fromUtf8(methodStart->oContactDetailsExtended.sNotes.szValue);
   Contact.homeGeoCodeLatLongValidity = methodStart->oContactDetailsExtended.oHomeAddressLatLong.bLatLongValidity;
   Contact.homeGeoCodeLatitude = methodStart->oContactDetailsExtended.oHomeAddressLatLong.s32Latitude;
   Contact.homeGeocodeLongitude = methodStart->oContactDetailsExtended.oHomeAddressLatLong.s32Longitude;
   Contact.workGeoCodeLatLongValidity = methodStart->oContactDetailsExtended.oWorkAddressLatLong.bLatLongValidity;
   Contact.workGeoCodeLatitude = methodStart->oContactDetailsExtended.oWorkAddressLatLong.s32Latitude;
   Contact.workGeocodeLongitude = methodStart->oContactDetailsExtended.oWorkAddressLatLong.s32Longitude;
   Contact.otherGeoCodeLatLongValidity = methodStart->oContactDetailsExtended.oOtherAddressLatLong.bLatLongValidity;
   Contact.otherGeoCodeLatitude = methodStart->oContactDetailsExtended.oOtherAddressLatLong.s32Latitude;
   Contact.otherGeocodeLongitude = methodStart->oContactDetailsExtended.oOtherAddressLatLong.s32Longitude;
   Contact.photoDlSize = 0;
   Contact.photo = "";   //.clear();
   Contact.photoSize = 0;
   Contact.photoType = "";   //.clear();
   Contact.photoCRC = 0;
   Contact.photoDlCRC = 0;
   Contact.APL1Size = 0;
   Contact.crcAPL1 = 0;
   bool bRetValue = false;
   if(!m_sqlite->UpdateRecord(Contact, ContactHandle))
   {
      ETG_TRACE_ERR((" Edit contact extended failed "));
      methodResult->u32ContactHandle= ContactHandle;
      methodResult->e8ContactEditStatusExtended.enType= most_fi_tcl_e8_PhonBkContactEditStatus::FI_EN_E8CONTACT_EDIT_FAILURE;
   }
   else
   {
      methodResult->u32ContactHandle= ContactHandle;
      methodResult->e8ContactEditStatusExtended.enType= most_fi_tcl_e8_PhonBkContactEditStatus::FI_EN_E8CONTACT_EDIT_SUCCESS;
      bRetValue = true;
   }
   return bRetValue;
}

/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::GetContactDetailsExtended
*
* DESCRIPTION: Gets the contact detail of VPBx where x intended for 2 to 11
**
*
* PARAMETER: [IN] methodStart :-
*            [OUT] methodResult :- Error if any
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_List::GetContactDetailsExtended(
   const most_phonbkfi_tclMsgGetContactDetailsExtendedMethodStart* methodStart,
   most_phonbkfi_tclMsgGetContactDetailsExtendedMethodResult* methodResult)
{
   ETG_TRACE_USR4(( " GetContactDetailsExtended entered" ));
   bool bRetValue = false;
   FC_PhoneBook_Contact_Detail contact;
   quint32 ContactHandle = methodStart->u32ContactHandle;
   QByteArray tmpArray;

   if(m_sqlite != NULL && true == m_sqlite->GetContactDetails(ContactHandle, contact))
   {
      ETG_TRACE_USR4((" m_sqlite->GetContactDetais Success"));
      tmpArray = contact.firstName.toUtf8();
      methodResult->oContactDetailsExtended.sFirstName.bSet( tmpArray.constData());

      tmpArray = contact.lastName.toUtf8();
      methodResult->oContactDetailsExtended.sLastName.bSet( tmpArray.constData());

      tmpArray = contact.poiName.toUtf8();
      methodResult->oContactDetailsExtended.sPOIName.bSet(tmpArray.constData());

      tmpArray = contact.PhoneNum1.toUtf8();
      methodResult->oContactDetailsExtended.oPhoneNumber1.sNumber.bSet(tmpArray.constData());

      tmpArray = contact.PhoneNum2.toUtf8();
      methodResult->oContactDetailsExtended.oPhoneNumber2.sNumber.bSet(tmpArray.constData());

      tmpArray = contact.PhoneNum3.toUtf8();
      methodResult->oContactDetailsExtended.oPhoneNumber3.sNumber.bSet(tmpArray.constData());

      tmpArray = contact.PhoneNum4.toUtf8();
      methodResult->oContactDetailsExtended.oPhoneNumber4.sNumber.bSet(tmpArray.constData());

      tmpArray = contact.PhoneNum5.toUtf8();
      methodResult->oContactDetailsExtended.oPhoneNumber5.sNumber.bSet(tmpArray.constData());

      tmpArray = contact.PhoneNum6.toUtf8();
      methodResult->oContactDetailsExtended.oPhoneNumber6.sNumber.bSet(tmpArray.constData());

      tmpArray = contact.PhoneNum7.toUtf8();
      methodResult->oContactDetailsExtended.oPhoneNumber7.sNumber.bSet(tmpArray.constData());

      tmpArray = contact.PhoneNum8.toUtf8();
      methodResult->oContactDetailsExtended.oPhoneNumber8.sNumber.bSet(tmpArray.constData());

      methodResult->oContactDetailsExtended.oPhoneNumber1.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)contact.NumberType1;
      methodResult->oContactDetailsExtended.oPhoneNumber2.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)contact.NumberType2;
      methodResult->oContactDetailsExtended.oPhoneNumber3.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)contact.NumberType3;
      methodResult->oContactDetailsExtended.oPhoneNumber4.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)contact.NumberType4;
      methodResult->oContactDetailsExtended.oPhoneNumber5.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)contact.NumberType5;
      methodResult->oContactDetailsExtended.oPhoneNumber6.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)contact.NumberType6;
      methodResult->oContactDetailsExtended.oPhoneNumber7.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)contact.NumberType7;
      methodResult->oContactDetailsExtended.oPhoneNumber8.e8Type.enType = (most_fi_tcl_e8_PhonBkNumberType::tenType)contact.NumberType8;

      tmpArray = contact.PPPhoneNum1.toUtf8();
      methodResult->oContactDetailsExtended.oPhoneNumber1.sNumber.bSet(tmpArray.constData());

      tmpArray = contact.PPPhoneNum2.toUtf8();
      methodResult->oContactDetailsExtended.oPhoneNumber2.sNumber.bSet(tmpArray.constData());

      tmpArray = contact.PPPhoneNum3.toUtf8();
      methodResult->oContactDetailsExtended.oPhoneNumber3.sNumber.bSet(tmpArray.constData());

      tmpArray = contact.PPPhoneNum4.toUtf8();
      methodResult->oContactDetailsExtended.oPhoneNumber4.sNumber.bSet(tmpArray.constData());

      tmpArray = contact.PPPhoneNum5.toUtf8();
      methodResult->oContactDetailsExtended.oPhoneNumber5.sNumber.bSet(tmpArray.constData());

      tmpArray = contact.PPPhoneNum6.toUtf8();
      methodResult->oContactDetailsExtended.oPhoneNumber6.sNumber.bSet(tmpArray.constData());

      tmpArray = contact.PPPhoneNum7.toUtf8();
      methodResult->oContactDetailsExtended.oPhoneNumber7.sNumber.bSet(tmpArray.constData());

      tmpArray = contact.PPPhoneNum8.toUtf8();
      methodResult->oContactDetailsExtended.oPhoneNumber8.sNumber.bSet(tmpArray.constData());

      tmpArray =  contact.emailAdd1.toUtf8();
      methodResult->oContactDetailsExtended.sEmailAddress1.bSet(tmpArray.constData());

      tmpArray = contact.emailAdd2.toUtf8();
      methodResult->oContactDetailsExtended.sEmailAddress2.bSet(tmpArray.constData());

      tmpArray = contact.emailAdd3.toUtf8();
      methodResult->oContactDetailsExtended.sEmailAddress3.bSet(tmpArray.constData());

      tmpArray = contact.homeAdd.toUtf8();
      methodResult->oContactDetailsExtended.sHomeAddress.bSet(tmpArray.constData());

      tmpArray = contact.workAdd.toUtf8();
      methodResult->oContactDetailsExtended.sWorkAddress.bSet(tmpArray.constData());

      tmpArray = contact.otherAdd.toUtf8();
      methodResult->oContactDetailsExtended.sOtherAddress.bSet(tmpArray.constData());

      tmpArray = contact.category.toUtf8();
      methodResult->oContactDetailsExtended.sCategory.bSet(tmpArray.constData());

      tmpArray = contact.notes.toUtf8();
      methodResult->oContactDetailsExtended.sNotes.bSet(tmpArray.constData());

      methodResult->oContactDetailsExtended.bPhotoAvailable = (contact.photoSize > 0)?true:false;

      methodResult->oContactDetailsExtended.oHomeAddressLatLong.bLatLongValidity = contact.homeGeoCodeLatLongValidity;

      methodResult->oContactDetailsExtended.oHomeAddressLatLong.s32Latitude = contact.homeGeoCodeLatitude;

      methodResult->oContactDetailsExtended.oHomeAddressLatLong.s32Longitude = contact.homeGeocodeLongitude;

      methodResult->oContactDetailsExtended.oWorkAddressLatLong.bLatLongValidity = contact.workGeoCodeLatLongValidity;

      methodResult->oContactDetailsExtended.oWorkAddressLatLong.s32Latitude = contact.workGeoCodeLatitude;

      methodResult->oContactDetailsExtended.oWorkAddressLatLong.s32Longitude = contact.workGeocodeLongitude;

      methodResult->oContactDetailsExtended.oOtherAddressLatLong.bLatLongValidity = contact.otherGeoCodeLatLongValidity;

      methodResult->oContactDetailsExtended.oOtherAddressLatLong.s32Latitude = contact.otherGeoCodeLatitude;

      methodResult->oContactDetailsExtended.oOtherAddressLatLong.s32Longitude = contact.otherGeocodeLongitude;
      bRetValue = true;
   }
   else
   {
      ETG_TRACE_ERR((" m_sqlite->GetContactDetailsExtended Failed "));
   }
   return bRetValue;
}
//End of Fix CMG3G-8365 IS2421_Configurable phone number type representation@FC_Phonebook


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::bClearVehiclePhonebookExtended
*
* DESCRIPTION: This method clears all vehicle phonebook entries from a specific vehicle phonebook. [VPBx]
*   It can also clear all the entries from all the vehicle phonebooks at once [1-11] (For reset to factory default.)
*
* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_List::bClearVehiclePhonebookExtended(tU8 u8DeviceHandle)
{
   ETG_TRACE_USR4((" bClearVehiclePhonebookExtended entered with device handle: %d", u8DeviceHandle));

   bool bRet = TRUE;

   if (0 != u8DeviceHandle)
   {
      //Delete the entries from VPB or particular VPBx
      bRet = ClearVehiclePhoneBook(u8DeviceHandle);
   }
   else
   {
      //Device Handle = 0 : Clear all the entries from all the vehicle phonebooks at once [1-11] (For reset to factory default)
      //Delete VPB
      bRet = ClearVehiclePhoneBook(FC_PB_VEHICLE_PBOOK);
      //Delete VPB2 to VPB11
      for (tU8 index = FC_VPB_DEVICE_HANDLE2; index <= FC_VPB_DEVICE_HANDLE11; index++)
      {
         bRet = bRet && ClearVehiclePhoneBook(index);
      }
   }

   return bRet;
}


/*******************************************************************************
 *
 * FUNCTION:  FC_PhoneBook_List::GetPositionForContactHandle
 *
 * DESCRIPTION:
 *   This method can be called for a dedicated contact list (identified by the u16ListHandle)
 *   to find out at which position in this list a contact (identified via its u32ContactHandle) is located.
 *
 *
 * PARAMETER: [IN] methodStart
 *            [OUT] methodResult
 * RETURNVALUE: bool - indicating success or failure of this function
 *
 *******************************************************************************/
bool FC_PhoneBook_List::GetPositionForContactHandle(const most_phonbkfi_tclMsgGetPositionForContactHandleMethodStart * methodStart, most_phonbkfi_tclMsgGetPositionForContactHandleMethodResult* methodResult)
{
   ETG_TRACE_USR4((" GetPositionForContactHandle entered with ListHandle: 0x%x", methodStart->u16ListHandle));
   ETG_TRACE_USR4((" ContactHandle: %u", methodStart->u32ContactHandle));

   bool bRetValue = FALSE;
   if (FALSE == m_hashPbList.contains(methodStart->u16ListHandle))
   {
      ETG_TRACE_ERR((" List handle not found in m_hashPbList!!!"));
   }
   else
   {
      FC_PhoneBook_List_Data* PBookList =
            m_hashPbList.value(methodStart->u16ListHandle, NULLPTR);
      quint32 u32Position = 0;
      ListChange_Handler pListChange_Handler;
      //CMG3G-14313-Coverity Fix
      if(!PBookList )
      {
         ETG_TRACE_ERR((" PBookList is NULL!!!"));
      }
      else
      {
         if (pListChange_Handler.FindIndexOfContactHandle(PBookList->contactList, methodStart->u32ContactHandle, u32Position))
         {
            methodResult->u32Position = u32Position;
            ETG_TRACE_USR2((" Contact Position in the list : %d", methodResult->u32Position));
            ETG_TRACE_USR4((" firstName : %s", PBookList->contactList.at(u32Position).firstName.toUtf8().constData()));
            ETG_TRACE_USR4((" lastName : %s", PBookList->contactList.at(u32Position).lastName.toUtf8().constData()));
            bRetValue = TRUE;
         }
         else
         {
            ETG_TRACE_ERR((" Contact handle not found in list!!!"));
         }
      }//End of Fix
   }
   return bRetValue;
}
