/*******************************************************************************
*
* FILE:          FC_PhoneBook_Database.cpp
*
* SW-COMPONENT:  FC_PhoneBook application
*
* PROJECT:
*
* DESCRIPTION:   Handles the sqlite database using QtSql module
*
* AUTHOR:
*
* COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
*
*******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/
#include "../FC_PhoneBook_Debug.h"
#include "../FC_PhoneBook_Device_Details.h"
#include "../DbusClient/FC_PhoneBook_EvoDbusClient.h"//CMG3G-14314
#include "FC_PhoneBook_Database.h"
#include "FC_PhoneBook_List.h"
#include "FC_PhoneBook_CallHistory_List.h"
#include "../FC_PhoneBook_Dnl_Manager.h"
#include "../FC_PhoneBook_FavoritesProvider.h"
#include "../FC_PhoneBook_Debug.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC

#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONEBOOK_SERVICE
#include "trcGenProj/Header/FC_PhoneBook_Database.cpp.trc.h"
#endif
/******************************************************************************/
/*                                                                            */
/* METHODS                                                                    */
/*                                                                            */
/******************************************************************************/

FC_PhoneBook_Database* FC_PhoneBook_Database::m_self = NULL;

/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_Database::Instance()
*
* DESCRIPTION: Creates or returns the singleton object.
*
* PARAMETER: None.
*
* RETURNVALUE: FC_PhoneBook_Database* :- The singleton object
*
*******************************************************************************/
FC_PhoneBook_Database* FC_PhoneBook_Database::Instance()
{
   if ( m_self == NULL )
      m_self = new FC_PhoneBook_Database;
   return m_self;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_Database::DeleteInstance()
*
* DESCRIPTION: Deletes the singleton object.
*
* PARAMETER: None.
*
* RETURNVALUE: None
*
*******************************************************************************/
void FC_PhoneBook_Database::DeleteInstance()
{
   delete m_self;
   m_self = NULLPTR;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_Database::FC_PhoneBook_Database()
*
* DESCRIPTION: Constructor
*
* PARAMETER: None.
*
* RETURNVALUE: None
*
*******************************************************************************/
FC_PhoneBook_Database::FC_PhoneBook_Database()
{
   //CLONE TICKET FIX - GMMY15-1545 - Singleton creation for FC_PhoneBook_List and FC_PhoneBook_CallHistory_List
   m_list = FC_PhoneBook_List::pGetInstance();
   m_callHistory = FC_PhoneBook_CallHistory_List::GetCurrentInstance();
   m_fav= FC_Phonebook_tclFavorite_Provider::GetInstance();
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_Database::~FC_PhoneBook_Database()
*
* DESCRIPTION: Destructor
*
* PARAMETER: None.
*
* RETURNVALUE: None
*
*******************************************************************************/
FC_PhoneBook_Database::~FC_PhoneBook_Database()
{
   //CLONE TICKET FIX - GMMY15-1545 - Singleton creation for FC_PhoneBook_List and FC_PhoneBook_CallHistory_List
   FC_PhoneBook_List::vDestroyInstance();
   FC_PhoneBook_CallHistory_List::vDestroyInstance();
   m_fav =NULLPTR;
   m_list = NULLPTR;
   m_callHistory = NULLPTR;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::CreatePhoneBookList
*
* DESCRIPTION:
*              Creates a new contact info list from the database.
*
*
* PARAMETER: [IN] methodStart :- Input arguments like sort type etc are in this class
*            [OUT] methodResult :- new list handle and list length are in this class
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::CreatePhoneBookList(
   const most_phonbkfi_tclMsgCreatePhoneBookListMethodStart* methodStart,
   most_phonbkfi_tclMsgCreatePhoneBookListMethodResult* methodResult, 
   tU32 u32RegisterID)//FIX GMMY15-1578 RegisterId is passed to identify the requested client
{
   ETG_TRACE_USR4(( " CreatePhoneBookList entered" ));
   if ( !methodStart || !methodResult )
      return false;
   //FIX GMMY15-1578 Bosch Internal: Proper usage of SQLite database instead of RAM -- removing redundant objects which are created in RAM for different clients & sorted lists.
   if(m_list)
   {
      return m_list->CreatePhoneBookList( methodStart, methodResult, u32RegisterID );
   }
   return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::CreateContactList
*
* DESCRIPTION:
*              Creates a new contact info list from the database.
*
*
* PARAMETER: [IN] methodStart :- Input arguments like sort type etc are in this class
*            [OUT] methodResult :- new list handle and list length are in this class
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::CreateContactList(
   const most_phonbkfi_tclMsgCreateContactListMethodStart* methodStart,
   most_phonbkfi_tclMsgCreateContactListMethodResult* methodResult,
   tU32 u32RegisterID)//FIX GMMY15-1578 RegisterId is passed to identify the requested client
{
   ETG_TRACE_USR4(( " CreateContactList entered" ));

  
   if(m_list)
   {
      return m_list->CreateContactList( methodStart, methodResult, u32RegisterID );
   }
   return false;
}

/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::CreateCallHistoryList
*
* DESCRIPTION:
*              Creates a Call History list from the database.
*
*
* PARAMETER: [IN] methodStart :- Input arguments like sort type etc are in this class
*            [OUT] methodResult :- new list handle and list length are in this class
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::CreateCallHistoryList(
   const most_phonbkfi_tclMsgCreateCallHistoryListMethodStart* methodStart,
   most_phonbkfi_tclMsgCreateCallHistoryListMethodResult* methodResult,
   tU32 u32RegisterID)//FIX GMMY15-1578 RegisterId is passed to identify the requested client
{
   ETG_TRACE_USR4(( " CreateCallHistoryList entered" ));
   if ( !methodStart || !methodResult )
      return false;

   if(m_callHistory)
   {
      return m_callHistory->CreateCallHistoryList( methodStart, methodResult, u32RegisterID );//FIX GMMY15-1578 RegisterId is passed to identify the requested client
   }

   return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::RequestSlicePhoneBookList
*
* DESCRIPTION:
*              Creates a subset of the list pointed by listhandle
*
*
* PARAMETER: [IN] methodStart :- Input arguments like listhandles are in this class
*            [OUT] methodResult :- The acutal sub list is passed in this class
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::RequestSlicePhoneBookList(
   const most_phonbkfi_tclMsgRequestPhoneBookListSliceMethodStart* methodStart,
   most_phonbkfi_tclMsgRequestPhoneBookListSliceMethodResult* methodResult )

{
   ETG_TRACE_USR4(( " RequestSlicePhoneBookList entered" ));
   if ( !methodStart || !methodResult )
      return false;

   if(m_list)
   {
      return m_list->RequestSlicePhoneBookList( methodStart, methodResult );
   }

   return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::RequestSliceCallHistoryList
*
* DESCRIPTION:
*              Creates a subset of the list pointed by listhandle
*
*
* PARAMETER: [IN] methodStart :- Input arguments like listhandles are in this class
*            [OUT] methodResult :- The acutal sub list is passed in this class
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::RequestSliceCallHistoryList(
   const most_phonbkfi_tclMsgRequestSliceCallHistoryListMethodStart* methodStart,
   most_phonbkfi_tclMsgRequestSliceCallHistoryListMethodResult* methodResult,
   tU32 registerID)////FIX NCG3D-55494 RegisterId is passed to identify the requested client
{
   ETG_TRACE_USR4(( " RequestSliceCallHistoryList entered" ));
   if ( !methodStart || !methodResult )
      return false;

   if(m_callHistory)
   {
      return m_callHistory->RequestSliceCallHistoryList( methodStart, methodResult,registerID );
   }
   return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::ReleasePhoneBookList
*
* DESCRIPTION:
*              Free the memory occupied by the list pointed by the listhandle.
*
*
* PARAMETER: [IN] methodStart :- Input arguments like listhandles are in this class
*            [OUT] methodResult :- Error if any
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::ReleasePhoneBookList(
   const most_phonbkfi_tclMsgReleasePhoneBookListMethodStart* methodStart,
   most_phonbkfi_tclMsgReleasePhoneBookListMethodResult* methodResult,
   tU32 u32RegisterID )//FIX GMMY15-1578 RegisterID is passed to identify the requested client
{
   ETG_TRACE_USR4(( " ReleasePhoneBookList entered" ));
   if ( !methodStart || !methodResult )
      return false;

   if(m_list)
   {
      return m_list->ReleasePhoneBookList( methodStart, methodResult, u32RegisterID );//FIX GMMY15-1578 RegisterID is passed to identify the requested client
   }
   return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::ReleaseCallHistoryList
*
* DESCRIPTION:
*              Free the memory occupied by the list pointed by the listhandle.
*
*
* PARAMETER: [IN] methodStart :- Input arguments like listhandles are in this class
*            [OUT] methodResult :- Error if any
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::ReleaseCallHistoryList(
   const most_phonbkfi_tclMsgReleaseCallHistoryListMethodStart* methodStart,
   most_phonbkfi_tclMsgReleaseCallHistoryListMethodResult* methodResult,
   tU32 registerID)////FIX GMMY15-1578 RegisterId is passed to identify the requested client
{
   ETG_TRACE_USR4(( " ReleaseCallHistoryList entered" ));
   if ( !methodStart || !methodResult )
      return false;

   if(m_callHistory)
   {
      return m_callHistory->ReleaseCallHistoryList( methodStart, methodResult, registerID );//FIX GMMY15-1578 RegisterID is passed to identify the requested client
   }
   return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::SearchKeyboardPhoneBookList
*
* DESCRIPTION:
*              Returns the keyboard list
*
*
* PARAMETER: [IN] methodStart :- Input arguments like listhandles are in this class
*            [OUT] methodResult :- Error if any
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::SearchKeyboardPhoneBookList(
   const most_phonbkfi_tclMsgSearchKeyboardPhoneBookListMethodStart* methodStart,
   most_phonbkfi_tclMsgSearchKeyboardPhoneBookListMethodResult* methodResult )
{
   ETG_TRACE_USR4(( " SearchKeyboardPhoneBookList entered" ));
   if ( !methodStart || !methodResult )
      return false;

   if(m_list)
   {
      return m_list->SearchKeyboardPhoneBookList(methodStart,methodResult);
   }
   return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::SearchPhoneBook
*
* DESCRIPTION:
**
*
* PARAMETER: [IN] methodStart :-
*            [OUT] methodResult :- Error if any
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::SearchPhoneBook(
   const most_phonbkfi_tclMsgSearchPhoneBookMethodStart* methodStart,
   most_phonbkfi_tclMsgSearchPhoneBookMethodResult* methodResult)
{
   ETG_TRACE_USR4(( " SearchPhoneBook entered" ));
   if ( !methodStart || !methodResult )
      return false;

   if(m_list)
   {
      return m_list->SearchPhoneBook(methodStart,methodResult);
   }
   return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::SearchContacts
*
* DESCRIPTION:
**
*
* PARAMETER: [IN] methodStart :-
*            [OUT] methodResult :- Error if any
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::SearchContacts(
   const most_phonbkfi_tclMsgSearchContactsMethodStart* methodStart,
   most_phonbkfi_tclMsgSearchContactsMethodResult* methodResult)
{
   ETG_TRACE_USR4(( " FC_PhoneBook_Database::SearchContacts entered" ));
   if(m_list)
   {
      return m_list->SearchContacts(methodStart,methodResult);
   }
   return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::ClearPhoneBookData
*
* DESCRIPTION: Deletes all Phonebook table contents including vehicle PB.
**
*
* PARAMETER: [IN]  :- None.
*            [OUT]  :- None.
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::ClearPhoneBookData()//FIX GMMY16-23858
{
   ETG_TRACE_USR4(( " ClearPhoneBookData entered" ));
   if(m_list)
   {
      return m_list->ClearPhoneBookData();//FIX GMMY16-23858
   }
   return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::ClearVehiclePhoneBook
*
* DESCRIPTION: Deletes content of vehicle phone book only.
**
*
* PARAMETER: [IN]  :- None.
*            [OUT]  :- None.
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::ClearVehiclePhoneBook()
{
   ETG_TRACE_USR4(( " ClearVehiclePhoneBook entered" ));
   if(m_list)
   {
      return m_list->ClearVehiclePhoneBook(FC_PB_VEHICLE_PBOOK);
   }
   return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::AddContact
*
* DESCRIPTION:
**
*
* PARAMETER: [IN] methodStart :-
*            [OUT] methodResult :- 
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::AddContact(
                                       const most_phonbkfi_tclMsgAddContactMethodStart* methodStart,
                                       most_phonbkfi_tclMsgAddContactMethodResult* methodResult)
{
   ETG_TRACE_USR4(( " AddContact entered" ));
   if ( !methodStart || !methodResult )
      return false;

   if(m_list)
   {
      return m_list->AddContact(methodStart,methodResult);
   }
   return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::EditContact
*
* DESCRIPTION:
**
*
* PARAMETER: [IN] methodStart :-
*            [OUT] methodResult :- Error if any
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::EditContact(
                                        const most_phonbkfi_tclMsgEditContactMethodStart* methodStart,
                                        most_phonbkfi_tclMsgEditContactMethodResult* methodResult)
{
   ETG_TRACE_USR4(( " EditContact entered" ));
   if ( !methodStart || !methodResult )
      return false;

   if(m_list)
   {
      return m_list->EditContact(methodStart,methodResult);
   }
   return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::DeleteContact
*
* DESCRIPTION:
**
*
* PARAMETER: [IN] methodStart :-
*            [OUT] methodResult :- Error if any
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::DeleteContact(
   const most_phonbkfi_tclMsgDeleteContactMethodStart* methodStart,
   most_phonbkfi_tclMsgDeleteContactMethodResult* methodResult)
{
   ETG_TRACE_USR4(( " DeleteContact entered" ));
   if(!methodStart || !methodResult)
      return false;

   if(m_list)
   {
      return m_list->DeleteContact(methodStart,methodResult);
   }
   return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::GetContactDetails
*
* DESCRIPTION:
**
*
* PARAMETER: [IN] methodStart :-
*            [OUT] methodResult :- Error if any
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::GetContactDetails(
   const most_phonbkfi_tclMsgGetContactDetailsMethodStart* methodStart,
   most_phonbkfi_tclMsgGetContactDetailsMethodResult* methodResult)
{
   ETG_TRACE_USR4(( " GetContactDetails entered" ));
   if(!methodStart || !methodResult)
      return false;

   if(m_list)
   {
      return m_list->GetContactDetails(methodStart,methodResult);
   }
   return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::ParseVCard
*
* DESCRIPTION:
*              Parses the the vcard. The data is not deleted, but a copy is made in
*              QByteArray internaly. So the caller must delete data. The actual parsing
*              starts when completed is true.

* PARAMETER:
*
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_Database::ParseVCard(GByteArray* data, unsigned char deviceHandle, unsigned char connectionid)
{
   ETG_TRACE_USR4((" ParseVCard entered"));
   quint8
         u8phonebookType =
               FC_PhoneBook_Dnl_Manager::getCurDownloadTypeForConnectionID(connectionid);

   switch (u8phonebookType)
   {
      case DownLoadPBSim:
      case DownLoadPBLocal:
      case DownloadPBSpeedDial:
      case DownloadPBFavorites:
         if (m_list)
            m_list->AddNewVCardData(data, u8phonebookType, deviceHandle);
      break;
      case DownloadOutGoingCallHistorySimLocal:
      case DownloadMissedCallHistorySimLocal:
      case DownloadIncomingCallHistorySimLocal:
      case DownloadCombinedCallHistorySimLocal:
         if (m_callHistory)
            m_callHistory->AddNewVCardData(data, u8phonebookType, deviceHandle);
      break;
      default:
      {
         ETG_TRACE_USR4(("Not Valid u8phonebookType "));
      }
   }
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::MethodResultPosted
*
* DESCRIPTION: This function is called when a method result is posted successfully
**
*
* PARAMETER: [IN] methodStart :-
*            [OUT] methodResult :-
* RETURNVALUE: bool -
*
*******************************************************************************/
void FC_PhoneBook_Database::MethodResultPosted(const guint16 u16FuntionID,
                                               const most_phonbkfi_tclMsgBaseMessage* MethodStart,
                                               const most_phonbkfi_tclMsgBaseMessage* MethodResult)
{
   ETG_TRACE_USR4(("Function MethodResultPosted entered"));

   switch(u16FuntionID)
   {
   case MOST_PHONBKFI_C_U16_ADDCONTACT:
   case MOST_PHONBKFI_C_U16_DELETECONTACT:
   case MOST_PHONBKFI_C_U16_EDITCONTACT:
   case MOST_PHONBKFI_C_U16_CREATEPHONEBOOKLIST:
   case MOST_PHONBKFI_C_U16_CREATECONTACTLIST:
   case MOST_PHONBKFI_C_U16_SETCONTACTPOSITION:
   case MOST_PHONBKFI_C_U16_SETCONTACTPHOTO:
   case MOST_PHONBKFI_C_U16_ADDCONTACTEXTENDED:
   case MOST_PHONBKFI_C_U16_EDITCONTACTEXTENDED:
      if(m_list)
         m_list->MethodResultPosted(u16FuntionID, MethodStart, MethodResult);
      break;
   case MOST_PHONBKFI_C_U16_STOREFAVORITE:
   case MOST_PHONBKFI_C_U16_DELETEFAVORITE:
   case MOST_PHONBKFI_C_U16_RESETFAVORITES:
      if(m_fav)
         m_fav->MethodResultPosted(u16FuntionID, MethodStart, MethodResult);
      break;
   default:
      {
         ETG_TRACE_ERR(("Not a valid function ID!!! "));
      }
   }
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::DeleteCallHistoryRecordsGivenCallType
*
* DESCRIPTION: Deletes the call history record for a given device (connection ID) and
* given call type
*
* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::DeleteCallHistoryRecordsGivenCallType(guint8 u8ConnectionID,
                                                                  guint8 u8CallType)
{
   ETG_TRACE_USR4(("DeleteCallHistoryRecordsGivenCallType entered with u8ConnectionID %d ",u8ConnectionID));
   if(m_callHistory)
   {
      return m_callHistory->DeleteCallHistoryRecordsGivenCallType(u8ConnectionID,
         u8CallType);
   }
   return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::DeviceDisconnected
*
* DESCRIPTION: This function is called when a device is disconnected
**
*
* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_Database::DeviceDisconnected(FC_Device_Details* DeviceDetails)
{
   ETG_TRACE_USR4(("DeviceDisconnected entered "));
   if(m_list)
   {
      m_list->DeviceDisconnected(DeviceDetails);
   }
   if(m_fav)
   {
      m_fav->RefreshFavoritesOnDeviceDisconnection(
         DeviceDetails->getDeviceHandle());
   }
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::DownloadCompleted
*
* DESCRIPTION: This function is called when the download is completed for a device
**
*
* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_Database::DownloadCompleted(guint8 u8DeviceHandle)
{
   ETG_TRACE_USR4(("DownloadCompleted entered "));
   if(m_list)
      m_list->DownloadCompleted(u8DeviceHandle);

   if(m_fav)
      m_fav->RefreshFavoritesOnDownloadComplete(u8DeviceHandle);
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::PhoneBookDownloadInitiated
*
* DESCRIPTION: This function is called when the phonebook download is
* started for a device
*
* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_Database::PhoneBookDownloadInitiated(FC_Device_Details* DeviceDetails)
{
   ETG_TRACE_USR4(("PhoneBookDownloadInitiated entered "));
   if(m_list)
      m_list->PhoneBookDownloadInitiated(DeviceDetails);
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::getListInstance
*
* DESCRIPTION: This function gives you the current list of contacts
*
* PARAMETER: [IN] None
*
* RETURNVALUE: Pointer to the list
*
*******************************************************************************/
FC_PhoneBook_List* FC_PhoneBook_Database::getCurrentListInstance()
{
   return m_list;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::DeviceDeleted
*
* DESCRIPTION: This function is when a device is deleted( unpaired )
*
* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_Database::DeviceDeleted(guint8 u8DeviceHandle)
{
   ETG_TRACE_USR4(("DeviceDeleted entered "));
   if(m_list)
      m_list->DeviceDeleted(u8DeviceHandle);
}


#ifdef BUGZID_206322
/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::DeviceConnected
*
* DESCRIPTION: This function is called when a device is connected
**
*
* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_Database::DeviceConnected(FC_Device_Details* DeviceDetails)
{
   ETG_TRACE_USR4(("DeviceConnected entered "));
   // Upon new device connetion, get the total no. records count
   if(m_list)
   {
      m_list->DeviceConnected(DeviceDetails);
      if(m_callHistory)
      {
         m_callHistory->DeviceConnected(DeviceDetails);
      }
   }
}
#endif


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::GetContactPhotoInfo
*
* DESCRIPTION:
**
*
* PARAMETER: [IN] methodStart :-
*            [OUT] methodResult :- Error if any
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::GetContactPhotoInfo(
   const most_phonbkfi_tclMsgGetContactPhotoInfoMethodStart* methodStart,
   most_phonbkfi_tclMsgGetContactPhotoInfoMethodResult* methodResult,
   tU16& u16ErrorCode)
{
   ETG_TRACE_USR4(("GetContactPhotoInfo entered "));
   if(!methodStart || !methodResult)
   {
      u16ErrorCode= CCA_C_U16_ERROR_INTERNAL_FAILURE;
      return false;
   }
   if(m_list)
   {
      return m_list->GetContactPhotoInfo(methodStart, methodResult, u16ErrorCode);
   }
   return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::GetContactPhoto
*
* DESCRIPTION:
**
*
* PARAMETER: [IN] methodStart :-
*            [OUT] methodResult :- Error if any
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::GetContactPhoto(
   const most_phonbkfi_tclMsgGetContactPhotoMethodStart* methodStart,
   most_phonbkfi_tclMsgGetContactPhotoMethodResult* methodResult,
   tU16& u16ErrorCode)
{
   ETG_TRACE_USR4(("GetContactPhoto entered "));
   if(!methodStart || !methodResult)
   {
      u16ErrorCode= CCA_C_U16_ERROR_INTERNAL_FAILURE;
      return false;
   }
   if(m_list)
   {
      return m_list->GetContactPhoto(methodStart, methodResult, u16ErrorCode);
   }
   return false;

}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::SearchPhoneNumberEntries
*
* DESCRIPTION:
**
*
* PARAMETER: [IN] methodStart :- Phone Number
*            [OUT] methodResult :- Matched records from CallHistory Database for the given phone number
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::SearchPhoneNumberEntries(
   const most_phonbkfi_tclMsgAutocompletePhoneNumberEntryMethodStart* methodStart,
   most_phonbkfi_tclMsgAutocompletePhoneNumberEntryMethodResult* methodResult,guint8 ConnectionID)
{
   ETG_TRACE_USR4(("SearchPhoneNumberEntries entered "));
   if(!methodStart || !methodResult)
      return false;

   if(m_callHistory)
      m_callHistory->SearchPhoneNumberEntries(methodStart,methodResult,ConnectionID);

   return true;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::clearAutocompleteListsOnOutgoingSourceChange
*
* DESCRIPTION:
**
*
* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_Database::clearAutocompleteListsOnOSChange()
{
   if(m_callHistory)
      m_callHistory->clearAutocompleteListsOnOSChange();
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::CallHistoryUpdated
*
* DESCRIPTION: This function is called on call history update
**
*
* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_Database::CallHistoryUpdated(
   guint8 u8ConnectionID, guint8 u8CallHistoryDnlCmd)
{
   if(m_callHistory)
      m_callHistory->RefreshCallHistoryOnCallHistoryUpdate(
      u8ConnectionID, u8CallHistoryDnlCmd);
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::getCallHistoryListInstance
*
* DESCRIPTION: 
*
* PARAMETER: [IN] None
*
* RETURNVALUE: Pointer to the list
*
*******************************************************************************/
FC_PhoneBook_CallHistory_List* FC_PhoneBook_Database::getCallHistoryListInstance()
{
   return m_callHistory;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::StoreFavorite
*
* DESCRIPTION: This function stores a new favorite in database
**
*
* PARAMETER: [IN] methodStart :- Object ID
*            [OUT] methodResult :- FavoritesID
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::StoreFavorite(
   const most_phonbkfi_tclMsgStoreFavoriteMethodStart* methodStart,
   most_phonbkfi_tclMsgStoreFavoriteMethodResult *methodResult, tU16& u16ErrorCode, tU8& u8ErrorInfo)
{
   ETG_TRACE_USR4(("StoreFavorite entered "));

   if(!methodStart || !methodResult)
   {
      return false;
   }
   if(m_fav)
   {
      return m_fav->Store_Favorite(methodStart, methodResult, u16ErrorCode, u8ErrorInfo);
   }
   return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::GetFavoriteInfo
*
* DESCRIPTION: This function fetches details of a favorite from database
**
* PARAMETER: [IN] methodStart :- FavoritesID
*            [OUT] methodResult
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::GetFavoriteInfo(
   const most_phonbkfi_tclMsgGetFavoriteInfoMethodStart* methodStart,
   most_phonbkfi_tclMsgGetFavoriteInfoMethodResult *methodResult,tU16& u16ErrorCode, tU8& u8ErrorInfo)
{

   ETG_TRACE_USR3(("FC_PhoneBook_Database::GetFavoriteInfo entered."));
   if(!methodStart || !methodResult)
   {
      return false;
   }
   if(m_fav)
   {
      return m_fav->GetFavoriteInfo(methodStart, methodResult, u16ErrorCode, u8ErrorInfo);
   }
   return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::ClearFavoritesAvailableAndActiveStatus
*
* DESCRIPTION: This function clears the available and active status of all
* favorites
**
* PARAMETER: [IN] methodStart :-
*            [OUT] methodResult
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_Database::ClearFavoritesAvailableAndActiveStatus()
{
   ETG_TRACE_USR4(("ClearFavoritesAvailableAndActiveStatus entered "));
   if(m_fav)
   {
      m_fav->ResetAllFavoritesAsUnavailable();
   }
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::MakeFavoritesAvailable
*
* DESCRIPTION: Makes all favorites of a particular device handle available.
* favorites
**
* PARAMETER: [IN] DeviceHandle
*            [OUT] None.
* RETURNVALUE: None.
*
*******************************************************************************/
void FC_PhoneBook_Database::MakeFavoritesAvailable(quint8 DeviceHandle)
{
   ETG_TRACE_USR4(("MakeFavoritesAvailable entered "));
   if(m_fav)
   {
      m_fav->MakeFavoritesAvailable(DeviceHandle);
      m_fav->triggerFavoritesChange(DeviceHandle);
   }
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::MakeFavoritesUnAvailable
*
* DESCRIPTION: Makes all favorites of a particular device handle unavailable.
* favorites
**
* PARAMETER: [IN] DeviceHandle
*            [OUT] None.
* RETURNVALUE: None.
*
*******************************************************************************/
void FC_PhoneBook_Database::MakeFavoritesUnAvailable(quint8 DeviceHandle)
{
   ETG_TRACE_USR4(("MakeFavoritesUnAvailable entered "));
   if(m_fav)
   {
      m_fav->MakeFavoritesUnAvailable(DeviceHandle);
      m_fav->triggerFavoritesChange(DeviceHandle);
   }
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::ChangefavoriteActiveStatus
*
* DESCRIPTION: Makes all favorites of a particular device handle available.
* favorites
**
* PARAMETER: [IN] DeviceHandle
*            [OUT] None.
* RETURNVALUE: None.
*
*******************************************************************************/
void FC_PhoneBook_Database::ChangefavoriteActiveStatus(quint8 u8Status)
{
   ETG_TRACE_USR4(("ChangefavoriteActiveStatus entered "));
   if(m_fav)
   {
      m_fav->ChangefavoriteActiveStatus(u8Status);
   }
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::RegisterFavoritesInstance
*
* DESCRIPTION:
**
* PARAMETER: [IN] methodStart :-
*            [OUT] methodResult
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_Database::RegisterFavoritesInstance(
   FC_Phonebook_tclFavorite_Provider* pFavInstance)
{
   ETG_TRACE_USR4(("RegisterFavoritesInstance entered "));
   m_fav= pFavInstance;
   if(m_list)
      m_list->RegisterFavoritesProvider(m_fav);
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::GetTotalFavoritesCount
*
* DESCRIPTION:
**
* PARAMETER: [IN] methodStart :-
*            [OUT] methodResult
* RETURNVALUE:
*
*******************************************************************************/
guint32 FC_PhoneBook_Database::GetTotalFavoritesCount()
{
   ETG_TRACE_USR4(("GetTotalFavoritesCount entered "));
   if(m_fav)
   {
      return m_fav->GetTotalFavoritesCount();
   }
   return 0;
}


/***************************************************************************************
*
* FUNCTION:  DeleteCallHistoryRecordsGivenConnectionID
*
* DESCRIPTION: This function clears all the call history related data for the
* given connection id.
*
* PARAMETER:   [IN] u8ConnectionID
*              [OUT] None.
* RETURNVALUE: None.
*
****************************************************************************************/
void FC_PhoneBook_Database::DeleteCallHistoryRecordsGivenConnectionID(
   quint8 u8ConnectionID)
{
   ETG_TRACE_USR4(("DeleteCallHistoryRecordsGivenConnectionID entered "));
   if(m_callHistory)
      m_callHistory->DeleteCallHistoryRecordsGivenConnectionID(u8ConnectionID);
}


/***************************************************************************************
*
* FUNCTION:  UpdateSortView
*
* DESCRIPTION: This function clears all the call history related data for the
* given connection id.
*
* PARAMETER:   [IN] u8DeviceHandle
*              [OUT] None.
* RETURNVALUE: SUCCESS or FAILURE.
*
****************************************************************************************/
bool FC_PhoneBook_Database::UpdateSortView(quint8 u8DeviceHandle)
{
   ETG_TRACE_USR4(("UpdateSortView entered "));
   if(m_list)
      return m_list->UpdateSortView(u8DeviceHandle);
   return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::SearchPhoneNumberEntriesInPB
*
* DESCRIPTION:
**
*
* PARAMETER: [IN] methodStart :- Phone Number
*            [OUT] methodResult :- Matched records from PhoneBook Database for the given phone number
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::SearchPhoneNumberEntriesInPB(
   const most_phonbkfi_tclMsgAutocompletePhoneNumberEntryMethodStart* methodStart,
   most_phonbkfi_tclMsgAutocompletePhoneNumberEntryMethodResult* methodResult,guint8 u8DeviceHandle)
{
   ETG_TRACE_USR4(("SearchPhoneNumberEntriesInPB entered "));
   if(!methodStart || !methodResult)
      return false;

   if(m_list)
      m_list->SearchPhoneNumberEntriesInPB(methodStart,methodResult,u8DeviceHandle);

   return true;
}

//FIX SUZUKI-1114 API for numeric value to indicate number of dial-able phone numbers available to a contact item in Phonebook
//RequestSliceCallHistoryListExtended API ADDED
/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::RequestSliceCallHistoryListExtended
*
* DESCRIPTION:
*              Creates a subset of the list pointed by listhandle
*
*
* PARAMETER: [IN] methodStart :- Input arguments like listhandles are in this class
*            [OUT] methodResult :- The acutal sub list is passed in this class
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::RequestSliceCallHistoryListExtended(
   const most_phonbkfi_tclMsgRequestSliceCallHistoryListExtendedMethodStart* methodStart,
   most_phonbkfi_tclMsgRequestSliceCallHistoryListExtendedMethodResult* methodResult,
   tU32 registerID)////FIX NCG3D-55494 RegisterId is passed to identify the requested client
{
   ETG_TRACE_USR4(( " RequestSliceCallHistoryListExtended entered" ));
   if ( !methodStart || !methodResult )
      return false;

   if(m_callHistory)
   {
      return m_callHistory->RequestSliceCallHistoryListExtended( methodStart, methodResult, registerID );
   }
   return false;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::RequestSlicePhoneBookListExtended
*
* DESCRIPTION:
*              Creates a subset of the list pointed by listhandle
*
*
* PARAMETER: [IN] methodStart :- Input arguments like listhandles are in this class
*            [OUT] methodResult :- The acutal sub list is passed in this class
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::RequestSlicePhoneBookListExtended(
   const most_phonbkfi_tclMsgRequestPhoneBookListSliceExtendedMethodStart* methodStart,
   most_phonbkfi_tclMsgRequestPhoneBookListSliceExtendedMethodResult* methodResult )

{
   ETG_TRACE_USR4(( " RequestSlicePhoneBookListExtended entered" ));
   if ( !methodStart || !methodResult )
      return false;

   if(m_list)
   {
      return m_list->RequestSlicePhoneBookListExtended( methodStart, methodResult );
   }

   return false;
}
//End of fix
//Fix CMG3G-8365 IS2421_Configurable phone number type representation@FC_Phonebook
/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::AddContactExtended
*
* DESCRIPTION: Adds the contact in VPBx where x intended for 2 to 11
**
*
* PARAMETER: [IN] methodStart :-
*            [OUT] methodResult :-
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::AddContactExtended(
                                       const most_phonbkfi_tclMsgAddContactExtendedMethodStart* methodStart,
                                       most_phonbkfi_tclMsgAddContactExtendedMethodResult* methodResult)
{
   ETG_TRACE_USR4(( " AddContactExtended entered" ));
   if ( !methodStart || !methodResult )
      return false;
   if(m_list)
   {
      return m_list->AddContactExtended(methodStart,methodResult);
   }
   return false;
}

/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::EditContactExtended
*
* DESCRIPTION:Edits the contact in VPBx where x intended for 2 to 11
**
*
* PARAMETER: [IN] methodStart :-
*            [OUT] methodResult :- Error if any
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::EditContactExtended(
                                        const most_phonbkfi_tclMsgEditContactExtendedMethodStart* methodStart,
                                        most_phonbkfi_tclMsgEditContactExtendedMethodResult* methodResult)
{
   ETG_TRACE_USR4(( " EditContactExtended entered" ));
   if ( !methodStart || !methodResult )
      return false;

   if(m_list)
   {
      return m_list->EditContactExtended(methodStart,methodResult);
   }
   return false;
}

/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::GetContactDetailsExtended
*
* DESCRIPTION: Gets the contact detail of VPBx where x intended for 2 to 11
**
*
* PARAMETER: [IN] methodStart :-
*            [OUT] methodResult :- Error if any
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::GetContactDetailsExtended(
   const most_phonbkfi_tclMsgGetContactDetailsExtendedMethodStart* methodStart,
   most_phonbkfi_tclMsgGetContactDetailsExtendedMethodResult* methodResult)
{
   ETG_TRACE_USR4(( " GetContactDetailsExtended entered" ));
   if(!methodStart || !methodResult)
      return false;

   if(m_list)
   {
      return m_list->GetContactDetailsExtended(methodStart,methodResult);
   }
   return false;
}
//End of Fix CMG3G-8365 IS2421_Configurable phone number type representation@FC_Phonebook


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_Database::GetPositionForContactHandle
*
* DESCRIPTION:
*   This method can be called for a dedicated contact list (identified by the u16ListHandle)
*   to find out at which position in this list a contact (identified via its u32ContactHandle) is located.
*
* PARAMETER: [IN] methodStart :-
*            [OUT] methodResult :-
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_Database::GetPositionForContactHandle(
   const most_phonbkfi_tclMsgGetPositionForContactHandleMethodStart* methodStart,
   most_phonbkfi_tclMsgGetPositionForContactHandleMethodResult* methodResult)
{
   ETG_TRACE_USR4(( " GetPositionForContactHandle entered" ));
   if(!methodStart || !methodResult)
      return false;

   if(m_list)
   {
      return m_list->GetPositionForContactHandle(methodStart,methodResult);
   }
   return false;
}
