/*******************************************************************************
*
* FILE:          FC_PhoneBook_CallHistory_List.cpp
*
* SW-COMPONENT:  FC_PhoneBook application
*
* PROJECT:
*
* DESCRIPTION:   Function to handle call history list
*
* AUTHOR:		 Balasubramani.V
*
* COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
*
*******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/
#include "../FC_PhoneBook_Debug.h"
#include "../FC_PhoneBook_Device_Details.h"
#include "../DbusClient/FC_PhoneBook_EvoDbusClient.h" //CMG3G-14314
#include "FC_PhoneBook_CallHistory_List.h"
#include "FC_PhoneBook_List.h"
#include "../VCard_Parser/FC_PhoneBook_VCard_Parser.h"
#include "../FC_PhoneBook_Dnl_Manager.h"
#include "../FC_PhoneBook_service_PhoneBook.h"
#include "FC_PhoneBook_Database.h"

#define UTFUTIL_S_IMPORT_INTERFACE_GENERIC
#include "utf_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC

#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONEBOOK_SQLITE
#include "trcGenProj/Header/FC_PhoneBook_CallHistory_List.cpp.trc.h"
#endif

#define MAX_DATETIME_FIELD_SIZE 15
#define MAX_DATE_FIELD_SIZE 8
#define MAX_TIME_FIELD_SIZE 6




/******************************************************************************/
/*                                                                            */
/* Constants and enum                                                                    */
/*                                                                            */
/******************************************************************************/
const char invalidDateStamp[] = "00000000";
const char invalidTimeStamp[] = "999999";

FC_PhoneBook_CallHistory_List* FC_PhoneBook_CallHistory_List::m_self = NULL;


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_CallHistory_List::
*
* DESCRIPTION:
*
*
* PARAMETER: None.
*
* RETURNVALUE: None
*
*******************************************************************************/
FC_PhoneBook_CallHistory_List::FC_PhoneBook_CallHistory_List()
{
   ETG_TRACE_USR4(("FC_PhoneBook_CallHistory_List() entered."));

   m_poSqlite= FC_PhoneBook_SQLite::GetSQLiteInstance();
   if(!m_poSqlite)
   {

      ETG_TRACE_ERR((" Failed to get SQlite instance for CallHistory "));
   }


   //FIX CMG3GB-505 Upon disconnecting a call/call is established/entering numbers in dialpad, resets when doing handsfree operation was observed
   AutoCompletePhNoSearchIndex = 0;
   //End of fix

   memset(blIgnoreCCHData, false, sizeof(blIgnoreCCHData));

   //FIX CMG3G-4786 CCHListSupport is shown inconsistently -- CallList loading is shown on Autoconnect
#ifdef UPDATE_CCH_SUPPORT_ON_INVALID_VCARD
   /*FIX GMMY16-8563 With paired samsung phone, the recent call list is not in sync with the device
   JAC-5006  Entry is not added to call list if incoming call is rejected from phone
   While downloading v-cards of a device which supports DateTime, when few v-cards are received without Date and Time, they are not ignored. The same is put into Phonebook DB.
   And it is informed to clients that only separate call history lists can be requested (as per GIS 327)*/
   bSendSCHSupport = FALSE;
   //End of fix
#endif
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_CallHistory_List::GetCurrentInstance()
*
* DESCRIPTION: Returns instance of Call history list
*
*
* PARAMETER: None.
*
* RETURNVALUE: Pointer to FC_PhoneBook_CallHistory_List
*
*******************************************************************************/
FC_PhoneBook_CallHistory_List* FC_PhoneBook_CallHistory_List::GetCurrentInstance()
{
   //CLONE TICKET FIX - GMMY15-1545 - Singleton creation for FC_PhoneBook_List and FC_PhoneBook_CallHistory_List
   if(m_self == NULL)
   {
      m_self = new FC_PhoneBook_CallHistory_List;
   }
   return m_self;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_CallHistory_List::vDestroyInstance
*
* DESCRIPTION: Deletes instance of Call History List
*
*
* PARAMETER: None.
*
* RETURNVALUE: None
*
*******************************************************************************/
tVoid FC_PhoneBook_CallHistory_List::vDestroyInstance()
{
   //CLONE TICKET FIX - GMMY15-1545 - Singleton creation for FC_PhoneBook_List and FC_PhoneBook_CallHistory_List
   delete m_self;
   m_self = NULLPTR;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_CallHistory_List::
*
* DESCRIPTION:
*
*
* PARAMETER: None.
*
* RETURNVALUE: None
*
*******************************************************************************/
FC_PhoneBook_CallHistory_List::~FC_PhoneBook_CallHistory_List()
{
   qDeleteAll(m_hashCHList);
   m_hashCHList.clear();
   m_hashCHList.squeeze();

   //FIX GMMY15-1578 Bosch Internal: Proper usage of SQLite database instead of RAM -- removing redundant objects which are created in RAM for different clients & sorted lists
   //Clear Multihash contents
   m_multiHashCHClientRequests.clear();
   m_multiHashCHClientRequests.squeeze();
   m_poSqlite = NULLPTR;

}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::CreateCallHistoryList
*
* DESCRIPTION:
*              Creates a call History list from the database.
*
*
* PARAMETER: [IN] methodStart :- Input arguments like sort type etc are in this class
*            [OUT] methodResult :- new list handle and list length are in this class
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_CallHistory_List::CreateCallHistoryList(
   const most_phonbkfi_tclMsgCreateCallHistoryListMethodStart* methodStart,
   most_phonbkfi_tclMsgCreateCallHistoryListMethodResult* methodResult,
   tU32 u32RegisterID)//FIX GMMY15-1578 RegisterID is passed to identify the requested client
{
   ETG_TRACE_USR4(("CreateCallHistoryList() entered."));
   bool bRetValue = false;
   FC_PhoneBook_SQLite::SortType sortType =
      getCallHistorySortType(&methodStart->e8CallHistorySortType);

   FC_PhoneBook_SQLite::FC_PhoneBook_enCallHistoryType callhistoryType =
      getCallHistoryType(&methodStart->e8CallHistoryType);
   FC_Device_Details * deviceObject = FC_Device_Details::deviceHandleToObjectInstance(methodStart->u8DeviceHandle);
   if(deviceObject == NULL)
   {
      ETG_TRACE_USR4((" Device object not found for given device handle"));
   }
   else
   {
      //FIX GMMY15-1578 Bosch Internal: Proper usage of SQLite database instead of RAM -- removing redundant objects which are created in RAM for different clients & sorted lists.
      //Compute the list handle from the parameters in methodstart
      tU16 u16ListHandle = (tU16)(methodStart->u8DeviceHandle * 100 + callhistoryType * 10 + sortType);//This format is chosen for creating list handle; as DeviceHandle, call history type and sortType can be identified easily as separate digits on debugging via ETG traces.
      ETG_TRACE_USR4 (( "List handle generated is: %d",u16ListHandle));

      FC_CallHistory_List_Data* pCHListData = NULL;
      if(m_multiHashCHClientRequests.contains(u16ListHandle))
      {
         //Retrieve the old List from Hashtable
         ETG_TRACE_USR4 (( "Retrieving old List Handle as same list has been requested before "));
         pCHListData = m_hashCHList.value(u16ListHandle);
      }
      else
      {
         ETG_TRACE_USR4 (( "New list created in RAM as list was not requested before"));
         pCHListData = new FC_CallHistory_List_Data;

         if(pCHListData)
         {
            //It is a newly created list, add it to the hash tables
            m_hashCHList.insert( u16ListHandle, pCHListData );
            ETG_TRACE_USR4 (( "m_hashCHList inserted with entry for Listhandle: %d",u16ListHandle));

            //Store values specific to the newly created list
            pCHListData->sortType = sortType;
            pCHListData->callHistoryTpye = callhistoryType;
            pCHListData->ConnectionID= deviceObject->getConnectionID();
         }
      }

      //FIX GMMY15-8739 Contacts are Displayed on the HMI, even though PBAP request was not accepted on the phone device.
      //HMI provided result for Create PB and CH list only if Prepare result has been received from DBus as success.
      if( TRUE == deviceObject->getPhBkPrepareCallBackStatus() )
      {
         //Get all records from Database if the list is not yet created
         //On subsequent changes to list via editing of a contact/storing a new contact or download of CH,
         //it will be updated via GetCallHistoryRecords() invocation in UpdateList on CallHistory HashTable via actions like enAction_Contact_Edited/enAction_Content_Refreshed
         if( (pCHListData) && (0 == pCHListData->callhistoryList.size()) )
         {
            m_poSqlite->GetCallHistoryRecords( sortType, callhistoryType , deviceObject->getConnectionID() , pCHListData->callhistoryList,
                  deviceObject->Device_DateTime_Support.CHDateTimeSupport);
         }
      }

      //If it is not a duplicate request, it needs to stored in the ClientRequests Hashtable
      if(!(m_multiHashCHClientRequests.contains(u16ListHandle,u32RegisterID)))
      {
         m_multiHashCHClientRequests.insert(u16ListHandle,u32RegisterID);
         ETG_TRACE_USR4 (( "Added entry to multiHashCHClientRequests for RegisterID: %d ",u32RegisterID));
      }

      //Update methodResult with values
      if(pCHListData)
      {
         methodResult->u16ListLength = (quint16)(pCHListData->callhistoryList.size());
         ETG_TRACE_USR4 (( "Updated methodResult with ListLength: %d",methodResult->u16ListLength));

         // CMG3G-6838: Resolution of LINT Warning 429 : Custodial Pointer
         tVoid*  pCustodialLose = pCHListData ;
         (tVoid) pCustodialLose ;
      }

      methodResult->u16ListHandle = u16ListHandle;
      methodResult->u8DeviceHandle = methodStart->u8DeviceHandle;
      //End of fix GMMY15-1578
      bRetValue = true;
   }
   return bRetValue;
}


/*******************************************************************************
*
* FUNCTION: function for Call History FC_PhoneBook_CallHistory_List::getCallHistoryType()
*
* DESCRIPTION:
*
*
*
* PARAMETER:
*
* RETURNVALUE: None.
*
*******************************************************************************/
FC_PhoneBook_SQLite::FC_PhoneBook_enCallHistoryType FC_PhoneBook_CallHistory_List::getCallHistoryType(
   const most_fi_tcl_e8_PhonBkCallHistoryType* type )
{
   ETG_TRACE_USR4(("getCallHistoryType() entered."));
   int val = type->enType;
   FC_PhoneBook_SQLite::FC_PhoneBook_enCallHistoryType tempCallHistoryType;

   //TODO
   // Need to know what to do in default
   switch ( val )
   {
   default:
   case most_fi_tcl_e8_PhonBkCallHistoryType::FI_EN_E8CCH:

      ETG_TRACE_USR4(("FC_PhoneBook_enCallHistoryType == callhisttype_combined"));
      tempCallHistoryType =  FC_PhoneBook_SQLite::callhisttype_combined;
      break;
   case most_fi_tcl_e8_PhonBkCallHistoryType::FI_EN_E8ICH:

      ETG_TRACE_USR4(("FC_PhoneBook_enCallHistoryType == callhisttype_incoming"));
      tempCallHistoryType =  FC_PhoneBook_SQLite::callhisttype_incoming;
      break;
   case most_fi_tcl_e8_PhonBkCallHistoryType::FI_EN_E8MCH:

      ETG_TRACE_USR4(("FC_PhoneBook_enCallHistoryType == callhisttype_missed"));
      tempCallHistoryType =  FC_PhoneBook_SQLite::callhisttype_missed;
      break;
   case most_fi_tcl_e8_PhonBkCallHistoryType::FI_EN_E8OCH:

      ETG_TRACE_USR4(("FC_PhoneBook_enCallHistoryType == callhisttype_outgoing"));
      tempCallHistoryType =  FC_PhoneBook_SQLite::callhisttype_outgoing;
      break;
   }

   return tempCallHistoryType;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::RequestSlicePhoneBookList
*
* DESCRIPTION:
*              Creates a subset of the list pointed by listhandle
*
*
* PARAMETER: [IN] methodStart :- Input arguments like listhandles are in this class
*            [OUT] methodResult :- The acutal sub list is passed in this class
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_CallHistory_List::RequestSliceCallHistoryList(
   const most_phonbkfi_tclMsgRequestSliceCallHistoryListMethodStart* methodStart,
   most_phonbkfi_tclMsgRequestSliceCallHistoryListMethodResult* methodResult,
   tU32 u32RegisterID)//FIX NCG3D-55494 RegisterID is used to identify the requested client)
{
   ETG_TRACE_USR4(("RequestSliceCallHistoryList() entered."));
   bool bRetValue = false;
   tU16 l_u16CorrectedListHandle =0;
   if(m_multiHashCHClientRequests.contains(methodStart->u16ListHandle,u32RegisterID))
   {
      if(false == m_hashCHList.contains(methodStart->u16ListHandle))
      {
         ETG_TRACE_USR4(("Wrong list handle sent by client, sending combined call history data"));
         return bRetValue;

         //Fix for NCG3D-55491
         /*QHash< quint16, FC_CallHistory_List_Data* >::iterator l_it;
         for(l_it = m_hashCHList.begin();l_it != m_hashCHList.end();l_it++)
         {
            if(l_it.value()->callHistoryTpye == FC_PhoneBook_SQLite::callhisttype_combined)
            {
               l_u16CorrectedListHandle = l_it.key();
               break;
            }
         }*/
      }
      else
      {
         ETG_TRACE_USR4(("Correct list handle sent by client"));
         l_u16CorrectedListHandle = methodStart->u16ListHandle;
      }
   }
   else
   {
      ETG_TRACE_ERR((" Client register ID is not availble "));
      return bRetValue;
   }
   FC_CallHistory_List_Data* data = m_hashCHList.value( l_u16CorrectedListHandle, NULLPTR );

   if(!data ) // BUGZID: 208352
   {

      ETG_TRACE_ERR((" FC_PhoneBook_CallHistory_List :- WARNING - Null Data for request List Handle "));
   }
   else
   {
      unsigned listLen = (unsigned)data->callhistoryList.size();
      ETG_TRACE_USR4(("listLen: %d",listLen));

      if ( (methodStart->u16WindowSize < 1) ||
         // methodStart->u32WindowStart < 0 || // This check is not needed
         (methodStart->u32WindowStart >= listLen) )
      {

         ETG_TRACE_USR4(("WARNING - negative number requested in windowStart or WindowSize, or zero windowSize, returning empty list"));
      }
      else
      {
         unsigned end = methodStart->u32WindowStart + methodStart->u16WindowSize;

         if ( end > listLen )
            end = listLen;

         ETG_TRACE_USR4(("TOTAL Items = %u, START = %d, END = %u, SIZE = %u", listLen, methodStart->u32WindowStart, end
               - 1, methodStart->u16WindowSize));

         //Fix CMG3G-12880
         FC_Device_Details* pDeviceObject =
               FC_Device_Details::getDeviceObject((INDEX) (data->ConnectionID
                     - 1));
         if (pDeviceObject)
         {
            //If device is not locked, continue as normal
            if (FALSE == pDeviceObject->bIsDeviceProbablyLocked())
            {
               for (unsigned counter = methodStart->u32WindowStart; counter
                     < end; ++counter)
               {
                  most_fi_tcl_PhonBkCallHistoryListSliceResultItem
                        sliceCallHistoryitem;

                  Convert(&data->callhistoryList.at((int) counter), &sliceCallHistoryitem);

                  methodResult->oCallHistoryListSliceResult.oItems.push_back(sliceCallHistoryitem);

               }
            }
            else
            {
               //Fix : CMG3G-12880 Mock call history list as empty since device is locked. so returning true
               ETG_TRACE_USR4(("Device is locked"));
            }
            bRetValue = true;
         }
         else
         {
            ETG_TRACE_USR4((" Device object not found for given device handle"));
         }
      }
   }

   return bRetValue;

}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::ReleaseCallHistoryList
*
* DESCRIPTION: If list handle received is available in CH Hash list,
*			   retrun true, else return false.
*
* PARAMETER: [IN] methodStart :- Input arguments like listhandles are in this class
*            [OUT] methodResult :- To add list handle parameter to same.

* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_CallHistory_List::ReleaseCallHistoryList(
   const most_phonbkfi_tclMsgReleaseCallHistoryListMethodStart* methodStart,
   most_phonbkfi_tclMsgReleaseCallHistoryListMethodResult* methodResult,
   tU32 u32RegisterID)//FIX GMMY15-1578 RegisterID is used to identify the requested client
{
   ETG_TRACE_USR4(("ReleaseCallHistoryList() entered."));

   bool bRetValue = false;

   //FIX GMMY15-1578 Bosch Internal: Proper usage of SQLite database instead of RAM -- removing redundant objects which are created in RAM for different clients & sorted lists.
   ETG_TRACE_USR4(( "Request from Client with RegisterID: %d",u32RegisterID));
   if(m_multiHashCHClientRequests.contains(methodStart->u16ListHandle,u32RegisterID))
   {
      ETG_TRACE_USR4(( "Client has requested for the list."));
      if(m_multiHashCHClientRequests.count(methodStart->u16ListHandle) > 1)
      {
         //Same list requested more than once. So we don't release the list from RAM
         ETG_TRACE_USR4(( "Multiple clients have requested for the same list."));
      }
      else
      {
         //Only this client has requested the list before so release list from RAM
         ETG_TRACE_USR4(( "Deleting the list and pointer to list. "));
         FC_CallHistory_List_Data* CHListData = m_hashCHList.take(methodStart->u16ListHandle);//Deleting entry from CH hash
         if(CHListData)
         {
            CHListData->callhistoryList.clear();//Deleting CH list of FC_PhoneBook_Callhistory_Detail inside structure FC_CallHistory_List_Data
            delete CHListData;//Deleting structure FC_CallHistory_List_Data
            m_hashCHList.squeeze();
         }
      }
      //For all valid requests, we delete the entry corresponding to the request from the particular client from ClientRequests HashList
      m_multiHashCHClientRequests.remove(methodStart->u16ListHandle,u32RegisterID);
      m_multiHashCHClientRequests.squeeze();

      //Fill values in methodResult
      methodResult->u16ListHandle = methodStart->u16ListHandle;
      bRetValue = true;
   }
   else
   {
      //List was not requested before by the client
      ETG_TRACE_USR4(( "Client has not requested for the list."));
      bRetValue = false;
   }
   //End of FIX GMMY15-1578

   return bRetValue;
}

/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_CallHistory_List::Convert()
*
* DESCRIPTION:
*              Converts the Qt format structure into the MOST format structure
*
*
* PARAMETER: [IN] contact :- The Qt format contact info
*            [OUT] item :- MOST format contact info
* RETURNVALUE: None.
*
*******************************************************************************/
void FC_PhoneBook_CallHistory_List::Convert(
   const FC_PhoneBook_Callhistory_Detail* callHistorydata,
   most_fi_tcl_PhonBkCallHistoryListSliceResultItem* CallHistoryitem )
{
   ETG_TRACE_USR4(("Convert() entered."));
   QByteArray byteArray;
   QByteArray tmpByteArray;

   CallHistoryitem->u32ContactHandle = callHistorydata->contactHandle;

   // First Name
   byteArray = callHistorydata->firstName.toUtf8();
   CallHistoryitem->sFirstName.bSet( byteArray.constData() );
   ETG_TRACE_USR4((" FIRST NAME = %s", byteArray.constData() ));

   // Last Name
   byteArray = callHistorydata->lastName.toUtf8();
   CallHistoryitem->sLastName.bSet( byteArray.constData() );

   ETG_TRACE_USR4((" LAST NAME = %s", byteArray.constData()));

   // Call Type
   CallHistoryitem->e8CallHistoryType.enType=
      (most_fi_tcl_e8_PhonBkCallHistoryType::tenType)callHistorydata->callType;

   ETG_TRACE_USR4(( " CallHistory Type = %d", (quint8)CallHistoryitem->e8CallHistoryType.enType));

   // Number Type
   //Fix NCG3D-13311 Not display icon registered phonebook of phone on the Call History list.(Receved calls/Dialled numbers/missed calls)
   //Mapping of call history number type to most_fi_tcl_e8_PhonBkCallPhoneType was missing.
   quint8 u8NumberType = callHistorydata->numberType;
   MapContactDetailTypeToCallPhonetype(u8NumberType);
   CallHistoryitem->e8CallPhoneType.enType
         = (most_fi_tcl_e8_PhonBkCallPhoneType::tenType) u8NumberType;
   //End of fix

   ETG_TRACE_USR4(( " Number Type = %d", (quint8)CallHistoryitem->e8CallPhoneType.enType));

   // Phone Number
   byteArray = callHistorydata->phoneNumber.toUtf8();
   CallHistoryitem->sPhoneNumber.bSet( byteArray.constData());

   ETG_TRACE_USR4(( " PhoneNumber = %s", byteArray.constData()));

   // Year
   byteArray = callHistorydata->dateStamp.toUtf8();
   tmpByteArray = byteArray.mid(0,4);
   CallHistoryitem->oCallDateTime.oCallDate.sCldrYear.bSet(tmpByteArray.constData());

   ETG_TRACE_USR2(( " sCldr Year = %s", tmpByteArray.constData() ));

   // month
   tmpByteArray = byteArray.mid(4,2);
   CallHistoryitem->oCallDateTime.oCallDate.sCldrMonth.bSet(tmpByteArray.constData());

   ETG_TRACE_USR2((" sCldr Month = %s", tmpByteArray.constData() ));

   // Day
   tmpByteArray = byteArray.mid(6,2);
   CallHistoryitem->oCallDateTime.oCallDate.sCldrDay.bSet(tmpByteArray.constData());

   ETG_TRACE_USR2(( " sCldr Day = %s", tmpByteArray.constData() ));

   // Hours
   byteArray = callHistorydata->timeStamp.toUtf8();
   tmpByteArray = byteArray.mid(0,2);
   CallHistoryitem->oCallDateTime.oCallTime.sHours.bSet(tmpByteArray.constData());

   ETG_TRACE_USR2(( " s Hours = %s", tmpByteArray.constData() ));

   // Minutes
   tmpByteArray = byteArray.mid(2,2);
   CallHistoryitem->oCallDateTime.oCallTime.sMinutes.bSet(tmpByteArray.constData());

   ETG_TRACE_USR2(( " s Minutes = %s", tmpByteArray.constData() ));

   // Seconds
   tmpByteArray = byteArray.mid(4,2);
   CallHistoryitem->oCallDateTime.oCallTime.sSeconds.bSet(tmpByteArray.constData());

   ETG_TRACE_USR2(( " s Seconds = %s", tmpByteArray.constData() ));

}


/*******************************************************************************
*
* FUNCTION: function for Call History FC_PhoneBook_CallHistory_List::getCallHistorySortType()
*
* DESCRIPTION:
*
*
*
* PARAMETER:
*
* RETURNVALUE: None.
*
*******************************************************************************/
FC_PhoneBook_SQLite::SortType FC_PhoneBook_CallHistory_List::getCallHistorySortType(
   const most_fi_tcl_e8_PhonBkCallHistorySortType* type )
{
   ETG_TRACE_USR4(("getCallHistorySortType() entered."));
   int val = type->enType;
   FC_PhoneBook_SQLite::SortType tempSortTpye;

   //TODO
   // Need to know what to do in default
   switch ( val )
   {
   default:
   case most_fi_tcl_e8_PhonBkCallHistorySortType::FI_EN_E8CH_SORT_DATETIME:

      ETG_TRACE_USR4(("CallhistaroySortType == DateTime"));
      tempSortTpye =  FC_PhoneBook_SQLite::DateTime;
      break;
   case most_fi_tcl_e8_PhonBkCallHistorySortType::FI_EN_E8CH_SORT_FIRSTNAME:

      ETG_TRACE_USR4(("CallhistaroySortType == FirstName"));
      tempSortTpye =  FC_PhoneBook_SQLite::FirstName;
      break;
   case most_fi_tcl_e8_PhonBkCallHistorySortType::FI_EN_E8CH_SORT_LASTNAME:

      ETG_TRACE_USR4(("CallhistaroySortType == Address_LastName"));
      tempSortTpye =  FC_PhoneBook_SQLite::LastName;
      break;
   }

   return tempSortTpye;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::AddNewVCardData
*
* DESCRIPTION:
*              Add new data to ByteArray and start parsing if completed is true.
*
*
* PARAMETER:
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_CallHistory_List::AddNewVCardData(GByteArray* data,
                                                    int phonebookType, unsigned char deviceHandle )
{
   ETG_TRACE_USR4(("AddNewVCardData() entered."));
   QByteArray ba(( const char*)(data->data), (int)data->len );
   FC_Device_Details* deviceObject;
   //static int i = 0;
   //Fix for GMMY15-11113 Reset on Phonebook
   switch(phonebookType)
   {
   case DownloadOutGoingCallHistorySimLocal:
   case DownloadMissedCallHistorySimLocal:
   case DownloadIncomingCallHistorySimLocal:
   case DownloadCombinedCallHistorySimLocal:
      {
         QList<FC_PhoneBook_Callhistory_Detail> list;

         if (true == ParseVCards( ba, phonebookType, deviceHandle, list))
         {
            if ( !list.isEmpty() )
            {
               deviceObject = FC_Device_Details::deviceHandleToObjectInstance(deviceHandle);

               if(deviceObject)
               {
                  // Special processing for first block of vCards of a CH list
                  if(deviceObject->m_countDownload == 0)
                  {
                     deviceObject->m_countDownload = 1;

                     ETG_TRACE_USR4(("download type == %u",deviceObject->downloadType));
                     tBool l_bMissedCallPresent = false;

                     // CH list might not be empty, therefore delete current entries before adding new data
                     switch(deviceObject->downloadType)
                     {
                     case DownloadOutGoingCallHistorySimLocal:
                        DeleteCallHistoryRecordsGivenCallType(deviceObject->getConnectionID(),FC_PhoneBook_SQLite::callhisttype_outgoing);
                        break;

                     case DownloadMissedCallHistorySimLocal:
                        l_bMissedCallPresent = true;
                        DeleteCallHistoryRecordsGivenCallType(deviceObject->getConnectionID(),FC_PhoneBook_SQLite::callhisttype_missed);
                        break;

                     case DownloadIncomingCallHistorySimLocal:
                        DeleteCallHistoryRecordsGivenCallType(deviceObject->getConnectionID(),FC_PhoneBook_SQLite::callhisttype_incoming);
                        break;

                     case DownloadCombinedCallHistorySimLocal:
                        l_bMissedCallPresent = true;
                        DeleteCallHistoryRecordsGivenConnectionID(deviceObject->getConnectionID());
                        break;

                     default:
                        ETG_TRACE_ERR((" deviceObject->downloadType is NOT VALID "));
                     }

                     // Check if a new Missed Call is present in MCH or CCH list
                     if(l_bMissedCallPresent)
                     {
                        vCheckMissedCallPresence(deviceObject, list, deviceObject->downloadType);
                     }
                  }  // end of: Special processing for first block of vCards in list


                  m_poSqlite->InsertCallHistoryRecords( list, deviceObject->getConnectionID() );
               }
               else
               {
                  ETG_TRACE_ERR(("Device Object is NULL"));
               }
            }  // end of: if ( !list.isEmpty() )
         }
         else
         {
            ETG_TRACE_USR4(("no documents"));
         }

      }
      break;

   default:
      {
         ETG_TRACE_USR4(("Not Valid download  Type "));
      }
   }
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::ParseVCards
*
* DESCRIPTION:
*              Parses the vcard stream.
*
*
* PARAMETER:
* RETURNVALUE:
*
*******************************************************************************/
bool FC_PhoneBook_CallHistory_List::ParseVCards(QByteArray ba, int phonebookType,
                                                unsigned char deviceHandle,QList<FC_PhoneBook_Callhistory_Detail> &list)
{
   ETG_TRACE_USR4(("ParseVCards() entered."));
   FC_PhoneBook_VCard_Parser parser;
   quint8 u8CallTypeDownloaded;
   quint8 u8ConnectionID;

   FC_Device_Details* DevDetails=
      FC_Device_Details::deviceHandleToObjectInstance(deviceHandle);

   if(DevDetails)
   {
      u8ConnectionID= DevDetails->getConnectionID();
      u8CallTypeDownloaded= MapCallHistoryTypeFromDnlCommand((FC_PhoneBookDwldCallType)
         FC_PhoneBook_Dnl_Manager::getCurDownloadTypeForConnectionID(
         u8ConnectionID));
   }
   else
   {
      ETG_TRACE_USR4(("Device object is NULL"));
      ETG_TRACE_ERR(("Device object is NULL"));
      return false;
   }

   QTime CallHistParseVCardsTime;
   CallHistParseVCardsTime.start();

   QList<FC_PhoneBook_VCard_Document> docs = parser.parseData(ba);
   int docsSize = docs.size();

   ETG_TRACE_USR2(( " -PBDL.Perf- CallHist 'parser.parseData' needed time= %d ms for %d vCards", CallHistParseVCardsTime.elapsed(), docsSize ));

   if ( !docsSize )
   {
      ETG_TRACE_USR4(("DATABASE : Warning, doc list empty on parsing vcards"));
      return false;
   }

   QTime CallHistMatchContactTime;
   CallHistMatchContactTime.start();
   for ( int docCounter = 0; docCounter < docsSize; ++docCounter )
   {
      /*FIX GMMY16-8563 With paired samsung phone, the recent call list is not in synch with the device
      JAC-5006  Entry is not added to call list if incoming call is rejected from phone
      While downloading v-cards of a device which supports DateTime, when few v-cards are received without Date and Time, they are not ignored. The same is put into Phonebook DB.
      And it is informed to clients that only separate call history lists can be requested (as per GIS 327)*/

      //Initialise as true. Clear it, if the data received has date and time support
      blIgnoreCCHData[u8ConnectionID-1]= TRUE;

      const FC_PhoneBook_VCard_Document* doc;
      doc = &docs.at( docCounter );

      if ( !doc->isEmpty() )
      {
         QList<FC_PhoneBook_VCard_Property> props = doc->properties();
         int propSize = props.size();

         FC_PhoneBook_Callhistory_Detail contact;
         //Using the download type to determine the call type
         contact.callType= u8CallTypeDownloaded;

         for ( int propCounter = 0; propCounter < propSize; ++propCounter )
         {
            AddPropertyToCallhistory( &(props.at( propCounter )), &contact,
               phonebookType ,deviceHandle);
         }

         //FIX CMG3G-4786 CCHListSupport is shown inconsistently -- CallList loading is shown on Autoconnect
         //CCHSupport to be updated as FALSE on receiving an invalid v card amidst valid ones.
#ifdef UPDATE_CCH_SUPPORT_ON_INVALID_VCARD
         if((FC_PhoneBook_SQLite::callhisttype_combined == u8CallTypeDownloaded) &&
            (blIgnoreCCHData[u8ConnectionID-1]))
         {
            bSendSCHSupport = TRUE;//Send separate call history support as TRUE
         }
#endif

         // Check if the call history entry already part of Phonebook contacts, if so use the
         // same contact handle as phonebook contact
         FC_PhoneBook_Search_Result PhoneContact;
         tU8 u8NameEdited_dummy = 0;//Fix for GMMY16-782

         if(GetMatchingContactFromExistingCHtable(contact, deviceHandle, PhoneContact))
         {
             if(PhoneContact.contactHandle >= FC_PB_PHONEBOOK_CNT_HANDLE_OFFSET)
             {
                contact.contactHandle = PhoneContact.contactHandle;  // PhoneContact.contactHandle stems from contact in phonebook database
             }
             else
             {
                contact.contactHandle = 0;  // By setting the .contactHandle to 0 here, we ensure that a new ContactHandle is created consistently when the CallHistory entry is stored into the database.
             }
             contact.numberType= PhoneContact.contactdetailtype;
             ETG_TRACE_USR4(("contact.numberType = %d",contact.numberType));
         }
         else if(GetMatchingContactForCallHistoryEntry(contact, deviceHandle, PhoneContact, u8NameEdited_dummy))//Fix for GMMY16-782
         {
            contact.contactHandle= PhoneContact.contactHandle;
            contact.numberType= PhoneContact.contactdetailtype;
            ETG_TRACE_USR4(("contact.numberType = %d",contact.numberType));
         }
         else
         {
            ETG_TRACE_USR4(("Clearing name as related call history entry not found in contacts."));
            ETG_TRACE_USR4(("firstName = %s",contact.firstName.toLatin1().constData()));
            ETG_TRACE_USR4(("lastName = %s",contact.lastName.toLatin1().constData()));
            contact.contactHandle= 0;
            contact.firstName = "";   //.clear();
            contact.lastName =  "";   //.clear();
         }

         //FIX GMMY16-12117 recent call list is updated with wrong date and time
         /*Macros as invalidDateStamp and invalidTimeStamp are defined as 00000000 and 999999.
         But the same is updated only when there exists X-IRMC-CALL-DATETIME field in the V-card and when the values in it are corrupted/does not exist.
         Case when there does not exist such a field in the V-card is not handled in FC_Phonebook.
         Blank values are sent to clients as part of Date and Time.*/

         if((contact.dateStamp.isEmpty()) && (contact.timeStamp.isEmpty()) )
         {
            ETG_TRACE_USR4((" Assigning Date and Time as Invalid "));
            contact.dateStamp = invalidDateStamp;
            contact.timeStamp = invalidTimeStamp;

            //FIX GMMY16-19854 Recent call list not available after incoming /outgoing Call.
            //Case 1:
            //When device supports CCH, and some of the v-cards received does not have date and time.
            //Case 2:
            //When device supports CCH, and none of the v-cards have date and time.
            //    - CCH support is to be sent as FALSE to clients.
            //    - Separate call history lists is to be downloaded from device.
            //    - Already downloaded CCH is to be replaced with new separate CH list data.

            //Here we set CHDateTimeSupport as FALSE on encountering any v-card with no date and time.
            //After completing of current download sequence, based on this variable we construct new download array.
            FC_Device_Details *p_deviceObject =  FC_Device_Details::deviceHandleToObjectInstance(deviceHandle);
            if(p_deviceObject != NULL)
            {
               if(FALSE == p_deviceObject->Device_DateTime_Support.DateTimesupportUpdated)
               {
                  ETG_TRACE_USR4(("Setting CHDateTimeSupport as FALSE"));
                  p_deviceObject->Device_DateTime_Support.CHDateTimeSupport  =  FALSE;
                  p_deviceObject->Device_DateTime_Support.DateTimesupportUpdated = TRUE;
               }
            }
            //End of fix
         }
         //End of fix

         ETG_TRACE_USR4(("  Inserting the following entry into Call history List  "));
         ETG_TRACE_USR2(("callType = %u",contact.callType));
         ETG_TRACE_USR2(("phoneNumber = %s",contact.phoneNumber.toLatin1().constData()));
         ETG_TRACE_USR2(("contactHandle = %d",contact.contactHandle));
         ETG_TRACE_USR2(("firstName = %s",contact.firstName.toUtf8().constData()));
         ETG_TRACE_USR2(("lastName = %s",contact.lastName.toUtf8().constData()));
         ETG_TRACE_USR2(("DateTime = %s",(contact.dateStamp + contact.timeStamp).toLatin1().constData()));

         //Insert the entry into call history list.
         list << contact;
      }
   }

   ETG_TRACE_USR2(( " -PBDL.Perf- CallHist 'GetMatchingContacts all CH entries' needed time= %d ms for %d vCards", CallHistMatchContactTime.elapsed(), docsSize ));

   //FIX CMG3G-4786 CCHListSupport is shown inconsistently -- CallList loading is shown on Autoconnect
   //CCHSupport not to be updated as FALSE on receiving an invalid v card amidst valid ones for SUZUKI/JAC.
#ifdef UPDATE_CCH_SUPPORT_ON_INVALID_VCARD
   if(TRUE == bSendSCHSupport)
   {
      ETG_TRACE_USR4(("Updating FS as CCH not supported"));
      tU8 ucDnlType= DevDetails->getDeviceCapabilities() & 0x1f;//AND-ing with '1f' will set CCHSupported to FALSE.
      DevDetails->setDeviceCapabilities(ucDnlType);
      fc_phonebook_tclService_PhoneBook::pGetInstance()->vUpdatePhoneBookFeatureSupport(DevDetails);
      bSendSCHSupport = FALSE;
   }
   //End of fix GMMY16-8563 JAC-5006
#endif

   ETG_TRACE_USR4(("ParseVCards() exited."));
   return true;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::AddPropertyToCallhistory
*
* DESCRIPTION:
*              Parses the vcard stream. for call history
*
*
* PARAMETER:
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_CallHistory_List::AddPropertyToCallhistory(const FC_PhoneBook_VCard_Property* prop,
                                                             FC_PhoneBook_Callhistory_Detail* contact, const int& phonebookType ,unsigned char deviceHandle)
{
   ETG_TRACE_USR4(("AddPropertyToCallhistory() entered."));
   (tVoid)phonebookType;
   QString text;
   const QChar dateTime_separator = 'T';

   FC_PhoneBook_VCard_Property::Type propType;


   FC_PhoneBook_List* poFC_PhoneBook_List = FC_PhoneBook_List::pGetInstance();

   QMultiHash <QString,QString> full_m_Parameters = prop->parameters();
   FC_Device_Details *p_checkObject;

   propType = prop->getType();

   QString name = prop->name();
   QChar space(' ');

   if ( name == "FN" )
   {
      if ( propType == FC_PhoneBook_VCard_Property::List || propType == FC_PhoneBook_VCard_Property::Compound)
         text = prop->value<QStringList>().join(" ").simplified();
      else
         text = prop->value().simplified();

      if ( (!text.isEmpty()) && ((contact->firstName.isEmpty()) && (contact->lastName.isEmpty())) )
      {
#ifdef LIMIT_ATTRIBUTE_LENGTH //FIX NCG3D-11656 Limiting of characters not required for AIVI
         contact->lastName =  text.left(25);

         //FIX SUZUKI-18215 strange "?" showed in Phone screen for a long Chinese contact
         //The replacement character is found if any and then string is truncated at the replacement character, to make it a valid utf8 string.
         vUtf8Validate(contact->lastName);
         //End of fix SUZUKI-18215
#else
         contact->lastName =  text;
#endif
      }
      ETG_TRACE_USR2(("FN lastName =%s",contact->lastName.toUtf8().constData()));
   }

   if ( name == "N" )
   {
      contact->lastName =  "";   //.clear();
      contact->firstName = "";   //.clear();
      if ( propType == FC_PhoneBook_VCard_Property::List || propType == FC_PhoneBook_VCard_Property::Compound)
      {
         QStringList NameValue= prop->value<QStringList>();
         ETG_TRACE_USR4(("Vcard property N: is a string list. Values are "));

         for(qint16 listIndex= 0; (listIndex< NameValue.size()&&
            (listIndex<= (qint16)FC_PHONEBOOK_PROP_N_FIRSTNAME_INDEX)); listIndex++)
         {

            ETG_TRACE_USR2(("NameValue[%d]- %s", listIndex, NameValue.at(listIndex).toUtf8().constData()));
            switch(listIndex)
            {
            case FC_PHONEBOOK_PROP_N_LASTNAME_INDEX:
#ifdef LIMIT_ATTRIBUTE_LENGTH //FIX NCG3D-11656 Limiting of characters not required for AIVI
               contact->lastName= NameValue.at(listIndex).left(25);

               //FIX SUZUKI-18215 strange "?" showed in Phone screen for a long Chinese contact
               //The replacement character is found if any and then string is truncated at the replacement character, to make it a valid utf8 string.
               vUtf8Validate(contact->lastName);
               //End of fix SUZUKI-18215
#else
               contact->lastName= NameValue.at(listIndex);
#endif
               break;
            case FC_PHONEBOOK_PROP_N_FIRSTNAME_INDEX:
#ifdef LIMIT_ATTRIBUTE_LENGTH //FIX NCG3D-11656 Limiting of characters not required for AIVI
               contact->firstName= NameValue.at(listIndex).left(25);

               //FIX SUZUKI-18215 strange "?" showed in Phone screen for a long Chinese contact
               //The replacement character is found if any and then string is truncated at the replacement character, to make it a valid utf8 string.
               vUtf8Validate(contact->firstName);
               //End of fix SUZUKI-18215
#else
               contact->firstName= NameValue.at(listIndex);
#endif
               break;
            default:
               {

                  ETG_TRACE_USR4(("Not Valid listIndex "));
               }
            }
         }
      }
      else
      {
         contact->lastName= prop->value().simplified();

         ETG_TRACE_ERR(("Vcard property N: is not a string list- %s", contact->lastName.toLatin1().constData()));
      }

      ETG_TRACE_USR2(("N firstName =%s",contact->firstName.toUtf8().constData()));
      ETG_TRACE_USR2(("N lastName =%s",contact->lastName.toUtf8().constData()));
   }
   else if ( name == "TEL" )
   {
      if ( propType == FC_PhoneBook_VCard_Property::List || propType == FC_PhoneBook_VCard_Property::Compound)
         text = prop->value<QStringList>().join(" ").simplified();
      else
         text = prop->value().simplified();

      // using qhash value for key TYPE
      contact->numberType =  MapTextToNumberForNumberType(prop->parameters().value("TYPE"));
      ETG_TRACE_USR2(("contact->numberType=%d",contact->numberType));

      if (contact->numberType == enNumType_Preffered)
      {

         contact->PPphoneNumber = text;
         ETG_TRACE_USR2(("contact->PPphoneNumber =%s", contact->PPphoneNumber.toUtf8().constData()));

         if((poFC_PhoneBook_List != NULL)){
         contact->phoneNumber = poFC_PhoneBook_List->StripTelephoneNumber(text);
         }

      }
      else
      {

         if (contact->PPphoneNumber.isEmpty())
         {
            contact->PPphoneNumber = text;
            ETG_TRACE_USR2(("contact->PPphoneNumber =%s", contact->PPphoneNumber.toUtf8().constData()));
            if((poFC_PhoneBook_List != NULL)){
            contact->phoneNumber = poFC_PhoneBook_List->StripTelephoneNumber(text);

            }
         }

      }

   }
   else if ( name == "SORT STRING" )
   {

   }
   else if(name == "X-IRMC-CALL-DATETIME")
   {
      //FIX GMMY16-19854 Recent call list not available after incoming /outgoing
      //Send Datetime support as TRUE to clients if some of the v-cards support datetime.
      //Only then those v-cards with date and time will be displayed with date and time in UI.
      FC_Device_Details *p_deviceObject =  FC_Device_Details::deviceHandleToObjectInstance(deviceHandle);
      if(p_deviceObject != NULL)
      {
         if(FALSE == p_deviceObject->Device_DateTime_Support.VcardsHaveDatetimeupdated)
         {
            ETG_TRACE_USR4(("Atleast one v-card has date and time"));
            p_deviceObject->Device_DateTime_Support.VcardsHaveDatetime = TRUE;
            p_deviceObject->Device_DateTime_Support.VcardsHaveDatetimeupdated = TRUE;
         }
      }
      //End of fix
      if ( propType == FC_PhoneBook_VCard_Property::List || propType == FC_PhoneBook_VCard_Property::Compound)
         text = prop->value<QStringList>().join(" ").simplified();
      else
         text = prop->value().simplified();

      p_checkObject =  FC_Device_Details::deviceHandleToObjectInstance(deviceHandle);
      if(p_checkObject != NULL)
      {
         blIgnoreCCHData[p_checkObject->getConnectionID()-1]= FALSE;
         ETG_TRACE_USR4(("Ignore combined call history cleared"));
      }
      else
      {
         ETG_TRACE_USR2((" 1Device object do not exists for device handle = %u ", deviceHandle));
      }

      // callType = missed, receive, dial
      contact->callType = MapTextToNumberForCallType(
         prop->parameters().value("TYPE"), deviceHandle);

      if((contact->dateStamp.isEmpty()) && (contact->timeStamp.isEmpty()) )
      {
         // date and time stamp is not present
         // 1. dateTime Field don't have T separator
         // 2. date or time have don't have proper field

         if (text.isEmpty())
         {
            contact->dateStamp = invalidDateStamp;
            contact->timeStamp = invalidTimeStamp;
         }
         else
         {
            // search the separator is present or not
            if (text.indexOf(dateTime_separator, 0, Qt::CaseSensitive) != -1)
            {
               QString str_datetime = text;
               QStringList qlist_temp = str_datetime.split("T");
               // check for date field
               if (qlist_temp[0].size() < MAX_DATE_FIELD_SIZE)
               {
                  contact->dateStamp = invalidDateStamp;
               }
               else
               {
                  contact->dateStamp = (qlist_temp[0]);
               }

               // check for time field
               if (qlist_temp[1].size() < MAX_TIME_FIELD_SIZE)
               {
                  contact->timeStamp = invalidTimeStamp;
               }
               else
               {
                  contact->timeStamp = qlist_temp[1];
               }
            }
            else
            {
               ETG_TRACE_USR2(("Separator not present complete field is corrupted"));
               contact->dateStamp = invalidDateStamp;
               contact->timeStamp = invalidTimeStamp;
            }
         }
      }
   }

}



/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::MapTextToNumberForCallType
*
* DESCRIPTION: Maps the Call_Type from String to a Number
*
*
*
* PARAMETER: [IN] -
*            [OUT] -
* RETURNVALUE: bool -
*
*******************************************************************************/
quint8 FC_PhoneBook_CallHistory_List::MapTextToNumberForCallType(
   QString CallType, quint8 u8DeviceHandle)
{
   ETG_TRACE_USR4(("MapTextToNumberForCallType() entered."));
   quint8 u8Ret = FC_PhoneBook_SQLite::callhisttype_invalid;

   if(0 == CallType.compare("MISSED"))
   {
      u8Ret= FC_PhoneBook_SQLite::callhisttype_missed;
   }
   else if(0 == CallType.compare("RECEIVED"))
   {
      u8Ret= FC_PhoneBook_SQLite::callhisttype_incoming;
   }
   else if(0 == CallType.compare("DIALED"))
   {
      u8Ret= FC_PhoneBook_SQLite::callhisttype_outgoing;
   }
   else
   {
      ETG_TRACE_USR4((" FC_PhoneBook_List: Call History Type Invalid "));
      ETG_TRACE_USR4(("Fetch the call type from the dowload command used"));

      FC_Device_Details* DeviceDetails =
         FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);

      if(DeviceDetails)
      {
         u8Ret= MapCallHistoryTypeFromDnlCommand(
            (FC_PhoneBookDwldCallType)FC_PhoneBook_Dnl_Manager::getCurDownloadTypeForConnectionID(
            DeviceDetails->getConnectionID()));
      }
      else
      {
         ETG_TRACE_ERR((" Device Object is NULL for device handle - %d", u8DeviceHandle));
      }
   }

   return u8Ret;
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::MapTextToNumberForNumberType
*
* DESCRIPTION: Maps the Number_Type from String to a Number
*
*
*
* PARAMETER: [IN] -
*            [OUT] -
* RETURNVALUE: bool -
*
*******************************************************************************/
quint8 FC_PhoneBook_CallHistory_List::MapTextToNumberForNumberType(QString NumberType)
{
   ETG_TRACE_USR4(("MapTextToNumberForNumberType() entered with NumberType %s", NumberType.toUtf8().constData()));
   quint8 u8Ret= enNumType_Unknown;

   if(0== NumberType.compare("PREF"))
   {
      u8Ret= enNumType_Preffered;
   }
   else if(0 == NumberType.compare("HOME"))
   {
      u8Ret= enNumType_Home;
   }
   else if(0 == NumberType.compare("WORK"))
   {
      u8Ret= enNumType_Work;
   }
   else if(0 == NumberType.compare("CELL"))
   {
      u8Ret= enNumType_Cell;
   }
   else
   {

      ETG_TRACE_USR4((" FC_PhoneBook_CallHistory_List: Call History Type Invalid "));
   }

   return u8Ret;
}


/**********************************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::GetMatchingContactFromExistingCHtable
*
* DESCRIPTION: Checks for a matching entry in the existing Call history table
*
* PARAMETER: [IN]   -
*            [OUT]  -
* RETURNVALUE: bool -
*
**********************************************************************************************/
bool FC_PhoneBook_CallHistory_List::GetMatchingContactFromExistingCHtable(
   FC_PhoneBook_Callhistory_Detail& CallHistoryRecord,
   quint8 u8DeviceHandle, FC_PhoneBook_Search_Result& PhoneContact)
{
   ETG_TRACE_USR4(("GetMatchingContactFromExistingCHtable() entered."));
   bool bContactFound = false;

   if (TRUE == (CallHistoryRecord.phoneNumber.isEmpty()))
   {
      ETG_TRACE_USR4((" GetMatchingContactFromExistingCHtable::Number is empty!!! "));
      return FALSE;
   }

   FC_PhoneBook_Search_Result MatchingRecord;

   //Searching Contact Based on Phone Number in Existing Call History table
   FC_Device_Details *deviceObject= FC_Device_Details::deviceHandleToObjectInstance(u8DeviceHandle);
   if( NULL != deviceObject )
   {
      if ((m_poSqlite != NULL)
            && (m_poSqlite->GetMatchingRecordBasedOnNumberfromCHtable(
                  MatchingRecord,
                  CallHistoryRecord.phoneNumber.toUtf8().constData(),
                  deviceObject->getConnectionID())))
      {
         ETG_TRACE_USR4(("Entry found in (old) call history table using phone number"));
         PhoneContact = MatchingRecord;

         CallHistoryRecord.firstName = MatchingRecord.firstname;
         CallHistoryRecord.lastName =  MatchingRecord.lastname;

         ETG_TRACE_USR2(("PhoneContact.firstName - %s", PhoneContact.firstname.toUtf8().constData()));
         bContactFound =  true;
      }
	  else
	  {
         ETG_TRACE_USR4(("Entry not found"));
	  }
   }
   else
   {
      ETG_TRACE_USR4(("DeviceObject is NULL"));
   }

   return bContactFound;
}


/**********************************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::GetMatchingContactForCallHistoryEntry
*
* DESCRIPTION: Checks for a matching entry in the phonebook contacts of the respective device and
*              then in the vehicle phonebook.
*              If a matching entry is found, then the same contact handle as used by the contact is
*              used for call history entry
*              If the first Name, Last Name of the Call History records does not match the
*              SearchResult,then Callhistory details/records are updated
*
* PARAMETER: [IN]   -
*            [OUT]  -
* RETURNVALUE: bool -
*
**********************************************************************************************/
bool FC_PhoneBook_CallHistory_List::GetMatchingContactForCallHistoryEntry(
   FC_PhoneBook_Callhistory_Detail& CallHistoryRecord,
   quint8 u8DeviceHandle, FC_PhoneBook_Search_Result& PhoneContact,
   tU8 &u8NameEdited)//FIX GMMY16-782
{
   ETG_TRACE_USR4(("GetMatchingContactForCallHistoryEntry() entered."));

   bool bRet = false;
   FC_PhoneBook_Search_Result MatchingRecord;

   //FIX GMMY17-3682 [CT] Having received a call from an Unknown or Restricted number the display shows the call being from myself and the number being unknown.
   //Do not check the DB if search string is empty.
   if (TRUE == (CallHistoryRecord.phoneNumber.isEmpty()))
   {
      ETG_TRACE_USR4((" CallHistory entry is from a private number - Number is empty!!! "));
      return false;
   }
   //End of fix

   QTime DebugTime;
   DebugTime.start();

   bool blEmptyName = false;

   if( CallHistoryRecord.firstName.isEmpty() && CallHistoryRecord.lastName.isEmpty() )
   {
      blEmptyName = true;
   }

   if (m_poSqlite != NULL)
   {
      if ((!blEmptyName) && (m_poSqlite->GetMatchingRecordBasedOnName(MatchingRecord,
                                                                      CallHistoryRecord.firstName.toUtf8().constData(),
                                                                      CallHistoryRecord.lastName.toUtf8().constData(),
                                                                      CallHistoryRecord.phoneNumber.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN).toUtf8().constData(),
                                                                      u8DeviceHandle)))
      {
         ETG_TRACE_USR4(( "Call History entry found via 'Name' in DeviceTable DH= %d", u8DeviceHandle ));
         PhoneContact = MatchingRecord;
         bRet = true;
      }
      else if ((m_poSqlite->GetMatchingRecord(MatchingRecord, CallHistoryRecord.phoneNumber.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN).toUtf8().constData(),
                                              FC_PhoneBook_SQLite::PhoneNumber, u8DeviceHandle)))
      {
          ETG_TRACE_USR4(( "Call History entry found via 'Number' in DeviceTable DH= %d", u8DeviceHandle ));
          bRet = true;
      }
      else if (m_poSqlite->GetMatchingRecord(MatchingRecord,CallHistoryRecord.phoneNumber.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN).toUtf8().constData(),
                                             FC_PhoneBook_SQLite::PhoneNumber,FC_PHONEBOOK_VEHICLE_DEVICEHANDLE))
      {
          ETG_TRACE_USR4(( "Call History entry found via 'Number' in VehicleTable(VPB)" ));
          bRet = true;
      }
      else if (m_poSqlite->GetMatchingRecord(MatchingRecord, CallHistoryRecord.phoneNumber.right(FC_PHONEBOOK_LIST_SEARCH_NUM_LEN).toUtf8().constData(),
                                             FC_PhoneBook_SQLite::PhoneNumber, (quint8)(u8DeviceHandle + FC_PB_TOTAL_NUMBER_OF_VPB)))
      {
          ETG_TRACE_USR4(( "Call History entry found via 'Number' in VPBx (related DH= %d)", u8DeviceHandle ));
          bRet = true;
      }

      if(true == bRet )
      {
          if( (CallHistoryRecord.firstName == MatchingRecord.firstname) && (CallHistoryRecord.lastName == MatchingRecord.lastname ) )
          {
              ETG_TRACE_USR4(("Contact FirstName and LastName matches with the CallHistory Record"));
          }
          else
          {
              u8NameEdited = 1;
              CallHistoryRecord.firstName = MatchingRecord.firstname;
              CallHistoryRecord.lastName = MatchingRecord.lastname;
          }
          PhoneContact = MatchingRecord;
      }
      else
      {
           ETG_TRACE_USR4(( "Call History Entry not found in Contact lists (DHx, VPB or VPBx)" ));
      }
   }
   else
   {
	   ETG_TRACE_ERR(("m_poSqlite is NULL"));
   }

   ETG_TRACE_USR2(( " -PBDL.Perf- 'GetMatchingContactForCallHistoryEntry' exited after %d ms ", DebugTime.elapsed() ));

   return bRet;
}

/**********************************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::ClearcallHistory
*
* DESCRIPTION: Clears all call history records from the database
*
* PARAMETER: [IN]   -
*            [OUT]  -
* RETURNVALUE:
*
**********************************************************************************************/

void FC_PhoneBook_CallHistory_List::ClearcallHistory()
{
   ETG_TRACE_USR4(("FC_PhoneBook_CallHistory_List: ClearcallHistory Entered"));
   quint8 i;
   ListChange_Handler mListChangeHandler;
   for(i = 1 ; i <= FC_PB_TOTAL_NUMBER_OF_CH; i++)
   {
      mListChangeHandler.UpdateList(&m_hashCHList, enAction_Content_Cleared,i, 0, 0, 0);
   }
}


/**********************************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::DeleteCallHistoryRecordsGivenCallType
*
* DESCRIPTION: deletes all call history records for a particular call type for a given
* device from the database
*
* PARAMETER: [IN]   - Connecton_ID and Call Type
*            [OUT]  -
* RETURNVALUE:
*
**********************************************************************************************/
bool FC_PhoneBook_CallHistory_List::DeleteCallHistoryRecordsGivenCallType(
   quint8 u8ConnectionID,	quint8 u8CallType)
{
   ETG_TRACE_USR4(("DeleteCallHistoryRecordsGivenCallType() entered."));
   if (m_poSqlite != NULL)
   {
      m_poSqlite->DeleteCallHistoryRecordsOfCallType(u8ConnectionID, u8CallType);
      return TRUE;
   }
   return FALSE;
}


/**********************************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::RefreshCallHistoryOnDeviceDisconnect
*
* DESCRIPTION:  This fucntion is triggered when a device is disconnected
*
* PARAMETER: [IN]   - Connection ID of the device disconnected
*            [OUT]  -
* RETURNVALUE:
*
**********************************************************************************************/
void FC_PhoneBook_CallHistory_List::
RefreshCallHistoryOnDeviceDisconnect(quint8 u8ConnectionID)
{
   ETG_TRACE_USR4(("RefreshCallHistoryOnDeviceDisconnect() entered."));
   if (m_poSqlite != NULL)
   {
      m_poSqlite->DeleteCallHistoryRecords(u8ConnectionID); //No call history type passed
      //to delete all call types

      ListChange_Handler mListChangeHandler;
      mListChangeHandler.UpdateList(&m_hashCHList, enAction_Content_Cleared, u8ConnectionID,
         0, 0);
   }
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::SearchPhoneNumberEntries
*
* DESCRIPTION:
*
*
*
* PARAMETER: [IN] methodStart :- Phone Number to search
*            [OUT] methodResult :- Matched PhoneNumber with first name, last name sorted by date time of recent call history lists(success or failure)
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
void FC_PhoneBook_CallHistory_List::SearchPhoneNumberEntries(const most_phonbkfi_tclMsgAutocompletePhoneNumberEntryMethodStart* methodStart,
                                                             most_phonbkfi_tclMsgAutocompletePhoneNumberEntryMethodResult* methodResult,guint8 ConnectionID)
{
   ETG_TRACE_USR4(("FC_PhoneBook_CallHistory_List::SearchPhoneNumberEntries; Entered "));
   quint32 RecIndex = 0;
   quint8 DataCache = 0;
   quint8 SearchIndex = 0;
   quint8 u8ListSize = 0;
   QString ExistingPhNo;
   QString SearchPhNo;
   FC_PhoneBook_List* poFC_PhoneBook_List = FC_PhoneBook_List::pGetInstance();
   tString PhoneNumberDigits = methodStart->sPhoneNumberDigits.szValue;
   QString PhNoToSearch(PhoneNumberDigits);
   if (poFC_PhoneBook_List != NULL)
   {
    SearchPhNo =poFC_PhoneBook_List->StripTelephoneNumber(PhNoToSearch);
   }


   ETG_TRACE_USR2(("PhoneNumberDigits to Search  = %s",PhoneNumberDigits));

   for(RecIndex = 0; RecIndex < MAX_AUTO_COMPLETE_LISTS; RecIndex++)
   {
      //		if(!m_poSqlite->AutoCompletePhNoSearchRecords[RecIndex].empty())
      if(m_poSqlite != NULL)
      {
         ExistingPhNo =  m_poSqlite->SearchedPhoneNumberDigit[RecIndex].toUtf8().constData();
         ETG_TRACE_USR2(("EXISTING Phone Numbers before Match =%s",ExistingPhNo.toLatin1().constData()));
         if(ExistingPhNo.compare(SearchPhNo) == 0)
         {

            ETG_TRACE_USR2(("Matched Phone Number = %s",ExistingPhNo.toLatin1().constData()));
            DataCache = 1;
            break;
         }
      }
   }
   if(( m_poSqlite != NULL) && (DataCache == 1))
   {

      ETG_TRACE_USR4(("DATA Matched  "));

      u8ListSize = (quint8)m_poSqlite->AutoCompletePhNoSearchRecords[RecIndex].size();

      if(u8ListSize != VALUE_ZERO)
      {
         methodResult->bAutocompleteMatch = TRUE;
      }
      else
      {
         methodResult->bAutocompleteMatch = FALSE;
      }

      methodResult->u8NumberOfMatches = u8ListSize;

      if(!m_poSqlite->AutoCompletePhNoSearchRecords[RecIndex].empty())
      {
         for(quint8 u8Index= 0; u8Index < u8ListSize; u8Index++)
         {
            QByteArray ba;
            AutoCompletePhoneNumberEntryResult MatchedContacts;
            most_fi_tcl_PhonBkAutocompleteMultipleMatchListItem	items;

            MatchedContacts = m_poSqlite->AutoCompletePhNoSearchRecords[RecIndex].at(u8Index);
            ba = MatchedContacts.FirstName.toUtf8();
            items.sFirstName.bSet(ba.constData());
            ba = MatchedContacts.LastName.toUtf8();
            items.sLastName.bSet(ba.constData());
            ba = MatchedContacts.PhoneNumber.toUtf8();
            items.sPhoneNumber.bSet(ba.constData());

            methodResult->oAutocompleteMultipleMatchList.oItems.push_back(items);
         }
         //FIX CMG3GB-505 Upon disconnecting a call/call is established/entering numbers in dialpad, resets when doing handsfree operation was observed
         methodResult->sAutocompleteMatchPhoneNumber.bSet(methodResult->oAutocompleteMultipleMatchList.oItems[0].sPhoneNumber.szValue);
         //End of fix
      }
   }
   else
   {

      ETG_TRACE_USR4(("DATA MISMatched  "));

      SearchIndex = (quint8)AutoCompletePhNoSearchIndex;
      if(( m_poSqlite != NULL ) && (!m_poSqlite->SearchPhoneNumberEntries(SearchPhNo,SearchIndex,ConnectionID)))
      {

         ETG_TRACE_USR4((" Search for Phone Number Digits Failed "));

         methodResult->bAutocompleteMatch = FALSE;
         methodResult->u8NumberOfMatches = u8ListSize;
         //FIX CMG3GB-505 Upon disconnecting a call/call is established/entering numbers in dialpad, resets when doing handsfree operation was observed
         methodResult->sAutocompleteMatchPhoneNumber.bSet(NULLPTR);
         //End of fix
      }
      else
      {
         if( m_poSqlite != NULL)
         {
            u8ListSize = (quint8)m_poSqlite->AutoCompletePhNoSearchRecords[SearchIndex].size();
            methodResult->bAutocompleteMatch = TRUE;
            methodResult->u8NumberOfMatches = u8ListSize;
            for(quint8 u8Index= 0; u8Index < u8ListSize; u8Index++)
            {
               QByteArray ba;
               AutoCompletePhoneNumberEntryResult MatchedContacts;
               most_fi_tcl_PhonBkAutocompleteMultipleMatchListItem	items;

               MatchedContacts = m_poSqlite->AutoCompletePhNoSearchRecords[SearchIndex].at(u8Index);

               ba = MatchedContacts.FirstName.toUtf8();
               items.sFirstName.bSet(ba.constData());
               ba = MatchedContacts.LastName.toUtf8();
               items.sLastName.bSet(ba.constData());
               ba = MatchedContacts.PhoneNumber.toUtf8();
               items.sPhoneNumber.bSet(ba.constData());

               methodResult->oAutocompleteMultipleMatchList.oItems.push_back(items);
            }
            //FIX CMG3GB-505 Upon disconnecting a call/call is established/entering numbers in dialpad, resets when doing handsfree operation was observed
            methodResult->sAutocompleteMatchPhoneNumber.bSet(methodResult->oAutocompleteMultipleMatchList.oItems[0].sPhoneNumber.szValue);
            //End of fix
         }
      }
      //FIX CMG3GB-505 Upon disconnecting a call/call is established/entering numbers in dialpad, resets when doing handsfree operation was observed
      AutoCompletePhNoSearchIndex = (AutoCompletePhNoSearchIndex==0)?1:0;//Toggle the values
      //End of fix
   }
}


/**********************************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::clearAutocompleteListsOnOSChange
*
* DESCRIPTION:
*
* PARAMETER: [IN]   -
*            [OUT]  -
* RETURNVALUE:
*
**********************************************************************************************/
void FC_PhoneBook_CallHistory_List::clearAutocompleteListsOnOSChange()
{
   ETG_TRACE_USR4(("clearAutocompleteListsOnOSChange() entered."));
   for(int RecIndex = 0;(( m_poSqlite != NULL) && ( RecIndex < MAX_AUTO_COMPLETE_LISTS )); RecIndex++)
   {
      m_poSqlite->AutoCompletePhNoSearchRecords[RecIndex].clear();
   }
   //	m_poSqlite->AutoCompletePhNoSearchRecords[] = {NULL, NULL};
}


/*******************************************************************************
*
* FUNCTION: MapCallHistoryTypeFromDnlCommand(FC_PhoneBookDwldCallType DnlCmd)
*
* DESCRIPTION:
* PARAMETER:
*
* RETURNVALUE: None.
*
*******************************************************************************/
FC_PhoneBook_SQLite::FC_PhoneBook_enCallHistoryType FC_PhoneBook_CallHistory_List::
MapCallHistoryTypeFromDnlCommand(FC_PhoneBookDwldCallType DnlCmd)
{
   ETG_TRACE_USR4(("MapCallHistoryTypeFromDnlCommand() entered."));
   FC_PhoneBook_SQLite::FC_PhoneBook_enCallHistoryType tempCallHistoryType;

   switch ( DnlCmd )
   {
   case DownloadIncomingCallHistorySimLocal:
      tempCallHistoryType =  FC_PhoneBook_SQLite::callhisttype_incoming;
      break;
   case DownloadMissedCallHistorySimLocal:
      tempCallHistoryType =  FC_PhoneBook_SQLite::callhisttype_missed;
      break;
   case DownloadOutGoingCallHistorySimLocal:
      tempCallHistoryType =  FC_PhoneBook_SQLite::callhisttype_outgoing;
      break;
   case DownloadCombinedCallHistorySimLocal:
      tempCallHistoryType =  FC_PhoneBook_SQLite::callhisttype_combined;
      break;
   default:
      tempCallHistoryType = FC_PhoneBook_SQLite::callhisttype_invalid;
   }

   return tempCallHistoryType;
}


/**********************************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::RefreshCallHistoryOnCallHistoryUpdate
*
* DESCRIPTION: Triggers list change on update of call history
*
* PARAMETER: [IN]   - connectionid of the device and callhistory type updated
*            [OUT]  -
* RETURNVALUE:
*
**********************************************************************************************/
void FC_PhoneBook_CallHistory_List::RefreshCallHistoryOnCallHistoryUpdate(
   quint8 u8ConnectionID, quint8 u8CallHistoryDnlCmd)
{
   ETG_TRACE_USR4(("RefreshCallHistoryOnCallHistoryUpdate() entered."));
   ListChange_Handler mListChangeHandler;

   quint8 CallHistoryType= (quint8)MapCallHistoryTypeFromDnlCommand(
      (FC_PhoneBookDwldCallType)u8CallHistoryDnlCmd);
   mListChangeHandler.UpdateList(&m_hashCHList, enAction_Content_Refreshed, u8ConnectionID,
      0, 0, CallHistoryType);
}


/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_List::ClearCallHistoryRecords
*
* DESCRIPTION:
**
*
* PARAMETER: [IN]
*            [OUT]
* RETURNVALUE:
*
*******************************************************************************/
void FC_PhoneBook_CallHistory_List::ClearCallHistoryRecords()
{
   ETG_TRACE_USR4(("Clearing all Call History data entered"));
   for( quint8 u8ConnID= 1;(( m_poSqlite != NULL) && ( u8ConnID <= FC_PB_TOTAL_NUMBER_OF_CH)); u8ConnID++)
   {
      m_poSqlite->DeleteCallHistoryRecords(u8ConnID);
   }
}


#ifdef BUGZID_206322 // Workaround for device disconection status
/***************************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::DeviceConnected
*
* DESCRIPTION: This function is called when a device is connected from IVI
*
* PARAMETER: [IN] -[OUT]
* RETURNVALUE:
*
****************************************************************************************/
void FC_PhoneBook_CallHistory_List::DeviceConnected(FC_Device_Details* DeviceDetails)
{
   ETG_TRACE_USR2(("DeviceConnected entered "));
   if((DeviceDetails) && (m_poSqlite))
   {
      quint8 u8ConnectionID= DeviceDetails->getConnectionID();
      //Fix for GMMY15-11511
      m_poSqlite->clearSearchPhoneDigitList();
      if(m_poSqlite->DeleteCallHistoryRecords(u8ConnectionID))
      {
         /*FIX GMMY16-8563 With paired samsung phone, the recent call list is not in synch with the device
         JAC-5006  Entry is not added to call list if incoming call is rejected from phone
         While downloading v-cards of a device which supports DateTime, when few v-cards are received without Date and Time, they are not ignored. The same is put into Phonebook DB.
         And it is informed to clients that only separate call history lists can be requested (as per GIS 327)*/
         blIgnoreCCHData[u8ConnectionID-1]= FALSE;

         //FIX CMG3G-4786 CCHListSupport is shown inconsistently -- CallList loading is shown on Autoconnect
         //CCHSupport to be updated as FALSE on receiving an invalid v card amidst valid ones.
#ifdef UPDATE_CCH_SUPPORT_ON_INVALID_VCARD
         bSendSCHSupport = FALSE;
#endif
         //End of fix GMMY16-8563 JAC-5006
      }
      else
      {
         ETG_TRACE_ERR(("Failed to delete call history records for ConnectionID= %d", u8ConnectionID));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Device Object is NULL"));
   }
}
#endif


/***************************************************************************************
* FUNCTION:  FC_PhoneBook_CallHistory_List::MapContactDetailTypeToCallPhonetype
*
* DESCRIPTION: Mapping for contactdeatailtype to callphonetype enum
*
* PARAMETER: [IN] u8NumberType :- u8NumberType of type contactdeatailtype
*            [OUT] u8NumberType :- u8NumberType of type callphonetype
* RETURNVALUE: None
*
*******************************************************************************/
void FC_PhoneBook_CallHistory_List::MapContactDetailTypeToCallPhonetype(quint8& u8NumberType)
{
   ETG_TRACE_USR4(("FC_PhoneBook_List::MapContactDetailTypeToCallPhonetype entered with numbertype %d", u8NumberType));

   switch (u8NumberType)
   {
      case FC_PhoneBook_SQLite::PrefferedNum:
         u8NumberType = most_fi_tcl_e8_PhonBkCallPhoneType::FI_EN_E8PREFERRED;
      break;

      case FC_PhoneBook_SQLite::CellNumber:
         u8NumberType = most_fi_tcl_e8_PhonBkCallPhoneType::FI_EN_E8CELL1;
      break;

      case FC_PhoneBook_SQLite::HomeNumber:
         u8NumberType = most_fi_tcl_e8_PhonBkCallPhoneType::FI_EN_E8HOME1;
      break;

      case FC_PhoneBook_SQLite::WorkNumber:
         u8NumberType = most_fi_tcl_e8_PhonBkCallPhoneType::FI_EN_E8WORK1;
      break;

      case FC_PhoneBook_SQLite::OtherNumber:
         u8NumberType = most_fi_tcl_e8_PhonBkCallPhoneType::FI_EN_E8OTHER;
      break;

      default:
         u8NumberType = most_fi_tcl_e8_PhonBkCallPhoneType::FI_EN_E8UNKNOWN;
   }
}


/***************************************************************************************

* FUNCTION:  DeleteCallHistoryRecordsGivenConnectionID
*
* DESCRIPTION: This function clears all the call history related data for the
* given connection id.
*
* PARAMETER: [IN] -[OUT]
* RETURNVALUE:
*
****************************************************************************************/
void FC_PhoneBook_CallHistory_List::DeleteCallHistoryRecordsGivenConnectionID(
   quint8 u8ConnectionID)
{
   ETG_TRACE_USR4(("DeleteCallHistoryRecordsGivenConnectionID entered "));
   if ( m_poSqlite != NULL)
   {
      if( m_poSqlite->DeleteCallHistoryRecords(u8ConnectionID))
      {
         ETG_TRACE_USR2(("Call History records deleted for uConnectionID= %d",
            u8ConnectionID));
      }
      else
      {
         ETG_TRACE_ERR(("Failed to delete call history records for ConnectionID= %d", u8ConnectionID));
      }
   }
   else
   {
      ETG_TRACE_FATAL(("m_poSqlite IS NULL"));
   }
}
void FC_PhoneBook_CallHistory_List::ClearVehicleCallhistorylist()
{

   ETG_TRACE_USR4(("ClearVehicleCallhistorylist function entered"));
   // FC_PhoneBook_CallHistory_List* l_poFC_PhoneBook_CallHistory_List;  Commented due to Lint warning
   QHash< quint16, FC_CallHistory_List_Data* >* l_poCallHistoryHashTable;

   l_poCallHistoryHashTable = &(m_hashCHList);



   QHashIterator<quint16, FC_CallHistory_List_Data*>
      l_itCallHistoryHashTable(*l_poCallHistoryHashTable);

   while (l_itCallHistoryHashTable.hasNext())
   {
      FC_CallHistory_List_Data* l_poFC_CallHistory_List_Data;

      l_itCallHistoryHashTable.next();

      ETG_TRACE_USR4(("Deleting the Call History list handle::%u",
         l_itCallHistoryHashTable.key()));

      l_poFC_CallHistory_List_Data = l_itCallHistoryHashTable.value();

      ETG_TRACE_USR4(("ClearVehicleCallhistory function l_poFC_CallHistory_List_Data->ConnectionID = %u",l_poFC_CallHistory_List_Data->ConnectionID));
      if(l_poFC_CallHistory_List_Data->ConnectionID == 1)
      {
         QList<FC_PhoneBook_Callhistory_Detail> *PBContactList=
            &l_poFC_CallHistory_List_Data->callhistoryList;

         most_phonbkfi_tclMsgListChangeStatus ListChangeStatus;

         ListChangeStatus.u16ListHandle= l_itCallHistoryHashTable.key();
         ListChangeStatus.e8ListType.enType=
            (most_fi_tcl_e8_PhonBkListType::tenType)enCallHistory_list;
         ListChangeStatus.e8ListChangeType.enType=
            (most_fi_tcl_e8_PhonBkListChangeType::tenType)enItems_removed;
         ListChangeStatus.u16ListLength= (quint16)PBContactList->size();
         ListChangeStatus.u32NumItems= 0xFFFFFFFF;
         ListChangeStatus.oChangeItems.u32Items.clear();
         ETG_TRACE_USR4(("Listhandle for VehicleCallHistory = %u",ListChangeStatus.u16ListHandle));
         (fc_phonebook_tclService_PhoneBook::pGetInstance())->vPhoneBook_UpdateListChange(ListChangeStatus);

         delete l_poFC_CallHistory_List_Data;
      }
   }
   l_poCallHistoryHashTable->clear();
}

/**********************************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::vUpdateDateAndTimeDetails
*
* DESCRIPTION: Updates the date and Time stamps for the deviceObject
*
* PARAMETER: [IN]   - Device Object of the connected device.
*            [IN]   - Date stamp
*            [IN]   - Time stamp
*            [IN]   - Download Type
*            [IN]   - isUpdateForFirst
*            [OUT]  -
* RETURNVALUE:
*
**********************************************************************************************/
void FC_PhoneBook_CallHistory_List::vUpdateDateAndTimeDetails(FC_Device_Details* deviceObject, QString dateStamp,
                                                              QString timeStamp, int iDownloadType,
                                                              tBool isUpdateForFirst)
{
   ETG_TRACE_USR4(( " vUpdateDateAndTimeDetails ENTERED "));

   ETG_TRACE_USR4(( " -PBDL.MissCallInd- Storing Date and Time stamps for iDownloadType:%d",iDownloadType ));
   ETG_TRACE_USR1(( " -PBDL.MissCallInd- dateStamp= '%s' ", dateStamp.toUtf8().constData() ));
   ETG_TRACE_USR1(( " -PBDL.MissCallInd- timeStamp= '%s' ", timeStamp.toUtf8().constData() ));
   ETG_TRACE_USR1(( " (vUpdateDateAndTimeDetails) deviceObject->m_sFirstMissedCallDate= '%s' ", deviceObject->m_sFirstMissedCallDate.toUtf8().constData() ));
   ETG_TRACE_USR1(( " (vUpdateDateAndTimeDetails) deviceObject->m_sFirstMissedCallTime = '%s' ", deviceObject->m_sFirstMissedCallTime.toUtf8().constData() ));
   ETG_TRACE_USR1(( " (vUpdateDateAndTimeDetails) deviceObject->m_sRecentMissedCallDate= '%s' ", deviceObject->m_sRecentMissedCallDate.toUtf8().constData() ));
   ETG_TRACE_USR1(( " (vUpdateDateAndTimeDetails) deviceObject->m_sRecentMissedCallTime = '%s' ", deviceObject->m_sRecentMissedCallTime.toUtf8().constData() ));

   switch (iDownloadType)
   {
      case DownloadCombinedCallHistorySimLocal: // CCH 5
      case DownloadMissedCallHistorySimLocal: // MCH 3
         deviceObject->m_sRecentMissedCallDate = dateStamp;
         deviceObject->m_sRecentMissedCallTime = timeStamp;
         if (isUpdateForFirst)
         {
            ETG_TRACE_USR4(( " Updating FirstMissedCall Date and Time"));
            deviceObject->m_sFirstMissedCallDate = dateStamp;
            deviceObject->m_sFirstMissedCallTime = timeStamp;
         }
      break;
      default:
         ETG_TRACE_ERR(( " Invalid Download Type " ));
   }
}

/**********************************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::vCheckMissedCallPresence
*
* DESCRIPTION: Checks the missed call count from the List and Updates the DeviceObject
*
* PARAMETER: [IN]   - Device Object.
*            [IN]   - CallHistory List
*            [IN]   - Download Type
*            [OUT]  -
* RETURNVALUE:
*
**********************************************************************************************/
void FC_PhoneBook_CallHistory_List::vCheckMissedCallPresence(FC_Device_Details* deviceObject,
                                                             QList<FC_PhoneBook_Callhistory_Detail>& list,
                                                             int iDownloadType)
{
   ETG_TRACE_USR4(( " vCheckMissedCallPresence ENTERED "));
   ETG_TRACE_USR4(( " m_deviceHandle:%d",deviceObject->getDeviceHandle()));

   QString l_sLatestDate = EMPTY_STRING;
   QString l_sLatestTime = EMPTY_STRING;
   int iter_count = 0;
   tBool l_bMissedCallFound = false;
   tU32  l_MissedcallCount = 0;

   QList<FC_PhoneBook_Callhistory_Detail>::iterator l_it;

   for (l_it = list.begin(); l_it != list.end(); l_it++)
   {
      iter_count++;
      ETG_TRACE_USR4(( " -PBDL.MissCallInd- iter_count:%d ",iter_count));

      //scenario: Device connected with No missed in its callHistory List
      //Get the latest Time stamp of any calltype if there is no missed call in the entire list
      if((deviceObject->m_sFirstMissedCallTime == QString(EMPTY_STRING)) && (l_sLatestTime == QString(EMPTY_STRING)))
      {
         ETG_TRACE_USR1(( " -PBDL.MissCallInd-Previous dateStamp= '%s' ", l_it->dateStamp.toUtf8().constData() ));
         ETG_TRACE_USR1(( " -PBDL.MissCallInd-Previous timeStamp= '%s' ", l_it->timeStamp.toUtf8().constData() ));
         l_sLatestDate = l_it->dateStamp;
         l_sLatestTime = l_it->timeStamp;
      }

      if (l_it->callType == FC_PhoneBook_SQLite::callhisttype_missed)
      {
         l_bMissedCallFound = true;

         //scenario:Device connected with at least One Missed call in the list
         if(deviceObject->m_sFirstMissedCallTime == QString(EMPTY_STRING))
         {
            ETG_TRACE_USR4(( " -PBDL.MissCallInd- Storing top most missed call during device connection" ));
            vUpdateDateAndTimeDetails(deviceObject,l_it->dateStamp,l_it->timeStamp,iDownloadType,TRUE);
            return;
         }
         else //Device receives missed call after connection
         {
            ETG_TRACE_USR1(( " -PBDL.MissCallInd-m_sFirstMissedCallDate= '%s' ", deviceObject->m_sFirstMissedCallDate.toUtf8().constData() ));
            ETG_TRACE_USR1(( " -PBDL.MissCallInd-m_sFirstMissedCallTime = '%s' ", deviceObject->m_sFirstMissedCallTime.toUtf8().constData() ));
            ETG_TRACE_USR1(( " -PBDL.MissCallInd-Latest dateStamp= '%s' ", l_it->dateStamp.toUtf8().constData() ));
            ETG_TRACE_USR1(( " -PBDL.MissCallInd-Latest timeStamp= '%s' ", l_it->timeStamp.toUtf8().constData() ));
            // Check if we have "invalid" Date/Time information in latest stored 'RecentMissed: CallDate / CallTime' or
            // in the 'first missed call' (of currently downloaded list).
            if ((deviceObject->m_sFirstMissedCallDate == QString(invalidDateStamp)) // invalidDateStamp = "00000000"
                  || (deviceObject->m_sFirstMissedCallTime == QString(invalidTimeStamp)) // invalidTimeStamp = "999999"
                  || (l_it->dateStamp == QString(invalidDateStamp)) // invalidDateStamp = "00000000"
                  || (l_it->timeStamp == QString(invalidTimeStamp)) // invalidTimeStamp = "999999";
            )
            { // INVALID Date/Time information found. Therefore we skip the Date/Time comparison ..
               // because invalid Date/Time stamps could result in a wrong 'MissedCallIndicator' update.
               ETG_TRACE_USR1(( " -PBDL.MissCallInd- INVALID Date/Time stamps, therefore we skip any update of 'MissedCallIndicator'. " ));
            }
            else
            {
               // VALID Date/Time information found for both 'missed calls'.
               // Now we check if the 'latest' missed call Date/Time is differs from the First stored missed call.

               //286966- 1. When the latest DateStamp and the "m_sFirstMissedCallDate" is same, then the "m_sFirstMissedCallTime" should be less than the latest timestamp.
               //        2. Similarly, when the latest DateStamp is greater than the "m_sFirstMissedCallDate", irrespective of the timeStamp, it should increment the counter.
               //           This happens only during the first missed call in the day.
               //Modified the behaviour where missedcall count is incremented only when the corresponding timestamp is less than the first timestamp irrespective of
               //whether the first DateStamp is less than the latest DateStamp.
               ETG_TRACE_USR1(( " PBDL.MissCallInd- Date/Time stamps are VALID " ));
               if( ((deviceObject->m_sFirstMissedCallDate == l_it->dateStamp) && (deviceObject->m_sFirstMissedCallTime < l_it->timeStamp) )
                     ||  (deviceObject->m_sFirstMissedCallDate < l_it->dateStamp)  ) //this(FirstMissedCallDate < LatestDate) happens during first Missed Call after DateChange
               {
                  ETG_TRACE_USR1(( " -PBDL.MissCallInd- Incrementing the MissedcallCount " ));
                  l_MissedcallCount++;
                  if(l_it == list.begin())
                  {
                     vUpdateDateAndTimeDetails(deviceObject,l_it->dateStamp,l_it->timeStamp,iDownloadType,FALSE);
                  }
               }
               ETG_TRACE_USR4(( " -PBDL.MissCallInd- l_MissedcallCount:%d", l_MissedcallCount));
               ETG_TRACE_USR1(( " -PBDL.MissCallInd- dateStamp= '%s' ", l_it->dateStamp.toUtf8().constData() ));
               ETG_TRACE_USR1(( " -PBDL.MissCallInd- timeStamp= '%s' ", l_it->timeStamp.toUtf8().constData() ));
            }
         }
      }
   }

   if (deviceObject->m_sFirstMissedCallTime == QString(EMPTY_STRING))
   {
      //No missed call Present while connecting the device. so storing the latest time and date of
      //any calltype(either incoming or outgoing)
      ETG_TRACE_USR4(( " -PBDL.MissCallInd- No Missed call present during device connection" ));
      vUpdateDateAndTimeDetails(deviceObject,l_sLatestDate,l_sLatestTime,iDownloadType,TRUE);
   }

   if (l_bMissedCallFound)
   {
      ETG_TRACE_USR4(( " vSetMissedcallCount called " ));
      deviceObject->vSetMissedcallCount(l_MissedcallCount);
      fc_phonebook_tclService_PhoneBook::pGetInstance()->vMissedCallIndicatorUpdate(l_MissedcallCount, deviceObject);
      fc_phonebook_tclService_PhoneBook::pGetInstance()->vUpdatePhonebookMissedCallIndicatorList(deviceObject->getDeviceHandle());
   }

   ETG_TRACE_USR4(( " vCheckMissedCallPresence EXIT " ));
}

//FIX CMG3G-8367 IS2424_DownloadOnOff behaviour@FC_Phonebook
// The user shall have possibility to switch the phonebook and call history list download On or Off individual for each paired phone device.
/*******************************************************************************
 *
 * FUNCTION: FC_PhoneBook_CallHistory_List::bClearCHRecords
 *
 * DESCRIPTION: DESCRIPTION: This function deletes the call history contents from database, frees related RAM content and triggers list change update.
 *
 *
 * PARAMETER: [IN] - u8ConnectionID - Connection ID of the device connected
 *            [OUT]
 * RETURNVALUE: - bRetValCH - Boolean return value of the deleting call history records in database
 *
 *******************************************************************************/
bool FC_PhoneBook_CallHistory_List::bClearCHRecords(quint8 u8ConnectionID)
{
   ETG_TRACE_USR4(("bClearCHRecords entered"));
   bool bRetValCH = FALSE;
   if (m_poSqlite)
   {
      //Delete Call history
      bRetValCH = m_poSqlite->DeleteCallHistoryRecords(u8ConnectionID);
      if (TRUE == bRetValCH)
      {
         /*FIX PSARCCB-6489 ListChange not update for call list if PBAP request in phone was declined before
         When PBAP is denied, the list handle is deleted from the hash list in addition to clearing the list.
          This is to be avoided. Deletion from hash is expected only on device disconnection.*/
         ETG_TRACE_USR4(("CH records deleted successfully"));
         RefreshCallHistoryOnCallHistoryUpdate(u8ConnectionID, DownloadCombinedCallHistorySimLocal);
         //End of fix PSARCCB-6489
      }
      else
      {
         ETG_TRACE_ERR(("CH records deletion FAILED"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("m_poSqlite is NULL. Hence returning"));
   }
   return bRetValCH;
}
//End of fix FIX CMG3G-8367

/**********************************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::RefreshCallHistoryOnChangeInContact
*
* DESCRIPTION: Call history list will be updated wrt the contact added/edited/deleted.
*
* PARAMETER: [IN]   - Details of the contact edited
*            [OUT]  -
* RETURNVALUE:
*
**********************************************************************************************/
void FC_PhoneBook_CallHistory_List::RefreshCallHistoryOnChangeInContact(
   FC_PhoneBook_Contact_Detail& PBook_Contact)
{
   ETG_TRACE_USR4(("RefreshCallHistoryOnChangeInContact() entered."));

   /*FIX GMMY15-11882 SHUTDOWN_QUEUE_FULL - Reducing execution time by removing redundant execution of for loops for separate call history lists.
   Single execution for type FC_PhoneBook_SQLite::callhisttype_combined is sufficient instead of for each call history type.

   Database update is done per record change in CH list, but List change update is sent only once.
   Maximum number of CH records is 50. Hence list change for content refreshed is sent instead of multiple updates for individual change.
   This will drastically reduce the number List change updates avoiding queue full condition.

   Also changing multiple functions for add,delete and edit on a contact to a single function which takes care of all changes.

   Added function bCHPhoneNumberAvailableInGivenPBContact to check if call history phone number is available in Phone contact.
   It also adds number type to the call history record if the corresponding phone number is found.*/

   QList<FC_PhoneBook_Callhistory_Detail> CallHistoryDetailList;

   //To check if PB DB update and List change update is required or not
   tBool blDBUpdateRequired = FALSE;
   tBool blListUpdateRequired = FALSE;

   //Old contact handle is saved to update the corresponding record in CH list with new one
   quint32 u32OldContactHandle= 0;
   if(m_poSqlite != NULL)
   {
      for(quint8 u8Index= 1; u8Index <= FC_PB_TOTAL_NUMBER_OF_CH; u8Index++)
      {
         CallHistoryDetailList.clear();
         CallHistoryDetailList = m_poSqlite->GetAllCallHistoryRecords(u8Index);

         quint16 u16ListSize= (quint16)CallHistoryDetailList.size();
         ETG_TRACE_USR4(("u16ListSize - %d", u16ListSize));
         FC_PhoneBook_Callhistory_Detail *pCallHistoryDetail= NULL;

         for(quint16 u16ListIndex = 0; u16ListIndex < u16ListSize; u16ListIndex++)
         {
            pCallHistoryDetail= &CallHistoryDetailList[u16ListIndex];

            //Checking whether PBook_Contact is available in pCallHistoryDetail
            if(TRUE == bCHPhoneNumberAvailableInGivenPBContact(*pCallHistoryDetail,PBook_Contact))
            {
               //Current CH Phone number available in given PB contact
               if( pCallHistoryDetail->firstName.compare(PBook_Contact.firstName) || pCallHistoryDetail->lastName.compare(PBook_Contact.lastName) )
               {
                  //Name has been edited by user or CH phone number added to existing contact in VPB or CH phone number added as new contact to VPB.
                  //Hence CH record is to be updated with new details.
                  pCallHistoryDetail->firstName= PBook_Contact.firstName;
                  pCallHistoryDetail->lastName= PBook_Contact.lastName;
                  //Update contact handle
                  u32OldContactHandle = pCallHistoryDetail->contactHandle;
                  pCallHistoryDetail->contactHandle= PBook_Contact.contactHandle;

                  ETG_TRACE_USR2((" FC_PhoneBook_CallHistory_List: New Call History Detail - "
                     "FirstName= %s", pCallHistoryDetail->firstName.toLatin1().constData()));
                  ETG_TRACE_USR2((" FC_PhoneBook_CallHistory_List: New Call History Detail - "
                     "LastName=  %s",pCallHistoryDetail->lastName.toLatin1().constData()));
                  ETG_TRACE_USR2((" FC_PhoneBook_CallHistory_List: New Call History Detail - "
                     "PhoneNumber=  %s",pCallHistoryDetail->phoneNumber.toLatin1().constData()));
                  ETG_TRACE_USR2((" FC_PhoneBook_CallHistory_List: New Call History Detail  - "
                     "NumberType=  %d",pCallHistoryDetail->numberType));
                  blDBUpdateRequired = TRUE;
               }
            }
            else
            {
               //Current CH Phone number not available in given PB contact
               if(pCallHistoryDetail->contactHandle == PBook_Contact.contactHandle)
               {
                  quint8 u8DeviceHandle = 0;

                  //Phone number has been edited by user or contact is deleted by user.
                  //Current phone number in CH does not exist anymore in PB. Hence clear name, contact handle and number type.
                  ETG_TRACE_USR4(("Clearing CH record details for phoneNumber: %s", pCallHistoryDetail->phoneNumber.toLatin1().constData()));
                  pCallHistoryDetail->firstName = "";   //.clear();
                  pCallHistoryDetail->lastName  = "";   //.clear();
                  //Update contact handle
                  u32OldContactHandle = pCallHistoryDetail->contactHandle;
                  pCallHistoryDetail->contactHandle= 0;
                  pCallHistoryDetail->numberType= (quint8)enNumType_Unknown;

                  //Fix for NCG3D-16229
                  //To get the device handle
                  FC_Device_Details
                        * pDeviceObject =
                              FC_Device_Details::getDeviceObject((INDEX)(u8Index - 1));
                  if (pDeviceObject)
                  {
                     u8DeviceHandle = pDeviceObject->getDeviceHandle();

                     FC_PhoneBook_Search_Result contactSearchResult;
                     tU8 u8NameEdited_dummy = 0;
                     if (GetMatchingContactForCallHistoryEntry(*pCallHistoryDetail, u8DeviceHandle, contactSearchResult, u8NameEdited_dummy))
                     {
                        pCallHistoryDetail->firstName
                              = contactSearchResult.firstname;
                        pCallHistoryDetail->lastName = contactSearchResult.lastname;
                        pCallHistoryDetail->contactHandle
                              = contactSearchResult.contactHandle;
                     }
                  }
                  //End of fix NCG3D-16229
                  blDBUpdateRequired = TRUE;
               }
            }

            if(TRUE == blDBUpdateRequired)
            {
               ETG_TRACE_USR2(("FC_PhoneBook_CallHistory_List: Updating DB for ContactHandle = %u",u32OldContactHandle));
               //GMMY15-1583 Use update instead of insert delete combination.
               //Update CH table wrt the record
               if(!(m_poSqlite->UpdateCallHistoryRecord(*pCallHistoryDetail,u32OldContactHandle,u8Index)))
               {
                  ETG_TRACE_ERR(("FC_PhoneBook_CallHistory_List:: Error - Failed "
                     "to Update call history record"));
               }
               u32OldContactHandle = 0;
               //Updating inner loop variable blDBUpdateRequired to false as DB update is done and setting that list change update is required.
               blDBUpdateRequired = FALSE;
               blListUpdateRequired = TRUE;
            }
         }

         if(TRUE == blListUpdateRequired)
         {
            ETG_TRACE_USR4(("Updating Call History List"));

            //Send list change for Refresh CH
            //Download type sent as DownloadCombinedCallHistorySimLocal (combined call history) so that list change will be sent for list handles of all call history types
            RefreshCallHistoryOnCallHistoryUpdate(u8Index, DownloadCombinedCallHistorySimLocal);
            blListUpdateRequired = FALSE;
         }
      }
   }
}


/**********************************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::bCHPhoneNumberAvailableInGivenPBContact
*
* DESCRIPTION: Searches if the existing call history phone number matches any phone number entry
*              in the given contact. If found, numberType of CH record is also updated.
*
* PARAMETER: [IN]   - Call history record's phone number to be searched, Complete Phone contact
*            [OUT]  -
*
* RETURNVALUE: SearchResult: TRUE or FALSE denoting availability
*
**********************************************************************************************/
tBool FC_PhoneBook_CallHistory_List::bCHPhoneNumberAvailableInGivenPBContact(FC_PhoneBook_Callhistory_Detail& CallHistoryDetail, FC_PhoneBook_Contact_Detail& PBook_Contact) //CMG3G-14313-FC_PhoneBook_Contact_Detail-Pass by Reference-Coverity Fix
{
   ETG_TRACE_USR4(("bCHPhoneNumberAvailableInGivenPBContact entered."));

   tBool blContactFound = TRUE;

   if (TRUE == CallHistoryDetail.phoneNumber.isEmpty()) // FIX NCG3D-19789
   {
      //If the call history phone number is empty then no need to compare further for finding a match in the contact's phone numbers.
      ETG_TRACE_USR4((" Call history entry is from Private number!!!"));
      blContactFound = FALSE;
   }
   else
   {
      if (!PBook_Contact.PhoneNum1.compare(CallHistoryDetail.phoneNumber))
      {
         CallHistoryDetail.numberType = PBook_Contact.NumberType1;
      }
      else if (!PBook_Contact.PhoneNum2.compare(CallHistoryDetail.phoneNumber))
      {
         CallHistoryDetail.numberType = PBook_Contact.NumberType2;
      }
      else if (!PBook_Contact.PhoneNum3.compare(CallHistoryDetail.phoneNumber))
      {
         CallHistoryDetail.numberType = PBook_Contact.NumberType3;
      }
      else if (!PBook_Contact.PhoneNum4.compare(CallHistoryDetail.phoneNumber))
      {
         CallHistoryDetail.numberType = PBook_Contact.NumberType4;
      }
      else if (!PBook_Contact.PhoneNum5.compare(CallHistoryDetail.phoneNumber))
      {
         CallHistoryDetail.numberType = PBook_Contact.NumberType5;
      }
      else if (!PBook_Contact.PhoneNum6.compare(CallHistoryDetail.phoneNumber))
      {
         CallHistoryDetail.numberType = PBook_Contact.NumberType6;
      }
      else if (!PBook_Contact.PhoneNum7.compare(CallHistoryDetail.phoneNumber))
      {
         CallHistoryDetail.numberType = PBook_Contact.NumberType7;
      }
      else if (!PBook_Contact.PhoneNum8.compare(CallHistoryDetail.phoneNumber))
      {
         CallHistoryDetail.numberType = PBook_Contact.NumberType8;
      }
      else
      {
         blContactFound = FALSE;
      }
   }

   ETG_TRACE_USR2(("blContactFound - %d", blContactFound));
   return blContactFound;
}
//End of fix GMMY15-11882


//FIX GMMY16-782 GMMY16-1106 After contacts downloaded, recent calls list still only shows phone numbers (no names).
//During second download check if call history contains any record whose number is available in PB. If yes update the same with PB contact's details
/**********************************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::SyncCallHistoryList
*
* DESCRIPTION: Searches if the existing call history matches any contact entry and updates the call
* history entry
*
* PARAMETER: [IN]   - Device handle
*            [OUT]  -
* RETURNVALUE:
*
**********************************************************************************************/
tVoid FC_PhoneBook_CallHistory_List::SyncCallHistoryList(tU8 deviceHandle)
{
   ETG_TRACE_USR4(("SyncCallHistoryList() entered."));

   FC_Device_Details *deviceObject=
      FC_Device_Details::deviceHandleToObjectInstance(deviceHandle);

   if((deviceObject) && (m_poSqlite) && !deviceObject->blIsItFirstTimeDownload())//FIX GMMY16-6009 Nullpointerexception - Adding null checks to pCallHistory and m_poSqlite
   {
      QTime DebugTime1;
      DebugTime1.start();

      //Get all CH records
      QList<FC_PhoneBook_Callhistory_Detail> callHistory = m_poSqlite->GetAllCallHistoryRecords(deviceObject->getConnectionID());//Fix for CMG3GB-621. We want all the records from the CH table.

      tU8 u8NameEdited = 0;
      FC_PhoneBook_Search_Result PhoneContact;
      FC_PhoneBook_Callhistory_Detail *pCallHistory = NULL;

      for(qint16 CHListIndex= 0; CHListIndex < callHistory.size(); CHListIndex++)
      {
         pCallHistory = &callHistory[CHListIndex];

         /* FIX GMMY16-14261 Recent Call List entries often appears as just Numbers, without resolved Contact Name, although contact exists with names */

         // CMG3G-5891: Prio1 LINT fix (Info 774) - Removing the NULL check for pCallHistory since it is always not NULL at this part of the code.

         //Check if same entry is found in PB
         if( GetMatchingContactForCallHistoryEntry( *pCallHistory, (quint8)deviceHandle, PhoneContact, u8NameEdited ))//FIX GMMY16-6009 Nullpointerexception - Adding null checks to pCallHistory and m_poSqlite
         {
            //If same entry found in PB, sync/update CH record
            pCallHistory->contactHandle= PhoneContact.contactHandle;
            pCallHistory->numberType= PhoneContact.contactdetailtype;
            ETG_TRACE_USR4(("pCallHistory->numberType = %d",pCallHistory->numberType));
         }
         else
         {
            //If not, clear the name associated with it
            ETG_TRACE_USR4(("Clearing name as related call history entry not found in contacts."));
            ETG_TRACE_USR4(("firstName = %s",pCallHistory->firstName.toLatin1().constData()));
            ETG_TRACE_USR4(("lastName = %s",pCallHistory->lastName.toLatin1().constData()));
            pCallHistory->contactHandle= 0;
            pCallHistory->firstName = "";   //.clear();
            pCallHistory->lastName =  "";   //.clear();
            u8NameEdited = 1;
         }
         //End of fix
      }

      ETG_TRACE_USR2(( " -PBDL.Perf- 'SyncCallHistoryList' [Get matching Records] needed %d ms ", DebugTime1.elapsed() ));   // NEW

      ETG_TRACE_USR4(("u8NameEdited: %d",u8NameEdited));
      if(1 == u8NameEdited)//Empty call histroy name replaced with name from contact
      {
         ETG_TRACE_USR4(("Syncing Call History"));

         QTime DebugTime1;
         DebugTime1.start();

         //Update PB DB
         DeleteCallHistoryRecordsGivenConnectionID(deviceObject->getConnectionID());
         m_poSqlite->InsertCallHistoryRecords( callHistory, deviceObject->getConnectionID() );
         ETG_TRACE_USR2(( " -PBDL.Perf- 'SyncCallHistoryList' [Delete and Insert in DB] needed %d ms ", DebugTime1.elapsed() ));   // NEW

         //Send list change for Refresh CH
         RefreshCallHistoryOnCallHistoryUpdate(deviceObject->getConnectionID(), DownloadCombinedCallHistorySimLocal);
      }
   }

   ETG_TRACE_USR4(("SyncCallHistoryList() exit."));
}
//End of fix


//FIX SUZUKI-1113 API for numeric value to indicate number of calls against a CallHistory item in Phonebook
/*******************************************************************************
*
* FUNCTION:  FC_PhoneBook_CallHistory_List::RequestSliceCallHistoryListExtended
*
* DESCRIPTION:
*              Creates a subset of the list pointed by listhandle
*
*
* PARAMETER: [IN] methodStart :- Input arguments like listhandles are in this class
*            [OUT] methodResult :- The acutal sub list is passed in this class
* RETURNVALUE: bool - indicating success or failure of this function
*
*******************************************************************************/
bool FC_PhoneBook_CallHistory_List::RequestSliceCallHistoryListExtended(
   const most_phonbkfi_tclMsgRequestSliceCallHistoryListExtendedMethodStart* methodStart,
   most_phonbkfi_tclMsgRequestSliceCallHistoryListExtendedMethodResult* methodResult,
   tU32 u32RegisterID)//FIX NCG3D-55494 RegisterID is used to identify the requested client
  {

      ETG_TRACE_USR4(("RequestSliceCallHistoryListExtended() entered."));
      tU16 l_u16CorrectedListHandle = 0;
      if(m_multiHashCHClientRequests.contains(methodStart->u16ListHandle,u32RegisterID))
      {

         if (false == m_hashCHList.contains(methodStart->u16ListHandle))
         {
            ETG_TRACE_USR4(("Wrong list handle sent by client, sending combined call history data"));
            return false;

            //Fix for NCG3D-55491
            /*QHash<quint16, FC_CallHistory_List_Data*>::iterator l_it;
            for (l_it = m_hashCHList.begin(); l_it != m_hashCHList.end(); l_it++)
            {
               if (l_it.value()->callHistoryTpye == FC_PhoneBook_SQLite::callhisttype_combined)
               {
                     l_u16CorrectedListHandle = l_it.key();
                     break;
               }
            }
            */
         }
         else
         {
            ETG_TRACE_USR4(("Correct list handle sent by client"));
            l_u16CorrectedListHandle = methodStart->u16ListHandle;
         }
      }
      else
      {
         ETG_TRACE_ERR((" Client register ID is not availble "));
         return false;
      }
      FC_CallHistory_List_Data* data = m_hashCHList.value(
         l_u16CorrectedListHandle, NULLPTR);

      if (!data) // BUGZID: 208352
      {

         ETG_TRACE_ERR(
            (" FC_PhoneBook_CallHistory_List :- WARNING - Null Data for request List Handle "));
         return false;
      }

      unsigned listLen = (unsigned)data->callhistoryList.size();

      if ((methodStart->u16WindowSize < 1) ||
         (methodStart->u32WindowStart >= listLen)) {

            ETG_TRACE_USR4(
               ("WARNING - negative number requested in windowStart or WindowSize, or zero windowSize, returning empty list"));
            return false;
      }

      unsigned end = methodStart->u32WindowStart + methodStart->u16WindowSize;

      if (end > listLen)
         end = listLen;

      ETG_TRACE_USR4(
         ("TOTAL Items = %u, START = %d, END = %u, SIZE = %u", listLen, methodStart->u32WindowStart, end
         - 1, methodStart->u16WindowSize));

      //Fix CMG3G-12880
   FC_Device_Details* pDeviceObject =
         FC_Device_Details::getDeviceObject((INDEX) (data->ConnectionID - 1));
   if (pDeviceObject)
   {
      //If device is not locked, continue as normal
      if (FALSE == pDeviceObject->bIsDeviceProbablyLocked())
      {
         for (unsigned counter = methodStart->u32WindowStart; counter < end; ++counter)
         {
            most_fi_tcl_PhonBkCallHistoryListSliceResultExtenedItem
                  sliceCallHistoryitemExtended;

            Convert_Extended(&data->callhistoryList.at((int) counter), &sliceCallHistoryitemExtended);

            methodResult->oCallHistoryListSliceResultExtended.oItems.push_back(sliceCallHistoryitemExtended);
         }
      }
      else
      {
         //Fix : CMG3G-12880 Mock call history list as empty since device is locked. so returning true
         ETG_TRACE_USR4(("Device is locked"));
      }
   }
   else
   {
      ETG_TRACE_USR4((" Device object not found for given device handle"));
   }
      return true;
}


/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_CallHistory_List::Convert_Extended()
*
* DESCRIPTION:
*              Converts the Qt format structure into the MOST format structure
*
*
* PARAMETER: [IN] contact :- The Qt format contact info
*            [OUT] item :- MOST format contact info
* RETURNVALUE: None.
*
*******************************************************************************/
void FC_PhoneBook_CallHistory_List::Convert_Extended(
   const FC_PhoneBook_Callhistory_Detail* callHistorydata,
   most_fi_tcl_PhonBkCallHistoryListSliceResultExtenedItem* CallHistoryitemExtended) {
      ETG_TRACE_USR4(("Convert_Extended() entered."));
      QByteArray byteArray;
      QByteArray tmpByteArray;

      CallHistoryitemExtended->u32ContactHandle = callHistorydata->contactHandle;

      // First Name
      byteArray = callHistorydata->firstName.toUtf8();
      CallHistoryitemExtended->sFirstName.bSet(byteArray.constData());
      ETG_TRACE_USR4((" FIRST NAME = %s", byteArray.constData()));

      // Last Name
      byteArray = callHistorydata->lastName.toUtf8();
      CallHistoryitemExtended->sLastName.bSet(byteArray.constData());

      ETG_TRACE_USR4((" LAST NAME = %s", byteArray.constData()));

      // Call Type
      CallHistoryitemExtended->e8CallHistoryType.enType
         = (most_fi_tcl_e8_PhonBkCallHistoryType::tenType) callHistorydata->callType;

      ETG_TRACE_USR4(
         (" CallHistory Type = %d", (quint8) CallHistoryitemExtended->e8CallHistoryType.enType));

      //Fix NCG3D-27060
      // Number Type
      CallHistoryitemExtended->e8CallPhoneType.enType
         = (most_fi_tcl_e8_PhonBkNumberType::tenType) callHistorydata->numberType;

      //End of fix NCG3D-27060
      ETG_TRACE_USR4(
         (" Number Type = %d", (quint8) CallHistoryitemExtended->e8CallPhoneType.enType));

      // Phone Number
#ifdef PB_PHONENUMBERS_IN_PRETTY_PRINT
      byteArray = callHistorydata->PPphoneNumber.toUtf8();
#else
      byteArray = callHistorydata->phoneNumber.toUtf8();
#endif
      CallHistoryitemExtended->sPhoneNumber.bSet(byteArray.constData());

      ETG_TRACE_USR4((" PhoneNumber = %s", byteArray.constData()));

      // Year
      byteArray = callHistorydata->dateStamp.toUtf8();
      tmpByteArray = byteArray.mid(0, 4);
      CallHistoryitemExtended->oCallDateTime.oCallDate.sCldrYear.bSet(
         tmpByteArray.constData());

      ETG_TRACE_USR2((" sCldr Year = %s", tmpByteArray.constData()));

      // month
      tmpByteArray = byteArray.mid(4, 2);
      CallHistoryitemExtended->oCallDateTime.oCallDate.sCldrMonth.bSet(
         tmpByteArray.constData());

      ETG_TRACE_USR2((" sCldr Month = %s", tmpByteArray.constData()));

      // Day
      tmpByteArray = byteArray.mid(6, 2);
      CallHistoryitemExtended->oCallDateTime.oCallDate.sCldrDay.bSet(
         tmpByteArray.constData());

      ETG_TRACE_USR2((" sCldr Day = %s", tmpByteArray.constData()));

      // Hours
      byteArray = callHistorydata->timeStamp.toUtf8();
      tmpByteArray = byteArray.mid(0, 2);
      CallHistoryitemExtended->oCallDateTime.oCallTime.sHours.bSet(
         tmpByteArray.constData());

      ETG_TRACE_USR2((" s Hours = %s", tmpByteArray.constData()));

      // Minutes
      tmpByteArray = byteArray.mid(2, 2);
      CallHistoryitemExtended->oCallDateTime.oCallTime.sMinutes.bSet(
         tmpByteArray.constData());

      ETG_TRACE_USR2((" s Minutes = %s", tmpByteArray.constData()));

      // Seconds
      tmpByteArray = byteArray.mid(4, 2);
      CallHistoryitemExtended->oCallDateTime.oCallTime.sSeconds.bSet(
         tmpByteArray.constData());

      ETG_TRACE_USR2((" s Seconds = %s", tmpByteArray.constData()));

      //CallCount
      CallHistoryitemExtended->u8CallCount = callHistorydata->callCount;
      ETG_TRACE_USR2((" u8CallCount = %d", (quint8) CallHistoryitemExtended->u8CallCount));
}
//End of fix

//FIX GMMY15-1578 Removing redundant objects which are created in RAM for different clients & sorted lists.
/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_CallHistory_List::vClearCHListAndHashListEntriesForListHandle()
*
* DESCRIPTION: Deletes entries from m_hashPbList and m_multiHashPBClientRequests
*              for a particular list handle.
*
* PARAMETER: [IN] u16ListHandle :- List handle whose corresponding entries are to be deleted
*                 from hash tables.
*            [OUT] None.
* RETURNVALUE: None.
*
*******************************************************************************/
void FC_PhoneBook_CallHistory_List::vClearCHListAndHashListEntriesForListHandle(tU16 u16ListHandle)
{
   ETG_TRACE_USR4(( " vClearCHListAndHashListEntriesForListHandle entered with list handle: %d", u16ListHandle));

   FC_CallHistory_List_Data* toDelete;
   toDelete = m_hashCHList.take(u16ListHandle);
   if(toDelete)
   {
      toDelete->callhistoryList.clear();
   }
   delete toDelete;
   m_hashCHList.squeeze();

   //Delete entries of all requests from this device which is stored in Multihash as it is no longer required
   m_multiHashCHClientRequests.remove(u16ListHandle);
   m_multiHashCHClientRequests.squeeze();
}
//End of fix GMMY15-1578


//FIX SUZUKI-18215 strange "?" showed in Phone screen for a long Chinese contact
//The replacement character is found if any and then string is truncated at the replacement character, to make it a valid Utf8 string.
/*******************************************************************************
*
* FUNCTION: FC_PhoneBook_CallHistory_List::vUtf8Validate()
*
* DESCRIPTION: Validates UTF-8 encoded text and cuts off string
*              after first encounter of invalid character
*
*
* PARAMETER: [IN] sCandidateToValidate :- QString to be validated.
*
* RETURNVALUE: None.
*
*******************************************************************************/
void FC_PhoneBook_CallHistory_List::vUtf8Validate(QString &sCandidateToValidate)
{
   ETG_TRACE_USR4((" vUtf8Validate entered with sCandidateToValidate = %s", sCandidateToValidate.toUtf8().constData() ));

   //We truncate at the 25th character as per GIS 361 for first name and last name.
   //Utf8 permits two,three and four byte representation of characters
   //Hence there can be an incomplete last character at the end of the string.
   //Each byte of the incomplete representation would be replaced by QT with a 'replacement character' (0xEF 0xBF 0xBD).
   //They in turn takes 3 bytes.

   //The truncation will happen on more than 25 characters. Hence the check for faulty character need not be done for shorter strings.
   if(sCandidateToValidate.toUtf8().size() >= 25)
   {
      QByteArray ba = sCandidateToValidate.toUtf8();

      //Cut off the faulty character if any
      //u32BufferLength for UTF8_vCutOffFaultyCharacter should be length of string in bytes plus 1.
      UTF8_vCutOffFaultyCharacter(ba.data(),(unsigned)ba.size() + 1);

      sCandidateToValidate = QString::fromUtf8(ba.data());

      ETG_TRACE_USR4(( " Validated String = %s", sCandidateToValidate.toUtf8().constData() ));
   }
}
//End of fix SUZUKI-18215
