/*******************************************************************************
 *
 * FILE:			FC_Phone_WorkQueue_Scheduler.cpp
 *
 * SW-COMPONENT:	FC_Phone application
 *
 * PROJECT:			BOSCH-GM-NEXTGEN
 *
 * DESCRIPTION:		Contains functions to 1)Lock and unlock mutex 2)Initialize and destroy mutex 3) Work Queue Scheduler
 *
 * AUTHOR:			Priya Sekhar
 *
 * COPYRIGHT:		(c) 2013 Robert Bosch Coimbatore
 *
 *******************************************************************************/

#ifndef __PLACEMENT_NEW_INLINE
#define __PLACEMENT_NEW_INLINE 1
#endif

#include "../FC_Phone_main.h"
#include "FC_Phone_WorkQueueScheduler.h"
#include "../FC_Phone_service_Telephone.h"
#include "unistd.h"
#include "signal.h"
#include "FC_Phone_WorkQueue.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONE_APPLICATION
#include "trcGenProj/Header/FC_Phone_WorkQueueScheduler.cpp.trc.h"
#endif

fc_phone_tclPhoneMutex* fc_phone_tclPhoneMutex::m_poPhoneMutexObject = NULLPTR;
pthread_t fc_phone_tclPhoneMutex::m_WorkQueueThreadID = 0;

/*******************************************************************************
 * FUNCTION:  	unlockPhoneDbusMutex
 * DESCRIPTION:  function is used to unlock mutex on timer expiry.
 *
 * PARAMETER:
 *
 * RETURNVALUE: none
 ********************************************************************************/
void unlockPhoneDbusMutex(int signalNumber)
{
   ETG_TRACE_USR4(("unlockPhoneDbusMutex : Called"));
   (void) signalNumber;

   ETG_TRACE_USR4(("Timer is expired"));

   //Fix for task CMG3G-5839: Method Result/Method Error not posted from FC_Phone
   /* Post MethodError for the unresponsive request from the CCA Context (in a loopback),
    * Mutex lock is released after posting the MethodError */
   fc_phone_tclWorkQueue::getWorkQueueInstance()->vSendToCCAThread(NULLPTR, TRUE);
}

/*******************************************************************************
 * FUNCTION:  	PhoneMutexInit
 * DESCRIPTION:  function is used to create MessageMutex object.
 *
 * PARAMETER:
 *
 * RETURNVALUE: object address
 ********************************************************************************/
fc_phone_tclPhoneMutex* fc_phone_tclPhoneMutex::PhoneMutexInit()
{
   ETG_TRACE_USR4(("PhoneMutexInit : Called"));
   if (!m_poPhoneMutexObject)
   {
      m_poPhoneMutexObject = new fc_phone_tclPhoneMutex;
   }
   return m_poPhoneMutexObject;
}

/*******************************************************************************
 * FUNCTION:  	PhoneMutexDestroy
 * DESCRIPTION:  function is used to destroy MessageMutex object.
 *
 * PARAMETER:
 *
 * RETURNVALUE: none
 ********************************************************************************/
void fc_phone_tclPhoneMutex::PhoneMutexDestroy()
{
   ETG_TRACE_USR4(("PhoneMutexDestroy : Called"));

   if(m_poPhoneMutexObject)
   {
      delete m_poPhoneMutexObject;
      m_poPhoneMutexObject = NULLPTR;
   }

   m_WorkQueueThreadID = 0;
}

/*******************************************************************************
 * FUNCTION:  	fc_phone_tclPhoneMutex
 * DESCRIPTION:  constructor.
 *
 * PARAMETER:
 *
 * RETURNVALUE: none
 ********************************************************************************/
fc_phone_tclPhoneMutex::fc_phone_tclPhoneMutex()
{
   ETG_TRACE_USR4(("fc_phone_tclPhoneMutex : Called"));
   pthread_mutex_init(&m_PhoneDbusMutex, NULLPTR);

   pthread_create(&m_WorkQueueThreadID, NULLPTR, fc_phone_tclPhoneMutex::WorkQueueScheduler,
         NULLPTR);
}

/*******************************************************************************
 * FUNCTION:  	~fc_phone_tclPhoneMutex
 * DESCRIPTION:  destructor.
 *
 * PARAMETER:
 *
 * RETURNVALUE: none
 ********************************************************************************/
fc_phone_tclPhoneMutex::~fc_phone_tclPhoneMutex()
{
   ETG_TRACE_USR4(("~fc_phone_tclPhoneMutex : Called"));
   pthread_mutex_destroy(&m_PhoneDbusMutex);
}

/*******************************************************************************
 * FUNCTION:  	PhoneMutexTrylock
 * DESCRIPTION:  function to lock the mutex.
 *
 * PARAMETER:
 *
 * RETURNVALUE: success or failure
 ********************************************************************************/
int fc_phone_tclPhoneMutex::PhoneMutexTrylock(struct Job *f_poJob, tU8 u8TimerValue)
{
   ETG_TRACE_USR4(("PhoneMutexTrylock : Called"));
   int ret = -1;
   ret = pthread_mutex_trylock(&m_poPhoneMutexObject->m_PhoneDbusMutex);
   if (ret != 0)
   {
      ETG_TRACE_ERR(("Unable to acquire lock, Inserting the job in WorkQueue"));
      // Insert the job in the queue
      if (f_poJob)
      {
         fc_phone_tclWorkQueue::getWorkQueueInstance()->insertNewJob(f_poJob);
      }
   }
   else
   {
      //Start the timer
      ETG_TRACE_USR4(("Timer started for value: %d", u8TimerValue));

      //Fix for task CMG3G-5839: Method Result/Method Error not posted from FC_Phone
      //Keep track of current FunctionID holding the lock
      fc_phone_tclWorkQueue::getWorkQueueInstance()->vSetCurrentRequestFID(f_poJob->u16FunctionID);
      alarm(u8TimerValue);
      signal(SIGALRM, unlockPhoneDbusMutex);
   }

   return ret;
}

/*******************************************************************************
 * FUNCTION:  	PhoneMutexUnlock
 * DESCRIPTION:  function to unlock the mutex.
 *
 * PARAMETER:
 *
 * RETURNVALUE: success or failure
 ********************************************************************************/
int fc_phone_tclPhoneMutex::PhoneMutexUnlock()
{
   ETG_TRACE_USR4(("PhoneMutexUnlock : Called"));
   // Stop the timer
   alarm(0);

   if (m_WorkQueueThreadID == pthread_self())
   {
      pthread_mutex_unlock(&m_poPhoneMutexObject->m_PhoneDbusMutex);
   }

   pthread_kill(m_WorkQueueThreadID, SIGUSR1);
   ETG_TRACE_USR4(("Signal sent"));
   return 0;
}

/*******************************************************************************
 * FUNCTION:  	dummy
 * DESCRIPTION:  dummy function.
 *
 * PARAMETER:
 *
 * RETURNVALUE: none
 ********************************************************************************/
void dummy(int signalNumber)
{
   (void) signalNumber;
   ETG_TRACE_USR4(("dummy : Called"));
}

/*******************************************************************************
 * FUNCTION:  	WorkQueueScheduler
 * DESCRIPTION:  function run by workqueue thread.
 *
 * PARAMETER:
 *
 * RETURNVALUE: none
 ********************************************************************************/
void *fc_phone_tclPhoneMutex::WorkQueueScheduler(void *ptr)
{
   ETG_TRACE_USR4(("WorkQueueScheduler : Called"));
   (void) ptr;

   signal(SIGUSR1, dummy);

   while (true)
   {
      // Wait for the signal
      pause();

      pthread_mutex_unlock(&m_poPhoneMutexObject->m_PhoneDbusMutex);

      ETG_TRACE_USR4(("Got the signal"));

      // Fetch the next job
      fc_phone_tclWorkQueue::getWorkQueueInstance()->fetchNextJob(true);
   }
   return NULLPTR;
}
