/******************************************************************************
 *
 * FILE:          FC_Phone_AudioManager.cpp
 *
 * SW-COMPONENT:  FC_Phone application
 *
 * PROJECT:
 *
 * DESCRIPTION:   Abstracts Audio Manager Service provided by PhoneService class
 *
 * AUTHOR:        Guruprasad G R
 *
 * COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
 *
 *******************************************************************************/
#include "../FC_Phone_main.h"
#include "FC_Phone_AudioManager.h"
#include "../FC_Phone_clienthandler_BTSettings.h"
#include "FC_Phone_CallManager.h"

#include "stdio.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONE_APPLICATION
#include "trcGenProj/Header/FC_Phone_AudioManager.cpp.trc.h"
#endif

#define FC_PHONE_AV_METHOD_ERROR_MAX_RETRY_COUNTER 2
fc_phone_tclAudioManager* fc_phone_tclAudioManager::m_poInstance = NULLPTR;

/*******************************************************************************
 ** FUNCTION:   vAudioCallBackDummy(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   Internal method to handle a dummy callback.
 *  PARAMETERS:
 *     IN:   u16TimerId - Timer Id for the timer being expired.
 *
 *  RETURNVALUE: NONE
 ********************************************************************************/
void vAudioCallBackDummy(tU16 u16AudioReqID,tU16 u16AudioChannelId, tU16 u16Status)
{
   ETG_TRACE_USR4(("vAudioCallBackDummy: entered "));
   /* Print Error */
   //for removing warnings
   (tVoid)u16AudioReqID ;
   (tVoid)u16AudioChannelId ;
   (tVoid)u16Status ;
}

/*******************************************************************************
 ** FUNCTION:   fc_phone_tclAudioManager(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   Constructor
 *  PARAMETERS:
 *     IN:   rfpTelService - Reference to service which provides timer service
 *
 *  RETURNVALUE: None
 ********************************************************************************/
fc_phone_tclAudioManager::fc_phone_tclAudioManager()
{
   ETG_TRACE_USR4(("fc_phone_tclAudioManager::fc_phone_tclAudioManager: entered "));
   m_poInstance = this;

   for (tU16 u16Index = 0x00; u16Index < FC_PHONE_AUDIOCHANNEL_LIMIT; u16Index++)
   {
      m_rAudioCallBackMap[u16Index].u16AudioReqID     = FC_PHONE_AUDIOCHANNEL_REQUESTOR_ID_UNKNOWN ;
      m_rAudioCallBackMap[u16Index].u16AudioChannelId = FC_PHONE_AUDIOCHANNEL_ID_UNUSED;
      m_rAudioCallBackMap[u16Index].FpCallBack = vAudioCallBackDummy;
   }

   for (tU16 u16Index = 0x00; u16Index < FC_PHONE_NUM_OFCHANNELS; u16Index++)
   {
      tChannelAquAttr[u16Index].u16ChannelAquCounter     = 0 ;
      tChannelAquAttr[u16Index].u16ChannelAquState = FC_PHONE_CHANNELAQUISITION_STATE_IDLE;
      tChannelAquAttr[u16Index].bPendingReleaseRequest = FALSE;
      tChannelAquAttr[u16Index].bPendingGetRequest = FALSE;
   }

   m_bPriorityChannelFlag = FALSE ;

   //Fix for GMMY17-11124
   m_u8GetRequestCounter = 0;
}
/*******************************************************************************
 ** FUNCTION:   ~fc_phone_tclAudioManager(..)
 *******************************************************************************/
/* DESCRIPTION:
 *   Destructor
 *  PARAMETERS:
 *     IN:   None
 *
 *  RETURNVALUE: None
 ********************************************************************************/
/* Destructor */

fc_phone_tclAudioManager::~fc_phone_tclAudioManager()
{
   ETG_TRACE_USR4(("fc_phone_tclAudioManager::~fc_phone_tclAudioManager: entered "));
}

/*******************************************************************************
 *
 * FUNCTION: s32GetRequest
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tS32 fc_phone_tclAudioManager::s32GetRequest(tU16 u16AudioReqID,tU16 u16AudioChannelId,AudioCallBack fpCallBack)
{
   tS32 s32Error = FC_PHONE_AUDIOCHANNEL_OK;

   ETG_TRACE_USR4(("  s32GetRequest for u16AudioReqID : %d and u16AudioChannelId: %d in slot list",u16AudioReqID,u16AudioChannelId));

   //check ,Valid Channel Requested
   if((FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE != u16AudioChannelId)
         &&(FC_PHONE_AUDIOCHANNEL_PHONE != u16AudioChannelId)
         &&(FC_PHONE_AUDIOCHANNEL_ALERTTONE != u16AudioChannelId)
         && (FC_PHONE_AUDIOCHANNEL_SPEECH != u16AudioChannelId)
   )
   {
      ETG_TRACE_USR4(("  Invalid Get Channel Requested u16AudioChannelId :%u",u16AudioChannelId));
      return FC_PHONE_AUDIOCHANNEL_INVALID;
   }

   //remove Release/GET slot , if present for same requestor ID and Channel ID
   s32Error = s32RemoveGetRequest(u16AudioReqID, u16AudioChannelId);
   if(FC_PHONE_AUDIOCHANNEL_OK == s32Error)
   {
      ETG_TRACE_USR4(("  Removed GET/RELEASE request "));
   }

   //Get free slot to register
   tU16 u16SlotId = 0x00;
   s32Error = s32GetNextSlot(u16SlotId);
   ETG_TRACE_USR4((" fc_phone_tclAudioManager::s32GetRequest ->s32Error : %d ",s32Error));

   if (FC_PHONE_AUDIOCHANNEL_OK == s32Error)
   {
      /* Register the callback */
      m_rAudioCallBackMap[u16SlotId].u16AudioReqID = u16AudioReqID ;
      m_rAudioCallBackMap[u16SlotId].u16AudioChannelId = u16AudioChannelId;
      m_rAudioCallBackMap[u16SlotId].FpCallBack = fpCallBack;

      //update channel attributes
      vUpdateChannelAttributes(u16AudioChannelId, FC_PHONE_REQUESTTYPE_GET);

      //If Get for this channel First Time Called , Forward Request to Client handler
      //else send Ack to corresponding SM ,As channel is Aquired
      if(FC_PHONE_AUDIOMANAGER_FIRST_GETREQUEST == tChannelAquAttr[u16AudioChannelId].u16ChannelAquCounter)
      {
         //Fix for GMMY17-11124 and GMMY17-11964
         if(FC_PHONE_AUDIOCHANNEL_PHONE == u16AudioChannelId)
         {
            m_u8GetRequestCounter++;
            ETG_TRACE_USR4((" m_u8GetRequestCounter for LC_Phone =  %u ",m_u8GetRequestCounter));
         }

         if(FC_PHONE_CHANNELAQUISITION_STATE_IDLE == tChannelAquAttr[u16AudioChannelId].u16ChannelAquState )
         {
            //update channel State As Processing
            tChannelAquAttr[u16AudioChannelId].u16ChannelAquState = FC_PHONE_CHANNELAQUISITION_STATE_PROCESSING_GET ;
            vPrintChannelAttributes();

            if(u16AudioChannelId == FC_PHONE_AUDIOCHANNEL_ALERTTONE)
            {
               ETG_TRACE_USR4((" AlertTone not supported currently - so grant channel"));
               s32GetRequestAck(u16AudioChannelId, FC_PHONE_AUDIOCHANNEL_GRANTED);
            }
            else if(u16AudioChannelId == FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE)
            {
               ETG_TRACE_USR4((" Audio channel is Ringtone"));
               fc_phone_tclClientHandler_BTSettings *m_poBTClient = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings;
               tU8 u8DeviceAddrId = m_poBTClient->vGetRingToneDeviceAddrId();
               m_poBTClient->RequestAvailableRingtonesList((fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->u8GetDeviceHandleFromBtAddressId(u8DeviceAddrId));
            }
            else
            {
               ETG_TRACE_USR4((" Audio channel other that Ringtone"));
               bool bARLRootRequest = (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioRouting->vTriggerPhoneAudioAllocation(u16AudioChannelId);
               if(bARLRootRequest)
               {
                  ETG_TRACE_USR4((" Request root success: sending Ack for u16AudioChannelId :: %d ",u16AudioChannelId));
                  //s32GetRequestAck(u16AudioChannelId, FC_PHONE_AUDIOCHANNEL_GRANTED);
               }
               else
               {
                  ETG_TRACE_USR4((" Request root failed: for u16AudioChannelId :: %d ",u16AudioChannelId));
                  s32GetRequestAck(u16AudioChannelId, FC_PHONE_AUDIOCHANNEL_DENIED);
               }
            }
         }
         else
         {
            //if State is not idle ,it means Release for Same Channel is in Progerss
            //Get request for this channel will be initiated After Release ACK
            ETG_TRACE_USR4(("Already release for this channel is in process, Get Request to ClientHandler will be sent in ReleaseRequestAck"));

            //Get request pended and Release request cleared

            //Fix for GMMY17-11124
            //The release request may be pending for LC_Phone. So we should not make PendingReleaseRequest as false.
            //tChannelAquAttr[u16AudioChannelId].bPendingReleaseRequest = FALSE ;
            tChannelAquAttr[u16AudioChannelId].bPendingGetRequest = TRUE ;
            vPrintChannelAttributes();
         }
      }
      else //send Ack to corresponding SM ,As channel is Already Aquired
      {
         ETG_TRACE_USR4(("Channel is Already with FC_Phone, send Ack"));

         if (FC_PHONE_CHANNELAQUISITION_STATE_IDLE == tChannelAquAttr[u16AudioChannelId].u16ChannelAquState)
         {
            ETG_TRACE_USR4(("Sending acknowledgement"));

            m_rAudioCallBackMap[u16SlotId].FpCallBack(
                  m_rAudioCallBackMap[u16SlotId].u16AudioReqID,
                  m_rAudioCallBackMap[u16SlotId].u16AudioChannelId,
                  FC_PHONE_AUDIOCHANNEL_GRANTED);

            m_rAudioCallBackMap[u16SlotId].u16AudioReqID = FC_PHONE_AUDIOCHANNEL_REQUESTOR_ID_UNKNOWN;
            m_rAudioCallBackMap[u16SlotId].u16AudioChannelId = FC_PHONE_AUDIOCHANNEL_ID_UNUSED;
            m_rAudioCallBackMap[u16SlotId].FpCallBack = vAudioCallBackDummy;
         }
         else
         {
            ETG_TRACE_USR4(("Channel Aquisition is Still in Progress, Once Aquired Ack Will be sent"));
         }
      }
   }
   return(s32Error);
}

/*******************************************************************************
 *
 * FUNCTION: s32ReleaseRequest
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tS32 fc_phone_tclAudioManager::s32ReleaseRequest(tU16 u16AudioReqID, tU16 u16AudioChannelId, AudioCallBack fpCallBack)
{
   ETG_TRACE_USR4(("fc_phone_tclAudioManager::s32ReleaseRequest AudioReqID: %u AudioChannelId: %u",
         u16AudioReqID, u16AudioChannelId));

   tS32 s32Error = FC_PHONE_AUDIOCHANNEL_OK;

   //Check if request received is for valid channel
   if ((FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE != u16AudioChannelId)
         && (FC_PHONE_AUDIOCHANNEL_PHONE != u16AudioChannelId)
         && (FC_PHONE_AUDIOCHANNEL_ALERTTONE != u16AudioChannelId)
         && (FC_PHONE_AUDIOCHANNEL_SPEECH != u16AudioChannelId))
   {
      ETG_TRACE_USR4(("Release request received for invalid channel, AudioChannelId : '%u' !!", u16AudioChannelId));
      return FC_PHONE_AUDIOCHANNEL_INVALID;
   }

   //remove Release/GET slot , if present for same requestor ID and Channel ID
   s32Error = s32RemoveGetRequest(u16AudioReqID, u16AudioChannelId);
   if (FC_PHONE_AUDIOCHANNEL_OK == s32Error)
   {
      ETG_TRACE_USR4((" Removed GET/RELEASE request "));
   }

   tU16 u16SlotId = 0x00;

   s32Error = s32GetNextSlot(u16SlotId);

   if (FC_PHONE_AUDIOCHANNEL_OK == s32Error)
   {
      /* Register the release callback */
      m_rAudioCallBackMap[u16SlotId].u16AudioReqID = u16AudioReqID;
      m_rAudioCallBackMap[u16SlotId].u16AudioChannelId = u16AudioChannelId;
      m_rAudioCallBackMap[u16SlotId].FpCallBack = fpCallBack;

      //If this is last Release Request for this channel
      //forward request to ClientHandler
      //else send Release Ack to corresponding SM ,As channel is Released
      if (FC_PHONE_AUDIOMANAGER_LAST_RELEASEREQUEST == tChannelAquAttr[u16AudioChannelId].u16ChannelAquCounter)
      {
         //update channel attributes
         vUpdateChannelAttributes(u16AudioChannelId, FC_PHONE_REQUESTTYPE_RELEASE);

         if (FC_PHONE_CHANNELAQUISITION_STATE_IDLE == tChannelAquAttr[u16AudioChannelId].u16ChannelAquState)
         {
            ETG_TRACE_USR4((" Sending Release Request to ClientHandler"));

            //update channel State As Processing
            tChannelAquAttr[u16AudioChannelId].u16ChannelAquState = FC_PHONE_CHANNELAQUISITION_STATE_PROCESSING_RELEASE;

            vPrintChannelAttributes();

            if ((u16AudioChannelId == FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE) && (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->bIsSuppressRingtoneEnabled())
            {
               ETG_TRACE_USR4((" Releasing Suppressed Ringtone u16AudioChannelId = %d ",u16AudioChannelId));
               (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->vStopRingtone();
               s32ReleaseRequestAck(u16AudioChannelId);
               return (s32Error);
            }

            if (u16AudioChannelId == FC_PHONE_AUDIOCHANNEL_ALERTTONE)
            {
               ETG_TRACE_USR4(("AlertTone not supported currently - so release channel"));
               s32ReleaseRequestAck(u16AudioChannelId);
            }

            else if (u16AudioChannelId == FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE)
            {
               ETG_TRACE_USR4(("s32ReleaseRequest for FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE channel"));

               tU16 bInBandRingtone =
                     (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->vIsInBandRingtone();

               if (bInBandRingtone)
               {
                  ETG_TRACE_USR4(("For Inband ringtone no need to release LC_Phone channel"));
                  //The attributes are reset before sending Audio Route Request in clienthandler of BTSettings
               }
               else
               {
                  ETG_TRACE_USR4(("For Vehicle ringtone trigger release request"));

                  // Fix for SUZUKI-16521
                  /* No need to wait for the StopRingtone.MethodResult to post Release Acknowledgement */
                  s32ReleaseRequestAck(u16AudioChannelId);
                  //stop ringtone interface provided by BT.
                  (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->vStopRingtone();
               }
            }
            else
            {
               ETG_TRACE_ERR(("For other channel trigger release request"));

               tBool bARLRootRequest =
                     (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioRouting->vTriggerPhoneAudioDeAllocation(
                           u16AudioChannelId);

               if (bARLRootRequest)
               {
                  ETG_TRACE_USR4(("PhoneAudioDeAllocation request is posted"));
               }
               else
               {
                  ETG_TRACE_USR4(("PhoneAudioDeAllocation request posting has failed: send ReleaseRequestAck"));
                  s32ReleaseRequestAck(u16AudioChannelId);
               }
            }
         }
         else
         {
            //if State is not idle ,it means Get for Same Channel is in Progerss
            //Release request for this channel will be initiated After Get ACK
            ETG_TRACE_USR4(("Already Get for this channel is in process, "
                  "Release Request to ClientHandler will be sent in GetRequestAck"));

            //Fix for GMMY17-11124.
            //The Get request may be pending for LC_Phone. So we should not make PendingGetRequest as false.
            //tChannelAquAttr[u16AudioChannelId].bPendingGetRequest = FALSE;
            tChannelAquAttr[u16AudioChannelId].bPendingReleaseRequest = TRUE;
            vPrintChannelAttributes();
         }
      }
      else //send Fake release to the requestor
      {
         ETG_TRACE_USR4(("Sending Fake Release Ack to SM"));

         //update channel attributes
         vUpdateChannelAttributes(u16AudioChannelId, FC_PHONE_REQUESTTYPE_RELEASE);

         AudioCallBack pAudioCallBack = m_rAudioCallBackMap[u16SlotId].FpCallBack;
         tU16 AudioReqID = m_rAudioCallBackMap[u16SlotId].u16AudioReqID;

         pAudioCallBack(AudioReqID, u16AudioChannelId, FC_PHONE_AUDIOCHANNEL_GRANTED);

         m_rAudioCallBackMap[u16SlotId].u16AudioChannelId = FC_PHONE_AUDIOCHANNEL_ID_UNUSED;
         m_rAudioCallBackMap[u16SlotId].u16AudioReqID = FC_PHONE_AUDIOCHANNEL_REQUESTOR_ID_UNKNOWN;
         m_rAudioCallBackMap[u16SlotId].FpCallBack = vAudioCallBackDummy;
      }
   }

   return (s32Error);
}


/*******************************************************************************
 *
 * FUNCTION: s32GetRequestAck
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tS32 fc_phone_tclAudioManager::s32GetRequestAck(tU16 u16AudioChannelId, tU16 u16RequestState)
{
   ETG_TRACE_USR4(("fc_phone_tclAudioManager::s32GetRequestAck AudioChannelId: %u RequestState: %u",
         u16AudioChannelId, u16RequestState));

   //Fix for GMMY17-11124 and GMMY17-11964
   if((m_u8GetRequestCounter > 0) && (FC_PHONE_AUDIOCHANNEL_PHONE == u16AudioChannelId))
   {
      m_u8GetRequestCounter--;
      ETG_TRACE_USR4(("m_u8GetRequestCounter for LC_Phone = %u",m_u8GetRequestCounter));
   }

   tS32 s32Error = FC_PHONE_AUDIOCHANNEL_OK;
   fc_phone_tclService_Telephone *m_TelService_instance = (fc_phone_tclApp::m_poMainAppInstance)->m_poTelephone;

   //update channel attributes
   //Fix for GMMY17-11124 and GMMY17-11964
   if(((m_u8GetRequestCounter == 0) && (FC_PHONE_AUDIOCHANNEL_PHONE == u16AudioChannelId)) || (FC_PHONE_AUDIOCHANNEL_PHONE != u16AudioChannelId))
   {
      tChannelAquAttr[u16AudioChannelId].bPendingGetRequest = FALSE;
   }
   tChannelAquAttr[u16AudioChannelId].u16ChannelAquState = FC_PHONE_CHANNELAQUISITION_STATE_IDLE;

   vPrintChannelAttributes();

   if ((FC_PHONE_AUDIOCHANNEL_DENIED == u16RequestState) || (FC_PHONE_AUDIOCHANNEL_SUSPENDED == u16RequestState))
   {
      ETG_TRACE_USR4(("Channel Get Request failed, updating Channel Attribute Table"));

      if (TRUE == tChannelAquAttr[u16AudioChannelId].bPendingReleaseRequest)
      {
         ETG_TRACE_USR4(("Channel Get Request failed, PENDING Release request CASE"));

         //If Get request is denied and Release request is pending
         //Do not sent Release rrequest to ARL , fake the release ack
         tChannelAquAttr[u16AudioChannelId].bPendingReleaseRequest = FALSE;
         s32ReleaseRequestAck(u16AudioChannelId);
         return s32Error;
      }
      else
      {
         if (0x00 != tChannelAquAttr[u16AudioChannelId].u16ChannelAquCounter)
         {
            tChannelAquAttr[u16AudioChannelId].u16ChannelAquCounter = 0;
            vPrintChannelAttributes();
         }
      }
   }

   //check and process pending Release request for this channel if any
   if (TRUE == tChannelAquAttr[u16AudioChannelId].bPendingReleaseRequest)
   {
      ETG_TRACE_USR4(("Processing Pending Release Request for AudioChannelId: %u", u16AudioChannelId));

      tChannelAquAttr[u16AudioChannelId].bPendingReleaseRequest = FALSE;
      tChannelAquAttr[u16AudioChannelId].u16ChannelAquState = FC_PHONE_CHANNELAQUISITION_STATE_PROCESSING_RELEASE;

      vPrintChannelAttributes();

      if (u16AudioChannelId == FC_PHONE_AUDIOCHANNEL_ALERTTONE)
      {
         ETG_TRACE_USR4((" AlertTone not supported currently - so release channel"));
         s32ReleaseRequestAck(u16AudioChannelId);
      }

      else if (u16AudioChannelId == FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE)
      {
         ETG_TRACE_USR4(("s32ReleaseRequest for FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE channel"));

         fc_phone_tclClientHandler_BTSettings *p_clientHndler_bluetooth = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings;
         tU16 bInBandRingtone = p_clientHndler_bluetooth->vIsInBandRingtone();

         if (bInBandRingtone)
         {
            ETG_TRACE_ERR(("For Inband ringtone no need to release LC_Phone channel"));
         }
         else
         {
            ETG_TRACE_ERR(("For Vehicle ringtone trigger release request"));
            s32ReleaseRequestAck(u16AudioChannelId);
            //SUZUKI-20931- stop ringtone interface provided by BT.
            (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->vStopRingtone();
         }
      }
      else
      {
         ETG_TRACE_ERR(("For other channel trigger release request"));
         bool bARLRootRequest = (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioRouting->vTriggerPhoneAudioDeAllocation(u16AudioChannelId);
         if (bARLRootRequest)
         {
            ETG_TRACE_USR4(("PhoneAudioDeAllocation request is posted"));
         }
         else
         {
            ETG_TRACE_USR4(("PhoneAudioDeAllocation request posting has failed: send ReleaseRequestAck"));
            s32ReleaseRequestAck(u16AudioChannelId);
         }
      }
   }
   else
   {
      ETG_TRACE_USR4(("No Pending Release Request for AudioChannelId: %u, sending ACKs", u16AudioChannelId));

      //send ACK to all registered CallBacks, for this channel
      for (tU16 u16SlotId = 0x00; u16SlotId < FC_PHONE_AUDIOCHANNEL_LIMIT;
            u16SlotId++)
      {
         if (u16AudioChannelId == m_rAudioCallBackMap[u16SlotId].u16AudioChannelId)
         {
            ETG_TRACE_USR4(("Sending Ack for AudioChannelId: %u SlotId: %u", u16AudioChannelId, u16SlotId));

            LoopBackData LB_data;
            LB_data.u16FunctionID = FC_PHONE_CB_FID_AUDIOMANAGER;
            LB_data.AudioMgr_Data.u16AudioChannelId = u16AudioChannelId;
            LB_data.AudioMgr_Data.u16AudioReqID = m_rAudioCallBackMap[u16SlotId].u16AudioReqID;
            LB_data.AudioMgr_Data.u16RequestState = u16RequestState;
            LB_data.AudioMgr_Data.AudioCallBack = m_rAudioCallBackMap[u16SlotId].FpCallBack;

            m_TelService_instance->push_back_LB_data(LB_data);
            m_TelService_instance->vPrepareLoopBackMsg(FC_PHONE_CB_FID_AUDIOMANAGER);

            m_rAudioCallBackMap[u16SlotId].u16AudioChannelId = FC_PHONE_AUDIOCHANNEL_ID_UNUSED;
            m_rAudioCallBackMap[u16SlotId].FpCallBack = vAudioCallBackDummy;
            m_rAudioCallBackMap[u16SlotId].u16AudioReqID = FC_PHONE_AUDIOCHANNEL_REQUESTOR_ID_UNKNOWN;
         }
      }
   }
   return (s32Error);
}

/*******************************************************************************
 *
 * FUNCTION: s32ReleaseRequestAck
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tS32 fc_phone_tclAudioManager::s32ReleaseRequestAck(tU16 u16AudioChannelId)
{
   ETG_TRACE_USR4(("fc_phone_tclAudioManager::s32ReleaseRequestAck u16AudioChannelId is : %d",u16AudioChannelId ));
   fc_phone_tclService_Telephone *m_TelService_instance = (fc_phone_tclApp::m_poMainAppInstance)->m_poTelephone;
   tS32 s32Error = FC_PHONE_AUDIOCHANNEL_OK;

   ETG_TRACE_USR4(("bPendingGetRequest : %d bPendingReleaseRequest : %d",tChannelAquAttr[u16AudioChannelId].bPendingGetRequest,tChannelAquAttr[u16AudioChannelId].bPendingReleaseRequest ));
   if ((FC_PHONE_AUDIOCHANNEL_PHONE == u16AudioChannelId)&&((TRUE == tChannelAquAttr[u16AudioChannelId].bPendingGetRequest) \
         && (TRUE == tChannelAquAttr[u16AudioChannelId].bPendingReleaseRequest)))
   {
      ETG_TRACE_USR4(("Reset the bPendingGetRequest for Phone Channel"));
      tChannelAquAttr[u16AudioChannelId].bPendingGetRequest = FALSE ;
   }
   tChannelAquAttr[u16AudioChannelId].bPendingReleaseRequest = FALSE ;
   //update channel attributes
   tChannelAquAttr[u16AudioChannelId].u16ChannelAquState = FC_PHONE_CHANNELAQUISITION_STATE_IDLE ;
   vPrintChannelAttributes();

   //check and process pending Get request for this channel, if Any
   if(TRUE == tChannelAquAttr[u16AudioChannelId].bPendingGetRequest)
   {
      ETG_TRACE_USR4(("  Processing Pending Get Request for u16AudioChannelId: %d",u16AudioChannelId ));

      //Fix for GMMY17-11124 and GMMY17-11964.
      if(((m_u8GetRequestCounter == 0) && (FC_PHONE_AUDIOCHANNEL_PHONE == u16AudioChannelId)) || (FC_PHONE_AUDIOCHANNEL_PHONE != u16AudioChannelId))
      {
         tChannelAquAttr[u16AudioChannelId].bPendingGetRequest = FALSE ;
      }

      tChannelAquAttr[u16AudioChannelId].u16ChannelAquState = FC_PHONE_CHANNELAQUISITION_STATE_PROCESSING_GET ;
      vPrintChannelAttributes();

      //update channel State As Processing
      tChannelAquAttr[u16AudioChannelId].u16ChannelAquState = FC_PHONE_CHANNELAQUISITION_STATE_PROCESSING_GET ;
      vPrintChannelAttributes();

      if(u16AudioChannelId == FC_PHONE_AUDIOCHANNEL_ALERTTONE)
      {
         ETG_TRACE_USR4((" AlertTone not supported currently - so grant channel"));
         s32GetRequestAck(u16AudioChannelId, FC_PHONE_AUDIOCHANNEL_GRANTED);
      }
      else if(u16AudioChannelId == FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE)
      {
         ETG_TRACE_USR4((" Audio channel is Ringtone"));
         fc_phone_tclClientHandler_BTSettings *m_poBTClient = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings;
         tU8 u8DeviceAddrId = m_poBTClient->vGetRingToneDeviceAddrId();
         m_poBTClient->RequestAvailableRingtonesList((fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->u8GetDeviceHandleFromBtAddressId(u8DeviceAddrId));
      }
      else
      {
         ETG_TRACE_USR4((" Audio channel other that Ringtone"));
         bool bARLRootRequest = (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioRouting->vTriggerPhoneAudioAllocation(u16AudioChannelId);
         if(bARLRootRequest)
         {
            ETG_TRACE_USR4((" Request root success: sending Ack for u16AudioChannelId :: %d ",u16AudioChannelId));
            //s32GetRequestAck(u16AudioChannelId, FC_PHONE_AUDIOCHANNEL_GRANTED);
         }
         else
         {
            ETG_TRACE_USR4((" Request root failed: for u16AudioChannelId :: %d ",u16AudioChannelId));
            s32GetRequestAck(u16AudioChannelId, FC_PHONE_AUDIOCHANNEL_DENIED);
         }
      }
   }
   else
   {
      ETG_TRACE_USR4(("No Pending Get Request for AudioChannelId: %u, sending ACKs", u16AudioChannelId));

      //send ACK to all registered CallBacks, for this channel
      for (tU16 u16SlotId = 0x00; u16SlotId < FC_PHONE_AUDIOCHANNEL_LIMIT;
            u16SlotId++)
      {
         if (u16AudioChannelId == m_rAudioCallBackMap[u16SlotId].u16AudioChannelId)
         {
            ETG_TRACE_USR4(("Sending Ack for AudioChannelId: %u SlotId: %u", u16AudioChannelId, u16SlotId));

            LoopBackData LB_data;
            LB_data.u16FunctionID = FC_PHONE_CB_FID_AUDIOMANAGER;
            LB_data.AudioMgr_Data.u16AudioChannelId = u16AudioChannelId;
            LB_data.AudioMgr_Data.u16AudioReqID = m_rAudioCallBackMap[u16SlotId].u16AudioReqID;
            LB_data.AudioMgr_Data.u16RequestState = FC_PHONE_AUDIOCHANNEL_GRANTED;
            LB_data.AudioMgr_Data.AudioCallBack = m_rAudioCallBackMap[u16SlotId].FpCallBack;

            m_TelService_instance->push_back_LB_data(LB_data);
            m_TelService_instance->vPrepareLoopBackMsg(FC_PHONE_CB_FID_AUDIOMANAGER);

            m_rAudioCallBackMap[u16SlotId].u16AudioChannelId = FC_PHONE_AUDIOCHANNEL_ID_UNUSED;
            m_rAudioCallBackMap[u16SlotId].FpCallBack = vAudioCallBackDummy;
            m_rAudioCallBackMap[u16SlotId].u16AudioReqID = FC_PHONE_AUDIOCHANNEL_REQUESTOR_ID_UNKNOWN;
         }
      }
   }

   return(s32Error);
}


/*******************************************************************************
 *
 * FUNCTION: s32GetNextSlot
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tS32 fc_phone_tclAudioManager::s32GetNextSlot(tU16& rfSlotId)
{
   ETG_TRACE_USR4(("fc_phone_tclAudioManager::s32GetNextSlot entered"));

   tS32 s32Error = FC_PHONE_AUDIOCHANNEL_LIMIT_EXCEEDED;

   for (tU16 u16Index = 0x00; u16Index < FC_PHONE_AUDIOCHANNEL_LIMIT;
         u16Index++)
   {
      if (FC_PHONE_AUDIOCHANNEL_ID_UNUSED == m_rAudioCallBackMap[u16Index].u16AudioChannelId)
      {
         rfSlotId = u16Index;
         s32Error = FC_PHONE_AUDIOCHANNEL_OK;
         ETG_TRACE_USR4(("s32GetNextSlot SlotId: '%u'", u16Index));
         break;
      }
   }

   return (s32Error);
}

/*******************************************************************************
 *
 * FUNCTION: s32GetSlotIdFromAudioChannelId
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tS32 fc_phone_tclAudioManager::s32GetSlotIdFromAudioChannelId(tU16 u16AudioChannelId, tU16& rfSlotId)
{
   ETG_TRACE_USR4(("fc_phone_tclAudioManager::s32GetSlotIdFromAudioChannelId: entered "));

   tS32 s32Error = FC_PHONE_AUDIOCHANNEL_INVALID;

   for (tU16 u16Index = 0x00; u16Index < FC_PHONE_AUDIOCHANNEL_LIMIT;
         u16Index++)
   {
      if (u16AudioChannelId == m_rAudioCallBackMap[u16Index].u16AudioChannelId)
      {
         rfSlotId = u16Index;
         s32Error = FC_PHONE_AUDIOCHANNEL_OK;
         ETG_TRACE_USR4(("s32GetSlotIdFromAudioChannelId SlotId: '%u'", u16Index));
         break;
      }
   }

   return (s32Error);
}

/*******************************************************************************
 *
 * FUNCTION: RemoveGetRequest
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tS32 fc_phone_tclAudioManager::s32RemoveGetRequest(tU16 u16AudioReqID, tU16 u16AudioChannelId)
{
   ETG_TRACE_USR4(("fc_phone_tclAudioManager::s32RemoveGetRequest AudioReqID: %u AudioChannelId: %u",
         u16AudioReqID, u16AudioChannelId));

   tS32 s32Error = FC_PHONE_AUDIOCHANNEL_INVALID;
   tBool bMatchFound = FALSE;

   for (tU16 u16Index = 0x00; u16Index < FC_PHONE_AUDIOCHANNEL_LIMIT;
         u16Index++)
   {
      if ((u16AudioChannelId == m_rAudioCallBackMap[u16Index].u16AudioChannelId)
            && (u16AudioReqID == m_rAudioCallBackMap[u16Index].u16AudioReqID))

      {
         ETG_TRACE_USR4(("s32RemoveGetRequest match found at SlotId: %u", u16Index));

         m_rAudioCallBackMap[u16Index].u16AudioReqID = FC_PHONE_AUDIOCHANNEL_REQUESTOR_ID_UNKNOWN;
         m_rAudioCallBackMap[u16Index].u16AudioChannelId = FC_PHONE_AUDIOCHANNEL_ID_UNUSED;
         m_rAudioCallBackMap[u16Index].FpCallBack = vAudioCallBackDummy;
         s32Error = FC_PHONE_AUDIOCHANNEL_OK;
         bMatchFound = TRUE;
         break;
      }
   }

   if (FALSE == bMatchFound)
   {
      //this line should never execute
      ETG_TRACE_ERR(("Error :: No Match found for u16AudioReqID : %d and u16AudioChannelId: %d in slot list",
            u16AudioReqID, u16AudioChannelId));
   }

   return (s32Error);
}

/*******************************************************************************
 *
 * FUNCTION: vSetPriorityChannel
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tVoid fc_phone_tclAudioManager::vSetPriorityChannel(tBool b_flag_EmerOrAdvisor)
{
   ETG_TRACE_USR4(("fc_phone_tclAudioManager::vSetPriorityChannel: entered "));
   m_bPriorityChannelFlag = b_flag_EmerOrAdvisor ;
}

/*******************************************************************************
 *
 * FUNCTION: bGetPriorityChannel
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tBool fc_phone_tclAudioManager::bGetPriorityChannel()
{
   ETG_TRACE_USR4(("fc_phone_tclAudioManager::bGetPriorityChannel: entered "));
   return m_bPriorityChannelFlag ;
}
/*******************************************************************************
 ** FUNCTION:   pGetInstance
 *******************************************************************************/
/* DESCRIPTION:
 *   Gets the current Instance
 *  PARAMETERS:None
 *
 *
 *  RETURNVALUE: fc_phone_tclAudioManager* - Instance to singleton.
 ********************************************************************************/
fc_phone_tclAudioManager* fc_phone_tclAudioManager::pGetInstance(tVoid)
{
   ETG_TRACE_USR4(("fc_phone_tclAudioManager::pGetInstance: entered "));
   return m_poInstance;
}
/*******************************************************************************
 *
 * FUNCTION: vHandleError
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclAudioManager::vHandleError(tVoid)
{
   ETG_TRACE_USR4(("  fc_phone_tclAudioManager::vHandleError "));
   fc_phone_tclService_Telephone *m_TelService_instance = (fc_phone_tclApp::m_poMainAppInstance)->m_poTelephone;

   /*Resets Audio Manager*/
   ETG_TRACE_USR4(("  fc_phone_tclAudioManager::Resets Audio Manager"));
   for (tU16 u16Index = 0x00; u16Index < FC_PHONE_AUDIOCHANNEL_LIMIT; u16Index++)
   {
      m_rAudioCallBackMap[u16Index].u16AudioReqID     = FC_PHONE_AUDIOCHANNEL_REQUESTOR_ID_UNKNOWN ;
      m_rAudioCallBackMap[u16Index].u16AudioChannelId = FC_PHONE_AUDIOCHANNEL_ID_UNUSED;
      m_rAudioCallBackMap[u16Index].FpCallBack = vAudioCallBackDummy;
   }
   for (tU16 u16Index = 0x00; u16Index < FC_PHONE_NUM_OFCHANNELS; u16Index++)
   {
      tChannelAquAttr[u16Index].u16ChannelAquCounter     = 0 ;
      tChannelAquAttr[u16Index].u16ChannelAquState = FC_PHONE_CHANNELAQUISITION_STATE_IDLE;
      tChannelAquAttr[u16Index].bPendingReleaseRequest = FALSE;
      tChannelAquAttr[u16Index].bPendingGetRequest = FALSE;
   }

   /*Resets State Machines*/
   if(m_TelService_instance)
   {
      ETG_TRACE_USR4((" Resets State Machines"));
      m_TelService_instance->vResetStateMachines();
   }
}

/*******************************************************************************
 *
 * FUNCTION: vReleaseUsedAudioChannels
 *
 * DESCRIPTION:Release Used Audio Channels
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclAudioManager::vReleaseUsedAudioChannels(tVoid)
{
   /*Resets Audio Manager*/
   ETG_TRACE_USR4(("  fc_phone_tclAudioManager::Release Used Audio Channels"));

   for (tU16 u16Index = 0x00; u16Index < FC_PHONE_AUDIOCHANNEL_LIMIT; u16Index++)
   {
      m_rAudioCallBackMap[u16Index].u16AudioReqID     = FC_PHONE_AUDIOCHANNEL_REQUESTOR_ID_UNKNOWN ;
      m_rAudioCallBackMap[u16Index].u16AudioChannelId = FC_PHONE_AUDIOCHANNEL_ID_UNUSED;
      m_rAudioCallBackMap[u16Index].FpCallBack = vAudioCallBackDummy;
   }

   for (tU16 u16Index = 0x00; u16Index < FC_PHONE_NUM_OFCHANNELS; u16Index++)
   {
      if ((tChannelAquAttr[u16Index].u16ChannelAquCounter > 0) ||
            (tChannelAquAttr[u16Index].u16ChannelAquState == FC_PHONE_CHANNELAQUISITION_STATE_PROCESSING_GET))
      {
         ETG_TRACE_USR4((" sending Release request for channel : %d",u16Index));
         //Added for GMMY15-10541- If Channel is released without setting the Aqustate later
         //when trying to acquire the channel activation will be posted to AVManager which will be replied with an error.
         //Now the channel is in releasing state so Activation request will not be posted.
         tChannelAquAttr[u16Index].u16ChannelAquCounter     = 0 ;
         tChannelAquAttr[u16Index].bPendingReleaseRequest = FALSE;
         tChannelAquAttr[u16Index].bPendingGetRequest = FALSE;

         if( tChannelAquAttr[u16Index].u16ChannelAquState == FC_PHONE_CHANNELAQUISITION_STATE_PROCESSING_GET )
         {
            tChannelAquAttr[u16Index].u16ChannelAquState = FC_PHONE_CHANNELAQUISITION_STATE_IDLE;
         }
         else
         {
            tChannelAquAttr[u16Index].u16ChannelAquState = FC_PHONE_CHANNELAQUISITION_STATE_PROCESSING_RELEASE;
         }

         if((u16Index == FC_PHONE_AUDIOCHANNEL_PHONE) || (u16Index == FC_PHONE_AUDIOCHANNEL_SPEECH))
         {
            bool bARLRootRequest = (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioRouting->vTriggerPhoneAudioDeAllocation(u16Index);
            if (bARLRootRequest)
            {
               ETG_TRACE_USR4(("PhoneAudioDeAllocation request is posted"));
            }
            else
            {
               ETG_TRACE_USR4(("PhoneAudioDeAllocation request posting has failed: send ReleaseRequestAck"));
            }
         }
         else if(u16Index == FC_PHONE_AUDIOCHANNEL_ALERTTONE)
         {
            tChannelAquAttr[u16Index].u16ChannelAquState = FC_PHONE_CHANNELAQUISITION_STATE_IDLE;
         }
         else if(u16Index == FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE)
         {
            ETG_TRACE_USR4(("OutBand Ringtone is playing. Hence need to stop it."));
            (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerBTSettings->vStopRingtone();
            s32ReleaseRequestAck(FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE);
         }
      }
      //GMMY15-243 If channel is already being released, don't reset the channel related parameters otherwise,
      //later when we post AV-Activation to AVManager (as new callstatus is received), it will be denied and
      //AV-Activation error will be posted to FC-Phone Fblock, as the release request is in Processing state.
      if(tChannelAquAttr[u16Index].u16ChannelAquState != FC_PHONE_CHANNELAQUISITION_STATE_PROCESSING_RELEASE)
      {
         tChannelAquAttr[u16Index].u16ChannelAquCounter     = 0 ;
         tChannelAquAttr[u16Index].u16ChannelAquState = FC_PHONE_CHANNELAQUISITION_STATE_IDLE;
         tChannelAquAttr[u16Index].bPendingReleaseRequest = FALSE;
         tChannelAquAttr[u16Index].bPendingGetRequest = FALSE;
      }
   }
   m_u8GetRequestCounter = 0;
}

/*******************************************************************************
 *
 * FUNCTION: vReleasePhoneChannel
 *
 * DESCRIPTION: GMMY17-11914 and NCG3D-4727.
 *              Called upon the Error condition on expiry of AudioWatchDog
 *              timer.
 *              Release the Phone audio Channel and update the channel attributes.
 *
 * PARAMETER:None
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclAudioManager::vReleasePhoneChannel(tVoid)
{
   ETG_TRACE_USR4((" vReleasePhoneChannel Entry"));

   tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_PHONE].u16ChannelAquState = FC_PHONE_CHANNELAQUISITION_STATE_PROCESSING_RELEASE;

   tBool bARLRootRequest =
         (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioRouting->vTriggerPhoneAudioDeAllocation(
               FC_PHONE_AUDIOCHANNEL_PHONE);

   if (bARLRootRequest)
   {
      ETG_TRACE_USR4(("PhoneAudioDeAllocation request is posted"));
   }
   else
   {
      ETG_TRACE_USR4(("PhoneAudioDeAllocation request posting has failed"));
   }

   tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_PHONE].bPendingGetRequest = FALSE;
   tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_PHONE].bPendingReleaseRequest = FALSE;
   tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_PHONE].u16ChannelAquCounter = 0;

   m_u8GetRequestCounter = 0;

   ETG_TRACE_USR4((" vReleasePhoneChannel Exit"));
}
/*******************************************************************************
 *
 * FUNCTION: vUpdateChannelAttributes
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclAudioManager::vUpdateChannelAttributes(tU16 u16AudioChannelId ,tU16 u16RequestType)
{
   ETG_TRACE_USR4(("  fc_phone_tclAudioManager::vUpdateChannelAttributes   u16AudioChannelId:: %d  "
         " u16RequestType %d",u16AudioChannelId,u16RequestType));

   //using channel ID as Index for updating the Attributes of corresponding channel.
   //As Channel Ids are 0,1 and 2

   if(FC_PHONE_REQUESTTYPE_GET == u16RequestType)
   {
      tChannelAquAttr[u16AudioChannelId].u16ChannelAquCounter++ ;
   }

   else if(FC_PHONE_REQUESTTYPE_RELEASE == u16RequestType)
   {
      if(0x00 != tChannelAquAttr[u16AudioChannelId].u16ChannelAquCounter)
      {
         tChannelAquAttr[u16AudioChannelId].u16ChannelAquCounter-- ;
      }
      else
      {
         /* Release request for audio without a get request */
         ETG_TRACE_USR4((" Release request for audio without a get request "));
      }
   }
   else
   {
      //this line should never execute
      ETG_TRACE_ERR(("  Wrong u16RequestType :; %d ",u16RequestType));
   }

   vPrintChannelAttributes();

}
/*******************************************************************************
 *
 * FUNCTION: vPrintChannelAttributes
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclAudioManager::vPrintChannelAttributes()
{
   ETG_TRACE_USR4((" Printing Audio Channel Attributes"));

   for (tU16 u16Index = 0x00; u16Index < FC_PHONE_NUM_OFCHANNELS; u16Index++)
   {
      ETG_TRACE_USR4((" tChannelAquAttr[%d].u16ChannelAquCounter %d ",u16Index,tChannelAquAttr[u16Index].u16ChannelAquCounter));
      ETG_TRACE_USR4((" tChannelAquAttr[%d].u16ChannelAquState %d ",u16Index,tChannelAquAttr[u16Index].u16ChannelAquState));
      ETG_TRACE_USR4((" tChannelAquAttr[%d].bPendingReleaseRequest %d ",u16Index,tChannelAquAttr[u16Index].bPendingReleaseRequest));
      ETG_TRACE_USR4((" tChannelAquAttr[%d].bPendingGetRequest %d ",u16Index,tChannelAquAttr[u16Index].bPendingGetRequest));
   }
}

/*******************************************************************************
 *
 * FUNCTION: s32SyncAudioMangerOnChannelRelease
 *
 * DESCRIPTION: Phone channel is released without sending release request to AVManager
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tS32 fc_phone_tclAudioManager::s32SyncAudioMangerOnChannelRelease(tU16 u16AudioReqID,tU16 u16AudioChannelId)
{

   ETG_TRACE_USR4((" fc_phone_tclAudioManager::s32SyncAudioMangerOnChannelRelease:%u",u16AudioChannelId));

   //check ,Valid Channel Requested
   if((FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE != u16AudioChannelId)
         &&(FC_PHONE_AUDIOCHANNEL_PHONE != u16AudioChannelId)
         &&(FC_PHONE_AUDIOCHANNEL_ALERTTONE != u16AudioChannelId)
         &&(FC_PHONE_AUDIOCHANNEL_SPEECH != u16AudioChannelId)
   )
   {
      ETG_TRACE_USR4(("  Invalid release Channel Requested u16AudioChannelId :%u",u16AudioChannelId));
      return FC_PHONE_AUDIOCHANNEL_INVALID;
   }
   for(tU16 u16Index = 0x00; u16Index < FC_PHONE_AUDIOCHANNEL_LIMIT; u16Index++)
   {
      if((u16AudioChannelId == m_rAudioCallBackMap[u16Index].u16AudioChannelId)
            &&(u16AudioReqID == m_rAudioCallBackMap[u16Index].u16AudioReqID))
      {
         m_rAudioCallBackMap[u16Index].u16AudioReqID = FC_PHONE_AUDIOCHANNEL_REQUESTOR_ID_UNKNOWN ;
         m_rAudioCallBackMap[u16Index].u16AudioChannelId = FC_PHONE_AUDIOCHANNEL_ID_UNUSED;
         m_rAudioCallBackMap[u16Index].FpCallBack = vAudioCallBackDummy;
         break;
      }
   }
   //update channel attributes
   vUpdateChannelAttributes(u16AudioChannelId, FC_PHONE_REQUESTTYPE_RELEASE);

   tChannelAquAttr[u16AudioChannelId].bPendingReleaseRequest = FALSE ;
   //update channel attributes
   tChannelAquAttr[u16AudioChannelId].u16ChannelAquState = FC_PHONE_CHANNELAQUISITION_STATE_IDLE ;
   vPrintChannelAttributes();

   return FC_PHONE_AUDIOCHANNEL_OK;
}

/*******************************************************************************
 *
 * FUNCTION: s32SyncAudioMangerOnChannelGranted
 *
 * DESCRIPTION: Phone channel is granted without sending get request to AVManager
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tS32 fc_phone_tclAudioManager::s32SyncAudioMangerOnChannelGranted(tU16 u16AudioReqID,tU16 u16AudioChannelId)
{
   ETG_TRACE_USR4(("fc_phone_tclAudioManager::s32SyncAudioMangerOnChannelGranted:%u %u",u16AudioChannelId,u16AudioReqID));

   (tVoid) u16AudioReqID;

   //check ,Valid Channel Requested
   if((FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE != u16AudioChannelId)
         &&(FC_PHONE_AUDIOCHANNEL_PHONE != u16AudioChannelId)
         &&(FC_PHONE_AUDIOCHANNEL_ALERTTONE != u16AudioChannelId)
         &&(FC_PHONE_AUDIOCHANNEL_SPEECH != u16AudioChannelId)
   )
   {

      ETG_TRACE_USR4(("Invalid release Channel Requested u16AudioChannelId :%u ",u16AudioChannelId));
      return FC_PHONE_AUDIOCHANNEL_INVALID;
   }
   ETG_TRACE_USR4(("Assigning channel attributes back when SA_ON received"));
   for(tU16 u16Index = 0x00; u16Index < FC_PHONE_AUDIOCHANNEL_LIMIT; u16Index++)
   {
      if(m_rAudioCallBackMap[u16Index].u16AudioReqID == FC_PHONE_AUDIOCHANNEL_REQUESTOR_ID_UNKNOWN)

      {
         m_rAudioCallBackMap[u16Index].u16AudioChannelId = u16AudioChannelId;
         m_rAudioCallBackMap[u16Index].u16AudioReqID = u16AudioReqID;
         break;
      }
   }
   //update channel attributes
   vUpdateChannelAttributes(u16AudioChannelId, FC_PHONE_REQUESTTYPE_GET);

   tChannelAquAttr[u16AudioChannelId].bPendingReleaseRequest = FALSE;
   tChannelAquAttr[u16AudioChannelId].bPendingGetRequest = FALSE;

   //update channel attributes
   tChannelAquAttr[u16AudioChannelId].u16ChannelAquState = FC_PHONE_CHANNELAQUISITION_STATE_IDLE;
   vPrintChannelAttributes();

   if(FC_PHONE_AUDIOCHANNEL_PHONE == u16AudioChannelId)
   {
      m_u8GetRequestCounter = 0;
   }

   return FC_PHONE_AUDIOCHANNEL_OK;
}


/*******************************************************************************
 *
 * FUNCTION: bIsPhoneChannelRelease
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tBool fc_phone_tclAudioManager::bIsPhoneChannelRelease()
{
   ETG_TRACE_USR4(("fc_phone_tclAudioManager::bIsPhoneChannelRelease"));
   ETG_TRACE_USR4((" u16ChannelAquState : %d ", tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_PHONE].u16ChannelAquState));

   if (FC_PHONE_CHANNELAQUISITION_STATE_PROCESSING_RELEASE == tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_PHONE].u16ChannelAquState)
   {
      return TRUE;
   }

   return FALSE;
}

/*******************************************************************************
 *
 * FUNCTION: bIsPhoneChannelGet
 *
 * DESCRIPTION:
 *
 * PARAMETER: None
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tBool fc_phone_tclAudioManager::bIsPhoneChannelGet()
{
   ETG_TRACE_USR4(("fc_phone_tclAudioManager::bIsPhoneChannelGet"));
   ETG_TRACE_USR4((" u16ChannelAquState : %d ", tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_PHONE].u16ChannelAquState));
   ETG_TRACE_USR4((" u16ChannelAquCounter : %d ", tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_PHONE].u16ChannelAquCounter));

   if (FC_PHONE_CHANNELCOUNTER_DEFAULT != tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_PHONE].u16ChannelAquCounter)
   {
      return TRUE;
   }

   return FALSE;
}

/*******************************************************************************
 *
 * FUNCTION: tSpeechChannelStatus
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE: tenSpeechChannelStatus.
 *
 ********************************************************************************/
tU8 fc_phone_tclAudioManager::u8SpeechChannelStatus()
{
   ETG_TRACE_USR4(("fc_phone_tclAudioManager::u8SpeechChannelStatus"));
   ETG_TRACE_USR4((" u16ChannelAquState : %d ", tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_SPEECH].u16ChannelAquState));

   tU8 u8RetStatus = EN_SPEECH_CHNL_NOTACTIVE;

   if( FC_PHONE_CHANNELCOUNTER_DEFAULT != tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_SPEECH].u16ChannelAquCounter)
   {
      if( (FC_PHONE_CHANNELAQUISITION_STATE_IDLE == tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_SPEECH].u16ChannelAquState) )
      {
         u8RetStatus = EN_SPEECH_CHNL_ACTIVE;
      }
      else if( FC_PHONE_CHANNELAQUISITION_STATE_PROCESSING_GET == tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_SPEECH].u16ChannelAquState )
      {
         u8RetStatus = EN_SPEECH_CHNL_GET_PROGRESS;
      }
      else if( FC_PHONE_CHANNELAQUISITION_STATE_PROCESSING_RELEASE == tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_SPEECH].u16ChannelAquState )
      {
         u8RetStatus = EN_SPEECH_CHNL_RELEASE_PROGRESS;
      }
   }

   ETG_TRACE_USR4((" u8RetStatus : %d ", u8RetStatus));
   return u8RetStatus;
}
/*******************************************************************************
 *
 * FUNCTION: bIsPhoneOrRingtoneChannelAcquistionIdle
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tBool fc_phone_tclAudioManager::bIsPhoneOrRingtoneChannelAcquistionIdle()
{

   ETG_TRACE_USR4(("fc_phone_tclAudioManager::bIsPhoneOrRingtoneChannelAcquistionIdle"));


   ETG_TRACE_USR4((" u16ChannelAquState : %d ", tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_PHONE].u16ChannelAquState));

   if ( (FC_PHONE_CHANNELAQUISITION_STATE_IDLE == tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_PHONE].u16ChannelAquState) &&
         (FC_PHONE_CHANNELAQUISITION_STATE_IDLE == tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE].u16ChannelAquState) )
   {
      return TRUE;
   }
   return FALSE;
}

/*******************************************************************************
 *
 * FUNCTION: bIsPhoneOrRingtoneChannelAcquistionIdle
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tBool fc_phone_tclAudioManager::bIsPhoneOrRingtoneChannelAcquistionInRelease()
{

   ETG_TRACE_USR4(("fc_phone_tclAudioManager::bIsPhoneOrRingtoneChannelAcquistionInRelease"));
   ETG_TRACE_USR4((" u16ChannelAquStateFC_PHONE_AUDIOCHANNEL_PHONE : %d ", tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_PHONE].u16ChannelAquState));
   ETG_TRACE_USR4((" u16ChannelAquState FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE: %d ", tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE].u16ChannelAquState));

   if ( (FC_PHONE_CHANNELAQUISITION_STATE_PROCESSING_RELEASE == tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_PHONE].u16ChannelAquState) ||
         (FC_PHONE_CHANNELAQUISITION_STATE_PROCESSING_RELEASE == tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE].u16ChannelAquState) )
   {
      return TRUE;
   }
   return FALSE;
}

/*******************************************************************************
 *
 * FUNCTION: vResetChannelAttributes
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclAudioManager::vResetChannelAttributes(tU16 u16AudioChannelId)
{
   ETG_TRACE_USR4(("fc_phone_tclAudioManager::vResetChannelAttributes u16AudioChannelId: %d", u16AudioChannelId));
   for (tU16 u16Index = 0x00; u16Index < FC_PHONE_NUM_OFCHANNELS; u16Index++)
   {
      if(u16AudioChannelId == u16Index)
      {
         tChannelAquAttr[u16Index].u16ChannelAquCounter = 0;
         tChannelAquAttr[u16Index].u16ChannelAquState = FC_PHONE_CHANNELAQUISITION_STATE_IDLE;
         tChannelAquAttr[u16Index].bPendingReleaseRequest = FALSE;
         tChannelAquAttr[u16Index].bPendingGetRequest = FALSE;

         if(FC_PHONE_AUDIOCHANNEL_PHONE == u16AudioChannelId)
         {
            m_u8GetRequestCounter = 0;
         }

         break;
      }
   }
   vPrintChannelAttributes();
}

/*******************************************************************************
 *
 * FUNCTION:    vResetAllChannelAttributes
 *
 * DESCRIPTION: Fix for NCG3D-5406.
 *              RESET all channel attributes on receiving SLC_ON
 *
 * PARAMETER:   None
 *
 * RETURNVALUE: None
 *
 ********************************************************************************/
tVoid fc_phone_tclAudioManager::vResetAllChannelAttributes()
{
   ETG_TRACE_USR4(("fc_phone_tclAudioManager::vResetAllChannelAttributes called"));

   tU16 u16Index = 0x00;

   for (u16Index = 0x00; u16Index < FC_PHONE_AUDIOCHANNEL_LIMIT; u16Index++)
   {
      m_rAudioCallBackMap[u16Index].u16AudioReqID     = FC_PHONE_AUDIOCHANNEL_REQUESTOR_ID_UNKNOWN ;
      m_rAudioCallBackMap[u16Index].u16AudioChannelId = FC_PHONE_AUDIOCHANNEL_ID_UNUSED;
      m_rAudioCallBackMap[u16Index].FpCallBack = vAudioCallBackDummy;
   }

   for (u16Index = 0x00; u16Index < FC_PHONE_NUM_OFCHANNELS; u16Index++)
   {
      tChannelAquAttr[u16Index].u16ChannelAquCounter = 0;
      tChannelAquAttr[u16Index].u16ChannelAquState = FC_PHONE_CHANNELAQUISITION_STATE_IDLE;
      tChannelAquAttr[u16Index].bPendingReleaseRequest = FALSE;
      tChannelAquAttr[u16Index].bPendingGetRequest = FALSE;
   }

   m_u8GetRequestCounter = 0;

   vPrintChannelAttributes();
   ETG_TRACE_USR4(("fc_phone_tclAudioManager::vResetAllChannelAttributes Exit"));
}

/*******************************************************************************
 *
 * FUNCTION: vSetChannelAttributesForGetRequest
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclAudioManager::vSetChannelAttributesForGetRequest(tU16 u16AudioChannelId)
{
   ETG_TRACE_USR4(("fc_phone_tclAudioManager::vSetChannelAttributesForGetRequest u16AudioChannelId: %d", u16AudioChannelId));

   if(u16AudioChannelId >= FC_PHONE_NUM_OFCHANNELS)
   {
      ETG_TRACE_USR4(("Received INVALID u16AudioChannelId"));
      return;
   }
//NCG3D-142568 : Channel allocation request would be made if it is the first request.
   tChannelAquAttr[u16AudioChannelId].u16ChannelAquCounter++;

   if(FC_PHONE_AUDIOMANAGER_FIRST_GETREQUEST == tChannelAquAttr[u16AudioChannelId].u16ChannelAquCounter)
   {
      if(FC_PHONE_AUDIOCHANNEL_PHONE == u16AudioChannelId)
      {
         ++m_u8GetRequestCounter;
         ETG_TRACE_USR4((" m_u8GetRequestCounter for LC_Phone = %u ", m_u8GetRequestCounter));
      }

      if (FC_PHONE_CHANNELAQUISITION_STATE_IDLE != tChannelAquAttr[u16AudioChannelId].u16ChannelAquState)
      {
         ETG_TRACE_USR4(("Already release for this channel is in process, Get Request will be handled in ReleaseRequestAck"));
         tChannelAquAttr[u16AudioChannelId].bPendingGetRequest = TRUE;
      }
      else
      {
         tChannelAquAttr[u16AudioChannelId].u16ChannelAquState = FC_PHONE_CHANNELAQUISITION_STATE_PROCESSING_GET;

         bool bARLRootRequest =
               (fc_phone_tclApp::m_poMainAppInstance)->m_poPhoneAudioRouting->vTriggerPhoneAudioAllocation(
                     u16AudioChannelId);

         if (bARLRootRequest)
         {
            ETG_TRACE_USR4((" PhoneAudioAllocation Request is posted successfully "));
         }
         else
         {
            ETG_TRACE_USR4((" PhoneAudioAllocation Request posting failed "));
            s32GetRequestAck(u16AudioChannelId, FC_PHONE_AUDIOCHANNEL_DENIED);
         }
      }
   }
   for(tU16 u16Index = 0x00; u16Index < FC_PHONE_AUDIOCHANNEL_LIMIT; u16Index++)
   {
      if((FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE == m_rAudioCallBackMap[u16Index].u16AudioChannelId))
      {
         m_rAudioCallBackMap[u16Index].u16AudioChannelId = FC_PHONE_AUDIOCHANNEL_PHONE;
         break;
      }
   }
   vPrintChannelAttributes();
}

/*******************************************************************************
 *
 * FUNCTION: vUpdateAcquisitionCounter
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclAudioManager::vUpdateAcquisitionCounter()
{
   ETG_TRACE_USR4(("  fc_phone_tclAudioManager::vUpdateAcquisitionCounter entered "));

   //using channel ID as Index for updating the Attributes of corresponding channel.
   //As Channel Ids are 0,1 and 2

   if (1 == fc_phone_tclCallManager::m_u16Count)
   {
      ETG_TRACE_USR4(("Updating u16ChannelAquCounter as 1..."));
      tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_PHONE].u16ChannelAquCounter
            = 1;
   }
   else if (2 == fc_phone_tclCallManager::m_u16Count)
   {
      ETG_TRACE_USR4(("Updating u16ChannelAquCounter as 2..."));
      tChannelAquAttr[FC_PHONE_AUDIOCHANNEL_PHONE].u16ChannelAquCounter
            = 2;
   }

   vPrintChannelAttributes();
}

/*******************************************************************************
 *
 * FUNCTION: vGetChannelStatus
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclAudioManager::vGetChannelStatus(const tU16 u16AudioChannelId,tU8& u8ChannelStatus)
{
   ETG_TRACE_USR4(("fc_phone_tclAudioManager::vGetChannelStatus"));
   u8ChannelStatus = EN_SPEECH_CHNL_NOTACTIVE;
   //check ,Valid Channel Requested
   if((FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE != u16AudioChannelId)
         &&(FC_PHONE_AUDIOCHANNEL_PHONE != u16AudioChannelId)
         &&(FC_PHONE_AUDIOCHANNEL_ALERTTONE != u16AudioChannelId)
         && (FC_PHONE_AUDIOCHANNEL_SPEECH != u16AudioChannelId)
   )
   {
      ETG_TRACE_USR4(("  Invalid u16AudioChannelId :%u", u16AudioChannelId));
      return;
   }

   ETG_TRACE_USR4((" u16ChannelAquState : %d ", tChannelAquAttr[u16AudioChannelId].u16ChannelAquState));

   if( FC_PHONE_CHANNELAQUISITION_STATE_PROCESSING_GET == tChannelAquAttr[u16AudioChannelId].u16ChannelAquState )
   {
      u8ChannelStatus = EN_SPEECH_CHNL_GET_PROGRESS;
   }
   else if( FC_PHONE_CHANNELAQUISITION_STATE_PROCESSING_RELEASE == tChannelAquAttr[u16AudioChannelId].u16ChannelAquState )
   {
      u8ChannelStatus = EN_SPEECH_CHNL_RELEASE_PROGRESS;
   }
   else
   {
      ETG_TRACE_USR4((" AudioChannelId is IDLE"));
   }
}
