/************************************************************************
 * FILE:         Phone_DefSetHandler.h
 * PROJECT:
 * SW-COMPONENT:
 *----------------------------------------------------------------------
 *
 * DESCRIPTION: Example implementation of a system set listener.
 * This handler sets the demodata value used for IncreaseDecrease demo
 * back to default on a DefSet request.
 * Take this class as a template for your own listener classes and
 * adjust the parts marked with
 * // +++ comment +++
 *
 * - ccademo_diaghandler.x              => // +++ Comment +++
 * - ccademo_IncreaseDecreaseHandler.x  => // +++ Comment +++
 * - Phone_DefSetHandler.x            => // +++ Comment +++
 * - ccademo_main.x                     => // +++diaglib+++ comment
 * - ccademo_msgtypes.h                 => // +++diaglib+++ comment
 *
 *----------------------------------------------------------------------
 * COPYRIGHT:    (c) 2008 Robert Bosch GmbH, Hildesheim
 * HISTORY:
 * Date      | Author                 | Modification
 * 02.10.08  | CM-AI/PJ-FO45 Evers    | Inital version
 * 06.06.11  | TMS Plischke           | Demoserver for GMG2
 *
 *************************************************************************/
// +++ Remove feature switch when not needed. Used to turn diaglib support on/off +++
// +++ Specify what part of diaglib you need in this file +++

#include "../Configuration/Phone/Phone_FeatureConfig.h"

#ifdef CONNECTIVITY_DIAGNOSIS_GM
/* Due to non availability of diagnosis
		library, this section will be compiled out for JAC  */
#define DIAGLIB_INCLUDE_SYSTEM_SET
#include "diaglib_if.h"

#include "FC_Phone_main.h"
#include "FC_Phone_clienthandler_DefSetHandler.h"
#include "FC_Phone_service_Telephone.h"
#include "Audio/FC_Phone_ECNR.h"

/* Needed for Trace */
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"



#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define  ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONE_APPLICATION
#include "trcGenProj/Header/FC_Phone_clienthandler_DefSetHandler.cpp.trc.h"
#endif


Phone_DefSetHandler::Phone_DefSetHandler(diaglib::tclServiceDiaglib* pDiaglibService): mpDiaglibService(pDiaglibService)
{
   ETG_TRACE_USR4(("ENTER Phone_DefSetHandler::vOnSystemSet =>DefSetHandler"));
   if(mpDiaglibService)
   {
      ETG_TRACE_USR4(("!!! Phone_DefSetHandler::vOnSystemSet =>vRegisterListener SYSSETID"));
      mpDiaglibService->poGetSysSet()->vRegisterListener(SYSSETID_PHON, this);
      mpDiaglibService->poGetSysSet()->vRegisterListener(SYSSETID_ALL_COMP, this);
   }

}

Phone_DefSetHandler::~Phone_DefSetHandler()
{
   mpDiaglibService = NULLPTR;
}

tU32 Phone_DefSetHandler::vOnSystemSet   ( 
      tU32 u32SystemSetID,
      diaglib::tenSystemSetType u32SystemSetType,
      diaglib::tContext MsgContext
)
{
   ETG_TRACE_USR4(("--> Phone_DefSetHandler::vOnSystemSet"));

   tU32 u32Return = diaglib::U32_DIAGLIB_RETURN_OK;
   diaglib::tclParameterVector oTmpVec;
   diaglib::tenSystemSetResult oSysSetResult = diaglib::EN_SYSTEMSET_OK; // default response succefully
   tBool bResetPrivateData = FALSE;
   tBool bResetRadioSetting = FALSE;

   // +++ Set values back to default, then send positive response +++

   switch(u32SystemSetID)
   {
      case SYSSETID_ALL_COMP:
      {
         ETG_TRACE_USR4(("--- Phone_DefSetHandler::vOnSystemSet => SYSSETID_ALL_COMP"));
         switch(u32SystemSetType)
         {
            case diaglib::EN_TYPE_DEFAULT_TEF:
            {
               ETG_TRACE_USR4(("--- Phone_DefSetHandler::vOnSystemSet => EN_TYPE_DEFAULT_TEF"));
               bResetPrivateData = TRUE;
               bResetRadioSetting = TRUE;
               break;
            }// case EN_TYPE_DEFAULT_TEF:

            case diaglib::EN_TYPE_DEFAULT_HMI:
            {
               ETG_TRACE_USR4(("--- Phone_DefSetHandler::vOnSystemSet => EN_TYPE_DEFAULT_HMI"));
               bResetPrivateData = TRUE;
               bResetRadioSetting = TRUE;
               break;
            }// case EN_TYPE_DEFAULT_HMI:

            // RTC- 681736 - SMART
            case diaglib::EN_TYPE_DEFAULT_CUSTOMER:
            {
               ETG_TRACE_USR4(("--- Phone_DefSetHandler::vOnSystemSet => EN_TYPE_DEFAULT_CUSTOMER"));
               bResetPrivateData = TRUE;
               bResetRadioSetting = TRUE;
               break;
            }// case EN_TYPE_DEFAULT_CUSTOMER:

            case diaglib::EN_TYPE_CODING: // Customer EOL
            {
               ETG_TRACE_USR4(("--- Phone_DefSetHandler::vOnSystemSet => EN_TYPE_CODING"));
               bResetPrivateData = TRUE;
               bResetRadioSetting = TRUE;
               break;
            }// case EN_TYPE_CODING:

            default:
            {
               ETG_TRACE_ERR(("!!! Phone_DefSetHandler::vOnSystemSet => ERROR: not supportet Systemset request"));
               oSysSetResult = diaglib::EN_SYSTEMSET_NOT_OK;

               diaglib::trParameter paramError;
               paramError.enType         = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
               paramError.u8Value        = diaglib::EN_ERROR_INCOMPATIBLE_PARAMETER_SIGNATURE;
               oTmpVec.push_back(paramError);
               break;
            }// default:
         }// switch(u32SystemSetType)
         break;
      }// case SYSSETID_ALL_COMP:

      case SYSSETID_PHON: // single Group
      {
         ETG_TRACE_USR4(("--- Phone_DefSetHandler::vOnSystemSet => SYSSETID_PHON"));
         switch(u32SystemSetType)
         {
            case diaglib::EN_TYPE_DEFAULT_TEF:
            {
               ETG_TRACE_USR4(("--- Phone_DefSetHandler::vOnSystemSet => EN_TYPE_DEFAULT_TEF"));
               bResetPrivateData = TRUE;
               bResetRadioSetting = TRUE;
               break;
            }// case EN_TYPE_DEFAULT_TEF:

            case diaglib::EN_TYPE_DEFAULT_HMI:
            {
               ETG_TRACE_USR4(("--- Phone_DefSetHandler::vOnSystemSet => EN_TYPE_DEFAULT_HMI"));
               bResetPrivateData = TRUE;
               bResetRadioSetting = TRUE;
               break;
            }// case EN_TYPE_DEFAULT_HMI:

            // RTC- 681736 - SMART
            case diaglib::EN_TYPE_DEFAULT_CUSTOMER:
            {
               ETG_TRACE_USR4(("--- Phone_DefSetHandler::vOnSystemSet => EN_TYPE_DEFAULT_CUSTOMER"));
               bResetPrivateData = TRUE;
               bResetRadioSetting = TRUE;
               break;
            }// case EN_TYPE_DEFAULT_CUSTOMER:

            case diaglib::EN_TYPE_CODING: // Customer EOL
            {
               ETG_TRACE_USR4(("--- Phone_DefSetHandler::vOnSystemSet => EN_TYPE_CODING"));
               bResetPrivateData = TRUE;
               bResetRadioSetting = TRUE;
               break;
            }// case EN_TYPE_CODING:

            default:
            {
               ETG_TRACE_ERR(("!!! Phone_DefSetHandler::vOnSystemSet => ERROR: not supportet Systemset request"));
               oSysSetResult = diaglib::EN_SYSTEMSET_NOT_OK;

               diaglib::trParameter paramError;
               paramError.enType         = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
               paramError.u8Value        = diaglib::EN_ERROR_INCOMPATIBLE_PARAMETER_SIGNATURE;
               oTmpVec.push_back(paramError);
               break;
            }// default:
         }// switch(u32SystemSetType)
         break;
      }// case SYSSETID_PHON:

      default:
      {
         ETG_TRACE_ERR(("!!! Phone_DefSetHandler::vOnSystemSet => ERROR: not supportet ID"));
         oSysSetResult = diaglib::EN_SYSTEMSET_NOT_OK;

         diaglib::trParameter paramError;
         paramError.enType         = static_cast<diaglib::tenParameterType>(diaglib::EN_PARAMETER_TYPE_U8ERROR);
         paramError.u8Value        = diaglib::EN_ERROR_ID_NOT_SUPPORTED;
         oTmpVec.push_back(paramError);
         break;
      }// default:
   }// switch(u32SystemSetID)

   if(bResetPrivateData)
   {
      ETG_TRACE_USR4(("!!! Phone_DefSetHandler::vOnSystemSet =>PHONE DOESNT RESET PRIVATE DATA  "));
   }
   if(bResetRadioSetting)
   {
      ETG_TRACE_USR4(("!!! Phone_DefSetHandler::vOnSystemSet =>PHONE DOESNT RESET RADIO SETTINGS "));
   }

   // send Data back to Server
   if(mpDiaglibService)
   {
      if(TRUE == mpDiaglibService->poGetSysSet()->bSendSystemSetResult(  oSysSetResult, 
            oTmpVec,
            MsgContext) )
      {
         ETG_TRACE_USR4(("--- Phone_DefSetHandler::vOnSystemSet => bSendSystemSetResult successfull"));
      }
      else
      {
         ETG_TRACE_ERR(("!!! Phone_DefSetHandler::vOnSystemSet => ERROR : bSendSystemSetResult failed"));
      }
   }
   else
   {
      ETG_TRACE_ERR(("!!! Phone_DefSetHandler::vOnSystemSet => ERROR : mpDiaglibService == OSAL_NULL"));
   }

   ETG_TRACE_USR3(("<-- Phone_DefSetHandler::vOnSystemSet(%x)",u32Return));
   return u32Return;
}

tU32 Phone_DefSetHandler::vOnSystemSetFinished   ( 
      tU32 u32SystemSetID,
      diaglib::tenSystemSetType u32SystemSetType,
      diaglib::tContext MsgContext
)
{
   ETG_TRACE_USR4(("--> Phone_DefSetHandler::vOnSystemSetFinished"));
   // +++ Reallow access to values that needed default setting, then send positive response +++
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32SystemSetID );
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32SystemSetType );
   /*
      Respond does not contain any data and is just an acknowledgement.
    */

   if(mpDiaglibService)
   {
      if(TRUE == mpDiaglibService->poGetSysSet()->bAcknowledgeSystemSetFinished(MsgContext))
      {
         // +++ Trace info +++
         ETG_TRACE_USR4(("mpDiaglibService->poGetSysSet()->bAcknowledgeSystemSetFinished= TRUE "));
      }
      else
      {
         // +++ Trace error +++
         ETG_TRACE_ERR(("Error "));
      }
   }
   else
   {
      // +++ Trace error +++
      ETG_TRACE_ERR(("Error "));
   }

   ETG_TRACE_USR4(("<-- Phone_DefSetHandler::vOnSystemSetFinished"));
   return diaglib::U32_DIAGLIB_RETURN_OK;
}

tU32 Phone_DefSetHandler::vOnSystemSetPrepare   ( 
      tU32 u32SystemSetID,
      diaglib::tenSystemSetType u32SystemSetType,
      diaglib::tContext MsgContext
)
{
   ETG_TRACE_USR4(("--> Phone_DefSetHandler::vOnSystemSetPrepare"));
   // +++ Stop access to values that need default setting, then send positive response +++
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32SystemSetID );
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32SystemSetType );
   /*
      Response can be one of the following:
      EN_SYSTEMSET_OK            -> Empty error vector

      Actually the diaglib already checks if ID is supported, so we don't need
      to do it here again.
    */
   diaglib::tclParameterVector oTmpVec;


   if(mpDiaglibService)
   {
      if(TRUE == mpDiaglibService->poGetSysSet()->bSendSystemSetResult(diaglib::EN_SYSTEMSET_OK,
            oTmpVec,
            MsgContext) )
      {
         // +++ Trace info +++
         ETG_TRACE_USR4(("mpDiaglibService->poGetSysSet()->bSendSystemSetResult= TRUE "));
      }
      else
      {
         // +++ Trace error +++
         ETG_TRACE_ERR(("Error "));
      }
   }
   else
   {
      // +++ Trace error +++
      ETG_TRACE_ERR(("Error "));
   }
   ETG_TRACE_USR3(("<-- Phone_DefSetHandler::vOnSystemSetPrepare"));
   return diaglib::U32_DIAGLIB_RETURN_OK;
}

tU32 Phone_DefSetHandler::vOnSystemSetCheck ( 
      tU32 u32SystemSetID,
      diaglib::tenSystemSetType u32SystemSetType,
      diaglib::tContext MsgContext
)
{
   ETG_TRACE_USR4(("--> Phone_DefSetHandler::vOnSystemSetCheck"));
   // +++ Check if all values are set to default. +++
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32SystemSetID );
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED( u32SystemSetType );
   /*
      Response can be one of the following:
      EN_SYSTEMSET_OK            -> Empty error vector
      EN_SYSTEMSET_NOT_OK        -> List of wrong values

      diaglib already checks if ID is supported, so we don't need
      to do it here again.
    */
   diaglib::tclParameterVector oTmpVec;

   if(mpDiaglibService)
   {
      if(TRUE == mpDiaglibService->poGetSysSet()->bSendSystemSetCheckResult(diaglib::EN_SYSTEMSET_OK,
            oTmpVec,
            MsgContext) )
      {
         // +++ Trace info +++
         ETG_TRACE_USR4(("mpDiaglibService->poGetSysSet()->bSendSystemSetCheckResult= TRUE "));
      }
      else
      {
         // +++ Trace error +++
         ETG_TRACE_ERR(("Error "));
      }
   }
   else
   {
      // +++ Trace error +++
      ETG_TRACE_ERR(("Error "));
   }
   ETG_TRACE_USR3(("<-- Phone_DefSetHandler::vOnSystemSetCheck"));
   return diaglib::U32_DIAGLIB_RETURN_OK;
}
#endif /* CONNECTIVITY_DIAGNOSIS_GM */
