/*******************************************************************************
 *
 * FILE:          FC_Phone_clienthandler_BTSettings.cpp
 *
 * SW-COMPONENT:  FC_Phone Client Handler for Blue Tooth
 *
 * PROJECT:
 *
 * DESCRIPTION:   Client-handler for Blue Tooth Interface.
 *
 * AUTHOR:
 *
 * COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim
 *
 *******************************************************************************/
#include "FC_Phone_main.h"
#include "FC_Phone_clienthandler_BTSettings.h"
#include "FC_Phone_clienthandler_PhoneBook.h"
#include "FC_Phone_service_Telephone.h"
#include "Interface/FC_Phone_DBusInterface.h"
#include "HelperClasses/FC_Phone_AudioManager.h"
#include "FC_Phone_clienthandler_SystemState.h"
#ifdef PHONE_WAITINGMODE
#include "WaitingMode/FC_Phone_WaitingMode.h"
#endif

#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
//Include FI interface of Blue Tooth Settings
#define MOST_FI_S_IMPORT_INTERFACE_MOST_BTSETFI_TYPES
#define MOST_FI_S_IMPORT_INTERFACE_MOST_BTSETFI_FUNCTIONIDS
#define MOST_FI_S_IMPORT_INTERFACE_MOST_BTSETFI_ERRORCODES
#define MOST_FI_S_IMPORT_INTERFACE_MOST_BTSETFI_SERVICEINFO


#include "conn_most_fi_if.h"
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_PHONE_APPLICATION
#include "trcGenProj/Header/FC_Phone_clienthandler_BTSettings.cpp.trc.h"
#endif

#define INVALID_RINGTONE 0xFFFF
#define NO_RINGTONE_ID 0xFF
#define NARROW_BAND_SPEECH 1
#define WIDE_BAND_SPEECH 2

/******************************************************************************/
/*                         DEFINES                          */
/******************************************************************************/
/******************************************************************************/
/*                         CCA MESSAGE MAP                        */
/******************************************************************************/
BEGIN_MSG_MAP(fc_phone_tclClientHandler_BTSettings, ahl_tclBaseWork)

ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_CHANGEDEVICESTATE, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT,vHandle_ChangeDeviceState_MethodResult)
ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_CHANGEDEVICESTATE, AMT_C_U8_CCAMSG_OPCODE_ERROR,vHandle_ChangeDeviceState_MethodError)
ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_REQUESTAVAILABLERINGTONESLIST, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT,vHandle_RequestAvailableRingtonesList_MethodResult)
ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_REQUESTAVAILABLERINGTONESLIST, AMT_C_U8_CCAMSG_OPCODE_ERROR,vHandle_RequestAvailableRingtonesList_MethodError)
ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_PLAYRINGTONE, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT,vHandle_PlayRingtone_MethodResult)
ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_PLAYRINGTONE, AMT_C_U8_CCAMSG_OPCODE_ERROR,vHandle_PlayRingtone_MethodError)
ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_STOPRINGTONE, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT,vHandle_RequestStopRingtone_MethodResult)
ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_STOPRINGTONE, AMT_C_U8_CCAMSG_OPCODE_ERROR,vHandle_RequestStopRingtone_MethodError)

ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_DEVICELISTEXTENDED, AMT_C_U8_CCAMSG_OPCODE_STATUS , vHandle_DeviceListExtended_Status)
ON_MESSAGE_SVCDATA( MOST_BTSETFI_C_U16_VEHICLEBTADDRESS, AMT_C_U8_CCAMSG_OPCODE_STATUS , vHandle_VehicleBtAddress_Status)
ON_MESSAGE_SVCDATA( MOST_BTSETFI_C_U16_VEHICLECONFIGURATION, AMT_C_U8_CCAMSG_OPCODE_STATUS , vHandle_VehicleConfiguration_Status)

ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_SETPRIMARY, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT,vHandle_SetPrimary_MethodResult)
ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_SETPRIMARY, AMT_C_U8_CCAMSG_OPCODE_ERROR,vHandle_SetPrimary_MethodError)

ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_SETBTPROFILESUSAGE, AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vHandle_SetBTProfileUsage_MethodResult)
ON_MESSAGE_SVCDATA(MOST_BTSETFI_C_U16_SETBTPROFILESUSAGE, AMT_C_U8_CCAMSG_OPCODE_ERROR, vHandle_SetBTProfileUsage_MethodError)

END_MSG_MAP()

/*******************************************************************************
 *
 * FUNCTION: fc_phone_tclClientHandler_BTSettings::
 *             fc_phone_tclClientHandler_BTSettings(fc_phone_tclApp* poMainAppl)
 *
 * DESCRIPTION: Constructor.
 *
 *              Create an object of the base class
 *              ahl_tclBaseOneThreadClientHandler with a pointer to this
 *              application, the to be used service identifier and the service
 *              version as parameters.
 *
 * PARAMETER: [IN] poMainAppl = Pointer to the object of this application.
 *
 * RETURNVALUE: None.
 *
 ******************************************************************************/
fc_phone_tclClientHandler_BTSettings::fc_phone_tclClientHandler_BTSettings(fc_phone_tclApp* poMainAppl)
: ahl_tclBaseOneThreadClientHandler(
      /* Application Pointer          */ poMainAppl,
      /* ID of used Service           */ CCA_C_U16_SRV_FB_BLUETOOTHSETTINGS,
      /* MajorVersion of used Service */ FC_PHONE_SERVICE_BLUETOOTHSETTINGS_FI_MAJOR_VERSION,
      /* MinorVersion of used Service */ FC_PHONE_SERVICE_BLUETOOTHSETTINGS_FI_MINOR_VERSION)
{
   ETG_TRACE_USR4(("  fc_phone_tclClientHandler_BTSettings entered. "));

   vAddAutoRegisterForProperty(MOST_BTSETFI_C_U16_VEHICLEBTADDRESS);
   vAddAutoRegisterForProperty(MOST_BTSETFI_C_U16_VEHICLECONFIGURATION);
   vAddAutoRegisterForProperty(MOST_BTSETFI_C_U16_DEVICELISTEXTENDED);

   // m_ActiveDeviceBD_Address[0] = OSAL_NULL;
   //m_u8DeviceHandle = FC_PHONE_DEVICEHANDLE_UNKNOWN ;
   //m_DeviceName[0] = 0;
   m_TelService_instance = NULLPTR ;
   m_bOutGoingSourceChange = FALSE;
   m_bCallSwitchStatusOnIncoming = FALSE;
   m_bInBandRingtone = FALSE;

   //connected device status : default
   for(tU16 u16index = 0 ; u16index< FC_PHONE_MAX_DEVICECONNECTION; u16index++)
   {
      tConnectedDeviceList[u16index].u8BDAddressId=INVALID_BDADDRESS_ID;
      memset(tConnectedDeviceList[u16index].DeviceName,0,sizeof(tConnectedDeviceList[u16index].DeviceName));
      memset(tConnectedDeviceList[u16index].sVoiceMailNumber,0,sizeof(tConnectedDeviceList[u16index].sVoiceMailNumber));
      tConnectedDeviceList[u16index].u8DeviceHandle = FC_PHONE_DEVICEHANDLE_UNKNOWN ;
      tConnectedDeviceList[u16index].bOutgoingSourceStatus = FALSE ;
      tConnectedDeviceList[u16index].bSlotFree = TRUE ;
      tConnectedDeviceList[u16index].u8SCOConnectionStatus = FC_PHONE_DBUS_SCO_CONN_STATUS_UNKNOWN;
      tConnectedDeviceList[u16index].u8Codec = NARROW_BAND_SPEECH;
      tConnectedDeviceList[u16index].enDisconnectionStatus = DISCONNECTION_IDLE;
      tConnectedDeviceList[u16index].bIsAppleDevice = FALSE;
   }

   m_u8RingToneDeviceAddrId = INVALID_BDADDRESS_ID;
   m_bSuppressRingtone = FALSE; // Task CMG3G-9536

   m_poAudioManager = NULLPTR;
}
/*******************************************************************************
 *
 * FUNCTION: fc_phone_tclClientHandler_BTSettings::
 *             ~fc_phone_tclClientHandler_BTSettings(tVoid)
 *
 * DESCRIPTION: Destructor.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
fc_phone_tclClientHandler_BTSettings::
~fc_phone_tclClientHandler_BTSettings(tVoid)
{
   ETG_TRACE_USR4(("~fc_phone_tclClientHandler_BTSettings entered."));
   m_TelService_instance = NULLPTR;
   m_bInBandRingtone = FALSE;

   m_poAudioManager = NULLPTR;

}


/*******************************************************************************
 *
 * FUNCTION: tVoid fc_phone_tclClientHandler_BTSettings::vOnServiceAvailable()
 *
 * DESCRIPTION: This function is called by the CCA framework when the service
 *              this client-handler has registered for has become available.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************
 * Overrides method ahl_tclBaseOneThreadClientHandler::vOnServiceAvailable().
 *******************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vOnServiceAvailable()
{
   ETG_TRACE_USR4((" vOnServiceAvailable Getting the DeviceConnection Property     "));

}
/*******************************************************************************
 *
 * FUNCTION: tChar* fc_phone_tclClientHandler_BTSettings::pGet_ActiveDevice_BDAddressId
 *
 * DESCRIPTION: This function provides active BT Device address
 *
 *
 * PARAMETER:
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tU8 fc_phone_tclClientHandler_BTSettings::pGet_ActiveDevice_BDAddressId()
{
   ETG_TRACE_USR4((" fc_phone_tclClientHandler_BTSettings::pGet_ActiveDevice_BDAddressId entered."));
   for(tU16 u16Index = 0 ;u16Index <FC_PHONE_MAX_DEVICECONNECTION ;u16Index++)
   {
      if(TRUE == tConnectedDeviceList[u16Index].bOutgoingSourceStatus)
      {

         return tConnectedDeviceList[u16Index].u8BDAddressId;

      }
   }

   ETG_TRACE_ERR((" WARNING :: DeviceList do not have any deviceList entry with outgoing src enabled "));

   vPrintDeviceList();

   return INVALID_BDADDRESS_ID ;
}

/*******************************************************************************
 *
 * FUNCTION: tChar* fc_phone_tclClientHandler_BTSettings::bIsOutGoingDevicePresent
 *
 * DESCRIPTION: To find if any outgoing source is present
 *
 * PARAMETER:
 *
 * RETURNVALUE: True - If outgoing device is present.
 *              False - If no outgoing device is present.
 *
 ********************************************************************************/
tBool fc_phone_tclClientHandler_BTSettings::bIsOutGoingDevicePresent()
{
   ETG_TRACE_USR4(("fc_phone_tclClientHandler_BTSettings::bIsOutGoingDevicePresent."));
   tBool bIsOutGoingSource = FALSE;

   for(tU16 u16Index = 0 ;u16Index <FC_PHONE_MAX_DEVICECONNECTION ;u16Index++)
   {
      if(FC_PHONE_DEVICEHANDLE_UNKNOWN != tConnectedDeviceList[u16Index].u8DeviceHandle)
      {
         ETG_TRACE_USR4(("HFP connected device is present"));
         bIsOutGoingSource = TRUE;
      }
   }

   return bIsOutGoingSource;
}

/*******************************************************************************
 *
 * FUNCTION: tVoid fc_phone_tclClientHandler_BTSettings::vOnServiceUnavailable()
 *
 * DESCRIPTION: This function is called by the CCA framework when the service
 *              this client-handler has registered for has become unavailable.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************
 * Overrides method ahl_tclBaseOneThreadClientHandler::vOnServiceUnavailable()
 *******************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vOnServiceUnavailable()
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclClientHandler_BTSettings::vOnServiceUnavailable."));
   //ETG_TRACE_USR4((" vOnServiceUnavailable entered. AppID = %s.",u16GetServerAppID()));
}

/*******************************************************************************
 *
 * FUNCTION: tVoid  fc_phone_tclClientHandler_BTSettings::vSetTelServiceInstance(fc_phone_tclService_Telephone*& rfpTelServiceInstance)
 *
 * DESCRIPTION: Stores reference to service
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid  fc_phone_tclClientHandler_BTSettings::vSetTelServiceInstance(fc_phone_tclService_Telephone*& rfpTelServiceInstance)
{
   ETG_TRACE_USR4((" ENTER fc_phone_tclClientHandler_BTSettings::vSetTelServiceInstance"));
   m_TelService_instance = rfpTelServiceInstance ;
}

/*******************************************************************************
 *
 * FUNCTION:   vHandle_ChangeDeviceState_MethodResult(..)
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *    IN:   poMessage        : This is the received message To client
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/

tVoid fc_phone_tclClientHandler_BTSettings::vHandle_ChangeDeviceState_MethodResult(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4((" vHandle_ChangeDeviceState_MethodResult entered."));
   fi_tclVisitorMessage oVisitorMsg( poMessage );
   // Create the (message related) FI data object
   most_btsetfi_tclMsgChangeDeviceStateMethodResult    o_ChangeDeviceStateMR ;
   if(oVisitorMsg.s32GetData(o_ChangeDeviceStateMR, FC_PHONE_SERVICE_BLUETOOTHSETTINGS_FI_MAJOR_VERSION) != OSAL_ERROR)
   {
      ETG_TRACE_USR1(("                    bReplaceDeviceRequired: %d", o_ChangeDeviceStateMR.bReplaceDeviceRequired));
      ETG_TRACE_USR1((" ConnectionResult:: %u",ETG_ENUM(TR_CONNECTIONRESULT,o_ChangeDeviceStateMR.e8ConnectionResult.enType)));
      if(o_ChangeDeviceStateMR.e8ConnectionResult.enType == most_fi_tcl_e8_BTSetConnectionResult::FI_EN_E8DEVICE_CONNECTED)
      {
         if(m_TelService_instance)
         {
            m_bCallSwitchStatusOnIncoming = !m_bCallSwitchStatusOnIncoming;
            m_TelService_instance->ChangeDeviceStateResponse();
         }
      }
      else
      {
         //Handle when BT sends other than device connected status
         // Call is accepted and the state remains waiting for BT ChangeDeviceStateResponse which it fails
         SetOutGoingSourceChangeStatus(FALSE);
         ETG_TRACE_ERR((" WARNING :: Source Change Request Doesn't Happen"));
         ETG_TRACE_USR1(("o_ChangeDeviceStateMR.bReplaceDeviceRequired :: %d ",o_ChangeDeviceStateMR.bReplaceDeviceRequired));
      }
   }
   o_ChangeDeviceStateMR.vDestroy();
}
/*******************************************************************************
 *
 * FUNCTION:   vHandle_ChangeDeviceState_MethodError(..)
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *    IN:   poMessage        : This is the received message To client
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vHandle_ChangeDeviceState_MethodError(amt_tclServiceData* poMessage)
{
   //If TODo Method error comes handle changing the variable to false
   (tVoid) poMessage;
   ETG_TRACE_USR4(("   MethodError:vHandle_ChangeDeviceState_MethodError  entered  "));
   SetOutGoingSourceChangeStatus(FALSE);
}
/*******************************************************************************
 *
 * FUNCTION:   u8GetDeviceHandle()
 *
 * DESCRIPTION: Return the device handle for the OutgoingSource device
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: device handle for the connected device.
 *
 ********************************************************************************/
tU8 fc_phone_tclClientHandler_BTSettings::u8GetOutgoingSourceDeviceHandle()
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclClientHandler_BTSettings::u8GetOutgoingSourceDeviceHandle"));
   for(tU16 u16Index = 0 ;u16Index <FC_PHONE_MAX_DEVICECONNECTION ;u16Index++)
   {
      if(TRUE == tConnectedDeviceList[u16Index].bOutgoingSourceStatus)
      {
         return tConnectedDeviceList[u16Index].u8DeviceHandle;
      }
   }
   ETG_TRACE_ERR((" WARNING :: DeviceList do not have any deviceList entry with outgoingsrc enabled "));
   //vPrintDeviceList();
   return FC_PHONE_DEVICEHANDLE_UNKNOWN ;
}
/*******************************************************************************
 *
 * FUNCTION:   RequestChangeDeviceState()
 *
 * DESCRIPTION: Return the device Name for the connected device
 *
 * PARAMETER:
 *
 * RETURNVALUE: Device Name for the connected device.
 *
 ********************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::SwitchOutgoingSource()
{
   //Write a function to get Device Handle corresponding to this device address
   ETG_TRACE_USR4((" ENTER fc_phone_tclClientHandler_BTSettings::SwitchOutgoingSource"));
   most_btsetfi_tclMsgChangeDeviceStateMethodStart o_ChangeDeviceStateMS;
   o_ChangeDeviceStateMS.e8ChangeDeviceStateAction.enType = most_fi_tcl_e8_BTSetChangeDeviceStateAction::FI_EN_E8SELECT;
   o_ChangeDeviceStateMS.u8DeviceHandle = u8GetDeviceHandleOfNonOutGoingSrcDevice();
   if(o_ChangeDeviceStateMS.u8DeviceHandle ==  FC_PHONE_NONOUTGOING_SRC_DEVICEHANDLE_NOTFOUND)
   {
      //Handle Invalid Device Handle Here
      ETG_TRACE_ERR(("INVALID DEVICE HANDLE"));
   }
   ETG_TRACE_USR1((" o_ChangeDeviceStateMS.u8DeviceHandle = %d     ",o_ChangeDeviceStateMS.u8DeviceHandle));
   fi_tclVisitorMessage oMsg(o_ChangeDeviceStateMS,FC_PHONE_SERVICE_BLUETOOTHSETTINGS_FI_MAJOR_VERSION);
   vInitServiceData(oMsg,                                    // ServiceDataMsg
         0,                                      // CmdCounter
         MOST_BTSETFI_C_U16_CHANGEDEVICESTATE,   // Function ID
         AMT_C_U8_CCAMSG_OPCODE_METHODSTART      // Opcode
   );
   // Post message
   ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMsg, TRUE);
   if(enResult != AIL_EN_N_NO_ERROR)
   {
      ETG_TRACE_ERR(("  enPostMessage failed for 'o_ChangeDeviceStateMS - METHODSTART' message, 'ail_tenCommunicationError' = %u      ", (tU32)enResult ));
   }
   o_ChangeDeviceStateMS.vDestroy();
}

/*******************************************************************************
 *
 * FUNCTION:   GetOutGoingSourceChangeStatus()
 *
 * DESCRIPTION: Return the  Outgoing Source Change Status
 *
 * PARAMETER:
 *
 * RETURNVALUE: Outgoing Source Change Status (TRUE or FALSE) for the requested device.
 *
 ********************************************************************************/
tBool fc_phone_tclClientHandler_BTSettings::GetOutGoingSourceChangeStatus()
{
   ETG_TRACE_USR4(("ENTER GetOutGoingSourceChangeStatus"));
   ETG_TRACE_USR3(("m_bOutGoingSourceChange= %d",m_bOutGoingSourceChange));
   return m_bOutGoingSourceChange;
}

/*******************************************************************************
 *
 * FUNCTION:   SetOutGoingSourceChangeStatus()
 *
 * DESCRIPTION: Return the  Outgoing Source Change Status
 *
 * PARAMETER:
 *
 * RETURNVALUE: Outgoing Source Change Status (TRUE or FALSE) for the requested device.
 *
 ********************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::SetOutGoingSourceChangeStatus(tBool SourceSwitchingStatus)
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclClientHandler_BTSettings::SetOutGoingSourceChangeStatus"));
   m_bOutGoingSourceChange  = SourceSwitchingStatus;
   ETG_TRACE_USR3(("SetOutGoingSourceChangeStatus = %d",m_bOutGoingSourceChange));
}
/*******************************************************************************
 *
 * FUNCTION:   u8GetBtAddressFromDeviceHandle()
 *
 * DESCRIPTION: Return the BTAddress handle for the Device Handle
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: device handle for the connected device.
 *
 ********************************************************************************/
tU8 fc_phone_tclClientHandler_BTSettings::u8GetBTAddressIdFromDeviceHandle(tU8 u8DeviceHandle)
{
   ETG_TRACE_USR4((" fc_phone_tclClientHandler_BTSettings::u8GetBTAddressFromDeviceHandle Input deviceHandle :: %d",u8DeviceHandle));

   for(tU16 u16Index = 0 ;u16Index <FC_PHONE_MAX_DEVICECONNECTION ;u16Index++)
   {
      if(u8DeviceHandle == tConnectedDeviceList[u16Index].u8DeviceHandle)
      {
         return tConnectedDeviceList[u16Index].u8BDAddressId;
      }
   }

   ETG_TRACE_ERR((" WARNING :: matching device handle is not found in deviceList Table "));

   vPrintDeviceList();

   return INVALID_BDADDRESS_ID;
}

/*******************************************************************************
 *
 * FUNCTION:   u8GetDeviceHandleFromBtAddressId()
 *
 * DESCRIPTION: Return the BTAddress handle for the Device Handle
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: device handle for the connected device.
 *
 ********************************************************************************/
tU8 fc_phone_tclClientHandler_BTSettings::u8GetDeviceHandleFromBtAddressId(tU8 deviceAddressId)
{
   ETG_TRACE_USR4((" Input deviceAddressId :: %d ",deviceAddressId));

   for(tU16 u16Index = 0 ;u16Index <FC_PHONE_MAX_DEVICECONNECTION ;u16Index++)
   {
      if(deviceAddressId==tConnectedDeviceList[u16Index].u8BDAddressId)
      {
         ETG_TRACE_USR4((" Input deviceHandle :: %d ",tConnectedDeviceList[u16Index].u8DeviceHandle));
         return tConnectedDeviceList[u16Index].u8DeviceHandle;
      }
   }

   ETG_TRACE_ERR((" WARNING :: device handle not found "));

   return FC_PHONE_DEVICEHANDLE_UNKNOWN;
}

/*******************************************************************************
 *
 * FUNCTION:   sGetDeviceNameFromBtAddressId()
 *
 * DESCRIPTION: Return the device Name of the given BD Address ID
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: device Name for the connected device.
 *
 ********************************************************************************/
const tChar* fc_phone_tclClientHandler_BTSettings::sGetDeviceNameFromBtAddressId(tU8 deviceAddressId)
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclClientHandler_BTSettings::sGetDeviceNameFromBtAddressId with deviceAddressId: %d", deviceAddressId));
   vPrintDeviceList();

   for(tU16 u16Index = 0 ;u16Index <FC_PHONE_MAX_DEVICECONNECTION ;u16Index++)
   {
      if(deviceAddressId == tConnectedDeviceList[u16Index].u8BDAddressId)
      {
         return tConnectedDeviceList[u16Index].DeviceName;
      }
   }

   ETG_TRACE_ERR((" WARNING :: DeviceList do not have any deviceList entry with the given deviceAddressId "));
   return NULLPTR ;
}


/*******************************************************************************
 *
 * FUNCTION:   vPrintDeviceList
 *
 * DESCRIPTION: Prints Device List
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: None
 *
 ********************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vPrintDeviceList()
{
   //connected device status : default
   ETG_TRACE_USR4(("ENTER  fc_phone_tclClientHandler_BTSettings::vPrintDeviceList"));
   for(tU16 u16index = 0 ; u16index< FC_PHONE_MAX_DEVICECONNECTION; u16index++)
   {
      ETG_TRACE_USR3(("Printing Device List for SLOT:: %d",u16index));

      ETG_TRACE_USR2(("DeviceBD_AddressId ::%d ",tConnectedDeviceList[u16index].u8BDAddressId));
      ETG_TRACE_USR2(("DeviceName :: %s ",tConnectedDeviceList[u16index].DeviceName));
      ETG_TRACE_USR2(("sVoiceMailNumber :: %s ",tConnectedDeviceList[u16index].sVoiceMailNumber));
      ETG_TRACE_USR2(("u8DeviceHandle ::%d ",tConnectedDeviceList[u16index].u8DeviceHandle));
      ETG_TRACE_USR2(("bOutgoingSourceStatus :: %d ",tConnectedDeviceList[u16index].bOutgoingSourceStatus));
      ETG_TRACE_USR2(("bSlotFree :: %d ",tConnectedDeviceList[u16index].bSlotFree));
      ETG_TRACE_USR2(("enDisconnectionStatus :: %d ",tConnectedDeviceList[u16index].enDisconnectionStatus));
   }
}

/*******************************************************************************
 *
 * FUNCTION:   tGetDevicelistInstanceFromDeviceHandle
 *
 * DESCRIPTION: Return the device list instance for corresponding devicehandle
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: Return the device list instance
 *
 ********************************************************************************/
FcphoneConnectedDeviceList* fc_phone_tclClientHandler_BTSettings::tGetDevicelistInstanceFromDeviceHandle(tU8 u8DeviceHandle)
{
   ETG_TRACE_USR4((" tGetDevicelistInstanceFromDeviceHandle  ENTER"));

   for(tU16 u16Index = 0 ;u16Index <FC_PHONE_MAX_DEVICECONNECTION ;u16Index++)
   {
      if(u8DeviceHandle == tConnectedDeviceList[u16Index].u8DeviceHandle)
      {
         ETG_TRACE_USR3((" Returning DeviceList instance for index ::%d ",u16Index));

         return &tConnectedDeviceList[u16Index];
      }
   }

   return NULLPTR ;
}

/*******************************************************************************
 *
 * FUNCTION:   tGetFreeDevicelistInstanceFromDeviceListTable
 *
 * DESCRIPTION: Return the device list instance for free deviceList Slot
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: Return the device list instance
 *
 ********************************************************************************/
FcphoneConnectedDeviceList* fc_phone_tclClientHandler_BTSettings::tGetFreeDevicelistInstanceFromDeviceListTable()
{
   ETG_TRACE_USR4((" tGetFreeDevicelistInstanceFromDeviceListTable  ENTER"));

   for(tU16 u16Index = 0 ;u16Index <FC_PHONE_MAX_DEVICECONNECTION ;u16Index++)
   {
      if(TRUE == tConnectedDeviceList[u16Index].bSlotFree)
      {
         ETG_TRACE_USR3((" Returning DeviceList instance for index ::%d ",u16Index));

         tConnectedDeviceList[u16Index].bSlotFree = FALSE ;
         return &tConnectedDeviceList[u16Index];
      }
   }

   return NULLPTR ;
}

/*******************************************************************************
 *
 * FUNCTION:   u8GetDeviceHandleOfNonOutGoingSrcDevice
 *
 * DESCRIPTION: Return the deviceHandle of non outgoing src device
 *
 * PARAMETER:
 *
 *
 * RETURNVALUE: Return the device list instance
 *
 ********************************************************************************/
tU8 fc_phone_tclClientHandler_BTSettings::u8GetDeviceHandleOfNonOutGoingSrcDevice()
{
   ETG_TRACE_USR4((" tGetFreeDevicelistInstanceFromDeviceListTable  ENTER"));

   for(tU16 u16Index = 0 ;u16Index <FC_PHONE_MAX_DEVICECONNECTION ;u16Index++)
   {
      if((FC_PHONE_DEVICEHANDLE_UNKNOWN != tConnectedDeviceList[u16Index].u8DeviceHandle)
            &&(FALSE == tConnectedDeviceList[u16Index].bOutgoingSourceStatus)
      )
      {
         ETG_TRACE_USR3((" Returning deviceHandle for index ::%d ",u16Index));

         return tConnectedDeviceList[u16Index].u8DeviceHandle;
      }
   }

   ETG_TRACE_ERR((" Warning :: Non Outgoing src device is not found"));

   return (tU8)FC_PHONE_NONOUTGOING_SRC_DEVICEHANDLE_NOTFOUND ;
}

/*******************************************************************************
 *
 * FUNCTION:   RequestAvailableRingtonesList()
 *
 * DESCRIPTION: Return the ringtone liste for the connected device
 *
 * PARAMETER: deviceAddress - device for which incoming call has come
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::RequestAvailableRingtonesList(tU8 u8DeviceHandle)
{
   //Write a function to get ringtone list corresponding to this device handle
   ETG_TRACE_USR4((" fc_phone_tclClientHandler_BTSettings::RequestAvailableRingtonesList ENTER"));
   most_btsetfi_tclMsgRequestAvailableRingtonesListMethodStart oAvailableRingtoneList;

   oAvailableRingtoneList.u8DeviceHandle = u8DeviceHandle;

   ETG_TRACE_USR1((" oAvailableRingtoneList.u8DeviceHandle = %d     ",oAvailableRingtoneList.u8DeviceHandle));
   fi_tclVisitorMessage oMsg(oAvailableRingtoneList,FC_PHONE_SERVICE_BLUETOOTHSETTINGS_FI_MAJOR_VERSION);
   vInitServiceData(oMsg,                                    // ServiceDataMsg
         0,                                      // CmdCounter
         MOST_BTSETFI_C_U16_REQUESTAVAILABLERINGTONESLIST,   // Function ID
         AMT_C_U8_CCAMSG_OPCODE_METHODSTART      // Opcode
   );
   // Post message
   ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMsg, TRUE);
   if(enResult != AIL_EN_N_NO_ERROR)
   {
      ETG_TRACE_ERR(("  enPostMessage() failed for 'oAvailableRingtoneList - METHODSTART' message, 'ail_tenCommunicationError' = %u      ", (tU32)enResult ));
   }
   oAvailableRingtoneList.vDestroy();
}

/*******************************************************************************
 *
 * FUNCTION:   vHandle_RequestAvailableRingtonesList_MethodResult
 *
 * DESCRIPTION: This is a handler function for RequestAvailableRingtonesList
 *              method Result
 *
 * PARAMETER: poMessage -This is the received message from a client
 *
 * RETURNVALUE: None
 *
 ********************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vHandle_RequestAvailableRingtonesList_MethodResult(amt_tclServiceData* poMessage)
{
   /* Extract the Call instance and other required info from poMessage */
   // Create a FI visitor message for the received CCA message
   ETG_TRACE_USR4((" fc_phone_tclClientHandler_BTSettings::vHandle_RequestAvailableRingtonesList_MethodResult ENTER"));
   fi_tclVisitorMessage oVisitorMsg( poMessage );
   most_btsetfi_tclMsgRequestAvailableRingtonesListMethodResult oRingtoneList;
   // Unmarshal the FI visitor message to the given FI data object.
   if ( oVisitorMsg.s32GetData(oRingtoneList, FC_PHONE_SERVICE_BLUETOOTHSETTINGS_FI_MAJOR_VERSION)
         != OSAL_ERROR)
   {
      ETG_TRACE_USR2((" m_bInBandRingtone = %d",m_bInBandRingtone));
      if(  0 != oRingtoneList.oRingtoneList.oItems.size())
      {
         //PSARCC30-1590. If system state is STANDBY, then ARL would not respond anything for our request. Hence, returning here itself with FALSE
         fc_phone_tclClientHandler_SystemState *p_ClientHdlerSystemState = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerSystemState;
         if(p_ClientHdlerSystemState->bIsSysteminStandByState())
         {
            (fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->s32GetRequestAck(FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE, FC_PHONE_AUDIOCHANNEL_DENIED);
            oRingtoneList.vDestroy();
            return;
         }

         for(tU8 u8Index = 0 ; u8Index < oRingtoneList.oRingtoneList.oItems.size();u8Index ++)
         {
            ETG_TRACE_USR1(("BTRingtoneListItem[%u]::u16RingtoneID :%u", u8Index, oRingtoneList.oRingtoneList.oItems[u8Index].u16RingtoneID));
            ETG_TRACE_USR1(("BTRingtoneListItem[%u]::sRingtoneName.szValue :%s", u8Index, oRingtoneList.oRingtoneList.oItems[u8Index].sRingtoneName.szValue));
            ETG_TRACE_USR1(("BTRingtoneListItem[%u]::bRingtoneAvailable :%d", u8Index, oRingtoneList.oRingtoneList.oItems[u8Index].bRingtoneAvailable));
            ETG_TRACE_USR1(("BTRingtoneListItem[%u]::bPlayRingtoneAvailable :%d", u8Index, oRingtoneList.oRingtoneList.oItems[u8Index].bPlayRingtoneAvailable));
            ETG_TRACE_USR1(("BTRingtoneListItem[%u]::bActiveRingtone :%d", u8Index, oRingtoneList.oRingtoneList.oItems[u8Index].bActiveRingtone));

            if ( (oRingtoneList.oRingtoneList.oItems[u8Index].bActiveRingtone == TRUE) &&
                  ( (0 == u8StringCompare(oRingtoneList.oRingtoneList.oItems[u8Index].sRingtoneName.szValue , "No Ringtone")) && oRingtoneList.oRingtoneList.oItems[u8Index].u16RingtoneID == NO_RINGTONE_ID))
            {
               m_bSuppressRingtone = TRUE;
            }
         }
         if(TRUE == m_bSuppressRingtone)
         {
            // CMG3G-9536
            // This is to suppress ringtone, when HMI setting is done, not to hear ringtone while receiving an incoming call.
            // By this, we would bypass vPlayRingtone. Hence no ringtone will be heard.
            ETG_TRACE_USR2((" Suppressing Ringtone"));
            (fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->s32GetRequestAck(FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE, FC_PHONE_AUDIOCHANNEL_GRANTED);
            oRingtoneList.vDestroy();
            return;
         }

         if ( (oRingtoneList.oRingtoneList.oItems[0].bActiveRingtone == TRUE) &&
               ( oRingtoneList.oRingtoneList.oItems[0].bRingtoneAvailable == TRUE) )
         {

            if (m_TelService_instance)
            {
               //JAC2-5552. RequestAvailableRingtonesList_MethodResult received after SLC disconnection(after receiving HFP Disconnection Event) or IDLE call status.
               if (m_TelService_instance->bCheckIdleCallStatus())
               {
                  ETG_TRACE_USR4((" Call status is IDLE or SLC disconnected. Hence not acquiring LC_Phone channel"));
                  (fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->s32GetRequestAck(FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE, FC_PHONE_AUDIOCHANNEL_DENIED);
               }
               else
               {
                  //Inband ringtone is enabled
                  m_bInBandRingtone = TRUE;
                  // When s32getrequest was called "m_poCallInstance->m_u8AudioChannel = FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE" is done
                  // necessary changes are done in Idle SM
                  //Reset IncomingRingtone channel attributes here and set attributes for Phone channel
                  (fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->vResetChannelAttributes(FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE);

                  //JAC2-5552. Setting the Phone channel to the call instance Ringing before requesting for LC_Phone channel
                  //from ClientHandler_BTSettings in case of In-Band Ringtone.
                  m_TelService_instance->vSetPhoneChannelforRingingCallInstance();

                  (fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->vSetChannelAttributesForGetRequest(FC_PHONE_AUDIOCHANNEL_PHONE);
               }
            }
            else
            {
               ETG_TRACE_USR4(("m_TelService_instance is NULL"));
            }
         }
         else
         {
            //System ringtone is enabled
            m_bInBandRingtone = FALSE;
            //GMMY16-13894 method result will be posted in play ringtone method result
            //JAC2-3831 Trigger PlayRingtone with INVALID RINGTONE ID for incoming call.
            vPlayRingtone(INVALID_RINGTONE,LC_INCOMING_RING_TONE_CHANNEL_ID);
         }

      }
      else
      {
         ETG_TRACE_USR4(("Available RingTone List Empty"));
      }
   }
   oRingtoneList.vDestroy();
}

/*******************************************************************************
 *
 * FUNCTION:   vHandle_RequestAvailableRingtonesList_MethodError
 *
 * DESCRIPTION: This is a handler function for RequestAvailableRingtonesList
 *              method Error
 *
 * PARAMETER: poMessage -This is the received message from a client
 *
 * RETURNVALUE: None
 *
 ********************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vHandle_RequestAvailableRingtonesList_MethodError(amt_tclServiceData* poMessage)
{
   (tVoid) poMessage;
   ETG_TRACE_USR4(("vHandle_RequestAvailableRingtonesList_MethodError entered"));
   (fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->s32GetRequestAck(FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE, FC_PHONE_AUDIOCHANNEL_DENIED);
}

/*******************************************************************************
 *
 * FUNCTION:   vUpdateRingToneDeviceAddrId
 *
 * DESCRIPTION: Update m_u8RingToneDeviceAddrId
 *
 * PARAMETER: u8DeviceHandle - device handle of device for which incoming call
 *                             has come.
 *
 *
 * RETURNVALUE: None
 *
 ********************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vUpdateRingToneDeviceAddrId(tU8 u8BDAddrId)
{
   ETG_TRACE_USR4(("ENTER fc_phone_tclClientHandler_BTSettings::vUpdateRingToneDeviceAddrId"));
   m_u8RingToneDeviceAddrId =  u8BDAddrId;
}

/*******************************************************************************
 *
 * FUNCTION:  vGetRingToneDeviceAddrId
 *
 * DESCRIPTION: Return m_u8RingToneDeviceAddrId
 *
 * PARAMETER: None
 *
 * RETURNVALUE: tU8 - ringtone device handle
 *
 ********************************************************************************/
tU8 fc_phone_tclClientHandler_BTSettings::vGetRingToneDeviceAddrId()
{
   ETG_TRACE_USR4((" ENTER fc_phone_tclClientHandler_BTSettings::vGetRingToneDeviceAddrId"));
   return m_u8RingToneDeviceAddrId;
}

/*******************************************************************************
 *
 * FUNCTION:   vIsInBandRingtone
 *
 * DESCRIPTION:
 *
 * PARAMETER: None
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tU16 fc_phone_tclClientHandler_BTSettings::vIsInBandRingtone()
{
   ETG_TRACE_USR4((" vIsInBandRingtone :: m_bInBandRingtone = %d     ",m_bInBandRingtone));
   return m_bInBandRingtone;
}

/*******************************************************************************
 *
 * FUNCTION:   vIsInBandRingtone
 *
 * DESCRIPTION:
 *
 * PARAMETER: None
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vUpdateInBandRingTone(tU16 isInBandRingtone)
{
   ETG_TRACE_USR4((" vUpdateInBandRingTone :: isInBandRingtone = %d     ",isInBandRingtone));
   m_bInBandRingtone = static_cast<tBool>(isInBandRingtone);
}

/*******************************************************************************
 *
 * FUNCTION:   pu8GetVoiceMailNumberFromDeviceName
 *
 * DESCRIPTION:
 *
 * PARAMETER: None
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
const tChar* fc_phone_tclClientHandler_BTSettings::pu8GetVoiceMailNumberFromDeviceName(tChar* pu8DeviceName)
{
   ETG_TRACE_USR4(("pu8GetVoiceMailNumberFromDeviceName :: pu8DeviceName  = %s",pu8DeviceName));

   for(tU8 u8Index2 = 0 ;u8Index2 < FC_PHONE_MAX_DEVICECONNECTION ; u8Index2++)
   {
      if(0 == u8StringCompare(pu8DeviceName , tConnectedDeviceList[u8Index2].DeviceName))
      {
         ETG_TRACE_USR3(("returning voicemail number  : %s",tConnectedDeviceList[u8Index2].sVoiceMailNumber));
         return  (tConnectedDeviceList[u8Index2].sVoiceMailNumber) ;
      }
   }

   ETG_TRACE_USR3(("No Match found for input  pu8DeviceName ::  = %s ",pu8DeviceName));

   vPrintDeviceList();
   return NULLPTR ;
}
/*******************************************************************************
 *
 * FUNCTION:   vGetSCOConnectionStatus
 *
 * DESCRIPTION:
 *
 * PARAMETER: None
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tU8 fc_phone_tclClientHandler_BTSettings::vGetSCOConnectionStatus(tString deviceAddress)
{
   tU8 u8index = 0;
   ETG_TRACE_USR4(("Device Address Matches and SCO Connection Status is updated ::  = %s ",deviceAddress));
   for(u8index = 0; u8index < FC_PHONE_MAX_DEVICECONNECTION; u8index++)
   {
      if(u8StringCompare(deviceAddress,fc_phone_tclDeviceList::getInstance()->u8GetBDAddressFromDeviceList(tConnectedDeviceList[u8index].u8BDAddressId)) == 0)
      {
         ETG_TRACE_USR3(("SCO Connection Status is updated ::  = %d ",tConnectedDeviceList[u8index].u8SCOConnectionStatus));
         return tConnectedDeviceList[u8index].u8SCOConnectionStatus;
      }
   }
   return FC_PHONE_SCOCONNEXNSTATUS_UNKNOWN;
}
/*******************************************************************************
 *
 * FUNCTION:   vUpdateSCOConnectionStatus
 *
 * DESCRIPTION:
 *
 * PARAMETER: None
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vUpdateSCOConnectionStatus(
      T_SCOConnectionStatus* SCOConexnStatus)
{
   ETG_TRACE_USR4(("fc_phone_tclClientHandler_BTSettings::vUpdateSCOConnectionStatus entered"));

   for (tU16 u16index = 0; u16index < FC_PHONE_MAX_DEVICECONNECTION; u16index++)
   {
      if (u8StringCompare(SCOConexnStatus->BD_Address,
            fc_phone_tclDeviceList::getInstance()->u8GetBDAddressFromDeviceList(
                  tConnectedDeviceList[u16index].u8BDAddressId)) == 0)
      {
         tConnectedDeviceList[u16index].u8SCOConnectionStatus = SCOConexnStatus->SCOConnectionStatus;
         tConnectedDeviceList[u16index].u8Codec = SCOConexnStatus->Codec;
      }
   }
}

/* Fix for the task CMG3G-3098: Audio adaptation for G3g FC_phone */
/*******************************************************************************
 *
 * FUNCTION:   vPlayRingtone
 *
 * DESCRIPTION:  Request to BT settings to start play vehicle ringtone
 *
 * PARAMETER: None
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vPlayRingtone(tU16 u16RingtoneId, tU8 u8LogicalAvChannel)
{
   ETG_TRACE_USR4(("fc_phone_tclClientHandler_BTSettings::vPlayRingtone entered"));
   most_btsetfi_tclMsgPlayRingtoneMethodStart oPlayRingtone;

   oPlayRingtone.u16RingtoneID = u16RingtoneId;
   oPlayRingtone.u8LogicalAVChannel = u8LogicalAvChannel;

   ETG_TRACE_USR1((" oPlayRingtone.u16RingtoneID = %d ", oPlayRingtone.u16RingtoneID));
   ETG_TRACE_USR1((" oPlayRingtone.u8LogicalAVChannel = %d ", oPlayRingtone.u8LogicalAVChannel));
   fi_tclVisitorMessage oMsg(oPlayRingtone, FC_PHONE_SERVICE_BLUETOOTHSETTINGS_FI_MAJOR_VERSION);
   vInitServiceData(oMsg, 0, MOST_BTSETFI_C_U16_PLAYRINGTONE, AMT_C_U8_CCAMSG_OPCODE_METHODSTART);
   ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMsg, TRUE);
   if(enResult != AIL_EN_N_NO_ERROR)
   {
      ETG_TRACE_ERR(("  enPostMessage() failed for 'oPlayRingtone - METHODSTART' message, 'ail_tenCommunicationError' = %u      ", (tU32)enResult ));
   }
   oPlayRingtone.vDestroy();
}
/*******************************************************************************
 *
 * FUNCTION:   vStopRingtone
 *
 * DESCRIPTION: Request to BT settings to stop play vehicle ringtone
 *
 * PARAMETER: None
 *
 * RETURNVALUE:
 *
 ********************************************************************************/

tVoid fc_phone_tclClientHandler_BTSettings::vStopRingtone()
{
   ETG_TRACE_USR4(("fc_phone_tclClientHandler_BTSettings::vStopRingtone entered"));
   if(bIsSuppressRingtoneEnabled())
   {
      m_bSuppressRingtone = FALSE;
      return;
   }
   most_btsetfi_tclMsgStopRingtoneMethodStart oStopRingtone;
   fi_tclVisitorMessage oMsg(oStopRingtone, FC_PHONE_SERVICE_BLUETOOTHSETTINGS_FI_MAJOR_VERSION);
   vInitServiceData(oMsg, 0, MOST_BTSETFI_C_U16_STOPRINGTONE, AMT_C_U8_CCAMSG_OPCODE_METHODSTART);
   ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMsg, TRUE);
   if(enResult != AIL_EN_N_NO_ERROR)
   {
      ETG_TRACE_ERR(("  enPostMessage() failed for 'oStopRingtone - METHODSTART' message, 'ail_tenCommunicationError' = %u      ", (tU32)enResult ));
   }
   oStopRingtone.vDestroy();
}
/* Fix for the task CMG3G-3098: Audio adaptation for G3g FC_phone */
/*******************************************************************************
 *
 * FUNCTION:   vHandle_PlayRingtone_MethodResult
 *
 * DESCRIPTION:  Send by BT settings once vehicle ringtone start playing
 *
 * PARAMETER: None
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vHandle_PlayRingtone_MethodResult(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("fc_phone_tclClientHandler_BTSettings::vHandle_PlayRingtone_MethodResult entered"));
   fi_tclVisitorMessage oVisitorMsg( poMessage );
   // Create the (message related) FI data object
   most_btsetfi_tclMsgPlayRingtoneMethodResult    oPlayRingtoneMR;

   ETG_TRACE_USR4(("vHandle_PlayRingtone_MethodResult ringtone is played successfully"));
   (fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->s32GetRequestAck(FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE, FC_PHONE_AUDIOCHANNEL_GRANTED);
   //if play ringtone MR is coming after call becomes idle and stop ringtone already request and acknowledged by BT (issue)
   //ringtone should stop playing
   if(m_TelService_instance)
   {
      if(TRUE == m_TelService_instance->bCheckIdleCallStatus())
      {
         ETG_TRACE_USR4(("Both call instance are IDLE release the ringtone.."));
         vStopRingtone();
      }
   }
   oPlayRingtoneMR.vDestroy();
}

/* Fix for the task CMG3G-3098: Audio adaptation for G3g FC_phone */
/*******************************************************************************
 *
 * FUNCTION:   vHandle_PlayRingtone_MethodError
 *
 * DESCRIPTION:
 *
 * PARAMETER: None
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vHandle_PlayRingtone_MethodError(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("fc_phone_tclClientHandler_BTSettings::vHandle_PlayRingtone_MethodError entered"));
   most_btsetfi_tclMsgPlayRingtoneError    oPlayRingtoneME;
   ETG_TRACE_USR4(("vHandle_PlayRingtone_MethodError entered Error code: %u", oPlayRingtoneME.e8ErrorCode.enType));
   (tVoid) poMessage;
   (fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->s32GetRequestAck(FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE, FC_PHONE_AUDIOCHANNEL_DENIED);
   oPlayRingtoneME.vDestroy();
}
/*******************************************************************************
 *
 * FUNCTION:   vHandle_RequestStopRingtone_MethodResult
 *
 * DESCRIPTION: Send by BT settings once vehicle ringtone stopped playing
 *
 * PARAMETER: None
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vHandle_RequestStopRingtone_MethodResult(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("fc_phone_tclClientHandler_BTSettings::vHandle_RequestStopRingtone_MethodResult entered"));
   fi_tclVisitorMessage oVisitorMsg( poMessage );
   most_btsetfi_tclMsgStopRingtoneMethodResult oStopRingtoneMethodResult_MR;
   if ( oVisitorMsg.s32GetData(oStopRingtoneMethodResult_MR, FC_PHONE_SERVICE_BLUETOOTHSETTINGS_FI_MAJOR_VERSION)
         != OSAL_ERROR)
   {
      ETG_TRACE_USR4(("most_btsetfi_tclMsgStopRingtoneMethodResult status : %u", oStopRingtoneMethodResult_MR.bStopPlayRingtoneResult));
      // Release Request Acknowledgement is sent before calling StopRingtone in AudioManager
   }
   oStopRingtoneMethodResult_MR.vDestroy();
}

/*******************************************************************************
 *
 * FUNCTION:   vHandle_RequestStopRingtone_MethodError
 *
 * DESCRIPTION:
 *
 * PARAMETER: None
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vHandle_RequestStopRingtone_MethodError(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("fc_phone_tclClientHandler_BTSettings::vHandle_RequestStopRingtone_MethodError entered"));
   fi_tclVisitorMessage oVisitorMsg( poMessage );
   // Create the (message related) FI data object
   most_btsetfi_tclMsgStopRingtoneError oStopRingtoneMethodError_ME;
   if ( oVisitorMsg.s32GetData(oStopRingtoneMethodError_ME, FC_PHONE_SERVICE_BLUETOOTHSETTINGS_FI_MAJOR_VERSION)
         != OSAL_ERROR)
   {
      ETG_TRACE_USR4(("most_btsetfi_tclMsgStopRingtoneMethodError error code : %u", oStopRingtoneMethodError_ME.e8ErrorCode.enType));
   }
   //m_poAudioManager->s32ReleaseRequestAck(FC_PHONE_AUDIOCHANNEL_INCOMINGRINGTONE);
   oStopRingtoneMethodError_ME.vDestroy();
}

#ifdef PHONE_SIRI
/*******************************************************************************
 *
 * FUNCTION:   bGetAppleDeviceStatusfromBDAddressID
 *
 * DESCRIPTION: Reports whether the device corresponding to the given DeviceHandle
 *                is an APPLE device or not
 *
 * PARAMETER: None
 *
 * RETURNVALUE: TRUE-  The connected Device is an APPLE device
 *              FALSE- The Connected device is not an APPLE device
 *
 ********************************************************************************/
tBool fc_phone_tclClientHandler_BTSettings::bGetAppleDeviceStatusfromBDAddressID(tU8 u8BDAddressID)
{
   ETG_TRACE_USR4(("fc_phone_tclClientHandler_BTSettings::bGetAppleDeviceStatusfromBDAddressID with u8BDAddressID: %d", u8BDAddressID));

   for(tU16 u16Index = 0 ;u16Index <FC_PHONE_MAX_DEVICECONNECTION ;u16Index++)
   {
      if(u8BDAddressID == tConnectedDeviceList[u16Index].u8BDAddressId)
      {
         ETG_TRACE_USR3((" APPLE device status for the given u8BDAddressID:: %d ",tConnectedDeviceList[u16Index].u8BDAddressId));

         return tConnectedDeviceList[u16Index].bIsAppleDevice;
      }
   }

   ETG_TRACE_USR4(("No matching device for the given DeviceHandle"));
   return false ;
}
#endif //PHONE_SIRI

/*******************************************************************************
 ** FUNCTION:   vHandle_DeviceListExtended_Status(..)
*******************************************************************************/
/* DESCRIPTION:
 *   This is a handler function for DeviceListExtended property status
 *
 *  PARAMETERS:
 *     IN:   poMessage        : This is the received message from a client
 *  RETURNVALUE: None
 ******************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vHandle_DeviceListExtended_Status(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4((" vHandle_DeviceListExtended_Status entered "));

   tBool bHFPConnectionStatus = false;

   // Extract required info from poMessage
   // Create a FI visitor message for the received CCA message
   fi_tclVisitorMessage oVisitorMsg(poMessage);

   // Create the (message related) FI data object
   most_btsetfi_tclMsgDeviceListExtendedStatus oDeviceListExtendedStatus;

   // Unmarshal the FI visitor message to the given FI data object.
   // The parameters are copied individually from shared memory to the FI data
   // object.
   if (oVisitorMsg.s32GetData(oDeviceListExtendedStatus, MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION)
         != OSAL_ERROR)
   {

      tU8 u8DeviceListExtendedSize =
            (tU8) oDeviceListExtendedStatus.oDeviceListExtendedResult.oItems.size();
      if (0 != u8DeviceListExtendedSize)
      {
         for (tU8 u8DeviceListExtendedIndex = 0; u8DeviceListExtendedIndex < u8DeviceListExtendedSize; u8DeviceListExtendedIndex++)
         {
            most_fi_tcl_BTSetDeviceListExtendedResultItem DeviceListExtendedItem =
                  oDeviceListExtendedStatus.oDeviceListExtendedResult.oItems[u8DeviceListExtendedIndex];

            //Copy the HFP connected status.
            bHFPConnectionStatus = DeviceListExtendedItem.oDeviceProfileStatus.bHFP;

            //fill the device address
            tChar acDeviceAddress[FC_PHONE_MAXDEVICENAME_LENGTH];
            tChar acTempActiveDeviceBD_Address[FC_PHONE_MAXDEVICENAME_LENGTH];
            memset(acDeviceAddress, 0, FC_PHONE_MAXDEVICENAME_LENGTH);

            sscanf(DeviceListExtendedItem.sDeviceAddress.szValue, "%02x%02x%02x%02x%02x%02x",
                  (tUInt *) &acTempActiveDeviceBD_Address[0], (tUInt *) &acTempActiveDeviceBD_Address[1],
                  (tUInt *) &acTempActiveDeviceBD_Address[2], (tUInt *) &acTempActiveDeviceBD_Address[3],
                  (tUInt *) &acTempActiveDeviceBD_Address[4], (tUInt *) &acTempActiveDeviceBD_Address[5]);

            //for string print safety , making 7th char NULL
            acTempActiveDeviceBD_Address[6] = 0;

            //store Active Device Address in proper format
            //Fix for CMG3GB-2367.
            //Addresses are basically in char datatype. Hence if the value crosses 127(max value of char),
            //the x86 compilers of LSIM uppercasts to int and interprets the values wrongly. Hence they are typecasted to unsigned char.
            snprintf(acDeviceAddress, FC_PHONE_MAXDEVICENAME_LENGTH, "dev_%02x_%02x_%02x_%02x_%02x_%02x",
                  (unsigned char) acTempActiveDeviceBD_Address[0], (unsigned char) acTempActiveDeviceBD_Address[1],
                  (unsigned char) acTempActiveDeviceBD_Address[2], (unsigned char) acTempActiveDeviceBD_Address[3],
                  (unsigned char) acTempActiveDeviceBD_Address[4], (unsigned char) acTempActiveDeviceBD_Address[5]);

            ETG_TRACE_USR4((" HFP connected status = %d", bHFPConnectionStatus));
            ETG_TRACE_USR4((" Formatted device address = %s", acDeviceAddress));

            if (!m_TelService_instance)
            {
               ETG_TRACE_USR4((" m_TelService_instance is NULL "));
               return;
            }
            //Calling Evolution instance bCreateProxyAndRegisterSignals.

            ETG_TRACE_USR3(("  Property Status:DeviceListStatus  "));
            ETG_TRACE_USR1((" B_BTDeviceListItem[%u]::u8DeviceHandle :%u", u8DeviceListExtendedIndex, DeviceListExtendedItem.u8DeviceHandle));
            ETG_TRACE_USR1((" B_BTDeviceListItem[%u]::sDeviceName.szValue :%s", u8DeviceListExtendedIndex, DeviceListExtendedItem.sDeviceName.szValue));
            ETG_TRACE_USR1((" B_BTDeviceListItem[%u]::bDeviceConnectedStatus :%u", u8DeviceListExtendedIndex, DeviceListExtendedItem.bDeviceConnectedStatus));
            ETG_TRACE_USR1((" B_BTDeviceListItem[%u]::bOutgoingSourceDeviceStatus :%u", u8DeviceListExtendedIndex, DeviceListExtendedItem.bOutgoingSourceDeviceStatus));

            // Fix for NCG3D-103060
            most_fi_tcl_e8_BTConnectionStatus HFPProfConnectionStatus;
            HFPProfConnectionStatus.enType = DeviceListExtendedItem.oDeviceProfileConnectionStatus.e8HFP.enType;

            // 0- e8STATUS_NOT_CONNECT; 1 - e8STATUS_CONNECTING; 2 - e8STATUS_CONNECTED
            // 3 - e8STATUS_DISCONNECTING; 4 - e8STATUS_DISCONNECTED
            ETG_TRACE_USR4(("HFPProfConnectionStatus : %u", HFPProfConnectionStatus.enType));
            if (bHFPConnectionStatus)
            {
               ETG_TRACE_USR4((" HFP Connected "));

               if( (tConnectedDeviceList[0].u8DeviceHandle != DeviceListExtendedItem.u8DeviceHandle)
                     &&(tConnectedDeviceList[1].u8DeviceHandle != DeviceListExtendedItem.u8DeviceHandle)
               )
               {
                  ETG_TRACE_USR4((" Different Device Handle..get the device Active Address From BT by calling DeviceInfo MethodStart "));

                  //get freeDeviceList instance from deviceList Table
                  FcphoneConnectedDeviceList *pDeviceListInstance = NULLPTR ;
                  pDeviceListInstance = tGetFreeDevicelistInstanceFromDeviceListTable();

                  if(!pDeviceListInstance)
                  {
                     ETG_TRACE_FATAL((" fatal Error :: No Device List slot is free, exiting from the function "));

                     vPrintDeviceList();
                     return ;
                  }

                  //update the device list
                  pDeviceListInstance->bOutgoingSourceStatus = DeviceListExtendedItem.bOutgoingSourceDeviceStatus ;
                  pDeviceListInstance->u8DeviceHandle        = DeviceListExtendedItem.u8DeviceHandle ;
                  vStringCopy (pDeviceListInstance->DeviceName ,DeviceListExtendedItem.sDeviceName.szValue,
                        FC_PHONE_MAXDEVICENAME_LENGTH);

                  pDeviceListInstance->enDisconnectionStatus = DISCONNECTION_IDLE;
                  m_TelService_instance->m_pDBus_CCA_If->bCreateProxyAndRegisterSignals(acDeviceAddress);

                  //GMMY16-12538: SIRI Inconsistencies
#ifdef PHONE_SIRI

                  pDeviceListInstance->bIsAppleDevice =
                        (FC_PHONE_BLUETOOTH_SIG_ID_SOURCE == DeviceListExtendedItem.oDeviceInfo.u16VendorIDSource
                              && FC_PHONE_APPLE_BLUETOOTH_SIG_VENDOR_ID == DeviceListExtendedItem.oDeviceInfo.u16VendorID) ||
                              (FC_PHONE_USB_IMPLEMENTER_FORUM_ID_SOURCE == DeviceListExtendedItem.oDeviceInfo.u16VendorIDSource
                                    && FC_PHONE_APPLE_USB_IMPLEMENTER_VENDOR_ID == DeviceListExtendedItem.oDeviceInfo.u16VendorID);

                  ETG_TRACE_USR4(("Is Apple Device : %d", pDeviceListInstance->bIsAppleDevice));

                  if(pDeviceListInstance->bIsAppleDevice)
                  {
                     tU8 u8_BDAddressId = fc_phone_tclDeviceList::getInstance()->u8GetBDAddressIdFromBDAddress(acDeviceAddress);
                     ETG_TRACE_USR4(("u8_BDAddressId: %d", u8_BDAddressId));

                     if(INVALID_BDADDRESS_ID != u8_BDAddressId)
                     {
                        //TODO: Need to check the possibility of removing the registration of ConnectionComplete signal since
                        //vEnableSiri function is called from this context too.
                        //If the BDADDRESS ID is valid, it internally means the ConnectionComplete signal is received for HFP
                        //Hence invoke the SIRI function directly
                        m_TelService_instance->m_oSiriData[pDeviceListInstance->u8BDAddressId].vEnableSiri(acDeviceAddress);
                     }
                  }

#endif //PHONE_SIRI

                  pDeviceListInstance->u8BDAddressId = fc_phone_tclDeviceList::getInstance()->u8AddDeviceToDeviceList((const char*)acDeviceAddress);
                  //Decide here whether the device should be ACTIVE
                  if(FC_PHONE_DEVICEHANDLE_UNKNOWN == m_TelService_instance->u8GetActiveDeviceHandle())
                  {
                     m_TelService_instance->vSetasActiveDevice(pDeviceListInstance->u8DeviceHandle);
                  }

                  if(pDeviceListInstance->u8DeviceHandle == m_TelService_instance->u8GetActiveDeviceHandle())
                  {
                     //Fix for NCG3D-5406. RESET all channel attributes on SLC_ON
                     (fc_phone_tclApp::m_poMainAppInstance)->m_poAudioManager->vResetAllChannelAttributes();
                  }

                  // Creating CS in the CS list vector and setting the recently connected device as the Active device
                  m_TelService_instance->vInsertCallStatusinCSList(pDeviceListInstance->u8BDAddressId);

                  fc_phone_tclClientHandler_PhoneBook *p_clientHndler_phonebook = (fc_phone_tclApp::m_poMainAppInstance)->m_poClientHandlerPhoneBook;
                  p_clientHndler_phonebook->vInsertPBListItem(pDeviceListInstance->u8DeviceHandle);

                  m_TelService_instance->vInitPhoneData(pDeviceListInstance->u8BDAddressId);

               }
               else
               {
                  // Fix for NCG3D-103060
                  // This is to check for the Virtual disconnection
                  FcphoneConnectedDeviceList *pDeviceListInstance = NULLPTR ;
                  pDeviceListInstance = tGetDevicelistInstanceFromDeviceHandle(DeviceListExtendedItem.u8DeviceHandle) ;

                  if(pDeviceListInstance)
                  {
                     // Fix for NCG3D-103060.
                     if (most_fi_tcl_e8_BTConnectionStatus::FI_EN_E8STATUS_DISCONNECTING == HFPProfConnectionStatus.enType)
                     {
                        ETG_TRACE_USR4((" Disconnection Started "));
                        pDeviceListInstance->enDisconnectionStatus = DISCONNECTION_STARTED;
                     }
                     else if (most_fi_tcl_e8_BTConnectionStatus::FI_EN_E8STATUS_CONNECTED == HFPProfConnectionStatus.enType)
                     {
                        if (DISCONNECTION_INCOMPLETE == pDeviceListInstance->enDisconnectionStatus)
                        {
                           ETG_TRACE_USR4(("Virtual disconnection(incomplete Disconnection) happened and then connection happened"));

                           if ((m_TelService_instance) && (FALSE == (m_TelService_instance->bCheckIdleCallStatus())))
                           {
                              if (TRUE == m_TelService_instance->bIsExternalTriggerReceivedForPhoneChannel())
                              {
                                 ETG_TRACE_USR4(("External trigger happened and SA-OFF received without Release request"));

                                 m_TelService_instance->m_pDBus_CCA_If->bAudioTransfer(acDeviceAddress, FC_PHONE_CALLTRANSFER_FROMHANDSET_TOHANSFREE);
                              }
                              else if (TRUE == m_TelService_instance->bIsFirstIncomingCallPresent())
                              {
                                 ETG_TRACE_USR4(("Out-band ring-tone would be stopped. So calling RequestAvailableRingtonesList"));
                                 RequestAvailableRingtonesList(pDeviceListInstance->u8DeviceHandle);
                              }
                           }
                           else
                           {
                              ETG_TRACE_USR4(("m_TelService_instance is NULL or IDLE calls"));
                           }
                        }

#ifdef PHONE_SIRI
                        //Check if device is Apple device in case vendor ID update is delayed and Enable SIRI
                        if(!(pDeviceListInstance->bIsAppleDevice))
                        {
                           pDeviceListInstance->bIsAppleDevice =
                                 (FC_PHONE_BLUETOOTH_SIG_ID_SOURCE == DeviceListExtendedItem.oDeviceInfo.u16VendorIDSource
                                       && FC_PHONE_APPLE_BLUETOOTH_SIG_VENDOR_ID == DeviceListExtendedItem.oDeviceInfo.u16VendorID) ||
                                       (FC_PHONE_USB_IMPLEMENTER_FORUM_ID_SOURCE == DeviceListExtendedItem.oDeviceInfo.u16VendorIDSource
                                             && FC_PHONE_APPLE_USB_IMPLEMENTER_VENDOR_ID == DeviceListExtendedItem.oDeviceInfo.u16VendorID);

                           ETG_TRACE_USR4(("Is Apple Device : %d", pDeviceListInstance->bIsAppleDevice));

                           if(pDeviceListInstance->bIsAppleDevice)
                           {
                              tU8 u8_BDAddressId = fc_phone_tclDeviceList::getInstance()->u8GetBDAddressIdFromBDAddress(acDeviceAddress);
                              ETG_TRACE_USR4(("u8_BDAddressId: %d", u8_BDAddressId));

                              if(INVALID_BDADDRESS_ID != u8_BDAddressId)
                              {
                                 //TODO: Need to check the possibility of removing the registration of ConnectionComplete signal since
                                 //vEnableSiri function is called from this context too.
                                 //If the BDADDRESS ID is valid, it internally means the ConnectionComplete signal is received for HFP
                                 //Hence invoke the SIRI function directly
                                 m_TelService_instance->m_oSiriData[pDeviceListInstance->u8BDAddressId].vEnableSiri(acDeviceAddress);
                              }
                           }

                        }

#endif //PHONE_SIRI
                        pDeviceListInstance->enDisconnectionStatus = DISCONNECTION_IDLE;
                     }
                     vPrintDeviceList();
                  }
                  else
                  {
                     ETG_TRACE_USR4(("pDeviceListInstance is NULL"));
                  }
               }
            }
            else
            {
               ETG_TRACE_USR4((" HFP Disconnected "));
               FcphoneConnectedDeviceList *pDeviceListInstance = NULLPTR ;
               pDeviceListInstance = tGetDevicelistInstanceFromDeviceHandle(DeviceListExtendedItem.u8DeviceHandle) ;

               if(pDeviceListInstance)
               {
                  if (most_fi_tcl_e8_BTConnectionStatus::FI_EN_E8STATUS_DISCONNECTED == HFPProfConnectionStatus.enType)
                  {
                     ETG_TRACE_USR4((" Delete Entry from Connected DeviceList Table "));

                     m_TelService_instance->m_pDBus_CCA_If->bDeleteProxyAndDeRegisterSignals(acDeviceAddress);

                     (fc_phone_tclApp::m_poMainAppInstance)->m_poTelephone->vSendSLCOFF(pDeviceListInstance->u8BDAddressId, pDeviceListInstance->u8DeviceHandle);

                     //Fix for GMMY15-6181-ProposeSolution
                     //fc_phone_tclDeviceList::getInstance()->u8DeleteDeviceFromDeviceList(pDeviceListInstance->u8BDAddressId);
                     pDeviceListInstance->u8BDAddressId=INVALID_BDADDRESS_ID;
                     memset(pDeviceListInstance->DeviceName,0,sizeof(pDeviceListInstance->DeviceName));
                     memset(pDeviceListInstance->sVoiceMailNumber,0,sizeof(pDeviceListInstance->sVoiceMailNumber));
                     pDeviceListInstance->u8DeviceHandle = FC_PHONE_DEVICEHANDLE_UNKNOWN ;
                     pDeviceListInstance->bOutgoingSourceStatus = FALSE ;
                     pDeviceListInstance->bSlotFree = TRUE ;
                     pDeviceListInstance->u8SCOConnectionStatus = FC_PHONE_DBUS_SCO_CONN_STATUS_UNKNOWN;
                     pDeviceListInstance->enDisconnectionStatus = DISCONNECTION_IDLE;
                     pDeviceListInstance->bIsAppleDevice = FALSE;
                  }
                  else if (most_fi_tcl_e8_BTConnectionStatus::FI_EN_E8STATUS_CONNECTING == HFPProfConnectionStatus.enType)
                  {
                     // Fix for NCG3D-103060
                     // This is to check for the Virtual disconnection
                     pDeviceListInstance->enDisconnectionStatus = DISCONNECTION_INCOMPLETE;
                     ETG_TRACE_USR4(("Disconnection is incomplete"));
                  }
                  vPrintDeviceList();
               }
               else
               {
                  ETG_TRACE_USR4(("pDeviceListInstance is NULL"));
               }
            }
         }
      }
      else
      {
         ETG_TRACE_USR4((" Size of the extended list is Zero "));
      }
   }
   else
   {
      ETG_TRACE_USR4((" Get data Failed "));
   }

   oDeviceListExtendedStatus.vDestroy();
}


/*******************************************************************************
 ** FUNCTION:   vHandle_VehicleBtAddress_Status
*******************************************************************************/
/* DESCRIPTION: Get the VehicleBtAddress_Status from BT Settings.
 *
 *  PARAMETERS:
 *  IN:   poMessage        :
 *  RETURNVALUE:
*******************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vHandle_VehicleBtAddress_Status(amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR4((" MethodResult: vHandle_VehicleBtAddress_Status  entered "));

   char ucVehicleBtAddress[FC_PHONE_STRING_LENGTH_MAX]; //FIXME:tharani Length needs to checked
   most_btsetfi_tclMsgVehicleBTAddressStatus oVehicleBtAddress;

   fi_tclVisitorMessage oVisitorMsg(poMessage);

   if (oVisitorMsg.s32GetData(oVehicleBtAddress, MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION)
         != OSAL_ERROR)
   {
      memset(ucVehicleBtAddress, 0, FC_PHONE_STRING_LENGTH_MAX);
      //copying vehicle BT address
      vStringCopy(ucVehicleBtAddress, oVehicleBtAddress.sVehicleBTAddress.szValue, FC_PHONE_STRING_LENGTH_MAX);
      if (strcmp(oVehicleBtAddress.sVehicleBTAddress.szValue, "") != 0)
      {
         if ((m_TelService_instance))
         {
            ETG_TRACE_USR1((" Local BT Device address to be updated = %s",ucVehicleBtAddress));
            m_TelService_instance->vSetVehicleBtAddress(ucVehicleBtAddress);
         }
         else
         {

            ETG_TRACE_USR1(("m_TelService_instance instance is not available"));
         }
      }
      else
      {
         ETG_TRACE_USR1(("Vehicle address is Empty"));
      }
   }
   else
   {
      amt_tclServiceDataError
            oErrorMsg(*poMessage, CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED);
      ail_tenCommunicationError enResult =
            _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

      ETG_TRACE_FATAL((" Error - Version not supported "));

      if (enResult != AIL_EN_N_NO_ERROR)
      {
      }
      // NORMAL_M_ASSERT_ALWAYS();
   }
   oVehicleBtAddress.vDestroy();
}

/*******************************************************************************
 ** FUNCTION:   vHandle_VehicleConfiguration_Status
*******************************************************************************/
/* DESCRIPTION: Get the VehicleInformation in VehicleConfiguration_Status of BT Settings.
 *
 *  PARAMETERS:
 *  IN:   poMessage        :
 *  RETURNVALUE:
*******************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vHandle_VehicleConfiguration_Status(amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR4((" Status: vHandle_VehicleConfiguration_Status  entered "));


   most_btsetfi_tclMsgVehicleConfigurationStatus oVehicleConfiguration;

   fi_tclVisitorMessage oVisitorMsg(poMessage);

   if (oVisitorMsg.s32GetData(oVehicleConfiguration, MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION)
         != OSAL_ERROR)
   {
      m_DeviceInformation.m_u16ProductID = oVehicleConfiguration.oVehicleIdentification.u16ProductID;
      m_DeviceInformation.m_u16VendorID = oVehicleConfiguration.oVehicleIdentification.u16VendorID;
      m_DeviceInformation.m_u16ProductVersion = oVehicleConfiguration.oVehicleIdentification.u16ProductVersion;
      ETG_TRACE_USR4(("m_DeviceInformation.u16ProductID: %x",m_DeviceInformation.m_u16ProductID));
      ETG_TRACE_USR4(("m_DeviceInformation.u16VendorID: %x",m_DeviceInformation.m_u16VendorID));
      ETG_TRACE_USR4(("m_DeviceInformation.u16ProductVersion: %x",m_DeviceInformation.m_u16ProductVersion));
   }
   else
   {
      amt_tclServiceDataError
            oErrorMsg(*poMessage, CCA_C_U16_ERROR_VERSION_NOT_SUPPORTED);
      ail_tenCommunicationError enResult =
            _poMainAppl->enPostMessage(&oErrorMsg, TRUE);

      ETG_TRACE_FATAL((" Error - MOST_BTSETFI_C_U16_SERVICE_MAJORVERSION version not supported "));

      if (enResult != AIL_EN_N_NO_ERROR)
      {
      }
      // NORMAL_M_ASSERT_ALWAYS();
   }
   oVehicleConfiguration.vDestroy();
}

/*******************************************************************************
 *
 * FUNCTION:   vSetPrimaryDevice()
 *
 * DESCRIPTION: -
 *
 * PARAMETER: deviceHandle - device for which to set the primary
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vSetPrimaryDevice(tU8 u8DeviceHandle)
{
   //Write a function to get ringtone list corresponding to this device handle
   ETG_TRACE_USR4((" fc_phone_tclClientHandler_BTSettings::vSetPrimaryDevice ENTER"));

   most_btsetfi_tclMsgSetPrimaryMethodStart oSetPrimary;

   oSetPrimary.u8DeviceHandle = u8DeviceHandle;

   ETG_TRACE_USR1((" oSetPrimary.u8DeviceHandle = %d     ",oSetPrimary.u8DeviceHandle));

   fi_tclVisitorMessage oMsg(oSetPrimary,FC_PHONE_SERVICE_BLUETOOTHSETTINGS_FI_MAJOR_VERSION);
   vInitServiceData(oMsg,                                   // ServiceDataMsg
         0,                                                 // CmdCounter
         MOST_BTSETFI_C_U16_SETPRIMARY,                     // Function ID
         AMT_C_U8_CCAMSG_OPCODE_METHODSTART                 // Opcode
   );

   // Post message
   ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMsg, TRUE);
   if(enResult != AIL_EN_N_NO_ERROR)
   {
      ETG_TRACE_ERR(("  enPostMessage() failed for 'setPrimaryDevice - METHODSTART' message, 'ail_tenCommunicationError' = %u      ", (tU32)enResult ));
   }

   oSetPrimary.vDestroy();
}

/*******************************************************************************
 *
 * FUNCTION:   vHandle_SetPrimary_MethodResult
 *
 * DESCRIPTION: Send by BT settings once Primary set for the corresponding device
 *
 * PARAMETER: None
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vHandle_SetPrimary_MethodResult(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("fc_phone_tclClientHandler_BTSettings::vHandle_SetPrimary_MethodResult entered"));

   fi_tclVisitorMessage oVisitorMsg( poMessage );
   most_btsetfi_tclMsgSetPrimaryMethodResult oSetPrimaryMethodResult_MR;

   if ( oVisitorMsg.s32GetData(oSetPrimaryMethodResult_MR, FC_PHONE_SERVICE_BLUETOOTHSETTINGS_FI_MAJOR_VERSION)
         != OSAL_ERROR)
   {
      ETG_TRACE_USR4(("vHandle_SetPrimary_MethodResult status : %u", oSetPrimaryMethodResult_MR.bResult));
   }

   oSetPrimaryMethodResult_MR.vDestroy();
}

/*******************************************************************************
 *
 * FUNCTION:   vHandle_SetPrimary_MethodError
 *
 * DESCRIPTION:
 *
 * PARAMETER: None
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vHandle_SetPrimary_MethodError(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("fc_phone_tclClientHandler_BTSettings::vHandle_SetPrimary_MethodError entered"));

   fi_tclVisitorMessage oVisitorMsg( poMessage );
   // Create the (message related) FI data object
   most_btsetfi_tclMsgSetPrimaryError oSetPrimaryMethodError_ME;

   if ( oVisitorMsg.s32GetData(oSetPrimaryMethodError_ME, FC_PHONE_SERVICE_BLUETOOTHSETTINGS_FI_MAJOR_VERSION)
         != OSAL_ERROR)
   {
      ETG_TRACE_USR4(("vHandle_SetPrimary_MethodError error code : %u", oSetPrimaryMethodError_ME.e8ErrorCode.enType));
   }

   oSetPrimaryMethodError_ME.vDestroy();
}

/*******************************************************************************
 *
 * FUNCTION:   vSetBTProfilesUsage()
 *
 * DESCRIPTION: -
 *
 * PARAMETER:  u8DeviceHandle - device for which BT Profile usage to be set
 *             bProfileUsageInfo- BT profile Usage info
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vSetBTProfilesUsage(tU8 u8DeviceHandle, tBool bProfileUsageInfo)
{
   ETG_TRACE_USR4((" vSetBTProfilesUsage Entered with u8DeviceHandle: %u and bProfileUsageInfo: %d", u8DeviceHandle, bProfileUsageInfo));

   most_btsetfi_tclMsgSetBTProfilesUsageMethodStart oSetBTProfilesUsage_MS;

   most_fi_tcl_BTProfilesUsageInfoList o_BTProfilesUsageInfoList;

   o_BTProfilesUsageInfoList.u8DeviceHandle = u8DeviceHandle;
   o_BTProfilesUsageInfoList.oBTProfilesUsageInfo.e8BTProfile.enType = (most_fi_tcl_e8_BTProfile::FI_EN_E8BT_PROFILE_HFP);
   o_BTProfilesUsageInfoList.oBTProfilesUsageInfo.e8BTProfileUsageType.enType = (most_fi_tcl_e8_BTProfileUsageType::FI_EN_E8PROFILE_USAGE_ACTIVE_CALL);

   if (bProfileUsageInfo)
   {
      o_BTProfilesUsageInfoList.oBTProfilesUsageInfo.bInUse = TRUE;
   }
   else
   {
      o_BTProfilesUsageInfoList.oBTProfilesUsageInfo.bInUse = FALSE;
   }

   ETG_TRACE_USR1((" vSetBTProfilesUsage::u8DeviceHandle = %d ", o_BTProfilesUsageInfoList.u8DeviceHandle));
   ETG_TRACE_USR1((" vSetBTProfilesUsage::e8BTProfileUsageType = %d ", o_BTProfilesUsageInfoList.oBTProfilesUsageInfo.e8BTProfileUsageType.enType));
   ETG_TRACE_USR1((" vSetBTProfilesUsage::oBTProfilesUsageInfo.bInUse = %d ", o_BTProfilesUsageInfoList.oBTProfilesUsageInfo.bInUse));

   oSetBTProfilesUsage_MS.oBTProfilesUsageInfoList.oItems.push_back(o_BTProfilesUsageInfoList);

   fi_tclVisitorMessage oMsg(oSetBTProfilesUsage_MS,FC_PHONE_SERVICE_BLUETOOTHSETTINGS_FI_MAJOR_VERSION);
   vInitServiceData(oMsg,                                   // ServiceDataMsg
         0,                                                 // CmdCounter
         MOST_BTSETFI_C_U16_SETBTPROFILESUSAGE,                     // Function ID
         AMT_C_U8_CCAMSG_OPCODE_METHODSTART                 // Opcode
   );

   // Post message
   ail_tenCommunicationError enResult = _poMainAppl->enPostMessage(&oMsg, TRUE);
   if(enResult != AIL_EN_N_NO_ERROR)
   {
      ETG_TRACE_ERR(("  enPostMessage() failed for 'SetBTProfilesUsage - METHODSTART' message, 'ail_tenCommunicationError' = %u      ", (tU32)enResult ));
   }

   oSetBTProfilesUsage_MS.vDestroy();
}

/*******************************************************************************
 *
 * FUNCTION:   vHandle_SetBTProfileUsage_MethodResult
 *
 * DESCRIPTION: Informs whether the SetBTProfilesUsage Request is processed successfully or not
 *
 * PARAMETER: None
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vHandle_SetBTProfileUsage_MethodResult(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("fc_phone_tclClientHandler_BTSettings::vHandle_SetBTProfileUsage_MethodResult entered"));

   fi_tclVisitorMessage oVisitorMsg( poMessage );
   most_btsetfi_tclMsgSetBTProfilesUsageMethodResult oSetBTProfilesUsage_MR;

   if ( oVisitorMsg.s32GetData(oSetBTProfilesUsage_MR, FC_PHONE_SERVICE_BLUETOOTHSETTINGS_FI_MAJOR_VERSION)
         != OSAL_ERROR)
   {
      ETG_TRACE_USR4(("vHandle_SetBTProfileUsage_MethodResult status : %u", oSetBTProfilesUsage_MR.bResult));
   }

   oSetBTProfilesUsage_MR.vDestroy();
}

/*******************************************************************************
 *
 * FUNCTION:   vHandle_SetBTProfileUsage_MethodError
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *
 * RETURNVALUE:
 *
 ********************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vHandle_SetBTProfileUsage_MethodError(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR4(("fc_phone_tclClientHandler_BTSettings::vHandle_SetBTProfileUsage_MethodError entered"));

   fi_tclVisitorMessage oVisitorMsg( poMessage );
   // Create the (message related) FI data object
   most_btsetfi_tclMsgSetBTProfilesUsageError oSetBTProfilesUsage_ME;

   if ( oVisitorMsg.s32GetData(oSetBTProfilesUsage_ME, FC_PHONE_SERVICE_BLUETOOTHSETTINGS_FI_MAJOR_VERSION)
         != OSAL_ERROR)
   {
      ETG_TRACE_USR4(("vHandle_SetBTProfileUsage_MethodError error code : %u", oSetBTProfilesUsage_ME.e8ErrorCode.enType));
   }

   oSetBTProfilesUsage_ME.vDestroy();
}

//Task CMG3G-7981: IS2278_Siri and other VR with different ECNR settings@FC_Phone
/*******************************************************************************
 *
 * FUNCTION:   vGetVRDataSet(..)
 *
 * DESCRIPTION:
 *
 * PARAMETER:
 *    IN:
 *       u8Codec : Codec information (Narrow band or Wideband)
 *    OUT:
 *       u8DataSet : SCO connection status
 *
 * RETURNVALUE: None.
 *
 ********************************************************************************/
tVoid fc_phone_tclClientHandler_BTSettings::vGetVRDataSet(tU8& u8DataSet, tU8 u8Codec)
{
   ETG_TRACE_USR4(("fc_phone_tclClientHandler_BTSettings::vGetVRDataSet Codec: '%u'", u8Codec));

   if(!m_TelService_instance)
   {
      ETG_TRACE_USR4(("m_TelService_instance is NULL. Hence returning"));
      u8DataSet = 0;
      return;
   }

   tU8 u8ActDeviceHandle = m_TelService_instance->u8GetActiveDeviceHandle();
   if(FC_PHONE_DEVICEHANDLE_UNKNOWN == u8ActDeviceHandle)
   {
      ETG_TRACE_USR4(("No Active device present. Hence returning"));
      u8DataSet = 0;
      return;
   }

   FcphoneConnectedDeviceList *pDeviceListInstance = NULLPTR ;
   pDeviceListInstance = tGetDevicelistInstanceFromDeviceHandle(u8ActDeviceHandle) ;

   // CMG3G-12981- Fix for COVERITY
   if(!pDeviceListInstance)
   {
      ETG_TRACE_USR4(("pDeviceListInstance is NULL. No OutgoingSource device present"));
      u8DataSet = 0;
      return;
   }

   if (0 == u8Codec)
   {
      //Fix for NCG3D-6029, setting codec = WIDEBAND as default for SIRI
#ifdef PHONE_SIRI
      if (TRUE == pDeviceListInstance->bIsAppleDevice)
      {
         u8Codec = WIDE_BAND_SPEECH;
         ETG_TRACE_USR4(("updated codec as WIDEBAND for SIRI"));
      }
      else
      {
#endif
         u8Codec = pDeviceListInstance->u8Codec;
#ifdef PHONE_SIRI
      }
#endif
      ETG_TRACE_USR4(("Codec of Outgoing device: %u", u8Codec));
   }

   if (WIDE_BAND_SPEECH == u8Codec)
   {
#ifdef PHONE_SIRI
      if (TRUE == pDeviceListInstance->bIsAppleDevice)
      {
         u8DataSet = SSE_CONFIG_PHONE_SIRI_WB_FROM_FILE;
      }
      else
      {
#endif
         u8DataSet = PHONE_DATASET_4;
#ifdef PHONE_SIRI
      }
#endif
   }
   else
   {
#ifdef PHONE_SIRI
      if (TRUE == pDeviceListInstance->bIsAppleDevice)
      {
         u8DataSet = SSE_CONFIG_PHONE_SIRI_NB_FROM_FILE;
      }
      else
      {
#endif
         u8DataSet = PHONE_DATASET_3;
#ifdef PHONE_SIRI
      }
#endif
   }
}
/*******************************************************************************
 *
 * FUNCTION:   bIsSuppressRingtoneEnabled
 *
 * DESCRIPTION: To check whether the suppress ringtone is enabled or not.
 *
 * PARAMETER: None
 *
 * RETURNVALUE: m_bSuppressRingtone
 *
 ********************************************************************************/
tBool fc_phone_tclClientHandler_BTSettings::bIsSuppressRingtoneEnabled()
{
   ETG_TRACE_USR4(("m_bSuppressRingtone: %d", m_bSuppressRingtone));
   return m_bSuppressRingtone;
}

/*******************************************************************************
 *
 * FUNCTION:   vGetDeviceInformation
 *
 * DESCRIPTION: Get the device information like vendor ID, Product ID and SW version.
 *
 * PARAMETER: VehicleInformation
 *
 * RETURNVALUE: none
 *
 ********************************************************************************/

tVoid fc_phone_tclClientHandler_BTSettings::vGetDeviceInformation(VehicleInformation  &devInfo)
{
	ETG_TRACE_USR4(("fc_phone_tclClientHandler_BTSettings::vGetDeviceInformation"));
	devInfo.m_u16ProductID = m_DeviceInformation.m_u16ProductID;
	devInfo.m_u16VendorID = m_DeviceInformation.m_u16VendorID;
	devInfo.m_u16ProductVersion = m_DeviceInformation.m_u16ProductVersion;
	ETG_TRACE_USR4(("vGetDeviceInformation devInfo.m_u16ProductID: %x", devInfo.m_u16ProductID));
	ETG_TRACE_USR4(("vGetDeviceInformation devInfo.m_u16VendorID: %x", devInfo.m_u16VendorID));
	ETG_TRACE_USR4(("vGetDeviceInformation devInfo.m_u16ProductVersion: %x", devInfo.m_u16ProductVersion));
}
