/*******************************************************************************
 *
 * FILE:        	FC_Messaging_ReadMessageAloud.cpp
 *
 * SW-COMPONENT:  FC_Messaging application
 *
 * PROJECT:       Gen3
 *
 * DESCRIPTION:   Handles text message read aloud, interacts with the ARL and SDS
 *
 * AUTHOR:
 *
 * COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
 *
 *******************************************************************************/

#include "../../FC_Messaging_Debug.h"
#include "../../GenericWrapper/FC_Messaging_GenericWrapper.h"
#include "../../FC_Messaging_service_Messaging.h"
#include "../../FC_Messaging_StateMachine.h"
#include "../../Audio/FC_Messaging_AudioRouteHandler.h"
#include "FC_Messaging_ReadMessageAloud.h"
#include "../FC_Messaging_main.h"
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_MESSAGING_APPLICATION
#include "trcGenProj/Header/FC_Messaging_ReadMessageAloud.cpp.trc.h"
#endif

#define READ_MESSAGE_ALOUD_CONTROL_SEQUENCE  			"\x1B\\tn=sms\\ "

ReadMessageAloud* ReadMessageAloud::s_poReadMessageAloudObject = NULLPTR;

/*******************************************************************************
 * FUNCTION:  	   ReadMessageAloud
 * DESCRIPTION:   constructor
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
ReadMessageAloud::ReadMessageAloud()
{
   ETG_TRACE_USR4(("ReadMessageAloud::ReadMessageAloud"));

   m_bNewReadRequestWaiting = FALSE;
   m_bAudioSrcChangedByArl = FALSE;
   m_bAudioRouteTimerRunning = FALSE;
   m_e8ReadMessageAloudCaller = e8ReadMessageAloudMethod;
   m_enReadAloudStatus = FC_MSG_READALOUDSTATE_IDLE;
   m_enReadAloudSubStatus = FC_MSG_READALOUDSUBSTATE_IDLE;
   ETG_TRACE_USR4(("ReadAloudStatus::FC_MSG_READALOUDSTATE_IDLE"));
}

/*******************************************************************************
 * FUNCTION:  	   ~ReadMessageAloud
 * DESCRIPTION:   destructor
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
ReadMessageAloud::~ReadMessageAloud()
{
   ETG_TRACE_USR4(("ReadMessageAloud::~ReadMessageAloud"));

   if (TRUE == m_bAudioRouteTimerRunning)
   {
      if (FALSE == fc_messaging_tclApp::m_poMessaging->bStopTimer(FC_MESSAGING_AUDIOROUTEREQUEST_WAIT_TIMER_ID))
      {
         ETG_TRACE_ERR(("Unable to stop the timer FC_MESSAGING_AUDIOROUTEREQUEST_WAIT_TIMER_ID"));
      }
      else
      {
         m_bAudioRouteTimerRunning = FALSE;
      }
   }
}

/*******************************************************************************
 * FUNCTION:      poGetInstance
 * DESCRIPTION:   Returns a pointer to an object of ReadMessageAloud
 *
 * PARAMETER:
 *
 * RETURNVALUE:   ReadMessageAloud instance pointer
 ********************************************************************************/
ReadMessageAloud* ReadMessageAloud::poGetInstance()
{
   ETG_TRACE_USR4(("ReadMessageAloud::poGetInstance"));

   if (!s_poReadMessageAloudObject)
   {
      s_poReadMessageAloudObject = OSAL_NEW ReadMessageAloud;
   }
   return s_poReadMessageAloudObject;
}

/*******************************************************************************
 * FUNCTION:  	   vDestroyReadMessageAloudObject
 * DESCRIPTION:   function is used to destroy ReadMessageAloud object
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vDestroyReadMessageAloudObject()
{
   ETG_TRACE_USR4(("ReadMessageAloud::vDestroyReadMessageAloudObject"));

   if (s_poReadMessageAloudObject)
   {
      OSAL_DELETE s_poReadMessageAloudObject;
      s_poReadMessageAloudObject = NULLPTR;
   }
}

/*******************************************************************************
 * FUNCTION:  	   vGetMessageForReadAloud
 * DESCRIPTION:   function is used to get text message to read.
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vGetMessageForReadAloud(most_msgfi_tclMsgReadMessageAloudMethodStart& roMsgReadMessageAloudMethodStart)
{
   ETG_TRACE_USR4(("ReadMessageAloud::vGetMessageForReadAloud Entered"));

   m_e8ReadMessageAloudCaller = e8ReadMessageAloudMethod;

   switch (m_enReadAloudStatus)
   {
      case FC_MSG_READALOUDSTATE_IDLE:
      {
         m_oCurrentMessageHandle = roMsgReadMessageAloudMethodStart.oMessageHandle;
         FileSystemManager* poFileSystemManager = FileSystemManager::poFileSystemManager(roMsgReadMessageAloudMethodStart.oMessageHandle);

         if (poFileSystemManager)
         {
            m_enReadAloudStatus = FC_MSG_GET_MSG_IN_PROGRESS;
            ETG_TRACE_USR4(("ReadAloudStatus::FC_MSG_GET_MSG_IN_PROGRESS"));
            poFileSystemManager->vGetMessage(roMsgReadMessageAloudMethodStart.oMessageHandle, e8ReadMessageAloud);
         }
         else
         {
            vSendReadMessageAloudError();
         }
      }
      break;

      case FC_MSG_READALOUD_IN_PROGRESS:
      {
         // Waiting request will be processed on receiving PrepareVoiceOutput/SynthesizeSpeak response
         m_bNewReadRequestWaiting = TRUE;
         m_oNewRequestMessageHandle = roMsgReadMessageAloudMethodStart.oMessageHandle;

         if (FC_MSG_SYNTHESIS_IN_PROGRESS == m_enReadAloudSubStatus)
         {
            //Abort read aloud if synthesis speak triggered

            fc_messaging_tclGenericWrapper* poGenericWrapper = fc_messaging_tclApp::poGetInstance()->poGetGenericWrapper();

            if (poGenericWrapper)
            {
               if (FALSE == poGenericWrapper->bAbortPrompt())
               {
                  ETG_TRACE_ERR(("AbortPrompt request posting failed!!"));
                  vUpdateStatesAndReleaseAudio();
               }
               else
               {
                  m_enReadAloudStatus = FC_MSG_ABORT_IN_PROGRESS;
                  ETG_TRACE_USR4(("ReadAloudStatus::FC_MSG_ABORT_IN_PROGRESS"));
                  m_enReadAloudSubStatus = FC_MSG_READALOUDSUBSTATE_IDLE;
               }
            }
         }
      }
      break;

      case FC_MSG_ABORT_IN_PROGRESS:
      case FC_MSG_CHANNEL_RELEASE_IN_PROGRESS:
      {
         // Waiting request will be processed on receiving AbortPrompt/SynthesizeSpeak/ response or Audio release
         m_bNewReadRequestWaiting = TRUE;
         m_oNewRequestMessageHandle = roMsgReadMessageAloudMethodStart.oMessageHandle;
      }
      break;

      default:
      {
         // Cannot come here
      }
   }
}

/*******************************************************************************
 * FUNCTION:  	   vGetMessageForReadAloudReturn
 * DESCRIPTION:   function is used to process the text message data for read aloud
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vGetMessageForReadAloudReturn(MsgRead_t* poMsgReadFromDevice, FolderNode* poFolderNode)
{
   ETG_TRACE_USR4(("ReadMessageAloud::vGetMessageForReadAloudReturn Entered"));

   QByteArray baReadAloudMessageData;
   QByteArray baReadMsgAloudControlSequence(READ_MESSAGE_ALOUD_CONTROL_SEQUENCE);

   if (poFolderNode->m_sFolderName == MSG_DBUS_FOLDER_TYPE_INBOX)
   {
      baReadAloudMessageData = baReadMsgAloudControlSequence + poMsgReadFromDevice->sGetDataPayLoad();
   }
   else
   {
      // Message is from Sent Items Folder
      baReadAloudMessageData = baReadMsgAloudControlSequence + poMsgReadFromDevice->sGetDataPayLoad();
   }

   m_sText.bSet(baReadAloudMessageData.constData(), most_fi_tcl_String::FI_EN_UTF8);

   vProcessEvent(FC_MESSAGING_SM_EVENT_GETMESSAGE_SUCCESS);
}

/*******************************************************************************
 * FUNCTION:      vStopReadAloud
 * DESCRIPTION:   function is used to stop the current read aloud
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vStopReadAloud(e8ReadMessageAloudCaller readMessageAloudCaller)
{
   ETG_TRACE_USR4(("ReadMessageAloud::vStopReadAloud Entered"));

   m_e8ReadMessageAloudCaller = readMessageAloudCaller;

   switch (m_enReadAloudStatus)
   {
      case FC_MSG_READALOUD_IN_PROGRESS:
      {
         if (TRUE == m_bAudioSrcChangedByArl)
         {
            //Audio source not available, send MethodError to the request
            vSendReadMessageAloudError();
         }
         else if (FC_MSG_PREPARE_IN_PROGRESS == m_enReadAloudSubStatus)
         {
            //Handle Stop request on PrepareVoiceOutput response
            m_enReadAloudSubStatus = FC_MSG_STOP_IN_WAIT;
         }
         else if (FC_MSG_SYNTHESIS_IN_PROGRESS == m_enReadAloudSubStatus)
         {
            //Abort the read aloud

            fc_messaging_tclGenericWrapper* poGenericWrapper = fc_messaging_tclApp::poGetInstance()->poGetGenericWrapper();

            if (poGenericWrapper)
            {
               if (FALSE == poGenericWrapper->bAbortPrompt())
               {
                  ETG_TRACE_ERR(("AbortPrompt request posting failed!!"));
                  vUpdateStatesAndReleaseAudio();
               }
               else
               {
                  m_enReadAloudStatus = FC_MSG_ABORT_IN_PROGRESS;
                  ETG_TRACE_USR4(("ReadAloudStatus::FC_MSG_ABORT_IN_PROGRESS"));
                  m_enReadAloudSubStatus = FC_MSG_STOP_IN_PROGRESS;
               }
            }
         }
         else
         {
            //No other possible states
         }
      }
      break;

      case FC_MSG_ABORT_IN_PROGRESS:
      case FC_MSG_CHANNEL_RELEASE_IN_PROGRESS:
      {
         //Since abort/audio release is in progress, send Error response
         vSendReadMessageAloudError();
      }
      break;

      default:
      {
         ETG_TRACE_ERR(("Stop request received in Invalid State: '%u'!!", ETG_ENUM(TR_MSG_READALOUDSTATUS, m_enReadAloudStatus)));

         vSendReadMessageAloudError();
      }
   }
}

/*******************************************************************************
 * FUNCTION:      vReadAloudContentFromClient
 * DESCRIPTION:   function is used to process the text message data for read aloud
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vReadAloudContentFromClient(tString sReadAloudContent)
{
   ETG_TRACE_USR4(("vReadAloudContentFromClient : %s", sReadAloudContent));

   m_e8ReadMessageAloudCaller = e8ReadMessageAloudExtendedMethod;

   switch (m_enReadAloudStatus)
   {
      case FC_MSG_READALOUDSTATE_IDLE:
      {
         m_enReadAloudStatus = FC_MSG_GET_MSG_IN_PROGRESS;
         ETG_TRACE_USR4(("ReadAloudStatus::FC_MSG_GET_MSG_IN_PROGRESS"));

         m_sText.bSet((tCString)sReadAloudContent, most_fi_tcl_String::FI_EN_UTF8);

         vProcessEvent(FC_MESSAGING_SM_EVENT_GETMESSAGE_SUCCESS);
      }
      break;

      case FC_MSG_READALOUD_IN_PROGRESS:
      {
         // Waiting request will be processed on receiving PrepareVoiceOutput/SynthesizeSpeak response
         m_bNewReadRequestWaiting = TRUE;
         m_sNewRequestMessageText.bSet((tCString)sReadAloudContent, most_fi_tcl_String::FI_EN_UTF8);

         if (FC_MSG_SYNTHESIS_IN_PROGRESS == m_enReadAloudSubStatus)
         {
            //Abort read aloud if synthesis speak triggered
            fc_messaging_tclGenericWrapper* poGenericWrapper = fc_messaging_tclApp::poGetInstance()->poGetGenericWrapper();

            if (poGenericWrapper)
            {
               if (FALSE == poGenericWrapper->bAbortPrompt())
               {
                  ETG_TRACE_ERR(("AbortPrompt request posting failed!!"));
                  vUpdateStatesAndReleaseAudio();
               }
               else
               {
                  m_enReadAloudStatus = FC_MSG_ABORT_IN_PROGRESS;
                  ETG_TRACE_USR4(("ReadAloudStatus::FC_MSG_ABORT_IN_PROGRESS"));
                  m_enReadAloudSubStatus = FC_MSG_READALOUDSUBSTATE_IDLE;
               }
            }
         }
      }
      break;

      case FC_MSG_ABORT_IN_PROGRESS:
      case FC_MSG_CHANNEL_RELEASE_IN_PROGRESS:
      {
         // Waiting request will be processed on receiving AbortPrompt/SynthesizeSpeak/ response or Audio release
         m_bNewReadRequestWaiting = TRUE;
         m_sNewRequestMessageText.bSet((tCString)sReadAloudContent, most_fi_tcl_String::FI_EN_UTF8);
      }
      break;

      default:
      {
         // should never happen
      }
   }
}

/*******************************************************************************
 * FUNCTION:      vHandleReadMessageAloudOnDeviceDisconnection
 * DESCRIPTION:   function is used to check and stop read aloud for disconnected device
 *
 * PARAMETER: deviceHandle - disconnected device handle
 *
 * RETURNVALUE:   none
 ********************************************************************************/
void ReadMessageAloud::vHandleReadMessageAloudOnDeviceDisconnection(uint8_t deviceHandle)
{
   ETG_TRACE_USR4(("ReadMessageAloud::vHandleReadMessageAloudOnDeviceDisconnection DeviceHandle: %d", deviceHandle));

   if((true == m_bNewReadRequestWaiting) && (deviceHandle == m_oNewRequestMessageHandle.u8DeviceHandle))
   {
      ETG_TRACE_USR4(("ReadMessageAloud::m_bNewReadRequestWaiting=false and ReadMessageAloudError returned"));

      m_bNewReadRequestWaiting = false;
      vSendReadMessageAloudError();
   }

   if(deviceHandle == m_oCurrentMessageHandle.u8DeviceHandle)
   {
      ETG_TRACE_USR4(("ReadMessageAloud::Device Disconnected Stopping ReadAloud"));

      vStopReadAloud(e8DeviceDisconnected);
   }
}

/*******************************************************************************
 * FUNCTION:      vProcessEvent
 * DESCRIPTION:   function is used to process various events of the Read message aloud state machine
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vProcessEvent(tU16 u16EventId, tVoid *varg)
{
   ETG_TRACE_USR4(("ReadMessageAloud::vProcessEvent EventID: '%d'", u16EventId));

   switch (m_enReadAloudStatus)
   {
      case FC_MSG_READALOUDSTATE_IDLE:
      {
         vHandleEventInIdle(u16EventId, varg);
      }
      break;

      case FC_MSG_GET_MSG_IN_PROGRESS:
      {
         vHandleEventInGetMsgInProgress(u16EventId, varg);
      }
      break;

      case FC_MSG_CHANNEL_ACQUISITION_IN_PROGRESS:
      {
         vHandleEventInChannelAcquisitionInProgress(u16EventId, varg);
      }
      break;

      case FC_MSG_READALOUD_IN_PROGRESS:
      {
         vHandleEventInReadAloudInProgress(u16EventId, varg);
      }
      break;

      case FC_MSG_ABORT_IN_PROGRESS:
      {
         vHandleEventInAbortInProgress(u16EventId, varg);
      }
      break;

      case FC_MSG_CHANNEL_RELEASE_IN_PROGRESS:
      {
         vHandleEventInChannelReleaseInProgress(u16EventId, varg);
      }
      break;

      default:
      {
         ETG_TRACE_ERR(("Invalid Read Aloud Status: '%d'!!", m_enReadAloudStatus));
      }
   }
}

/*******************************************************************************
 * FUNCTION:      vAudioRouteRequest
 * DESCRIPTION:   function is used to send audio route request to ARL
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vAudioRouteRequest()
{
   ETG_TRACE_USR4(("ReadMessageAloud::vAudioRouteRequest entered"));

   fc_messaging_tclGenericWrapper* poGenericWrapper = fc_messaging_tclApp::poGetInstance()->poGetGenericWrapper();

   if (poGenericWrapper)
   {
      //Trigger audio route request

      if (FALSE == poGenericWrapper->bRequestAVActivation(ARL_SRC_SMS))
      {
         ETG_TRACE_ERR(("Audio route request posting failed!!"));

         //Audio request posting failed, next request will be processed
         m_enReadAloudStatus = FC_MSG_READALOUDSTATE_IDLE;
         ETG_TRACE_USR4(("ReadAloudStatus::FC_MSG_READALOUDSTATE_IDLE"));

         //Discard Msg handle from the list
         vUpdateReadAloudMessageList(m_oCurrentMessageHandle);

         //ReadMessageAloud is completed, reset the DeviceHandle to default value.
         m_oCurrentMessageHandle.u8DeviceHandle = DEFAULT_DEVICE_HANDLE;

         //Send ReadMessageAloud response for the current request
         vSendReadMessageAloudError();
      }
      else
      {
         m_enReadAloudStatus = FC_MSG_CHANNEL_ACQUISITION_IN_PROGRESS;
         ETG_TRACE_USR4(("ReadAloudStatus::FC_MSG_CHANNEL_ACQUISITION_IN_PROGRESS"));

         //Start the timer for AudioRouteRequest response
         if (FALSE == m_bAudioRouteTimerRunning)
         {
            if (FALSE == fc_messaging_tclApp::m_poMessaging->bStartTimer(FC_MESSAGING_AUDIOROUTEREQUEST_WAIT_TIMER_ID,
                  FC_MESSAGING_AUDIOROUTEREQUEST_WAIT_TIMEOUT, FC_MESSAGING_AUDIOROUTEREQUEST_WAIT_TIMEOUT))
            {
               ETG_TRACE_ERR(("Unable to start the timer - FC_MESSAGING_AUDIOROUTEREQUEST_WAIT_TIMER_ID"));
            }
            else
            {
               m_bAudioRouteTimerRunning = TRUE;
            }
         }
      }
   }
}

/*******************************************************************************
 * FUNCTION:      vHandleAudioRouted
 * DESCRIPTION:   function is used to handle audio routed state
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vHandleAudioRouted()
{
   ETG_TRACE_USR4(("ReadMessageAloud::vHandleAudioRouted entered"));

   //Send ReadMessageAloud.MR
   vSendReadMessageAloudResult(most_fi_tcl_e8_MsgPlaybackAction::FI_EN_E8PBA_PLAY);
   vPrepareVoiceOutput();
}

/*******************************************************************************
 * FUNCTION:      vAudioReleaseRequest
 * DESCRIPTION:   function is used to send audio release request to ARL
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vAudioReleaseRequest()
{
   ETG_TRACE_USR4(("ReadMessageAloud::vAudioReleaseRequest entered"));

   fc_messaging_tclGenericWrapper* poGenericWrapper = fc_messaging_tclApp::poGetInstance()->poGetGenericWrapper();

   if (poGenericWrapper)
   {
      if (FALSE == poGenericWrapper->bRequestAVDeActivation(ARL_SRC_SMS))
      {
         ETG_TRACE_ERR(("Audio release request posting failed!!"));

         m_enReadAloudStatus = FC_MSG_READALOUDSTATE_IDLE;
         ETG_TRACE_USR4(("ReadAloudStatus::FC_MSG_READALOUDSTATE_IDLE"));

         // Process any waiting requests
         vHandleWaitingRequestsOnError();
      }
      else
      {
         m_enReadAloudStatus = FC_MSG_CHANNEL_RELEASE_IN_PROGRESS;
         ETG_TRACE_USR4(("ReadAloudStatus::FC_MSG_CHANNEL_RELEASE_IN_PROGRESS"));

         //Start the timer for vAudioReleaseRequest response
         if ((FALSE == m_bAudioRouteTimerRunning) && (FALSE == m_bAudioSrcChangedByArl))
         {
            if (FALSE == fc_messaging_tclApp::m_poMessaging->bStartTimer(FC_MESSAGING_AUDIOROUTEREQUEST_WAIT_TIMER_ID,
                  FC_MESSAGING_AUDIOROUTEREQUEST_WAIT_TIMEOUT, FC_MESSAGING_AUDIOROUTEREQUEST_WAIT_TIMEOUT))
            {
               ETG_TRACE_ERR(("Unable to start the timer - FC_MESSAGING_AUDIOROUTEREQUEST_WAIT_TIMER_ID"));
            }
            else
            {
               m_bAudioRouteTimerRunning = TRUE;
            }
         }
      }

      if (TRUE == m_bAudioSrcChangedByArl)
      {
         m_bAudioSrcChangedByArl = FALSE;

         //If any Stop/Read request pending then send the MethodResult for the request, check
         if ((TRUE == m_bNewReadRequestWaiting) || (FC_MSG_STOP_IN_WAIT == m_enReadAloudSubStatus))
         {
            m_bNewReadRequestWaiting = FALSE;
            vSendReadMessageAloudError();
         }

         m_enReadAloudStatus = FC_MSG_READALOUDSTATE_IDLE;
         ETG_TRACE_USR4(("ReadAloudStatus::FC_MSG_READALOUDSTATE_IDLE"));
         m_enReadAloudSubStatus = FC_MSG_READALOUDSUBSTATE_IDLE;

         //ReadMessageAloud is completed, reset the DeviceHandle to default value.
         m_oCurrentMessageHandle.u8DeviceHandle = DEFAULT_DEVICE_HANDLE;
      }
   }
}

/*******************************************************************************
 * FUNCTION:      vHandleAudioReleased
 * DESCRIPTION:   function is used to handle audio release state
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vHandleAudioReleased()
{
   ETG_TRACE_USR4(("ReadMessageAloud::vHandleAudioReleased entered"));

   switch (m_enReadAloudStatus)
   {
      case FC_MSG_CHANNEL_RELEASE_IN_PROGRESS:
      {
         if (TRUE == m_bNewReadRequestWaiting)
         {
            vHandleWaitingReadRequest();
         }
         else
         {
            m_enReadAloudStatus = FC_MSG_READALOUDSTATE_IDLE;
            ETG_TRACE_USR4(("ReadAloudStatus::FC_MSG_READALOUDSTATE_IDLE"));
            m_enReadAloudSubStatus = FC_MSG_READALOUDSUBSTATE_IDLE;

            //ReadMessageAloud is completed, reset the DeviceHandle to default value.
            m_oCurrentMessageHandle.u8DeviceHandle = DEFAULT_DEVICE_HANDLE;
         }
      }
      break;

      case FC_MSG_CHANNEL_ACQUISITION_IN_PROGRESS:
      {
         ETG_TRACE_USR4(("FC_MSG_CHANNEL_ACQUISITION_IN_PROGRESS"));

         // Audio source not Routed, Send ReadMessageAloud.Error for current request
         vSendReadMessageAloudError();

         // Discard message handle from list, send message list change update (not required to send MessageListChange update since PB:Pause not sent)
         vSendMessageListChangeUpdate(m_oCurrentMessageHandle, most_fi_tcl_e8_MsgPlaybackState::FI_EN_E8PBS_STOPPED);
      }
      break;

      default:
      {
         ETG_TRACE_ERR(("Invalid State: '%u'!!", ETG_ENUM(TR_MSG_READALOUDSTATUS, m_enReadAloudStatus)));
      }
   }
}

/*******************************************************************************
 * FUNCTION:      vPrepareVoiceOutput
 * DESCRIPTION:   function is used to send PrepareVoiceOutput request to ARL
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vPrepareVoiceOutput()
{
   ETG_TRACE_USR4(("ReadMessageAloud::vPrepareVoiceOutput"));

   fc_messaging_tclGenericWrapper* poGenericWrapper = fc_messaging_tclApp::poGetInstance()->poGetGenericWrapper();

   if (poGenericWrapper)
   {
      if (FALSE == poGenericWrapper->bPrepareVoiceOutput())
      {
         ETG_TRACE_ERR(("PrepareVoiceOutput request posting failed!!"));
         vUpdateStatesAndReleaseAudio();
      }
      else
      {
         m_enReadAloudStatus = FC_MSG_READALOUD_IN_PROGRESS;
         ETG_TRACE_USR4(("ReadAloudStatus::FC_MSG_READALOUD_IN_PROGRESS"));

         m_enReadAloudSubStatus = FC_MSG_PREPARE_IN_PROGRESS;
         ETG_TRACE_USR4(("FC_MSG_PREPARE_IN_PROGRESS"));
      }
   }
}

/*******************************************************************************
 * FUNCTION:      vPrepareVoiceOutputAck
 * DESCRIPTION:   function is used to handle PrepareVoiceOutput acknowledgement
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vPrepareVoiceOutputAck(tBool bSuccess)
{
   ETG_TRACE_USR4(("ReadMessageAloud::vPrepareVoiceOutputAck Entered"));

   // Check if any other request(s) is/are waiting

   if (TRUE == m_bAudioSrcChangedByArl)
   {
      vUpdateStatesAndReleaseAudio();
   }
   else if (TRUE == m_bNewReadRequestWaiting)
   {
      vUpdateReadAloudMessageList(m_oCurrentMessageHandle);
      vHandleWaitingReadRequest();
   }
   else if (FC_MSG_STOP_IN_WAIT == m_enReadAloudSubStatus)
   {
      m_enReadAloudSubStatus = FC_MSG_READALOUDSUBSTATE_IDLE;
      vAudioReleaseRequest();
      vSendReadMessageAloudResult(most_fi_tcl_e8_MsgPlaybackAction::FI_EN_E8PBA_STOP);
   }
   else
   {
      if (TRUE == bSuccess)
      {
         fc_messaging_tclGenericWrapper* poGenericWrapper = fc_messaging_tclApp::poGetInstance()->poGetGenericWrapper();
         if (poGenericWrapper)
         {
            //Read the text message
            if (FALSE == poGenericWrapper->bSynthesizeSpeak(m_sText))
            {
               ETG_TRACE_ERR(("SynthesizeSpeak request posting failed!!"));
               vUpdateStatesAndReleaseAudio();
            }
            else
            {
               m_enReadAloudSubStatus = FC_MSG_SYNTHESIS_IN_PROGRESS;
            }
         }
      }
      else
      {
         //Prepare failed
         vUpdateStatesAndReleaseAudio();
      }
   }
}

/*******************************************************************************
 * FUNCTION:      vSynthesizeSpeakAck
 * DESCRIPTION:   function is used to handle Synthesize Speak acknowledgement
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vSynthesizeSpeakAck(tenSynthesizeSpeakResult enSynthesizeSpeakResult)
{
   ETG_TRACE_USR4(("ReadMessageAloud::vSynthesizeSpeakAck Entered SynthesizeResult '%u'", ETG_ENUM(TR_SYNTHESIZESPEAKRESULT, enSynthesizeSpeakResult)));

   switch (enSynthesizeSpeakResult)
   {
      case FC_MSG_SYNTHESIZESPEAKRESULT_FIRST:
      case FC_MSG_SYNTHESIZESPEAKRESULT_MIDDLE:
      {
         fc_messaging_tclService_Messaging::m_poMessagingService->messagingAudioPlaybackState(most_fi_tcl_e8_MsgPlaybackState::FI_EN_E8PBS_PLAYING);
         vSendMessageListChangeUpdate(m_oCurrentMessageHandle, most_fi_tcl_e8_MsgPlaybackState::FI_EN_E8PBS_PLAYING);
      }
      break;

      case FC_MSG_SYNTHESIZESPEAKRESULT_LAST:
      case FC_MSG_SYNTHESIZESPEAKRESULT_ERROR:
      {
         fc_messaging_tclService_Messaging::m_poMessagingService->messagingAudioPlaybackState(
               most_fi_tcl_e8_MsgPlaybackState::FI_EN_E8PBS_STOPPED);
         vSendMessageListChangeUpdate(m_oCurrentMessageHandle, most_fi_tcl_e8_MsgPlaybackState::FI_EN_E8PBS_STOPPED);

         if (TRUE == m_bAudioSrcChangedByArl)
         {
            vAudioReleaseRequest();
         }
         else
         {
            if (FC_MSG_STOP_IN_PROGRESS == m_enReadAloudSubStatus)
            {
               m_enReadAloudSubStatus = FC_MSG_READALOUDSUBSTATE_IDLE;
               vSendReadMessageAloudResult(most_fi_tcl_e8_MsgPlaybackAction::FI_EN_E8PBA_STOP);
            }

            if (TRUE == m_bNewReadRequestWaiting)
            {
               vHandleWaitingReadRequest();
            }
            else
            {
               vAudioReleaseRequest();
            }
         }
      }
      break;

      default:
      {
         // Cannot come here
      }
   }
}

/*******************************************************************************
 * FUNCTION:      vAbortPromptAck
 * DESCRIPTION:   function is used to handle Abort Prompt acknowledgement
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vAbortPromptAck(tBool bSuccess)
{
   ETG_TRACE_USR4(("ReadMessageAloud::vAbortPromptAck Entered"));

   fc_messaging_tclGenericWrapper* poGenericWrapper = fc_messaging_tclApp::poGetInstance()->poGetGenericWrapper();

   if (poGenericWrapper)
   {
      if ((TRUE == m_bAudioSrcChangedByArl) && (FALSE == bSuccess)) //Abort Error on Source change
      {
         vUpdateStatesAndReleaseAudio();

         //If any Stop/Read request pending then send the MethodResult for the request, check
         if ((TRUE == m_bNewReadRequestWaiting) || (m_enReadAloudSubStatus == FC_MSG_STOP_IN_WAIT))
         {
            m_bNewReadRequestWaiting = FALSE;
            vSendReadMessageAloudError();
         }

         m_enReadAloudStatus = FC_MSG_READALOUDSTATE_IDLE;
         ETG_TRACE_USR4(("ReadAloudStatus::FC_MSG_READALOUDSTATE_IDLE"));
         m_enReadAloudSubStatus = FC_MSG_READALOUDSUBSTATE_IDLE;

         //ReadMessageAloud is completed, reset the DeviceHandle to default value.
         m_oCurrentMessageHandle.u8DeviceHandle = DEFAULT_DEVICE_HANDLE;
      }
      else if (FC_MSG_STOP_IN_PROGRESS == m_enReadAloudSubStatus)
      {
         m_enReadAloudSubStatus = FC_MSG_READALOUDSUBSTATE_IDLE;
         vSendReadMessageAloudResult(most_fi_tcl_e8_MsgPlaybackAction::FI_EN_E8PBA_STOP);

         // Abort failed so send Audio release request
         if (FALSE == bSuccess)
         {
            vUpdateStatesAndReleaseAudio();
         }
      }
      else if (TRUE == m_bNewReadRequestWaiting)
      {
         if (FALSE == bSuccess)
         {
            // Update current message status and proceed with the next request
            fc_messaging_tclService_Messaging::m_poMessagingService->messagingAudioPlaybackState(
                  most_fi_tcl_e8_MsgPlaybackState::FI_EN_E8PBS_STOPPED);
            vSendMessageListChangeUpdate(m_oCurrentMessageHandle, most_fi_tcl_e8_MsgPlaybackState::FI_EN_E8PBS_STOPPED);
            vHandleWaitingReadRequest();
         }

         // Incase of AbortPrompt success, we will wait for SynthesizeSpeak.Method Response
      }
      else
      {
         // Cannot come here
      }
   }
}

/*******************************************************************************
 * FUNCTION:      vHandleEventInIdle
 * DESCRIPTION:   function is used to handle events received in Idle Read aloud state
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vHandleEventInIdle(tU16 u16EventId, tVoid *varg)
{
   ETG_TRACE_USR4(("ReadMessageAloud::vHandleEventInIdle EventID: %d", u16EventId));
   (tVoid) varg; //lint fix

   //Stop the timer and reset the variable
   if (TRUE == m_bAudioRouteTimerRunning)
   {
      if (FALSE == fc_messaging_tclApp::m_poMessaging->bStopTimer(FC_MESSAGING_AUDIOROUTEREQUEST_WAIT_TIMER_ID))
      {
         ETG_TRACE_ERR(("Unable to stop the timer FC_MESSAGING_AUDIOROUTEREQUEST_WAIT_TIMER_ID"));
      }
      else
      {
         m_bAudioRouteTimerRunning = FALSE;
      }
   }
}

/*******************************************************************************
 * FUNCTION:      vHandleEventInGetMsgInProgress
 * DESCRIPTION:   function is used to handle events received while text message
 *                retrieval for Read aloud is in progress
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vHandleEventInGetMsgInProgress(tU16 u16EventId, tVoid *varg)
{
   ETG_TRACE_USR4(("ReadMessageAloud::vHandleEventInGetMsgInProgress EventID: %d", u16EventId));
   (tVoid) varg; //lint fix

   switch (u16EventId)
   {
      case FC_MESSAGING_SM_EVENT_GETMESSAGE_SUCCESS:
      {
         if(e8ReadMessageAloudMethod == m_e8ReadMessageAloudCaller)
         {
            most_fi_tcl_MsgMessageHandle oMessageHandle = m_oCurrentMessageHandle;
            m_aoReadAloudMessageHandle.append(oMessageHandle); //discard message handle once play completed/failed to play
         }

         if (FC_MSG_CHANNEL_ALREADY_ACQUIRED == m_enReadAloudSubStatus)
         {
            //Since audio is already routed, send ReadMessageAloud.MR
            vSendReadMessageAloudResult(most_fi_tcl_e8_MsgPlaybackAction::FI_EN_E8PBA_PLAY);

            vPrepareVoiceOutput(); //Audio source already routed
         }
         else
         {
            vAudioRouteRequest();
         }
      }
      break;

      case FC_MESSAGING_SM_EVENT_GETMESSAGE_FAILURE:
      {
         if (FC_MSG_CHANNEL_ALREADY_ACQUIRED == m_enReadAloudSubStatus)
         {
            vAudioReleaseRequest();
         }
         else
         {
            m_enReadAloudStatus = FC_MSG_READALOUDSTATE_IDLE;
            ETG_TRACE_USR4(("ReadAloudStatus::FC_MSG_READALOUDSTATE_IDLE"));
            m_enReadAloudSubStatus = FC_MSG_READALOUDSUBSTATE_IDLE;

            //ReadMessageAloud is completed, reset the DeviceHandle to default value.
            m_oCurrentMessageHandle.u8DeviceHandle = DEFAULT_DEVICE_HANDLE;
         }
      }
      break;

      default:
      {
         ETG_TRACE_ERR(("Invalid Event received in State FC_MSG_GET_MSG_IN_PROGRESS"));
      }
   }
}

/*******************************************************************************
 * FUNCTION:      vHandleEventInChannelAcquisitionInProgress
 * DESCRIPTION:   function is used to handle events received while audio route
 *                allocation is in progress
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vHandleEventInChannelAcquisitionInProgress(tU16 u16EventId, tVoid *varg)
{
   ETG_TRACE_USR4(("ReadMessageAloud::vHandleEventInChannelAcquisitionInProgress EventID: %d", u16EventId));

   switch (u16EventId)
   {
      case FC_MESSAGING_SM_EVENT_SOURCEACTIVITY_RESPONSE:
      {
         trFC_Msg_ProcessEventArg* rFC_Msg_ProcessEventArg = (trFC_Msg_ProcessEventArg*) varg;

         if (rFC_Msg_ProcessEventArg)
         {
            if (ARL_EN_ISRC_ACT_ON == rFC_Msg_ProcessEventArg->enArlSrcActivity)
            {
               vHandleAudioRouted();
            }
            else
            {
               //ARL_EN_ISRC_ACT_OFF/ARL_EN_ISRC_ACT_PAUSE received
               vHandleAudioReleased();
            }

            //Stop the timer and reset the variable on src activity update
            if (TRUE == m_bAudioRouteTimerRunning)
            {
               if (FALSE == fc_messaging_tclApp::m_poMessaging->bStopTimer(FC_MESSAGING_AUDIOROUTEREQUEST_WAIT_TIMER_ID))
               {
                  ETG_TRACE_ERR(("Unable to stop the timer FC_MESSAGING_AUDIOROUTEREQUEST_WAIT_TIMER_ID"));
               }
               else
               {
                  m_bAudioRouteTimerRunning = FALSE;
               }
            }
         }
      }
      break;

      case FC_MESSAGING_SM_EVENT_AUDIOROUTE_TIMER_EXPIRED:
      {
         vAudioTimerExpired();
      }
      break;

      default:
      {
         ETG_TRACE_ERR(("Invalid Event received in State FC_MSG_CHANNEL_ACQUISITION_IN_PROGRESS"));
      }
   }
}

/*******************************************************************************
 * FUNCTION:      vHandleEventInReadAloudInProgress
 * DESCRIPTION:   function is used to handle events received while text message
 *                read aloud is in progress
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vHandleEventInReadAloudInProgress(tU16 u16EventId, tVoid *varg)
{
   ETG_TRACE_USR4(("ReadMessageAloud::vHandleEventInReadAloudInProgress EventID: %d", u16EventId));

   switch (u16EventId)
   {
      case FC_MESSAGING_SM_EVENT_PREPAREVOICEOUTPUT_RESPONSE:
      {
         trFC_Msg_ProcessEventArg* rFC_Msg_ProcessEventArg = (trFC_Msg_ProcessEventArg*) varg;

         if (rFC_Msg_ProcessEventArg)
         {
            vPrepareVoiceOutputAck(rFC_Msg_ProcessEventArg->bSuccess);
         }
      }
      break;

      case FC_MESSAGING_SM_EVENT_SYNTHESIZESPEAK_RESPONSE:
      {
         trFC_Msg_ProcessEventArg* rFC_Msg_ProcessEventArg = (trFC_Msg_ProcessEventArg*) varg;

         if (rFC_Msg_ProcessEventArg)
         {
            vSynthesizeSpeakAck(rFC_Msg_ProcessEventArg->enSynthesizeSpeakResult);
         }
      }
      break;

      case FC_MESSAGING_SM_EVENT_SOURCEACTIVITY_RESPONSE:
      {
         //Channel released by ARL
         trFC_Msg_ProcessEventArg* rFC_Msg_ProcessEventArg = (trFC_Msg_ProcessEventArg*) varg;

         if (rFC_Msg_ProcessEventArg)
         {
            if (ARL_EN_ISRC_ACT_ON != rFC_Msg_ProcessEventArg->enArlSrcActivity)
            {
               vHandleSourceChangeDuringReadAloud();
            }
         }
      }
      break;

      default:
      {
         ETG_TRACE_ERR(("Invalid Event received in State FC_MSG_READALOUD_IN_PROGRESS"));
      }
   }
}

/*******************************************************************************
 * FUNCTION:      vHandleEventInAbortInProgress
 * DESCRIPTION:   function is used to handle events received abort text message
 *                read aloud is in progress
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vHandleEventInAbortInProgress(tU16 u16EventId, tVoid *varg)
{
   ETG_TRACE_USR4(("ReadMessageAloud::vHandleEventInAbortInProgress EventID: %d", u16EventId));

   switch (u16EventId)
   {
      case FC_MESSAGING_SM_EVENT_ABORTPROMPT_RESPONSE:
      {
         trFC_Msg_ProcessEventArg* rFC_Msg_ProcessEventArg = (trFC_Msg_ProcessEventArg*) varg;

         if (rFC_Msg_ProcessEventArg)
         {
            vAbortPromptAck(rFC_Msg_ProcessEventArg->bSuccess);
         }
      }
      break;

      case FC_MESSAGING_SM_EVENT_SYNTHESIZESPEAK_RESPONSE:
      {
         trFC_Msg_ProcessEventArg* rFC_Msg_ProcessEventArg = (trFC_Msg_ProcessEventArg*) varg;

         if (rFC_Msg_ProcessEventArg)
         {
            vSynthesizeSpeakAck(rFC_Msg_ProcessEventArg->enSynthesizeSpeakResult);
         }
      }
      break;

      case FC_MESSAGING_SM_EVENT_SOURCEACTIVITY_RESPONSE:
      {
         //Channel released by ARL
         trFC_Msg_ProcessEventArg* rFC_Msg_ProcessEventArg = (trFC_Msg_ProcessEventArg*) varg;

         if (rFC_Msg_ProcessEventArg)
         {
            if (ARL_EN_ISRC_ACT_ON != rFC_Msg_ProcessEventArg->enArlSrcActivity)
            {
               m_bAudioSrcChangedByArl = TRUE;
               // Rest handled on Abort/Synthesis response
            }
         }
      }
      break;

      default:
      {
         ETG_TRACE_ERR(("Invalid Event received in State FC_MSG_ABORT_IN_PROGRESS"));
      }
   }
}

/*******************************************************************************
 * FUNCTION:      vHandleEventInChannelReleaseInProgress
 * DESCRIPTION:   function is used to handle events received audio route release
 *                is in progress
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vHandleEventInChannelReleaseInProgress(tU16 u16EventId, tVoid *varg)
{
   ETG_TRACE_USR4(("ReadMessageAloud::vHandleEventInChannelReleaseInProgress EventID: %d", u16EventId));

   switch (u16EventId)
   {
      case FC_MESSAGING_SM_EVENT_SOURCEACTIVITY_RESPONSE:
      {
         trFC_Msg_ProcessEventArg* rFC_Msg_ProcessEventArg = (trFC_Msg_ProcessEventArg*) varg;

         if (rFC_Msg_ProcessEventArg)
         {
            if (ARL_EN_ISRC_ACT_ON != rFC_Msg_ProcessEventArg->enArlSrcActivity)
            {
               vHandleAudioReleased();
            }
            else
            {
               ETG_TRACE_USR4(("Audio Routed during channel release!!"));
            }

            //Stop the timer and reset the variable on src activity update
            if (TRUE == m_bAudioRouteTimerRunning)
            {
               if (FALSE == fc_messaging_tclApp::m_poMessaging->bStopTimer(FC_MESSAGING_AUDIOROUTEREQUEST_WAIT_TIMER_ID))
               {
                  ETG_TRACE_ERR(("Unable to stop the timer FC_MESSAGING_AUDIOROUTEREQUEST_WAIT_TIMER_ID"));
               }
               else
               {
                  m_bAudioRouteTimerRunning = FALSE;
               }
            }
         }
      }
      break;

      case FC_MESSAGING_SM_EVENT_AUDIOROUTE_TIMER_EXPIRED:
      {
         vAudioTimerExpired();
      }
      break;

      default:
      {
         ETG_TRACE_ERR(("Invalid Event received in State FC_MSG_CHANNEL_RELEASE_IN_PROGRESS"));
      }
   }
}

/*******************************************************************************
 * FUNCTION:      vHandleWaitingReadRequest
 * DESCRIPTION:   function is used to handle any waiting read aloud request
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vHandleWaitingReadRequest()
{
   ETG_TRACE_USR4(("ReadMessageAloud::vHandleWaitingReadRequest"));

   if(e8ReadMessageAloudMethod == m_e8ReadMessageAloudCaller)
   {
      m_oCurrentMessageHandle = m_oNewRequestMessageHandle;
      FileSystemManager* poFileSystemManager = FileSystemManager::poFileSystemManager(m_oCurrentMessageHandle);

      if (poFileSystemManager)
      {
         if (FC_MSG_CHANNEL_RELEASE_IN_PROGRESS != m_enReadAloudStatus)
         {
            m_enReadAloudSubStatus = FC_MSG_CHANNEL_ALREADY_ACQUIRED;
         }

         m_enReadAloudStatus = FC_MSG_GET_MSG_IN_PROGRESS;
         ETG_TRACE_USR4(("ReadAloudStatus::FC_MSG_GET_MSG_IN_PROGRESS"));
         poFileSystemManager->vGetMessage(m_oCurrentMessageHandle, e8ReadMessageAloud);
      }
      else
      {
         m_enReadAloudStatus = FC_MSG_READALOUDSTATE_IDLE;
         ETG_TRACE_USR4(("ReadAloudStatus::FC_MSG_READALOUDSTATE_IDLE"));
         m_enReadAloudSubStatus = FC_MSG_READALOUDSUBSTATE_IDLE;

         //ReadMessageAloud is completed, reset the DeviceHandle to default value.
         m_oCurrentMessageHandle.u8DeviceHandle = DEFAULT_DEVICE_HANDLE;

         vSendReadMessageAloudError();
      }
   }
   else
   {
      //start playback for the received text
      if (FC_MSG_CHANNEL_RELEASE_IN_PROGRESS != m_enReadAloudStatus)
      {
         m_enReadAloudSubStatus = FC_MSG_CHANNEL_ALREADY_ACQUIRED;
      }

      m_enReadAloudStatus = FC_MSG_GET_MSG_IN_PROGRESS;
      ETG_TRACE_USR4(("ReadAloudStatus::FC_MSG_GET_MSG_IN_PROGRESS"));

      m_sText = m_sNewRequestMessageText;

      vProcessEvent(FC_MESSAGING_SM_EVENT_GETMESSAGE_SUCCESS);
   }

   m_bNewReadRequestWaiting = FALSE;
}

/*******************************************************************************
 * FUNCTION:      vHandleSourceChangeDuringReadAloud
 * DESCRIPTION:   function is used to handle read aloud when audio source is
 *                changed by the ARL
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vHandleSourceChangeDuringReadAloud()
{
   ETG_TRACE_USR4(("ReadMessageAloud::vHandleSourceChangeDuringReadAloud"));

   switch (m_enReadAloudSubStatus)
   {
      case FC_MSG_PREPARE_IN_PROGRESS:
      case FC_MSG_STOP_IN_WAIT:
      {
         m_bAudioSrcChangedByArl = TRUE;
         vUpdateStatesAndReleaseAudio();
      }
      break;

      case FC_MSG_SYNTHESIS_IN_PROGRESS:
      {
         fc_messaging_tclGenericWrapper* poGenericWrapper = fc_messaging_tclApp::poGetInstance()->poGetGenericWrapper();

         if(poGenericWrapper)
         {
            if (FALSE == poGenericWrapper->bAbortPrompt())
            {
               ETG_TRACE_ERR(("AbortPrompt request posting failed!!"));
               m_bAudioSrcChangedByArl = TRUE;
               vUpdateStatesAndReleaseAudio();
            }
            else
            {
               m_enReadAloudStatus = FC_MSG_ABORT_IN_PROGRESS;
               ETG_TRACE_USR4(("ReadAloudStatus::FC_MSG_ABORT_IN_PROGRESS"));
               m_bAudioSrcChangedByArl = TRUE;
               m_enReadAloudSubStatus = FC_MSG_READALOUDSUBSTATE_IDLE;
            }

         }
      }
      break;

      default:
      {
         ETG_TRACE_USR4(("SourceActivity Off received in Invalid state!!"));
      }
   }
}

/*******************************************************************************
 * FUNCTION:      vAudioTimerExpired
 * DESCRIPTION:   function is used handle read aloud if ARL response timer expired
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vAudioTimerExpired()
{
   ETG_TRACE_USR4(("ReadMessageAloud::vAudioTimerExpired"));

   m_bAudioRouteTimerRunning = FALSE;

   if (FC_MSG_CHANNEL_ACQUISITION_IN_PROGRESS == m_enReadAloudStatus)
   {
      vSendMessageListChangeUpdate(m_oCurrentMessageHandle, most_fi_tcl_e8_MsgPlaybackState::FI_EN_E8PBS_STOPPED);
      vSendReadMessageAloudError();
   }
   else if (FC_MSG_CHANNEL_RELEASE_IN_PROGRESS == m_enReadAloudStatus)
   {
      fc_messaging_tclService_Messaging::m_poMessagingService->messagingAudioPlaybackState(
            most_fi_tcl_e8_MsgPlaybackState::FI_EN_E8PBS_STOPPED);
      vSendMessageListChangeUpdate(m_oCurrentMessageHandle, most_fi_tcl_e8_MsgPlaybackState::FI_EN_E8PBS_STOPPED);
   }

   m_enReadAloudStatus = FC_MSG_READALOUDSTATE_IDLE;
   ETG_TRACE_USR4(("ReadAloudStatus::FC_MSG_READALOUDSTATE_IDLE"));
   m_enReadAloudSubStatus = FC_MSG_READALOUDSUBSTATE_IDLE;

   //ReadMessageAloud is completed, reset the DeviceHandle to default value.
   m_oCurrentMessageHandle.u8DeviceHandle = DEFAULT_DEVICE_HANDLE;

   if (TRUE == m_bNewReadRequestWaiting)
      vHandleWaitingReadRequest();
}

/*******************************************************************************
 * FUNCTION:      vUpdateStatesAndReleaseAudio
 * DESCRIPTION:   function is used update current read aloud state and release
 *                audio route
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vUpdateStatesAndReleaseAudio()
{
   ETG_TRACE_USR4(("ReadMessageAloud::vUpdateStatesAndReleaseAudio"));

   fc_messaging_tclService_Messaging::m_poMessagingService->messagingAudioPlaybackState(
         most_fi_tcl_e8_MsgPlaybackState::FI_EN_E8PBS_STOPPED);
   vSendMessageListChangeUpdate(m_oCurrentMessageHandle, most_fi_tcl_e8_MsgPlaybackState::FI_EN_E8PBS_STOPPED);
   vAudioReleaseRequest();
}

/*******************************************************************************
 * FUNCTION:      vSendMessageListChangeUpdate
 * DESCRIPTION:   function is used to send MessageListChange status property
 *                update
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vSendMessageListChangeUpdate(most_fi_tcl_MsgMessageHandle oMessageHandle,
      most_fi_tcl_e8_MsgPlaybackState::tenType enPlayState)
{
   ETG_TRACE_USR4(("ReadMessageAloud::vSendMessageListChangeUpdate PlayState '%u' received",
         ETG_ENUM(TR_MSG_PLAYBACKSTATE, enPlayState)));

   if (!s_poReadMessageAloudObject)
   {
      ETG_TRACE_USR4(("vSendMessageListChangeUpdate: ReadMessageAloudObject not found"));
      return;
   }

   if (m_aoReadAloudMessageHandle.isEmpty())
   {
      ETG_TRACE_USR4(("vSendMessageListChangeUpdate: ReadMessageAloudList is empty"));
      return;
   }

   most_fi_tcl_MsgMessageHandle oCurrMessageHandle;
   QList<most_fi_tcl_MsgMessageHandle>::iterator iterator;

   for (iterator = m_aoReadAloudMessageHandle.begin();
         iterator != m_aoReadAloudMessageHandle.end(); iterator++)
   {
      if (*iterator == oMessageHandle)
      {
         oCurrMessageHandle = *iterator;
         break;
      }
   }

   if (iterator == m_aoReadAloudMessageHandle.end())
   {
      return; // No Message found for this info
   }

   if (most_fi_tcl_e8_MsgPlaybackState::FI_EN_E8PBS_STOPPED == enPlayState)
   {
      ETG_TRACE_USR4(("Stop received so remove the info of message"));
      m_aoReadAloudMessageHandle.erase(iterator);
   }

   FileSystemManager* poFileSystemManager = FileSystemManager::poFileSystemManager(oCurrMessageHandle);

   if (poFileSystemManager)
   {
      ETG_TRACE_USR4(("FileSystem found"));

      tU32 u32ListHandle = oCurrMessageHandle.u32MsgHandleLower;
      FolderNode* poFolderNode = poFileSystemManager->poSearchNodeByListHandle(u32ListHandle);

      if (poFolderNode)
      {
         tU32 u32MessagePosition = 0;
         QVector<most_fi_tcl_MsgMessageListSliceResultItem>::iterator it;

         for (it = poFolderNode->m_pvMessageList->begin();
               it != poFolderNode->m_pvMessageList->end();
               it++, u32MessagePosition++)
         {
            if (it->oMessageHandle == oCurrMessageHandle)
            {
               it->e8PlaybackState.enType = enPlayState;
               break;
            }
         }

         most_msgfi_tclMsgMessageListChangeStatus oMessageListChangeStatus;

         oMessageListChangeStatus.u16ListHandle = static_cast<tU16>(u32ListHandle);
         oMessageListChangeStatus.e8ListChangeType.enType = most_fi_tcl_e8_MsgListChangeType::FI_EN_E8LCH_ITEMS_CHANGED;
         oMessageListChangeStatus.u16ListLength = static_cast<tU16>(poFolderNode->m_pvMessageList->size());
         oMessageListChangeStatus.u32NumItems = 1; // Only one can play at a time
         oMessageListChangeStatus.oChangeItems.u32Items.push_back(u32MessagePosition);

         ETG_TRACE_USR4(("List change sent"));
         fc_messaging_tclService_Messaging::m_poMessagingService->vMessageListChange(oMessageListChangeStatus);
      }
   }
}

/*******************************************************************************
 * FUNCTION:      vUpdateReadAloudMessageList
 * DESCRIPTION:   function is used to update the list containing the message
 *                handles of all Read Aloud requests
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vUpdateReadAloudMessageList(most_fi_tcl_MsgMessageHandle oMessageHandle)
{
   ETG_TRACE_USR4(("ReadMessageAloud::vUpdateReadAloudMessageList"));

   if (!s_poReadMessageAloudObject)
   {
      ETG_TRACE_USR4(("vSendMessageListChangeUpdate: ReadMessageAloudObject not found"));
      return;
   }

   if (m_aoReadAloudMessageHandle.isEmpty())
   {
      ETG_TRACE_USR4(("vSendMessageListChangeUpdate: ReadMessageAloudList is empty"));
      return;
   }

   most_fi_tcl_MsgMessageHandle oCurrMessageHandle;
   QList<most_fi_tcl_MsgMessageHandle>::iterator iterator;

   for (iterator = m_aoReadAloudMessageHandle.begin();
         iterator != m_aoReadAloudMessageHandle.end(); iterator++)
   {
      if (*iterator == oMessageHandle)
      {
         ETG_TRACE_USR4(("Match found for the Message Handle"));
         oCurrMessageHandle = *iterator;
         break;
      }
   }

   if (iterator == m_aoReadAloudMessageHandle.end())
   {
      return; // No Message found for this info
   }

   m_aoReadAloudMessageHandle.erase(iterator);
}

/*******************************************************************************
 * FUNCTION:      vHandleWaitingRequestsOnError
 * DESCRIPTION:   function is used to send response for any pending read/stop
 *                requests on any error
 *
 * PARAMETER:
 *
 * RETURNVALUE:   none
 ********************************************************************************/
tVoid ReadMessageAloud::vHandleWaitingRequestsOnError()
{
   ETG_TRACE_USR4(("ReadMessageAloud::vHandleWaitingRequestsOnError"));

   if (TRUE == m_bNewReadRequestWaiting)
   {
      m_bNewReadRequestWaiting = FALSE;
      vSendReadMessageAloudError();
   }
   else if (FC_MSG_STOP_IN_WAIT == m_enReadAloudSubStatus)
   {
      vSendReadMessageAloudError();
   }
   m_enReadAloudStatus = FC_MSG_READALOUDSTATE_IDLE;
   m_enReadAloudSubStatus = FC_MSG_READALOUDSUBSTATE_IDLE;

   //ReadMessageAloud is completed, reset the DeviceHandle to default value.
   m_oCurrentMessageHandle.u8DeviceHandle = DEFAULT_DEVICE_HANDLE;
}

tVoid ReadMessageAloud::vSendReadMessageAloudResult(most_fi_tcl_e8_MsgPlaybackAction::tenType enPlaybackAction)
{
   if(e8ReadMessageAloudExtendedMethod == m_e8ReadMessageAloudCaller)
   {
      fc_messaging_tclService_Messaging::m_poMessagingService->vHandle_ReadMessageAloudExtended_Return(enPlaybackAction);
   }
   else if(e8ReadMessageAloudMethod == m_e8ReadMessageAloudCaller)
   {
      fc_messaging_tclService_Messaging::m_poMessagingService->vHandle_ReadMessageAloud_Return(enPlaybackAction);
   }
   else
   {
      //ReadMessageAloud Stop requested due to device disconnection
   }
}

tVoid ReadMessageAloud::vSendReadMessageAloudError()
{
   if(e8ReadMessageAloudExtendedMethod == m_e8ReadMessageAloudCaller)
   {
      fc_messaging_tclService_Messaging::m_poMessagingService->vHandle_ReadMessageAloudExtended_Error();
   }
   else if(e8ReadMessageAloudMethod == m_e8ReadMessageAloudCaller)
   {
      fc_messaging_tclService_Messaging::m_poMessagingService->vHandle_ReadMessageAloud_Error();
   }
   else
   {
      //ReadMessageAloud Stop requested due to device disconnection
   }
}
