/*******************************************************************************
*                                                                               
* FILE:          FC_Messaging_main.cpp                                   
*                                                                               
* SW-COMPONENT:  FC_Messaging application                                  
*                                                                               
* PROJECT:       Bosch                                                               
*                                                                               
* DESCRIPTION:   CCA Application.                                               
*                                                                               
* AUTHOR:        Arun Choudhary                                                               
*                                                                               
* COPYRIGHT:    (c) 2010 Robert Bosch GmbH, Hildesheim                          
*                                                                               
*******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/
#include "FC_Messaging_Debug.h"
#include "FC_Messaging_main.h"

#include "Audio/FC_Messaging_AudioRouteHandler.h"
#include "GenericWrapper/FC_Messaging_GenericWrapper.h"
#include "FC_Messaging_clienthandler_SDS.h"
#include "ReadMessageAloud/Gen3/FC_Messaging_ReadMessageAloud.h"

#include "FC_Messaging_clienthandler_BluetoothSetings.h"
#include "FC_Messaging_WorkQueue.h"
#include "FC_Messaging_clienthandler_Phonebook.h"

#include "FC_Messaging_clienthandler_DefSetHandler.h"
#include "FC_Messaging_clienthandler_Diaglib.h"

#include "FC_Messaging_service_Messaging.h"
#include "DbusClient/FC_Messaging_DbusClientInterface.h"
#include "FC_Messaging_StateMachine.h"
#include "./Database/FC_Messaging_SQLite.h"
#include "SendMessage/FC_Messaging_SendMessage.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_MESSAGING_APPLICATION
#include "trcGenProj/Header/FC_Messaging_main.cpp.trc.h"
#endif

/******************************************************************************/
/*                                                                            */
/* DEFINES                                                                    */
/*                                                                            */
/******************************************************************************/

//#define FC_MESSAGING_TIMER_TIME_UNTIL_FIRST_TICK 2000
//#define FC_MESSAGING_TIMER_TICK_INTERVALL 2000

/******************************************************************************/
/*                                                                            */
/* GLOBAL VARIABLES                                                           */
/*                                                                            */
/******************************************************************************/

//Messaging build for G3G
static GMainLoop* mainloop = NULLPTR;

// Static self reference to be used in function callbacks.
fc_messaging_tclService_Messaging* fc_messaging_tclApp::m_poMessaging = NULLPTR;
fc_messaging_tclApp* fc_messaging_tclApp::m_poMainAppInstance = NULLPTR;

fc_messaging_tclClientHandler_Phonebook* fc_messaging_tclApp::m_poClientHandlerPhonebook = NULLPTR;

Messaging_DefSetHandler*                    fc_messaging_tclApp::m_poDefSet = NULLPTR;
tU16    fc_messaging_tclApp::m_UCID = 0;
/******************************************************************************/
/*                                                                            */
/* METHODS                                                                    */
/*                                                                            */
/******************************************************************************/

//Messaging build for G3G
#ifdef CONNECTIVITY_PROCESS_STARTUP_GEN3
fc_messaging_tclApp* fc_messaging_tclApp::theServer()
{
   if(!m_poMainAppInstance)
   {
      m_poMainAppInstance = OSAL_NEW fc_messaging_tclApp();
   }
   return m_poMainAppInstance;
}
#endif //CONNECTIVITY_PROCESS_STARTUP_GEN3

/*******************************************************************************
*
* FUNCTION: ahl_fc_messaging_tclApp::ahl_fc_messaging_tclApp()
*
* DESCRIPTION: Constructor.
*
*              Initialize static self reference to be used in function
*              callbacks.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
fc_messaging_tclApp::fc_messaging_tclApp() :
      ahl_tclBaseOneThreadApp()
{
   ETG_TRACE_USR4(("fc_messaging_tclApp::fc_messaging_tclApp()"));

   fc_messaging_tclApp::m_poMainAppInstance = this;
   u32DoUpdateInboxTime = 0;

   m_poClientHandlerBluetoothSetings = NULLPTR;
   m_poDiaglibService = NULLPTR;
   m_poClientHandlerDiaglib = NULLPTR;
   m_poDefSet = NULLPTR;
   m_poMessaging = NULLPTR;

   m_poAudioRouteHandler = NULLPTR;
   m_poGenericWrapper = NULLPTR;
   m_poClientHandler_SDS = NULLPTR;

   //Messaging build for G3G
#ifdef CONNECTIVITY_PROCESS_STARTUP_GEN3
   m_hRxThreadId = OSAL_NULL;
#endif
}

//Messaging build for G3G
#ifdef CONNECTIVITY_PROCESS_STARTUP_GEN3

tVoid fc_messaging_tclApp::vInitializeThread()
{
   ETG_TRACE_USR4(("fc_messaging_tclApp::vInitializeThread"));

   tU32 u32ThreadPrio = 0;
   tU32 u32StackSize  = 0;
   OSAL_trThreadAttribute	rAttr;
   tChar szThreadName[OSAL_C_U32_MAX_NAMELENGTH] = "\0";
   tString tStrTemp = OSAL_szStringCopy( szThreadName, FC_MESSAGING_THREAD_ID);
   tString tStrDummy = tStrTemp;//Added to remove lint warning.
   tStrTemp = tStrDummy;
   u32ThreadPrio = FC_MESSAGING_DEFAULT_PRIORITY ;
   u32StackSize  = FC_MESSAGING_DEFAULT_STACK_SIZE;
   rAttr.szName	   = szThreadName;
   rAttr.s32StackSize = (tS32)u32StackSize;
   rAttr.u32Priority  = u32ThreadPrio;
   rAttr.pfEntry	   = (OSAL_tpfThreadEntry)fc_messaging_tclApp::vStartMainLoop;
   rAttr.pvArg 	   = (tPVoid)this;
   m_hRxThreadId = OSAL_ThreadSpawn(&rAttr);
   if ( m_hRxThreadId == OSAL_ERROR )
   {
      ETG_TRACE_USR4(("fc_messaging_tclApp::vInitializeThread() -> the thread NOT create"));
   }
   else
   {
      ETG_TRACE_USR4(("fc_messaging_tclApp::vInitializeThread() -> Thread creation successful"));
   }
}

tVoid fc_messaging_tclApp::vStartMainLoop(tVoid * poArg)
{
   ETG_TRACE_USR4(("fc_messaging_tclApp::vStartMainLoop"));
   (tVoid) poArg;

   mainloop = g_main_loop_new(NULLPTR, FALSE);
   if (!mainloop)
   {
      ETG_TRACE_ERR(("g_main_loop_new fails"));
      NORMAL_M_ASSERT_ALWAYS();
   }

   ETG_TRACE_USR4(("running main loop of fc_messaging_tclApp"));
   g_main_loop_run(mainloop);
}
#endif //CONNECTIVITY_PROCESS_STARTUP_GEN3

/*******************************************************************************
*
* FUNCTION: ahl_fc_messaging_tclApp::~ahl_fc_messaging_tclApp()
*
* DESCRIPTION: Destructor.
*
*              Invalidate static self reference.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
fc_messaging_tclApp::~fc_messaging_tclApp()
{
   ETG_TRACE_USR4(("fc_messaging_tclApp::~fc_messaging_tclApp"));

   if(m_poAudioRouteHandler)
   {
      OSAL_DELETE m_poAudioRouteHandler;
      m_poAudioRouteHandler = NULLPTR;
   }

   if(m_poGenericWrapper)
   {
      OSAL_DELETE m_poGenericWrapper;
      m_poGenericWrapper = NULLPTR;
   }

   if(m_poClientHandler_SDS)
   {
      OSAL_DELETE m_poClientHandler_SDS;
      m_poClientHandler_SDS = NULLPTR;
   }

   if (m_poMessaging)
   {
      OSAL_DELETE m_poMessaging;
      m_poMessaging = NULLPTR;
   }

   if (m_poClientHandlerBluetoothSetings)
   {
      OSAL_DELETE m_poClientHandlerBluetoothSetings;
      m_poClientHandlerBluetoothSetings = NULLPTR;
   }

   fc_messaging_tclApp::m_poMainAppInstance = NULLPTR;

   if (m_poDiaglibService)
   {
      OSAL_DELETE m_poDiaglibService;
      m_poDiaglibService = NULLPTR;
   }

   if (m_poClientHandlerDiaglib)
   {
      OSAL_DELETE m_poClientHandlerDiaglib;
      m_poClientHandlerDiaglib = NULLPTR;
   }

//Messaging build for G3G
#ifdef CONNECTIVITY_PROCESS_STARTUP_GEN3
   OSAL_s32ThreadDelete(m_hRxThreadId);
   m_hRxThreadId = OSAL_NULL;
#endif //CONNECTIVITY_PROCESS_STARTUP_GEN3

}

Messaging_DefSetHandler* fc_messaging_tclApp::poGetDefSetInstance()
 {
     return m_poDefSet;
 }

fc_messaging_tclApp* fc_messaging_tclApp::poGetInstance()
{
    return m_poMainAppInstance;
}

/*******************************************************************************
*
* FUNCTION: tBool fc_messaging_tclApp::bOnInit()
*
* DESCRIPTION: This function is called by the CCA framework before starting any
*              CCA message related communication action. As default this
*              function always returns TRUE. The user has to implement all his
*              local application initialization (create client and service
*              handler, create worker threads, allocate application memory,...).
*              If returns TRUE, initialization could be successfully performed
*              and framework will register application, else, a reset of the
*              system is forced immediately.
*
*              This function is the counterpart of vOnApplicationClose().
*
* PARAMETER: None.
*
* RETURNVALUE: TRUE  = Application successfully initialized.
*              FALSE = Application NOT successfully initialized.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::bOnInit().
*******************************************************************************/
tBool fc_messaging_tclApp::bOnInit()
{
   ETG_TRACE_USR1(("bOnInit() entered."));

   tBool bSuccess = TRUE;

//Messaging build for G3G
#ifdef CONNECTIVITY_PROCESS_STARTUP_GEN3
   vInitializeThread();
#endif //CONNECTIVITY_PROCESS_STARTUP_GEN3

   m_poAudioRouteHandler = OSAL_NEW fc_messaging_tclAudioRouteHandler(this);
   if (!m_poAudioRouteHandler)
   {
      bSuccess = FALSE;

      //FATAL_M_ASSERT_ALWAYS(); // This option with the fatal assert would issue a direct reset and could be used to avoid
      ETG_TRACE_FATAL(("Creation of fc_messaging_tclAudioRouteHandler failed"));
   }

   m_poGenericWrapper = OSAL_NEW fc_messaging_tclGenericWrapper(this);
   if (!m_poGenericWrapper)
   {
      bSuccess = FALSE;

      //FATAL_M_ASSERT_ALWAYS();
      ETG_TRACE_FATAL(("Creation of fc_messaging_tclGenericWrapper failed"));
   }

   m_poClientHandler_SDS = OSAL_NEW fc_messaging_ClientHandler_SDS(this);
   if (!m_poClientHandler_SDS)
   {
      bSuccess = FALSE;

      //FATAL_M_ASSERT_ALWAYS();
      ETG_TRACE_FATAL(("Creation of fc_messaging_ClientHandler_SDS failed"));
   }

   m_poClientHandlerPhonebook = OSAL_NEW fc_messaging_tclClientHandler_Phonebook(this);
   if (!m_poClientHandlerPhonebook)
   {
      bSuccess = FALSE;

      //FATAL_M_ASSERT_ALWAYS();
      ETG_TRACE_FATAL(("Creation of fc_messaging_tclClientHandler_Phonebook failed"));
   }

   m_poMessaging = OSAL_NEW fc_messaging_tclService_Messaging(this);
   if (!m_poMessaging)
   {
      bSuccess = FALSE;

      //FATAL_M_ASSERT_ALWAYS();
      ETG_TRACE_FATAL(("Creation of fc_messaging_tclService_Messaging failed"));
   }

   m_poClientHandlerBluetoothSetings = OSAL_NEW fc_messaging_tclClientHandler_BluetoothSetings(this);
   if (!m_poClientHandlerBluetoothSetings)
   {
      bSuccess = FALSE;

      //FATAL_M_ASSERT_ALWAYS();
      ETG_TRACE_FATAL(("Creation of fc_messaging_tclClientHandler_BluetoothSetings failed"));
   }

   m_poDiaglibService = OSAL_NEW diaglib::tclServiceDiaglib(this, 1);
   if (!m_poDiaglibService)
   {
      ETG_TRACE_FATAL(("Creation of diaglib::tclServiceDiaglib failed"));
      // bSuccess = FALSE;
      //FATAL_M_ASSERT_ALWAYS();
   }
   else
   {
      m_poClientHandlerDiaglib = OSAL_NEW fc_messaging_tclClientHandler_Diaglib(m_poDiaglibService);
      if (!m_poClientHandlerDiaglib)
      {
         ETG_TRACE_FATAL(("Creation of fc_messaging_tclClientHandler_Diaglib failed"));
         bSuccess = FALSE;
         //FATAL_M_ASSERT_ALWAYS();
      }
   }

   m_poDefSet = OSAL_NEW Messaging_DefSetHandler(m_poDiaglibService);
   if (!m_poDefSet)
   {
      bSuccess = FALSE;
      //FATAL_M_ASSERT_ALWAYS();
      ETG_TRACE_FATAL(("Creation of Messaging_DefSetHandler failed"));
   }

   return bSuccess;
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_messaging_tclApp::vOnApplicationClose()
*
* DESCRIPTION: This function is called by the CCA framework to indicate the
*              imminent shutdown of the application. The user has to implement
*              the de-initialization of all his local application data (destroy
*              client and service handler, destroy worker threads, free
*              application memory, ... ). After this function call the
*              destructor of this application will be called.
*
*              This function is the counterpart of bOnInit().
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::vOnApplicationClose().
*******************************************************************************/
tVoid fc_messaging_tclApp::vOnApplicationClose()
{
   ETG_TRACE_USR1(("fc_messaging_tclApp::vOnApplicationClose"));

   _BP_TRY_BEGIN
   {
      if (m_poClientHandlerPhonebook)
      {
         OSAL_DELETE m_poClientHandlerPhonebook;
         m_poClientHandlerPhonebook = NULLPTR;
      }

      if(m_poAudioRouteHandler)
      {
         OSAL_DELETE m_poAudioRouteHandler;
         m_poAudioRouteHandler = NULLPTR;
      }

      if(m_poGenericWrapper)
      {
         OSAL_DELETE m_poGenericWrapper;
         m_poGenericWrapper = NULLPTR;
      }

      if(m_poClientHandler_SDS)
      {
         OSAL_DELETE m_poClientHandler_SDS;
         m_poClientHandler_SDS = NULLPTR;
      }

      if (m_poClientHandlerBluetoothSetings)
      {
         OSAL_DELETE m_poClientHandlerBluetoothSetings;
         m_poClientHandlerBluetoothSetings = NULLPTR;
      }

      if (m_poMessaging)
      {
         OSAL_DELETE m_poMessaging;
         m_poMessaging = NULLPTR;
      }

      if (m_poDiaglibService)
      {
         OSAL_DELETE m_poDiaglibService;
         m_poDiaglibService = NULLPTR;
      }

      if (m_poClientHandlerDiaglib)
      {
         OSAL_DELETE m_poClientHandlerDiaglib;
         m_poClientHandlerDiaglib = NULLPTR;
      }

      if (m_poDefSet)
      {
         OSAL_DELETE m_poDefSet;
         m_poDefSet = NULLPTR;
      }
   }

   _BP_CATCH_ALL
   { }

   _BP_CATCH_END

   ahl_tclBaseOneThreadApp::vOnApplicationClose();
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_messaging_tclApp::vOnLoadSettings()
*
* DESCRIPTION: This function is called by the CCA framework to trigger the 
*              loading of last mode settings. The user has to load the
*              previously stored last mode settings from persistent memory.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::vOnLoadSettings().
*******************************************************************************/
tVoid fc_messaging_tclApp::vOnLoadSettings()
{
   ETG_TRACE_USR1(("vOnLoadSettings() entered"));

   // Start time measurement
   struct timespec time1 = { 0, 0 };
   struct timespec time2 = { 0, 0 };
   struct timespec diff_time = { 0, 0 };
   clock_gettime(CLOCK_MONOTONIC, &time1);

   /*
   ETG_TRACE_FATAL ((" -MSG_APP_STATE_CHANGE- vOnLoadSettings() entered. up-time= [%7lu.%03lu] sec ", time1.tv_sec, time1.tv_nsec / 1000000 ));
   // EM trace added temporarily for tracking of PSARCC21-3492 and related issues
   ETG_TRACE_ERRMEM((" -MSG_APP_STATE_CHANGE- vOnLoadSettings() entered. up-time= [%7lu.%03lu] sec ", time1.tv_sec, time1.tv_nsec / 1000000 ));
    */
   if (m_poMessaging)
   {
      m_poMessaging->vOff2Normal();
   }

   if(m_poAudioRouteHandler)
   {
      m_poAudioRouteHandler->vSetSourceAvailable(TRUE);
   }

   // Evaluation of needed time
   clock_gettime(CLOCK_MONOTONIC, &time2);
   if ((time2.tv_nsec - time1.tv_nsec) < 0)
   {
      diff_time.tv_sec = time2.tv_sec - time1.tv_sec - 1;
      diff_time.tv_nsec = 1000000000 + time2.tv_nsec - time1.tv_nsec;
   }
   else
   {
      diff_time.tv_sec = time2.tv_sec - time1.tv_sec;
      diff_time.tv_nsec = time2.tv_nsec - time1.tv_nsec;
   }
   /*
   ETG_TRACE_FATAL ((" -MSG_APP_STATE_CHANGE- vOnLoadSettings() exited, needed time= [%5lu] msec ",
      diff_time.tv_sec * 1000 + diff_time.tv_nsec / 1000000));

   // EM trace added temporarily for tracking of PSARCC21-3492 and related issues
   ETG_TRACE_ERRMEM((" -MSG_APP_STATE_CHANGE- vOnLoadSettings() exited, needed time= [%5lu] msec ",
      diff_time.tv_sec * 1000 + diff_time.tv_nsec / 1000000));
    */
   (tVoid)diff_time;
}


/*******************************************************************************
*
* FUNCTION: tVoid fc_messaging_tclApp::vOnSaveSettings()
*
* DESCRIPTION: This function is called by the CCA framework to trigger the 
*              storage of last mode settings. The user has to store the last
*              mode settings to persistent memory.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::vOnSaveSettings().
*******************************************************************************/
tVoid fc_messaging_tclApp::vOnSaveSettings()
{
   ETG_TRACE_USR1(( "vOnSaveSettings(): entered" ));

   // Start time measurement
   struct timespec time1     = {0, 0};
   struct timespec time2     = {0, 0};
   struct timespec diff_time = {0, 0};
   clock_gettime(CLOCK_MONOTONIC, &time1);

   /*
   ETG_TRACE_FATAL ((" -MSG_APP_STATE_CHANGE- vOnSaveSettings() entered, up-time= [%7lu.%03lu] sec ", time1.tv_sec, time1.tv_nsec / 1000000));
   // EM trace added temporarily for tracking of PSARCC21-3492 and related issues
   ETG_TRACE_ERRMEM((" -MSG_APP_STATE_CHANGE- vOnSaveSettings() entered, up-time= [%7lu.%03lu] sec ", time1.tv_sec, time1.tv_nsec / 1000000));
    */

   //   // Here the saving of last mode settings shall be implemented
   //   FC_Messaging_SQLite* dataBase= FC_Messaging_SQLite::poGetFC_Messaging_SQLite();
   //
   //   MSG_DEBUG("CLOSING THE DATABASE ON SPM vOnSaveSettings");
   //   dataBase->setDataBaseState(false);
   //   m_poMessaging->m_pMsgListInterface->poGetAudioRecording()->vRequestRecordAudioMessage(
   // 	  	   most_fi_tcl_e8_MsgRecordMessageAction::FI_EN_E8REC_MSG_ACTION_STOP);
   //
   //   for(int loop=MINIMUM_DEVICE_HANDLE;loop<=MAXIMUM_DEVICE_HANDLE;loop++)
   //   {
   //	   MessagingFunctionBlockDbusClient::messagingDeviceDisconnected(loop);
   //   }
   if(m_poMessaging)
   {
      m_poMessaging->vNormal2Off();
   }

   //Unregister Evo Signals and Destroy Evo Proxies
   DbusClientInterface::getInstance().messagingRemoveEvoProxies();

   DbusClientInterface::destroyInstance();

   // Evaluation of needed time
   clock_gettime(CLOCK_MONOTONIC, &time2);
   if( (time2.tv_nsec - time1.tv_nsec) < 0 )
   {
      diff_time.tv_sec  = time2.tv_sec  - time1.tv_sec-1;
      diff_time.tv_nsec = 1000000000 + time2.tv_nsec - time1.tv_nsec;
   }
   else
   {
      diff_time.tv_sec  = time2.tv_sec  - time1.tv_sec;
      diff_time.tv_nsec = time2.tv_nsec - time1.tv_nsec;
   }

   /*
   ETG_TRACE_FATAL (( " -MSG_APP_STATE_CHANGE- vOnSaveSettings() exited, needed time= [%5lu] msec ",
         diff_time.tv_sec *1000 + diff_time.tv_nsec / 1000000 ));

   // EM trace added temporarily for tracking of PSARCC21-3492 and related issues
   ETG_TRACE_ERRMEM(( " -MSG_APP_STATE_CHANGE- vOnSaveSettings() exited, needed time= [%5lu] msec ",
         diff_time.tv_sec *1000 + diff_time.tv_nsec / 1000000 ));
    */
   (tVoid)diff_time;
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_messaging_tclApp::vOnTimer(tU16 u16TimerId)
*
* DESCRIPTION: This function is called by the CCA framework on the expiration
*              of a previously via function bStartTimer() started timer. The
*              expired timer is forwarded to the respective service or
*              client-handler via a call of vProcessTimer().
*
*              The function is called from this applications context and
*              therefore no interthread programming rules must be considered
*              and the application methods and/or member variables can be
*              accessed without using the static self reference
*              'm_poMainAppInstance'.
*
* PARAMETER: [IN] u16TimerId = Identifier of the expired timer.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::vOnTimer().
*******************************************************************************/
tVoid fc_messaging_tclApp::vOnTimer(tU16 u16TimerId)
{
   ETG_TRACE_USR4(("fc_messaging_tclApp::vOnTimer ID :%d:", u16TimerId));

   static tU32 u32TimerCount = 0;
   (void) u32TimerCount;
   switch (u16TimerId)
   {
      case FC_MESSAGING_UPDATEINBOX_TIMER_ID:
      {
         ETG_TRACE_USR4(("vOnTimer FC_MESSAGING_UPDATEINBOX_TIMER_ID"));

         u32TimerCount++;

         if (u32TimerCount >= (u32DoUpdateInboxTime / 5)) // Bug in Timer
         {
            if ((MessagingMutex::messagingMutexTrylock(NULLPTR) == 0)) // Do not queue it
            {
               DbusClientInterface::getInstance().messagingMapUpdateInbox();
            }

            u32TimerCount = 0;
         }
      }
      break;

      case FC_MESSAGING_MAPERRORRETRY_TIMER_ID:
      {
         ETG_TRACE_USR4(("vOnTimer FC_MESSAGING_MAPERRORRETRY_TIMER_ID"));

         if (FALSE == bStopTimer(FC_MESSAGING_MAPERRORRETRY_TIMER_ID))
         {
            ETG_TRACE_ERR(("Unable to stop the timer FC_MESSAGING_MAPERRORRETRY_TIMER_ID"));
         }

         DbusClientInterface::getInstance().messagingMapStart();
      }
      break;

      case FC_MESSAGING_MAPCAPABILITYRETRY_TIMER_ID:
      {
         ETG_TRACE_USR4(("vOnTimer FC_MESSAGING_MAPCAPABILITYRETRY_TIMER_ID"));

         DbusClientInterface::getInstance().messagingMapGetCapabilities(DbusClientInterface::getInstance().getCurrentOutgoingDeviceId());

         if (FALSE == bStopTimer(FC_MESSAGING_MAPCAPABILITYRETRY_TIMER_ID))
         {
            ETG_TRACE_ERR(("Unable to stop the timer FC_MESSAGING_MAPCAPABILITYRETRY_TIMER_ID"));
         }
      }
      break;

      case FC_MESSAGING_AUDIOROUTEREQUEST_WAIT_TIMER_ID:
      {
         ETG_TRACE_USR4(("vOnTimer FC_MESSAGING_AUDIOROUTEREQUEST_WAIT_TIMER_ID"));

         if (FALSE == bStopTimer(FC_MESSAGING_AUDIOROUTEREQUEST_WAIT_TIMER_ID))
         {
            ETG_TRACE_ERR(("Unable to stop the timer FC_MESSAGING_AUDIOROUTEREQUEST_WAIT_TIMER_ID"));
         }

         ReadMessageAloud* poReadMessageAloud = ReadMessageAloud::poGetInstance();
         poReadMessageAloud->vProcessEvent(FC_MESSAGING_SM_EVENT_AUDIOROUTE_TIMER_EXPIRED);
      }
      break;

      default:
      {
         //NCG3D-68657: Workaround Start: Incomplete MNS Connection
         if((u16TimerId >= (FC_MESSAGING_MNSCONNECTIONRETRY_TIMER_ID + MINIMUM_DEVICE_HANDLE)) &&
               (u16TimerId <= (FC_MESSAGING_MNSCONNECTIONRETRY_TIMER_ID + MAXIMUM_DEVICE_HANDLE)))
         {
            ETG_TRACE_USR4(("vOnTimer FC_MESSAGING_MNSCONNECTIONRETRY_TIMER_ID"));

            if (FALSE == bStopTimer(u16TimerId))
            {
               ETG_TRACE_ERR(("Unable to stop the timer FC_MESSAGING_MNSCONNECTIONRETRY_TIMER_ID"));
            }

            //Disconnect the MAP profile, Reconnection will happen after disconnect MAP profile completed.
            tU8 deviceHandle = (tU8)(u16TimerId - FC_MESSAGING_MNSCONNECTIONRETRY_TIMER_ID);
            if(m_poClientHandlerBluetoothSetings)
            {
               m_poClientHandlerBluetoothSetings->vReconnectMAPProfile(deviceHandle);
            }
         }//Workaround End
         else if((u16TimerId >= (FC_MESSAGING_OUTBOXCHECK_WAIT_TIMER_ID + MINIMUM_DEVICE_HANDLE)) &&
                    (u16TimerId <= (FC_MESSAGING_OUTBOXCHECK_WAIT_TIMER_ID + MAXIMUM_DEVICE_HANDLE)))
         {
            ETG_TRACE_USR4(("FC_MESSAGING_OUTBOXCHECK_WAIT_TIMER_ID"));
            if (TRUE == bStopTimer(u16TimerId))
            {
               ETG_TRACE_ERR(("The timer FC_MESSAGING_OUTBOXCHECK_WAIT_TIMER_ID is Stopped"));
            }
            most_msgfi_tclMsgCreateMessageListMethodStart* createMessageListMethodStart = new most_msgfi_tclMsgCreateMessageListMethodStart;
            if (createMessageListMethodStart)
            {
               tU8 deviceHandle = (tU8)(u16TimerId - FC_MESSAGING_OUTBOXCHECK_WAIT_TIMER_ID);
               createMessageListMethodStart->u8DeviceHandle = deviceHandle;
               createMessageListMethodStart->e8MessageListType.enType = most_fi_tcl_e8_MsgMessageListType::FI_EN_E8MSG_LIST_SMS_MMS;
               createMessageListMethodStart->e8FolderType.enType=most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_OUTBOX;
               _BP_TRY_BEGIN
               {
                  DbusClientInterface::getInstance().messagingMapListMsg(*createMessageListMethodStart,MAXIMUM_LIST_COUNT,e8MapListMsgCallerOutboxCheck);
               }
               _BP_CATCH_ALL
               {
                  createMessageListMethodStart->vDestroy();
                  delete createMessageListMethodStart;
                  ETG_TRACE_ERR(("FC_MESSAGING_OUTBOXCHECK_WAIT_TIMER_ID caught exception"));
                  return;
               }
               _BP_CATCH_END
			   createMessageListMethodStart->vDestroy();
			   delete createMessageListMethodStart;
            }
         }
         else
         {
            ETG_TRACE_USR4(("Invalid Timer ID :%d:", u16TimerId));
         }
      }
   }
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_messaging_tclApp::vOnLoopback
*                 (tU16 u16ServiceID, amt_tclServiceData* poMessage)
*
* DESCRIPTION: This function is called by the CCA framework if a message from
*              one if this applications services is received (so called self
*              or loopback messages). The message is forwarded to the respective
*              service via a call of vOnLoopbackService().
*
* PARAMETER: [IN] u16ServiceID = Identifier of the service from where the
*                                message was sent.
*
*            [IN] poMessage = Message object.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::vOnLoopback().
*******************************************************************************/
tVoid fc_messaging_tclApp::vOnLoopback
(tU16 u16ServiceID, amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR1(("vOnLoopback() entered."));

   switch(u16ServiceID)
   {
      case CCA_C_U16_SRV_FB_MESSAGING:
      {
         if(m_poMessaging)
         {
            m_poMessaging->vOnLoopbackService(poMessage);
         }
         else
         {
            NORMAL_M_ASSERT_ALWAYS();
         }
      }
      break;

      default:
      {
         ETG_TRACE_ERR(("vOnLoopback(). Unknown ServiceID %u.", u16ServiceID));
      }
   }
}

/*******************************************************************************
*
* FUNCTION: fc_messaging_tclApp::fc_messaging_set_service (tBool on))
*
* DESCRIPTION: This function is used to enable/disable service of Phonebook
*
* PARAMETER: [IN] tBool on = Boolean variable to enable/disable Service 
*
* RETURNVALUE: None.
*
********************************************************************************/

tVoid fc_messaging_tclApp::fc_messaging_set_service(tBool on)
{
    ETG_TRACE_USR4(("fc_messaging_set_service : Called"));
    m_poMainAppInstance->m_poMessaging->vSetPrivateServiceAvailableAllowed(on);
}
