/**
 * @file FC_Messaging_EvoDbusClient.cpp
 * @author
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */
#include "FC_Messaging_EvoDbusClient.h"
#include "FC_Messaging_DbusClient_MsgRead.h"
#include "FC_Messaging_Macro.h"
#include "../FC_Messaging_clienthandler_BluetoothSetings.h"
#include "../SendMessage/FC_Messaging_SendMessage.h"
#include "../MsgList/FC_Messaging_Settings.h"
#include "../FC_Messaging_StateMachine.h"
#include "../FC_Messaging_WorkQueue.h"
#include <iostream>
#include <fstream>
#include <sstream>
using namespace std;

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_MESSAGING_APPLICATION
#include "trcGenProj/Header/FC_Messaging_EvoDbusClient.cpp.trc.h"
#endif

EvolutionDbusClient* EvolutionDbusClient::_bluetoothDeviceObject[MAXIMUM_DEVICE_SUPPORT];
GDBusConnection* EvolutionDbusClient::_evoMsgBus = NULLPTR;
OrgBluezObexClient* EvolutionDbusClient::_proxyClient = NULLPTR;
OrgFreedesktopDBusObjectManager* EvolutionDbusClient::_proxyObjectManager = NULLPTR;
tU8 EvolutionDbusClient::_currentOutgoingDeviceId = 0;
tBool EvolutionDbusClient::_isRetryTimerRunning = FALSE;
QMutex EvolutionDbusClient::_deviceListMutex;
trFC_Msg_EvoSignalHandlerId EvolutionDbusClient::_evoSignalHandlerId;
::std::map<OrgBluezObexMessageAccess*, notificationFunctionMap> EvolutionDbusClient::_newMessageNotificationMap;
tBool EvolutionDbusClient::_isNewMsgNotification1inUse = false;
tBool EvolutionDbusClient::_isNewMsgNotification2inUse = false;

EvolutionDbusClient::EvolutionDbusClient()
{
   _proxyMessageAccess = NULLPTR;
   _proxyDBusPropertiesForMessageAccess = NULLPTR;
   _proxySetMessage = NULLPTR;
   _proxyGetMessage = NULLPTR;

   _currentTransfer.clear();
   _listFilterFields.clear();
   _clientSessionPath = NULLPTR;
   _mapSentMsgAlertSent = false;
   _msgCountDeviceInbox = 0;

   _currentActiveEmailService = "";
   _currentActiveSMSMMSService = "";
   _deviceHandle = 0;
   _filterMessageType = 0;
   _msgCapabilityInfo.m_u8ServiceNum = 0;
   _mapListMsgCaller = e8MapListMsgCallerCreateMessageList;
   _newMessageHandleList.clear();
   _messageHandleList.clear();
   _msgNotifyMap.clear();
   _mapNotifyStatus = FALSE;
   _waitForMapListMsgCompSignal = false;
   _folderListRequestStatus = FC_MSG_FOLDER_LIST_NOT_REQUESTED;
   _inboxSearchType = e8Forward;
   _isDeleteMessage = false;
   _retryCount = 0;
   _signatureText = "";
   m_MaxMapRetryCount = MAX_MAP_RETRY;
   _msgReadObject.vClear();
   _msgListObject.vClear();

   memset(&_deviceConnection, 0, sizeof(_deviceConnection));

   //Handler ID's
   ulNewMsgNotification = 0;

   resetMapDeviceCapabilities();
}

EvolutionDbusClient::~EvolutionDbusClient()
{
   ETG_TRACE_USR4(("~EvolutionDbusClient() called"));
   _BP_TRY_BEGIN
   {
   _proxyMessageAccess = NULLPTR;
   _proxyDBusPropertiesForMessageAccess = NULLPTR;
   _proxyGetMessage = NULLPTR;
   _proxySetMessage = NULLPTR;
   _newMessageHandleList.clear();
   _messageHandleList.clear();
   _msgNotifyMap.clear();
   _listFilterFields.clear();
   }
   _BP_CATCH_ALL
   {
      ETG_TRACE_ERR(("BadAlloc Exception Caught"));
   }
   _BP_CATCH_END

   //clearClientSessionPath(); Not required, SessionPath cleared on device disconnection.
}

void EvolutionDbusClient::clientSessionCreatedCallback(OrgBluezObexClient *object, const gchar *arg_address, const gchar *arg_session, guchar arg_remode_instance, const gchar *arg_UUID)
{
   ETG_TRACE_USR4(("clientSessionCreatedCallback() called"));

   (void)object; //Lint fix
   (void)arg_address;
   (void)arg_session;
   (void)arg_remode_instance;
   (void)arg_UUID;
}

void EvolutionDbusClient::clientSessionRemovedCallback(OrgBluezObexClient *object, const gchar *arg_address, const gchar *arg_session, guchar arg_remode_instance, const gchar *arg_UUID)
{
   ETG_TRACE_USR4(("clientSessionRemovedCallback() called"));

   (void)object; //Lint fix
   (void)arg_address;
   (void)arg_remode_instance;
   (void)arg_UUID;

   EvolutionDbusClient* client = getBluetoothDeviceObject(arg_session);

   if(client)
   {
      client->messagingMapSessionDisconnected();
   }
}

tBool EvolutionDbusClient::messagingMapCreateSession(::std::string bluetoothDeviceAddress)
{
   ETG_TRACE_USR4(("messagingMapCreateSession called"));

   if(!_proxyClient)
   {
      ETG_TRACE_USR4(("Client Proxy NULL"));
      return false;
   }

   GError* error = NULLPTR;
   gchar* session = NULLPTR;
   gboolean result;
   GVariantBuilder* builder;
   GVariant* value;
   tU8 firstCounter, secondCounter;
   ::std::string targetProxyDeviceAddress;
   char targetAddress[18] = {0};
   char deviceAddress[18] = {0};

   if((fc_messaging_tclApp::m_poMainAppInstance)->m_poClientHandlerBluetoothSetings)
   {
      targetProxyDeviceAddress = (fc_messaging_tclApp::m_poMainAppInstance)->m_poClientHandlerBluetoothSetings->getVehicleBTAddress();
   }

   for (firstCounter = 0, secondCounter = 0; firstCounter < targetProxyDeviceAddress.size(); firstCounter++, secondCounter++)
   {
      targetAddress[secondCounter++] = targetProxyDeviceAddress[firstCounter++];
      targetAddress[secondCounter++] = targetProxyDeviceAddress[firstCounter];
      targetAddress[secondCounter] = ':';
   }
   targetAddress[secondCounter-1] = '\0';

   for (firstCounter = 0, secondCounter = 0; firstCounter < bluetoothDeviceAddress.size(); firstCounter++, secondCounter++)
   {
      deviceAddress[secondCounter++] = bluetoothDeviceAddress[firstCounter++];
      deviceAddress[secondCounter++] = bluetoothDeviceAddress[firstCounter];
      deviceAddress[secondCounter] = ':';
   }
   deviceAddress[secondCounter-1] = '\0';

   ETG_TRACE_USR4(("LocalAddress: %s", targetAddress));
   ETG_TRACE_USR4(("DeviceAddress: %s", deviceAddress));

   builder = g_variant_builder_new(G_VARIANT_TYPE("a{sv}"));
   g_variant_builder_add(builder, "{sv}", "Target", g_variant_new_string("MAP"));
   g_variant_builder_add(builder, "{sv}", "Source", g_variant_new_string(targetAddress));
   g_variant_builder_add(builder, "{sv}", "Channel", g_variant_new_byte(0));
   g_variant_builder_add(builder, "{sv}", "InstanceID", g_variant_new_byte(0));
   value = g_variant_new("a{sv}", builder);

   result = org_bluez_obex_client_call_create_session_sync (_proxyClient, deviceAddress, value, &session, NULLPTR, &error);

   if (error && (result != TRUE))
   {
      ETG_TRACE_ERR(("ClientProxyError: %s", g_dbus_error_get_remote_error (error)));
      g_error_free(error);

      doMapRetry();

      return false;
   }

   //device connected, set the retry to default values.
   _retryCount = 0;
   setRetryTimerRunningStatus(FALSE);

   //also remove this device from retry list
   fc_messaging_tclClientHandler_BluetoothSetings::removeDeviceFromRetryList(_deviceHandle);

   if(session)
   {
      ETG_TRACE_USR4(("ClientSessionPath: %s", session));

      //If session path is available, then clear and allocate fresh memory
      ::std::size_t len = strlen(session) + 1;
      tString sessionPath = new tChar[len];
      if(sessionPath)
      {
         memset(sessionPath, 0, len);
         strncpy(sessionPath, session, len);

         setClientSessionPath(sessionPath);
      }
   }
   else
   {
      ETG_TRACE_USR4(("ClientSession is NULL"));
      return false;
   }

   return createMessageAccessProxy();
}

void EvolutionDbusClient::messagingMapRemoveSession()
{
   ETG_TRACE_USR4(("messagingMapRemoveSession called"));

   if(!_proxyClient)
   {
      ETG_TRACE_USR4(("Client Proxy NULL"));
      return;
   }

   if(!_clientSessionPath)
   {
      ETG_TRACE_USR4(("MAP Session not connected"));
      return;
   }

   GError* error = NULLPTR;
   gboolean result;

   result = org_bluez_obex_client_call_remove_session_sync(_proxyClient, _clientSessionPath, NULLPTR, &error);

   if ((result != TRUE) && error)
   {
      ETG_TRACE_ERR(("RemoveSession Error: %s", g_dbus_error_get_remote_error(error)));
      g_error_free(error);
      return;
   }
   else
   {
      ETG_TRACE_USR4(("RemoveSession Succeeded"));
   }
}

void EvolutionDbusClient::doMapRetry()
{
   quint16 delayTimeout = EVO_MAP_SESSION_RETRY_TIMER;

   if(_retryCount < vGetMapRetryCount())
   {
      if (FALSE == fc_messaging_tclApp::m_poMessaging->bStartTimer(
            FC_MESSAGING_MAPERRORRETRY_TIMER_ID,
            delayTimeout,
            delayTimeout))
      {
         ETG_TRACE_ERR(("Unable to start the timer FC_MESSAGING_MAPERRORRETRY_TIMER_ID"));

         updateMapDeviceConnectionStatus(FC_MSG_MAP_CONNECTION_FAILED);
      }
      else
      {
         fc_messaging_tclClientHandler_BluetoothSetings::addDeviceToRetryList(_deviceHandle);

         _retryCount++;
         setRetryTimerRunningStatus(TRUE);
      }
   }
   else
   {
      //Set retry values to default
      _retryCount = 0;
      setRetryTimerRunningStatus(FALSE);
      vSetMapRetryCount();

      //Remove this device from retry list
      fc_messaging_tclClientHandler_BluetoothSetings::removeDeviceFromRetryList(_deviceHandle);
      fc_messaging_tclClientHandler_BluetoothSetings::removeDeviceFromMAPConnRetryForMNSList(_deviceHandle);

      //Maximum number of retries reached, connection failed
      updateMapDeviceConnectionStatus(FC_MSG_MAP_CONNECTION_FAILED);
   }
}

tBool EvolutionDbusClient::createMessageAccessProxy()
{
   tBool bRetVal = false;

   if(!_clientSessionPath)
   {
      return bRetVal;
   }

   GError* error = NULLPTR;

   //Message Access Proxy Creation
   _proxyMessageAccess = org_bluez_obex_message_access_proxy_new_sync(_evoMsgBus,
         G_DBUS_PROXY_FLAGS_NONE, BASE_EVO_SERVICE_NAME,
         _clientSessionPath, NULLPTR, &error);

   if (!_proxyMessageAccess)
   {
      ETG_TRACE_ERR(("MessageAccessProxyError: %s", g_dbus_error_get_remote_error(error)));
      g_error_free(error);
   }
   else
   {
      ETG_TRACE_USR4(("MessageAccessProxy created"));

      //lint --e{826} Suppress Info 826: Suspicious pointer-to-pointer conversion (area too small)
      // this removes the lint warning for this block only
      OrgBluezObexMessageAccessProxy* msgAccessProxy = ORG_BLUEZ_OBEX_MESSAGE_ACCESS_PROXY(_proxyMessageAccess);
      if(msgAccessProxy)
      {
         g_dbus_proxy_set_default_timeout(&msgAccessProxy->parent_instance, DEFAULT_TIMEOUT_VAL_DBUS_CALL);
      }
      else
      {
         ETG_TRACE_ERR(("set timeout failed for MessageAccess proxy"));
      }

      if(false == _isNewMsgNotification1inUse)
      {
         ulNewMsgNotification = g_signal_connect(_proxyMessageAccess, "new-message-notification", G_CALLBACK(onNewMessageNotification1), NULLPTR);
         _isNewMsgNotification1inUse = true;
         _newMessageNotificationMap.insert(std::pair<OrgBluezObexMessageAccess*, notificationFunctionMap>(_proxyMessageAccess, NOTIFICATION_FUNC_1));

         ETG_TRACE_USR4(("onNewMessageNotification1() used for deviceHandle: %d", _deviceHandle));
      }
      else if(false ==_isNewMsgNotification2inUse)
      {
         ulNewMsgNotification = g_signal_connect(_proxyMessageAccess, "new-message-notification", G_CALLBACK(onNewMessageNotification2), NULLPTR);
         _isNewMsgNotification2inUse = true;
         _newMessageNotificationMap.insert(std::pair<OrgBluezObexMessageAccess*, notificationFunctionMap>(_proxyMessageAccess, NOTIFICATION_FUNC_2));

         ETG_TRACE_USR4(("onNewMessageNotification2() used for deviceHandle: %d", _deviceHandle));
      }

      //DBusPropertiesChanged should be registered to get the Notification update
      createDBusPropertiesProxyForMessageAccessPropertiesChanged(_clientSessionPath);

      messagingMapNotificationRegistry();

      bRetVal = true;
   }

   return bRetVal;
}

void EvolutionDbusClient::messagingMapSessionConnected()
{
   ETG_TRACE_USR4(("messagingMapSessionConnected called"));

   if(!_clientSessionPath)
   {
      return;
   }

   _serviceNameAccountManager.vCreateFileSystemManager(_clientSessionPath);

   _serviceNameAccountManager.vSetCurrentActiveService(_clientSessionPath, NULLPTR);

   _currentActiveSMSMMSService = _clientSessionPath;

   FileSystemManager* fileSystemManager = _serviceNameAccountManager.poGetFileSystemManager(QString::fromStdString (_currentActiveSMSMMSService));

   if (fileSystemManager)
   {
      fileSystemManager->vAddFolderNode("Root", MSG_DBUS_FOLDER_TYPE_INBOX, _deviceHandle); // SMS RequestFolderList
      fileSystemManager->vAddFolderNode("Root", MSG_DBUS_FOLDER_TYPE_SENT, _deviceHandle);

#ifdef CONVERSATION_MESSAGES_SUPPORT //enable conversation messages support
      fileSystemManager->vAddFolderNode("Root", MSG_DBUS_FOLDER_TYPE_CONVERSATION, _deviceHandle);
#endif
   }

   _filterMessageType |= DBUS_MSG_TYPE_SMS_GSM;

   messagingMapListFilterFields();

   sendFolderListRequest(FC_MSG_FOLDER_LIST_REQUESTED_FOR_SMSMMS_SERVICE);

   //Update Messaging Device Connection Status
   _deviceConnection.m_u8DeviceHandle = _deviceHandle;
   _deviceConnection.m_bEmailSupport = false;
   _deviceConnection.m_bMMSSupport = true;
   _deviceConnection.m_bMessagingMemoryStatus = true;
   _deviceConnection.m_bSMSSupport = true;
   _deviceConnection.m_bUpdateInboxSupport = false;
   _deviceConnection.m_u16EmailAccountId = 0;
   fc_messaging_tclService_Messaging::m_poMessagingService->_MessagingDeviceConnection(&_deviceConnection);

   //Update Map Device Capabilities Status
   _mapDeviceCapabilities.u8DeviceHandle = _deviceHandle;
   _mapDeviceCapabilities.u8MapConnectionState = FC_MSG_MAP_CONNECTED;
   _mapDeviceCapabilities.u16SupportedMessageTypes |= FC_MSG_MMS_SUPPORT;
   _mapDeviceCapabilities.u16SupportedMessageTypes |= FC_MSG_SMS_SUPPORT;
   _mapDeviceCapabilities.rFC_Msg_ActiveEmailAccount.u16AccountHandle = 0;
   _mapDeviceCapabilities.rFC_Msg_ActiveEmailAccount.sAccountName = "";

   sendMapDeviceCapabilitiesStatus();
}

void EvolutionDbusClient::messagingMapSessionDisconnected()
{
   ETG_TRACE_USR4(("messagingMapSessionDisconnected() called"));

   //send MAP session as disconnected.
   _serviceNameAccountManager.vSetCurrentActiveService(NULLPTR, NULLPTR);

   _folderListRequestStatus = FC_MSG_FOLDER_LIST_NOT_REQUESTED;
   resetMapDeviceCapabilities();
   sendMapDeviceCapabilitiesStatus();

   _deviceConnection.m_u8DeviceHandle = _deviceHandle;
   _deviceConnection.m_bEmailSupport = false;
   _deviceConnection.m_bMMSSupport = false;
   _deviceConnection.m_bMessagingMemoryStatus = false;
   _deviceConnection.m_bSMSSupport = false;
   _deviceConnection.m_bUpdateInboxSupport = false;
   _deviceConnection.m_u16EmailAccountId = 0;
   fc_messaging_tclService_Messaging::m_poMessagingService->_MessagingDeviceConnection(&_deviceConnection);

   _currentTransfer.clear();
}

void EvolutionDbusClient::setClientSessionPath(tString sessionPath)
{
   clearClientSessionPath();

   _clientSessionPath = sessionPath;
}

void EvolutionDbusClient::clearClientSessionPath()
{
   ETG_TRACE_USR4(("clearClientSessionPath called"));

   if(_clientSessionPath)
   {
      delete[] _clientSessionPath;
      _clientSessionPath = NULLPTR;
   }
}

void EvolutionDbusClient::onObjectManagerInterfacesAdded(
         OrgFreedesktopDBusObjectManager *object,
         const gchar *arg_object,
         GVariant *arg_interfaces)
{
   ETG_TRACE_USR4(("onObjectManagerInterfacesAdded() called"));

   (void)object; //Lint fix

   ::std::string objectPath = arg_object;

   GVariantIter iter1, iter2;
   GVariant* dict;
   gchar* interface;
   gchar* key;
   GVariant* value;
   gsize length = 0;
   EvolutionDbusClient* client = NULLPTR;
   tU8 deviceHandle = 0;

   g_variant_iter_init (&iter1, arg_interfaces);
   while (g_variant_iter_next (&iter1, "{s@a{sv}}", &interface, &dict))
   {
      ETG_TRACE_USR4(("onObjectManagerInterfacesAdded::Interface : %s", interface));

      if(strcmp(interface, BASE_EVO_TRANSFER_INTERFACE_NAME) == 0)
      {
         g_variant_iter_init (&iter2, dict);
         while (g_variant_iter_next (&iter2, "{sv}", &key, &value))
         {
            if (strcmp (key, "Session") == 0 && g_variant_is_of_type (value, G_VARIANT_TYPE_OBJECT_PATH))
            {
               tString session  = g_variant_dup_string(value,&length);
               ETG_TRACE_USR4(("onObjectManagerInterfacesAdded::Session : %s", session));

               QMutexLocker lock(&_deviceListMutex);
               client = getBluetoothDeviceObject(session);
               if(!client)
               {
                  ETG_TRACE_USR4(("This is not a MAP Session"));
                  break;
               }
               g_free(session);
            }
            if (strcmp (key, "Type") == 0 && g_variant_is_of_type (value, G_VARIANT_TYPE_STRING))
            {
               if(client)
               {
                  tBool isTransferActive = false;
                  tString type  = g_variant_dup_string(value,&length);

                  ETG_TRACE_USR4(("onObjectManagerInterfacesAdded::Type : %s", type));

                  if(strcmp(type, EVO_TRANSFER_TYPE_LIST_FOLDER) == 0)
                  {
                     //No need to store/create the transfer path/proxy for the ListFolder.
                     //client->_currentTransfer.insert(::std::pair<std::string, msgCallback>(objectPath, LIST_FOLDER_CB));
                     //isTransferActive = true;
                  }
                  else if(strcmp(type, EVO_TRANSFER_TYPE_LIST_MESSAGES) == 0)
                  {
                     client->_currentTransfer.insert(::std::pair<std::string, msgCallback>(objectPath, LIST_MESSAGES_CB));
                     isTransferActive = true;
                  }
                  else if((strcmp(type, EVO_TRANSFER_TYPE_SET_MSG_STATUS) == 0) && client->_isDeleteMessage)
                  {
                     client->_currentTransfer.insert(::std::pair<std::string, msgCallback>(objectPath, SET_MSG_STATUS_DELETED_CB));
                     isTransferActive = true;
                     client->_isDeleteMessage = false;
                  }
                  else if((strcmp(type, EVO_TRANSFER_TYPE_SET_MSG_STATUS) == 0) && !(client->_isDeleteMessage))
                  {
                     client->_currentTransfer.insert(::std::pair<std::string, msgCallback>(objectPath, SET_MSG_STATUS_READ_CB));
                     isTransferActive = true;
                  }
                  g_free(type);

                  if(isTransferActive == true)
                  {
                     client->createDBusPropertiesProxyForTransferPropertiesChanged(objectPath);
                  }
               }
            }
         }
      }
#ifdef CONNECT_PBAP_MAP_VIA_BM_APPL
      else if(strcmp(interface, BASE_EVO_SESSION_INTERFACE_NAME) == 0)
      {
         //Interface: Session
         g_variant_iter_init (&iter2, dict);
         while (g_variant_iter_next (&iter2, "{sv}", &key, &value))
         {
            if (strcmp (key, "Destination") == 0 && g_variant_is_of_type (value, G_VARIANT_TYPE_STRING))
            {
               tString destination  = g_variant_dup_string(value,&length);
               ETG_TRACE_USR4(("onObjectManagerInterfacesAdded::Destination : %s", destination));

               //find the device handle
               std::map<tU8, std::string>::iterator it;

               for(it = fc_messaging_tclClientHandler_BluetoothSetings::m_oConnectedDeviceList.begin();
                     it != fc_messaging_tclClientHandler_BluetoothSetings::m_oConnectedDeviceList.end(); it++)
               {
                  tU8 firstCounter, secondCounter;
                  char deviceAddress[18] = {0};
                  ::std::string bluetoothDeviceAddress = it->second;
                  for (firstCounter = 0, secondCounter = 0; firstCounter < bluetoothDeviceAddress.size(); firstCounter++, secondCounter++)
                  {
                     deviceAddress[secondCounter++] = static_cast<char>(toupper(bluetoothDeviceAddress[firstCounter++]));
                     deviceAddress[secondCounter++] = static_cast<char>(toupper(bluetoothDeviceAddress[firstCounter]));
                     deviceAddress[secondCounter] = ':';
                  }
                  deviceAddress[secondCounter-1] = '\0';

                  if(strcmp(deviceAddress, destination) == 0)
                  {
                     deviceHandle = it->first;
                  }
               }

               g_free(destination);
            }
         }
      }
      else if(strcmp(interface, BASE_EVO_MESSAGEACCESS_INTERFACE_NAME) == 0)
      {
         //Interface: MessageAccess

         //use the device handle which is identified in Session Interface
         client = getDeviceInstance(deviceHandle);
         if(client)
         {
            ::std::size_t len = strlen(objectPath.c_str()) + 1;
            tString sessionPath = new tChar[len];
            if(sessionPath)
            {
               memset(sessionPath, 0, len);
               strncpy(sessionPath, objectPath.c_str(), len);

               ETG_TRACE_USR4(("onObjectManagerInterfacesAdded::Session : %s", sessionPath));

               LoopBackMessageData loopBackMessageData;

               loopBackMessageData.ucStatus = (tU8)TRUE;
               loopBackMessageData.u8DeviceHandle = deviceHandle;
               loopBackMessageData.sServiceName = "";
               loopBackMessageData.vpData = (tVoid*)sessionPath;
               loopBackMessageData.u32FunctionId = FC_MSG_LB_FID_DEVICE_MAP_CONNECTION_STATUS;

               fc_messaging_tclService_Messaging::m_poMessagingService->vPostLoopBackMessage(loopBackMessageData);
            }
         }
      }
#endif
   }

   (void)deviceHandle; //Lint fix
}

void EvolutionDbusClient::onObjectManagerInterfacesRemoved(
         OrgFreedesktopDBusObjectManager *object,
         const gchar *arg_object,
         const gchar *const *arg_interfaces)
{
   ETG_TRACE_USR4(("onObjectManagerInterfacesRemoved() called"));

   (void)object;
   (void)arg_interfaces; //Lint fix

   std::string objectPath = arg_object;

   if(objectPath.empty())
   {
      return; //This should never happen
   }

   QMutexLocker lock(&_deviceListMutex);
   EvolutionDbusClient* bluetoothDeviceObject;

#ifdef CONNECT_PBAP_MAP_VIA_BM_APPL
   bluetoothDeviceObject = getBluetoothDeviceObject(objectPath.c_str());

   if(bluetoothDeviceObject)
   {
      //MessageAccess interface is removed (MAP Session closed)
      ETG_TRACE_USR4(("onObjectManagerInterfacesRemoved() for MAP Session"));

      LoopBackMessageData loopBackMessageData;

      loopBackMessageData.ucStatus = (tU8)FALSE;
      loopBackMessageData.u8DeviceHandle = bluetoothDeviceObject->_deviceHandle;
      loopBackMessageData.sServiceName = "";
      loopBackMessageData.u32FunctionId = FC_MSG_LB_FID_DEVICE_MAP_CONNECTION_STATUS;

      fc_messaging_tclService_Messaging::m_poMessagingService->vPostLoopBackMessage(loopBackMessageData);
   }
   else
   {
#endif
      bluetoothDeviceObject = getDeviceInstanceFromTransferPath(objectPath);

      if(bluetoothDeviceObject)
      {
         //MAP related Transfer interface is removed
         ETG_TRACE_USR4(("onObjectManagerInterfacesRemoved() for MAP related Transfer"));

         ::std::map<std::string, msgCallback>::iterator it;

         it = bluetoothDeviceObject->_currentTransfer.find(objectPath);
         if(it != bluetoothDeviceObject->_currentTransfer.end())
         {
            if(GET_MESSAGE_CB == it->second)
            {
               ETG_TRACE_USR4(("onObjectManagerInterfacesRemoved() for GetMessage"));
               if(access(MSG_DBUS_EVO_GETMESSAGE_PATH, F_OK) != -1)
               {
                  //parse the file and update
                  bluetoothDeviceObject->sendGetMessageSuccess();
               }
               else
               {
                  //Get message error
                  bluetoothDeviceObject->sendGetMessageError();
               }

               //Transfer PropertiesChanged has been missed, dereference the proxy
               if(bluetoothDeviceObject->_proxyGetMessage)
               {
                  g_object_unref(bluetoothDeviceObject->_proxyGetMessage);
               }
            }

            else if(PUSH_MESSAGE_CB == it->second)
            {
               ETG_TRACE_USR4(("onObjectManagerInterfacesRemoved() for PushMessage"));
               if(access(MSG_DBUS_EVO_PUSHMESSAGE_PATH, F_OK) != -1)
               {
                  //parse the file and update
                  bluetoothDeviceObject->sendPushMessageStatus(SUCCESS);
               }
               else
               {
                  //Push message error
                  bluetoothDeviceObject->sendPushMessageStatus(FAILURE);
               }
            }

            else if(SET_MSG_STATUS_READ_CB == it->second)
            {
               ETG_TRACE_USR4(("onObjectManagerInterfacesRemoved() for SetMsgStatus Read"));

               //Message read transfer removed, unlock the workqueue
               bluetoothDeviceObject->unlockDBusMutex();

               //Transfer PropertiesChanged has been missed, dereference the proxy
               if(bluetoothDeviceObject->_proxySetMessage)
               {
                  g_object_unref(bluetoothDeviceObject->_proxySetMessage);
               }
            }

            bluetoothDeviceObject->_currentTransfer.erase(it); //erase the current transfer path
         }
      }
#ifdef CONNECT_PBAP_MAP_VIA_BM_APPL
   }
#endif
}

void EvolutionDbusClient::onDBusMessageAccessPropertiesChangedCallback(OrgFreedesktopDBusProperties *object,
      const gchar *arg_interface,
      GVariant *arg_changed_properties,
      const gchar *const *arg_invalidated_properties)
{
   ETG_TRACE_USR4(("onDBusMessageAccessPropertiesChangedCallback: Interface : %s", arg_interface));

   (void)arg_invalidated_properties; //Lint fix

   GVariantIter iter;
   GVariant *value;
   gchar *key;
   gsize length = 0;

   std::string sessionPath = g_dbus_proxy_get_object_path((GDBusProxy*)object);

   if(sessionPath.empty())
   {
      return;
   }

   ETG_TRACE_USR4(("onDBusMessageAccessPropertiesChangedCallback: SessionPath : %s", sessionPath.c_str()));

#ifdef CONNECT_PBAP_MAP_VIA_BM_APPL
   //NCG3D-68657: Workaround Start: Incomplete MNS Connection
   if(strcmp(arg_interface, BASE_EVO_MESSAGEACCESS_INTERFACE_NAME) == 0)
   {
      EvolutionDbusClient* bluetoothDeviceObject = getBluetoothDeviceObject(sessionPath.c_str());

      if(!bluetoothDeviceObject)
      {
         return;
      }

      g_variant_iter_init (&iter, arg_changed_properties);
      while (g_variant_iter_next (&iter, "{sv}", &key, &value))
      {
         ETG_TRACE_USR4(("onDBusMessageAccessPropertiesChangedCallback: Property : %s", key));

         if (strcmp (key, "Notification") == 0 && g_variant_is_of_type (value, G_VARIANT_TYPE_STRING))
         {
            gchar* notification  = g_variant_dup_string(value,&length);
            ETG_TRACE_USR4(("Notification : %s", notification));

            if((strcmp(notification, "On") == 0) || (strcmp(notification, "on") == 0))
            {
               //stop the timer
               if (FALSE == fc_messaging_tclApp::m_poMessaging->bStopTimer(static_cast<tU16>(
                     FC_MESSAGING_MNSCONNECTIONRETRY_TIMER_ID + bluetoothDeviceObject->_deviceHandle)))
               {
                  ETG_TRACE_ERR(("Unable to stop the timer FC_MESSAGING_MNSCONNECTIONRETRY_TIMER_ID+%d",
                        bluetoothDeviceObject->_deviceHandle));
               }

               if((true == bluetoothDeviceObject->_currentActiveSMSMMSService.empty()) &&
                     (true == bluetoothDeviceObject->_currentActiveEmailService.empty()))
               {
                  LoopBackMessageData loopBackMessageData;
                  loopBackMessageData.u8DeviceHandle = bluetoothDeviceObject->_deviceHandle;
                  loopBackMessageData.sServiceName = "";
                  loopBackMessageData.u32FunctionId = FC_MSG_LB_FID_CONTINUE_MAP_SESSION;

                  fc_messaging_tclService_Messaging::m_poMessagingService->vPostLoopBackMessage(loopBackMessageData);
               }
            }
            else if((strcmp(notification, "Off") == 0) || (strcmp(notification, "off") == 0))
            {
               //On first time, don't stop the timer, after timeout retry will happen once.
               //If StartTimer failed (or) End of retry, MAS connection will be continued.
            }

            g_free(notification);
         }
      }
   } //Workaround End
#endif
}

void EvolutionDbusClient::createDBusPropertiesProxyForMessageAccessPropertiesChanged(std::string sessionPath)
{
   ETG_TRACE_USR4(("createDBusPropertiesProxyForMessageAccessPropertiesChanged() called"));

   if(sessionPath.empty())
   {
      ETG_TRACE_USR4(("Session Path Empty"));
      return;
   }

   GError* error = NULLPTR;

   //Create gdbus system proxy
   _proxyDBusPropertiesForMessageAccess = org_freedesktop_dbus_properties_proxy_new_sync(
         _evoMsgBus,
         G_DBUS_PROXY_FLAGS_NONE,
         BASE_EVO_SERVICE_NAME, //bus path
         sessionPath.c_str(), //obj name
         NULLPTR,
         &error);

   if (!_proxyDBusPropertiesForMessageAccess && error)
   {
      ETG_TRACE_ERR(("DBusPropertiesProxy Creation Failed for MessageAccess: Error : %s",
            g_dbus_error_get_remote_error (error)));
      g_error_free(error);

      NORMAL_M_ASSERT_ALWAYS();
   }
   else
   {
      g_signal_connect(_proxyDBusPropertiesForMessageAccess, "properties-changed",
            G_CALLBACK(onDBusMessageAccessPropertiesChangedCallback), NULLPTR);
   }
}

void EvolutionDbusClient::onDBusTransferPropertiesChangedCallback(OrgFreedesktopDBusProperties *object,
      const gchar *arg_interface,
      GVariant *arg_changed_properties,
      const gchar *const *arg_invalidated_properties)
{
   ETG_TRACE_USR4(("onDBusTransferPropertiesChangedCallback: Interface : %s", arg_interface));

   (void)arg_invalidated_properties; //Lint fix

   GVariantIter iter;
   GVariant *value;
   gchar *key;
   gsize length = 0;

   std::string transferPath = g_dbus_proxy_get_object_path((GDBusProxy*)object);

   if(transferPath.empty())
   {
      return;
   }

   ETG_TRACE_USR4(("onDBusTransferPropertiesChangedCallback: TransferPath : %s", transferPath.c_str()));

   if(strcmp(arg_interface, BASE_EVO_TRANSFER_INTERFACE_NAME) == 0)
   {
      EvolutionDbusClient* bluetoothDeviceObject = getDeviceInstanceFromTransferPath(transferPath);

      if(!bluetoothDeviceObject)
      {
         return;
      }

      g_variant_iter_init (&iter, arg_changed_properties);
      while (g_variant_iter_next (&iter, "{sv}", &key, &value))
      {
         ETG_TRACE_USR4(("onDBusTransferPropertiesChangedCallback: Property : %s", key));

         if (strcmp (key, "Status") == 0 && g_variant_is_of_type (value, G_VARIANT_TYPE_STRING))
         {
            gchar* status  = g_variant_dup_string(value,&length);
            ETG_TRACE_USR4(("Status : %s", status));

            ::std::map<std::string, msgCallback>::iterator it;

            QMutexLocker lock(&_deviceListMutex);
            ETG_TRACE_USR4(("Current Transfer Size : %d", (int)bluetoothDeviceObject->_currentTransfer.size()));

            it = bluetoothDeviceObject->_currentTransfer.find(transferPath);

            if(it != bluetoothDeviceObject->_currentTransfer.end())
            {
               ETG_TRACE_USR4(("Current Transfer Callback : %d", it->second));
               ETG_TRACE_USR4(("onDBusTransferPropertiesChangedCallback: TransferPath Found"));

               if((strcmp(status, "Complete") == 0) || (strcmp(status, "complete") == 0))
               {
                  if(PUSH_MESSAGE_CB == it->second)
                  {
                     bluetoothDeviceObject->sendPushMessageStatus(SUCCESS);
                  }
                  else if(GET_MESSAGE_CB == it->second)
                  {
                     //parse the file and update
                     bluetoothDeviceObject->sendGetMessageSuccess();
                     g_object_unref(bluetoothDeviceObject->_proxyGetMessage);
                  }
                  else if(SET_MSG_STATUS_DELETED_CB == it->second || SET_MSG_STATUS_READ_CB == it->second)
                  {
                     //Message deleted/read successfully
                     bluetoothDeviceObject->messagingMapSetMsgStatusCallback(it->second, SUCCESS);
                     g_object_unref(bluetoothDeviceObject->_proxySetMessage);

                  }

                  //dereference the DBusPropertiesProxy for Transfer
                  g_object_unref(object);
                  bluetoothDeviceObject->_currentTransfer.erase(it); //erase the current transfer path
                  break;
               }
               else if((strcmp(status, "Error") == 0) || (strcmp(status, "error") == 0))
               {
                  if(PUSH_MESSAGE_CB == it->second)
                  {
                     bluetoothDeviceObject->sendPushMessageStatus(FAILURE);
                  }
                  else if(GET_MESSAGE_CB == it->second)
                  {
                     //Get message error
                     bluetoothDeviceObject->sendGetMessageError();
                     g_object_unref(bluetoothDeviceObject->_proxyGetMessage);
                  }
                  else if(SET_MSG_STATUS_DELETED_CB == it->second || SET_MSG_STATUS_READ_CB == it->second)
                  {
                     //Message not deleted/read
                     bluetoothDeviceObject->messagingMapSetMsgStatusCallback(it->second, FAILURE);
                     g_object_unref(bluetoothDeviceObject->_proxySetMessage);
                  }

                  //dereference the DBusPropertiesProxy for Transfer
                  g_object_unref(object);
                  bluetoothDeviceObject->_currentTransfer.erase(it); //erase the current transfer path
                  break;
               }
            }
            g_free(status);
         }
      }
   }
}

void EvolutionDbusClient::createDBusPropertiesProxyForTransferPropertiesChanged(std::string transferPath)
{
   ETG_TRACE_USR4(("createDBusPropertiesProxyForTransferPropertiesChanged() called"));

   if(transferPath.empty())
   {
      ETG_TRACE_USR4(("Transfer Path Empty"));
      return;
   }

   GError* error = NULLPTR;

   //Create gdbus system proxy
   OrgFreedesktopDBusProperties* dbusSystemProxy = org_freedesktop_dbus_properties_proxy_new_sync(
         _evoMsgBus,
         G_DBUS_PROXY_FLAGS_NONE,
         BASE_EVO_SERVICE_NAME, //bus path
         transferPath.c_str(), //obj name
         NULLPTR,
         &error);

   if (!dbusSystemProxy && error)
   {
      ETG_TRACE_ERR(("DBusPropertiesProxy Creation Failed for Transfer: Error : %s", g_dbus_error_get_remote_error (error)));
      g_error_free(error);

      NORMAL_M_ASSERT_ALWAYS();
   }
   else
   {
      g_signal_connect(dbusSystemProxy, "properties-changed", G_CALLBACK(onDBusTransferPropertiesChangedCallback), NULLPTR);
   }
}

void EvolutionDbusClient::messagingMapNotificationRegistry()
{
   ETG_TRACE_USR4(("messagingMapNotificationRegistry() called"));

   gboolean result;
   gboolean action = BASE_EVO_MESSAGE_NOTIFICATION_ON;
   GError* error = NULLPTR;
   bool continueMAPSession = false;

   result = org_bluez_obex_message_access_call_notification_registry_sync(_proxyMessageAccess, action, NULLPTR, &error);
   if (error && (FALSE == result))
   {
      ETG_TRACE_USR4(("NotificationRegistry: Error: %s", g_dbus_error_get_remote_error (error)));

      g_error_free(error);

      //NotificationRegistry Error received, continue with MAS connection since MAP Notification On/Off will not be received.
      continueMAPSession = true;
   }
   else
   {
      ETG_TRACE_USR4(("NotificationRegistry: Success"));

      //NCG3D-68657: Workaround Start: Incomplete MNS Connection
      //Timer 3000ms should be started here to detect MNS connection failure
      std::vector< tU8 >::iterator it = std::find(fc_messaging_tclClientHandler_BluetoothSetings::m_u8MAPConnRetryReqForMNSList.begin(),
            fc_messaging_tclClientHandler_BluetoothSetings::m_u8MAPConnRetryReqForMNSList.end(), _deviceHandle);

      if(it == fc_messaging_tclClientHandler_BluetoothSetings::m_u8MAPConnRetryReqForMNSList.end())
      {
         if (FALSE == fc_messaging_tclApp::m_poMessaging->bStartTimer(
               (tU16)(FC_MESSAGING_MNSCONNECTIONRETRY_TIMER_ID+_deviceHandle),
               EVO_MAP_NOTIFICATION_WAIT_TIMER,
               EVO_MAP_NOTIFICATION_WAIT_TIMER))
         {
            ETG_TRACE_ERR(("Unable to start the timer FC_MESSAGING_MNSCONNECTIONRETRY_TIMER_ID+%d", _deviceHandle));

            //StartTimer failed, continue with MAS connection.
            continueMAPSession = true;
         }
      }
      else
      {
         fc_messaging_tclClientHandler_BluetoothSetings::removeDeviceFromMAPConnRetryForMNSList(_deviceHandle);

         //End of retry (MAP session reconnected), continue with MAS connection.
         continueMAPSession = true;
      }
      //Workaround End
   }

   if(true == continueMAPSession)
   {
#ifdef CONNECT_PBAP_MAP_VIA_BM_APPL
      //NotificationRegistry Error received, continue with MAS connection since MAP Notification On/Off will not be received.
      messagingMapSessionConnected();
#endif
   }
}

void EvolutionDbusClient::handleNewMessageNotification(const most_msgfi_tclMsgCreateMessageListMethodStart* createMessageListMethodStart)
{
   ETG_TRACE_USR4(("handleNewMessageNotification() called"));

   QMutexLocker lock(&_newMsgHandleListMutex);

   if (TRUE == _newMessageHandleList.isEmpty())
   {
      //All new text messages are resolved, release Mutex lock
      MessagingMutex::messagingMutexUnlock();
      return;
   }

   if(most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_SENT == createMessageListMethodStart->e8FolderType.enType)
   {
      messagingMapListMsg(*createMessageListMethodStart, 1, e8MapListMsgCallerSendMessage);
   }
   else
   {
      messagingMapListMsg(*createMessageListMethodStart,
            static_cast<tU16>(_newMessageHandleList.size() * 5),   //To handle simultaneous multiple new messages
            e8MapListMsgCallerNewIncomingMessage);
   }
}

void EvolutionDbusClient::getNotificationType(::std::string data, guchar& type)
{
   //find the type of notification received from device
   //if type not identified then default value will be set (in calling function itself)
   //default value: 0xFF

   //types are coded as per MAP specification
   if (data.find("NewMessage") != std::string::npos)
   {
      type = TYPE_NEW_MESSAGE;
   }
   else if (data.find("DeliverySuccess") != std::string::npos)
   {
      type = TYPE_DELIVERY_SUCCESS;
   }
   else if (data.find("SendingSuccess") != std::string::npos)
   {
      type = TYPE_SENDING_SUCCESS;
   }
   else if (data.find("DeliveryFailure") != std::string::npos)
   {
      type = TYPE_DELIVERY_FAILURE;
   }
   else if (data.find("SendingFailure") != std::string::npos)
   {
      type = TYPE_SENDING_FAILURE;
   }
   else if (data.find("MemoryFull") != std::string::npos)
   {
      type = TYPE_MEMORY_FULL;
   }
   else if (data.find("MemoryAvailable") != std::string::npos)
   {
      type = TYPE_MEMORY_AVAILABLE;
   }
   else if (data.find("MessageDeleted") !=std::string::npos)
   {
      type = TYPE_MESSAGE_DELETED;
   }
   else if (data.find("MessageShift") != std::string::npos)
   {
      type = TYPE_MESSAGE_SHIFT;
   }
   else if (data.find("ReadStatusChanged") != std::string::npos)
   {
         type = TYPE_READ_STATUS_CHANGED;
   }
}

void EvolutionDbusClient::getNotificationMsgType(::std::string data, guchar& msgType)
{
   //find the message type
   //if message type not identified then default value will be set (in calling function itself)
   //default value: NOTIFY_MSG_IGNORE (0xFF)

   //msg types are coded as per MAP specification
   if (data.find("EMAIL") != std::string::npos)
   {
      msgType = NOTIFY_MSG_EMAIL;
   }
   else if (data.find("SMS_GSM") != std::string::npos)
   {
      msgType = NOTIFY_MSG_SMS_GSM;
   }
   else if (data.find("SMS_CDMA") != std::string::npos)
   {
      msgType = NOTIFY_MSG_SMS_CDMA;
   }
   else if (data.find("MMS") != std::string::npos)
   {
      msgType = NOTIFY_MSG_MMS;
   }
}

void EvolutionDbusClient::getNotificationData(::std::string data, ::std::size_t position, tString* value)
{
   //generic handling for "handle", "folder", "old_folder"
   //if "handle", "folder", "old_folder" not identified then NULL will be set in calling function
   if (position != std::string::npos)
   {
      std::size_t valueStart = data.find("\"", position) + 1; //value starts after first double apostrophe(")
      std::size_t valueEnd = data.find("\"", valueStart);     //value ends before second double apostrophe(")
      std::size_t valueLength = (valueEnd-valueStart);

      tString buffer = new tChar[valueLength+1];
      if(buffer)
      {
         std::size_t length = data.copy(buffer, valueLength, valueStart);
         buffer[length]='\0';
         *value = buffer;
      }
   }
}

void EvolutionDbusClient::getNotificationFolderType(::std::string data, ::std::size_t position, most_fi_tcl_e8_MsgFolderType::tenType& folder)
{
   //generic handling for "handle", "folder", "old_folder"
   //if "handle", "folder", "old_folder" not identified then NULL will be set in calling function
   if (position != std::string::npos)
   {
      std::size_t valueStart = data.find("\"", position) + 1; //value starts after first double apostrophe(")
      std::size_t valueEnd = data.find("\"", valueStart);     //value ends before second double apostrophe(")
      std::size_t valueLength = (valueEnd-valueStart);

      tString buffer = new tChar[valueLength+1];
      if(buffer)
      {
         QString folderName;

         std::size_t length = data.copy(buffer, valueLength, valueStart);
         buffer[length]='\0';
         folderName = convertFromNCR(buffer);

         if(folderName.contains(MSG_DBUS_FOLDER_TYPE_INBOX, Qt::CaseInsensitive))
         {
            folder = most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_INBOX;
         }
         else if(folderName.contains(MSG_DBUS_FOLDER_TYPE_OUTBOX, Qt::CaseInsensitive))
         {
            folder = most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_OUTBOX;
         }
         else if(folderName.contains(MSG_DBUS_FOLDER_TYPE_SENT, Qt::CaseInsensitive))
         {
            folder = most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_SENT;
         }
         else if(folderName.contains(MSG_DBUS_FOLDER_TYPE_DRAFT, Qt::CaseInsensitive))
         {
            folder = most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_DRAFTS;
         }
         else if(folderName.contains(MSG_DBUS_FOLDER_TYPE_DELETED, Qt::CaseInsensitive))
         {
            folder = most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_DELETED;
         }
         else
         {
            folder = most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_USERDEFINED;
         }

         delete[] buffer;
      }
   }
}

void EvolutionDbusClient::onNewMessageNotification1(OrgBluezObexMessageAccess *object, GVariant *arg_data, const gchar *arg_path)
{
   ETG_TRACE_USR4(("onNewMessageNotification1() called"));

   onNewMessageNotification(object, arg_data, arg_path);
}

void EvolutionDbusClient::onNewMessageNotification2(OrgBluezObexMessageAccess *object, GVariant *arg_data, const gchar *arg_path)
{
   ETG_TRACE_USR4(("onNewMessageNotification2() called"));

   onNewMessageNotification(object, arg_data, arg_path);
}

void EvolutionDbusClient::onNewMessageNotification(OrgBluezObexMessageAccess *object, GVariant *arg_data, const gchar *arg_path)
{
   ETG_TRACE_USR4(("onNewMessageNotification() called"));

   (void)arg_path; //Lint Fix

   QMutexLocker lock(&_deviceListMutex);
   EvolutionDbusClient* bluetoothDeviceObject = getBluetoothDeviceObject(object);
   if(!bluetoothDeviceObject)
   {
      ETG_TRACE_USR4(("Proxy not found"));
      return;
   }

   gsize size = g_variant_get_size(arg_data);
   ETG_TRACE_USR4(("onNewMessageNotification() %d bytes received", (int)size));

   gconstpointer variable = g_variant_get_data(arg_data);
   tString data = new tChar[size+1];

   guchar type = 0xFF;
   tString messageHandle = NULLPTR;
   most_fi_tcl_e8_MsgFolderType::tenType folder = most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_INBOX;
   most_fi_tcl_e8_MsgFolderType::tenType oldFolder = most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_INBOX;
   guchar msgType = NOTIFY_MSG_IGNORE;

   if(data)
   {
      memset(data, 0, size+1);
      memcpy(data, variable, size);

      ::std::string strdata(data);

      _BP_TRY_BEGIN
      {
         ETG_TRACE_USR4(("onNewMessageNotification() : %s", strdata.c_str()));

         bluetoothDeviceObject->getNotificationType(strdata, type);

         bluetoothDeviceObject->getNotificationData(strdata, strdata.find("handle"), &messageHandle);

		 bluetoothDeviceObject->getNotificationFolderType(strdata, strdata.find("folder"), folder);

		 bluetoothDeviceObject->getNotificationFolderType(strdata, strdata.find("old_folder"), oldFolder);

         bluetoothDeviceObject->getNotificationMsgType(strdata, msgType);

         ETG_TRACE_USR4(("onNewMessageNotification() type : %u", ETG_ENUM(TR_MSGNOTIFICATIONTYPE, type)));
         ETG_TRACE_USR4(("onNewMessageNotification() folder : %u", ETG_ENUM(TR_FOLDER_TYPE, folder)));
         ETG_TRACE_USR4(("onNewMessageNotification() oldFolder : %u", ETG_ENUM(TR_FOLDER_TYPE, oldFolder)));
         ETG_TRACE_USR4(("onNewMessageNotification() msgType : %u", ETG_ENUM(TR_MSGTYPE, msgType)));
      }
      _BP_CATCH_ALL
      {
         delete[] data;
         delete[] messageHandle;

         ETG_TRACE_ERR(("onNewMessageNotification() Caught Exception"));
         return;
      }
      _BP_CATCH_END

      delete[] data;
   }
   else
   {
      return;
   }

   tU8 messageType = 0;
   tU8 notifyType = 0xFF;
   tBool notifySend = false;


#ifndef EMAIL_SUPPORT
   if (NOTIFY_MSG_EMAIL == msgType) //Email not supported in GM
   {
      ETG_TRACE_USR4(("Email Notification not supported"));
      delete[] messageHandle;
      return;
   }
#endif

   tU8 deviceHandle = getDeviceHandle((OrgBluezObexMessageAccess*)object);
   if (deviceHandle == 0)
   {
      delete[] messageHandle;
      return; // Invalid device handle
   }

   switch (type)
   {
      case TYPE_NEW_MESSAGE:
      {
         if ((most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_INBOX == folder) ||
               (most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_SENT == folder))
         {
            if(messageHandle)
            {
               ETG_TRACE_USR4(("onNewMessageNotification() messageHandle : %s", messageHandle));

               // Two consecutive New Incoming Message Bug-Fix GMMY15-1259
               QByteArray newMsgHandle(messageHandle, (tU8)strlen(messageHandle));

               QMutexLocker newmsglock(&(bluetoothDeviceObject->_newMsgHandleListMutex));
               bluetoothDeviceObject->_newMessageHandleList.append(newMsgHandle);

               // Bugfix for GMMY17-5280: The new text message received while reading previous
               //                         messages is not shown in the INBOX for the SMS text application.
               struct Job* prJob = new Job;

               if (prJob)
               {
                  most_msgfi_tclMsgCreateMessageListMethodStart* createMessageListMethodStart = new most_msgfi_tclMsgCreateMessageListMethodStart;

                  if (createMessageListMethodStart)
                  {
                     createMessageListMethodStart->u8DeviceHandle = bluetoothDeviceObject->_deviceHandle;

#ifdef EMAIL_SUPPORT //removal lint warning
                     if (msgType == NOTIFY_MSG_EMAIL)
                     {
                        createMessageListMethodStart->e8MessageListType.enType = most_fi_tcl_e8_MsgMessageListType::FI_EN_E8MSG_LIST_EMAIL;
                     }
                     else
#else
                     {
                        createMessageListMethodStart->e8MessageListType.enType = most_fi_tcl_e8_MsgMessageListType::FI_EN_E8MSG_LIST_SMS_MMS;
                     }
#endif

                     createMessageListMethodStart->e8FolderType.enType = folder;

                     prJob->m_pfMethod = &fc_messaging_tclService_Messaging::vHandle_NewMessageNotification;
                     prJob->u8DeviceHandle = bluetoothDeviceObject->_deviceHandle;
                     prJob->u16CmdCounter = 0;
                     prJob->u16FunctionID = 0;
                     prJob->u16RegisterID = 0;
                     prJob->u16SourceAppID = 0;
                     prJob->m_pvMethodStart = createMessageListMethodStart;

                     // CMG3G-7699 - IS2202_message send notification (network level)@FC_Messaging
                     bluetoothDeviceObject->_msgNotifyMap.insert(newMsgHandle, (tU8)most_fi_tcl_e8_MsgNotificationType::FI_EN_E8NOTIFY_NEW_MSG);

                     if (0 != MessagingMutex::messagingMutexTrylock(prJob))
                     {
                        tVoid* pCustodialLose = prJob; //To remove LINT warning : Custodial Pointer
                        (tVoid) pCustodialLose;
                        break; // Didn't get the lock
                     }

                     if (most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_SENT == folder)
                     {
                        bluetoothDeviceObject->messagingMapListMsg(*createMessageListMethodStart, 1, e8MapListMsgCallerSendMessage);
                     }
                     else
                     {
                        //Request for the Message Details of New Text Message(s)
                        bluetoothDeviceObject->messagingMapListMsg(*createMessageListMethodStart, static_cast<tU16>(bluetoothDeviceObject->_newMessageHandleList.size() * 2), e8MapListMsgCallerNewIncomingMessage);
                     }

                     createMessageListMethodStart->vDestroy();

                     OSAL_DELETE createMessageListMethodStart;
                  }
                  OSAL_DELETE prJob;
               }
            }
         }
      }
      break;

      case TYPE_DELIVERY_SUCCESS:
      case TYPE_SENDING_SUCCESS:
      case TYPE_MESSAGE_SHIFT:
      {
         if (TRUE == fc_messaging_tclApp::m_poMessaging->bStopTimer(static_cast<tU16>(FC_MESSAGING_OUTBOXCHECK_WAIT_TIMER_ID + bluetoothDeviceObject->_deviceHandle)))
         {
            ETG_TRACE_ERR(("The timer FC_MESSAGING_OUTBOXCHECK_WAIT_TIMER_ID is Stopped"));
         }
         if (((most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_SENT == folder)
               || (most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_OUTBOX == folder)) && (folder != oldFolder))
         {
            //Bug Fix for GMMY15-10258 - Replying to a message when phone is in No Network condition displays
            //                           "Text Message Sent" alert instead of "Text Message Delivery Failure" alert.
            bluetoothDeviceObject->_mapNotifyStatus = TRUE;

            //CMG3G-1494 - Changes for Email support and sent items folder
            most_fi_tcl_e8_MsgMessageListType::tenType enMsgType;
#ifdef EMAIL_SUPPORT //removal lint warning
            if (msgType == NOTIFY_MSG_EMAIL)
            {
               enMsgType = most_fi_tcl_e8_MsgMessageListType::FI_EN_E8MSG_LIST_EMAIL;
               messageType = most_fi_tcl_e8_MsgMessageType::FI_EN_E8MSG_TYPE_EMAIL;
            }
            else
            {
#endif
               if(msgType == NOTIFY_MSG_MMS)
               {
                  messageType = most_fi_tcl_e8_MsgMessageType::FI_EN_E8MSG_TYPE_MMS;
               }
               else
               {
                  messageType = most_fi_tcl_e8_MsgMessageType::FI_EN_E8MSG_TYPE_SMS;
               }

               enMsgType = most_fi_tcl_e8_MsgMessageListType::FI_EN_E8MSG_LIST_SMS_MMS;
#ifdef EMAIL_SUPPORT //removal lint warning
            }
#endif

            if(messageHandle)
            {
               ETG_TRACE_USR4(("onNewMessageNotification() messageHandle : %s", messageHandle));

               // CMG3G-7699 - IS2202_message send notification (network level)@FC_Messaging
               QByteArray baMsgHandle(messageHandle, (tU8)strlen(messageHandle));
               if(type == TYPE_SENDING_SUCCESS || type == TYPE_MESSAGE_SHIFT)
               {
                  bluetoothDeviceObject->getSendMessageList(folder, enMsgType, baMsgHandle, (tU8)most_fi_tcl_e8_MsgNotificationType::FI_EN_E8NOTIFY_MSG_SEND_SUCCESS);
               }
               else
               {
                  if(true != bluetoothDeviceObject->_messageHandleList.contains(baMsgHandle))
                  {
                     bluetoothDeviceObject->getSendMessageList(folder, enMsgType, baMsgHandle, (tU8)most_fi_tcl_e8_MsgNotificationType::FI_EN_E8NOTIFY_MSG_DELIVERY_SUCCESS);
                  }
                  else
                  {
                     notifyType = (tU8)most_fi_tcl_e8_MsgNotificationType::FI_EN_E8NOTIFY_MSG_DELIVERY_SUCCESS;
                     notifySend = true;
                  }
               }
            }
         }
      }

      break;

      // Bugfix for GMMY16-9063: Text message sent failure alert is not shown when reply to an invalid number
      case TYPE_DELIVERY_FAILURE:
      case TYPE_SENDING_FAILURE:
      {
         if (TRUE == fc_messaging_tclApp::m_poMessaging->bStopTimer(static_cast<tU16>(FC_MESSAGING_OUTBOXCHECK_WAIT_TIMER_ID + bluetoothDeviceObject->_deviceHandle)))
         {
            ETG_TRACE_ERR(("The timer FC_MESSAGING_OUTBOXCHECK_WAIT_TIMER_ID is Stopped"));
         }
         // Bug Fix for GMMY15-10258 - Replying to a message when phone is in No Network condition displays
         //                            "Text Message Sent" alert instead of "Text Message Delivery Failure" alert

         if ((most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_SENT == folder)
               || (most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_OUTBOX == folder))
         {
            ETG_TRACE_USR1(("Msg found in outbox/sent folder"));
            bluetoothDeviceObject->_mapNotifyStatus = FALSE;

            //CMG3G-1494 - Changes for Email support and sent items folder
            most_fi_tcl_e8_MsgMessageListType::tenType enMsgType;
#ifdef EMAIL_SUPPORT //removal lint warning
            if (msgType == NOTIFY_MSG_EMAIL)
            {
               enMsgType = most_fi_tcl_e8_MsgMessageListType::FI_EN_E8MSG_LIST_EMAIL;
               messageType = most_fi_tcl_e8_MsgMessageType::FI_EN_E8MSG_TYPE_EMAIL;
            }
            else
            {
#endif
               if(msgType == NOTIFY_MSG_MMS)
               {
                  messageType = most_fi_tcl_e8_MsgMessageType::FI_EN_E8MSG_TYPE_MMS;
               }
               else
               {
                  messageType = most_fi_tcl_e8_MsgMessageType::FI_EN_E8MSG_TYPE_SMS;
               }

               enMsgType = most_fi_tcl_e8_MsgMessageListType::FI_EN_E8MSG_LIST_SMS_MMS;
#ifdef EMAIL_SUPPORT
            }
#endif

            if(messageHandle)
            {
               ETG_TRACE_USR4(("onNewMessageNotification() messageHandle : %s", messageHandle));

               // CMG3G-7699 - IS2202_message send notification (network level)@FC_Messaging
               QByteArray baMsgHandle(messageHandle, (tU8)strlen(messageHandle));
               if(type == TYPE_SENDING_FAILURE)
               {
                  bluetoothDeviceObject->getSendMessageList(folder, enMsgType, baMsgHandle, (tU8)most_fi_tcl_e8_MsgNotificationType::FI_EN_E8NOTIFY_MSG_SEND_FAILURE);
               }
               else
               {
                  if(true != bluetoothDeviceObject->_messageHandleList.contains(baMsgHandle))
                  {
                     bluetoothDeviceObject->getSendMessageList(folder, enMsgType, baMsgHandle, (tU8)most_fi_tcl_e8_MsgNotificationType::FI_EN_E8NOTIFY_MSG_DELIVERY_FAILURE);
                  }
                  else
                  {
                     notifyType = (tU8)most_fi_tcl_e8_MsgNotificationType::FI_EN_E8NOTIFY_MSG_DELIVERY_FAILURE;
                     notifySend = true;
                  }
               }
            }
         }
      }
      break;

      case TYPE_MEMORY_FULL:
      case TYPE_MEMORY_AVAILABLE:
      {
         if (TYPE_MEMORY_FULL == type)
         {
            bluetoothDeviceObject->_deviceConnection.m_bMessagingMemoryStatus = FALSE;
         }
         else
         {
            bluetoothDeviceObject->_deviceConnection.m_bMessagingMemoryStatus = TRUE;
         }

         LoopBackMessageData loopBackMessageData;
         loopBackMessageData.vpData = (tVoid*) (&bluetoothDeviceObject->_deviceConnection);
         loopBackMessageData.u32FunctionId = FC_MSG_LB_FID_DEVICE_MEMORY_STATUS;
         loopBackMessageData.u8DeviceHandle = deviceHandle;
         loopBackMessageData.sServiceName = bluetoothDeviceObject->_clientSessionPath;
         fc_messaging_tclService_Messaging::m_poMessagingService->vPostLoopBackMessage(loopBackMessageData);
      }
      break;

      case TYPE_MESSAGE_DELETED:
      {
         //GMMY15_8172 : Text message remains in inbox when deleted from phone
         //CMG3G-1494 - Changes for Email support and sent items folder
         MapNotifySignalLoopBackData *notifyData = new MapNotifySignalLoopBackData;

         if (messageHandle && notifyData)
         {
            if ((most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_INBOX == folder) ||
                  (most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_SENT == folder))
            {
               notifyData->u8FolderType = folder;
            }
            else
            {
               OSAL_DELETE notifyData; //removal lint warning
               return;
            }

            //Remove the message handle from the list, if it is deleted from device.
            QByteArray baMsgHandle(messageHandle, (tU8)strlen(messageHandle));
            if(true == bluetoothDeviceObject->_messageHandleList.contains(baMsgHandle))
            {
               bluetoothDeviceObject->_messageHandleList.removeOne(baMsgHandle);
            }

            // CMG3G-7699 - IS2202_message send notification (network level)@FC_Messaging
            notifyData->u8MessageType = msgType;
            notifyData->pacMessageHandle = new QByteArray(messageHandle, (tU8)strlen(messageHandle));

            LoopBackMessageData loopBackMessageData;
            loopBackMessageData.vpData = (tVoid*) (notifyData);
            loopBackMessageData.u32FunctionId = FC_MSG_LB_FID_MESSAGE_DELETED_FROM_DEVICE;
            loopBackMessageData.u8DeviceHandle = deviceHandle;
            loopBackMessageData.sServiceName = bluetoothDeviceObject->_clientSessionPath;

            fc_messaging_tclService_Messaging::m_poMessagingService->vPostLoopBackMessage(loopBackMessageData);
         }
      }
      break;

      case TYPE_READ_STATUS_CHANGED:
            {
               MapNotifySignalLoopBackData *notifyData = new MapNotifySignalLoopBackData;

         if (messageHandle && notifyData)
         {
            if ((most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_INBOX == folder) ||
                  (most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_SENT == folder))
            {
               notifyData->u8FolderType = folder;
            }
            else
            {
               OSAL_DELETE notifyData; //removal lint warning
               return;
            }

                  notifyData->u8MessageType = msgType;
                  notifyData->pacMessageHandle = new QByteArray(messageHandle, (tU8)strlen(messageHandle));

                  LoopBackMessageData loopBackMessageData;
                  loopBackMessageData.vpData = (tVoid*) (notifyData);
                  loopBackMessageData.u32FunctionId = FC_MSG_LB_FID_READ_STATUS_CHANGED;
                  loopBackMessageData.u8DeviceHandle = deviceHandle;
                  loopBackMessageData.sServiceName = bluetoothDeviceObject->_clientSessionPath;

                  fc_messaging_tclService_Messaging::m_poMessagingService->vPostLoopBackMessage(loopBackMessageData);
               }
            }
            break;

      default:
         ETG_TRACE_USR1(("Wrong Type"));
   }

   // CMG3G-7699 - IS2202_message send notification (network level)@FC_Messaging
   if(notifySend == true)
   {
      trFC_Msg_Notification *prFC_Msg_Notification = new trFC_Msg_Notification;

      if(prFC_Msg_Notification)
      {
         prFC_Msg_Notification->u8MessgeType = messageType;
         prFC_Msg_Notification->u8MessgeNotificationType = notifyType;

         ETG_TRACE_USR1(("Msg Notify: %d", prFC_Msg_Notification->u8MessgeNotificationType));

         if ((most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_SENT == folder) ||
               (most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_OUTBOX == folder))
         {
            prFC_Msg_Notification->rMapNoitfyData.u8FolderType = most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_SENT;
         }

         prFC_Msg_Notification->u8DeviceHandle = bluetoothDeviceObject->_deviceHandle;
         if(messageHandle)
         {
            prFC_Msg_Notification->rMapNoitfyData.pacMessageHandle = new QByteArray(messageHandle, (tU8)strlen(messageHandle));
         }
         else
         {
            ETG_TRACE_USR4(("messageHandle is null"));
         }

         LoopBackMessageData loopBackMessageData;
         loopBackMessageData.vpData = (tVoid*) (prFC_Msg_Notification);
         loopBackMessageData.u32FunctionId = FC_MSG_LB_FID_MSG_NOTIFICATION_STATUS;
         loopBackMessageData.u8DeviceHandle = deviceHandle;
         loopBackMessageData.sServiceName = bluetoothDeviceObject->_clientSessionPath;
         fc_messaging_tclService_Messaging::m_poMessagingService->vPostLoopBackMessage(loopBackMessageData);
      }
   }

   delete[] messageHandle;
}

//CMG3G-1494 - Changes for Email support and sent items folder
void EvolutionDbusClient::getSendMessageList(most_fi_tcl_e8_MsgFolderType::tenType enFolderType, most_fi_tcl_e8_MsgMessageListType::tenType enMsgType,
      QByteArray msgHandle, tU8 msgNotificationType)
{
   ETG_TRACE_USR4(("getSendMessageList() called"));

   most_msgfi_tclMsgCreateMessageListMethodStart createMessageListMethodStart;

   createMessageListMethodStart.u8DeviceHandle = _deviceHandle;

   ETG_TRACE_USR4(("MessageType: %d",enMsgType));

   //CMG3G-7699 - IS2202_message send notification (network level)@FC_Messaging
   createMessageListMethodStart.e8MessageListType.enType = enMsgType;

   ETG_TRACE_USR4(("m_oCreateMessageMethodStart.e8MessageType.enType::FI_EN_E8MSG_FOLDER_SENT"));
   createMessageListMethodStart.e8FolderType.enType = enFolderType;

   //CMG3G-7699 - IS2202_message send notification (network level)@FC_Messaging
   _msgNotifyMap.insert(msgHandle, msgNotificationType);

   //Do not Download the message list when already list download is in progress
   if (MessagingMutex::messagingMutexTrylock(NULLPTR) != 0)
      return;

   messagingMapListMsg(createMessageListMethodStart, 1, e8MapListMsgCallerSendMessage);
}

void EvolutionDbusClient::messagingMapUpdateInboxCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR4(("messagingMapUpdateInboxCallback() called"));

   (void)user_data; //Lint fix

   gboolean result;
   GError* error = NULLPTR;

   result = org_bluez_obex_message_access_call_update_inbox_finish((OrgBluezObexMessageAccess*)source_object, res, &error);

   if (error && (FALSE == result))
   {
      ETG_TRACE_USR4(("UpdateInbox: Error: %s", g_dbus_error_get_remote_error (error)));
      fc_messaging_tclService_Messaging::m_poMessagingService->vHandle_UpdateInbox_Error();

      g_error_free(error);
      return;
   }

   fc_messaging_tclService_Messaging::m_poMessagingService-> vHandle_UpdateInbox_Return();
}

tBool EvolutionDbusClient::messagingMapUpdateInbox()
{
   ETG_TRACE_USR4(("messagingMapUpdateInbox() called"));

   if(!_proxyMessageAccess)
   {
      ETG_TRACE_USR4(("MessageAccess Proxy NULL"));
      return false;
   }

   org_bluez_obex_message_access_call_update_inbox(_proxyMessageAccess,
         NULLPTR, messagingMapUpdateInboxCallback, NULLPTR);

   return true;
}

void EvolutionDbusClient::handleMapListFolderComplete(tU16 listSize)
{
   ETG_TRACE_USR4(("handleMapListFolderComplete() called"));

   if (FC_MSG_FOLDER_LIST_REQUESTED_FOR_SMSMMS_SERVICE == _folderListRequestStatus)
   {
      //if email service available then check for the folder structure
      if (strlen(_currentActiveEmailService.c_str()) > 0)
      {
         //Lock is not required here as it is already acquired by SMSMMS Service
         _folderListRequestStatus = FC_MSG_FOLDER_LIST_REQUESTED_FOR_EMAIL_SERVICE;

         ::std::string folder = "/telecom/msg";
         messagingMapListFolder(folder);
      }
      else
      {
         folderListRequestComplete();
      }
   }
   else if (FC_MSG_FOLDER_LIST_REQUESTED_FOR_EMAIL_SERVICE == _folderListRequestStatus)
   {
      folderListRequestComplete();
   }
   else
   {
      LoopBackMessageData loopBackMessageData;

      loopBackMessageData.u32FunctionId = FC_MSG_LB_FID_FOLDER_LIST_COMPLETE;
      loopBackMessageData.u8DeviceHandle = _deviceHandle;
      loopBackMessageData.u16Private = listSize;
      loopBackMessageData.sServiceName = _clientSessionPath;

      fc_messaging_tclService_Messaging::m_poMessagingService->vPostLoopBackMessage(loopBackMessageData);
   }
}

void EvolutionDbusClient::handleMapListFolderData(GVariant* folderList, tU16& listSize)
{
   ETG_TRACE_USR4(("handleMapListFolderData() called"));

   //folderList type is aa{sv}
   GVariantIter iter1, iter2;
   GVariant* array;
   gchar* key;
   GVariant* value;
   gsize length;

   g_variant_iter_init (&iter1, folderList);
   while((array = g_variant_iter_next_value(&iter1)) != NULLPTR) //iterating first array
   {
      g_variant_iter_init (&iter2, array);
      while (g_variant_iter_next (&iter2, "{sv}", &key, &value)) //iterating second array
      {
         if (strcmp (key, "Name") == 0 && g_variant_is_of_type (value, G_VARIANT_TYPE_STRING))
         {
            gchar* folder  = const_cast<gchar*>(g_variant_get_string(value, &length));
            ETG_TRACE_USR4(("ListFolders::Folder : %s", folder));

            if (FC_MSG_FOLDER_LIST_NOT_REQUESTED != _folderListRequestStatus)
            {
               if ((strcmp(MSG_DBUS_FOLDER_TYPE_OUTBOX, folder) == 0) || (strcmp(MSG_DBUS_FOLDER_TYPE_OUTBOX_UC, folder) == 0))
               {
                  if (FC_MSG_FOLDER_LIST_REQUESTED_FOR_SMSMMS_SERVICE == _folderListRequestStatus)
                  {
                     if (FC_MSG_SMS_SUPPORT == (_mapDeviceCapabilities.u16SupportedMessageTypes & FC_MSG_SMS_SUPPORT))
                     {
                        _mapDeviceCapabilities.u16SupportedMapFeatures |= FC_MSG_SMS_SEND_SUPPORT;
                     }
                     if (FC_MSG_MMS_SUPPORT == (_mapDeviceCapabilities.u16SupportedMessageTypes & FC_MSG_MMS_SUPPORT))
                     {
                        _mapDeviceCapabilities.u16SupportedMapFeatures |= FC_MSG_MMS_SEND_SUPPORT;
                     }
                  }
                  else
                  {
                     if (FC_MSG_EMAIL_SUPPORT == (_mapDeviceCapabilities.u16SupportedMessageTypes & FC_MSG_EMAIL_SUPPORT))
                     {
                        _mapDeviceCapabilities.u16SupportedMapFeatures |= FC_MSG_EMAIL_SEND_SUPPORT;
                     }
                  }
               }
            }
            else
            {
               LoopBackMessageData loopBackMessageData;

               loopBackMessageData.u32FunctionId = FC_MSG_LB_FID_FOLDER_LIST;
               loopBackMessageData.u8DeviceHandle = _deviceHandle;
               tU16 stringLength = (tU16) strlen(folder);
               loopBackMessageData.vpData = new gchar[stringLength + 1];
               memset(loopBackMessageData.vpData, 0, static_cast<unsigned int>(stringLength + 1));
               strncpy((char*) loopBackMessageData.vpData, folder, stringLength);
               loopBackMessageData.sServiceName = _clientSessionPath;

               fc_messaging_tclService_Messaging::m_poMessagingService->vPostLoopBackMessage(loopBackMessageData);
            }
         }

         listSize++;
      }
   }
}

void EvolutionDbusClient::messagingMapListFoldersCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR4(("messagingMapListFoldersCallback() called"));

   (void)user_data; //Lint fix

   gboolean result;
   GError* error = NULLPTR;
   GVariant* folderList;
   tU16 listSize = 0;

   result = org_bluez_obex_message_access_call_list_folders_finish((OrgBluezObexMessageAccess*)source_object, &folderList, res, &error);

   EvolutionDbusClient* bluetoothDeviceObject = getBluetoothDeviceObject((OrgBluezObexMessageAccess*)source_object);

   if (error && (FALSE == result))
   {
      ETG_TRACE_USR4(("ListFolder: Error : %s", g_dbus_error_get_remote_error (error)));
      g_error_free(error);

      if (bluetoothDeviceObject)
      {
         if (FC_MSG_FOLDER_LIST_NOT_REQUESTED != bluetoothDeviceObject->_folderListRequestStatus)
         {
            // MapListFolder request not sent from HMI

            QMutexLocker lock(&_deviceListMutex);
            bluetoothDeviceObject->folderListRequestComplete();
            return;
         }
      }

      //MapListFolder request sent from HMI
      fc_messaging_tclService_Messaging::m_poMessagingService->vHandle_RequestFolderList_Error();
   }
   else
   {
      if (bluetoothDeviceObject)
      {
         //Update the MapDeviceCapabilities/FolderList to HMI
         bluetoothDeviceObject->handleMapListFolderData(folderList, listSize);

         bluetoothDeviceObject->handleMapListFolderComplete(listSize);
      }
   }
}

void EvolutionDbusClient::messagingMapSetFolderForListFoldersCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR4(("messagingMapSetFolderForListFoldersCallback() called"));

   (void)user_data; //Lint fix

   gboolean result;
   GError* error = NULLPTR;

   result = org_bluez_obex_message_access_call_set_folder_finish((OrgBluezObexMessageAccess*)source_object, res, &error);

   EvolutionDbusClient* bluetoothDeviceObject = getBluetoothDeviceObject((OrgBluezObexMessageAccess*)source_object);

   if (error && (FALSE == result))
   {
      ETG_TRACE_USR4(("SetFolder: Error : %s", g_dbus_error_get_remote_error (error)));
      g_error_free(error);

      if (bluetoothDeviceObject)
      {
         if (FC_MSG_FOLDER_LIST_NOT_REQUESTED != bluetoothDeviceObject->_folderListRequestStatus)
         {
            // MapListFolder request not sent from HMI

            QMutexLocker lock(&_deviceListMutex);
            bluetoothDeviceObject->folderListRequestComplete();
            return;
         }
      }

      // MapListFolder request sent from HMI
      fc_messaging_tclService_Messaging::m_poMessagingService->vHandle_RequestFolderList_Error();
   }
   else
   {
      //SetFolder is success, call ListFolders!!!
      GVariantBuilder* builder;
      GVariant* dict;

      guint16 offset = FOLDER_LIST_OFFSET; //default offset is '0'
      guint16 listMaxCount = MAXIMUM_LIST_COUNT; //default list max count is '1024', as per current requirement it is set to '100'

      builder = g_variant_builder_new(G_VARIANT_TYPE("a{sv}"));
      g_variant_builder_add (builder, "{sv}", "Offset", g_variant_new_uint16 (offset));
      g_variant_builder_add (builder, "{sv}", "MaxCount", g_variant_new_uint16(listMaxCount));
      dict = g_variant_builder_end (builder);

      org_bluez_obex_message_access_call_list_folders((OrgBluezObexMessageAccess*)source_object, dict,
            NULLPTR, messagingMapListFoldersCallback, NULLPTR);
   }
}

tBool EvolutionDbusClient::messagingMapListFolder(::std::string folderName)
{
   ETG_TRACE_USR4(("messagingMapListFolder() called for %s", folderName.c_str()));
   gchar* folder = const_cast<gchar*>(folderName.c_str()); //Lint fix

   org_bluez_obex_message_access_call_set_folder(_proxyMessageAccess,
         folder, NULLPTR, messagingMapSetFolderForListFoldersCallback, NULLPTR);

   return true;
}

void EvolutionDbusClient::messagingMapListFilterFields()
{
   ETG_TRACE_USR4(("messagingMapListFilterFields() called"));

   if(!_proxyMessageAccess)
   {
      ETG_TRACE_USR4(("MessageAccess Proxy NULL"));
      return;
   }

   GError* error = NULLPTR;
   gboolean result;
   gchar** fields;

   result = org_bluez_obex_message_access_call_list_filter_fields_sync(_proxyMessageAccess,
         &fields, NULLPTR, &error);

   if((FALSE == result) && error)
   {
      ETG_TRACE_USR4(("ListFilterFields: Error: %s", g_dbus_error_get_remote_error (error)));

      g_error_free(error);
      return;
   }

   if(fields)
   {
      for (tU8 index = 0; fields[index]; index++)
      {
         ETG_TRACE_USR4(("ListFilterFields: Field : %s", fields[index]));
         _listFilterFields.push_back(fields[index]);
      }
   }
}

void EvolutionDbusClient::sendListMessagesError()
{
   // Bugfix for GMMY16-5367 : Selected view on new message alert and nothing happened
   // and GMMY16-5368 : Message inbox was missing many of the newest messages.
   setWaitForMapListMsgCompSignal(FALSE);

   if (e8MapListMsgCallerCreateMessageList == _mapListMsgCaller)
   {
      fc_messaging_tclService_Messaging::m_poMessagingService->vHandle_CreateMessageList_Error();
   }
   else
   {
      MessagingMutex::messagingMutexUnlock();
   }
}

void EvolutionDbusClient::handleListMessagesComplete(tU16 listSize)
{
   ETG_TRACE_USR4(("handleListMessagesComplete() called"));

   setWaitForMapListMsgCompSignal(FALSE);

   if (_mapListMsgCaller == e8MapListMsgCallerCreateMessageList)
   {
      LoopBackMessageData loopBackMessageData;

      loopBackMessageData.u32FunctionId = FC_MSG_LB_FID_MESSAGE_CREATE_MESSAGE_LIST;
      loopBackMessageData.u8DeviceHandle = _deviceHandle;
      loopBackMessageData.u16Private = listSize;
      loopBackMessageData.vpData = NULLPTR;
      loopBackMessageData.sServiceName = _clientSessionPath;

      fc_messaging_tclService_Messaging::m_poMessagingService->vPostLoopBackMessage(loopBackMessageData);
   }
   else if(_mapListMsgCaller == e8MapListMsgCallerNewIncomingMessage)
   {
      //Two consecutive New Incoming Message Bug-Fix GMMY15-1259
      //MessagingMutex::messagingMutexUnlock();

      QMutexLocker newmsglock(&_newMsgHandleListMutex);

      if (false == _newMessageHandleList.isEmpty())
      {
         if(MessagingMutex::messagingMutexTrylock(NULLPTR) != 0)
         {
            _inboxSearchType=e8Forward;
            ETG_TRACE_FATAL(("mapListMsgCompSignalHandler :WARNING Unable To acquire mutex!!"));
            return;
         }

         if(_inboxSearchType == e8SearchByHandle)
         {
            MessageTypeDbusMessageHandleInfo msgHandleInfo;
            msgHandleInfo.m_oMessageListType.enType = most_fi_tcl_e8_MsgMessageListType::FI_EN_E8MSG_LIST_SMS_MMS;
            msgHandleInfo.m_sDbusMessageHandle = _newMessageHandleList.first();
            _msgListObject.vClear();
            _msgListObject.vSetHandle((tU8*) _newMessageHandleList.first().data(), static_cast<tU8>(_newMessageHandleList.first().size()));
            messagingMapGet(msgHandleInfo);
            _messageHandleList.append(_newMessageHandleList.first());
            _newMessageHandleList.removeOne(_newMessageHandleList.first());
         }
         else
         {
            tU16 listStartOffset = LIST_START_OFFSET;
            most_msgfi_tclMsgCreateMessageListMethodStart createMessageListMethodStart;
            createMessageListMethodStart.u8DeviceHandle = _deviceHandle;
            createMessageListMethodStart.e8MessageListType.enType = most_fi_tcl_e8_MsgMessageListType::FI_EN_E8MSG_LIST_SMS_MMS;
            createMessageListMethodStart.e8FolderType.enType = most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_INBOX;

            // Search 2 times the New message list count
            tU16 msgListCount = static_cast<tU16>(_newMessageHandleList.size() * 2);

            if(_inboxSearchType == e8Forward)
            {
               listStartOffset = LIST_START_OFFSET;
            }
            else if(_inboxSearchType == e8Reverse)
            {
               listStartOffset = static_cast<tU16>(_msgCountDeviceInbox - _newMessageHandleList.size());
            }

            messagingMapListMsg(createMessageListMethodStart,
                  msgListCount, e8MapListMsgCallerNewIncomingMessage, listStartOffset);
         }
      }
   }
   else if(_mapListMsgCaller == e8MapListMsgCallerSendMessage)
   {
      if(_mapSentMsgAlertSent == false)
      {
         SendMessage::poGetSendMessageInstance()->vRecentlySentMessageInfo(_deviceHandle, _clientSessionPath, NULLPTR);
      }
      else
      {
         //Sent message status alert already sent
         _mapSentMsgAlertSent = false;
      }

      //Unlock the acquired lock for getting send messages list
      //MessagingMutex::messagingMutexUnlock();
   }
}

void EvolutionDbusClient::handleListMessagesData(GVariant* msgList, tU16 listSize)
{
   ETG_TRACE_USR4(("handleListMessagesData() called"));

   if(!msgList || !_clientSessionPath)
   {
      return;
   }

   tBool msgHandleFoundInList = false;
   tBool sentMsgAddedInList = true;

   MsgListing_t* msgListing = new MsgListing_t[listSize];

   if (msgListing)
   {
      GVariantIter iter1, iter2;
      gchar* object;
      GVariant* dict;
      gchar* key;
      GVariant* value;
      gsize length = 0;
      tU8 index = 0;
      tU8 msgHandleStartPosition = static_cast<tU8>(strlen(_clientSessionPath)+8);
      g_variant_iter_init (&iter1, msgList);
      while (g_variant_iter_next (&iter1, "{o@a{sv}}", &object, &dict))
      {
         tU8 type =0, text = 0, status = 0, priority = 0, read = 0, sent = 0, protect = 0, notifyType = 0xFF;
         tU32 size = 0, attachmentSize = 0;

         if(!object || !dict)
         {
            ETG_TRACE_USR4(("Message Handle is NULL"));
            delete[] msgListing;

            return;
         }

         tU8 msgHandle[MESSAGE_HANDLE_LENGTH+1] = {0};
         tU8 i , j;
         tU8 msgPathLength = (tU8)strlen(object);
         tU8 msgHandleLength = static_cast<tU8>(msgPathLength-msgHandleStartPosition);

         // Bugfix for NCG3D-25456: MessageHandle should be in hexadecimal representation with upto 16 digits (as per MAP specification).
         // Hence, below condition will block the over writing of handle array in error scenario (handle with more than 16 digits).
         for (i=msgHandleStartPosition, j=0; ((i<msgPathLength) && (j<MESSAGE_HANDLE_LENGTH)); i++,j++)
         {
            msgHandle[j] = (tU8)object[i];
         }
         msgHandle[j]='\0';

         ETG_TRACE_USR4(("messagingMapListMessagesCallback::MsgHandleLength: %d, MsgHandle : %s", msgHandleLength, msgHandle));

         //------------------------------------------------------------------------------------------------------------------//

         QByteArray inboxMsgHandle((tString)msgHandle, msgHandleLength);

         if ((true == _messageHandleList.contains(inboxMsgHandle))
               && (_mapListMsgCaller == e8MapListMsgCallerNewIncomingMessage))
         {
            // Msg Handle is already available in inbox list, check for the next message
            continue;
         }
         else if ((true == _messageHandleList.contains(inboxMsgHandle))
               && (_mapListMsgCaller == e8MapListMsgCallerSendMessage))
         {
            sentMsgAddedInList = false;
         }
         else
         {
            if (_mapListMsgCaller == e8MapListMsgCallerNewIncomingMessage)
            {
               QMutexLocker newmsglock(&_newMsgHandleListMutex);

               msgHandleFoundInList = true;
               if (true == _newMessageHandleList.contains(inboxMsgHandle))
               {
                  ETG_TRACE_USR4(("mapListMsgDataSignalHandler Message handle exists in NewMessageHandleList"));
                  _messageHandleList.append(inboxMsgHandle);
                  _newMessageHandleList.removeOne(inboxMsgHandle);
               }
               else
               {
                  //Message Handle is already available in inbox list, check for the next message
                  continue;
               }
            }
            else if (_mapListMsgCaller == e8MapListMsgCallerSendMessage)
            {
               QMutexLocker newmsglock(&_newMsgHandleListMutex);

               if (true == _newMessageHandleList.contains(inboxMsgHandle))
               {
                  ETG_TRACE_USR4(("mapListMsgDataSignalHandler Message handle exists in NewMessageHandleList"));
                  _newMessageHandleList.removeOne(inboxMsgHandle);
               }
               _messageHandleList.append(inboxMsgHandle);
            }
            else
            {
               _messageHandleList.append(inboxMsgHandle);
            }
         }

         QMap<QByteArray, tU8>::iterator l_it;
         l_it = _msgNotifyMap.find(inboxMsgHandle);
         if (l_it != _msgNotifyMap.end())
         {
            notifyType = _msgNotifyMap.value(inboxMsgHandle);
            _msgNotifyMap.erase(l_it);
         }
         else
         {
            ETG_TRACE_USR4(("Msg Handle is not in the map"));
         }

         //------------------------------------------------------------------------------------------------------------------//

         msgListing[index].vSetHandle(msgHandle, msgHandleLength);

         g_variant_iter_init (&iter2, dict);
         while (g_variant_iter_next (&iter2, "{sv}", &key, &value))
         {
            ETG_TRACE_USR4(("Inside while loop 2"));
            if (strcmp (key, "Subject") == 0 && g_variant_is_of_type (value, G_VARIANT_TYPE_STRING))
            {
               tString subject  = const_cast<gchar*>(g_variant_get_string(value,&length));
               msgListing[index].vSetSubject(convertFromNCR(subject));
               ETG_TRACE_USR4(("messagingMapListMessagesCallback::Subject : %s", msgListing[index].sGetSubject().constData()));
            }
            else if (strcmp (key, "Timestamp") == 0 && g_variant_is_of_type (value, G_VARIANT_TYPE_STRING))
            {
               tString timestamp  = const_cast<gchar*>(g_variant_get_string(value,&length));
               msgListing[index].vSetDateTime(convertFromNCR(timestamp));
               ETG_TRACE_USR4(("messagingMapListMessagesCallback::Timestamp : %s", msgListing[index].sGetDateTime().constData()));
            }
            else if (strcmp (key, "Sender") == 0 && g_variant_is_of_type (value, G_VARIANT_TYPE_STRING))
            {
               tString sender  = const_cast<gchar*>(g_variant_get_string(value,&length));
               if (strnlen(sender, 255) > 0)
               {
                  msgListing[index].vSetSenderName(validatePhoneNumber(convertFromNCR(sender), FALSE));
                  ETG_TRACE_USR4(("messagingMapListMessagesCallback::Sender : %s", msgListing[index].sGetSenderName().constData()));
               }
            }
            else if (strcmp (key, "SenderAddress") == 0 && g_variant_is_of_type (value, G_VARIANT_TYPE_STRING))
            {
               tString senderAddress  = const_cast<gchar*>(g_variant_get_string(value,&length));
               if (strnlen(senderAddress, 255) > 0)
               {
                  msgListing[index].vSetSenderAddress(validatePhoneNumber(convertFromNCR(senderAddress), TRUE));
                  ETG_TRACE_USR4(("messagingMapListMessagesCallback::SenderAddress : %s", msgListing[index].sGetSenderAddress().constData()));

                  msgListing[index].vSetReplytoAddress(convertFromNCR(senderAddress));
                  ETG_TRACE_USR4(("messagingMapListMessagesCallback::ReplyTo : %s", msgListing[index].sGetReplytoAddress().constData()));
               }
            }
            else if (strcmp (key, "Recipient") == 0 && g_variant_is_of_type (value, G_VARIANT_TYPE_STRING))
            {
               tString recipient  = const_cast<gchar*>(g_variant_get_string(value,&length));
               if (strnlen(recipient, 255) > 0)
               {
                  msgListing[index].vSetRecipientName(validatePhoneNumber(convertFromNCR(recipient), FALSE));
                  ETG_TRACE_USR4(("messagingMapListMessagesCallback::Recipient : %s", msgListing[index].sGetRecipientName().constData()));
               }
            }
            else if (strcmp (key, "RecipientAddress") == 0 && g_variant_is_of_type (value, G_VARIANT_TYPE_STRING))
            {
               tString recipientAddress  = const_cast<gchar*>(g_variant_get_string(value,&length));
               if (strnlen(recipientAddress, 255) > 0)
               {
                  msgListing[index].vSetRecipientAddress(validatePhoneNumber(convertFromNCR(recipientAddress), TRUE));
                  ETG_TRACE_USR4(("messagingMapListMessagesCallback::RecipientAddress : %s", msgListing[index].sGetRecipientAddress().constData()));
               }
            }
            else if (strcmp (key, "Type") == 0 && g_variant_is_of_type (value, G_VARIANT_TYPE_STRING))
            {
               gchar* typestring  = const_cast<gchar*>(g_variant_get_string(value,&length));
               if(strcmp(typestring, "EMAIL") == 0)
               {
                  type = MSG_DBUS_MESSAGE_TYPE_EMAIL;
               }
               else if(strcmp(typestring, "SMS_GSM") == 0)
               {
                  type = MSG_DBUS_MESSAGE_TYPE_SMS_GSM;
               }
               else if(strcmp(typestring, "SMS_CDMA") == 0)
               {
                  type = MSG_DBUS_MESSAGE_TYPE_SMS_CDMA;
               }
               else if(strcmp(typestring, "MMS") == 0)
               {
                  type = MSG_DBUS_MESSAGE_TYPE_MMS;
               }

               ETG_TRACE_USR4(("messagingMapListMessagesCallback::Type : %s", typestring));
            }
            else if (strcmp (key, "Size") == 0 && g_variant_is_of_type (value, G_VARIANT_TYPE_UINT64))
            {
               size = (tU32)g_variant_get_uint64(value);
               ETG_TRACE_USR4(("messagingMapListMessagesCallback::Size : %d", size));
            }
            else if (strcmp (key, "Text") == 0 && g_variant_is_of_type (value, G_VARIANT_TYPE_BOOLEAN))
            {
               text = (g_variant_get_boolean(value) == TRUE)? 1 : 0;
               ETG_TRACE_USR4(("messagingMapListMessagesCallback::Text : %d", text));
            }
            else if (strcmp (key, "Status") == 0 && g_variant_is_of_type (value, G_VARIANT_TYPE_STRING))
            {
               gchar* statusstring = const_cast<gchar*>(g_variant_get_string(value,&length));

               if(strcmp(statusstring, "complete") == 0)
               {
                  status = MSG_STATUS_RECEIVED_COMPLETE;
               }
               else if(strcmp(statusstring, "fractioned") == 0)
               {
                  status = MSG_STATUS_RECEIVED_FRACTIONED;
               }
               else
               {
                  status = MSG_STATUS_RECEIVED_NOTIFICATION;
               }
               ETG_TRACE_USR4(("messagingMapListMessagesCallback::Status : %d", status));
            }
            else if (strcmp (key, "AttachmentSize") == 0 && g_variant_is_of_type (value, G_VARIANT_TYPE_UINT64))
            {
               attachmentSize = (tU32)g_variant_get_uint64(value);
               ETG_TRACE_USR4(("messagingMapListMessagesCallback::AttachmentSize : %d", attachmentSize));
            }
            else if (strcmp (key, "Priority") == 0 && g_variant_is_of_type (value, G_VARIANT_TYPE_BOOLEAN))
            {
               priority = (g_variant_get_boolean(value) == TRUE)? 1 : 0;
               ETG_TRACE_USR4(("messagingMapListMessagesCallback::Priority : %d", priority));
            }
            else if (strcmp (key, "Read") == 0 && g_variant_is_of_type (value, G_VARIANT_TYPE_BOOLEAN))
            {
               read = (g_variant_get_boolean(value) == TRUE)? 1 : 0;
               ETG_TRACE_USR4(("messagingMapListMessagesCallback::Read : %d", read));
            }
            else if (strcmp (key, "Sent") == 0 && g_variant_is_of_type (value, G_VARIANT_TYPE_BOOLEAN))
            {
               sent = (g_variant_get_boolean(value) == TRUE)? MSG_STATUS_DELIVERY_SUCCESS : MSG_STATUS_DELIVERY_FAILURE;
               ETG_TRACE_USR4(("messagingMapListMessagesCallback::Sent : %d", sent));
            }
            else if (strcmp (key, "Protected") == 0 && g_variant_is_of_type (value, G_VARIANT_TYPE_BOOLEAN))
            {
               protect = (g_variant_get_boolean(value) == TRUE)? 1 : 0;
               ETG_TRACE_USR4(("messagingMapListMessagesCallback::Protected : %d", protect));
            }
         }

         ETG_TRACE_USR4(("messagingMapListMessagesCallback::NotificationType : %d", notifyType));

         msgListing[index].vSetMsgListDataFlags(type, size, text, status, attachmentSize, priority, !(read), sent, protect, notifyType);

         index++;
      }
   }

   LoopBackMessageData loopBackMessageData;
   if (_mapListMsgCaller == e8MapListMsgCallerCreateMessageList)
   {
      loopBackMessageData.u32FunctionId = FC_MSG_LB_FID_MAP_LIST_MSG_DATA;
      loopBackMessageData.u8DeviceHandle = _deviceHandle;
      loopBackMessageData.u16Private = listSize;
      loopBackMessageData.vpData = msgListing;
      loopBackMessageData.sServiceName = _clientSessionPath;

      fc_messaging_tclService_Messaging::m_poMessagingService->vPostLoopBackMessage(loopBackMessageData);
   }
   else if (_mapListMsgCaller == e8MapListMsgCallerSendMessage)
   {
      MessagingMutex::messagingMutexUnlock();

      //CMG3G-1494 - Changes for Email support and sent items folder
      if (true == sentMsgAddedInList)
      {
         loopBackMessageData.u32FunctionId = FC_MSG_LB_FID_RECENTLY_SENT_MESSAGE_INFO;
         loopBackMessageData.u8DeviceHandle = _deviceHandle;
         loopBackMessageData.u16Private = listSize;
         loopBackMessageData.vpData = msgListing;
         loopBackMessageData.sServiceName = _clientSessionPath;
         fc_messaging_tclService_Messaging::m_poMessagingService->vPostLoopBackMessage(loopBackMessageData);

         _mapSentMsgAlertSent = true;
      }
      else
      {
         ETG_TRACE_USR4(("messagingMapListMessagesCallback:: sentMsgAddedInList False"));
         delete[] msgListing;
      }
   }
   else if (_mapListMsgCaller == e8MapListMsgCallerNewIncomingMessage)
   {
      MessagingMutex::messagingMutexUnlock();

      if (false == msgHandleFoundInList)
      {
         if (_inboxSearchType == e8Forward)
         {
            _inboxSearchType = e8Reverse;
         }
         else if (_inboxSearchType == e8Reverse)
         {
            _inboxSearchType = e8SearchByHandle;
         }
      }

      loopBackMessageData.u32FunctionId = FC_MSG_LB_FID_NEW_INCOMING_MESSAGE;
      loopBackMessageData.u8DeviceHandle = _deviceHandle;
      loopBackMessageData.u16Private = listSize;
      loopBackMessageData.vpData = msgListing;
      loopBackMessageData.sServiceName = _clientSessionPath;

      fc_messaging_tclService_Messaging::m_poMessagingService->vPostLoopBackMessage(loopBackMessageData);
   }
   else
   {
      ETG_TRACE_USR4(("messagingMapListMessagesCallback:: Invalid ListMsgCaller"));
      delete[] msgListing;
   }
}

void EvolutionDbusClient::messagingMapListMessagesCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR4(("messagingMapListMessagesCallback() called"));

   (void)user_data; //Lint fix

   GVariant* msgList = NULLPTR;
   GError* error = NULLPTR;
   gboolean result;

   EvolutionDbusClient* bluetoothDeviceObject =  getBluetoothDeviceObject((OrgBluezObexMessageAccess*)source_object);
   if (!bluetoothDeviceObject)
   {
      return;
   }
   
   result = org_bluez_obex_message_access_call_list_messages_finish((OrgBluezObexMessageAccess*)source_object, &msgList, res, &error);

   if(error && (FALSE == result))
   {
      ETG_TRACE_USR4(("ListMessages Error: %s", g_dbus_error_get_remote_error (error)));
      bluetoothDeviceObject->sendListMessagesError();

      g_error_free(error);
      return;
   }

   tU16 listSize = 0;
   if(!msgList)
   {
      return;
   }
   else
   {
      //Find the number of messages available in the folder.
      GVariantIter iter;
      gchar* tmpobject;
      GVariant* tmpdict;

      g_variant_iter_init (&iter, msgList);
      while (g_variant_iter_next (&iter, "{o@a{sv}}", &tmpobject, &tmpdict))
      {
         listSize++;
      }
   }

   ETG_TRACE_USR4(("messagingMapListMessagesCallback() size : %d", listSize));


   //msgList format a{oa{av}}
   //CRQ256355 : Checking the outbox listSize in case there is no Status Reporting from the Stack
   if(bluetoothDeviceObject->_mapListMsgCaller == e8MapListMsgCallerOutboxCheck)
   {
      trFC_Msg_Notification *prFC_Msg_Notification = new trFC_Msg_Notification;

      if(prFC_Msg_Notification)
      {
         prFC_Msg_Notification->u8MessgeType = most_fi_tcl_e8_MsgMessageType::FI_EN_E8MSG_TYPE_SMS;
         if(0 == listSize)
         {
            prFC_Msg_Notification->u8MessgeNotificationType = (tU8)most_fi_tcl_e8_MsgNotificationType::FI_EN_E8NOTIFY_MSG_SEND_SUCCESS;
         }
         else
         {
            prFC_Msg_Notification->u8MessgeNotificationType = (tU8)most_fi_tcl_e8_MsgNotificationType::FI_EN_E8NOTIFY_MSG_SEND_FAILURE;
         }

         ETG_TRACE_USR1(("Msg Notify: %d", prFC_Msg_Notification->u8MessgeNotificationType));
         prFC_Msg_Notification->rMapNoitfyData.u8FolderType = most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_OUTBOX;

         prFC_Msg_Notification->u8DeviceHandle = bluetoothDeviceObject->_deviceHandle;
         tU8 msgHandle[MESSAGE_HANDLE_LENGTH+1] = {0};
         _BP_TRY_BEGIN
         {
            prFC_Msg_Notification->rMapNoitfyData.pacMessageHandle = new QByteArray((tString)msgHandle,MESSAGE_HANDLE_LENGTH);
         }
         _BP_CATCH_ALL
         {
            delete prFC_Msg_Notification;
            ETG_TRACE_ERR(("MessageListCallback() Caught Exception"));
            return;
         }
         _BP_CATCH_END
         LoopBackMessageData loopBackMessageData;
         loopBackMessageData.vpData = (tVoid*) (prFC_Msg_Notification);
         loopBackMessageData.u32FunctionId = FC_MSG_LB_FID_MSG_NOTIFICATION_STATUS;
         loopBackMessageData.u8DeviceHandle = bluetoothDeviceObject->_deviceHandle;
         loopBackMessageData.sServiceName = bluetoothDeviceObject->_clientSessionPath;
         fc_messaging_tclService_Messaging::m_poMessagingService->vPostLoopBackMessage(loopBackMessageData);
      }
   }
   else
   {
      bluetoothDeviceObject->_msgCountDeviceInbox = listSize;
      //Bugfix for CMG3GB-3147 : Two messages are showing under "Text message" for single incoming message for iphone after fresh pairing.
      //After connecting iPhone, ListMessages has been invoked to download the messages and in callback,
      //the member variable _inboxSearchType is changed to "Search_By_Handle" due to empty message list.
      if(0 != listSize)
      {
         bluetoothDeviceObject->handleListMessagesData(msgList, listSize);
      }

      bluetoothDeviceObject->handleListMessagesComplete(listSize);
   }
}

tBool EvolutionDbusClient::messagingMapListMsg(most_msgfi_tclMsgCreateMessageListMethodStart createMessageListMethodStart,
      tU16 listMaxCount, e8MapListMsgCaller mapListMsgCaller, tU16 listStartOffset)
{
   ETG_TRACE_USR4(("messagingMapListMsg() called"));

   _waitForMapListMsgCompSignal = true;
   _mapListMsgCaller = mapListMsgCaller;

   if(!_proxyMessageAccess)
   {
      ETG_TRACE_USR4(("MessageAccess Proxy NULL"));

      sendListMessagesError();
      return false;
   }

   gboolean result;
   GError* error = NULLPTR;
   ::std::string folderName;

   ETG_TRACE_USR4((" Folder Type:: %u", ETG_ENUM(TR_FOLDER_TYPE, createMessageListMethodStart.e8FolderType.enType)));

   switch (createMessageListMethodStart.e8FolderType.enType)
   {
      case most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_INBOX:
         folderName = MSG_DBUS_EVO_FOLDER_TYPE_INBOX;
         break;
      case most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_OUTBOX:
         folderName = MSG_DBUS_EVO_FOLDER_TYPE_OUTBOX;
         break;
      case most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_DELETED:
         folderName = MSG_DBUS_EVO_FOLDER_TYPE_DELETED;
         break;
      case most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_DRAFTS:
         folderName = MSG_DBUS_EVO_FOLDER_TYPE_DRAFT;
         break;
      case most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_SENT:
         folderName = MSG_DBUS_EVO_FOLDER_TYPE_SENT;
         break;
      case most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_USERDEFINED:
         folderName = MSG_DBUS_EVO_FOLDER_TYPE_USERDEFINED;
         break;
      default:
         folderName = MSG_DBUS_EVO_FOLDER_TYPE_INBOX; // take inbox as default
         break;
   }

   result = org_bluez_obex_message_access_call_set_folder_sync(_proxyMessageAccess,
         folderName.c_str(), NULLPTR, &error); //Can we try sync request here!!!

   if(error && (TRUE != result))
   {
      ETG_TRACE_USR4(("SetFolder Error: %s", g_dbus_error_get_remote_error (error)));
      g_error_free(error);

      sendListMessagesError();
      return false;
   }

   GVariantBuilder* builder;
   GVariant* dict;
   //const gchar* fields[] = { "subject", "timestamp", "sender", "sender-address", "recipient", "recipient-address", "type", "size", "status", "text", "attachment", "priority", "read", "sent", "protected", "replyto", NULL };
   guchar subjectLen = MAXIMUM_SUBJECT_LENGTH; //default subject length is 256

   builder = g_variant_builder_new(G_VARIANT_TYPE("a{sv}"));
   g_variant_builder_add(builder, "{sv}", "Offset", g_variant_new_uint16(listStartOffset));
   g_variant_builder_add(builder, "{sv}", "MaxCount", g_variant_new_uint16(listMaxCount));
   g_variant_builder_add(builder, "{sv}", "SubjectLength", g_variant_new_byte(subjectLen));

   if(!_listFilterFields.empty())
   {
      const gchar** fields = new const gchar* [_listFilterFields.size()+1]; //NULL should be added at last for g_variant_new_strv.
      if(fields)
      {
         tU8 index;
         for (index = 0; index < _listFilterFields.size(); index++)
         {
            fields[index] = _listFilterFields.at(index).c_str();
         }
         fields[index] = NULLPTR;

         g_variant_builder_add(builder, "{sv}", "Fields", g_variant_new_strv(fields, -1));

         delete [] fields;
      }
   }

   //message types should be filled here
   if (createMessageListMethodStart.e8MessageListType.enType == most_fi_tcl_e8_MsgMessageListType::FI_EN_E8MSG_LIST_SMS_MMS)
   {
      const gchar* messageTypes[] = { "email", NULLPTR };
      g_variant_builder_add(builder, "{sv}", "Types", g_variant_new_strv(messageTypes, -1));
   }
   else
   {
      const gchar* messageTypes[] = { "sms", "mms", NULLPTR };
      g_variant_builder_add(builder, "{sv}", "Types", g_variant_new_strv(messageTypes, -1));
   }

   //No need to set below values
   //g_variant_builder_add(builder, "{sv}", "PeriodBegin", g_variant_new_string(""));
   //g_variant_builder_add(builder, "{sv}", "PeriodEnd", g_variant_new_string(""));
   //g_variant_builder_add(builder, "{sv}", "Read", g_variant_new_boolean(FALSE));
   //g_variant_builder_add(builder, "{sv}", "Recipient", g_variant_new_string(""));
   //g_variant_builder_add(builder, "{sv}", "Sender", g_variant_new_string(""));
   //g_variant_builder_add(builder, "{sv}", "Priority", g_variant_new_boolean(FALSE));
   dict = g_variant_builder_end (builder);

   org_bluez_obex_message_access_call_list_messages(_proxyMessageAccess, folderName.c_str(), dict, NULLPTR, messagingMapListMessagesCallback, NULLPTR);

   return true;
}

void EvolutionDbusClient::sendPushMessageStatus(tU8 status)
{
   ifstream message(MSG_DBUS_EVO_PUSHMESSAGE_PATH);
   string data;

   //Print the uploaded message data for verification
   if(message.is_open())
   {
      while(message.good())
      {
         getline(message,data);

         ETG_TRACE_USR4(("PushMessage Data: %s", data.c_str()));
      }
      message.close();
   }
   else
   {
      ETG_TRACE_ERR(("File open failed"));
   }

   LoopBackMessageData loopBackMessageData;
   loopBackMessageData.ucStatus = status;
   loopBackMessageData.u8DeviceHandle = _deviceHandle;
   loopBackMessageData.sServiceName = _clientSessionPath;
   loopBackMessageData.u32FunctionId = FC_MSG_LB_FID_PUSH_MESSAGE_STATUS;

   fc_messaging_tclService_Messaging::m_poMessagingService->vPostLoopBackMessage(loopBackMessageData);

   //Remove the message file for next time upload.
   if( remove( MSG_DBUS_EVO_PUSHMESSAGE_PATH ) != 0 )
   {
      ETG_TRACE_ERR(("File remove failed"));
   }
   else
   {
      ETG_TRACE_USR4(("File remove success"));
   }
}

tBool EvolutionDbusClient::constructbMessage(SendMessage* sendMessage)
{
   ETG_TRACE_USR4(("constructbMessage() called"));
   tBool retVal = false;

   if(!sendMessage)
   {
      return retVal;
   }

   tBool isSMSorMMS = false;

   ETG_TRACE_USR4(("MessageType: %d", _filterMessageType ));

   ofstream message (MSG_DBUS_EVO_PUSHMESSAGE_PATH);
   if (message.is_open())
   {
      message << "BEGIN:BMSG" << BMESSAGE_NEWLINE;
      message << "VERSION:1.0" << BMESSAGE_NEWLINE;
      message << "STATUS:UNREAD" << BMESSAGE_NEWLINE; //Same as Fusion, ReadStatus 'true' indicates that 'unread' in Fusion.

      if (sendMessage->m_oCreateMessageMethodStart.e8MessageType.enType == most_fi_tcl_e8_MsgMessageType::FI_EN_E8MSG_TYPE_EMAIL)
      {
         message << "TYPE:EMAIL" << BMESSAGE_NEWLINE;
      }
      else
      {
         if (sendMessage->m_oCreateMessageMethodStart.e8MessageType.enType == most_fi_tcl_e8_MsgMessageType::FI_EN_E8MSG_TYPE_MMS)
         {
            message << "TYPE:MMS" << BMESSAGE_NEWLINE;
         }
         else
         {
            if(_filterMessageType & DBUS_MSG_TYPE_SMS_GSM)
            {
               message << "TYPE:SMS_GSM" << BMESSAGE_NEWLINE;
            }
            else if(_filterMessageType & DBUS_MSG_TYPE_SMS_CDMA)
            {
               message << "TYPE:SMS_CDMA" << BMESSAGE_NEWLINE;
            }
         }
         isSMSorMMS = true;
      }

      message << "FOLDER:" << BMESSAGE_NEWLINE; //As per MAP Spec V12, this is not required to fill. As this is same as folder which is given as direct argument.

      //Sender information
      message << "BEGIN:VCARD" << BMESSAGE_NEWLINE;
      message << BMESSAGE_VCARD_VERSION << BMESSAGE_NEWLINE;
      message << "N:" << sendMessage->m_oSenderInfo.m_sLastName.constData() << BMESSAGE_NEWLINE;
      message << "FN:" << sendMessage->m_oSenderInfo.m_sFirstName.constData() << BMESSAGE_NEWLINE;
      if(true == isSMSorMMS)
      {
         message << "TEL:" << sendMessage->m_oSenderInfo.m_sPhoneNumber.constData() << BMESSAGE_NEWLINE;
      }
      else
      {
         message << "EMAIL:" << sendMessage->m_oSenderInfo.m_sEmailAddress.constData() << BMESSAGE_NEWLINE;
      }
      message << "END:VCARD" << BMESSAGE_NEWLINE;

      message << "BEGIN:BENV" << BMESSAGE_NEWLINE;

      //Recipient information
      tU8 index;
      for (index = 0; index < sendMessage->m_vRecipientInfo.size(); index++)
      {
         message << "BEGIN:VCARD" << BMESSAGE_NEWLINE;
         message << BMESSAGE_VCARD_VERSION << BMESSAGE_NEWLINE;
         message << "N:" << sendMessage->m_vRecipientInfo[index].m_sLastName.constData() << BMESSAGE_NEWLINE;
         message << "FN:" << sendMessage->m_vRecipientInfo[index].m_sFirstName.constData() << BMESSAGE_NEWLINE;
         if(true == isSMSorMMS)
         {
            message << "TEL:" << sendMessage->m_vRecipientInfo[index].m_sPhoneNumber.constData() << BMESSAGE_NEWLINE;
         }
         else
         {
            message << "EMAIL:" << sendMessage->m_vRecipientInfo[index].m_sEmailAddress.constData() << BMESSAGE_NEWLINE;
         }
         message << "END:VCARD" << BMESSAGE_NEWLINE;
      }

      //Body of the message
      message << "BEGIN:BBODY" << BMESSAGE_NEWLINE;
      message << "CHARSET:UTF-8" << BMESSAGE_NEWLINE;
      message << "LANGUAGE:UNKNOWN" << BMESSAGE_NEWLINE;

      ::std::string messageContent(sendMessage->m_sSentMessageData.constData(), static_cast<unsigned int>(sendMessage->m_sSentMessageData.length()));

      //trim trailing white spaces (TODO: Remove isspace function, as it gives undefined behavior if it gets Non-ASCII characters)
      messageContent.erase(std::find_if(messageContent.rbegin(), messageContent.rend(),
                  std::not1(std::ptr_fun<int, int>(std::isspace))).base(), messageContent.end());

      tU32 textLength = static_cast<unsigned int>(messageContent.size());
      ETG_TRACE_USR4(("MessageBodyContent: %50s Length: %d", messageContent.c_str(), textLength));

      if(TRUE == _signatureText.empty())
      {
         _signatureText = fc_messaging_tclService_Messaging::m_poMessagingService->getSignatureText();
      }

      tU32 signatureLength = static_cast<unsigned int>(_signatureText.size());
      ETG_TRACE_USR4(("MessageBodySignature: %50s Length: %d", _signatureText.c_str(), signatureLength));

      if((SEND_MESSAGE_CONTENT_MAX_LENGTH < (textLength + signatureLength)) || (0 == signatureLength))
      {
         message << "LENGTH:" << (textLength + BMESSAGE_MSG_BODY_LENGTH) << BMESSAGE_NEWLINE;
         message << "BEGIN:MSG" << BMESSAGE_NEWLINE;
         message << messageContent.c_str() << BMESSAGE_NEWLINE;
      }
      else
      {
         message << "LENGTH:" << (textLength + signatureLength + BMESSAGE_MSG_BODY_LENGTH) << BMESSAGE_NEWLINE;
         message << "BEGIN:MSG" << BMESSAGE_NEWLINE;
         message << messageContent.c_str();
         message << _signatureText.c_str() << BMESSAGE_NEWLINE;
      }
      message << "END:MSG" << BMESSAGE_NEWLINE;
      message << "END:BBODY" << BMESSAGE_NEWLINE;

      message << "END:BENV" << BMESSAGE_NEWLINE;
      message << "END:BMSG" << BMESSAGE_NEWLINE;

      message.close();

      retVal = true;
   }
   else
   {
      ETG_TRACE_ERR(("File open failed"));
      //sendPushMessageStatus(FAILURE); //Not required if returned false
   }

   return retVal;
}

void EvolutionDbusClient::messagingMapPushMessageCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR4(("messagingMapPushMessageCallback() called"));

   (void)user_data; //Lint fix

   gboolean result;
   GError* error = NULLPTR;
   gchar* transferPath;

   result = org_bluez_obex_message_access_call_push_message_finish((OrgBluezObexMessageAccess*)source_object, &transferPath, res, &error);

   QMutexLocker lock(&_deviceListMutex);
   EvolutionDbusClient* client = getBluetoothDeviceObject((OrgBluezObexMessageAccess*)source_object);

   if (error && (FALSE == result))
   {
      ETG_TRACE_USR4(("PushMessage Error: %s", g_dbus_error_get_remote_error (error)));
      g_error_free(error);

      if(client)
      {
         client->sendPushMessageStatus(FAILURE);
      }
   }
   else
   {
      ETG_TRACE_USR4(("PushMessage Success, Wait for transfer complete!!!"));

      if(client)
      {
         std::string str = transferPath;
         client->_currentTransfer.insert(::std::pair<std::string, msgCallback>(str, PUSH_MESSAGE_CB));
         client->createDBusPropertiesProxyForTransferPropertiesChanged(str);
      }
   }
}

tBool EvolutionDbusClient::messagingMapPushMessage(SendMessage* sendMessage)
{
   ETG_TRACE_USR4(("messagingMapPushMessage() called"));

   if(!sendMessage || !_proxyMessageAccess)
   {
      ETG_TRACE_USR4(("NULL Pointers"));
      return false;
   }

   gchar* folderName = const_cast<gchar*> (MSG_DBUS_EVO_FOLDER_TYPE_OUTBOX); //Default
   GError* error = NULLPTR;
   gboolean result;

   if (sendMessage->m_oSendMessageMethodStart.e8FolderType.enType == most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_OUTBOX)
   {
      folderName = const_cast<gchar*> (MSG_DBUS_EVO_FOLDER_TYPE_OUTBOX);
   }
   else if (sendMessage->m_oSendMessageMethodStart.e8FolderType.enType == most_fi_tcl_e8_MsgFolderType::FI_EN_E8MSG_FOLDER_DRAFTS)
   {
      folderName = const_cast<gchar*> (MSG_DBUS_EVO_FOLDER_TYPE_DRAFT);
   }

   ETG_TRACE_USR4(("SetFolder Folder: %s", folderName));

   result = org_bluez_obex_message_access_call_set_folder_sync(_proxyMessageAccess,
         folderName, NULLPTR, &error); //Can we try sync request here!!!

   if(error && (TRUE != result))
   {
      ETG_TRACE_USR4(("SetFolder Error: %s", g_dbus_error_get_remote_error (error)));
      g_error_free(error);

      //sendPushMessageStatus(FAILURE); //Not required here, if false returned.
      return false;
   }

   GVariantBuilder* builder;
   GVariant* dict;
   gboolean transparent = false; //equivalent of save sent setting
   gboolean retry = true;

   //Construct a message (in bMessgae format) with the available info
   if(true != constructbMessage(sendMessage))
   {
      return false;
   }

   MessagingSettings* messagingSettings = MessagingSettings::poGetMessagingSettingsInstance();

   if (sendMessage->m_oCreateMessageMethodStart.e8MessageType.enType == most_fi_tcl_e8_MsgMessageType::FI_EN_E8MSG_TYPE_EMAIL)
   {
      if(messagingSettings)
      {
         transparent = messagingSettings->bGetSaveSentEmailSetting();
      }
   }
   else
   {
      if(messagingSettings)
      {
         transparent = messagingSettings->bGetSaveSentTextSetting();
      }
   }

   builder = g_variant_builder_new(G_VARIANT_TYPE("a{sv}"));
   g_variant_builder_add(builder, "{sv}", "Transparent", g_variant_new_boolean(!transparent));
   g_variant_builder_add(builder, "{sv}", "Retry", g_variant_new_boolean(!retry));
   g_variant_builder_add(builder, "{sv}", "Charset", g_variant_new_string("utf8"));
   dict = g_variant_builder_end (builder);

   org_bluez_obex_message_access_call_push_message(_proxyMessageAccess,
         MSG_DBUS_EVO_PUSHMESSAGE_PATH, folderName, dict, NULLPTR, messagingMapPushMessageCallback, NULLPTR);

   return true;
}

void EvolutionDbusClient::messagingMapSetMsgStatusCallback(msgCallback callback, tU8 status)
{
   ETG_TRACE_USR4(("messagingMapSetMsgStatusCallback() %d with status %d", callback, status));

   LoopBackMessageData loopBackMessageData;
   loopBackMessageData.ucStatus = status;
   loopBackMessageData.u16Private = callback;
   loopBackMessageData.u8DeviceHandle = _deviceHandle;
   loopBackMessageData.sServiceName = _clientSessionPath;
   loopBackMessageData.u32FunctionId = FC_MSG_LB_FID_SET_MSG_STATUS;

   fc_messaging_tclService_Messaging::m_poMessagingService->vPostLoopBackMessage(loopBackMessageData);
}

tBool EvolutionDbusClient::messagingMapSetMsgStatus(::std::string messageHandle, most_fi_tcl_e8_MsgMessageListType messageListType, tU8 operation)
{
   ETG_TRACE_USR4(("messagingMapSetMsgStatus() called"));
   ETG_TRACE_USR4(("Message Handle: %16s, Operation: %d", messageHandle.c_str(), operation));

   (void)messageListType; //Lint fix

   if(!_clientSessionPath)
   {
      ETG_TRACE_USR4(("MAP Session not connected"));
      return false;
   }

   GError* error = NULLPTR;
   gchar* message = const_cast<gchar*>(MESSAGE_STRING);

   //Message path format: /org/bluez/obex/client/session(count)/message{handle}
   gchar* msgPath = g_strconcat(_clientSessionPath, message, const_cast<gchar*>(messageHandle.c_str()), NULLPTR);

   ETG_TRACE_USR4(("Message Object Path: %s", msgPath));

   //Create proxy for the message
   _proxySetMessage = org_bluez_obex_message_proxy_new_sync(_evoMsgBus,
         G_DBUS_PROXY_FLAGS_NONE, BASE_EVO_SERVICE_NAME, msgPath, NULLPTR, &error);

   g_free(msgPath);

   if (!_proxySetMessage && error)
   {
      ETG_TRACE_ERR(("messagingMapSetMsgStatus : failed: %s", g_dbus_error_get_remote_error(error)));
      g_error_free(error);
      return false;
   }

   QByteArray inboxMsgHandle(messageHandle.c_str(), (tU8)strlen(messageHandle.c_str()));

   if(_messageHandleList.contains(inboxMsgHandle))
   {
      if (operation == MSG_STATUS_DELETED)
      {
         //Clear the message handle of deleted message in the list
         _isDeleteMessage = true;
         _messageHandleList.removeOne(inboxMsgHandle);

         //Set delete flag to true (equivalent of yes in Fusion stack)
         org_bluez_obex_message_set_deleted(_proxySetMessage, true);
      }
      else if(operation == MSG_STATUS_READ)
      {
         org_bluez_obex_message_set_read(_proxySetMessage, true);
      }
      else if(operation == MSG_STATUS_UNREAD)
      {
         org_bluez_obex_message_set_read(_proxySetMessage, false);
      }
   }

   return true;
}

tBool EvolutionDbusClient::isMultipartMessage(::std::string subject, ::std::string& finalboundary)
{
   ::std::istringstream data(subject);
   ::std::string line;
   ::std::string boundary;
   tBool bRet = false;

   while (std::getline(data, line))
   {
      if(std::string::npos != line.find(CONTENT_TYPE))
      {
         if(std::string::npos != line.find(CONTENT_TYPE_MULTIPART))
         {
            bRet = true;

            //search boundary
            std::size_t found = line.find(CONTENT_NAME_BOUNDARY);
            if(std::string::npos != found)
            {
               line.erase(0, (found+BOUNDARY_TEXT_LENGTH));

               //Remove the " and \r characters in the boundary string and append --
               char removeChars[] = "\"\r";

               boundary.reserve(line.size()); // optional, avoids buffer reallocations in the loop
               for(size_t i = 0; i < line.size(); ++i)
               {
                  if((line[i] != removeChars[0]) && (line[i] != removeChars[1]))
                  {
                     boundary += line[i];
                  }
               }

               //Multipart message has boundary between each parts (E.g. image/jpeg, text/plain and etc)
               //No need to check final boundary as text/plain may not be available in the end of the message.
               finalboundary.append("--");
               finalboundary.append(boundary);
            }

            break;
         }
      }
   }

   return bRet;
}

void EvolutionDbusClient::parseMessageBodyContent(::std::string& subject)
{
   ::std::istringstream data(subject);
   ::std::string line;
   ::std::string finalboundary;
   ::std::string tmp;
   tBool isText = false;
   tBool isMultipartMsg = isMultipartMessage(subject, finalboundary);

   if (isMultipartMsg)
   {
      tBool isTextPlain = false;
      while (std::getline(data, line))
      {
         //search for text/plain part
         if(std::string::npos != line.find(CONTENT_TYPE_TEXT_PLAIN))
         {
            isTextPlain = true;
         }

         if(isTextPlain)
         {
            //search for an empty line before original text
            std::size_t found = line.find(finalboundary);
            if(isText && (std::string::npos == found)) //avoid boundary line in the message content
            {
               tmp += line;
               tmp += '\n';
            }

            if((0 == line.find(STRING_CR)) && (!isText))
            {
               //empty line found(\r - starts first), message content starts after this line
               isText = true;
            }
            else if((std::string::npos != found) && isText)
            {
               //message content parsing completed here as boundary line reached.
               break;
            }
         }
      }
   }
   else
   {
      while (std::getline(data, line))
      {
         //search for an empty line before original text
         if(isText && (0 != line.find(STRING_CR)))
         {
            tmp += line;
            tmp += '\n';
         }

         if((0 == line.find(STRING_CR)) && (!isText))
         {
            isText = true;
         }
      }
   }

   subject = tmp;
}

void EvolutionDbusClient::sendGetMessageError()
{
   ETG_TRACE_USR4(("sendGetMessageError() called"));

   if (_inboxSearchType == e8SearchByHandle)
   {
      _inboxSearchType = e8Forward;
      MessagingMutex::messagingMutexUnlock();
   }
   else
   {
      LoopBackMessageData loopBackMessageData;

      loopBackMessageData.u32FunctionId = FC_MSG_LB_FID_GET_MSG_ERROR;
      loopBackMessageData.u8DeviceHandle = _deviceHandle;
      loopBackMessageData.sServiceName = _clientSessionPath;

      fc_messaging_tclService_Messaging::m_poMessagingService->vPostLoopBackMessage(loopBackMessageData);
   }
}

void EvolutionDbusClient::sendGetMessageSuccess()
{
   ETG_TRACE_USR4(("sendGetMessageSuccess() called"));

   //Parse the message data
   ifstream message(MSG_DBUS_EVO_GETMESSAGE_PATH);
   string data;
   string subject;
   string folder;
   tBool msgFound = false;
   tBool isVCard = false;
   tBool isSenderVCard = true;

   guchar readStatus = 0;
   guchar type = NOTIFY_MSG_IGNORE;
   tString folderName = NULLPTR;
   tU8 attachmentNum = 0;
   tU8 receiverNum = 0;
   guint dataFlag = DATA_FOR_TEXT;
   //tU32 subjectLength = 0;

   SenderPtr_t sender = g_new0(Sender_t, 1);
   sender->pucFN = NULLPTR;
   sender->pucN = NULLPTR;
   sender->pucTEL = NULLPTR;
   sender->pucEMAIL = NULLPTR;

   if(message.is_open())
   {
      while(message.good())
      {
         getline(message,data);

         ETG_TRACE_USR4(("GetMessage Data: %s", data.c_str()));

         if (data.find("STATUS:") != std::string::npos)
         {
            if(data.find("STATUS:READ") != std::string::npos)
            {
               readStatus = 1;
            }
            else
            {
               readStatus = 0;
            }
         }
         else if (data.find("TYPE:") != std::string::npos)
         {
            getNotificationMsgType(data, type);
            ETG_TRACE_USR4(("msgType : %u", ETG_ENUM(TR_MSGTYPE, type)));
         }
         else if (data.find("FOLDER:telecom/msg/") != std::string::npos)
         {
            folder = data;
         }
         else if (data.find("BEGIN:VCARD") != std::string::npos)
         {
            isVCard = true;
            if(true == isSenderVCard)
            {
               ETG_TRACE_USR4(("Sender VCard"));
            }
            else
            {
               ETG_TRACE_USR4(("Receiver VCard"));
            }
         }
         else if (data.find("END:VCARD") != std::string::npos)
         {
            isVCard = false;
         }
         else if (data.find("BEGIN:BENV") != std::string::npos)
         {
            isSenderVCard = false;
         }
         else if(data.find("BEGIN:MSG") != std::string::npos)
         {
            msgFound = true;
            continue;
         }
         else if(data.find("END:MSG") != std::string::npos)
         {
            msgFound = false;
            break;
         }
         else if (isVCard && (data.find("FN") != std::string::npos))
         {
            std::size_t found = data.find(":");
            std::size_t size = data.size()-(found+1);
            sender->pucFN = new tChar[size+1];
            memset(sender->pucFN, 0, size+1);
            std::size_t length = data.copy(sender->pucFN,size,(found+1));
            sender->pucFN[length]='\0';
         }
         else if (isVCard && (data.find("N") != std::string::npos) && (data.find("N") == 0))
         {
            std::size_t found = data.find(":");
            std::size_t size = data.size()-(found+1);
            sender->pucN = new tChar[size+1];
            memset(sender->pucN, 0, size+1);
            std::size_t length = data.copy(sender->pucN,size,(found+1));
            sender->pucN[length]='\0';
         }
         else if (isVCard && (data.find("TEL") != std::string::npos))
         {
            std::size_t found = data.find(":");
            std::size_t size = data.size()- (found+1);
            sender->pucTEL = new tChar[size+1];
            memset(sender->pucTEL, 0, size+1);
            std::size_t length = data.copy(sender->pucTEL, size, (found+1));
            sender->pucTEL[length]='\0';
         }
         else if (isVCard && (data.find("EMAIL") != std::string::npos))
         {
            std::size_t found = data.find(":");
            std::size_t size = data.size()-(found+1);
            sender->pucEMAIL = new tChar[size+1];
            memset(sender->pucEMAIL, 0, size+1);
            std::size_t length = data.copy(sender->pucEMAIL,size, (found+1));
            sender->pucEMAIL[length]='\0';
         }

         if(msgFound == true)
         {
            //subject - append if n lines identified.
            if(subject.size() != 0)
            {
               subject += '\n';
            }
            subject.append(data);
         }
      }
      subject += '\n';
      message.close();
   }
   else
   {
      ETG_TRACE_USR4(("File open failed"));
   }

   if((NOTIFY_MSG_EMAIL == type) || (NOTIFY_MSG_MMS == type))
   {
      parseMessageBodyContent(subject);
   }

   std::size_t len = subject.size();
   tString dataPayload = new tChar[len+1];
   if(dataPayload)
   {
      memset(dataPayload, 0, len+1);
      std::size_t length = subject.copy(dataPayload, len);
      dataPayload[length]='\0';
   }

   if(_inboxSearchType == e8SearchByHandle)
   {
     // Set the default time as 00000000T999999. Since we do not receive this value from MapReadMsgData signal
     QByteArray defaultDateTime(DEFAULT_DATE_TIME);
     _msgListObject.vSetDateTime(const_cast<tString>(defaultDateTime.constData()));

     //Bugfix for GMMY16-18639, GMMY16-19799: FirstName, LastName returned should not be a phone number and PhoneNumber returned should always be valid
     if (sender->pucFN && (strnlen(sender->pucFN, 255) > 0))
     {
        _msgListObject.vSetSenderName(validatePhoneNumber(sender->pucFN, FALSE));
        ETG_TRACE_USR4(("sValidatePhoneNumber SenderName :%s:", _msgListObject.sGetSenderName().constData()));
     }

     if (sender->pucTEL && (strnlen(sender->pucTEL, 255) > 0))
     {
        _msgListObject.vSetSenderAddress(validatePhoneNumber(sender->pucTEL, TRUE));
        ETG_TRACE_USR4(("sValidatePhoneNumber SenderAddr :%s:", _msgListObject.sGetSenderAddress().constData()));
     }

     //it is not required here, since clear has been done in handleListMessagesComplete()
     //_msgListObject.m_sSubject.clear();
     _msgListObject.vSetSubject(dataPayload);

     // CMG3G-7699 - IS2202_message send notification (network level)@FC_Messaging
     _msgListObject.vSetMsgListDataFlags(type, 0, 0, 0, 0, 0, !(readStatus), 0, 0, 0xFF);
   }
   else
   {
      _msgReadObject.vClear();
      _msgReadObject.vSetReadStatus(readStatus);
      _msgReadObject.vSetType(type);
      _msgReadObject.vSetAttachNum(attachmentNum);
      _msgReadObject.vSetReceiverNum(receiverNum);

      if(TRUE != folder.empty())
      {
         _BP_TRY_BEGIN
         {
            //Remove the FOLDER:telecom/msg/ text and copy the folder name only E.g. inbox, sent
            std::size_t size = folder.size()-19;
            folderName = new tChar[size+1];
            if(folderName)
            {
               std::size_t length = folder.copy(folderName,size,19);
               folderName[length]='\0';
            }
         }
         _BP_CATCH_ALL
         {
            delete[] dataPayload;

            ETG_TRACE_ERR(("sendGetMessageSuccess() Caught Exception"));
            return;
         }
         _BP_CATCH_END
      }

      _msgReadObject.vSetFolderName(folderName);
      _msgReadObject.vSetSenderInfo(sender);
      _msgReadObject.vSetDataFlag(dataFlag);
      _msgReadObject.vSetDataPayLoad(dataPayload, static_cast<tU32>(len));
   }

   delete[] folderName;
   delete[] dataPayload;
   delete[] sender->pucN;
   delete[] sender->pucFN;
   delete[] sender->pucTEL;
   delete[] sender->pucEMAIL;
   g_free(sender);

   //Remove the message file for next time read.
   if( remove( MSG_DBUS_EVO_GETMESSAGE_PATH ) != 0 )
   {
      ETG_TRACE_ERR(("File remove failed"));
   }
   else
   {
      ETG_TRACE_USR4(("File remove success"));
   }

   //Send complete loopback message
   LoopBackMessageData loopBackMessageData;

   if (_inboxSearchType == e8SearchByHandle)
   {
      loopBackMessageData.u32FunctionId = FC_MSG_LB_FID_NEW_INC_MSG_HANDLE;
      _inboxSearchType = e8Forward;
      MessagingMutex::messagingMutexUnlock();
   }
   else
   {
      loopBackMessageData.u32FunctionId = FC_MSG_LB_FID_GETMESSAGE;
   }
   loopBackMessageData.u8DeviceHandle = _deviceHandle;
   loopBackMessageData.sServiceName = _clientSessionPath;

   fc_messaging_tclService_Messaging::m_poMessagingService->vPostLoopBackMessage(loopBackMessageData);
}

void EvolutionDbusClient::messagingMapGetCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR4(("messagingMapGetCallback() called"));

   gboolean result;
   GError* error = NULLPTR;
   gchar* transferPath;

   tU8* deviceHandle = (tU8*)user_data;

   result = org_bluez_obex_message_call_get_finish((OrgBluezObexMessage*)source_object, &transferPath, res, &error);

   QMutexLocker lock(&_deviceListMutex);
   EvolutionDbusClient* client = NULLPTR;

   if(deviceHandle)
   {
      client = getDeviceInstance(*deviceHandle);
      delete deviceHandle;
   }

   if (error && (FALSE == result))
   {
      ETG_TRACE_USR4(("GetMessage Error: %s", g_dbus_error_get_remote_error (error)));
      if(client)
      {
         client->sendGetMessageError();
      }

      g_error_free(error);
   }
   else
   {
      ETG_TRACE_USR4(("GetMessage Success, Wait for transfer complete!!!"));
      ETG_TRACE_USR4(("GetMessage Transfer path: %s", transferPath));

      if(client)
      {
         std::string str = transferPath;
         client->_currentTransfer.insert(::std::pair<std::string, msgCallback>(str, GET_MESSAGE_CB));
         client->createDBusPropertiesProxyForTransferPropertiesChanged(str);
      }

      //org_bluez_obex_message_set_read((OrgBluezObexMessage*)source_object, true);
   }
}

tBool EvolutionDbusClient::messagingMapGet(const MessageTypeDbusMessageHandleInfo& messageTypeDbusMessageHandleInfo)
{
   ETG_TRACE_USR4(("messagingMapGet() called"));

   if(!_clientSessionPath)
   {
      ETG_TRACE_USR4(("MAP Session not connected"));
      return false;
   }

   gboolean retrieveAttachment = DONOT_RETRIEVE_ATTACHMENTS; //Default: do not retrieve attachments
   GError* error = NULLPTR;
   gchar* message = const_cast<gchar*> (MESSAGE_STRING);
   const QByteArray tmp = messageTypeDbusMessageHandleInfo.m_sDbusMessageHandle.toUtf8();

   //Message path format: /org/bluez/obex/client/session(count)/message{handle}
   gchar* msgObjPath = g_strconcat(_clientSessionPath, message, const_cast<gchar*>(tmp.constData()), NULLPTR);

   ETG_TRACE_USR4(("MessagePath: %s", msgObjPath));

   //Create proxy for the message
   _proxyGetMessage = org_bluez_obex_message_proxy_new_sync(_evoMsgBus,
         G_DBUS_PROXY_FLAGS_NONE, BASE_EVO_SERVICE_NAME,
         msgObjPath, NULLPTR, &error);

   g_free(msgObjPath);

   if (!_proxyGetMessage && error)
   {
      ETG_TRACE_ERR(("messagingMapGet : failed: %s", g_dbus_error_get_remote_error(error)));
      g_error_free(error);

      //Not able to create proxy for the message, send the error
      sendGetMessageError();
      return false;
   }

   tU8* deviceHandle = new tU8[1];

   if(!deviceHandle)
   {
      //Not able to allocate memory, send the error (this should never happen)
      sendGetMessageError();
      return false;
   }

   *deviceHandle = _deviceHandle;

   org_bluez_obex_message_call_get(_proxyGetMessage, MSG_DBUS_EVO_GETMESSAGE_PATH, retrieveAttachment,
         NULLPTR, messagingMapGetCallback, deviceHandle);

   tVoid* pCustodialLose = deviceHandle; //To remove LINT warning : Custodial Pointer
   (tVoid) pCustodialLose;

   return true;
}

void EvolutionDbusClient::messagingMapCancelCallback(GObject *source_object, GAsyncResult *res, gpointer user_data)
{
   ETG_TRACE_USR4(("messagingMapCancelCallback() called"));

   (void)user_data; //Lint fix

   gboolean result;
   GError* error = NULLPTR;

   result = org_bluez_obex_transfer_call_cancel_finish((OrgBluezObexTransfer*)source_object, res, &error);

   if (error && (FALSE == result))
   {
      ETG_TRACE_USR4(("Cancel Error: %s", g_dbus_error_get_remote_error(error)));

      g_error_free(error);
   }

   //dereference the transfer proxy
   g_object_unref(source_object);
}

void EvolutionDbusClient::messagingMapCancel()
{
   ETG_TRACE_USR4(("messagingMapCancel() called"));

   GError* error = NULLPTR;
   std::string transferPath;

   //Get current active transfer path from the list
   ::std::map<std::string, msgCallback>::iterator it;
   it = _currentTransfer.begin();

   if(it != _currentTransfer.end())
   {
      transferPath = it->first;
   }
   else
   {
      ETG_TRACE_USR4(("No active transfer in Messaging"));
      return;
   }

   if(transferPath.empty())
   {
      ETG_TRACE_USR4(("Transfer path Empty"));
      return;
   }

   ETG_TRACE_USR4(("Active transfer path : %s", transferPath.c_str()));

   //Transfer path format: /org/bluez/obex/client/session(count1)/transfer(count2)
   //Create proxy for the current ongoing transfer
   OrgBluezObexTransfer* proxyTransfer = org_bluez_obex_transfer_proxy_new_sync(_evoMsgBus,
         G_DBUS_PROXY_FLAGS_NONE, BASE_EVO_SERVICE_NAME,
         transferPath.c_str(), NULLPTR, &error);

   if (!proxyTransfer && error)
   {
      ETG_TRACE_ERR(("messagingMapCancel : transfer function failed: %s", g_dbus_error_get_remote_error(error)));
      g_error_free(error);
      return;
   }

   //lint --e{826} Suppress Info 826: Suspicious pointer-to-pointer conversion (area too small)
   // this removes the lint warning for this block only
   OrgBluezObexTransferProxy *transferProxy = ORG_BLUEZ_OBEX_TRANSFER_PROXY(proxyTransfer);
   if (transferProxy)
   {
      //Default timeout 300 sec
      g_dbus_proxy_set_default_timeout(&transferProxy->parent_instance, DEFAULT_TIMEOUT_VAL_DBUS_CALL);
   }
   else
   {
      ETG_TRACE_ERR(("messagingMapCancel : set timeout failed for Transfer proxy"));
   }

   org_bluez_obex_transfer_call_cancel(proxyTransfer, NULLPTR, messagingMapCancelCallback, NULLPTR);
}

tBool EvolutionDbusClient::isValidDeviceHandle(tU8 deviceHandle)
{
   ETG_TRACE_USR4(("isValidDeviceHandle: deviceHandle = %d", deviceHandle));

   if ((deviceHandle >= MINIMUM_DEVICE_HANDLE) && (deviceHandle <= MAXIMUM_DEVICE_HANDLE))
   {
      if (_bluetoothDeviceObject[deviceHandle])
      {
         return true; // Valid device handle
      }
   }
   return false; // Invalid device handle
}

void EvolutionDbusClient::setCurrentOutgoingDeviceId(tU8 deviceHandle)
{
   ETG_TRACE_USR4(("setCurrentOutgoingDeviceId: %d", deviceHandle));

   if ( !((deviceHandle >= MINIMUM_DEVICE_HANDLE) &&
         (deviceHandle <= MAXIMUM_DEVICE_HANDLE)))
   {
      return; //Invalid device handle
   }

   _currentOutgoingDeviceId = deviceHandle;
}

tU8 EvolutionDbusClient::getCurrentOutgoingDeviceId()
{
   ETG_TRACE_USR4(("getCurrentOutgoingDeviceId: %d", _currentOutgoingDeviceId));
   return _currentOutgoingDeviceId;
}

void EvolutionDbusClient::messagingDeviceConnectionStatus(tU8 deviceHandle, std::string bluetoothDeviceAddress)
{
   ETG_TRACE_USR4(("messagingDeviceConnectionStatus() called"));

   ETG_TRACE_USR1(("DeviceHandle: %d, DeviceAddress: %s", deviceHandle, bluetoothDeviceAddress.c_str()));

   if (!((deviceHandle >= MINIMUM_DEVICE_HANDLE) && (deviceHandle <= MAXIMUM_DEVICE_HANDLE)))
   {
      //ETG_TRACE_USR4(("Invalid device handle"));
      return;
   }

   if (_bluetoothDeviceObject[deviceHandle])
   {
      QString currentActiveSMSMMSService = _bluetoothDeviceObject[deviceHandle]->_serviceNameAccountManager.sGetCurrentActiveSMSMMSService();
      QString currentActiveEmailService = _bluetoothDeviceObject[deviceHandle]->_serviceNameAccountManager.sGetCurrentActiveEmailService();

      if ((TRUE != currentActiveSMSMMSService.isEmpty()) || (TRUE != currentActiveEmailService.isEmpty()))
      {
         ETG_TRACE_USR1(("Device is already connected"));
         return;
      }

      delete _bluetoothDeviceObject[deviceHandle];
      _bluetoothDeviceObject[deviceHandle] = NULLPTR;
   }

   _bluetoothDeviceObject[deviceHandle] = new EvolutionDbusClient;
   if (!_bluetoothDeviceObject[deviceHandle])
   {
      ETG_TRACE_ERR(("Unable to allocate memory"));
      return;
   }

   _bluetoothDeviceObject[deviceHandle]->_deviceHandle = deviceHandle;

   //Improvement: to avoid blocking of Mutex lock in case MapListMsgCallback is not received for MapListMsg request
   _bluetoothDeviceObject[deviceHandle]->_waitForMapListMsgCompSignal = FALSE;

   _bluetoothDeviceObject[deviceHandle]->_folderListRequestStatus = FC_MSG_FOLDER_LIST_NOT_REQUESTED;
   _bluetoothDeviceObject[deviceHandle]->resetMapDeviceCapabilities();

#ifndef CONNECT_PBAP_MAP_VIA_BM_APPL
   //Create the proxies for Client, ObjectManager, MessageAccess
   if (false == _bluetoothDeviceObject[deviceHandle]->messagingMapCreateSession(bluetoothDeviceAddress))
   {
      ETG_TRACE_ERR(("Unable to Create MAP Session"));
      return;
   }

   //_bluetoothDeviceObject[deviceHandle]->messagingMapGetCapabilities();
   _bluetoothDeviceObject[deviceHandle]->messagingMapSessionConnected();
#else
   if((fc_messaging_tclApp::m_poMainAppInstance)->m_poClientHandlerBluetoothSetings)
   {
      (fc_messaging_tclApp::m_poMainAppInstance)->m_poClientHandlerBluetoothSetings->bConnectMAPProfile(deviceHandle);
   }

   _bluetoothDeviceObject[deviceHandle]->updateMapDeviceConnectionStatus(FC_MSG_MAP_CONNECTING);
#endif
}

void EvolutionDbusClient::messagingDeviceDisconnected(tU8 deviceHandle, tBool mapOnOff)
{
   ETG_TRACE_USR4(("messagingDeviceDisconnected() entered"));

   QMutexLocker lock(&_deviceListMutex);

   if (false == isValidDeviceHandle(deviceHandle))
   {
      ETG_TRACE_USR1(("messagingDeviceDisconnected deviceHandle '%d' is not valid", deviceHandle));
      return;
   }

   if (_bluetoothDeviceObject[deviceHandle])
   {
      ETG_TRACE_USR1(("Device got disconnected"));

      _bluetoothDeviceObject[deviceHandle]->stopTimers();
      _bluetoothDeviceObject[deviceHandle]->_retryCount = 0;
      _bluetoothDeviceObject[deviceHandle]->vSetMapRetryCount();

      // Bugfix for GMMY16-5367 : Selected view on new message alert and nothing happened
      // and GMMY16-5368 : Message inbox was missing many of the newest messages
      _bluetoothDeviceObject[deviceHandle]->unregisterMessageAccessSignals();
      _bluetoothDeviceObject[deviceHandle]->unrefMessageAccessProxy(deviceHandle);

      _bluetoothDeviceObject[deviceHandle]->setWaitForMapListMsgCompSignal(false);

      _bluetoothDeviceObject[deviceHandle]->clearClientSessionPath();

      // Two consecutive New Incoming Message Bug-Fix GMMY15-1259 - Clear the list when the phone gets disconnected
      _bluetoothDeviceObject[deviceHandle]->_messageHandleList.clear();

      // CMG3G-7699 - IS2202_message send notification (network level)@FC_Messaging
      _bluetoothDeviceObject[deviceHandle]->_msgNotifyMap.clear();

      QMutexLocker newmsglock(&(_bluetoothDeviceObject[deviceHandle]->_newMsgHandleListMutex));
      _bluetoothDeviceObject[deviceHandle]->_newMessageHandleList.clear();

      if (false == mapOnOff)
      {
         _bluetoothDeviceObject[deviceHandle]->messagingMapSessionDisconnected();

         delete _bluetoothDeviceObject[deviceHandle];
         _bluetoothDeviceObject[deviceHandle] = NULLPTR;
      }
   }
}

tVoid EvolutionDbusClient::stopTimers()
{
   if (fc_messaging_tclApp::m_poMessaging)
   {
      if (FALSE == fc_messaging_tclApp::m_poMessaging->bStopTimer(FC_MESSAGING_MAPCAPABILITYRETRY_TIMER_ID))
      {
         ETG_TRACE_ERR(("Unable to stop the timer FC_MESSAGING_MAPCAPABILITYRETRY_TIMER_ID"));
      }

      if(TRUE == fc_messaging_tclClientHandler_BluetoothSetings::m_u8RetryDeviceList.empty())
      {
         if (FALSE == fc_messaging_tclApp::m_poMessaging->bStopTimer(FC_MESSAGING_MAPERRORRETRY_TIMER_ID))
         {
            ETG_TRACE_ERR(("Unable to stop the timer FC_MESSAGING_MAPERRORRETRY_TIMER_ID"));
         }
         setRetryTimerRunningStatus(FALSE);
      }

      if (FALSE == fc_messaging_tclApp::m_poMessaging->bStopTimer((tU16)(FC_MESSAGING_MNSCONNECTIONRETRY_TIMER_ID+_deviceHandle)))
      {
         ETG_TRACE_ERR(("Unable to stop the timer FC_MESSAGING_MNSCONNECTIONRETRY_TIMER_ID+%d",_deviceHandle));
      }

      if (FALSE == fc_messaging_tclApp::m_poMessaging->bStopTimer(FC_MESSAGING_AUDIOROUTEREQUEST_WAIT_TIMER_ID))
      {
         ETG_TRACE_ERR(("Unable to stop the timer FC_MESSAGING_AUDIOROUTEREQUEST_WAIT_TIMER_ID"));
      }
   }
}

void EvolutionDbusClient::handleDeviceDisconnection()
{
   ETG_TRACE_USR4(("handleDeviceDisconnection() called"));

   //Lock is not required since it is already acquired in BluetoothSettings ClientHandler
   while (!fc_messaging_tclClientHandler_BluetoothSetings::m_u8DiscDeviceHandleList.isEmpty())
   {
      tU8 deviceHandle = fc_messaging_tclClientHandler_BluetoothSetings::m_u8DiscDeviceHandleList.front();
      EvolutionDbusClient* evolutionClient = EvolutionDbusClient::getDeviceInstance(deviceHandle);

      if(evolutionClient)
      {
#ifndef CONNECT_PBAP_MAP_VIA_BM_APPL
         //Remove MAP Session first (sync call used)
         evolutionClient->messagingMapRemoveSession();

         evolutionClient->messagingDeviceDisconnected(deviceHandle);
#else
         //Check MAP session is active
         QString currentActiveSMSMMSService = evolutionClient->_serviceNameAccountManager.sGetCurrentActiveSMSMMSService();
         QString currentActiveEmailService = evolutionClient->_serviceNameAccountManager.sGetCurrentActiveEmailService();

         if ((TRUE != currentActiveSMSMMSService.isEmpty()) || (TRUE != currentActiveEmailService.isEmpty()))
         {
            if((fc_messaging_tclApp::m_poMainAppInstance)->m_poClientHandlerBluetoothSetings)
            {
               (fc_messaging_tclApp::m_poMainAppInstance)->m_poClientHandlerBluetoothSetings->vDisconnectMAPProfile(deviceHandle);
            }
            //NCG3D-103057: The following line is commented so that the Service will be set to NULL only after the InterfacesRemoved callback is obtained.
            //evolutionClient->_serviceNameAccountManager.vSetCurrentActiveService(NULLPTR, NULLPTR);
         }
         else
         {
            evolutionClient->messagingDeviceDisconnected(deviceHandle);
         }
#endif
      }

      fc_messaging_tclClientHandler_BluetoothSetings::m_u8DiscDeviceHandleList.pop_front();
   }
}

void EvolutionDbusClient::switchMapOn()
{
   ETG_TRACE_USR4(("SwitchMapOn() called"));

   std::vector<tU8>::iterator iter;

   for(iter = fc_messaging_tclClientHandler_BluetoothSetings::m_u8ConnectedDevicesOrder.begin();
         iter != fc_messaging_tclClientHandler_BluetoothSetings::m_u8ConnectedDevicesOrder.end(); iter++)
   {
      vConnectDeviceToMAPSession(*iter);
   }
}

void EvolutionDbusClient::switchMapOff()
{
   ETG_TRACE_USR4(("SwitchMapOff() called"));

   std::vector<tU8>::iterator it;

   for(it = fc_messaging_tclClientHandler_BluetoothSetings::m_u8ConnectedDevicesOrder.begin();
         it != fc_messaging_tclClientHandler_BluetoothSetings::m_u8ConnectedDevicesOrder.end(); it++)
   {
      vDisconnectDeviceFromMAPSession(*it);
   }
}

void EvolutionDbusClient::vConnectDeviceToMAPSession(tU8 u8DeviceHandle)
{
   ETG_TRACE_USR4(("SwitchMapOn() called"));

   std::map<tU8, std::string>::iterator it = fc_messaging_tclClientHandler_BluetoothSetings::m_oConnectedDeviceList.find(u8DeviceHandle);

   if(it != fc_messaging_tclClientHandler_BluetoothSetings::m_oConnectedDeviceList.end())
   {
      if(TRUE == fc_messaging_tclClientHandler_BluetoothSetings::m_u8RetryDeviceList.empty())
      {
         EvolutionDbusClient* evolutionClient = EvolutionDbusClient::getDeviceInstance(it->first);

         if(evolutionClient)
         {
#ifndef CONNECT_PBAP_MAP_VIA_BM_APPL
            //Call CreateSession
            if(evolutionClient->messagingMapCreateSession(it->second))
            {
               evolutionClient->messagingMapSessionConnected();
            }
#else
            if((fc_messaging_tclApp::m_poMainAppInstance)->m_poClientHandlerBluetoothSetings)
            {
               (fc_messaging_tclApp::m_poMainAppInstance)->m_poClientHandlerBluetoothSetings->bConnectMAPProfile(it->first);
            }
#endif
         }
         else
         {
            EvolutionDbusClient::messagingDeviceConnectionStatus(it->first, it->second);
         }
      }
      else
      {
         fc_messaging_tclClientHandler_BluetoothSetings::addDeviceToRetryList(it->first);
      }
   }
}

void EvolutionDbusClient::vDisconnectDeviceFromMAPSession(tU8 u8DeviceHandle)
{
   ETG_TRACE_USR4(("SwitchMapOff() called"));

   EvolutionDbusClient* evolutionClient = EvolutionDbusClient::getDeviceInstance(u8DeviceHandle);

   if(evolutionClient)
   {
      //Check MAP session is active
      QString currentActiveSMSMMSService = evolutionClient->_serviceNameAccountManager.sGetCurrentActiveSMSMMSService();
      QString currentActiveEmailService = evolutionClient->_serviceNameAccountManager.sGetCurrentActiveEmailService();

#ifndef CONNECT_PBAP_MAP_VIA_BM_APPL
      if ((TRUE != currentActiveSMSMMSService.isEmpty()) || (TRUE != currentActiveEmailService.isEmpty()))
      {
         //Call Remove Session
         evolutionClient->messagingMapRemoveSession();
      }

      //unregister for map signals
      evolutionClient->messagingDeviceDisconnected(u8DeviceHandle, TRUE);
#else
      if ((TRUE != currentActiveSMSMMSService.isEmpty()) || (TRUE != currentActiveEmailService.isEmpty()) ||
            (TRUE != fc_messaging_tclClientHandler_BluetoothSetings::m_u8RetryDeviceList.empty()))
      {
         if((fc_messaging_tclApp::m_poMainAppInstance)->m_poClientHandlerBluetoothSetings)
         {
            (fc_messaging_tclApp::m_poMainAppInstance)->m_poClientHandlerBluetoothSetings->vDisconnectMAPProfile(u8DeviceHandle);
         }
      }
#endif
      fc_messaging_tclClientHandler_BluetoothSetings::removeDeviceFromRetryList(u8DeviceHandle);
      evolutionClient->stopTimers();
   }
   else
   {
      fc_messaging_tclClientHandler_BluetoothSetings::removeDeviceFromRetryList(u8DeviceHandle);
   }
}

tU8 EvolutionDbusClient::getDeviceHandle(const OrgBluezObexMessageAccess* proxy)
{
   ETG_TRACE_USR4(("getDeviceHandle() Called"));
   tU8 index;
   tU8 deviceHandle = 0;

   for (index = MINIMUM_DEVICE_HANDLE; index <= MAXIMUM_DEVICE_HANDLE; index++)
   {
      if (_bluetoothDeviceObject[index])
      {
         if (_bluetoothDeviceObject[index]->_proxyMessageAccess == proxy)
         {
            deviceHandle = _bluetoothDeviceObject[index]->_deviceHandle;
         }
      }
   }

   return deviceHandle;
}

EvolutionDbusClient* EvolutionDbusClient::getDeviceInstance(tU8 deviceHandle)
{
   if (deviceHandle >= MINIMUM_DEVICE_HANDLE && deviceHandle <= MAXIMUM_DEVICE_HANDLE)
   {
      return _bluetoothDeviceObject[deviceHandle]; // Even this may be NULL
   }
   else
   {
      ETG_TRACE_USR4(("getDeviceInstance : Device handle is not in range. NULL is returned"));
      return NULLPTR;
   }
}

EvolutionDbusClient* EvolutionDbusClient::getBluetoothDeviceObject(const OrgBluezObexMessageAccess* proxy)
{
   ETG_TRACE_USR4(("getBluetoothDeviceObject() Called"));
   tU8 index;
   for (index = MINIMUM_DEVICE_HANDLE; index <= MAXIMUM_DEVICE_HANDLE; index++)
   {
      if (_bluetoothDeviceObject[index])
      {
         if (_bluetoothDeviceObject[index]->_proxyMessageAccess == proxy)
         {
            return _bluetoothDeviceObject[index];
         }
      }
   }

   ETG_TRACE_ERR(("Proxy not found in the list"));
   return NULLPTR;
}

EvolutionDbusClient* EvolutionDbusClient::getBluetoothDeviceObject(const gchar* session)
{
   ETG_TRACE_USR4(("getBluetoothDeviceObject() called"));

   tU8 index;
   for (index = MINIMUM_DEVICE_HANDLE; index <= MAXIMUM_DEVICE_HANDLE; index++)
   {
      if (_bluetoothDeviceObject[index] && (_bluetoothDeviceObject[index]->_clientSessionPath) && session)
      {
         if (strcmp(_bluetoothDeviceObject[index]->_clientSessionPath, session) == 0)
         {
            return _bluetoothDeviceObject[index];
         }
      }
   }

   return NULLPTR;
}

EvolutionDbusClient* EvolutionDbusClient::getDeviceInstanceFromTransferPath(std::string transferPath)
{
   ETG_TRACE_USR4(("getDeviceInstanceFromTransferPath() called"));

   tU8 index;
   for (index = MINIMUM_DEVICE_HANDLE; index <= MAXIMUM_DEVICE_HANDLE; index++)
   {
      if (_bluetoothDeviceObject[index] &&
            (_bluetoothDeviceObject[index]->_currentTransfer.size() != 0))
      {
         ::std::map<std::string, msgCallback>::iterator it;

         it = _bluetoothDeviceObject[index]->_currentTransfer.find(transferPath);
         if (it != _bluetoothDeviceObject[index]->_currentTransfer.end())
         {
            return _bluetoothDeviceObject[index];
         }
      }
   }

   return NULLPTR;
}


void EvolutionDbusClient::unregisterMessageAccessSignals()
{
   ETG_TRACE_USR4(("unregisterMessageAccessSignals() called"));

   if(_proxyMessageAccess)
   {
      g_signal_handler_disconnect(_proxyMessageAccess, ulNewMsgNotification);

      ::std::map<OrgBluezObexMessageAccess*, notificationFunctionMap>::iterator it = _newMessageNotificationMap.find(_proxyMessageAccess);
      if(it != _newMessageNotificationMap.end())
      {
         if(NOTIFICATION_FUNC_1 == it->second)
         {
            ETG_TRACE_USR4(("onNewMessageNotification1() removed for deviceHandle: %d", _deviceHandle));
            _isNewMsgNotification1inUse = false;
         }
         else if (NOTIFICATION_FUNC_2 == it->second)
         {
            ETG_TRACE_USR4(("onNewMessageNotification2() removed for deviceHandle: %d", _deviceHandle));
            _isNewMsgNotification2inUse = false;
         }

         _newMessageNotificationMap.erase(it);
      }
   }
}

void EvolutionDbusClient::registerClientSignals()
{
   ETG_TRACE_USR4(("registerClientSignals() called"));

   if(_proxyClient)
   {
      _evoSignalHandlerId.ulSessionCreated = g_signal_connect(_proxyClient, "session-created", G_CALLBACK(clientSessionCreatedCallback), NULLPTR);
      _evoSignalHandlerId.ulSessionRemoved = g_signal_connect(_proxyClient, "session-removed", G_CALLBACK(clientSessionRemovedCallback), NULLPTR);
   }
}

void EvolutionDbusClient::unregisterClientSignals()
{
   ETG_TRACE_USR4(("unregisterClientSignals() called"));

   if(_proxyClient)
   {
      g_signal_handler_disconnect(_proxyClient, _evoSignalHandlerId.ulSessionCreated);
      g_signal_handler_disconnect(_proxyClient, _evoSignalHandlerId.ulSessionRemoved);
   }
}

void EvolutionDbusClient::registerObjectManagerSignals()
{
   ETG_TRACE_USR4(("registerObjectManagerSignals() called"));

   if(_proxyObjectManager)
   {
      _evoSignalHandlerId.ulInterfacesAdded = g_signal_connect(_proxyObjectManager, "interfaces-added", G_CALLBACK(onObjectManagerInterfacesAdded), NULLPTR);
      _evoSignalHandlerId.ulInterfacesRemoved = g_signal_connect(_proxyObjectManager, "interfaces-removed", G_CALLBACK(onObjectManagerInterfacesRemoved), NULLPTR);
   }
}

void EvolutionDbusClient::unregisterObjectManagerSignals()
{
   ETG_TRACE_USR4(("unregisterObjectManagerSignals() called"));

   if(_proxyObjectManager)
   {
      g_signal_handler_disconnect(_proxyObjectManager, _evoSignalHandlerId.ulInterfacesAdded);
      g_signal_handler_disconnect(_proxyObjectManager, _evoSignalHandlerId.ulInterfacesRemoved);
   }
}

void EvolutionDbusClient::unrefMessageAccessProxy(tU8 deviceHandle)
{
   ETG_TRACE_USR4(("unrefMessageAccessProxy() called"));

   if (_bluetoothDeviceObject[deviceHandle])
   {
      if(_bluetoothDeviceObject[deviceHandle]->_proxyDBusPropertiesForMessageAccess)
      {
         ETG_TRACE_USR4(("Dereferencing DBusPropertiesForMessageAccess Proxy"));

         g_object_unref(_bluetoothDeviceObject[deviceHandle]->_proxyDBusPropertiesForMessageAccess);
         _bluetoothDeviceObject[deviceHandle]->_proxyDBusPropertiesForMessageAccess = NULLPTR;
      }

      if(_bluetoothDeviceObject[deviceHandle]->_proxyMessageAccess)
      {
         ETG_TRACE_USR4(("Dereferencing MessageAccess Proxy"));

         g_object_unref(_bluetoothDeviceObject[deviceHandle]->_proxyMessageAccess);
         _bluetoothDeviceObject[deviceHandle]->_proxyMessageAccess = NULLPTR;
      }
   }
}

void EvolutionDbusClient::unrefClientProxy()
{
   ETG_TRACE_USR4(("unrefClientProxy() called"));

   if(_proxyClient)
   {
      ETG_TRACE_USR4(("Dereferencing Client Proxy"));

      g_object_unref((GObject*)_proxyClient);
      _proxyClient = NULLPTR;
   }
}

void EvolutionDbusClient::unrefObjectManagerProxy()
{
   ETG_TRACE_USR4(("unrefObjectManagerProxy() called"));

   if(_proxyObjectManager)
   {
      ETG_TRACE_USR4(("Dereferencing ObjectManager Proxy"));

      g_object_unref(_proxyObjectManager);
      _proxyObjectManager = NULLPTR;
   }
}

tBool EvolutionDbusClient::getWaitForMapListMsgCompSignal()
{
   ETG_TRACE_USR4(("getWaitForMapListMsgCompSignal() called: %d", _waitForMapListMsgCompSignal));
   return _waitForMapListMsgCompSignal;
}

tVoid EvolutionDbusClient::setWaitForMapListMsgCompSignal(tBool waitForMapListMsgCompSignal)
{
   ETG_TRACE_USR4(("setWaitForMapListMsgCompSignal() called: %d", waitForMapListMsgCompSignal));
   _waitForMapListMsgCompSignal = waitForMapListMsgCompSignal;
}

tString EvolutionDbusClient::validatePhoneNumber(tString sender, tBool isAddress)
{
   ETG_TRACE_USR4(("validatePhoneNumber() called : %s", sender));

   QString name = QString::fromUtf8(sender);
   tU16 senderLength = static_cast<tU16>(strlen(sender) + 1);
   ETG_TRACE_USR4(("senderLength : %d", senderLength));

   //Check if at least two digits are present
   QRegExp digit("[0-9]");
   tU8 u8Count = static_cast<tU8>(name.count(digit));

   if (u8Count >= 2)
   {
      //Check if any invalid characters for phone number are/is present
      QRegExp invalidCharacters("[^ ()-/\\\\*#+ABCD,TP!W@;pw0-9]");
      u8Count = static_cast<tU8>(name.count(invalidCharacters));
      ETG_TRACE_USR4(("count: %d", u8Count));

      if (u8Count == 0)
      {
         //A valid phone number found, remove tolerated characters and return
         name.remove(' ');
         name.remove(')');
         name.remove('(');
         name.remove('-');
         name.remove('/');
         name.remove('\\');
         QByteArray baSender = name.toUtf8();
         ETG_TRACE_USR4(("validatePhoneNumber baSender :%s:", baSender.constData()));
         tU16 tempStrLen = static_cast<tU16>(strlen(baSender.data()));
         memset(sender, OSAL_NULL, senderLength);
         memcpy(sender, baSender.data(), tempStrLen);
         ETG_TRACE_USR4(("validatePhoneNumber sending :%s:", sender));
         return sender;
      }
   }

   //Not a Valid phone number
   if (isAddress)
   {
      ETG_TRACE_USR4(("validatePhoneNumber send empty Address"));
      memset(sender, OSAL_NULL, senderLength);
   }

   ETG_TRACE_USR4(("validatePhoneNumber sending :%s:", sender));
   return sender;
}

tString EvolutionDbusClient::convertFromNCR(tString sString)
{
   ETG_TRACE_USR4(("convertFromNCR() Called"));

   if (NULLPTR == strchr(sString, '&'))
   {
      ETG_TRACE_USR4(("The string does not contain NCR data"));
      return sString;
   }

   ETG_TRACE_USR4(("convertFromNCR received string :%s:", sString));

   tU32 ncrValue = 0;
   tU32 destIdx = 0; // Always less than or equal to the Source length
   tU32 srcLen = static_cast<tU32>(OSAL_u32StringLength(sString) + 1);
   ETG_TRACE_USR4(("convertFromNCR srcLen: %u", srcLen));

   tChar tmpChar = '\0'; // To check for the ending semicolon(0x3B) of each NCR entity
   tChar cChar = '\0'; // To read every character of the Source string
   tChar validString[srcLen];
   memset(validString, OSAL_NULL, srcLen);

   tString sInString = sString;
   tString checkCData = NULLPTR;

   tBool bContainsNCR=FALSE;

   for (tU32 srcIdx = 0; srcIdx < srcLen-1; ) // lint fix
   {
      ETG_TRACE_USR4(("convertFromNCR srcIdx: %u", srcIdx));

      ETG_TRACE_USR4(("convertFromNCR sInString : %s", sInString));

      cChar = sInString[0];

      if (cChar == 0x26) //Check for '&'
      {
         checkCData = strchr(sInString, '&');

         if (checkCData)
         {
            ETG_TRACE_USR4(("convertFromNCR sCheckCData : %s", checkCData));

            tU8 tmpOffset = 0;

            //Read NCR value and Check for the ending semicolon
            if ((sscanf(checkCData, "&#x0%x%c", &ncrValue, &tmpChar) == 2) && (tmpChar == 0x3B))
            {
               ETG_TRACE_USR4(("convertFromNCR NCR date format &#x0XXX;")); //XXX is a hex value
               tmpOffset = 5;
            }
            else if ((sscanf(checkCData, "&#x%x%c", &ncrValue, &tmpChar) == 2) && (tmpChar == 0x3B))
            {
               ETG_TRACE_USR4(("convertFromNCR NCR date format &#xXXX;")); //XXX is a hex value
               tmpOffset = 4;
            }
            else if ((sscanf(checkCData, "&#0%d%c", &ncrValue, &tmpChar) == 2) && (tmpChar == 0x3B))
            {
               ETG_TRACE_USR4(("convertFromNCR NCR date format &#0XXX;")); //XXX is a decimal value
               tmpOffset = 4;
            }
            else if ((sscanf(checkCData, "&#%d%c", &ncrValue, &tmpChar) == 2) && (tmpChar == 0x3B))
            {
               ETG_TRACE_USR4(("convertFromNCR NCR date format &#XXX;")); //XXX is a decimal value
               tmpOffset = 3;
            }
            else
            {
               ETG_TRACE_USR4(("Not a NCR data format"));
               validString[destIdx++] = cChar;
               sInString++;
               srcIdx++; // lint fix
               continue;
            }
            if(tmpOffset>0)
            {
               bContainsNCR=TRUE;
            }
            ETG_TRACE_USR4(("convertFromNCR NcrValue : %u", ncrValue));

            tChar sUtf8[5] = { 0x00 }; //Char array to Hold UTF-8 data

            if (ncrValue < 0x80)
            {
               sUtf8[0] = static_cast<tChar>(ncrValue);
               sUtf8[1] = '\0';
               ETG_TRACE_USR4(("convertFromNCR Hex value for single byte data:   %02x", sUtf8[0]));
            }
            else if (ncrValue >= 0x80 && ncrValue < 0x800)
            {
               sUtf8[0] = static_cast<tChar>((ncrValue >> 6) | 0xC0);
               sUtf8[1] = static_cast<tChar>((ncrValue & 0x3F) | 0x80);
               sUtf8[2] = '\0';
               ETG_TRACE_USR4(("convertFromNCR Hex value for two bye data:  %02x  %02x", sUtf8[0], sUtf8[1]));
            }
            else if (ncrValue >= 0x800 && ncrValue < 0xFFFF)
            {
               if (ncrValue >= 0xD800 && ncrValue <= 0xDFFF)
               {
                  // Surrogatory codes
                  ETG_TRACE_USR4(("convertFromNCR, this could be a surrogatory code %u!", ncrValue));
               }
               else
               {
                  sUtf8[0] = static_cast<tChar>(((ncrValue >> 12)) | 0xE0);
                  sUtf8[1] = static_cast<tChar>(((ncrValue >> 6) & 0x3F) | 0x80);
                  sUtf8[2] = static_cast<tChar>(((ncrValue) & 0x3F) | 0x80);
                  sUtf8[3] = '\0';
                  ETG_TRACE_USR4(("convertFromNCR Hex value for three bye data: %02x  %02x  %02x", sUtf8[0], sUtf8[1], sUtf8[2]));
               }
            }
            else if (ncrValue >= 0x10000 && ncrValue < 0x10FFFF)
            {
               sUtf8[0] = static_cast<tChar>(((ncrValue >> 18)) | 0xF0);
               sUtf8[1] = static_cast<tChar>(((ncrValue >> 12) & 0x3F) | 0x80);
               sUtf8[2] = static_cast<tChar>(((ncrValue >> 6) & 0x3F) | 0x80);
               sUtf8[3] = static_cast<tChar>(((ncrValue) & 0x3F) | 0x80);
               sUtf8[4] = '\0';
               ETG_TRACE_USR4(("convertFromNCR Hex value for four bye data: %02x  %02x  %02x  %02x", sUtf8[0], sUtf8[1], sUtf8[2], sUtf8[3]));
            }
            else
            {
               ETG_TRACE_USR4(("convertFromNCR NCR value does not fit in range"));
               validString[destIdx++] = cChar;
               sInString++;
               srcIdx++; // lint fix
               continue;
            }

            for (tU8 index = 0; index < 5; index++)
            {
               if(sUtf8[index] != 0x0)
               {
                  validString[destIdx++] = sUtf8[index];
               }
            }

            tU16 offset = static_cast<tU16>(getIntLength(ncrValue) + tmpOffset);

            ETG_TRACE_USR4(("convertFromNCR Offset: %u", offset));

            sInString += offset;
            srcIdx = srcIdx + offset - 1;
            //checkCData = strchr(sInString, '&'); //Coverity Fix: Not required here, since it is done on starting of the loop
            ncrValue = 0;
            tmpChar = '\0';
         }
      }
      else
      {

         ETG_TRACE_USR4(("Encountered normal ASCII character, no need to convert to UTF-8"));
         validString[destIdx++] = cChar;
         sInString++;
      }
      srcIdx++; // lint fix
   }

   validString[destIdx] = '\0';

   memset(sString, OSAL_NULL, srcLen);
   memcpy(sString, validString, destIdx);


   ETG_TRACE_USR4(("convertFromNCR:: The string to be returned before truncation is %s",sString));
   if(TRUE==bContainsNCR)
   {
      // Traverse to one character before the '\0' and check if it is not ';'
      // TODO: Currently the invalid string in the last position alone is stripped.
      //       Need to check if the same has to be done for the complete message.
      tString sDatatoStrip_start = NULLPTR;
      tString sDatatoStrip_end = NULLPTR;

      sDatatoStrip_start=strrchr(sString, '&');

      if (sDatatoStrip_start)
      {
         sDatatoStrip_end = strrchr(sString, '\0');

         if (sDatatoStrip_end)
         {
            if(*(sDatatoStrip_end-1)!= ';')
            {
               //Strip the data, since there seems to be an invalid character at the end
               *(sDatatoStrip_start) = '\0';
               ETG_TRACE_USR4(("ConvertFromNCR:: The string has a malicious content at the end"));
            }
         }
      }
   }
   if (TRUE != g_utf8_validate((gchar*) sString, -1, NULLPTR))
   {
      ETG_TRACE_ERR(("String contains invalid UTF-8 data!"));
   }
   ETG_TRACE_USR4(("ConvertFromNCR:: The string to be returned is %s",sString));
   return sString;
}

tU8 EvolutionDbusClient::getIntLength(tU32 u32Value)
{
   tU8 u8Length = 1;

   while (u32Value > 9)
   {
      u8Length++;
      u32Value /= 10;
   }

   return u8Length;
}

tVoid EvolutionDbusClient::unlockDBusMutex()
{
   LoopBackMessageData loopBackMessageData;

   loopBackMessageData.u32FunctionId = FC_MSG_LB_FID_UNLOCK_DBUS_MUTEX;
   loopBackMessageData.u8DeviceHandle = _deviceHandle;
   loopBackMessageData.sServiceName = _clientSessionPath;

   fc_messaging_tclService_Messaging::m_poMessagingService->vPostLoopBackMessage(loopBackMessageData);
}

tBool EvolutionDbusClient::getMapNotifyStatus()
{
   ETG_TRACE_USR4(("getMapNotifyStatus() Called"));
   return _mapNotifyStatus;
}

MsgRead_t* EvolutionDbusClient::getMsgReadObject()
{
   return &_msgReadObject;
}

MsgListing_t* EvolutionDbusClient::getMsgListObject()
{
   return &_msgListObject;
}

tVoid EvolutionDbusClient::sendFolderListRequest(tenFC_Msg_FolderListRequestStatus folderListRequestStatus)
{
   ETG_TRACE_USR4(("sendFolderListRequest: %u", folderListRequestStatus));

   struct Job* prJob = new Job;
   if (prJob)
   {
      prJob->m_pfMethod = &fc_messaging_tclService_Messaging::vHandle_ListFolders;
      prJob->u8DeviceHandle = _deviceHandle;
      prJob->u16CmdCounter = 0;
      prJob->u16FunctionID = 0;
      prJob->u16RegisterID = 0;
      prJob->u16SourceAppID = 0;
      prJob->m_pvMethodStart = NULLPTR;

      if (0 != MessagingMutex::messagingMutexTrylock(prJob))
      {
         ETG_TRACE_ERR(("sendFolderListRequest : Unable To acquire mutex!!"));

         (tVoid) folderListRequestStatus;

         tVoid* pCustodialLose = prJob; //To remove LINT warning : Custodial Pointer
         (tVoid) pCustodialLose;

         return;
      }
      OSAL_DELETE prJob;

      _folderListRequestStatus = folderListRequestStatus;

      ::std::string folder = "/telecom/msg";
      messagingMapListFolder(folder);
   }
}

tVoid EvolutionDbusClient::folderListRequestComplete()
{
   ETG_TRACE_USR4(("folderListRequestComplete() called"));

   MessagingMutex::messagingMutexUnlock();

   _folderListRequestStatus = FC_MSG_FOLDER_LIST_NOT_REQUESTED;
   _mapDeviceCapabilities.u8DeviceHandle = _deviceHandle;

   sendMapDeviceCapabilitiesStatus();
}

tVoid EvolutionDbusClient::updateMapDeviceConnectionStatus(tU8 mapConnectionState)
{
   ETG_TRACE_USR4(("updateMapDeviceConnectionStatus() called : %d", mapConnectionState));

   _mapDeviceCapabilities.u8MapConnectionState = mapConnectionState;
   sendMapDeviceCapabilitiesStatus();
}

tVoid EvolutionDbusClient::sendMapDeviceCapabilitiesStatus()
{
   ETG_TRACE_USR4(("sendMapDeviceCapabilitiesStatus() called"));

   fc_messaging_tclService_Messaging::m_poMessagingService->_MapDeviceCapabilities(&_mapDeviceCapabilities);
}

tVoid EvolutionDbusClient::resetMapDeviceCapabilities()
{
   ETG_TRACE_USR4(("resetMapDeviceCapabilities() called"));

   _mapDeviceCapabilities.u8DeviceHandle = _deviceHandle;
   _mapDeviceCapabilities.u16SupportedMessageTypes = FC_MSG_NO_MSG_TYPES_SUPPORTED;
   _mapDeviceCapabilities.u16SupportedMapFeatures = FC_MSG_NO_FEATURES_SUPPORTED;
   _mapDeviceCapabilities.u8MapConnectionState = FC_MSG_MAP_DISCONNECTED;
   _mapDeviceCapabilities.rFC_Msg_ActiveEmailAccount.u16AccountHandle = 0;
   _mapDeviceCapabilities.rFC_Msg_ActiveEmailAccount.sAccountName.clear();
}

tVoid EvolutionDbusClient::setSignatureText(::std::string signatureText)
{
   ETG_TRACE_USR4(("setSignatureText() : %s", signatureText.c_str()));

   _signatureText = signatureText;
}

tBool EvolutionDbusClient::getRetryTimerRunningStatus()
{
   ETG_TRACE_USR4(("getRetryTimerRunningStatus() called: %d", _isRetryTimerRunning));
   return _isRetryTimerRunning;
}

tVoid EvolutionDbusClient::setRetryTimerRunningStatus(tBool retryTimerRunningStatus)
{
   ETG_TRACE_USR4(("setRetryTimerRunningStatus() called: %d", retryTimerRunningStatus));
   _isRetryTimerRunning = retryTimerRunningStatus;
}
void EvolutionDbusClient::vSetMapRetryCount(tU8 u8RetryCount)
{
   ETG_TRACE_USR4(("vSetMapRetryCount() called: %d", u8RetryCount));
   m_MaxMapRetryCount = u8RetryCount;
}
tU8 EvolutionDbusClient::vGetMapRetryCount()
{
   ETG_TRACE_USR4(("vGetMapRetryCount() called: %d", m_MaxMapRetryCount));
   return m_MaxMapRetryCount;
}
/** @} */
