/**
 * @file FC_Messaging_DbusClientInterface.h
 * @author
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */
#ifndef _FC_MESSAGING_DBUS_CLIENT_INTERFACE_H_
#define _FC_MESSAGING_DBUS_CLIENT_INTERFACE_H_

#include "../FileSystemManager/FC_Messaging_FileSystemManager.h"
#include <QString>


/**
 * defines for loopback messages function id's
 */
#define FC_MSG_LB_FID_BASE                                   0x01
#define FC_MSG_LB_FID_GETMESSAGE                           ( FC_MSG_LB_FID_BASE + 1)
#define FC_MSG_LB_FID_RECENTLY_SENT_MESSAGE_INFO           ( FC_MSG_LB_FID_BASE + 2)
#define FC_MSG_LB_FID_NEW_INCOMING_MESSAGE                 ( FC_MSG_LB_FID_BASE + 3)
#define FC_MSG_LB_FID_MESSAGE_SENT_FROM_DEVICE             ( FC_MSG_LB_FID_BASE + 4)
#define FC_MSG_LB_FID_MESSAGE_DELETED_FROM_DEVICE          ( FC_MSG_LB_FID_BASE + 5)
#define FC_MSG_LB_FID_MESSAGE_DELETED_MESSAGE              ( FC_MSG_LB_FID_BASE + 6)
#define FC_MSG_LB_FID_MESSAGE_CREATE_MESSAGE_LIST          ( FC_MSG_LB_FID_BASE + 7)
#define FC_MSG_LB_FID_UPLOAD_MESSAGE_DATA                  ( FC_MSG_LB_FID_BASE + 8)
#define FC_MSG_LB_FID_MAP_LIST_MSG_DATA                    ( FC_MSG_LB_FID_BASE + 9)
#define FC_MSG_LB_FID_FOLDER_LIST                          ( FC_MSG_LB_FID_BASE + 10)
#define FC_MSG_LB_FID_FOLDER_LIST_COMPLETE                 ( FC_MSG_LB_FID_BASE + 11)
#define FC_MSG_LB_FID_GET_TOTAL_MESSAGE_COUNT_IN_FOLDER    ( FC_MSG_LB_FID_BASE + 12)
#define FC_MSG_LB_FID_GET_UNREAD_MESSAGE_COUNT_IN_FOLDER   ( FC_MSG_LB_FID_BASE + 13)
#define FC_MSG_LB_FID_GET_SUBFOLDERS_IN_FOLDER             ( FC_MSG_LB_FID_BASE + 14)
#define FC_MSG_LB_FID_UNLOCK_DBUS_MUTEX                    ( FC_MSG_LB_FID_BASE + 15)
#define FC_MSG_LB_FID_NEW_INC_MSG_HANDLE                   ( FC_MSG_LB_FID_BASE + 16)
#define FC_MSG_LB_FID_GET_MSG_ERROR                        ( FC_MSG_LB_FID_BASE + 17)   // Bugfix for GMMY15-11135
#define FC_MSG_LB_FID_DEVICE_MEMORY_STATUS                 ( FC_MSG_LB_FID_BASE + 18)   // Bugfix for GMMY16-8827
#define FC_MSG_LB_FID_CONTINUE_MAP_SESSION                 ( FC_MSG_LB_FID_BASE + 19)
#define FC_MSG_LB_FID_JOB_FROM_WORKQUEUE                   ( FC_MSG_LB_FID_BASE + 20)
#define FC_MSG_LB_FID_MAP_DEVICE_CAPABILITIES_STATUS       ( FC_MSG_LB_FID_BASE + 21)   // Task CMG3G-7698
#define FC_MSG_LB_FID_MSG_NOTIFICATION_STATUS              ( FC_MSG_LB_FID_BASE + 22)   // Task CMG3G-7699
#define FC_MSG_LB_FID_PUSH_MESSAGE_STATUS                  ( FC_MSG_LB_FID_BASE + 23)
#define FC_MSG_LB_FID_SET_MSG_STATUS                       ( FC_MSG_LB_FID_BASE + 24)
#define FC_MSG_LB_FID_DEVICE_MAP_CONNECTION_STATUS         ( FC_MSG_LB_FID_BASE + 25)
#define FC_MSG_LB_FID_READ_STATUS_CHANGED         		   ( FC_MSG_LB_FID_BASE + 26)

/**
 * Enum declaration for DBus Client interfaces
 */
enum DbusClientIf
{
   DBUS_CLIENT_IF_UNDEFINED = 0, /**< DBus Client Not defined */
   DBUS_CLIENT_IF_EVOLUTION = 1, /**< DBus Client Evolution Interface */
   //---------------------------------------
   DBUS_CLIENT_IF_LAST /**< End marker, must be the last one */
};

/**
 * forward class declarations
 */
class EvolutionDbusClient;
class FileSystemManager;
class SendMessage;
class MsgRead_t;
class MsgListing_t;

/**
 * DbusClientInterface Singleton Class
 */
class DbusClientInterface
{
public:

   /**
    * Function to create a singleton for the DbusClientInterface
    *
    * @return pointer to instance of DbusClientInterface object
    */
   static DbusClientInterface& getInstance();

   /**
    * @brief Destroys singleton instance of this class
    *
    * @return void
    */
   static void destroyInstance();

   /**
    * getSystemBus
    *
    * @return void
    */
   void getSystemBus();

   tBool messagingCreateEvoProxies();

   tVoid messagingRemoveEvoProxies();

   /**
    * Messaging Map Get Capabilities
    *
    * @param[in] deviceHandle: device handle
    *
    * @return void
    */
   void messagingMapGetCapabilities(tU8 deviceHandle);

   /**
    * Messaging Map Start
    *
    * @return void
    */
   void messagingMapStart();

   /**
    * Messaging Map Stop
    *
    * @return void
    */
   void messagingMapStop();

   /**
    * Messaging Map Chg Instance
    *
    * @param[in] accountHandle: account handle
    *
    * @return boolean
    */
   tBool messagingMapChgInstance(tU16 accountHandle);

   /**
    * Messaging Map Update Inbox
    *
    * @return void
    */
   void messagingMapUpdateInbox();

   /**
    * Messaging Map List Folder
    *
    * @param[in] folderName: folder name
    *
    * @return void
    */
   void messagingMapListFolder(QString folderName);

   /**
    * Messaging Map Get Sub Folder Count
    *
    * @param[in] folderName: folder name
    *
    * @return void
    */
   void messagingMapGetSubFolderCount(tString folderName);

   /**
    * Messaging Map List Folder
    *
    * @param[in] deviceHandle: device handle
    * @param[in] messageTypeDbusMessageHandleInfo: message handle and list type info
    *
    * @return void
    */
   void messagingMapReadMsg(tU8 deviceHandle, MessageTypeDbusMessageHandleInfo& messageTypeDbusMessageHandleInfo);

   /**
    * Messaging Map List Folder
    *
    * @param[in] deviceHandle: device handle
    * @param[in] messageHandle: message handle
    * @param[in] messageListType: message list type
    * @param[in] operation: operation
    *
    * @return boolean
    */
   tBool messagingMapSetMsgStatus(tU8 deviceHandle, QString messageHandle, most_fi_tcl_e8_MsgMessageListType messageListType, tU8 operation);

   /**
    * Messaging Map Abort
    *
    * @param[in] deviceHandle: device handle
    *
    * @return void
    */
   void messagingMapAbort(tU8 deviceHandle);

   /**
    * Messaging Map List Msg
    *
    * @param[in] createMessageListMethodStart: create message list method start
    * @param[in] listMaxCount: max count, default is MAXIMUM_LIST_COUNT (100)
    * @param[in] mapListMsgCaller: list msg caller, default is e8MapListMsgCallerCreateMessageList (2)
    * @param[in] listStartOffset: list offset, default is LIST_START_OFFSET (0X00)
    *
    * @return void
    */
   void messagingMapListMsg(
         most_msgfi_tclMsgCreateMessageListMethodStart createMessageListMethodStart,
         tU16 listMaxCount = MAXIMUM_LIST_COUNT,
         e8MapListMsgCaller mapListMsgCaller = e8MapListMsgCallerCreateMessageList,
         tU16 listStartOffset = LIST_START_OFFSET);

   /**
    * Messaging Map Get Message Count
    *
    * @param[in] folderName: folder name
    * @param[in] listMaxCount: list max count
    * @param[in] statusFilter: status filter
    *
    * @return void
    */
   void messagingMapGetMessageCount(tCString folderName, tU16 listMaxCount, tU8 statusFilter);

   /**
    * Messaging Map Upload
    *
    * @param[in] deviceHandle: device handle
    * @param[in] sendMessage: send message
    *
    * @return boolean
    */
   tBool messagingMapUpload(tU8 deviceHandle, SendMessage* sendMessage);

   /**
    * Messaging Map Upload Data
    *
    * @param[in] deviceHandle: device handle
    * @param[in] sendMessage: send message
    *
    * @return void
    */
   void messagingMapUploadData(tU8 deviceHandle, SendMessage* sendMessage);

   /**
    * Handle New Message Notification
    *
    * @param[in] createMessageListMethodStart: most_msgfi_tclMsgCreateMessageListMethodStart data
    *
    * @return void
    */
   void handleNewMessageNotification(most_msgfi_tclMsgCreateMessageListMethodStart* createMessageListMethodStart);

   tVoid sendFolderListRequest(tU8 deviceHandle, tenFC_Msg_FolderListRequestStatus folderListRequestStatus);

   //--------------------------------------------------------------------------------------------------------

   /**
    * Set Current Outgoing Device Id
    *
    * @param[in] deviceHandle: device handle
    *
    * @return void
    */
   void setCurrentOutgoingDeviceId(tU8 deviceHandle);

   /**
    * Get Current Outgoing Device Id
    *
    * @return tU8 device handle
    */
   tU8 getCurrentOutgoingDeviceId();

   /**
    * Get Current Active Service
    *
    * @param[in] deviceHandle: device handle
    * @param[in] MsgListType: msg list type, default: FI_EN_E8MSG_LIST_SMS_MMS
    *
    * @return QString current active service
    */
   QString getCurrentActiveService(tU8 deviceHandle, most_fi_tcl_e8_MsgMessageListType::tenType MsgListType = most_fi_tcl_e8_MsgMessageListType::FI_EN_E8MSG_LIST_SMS_MMS);

   /**
    * Get File System Manager
    *
    * @param[in] deviceHandle: device handle
    * @param[in] serviceName: service name
    *
    * @return FileSystemManager pointer to file system manager
    */
   FileSystemManager* getFileSystemManager(tU8 deviceHandle, QString serviceName);

   /**
    * Get Msg Read Object
    *
    * @return MsgRead_t msg read pointer
    */
   MsgRead_t* getMsgReadObject(tU8 deviceHandle);

   /**
    * Get Msg List Object
    *
    * @return MsgListing_t msg list pointer
    */
   MsgListing_t* getMsgListObject(tU8 deviceHandle);

   /**
    * Get Map Notify Status
    *
    * @return tBool True/False
    */
   tBool getMapNotifyStatus(tU8 deviceHandle);

   /**
    * Get Map Msg Capability Info
    *
    * @param[in] deviceHandle: device handle
    * @param[in] msgCapabilityInfo: capability info
    *
    * @return tBool True/False
    */
   tBool getMapMsgCapabilityInfo(tU8 deviceHandle, MsgCapability_t& msgCapabilityInfo);

   /**
    * Wait For Map ListMsgComp Signal
    *
    * @return tBool True/False
    */
   tBool waitForMapListMsgCompSignal();

   /**
    * Is Dbus Client Available
    *
    * @param[in] deviceHandle: device handle
    *
    * @return tBool True/False
    */
   tBool isDbusClientAvailable(tU8 deviceHandle);

   /**
    * Get Client Interface
    *
    * @return DbusClientIf Enum value
    */
   DbusClientIf getClientInterface(void);

   /**
    * Set Signature Text
    *
    * @param[in] deviceHandle: device handle
    * @param[in] signatureText: signature text
    *
    * @return tBool True/False
    */
   tBool setSignatureText(tU8 deviceHandle, ::std::string signatureText);

protected:
   static DbusClientInterface* _dbusClientIf; /**< pointer to singleton instance of this class */

private:
   /**
    * @brief Default constructor of this class
    * @note Constructor is private since this class is a singleton.
    *
    * @return void
    */
   DbusClientInterface();

   /**
    * @brief Destructor of this class
    * @note Destructor is private since this class is a singleton.
    *
    * @return void
    */
   ~DbusClientInterface();

   /**
    * @brief Copy constructor of this class
    * @note Copy constructor explicitly declared to prevent from unintentional misuse.
    *
    * @return void
    */
   DbusClientInterface(const DbusClientInterface&);

   /**
    * @brief Assignment operator
    * @note Assignment operator explicitly declared to prevent from unintentional misuse.
    *
    * @return void
    */
   const DbusClientInterface& operator=(const DbusClientInterface&);

   /**
    * Get Evolution DBus Client
    *
    * @return Pointer to Evolution Client
    */
   EvolutionDbusClient* getEvolutionDBusClient();

   tBool messagingCreateClientProxy();

   tBool messagingCreateObjectManagerProxy();

   DbusClientIf _dbusClient; /**< enum variable for dbus client interface */
};

#endif // _FC_MESSAGING_DBUS_CLIENT_INTERFACE_H_

/** @} */
