/**
 * @file DiagSystemSettings.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the DiagSystemSettings class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file holds the class definition for DiagSystemSettings class
 *
 * @ingroup IpcWrapper
 */

#ifndef DiagSystemSettings_h
#define DiagSystemSettings_h

#include "IDiagSystemSettings.h"
#include <map>

namespace com
{
namespace bosch
{
namespace pmapp
{

using namespace std;

typedef std::string SystemSetID;
typedef std::string SystemSetType;

class ExecuteSettingsInfo
{
public:
   /**
    * Constructor of ExecuteSettingsInfo class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   ExecuteSettingsInfo();

   /**
    * Parameterized Constructor of ExecuteSettingsInfo class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   ExecuteSettingsInfo(const SystemSetID sysSetID, const SystemSetType sysSetType, const uint32 cookie);

   /**
    * Copy Constructor of ExecuteSettingsInfo class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   ExecuteSettingsInfo(const ExecuteSettingsInfo& other);

   /**
    * Destructor of ExecuteSettingsInfo class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   ~ExecuteSettingsInfo();

   /**
    * Friend function used when copying the object.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   friend void swap(ExecuteSettingsInfo& first, ExecuteSettingsInfo& second);

   /**
    * Overloading '=' operator
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   //lint -e{1720} suppress "assignment operator has non-const parameter", see copy-and-swap idiom
   ExecuteSettingsInfo& operator=(ExecuteSettingsInfo other);

   /**
    * Overloading '==' operator
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   bool operator==(const ExecuteSettingsInfo& rhs) const;

   /**
    * Overloading '!=' operator
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   bool operator!=(const ExecuteSettingsInfo& rhs) const;

   // The below set of information is received from the diagnosis component as part of the
   // onExecuteSystemSettingRequest. The data is stored to be filled & sent as part of the response.

   SystemSetID _sysSetID;

   SystemSetType _sysSetType;

   uint32 _cookie;
};

class DiagSystemSettings : public IDiagSystemSettings
{

public:
   /**
    * Constructor of interface class DiagSystemSettings
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   DiagSystemSettings();

   /**
    * Virtual destructor of interface class DiagSystemSettings
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual ~DiagSystemSettings();

   /**
    * This function is called to prepare the PM for executing the system settings.
    *
    * @param[in] request - Request information from Diagnosis component
    *
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void onPrepareSystemSettingRequest(
         const ::boost::shared_ptr< PrepareSystemSettingRequest >& request) override;

   /**
    * This function is called to execute the system settings. On this request, PM has to reset the user data (or)
    * restore the default data in the persistent memory of PM.
    *
    * @param[in] request - Request information from Diagnosis component
    *
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void onExecuteSystemSettingRequest(
         const ::boost::shared_ptr< ExecuteSystemSettingRequest >& request) override;

   /**
    * This function is called to finalize the execution of system settings.
    *
    * @param[in] request - Request information from Diagnosis component
    *
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void onFinalizeSystemSettingRequest(
         const ::boost::shared_ptr< FinalizeSystemSettingRequest >& request) override;

   /**
    * This function is invoked to send the response to Diagnosis component after restoring / clearing the
    * user data stored persistently in PM
    *
    * @param[in]  operationSuccess  -  Success / Failure response
    *             act_t             -  Acknowledgement token
    *
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   virtual void onRestoreDefaultSettingResponse(const bool operationSuccess, const act_t act) override;

private:

   std::map<act_t, ExecuteSettingsInfo> _actToExecuteSettingsInfoMap;   /**< Map - Defset act token to Request Info */

   DECLARE_CLASS_LOGGER(); //lint !e1516 Member declaration hides inherited member
};

} // namespace pmapp
} // namespace bosch
} // namespace com

#endif // DiagSystemSettings_h
