/**
 * @file IpcWrapper.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the IpcWrapper class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup IpcWrapper
 */

#include "IpcWrapper.h"
#include "PmCoreRequestIf.h"
#include "LoopbackController.h"
#include "asf/core/BaseComponent.h"
#include "PhoneCallManager.h"
#include "DeviceDetailsListHandler.h"
#include "PmUtils.h"
#include "FwBluetoothStringUtils.h"
#include "PmAppClientHandler.h"
#include "PmConfiguration.h"
#include "PmAppTrace.h"

using namespace pmcore;
using namespace com::bosch::pmapp;
using namespace com::bosch::pmcommon;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_SERVICE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/IpcWrapper.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_SERVICE
#endif
#endif

#define PHONE_SUBSTATE_SET_FOR_HANDSET_MODE 0x01

namespace pm_ipc_wrapper
{

#define IPC_REQUEST_TYPE_PLACE_HOLDER 0x07
#define WAITINGMODE_NB_TONE_DEFAULT "/var/opt/bosch/static/connectivity/tones/waitingmodetones/default_en_US.mp3"
#define WAITINGMODE_WB_TONE_DEFAULT "/var/opt/bosch/static/connectivity/tones/waitingmodetones/default_en_US.mp3"

IpcWrapper::IpcWrapper() : _ipcMessageDispatcher(nullptr), _pmComponent(nullptr)
{
   ETG_TRACE_USR4(("IpcWrapper::IpcWrapper() entered"));

   com::bosch::pmcommon::LoopbackController::getInstance().setLoopbackReceiver(this);
}

IpcWrapper::~IpcWrapper()
{
   ETG_TRACE_USR4(("IpcWrapper::~IpcWrapper() entered"));
}

void IpcWrapper::setPmApplication(ahl_tclBaseOneThreadApp* mainApplication)
{
   // PmCoreRequestIf object is created here and registered the callback.
   PmInterface::getInstance().registerPmCoreCallbackIf(this);
   PmInterface::getInstance().initializePmCore(mainApplication);
}

void IpcWrapper::setPmComponent(com::bosch::PmComponent* pmComponent)
{
   _pmComponent = pmComponent;
}

void IpcWrapper::registerIpcMessageDispatcher(IpcMessageDispatcher* ipcMessageDispatcher)
{
   ETG_TRACE_USR4(("IpcWrapper::registerIpcMessageDispatcher() entered"));
   _ipcMessageDispatcher = ipcMessageDispatcher;
}

ResponseToken IpcWrapper::postLoopbackMessage(com::bosch::pmcommon::LoopbackData* loopbackData)
{
   ETG_TRACE_USR4(("IpcWrapper::postLoopbackMessage() entered"));

   ResponseToken act = 0;

   if(nullptr != _pmComponent)
   {
      act = _pmComponent->sendLocalMessage<LoopbackData*>(*_pmComponent, loopbackData);
   }
   else
   {
      ETG_TRACE_ERR(("IpcWrapper::postLoopbackMessage() PmComponent null"));
   }

   return act;
}

unsigned short int IpcWrapper::getIpcRequestType(const ActType actToken)
{
   ETG_TRACE_USR4(("IpcWrapper::getIpcRequestType() entered with actToken: %u", actToken));

   unsigned short int reqType = 0;

   reqType = actToken & IPC_REQUEST_TYPE_PLACE_HOLDER;

   ETG_TRACE_USR4(("IpcWrapper::getIpcRequestType(): IpcRequestType: %u", reqType));

   return reqType;
}

void IpcWrapper::postResponseToMessageDispatcher(const std::shared_ptr<PmCoreResponseData>& message)
{
   ETG_TRACE_USR4(("IpcWrapper::postResponseToMessageDispatcher() entered"));

   unsigned short int ipcReqType = getIpcRequestType(message->_u64TokenId);
   MessageHandlerBase* msgHandler = nullptr;

   if(CCA == ipcReqType)
   {
      msgHandler = PhoneCallManager::getInstance().getPmAppCcaServer()->getCcaMsgHandler();
   }
   else if(ASF_DBUS == ipcReqType)
   {
      // TODO: ASF DBus message handler instance to be assigned here!
      ETG_TRACE_USR4(("ASF_DBUS"));
   }

   if (msgHandler)
   {
      _ipcMessageDispatcher->dispatchMessage(msgHandler, message.get());
   }
}

void IpcWrapper::postPmCoreResponseData(const PmResult& pmResult, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::postPmCoreResponseData() entered"));

   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<PmCoreResponseData> responseMsg  = std::make_shared<PmCoreResponseData> ();
      responseMsg->_pmResult = pmResult;
      responseMsg->_u64TokenId = act;

      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

template<typename TReponseType>
void IpcWrapper::postPmCoreResponseData(const PmResult& pmResult,
      const BdAddress& deviceAddress, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::postPmCoreResponseData() entered"));

   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<TReponseType> responseMsg =
            std::make_shared<TReponseType> ();

      responseMsg->_pmResult = pmResult;
      responseMsg->_u64TokenId = act;

      if(PM_RESULT_OK == responseMsg->_pmResult._pmResultCode)
      {
         responseMsg->_deviceAddress = deviceAddress;
      }

      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

template<typename TReponseType>
void IpcWrapper::postPmCoreResponseData(const PmResult& pmResult, const BdAddress& deviceAddress,
      const CallInstance callInstance, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::postPmCoreResponseData() entered"));

   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<TReponseType> responseMsg = std::make_shared<TReponseType> ();

      responseMsg->_pmResult = pmResult;
      responseMsg->_u64TokenId = act;

      if(PM_RESULT_OK == responseMsg->_pmResult._pmResultCode)
      {
         responseMsg->_deviceAddress = deviceAddress;
         responseMsg->_callInstance = callInstance;
      }

      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

void IpcWrapper::onSystemStateChanged(const SystemState systemState)
{
   ETG_TRACE_USR4(("IpcWrapper::onSystemStateChanged() entered"));
   PmInterface::getInstance().setSystemState(systemState);
}

void IpcWrapper::setVehicleConfiguration(const VehicleConfiguration& vehicleConfiguration)
{
   ETG_TRACE_USR4(("IpcWrapper::setVehicleConfiguration() entered"));
   PmInterface::getInstance().setVehicleConfiguration(vehicleConfiguration);
}

void IpcWrapper::onDeviceDeletion(const BdAddress& deviceAddress)
{
   ETG_TRACE_USR4(("onDeviceDeletion() entered with deviceAddress: %s", deviceAddress.c_str()));
   PmInterface::getInstance().onDeviceDeleted(deviceAddress);
}

void IpcWrapper::onDeviceIdentificationChanged(const BasicDeviceDetails& basicDeviceDetails)
{
   ETG_TRACE_USR4(("IpcWrapper::onDeviceIdentificationChanged() entered"));
   PmInterface::getInstance().onDeviceIdentificationUpdated(basicDeviceDetails);
}

void IpcWrapper::onDeviceConnectionStatusChanged(const DeviceDetails& deviceDetails)
{
   ETG_TRACE_USR4(("IpcWrapper::onDeviceConnectionStatusChanged() entered"));

   BasicDeviceDetails basicDeviceDetails(deviceDetails._deviceAddress, deviceDetails._deviceHandle,
         deviceDetails._deviceIdentification);

   // TODO: Need to check the error handling of the function call "setActivePhone" to PM core
   switch (deviceDetails._deviceConnectionStatus)
   {
      case DEVICE_HFP_CONNECTED:
      {
         PmInterface::getInstance().onDeviceConnected(basicDeviceDetails);
         IpcWrapper::getInstance().onDeviceFriendlyNameChanged(deviceDetails._deviceAddress,
               deviceDetails._deviceName);
      }
      break;

      case DEVICE_HFP_DISCONNECTED:
      {
         PmInterface::getInstance().onDeviceDisconnected(basicDeviceDetails._deviceAddress);

         // On device disconnection, all the AG specific values, such as Battery level, signal strength, etc..
         // should be updated to the clients with the Default values including DeviceName.
         IpcWrapper::getInstance().onDeviceFriendlyNameChanged(basicDeviceDetails._deviceAddress, FriendlyName(""));

         // VR status should be updated for CCA specific properties
         std::shared_ptr<VoiceRecognitionPropertyUpdate> btvrExtendedStatusresponseMsg =
               std::make_shared<VoiceRecognitionPropertyUpdate>
         (PmResult(), DEFAULT_ACT, PM_PROPERTYID_BTDEVICEVOICERECOGNITIONEXTENDED, basicDeviceDetails._deviceAddress,
               false, VoiceRecognitionStatus(), EnhancedVoiceRecognitionFeature(), VR_AVAILABILITY_STATUS);

         _ipcMessageDispatcher->updatePropertyChanged(btvrExtendedStatusresponseMsg.get());

         std::shared_ptr<VoiceRecognitionPropertyUpdate> btvrStatusresponseMsg =
               std::make_shared<VoiceRecognitionPropertyUpdate>
         (PmResult(), DEFAULT_ACT, PM_PROPERTYID_BTDEVICEVOICERECOGNITION, basicDeviceDetails._deviceAddress,
               false, VoiceRecognitionStatus(), EnhancedVoiceRecognitionFeature(), VR_AVAILABILITY_STATUS);

         _ipcMessageDispatcher->updatePropertyChanged(btvrStatusresponseMsg.get());
      }
      break;

      case DEVICE_NOT_HFP_CONNECTED:
      {
         PmInterface::getInstance().onDevicePaired(basicDeviceDetails);
      }
      break;

      default:
         ETG_TRACE_USR4(("deviceDetails._deviceConnectionStatus: %u", deviceDetails._deviceConnectionStatus));
   }
}

PmResult IpcWrapper::getSystemWideRingtonesList(const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getSystemWideRingtonesList() entered"));
   return (PmInterface::getInstance().getSystemWideRingtonesList(act));
}

PmResult IpcWrapper::getDeviceRingtoneList(const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getDeviceRingtoneList() entered"));
   return (PmInterface::getInstance().getDeviceRingtoneList(act));
}

PmResult IpcWrapper::getSuppressRingtoneOnOffList(const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getSuppressRingtoneOnOffList() entered"));
   return (PmInterface::getInstance().getSuppressRingtoneOnOffList(act));
}

PmResult IpcWrapper::postSuppressRingtoneOnOffRequest2PmCore(const BdAddress& bdAddress,
      const SuppressRingtoneState suppressRingtoneState, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::postSuppressRingtoneOnOffRequest2PmCore() entered"));
   return PmInterface::getInstance().suppressRingtoneOnOff(bdAddress, suppressRingtoneState, act);
}

void IpcWrapper::suppressRingtoneOnOffResponse(IN const PmResult& pmResult, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::suppressRingtoneOnOffResponse() entered"));
   postPmCoreResponseData(pmResult, act);
}

PmResult IpcWrapper::getAutoWaitingModeOnOffList(const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getAutoWaitingModeOnOffList() entered"));
   return PmInterface::getInstance().getAutoWaitingModeOnOffList(act);
}

PmResult IpcWrapper::postAutoWaitingModeOnOffRequest2PmCore(const BdAddress& bdAddress,
      const AutoWaitingModeState autoWaitingModeState, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::postAutoWaitingModeOnOffRequest2PmCore() entered"));
   return (PmInterface::getInstance().autoWaitingModeOnOff(bdAddress, autoWaitingModeState, act));
}

void IpcWrapper::autoWaitingModeOnOffResponse(IN const PmResult& pmResult, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::autoWaitingModeOnOffResponse() entered"));
   postPmCoreResponseData(pmResult, act);
}

PmResult IpcWrapper::getWaitingModeFilePath(const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getWaitingModeFilePath() entered"));
   return (PmInterface::getInstance().getWaitingModeFilePath(act));
}

PmResult IpcWrapper::setWaitingModeFilePath(const FilePath& filePathNB, const FilePath& filePathWB, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::setWaitingModeFilePath() entered"));

   FilePath completefilePathNB = "";
   FilePath completefilePathWB = "";

   ETG_TRACE_USR4(("filePathNB: %s", filePathNB.c_str()));
   ETG_TRACE_USR4(("filePathWB: %s", filePathWB.c_str()));

   validateWaitingModeFilePath(filePathNB, completefilePathNB);
   if (completefilePathNB.empty())
   {
      completefilePathNB = WAITINGMODE_NB_TONE_DEFAULT;
   }

   validateWaitingModeFilePath(filePathWB, completefilePathWB);
   if (completefilePathWB.empty())
   {
      completefilePathWB = WAITINGMODE_WB_TONE_DEFAULT;
   }

   ETG_TRACE_USR4(("completefilePathNB: %s", completefilePathNB.c_str()));
   ETG_TRACE_USR4(("completefilePathWB: %s", completefilePathWB.c_str()));

   WaitingModeFilePath waitingModeFilePath;
   waitingModeFilePath._filePathNB = completefilePathNB;
   waitingModeFilePath._filePathWB = completefilePathWB;

   return PmInterface::getInstance().setWaitingModeFilePath(waitingModeFilePath, act);
}

PmResult IpcWrapper::getPhoneCallAudioActive(const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getPhoneCallAudioActive() entered"));
   return (PmInterface::getInstance().getPhoneCallAudioActive(act));
}

PmResult IpcWrapper::getActivePassiveDeviceList(const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getActivePassiveDeviceList() entered"));
   return (PmInterface::getInstance().getActivePassiveDeviceList(act));
}

PmResult IpcWrapper::postSetActivePhoneRequest2PmCore(const BdAddressList& deviceAddressList, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::postsetActivePhoneRequest2PmCore() entered"));

   PmResult pmResult;
   BdAddressList deviceAddressListtoPmCore = deviceAddressList;

   // removes all empty strings
   deviceAddressListtoPmCore.erase( std::remove( deviceAddressListtoPmCore.begin(),
         deviceAddressListtoPmCore.end(), "" ), deviceAddressListtoPmCore.end() );

   if (!deviceAddressListtoPmCore.empty())
   {
      pmResult = PmInterface::getInstance().setActivePhone(deviceAddressListtoPmCore, act);
   }
   else
   {
      pmResult._pmResultCode = PM_RESULT_ERR_DEVICE_NOT_EXIST;
   }

   return pmResult;
}

void IpcWrapper::setActivePhoneResponse(IN const PmResult& pmResult, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::setActivePhoneResponse() entered"));
   postPmCoreResponseData(pmResult, act);
}

PmResult IpcWrapper::postRestoreDefaultSettingRequest2PmCore(const BdAddress& deviceAddress, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::postRestoreDefaultSettingRequest2PmCore() entered"));
   return (PmInterface::getInstance().restoreDefaultSetting(deviceAddress, act));
}

PmResult IpcWrapper::getSupportedFeatures(const BdAddress& deviceAddress, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getSupportedFeatures() entered"));
   return (PmInterface::getInstance().getSupportedFeatures(deviceAddress, act));
}

PmResult IpcWrapper::getNetworkStatus(const BdAddress& deviceAddress, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getNetworkStatus() entered"));
   return (PmInterface::getInstance().getNetworkStatus(deviceAddress, act));
}

PmResult IpcWrapper::getDeviceFriendlyName(IN const BdAddress& deviceAddress,
      IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getDeviceFriendlyName() entered"));

   PmResult pmResult;

   DeviceDetails deviceDetails;
   ::com::bosch::pmapp::DeviceDetailsListHandler::getInstance().getDeviceDetails(deviceAddress, deviceDetails);
   getDeviceFriendlyNameResponse(deviceAddress, deviceDetails._deviceName, act);

   return pmResult;
}

PmResult IpcWrapper::getSignalStrength(const BdAddress& deviceAddress, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getSignalStrength() entered"));
   return (PmInterface::getInstance().getSignalStrength(deviceAddress, act));
}

PmResult IpcWrapper::getBatteryChargeLevel(const BdAddress& deviceAddress, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getBatteryChargeLevel() entered"));
   return (PmInterface::getInstance().getBatteryChargeLevel(deviceAddress, act));
}

PmResult IpcWrapper::getNetworkOperator(const BdAddress& deviceAddress, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getNetworkOperator() entered"));
   return (PmInterface::getInstance().getNetworkOperator(deviceAddress, act));
}

PmResult IpcWrapper::getVoiceMailList(const BdAddress& deviceAddress, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getVoiceMailList() entered"));
   return (PmInterface::getInstance().voiceMailList(deviceAddress, act));
}

void IpcWrapper::voiceMailListResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
      IN const VoiceMailList& voiceMailList, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::voiceMailListResponse() entered"));

   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<VoiceMailListResponseMsg> responseMsg =
            std::make_shared<VoiceMailListResponseMsg> ();

      responseMsg->_pmResult = pmResult;
      responseMsg->_u64TokenId = act;

      if(PM_RESULT_OK == responseMsg->_pmResult._pmResultCode)
      {
         responseMsg->_deviceAddress = deviceAddress;
         responseMsg->_voiceMailList = voiceMailList;
      }

      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

PmResult IpcWrapper::getCallStatusList(const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getCallStatusList() entered"));
   return (PmInterface::getInstance().getCallStatusList(act));
}

PmResult IpcWrapper::getMicrophoneMuteState(const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getMicrophoneMuteState() entered"));
   return (PmInterface::getInstance().getMicrophoneMuteState(act));
}

PmResult IpcWrapper::setMicrophoneMuteState(const MuteState microphoneMuteState, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::setMicrophoneMuteState() entered"));
   return (PmInterface::getInstance().setMicrophoneMuteState(microphoneMuteState, act));
}

PmResult IpcWrapper::postMergeCallsRequest2PmCore(const DeviceCallInstanceListMap& mergeCallList, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::postMergeCallsRequest2PmCore() entered"));

   // TODO: Received call instance validity should be checked in PmCoreRequestIf.
   return (PmInterface::getInstance().mergeCalls(mergeCallList, act));
}

void IpcWrapper::mergeCallsResponse(const PmResult& pmResult, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::mergeCallsResponse() entered"));
   postPmCoreResponseData(pmResult, act);
}

PmResult IpcWrapper::postSplitCallsRequest2PmCore(const DeviceCallInstanceListMap& splitCallList, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::postSplitCallsRequest2PmCore() entered"));

   // TODO: Received call instance validity should be checked in PmCoreRequestIf.
   return (PmInterface::getInstance().splitCalls(splitCallList, act));
}

void IpcWrapper::splitCallsResponse(IN const PmResult& pmResult, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::splitCallsResponse() entered"));
   postPmCoreResponseData(pmResult, act);
}

PmResult IpcWrapper::postHangupCallsRequest2PmCore(const DeviceCallInstanceListMap hangupCallList, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::postHangupCallsRequest2PmCore() entered"));

   // TODO: Received call instance validity should be checked in PmCoreRequestIf.
   return (PmInterface::getInstance().hangupCalls(hangupCallList, act));
}

void IpcWrapper::hangupCallsResponse(const PmResult& pmResult, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::hangupCallsResponse() entered"));
   postPmCoreResponseData(pmResult, act);
}

PmResult IpcWrapper::postRedialRequest2PmCore(const BdAddress& deviceAddress,
      const HideCallerId hideCallerId, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::postRedialRequest2PmCore() entered"));
   return (PmInterface::getInstance().redial(deviceAddress, hideCallerId, act));
}

void IpcWrapper::redialResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
      IN const CallInstance callInstance, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::redialResponse() entered"));
   postPmCoreResponseData<RedialResponseMsg>(pmResult, deviceAddress, callInstance, act);
}

PmResult IpcWrapper::getSCOConnection(const BdAddress& deviceAddress, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getSCOConnection() entered"));
   return (PmInterface::getInstance().getSCOConnection(deviceAddress, act));
}

PmResult IpcWrapper::getRingtoneMuteState(const BdAddress& deviceAddress, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getRingtoneMuteState() entered"));
   return (PmInterface::getInstance().getRingtoneMuteState(deviceAddress, act));
}

PmResult IpcWrapper::setRingtoneMuteState(const BdAddress& deviceAddress,
      const MuteState ringtoneMuteState, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::setRingtoneMuteState() entered"));
   return (PmInterface::getInstance().setRingtoneMuteState(deviceAddress, ringtoneMuteState, act));
}

PmResult IpcWrapper::getWaitingModeState(const BdAddress& deviceAddress, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getWaitingModeState() entered"));
   return (PmInterface::getInstance().getWaitingModeState(deviceAddress, act));
}

PmResult IpcWrapper::getRingtonePlaybackStatus(const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getRingtonePlaybackStatus() entered"));
   return (PmInterface::getInstance().getRingtonePlaybackStatus(act));
}

PmResult IpcWrapper::postStartStopWaitingModeRequest2PmCore(const BdAddress& deviceAddress,
      const StartStop startStopWaitingMode, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::postStartStopWaitingModeRequest2PmCore() entered"));
   return (PmInterface::getInstance().startStopWaitingMode(deviceAddress, startStopWaitingMode, act));
}

void IpcWrapper::startStopWaitingModeResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
      IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::startStopWaitingModeResponse() entered"));
   postPmCoreResponseData<StartStopwaitingModeResponseMsg>(pmResult, deviceAddress, act);
}

PmResult IpcWrapper::postDialRequest2PmCore(const BdAddress& deviceAddress, const DialInfo& dialInfo,
      const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::postDialRequest2PmCore() entered"));

   PmResult pmResult;

   std::string validTelephoneNumber = "";

   ETG_TRACE_USR4(("dialInfo._telephoneNumber: %s", dialInfo._telephoneNumber.c_str()));
   validateDialNumber(dialInfo._telephoneNumber, validTelephoneNumber);
   ETG_TRACE_USR4(("validTelephoneNumber: %s", validTelephoneNumber.c_str()));

   if (!validTelephoneNumber.empty())
   {
      pmResult = PmInterface::getInstance().dial(deviceAddress, validTelephoneNumber, dialInfo._hideCallerId, act);

      // Record the Name received which should be used
      // while updating the call status or outgoing call failed event
      if (!dialInfo._firstName.empty())
      {
         QueriedContactInfo queriedContactInfo(dialInfo._firstName, dialInfo._lastName, CONTACT_HANDLE_DEFAULT);
         DeviceDetailsListHandler::getInstance().setContactInfoFromClients(deviceAddress, validTelephoneNumber,
               act, queriedContactInfo);
      }
   }
   else
   {
      pmResult._pmResultCode = PM_RESULT_ERR_INVALID_PARAMETER;
   }

   return pmResult;
}

void IpcWrapper::dialResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
      IN const CallInstance callInstance, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::dialResponse() entered"));
   postPmCoreResponseData<DialResponseMsg>(pmResult, deviceAddress, callInstance, act);

   if(PM_RESULT_OK != pmResult._pmResultCode)
   {
      QueriedContactInfo queriedContactInfo;
      TelephoneNumber telephoneNumber = "";
      DeviceDetailsListHandler::getInstance().resetContactInfoFromClientsByAct(deviceAddress, act,
            telephoneNumber, queriedContactInfo);

      // This telephone number info might be needed while updating the Dial response Error to the clients.
      ETG_TRACE_USR4(("IpcWrapper::dialResponse(): telephoneNumber: %s", telephoneNumber.c_str()));
   }
}

PmResult IpcWrapper::postAcceptRequest2PmCore(const CallInstance callInstance, const ActType act,
      const BdAddress& devAddress, const AcceptOperation operation)
{
   ETG_TRACE_USR4(("IpcWrapper::postAcceptRequest2PmCore() entered"));
   return (PmInterface::getInstance().acceptCall(devAddress, callInstance, operation, act));
}

void IpcWrapper::acceptCallResponse(const PmResult& pmResult, const BdAddress& deviceAddress,
      const CallInstance callInstance, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::acceptCallResponse() entered"));
   postPmCoreResponseData<AcceptCallResponseMsg>(pmResult, deviceAddress, callInstance, act);
}

PmResult IpcWrapper::postSwapCallRequest2PmCore(const BdAddress& deviceAddress,
      const SwapOperation operation, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::postSwapCallRequest2PmCore() entered"));
   return (PmInterface::getInstance().swapCall(deviceAddress, operation, act));
}

void IpcWrapper::swapCallResponse(const PmResult& pmResult, const BdAddress& deviceAddress, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::swapCallResponse() entered"));
   postPmCoreResponseData<SwapCallResponseMsg>(pmResult, deviceAddress, act);
}

PmResult IpcWrapper::postSpeedDialRequest2PmCore(const BdAddress& deviceAddress, const MemDialIndex memDialIndex,
      const HideCallerId hideCallerId, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::postSpeedDialRequest2PmCore() entered"));
   return (PmInterface::getInstance().speedDial(deviceAddress, memDialIndex, hideCallerId, act));
}

void IpcWrapper::speedDialResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
      IN const CallInstance callInstance, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::speedDialResponse() entered"));
   postPmCoreResponseData<SpeedDialResponseMsg>(pmResult, deviceAddress, callInstance, act);
}

PmResult IpcWrapper::sendDTMFRequest2PmCore(const BdAddress& deviceAddress, const DTMFTones& tones, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::sendDTMFRequest2PmCore() entered"));
   return (PmInterface::getInstance().sendDTMF(deviceAddress, tones, act));
}

void IpcWrapper::sendDTMFResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
      IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::sendDTMFResponse() entered"));
   postPmCoreResponseData<SendDTMFResponseMsg>(pmResult, deviceAddress, act);
}

PmResult IpcWrapper::postTransferAudioRequest2PmCore(const BdAddress& deviceAddress,
      const AudioDirection audioDirection, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::postTransferAudioRequest2PmCore() entered"));
   return (PmInterface::getInstance().transferAudio(deviceAddress, audioDirection, act));
}

void IpcWrapper::transferAudioResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
      IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::transferAudioResponse() entered"));
   postPmCoreResponseData<TransferAudioResponseMsg>(pmResult, deviceAddress, act);
}

PmResult IpcWrapper::getVoiceRecognitionStatus(const BdAddress& deviceAddress,
      const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getVoiceRecognitionStatus() entered"));
   return (PmInterface::getInstance().getVoiceRecognitionStatus(deviceAddress, act));
}

PmResult IpcWrapper::postStartStopVoiceRecognitionRequest2PmCore(const BdAddress& deviceAddress,
      const StartStop startStopVR, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::postStartStopVoiceRecognitionRequest2PmCore() entered"));
   return (PmInterface::getInstance().startStopVoiceRecognition(deviceAddress, startStopVR, act));
}

PmResult IpcWrapper::getEnhancedVoiceRecognitionFeature(const BdAddress& deviceAddress,
      const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getEnhancedVoiceRecognitionFeature() entered"));
   return (PmInterface::getInstance().getEnhancedVoiceRecognitionFeature(deviceAddress, act));
}

PmResult IpcWrapper::getExternalVRState(const BdAddress& deviceAddress, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getExternalVRState() entered"));
   return (PmInterface::getInstance().getExternalVRState(deviceAddress, act));
}

PmResult IpcWrapper::postActivateExternalVRRequest2PmCore(const BdAddress& deviceAddress,
      const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::postActivateExternalVRRequest2PmCore() entered"));
   return (PmInterface::getInstance().activateExternalVR(deviceAddress, act));
}

PmResult IpcWrapper::postDeActivateExternalVRRequest2PmCore(const BdAddress& deviceAddress,
      const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::postDeActivateExternalVRRequest2PmCore() entered"));
   return (PmInterface::getInstance().deActivateExternalVR(deviceAddress, act));
}

void IpcWrapper::stopExtendedVRSession(IN const BdAddress& deviceAddress)
{
   ETG_TRACE_USR4(("stopExtendedVRSession for deviceAddress: %s", deviceAddress.c_str()));
   PmInterface::getInstance().stopExtendedVR(deviceAddress);
}

void IpcWrapper::getSystemWideRingtonesListResponse(IN const PmResult& pmResult,
      IN const SystemWideRingtonesList& systemWideRingtonesList, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getSystemWideRingtonesListResponse() entered"));

   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<SystemWideRingtonesListPropertyUpdate> responseMsg =
            std::make_shared<SystemWideRingtonesListPropertyUpdate>
      (pmResult, act, PM_PROPERTYID_SYSTEMWIDERINGTONESLIST, systemWideRingtonesList);

      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

void IpcWrapper::onSystemWideRingtonesListChanged(IN const SystemWideRingtonesList& systemWideRingtonesList)
{
   ETG_TRACE_USR4(("IpcWrapper::onSystemWideRingtonesListChanged() entered"));

   std::shared_ptr<SystemWideRingtonesListPropertyUpdate> responseMsg =
         std::make_shared<SystemWideRingtonesListPropertyUpdate>
   (PmResult(), DEFAULT_ACT, PM_PROPERTYID_SYSTEMWIDERINGTONESLIST, systemWideRingtonesList);

   _ipcMessageDispatcher->updatePropertyChanged(responseMsg.get());
}

void IpcWrapper::getDeviceRingtoneListResponse(IN const PmResult& pmResult,
      IN const DeviceRingtoneList& deviceRingtoneList, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getDeviceRingtoneListResponse() entered"));

   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<DeviceRingtoneListPropertyUpdate> responseMsg =
            std::make_shared<DeviceRingtoneListPropertyUpdate>
      (pmResult, act, PM_PROPERTYID_DEVICERINGTONELIST, deviceRingtoneList);

      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

void IpcWrapper::onDeviceRingtoneListChanged(IN const DeviceRingtoneList& deviceRingtoneList)
{
   ETG_TRACE_USR4(("IpcWrapper::onDeviceRingtoneListChanged() entered"));

   std::shared_ptr<DeviceRingtoneListPropertyUpdate> responseMsg =
         std::make_shared<DeviceRingtoneListPropertyUpdate>
   (PmResult(), DEFAULT_ACT, PM_PROPERTYID_DEVICERINGTONELIST, deviceRingtoneList);

   _ipcMessageDispatcher->updatePropertyChanged(responseMsg.get());
}

PmResult IpcWrapper::postSetRingtoneRequest2PmCore(const BdAddress& deviceAddress,
      const RingtoneId ringtoneId, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::postSetRingtoneRequest2PmCore() entered"));
   return (PmInterface::getInstance().setRingtone(deviceAddress, ringtoneId, act));
}

void IpcWrapper::setRingtoneResponse(IN const PmResult& pmResult, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::setRingtoneResponse() entered"));
   postPmCoreResponseData(pmResult, act);
}

PmResult IpcWrapper::postPlayRingtoneRequest2PmCore(const RingtoneId ringtoneId,
      const PlayCount playCount, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::postPlayRingtoneRequest2PmCore() entered"));
   return (PmInterface::getInstance().playRingtone(ringtoneId, playCount, act));
}

void IpcWrapper::playRingtoneResponse(IN const PmResult& pmResult, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::playRingtoneResponse() entered"));
   postPmCoreResponseData(pmResult, act);
}

PmResult IpcWrapper::postStopRingtoneRequest2PmCore(const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::postStopRingtoneRequest2PmCore() entered"));
   return (PmInterface::getInstance().stopRingtone(act));
}

void IpcWrapper::stopRingtoneResponse(IN const PmResult& pmResult, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::stopRingtoneResponse() entered"));
   postPmCoreResponseData(pmResult, act);
}

void IpcWrapper::getSuppressRingtoneOnOffListResponse(IN const PmResult& pmResult,
      IN const SuppressRingtoneOnOffList& suppressRingtoneOnOffList, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getSuppressRingtoneOnOffListResponse() entered"));

   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<SuppressRingtoneOnOffListPropertyUpdate> responseMsg =
            std::make_shared<SuppressRingtoneOnOffListPropertyUpdate>
      (pmResult, act, PM_PROPERTYID_SUPPRESSRINGTONEONOFFLIST, suppressRingtoneOnOffList);

      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

void IpcWrapper::onSuppressRingtoneOnOffListChanged(IN const SuppressRingtoneOnOffList& suppressRingtoneOnOffList)
{
   ETG_TRACE_USR4(("IpcWrapper::onSuppressRingtoneOnOffListChanged() entered"));

   std::shared_ptr<SuppressRingtoneOnOffListPropertyUpdate> responseMsg =
         std::make_shared<SuppressRingtoneOnOffListPropertyUpdate>
   (PmResult(), DEFAULT_ACT, PM_PROPERTYID_SUPPRESSRINGTONEONOFFLIST, suppressRingtoneOnOffList);

   _ipcMessageDispatcher->updatePropertyChanged(responseMsg.get());
}

void IpcWrapper::getAutoWaitingModeOnOffListResponse(IN const PmResult& pmResult,
      IN const AutoWaitingModeOnOffList& autoWaitingModeOnOffList, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getAutoWaitingModeOnOffListResponse() entered"));

   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<AutoWaitingModeOnOffListPropertyUpdate> responseMsg =
            std::make_shared<AutoWaitingModeOnOffListPropertyUpdate>
      (pmResult, act, PM_PROPERTYID_AUTOWAITINGMODEONOFFLIST, autoWaitingModeOnOffList);

      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

void IpcWrapper::onAutoWaitingModeOnOffListChanged(IN AutoWaitingModeOnOffList& autoWaitingModeOnOffList)
{
   ETG_TRACE_USR4(("IpcWrapper::onAutoWaitingModeOnOffListChanged() entered"));

   std::shared_ptr<AutoWaitingModeOnOffListPropertyUpdate> responseMsg =
         std::make_shared<AutoWaitingModeOnOffListPropertyUpdate>
   (PmResult(), DEFAULT_ACT, PM_PROPERTYID_AUTOWAITINGMODEONOFFLIST, autoWaitingModeOnOffList);

   _ipcMessageDispatcher->updatePropertyChanged(responseMsg.get());
}

void IpcWrapper::setWaitingModeFilePathResponse(IN const PmResult& pmResult, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::setWaitingModeFilePathResponse() entered"));
   dispatchWaitingModeFilePathResponse(pmResult, WaitingModeFilePath(), act);
}

void IpcWrapper::getWaitingModeFilePathResponse(IN const PmResult& pmResult,
      IN const WaitingModeFilePath& waitingModeFilePath, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getWaitingModeFilePathResponse() entered"));
   dispatchWaitingModeFilePathResponse(pmResult, waitingModeFilePath, act);
}

void IpcWrapper::onWaitingModeFilePathChanged(IN const WaitingModeFilePath& waitingModeFilePath)
{
   ETG_TRACE_USR4(("IpcWrapper::onWaitingModeFilePathChanged() entered"));

   std::shared_ptr<WaitingModeFilePathPropertyUpdate> responseMsg =
         std::make_shared<WaitingModeFilePathPropertyUpdate>
   (PmResult(), DEFAULT_ACT, PM_PROPERTYID_WAITINGMODEFILEPATH, waitingModeFilePath);

   _ipcMessageDispatcher->updatePropertyChanged(responseMsg.get());
}

void IpcWrapper::dispatchWaitingModeFilePathResponse(const PmResult& pmResult,
      const WaitingModeFilePath& waitingModeFilePath, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::dispatchWaitingModeFilePathResponse() entered"));

   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<WaitingModeFilePathPropertyUpdate> responseMsg =
            std::make_shared<WaitingModeFilePathPropertyUpdate>
      (pmResult, act, PM_PROPERTYID_WAITINGMODEFILEPATH, waitingModeFilePath);

      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

void IpcWrapper::getPhoneCallAudioActiveResponse(IN const PmResult& pmResult,
      IN const PhoneCallAudioActive& phoneCallAudioActive, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getPhoneCallAudioActiveResponse() entered"));

   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<PhoneCallAudioPropertyUpdate> responseMsg = std::make_shared<PhoneCallAudioPropertyUpdate>
      (pmResult, act, PM_PROPERTYID_VEHICLEAUDIO, phoneCallAudioActive);
      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

void IpcWrapper::onPhoneCallAudioActiveChanged(IN const PhoneCallAudioActive& phoneCallAudioActive)
{
   ETG_TRACE_USR4(("IpcWrapper::onPhoneCallAudioActiveChanged() entered"));

   std::shared_ptr<PhoneCallAudioPropertyUpdate> responseMsgVehicleAudio =
         std::make_shared<PhoneCallAudioPropertyUpdate>
   (PmResult(), DEFAULT_ACT, PM_PROPERTYID_VEHICLEAUDIO, phoneCallAudioActive);

   _ipcMessageDispatcher->updatePropertyChanged(responseMsgVehicleAudio.get());

   std::shared_ptr<PhoneCallAudioPropertyUpdate> responseMsgActiveCallinHandset =
         std::make_shared<PhoneCallAudioPropertyUpdate>
   (PmResult(), DEFAULT_ACT, PM_PROPERTYID_ACTIVECALLINHANDSET, phoneCallAudioActive);

   _ipcMessageDispatcher->updatePropertyChanged(responseMsgActiveCallinHandset.get());
}

void IpcWrapper::getActivePassiveDeviceListResponse(IN const PmResult& pmResult,
      IN const ActivePassiveDeviceList& activePassiveDeviceList, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getActivePassiveDeviceListResponse() entered"));

   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<ActivePassiveDeviceListPropertyUpdate> responseMsg =
            std::make_shared<ActivePassiveDeviceListPropertyUpdate>
      (pmResult, act, PM_PROPERTYID_ACTIVEPHONEDEVICE, activePassiveDeviceList);

      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

void IpcWrapper::onActivePassiveDeviceListChanged(IN const ActivePassiveDeviceList& activePassiveDeviceList)
{
   ETG_TRACE_USR4(("IpcWrapper::onActivePassiveDeviceListChanged() entered"));

   ::com::bosch::pmapp::DeviceDetailsListHandler::getInstance().setDeviceRole
         (activePassiveDeviceList._activePassiveDeviceList);

   // TODO: Need to check the possibility of updating SLC status from onDeviceConnectionStatusChanged() and onDeviceDeleted().
   // Also need to check whether it is possible to remove this paramater.
   // If this parameter is needed, add a member varaible so that same value of SLC status shall not be updated again.

   std::shared_ptr<ActivePassiveDeviceListPropertyUpdate> activePhoneDeviceResponseMsg =
         std::make_shared<ActivePassiveDeviceListPropertyUpdate>
   (PmResult(), DEFAULT_ACT, PM_PROPERTYID_ACTIVEPHONEDEVICE, activePassiveDeviceList);

   _ipcMessageDispatcher->updatePropertyChanged(activePhoneDeviceResponseMsg.get());

   std::shared_ptr<ActivePassiveDeviceListPropertyUpdate> slcStatusResponseMsg =
         std::make_shared<ActivePassiveDeviceListPropertyUpdate>
   (PmResult(), DEFAULT_ACT, PM_PROPERTYID_SLCSTATUS, activePassiveDeviceList);

   _ipcMessageDispatcher->updatePropertyChanged(slcStatusResponseMsg.get());
}

void IpcWrapper::restoreDefaultSettingResponse(IN const PmResult& pmResult, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::restoreDefaultSettingResponse() entered"));
   (void) pmResult;
   (void) act;

   PmAppClientHandler* pmAppClientHandler = PhoneCallManager::getInstance().getPmAppClientHandler();

   if (pmAppClientHandler)
   {
      IDiagSystemSettings* iDiagSystemSettings = pmAppClientHandler->getDiagSystemSettings();

      if (iDiagSystemSettings)
      {
         iDiagSystemSettings->onRestoreDefaultSettingResponse((PM_RESULT_OK == pmResult._pmResultCode), act);
      }
   }
}

void IpcWrapper::onNotificationEvent(IN const NotificationEvent& notificationEvent)
{
   ETG_TRACE_USR4(("IpcWrapper::onNotificationEvent() entered with notificationEvent._eventName: %u",
         ETG_CENUM(NotificationEventName, notificationEvent._eventName)));

   DeviceDetails deviceDetails;

   // device address validity check
   if (DeviceDetailsListHandler::getInstance().getDeviceDetails(notificationEvent._bdAddress, deviceDetails))
   {
      std::string validTelephoneNumber = "";
      validateDialNumber(notificationEvent._telephoneNumber, validTelephoneNumber);

      unsigned short int propertyId = PROPERTY_ID_INVALID;
      std::map<std::string, std::string> eventMap;

      eventMap.emplace_hint(eventMap.end(), "DeviceAddress", notificationEvent._bdAddress);
      eventMap.emplace_hint(eventMap.end(), "TelephoneNumber", validTelephoneNumber);
      eventMap.emplace_hint(eventMap.end(), "FailureReason", notificationEvent._reason);

      switch (notificationEvent._eventName)
      {
         case INCOMING_CALL_REJECTED:
         {
            if (!validTelephoneNumber.empty())
            {
               ETG_TRACE_USR4(("IpcWrapper::onNotificationEvent() entered with validTelephoneNumber"));

               eventMap.emplace_hint(eventMap.end(), "EventName", "IncomingCallRejectedEvent");

               propertyId = PM_PROPERTYID_INCOMINGCALLREJECTEDEVENT;
            }
         }
         break;

         case OUTGOING_CALL_FAILED:
         {
            if (!validTelephoneNumber.empty())
            {
               eventMap.emplace_hint(eventMap.end(), "EventName", "OutgoingCallFailedEvent");
               // Name field will be filled in the corresponding Property class.

               propertyId = PM_PROPERTYID_OUTGOINGCALLFAILEDEVENT;
            }
         }
         break;

         case SPEED_DIAL_FAILED:
         {
            eventMap.emplace_hint(eventMap.end(), "EventName", "OutgoingCallFailedEvent");
            // Name field will be filled in the corresponding Property class.

            propertyId = PM_PROPERTYID_OUTGOINGCALLFAILEDEVENT;
         }
         break;

         case ACCEPT_CALL_FAILED:
         {
            eventMap.emplace_hint(eventMap.end(), "EventName", "AcceptCall Failed");

            propertyId = PM_PROPERTYID_CALLOPERATIONFAILEDEVENT;
         }
         break;

         case SWAP_CALL_FAILED:
         {
            eventMap.emplace_hint(eventMap.end(), "EventName", "SwapCall Failed");

            propertyId = PM_PROPERTYID_CALLOPERATIONFAILEDEVENT;
         }
         break;

         case CONFERENCE_CALL_FAILED:
         {
            eventMap.emplace_hint(eventMap.end(), "EventName", "ConferenceCall Failed");

            propertyId = PM_PROPERTYID_CALLOPERATIONFAILEDEVENT;
         }
         break;

         case SPLIT_CALL_FAILED:
         {
            eventMap.emplace_hint(eventMap.end(), "EventName", "SplitCall Failed");

            propertyId = PM_PROPERTYID_CALLOPERATIONFAILEDEVENT;
         }
         break;

         case HANGUP_CALL_FAILED:
         {
            eventMap.emplace_hint(eventMap.end(), "EventName", "HangupCall Failed");

            propertyId = PM_PROPERTYID_CALLOPERATIONFAILEDEVENT;
         }
         break;

         case TRANSFER_CALL_AUDIO_TO_AG_FAILED:
         {
            eventMap.emplace_hint(eventMap.end(), "EventName", "TransferCallAudioToAG Failed");

            propertyId = PM_PROPERTYID_AUDIOTRANSFERFAILEDEVENT;
         }
         break;

         case TRANSFER_CALL_AUDIO_TO_HF_FAILED:
         {
            eventMap.emplace_hint(eventMap.end(), "EventName", "TransferCallAudioToHF Failed");

            propertyId = PM_PROPERTYID_AUDIOTRANSFERFAILEDEVENT;
         }
         break;

         case SEND_DTMF_FAILED:
         {
            eventMap.emplace_hint(eventMap.end(), "EventName", "SendDTMF Failed");

            propertyId = PM_PROPERTYID_SENDDTMFFAILEDEVENT;
         }
         break;

         default:
         {
            ETG_TRACE_USR4(("IpcWrapper::onNotificationEvent() Default"));
         }
      }

      if (PROPERTY_ID_INVALID != propertyId)
      {
         std::shared_ptr<PmNotificationEventPropertyUpdate> responseMsg =
               std::make_shared<PmNotificationEventPropertyUpdate>
         (PmResult(), DEFAULT_ACT, propertyId, eventMap, notificationEvent._callInstanceList);

         _ipcMessageDispatcher->updatePropertyChanged(responseMsg.get());

         responseMsg = std::make_shared<PmNotificationEventPropertyUpdate>
         (PmResult(), DEFAULT_ACT, PM_PROPERTYID_NOTIFICATIONEVENT, eventMap, notificationEvent._callInstanceList);

         _ipcMessageDispatcher->updatePropertyChanged(responseMsg.get());
      }
   }
   else
   {
      ETG_TRACE_USR4(("IpcWrapper::DeviceDetails not found for :%s",notificationEvent._bdAddress.c_str()));
   }
}

void IpcWrapper::getSupportedFeaturesResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
      IN const SupportedFeatures& supportedFeatures, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getSupportedFeaturesResponse() entered"));

   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<AgInfoPropertyUpdate<SupportedFeatures>> responseMsg =
            std::make_shared<AgInfoPropertyUpdate<SupportedFeatures>>
            (pmResult, act, PM_PROPERTYID_HANDSFREEENHANCEDCALLCONTROL, deviceAddress, supportedFeatures);

      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

void IpcWrapper::onSupportedFeaturesChanged(IN const BdAddress& deviceAddress,
      IN const SupportedFeatures& supportedFeatures)
{
   ETG_TRACE_USR4(("IpcWrapper::onSupportedFeaturesChanged() entered"));

   std::shared_ptr<AgInfoPropertyUpdate<SupportedFeatures>> hfEnhancedCallControlresponseMsg =
         std::make_shared<AgInfoPropertyUpdate<SupportedFeatures>>
         (PmResult(), DEFAULT_ACT, PM_PROPERTYID_HANDSFREEENHANCEDCALLCONTROL, deviceAddress, supportedFeatures);

   _ipcMessageDispatcher->updatePropertyChanged(hfEnhancedCallControlresponseMsg.get());

   std::shared_ptr<AgInfoPropertyUpdate<SupportedFeatures>> hfServiceresponseMsg =
         std::make_shared<AgInfoPropertyUpdate<SupportedFeatures>>
         (PmResult(), DEFAULT_ACT, PM_PROPERTYID_HFSERVICE, deviceAddress, supportedFeatures);

   _ipcMessageDispatcher->updatePropertyChanged(hfServiceresponseMsg.get());

   auto iter = std::find(supportedFeatures._supportedFeatures.begin(), supportedFeatures._supportedFeatures.end(),
         "voice_recognition");

   if (supportedFeatures._supportedFeatures.end() != iter)
   {
      std::shared_ptr<VoiceRecognitionPropertyUpdate> btvrExtendedStatusresponseMsg =
            std::make_shared<VoiceRecognitionPropertyUpdate>
      (PmResult(), DEFAULT_ACT, PM_PROPERTYID_BTDEVICEVOICERECOGNITIONEXTENDED, deviceAddress, true,
            VoiceRecognitionStatus(),
            EnhancedVoiceRecognitionFeature(),
            VR_AVAILABILITY_STATUS);

      _ipcMessageDispatcher->updatePropertyChanged(btvrExtendedStatusresponseMsg.get());

      std::shared_ptr<VoiceRecognitionPropertyUpdate> btvrStatusresponseMsg =
            std::make_shared<VoiceRecognitionPropertyUpdate>
      (PmResult(), DEFAULT_ACT, PM_PROPERTYID_BTDEVICEVOICERECOGNITION, deviceAddress, true,
            VoiceRecognitionStatus(),
            EnhancedVoiceRecognitionFeature(),
            VR_AVAILABILITY_STATUS);

      _ipcMessageDispatcher->updatePropertyChanged(btvrStatusresponseMsg.get());
   }
}

void IpcWrapper::getDeviceFriendlyNameResponse(IN const BdAddress& deviceAddress, IN const FriendlyName& friendlyName,
      IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getDeviceFriendlyNameResponse() entered"));

   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<AgInfoPropertyUpdate<FriendlyName>> responseMsg =
            std::make_shared<AgInfoPropertyUpdate<FriendlyName>>
            (PmResult(), act, PM_PROPERTYID_CELLDEVICENAME, deviceAddress, friendlyName);

      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

void IpcWrapper::onDeviceFriendlyNameChanged(IN const BdAddress& deviceAddress, IN const FriendlyName& friendlyName)
{
   ETG_TRACE_USR4(("IpcWrapper::onDeviceFriendlyNameChanged() entered"));

   std::shared_ptr<AgInfoPropertyUpdate<FriendlyName>> responseMsg =
         std::make_shared<AgInfoPropertyUpdate<FriendlyName>>
         (PmResult(), DEFAULT_ACT, PM_PROPERTYID_CELLDEVICENAME, deviceAddress, friendlyName);

   _ipcMessageDispatcher->updatePropertyChanged(responseMsg.get());
}

void IpcWrapper::getNetworkStatusResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
      IN const NetworkStatus& networkStatus, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getNetworkStatusResponse() entered"));

   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<AgInfoPropertyUpdate<NetworkStatus>> responseMsg =
            std::make_shared<AgInfoPropertyUpdate<NetworkStatus>>
            (pmResult, act, PM_PROPERTYID_CELLNETWORKSTATUS, deviceAddress, networkStatus);

      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

void IpcWrapper::onNetworkStatusChanged(IN const BdAddress& deviceAddress, IN const NetworkStatus& networkStatus)
{
   ETG_TRACE_USR4(("IpcWrapper::onNetworkStatusChanged() entered"));

   std::shared_ptr<AgInfoPropertyUpdate<NetworkStatus>> responseMsg =
         std::make_shared<AgInfoPropertyUpdate<NetworkStatus>>
         (PmResult(), DEFAULT_ACT, PM_PROPERTYID_CELLNETWORKSTATUS, deviceAddress, networkStatus);

   _ipcMessageDispatcher->updatePropertyChanged(responseMsg.get());
}

void IpcWrapper::getSignalStrengthResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
      IN const SignalStrength& signalStrength, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getSignalStrengthResponse() entered"));

   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<AgInfoPropertyUpdate<SignalStrength>> responseMsg =
            std::make_shared<AgInfoPropertyUpdate<SignalStrength>>
            (pmResult, act, PM_PROPERTYID_CELLSIGNALSTRENGTH, deviceAddress, signalStrength);

      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

void IpcWrapper::onSignalStrengthChanged(IN const BdAddress& deviceAddress,
      IN const SignalStrength& signalStrength)
{
   ETG_TRACE_USR4(("IpcWrapper::onSignalStrengthChanged() entered"));

   std::shared_ptr<AgInfoPropertyUpdate<SignalStrength>> responseMsg =
         std::make_shared<AgInfoPropertyUpdate<SignalStrength>>
         (PmResult(), DEFAULT_ACT, PM_PROPERTYID_CELLSIGNALSTRENGTH, deviceAddress, signalStrength);

   _ipcMessageDispatcher->updatePropertyChanged(responseMsg.get());
}

void IpcWrapper::getBatteryChargeLevelResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
      IN const BatteryChargeLevel& batteryChargeLevel, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getBatteryChargeLevelResponse() entered"));

   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<AgInfoPropertyUpdate<BatteryChargeLevel>> responseMsg =
            std::make_shared<AgInfoPropertyUpdate<BatteryChargeLevel>>
            (pmResult, act, PM_PROPERTYID_CELLSTATEOFCHARGE, deviceAddress, batteryChargeLevel);

      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

void IpcWrapper::onBatteryChargeLevelChanged(IN const BdAddress& deviceAddress,
      IN const BatteryChargeLevel& batteryChargeLevel)
{
   ETG_TRACE_USR4(("IpcWrapper::onBatteryChargeLevelChanged() entered"));

   std::shared_ptr<AgInfoPropertyUpdate<BatteryChargeLevel>> responseMsg =
         std::make_shared<AgInfoPropertyUpdate<BatteryChargeLevel>>
         (PmResult(), DEFAULT_ACT, PM_PROPERTYID_CELLSTATEOFCHARGE, deviceAddress, batteryChargeLevel);

   _ipcMessageDispatcher->updatePropertyChanged(responseMsg.get());
}

void IpcWrapper::getNetworkOperatorResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
      IN const NetworkOperator& networkOperator, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getNetworkOperatorResponse() entered"));

   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<AgInfoPropertyUpdate<NetworkOperator>> responseMsg =
            std::make_shared<AgInfoPropertyUpdate<NetworkOperator>>
            (pmResult, act, PM_PROPERTYID_NETWORKOPERATOR, deviceAddress, networkOperator);

      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

void IpcWrapper::onNetworkOperatorChanged(IN const BdAddress& deviceAddress,
      IN const NetworkOperator& networkOperator)
{
   ETG_TRACE_USR4(("IpcWrapper::onNetworkOperatorChanged() entered"));

   std::shared_ptr<AgInfoPropertyUpdate<NetworkOperator>> responseMsg =
         std::make_shared<AgInfoPropertyUpdate<NetworkOperator>>
         (PmResult(), DEFAULT_ACT, PM_PROPERTYID_NETWORKOPERATOR, deviceAddress, networkOperator);

   _ipcMessageDispatcher->updatePropertyChanged(responseMsg.get());
}

void IpcWrapper::getCallStatusListResponse(IN const PmResult& pmResult, IN const CallStatusList& callStatusList,
      IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getCallStatusListResponse() entered"));

   if(DEFAULT_ACT != act)
   {
      if (PM_RESULT_OK == pmResult._pmResultCode)
      {
         DeviceDetailsListHandler::getInstance().updateCallStatusListFromPmCore(callStatusList, act);
      }
      else
      {
         std::shared_ptr<CallsReportListPropertyUpdate> responseMsg =
               std::make_shared<CallsReportListPropertyUpdate>
         (pmResult, act, PM_PROPERTYID_CALLSTATUSNOTICEEXTENDED, CallsReportList());

         postResponseToMessageDispatcher(responseMsg);
      }
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

void IpcWrapper::onCallStatusListChanged(IN const CallStatusList& callStatusList)
{
   ETG_TRACE_USR4(("IpcWrapper::onCallStatusListChanged() entered"));

   DeviceDetailsListHandler::getInstance().updateCallStatusListFromPmCore(callStatusList);
}

void IpcWrapper::updateCallsReportListToClients(const CallsReportList& callsReportList, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::updateCallsReportListToClients() entered"));

   if (PM_DEFAULT_ACT == act)
   {
      //CallStatusNoticeExtended property update
      std::shared_ptr<CallsReportListPropertyUpdate> callStatusExtendedUpdateMsg =
            std::make_shared<CallsReportListPropertyUpdate>
      (PmResult(), DEFAULT_ACT, PM_PROPERTYID_CALLSTATUSNOTICEEXTENDED, callsReportList);

      _ipcMessageDispatcher->updatePropertyChanged(callStatusExtendedUpdateMsg.get());

      //CallStatusNotice property update
      std::shared_ptr<CallsReportListPropertyUpdate> callStatusUpdateMsg =
            std::make_shared<CallsReportListPropertyUpdate>
      (PmResult(), DEFAULT_ACT, PM_PROPERTYID_CALLSTATUSNOTICE, callsReportList);

      _ipcMessageDispatcher->updatePropertyChanged(callStatusUpdateMsg.get());

      std::shared_ptr<CallsReportListPropertyUpdate> incomingCallEventUpdate =
            std::make_shared<CallsReportListPropertyUpdate>
      (PmResult(), DEFAULT_ACT, PM_PROPERTYID_INCOMINGCALLEVENT, callsReportList);

      _ipcMessageDispatcher->updatePropertyChanged(incomingCallEventUpdate.get());
   }
   else
   {
      std::shared_ptr<CallsReportListPropertyUpdate> responseMsg =
            std::make_shared<CallsReportListPropertyUpdate>
      (PmResult(), act, PM_PROPERTYID_CALLSTATUSNOTICEEXTENDED, callsReportList);

      postResponseToMessageDispatcher(responseMsg);
   }
}

void IpcWrapper::getMicrophoneMuteStateResponse(IN const PmResult& pmResult,
      IN const MicrophoneMuteState& microphoneMuteState, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getMicrophoneMuteStateResponse() entered"));
   dispatchMicMuteStateResponse(pmResult, microphoneMuteState, act);
}

void IpcWrapper::onMicrophoneMuteStateChanged(IN const MicrophoneMuteState& microphoneMuteState)
{
   ETG_TRACE_USR4(("IpcWrapper::onMicrophoneMuteStateChanged() entered"));

   std::shared_ptr<MicMuteStatePropertyUpdate> responseMsg =
         std::make_shared<MicMuteStatePropertyUpdate>
   (PmResult(), DEFAULT_ACT, PM_PROPERTYID_MICROPHONEMUTESTATE, microphoneMuteState);

   _ipcMessageDispatcher->updatePropertyChanged(responseMsg.get());
}

void IpcWrapper::setMicrophoneMuteStateResponse(IN const PmResult& pmResult, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::setMicrophoneMuteStateResponse() entered"));
   dispatchMicMuteStateResponse(pmResult, MicrophoneMuteState(), act);
}

void IpcWrapper::dispatchMicMuteStateResponse(const PmResult& pmResult,
      const MicrophoneMuteState& microphoneMuteState, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::dispatchMicMuteStateResponse() entered"));
   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<MicMuteStatePropertyUpdate> responseMsg =
            std::make_shared<MicMuteStatePropertyUpdate>
      (pmResult, act, PM_PROPERTYID_MICROPHONEMUTESTATE, microphoneMuteState);

      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

void IpcWrapper::getSCOConnectionResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
      IN const SCOConnection& scoConnection, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getSCOConnectionResponse() entered"));
   ETG_TRACE_USR4(("deviceAddress: %s", deviceAddress.c_str()));

   (void) pmResult;
   (void) scoConnection;
   (void) act;
}

void IpcWrapper::onSCOConnectionChanged(IN const BdAddress& deviceAddress, IN const SCOConnection& scoConnection)
{
   ETG_TRACE_USR4(("IpcWrapper::onSCOConnectionChanged() entered"));
   ETG_TRACE_USR4(("deviceAddress: %s", deviceAddress.c_str()));

   (void) scoConnection;
}

void IpcWrapper::getRingtoneMuteStateResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
      IN const RingtoneMuteState& ringtoneMuteState, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getRingtoneMuteStateResponse() entered"));
   dispatchRingtoneMuteStateResponse(pmResult, deviceAddress, ringtoneMuteState, act);
}

void IpcWrapper::onRingtoneMuteStateChanged(IN const BdAddress& deviceAddress,
      IN const RingtoneMuteState& ringtoneMuteState)
{
   ETG_TRACE_USR4(("IpcWrapper::onRingtoneMuteStateChanged() entered"));

   std::shared_ptr<RingtoneMuteStatePropertyUpdate> responseMsg = std::make_shared<RingtoneMuteStatePropertyUpdate>
   (PmResult(), DEFAULT_ACT, PM_PROPERTYID_RINGTONEMUTESTATE, deviceAddress, ringtoneMuteState);

   _ipcMessageDispatcher->updatePropertyChanged(responseMsg.get());
}

void IpcWrapper::setRingtoneMuteStateResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
      IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::setRingtoneMuteStateResponse() entered"));
   dispatchRingtoneMuteStateResponse(pmResult, deviceAddress, RingtoneMuteState(), act);
}

void IpcWrapper::dispatchRingtoneMuteStateResponse(const PmResult& pmResult, const BdAddress& deviceAddress,
      const RingtoneMuteState& ringtoneMuteState, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::dispatchRingtoneMuteStateResponse() entered"));

   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<RingtoneMuteStatePropertyUpdate> responseMsg = std::make_shared<RingtoneMuteStatePropertyUpdate>
      (pmResult, act, PM_PROPERTYID_RINGTONEMUTESTATE, deviceAddress, ringtoneMuteState);

      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

void IpcWrapper::getWaitingModeStateResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
      IN const WaitingModeState& waitingModeState, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getWaitingModeStateResponse() entered"));

   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<WaitingModeStatePropertyUpdate> responseMsg =
            std::make_shared<WaitingModeStatePropertyUpdate>
      (pmResult, act, PM_PROPERTYID_WAITINGMODESTATE, deviceAddress, waitingModeState);

      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

void IpcWrapper::onWaitingModeStateChanged(IN const BdAddress& deviceAddress,
      IN const WaitingModeState& waitingModeState)
{
   ETG_TRACE_USR4(("IpcWrapper::onWaitingModeStateChanged() entered"));

   std::shared_ptr<WaitingModeStatePropertyUpdate> responseMsg =
         std::make_shared<WaitingModeStatePropertyUpdate>
   (PmResult(), DEFAULT_ACT, PM_PROPERTYID_WAITINGMODESTATE, deviceAddress, waitingModeState);

   _ipcMessageDispatcher->updatePropertyChanged(responseMsg.get());
}

void IpcWrapper::getRingtonePlaybackStatusResponse(IN const PmResult& pmResult,
      IN const RingtonePlaybackState& ringtonePlaybackState, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getRingtonePlaybackStatusResponse() entered"));

   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<RingtonePlaybackStatusPropertyUpdate> responseMsg =
            std::make_shared<RingtonePlaybackStatusPropertyUpdate>
      (pmResult, act, PM_PROPERTYID_RINGTONEPLAYBACKSTATUS, ringtonePlaybackState);

      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

void IpcWrapper::onRingtonePlaybackStatusChanged(IN const RingtonePlaybackState& ringtonePlaybackState)
{
   ETG_TRACE_USR4(("IpcWrapper::onRingtonePlaybackStatusChanged() entered"));

   std::shared_ptr<RingtonePlaybackStatusPropertyUpdate> responseMsg =
         std::make_shared<RingtonePlaybackStatusPropertyUpdate>
   (PmResult(), DEFAULT_ACT, PM_PROPERTYID_RINGTONEPLAYBACKSTATUS, ringtonePlaybackState);

   _ipcMessageDispatcher->updatePropertyChanged(responseMsg.get());
}

void IpcWrapper::getVoiceRecognitionStatusResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
      IN const VoiceRecognitionStatus& voiceRecognitionStatus, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getVoiceRecognitionStatusResponse() entered"));
   dispatchVoiceRecognitionStateResponse(pmResult, deviceAddress, voiceRecognitionStatus, act);
}

void IpcWrapper::onVoiceRecognitionStatusChanged(IN const BdAddress& deviceAddress,
      IN const VoiceRecognitionStatus& voiceRecognitionStatus)
{
   ETG_TRACE_USR4(("IpcWrapper::onVoiceRecognitionStatusChanged() entered"));

   std::shared_ptr<VoiceRecognitionPropertyUpdate> btvrExtendedStatusresponseMsg =
         std::make_shared<VoiceRecognitionPropertyUpdate>
   (PmResult(), DEFAULT_ACT, PM_PROPERTYID_BTDEVICEVOICERECOGNITIONEXTENDED, deviceAddress, false,
         voiceRecognitionStatus,
         EnhancedVoiceRecognitionFeature());

   _ipcMessageDispatcher->updatePropertyChanged(btvrExtendedStatusresponseMsg.get());

   std::shared_ptr<VoiceRecognitionPropertyUpdate> btvrStatusresponseMsg =
         std::make_shared<VoiceRecognitionPropertyUpdate>
   (PmResult(), DEFAULT_ACT, PM_PROPERTYID_BTDEVICEVOICERECOGNITION, deviceAddress, false, voiceRecognitionStatus,
         EnhancedVoiceRecognitionFeature());

   _ipcMessageDispatcher->updatePropertyChanged(btvrStatusresponseMsg.get());
}

void IpcWrapper::startStopVoiceRecognitionResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
      IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::startStopVoiceRecognitionResponse() entered"));
   dispatchVoiceRecognitionStateResponse(pmResult, deviceAddress, VoiceRecognitionStatus(), act);
}

void IpcWrapper::dispatchVoiceRecognitionStateResponse(const PmResult& pmResult, const BdAddress& deviceAddress,
      const VoiceRecognitionStatus& voiceRecognitionStatus, const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::dispatchVoiceRecognitionStateResponse() entered"));

   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<VoiceRecognitionPropertyUpdate> responseMsg =
            std::make_shared<VoiceRecognitionPropertyUpdate>
      (pmResult, act, PM_PROPERTYID_BTDEVICEVOICERECOGNITIONEXTENDED, deviceAddress, false, voiceRecognitionStatus,
            EnhancedVoiceRecognitionFeature());

      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

// TODO: The below function should be able to inform about the SIRI availability status to the clients.
void IpcWrapper::getEnhancedVoiceRecognitionFeatureResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
      IN const EnhancedVoiceRecognitionFeature& enhancedVoiceRecognitionFeature, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getEnhancedVoiceRecognitionFeatureResponse() entered"));
   ETG_TRACE_USR4(("deviceAddress: %s", deviceAddress.c_str()));

   (void) pmResult;
   (void) enhancedVoiceRecognitionFeature;
   (void) act;
}

void IpcWrapper::onEnhancedVoiceRecognitionFeatureChanged(IN const BdAddress& deviceAddress,
      IN const EnhancedVoiceRecognitionFeature& enhancedVoiceRecognitionFeature)
{
   ETG_TRACE_USR4(("IpcWrapper::onEnhancedVoiceRecognitionFeatureChanged() entered"));
   ETG_TRACE_USR4(("deviceAddress: %s", deviceAddress.c_str()));

   std::shared_ptr<VoiceRecognitionPropertyUpdate> btvrExtendedStatusresponseMsg =
         std::make_shared<VoiceRecognitionPropertyUpdate>
   (PmResult(), DEFAULT_ACT, PM_PROPERTYID_BTDEVICEVOICERECOGNITIONEXTENDED, deviceAddress, true,
         VoiceRecognitionStatus(),
         enhancedVoiceRecognitionFeature,
         ENHANCEDVOICERECOGNITION_FEATURE_STATUS);

   _ipcMessageDispatcher->updatePropertyChanged(btvrExtendedStatusresponseMsg.get());
}

void IpcWrapper::getExternalVRStateResponse(IN const PmResult& pmResult, IN const BdAddress& deviceAddress,
      IN const ExtVoiceRecognitionStatus& extVoiceRecognitionStatus, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::getExternalVRStateResponse() entered with DeviceAddress: %50s", deviceAddress.c_str()));

   if(DEFAULT_ACT != act)
   {
      std::shared_ptr<ExternalVRStatePropertyUpdate> responseMsg =
            std::make_shared<ExternalVRStatePropertyUpdate>
      (pmResult, act, PM_PROPERTYID_EXTERNALVRSTATE, deviceAddress, extVoiceRecognitionStatus._extVRStatus,
            VRInterruptedReason::VR_INTERRUPT_NO_ERROR);

      postResponseToMessageDispatcher(responseMsg);
   }
   else
   {
      // This portion will be executed only when error in coding
      ETG_TRACE_ERR(("Invalid Ack token"));
   }
}

void IpcWrapper::onExternalVRStateChanged(IN const BdAddress& deviceAddress,
      IN const ExtVoiceRecognitionStatus& extVoiceRecognitionStatus)
{
   ETG_TRACE_USR4(("IpcWrapper::onExternalVRStateChanged() entered with DeviceAddress: %50s", deviceAddress.c_str()));

   std::shared_ptr<ExternalVRStatePropertyUpdate> externalVRStateMsg =
         std::make_shared<ExternalVRStatePropertyUpdate>
   (PmResult(), DEFAULT_ACT, PM_PROPERTYID_EXTERNALVRSTATE, deviceAddress, extVoiceRecognitionStatus._extVRStatus,
         VRInterruptedReason::VR_INTERRUPT_NO_ERROR);

   _ipcMessageDispatcher->updatePropertyChanged(externalVRStateMsg.get());
}

void IpcWrapper::activateExternalVRResponse(IN const PmResult& pmResult,
      IN const BdAddress& deviceAddress, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::activateExternalVRResponse() entered"));
   postPmCoreResponseData<ActivateExternalVRResponseMsg>(pmResult, deviceAddress, act);
}

void IpcWrapper::deActivateExternalVRResponse(IN const PmResult& pmResult,
      IN const pmcore::BdAddress& deviceAddress, IN const ActType act)
{
   ETG_TRACE_USR4(("IpcWrapper::deActivateExternalVRResponse() entered"));
   postPmCoreResponseData<DeActivateExternalVRResponseMsg>(pmResult, deviceAddress, act);
}

void IpcWrapper::onPmStateChanged(IN const BdAddress& deviceAddress, IN const CallExistenceState
      callExistenceState)
{
   ETG_TRACE_USR4(("IpcWrapper::onPMStateChanged() entered: callExistenceState: %u",
         ETG_CENUM(CallExistenceState, callExistenceState)));
   ETG_TRACE_USR4(("deviceAddress: %s", deviceAddress.c_str()));
   (void) callExistenceState;

   PmAppClientHandler* pmAppClientHandler = PhoneCallManager::getInstance().getPmAppClientHandler();
   if (pmAppClientHandler)
   {
      bool phoneSubState = false;
      bool profileUsageInfo = false;

      if ((CALL_PRESENT_WITH_SCO == callExistenceState) || (INCOMING_CALL_PRESENT == callExistenceState))
      {
         phoneSubState = true;
         profileUsageInfo = true;
      }
      else if (CALL_PRESENT_WITHOUT_SCO == callExistenceState)
      {
         profileUsageInfo = true;

         unsigned char isPhoneSubstateSetForHandsetMode =
               com::bosch::pmcommon::PmConfiguration::getInstance().getPhoneSubstateSetForHandsetMode();

         if (PHONE_SUBSTATE_SET_FOR_HANDSET_MODE == isPhoneSubstateSetForHandsetMode)
         {
            phoneSubState = true;
         }
      }

      // Setting Phone Sub state
      pmAppClientHandler->getLcmRequestIf()->setSpmPhoneSubState(deviceAddress, phoneSubState);

      // Setting BT-Profile usage
      pmAppClientHandler->getBmAppRequestIf()->setBTProfilesUsage(deviceAddress, profileUsageInfo);
   }
}

} // namespace pm_ipc_wrapper
