/**
 * @file DeviceDetailsListHandler.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the DeviceDetailsListHandler class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file holds the class definition for DeviceDetailsListHandler class
 *
 * @ingroup IpcWrapper
 */

#include "DeviceDetailsListHandler.h"
#include "PhoneCallManager.h"
#include "IpcWrapper.h"
#include <algorithm>
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_SERVICE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/DeviceDetailsListHandler.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_SERVICE
#endif
#endif

using namespace pmcore;

namespace com
{
namespace bosch
{
namespace pmapp
{

DeviceDetailsList DeviceDetailsListHandler::_deviceDetailsList;
std::map<act_t, QueryRequestedContact> DeviceDetailsListHandler::_pendingQueryContactsList;
std::map<DialRequestInfo, QueriedContactInfo> DeviceDetailsListHandler::_contactInfoFromClients;

DeviceDetailsListHandler::DeviceDetailsListHandler(): _deviceDetailsListLock(), _activeDevicesList()
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::DeviceDetailsListHandler() entered"));
}

DeviceDetailsListHandler::~DeviceDetailsListHandler()
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::~DeviceDetailsListHandler() entered"));

   _deviceDetailsList.clear();

   _pendingQueryContactsList.clear();
   _contactInfoFromClients.clear();
   _activeDevicesList.clear();
}

void DeviceDetailsListHandler::insertDeviceDetails(const DeviceDetails& deviceDetails)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::insertDeviceDetails() entered"));
   _deviceDetailsList.push_back(deviceDetails);
}

void DeviceDetailsListHandler::updateDeviceDetails(const DeviceDetails& deviceDetails)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::updateDeviceDetails() entered"));

   Locker locker(&_deviceDetailsListLock);

   auto iter = std::find_if(_deviceDetailsList.begin(), _deviceDetailsList.end(),
         [&deviceDetails](DeviceDetails const& obj){return obj._deviceAddress == deviceDetails._deviceAddress;});

   if (_deviceDetailsList.end() != iter)
   {
      iter->_deviceHandle = deviceDetails._deviceHandle;
      iter->_deviceConnectionStatus = deviceDetails._deviceConnectionStatus;
      iter->_deviceIdentification = deviceDetails._deviceIdentification;
      iter->_deviceName = deviceDetails._deviceName;

      // Device role and callstatuslist should not be changed here. It should be updated only from PM core.
   }
   else
   {
      insertDeviceDetails(deviceDetails);
   }

   printDeviceDetailsList();
}

bool DeviceDetailsListHandler::updateDeviceConnectionStatus(const BdAddress& deviceAddress,
      const DeviceConnectionStatus deviceConnectionStatus)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::updateDeviceConnectionStatus() entered"));

   bool isDeviceConnectionStatusUpdated = false;

   auto iter = std::find_if(_deviceDetailsList.begin(), _deviceDetailsList.end(),
         [&deviceAddress](DeviceDetails const& obj){return obj._deviceAddress == deviceAddress;});

   if (_deviceDetailsList.end() != iter)
   {
      iter->_deviceConnectionStatus = deviceConnectionStatus;
      isDeviceConnectionStatusUpdated = true;
   }

   return isDeviceConnectionStatusUpdated;
}

bool DeviceDetailsListHandler::updateDeviceIdentification(const BdAddress& deviceAddress,
      const DeviceIdentification& deviceIdentification)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::updateDeviceIdentification() entered"));

   bool isDeviceIdentificationInfoUpdated = false;

   auto iter = std::find_if(_deviceDetailsList.begin(), _deviceDetailsList.end(),
         [&deviceAddress](DeviceDetails const& obj){return obj._deviceAddress == deviceAddress;});

   if (_deviceDetailsList.end() != iter)
   {
      iter->_deviceIdentification = deviceIdentification;
      isDeviceIdentificationInfoUpdated = true;
   }

   return isDeviceIdentificationInfoUpdated;
}

bool DeviceDetailsListHandler::removeDevice(const DeviceHandle deletedDeviceHandle, BdAddress& deletedDeviceAddress)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::removeDevice() entered"));

   Locker locker(&_deviceDetailsListLock);

   if (_deviceDetailsList.empty())
      return false;

   printDeviceDetailsList();

   bool isDeviceRemoved = false;

   if (PM_DEVICEHANDLE_ZERO == deletedDeviceHandle)
   {
      deletedDeviceAddress = "ALL";

      for (auto& deviceDetailsListIter : _deviceDetailsList)
      {
         clearDeviceInfo(deviceDetailsListIter._deviceAddress);
      }

      _deviceDetailsList.clear();
      isDeviceRemoved = true;
   }
   else
   {
      auto iter = std::find_if(_deviceDetailsList.begin(), _deviceDetailsList.end(),
            [&deletedDeviceHandle](DeviceDetails const& obj){return obj._deviceHandle == deletedDeviceHandle;});

      if (_deviceDetailsList.end() != iter)
      {
         deletedDeviceAddress = iter->_deviceAddress;

         clearDeviceInfo(deletedDeviceAddress);
         _deviceDetailsList.erase(iter);

         isDeviceRemoved = true;
      }
   }

   printDeviceDetailsList();
   return isDeviceRemoved;
}

bool DeviceDetailsListHandler::removeDevice(const BdAddress& deletedDeviceAddress)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::removeDevice() entered"));

   Locker locker(&_deviceDetailsListLock);

   if (_deviceDetailsList.empty())
      return false;

   printDeviceDetailsList();

   bool isDeviceRemoved = false;

   if ("ALL" == deletedDeviceAddress)
   {
      for (auto& deviceDetailsListIter : _deviceDetailsList)
      {
         clearDeviceInfo(deviceDetailsListIter._deviceAddress);
      }

      _deviceDetailsList.clear();
      isDeviceRemoved = true;
   }
   else
   {
      auto iter = std::find_if(_deviceDetailsList.begin(), _deviceDetailsList.end(),
            [&deletedDeviceAddress](DeviceDetails const& obj){return obj._deviceAddress == deletedDeviceAddress;});

      if (_deviceDetailsList.end() != iter)
      {
         ETG_TRACE_USR4(("Device is present in the container"));

         clearDeviceInfo(deletedDeviceAddress);
         _deviceDetailsList.erase(iter);

         isDeviceRemoved = true;
      }
   }

   return isDeviceRemoved;
}

bool DeviceDetailsListHandler::getDeviceDetails(const BdAddress& deviceAddress, DeviceDetails& deviceDetails)
{
   Locker locker(&_deviceDetailsListLock);

   printDeviceDetailsList();

   bool isDevicePresent = false;
   auto iter = std::find_if(_deviceDetailsList.begin(), _deviceDetailsList.end(),
         [&deviceAddress](DeviceDetails const& obj){return obj._deviceAddress == deviceAddress;});

   if (_deviceDetailsList.end() != iter)
   {
      deviceDetails = *iter;
      isDevicePresent = true;
   }

   return isDevicePresent;
}

bool DeviceDetailsListHandler::getDeviceDetails(const DeviceHandle& deviceHandle, DeviceDetails& deviceDetails)
{
   Locker locker(&_deviceDetailsListLock);

   printDeviceDetailsList();

   bool isDevicePresent = false;
   auto iter = std::find_if(_deviceDetailsList.begin(), _deviceDetailsList.end(),
         [&deviceHandle](DeviceDetails const& obj){return obj._deviceHandle == deviceHandle;} );

   if (_deviceDetailsList.end() != iter)
   {
      deviceDetails = *iter;
      isDevicePresent = true;
   }

   return isDevicePresent;
}

void DeviceDetailsListHandler::getDeviceHandle(const BdAddress& deviceAddress, DeviceHandle& deviceHandle)
{
   ETG_TRACE_USR4(("getDeviceHandle() entered with deviceAddress: %s", deviceAddress.c_str()));

   Locker locker(&_deviceDetailsListLock);

   printDeviceDetailsList();

   auto iter = std::find_if(_deviceDetailsList.begin(), _deviceDetailsList.end(),
         [&deviceAddress](DeviceDetails const& obj){return obj._deviceAddress == deviceAddress;});

   if (_deviceDetailsList.end() != iter)
   {
      deviceHandle = iter->_deviceHandle;
   }
}

void DeviceDetailsListHandler::getDeviceAddress(const DeviceHandle deviceHandle, BdAddress& deviceAddress)
{
   ETG_TRACE_USR4(("getDeviceAddress() entered with deviceHandle: %u", deviceHandle));

   Locker locker(&_deviceDetailsListLock);

   printDeviceDetailsList();

   auto iter = std::find_if(_deviceDetailsList.begin(), _deviceDetailsList.end(),
         [&deviceHandle](DeviceDetails const& obj){return obj._deviceHandle == deviceHandle;});

   if (_deviceDetailsList.end() != iter)
   {
      deviceAddress = iter->_deviceAddress;
   }
}

void DeviceDetailsListHandler::printDeviceDetailsList()
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::printDeviceDetailsList() entered"));

   for_each(_deviceDetailsList.begin(), _deviceDetailsList.end(), [](DeviceDetails & obj){obj.printDeviceIdentity();});
}

void DeviceDetailsListHandler::setActiveDevices(const BdAddressList& deviceAddressList)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::setActiveDevices() entered"));

   Locker locker(&_deviceDetailsListLock);

   for (auto& receivedDeviceAddress : deviceAddressList)
   {
      auto iter = std::find_if(_deviceDetailsList.begin(), _deviceDetailsList.end(),
            [&receivedDeviceAddress](DeviceDetails const& obj){return obj._deviceAddress == receivedDeviceAddress;});

      if (_deviceDetailsList.end() != iter)
      {
         iter->_deviceRole = DEVICEROLE_ACTIVE;
      }
   }
}

void DeviceDetailsListHandler::getActiveDevices(BdAddressList& deviceAddressList)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::getActiveDevices() entered"));

   Locker locker(&_deviceDetailsListLock);
   deviceAddressList = _activeDevicesList;
}

void DeviceDetailsListHandler::setDeviceRole(const ActivePassiveDeviceListMap& activePassiveDeviceList)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::setDeviceRole() entered"));

   Locker locker(&_deviceDetailsListLock);

   ETG_TRACE_USR4(("DeviceDetailsListHandler::ActivePassiveDeviceListMap :%d",activePassiveDeviceList.size()));

   // The device role of all the devices in "_deviceDetailsList" is set to "DEVICEROLE_DEFAULT".
   for_each(_deviceDetailsList.begin(), _deviceDetailsList.end(),
         [](DeviceDetails & obj){obj._deviceRole = DEVICEROLE_DEFAULT;});

   for (auto& activePassiveDeviceListElement : activePassiveDeviceList)
   {
      auto iter = std::find_if(_deviceDetailsList.begin(), _deviceDetailsList.end(),
            [&activePassiveDeviceListElement](DeviceDetails const& obj)
            {return obj._deviceAddress == activePassiveDeviceListElement.first;});

      if (_deviceDetailsList.end() != iter)
      {
         iter->_deviceRole = activePassiveDeviceListElement.second;
      }
   }

   printDeviceDetailsList();

   // The below set of code is to maintain the list of active devices
   bool activeDeviceListChanged = false;

   // Removing the non-active devices
   for (auto activeDevicesListIter = _activeDevicesList.begin(); activeDevicesListIter != _activeDevicesList.end(); )
   {
      auto activePassiveDeviceListMapIter = activePassiveDeviceList.find(*activeDevicesListIter);
      if ((activePassiveDeviceListMapIter == activePassiveDeviceList.end()) ||
            (pmcore::DEVICEROLE_ACTIVE != activePassiveDeviceListMapIter->second))
      {
         activeDevicesListIter = _activeDevicesList.erase(activeDevicesListIter);
         activeDeviceListChanged = true;
      }
      else
      {
         ++activeDevicesListIter;
      }
   }

   // Adding the new Active devices
   for (auto& activePassiveDeviceListMapIter : activePassiveDeviceList)
   {
      if (pmcore::DEVICEROLE_ACTIVE == activePassiveDeviceListMapIter.second)
      {
         auto activeDeviceIter = std::find(_activeDevicesList.begin(), _activeDevicesList.end(),
               activePassiveDeviceListMapIter.first);

         if (activeDeviceIter == _activeDevicesList.end())
         {
            // Device is not present
            _activeDevicesList.emplace_back(activePassiveDeviceListMapIter.first);
            activeDeviceListChanged = true;
         }
      }
   }

   if (activeDeviceListChanged)
   {
      PmAppClientHandler* pmAppClientHandler = PhoneCallManager::getInstance().getPmAppClientHandler();
      if (pmAppClientHandler)
      {
         pmAppClientHandler->getBmAppRequestIf()->setPrimaryDevice(_activeDevicesList);
      }
   }

}

void DeviceDetailsListHandler::getActiveDeviceAddressList(BdAddressList& deviceAddressList)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::getActiveDeviceAddressList() entered"));

   Locker locker(&_deviceDetailsListLock);

   for (auto& iter : _deviceDetailsList)
   {
      if (DEVICEROLE_ACTIVE == iter._deviceRole)
      {
         deviceAddressList.push_back(iter._deviceAddress);
      }
   }

}

void DeviceDetailsListHandler::getDeviceRole(const BdAddress& deviceAddress, DeviceRole& deviceRole)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::getDeviceRole() entered"));

   Locker locker(&_deviceDetailsListLock);

   auto iter = std::find_if(_deviceDetailsList.begin(), _deviceDetailsList.end(),
         [&deviceAddress](DeviceDetails const& obj){return obj._deviceAddress == deviceAddress;});

   if (_deviceDetailsList.end() != iter)
   {
      deviceRole = iter->_deviceRole;
   }
}

// TODO: This function is used only by CCA side. Once CCA is removed, this function should be removed.
void DeviceDetailsListHandler::getActiveDeviceAddress(BdAddress& deviceAddress)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::getActiveDeviceAddress() entered"));

   Locker locker(&_deviceDetailsListLock);

   for (auto& iter : _deviceDetailsList)
   {
      if (DEVICEROLE_ACTIVE == iter._deviceRole)
      {
         deviceAddress = iter._deviceAddress;
         break;
      }
   }
}

void DeviceDetailsListHandler::updateCallStatusListFromPmCore(const CallStatusList& callStatusList,
      const ActType act)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::updateCallStatusListFromPmCore() entered"));

   bool isQueryContactRequested = false;

   for (auto& deviceIter : callStatusList._callStatusList)
   {
      bool isQueryContactRequestedForDeviceIter = false;
      updateDeviceCallStatus(deviceIter.first, deviceIter.second, isQueryContactRequestedForDeviceIter);
      isQueryContactRequested |= isQueryContactRequestedForDeviceIter;
   }

   clearCallReportOfRemovedDevice(callStatusList);

   if ((PM_DEFAULT_ACT != act) || (!isQueryContactRequested))
   {
      Locker locker(&_deviceDetailsListLock);
      sendCallsReportList(act);
   }
}

void DeviceDetailsListHandler::updateDeviceCallStatus(const BdAddress& deviceAddress,
      const CallStatus& callStatus, bool &isQueryContactRequested)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::updateDeviceCallStatus() entered with DeviceAddress: %s",
         deviceAddress.c_str()));

   bool isQueryContactInfoSuccess = false;
   isQueryContactRequested = false;

   Locker locker(&_deviceDetailsListLock);

   auto deviceIter = std::find_if(_deviceDetailsList.begin(), _deviceDetailsList.end(),
         [&deviceAddress](DeviceDetails const& obj){return obj._deviceAddress == deviceAddress;});

   if (_deviceDetailsList.end() != deviceIter)
   {
      CallAttributesList& callAttributesList = deviceIter->_callsReport._callAttributesList;
      deviceIter->_callsReport._multiparty = callStatus._multiparty;

      ETG_TRACE_USR4(("callStatus._multiparty : %d", callStatus._multiparty));

      CallInfoList pmCoreCallInfoList = callStatus._callInfoList;
      for_each(pmCoreCallInfoList.begin(), pmCoreCallInfoList.end(), [](CallInfo & obj){obj.printCallInfo();});

      printCallAttributesList(callAttributesList);

      // updating callAttributesList with the new calls and updated attributes for the existing calls
      for (auto& pmCoreCallInfo : pmCoreCallInfoList)
      {
         auto callAttributesListIter = callAttributesList.find(pmCoreCallInfo._telephoneNumber);

         if (callAttributesList.end() != callAttributesListIter)
         {
            // The contact is already present. Hence updating the information specific to PM Core
            callAttributesListIter->second.adaptToPmCoreCallInfo(pmCoreCallInfo);
         }
         else
         {
            if ("Idle" == pmCoreCallInfo._state)
            {
               ETG_TRACE_USR4(("Ignoring the Idle call status"));
            }
            else
            {
               // the contact is not present. Hence adding it to the "callAttributesList" map
               ETG_TRACE_USR4(("Adding the contact to \"callAttributesList\""));
               CallAttributes callAttributes;
               callAttributes.adaptToPmCoreCallInfo(pmCoreCallInfo);

               QueriedContactInfo queriedContactInfo;
               ActType act = PM_DEFAULT_ACT;

               if (resetContactInfoFromClientsByTelephoneNumber(deviceAddress, pmCoreCallInfo._telephoneNumber,
                     act, queriedContactInfo))
               {
                  callAttributes._firstName = queriedContactInfo._firstName;
                  callAttributes._lastName = queriedContactInfo._lastName;
                  callAttributes._contactHandle = queriedContactInfo._contactHandle;
                  callAttributes._isContactProvidedByClient = true;
               }
               else
               {
                  requestQueryContactInfo(deviceAddress, pmCoreCallInfo._telephoneNumber, isQueryContactInfoSuccess);
                  if(isQueryContactInfoSuccess)
                  {
                     ETG_TRACE_USR4(("requestQueryContactInfo success"));
                     isQueryContactRequested = true;
                  }
               }

               callAttributesList.emplace_hint(callAttributesList.end(), pmCoreCallInfo._telephoneNumber,
                     callAttributes);
            }
         }
      }

      printCallAttributesList(callAttributesList);
      printPendingQueryContactsList();

      // Removing ended calls from callAttributesList
      for (auto callAttributesListIter = callAttributesList.begin(); callAttributesListIter != callAttributesList.end(); )
      {
         auto pmcoreCallInfoListIter = std::find_if(pmCoreCallInfoList.begin(), pmCoreCallInfoList.end(),
               [&callAttributesListIter](CallInfo const& obj){return obj._telephoneNumber == callAttributesListIter->first;});

         if ((pmcoreCallInfoListIter == pmCoreCallInfoList.end()) || ("Idle" == pmcoreCallInfoListIter->_state))
         {
            ETG_TRACE_USR4(("one call has ended"));

            auto iter = std::find_if(_pendingQueryContactsList.begin(), _pendingQueryContactsList.end(),
                  [&callAttributesListIter, &deviceAddress](std::pair<act_t, QueryRequestedContact> const& obj)
                  {return ((obj.second._telephoneNumber == callAttributesListIter->first) &&
                        (obj.second._deviceAddress == deviceAddress));});

            if (iter != _pendingQueryContactsList.end())
            {
               _pendingQueryContactsList.erase(iter->first);
            }
            callAttributesListIter = callAttributesList.erase(callAttributesListIter);
         }
         else
         {
            ++callAttributesListIter;
         }
      }

      // TODO: To be removed after testing
      printCallAttributesList(callAttributesList);
      printPendingQueryContactsList();
   }
}

void DeviceDetailsListHandler::clearCallReportOfRemovedDevice(const CallStatusList& callStatusList)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::clearCallReportOfRemovedDevice() entered"));

   for (auto& iter : _deviceDetailsList)
   {
      BdAddress deviceAddress = iter._deviceAddress;

      ETG_TRACE_USR4(("clearCallReportOfRemovedDevice() deviceAddress: %s", deviceAddress.c_str()));

      CallStatusListMap callStatusListMap = callStatusList._callStatusList;

      auto deviceIter = std::find_if(callStatusListMap.begin(), callStatusListMap.end(),
            [&deviceAddress](std::pair<BdAddress, CallStatus> const& obj){return obj.first == deviceAddress;});

      if (callStatusListMap.end() == deviceIter)
      {
         iter._callsReport._callAttributesList.clear();
         iter._callsReport._multiparty = false;
      }
   }
}

void DeviceDetailsListHandler::sendCallsReportList(const ActType act)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::sendCallsReportList() entered"));

   CallsReportList callsReportList;

   for (auto& deviceDetailsIter : _deviceDetailsList)
   {
      // Updating if the contact is not present in _pendingQueryContactsList
      CallAttributesList& actualCallAttributesList = deviceDetailsIter._callsReport._callAttributesList;

      ETG_TRACE_USR4(("DeviceDetailsListHandler::deviceDetailsIter._deviceAddress: %s", deviceDetailsIter._deviceAddress.c_str()));
      ETG_TRACE_USR4(("DeviceDetailsListHandler::actualCallAttributesList.size(): %u", actualCallAttributesList.size()));

      CallAttributesList callAttributesListToClients;

      printPendingQueryContactsList();

      // If the call status is requested by client(PM_DEFAULT_ACT != act), then all the call status info should be
      // reported even if the queried contact info is not yet received for a contact.
      if (PM_DEFAULT_ACT != act)
      {
         callAttributesListToClients = actualCallAttributesList;
      }
      else
      {
         for (auto& actualCallAttributesListIter : actualCallAttributesList)
         {
            TelephoneNumber telephoneNumber = actualCallAttributesListIter.first;

            auto iter = std::find_if(_pendingQueryContactsList.begin(), _pendingQueryContactsList.end(),
                  [&telephoneNumber, &deviceDetailsIter] (std::pair<act_t, QueryRequestedContact> const& obj)
                  {return ((obj.second._telephoneNumber == telephoneNumber) &&
                        (obj.second._deviceAddress == deviceDetailsIter._deviceAddress) &&
                        (obj.first != PM_DEFAULT_ACT));});

            bool updateToClients = false;

            if (iter == _pendingQueryContactsList.end())
            {
               // Check if some other contact number is waiting for the response from PB client.
               // If that is multi-party, then this must not be updated to the clients
               // as of now. Only on receiving the response from PB client, this status must be updated.
               auto otherContactIter = std::find_if(_pendingQueryContactsList.begin(), _pendingQueryContactsList.end(),
                     [&telephoneNumber, &deviceDetailsIter](std::pair<act_t, QueryRequestedContact> const& obj)
                     {return ((obj.second._telephoneNumber != telephoneNumber) &&
                           (obj.second._deviceAddress == deviceDetailsIter._deviceAddress));});

               if ((_pendingQueryContactsList.end() != otherContactIter) &&
                     (deviceDetailsIter._callsReport._multiparty))
               {
                  // Update is blocked even if it has a queried contact info.
                  updateToClients = false;
               }
               else
               {
                  updateToClients = true;
               }
            }

            if (updateToClients)
            {
               callAttributesListToClients.emplace_hint(callAttributesListToClients.end(), telephoneNumber,
                     actualCallAttributesListIter.second);
            }
         }
      }

      printCallAttributesList(callAttributesListToClients);

      callsReportList.emplace_hint(callsReportList.end(), deviceDetailsIter._deviceAddress,
            CallsReport(deviceDetailsIter._callsReport._multiparty, callAttributesListToClients));
   }

   // Updating the call status
   pm_ipc_wrapper::IpcWrapper::getInstance().updateCallsReportListToClients(callsReportList, act);
}

void DeviceDetailsListHandler::requestQueryContactInfo(const pmcore::BdAddress& deviceAddress,
      const pmcore::TelephoneNumber& telephoneNumber, bool& isQueryContactInfoSuccess)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::requestQueryContactInfo() entered"));

   PmAppClientHandler* pmAppClientHandler = PhoneCallManager::getInstance().getPmAppClientHandler();

   if (pmAppClientHandler)
   {
      IPhoneBookRequestIf* phoneBookRequestIf = pmAppClientHandler->getPhoneBookRequestIf();
      if (phoneBookRequestIf)
      {
         DeviceHandle deviceHandle;
         getDeviceHandle(deviceAddress, deviceHandle);

         act_t act = phoneBookRequestIf->searchPhoneBook(deviceHandle, telephoneNumber);

         _pendingQueryContactsList.emplace_hint(_pendingQueryContactsList.end(), act,
               QueryRequestedContact(deviceAddress, telephoneNumber));

         printPendingQueryContactsList();

         if (DEFAULT_ACT == act)
         {
            ETG_TRACE_ERR(("Error in requestQueryContactInfo, Sending the default CallStatusList as is"));
         }
         else
         {
            isQueryContactInfoSuccess = true;
         }

      }
   }
}

void DeviceDetailsListHandler::queryContactInfoResponse(const act_t act,
      const QueriedContactInfo& queriedContactInfo, const PhonebookErrorResponseEnum phonebookErrorResponse)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::queryContactInfoResponse() entered"));

   // FIXME: Need to check the possibility for retry after checking the error
   if (NO_ERROR != phonebookErrorResponse)
   {
      ETG_TRACE_USR4(("phonebookErrorResponse :%u", ETG_CENUM(PhonebookErrorResponseEnum, phonebookErrorResponse)));
   }

   BdAddress deviceAddress = "";

   auto pendingQueryContact = _pendingQueryContactsList.find(act);

   if (_pendingQueryContactsList.end() != pendingQueryContact)
   {
      deviceAddress = pendingQueryContact->second._deviceAddress;
   }

   bool sendUpdateToClients = true;

   if (updateQueriedContactInfo(act, queriedContactInfo, sendUpdateToClients))
   {
      // Before updating, check whether some other contact is also waiting for queried info from PB in the same device.
      if (sendUpdateToClients)
      {
         sendCallsReportList();
      }
   }
}

bool DeviceDetailsListHandler::updateQueriedContactInfo(const act_t act,
      const QueriedContactInfo& queriedContactInfo, bool &sendUpdateToClients)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::updateQueriedContactInfo() entered"));

   Locker locker(&_deviceDetailsListLock);

   bool updateSuccess = false;
   sendUpdateToClients = true;

   printPendingQueryContactsList();

   auto pendingQueryContact = _pendingQueryContactsList.find(act);

   if (_pendingQueryContactsList.end() != pendingQueryContact)
   {
      BdAddress deviceAddress = pendingQueryContact->second._deviceAddress;
      TelephoneNumber telephoneNumber = pendingQueryContact->second._telephoneNumber;

      auto deviceDetailsIter = std::find_if(_deviceDetailsList.begin(), _deviceDetailsList.end(),
               [&deviceAddress](DeviceDetails const& obj){return obj._deviceAddress == deviceAddress;});

      if (_deviceDetailsList.end() != deviceDetailsIter)
      {
         CallAttributesList& callAttributesList = deviceDetailsIter->_callsReport._callAttributesList;

         auto callAttributesIter = callAttributesList.find(telephoneNumber);

         if (callAttributesList.end() != callAttributesIter)
         {
            callAttributesIter->second._firstName = queriedContactInfo._firstName;
            callAttributesIter->second._lastName = queriedContactInfo._lastName;
            callAttributesIter->second._contactHandle = queriedContactInfo._contactHandle;

            updateSuccess = true;
         }
      }

      _pendingQueryContactsList.erase(act);

      auto iter = std::find_if(_pendingQueryContactsList.begin(), _pendingQueryContactsList.end(),
            [&deviceAddress](std::pair<act_t, QueryRequestedContact> const& obj)
            {return (obj.second._deviceAddress == deviceAddress);});

      if (_pendingQueryContactsList.end() != iter)
      {
         // Some other contact is pending for query info from Phonebook for the same device.
         // Hence ignoring the update for this device as of now.
         sendUpdateToClients = false;
      }
   }

   return updateSuccess;
}

void DeviceDetailsListHandler::setContactInfoFromClients(const BdAddress& deviceAddress,
      const TelephoneNumber& telephoneNumber, const ActType act,
      QueriedContactInfo& queriedContactInfo)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::setContactInfoFromClients() entered"));

   Locker locker(&_deviceDetailsListLock);

   _contactInfoFromClients.emplace_hint(_contactInfoFromClients.end(),
         DialRequestInfo(deviceAddress, telephoneNumber, act), queriedContactInfo);
}

bool DeviceDetailsListHandler::resetContactInfoFromClientsByAct(const BdAddress& deviceAddress, const ActType act,
      TelephoneNumber& telephoneNumber, QueriedContactInfo& queriedContactInfo)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::resetContactInfoFromClientsByAct() entered"));
   bool isOperationSuccess = false;

   Locker locker(&_deviceDetailsListLock);

   auto iter = std::find_if(_contactInfoFromClients.begin(), _contactInfoFromClients.end(),
         [&deviceAddress, &act](std::pair<DialRequestInfo, QueriedContactInfo> const& obj)
         {return ((obj.first._deviceAddress == deviceAddress) && (obj.first._act == act));});

   if (_contactInfoFromClients.end() != iter)
   {
      queriedContactInfo = iter->second;
      telephoneNumber = iter->first._telephoneNumber;
      _contactInfoFromClients.erase(iter);
      isOperationSuccess = true;
   }

   return isOperationSuccess;
}

bool DeviceDetailsListHandler::resetContactInfoFromClientsByTelephoneNumber(const BdAddress& deviceAddress,
      const TelephoneNumber& telephoneNumber, ActType& act,
      QueriedContactInfo& queriedContactInfo)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::resetContactInfoFromClientsByTelephoneNumber() entered"));
   bool isOperationSuccess = false;

   Locker locker(&_deviceDetailsListLock);

   auto iter = std::find_if(_contactInfoFromClients.begin(), _contactInfoFromClients.end(),
         [&deviceAddress, &telephoneNumber](std::pair<DialRequestInfo, QueriedContactInfo> const& obj)
         {return ((obj.first._deviceAddress == deviceAddress) &&
               (obj.first._telephoneNumber == telephoneNumber));});

   if (_contactInfoFromClients.end() != iter)
   {
      queriedContactInfo = iter->second;
      act = iter->first._act;
      _contactInfoFromClients.erase(iter);
      isOperationSuccess = true;
   }

   return isOperationSuccess;
}

void DeviceDetailsListHandler::clearDeviceInfo(const BdAddress& deviceAddress)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::clearDeviceInfo() entered"));

   // Need to update the corresponding detail in "_pendingQueryContactsList" when the device
   // deletion happens and then update the call reports list to the clients
   for (auto iter = _pendingQueryContactsList.begin(); iter != _pendingQueryContactsList.end(); )
   {
      if (iter->second._deviceAddress == deviceAddress)
         iter = _pendingQueryContactsList.erase(iter);
      else
         ++iter;
   }

   auto deviceDetailsListIter = std::find_if(_deviceDetailsList.begin(), _deviceDetailsList.end(),
            [&deviceAddress](DeviceDetails const& obj)
            {return (obj._deviceAddress == deviceAddress);});

   if (_deviceDetailsList.end() != deviceDetailsListIter)
   {
      // Resetting the calls report for this device
      deviceDetailsListIter->_callsReport = CallsReport();
      sendCallsReportList();
   }

   for (auto iter = _contactInfoFromClients.begin(); iter != _contactInfoFromClients.end(); )
   {
      if (iter->first._deviceAddress == deviceAddress)
         iter = _contactInfoFromClients.erase(iter);
      else
         ++iter;
   }
}

void DeviceDetailsListHandler::printCallAttributesList(const CallAttributesList& callAttributesList)
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::printCallAttributesList() entered"));

   for (auto iter : callAttributesList)
   {
      ETG_TRACE_USR4(("printCallAttributesList:telephoneNumber: %s", iter.first.c_str()));
      iter.second.printCallAttributes();
   }
}

void DeviceDetailsListHandler::printPendingQueryContactsList()
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::printPendingQueryContactsList() entered"));

   unsigned short instance = 0;

   for (auto& iter : _pendingQueryContactsList)
   {
      instance++;
      ETG_TRACE_USR4(("instance: %u", instance));

      ETG_TRACE_USR4(("printPendingQueryContactsList:act: %u", iter.first));

      ETG_TRACE_USR4(("printPendingQueryContactsList:_deviceAddress: %s", iter.second._deviceAddress.c_str()));
      ETG_TRACE_USR4(("printPendingQueryContactsList:_telephoneNumber: %s", iter.second._telephoneNumber.c_str()));
   }
}

void DeviceDetailsListHandler::checkPendingQueryContactsList()
{
   ETG_TRACE_USR4(("DeviceDetailsListHandler::checkPendingQueryContactsList() entered"));

   printPendingQueryContactsList();

   PmAppClientHandler* pmAppClientHandler = PhoneCallManager::getInstance().getPmAppClientHandler();

   if (pmAppClientHandler)
   {
      IPhoneBookRequestIf* phoneBookRequestIf = pmAppClientHandler->getPhoneBookRequestIf();
      if (phoneBookRequestIf)
      {
         std::map<act_t, QueryRequestedContact> _pendingQueryContactsListTemp;
         _pendingQueryContactsListTemp = _pendingQueryContactsList;
         _pendingQueryContactsList.clear();

         for (auto iter = _pendingQueryContactsListTemp.begin(); iter != _pendingQueryContactsListTemp.end();)
         {
            DeviceHandle deviceHandle;
            getDeviceHandle(iter->second._deviceAddress, deviceHandle);
            act_t act = phoneBookRequestIf->searchPhoneBook(deviceHandle, iter->second._telephoneNumber);

            if (DEFAULT_ACT != act)
            {
               ETG_TRACE_USR4(("searchPhoneBook called"));
               auto const second_value = std::move(iter->second);
               iter = _pendingQueryContactsListTemp.erase(iter);
               _pendingQueryContactsList.insert({act, std::move(second_value)});
            }
            else
            {
               auto const second_value = std::move(iter->second);
               _pendingQueryContactsList.insert({act, std::move(second_value)});
               iter++;
            }
         }
      }
   }
   printPendingQueryContactsList();
}

} // namespace pmapp
} // namespace bosch
} // namespace com
