/**
 * @file DeviceDetails.cpp
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the DeviceDetails class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details This file holds the class definition for DeviceDetails class
 *
 * @ingroup IpcWrapper
 */

#include "DeviceDetails.h"
#include "PmAppTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_SERVICE
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/DeviceDetails.cpp.trc.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_PM_SERVICE
#endif
#endif

using namespace pmcore;

namespace com
{
namespace bosch
{
namespace pmapp
{

DeviceDetails::DeviceDetails()
: _deviceAddress(""), _deviceHandle(PM_DEVICEHANDLE_ZERO), _deviceName(""),
  _deviceConnectionStatus(DEVICE_CONNECTIONSTATUS_DEFAULT),
  _deviceIdentification(VENDOR_ID_INVALID, VENDOR_ID_SOURCE_INVALID),
  _deviceRole(DEVICEROLE_DEFAULT), _callsReport()
{
   ETG_TRACE_USR4(("DeviceDetails::DeviceDetails() entered"));
}

DeviceDetails::DeviceDetails(const BdAddress& deviceAddress, const DeviceHandle deviceHandle,
      const DeviceName& deviceName, const DeviceConnectionStatus deviceConnectionStatus,
      const DeviceIdentification& deviceIdentification, const DeviceRole deviceRole,
      const CallsReport& callsReport) :
            _deviceAddress(deviceAddress), _deviceHandle(deviceHandle),
            _deviceName(deviceName), _deviceConnectionStatus(deviceConnectionStatus),
            _deviceIdentification(deviceIdentification), _deviceRole(deviceRole), _callsReport(callsReport)
{
   ETG_TRACE_USR4(("DeviceDetails::DeviceDetails() Parameterized constructor entered"));
}

DeviceDetails::DeviceDetails(const DeviceDetails& other)
: _deviceAddress(other._deviceAddress), _deviceHandle(other._deviceHandle), _deviceName(other._deviceName),
  _deviceConnectionStatus(other._deviceConnectionStatus), _deviceIdentification(other._deviceIdentification),
  _deviceRole(other._deviceRole), _callsReport(other._callsReport)
{
   ETG_TRACE_USR4(("DeviceDetails::DeviceDetails() Copy constructor entered"));
}

DeviceDetails::~DeviceDetails()
{
   ETG_TRACE_USR4(("DeviceDetails::~DeviceDetails() entered"));
}

void swap(DeviceDetails& first, DeviceDetails& second)
{
   using std::swap;

   swap(first._deviceAddress, second._deviceAddress);
   swap(first._deviceHandle, second._deviceHandle);
   swap(first._deviceName, second._deviceName);
   swap(first._deviceConnectionStatus, second._deviceConnectionStatus);
   swap(first._deviceIdentification, second._deviceIdentification);
   swap(first._deviceRole, second._deviceRole);
   swap(first._callsReport, second._callsReport);
}

//lint -e{1529} suppress "assignment operator not first checking for assignment to this", see copy-and-swap idiom
DeviceDetails& DeviceDetails::operator=(DeviceDetails other)
{
   swap(*this, other);
   return *this;
}

bool DeviceDetails::operator==(const DeviceDetails& rhs) const
{
   return ((this->_deviceAddress == rhs._deviceAddress)
         && (this->_deviceHandle == rhs._deviceHandle) && (this->_deviceName == rhs._deviceName)
         && (this->_deviceConnectionStatus == rhs._deviceConnectionStatus)
         && (this->_deviceIdentification == rhs._deviceIdentification)
         && (this->_deviceRole == rhs._deviceRole)
         && (this->_callsReport == rhs._callsReport));
}

void DeviceDetails::printDeviceIdentity()
{
   ETG_TRACE_USR4(("DeviceDetails::printDeviceIdentity() entered"));

   ETG_TRACE_USR4(("DeviceDetails: deviceAddress: %s", _deviceAddress.c_str()));
   ETG_TRACE_USR4(("DeviceDetails: deviceHandle: %u", _deviceHandle));
   ETG_TRACE_USR4(("DeviceDetails: deviceName: %s", _deviceName.c_str()));
   ETG_TRACE_USR4(("DeviceDetails: deviceConnectionStatus: %u",
         ETG_CENUM(DeviceConnectionStatus, _deviceConnectionStatus)));
   ETG_TRACE_USR4(("DeviceDetails: deviceRole: %u", ETG_CENUM(DeviceRoleEnumType, _deviceRole)));

   _callsReport.printCallsReport();
   _deviceIdentification.printDeviceIdentificationInfo();
}

} // namespace pmapp
} // namespace bosch
} // namespace com
