/**
 * @file CallAttributes.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the type definition of the CallAttributes class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup IpcWrapper
 */

#ifndef CallAttributes_h
#define CallAttributes_h

#include "PmCoreIfTypes.h"
#include <vector>

#define CONTACT_HANDLE_DEFAULT 0x00

typedef unsigned int ContactHandle;

namespace com
{
namespace bosch
{
namespace pmapp
{

class CallAttributes
{
public:
   /**
    * Constructor of CallAttributes class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   CallAttributes();

   /**
    * Copy Constructor of CallAttributes class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   CallAttributes(const CallAttributes& other);

   /**
    * Destructor of CallAttributes class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   ~CallAttributes();

   /**
    * Friend function used when copying the object.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   friend void swap(CallAttributes& first, CallAttributes& second);

   /**
    * Overloading '=' operator
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   //lint -e{1720} suppress "assignment operator has non-const parameter", see copy-and-swap idiom
   CallAttributes& operator=(CallAttributes other);

   /**
    * This function is used to set the parameters of CallAttributes in accordance with the corresponding
    * call info from PM Core
    *
    * @param[in]  pmcore::CallInfo& pmCoreCallInfo - Call Info corresponding to this CallAttribute object
    *             received from PM Core
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void adaptToPmCoreCallInfo(const pmcore::CallInfo& pmCoreCallInfo);

   /**
    * Overloading '==' operator
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   bool operator==(const CallAttributes& rhs) const;

   /**
    * Overloading '!=' operator
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   bool operator!=(const CallAttributes& rhs) const;

   /**
    * This function is used to print the call attributes
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void printCallAttributes();

   pmcore::CallInstance    _instance;           // call instance
   pmcore::CallState       _state;              // call state
   pmcore::Duration        _durationHr;         // call duration in hours
   pmcore::Duration        _durationMin;        // call duration in minutes
   pmcore::Duration        _durationSec;        // call duration in seconds
   pmcore::CallMode        _mode;               // call mode
   pmcore::NumberType      _type;               // call number type
   pmcore::CallDirection   _direction;          // call direction
   pmcore::FirstName       _firstName;          // First name
   pmcore::LastName        _lastName;           // Last name
   ContactHandle           _contactHandle;      // Last name
   bool _isContactProvidedByClient;             // true if the contact information(name) is provided by the clients itself
};

// key - Telephone number and value - Corresponding call attributes
typedef std::map<pmcore::TelephoneNumber, CallAttributes> CallAttributesList;

} // namespace pmapp
} // namespace bosch
} // namespace com

#endif // CallAttributes_h
