/**
 * @file IpcWrapper.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the IpcWrapper class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup IpcWrapper
 */

#ifndef IpcWrapper_h
#define IpcWrapper_h

#include "IpcMsgTypes.h"
#include "PmSingleton.h"
#include "IPmCoreCallbackIf.h"
#include "IPmCoreRequestIf.h"
#include "DeviceDetails.h"
#include "ILoopbackReceiver.h"
#include "PmComponent.h"
#include <memory>

class IpcMessageDispatcher;

namespace pm_ipc_wrapper
{

/*
 * This class takes the responsibility to forward the IPC messages to PM-Core &
 * back from PM_Core to IPC Message types.
 *
 */
class IpcWrapper final : public PmSingleton<IpcWrapper>, public pmcore::IPmCoreCallbackIf,
   public com::bosch::pmcommon::ILoopbackReceiver
{
public:
   /**
    * This method is used to set the pm application.
    *
    * @param[in] mainApplication - Pointer to the AhlApp class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void setPmApplication(ahl_tclBaseOneThreadApp* mainApplication);

   /**
    * This method is used to set the pm component.
    *
    * @param[in] pmComponent - Pointer to the pm component
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void setPmComponent(com::bosch::PmComponent* pmComponent);

   /**
    * This function is invoked to register the IpcMessageDispatcher which is used to post the responses
    * received from PM Core layer
    *
    * @param[in] IpcMessageDispatcher* ipcMessageDispatcher - Instance to IpcMessageDispatcher
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void registerIpcMessageDispatcher(IpcMessageDispatcher* ipcMessageDispatcher);

   /**
    * This function is invoked to inform the changed system state
    * so that the corresponding actions shall be taken in PM core.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void onSystemStateChanged(const pmcore::SystemState systemState);

   /**
    * This function is invoked to inform the vehicle configuration.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void setVehicleConfiguration(const pmcore::VehicleConfiguration& vehicleconfig);
   /**
    * This function is invoked to inform the changed device connection status.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void onDeviceConnectionStatusChanged(const ::com::bosch::pmapp::DeviceDetails& deviceDetails);

   /**
    * This function is invoked to inform the changed device identification status.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   void onDeviceIdentificationChanged(const pmcore::BasicDeviceDetails& basicDeviceDetails);

   /**
    * This function is invoked to get the System Ring-tones list.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult getSystemWideRingtonesList(const pmcore::ActType act = 0u);

   /**
    * This function is invoked to get the Device Ring-tones list.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult getDeviceRingtoneList(const pmcore::ActType act = 0u);

   /**
    * This function is used to get the suppress ring-tone on/off list for all the paired devices.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult getSuppressRingtoneOnOffList(const pmcore::ActType act = 0u);

   /**
    * This function is used to get the auto-waiting mode setting of all the paired devices.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult getAutoWaitingModeOnOffList(const pmcore::ActType act = 0u);

   /**
    * This function is used to get the waiting mode file path.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult getWaitingModeFilePath(const pmcore::ActType act = 0u);

   /**
    * This function is used to set the waiting mode file path.
    *
    * @param[in] pmcore::FilePath filePathNB - Narrow band file path
    *            pmcore::FilePath filePathWB - Wide band file path
    *            pmcore::ActType act - Generated Token
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult setWaitingModeFilePath(const pmcore::FilePath& filePathNB, const pmcore::FilePath& filePathWB,
         const pmcore::ActType act = 0u);

   /**
    * This function is used to get the "pmcore::PhoneCallAudioActive" property.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult getPhoneCallAudioActive(const pmcore::ActType act = 0u);

   /**
    * This function is used to get the "pmcore::ActivePassiveDeviceList" status.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult getActivePassiveDeviceList(const pmcore::ActType act = 0u);

   /**
    * This function is invoked to make a Play ring tone in the HFP connected device
    * corresponding to the given device address.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult postSetRingtoneRequest2PmCore(const pmcore::BdAddress& deviceAddress,
         const pmcore::RingtoneId ringtoneId, const pmcore::ActType act = 0u);

   /**
    * This function is used to suppress the ring-tone either to on/off list for the paired device.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult postSuppressRingtoneOnOffRequest2PmCore(const pmcore::BdAddress& bdAddress,
         const pmcore::SuppressRingtoneState suppressRingtoneState, const pmcore::ActType act = 0u);

   /**
    * This function is used to set the auto-waiting mode setting either to on/off for the paired device.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult postAutoWaitingModeOnOffRequest2PmCore(const pmcore::BdAddress& bdAddress,
         const pmcore::AutoWaitingModeState autoWaitingModeState, const pmcore::ActType act = 0u);

   /**
    * This function is used to set the given device address as the Active device.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult postSetActivePhoneRequest2PmCore(const pmcore::BdAddressList& deviceAddressList,
         const pmcore::ActType act = 0u);

   /**
    * This function is invoked to make a Play ring tone in the HFP connected device
    * corresponding to the given device address.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult postPlayRingtoneRequest2PmCore(const pmcore::RingtoneId ringtoneId,
         const pmcore::PlayCount, const pmcore::ActType act = 0u);

   /**
    * This function is invoked to make a Play ring tone in the HFP connected device
    * corresponding to the given device address.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult postStopRingtoneRequest2PmCore(const pmcore::ActType act = 0u);

   /**
    * This function is used to restore the settings of PM Application such as Suppress Ring-tone,
    *  Device Ring-tone, Auto-waiting mode to their default values.
    *
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult postRestoreDefaultSettingRequest2PmCore(const pmcore::BdAddress& deviceAddress,
         const pmcore::ActType act = 0u);

   /**
    * This function is used to get the supported features of the connected HFP device
    * corresponding to the given device address.
    *
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult getSupportedFeatures(const pmcore::BdAddress& deviceAddress, const pmcore::ActType act = 0u);

   /**
    * This function is used to get the network status of the connected HFP device
    * corresponding to the given device address.
    *
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult getNetworkStatus(const pmcore::BdAddress& deviceAddress, const pmcore::ActType act = 0u);

   /**
    * This function is used to get the signal strength of the connected HFP device
    * corresponding to the given device address.
    *
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult getSignalStrength(const pmcore::BdAddress& deviceAddress, const pmcore::ActType act = 0u);

   /**
    * This function is used to get the pmcore::BatteryChargeLevel of the connected HFP device
    * corresponding to the given device address.
    *
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult getBatteryChargeLevel(const pmcore::BdAddress& deviceAddress, const pmcore::ActType act = 0u);

   /**
    * This function is used to get the pmcore::NetworkOperator of the connected HFP device
    * corresponding to the given device address.
    *
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult getNetworkOperator(const pmcore::BdAddress& deviceAddress, const pmcore::ActType act = 0u);

   /**
    * This function is used to get the Voice mail list of the connected HFP device
    * corresponding to the given device address.
    *
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult getVoiceMailList(const pmcore::BdAddress& deviceAddress, const pmcore::ActType act = 0u);

   /**
    * This function is used to get the calls' status of all the HFP connected devices.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult getCallStatusList(const pmcore::ActType act = 0u);

   /**
    * This function is used to get the microphone mute status of the vehicle.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult getMicrophoneMuteState(const pmcore::ActType act = 0u);

   /**
    * This function is used to set the microphone mute status of the vehicle.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult setMicrophoneMuteState(const pmcore::MuteState microphoneMuteState,
         const pmcore::ActType act = 0u);

   /**
    * This function is invoked to merge the given calls in the given devices.
    *
    * @param[in] pmcore::DeviceCallInstanceListMap& mergeCallList
    *             - Object containing the calls list and their corresponding device addresses.
    *            pmcore::ActType act
    *             - asynchronous completion token
    * @param[out] pmcore::PmResult - Result containing the response message
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult postMergeCallsRequest2PmCore(const pmcore::DeviceCallInstanceListMap& mergeCallList,
         const pmcore::ActType act = 0u);

   /**
    * This function is invoked to split the merged calls and make the given calls Active.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult postSplitCallsRequest2PmCore(const pmcore::DeviceCallInstanceListMap& splitCallList,
         const pmcore::ActType act = 0u);

   /**
    * This function is invoked to hangup the given calls in the given device addresses.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult postHangupCallsRequest2PmCore(const pmcore::DeviceCallInstanceListMap hangupCallList,
         const pmcore::ActType act = 0u);

   /**
    * This function is invoked to make a re-dial in the HFP connected device
    * corresponding to the given device address.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult postRedialRequest2PmCore(const pmcore::BdAddress& deviceAddress,
         const pmcore::HideCallerId hideCallerId = 0u, const pmcore::ActType act = 0u);

   /**
    * This function is invoked to get the SCO connection status of the HFP connected device
    * corresponding to the given device address.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult getSCOConnection(const pmcore::BdAddress& deviceAddress, const pmcore::ActType act = 0u);

   /**
    * This function is invoked to get the ring-tone mute status of the paired device
    * corresponding to the given device address.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult getRingtoneMuteState(const pmcore::BdAddress& deviceAddress, const pmcore::ActType act = 0u);

   /**
    * This function is invoked to set the ring-tone mute status of the paired device
    * corresponding to the given device address.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult setRingtoneMuteState(const pmcore::BdAddress& deviceAddress,
         const pmcore::MuteState ringtoneMuteState, const pmcore::ActType act = 0u);

   /**
    * This function is invoked to get the waiting mode status.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult getWaitingModeState(const pmcore::BdAddress& deviceAddress, const pmcore::ActType act = 0u);

   /**
    * This function is invoked to get the ringtone playing status.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult getRingtonePlaybackStatus(const pmcore::ActType act = 0u);

   /**
    * This function is invoked to either start or stop the waiting mode.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult postStartStopWaitingModeRequest2PmCore(const pmcore::BdAddress& deviceAddress,
         const pmcore::StartStop startStopWaitingMode, const pmcore::ActType act = 0u);

   /**
    * This function is invoked to dial a call in the HFP connected device
    * corresponding to the given device address.
    *
    * @param[in] BdAddress& deviceAddress - BT device address
    *            DialInfo& dialInfo - Reference to DialInfo
    *            ActType act - Generated token
    * @param[out] PmResult - Reference to PmResult
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult postDialRequest2PmCore(const pmcore::BdAddress& deviceAddress, const DialInfo& dialInfo,
         const pmcore::ActType act = 0u);

   /**
    * Post the request to PmCore - CPP interface
    *
    * @param[in] CallInstance - call instance
    *            ActType - Token to match request and response
    *            AcceptOperation - specify the operation to be done on an incoming call
    *            BdAddress& - Device address
    * @param[out] PmResult - Gives the success or error message.
    * @param[in,out]
    *
    * @return
    *
    */
   pmcore::PmResult postAcceptRequest2PmCore(const pmcore::CallInstance callInstance,
         const pmcore::ActType act = 0u, const pmcore::BdAddress& devAddress = "",
         const pmcore::AcceptOperation operation = pmcore::ACCEPT_CALLS);

   /**
    * This function is invoked to swap the calls in the HFP connected device
    * corresponding to the given device address.
    *
    * @param[in] pmcore::BdAddress& deviceAddress - BD Address of the device
    *            pmcore::SwapOperation operation - swap call operation type
    *            pmcore::ActType act - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult postSwapCallRequest2PmCore(const pmcore::BdAddress& deviceAddress,
         const pmcore::SwapOperation operation, const pmcore::ActType act = 0u);

   /**
    * This function is used to invoke speed dial in the HFP connected device
    * corresponding to the given device address for the given memDialIndex.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult postSpeedDialRequest2PmCore(const pmcore::BdAddress& deviceAddress,
         const pmcore::MemDialIndex memDialIndex,
         const pmcore::HideCallerId hideCallerId, const pmcore::ActType act = 0u);

   /**
    * This function is invoked speed send the string of DTMF tones to the HFP connected device
    * corresponding to the given device address.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult sendDTMFRequest2PmCore(const pmcore::BdAddress& deviceAddress, const pmcore::DTMFTones& tones,
         const pmcore::ActType act = 0u);

   /**
    * This function is invoked to transfer the call to HF/HS in the HFP connected device
    * corresponding to the given device address.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult postTransferAudioRequest2PmCore(const pmcore::BdAddress& deviceAddress,
         const pmcore::AudioDirection audioDirection, const pmcore::ActType act = 0u);

   /**
    * This function is invoked to get the voice recognition status in the HFP connected device
    * corresponding to the given device address.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult getVoiceRecognitionStatus(const pmcore::BdAddress& deviceAddress, const pmcore::ActType act = 0u);

   /**
    * This function is invoked to start/stop the voice recognition status in the HFP connected device
    * corresponding to the given device address.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult postStartStopVoiceRecognitionRequest2PmCore(const pmcore::BdAddress& deviceAddress,
         const pmcore::StartStop startStopVR, const pmcore::ActType act = 0u);

   /**
    * This function is used to get the availability status of the Enhanced voice recognition feature
    * in the HFP connected device corresponding to the given device address.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult getEnhancedVoiceRecognitionFeature(const pmcore::BdAddress& deviceAddress,
         const pmcore::ActType act = 0u);

   /**
    * This function is invoked to get the external voice recognition status in the HFP connected device
    * corresponding to the given device address.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult getExternalVRState(const pmcore::BdAddress& deviceAddress, const pmcore::ActType act = 0u);

   /**
    * This function is invoked to activate the external voice recognition status in the HFP connected device
    * corresponding to the given device address.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult postActivateExternalVRRequest2PmCore(const pmcore::BdAddress& deviceAddress,
         const pmcore::ActType act = 0u);

   /**
    * This function is invoked to deactivate the external voice recognition status in the HFP connected device
    * corresponding to the given device address.
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   pmcore::PmResult postDeActivateExternalVRRequest2PmCore(const pmcore::BdAddress& deviceAddress,
         const pmcore::ActType act = 0u);

   /**
    * This function is used to stop the extended VR session since READYTOPLAY update is received from Media player.
    *
    * @param[in] BdAddress& deviceAddress - Device address.
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void stopExtendedVRSession(IN const pmcore::BdAddress& deviceAddress);

   /**
    * Generic method from where the responses are converted to PmCoreResponseData type
    * and posted to IpcMessageDispatcher.
    *
    * @param[in] pmcore::PmResult& pmResult - Result received from PM Core
    *            pmcore::ActType act - Token
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void postPmCoreResponseData(const pmcore::PmResult& pmResult, const pmcore::ActType act);

   /**
    * Generic method from where the responses with parameters including the Device address
    * are converted to PmCoreResponseData type and posted to IpcMessageDispatcher.
    *
    * @param[in] pmcore::PmResult& pmResult - Result received from PM Core
    *            pmcore::BdAddress& deviceAddress - BT address of the device
    *            pmcore::ActType act - Token
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   template<typename TReponseType>
   void postPmCoreResponseData(const pmcore::PmResult& pmResult, const pmcore::BdAddress& deviceAddress,
         const pmcore::ActType act = 0u);

   /**
    * Generic method from where the responses with parameters including the Device address and call instance
    * are converted to PmCoreResponseData type and posted to IpcMessageDispatcher.
    *
    * @param[in] pmcore::PmResult& pmResult - Result received from PM Core
    *            pmcore::BdAddress& deviceAddress - BT address of the device
    *            pmcore::CallInstance callInstance
    *            pmcore::ActType act - Token
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   template<typename TReponseType>
   void postPmCoreResponseData(const pmcore::PmResult& pmResult, const pmcore::BdAddress& deviceAddress,
         const pmcore::CallInstance callInstance, const pmcore::ActType act = 0u);

   /**
    * Generic method from where the responses are posted to IpcMessageDispatcher. From IpcMessageDispatcher,
    * the message is posted to the appropriate MessageHandler from where the result to clients are posted back.
    *
    * @param[in] const PmCoreResponseData* - pointer to the Base class for all MethodResult data type
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void postResponseToMessageDispatcher(const std::shared_ptr<PmCoreResponseData>& message);

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::SystemWideRingtonesList& - reference to pmcore::SystemWideRingtonesList class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void getSystemWideRingtonesListResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::SystemWideRingtonesList& systemWideRingtonesList, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::SystemWideRingtonesList& - reference to pmcore::SystemWideRingtonesList class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onSystemWideRingtonesListChanged(
         IN const pmcore::SystemWideRingtonesList& systemWideRingtonesList) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::DeviceRingtoneList& - reference to pmcore::DeviceRingtoneList class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void getDeviceRingtoneListResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::DeviceRingtoneList& deviceRingtoneList, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::DeviceRingtoneList& - reference to pmcore::DeviceRingtoneList class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onDeviceRingtoneListChanged(IN const pmcore::DeviceRingtoneList& deviceRingtoneList) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void setRingtoneResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void playRingtoneResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void stopRingtoneResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::SuppressRingtoneOnOffList& - reference to pmcore::SuppressRingtoneOnOffList class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void getSuppressRingtoneOnOffListResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::SuppressRingtoneOnOffList& suppressRingtoneOnOffList, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::SuppressRingtoneOnOffList& - reference to pmcore::SuppressRingtoneOnOffList class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onSuppressRingtoneOnOffListChanged(
         IN const pmcore::SuppressRingtoneOnOffList& suppressRingtoneOnOffList) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void suppressRingtoneOnOffResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::AutoWaitingModeOnOffList& - reference to pmcore::AutoWaitingModeOnOffList class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void getAutoWaitingModeOnOffListResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::AutoWaitingModeOnOffList& autoWaitingModeOnOffList, IN const pmcore::ActType act = 0u) override;

   /**
    *  Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::AutoWaitingModeOnOffList& - reference to pmcore::AutoWaitingModeOnOffList class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onAutoWaitingModeOnOffListChanged(
         IN pmcore::AutoWaitingModeOnOffList& autoWaitingModeOnOffList) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void autoWaitingModeOnOffResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void setWaitingModeFilePathResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::WaitingModeFilePath& - reference to pmcore::WaitingModeFilePath class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void getWaitingModeFilePathResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::WaitingModeFilePath& waitingModeFilePath, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::WaitingModeFilePath& - reference to pmcore::WaitingModeFilePath class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onWaitingModeFilePathChanged(IN const pmcore::WaitingModeFilePath& waitingModeFilePath) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::PhoneCallAudioActive& - reference to pmcore::PhoneCallAudioActive class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void getPhoneCallAudioActiveResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::PhoneCallAudioActive& phoneCallAudioActive, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PhoneCallAudioActive& - reference to pmcore::PhoneCallAudioActive class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onPhoneCallAudioActiveChanged(IN const pmcore::PhoneCallAudioActive& phoneCallAudioActive) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::ActivePassiveDeviceList& - reference to pmcore::ActivePassiveDeviceList class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void getActivePassiveDeviceListResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::ActivePassiveDeviceList& activePassiveDeviceList, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::ActivePassiveDeviceList& - reference to pmcore::ActivePassiveDeviceList class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onActivePassiveDeviceListChanged(
         IN const pmcore::ActivePassiveDeviceList& activePassiveDeviceList) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void setActivePhoneResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void restoreDefaultSettingResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::NotificationEvent& - reference to pmcore::NotificationEvent class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onNotificationEvent(IN const pmcore::NotificationEvent& notificationEvent) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::SupportedFeatures& - reference to pmcore::SupportedFeatures class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void getSupportedFeaturesResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::SupportedFeatures& supportedFeatures, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::SupportedFeatures& - reference to pmcore::SupportedFeatures class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onSupportedFeaturesChanged(IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::SupportedFeatures& supportedFeatures) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::NetworkStatus& - reference to pmcore::NetworkStatus class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void getNetworkStatusResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::NetworkStatus& networkStatus,
         IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::NetworkStatus& - reference to pmcore::NetworkStatus class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onNetworkStatusChanged(IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::NetworkStatus& networkStatus) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::SignalStrength& - reference to pmcore::SignalStrength class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void getSignalStrengthResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::SignalStrength& signalStrength, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::SignalStrength& - reference to pmcore::SignalStrength class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onSignalStrengthChanged(IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::SignalStrength& signalStrength) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::BatteryChargeLevel& - reference to pmcore::BatteryChargeLevel class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void getBatteryChargeLevelResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::BatteryChargeLevel& batteryChargeLevel, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::BatteryChargeLevel& - reference to pmcore::BatteryChargeLevel class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onBatteryChargeLevelChanged(IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::BatteryChargeLevel& batteryChargeLevel) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::NetworkOperator& - reference to pmcore::NetworkOperator class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void getNetworkOperatorResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::NetworkOperator& networkOperator, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::NetworkOperator& - reference to pmcore::NetworkOperator class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onNetworkOperatorChanged(IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::NetworkOperator& networkOperator) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::VoiceMailList& - reference to pmcore::VoiceMailList class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void voiceMailListResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::VoiceMailList& voiceMailList, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::CallStatusList& - reference to pmcore::CallStatusList class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void getCallStatusListResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::CallStatusList& callStatusList, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::CallStatusList& - reference to pmcore::CallStatusList class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onCallStatusListChanged(IN const pmcore::CallStatusList& callStatusList) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::MicrophoneMuteState& - reference to pmcore::MicrophoneMuteState class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void getMicrophoneMuteStateResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::MicrophoneMuteState& microphoneMuteState, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::MicrophoneMuteState& - reference to pmcore::MicrophoneMuteState class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onMicrophoneMuteStateChanged(IN const pmcore::MicrophoneMuteState& microphoneMuteState) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void setMicrophoneMuteStateResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void mergeCallsResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return
    */
   virtual void splitCallsResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void hangupCallsResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::CallInstance& - call instance
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void redialResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::CallInstance callInstance, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::SCOConnection& - reference to pmcore::SCOConnection class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void getSCOConnectionResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::SCOConnection& scoConnection, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::SCOConnection& - reference to pmcore::SCOConnection class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onSCOConnectionChanged(IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::SCOConnection& scoConnection) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::RingtoneMuteState& - reference to pmcore::RingtoneMuteState class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void getRingtoneMuteStateResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::RingtoneMuteState& ringtoneMuteState, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::RingtoneMuteState& - reference to pmcore::RingtoneMuteState class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onRingtoneMuteStateChanged(IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::RingtoneMuteState& ringtoneMuteState) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void setRingtoneMuteStateResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::WaitingModeState& - reference to pmcore::WaitingModeState class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void getWaitingModeStateResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::WaitingModeState& waitingModeState, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::WaitingModeState& - reference to pmcore::WaitingModeState class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onWaitingModeStateChanged(IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::WaitingModeState& waitingModeState) override;

   /**
    * This method is used to get the  current playback status of the vehicle ringtone
    *
    * @param[in] PmResult& - reference to PmResult class
    * @param[in] ringtonePlaybackState& - reference to RingtonePlaybackState class
    * @param[in] ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void getRingtonePlaybackStatusResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::RingtonePlaybackState& ringtonePlaybackState, IN const pmcore::ActType act = 0u) override;

   /**
    * This method is used to notify the changes in ringtonePlaybackState of the vehicle ringtone.
    *
    * @param[in] ringtonePlaybackState& - reference to RingtonePlaybackState class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onRingtonePlaybackStatusChanged(
         IN const pmcore::RingtonePlaybackState& ringtonePlaybackState) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void startStopWaitingModeResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::CallInstance - call instance
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void dialResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::CallInstance callInstance, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::CallInstance - call instance
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void acceptCallResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::CallInstance callInstance, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void swapCallResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::CallInstance - call instance
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void speedDialResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::CallInstance callInstance, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void sendDTMFResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void transferAudioResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::VoiceRecognitionStatus& - reference to pmcore::VoiceRecognitionStatus class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void getVoiceRecognitionStatusResponse(IN const pmcore::PmResult& pmResult, IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::VoiceRecognitionStatus& voiceRecognitionStatus, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::VoiceRecognitionStatus& - reference to pmcore::VoiceRecognitionStatus class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onVoiceRecognitionStatusChanged(IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::VoiceRecognitionStatus& voiceRecognitionStatus) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::BdAddress& deviceAddress - BT device address
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void startStopVoiceRecognitionResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::EnhancedVoiceRecognitionFeature& - reference to pmcore::EnhancedVoiceRecognitionFeature class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void getEnhancedVoiceRecognitionFeatureResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::BdAddress& deviceAddress, IN const pmcore::EnhancedVoiceRecognitionFeature& enhancedVoiceRecognitionFeature,
         IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::EnhancedVoiceRecognitionFeature& - reference to pmcore::EnhancedVoiceRecognitionFeature class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onEnhancedVoiceRecognitionFeatureChanged(IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::EnhancedVoiceRecognitionFeature& enhancedVoiceRecognitionFeature) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::PmResult& - reference to pmcore::PmResult class
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::ExtVoiceRecognitionStatus& - external VR status
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void getExternalVRStateResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::ExtVoiceRecognitionStatus& extVoiceRecognitionStatus,
         IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::ExtVoiceRecognitionStatus& - external VR status
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onExternalVRStateChanged(IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::ExtVoiceRecognitionStatus& extVoiceRecognitionStatus) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] PmResult& - reference to PmResult class
    * @param[in] BdAddress& deviceAddress
    * @param[in] ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void activateExternalVRResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::BdAddress& deviceAddress, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in] PmResult& - reference to PmResult class
    * @param[in] BdAddress& deviceAddress
    * @param[in] ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void deActivateExternalVRResponse(IN const pmcore::PmResult& pmResult,
         IN const pmcore::BdAddress& deviceAddress, IN const pmcore::ActType act = 0u) override;

   /**
    * Overridden from IPmCoreCallbackIf
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   virtual void onPmStateChanged(IN const pmcore::BdAddress& deviceAddres,
         IN const pmcore::CallExistenceState callExistenceState) override;

   /**
    * This method is used to notify the changes in pmcore::FriendlyName of AudioGateway.
    *
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::FriendlyName& - reference to pmcore::FriendlyName class
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void onDeviceFriendlyNameChanged(IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::FriendlyName& friendlyName);

   /**
    * This method is invoked to get the friendly name of the paired/connected device.
    *
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return pmcore::PmResult - Result
    */
   pmcore::PmResult getDeviceFriendlyName(IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::ActType act = 0u);

   /**
    * This method provides the friendly name of the paired/connected device.
    *
    * @param[in] pmcore::BdAddress& - device address
    * @param[in] pmcore::FriendlyName& - reference to pmcore::FriendlyName class
    * @param[in] pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return void
    */
   void getDeviceFriendlyNameResponse(IN const pmcore::BdAddress& deviceAddress,
         IN const pmcore::FriendlyName& friendlyName,
         IN const pmcore::ActType act = 0u);

   /**
    * This function is used to clear the HFP connected device in the STL container class and
    * inform PM core to handle the device deletion.
    *
    * @param[in] BdAddress& deviceAddress - Device address of the deleted device.
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void onDeviceDeletion(const pmcore::BdAddress& deviceAddress);

   /**
    * Member function overridden from ILoopbackReceiver class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return unsigned long long int - Acknowledgment token
    */
   ResponseToken postLoopbackMessage(com::bosch::pmcommon::LoopbackData* loopbackData) override;

   /**
    * This function is used to update the calls report list to the clients(call status + searchPhonebook response)
    *
    * @param[in]  CallsReport& CallsReport - Calls report received after search Phonebook response
    *             ActType act - If act is valid, then the corresponding response should be updated
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void updateCallsReportListToClients(const com::bosch::pmapp::CallsReportList& callsReportList,
         const pmcore::ActType act = pmcore::PM_DEFAULT_ACT);

private:

   friend class PmSingleton<IpcWrapper>;

   /**
    * Constructor of the IpcWrapper class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   IpcWrapper();

   /**
    * Destructor of the IpcWrapper class
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   ~IpcWrapper();

   /**
    * Method to detect the Ipc Request type - CCA / ASF-DBus. Based on this decision the appropriate Message handler
    * is specified in the response message and sent to IpcMessageDispatcher
    *
    * @param[in] const pmcore::ActType - asynchronous completion token
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   unsigned short int getIpcRequestType(const pmcore::ActType actToken);

   /**
    * This function is used to insert the paired device info into the STL container class and
    * inform PM core to handle the device pairing.
    *
    * @param[in] DeviceDetails& deviceDetails - Referenced object of the DeviceDetails class
    *
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void onDevicePaired(const ::com::bosch::pmapp::DeviceDetails& deviceDetails);

   /**
    * This function is used to update the HFP connected device info in the STL container class and
    * inform PM core to handle the device connection.
    *
    * @param[in] DeviceDetails& deviceDetails - Referenced object of the DeviceDetails class
    *
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void onDeviceConnected(const ::com::bosch::pmapp::DeviceDetails& deviceDetails);

   /**
    * This function is used to update the HFP connected device info in the STL container class and
    * inform PM core to handle the device connection.
    *
    * @param[in] pmcore::BdAddress& deletedDeviceAddress - Device address of the device that is deleted.
    *
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void onDeviceDisconnected(const pmcore::BdAddress& deletedDeviceAddress);

   /**
    * This function is used to dispatch the Get or Set response
    * of RingtoneMutestate property from PM core to the IPC dispatcher
    *
    * @param[in] pmcore::PmResult& pmResult - Result received from PM core
    *            pmcore::BdAddress& deviceAddress - Bluetooth address of the device
    *            pmcore::RingtoneMuteState& ringtoneMuteState - pmcore::RingtoneMuteState object
    *            pmcore::ActType act - Generated token
    *
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void dispatchRingtoneMuteStateResponse(const pmcore::PmResult& pmResult, const pmcore::BdAddress& deviceAddress,
         const pmcore::RingtoneMuteState& ringtoneMuteState, const pmcore::ActType act);

   /**
    * This function is used to dispatch the Get or Set response
    * of MicrophoneMutestate property from PM core to the IPC dispatcher
    *
    * @param[in] pmcore::PmResult& pmResult - Result received from PM core
    *            MicrophoneMutestate& microphoneMutestate - pmcore::RingtoneMuteState object
    *            pmcore::ActType act - Generated token
    *
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void dispatchMicMuteStateResponse(const pmcore::PmResult& pmResult,
         const pmcore::MicrophoneMuteState& microphoneMuteState, const pmcore::ActType act);

   /**
    * This function is used to dispatch the Get or Set response
    * of VoiceRecognition property from PM core to the IPC dispatcher
    *
    * @param[in] pmcore::PmResult& pmResult - Result received from PM core
    *            pmcore::VoiceRecognitionStatus& voiceRecognitionStatus - VR status object
    *            pmcore::ActType act - Generated token
    *
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void dispatchVoiceRecognitionStateResponse(const pmcore::PmResult& pmResult,
         const pmcore::BdAddress& deviceAddress,
         const pmcore::VoiceRecognitionStatus& voiceRecognitionStatus, const pmcore::ActType act);

   /**
    * This function is used to dispatch the Get or Set response
    * of pmcore::WaitingModeFilePath property from PM core to the IPC dispatcher
    *
    * @param[in] pmcore::PmResult& pmResult - Result received from PM core
    *            pmcore::WaitingModeFilePath& waitingModeFilePath - pmcore::WaitingModeFilePath status object
    *            pmcore::ActType act - Generated token
    *
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void dispatchWaitingModeFilePathResponse(const pmcore::PmResult& pmResult,
         const pmcore::WaitingModeFilePath& waitingModeFilePath,
         const pmcore::ActType act);

   IpcMessageDispatcher*    _ipcMessageDispatcher; /**< Pointer for the IPC message dispatcher */
   com::bosch::PmComponent* _pmComponent;          /**< Pointer for PmComponent */
};

} // namespace pm_ipc_wrapper

#endif // #define IpcWrapper_h
