/**
 * @file IpcMessageDispatcher.h
 *
 * @swcomponent PhoneCallManager
 *
 * @brief This file contains the definition of the IpcMessageDispatcher class
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *            The reproduction, distribution and utilization of this file as
 *            well as the communication of its contents to others without express
 *            authorization is prohibited. Offenders will be held liable for the
 *            payment of damages. All rights reserved in the event of the grant
 *            of a patent, utility model or design.
 *
 * @details
 *
 * @ingroup IpcWrapper
 */

#ifndef IpcMessageDispatcher_h
#define IpcMessageDispatcher_h

#include <list>
#include "IpcMessageHandler.h"

/*
 * This class is responsible for dispatching the IPC message request to the appropriate message handler.
 * E.X. A CCA message will be dispatched to a CCA Message handler.
 */
class IpcMessageDispatcher
{

public:

   /**
    * Regsiter message handlers to receive property updates from CPP layer
    *
    * @param[in] MessageHandlerBase* pointer to the message handler base class
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void registerMsgHandler(MessageHandlerBase* pMsgHandler);

   /**
    * UnRegsiter message handlers to stop receiving property updates from CPP layer
    * Right now, it removes all the message handlers from the list
    *
    * @param[in]
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void unRegisterMsgHandlers();

   /**
    * Process the property updates from CPP interfaces. This method shall be overridden in the
    * respective derived classes property handlers to update the property status to the clients
    *
    *
    * @param[in] PropertyUpdate* - Property data to be updated to the registered clients
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   void updatePropertyChanged(PropertyUpdate * pPropertyUpdate);

   /**
    * Method to dispatch the IPC message to the respective message handler class
    * Hence a template method, to support message handler of any IPC type
    *
    * @param[in] MessageHandler* pointer to the message handler class
    *            IpcMessageType*  <template> - place holder of the IPC message type
    * @param[out]
    * @param[in,out]
    *
    * @return
    *
    */
   template <typename TMessage>
   void dispatchMessage(MessageHandlerBase* handler,TMessage* message)
   {
      /**
       * This is where the dynamic dispatch happens, i.e. during runtime it is decided upon which message has
       * to be passed on to which message handler. e.x. a CCA message will be passed to a CcaMessageHandler class.
       * An ASF message request, shall be passed to the AsfMessageHandler class. This dispatcher is also accessed
       * from the bottom layer of IpcWrapper layer to dispatch the responses from CPP layer to the appropriate
       * message handlers based on the requests.
       *
       * Refer: https://en.wikipedia.org/wiki/Double_dispatch for the concept
       *
       * @attention This is a crucial portion, since any wrong casting or missing handler
       * will lead to a bad cast exception. To overcome this, casting is done to a pointer
       * which returns NULL instead of throwing exception!
       *
       */
      IpcMessageHandler<TMessage> *pMsgHandler = dynamic_cast<IpcMessageHandler<TMessage>*>(handler);

      if(pMsgHandler)
      {
         if(message)
         {
            pMsgHandler->processMessage(message);
         }
         else
         {
            // Code flow reaches here if the message received from CCA / PMCore is not a valid message
            // TODO: Add an ETG error trace
         }
      }
      else
      {
         // This code portion Should not be executed, execution reaches here only if coding is not done correctly
         // TODO: Add an ETG error trace
      }
   }

private:

   std::list<MessageHandlerBase*> _msgHandlerlist; /**< Message handler list that shall be used to dispatch messages */
};

#endif // IpcMessageDispatcher_h
